//%LICENSE////////////////////////////////////////////////////////////////
//
// Licensed to The Open Group (TOG) under one or more contributor license
// agreements.  Refer to the OpenPegasusNOTICE.txt file distributed with
// this work for additional information regarding copyright ownership.
// Each contributor licenses this file to you under the OpenPegasus Open
// Source License; you may not use this file except in compliance with the
// License.
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included
// in all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
// IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
// CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
// TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
// SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
//////////////////////////////////////////////////////////////////////////
//
//%/////////////////////////////////////////////////////////////////////////////

/****************************************************************************
**
**    This executable generates output to stdout for a message bundle
**    representing the help messages for Config Properties. It can be used
**    at any time to generate this text which must be then manually
**    inserted into the Server message bundle.
**    It generates the message bundle text from the help Strings defined
**    in the file src/Pegasus/Config/ConfigPropertyHelp.cpp
**
**    Normally not compiled. Compile and use only when the text in the table
**    changes.
**
****************************************************************************/
#include <Pegasus/Common/Config.h>
#include <Pegasus/Common/PegasusAssert.h>
#include <Pegasus/Common/Constants.h>
#include <Pegasus/Config/ConfigPropertyHelp.h>

PEGASUS_USING_PEGASUS;

PEGASUS_USING_STD;

static Boolean verbose;

//  Creates string of form:
//      Config.DefaultPropertyOwner.<propertyName>.DESCRIPTION
String _createKey(const char* propertyName)
{
   String rtn = "Config.ConfigPropertyHelp.";
   rtn.append("DESCRIPTION_");
   rtn.append(propertyName);
   return rtn;
}

String _descriptionWithPrefix(const char* description)
{
   String rtn = description;
   return rtn;
}

static String _fill(Uint32 count)
{
    String str;
    for (Uint32 i = 0 ; i < count ; i++)
    {
        str.append(' ');
    }
    return str;
}

// Indents a String including all
String indentString(
    const String& input,
    Uint32 leftMargin,
    Boolean indentFirstLine )
{
    String fill = _fill(leftMargin);
    String output;
    if (indentFirstLine)
    {
        output.append(fill);
    }
    for (Uint32 i = 0 ; i < input.size(); i++ )
    {
        if (input[i] == '\n')
        {
            output.append(input[i]);
            output.append(fill);
        }
        else
        {
            output.append(input[i]);
        }
    }
    return output;
}

int main()
{
    verbose = getenv ("PEGASUS_TEST_VERBOSE") ? true : false;
    /*
        Function to output the full text of all messages in the
        configPropertyDescriptionList to a file in a form compatible with
        the message bundle format.  This is used ONLY as a way to generate
        input for the Message bundle rather than to try to manually keep
        the bundle and these files in sync.
    */

     String out = indentString(
         "// ==START_CONFIG_PROPERTY_HELP MSGS==========================\n"
         "//  Description Messages for ConfigPropertyHelp\n"
         "//  These messages do not have a number prefix.\n"
         "//  These msgs are auto-generated by Config/tests/BldMsgBndl\n"
         "//  and are defined in source file\n"
         "//  Pegasus/Config/ConfigPropertyOwner.cpp\n"
         "//  Do NOT translate words or phrases enclosed in single\n"
         "//  quotes (ex 'required').\n"
         "//  PLEASE auto-generate this if config property help messages "
              " change.\n"
         "// ==========================================================\n",
         8, true);
     out.append("\n");

     for (Uint32 i = 0; i < configPropertyDescriptionListSize ; i++)
     {
         String description = _descriptionWithPrefix(
              configPropertyDescriptionList[i].Description);
         String fixedDescription;

         // Map selected characters to get correct output for the bundle
         // /n maps to new line representation in output string, quote
         // new line indent, newline to show the \n to the translator
         // It also fixes any internalized double quotes so that a \" is output
         //
         for (Uint32 j = 0; j < description.size(); j++)
         {
             switch (description[j])
             {
                 case '\n':
                     fixedDescription.append("\\n\"\n            \"");
                     break;
                 case '\"':
                     fixedDescription.append("\\\"");
                     break;
                 default:
                     fixedDescription.append(description[j]);
             }
         }

         // Output data for the pegasusServer_en.txt file for this
         // key where key includes config property name.
         // Indenting these lines is handled during the mapping to
         // fixedDescription.
         out.append( "        " );
         out.append(_createKey(
                     configPropertyDescriptionList[i].name));
         out.append( ":string {\"");
         out.append(fixedDescription);
         out.append("\"}");
         out.append("\n\n");
     }

     out.append(indentString(
             "// ==END_CONFIG_PROPERTY_HELP MSGS "
             "DO NOT HAND EDIT THE ABOVE==========",
                 8, true));
     out.append("\n\n");

     cout << out;

     return 0;
}
