from __future__ import print_function, division, absolute_import

#
# GUI Module for the Autobind Wizard
#
# Copyright (c) 2012 Red Hat, Inc.
#
# This software is licensed to you under the GNU General Public License,
# version 2 (GPLv2). There is NO WARRANTY for this software, express or
# implied, including the implied warranties of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. You should have received a copy of GPLv2
# along with this software; if not, see
# http://www.gnu.org/licenses/old-licenses/gpl-2.0.txt.
#
# Red Hat trademarks are not licensed under GPLv2. No permission is
# granted to use or replicate Red Hat trademarks that are incorporated
# in this software or its documentation.
#
import logging

log = logging.getLogger(__name__)


class DryRunResult(object):
    """ Encapsulates a dry-run autobind result from the server. """

    def __init__(self, service_level, server_json, cert_sorter):
        self.json = server_json
        self.sorter = cert_sorter
        self.service_level = service_level

    def covers_required_products(self):
        """
        Return True if this dry-run result would cover all installed
        products which are not covered by a valid entitlement.

        NOTE: we do not require full stacking compliance here. The server
        will return the best match it can find, but that may still leave you
        only partially entitled. We will still consider this situation a valid
        SLA to use, the key point being you have access to the content you
        need.
        """
        required_products = set(self.sorter.unentitled_products.keys())

        # The products that would be covered if we did this autobind:
        autobind_products = set()

        for pool_quantity in self.json:
            pool = pool_quantity['pool']
            # This is usually the MKT product and has no content, but it
            # doesn't hurt to include it:
            autobind_products.add(pool['productId'])
            for provided_prod in pool['providedProducts']:
                autobind_products.add(provided_prod['productId'])
        log.debug("Autobind would give access to: %s" % autobind_products)
        if required_products.issubset(autobind_products):
            log.debug("Found valid service level: %s" % self.service_level)
            return True
        else:
            log.debug("Service level does not cover required products: %s" %
                      self.service_level)
            return False

    def get_pool_quantities(self):
        """
        Returns a list of tuples, each of which is a pool ID and a quantity
        to consume. Used when we actually decide to bind to this pool.
        """
        tuples = []
        for pool_quantity in self.json:
            tuples.append((pool_quantity['pool']['id'],
                pool_quantity['quantity']))
        return tuples


class ServiceLevelNotSupportedException(Exception):
    """
    Exception for AutobindController.load. The remote candlepin doesn't
    support service levels.
    """
    pass


class AllProductsCoveredException(Exception):
    """
    Exception for AutobindController.load. The system doesn't have any
    products that are in need of entitlements.
    """
    pass


class NoProductsException(Exception):
    """
    Exception for AutobindController.load. The system has no products, and
    thus needs no entitlements.
    """
    pass
