#!/usr/bin/env python3
#
# Copyright (C) 2018 Google, Inc.
#
# Permission is hereby granted, free of charge, to any person obtaining a
# copy of this software and associated documentation files (the "Software"),
# to deal in the Software without restriction, including without limitation
# the rights to use, copy, modify, merge, publish, distribute, sublicense,
# and/or sell copies of the Software, and to permit persons to whom the
# Software is furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included
# in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
# THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
# DEALINGS IN THE SOFTWARE.

"""Generate Vulkan wrapper to support Android without libvulkan
"""

import os
import sys

class Command(object):
    PLATFORM = 0
    LOADER = 1
    INSTANCE = 2
    DEVICE = 3

    def __init__(self, name, dispatch):
        self.name = name
        self.dispatch = dispatch
        self.ty = self._get_type()

    @staticmethod
    def valid_c_typedef(c):
        return (c.startswith("typedef") and
                c.endswith(");") and
                "*PFN_vkVoidFunction" not in c)

    @classmethod
    def from_c_typedef(cls, c):
        name_begin = c.find("*PFN_vk") + 5 # instead of 7 to restore vk
        name_end = c.find(")(", name_begin)
        name = c[name_begin:name_end]

        dispatch_begin = name_end + 2
        dispatch_end = c.find(" ", dispatch_begin)
        dispatch = c[dispatch_begin:dispatch_end]
        if not dispatch.startswith("Vk"):
            dispatch = None

        return cls(name, dispatch)

    def _get_type(self):
        if self.dispatch:
            if self.dispatch in ["VkDevice", "VkQueue", "VkCommandBuffer"]:
                return self.DEVICE
            else:
                return self.INSTANCE
        else:
            if self.name in ["GetInstanceProcAddr"]:
                return self.PLATFORM
            else:
                return self.LOADER

    def __repr__(self):
        return "Command(name=%s, dispatch=%s)" % \
                (repr(self.name), repr(self.dispatch))

class Extension(object):
    def __init__(self, name, version, guard=None, commands=[]):
        self.name = name
        self.version = version
        self.guard = guard
        self.commands = commands[:]

    def add_command(self, cmd):
        self.commands.append(cmd)

    def __repr__(self):
        lines = []
        lines.append("Extension(name=%s, version=%s, guard=%s, commands=[" %
                (repr(self.name), repr(self.version), repr(self.guard)))

        for cmd in self.commands:
            lines.append("    %s," % repr(cmd))

        lines.append("])")

        return "\n".join(lines)

# generated by "generate_vulkan_wrapper.py parse vulkan.h"
VK_core_0 = Extension(name='VK_core_0', version=0, guard=None, commands=[
    Command(name='vkCreateInstance', dispatch=None),
    Command(name='vkDestroyInstance', dispatch='VkInstance'),
    Command(name='vkEnumeratePhysicalDevices', dispatch='VkInstance'),
    Command(name='vkGetPhysicalDeviceFeatures', dispatch='VkPhysicalDevice'),
    Command(name='vkGetPhysicalDeviceFormatProperties', dispatch='VkPhysicalDevice'),
    Command(name='vkGetPhysicalDeviceImageFormatProperties', dispatch='VkPhysicalDevice'),
    Command(name='vkGetPhysicalDeviceProperties', dispatch='VkPhysicalDevice'),
    Command(name='vkGetPhysicalDeviceQueueFamilyProperties', dispatch='VkPhysicalDevice'),
    Command(name='vkGetPhysicalDeviceMemoryProperties', dispatch='VkPhysicalDevice'),
    Command(name='vkGetInstanceProcAddr', dispatch='VkInstance'),
    Command(name='vkGetDeviceProcAddr', dispatch='VkDevice'),
    Command(name='vkCreateDevice', dispatch='VkPhysicalDevice'),
    Command(name='vkDestroyDevice', dispatch='VkDevice'),
    Command(name='vkEnumerateInstanceExtensionProperties', dispatch=None),
    Command(name='vkEnumerateDeviceExtensionProperties', dispatch='VkPhysicalDevice'),
    Command(name='vkEnumerateInstanceLayerProperties', dispatch=None),
    Command(name='vkEnumerateDeviceLayerProperties', dispatch='VkPhysicalDevice'),
    Command(name='vkGetDeviceQueue', dispatch='VkDevice'),
    Command(name='vkQueueSubmit', dispatch='VkQueue'),
    Command(name='vkQueueWaitIdle', dispatch='VkQueue'),
    Command(name='vkDeviceWaitIdle', dispatch='VkDevice'),
    Command(name='vkAllocateMemory', dispatch='VkDevice'),
    Command(name='vkFreeMemory', dispatch='VkDevice'),
    Command(name='vkMapMemory', dispatch='VkDevice'),
    Command(name='vkUnmapMemory', dispatch='VkDevice'),
    Command(name='vkFlushMappedMemoryRanges', dispatch='VkDevice'),
    Command(name='vkInvalidateMappedMemoryRanges', dispatch='VkDevice'),
    Command(name='vkGetDeviceMemoryCommitment', dispatch='VkDevice'),
    Command(name='vkBindBufferMemory', dispatch='VkDevice'),
    Command(name='vkBindImageMemory', dispatch='VkDevice'),
    Command(name='vkGetBufferMemoryRequirements', dispatch='VkDevice'),
    Command(name='vkGetImageMemoryRequirements', dispatch='VkDevice'),
    Command(name='vkGetImageSparseMemoryRequirements', dispatch='VkDevice'),
    Command(name='vkGetPhysicalDeviceSparseImageFormatProperties', dispatch='VkPhysicalDevice'),
    Command(name='vkQueueBindSparse', dispatch='VkQueue'),
    Command(name='vkCreateFence', dispatch='VkDevice'),
    Command(name='vkDestroyFence', dispatch='VkDevice'),
    Command(name='vkResetFences', dispatch='VkDevice'),
    Command(name='vkGetFenceStatus', dispatch='VkDevice'),
    Command(name='vkWaitForFences', dispatch='VkDevice'),
    Command(name='vkCreateSemaphore', dispatch='VkDevice'),
    Command(name='vkDestroySemaphore', dispatch='VkDevice'),
    Command(name='vkCreateEvent', dispatch='VkDevice'),
    Command(name='vkDestroyEvent', dispatch='VkDevice'),
    Command(name='vkGetEventStatus', dispatch='VkDevice'),
    Command(name='vkSetEvent', dispatch='VkDevice'),
    Command(name='vkResetEvent', dispatch='VkDevice'),
    Command(name='vkCreateQueryPool', dispatch='VkDevice'),
    Command(name='vkDestroyQueryPool', dispatch='VkDevice'),
    Command(name='vkGetQueryPoolResults', dispatch='VkDevice'),
    Command(name='vkCreateBuffer', dispatch='VkDevice'),
    Command(name='vkDestroyBuffer', dispatch='VkDevice'),
    Command(name='vkCreateBufferView', dispatch='VkDevice'),
    Command(name='vkDestroyBufferView', dispatch='VkDevice'),
    Command(name='vkCreateImage', dispatch='VkDevice'),
    Command(name='vkDestroyImage', dispatch='VkDevice'),
    Command(name='vkGetImageSubresourceLayout', dispatch='VkDevice'),
    Command(name='vkCreateImageView', dispatch='VkDevice'),
    Command(name='vkDestroyImageView', dispatch='VkDevice'),
    Command(name='vkCreateShaderModule', dispatch='VkDevice'),
    Command(name='vkDestroyShaderModule', dispatch='VkDevice'),
    Command(name='vkCreatePipelineCache', dispatch='VkDevice'),
    Command(name='vkDestroyPipelineCache', dispatch='VkDevice'),
    Command(name='vkGetPipelineCacheData', dispatch='VkDevice'),
    Command(name='vkMergePipelineCaches', dispatch='VkDevice'),
    Command(name='vkCreateGraphicsPipelines', dispatch='VkDevice'),
    Command(name='vkCreateComputePipelines', dispatch='VkDevice'),
    Command(name='vkDestroyPipeline', dispatch='VkDevice'),
    Command(name='vkCreatePipelineLayout', dispatch='VkDevice'),
    Command(name='vkDestroyPipelineLayout', dispatch='VkDevice'),
    Command(name='vkCreateSampler', dispatch='VkDevice'),
    Command(name='vkDestroySampler', dispatch='VkDevice'),
    Command(name='vkCreateDescriptorSetLayout', dispatch='VkDevice'),
    Command(name='vkDestroyDescriptorSetLayout', dispatch='VkDevice'),
    Command(name='vkCreateDescriptorPool', dispatch='VkDevice'),
    Command(name='vkDestroyDescriptorPool', dispatch='VkDevice'),
    Command(name='vkResetDescriptorPool', dispatch='VkDevice'),
    Command(name='vkAllocateDescriptorSets', dispatch='VkDevice'),
    Command(name='vkFreeDescriptorSets', dispatch='VkDevice'),
    Command(name='vkUpdateDescriptorSets', dispatch='VkDevice'),
    Command(name='vkCreateFramebuffer', dispatch='VkDevice'),
    Command(name='vkDestroyFramebuffer', dispatch='VkDevice'),
    Command(name='vkCreateRenderPass', dispatch='VkDevice'),
    Command(name='vkDestroyRenderPass', dispatch='VkDevice'),
    Command(name='vkGetRenderAreaGranularity', dispatch='VkDevice'),
    Command(name='vkCreateCommandPool', dispatch='VkDevice'),
    Command(name='vkDestroyCommandPool', dispatch='VkDevice'),
    Command(name='vkResetCommandPool', dispatch='VkDevice'),
    Command(name='vkAllocateCommandBuffers', dispatch='VkDevice'),
    Command(name='vkFreeCommandBuffers', dispatch='VkDevice'),
    Command(name='vkBeginCommandBuffer', dispatch='VkCommandBuffer'),
    Command(name='vkEndCommandBuffer', dispatch='VkCommandBuffer'),
    Command(name='vkResetCommandBuffer', dispatch='VkCommandBuffer'),
    Command(name='vkCmdBindPipeline', dispatch='VkCommandBuffer'),
    Command(name='vkCmdSetViewport', dispatch='VkCommandBuffer'),
    Command(name='vkCmdSetScissor', dispatch='VkCommandBuffer'),
    Command(name='vkCmdSetLineWidth', dispatch='VkCommandBuffer'),
    Command(name='vkCmdSetDepthBias', dispatch='VkCommandBuffer'),
    Command(name='vkCmdSetBlendConstants', dispatch='VkCommandBuffer'),
    Command(name='vkCmdSetDepthBounds', dispatch='VkCommandBuffer'),
    Command(name='vkCmdSetStencilCompareMask', dispatch='VkCommandBuffer'),
    Command(name='vkCmdSetStencilWriteMask', dispatch='VkCommandBuffer'),
    Command(name='vkCmdSetStencilReference', dispatch='VkCommandBuffer'),
    Command(name='vkCmdBindDescriptorSets', dispatch='VkCommandBuffer'),
    Command(name='vkCmdBindIndexBuffer', dispatch='VkCommandBuffer'),
    Command(name='vkCmdBindVertexBuffers', dispatch='VkCommandBuffer'),
    Command(name='vkCmdDraw', dispatch='VkCommandBuffer'),
    Command(name='vkCmdDrawIndexed', dispatch='VkCommandBuffer'),
    Command(name='vkCmdDrawIndirect', dispatch='VkCommandBuffer'),
    Command(name='vkCmdDrawIndexedIndirect', dispatch='VkCommandBuffer'),
    Command(name='vkCmdDispatch', dispatch='VkCommandBuffer'),
    Command(name='vkCmdDispatchIndirect', dispatch='VkCommandBuffer'),
    Command(name='vkCmdCopyBuffer', dispatch='VkCommandBuffer'),
    Command(name='vkCmdCopyImage', dispatch='VkCommandBuffer'),
    Command(name='vkCmdBlitImage', dispatch='VkCommandBuffer'),
    Command(name='vkCmdCopyBufferToImage', dispatch='VkCommandBuffer'),
    Command(name='vkCmdCopyImageToBuffer', dispatch='VkCommandBuffer'),
    Command(name='vkCmdUpdateBuffer', dispatch='VkCommandBuffer'),
    Command(name='vkCmdFillBuffer', dispatch='VkCommandBuffer'),
    Command(name='vkCmdClearColorImage', dispatch='VkCommandBuffer'),
    Command(name='vkCmdClearDepthStencilImage', dispatch='VkCommandBuffer'),
    Command(name='vkCmdClearAttachments', dispatch='VkCommandBuffer'),
    Command(name='vkCmdResolveImage', dispatch='VkCommandBuffer'),
    Command(name='vkCmdSetEvent', dispatch='VkCommandBuffer'),
    Command(name='vkCmdResetEvent', dispatch='VkCommandBuffer'),
    Command(name='vkCmdWaitEvents', dispatch='VkCommandBuffer'),
    Command(name='vkCmdPipelineBarrier', dispatch='VkCommandBuffer'),
    Command(name='vkCmdBeginQuery', dispatch='VkCommandBuffer'),
    Command(name='vkCmdEndQuery', dispatch='VkCommandBuffer'),
    Command(name='vkCmdResetQueryPool', dispatch='VkCommandBuffer'),
    Command(name='vkCmdWriteTimestamp', dispatch='VkCommandBuffer'),
    Command(name='vkCmdCopyQueryPoolResults', dispatch='VkCommandBuffer'),
    Command(name='vkCmdPushConstants', dispatch='VkCommandBuffer'),
    Command(name='vkCmdBeginRenderPass', dispatch='VkCommandBuffer'),
    Command(name='vkCmdNextSubpass', dispatch='VkCommandBuffer'),
    Command(name='vkCmdEndRenderPass', dispatch='VkCommandBuffer'),
    Command(name='vkCmdExecuteCommands', dispatch='VkCommandBuffer'),
])

VK_core_1 = Extension(name='VK_core_1', version=1, guard=None, commands=[
    Command(name='vkEnumerateInstanceVersion', dispatch=None),
    Command(name='vkBindBufferMemory2', dispatch='VkDevice'),
    Command(name='vkBindImageMemory2', dispatch='VkDevice'),
    Command(name='vkGetDeviceGroupPeerMemoryFeatures', dispatch='VkDevice'),
    Command(name='vkCmdSetDeviceMask', dispatch='VkCommandBuffer'),
    Command(name='vkCmdDispatchBase', dispatch='VkCommandBuffer'),
    Command(name='vkEnumeratePhysicalDeviceGroups', dispatch='VkInstance'),
    Command(name='vkGetImageMemoryRequirements2', dispatch='VkDevice'),
    Command(name='vkGetBufferMemoryRequirements2', dispatch='VkDevice'),
    Command(name='vkGetImageSparseMemoryRequirements2', dispatch='VkDevice'),
    Command(name='vkGetPhysicalDeviceFeatures2', dispatch='VkPhysicalDevice'),
    Command(name='vkGetPhysicalDeviceProperties2', dispatch='VkPhysicalDevice'),
    Command(name='vkGetPhysicalDeviceFormatProperties2', dispatch='VkPhysicalDevice'),
    Command(name='vkGetPhysicalDeviceImageFormatProperties2', dispatch='VkPhysicalDevice'),
    Command(name='vkGetPhysicalDeviceQueueFamilyProperties2', dispatch='VkPhysicalDevice'),
    Command(name='vkGetPhysicalDeviceMemoryProperties2', dispatch='VkPhysicalDevice'),
    Command(name='vkGetPhysicalDeviceSparseImageFormatProperties2', dispatch='VkPhysicalDevice'),
    Command(name='vkTrimCommandPool', dispatch='VkDevice'),
    Command(name='vkGetDeviceQueue2', dispatch='VkDevice'),
    Command(name='vkCreateSamplerYcbcrConversion', dispatch='VkDevice'),
    Command(name='vkDestroySamplerYcbcrConversion', dispatch='VkDevice'),
    Command(name='vkCreateDescriptorUpdateTemplate', dispatch='VkDevice'),
    Command(name='vkDestroyDescriptorUpdateTemplate', dispatch='VkDevice'),
    Command(name='vkUpdateDescriptorSetWithTemplate', dispatch='VkDevice'),
    Command(name='vkGetPhysicalDeviceExternalBufferProperties', dispatch='VkPhysicalDevice'),
    Command(name='vkGetPhysicalDeviceExternalFenceProperties', dispatch='VkPhysicalDevice'),
    Command(name='vkGetPhysicalDeviceExternalSemaphoreProperties', dispatch='VkPhysicalDevice'),
    Command(name='vkGetDescriptorSetLayoutSupport', dispatch='VkDevice'),
])

VK_KHR_surface = Extension(name='VK_KHR_surface', version=25, guard=None, commands=[
    Command(name='vkDestroySurfaceKHR', dispatch='VkInstance'),
    Command(name='vkGetPhysicalDeviceSurfaceSupportKHR', dispatch='VkPhysicalDevice'),
    Command(name='vkGetPhysicalDeviceSurfaceCapabilitiesKHR', dispatch='VkPhysicalDevice'),
    Command(name='vkGetPhysicalDeviceSurfaceFormatsKHR', dispatch='VkPhysicalDevice'),
    Command(name='vkGetPhysicalDeviceSurfacePresentModesKHR', dispatch='VkPhysicalDevice'),
])

VK_KHR_swapchain = Extension(name='VK_KHR_swapchain', version=70, guard=None, commands=[
    Command(name='vkCreateSwapchainKHR', dispatch='VkDevice'),
    Command(name='vkDestroySwapchainKHR', dispatch='VkDevice'),
    Command(name='vkGetSwapchainImagesKHR', dispatch='VkDevice'),
    Command(name='vkAcquireNextImageKHR', dispatch='VkDevice'),
    Command(name='vkQueuePresentKHR', dispatch='VkQueue'),
    Command(name='vkGetDeviceGroupPresentCapabilitiesKHR', dispatch='VkDevice'),
    Command(name='vkGetDeviceGroupSurfacePresentModesKHR', dispatch='VkDevice'),
    Command(name='vkGetPhysicalDevicePresentRectanglesKHR', dispatch='VkPhysicalDevice'),
    Command(name='vkAcquireNextImage2KHR', dispatch='VkDevice'),
])

VK_KHR_display = Extension(name='VK_KHR_display', version=21, guard=None, commands=[
    Command(name='vkGetPhysicalDeviceDisplayPropertiesKHR', dispatch='VkPhysicalDevice'),
    Command(name='vkGetPhysicalDeviceDisplayPlanePropertiesKHR', dispatch='VkPhysicalDevice'),
    Command(name='vkGetDisplayPlaneSupportedDisplaysKHR', dispatch='VkPhysicalDevice'),
    Command(name='vkGetDisplayModePropertiesKHR', dispatch='VkPhysicalDevice'),
    Command(name='vkCreateDisplayModeKHR', dispatch='VkPhysicalDevice'),
    Command(name='vkGetDisplayPlaneCapabilitiesKHR', dispatch='VkPhysicalDevice'),
    Command(name='vkCreateDisplayPlaneSurfaceKHR', dispatch='VkInstance'),
])

VK_KHR_display_swapchain = Extension(name='VK_KHR_display_swapchain', version=9, guard=None, commands=[
    Command(name='vkCreateSharedSwapchainsKHR', dispatch='VkDevice'),
])

VK_KHR_sampler_mirror_clamp_to_edge = Extension(name='VK_KHR_sampler_mirror_clamp_to_edge', version=1, guard=None, commands=[
])

VK_KHR_multiview = Extension(name='VK_KHR_multiview', version=1, guard=None, commands=[
])

VK_KHR_get_physical_device_properties2 = Extension(name='VK_KHR_get_physical_device_properties2', version=1, guard=None, commands=[
    Command(name='vkGetPhysicalDeviceFeatures2KHR', dispatch='VkPhysicalDevice'),
    Command(name='vkGetPhysicalDeviceProperties2KHR', dispatch='VkPhysicalDevice'),
    Command(name='vkGetPhysicalDeviceFormatProperties2KHR', dispatch='VkPhysicalDevice'),
    Command(name='vkGetPhysicalDeviceImageFormatProperties2KHR', dispatch='VkPhysicalDevice'),
    Command(name='vkGetPhysicalDeviceQueueFamilyProperties2KHR', dispatch='VkPhysicalDevice'),
    Command(name='vkGetPhysicalDeviceMemoryProperties2KHR', dispatch='VkPhysicalDevice'),
    Command(name='vkGetPhysicalDeviceSparseImageFormatProperties2KHR', dispatch='VkPhysicalDevice'),
])

VK_KHR_device_group = Extension(name='VK_KHR_device_group', version=3, guard=None, commands=[
    Command(name='vkGetDeviceGroupPeerMemoryFeaturesKHR', dispatch='VkDevice'),
    Command(name='vkCmdSetDeviceMaskKHR', dispatch='VkCommandBuffer'),
    Command(name='vkCmdDispatchBaseKHR', dispatch='VkCommandBuffer'),
])

VK_KHR_shader_draw_parameters = Extension(name='VK_KHR_shader_draw_parameters', version=1, guard=None, commands=[
])

VK_KHR_maintenance1 = Extension(name='VK_KHR_maintenance1', version=2, guard=None, commands=[
    Command(name='vkTrimCommandPoolKHR', dispatch='VkDevice'),
])

VK_KHR_device_group_creation = Extension(name='VK_KHR_device_group_creation', version=1, guard=None, commands=[
    Command(name='vkEnumeratePhysicalDeviceGroupsKHR', dispatch='VkInstance'),
])

VK_KHR_external_memory_capabilities = Extension(name='VK_KHR_external_memory_capabilities', version=1, guard=None, commands=[
    Command(name='vkGetPhysicalDeviceExternalBufferPropertiesKHR', dispatch='VkPhysicalDevice'),
])

VK_KHR_external_memory = Extension(name='VK_KHR_external_memory', version=1, guard=None, commands=[
])

VK_KHR_external_memory_fd = Extension(name='VK_KHR_external_memory_fd', version=1, guard=None, commands=[
    Command(name='vkGetMemoryFdKHR', dispatch='VkDevice'),
    Command(name='vkGetMemoryFdPropertiesKHR', dispatch='VkDevice'),
])

VK_KHR_external_semaphore_capabilities = Extension(name='VK_KHR_external_semaphore_capabilities', version=1, guard=None, commands=[
    Command(name='vkGetPhysicalDeviceExternalSemaphorePropertiesKHR', dispatch='VkPhysicalDevice'),
])

VK_KHR_external_semaphore = Extension(name='VK_KHR_external_semaphore', version=1, guard=None, commands=[
])

VK_KHR_external_semaphore_fd = Extension(name='VK_KHR_external_semaphore_fd', version=1, guard=None, commands=[
    Command(name='vkImportSemaphoreFdKHR', dispatch='VkDevice'),
    Command(name='vkGetSemaphoreFdKHR', dispatch='VkDevice'),
])

VK_KHR_push_descriptor = Extension(name='VK_KHR_push_descriptor', version=2, guard=None, commands=[
    Command(name='vkCmdPushDescriptorSetKHR', dispatch='VkCommandBuffer'),
    Command(name='vkCmdPushDescriptorSetWithTemplateKHR', dispatch='VkCommandBuffer'),
])

VK_KHR_16bit_storage = Extension(name='VK_KHR_16bit_storage', version=1, guard=None, commands=[
])

VK_KHR_incremental_present = Extension(name='VK_KHR_incremental_present', version=1, guard=None, commands=[
])

VK_KHR_descriptor_update_template = Extension(name='VK_KHR_descriptor_update_template', version=1, guard=None, commands=[
    Command(name='vkCreateDescriptorUpdateTemplateKHR', dispatch='VkDevice'),
    Command(name='vkDestroyDescriptorUpdateTemplateKHR', dispatch='VkDevice'),
    Command(name='vkUpdateDescriptorSetWithTemplateKHR', dispatch='VkDevice'),
])

VK_KHR_shared_presentable_image = Extension(name='VK_KHR_shared_presentable_image', version=1, guard=None, commands=[
    Command(name='vkGetSwapchainStatusKHR', dispatch='VkDevice'),
])

VK_KHR_external_fence_capabilities = Extension(name='VK_KHR_external_fence_capabilities', version=1, guard=None, commands=[
    Command(name='vkGetPhysicalDeviceExternalFencePropertiesKHR', dispatch='VkPhysicalDevice'),
])

VK_KHR_external_fence = Extension(name='VK_KHR_external_fence', version=1, guard=None, commands=[
])

VK_KHR_external_fence_fd = Extension(name='VK_KHR_external_fence_fd', version=1, guard=None, commands=[
    Command(name='vkImportFenceFdKHR', dispatch='VkDevice'),
    Command(name='vkGetFenceFdKHR', dispatch='VkDevice'),
])

VK_KHR_maintenance2 = Extension(name='VK_KHR_maintenance2', version=1, guard=None, commands=[
])

VK_KHR_get_surface_capabilities2 = Extension(name='VK_KHR_get_surface_capabilities2', version=1, guard=None, commands=[
    Command(name='vkGetPhysicalDeviceSurfaceCapabilities2KHR', dispatch='VkPhysicalDevice'),
    Command(name='vkGetPhysicalDeviceSurfaceFormats2KHR', dispatch='VkPhysicalDevice'),
])

VK_KHR_variable_pointers = Extension(name='VK_KHR_variable_pointers', version=1, guard=None, commands=[
])

VK_KHR_dedicated_allocation = Extension(name='VK_KHR_dedicated_allocation', version=3, guard=None, commands=[
])

VK_KHR_storage_buffer_storage_class = Extension(name='VK_KHR_storage_buffer_storage_class', version=1, guard=None, commands=[
])

VK_KHR_relaxed_block_layout = Extension(name='VK_KHR_relaxed_block_layout', version=1, guard=None, commands=[
])

VK_KHR_get_memory_requirements2 = Extension(name='VK_KHR_get_memory_requirements2', version=1, guard=None, commands=[
    Command(name='vkGetImageMemoryRequirements2KHR', dispatch='VkDevice'),
    Command(name='vkGetBufferMemoryRequirements2KHR', dispatch='VkDevice'),
    Command(name='vkGetImageSparseMemoryRequirements2KHR', dispatch='VkDevice'),
])

VK_KHR_image_format_list = Extension(name='VK_KHR_image_format_list', version=1, guard=None, commands=[
])

VK_KHR_sampler_ycbcr_conversion = Extension(name='VK_KHR_sampler_ycbcr_conversion', version=1, guard=None, commands=[
    Command(name='vkCreateSamplerYcbcrConversionKHR', dispatch='VkDevice'),
    Command(name='vkDestroySamplerYcbcrConversionKHR', dispatch='VkDevice'),
])

VK_KHR_bind_memory2 = Extension(name='VK_KHR_bind_memory2', version=1, guard=None, commands=[
    Command(name='vkBindBufferMemory2KHR', dispatch='VkDevice'),
    Command(name='vkBindImageMemory2KHR', dispatch='VkDevice'),
])

VK_KHR_maintenance3 = Extension(name='VK_KHR_maintenance3', version=1, guard=None, commands=[
    Command(name='vkGetDescriptorSetLayoutSupportKHR', dispatch='VkDevice'),
])

VK_EXT_debug_report = Extension(name='VK_EXT_debug_report', version=9, guard=None, commands=[
    Command(name='vkCreateDebugReportCallbackEXT', dispatch='VkInstance'),
    Command(name='vkDestroyDebugReportCallbackEXT', dispatch='VkInstance'),
    Command(name='vkDebugReportMessageEXT', dispatch='VkInstance'),
])

VK_NV_glsl_shader = Extension(name='VK_NV_glsl_shader', version=1, guard=None, commands=[
])

VK_EXT_depth_range_unrestricted = Extension(name='VK_EXT_depth_range_unrestricted', version=1, guard=None, commands=[
])

VK_IMG_filter_cubic = Extension(name='VK_IMG_filter_cubic', version=1, guard=None, commands=[
])

VK_AMD_rasterization_order = Extension(name='VK_AMD_rasterization_order', version=1, guard=None, commands=[
])

VK_AMD_shader_trinary_minmax = Extension(name='VK_AMD_shader_trinary_minmax', version=1, guard=None, commands=[
])

VK_AMD_shader_explicit_vertex_parameter = Extension(name='VK_AMD_shader_explicit_vertex_parameter', version=1, guard=None, commands=[
])

VK_EXT_debug_marker = Extension(name='VK_EXT_debug_marker', version=4, guard=None, commands=[
    Command(name='vkDebugMarkerSetObjectTagEXT', dispatch='VkDevice'),
    Command(name='vkDebugMarkerSetObjectNameEXT', dispatch='VkDevice'),
    Command(name='vkCmdDebugMarkerBeginEXT', dispatch='VkCommandBuffer'),
    Command(name='vkCmdDebugMarkerEndEXT', dispatch='VkCommandBuffer'),
    Command(name='vkCmdDebugMarkerInsertEXT', dispatch='VkCommandBuffer'),
])

VK_AMD_gcn_shader = Extension(name='VK_AMD_gcn_shader', version=1, guard=None, commands=[
])

VK_NV_dedicated_allocation = Extension(name='VK_NV_dedicated_allocation', version=1, guard=None, commands=[
])

VK_AMD_draw_indirect_count = Extension(name='VK_AMD_draw_indirect_count', version=1, guard=None, commands=[
    Command(name='vkCmdDrawIndirectCountAMD', dispatch='VkCommandBuffer'),
    Command(name='vkCmdDrawIndexedIndirectCountAMD', dispatch='VkCommandBuffer'),
])

VK_AMD_negative_viewport_height = Extension(name='VK_AMD_negative_viewport_height', version=1, guard=None, commands=[
])

VK_AMD_gpu_shader_half_float = Extension(name='VK_AMD_gpu_shader_half_float', version=1, guard=None, commands=[
])

VK_AMD_shader_ballot = Extension(name='VK_AMD_shader_ballot', version=1, guard=None, commands=[
])

VK_AMD_texture_gather_bias_lod = Extension(name='VK_AMD_texture_gather_bias_lod', version=1, guard=None, commands=[
])

VK_AMD_shader_info = Extension(name='VK_AMD_shader_info', version=1, guard=None, commands=[
    Command(name='vkGetShaderInfoAMD', dispatch='VkDevice'),
])

VK_AMD_shader_image_load_store_lod = Extension(name='VK_AMD_shader_image_load_store_lod', version=1, guard=None, commands=[
])

VK_IMG_format_pvrtc = Extension(name='VK_IMG_format_pvrtc', version=1, guard=None, commands=[
])

VK_NV_external_memory_capabilities = Extension(name='VK_NV_external_memory_capabilities', version=1, guard=None, commands=[
    Command(name='vkGetPhysicalDeviceExternalImageFormatPropertiesNV', dispatch='VkPhysicalDevice'),
])

VK_NV_external_memory = Extension(name='VK_NV_external_memory', version=1, guard=None, commands=[
])

VK_EXT_validation_flags = Extension(name='VK_EXT_validation_flags', version=1, guard=None, commands=[
])

VK_EXT_shader_subgroup_ballot = Extension(name='VK_EXT_shader_subgroup_ballot', version=1, guard=None, commands=[
])

VK_EXT_shader_subgroup_vote = Extension(name='VK_EXT_shader_subgroup_vote', version=1, guard=None, commands=[
])

VK_NVX_device_generated_commands = Extension(name='VK_NVX_device_generated_commands', version=3, guard=None, commands=[
    Command(name='vkCmdProcessCommandsNVX', dispatch='VkCommandBuffer'),
    Command(name='vkCmdReserveSpaceForCommandsNVX', dispatch='VkCommandBuffer'),
    Command(name='vkCreateIndirectCommandsLayoutNVX', dispatch='VkDevice'),
    Command(name='vkDestroyIndirectCommandsLayoutNVX', dispatch='VkDevice'),
    Command(name='vkCreateObjectTableNVX', dispatch='VkDevice'),
    Command(name='vkDestroyObjectTableNVX', dispatch='VkDevice'),
    Command(name='vkRegisterObjectsNVX', dispatch='VkDevice'),
    Command(name='vkUnregisterObjectsNVX', dispatch='VkDevice'),
    Command(name='vkGetPhysicalDeviceGeneratedCommandsPropertiesNVX', dispatch='VkPhysicalDevice'),
])

VK_NV_clip_space_w_scaling = Extension(name='VK_NV_clip_space_w_scaling', version=1, guard=None, commands=[
    Command(name='vkCmdSetViewportWScalingNV', dispatch='VkCommandBuffer'),
])

VK_EXT_direct_mode_display = Extension(name='VK_EXT_direct_mode_display', version=1, guard=None, commands=[
    Command(name='vkReleaseDisplayEXT', dispatch='VkPhysicalDevice'),
])

VK_EXT_display_surface_counter = Extension(name='VK_EXT_display_surface_counter', version=1, guard=None, commands=[
    Command(name='vkGetPhysicalDeviceSurfaceCapabilities2EXT', dispatch='VkPhysicalDevice'),
])

VK_EXT_display_control = Extension(name='VK_EXT_display_control', version=1, guard=None, commands=[
    Command(name='vkDisplayPowerControlEXT', dispatch='VkDevice'),
    Command(name='vkRegisterDeviceEventEXT', dispatch='VkDevice'),
    Command(name='vkRegisterDisplayEventEXT', dispatch='VkDevice'),
    Command(name='vkGetSwapchainCounterEXT', dispatch='VkDevice'),
])

VK_GOOGLE_display_timing = Extension(name='VK_GOOGLE_display_timing', version=1, guard=None, commands=[
    Command(name='vkGetRefreshCycleDurationGOOGLE', dispatch='VkDevice'),
    Command(name='vkGetPastPresentationTimingGOOGLE', dispatch='VkDevice'),
])

VK_NV_sample_mask_override_coverage = Extension(name='VK_NV_sample_mask_override_coverage', version=1, guard=None, commands=[
])

VK_NV_geometry_shader_passthrough = Extension(name='VK_NV_geometry_shader_passthrough', version=1, guard=None, commands=[
])

VK_NV_viewport_array2 = Extension(name='VK_NV_viewport_array2', version=1, guard=None, commands=[
])

VK_NVX_multiview_per_view_attributes = Extension(name='VK_NVX_multiview_per_view_attributes', version=1, guard=None, commands=[
])

VK_NV_viewport_swizzle = Extension(name='VK_NV_viewport_swizzle', version=1, guard=None, commands=[
])

VK_EXT_discard_rectangles = Extension(name='VK_EXT_discard_rectangles', version=1, guard=None, commands=[
    Command(name='vkCmdSetDiscardRectangleEXT', dispatch='VkCommandBuffer'),
])

VK_EXT_conservative_rasterization = Extension(name='VK_EXT_conservative_rasterization', version=1, guard=None, commands=[
])

VK_EXT_swapchain_colorspace = Extension(name='VK_EXT_swapchain_colorspace', version=3, guard=None, commands=[
])

VK_EXT_hdr_metadata = Extension(name='VK_EXT_hdr_metadata', version=1, guard=None, commands=[
    Command(name='vkSetHdrMetadataEXT', dispatch='VkDevice'),
])

VK_EXT_external_memory_dma_buf = Extension(name='VK_EXT_external_memory_dma_buf', version=1, guard=None, commands=[
])

VK_EXT_queue_family_foreign = Extension(name='VK_EXT_queue_family_foreign', version=1, guard=None, commands=[
])

VK_EXT_debug_utils = Extension(name='VK_EXT_debug_utils', version=1, guard=None, commands=[
    Command(name='vkSetDebugUtilsObjectNameEXT', dispatch='VkDevice'),
    Command(name='vkSetDebugUtilsObjectTagEXT', dispatch='VkDevice'),
    Command(name='vkQueueBeginDebugUtilsLabelEXT', dispatch='VkQueue'),
    Command(name='vkQueueEndDebugUtilsLabelEXT', dispatch='VkQueue'),
    Command(name='vkQueueInsertDebugUtilsLabelEXT', dispatch='VkQueue'),
    Command(name='vkCmdBeginDebugUtilsLabelEXT', dispatch='VkCommandBuffer'),
    Command(name='vkCmdEndDebugUtilsLabelEXT', dispatch='VkCommandBuffer'),
    Command(name='vkCmdInsertDebugUtilsLabelEXT', dispatch='VkCommandBuffer'),
    Command(name='vkCreateDebugUtilsMessengerEXT', dispatch='VkInstance'),
    Command(name='vkDestroyDebugUtilsMessengerEXT', dispatch='VkInstance'),
    Command(name='vkSubmitDebugUtilsMessageEXT', dispatch='VkInstance'),
])

VK_EXT_sampler_filter_minmax = Extension(name='VK_EXT_sampler_filter_minmax', version=1, guard=None, commands=[
])

VK_AMD_gpu_shader_int16 = Extension(name='VK_AMD_gpu_shader_int16', version=1, guard=None, commands=[
])

VK_AMD_mixed_attachment_samples = Extension(name='VK_AMD_mixed_attachment_samples', version=1, guard=None, commands=[
])

VK_AMD_shader_fragment_mask = Extension(name='VK_AMD_shader_fragment_mask', version=1, guard=None, commands=[
])

VK_EXT_shader_stencil_export = Extension(name='VK_EXT_shader_stencil_export', version=1, guard=None, commands=[
])

VK_EXT_sample_locations = Extension(name='VK_EXT_sample_locations', version=1, guard=None, commands=[
    Command(name='vkCmdSetSampleLocationsEXT', dispatch='VkCommandBuffer'),
    Command(name='vkGetPhysicalDeviceMultisamplePropertiesEXT', dispatch='VkPhysicalDevice'),
])

VK_EXT_blend_operation_advanced = Extension(name='VK_EXT_blend_operation_advanced', version=2, guard=None, commands=[
])

VK_NV_fragment_coverage_to_color = Extension(name='VK_NV_fragment_coverage_to_color', version=1, guard=None, commands=[
])

VK_NV_framebuffer_mixed_samples = Extension(name='VK_NV_framebuffer_mixed_samples', version=1, guard=None, commands=[
])

VK_NV_fill_rectangle = Extension(name='VK_NV_fill_rectangle', version=1, guard=None, commands=[
])

VK_EXT_post_depth_coverage = Extension(name='VK_EXT_post_depth_coverage', version=1, guard=None, commands=[
])

VK_EXT_validation_cache = Extension(name='VK_EXT_validation_cache', version=1, guard=None, commands=[
    Command(name='vkCreateValidationCacheEXT', dispatch='VkDevice'),
    Command(name='vkDestroyValidationCacheEXT', dispatch='VkDevice'),
    Command(name='vkMergeValidationCachesEXT', dispatch='VkDevice'),
    Command(name='vkGetValidationCacheDataEXT', dispatch='VkDevice'),
])

VK_EXT_descriptor_indexing = Extension(name='VK_EXT_descriptor_indexing', version=2, guard=None, commands=[
])

VK_EXT_shader_viewport_index_layer = Extension(name='VK_EXT_shader_viewport_index_layer', version=1, guard=None, commands=[
])

VK_EXT_global_priority = Extension(name='VK_EXT_global_priority', version=2, guard=None, commands=[
])

VK_EXT_external_memory_host = Extension(name='VK_EXT_external_memory_host', version=1, guard=None, commands=[
    Command(name='vkGetMemoryHostPointerPropertiesEXT', dispatch='VkDevice'),
])

VK_AMD_buffer_marker = Extension(name='VK_AMD_buffer_marker', version=1, guard=None, commands=[
    Command(name='vkCmdWriteBufferMarkerAMD', dispatch='VkCommandBuffer'),
])

VK_AMD_shader_core_properties = Extension(name='VK_AMD_shader_core_properties', version=1, guard=None, commands=[
])

VK_EXT_vertex_attribute_divisor = Extension(name='VK_EXT_vertex_attribute_divisor', version=1, guard=None, commands=[
])

VK_NV_shader_subgroup_partitioned = Extension(name='VK_NV_shader_subgroup_partitioned', version=1, guard=None, commands=[
])

VK_KHR_android_surface = Extension(name='VK_KHR_android_surface', version=6, guard='VK_USE_PLATFORM_ANDROID_KHR', commands=[
    Command(name='vkCreateAndroidSurfaceKHR', dispatch='VkInstance'),
])

VK_ANDROID_external_memory_android_hardware_buffer = Extension(name='VK_ANDROID_external_memory_android_hardware_buffer', version=3, guard='VK_USE_PLATFORM_ANDROID_KHR', commands=[
    Command(name='vkGetAndroidHardwareBufferPropertiesANDROID', dispatch='VkDevice'),
    Command(name='vkGetMemoryAndroidHardwareBufferANDROID', dispatch='VkDevice'),
])

VK_MVK_ios_surface = Extension(name='VK_MVK_ios_surface', version=2, guard='VK_USE_PLATFORM_IOS_MVK', commands=[
    Command(name='vkCreateIOSSurfaceMVK', dispatch='VkInstance'),
])

VK_MVK_macos_surface = Extension(name='VK_MVK_macos_surface', version=2, guard='VK_USE_PLATFORM_MACOS_MVK', commands=[
    Command(name='vkCreateMacOSSurfaceMVK', dispatch='VkInstance'),
])

VK_KHR_mir_surface = Extension(name='VK_KHR_mir_surface', version=4, guard='VK_USE_PLATFORM_MIR_KHR', commands=[
    Command(name='vkCreateMirSurfaceKHR', dispatch='VkInstance'),
    Command(name='vkGetPhysicalDeviceMirPresentationSupportKHR', dispatch='VkPhysicalDevice'),
])

VK_NN_vi_surface = Extension(name='VK_NN_vi_surface', version=1, guard='VK_USE_PLATFORM_VI_NN', commands=[
    Command(name='vkCreateViSurfaceNN', dispatch='VkInstance'),
])

VK_KHR_wayland_surface = Extension(name='VK_KHR_wayland_surface', version=6, guard='VK_USE_PLATFORM_WAYLAND_KHR', commands=[
    Command(name='vkCreateWaylandSurfaceKHR', dispatch='VkInstance'),
    Command(name='vkGetPhysicalDeviceWaylandPresentationSupportKHR', dispatch='VkPhysicalDevice'),
])

VK_KHR_win32_surface = Extension(name='VK_KHR_win32_surface', version=6, guard='VK_USE_PLATFORM_WIN32_KHR', commands=[
    Command(name='vkCreateWin32SurfaceKHR', dispatch='VkInstance'),
    Command(name='vkGetPhysicalDeviceWin32PresentationSupportKHR', dispatch='VkPhysicalDevice'),
])

VK_KHR_external_memory_win32 = Extension(name='VK_KHR_external_memory_win32', version=1, guard='VK_USE_PLATFORM_WIN32_KHR', commands=[
    Command(name='vkGetMemoryWin32HandleKHR', dispatch='VkDevice'),
    Command(name='vkGetMemoryWin32HandlePropertiesKHR', dispatch='VkDevice'),
])

VK_KHR_win32_keyed_mutex = Extension(name='VK_KHR_win32_keyed_mutex', version=1, guard='VK_USE_PLATFORM_WIN32_KHR', commands=[
])

VK_KHR_external_semaphore_win32 = Extension(name='VK_KHR_external_semaphore_win32', version=1, guard='VK_USE_PLATFORM_WIN32_KHR', commands=[
    Command(name='vkImportSemaphoreWin32HandleKHR', dispatch='VkDevice'),
    Command(name='vkGetSemaphoreWin32HandleKHR', dispatch='VkDevice'),
])

VK_KHR_external_fence_win32 = Extension(name='VK_KHR_external_fence_win32', version=1, guard='VK_USE_PLATFORM_WIN32_KHR', commands=[
    Command(name='vkImportFenceWin32HandleKHR', dispatch='VkDevice'),
    Command(name='vkGetFenceWin32HandleKHR', dispatch='VkDevice'),
])

VK_NV_external_memory_win32 = Extension(name='VK_NV_external_memory_win32', version=1, guard='VK_USE_PLATFORM_WIN32_KHR', commands=[
    Command(name='vkGetMemoryWin32HandleNV', dispatch='VkDevice'),
])

VK_NV_win32_keyed_mutex = Extension(name='VK_NV_win32_keyed_mutex', version=1, guard='VK_USE_PLATFORM_WIN32_KHR', commands=[
])

VK_KHR_xcb_surface = Extension(name='VK_KHR_xcb_surface', version=6, guard='VK_USE_PLATFORM_XCB_KHR', commands=[
    Command(name='vkCreateXcbSurfaceKHR', dispatch='VkInstance'),
    Command(name='vkGetPhysicalDeviceXcbPresentationSupportKHR', dispatch='VkPhysicalDevice'),
])

VK_KHR_xlib_surface = Extension(name='VK_KHR_xlib_surface', version=6, guard='VK_USE_PLATFORM_XLIB_KHR', commands=[
    Command(name='vkCreateXlibSurfaceKHR', dispatch='VkInstance'),
    Command(name='vkGetPhysicalDeviceXlibPresentationSupportKHR', dispatch='VkPhysicalDevice'),
])

VK_EXT_acquire_xlib_display = Extension(name='VK_EXT_acquire_xlib_display', version=1, guard='VK_USE_PLATFORM_XLIB_XRANDR_EXT', commands=[
    Command(name='vkAcquireXlibDisplayEXT', dispatch='VkPhysicalDevice'),
    Command(name='vkGetRandROutputDisplayEXT', dispatch='VkPhysicalDevice'),
])

extensions = [
    VK_core_0,
    VK_core_1,
    VK_KHR_surface,
    VK_KHR_swapchain,
    VK_KHR_display,
    VK_KHR_display_swapchain,
    VK_KHR_sampler_mirror_clamp_to_edge,
    VK_KHR_multiview,
    VK_KHR_get_physical_device_properties2,
    VK_KHR_device_group,
    VK_KHR_shader_draw_parameters,
    VK_KHR_maintenance1,
    VK_KHR_device_group_creation,
    VK_KHR_external_memory_capabilities,
    VK_KHR_external_memory,
    VK_KHR_external_memory_fd,
    VK_KHR_external_semaphore_capabilities,
    VK_KHR_external_semaphore,
    VK_KHR_external_semaphore_fd,
    VK_KHR_push_descriptor,
    VK_KHR_16bit_storage,
    VK_KHR_incremental_present,
    VK_KHR_descriptor_update_template,
    VK_KHR_shared_presentable_image,
    VK_KHR_external_fence_capabilities,
    VK_KHR_external_fence,
    VK_KHR_external_fence_fd,
    VK_KHR_maintenance2,
    VK_KHR_get_surface_capabilities2,
    VK_KHR_variable_pointers,
    VK_KHR_dedicated_allocation,
    VK_KHR_storage_buffer_storage_class,
    VK_KHR_relaxed_block_layout,
    VK_KHR_get_memory_requirements2,
    VK_KHR_image_format_list,
    VK_KHR_sampler_ycbcr_conversion,
    VK_KHR_bind_memory2,
    VK_KHR_maintenance3,
    VK_EXT_debug_report,
    VK_NV_glsl_shader,
    VK_EXT_depth_range_unrestricted,
    VK_IMG_filter_cubic,
    VK_AMD_rasterization_order,
    VK_AMD_shader_trinary_minmax,
    VK_AMD_shader_explicit_vertex_parameter,
    VK_EXT_debug_marker,
    VK_AMD_gcn_shader,
    VK_NV_dedicated_allocation,
    VK_AMD_draw_indirect_count,
    VK_AMD_negative_viewport_height,
    VK_AMD_gpu_shader_half_float,
    VK_AMD_shader_ballot,
    VK_AMD_texture_gather_bias_lod,
    VK_AMD_shader_info,
    VK_AMD_shader_image_load_store_lod,
    VK_IMG_format_pvrtc,
    VK_NV_external_memory_capabilities,
    VK_NV_external_memory,
    VK_EXT_validation_flags,
    VK_EXT_shader_subgroup_ballot,
    VK_EXT_shader_subgroup_vote,
    VK_NVX_device_generated_commands,
    VK_NV_clip_space_w_scaling,
    VK_EXT_direct_mode_display,
    VK_EXT_display_surface_counter,
    VK_EXT_display_control,
    VK_GOOGLE_display_timing,
    VK_NV_sample_mask_override_coverage,
    VK_NV_geometry_shader_passthrough,
    VK_NV_viewport_array2,
    VK_NVX_multiview_per_view_attributes,
    VK_NV_viewport_swizzle,
    VK_EXT_discard_rectangles,
    VK_EXT_conservative_rasterization,
    VK_EXT_swapchain_colorspace,
    VK_EXT_hdr_metadata,
    VK_EXT_external_memory_dma_buf,
    VK_EXT_queue_family_foreign,
    VK_EXT_debug_utils,
    VK_EXT_sampler_filter_minmax,
    VK_AMD_gpu_shader_int16,
    VK_AMD_mixed_attachment_samples,
    VK_AMD_shader_fragment_mask,
    VK_EXT_shader_stencil_export,
    VK_EXT_sample_locations,
    VK_EXT_blend_operation_advanced,
    VK_NV_fragment_coverage_to_color,
    VK_NV_framebuffer_mixed_samples,
    VK_NV_fill_rectangle,
    VK_EXT_post_depth_coverage,
    VK_EXT_validation_cache,
    VK_EXT_descriptor_indexing,
    VK_EXT_shader_viewport_index_layer,
    VK_EXT_global_priority,
    VK_EXT_external_memory_host,
    VK_AMD_buffer_marker,
    VK_AMD_shader_core_properties,
    VK_EXT_vertex_attribute_divisor,
    VK_NV_shader_subgroup_partitioned,
    VK_KHR_android_surface,
    VK_ANDROID_external_memory_android_hardware_buffer,
    VK_MVK_ios_surface,
    VK_MVK_macos_surface,
    VK_KHR_mir_surface,
    VK_NN_vi_surface,
    VK_KHR_wayland_surface,
    VK_KHR_win32_surface,
    VK_KHR_external_memory_win32,
    VK_KHR_win32_keyed_mutex,
    VK_KHR_external_semaphore_win32,
    VK_KHR_external_fence_win32,
    VK_NV_external_memory_win32,
    VK_NV_win32_keyed_mutex,
    VK_KHR_xcb_surface,
    VK_KHR_xlib_surface,
    VK_EXT_acquire_xlib_display,
]
# end of generated code

def generate_wrapper_header(guard):
    copyright = []
    copyright.append("/*                                                                         ")
    copyright.append(" * Copyright 2018 The Android Open Source Project                          ")
    copyright.append(" *                                                                         ")
    copyright.append(" * Licensed under the Apache License, Version 2.0 (the \"License\");       ")
    copyright.append(" * you may not use this file except in compliance with the License.        ")
    copyright.append(" * You may obtain a copy of the License at                                 ")
    copyright.append(" *                                                                         ")
    copyright.append(" *      http://www.apache.org/licenses/LICENSE-2.0                         ")
    copyright.append(" *                                                                         ")
    copyright.append(" * Unless required by applicable law or agreed to in writing, software     ")
    copyright.append(" * distributed under the License is distributed on an \"AS IS\" BASIS,     ")
    copyright.append(" * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.")
    copyright.append(" * See the License for the specific language governing permissions and     ")
    copyright.append(" * limitations under the License.                                          ")
    copyright.append(" */                                                                        ")
    lines = [line.rstrip() for line in copyright]

    lines.append("// This file is generated.")
    lines.append("#ifndef %s" % guard)
    lines.append("#define %s" % guard)
    lines.append("")
    lines.append("#ifdef __cplusplus")
    lines.append("extern \"C\" {")
    lines.append("#endif")
    lines.append("")
    lines.append("#define VK_NO_PROTOTYPES 1")
    lines.append("#include <vulkan/vulkan.h>")
    lines.append("")
    lines.append("/* Initialize the Vulkan function pointer variables declared in this header.")
    lines.append(" * Returns 0 if vulkan is not available, non-zero if it is available.")
    lines.append(" */")
    lines.append("int InitVulkan(void);")
    lines.append("")

    for ext in extensions:
        # Only wrap core and WSI functions
        wrapped_exts = {'VK_core', 'VK_KHR'}
        if not any(ext.name.startswith(s) for s in wrapped_exts):
            continue

        if ext.guard:
            lines.append("#ifdef %s" % ext.guard)

        lines.append("// %s" % ext.name)
        for cmd in ext.commands:
            lines.append("extern PFN_%s %s;" % (cmd.name, cmd.name))

        if ext.guard:
            lines.append("#endif")
        lines.append("")

    lines.append("")
    lines.append("")
    lines.append("#ifdef __cplusplus")
    lines.append("}")
    lines.append("#endif")
    lines.append("")
    lines.append("#endif  // %s" % guard)

    return "\n".join(lines)

def generate_wrapper_source(header):
    copyright = []
    copyright.append("/*                                                                         ")
    copyright.append(" * Copyright 2018 The Android Open Source Project                          ")
    copyright.append(" *                                                                         ")
    copyright.append(" * Licensed under the Apache License, Version 2.0 (the \"License\");       ")
    copyright.append(" * you may not use this file except in compliance with the License.        ")
    copyright.append(" * You may obtain a copy of the License at                                 ")
    copyright.append(" *                                                                         ")
    copyright.append(" *      http://www.apache.org/licenses/LICENSE-2.0                         ")
    copyright.append(" *                                                                         ")
    copyright.append(" * Unless required by applicable law or agreed to in writing, software     ")
    copyright.append(" * distributed under the License is distributed on an \"AS IS\" BASIS,     ")
    copyright.append(" * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.")
    copyright.append(" * See the License for the specific language governing permissions and     ")
    copyright.append(" * limitations under the License.                                          ")
    copyright.append(" */                                                                        ")
    lines = [line.rstrip() for line in copyright]

    lines.append("// This file is generated.")
    lines.append("#ifdef __cplusplus")
    lines.append("extern \"C\" {")
    lines.append("#endif")
    lines.append("")
    lines.append("#include \"%s\"" % header)
    lines.append("#include <dlfcn.h>")
    lines.append("")

    lines.append("int InitVulkan(void) {")
    lines.append("    void* libvulkan = dlopen(\"libvulkan.so\", RTLD_NOW | RTLD_LOCAL);")
    lines.append("    if (!libvulkan)")
    lines.append("        return 0;")
    lines.append("")
    lines.append("    // Vulkan supported, set function addresses")
    for ext in extensions:
        # Only wrap core and WSI functions
        wrapped_exts = {'VK_core', 'VK_KHR'}
        if not any(ext.name.startswith(s) for s in wrapped_exts):
            continue

        if ext.guard:
            lines.append("")
            lines.append("#ifdef %s" % ext.guard)

        for cmd in ext.commands:
            lines.append("    %s = reinterpret_cast<PFN_%s>(dlsym(libvulkan, \"%s\"));" % (cmd.name, cmd.name, cmd.name))

        if ext.guard:
            lines.append("#endif")

    lines.append("    return 1;")
    lines.append("}")
    lines.append("")

    lines.append("// No Vulkan support, do not set function addresses")
    for ext in extensions:
        if ext.guard:
            lines.append("")
            lines.append("#ifdef %s" % ext.guard)

        for cmd in ext.commands:
            lines.append("PFN_%s %s;" % (cmd.name, cmd.name))

        if ext.guard:
            lines.append("#endif")

    lines.append("")
    lines.append("#ifdef __cplusplus")
    lines.append("}")
    lines.append("#endif")

    return "\n".join(lines)

def parse_subheader(filename, ext_guard):
    sub_extensions = []

    with open(filename, "r") as f:
        current_ext = None
        spec_version = None

        for line in f:
            line = line.strip();

            if line.startswith("#define VK_API_VERSION"):
                minor_end = line.rfind(",")
                minor_begin = line.rfind(",", 0, minor_end) + 1
                spec_version = int(line[minor_begin:minor_end])
                # add core
                current_ext = Extension("VK_core_%s" % spec_version, spec_version)
                sub_extensions.append(current_ext)
            elif Command.valid_c_typedef(line):
                current_ext.add_command(Command.from_c_typedef(line))
            elif line.startswith("#define") and "SPEC_VERSION " in line:
                version_begin = line.rfind(" ") + 1
                spec_version = int(line[version_begin:])
            elif line.startswith("#define") and "EXTENSION_NAME " in line:
                name_end = line.rfind("\"")
                name_begin = line.rfind("\"", 0, name_end) + 1
                name = line[name_begin:name_end]
                # add extension
                current_ext = Extension(name, spec_version, ext_guard)
                sub_extensions.append(current_ext)

    return sub_extensions;

def parse_vulkan_h(filename):
    extensions = []

    with open(filename, "r") as f:
        ext_guard = None

        for line in f:
            line = line.strip();

            if line.startswith("#include \"vulkan_"):
                # Extract the filename and parse it.  Must be local to script file (no path).
                extensions.extend(parse_subheader(line[10:].replace('"', ''), ext_guard))
            elif line.startswith("#ifdef VK_USE_PLATFORM"):
                guard_begin = line.find(" ") + 1
                ext_guard = line[guard_begin:]
            elif ext_guard and line.startswith("#endif") and ext_guard in line:
                ext_guard = None

    for ext in extensions:
        print("%s = %s" % (ext.name, repr(ext)))
        print("")

    print("extensions = [")
    for ext in extensions:
        print("    %s," % ext.name)
    print("]")

if __name__ == "__main__":
    if sys.argv[1] == "parse":
        parse_vulkan_h(sys.argv[2])
    else:
        filename = sys.argv[1]
        base = os.path.basename(filename)
        contents = []

        if base.endswith(".h"):
            contents = generate_wrapper_header(base.replace(".", "_").upper())
        elif base.endswith(".cpp"):
            contents = generate_wrapper_source(base.replace(".cpp", ".h"))

        with open(filename, "w") as f:
            print(contents, file=f)
