/*
 * Copyright (c) 2008-2014 MongoDB, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.mongodb;

import org.junit.Before;
import org.junit.Test;

import java.io.IOException;
import java.net.UnknownHostException;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;

import static com.mongodb.ClusterConnectionMode.Multiple;
import static com.mongodb.ClusterType.ReplicaSet;
import static com.mongodb.ClusterType.Unknown;
import static com.mongodb.ServerConnectionState.Connected;
import static com.mongodb.ServerConnectionState.Connecting;
import static com.mongodb.ServerDescription.MAX_DRIVER_WIRE_VERSION;
import static com.mongodb.ServerDescription.builder;
import static com.mongodb.ServerType.ReplicaSetOther;
import static com.mongodb.ServerType.ReplicaSetPrimary;
import static com.mongodb.ServerType.ReplicaSetSecondary;
import static java.util.Arrays.asList;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

public class ClusterDescriptionTest {

    private ServerDescription primary, secondary, otherSecondary, uninitiatedMember, notOkMember;
    private ClusterDescription cluster;

    @Before
    public void setUp() throws IOException {
        TagSet tags1 = new TagSet(asList(new Tag("foo", "1"),
                                                         new Tag("bar", "2"),
                                                         new Tag("baz", "1")));
        TagSet tags2 = new TagSet(asList(new Tag("foo", "1"),
                                                         new Tag("bar", "2"),
                                                         new Tag("baz", "2")));
        TagSet tags3 = new TagSet(asList(new Tag("foo", "1"),
                                                         new Tag("bar", "3"),
                                                         new Tag("baz", "3")));

        primary = builder()
                  .state(Connected).address(new ServerAddress("localhost", 27017)).ok(true)
                  .type(ReplicaSetPrimary).tagSet(tags1)
                  .build();

        secondary = builder()
                    .state(Connected).address(new ServerAddress("localhost", 27018)).ok(true)
                    .type(ReplicaSetSecondary).tagSet(tags2)
                    .build();

        otherSecondary = builder()
                         .state(Connected).address(new ServerAddress("localhost", 27019)).ok(true)
                         .type(ReplicaSetSecondary).tagSet(tags3)
                         .build();
        uninitiatedMember = builder()
                            .state(Connected).address(new ServerAddress("localhost", 27020)).ok(true)
                            .type(ReplicaSetOther)
                            .build();

        notOkMember = builder().state(Connected).address(new ServerAddress("localhost", 27021)).ok(false)
                               .build();

        List<ServerDescription> nodeList = asList(primary, secondary, otherSecondary, uninitiatedMember, notOkMember);

        cluster = new ClusterDescription(Multiple, ReplicaSet, nodeList);
    }

    @Test
    public void testMode() {
        ClusterDescription description = new ClusterDescription(Multiple, Unknown, Collections.<ServerDescription>emptyList());
        assertEquals(Multiple, description.getConnectionMode());
    }

    @Test
    public void testAll() {
        ClusterDescription description = new ClusterDescription(Multiple, Unknown, Collections.<ServerDescription>emptyList());
        assertTrue(description.getAll().isEmpty());
        assertEquals(new HashSet<ServerDescription>(asList(primary, secondary, otherSecondary, uninitiatedMember, notOkMember)),
                     cluster.getAll());
    }

    @Test
    public void testAny() throws UnknownHostException {
        assertEquals(asList(primary, secondary, otherSecondary, uninitiatedMember), cluster.getAny());
    }

    @Test
    public void testPrimaryOrSecondary() throws UnknownHostException {
        assertEquals(asList(primary, secondary, otherSecondary), cluster.getAnyPrimaryOrSecondary());
        assertEquals(asList(primary, secondary), cluster.getAnyPrimaryOrSecondary(new TagSet(asList(new Tag("foo", "1"),
                                                                                                    new Tag("bar", "2")))));
    }

    @Test
    public void testServerAddress(){
        assertEquals(primary, cluster.getByServerAddress(primary.getAddress()));
        assertNull(cluster.getByServerAddress(notOkMember.getAddress()));
    }

    @Test
    public void testSortingOfAll() throws UnknownHostException {
        ClusterDescription description =
        new ClusterDescription(Multiple, Unknown, asList(
                                                        builder()
                                                        .state(Connecting)
                                                        .address(new ServerAddress("loc:27019"))
                                                        .build(),
                                                        builder()
                                                        .state(Connecting)
                                                        .address(new ServerAddress("loc:27018"))
                                                        .build(),
                                                        builder()
                                                        .state(Connecting)
                                                        .address(new ServerAddress("loc:27017"))
                                                        .build())
        );
        Iterator<ServerDescription> iter = description.getAll().iterator();
        assertEquals(new ServerAddress("loc:27017"), iter.next().getAddress());
        assertEquals(new ServerAddress("loc:27018"), iter.next().getAddress());
        assertEquals(new ServerAddress("loc:27019"), iter.next().getAddress());
    }

    @Test
    public void clusterDescriptionWithAnIncompatibleServerShouldBeIncompatible() throws UnknownHostException {
        ClusterDescription description =
        new ClusterDescription(Multiple, Unknown, asList(
                                                        builder()
                                                        .state(Connecting)
                                                        .address(new ServerAddress("loc:27019"))
                                                        .build(),
                                                        builder()
                                                        .state(Connected)
                                                        .ok(true)
                                                        .address(new ServerAddress("loc:27018"))
                                                        .minWireVersion(MAX_DRIVER_WIRE_VERSION + 1)
                                                        .maxWireVersion(MAX_DRIVER_WIRE_VERSION + 1)
                                                        .build(),
                                                        builder()
                                                        .state(Connecting)
                                                        .address(new ServerAddress("loc:27017"))
                                                        .build())
        );
        assertFalse(description.isCompatibleWithDriver());
    }

    @Test
    public void clusterDescriptionWithCompatibleServerShouldBeCompatible() throws UnknownHostException {
        ClusterDescription description =
        new ClusterDescription(Multiple, Unknown, asList(
                                                        builder()
                                                        .state(Connecting)
                                                        .address(new ServerAddress("loc:27019"))
                                                        .build(),
                                                        builder()
                                                        .state(Connecting)
                                                        .address(new ServerAddress("loc:27018"))
                                                        .build(),
                                                        builder()
                                                        .state(Connecting)
                                                        .address(new ServerAddress("loc:27017"))
                                                        .build())
        );
        assertTrue(description.isCompatibleWithDriver());
    }

    @Test
    public void testObjectOverrides() throws UnknownHostException {
        ClusterDescription description =
        new ClusterDescription(Multiple, Unknown, asList(
                                                        builder()
                                                        .state(Connecting)
                                                        .address(new ServerAddress("loc:27019"))
                                                        .build(),
                                                        builder()
                                                        .state(Connecting)
                                                        .address(new ServerAddress("loc:27018"))
                                                        .build(),
                                                        builder()
                                                        .state(Connecting)
                                                        .address(new ServerAddress("loc:27017"))
                                                        .build())
        );
        ClusterDescription descriptionTwo =
        new ClusterDescription(Multiple, Unknown, asList(
                                                        builder()
                                                        .state(Connecting)
                                                        .address(new ServerAddress("loc:27019"))
                                                        .build(),
                                                        builder()
                                                        .state(Connecting)
                                                        .address(new ServerAddress("loc:27018"))
                                                        .build(),
                                                        builder()
                                                        .state(Connecting)
                                                        .address(new ServerAddress("loc:27017"))
                                                        .build())
        );
        assertEquals(description, descriptionTwo);
        assertEquals(description.hashCode(), descriptionTwo.hashCode());
        assertTrue(description.toString().startsWith("ClusterDescription"));
    }
}