// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.
//

//
// ---------------------------------------------------------------------------
// Ex.cpp
// ---------------------------------------------------------------------------

#include "stdafx.h"
#include "string.h"
#include "ex.h"
#include "holder.h"

// error codes
#include "corerror.h"

#include "../dlls/mscorrc/resource.h"

#include "olectl.h"

#include "corexcep.h"

#define MAX_EXCEPTION_MSG   200

// Set if fatal error (like stack overflow or out of memory) occurred in this process.
GVAL_IMPL_INIT(HRESULT, g_hrFatalError, S_OK);

// Helper function to get an exception object from outside the exception.  In
//  the CLR, it may be from the Thread object.  Non-CLR users have no thread object,
//  and it will do nothing.                                                    
void GetLastThrownObjectExceptionFromThread(void **ppvException);

Exception *Exception::g_OOMException = NULL;

// avoid global constructors
static BYTE g_OOMExceptionInstance[sizeof(OutOfMemoryException)];

Exception * Exception::GetOOMException()
{
    LIMITED_METHOD_CONTRACT;
    
    if (!g_OOMException) {
        // Create a local copy on the stack and then copy it over to the static instance.
        // This avoids race conditions caused by multiple initializations of vtable in the constructor       

        OutOfMemoryException local(TRUE);  // Construct a "preallocated" instance.
        memcpy((void*)&g_OOMExceptionInstance, (void*)&local, sizeof(OutOfMemoryException));

        g_OOMException = (OutOfMemoryException*)&g_OOMExceptionInstance;
    }

    return g_OOMException;
}

/*virtual*/ Exception *OutOfMemoryException::Clone()
{
    LIMITED_METHOD_CONTRACT;

    return GetOOMException();
}

//------------------------------------------------------------------------------
void Exception::Delete(Exception* pvMemory)
{
    CONTRACTL
    {
        GC_NOTRIGGER;
        NOTHROW;
        SO_TOLERANT;
        SUPPORTS_DAC_HOST_ONLY;   // Exceptions aren't currently marshalled by DAC - just used in the host
    }
    CONTRACTL_END;

    if ((pvMemory == 0) || pvMemory->IsPreallocatedException())
    {
        return;
    }

    ::delete((Exception *) pvMemory);
}

void Exception::GetMessage(SString &result)
{
    WRAPPER_NO_CONTRACT;
    
    return GenerateTopLevelHRExceptionMessage(GetHR(), result);
}

void HRMsgException::GetMessage(SString &result)
{
    WRAPPER_NO_CONTRACT;
    
    if (m_msg.IsEmpty())
        HRException::GetMessage(result);
    else
        result = m_msg;
}

Exception *Exception::Clone()
{
    CONTRACTL
    {
        GC_NOTRIGGER;
        THROWS;
    }
    CONTRACTL_END;
    
    NewHolder<Exception> retExcep(CloneHelper());
    if (m_innerException)
    {
        retExcep->m_innerException = m_innerException->Clone();
    }

    retExcep.SuppressRelease();
    return retExcep;
}

Exception *Exception::CloneHelper()
{
    StackSString s;
    GetMessage(s);
    return new HRMsgException(GetHR(), s);
}

Exception *Exception::DomainBoundClone()
{
    CONTRACTL
    {
        // Because we may call DomainBoundCloneHelper() of ObjrefException or CLRLastThrownObjectException
        // this should be GC_TRIGGERS, but we can not include EE contracts in Utilcode.
        THROWS;
    }
    CONTRACTL_END;

    NewHolder<Exception> retExcep(DomainBoundCloneHelper());
    if (m_innerException)
    {
        retExcep->m_innerException = m_innerException->DomainBoundClone();
    }

    retExcep.SuppressRelease();
    return retExcep;
}

BOOL Exception::IsTerminal()
{
    CONTRACTL
    {
        GC_NOTRIGGER;
        NOTHROW;

        // CLRException::GetHR() can eventually call BaseDomain::CreateHandle(),
        // which can indirectly cause a lock if we get a miss in the handle table
        // cache (TableCacheMissOnAlloc).  Since CLRException::GetHR() is virtual,
        // SCAN won't find this for you (though 40 minutes of one of the sql stress
        // tests will :-))
        CAN_TAKE_LOCK;
    }
    CONTRACTL_END;
    
    HRESULT hr = GetHR();
    return (COR_E_THREADABORTED == hr);
}

BOOL Exception::IsTransient()
{
    WRAPPER_NO_CONTRACT;

    return IsTransient(GetHR());
}

/* static */
BOOL Exception::IsTransient(HRESULT hr)
{
    LIMITED_METHOD_CONTRACT;
    
    return (hr == COR_E_THREADABORTED
            || hr == COR_E_THREADINTERRUPTED
            || hr == COR_E_THREADSTOP
            || hr == COR_E_APPDOMAINUNLOADED
            || hr == E_OUTOFMEMORY
            || hr == HRESULT_FROM_WIN32(ERROR_COMMITMENT_LIMIT) // ran out of room in pagefile
            || hr == HRESULT_FROM_WIN32(ERROR_NOT_ENOUGH_MEMORY)
            || hr == (HRESULT)STATUS_NO_MEMORY
            || hr == COR_E_STACKOVERFLOW
            || hr == MSEE_E_ASSEMBLYLOADINPROGRESS);
}

//------------------------------------------------------------------------------
// Functions to manage the preallocated exceptions.
// Virtual 
BOOL Exception::IsPreallocatedException()
{   // Most exceptions can't be preallocated.  If they can be, their class
    //  should provide a virtual override of this function.
    return FALSE;
}

BOOL Exception::IsPreallocatedOOMException()
{   // This is the preallocated OOM if it is preallocated and is OOM.
    return IsPreallocatedException() && (GetInstanceType() == OutOfMemoryException::GetType());
}

//------------------------------------------------------------------------------
#ifdef _PREFAST_
#pragma warning(push)
#pragma warning(disable:21000) // Suppress PREFast warning about overly large function
#endif
LPCSTR Exception::GetHRSymbolicName(HRESULT hr)
{
    LIMITED_METHOD_CONTRACT;

#define CASE_HRESULT(hrname) case hrname: return #hrname;

    switch (hr)
    {
        CASE_HRESULT(S_OK)//                             0x00000000L
        CASE_HRESULT(S_FALSE)//                          0x00000001L

        CASE_HRESULT(E_UNEXPECTED)//                     0x8000FFFFL
        CASE_HRESULT(E_NOTIMPL)//                        0x80004001L
        CASE_HRESULT(E_OUTOFMEMORY)//                    0x8007000EL
        CASE_HRESULT(E_INVALIDARG)//                     0x80070057L
        CASE_HRESULT(E_NOINTERFACE)//                    0x80004002L
        CASE_HRESULT(E_POINTER)//                        0x80004003L
        CASE_HRESULT(E_HANDLE)//                         0x80070006L
        CASE_HRESULT(E_ABORT)//                          0x80004004L
        CASE_HRESULT(E_FAIL)//                           0x80004005L
        CASE_HRESULT(E_ACCESSDENIED)//                   0x80070005L

#ifdef FEATURE_COMINTEROP
        CASE_HRESULT(CO_E_INIT_TLS)//                    0x80004006L
        CASE_HRESULT(CO_E_INIT_SHARED_ALLOCATOR)//       0x80004007L
        CASE_HRESULT(CO_E_INIT_MEMORY_ALLOCATOR)//       0x80004008L
        CASE_HRESULT(CO_E_INIT_CLASS_CACHE)//            0x80004009L
        CASE_HRESULT(CO_E_INIT_RPC_CHANNEL)//            0x8000400AL
        CASE_HRESULT(CO_E_INIT_TLS_SET_CHANNEL_CONTROL)// 0x8000400BL
        CASE_HRESULT(CO_E_INIT_TLS_CHANNEL_CONTROL)//    0x8000400CL
        CASE_HRESULT(CO_E_INIT_UNACCEPTED_USER_ALLOCATOR)// 0x8000400DL
        CASE_HRESULT(CO_E_INIT_SCM_MUTEX_EXISTS)//       0x8000400EL
        CASE_HRESULT(CO_E_INIT_SCM_FILE_MAPPING_EXISTS)// 0x8000400FL
        CASE_HRESULT(CO_E_INIT_SCM_MAP_VIEW_OF_FILE)//   0x80004010L
        CASE_HRESULT(CO_E_INIT_SCM_EXEC_FAILURE)//       0x80004011L
        CASE_HRESULT(CO_E_INIT_ONLY_SINGLE_THREADED)//   0x80004012L

// ******************
// FACILITY_ITF
// ******************

        CASE_HRESULT(OLE_E_OLEVERB)//                    0x80040000L
        CASE_HRESULT(OLE_E_ADVF)//                       0x80040001L
        CASE_HRESULT(OLE_E_ENUM_NOMORE)//                0x80040002L
        CASE_HRESULT(OLE_E_ADVISENOTSUPPORTED)//         0x80040003L
        CASE_HRESULT(OLE_E_NOCONNECTION)//               0x80040004L
        CASE_HRESULT(OLE_E_NOTRUNNING)//                 0x80040005L
        CASE_HRESULT(OLE_E_NOCACHE)//                    0x80040006L
        CASE_HRESULT(OLE_E_BLANK)//                      0x80040007L
        CASE_HRESULT(OLE_E_CLASSDIFF)//                  0x80040008L
        CASE_HRESULT(OLE_E_CANT_GETMONIKER)//            0x80040009L
        CASE_HRESULT(OLE_E_CANT_BINDTOSOURCE)//          0x8004000AL
        CASE_HRESULT(OLE_E_STATIC)//                     0x8004000BL
        CASE_HRESULT(OLE_E_PROMPTSAVECANCELLED)//        0x8004000CL
        CASE_HRESULT(OLE_E_INVALIDRECT)//                0x8004000DL
        CASE_HRESULT(OLE_E_WRONGCOMPOBJ)//               0x8004000EL
        CASE_HRESULT(OLE_E_INVALIDHWND)//                0x8004000FL
        CASE_HRESULT(OLE_E_NOT_INPLACEACTIVE)//          0x80040010L
        CASE_HRESULT(OLE_E_CANTCONVERT)//                0x80040011L
        CASE_HRESULT(OLE_E_NOSTORAGE)//                  0x80040012L
        CASE_HRESULT(DV_E_FORMATETC)//                   0x80040064L
        CASE_HRESULT(DV_E_DVTARGETDEVICE)//              0x80040065L
        CASE_HRESULT(DV_E_STGMEDIUM)//                   0x80040066L
        CASE_HRESULT(DV_E_STATDATA)//                    0x80040067L
        CASE_HRESULT(DV_E_LINDEX)//                      0x80040068L
        CASE_HRESULT(DV_E_TYMED)//                       0x80040069L
        CASE_HRESULT(DV_E_CLIPFORMAT)//                  0x8004006AL
        CASE_HRESULT(DV_E_DVASPECT)//                    0x8004006BL
        CASE_HRESULT(DV_E_DVTARGETDEVICE_SIZE)//         0x8004006CL
        CASE_HRESULT(DV_E_NOIVIEWOBJECT)//               0x8004006DL
        CASE_HRESULT(DRAGDROP_E_NOTREGISTERED)//         0x80040100L
        CASE_HRESULT(DRAGDROP_E_ALREADYREGISTERED)//     0x80040101L
        CASE_HRESULT(DRAGDROP_E_INVALIDHWND)//           0x80040102L
        CASE_HRESULT(CLASS_E_NOAGGREGATION)//            0x80040110L
        CASE_HRESULT(CLASS_E_CLASSNOTAVAILABLE)//        0x80040111L
        CASE_HRESULT(VIEW_E_DRAW)//                      0x80040140L
        CASE_HRESULT(REGDB_E_READREGDB)//                0x80040150L
        CASE_HRESULT(REGDB_E_WRITEREGDB)//               0x80040151L
        CASE_HRESULT(REGDB_E_KEYMISSING)//               0x80040152L
        CASE_HRESULT(REGDB_E_INVALIDVALUE)//             0x80040153L
        CASE_HRESULT(REGDB_E_CLASSNOTREG)//              0x80040154L
        CASE_HRESULT(CACHE_E_NOCACHE_UPDATED)//          0x80040170L
        CASE_HRESULT(OLEOBJ_E_NOVERBS)//                 0x80040180L
        CASE_HRESULT(INPLACE_E_NOTUNDOABLE)//            0x800401A0L
        CASE_HRESULT(INPLACE_E_NOTOOLSPACE)//            0x800401A1L
        CASE_HRESULT(CONVERT10_E_OLESTREAM_GET)//        0x800401C0L
        CASE_HRESULT(CONVERT10_E_OLESTREAM_PUT)//        0x800401C1L
        CASE_HRESULT(CONVERT10_E_OLESTREAM_FMT)//        0x800401C2L
        CASE_HRESULT(CONVERT10_E_OLESTREAM_BITMAP_TO_DIB)// 0x800401C3L
        CASE_HRESULT(CONVERT10_E_STG_FMT)//              0x800401C4L
        CASE_HRESULT(CONVERT10_E_STG_NO_STD_STREAM)//    0x800401C5L
        CASE_HRESULT(CONVERT10_E_STG_DIB_TO_BITMAP)//    0x800401C6L
        CASE_HRESULT(CLIPBRD_E_CANT_OPEN)//              0x800401D0L
        CASE_HRESULT(CLIPBRD_E_CANT_EMPTY)//             0x800401D1L
        CASE_HRESULT(CLIPBRD_E_CANT_SET)//               0x800401D2L
        CASE_HRESULT(CLIPBRD_E_BAD_DATA)//               0x800401D3L
        CASE_HRESULT(CLIPBRD_E_CANT_CLOSE)//             0x800401D4L
        CASE_HRESULT(MK_E_CONNECTMANUALLY)//             0x800401E0L
        CASE_HRESULT(MK_E_EXCEEDEDDEADLINE)//            0x800401E1L
        CASE_HRESULT(MK_E_NEEDGENERIC)//                 0x800401E2L
        CASE_HRESULT(MK_E_UNAVAILABLE)//                 0x800401E3L
        CASE_HRESULT(MK_E_SYNTAX)//                      0x800401E4L
        CASE_HRESULT(MK_E_NOOBJECT)//                    0x800401E5L
        CASE_HRESULT(MK_E_INVALIDEXTENSION)//            0x800401E6L
        CASE_HRESULT(MK_E_INTERMEDIATEINTERFACENOTSUPPORTED)// 0x800401E7L
        CASE_HRESULT(MK_E_NOTBINDABLE)//                 0x800401E8L
        CASE_HRESULT(MK_E_NOTBOUND)//                    0x800401E9L
        CASE_HRESULT(MK_E_CANTOPENFILE)//                0x800401EAL
        CASE_HRESULT(MK_E_MUSTBOTHERUSER)//              0x800401EBL
        CASE_HRESULT(MK_E_NOINVERSE)//                   0x800401ECL
        CASE_HRESULT(MK_E_NOSTORAGE)//                   0x800401EDL
        CASE_HRESULT(MK_E_NOPREFIX)//                    0x800401EEL
        CASE_HRESULT(MK_E_ENUMERATION_FAILED)//          0x800401EFL
        CASE_HRESULT(CO_E_NOTINITIALIZED)//              0x800401F0L
        CASE_HRESULT(CO_E_ALREADYINITIALIZED)//          0x800401F1L
        CASE_HRESULT(CO_E_CANTDETERMINECLASS)//          0x800401F2L
        CASE_HRESULT(CO_E_CLASSSTRING)//                 0x800401F3L
        CASE_HRESULT(CO_E_IIDSTRING)//                   0x800401F4L
        CASE_HRESULT(CO_E_APPNOTFOUND)//                 0x800401F5L
        CASE_HRESULT(CO_E_APPSINGLEUSE)//                0x800401F6L
        CASE_HRESULT(CO_E_ERRORINAPP)//                  0x800401F7L
        CASE_HRESULT(CO_E_DLLNOTFOUND)//                 0x800401F8L
        CASE_HRESULT(CO_E_ERRORINDLL)//                  0x800401F9L
        CASE_HRESULT(CO_E_WRONGOSFORAPP)//               0x800401FAL
        CASE_HRESULT(CO_E_OBJNOTREG)//                   0x800401FBL
        CASE_HRESULT(CO_E_OBJISREG)//                    0x800401FCL
        CASE_HRESULT(CO_E_OBJNOTCONNECTED)//             0x800401FDL
        CASE_HRESULT(CO_E_APPDIDNTREG)//                 0x800401FEL
        CASE_HRESULT(CO_E_RELEASED)//                    0x800401FFL

        CASE_HRESULT(OLE_S_USEREG)//                     0x00040000L
        CASE_HRESULT(OLE_S_STATIC)//                     0x00040001L
        CASE_HRESULT(OLE_S_MAC_CLIPFORMAT)//             0x00040002L
        CASE_HRESULT(DRAGDROP_S_DROP)//                  0x00040100L
        CASE_HRESULT(DRAGDROP_S_CANCEL)//                0x00040101L
        CASE_HRESULT(DRAGDROP_S_USEDEFAULTCURSORS)//     0x00040102L
        CASE_HRESULT(DATA_S_SAMEFORMATETC)//             0x00040130L
        CASE_HRESULT(VIEW_S_ALREADY_FROZEN)//            0x00040140L
        CASE_HRESULT(CACHE_S_FORMATETC_NOTSUPPORTED)//   0x00040170L
        CASE_HRESULT(CACHE_S_SAMECACHE)//                0x00040171L
        CASE_HRESULT(CACHE_S_SOMECACHES_NOTUPDATED)//    0x00040172L
        CASE_HRESULT(OLEOBJ_S_INVALIDVERB)//             0x00040180L
        CASE_HRESULT(OLEOBJ_S_CANNOT_DOVERB_NOW)//       0x00040181L
        CASE_HRESULT(OLEOBJ_S_INVALIDHWND)//             0x00040182L
        CASE_HRESULT(INPLACE_S_TRUNCATED)//              0x000401A0L
        CASE_HRESULT(CONVERT10_S_NO_PRESENTATION)//      0x000401C0L
        CASE_HRESULT(MK_S_REDUCED_TO_SELF)//             0x000401E2L
        CASE_HRESULT(MK_S_ME)//                          0x000401E4L
        CASE_HRESULT(MK_S_HIM)//                         0x000401E5L
        CASE_HRESULT(MK_S_US)//                          0x000401E6L
        CASE_HRESULT(MK_S_MONIKERALREADYREGISTERED)//    0x000401E7L

// ******************
// FACILITY_WINDOWS
// ******************

        CASE_HRESULT(CO_E_CLASS_CREATE_FAILED)//         0x80080001L
        CASE_HRESULT(CO_E_SCM_ERROR)//                   0x80080002L
        CASE_HRESULT(CO_E_SCM_RPC_FAILURE)//             0x80080003L
        CASE_HRESULT(CO_E_BAD_PATH)//                    0x80080004L
        CASE_HRESULT(CO_E_SERVER_EXEC_FAILURE)//         0x80080005L
        CASE_HRESULT(CO_E_OBJSRV_RPC_FAILURE)//          0x80080006L
        CASE_HRESULT(MK_E_NO_NORMALIZED)//               0x80080007L
        CASE_HRESULT(CO_E_SERVER_STOPPING)//             0x80080008L
        CASE_HRESULT(MEM_E_INVALID_ROOT)//               0x80080009L
        CASE_HRESULT(MEM_E_INVALID_LINK)//               0x80080010L
        CASE_HRESULT(MEM_E_INVALID_SIZE)//               0x80080011L

// ******************
// FACILITY_DISPATCH
// ******************

        CASE_HRESULT(DISP_E_UNKNOWNINTERFACE)//          0x80020001L
        CASE_HRESULT(DISP_E_MEMBERNOTFOUND)//            0x80020003L
        CASE_HRESULT(DISP_E_PARAMNOTFOUND)//             0x80020004L
        CASE_HRESULT(DISP_E_TYPEMISMATCH)//              0x80020005L
        CASE_HRESULT(DISP_E_UNKNOWNNAME)//               0x80020006L
        CASE_HRESULT(DISP_E_NONAMEDARGS)//               0x80020007L
        CASE_HRESULT(DISP_E_BADVARTYPE)//                0x80020008L
        CASE_HRESULT(DISP_E_EXCEPTION)//                 0x80020009L
        CASE_HRESULT(DISP_E_OVERFLOW)//                  0x8002000AL
        CASE_HRESULT(DISP_E_BADINDEX)//                  0x8002000BL
        CASE_HRESULT(DISP_E_UNKNOWNLCID)//               0x8002000CL
        CASE_HRESULT(DISP_E_ARRAYISLOCKED)//             0x8002000DL
        CASE_HRESULT(DISP_E_BADPARAMCOUNT)//             0x8002000EL
        CASE_HRESULT(DISP_E_PARAMNOTOPTIONAL)//          0x8002000FL
        CASE_HRESULT(DISP_E_BADCALLEE)//                 0x80020010L
        CASE_HRESULT(DISP_E_NOTACOLLECTION)//            0x80020011L
        CASE_HRESULT(TYPE_E_BUFFERTOOSMALL)//            0x80028016L
        CASE_HRESULT(TYPE_E_INVDATAREAD)//               0x80028018L
        CASE_HRESULT(TYPE_E_UNSUPFORMAT)//               0x80028019L
        CASE_HRESULT(TYPE_E_REGISTRYACCESS)//            0x8002801CL
        CASE_HRESULT(TYPE_E_LIBNOTREGISTERED)//          0x8002801DL
        CASE_HRESULT(TYPE_E_UNDEFINEDTYPE)//             0x80028027L
        CASE_HRESULT(TYPE_E_QUALIFIEDNAMEDISALLOWED)//   0x80028028L
        CASE_HRESULT(TYPE_E_INVALIDSTATE)//              0x80028029L
        CASE_HRESULT(TYPE_E_WRONGTYPEKIND)//             0x8002802AL
        CASE_HRESULT(TYPE_E_ELEMENTNOTFOUND)//           0x8002802BL
        CASE_HRESULT(TYPE_E_AMBIGUOUSNAME)//             0x8002802CL
        CASE_HRESULT(TYPE_E_NAMECONFLICT)//              0x8002802DL
        CASE_HRESULT(TYPE_E_UNKNOWNLCID)//               0x8002802EL
        CASE_HRESULT(TYPE_E_DLLFUNCTIONNOTFOUND)//       0x8002802FL
        CASE_HRESULT(TYPE_E_BADMODULEKIND)//             0x800288BDL
        CASE_HRESULT(TYPE_E_SIZETOOBIG)//                0x800288C5L
        CASE_HRESULT(TYPE_E_DUPLICATEID)//               0x800288C6L
        CASE_HRESULT(TYPE_E_INVALIDID)//                 0x800288CFL
        CASE_HRESULT(TYPE_E_TYPEMISMATCH)//              0x80028CA0L
        CASE_HRESULT(TYPE_E_OUTOFBOUNDS)//               0x80028CA1L
        CASE_HRESULT(TYPE_E_IOERROR)//                   0x80028CA2L
        CASE_HRESULT(TYPE_E_CANTCREATETMPFILE)//         0x80028CA3L
        CASE_HRESULT(TYPE_E_CANTLOADLIBRARY)//           0x80029C4AL
        CASE_HRESULT(TYPE_E_INCONSISTENTPROPFUNCS)//     0x80029C83L
        CASE_HRESULT(TYPE_E_CIRCULARTYPE)//              0x80029C84L

// ******************
// FACILITY_STORAGE
// ******************

        CASE_HRESULT(STG_E_INVALIDFUNCTION)//            0x80030001L
        CASE_HRESULT(STG_E_FILENOTFOUND)//               0x80030002L
        CASE_HRESULT(STG_E_PATHNOTFOUND)//               0x80030003L
        CASE_HRESULT(STG_E_TOOMANYOPENFILES)//           0x80030004L
        CASE_HRESULT(STG_E_ACCESSDENIED)//               0x80030005L
        CASE_HRESULT(STG_E_INVALIDHANDLE)//              0x80030006L
        CASE_HRESULT(STG_E_INSUFFICIENTMEMORY)//         0x80030008L
        CASE_HRESULT(STG_E_INVALIDPOINTER)//             0x80030009L
        CASE_HRESULT(STG_E_NOMOREFILES)//                0x80030012L
        CASE_HRESULT(STG_E_DISKISWRITEPROTECTED)//       0x80030013L
        CASE_HRESULT(STG_E_SEEKERROR)//                  0x80030019L
        CASE_HRESULT(STG_E_WRITEFAULT)//                 0x8003001DL
        CASE_HRESULT(STG_E_READFAULT)//                  0x8003001EL
        CASE_HRESULT(STG_E_SHAREVIOLATION)//             0x80030020L
        CASE_HRESULT(STG_E_LOCKVIOLATION)//              0x80030021L
        CASE_HRESULT(STG_E_FILEALREADYEXISTS)//          0x80030050L
        CASE_HRESULT(STG_E_INVALIDPARAMETER)//           0x80030057L
        CASE_HRESULT(STG_E_MEDIUMFULL)//                 0x80030070L
        CASE_HRESULT(STG_E_ABNORMALAPIEXIT)//            0x800300FAL
        CASE_HRESULT(STG_E_INVALIDHEADER)//              0x800300FBL
        CASE_HRESULT(STG_E_INVALIDNAME)//                0x800300FCL
        CASE_HRESULT(STG_E_UNKNOWN)//                    0x800300FDL
        CASE_HRESULT(STG_E_UNIMPLEMENTEDFUNCTION)//      0x800300FEL
        CASE_HRESULT(STG_E_INVALIDFLAG)//                0x800300FFL
        CASE_HRESULT(STG_E_INUSE)//                      0x80030100L
        CASE_HRESULT(STG_E_NOTCURRENT)//                 0x80030101L
        CASE_HRESULT(STG_E_REVERTED)//                   0x80030102L
        CASE_HRESULT(STG_E_CANTSAVE)//                   0x80030103L
        CASE_HRESULT(STG_E_OLDFORMAT)//                  0x80030104L
        CASE_HRESULT(STG_E_OLDDLL)//                     0x80030105L
        CASE_HRESULT(STG_E_SHAREREQUIRED)//              0x80030106L
        CASE_HRESULT(STG_E_NOTFILEBASEDSTORAGE)//        0x80030107L
        CASE_HRESULT(STG_S_CONVERTED)//                  0x00030200L

// ******************
// FACILITY_RPC
// ******************

        CASE_HRESULT(RPC_E_CALL_REJECTED)//              0x80010001L
        CASE_HRESULT(RPC_E_CALL_CANCELED)//              0x80010002L
        CASE_HRESULT(RPC_E_CANTPOST_INSENDCALL)//        0x80010003L
        CASE_HRESULT(RPC_E_CANTCALLOUT_INASYNCCALL)//    0x80010004L
        CASE_HRESULT(RPC_E_CANTCALLOUT_INEXTERNALCALL)// 0x80010005L
        CASE_HRESULT(RPC_E_CONNECTION_TERMINATED)//      0x80010006L
        CASE_HRESULT(RPC_E_SERVER_DIED)//                0x80010007L
        CASE_HRESULT(RPC_E_CLIENT_DIED)//                0x80010008L
        CASE_HRESULT(RPC_E_INVALID_DATAPACKET)//         0x80010009L
        CASE_HRESULT(RPC_E_CANTTRANSMIT_CALL)//          0x8001000AL
        CASE_HRESULT(RPC_E_CLIENT_CANTMARSHAL_DATA)//    0x8001000BL
        CASE_HRESULT(RPC_E_CLIENT_CANTUNMARSHAL_DATA)//  0x8001000CL
        CASE_HRESULT(RPC_E_SERVER_CANTMARSHAL_DATA)//    0x8001000DL
        CASE_HRESULT(RPC_E_SERVER_CANTUNMARSHAL_DATA)//  0x8001000EL
        CASE_HRESULT(RPC_E_INVALID_DATA)//               0x8001000FL
        CASE_HRESULT(RPC_E_INVALID_PARAMETER)//          0x80010010L
        CASE_HRESULT(RPC_E_CANTCALLOUT_AGAIN)//          0x80010011L
        CASE_HRESULT(RPC_E_SERVER_DIED_DNE)//            0x80010012L
        CASE_HRESULT(RPC_E_SYS_CALL_FAILED)//            0x80010100L
        CASE_HRESULT(RPC_E_OUT_OF_RESOURCES)//           0x80010101L
        CASE_HRESULT(RPC_E_ATTEMPTED_MULTITHREAD)//      0x80010102L
        CASE_HRESULT(RPC_E_NOT_REGISTERED)//             0x80010103L
        CASE_HRESULT(RPC_E_FAULT)//                      0x80010104L
        CASE_HRESULT(RPC_E_SERVERFAULT)//                0x80010105L
        CASE_HRESULT(RPC_E_CHANGED_MODE)//               0x80010106L
        CASE_HRESULT(RPC_E_INVALIDMETHOD)//              0x80010107L
        CASE_HRESULT(RPC_E_DISCONNECTED)//               0x80010108L
        CASE_HRESULT(RPC_E_RETRY)//                      0x80010109L
        CASE_HRESULT(RPC_E_SERVERCALL_RETRYLATER)//      0x8001010AL
        CASE_HRESULT(RPC_E_SERVERCALL_REJECTED)//        0x8001010BL
        CASE_HRESULT(RPC_E_INVALID_CALLDATA)//           0x8001010CL
        CASE_HRESULT(RPC_E_CANTCALLOUT_ININPUTSYNCCALL)// 0x8001010DL
        CASE_HRESULT(RPC_E_WRONG_THREAD)//               0x8001010EL
        CASE_HRESULT(RPC_E_THREAD_NOT_INIT)//            0x8001010FL
        CASE_HRESULT(RPC_E_UNEXPECTED)//                 0x8001FFFFL   

// ******************
// FACILITY_CTL
// ******************

        CASE_HRESULT(CTL_E_ILLEGALFUNCTIONCALL)       
        CASE_HRESULT(CTL_E_OVERFLOW)                  
        CASE_HRESULT(CTL_E_OUTOFMEMORY)               
        CASE_HRESULT(CTL_E_DIVISIONBYZERO)            
        CASE_HRESULT(CTL_E_OUTOFSTRINGSPACE)          
        CASE_HRESULT(CTL_E_OUTOFSTACKSPACE)           
        CASE_HRESULT(CTL_E_BADFILENAMEORNUMBER)       
        CASE_HRESULT(CTL_E_FILENOTFOUND)              
        CASE_HRESULT(CTL_E_BADFILEMODE)               
        CASE_HRESULT(CTL_E_FILEALREADYOPEN)           
        CASE_HRESULT(CTL_E_DEVICEIOERROR)             
        CASE_HRESULT(CTL_E_FILEALREADYEXISTS)         
        CASE_HRESULT(CTL_E_BADRECORDLENGTH)           
        CASE_HRESULT(CTL_E_DISKFULL)                  
        CASE_HRESULT(CTL_E_BADRECORDNUMBER)           
        CASE_HRESULT(CTL_E_BADFILENAME)               
        CASE_HRESULT(CTL_E_TOOMANYFILES)              
        CASE_HRESULT(CTL_E_DEVICEUNAVAILABLE)         
        CASE_HRESULT(CTL_E_PERMISSIONDENIED)          
        CASE_HRESULT(CTL_E_DISKNOTREADY)              
        CASE_HRESULT(CTL_E_PATHFILEACCESSERROR)       
        CASE_HRESULT(CTL_E_PATHNOTFOUND)              
        CASE_HRESULT(CTL_E_INVALIDPATTERNSTRING)      
        CASE_HRESULT(CTL_E_INVALIDUSEOFNULL)          
        CASE_HRESULT(CTL_E_INVALIDFILEFORMAT)         
        CASE_HRESULT(CTL_E_INVALIDPROPERTYVALUE)      
        CASE_HRESULT(CTL_E_INVALIDPROPERTYARRAYINDEX) 
        CASE_HRESULT(CTL_E_SETNOTSUPPORTEDATRUNTIME)  
        CASE_HRESULT(CTL_E_SETNOTSUPPORTED)           
        CASE_HRESULT(CTL_E_NEEDPROPERTYARRAYINDEX)    
        CASE_HRESULT(CTL_E_SETNOTPERMITTED)           
        CASE_HRESULT(CTL_E_GETNOTSUPPORTEDATRUNTIME)  
        CASE_HRESULT(CTL_E_GETNOTSUPPORTED)           
        CASE_HRESULT(CTL_E_PROPERTYNOTFOUND)          
        CASE_HRESULT(CTL_E_INVALIDCLIPBOARDFORMAT)    
        CASE_HRESULT(CTL_E_INVALIDPICTURE)            
        CASE_HRESULT(CTL_E_PRINTERERROR)              
        CASE_HRESULT(CTL_E_CANTSAVEFILETOTEMP)        
        CASE_HRESULT(CTL_E_SEARCHTEXTNOTFOUND)        
        CASE_HRESULT(CTL_E_REPLACEMENTSTOOLONG)       
#endif // FEATURE_COMINTEROP

#ifdef _DEBUG  // @todo: do we want to burn strings for this in a free build?

    CASE_HRESULT(CEE_E_ENTRYPOINT)
    CASE_HRESULT(CEE_E_CVTRES_NOT_FOUND)
    CASE_HRESULT(MSEE_E_LOADLIBFAILED)
    CASE_HRESULT(MSEE_E_GETPROCFAILED)
    CASE_HRESULT(MSEE_E_MULTCOPIESLOADED)
    CASE_HRESULT(COR_E_APPDOMAINUNLOADED)
    CASE_HRESULT(COR_E_CANNOTUNLOADAPPDOMAIN)
    CASE_HRESULT(MSEE_E_ASSEMBLYLOADINPROGRESS)
    CASE_HRESULT(MSEE_E_CANNOTCREATEAPPDOMAIN)
    CASE_HRESULT(COR_E_FIXUPSINEXE)
    CASE_HRESULT(COR_E_NO_LOADLIBRARY_ALLOWED)
    CASE_HRESULT(COR_E_MODULE_HASH_CHECK_FAILED)
    CASE_HRESULT(FUSION_E_LOADFROM_BLOCKED)
    CASE_HRESULT(FUSION_E_CACHEFILE_FAILED)
    CASE_HRESULT(FUSION_E_REF_DEF_MISMATCH)
    CASE_HRESULT(FUSION_E_INVALID_PRIVATE_ASM_LOCATION)
    CASE_HRESULT(FUSION_E_ASM_MODULE_MISSING)
    CASE_HRESULT(FUSION_E_UNEXPECTED_MODULE_FOUND)
    CASE_HRESULT(FUSION_E_PRIVATE_ASM_DISALLOWED)
    CASE_HRESULT(FUSION_E_SIGNATURE_CHECK_FAILED)
    CASE_HRESULT(FUSION_E_DATABASE_ERROR)
    CASE_HRESULT(FUSION_E_INVALID_NAME)
    CASE_HRESULT(FUSION_E_CODE_DOWNLOAD_DISABLED)
    CASE_HRESULT(FUSION_E_UNINSTALL_DISALLOWED)
    CASE_HRESULT(CLDB_E_FILE_BADREAD)
    CASE_HRESULT(CLDB_E_FILE_BADWRITE)
    CASE_HRESULT(CLDB_E_FILE_READONLY)
    CASE_HRESULT(CLDB_E_NAME_ERROR)
    CASE_HRESULT(CLDB_S_TRUNCATION)
    CASE_HRESULT(CLDB_E_TRUNCATION)
    CASE_HRESULT(CLDB_E_FILE_OLDVER)
    CASE_HRESULT(CLDB_E_RELOCATED)
    CASE_HRESULT(CLDB_S_NULL)
    CASE_HRESULT(CLDB_E_SMDUPLICATE)
    CASE_HRESULT(CLDB_E_NO_DATA)
    CASE_HRESULT(CLDB_E_READONLY)
    CASE_HRESULT(CLDB_E_INCOMPATIBLE)
    CASE_HRESULT(CLDB_E_FILE_CORRUPT)
    CASE_HRESULT(CLDB_E_SCHEMA_VERNOTFOUND)
    CASE_HRESULT(CLDB_E_BADUPDATEMODE)
    CASE_HRESULT(CLDB_E_INDEX_NONULLKEYS)
    CASE_HRESULT(CLDB_E_INDEX_DUPLICATE)
    CASE_HRESULT(CLDB_E_INDEX_BADTYPE)
    CASE_HRESULT(CLDB_E_INDEX_NOTFOUND)
    CASE_HRESULT(CLDB_S_INDEX_TABLESCANREQUIRED)
    CASE_HRESULT(CLDB_E_RECORD_NOTFOUND)
    CASE_HRESULT(CLDB_E_RECORD_OVERFLOW)
    CASE_HRESULT(CLDB_E_RECORD_DUPLICATE)
    CASE_HRESULT(CLDB_E_RECORD_PKREQUIRED)
    CASE_HRESULT(CLDB_E_RECORD_DELETED)
    CASE_HRESULT(CLDB_E_RECORD_OUTOFORDER)
    CASE_HRESULT(CLDB_E_COLUMN_OVERFLOW)
    CASE_HRESULT(CLDB_E_COLUMN_READONLY)
    CASE_HRESULT(CLDB_E_COLUMN_SPECIALCOL)
    CASE_HRESULT(CLDB_E_COLUMN_PKNONULLS)
    CASE_HRESULT(CLDB_E_TABLE_CANTDROP)
    CASE_HRESULT(CLDB_E_OBJECT_NOTFOUND)
    CASE_HRESULT(CLDB_E_OBJECT_COLNOTFOUND)
    CASE_HRESULT(CLDB_E_VECTOR_BADINDEX)
    CASE_HRESULT(CLDB_E_TOO_BIG)
    CASE_HRESULT(META_E_DUPLICATE)
    CASE_HRESULT(META_E_GUID_REQUIRED)
    CASE_HRESULT(META_E_TYPEDEF_MISMATCH)
    CASE_HRESULT(META_E_MERGE_COLLISION)
    CASE_HRESULT(META_E_METHD_NOT_FOUND)
    CASE_HRESULT(META_E_FIELD_NOT_FOUND)
    CASE_HRESULT(META_S_PARAM_MISMATCH)
    CASE_HRESULT(META_E_PARAM_MISMATCH)
    CASE_HRESULT(META_E_BADMETADATA)
    CASE_HRESULT(META_E_INTFCEIMPL_NOT_FOUND)
    CASE_HRESULT(META_E_CLASS_LAYOUT_INCONSISTENT)
    CASE_HRESULT(META_E_FIELD_MARSHAL_NOT_FOUND)
    CASE_HRESULT(META_E_METHODSEM_NOT_FOUND)
    CASE_HRESULT(META_E_EVENT_NOT_FOUND)
    CASE_HRESULT(META_E_PROP_NOT_FOUND)
    CASE_HRESULT(META_E_BAD_SIGNATURE)
    CASE_HRESULT(META_E_BAD_INPUT_PARAMETER)
    CASE_HRESULT(META_E_METHDIMPL_INCONSISTENT)
    CASE_HRESULT(META_E_MD_INCONSISTENCY)
    CASE_HRESULT(META_E_CANNOTRESOLVETYPEREF)
    CASE_HRESULT(META_S_DUPLICATE)
    CASE_HRESULT(META_E_STRINGSPACE_FULL)
    CASE_HRESULT(META_E_UNEXPECTED_REMAP)
    CASE_HRESULT(META_E_HAS_UNMARKALL)
    CASE_HRESULT(META_E_MUST_CALL_UNMARKALL)
    CASE_HRESULT(TLBX_E_CANT_LOAD_MODULE)
    CASE_HRESULT(TLBX_E_CANT_LOAD_CLASS)
    CASE_HRESULT(TLBX_E_NULL_MODULE)
    CASE_HRESULT(TLBX_E_NO_CLSID_KEY)
    CASE_HRESULT(TLBX_E_CIRCULAR_EXPORT)
    CASE_HRESULT(TLBX_E_CIRCULAR_EXPORT2)
    CASE_HRESULT(TLBX_E_CIRCULAR_IMPORT)
    CASE_HRESULT(TLBX_E_BAD_NATIVETYPE)
    CASE_HRESULT(TLBX_E_BAD_VTABLE)
    CASE_HRESULT(TLBX_E_CRM_NON_STATIC)
    CASE_HRESULT(TLBX_E_CRM_INVALID_SIG)
    CASE_HRESULT(TLBX_E_CLASS_LOAD_EXCEPTION)
    CASE_HRESULT(TLBX_E_UNKNOWN_SIGNATURE)
    CASE_HRESULT(TLBX_E_REFERENCED_TYPELIB)
    CASE_HRESULT(TLBX_S_REFERENCED_TYPELIB)
    CASE_HRESULT(TLBX_E_LAYOUT_ERROR)
    CASE_HRESULT(TLBX_E_NOTIUNKNOWN)
    CASE_HRESULT(TLBX_E_NONVISIBLEVALUECLASS)
    CASE_HRESULT(TLBX_E_LPTSTR_NOT_ALLOWED)
    CASE_HRESULT(TLBX_E_AUTO_CS_NOT_ALLOWED)
    CASE_HRESULT(TLBX_S_NOSTDINTERFACE)
    CASE_HRESULT(TLBX_S_DUPLICATE_DISPID)
    CASE_HRESULT(TLBX_E_ENUM_VALUE_INVALID)
    CASE_HRESULT(TLBX_W_ENUM_VALUE_TOOBIG)
    CASE_HRESULT(TLBX_E_DUPLICATE_IID)
    CASE_HRESULT(TLBX_E_NO_NESTED_ARRAYS)
    CASE_HRESULT(TLBX_E_PARAM_ERROR_NAMED)
    CASE_HRESULT(TLBX_E_PARAM_ERROR_UNNAMED)
    CASE_HRESULT(TLBX_E_AGNOST_SIGNATURE)
    CASE_HRESULT(TLBX_E_CONVERT_FAIL)
    CASE_HRESULT(TLBX_W_DUAL_NOT_DISPATCH)
    CASE_HRESULT(TLBX_E_BAD_SIGNATURE)
    CASE_HRESULT(TLBX_E_ARRAY_NEEDS_NT_FIXED)
    CASE_HRESULT(TLBX_E_CLASS_NEEDS_NT_INTF)
    CASE_HRESULT(TLBX_E_INVALID_TYPEINFO)
    CASE_HRESULT(TLBX_E_INVALID_TYPEINFO_UNNAMED)
    CASE_HRESULT(TLBX_E_CTX_NESTED)
    CASE_HRESULT(TLBX_E_ERROR_MESSAGE)
    CASE_HRESULT(TLBX_E_CANT_SAVE)
    CASE_HRESULT(TLBX_W_LIBNOTREGISTERED)
    CASE_HRESULT(TLBX_E_CANTLOADLIBRARY)
    CASE_HRESULT(TLBX_E_BAD_VT_TYPE)
    CASE_HRESULT(TLBX_E_NO_MSCOREE_TLB)
    CASE_HRESULT(TLBX_E_BAD_MSCOREE_TLB)
    CASE_HRESULT(TLBX_E_TLB_EXCEPTION)
    CASE_HRESULT(TLBX_E_MULTIPLE_LCIDS)
    CASE_HRESULT(TLBX_I_TYPEINFO_IMPORTED)
    CASE_HRESULT(TLBX_E_AMBIGUOUS_RETURN)
    CASE_HRESULT(TLBX_E_PROPGET_WITHOUT_RETURN)
    CASE_HRESULT(TLBX_E_DUPLICATE_TYPE_NAME)
    CASE_HRESULT(TLBX_I_USEIUNKNOWN)
    CASE_HRESULT(TLBX_I_UNCONVERTABLE_ARGS)
    CASE_HRESULT(TLBX_I_UNCONVERTABLE_FIELD)
    CASE_HRESULT(TLBX_I_NONSEQUENTIALSTRUCT)
    CASE_HRESULT(TLBX_W_WARNING_MESSAGE)
    CASE_HRESULT(TLBX_I_RESOLVEREFFAILED)
    CASE_HRESULT(TLBX_E_ASANY)
    CASE_HRESULT(TLBX_E_INVALIDLCIDPARAM)
    CASE_HRESULT(TLBX_E_LCIDONDISPONLYITF)
    CASE_HRESULT(TLBX_E_NONPUBLIC_FIELD)
    CASE_HRESULT(TLBX_I_TYPE_EXPORTED)
    CASE_HRESULT(TLBX_I_DUPLICATE_DISPID)
    CASE_HRESULT(TLBX_E_BAD_NAMES)
    CASE_HRESULT(TLBX_E_BITNESS_MISMATCH)
    CASE_HRESULT(TLBX_I_REF_TYPE_AS_STRUCT)
    CASE_HRESULT(TLBX_E_GENERICINST_SIGNATURE)
    CASE_HRESULT(TLBX_E_GENERICPAR_SIGNATURE)
    CASE_HRESULT(TLBX_I_GENERIC_TYPE)
    CASE_HRESULT(TLBX_W_EXPORTING_AUTO_LAYOUT)
    CASE_HRESULT(TLBX_E_TYPED_REF)
    CASE_HRESULT(TLBX_W_DEFAULT_INTF_NOT_VISIBLE)
    CASE_HRESULT(TLBX_W_NON_INTEGRAL_CA_TYPE)
    CASE_HRESULT(TLBX_W_IENUM_CA_ON_IUNK)
    CASE_HRESULT(TLBX_E_NO_SAFEHANDLE_ARRAYS)
    CASE_HRESULT(TLBX_W_NO_PROPS_IN_EVENTS)
    CASE_HRESULT(META_E_CA_INVALID_TARGET)
    CASE_HRESULT(META_E_CA_INVALID_VALUE)
    CASE_HRESULT(META_E_CA_INVALID_BLOB)
    CASE_HRESULT(META_E_CA_REPEATED_ARG)
    CASE_HRESULT(META_E_CA_UNKNOWN_ARGUMENT)
    CASE_HRESULT(META_E_CA_VARIANT_NYI)
    CASE_HRESULT(META_E_CA_ARRAY_NYI)
    CASE_HRESULT(META_E_CA_UNEXPECTED_TYPE)
    CASE_HRESULT(META_E_CA_INVALID_ARGTYPE)
    CASE_HRESULT(META_E_CA_INVALID_ARG_FOR_TYPE)
    CASE_HRESULT(META_E_CA_INVALID_UUID)
    CASE_HRESULT(META_E_CA_INVALID_MARSHALAS_FIELDS)
    CASE_HRESULT(META_E_CA_NT_FIELDONLY)
    CASE_HRESULT(META_E_CA_NEGATIVE_PARAMINDEX)
    CASE_HRESULT(META_E_CA_NEGATIVE_MULTIPLIER)
    CASE_HRESULT(META_E_CA_NEGATIVE_CONSTSIZE)
    CASE_HRESULT(META_E_CA_FIXEDSTR_SIZE_REQUIRED)
    CASE_HRESULT(META_E_CA_CUSTMARSH_TYPE_REQUIRED)
    CASE_HRESULT(META_E_CA_FILENAME_REQUIRED)
    CASE_HRESULT(META_E_CA_BAD_FRIENDS_ARGS)
    CASE_HRESULT(VLDTR_S_WRN)
    CASE_HRESULT(VLDTR_S_ERR)
    CASE_HRESULT(VLDTR_S_WRNERR)
    CASE_HRESULT(VLDTR_E_RID_OUTOFRANGE)
    CASE_HRESULT(VLDTR_E_CDTKN_OUTOFRANGE)
    CASE_HRESULT(VLDTR_E_CDRID_OUTOFRANGE)
    CASE_HRESULT(VLDTR_E_STRING_INVALID)
    CASE_HRESULT(VLDTR_E_GUID_INVALID)
    CASE_HRESULT(VLDTR_E_BLOB_INVALID)
    CASE_HRESULT(VLDTR_E_MOD_MULTI)
    CASE_HRESULT(VLDTR_E_MOD_NULLMVID)
    CASE_HRESULT(VLDTR_E_TR_NAMENULL)
    CASE_HRESULT(VLDTR_E_TR_DUP)
    CASE_HRESULT(VLDTR_E_TD_NAMENULL)
    CASE_HRESULT(VLDTR_E_TD_DUPNAME)
    CASE_HRESULT(VLDTR_E_TD_DUPGUID)
    CASE_HRESULT(VLDTR_E_TD_NOTIFACEOBJEXTNULL)
    CASE_HRESULT(VLDTR_E_TD_OBJEXTENDSNONNULL)
    CASE_HRESULT(VLDTR_E_TD_EXTENDSSEALED)
    CASE_HRESULT(VLDTR_E_TD_DLTNORTSPCL)
    CASE_HRESULT(VLDTR_E_TD_RTSPCLNOTDLT)
    CASE_HRESULT(VLDTR_E_MI_DECLPRIV)
    CASE_HRESULT(VLDTR_E_AS_BADNAME)
    CASE_HRESULT(VLDTR_E_FILE_SYSNAME)
    CASE_HRESULT(VLDTR_E_MI_BODYSTATIC)
    CASE_HRESULT(VLDTR_E_TD_IFACENOTABS)
    CASE_HRESULT(VLDTR_E_TD_IFACEPARNOTNIL)
    CASE_HRESULT(VLDTR_E_TD_IFACEGUIDNULL)
    CASE_HRESULT(VLDTR_E_MI_DECLFINAL)
    CASE_HRESULT(VLDTR_E_TD_VTNOTSEAL)
    CASE_HRESULT(VLDTR_E_PD_BADFLAGS)
    CASE_HRESULT(VLDTR_E_IFACE_DUP)
    CASE_HRESULT(VLDTR_E_MR_NAMENULL)
    CASE_HRESULT(VLDTR_E_MR_VTBLNAME)
    CASE_HRESULT(VLDTR_E_MR_DELNAME)
    CASE_HRESULT(VLDTR_E_MR_PARNIL)
    CASE_HRESULT(VLDTR_E_MR_BADCALLINGCONV)
    CASE_HRESULT(VLDTR_E_MR_NOTVARARG)
    CASE_HRESULT(VLDTR_E_MR_NAMEDIFF)
    CASE_HRESULT(VLDTR_E_MR_SIGDIFF)
    CASE_HRESULT(VLDTR_E_MR_DUP)
    CASE_HRESULT(VLDTR_E_CL_TDAUTO)
    CASE_HRESULT(VLDTR_E_CL_BADPCKSZ)
    CASE_HRESULT(VLDTR_E_CL_DUP)
    CASE_HRESULT(VLDTR_E_FL_BADOFFSET)
    CASE_HRESULT(VLDTR_E_FL_TDNIL)
    CASE_HRESULT(VLDTR_E_FL_NOCL)
    CASE_HRESULT(VLDTR_E_FL_TDNOTEXPLCT)
    CASE_HRESULT(VLDTR_E_FL_FLDSTATIC)
    CASE_HRESULT(VLDTR_E_FL_DUP)
    CASE_HRESULT(VLDTR_E_MODREF_NAMENULL)
    CASE_HRESULT(VLDTR_E_MODREF_DUP)
    CASE_HRESULT(VLDTR_E_TR_BADSCOPE)
    CASE_HRESULT(VLDTR_E_TD_NESTEDNOENCL)
    CASE_HRESULT(VLDTR_E_TD_EXTTRRES)
    CASE_HRESULT(VLDTR_E_SIGNULL)
    CASE_HRESULT(VLDTR_E_SIGNODATA)
    CASE_HRESULT(VLDTR_E_MD_BADCALLINGCONV)
    CASE_HRESULT(VLDTR_E_MD_THISSTATIC)
    CASE_HRESULT(VLDTR_E_MD_NOTTHISNOTSTATIC)
    CASE_HRESULT(VLDTR_E_MD_NOARGCNT)
    CASE_HRESULT(VLDTR_E_SIG_MISSELTYPE)
    CASE_HRESULT(VLDTR_E_SIG_MISSTKN)
    CASE_HRESULT(VLDTR_E_SIG_TKNBAD)
    CASE_HRESULT(VLDTR_E_SIG_MISSFPTR)
    CASE_HRESULT(VLDTR_E_SIG_MISSFPTRARGCNT)
    CASE_HRESULT(VLDTR_E_SIG_MISSRANK)
    CASE_HRESULT(VLDTR_E_SIG_MISSNSIZE)
    CASE_HRESULT(VLDTR_E_SIG_MISSSIZE)
    CASE_HRESULT(VLDTR_E_SIG_MISSNLBND)
    CASE_HRESULT(VLDTR_E_SIG_MISSLBND)
    CASE_HRESULT(VLDTR_E_SIG_BADELTYPE)
    CASE_HRESULT(VLDTR_E_FD_BADCALLINGCONV)
    CASE_HRESULT(VLDTR_E_MD_NAMENULL)
    CASE_HRESULT(VLDTR_E_MD_PARNIL)
    CASE_HRESULT(VLDTR_E_MD_DUP)
    CASE_HRESULT(VLDTR_E_FD_NAMENULL)
    CASE_HRESULT(VLDTR_E_FD_PARNIL)
    CASE_HRESULT(VLDTR_E_FD_DUP)
    CASE_HRESULT(VLDTR_E_AS_MULTI)
    CASE_HRESULT(VLDTR_E_AS_NAMENULL)
    CASE_HRESULT(VLDTR_E_SIG_TOKTYPEMISMATCH)
    CASE_HRESULT(VLDTR_E_CL_TDINTF)
    CASE_HRESULT(VLDTR_E_ASOS_OSPLTFRMIDINVAL)
    CASE_HRESULT(VLDTR_E_AR_NAMENULL)
    CASE_HRESULT(VLDTR_E_TD_ENCLNOTNESTED)
    CASE_HRESULT(VLDTR_E_AROS_OSPLTFRMIDINVAL)
    CASE_HRESULT(VLDTR_E_FILE_NAMENULL)
    CASE_HRESULT(VLDTR_E_CT_NAMENULL)
    CASE_HRESULT(VLDTR_E_TD_EXTENDSCHILD)
    CASE_HRESULT(VLDTR_E_MAR_NAMENULL)
    CASE_HRESULT(VLDTR_E_FILE_DUP)
    CASE_HRESULT(VLDTR_E_FILE_NAMEFULLQLFD)
    CASE_HRESULT(VLDTR_E_CT_DUP)
    CASE_HRESULT(VLDTR_E_MAR_DUP)
    CASE_HRESULT(VLDTR_E_MAR_NOTPUBPRIV)
    CASE_HRESULT(VLDTR_E_TD_ENUMNOVALUE)
    CASE_HRESULT(VLDTR_E_TD_ENUMVALSTATIC)
    CASE_HRESULT(VLDTR_E_TD_ENUMVALNOTSN)
    CASE_HRESULT(VLDTR_E_TD_ENUMFLDNOTST)
    CASE_HRESULT(VLDTR_E_TD_ENUMFLDNOTLIT)
    CASE_HRESULT(VLDTR_E_TD_ENUMNOLITFLDS)
    CASE_HRESULT(VLDTR_E_TD_ENUMFLDSIGMISMATCH)
    CASE_HRESULT(VLDTR_E_TD_ENUMVALNOT1ST)
    CASE_HRESULT(VLDTR_E_FD_NOTVALUERTSN)
    CASE_HRESULT(VLDTR_E_FD_VALUEPARNOTENUM)
    CASE_HRESULT(VLDTR_E_FD_INSTINIFACE)
    CASE_HRESULT(VLDTR_E_FD_NOTPUBINIFACE)
    CASE_HRESULT(VLDTR_E_FMD_GLOBALNOTPUBPRIVSC)
    CASE_HRESULT(VLDTR_E_FMD_GLOBALNOTSTATIC)
    CASE_HRESULT(VLDTR_E_FD_GLOBALNORVA)
    CASE_HRESULT(VLDTR_E_MD_CTORZERORVA)
    CASE_HRESULT(VLDTR_E_FD_MARKEDNOMARSHAL)
    CASE_HRESULT(VLDTR_E_FD_MARSHALNOTMARKED)
    CASE_HRESULT(VLDTR_E_FD_MARKEDNODEFLT)
    CASE_HRESULT(VLDTR_E_FD_DEFLTNOTMARKED)
    CASE_HRESULT(VLDTR_E_FMD_MARKEDNOSECUR)
    CASE_HRESULT(VLDTR_E_FMD_SECURNOTMARKED)
    CASE_HRESULT(VLDTR_E_FMD_PINVOKENOTSTATIC)
    CASE_HRESULT(VLDTR_E_FMD_MARKEDNOPINVOKE)
    CASE_HRESULT(VLDTR_E_FMD_PINVOKENOTMARKED)
    CASE_HRESULT(VLDTR_E_FMD_BADIMPLMAP)
    CASE_HRESULT(VLDTR_E_IMAP_BADMODREF)
    CASE_HRESULT(VLDTR_E_IMAP_BADMEMBER)
    CASE_HRESULT(VLDTR_E_IMAP_BADIMPORTNAME)
    CASE_HRESULT(VLDTR_E_IMAP_BADCALLCONV)
    CASE_HRESULT(VLDTR_E_FMD_BADACCESSFLAG)
    CASE_HRESULT(VLDTR_E_FD_INITONLYANDLITERAL)
    CASE_HRESULT(VLDTR_E_FD_LITERALNOTSTATIC)
    CASE_HRESULT(VLDTR_E_FMD_RTSNNOTSN)
    CASE_HRESULT(VLDTR_E_MD_ABSTPARNOTABST)
    CASE_HRESULT(VLDTR_E_MD_NOTSTATABSTININTF)
    CASE_HRESULT(VLDTR_E_MD_NOTPUBININTF)
    CASE_HRESULT(VLDTR_E_MD_CTORININTF)
    CASE_HRESULT(VLDTR_E_MD_GLOBALCTORCCTOR)
    CASE_HRESULT(VLDTR_E_MD_CTORSTATIC)
    CASE_HRESULT(VLDTR_E_MD_CTORNOTSNRTSN)
    CASE_HRESULT(VLDTR_E_MD_CTORVIRT)
    CASE_HRESULT(VLDTR_E_MD_CTORABST)
    CASE_HRESULT(VLDTR_E_MD_CCTORNOTSTATIC)
    CASE_HRESULT(VLDTR_E_MD_ZERORVA)
    CASE_HRESULT(VLDTR_E_MD_FINNOTVIRT)
    CASE_HRESULT(VLDTR_E_MD_STATANDFINORVIRT)
    CASE_HRESULT(VLDTR_E_MD_ABSTANDFINAL)
    CASE_HRESULT(VLDTR_E_MD_ABSTANDIMPL)
    CASE_HRESULT(VLDTR_E_MD_ABSTANDPINVOKE)
    CASE_HRESULT(VLDTR_E_MD_ABSTNOTVIRT)
    CASE_HRESULT(VLDTR_E_MD_NOTABSTNOTIMPL)
    CASE_HRESULT(VLDTR_E_MD_NOTABSTBADFLAGSRVA)
    CASE_HRESULT(VLDTR_E_MD_PRIVSCOPENORVA)
    CASE_HRESULT(VLDTR_E_MD_GLOBALABSTORVIRT)
    CASE_HRESULT(VLDTR_E_SIG_LONGFORM)
    CASE_HRESULT(VLDTR_E_MD_MULTIPLESEMANTICS)
    CASE_HRESULT(VLDTR_E_MD_INVALIDSEMANTICS)
    CASE_HRESULT(VLDTR_E_MD_SEMANTICSNOTEXIST)
    CASE_HRESULT(VLDTR_E_MI_DECLNOTVIRT)
    CASE_HRESULT(VLDTR_E_FMD_GLOBALITEM)
    CASE_HRESULT(VLDTR_E_MD_MULTSEMANTICFLAGS)
    CASE_HRESULT(VLDTR_E_MD_NOSEMANTICFLAGS)
    CASE_HRESULT(VLDTR_E_FD_FLDINIFACE)
    CASE_HRESULT(VLDTR_E_AS_HASHALGID)
    CASE_HRESULT(VLDTR_E_AS_PROCID)
    CASE_HRESULT(VLDTR_E_AR_PROCID)
    CASE_HRESULT(VLDTR_E_CN_PARENTRANGE)
    CASE_HRESULT(VLDTR_E_AS_BADFLAGS)
    CASE_HRESULT(VLDTR_E_TR_HASTYPEDEF)
    CASE_HRESULT(VLDTR_E_IFACE_BADIMPL)
    CASE_HRESULT(VLDTR_E_IFACE_BADIFACE)
    CASE_HRESULT(VLDTR_E_TD_SECURNOTMARKED)
    CASE_HRESULT(VLDTR_E_TD_MARKEDNOSECUR)
    CASE_HRESULT(VLDTR_E_MD_CCTORHASARGS)
    CASE_HRESULT(VLDTR_E_CT_BADIMPL)
    CASE_HRESULT(VLDTR_E_MI_ALIENBODY)
    CASE_HRESULT(VLDTR_E_MD_CCTORCALLCONV)
    CASE_HRESULT(VLDTR_E_MI_BADCLASS)
    CASE_HRESULT(VLDTR_E_MI_CLASSISINTF)
    CASE_HRESULT(VLDTR_E_MI_BADDECL)
    CASE_HRESULT(VLDTR_E_MI_BADBODY)
    CASE_HRESULT(VLDTR_E_MI_DUP)
    CASE_HRESULT(VLDTR_E_FD_BADPARENT)
    CASE_HRESULT(VLDTR_E_MD_PARAMOUTOFSEQ)
    CASE_HRESULT(VLDTR_E_MD_PARASEQTOOBIG)
    CASE_HRESULT(VLDTR_E_MD_PARMMARKEDNOMARSHAL)
    CASE_HRESULT(VLDTR_E_MD_PARMMARSHALNOTMARKED)
    CASE_HRESULT(VLDTR_E_MD_PARMMARKEDNODEFLT)
    CASE_HRESULT(VLDTR_E_MD_PARMDEFLTNOTMARKED)
    CASE_HRESULT(VLDTR_E_PR_BADSCOPE)
    CASE_HRESULT(VLDTR_E_PR_NONAME)
    CASE_HRESULT(VLDTR_E_PR_NOSIG)
    CASE_HRESULT(VLDTR_E_PR_DUP)
    CASE_HRESULT(VLDTR_E_PR_BADCALLINGCONV)
    CASE_HRESULT(VLDTR_E_PR_MARKEDNODEFLT)
    CASE_HRESULT(VLDTR_E_PR_DEFLTNOTMARKED)
    CASE_HRESULT(VLDTR_E_PR_BADSEMANTICS)
    CASE_HRESULT(VLDTR_E_PR_BADMETHOD)
    CASE_HRESULT(VLDTR_E_PR_ALIENMETHOD)
    CASE_HRESULT(VLDTR_E_CN_BLOBNOTNULL)
    CASE_HRESULT(VLDTR_E_CN_BLOBNULL)
    CASE_HRESULT(VLDTR_E_EV_BADSCOPE)
    CASE_HRESULT(VLDTR_E_EV_NONAME)
    CASE_HRESULT(VLDTR_E_EV_DUP)
    CASE_HRESULT(VLDTR_E_EV_BADEVTYPE)
    CASE_HRESULT(VLDTR_E_EV_EVTYPENOTCLASS)
    CASE_HRESULT(VLDTR_E_EV_BADSEMANTICS)
    CASE_HRESULT(VLDTR_E_EV_BADMETHOD)
    CASE_HRESULT(VLDTR_E_EV_ALIENMETHOD)
    CASE_HRESULT(VLDTR_E_EV_NOADDON)
    CASE_HRESULT(VLDTR_E_EV_NOREMOVEON)
    CASE_HRESULT(VLDTR_E_CT_DUPTDNAME)
    CASE_HRESULT(VLDTR_E_MAR_BADOFFSET)
    CASE_HRESULT(VLDTR_E_DS_BADOWNER)
    CASE_HRESULT(VLDTR_E_DS_BADFLAGS)
    CASE_HRESULT(VLDTR_E_DS_NOBLOB)
    CASE_HRESULT(VLDTR_E_MAR_BADIMPL)
    CASE_HRESULT(VLDTR_E_MR_VARARGCALLINGCONV)
    CASE_HRESULT(VLDTR_E_MD_CTORNOTVOID)
    CASE_HRESULT(VLDTR_E_EV_FIRENOTVOID)
    CASE_HRESULT(VLDTR_E_AS_BADLOCALE)
    CASE_HRESULT(VLDTR_E_CN_PARENTTYPE)
    CASE_HRESULT(VLDTR_E_SIG_SENTINMETHODDEF)
    CASE_HRESULT(VLDTR_E_SIG_SENTMUSTVARARG)
    CASE_HRESULT(VLDTR_E_SIG_MULTSENTINELS)
    CASE_HRESULT(VLDTR_E_SIG_LASTSENTINEL)
    CASE_HRESULT(VLDTR_E_SIG_MISSARG)
    CASE_HRESULT(VLDTR_E_SIG_BYREFINFIELD)
    CASE_HRESULT(VLDTR_E_MD_SYNCMETHODINVTYPE)
    CASE_HRESULT(VLDTR_E_TD_NAMETOOLONG)
    CASE_HRESULT(VLDTR_E_AS_PROCDUP)
    CASE_HRESULT(VLDTR_E_ASOS_DUP)
    CASE_HRESULT(VLDTR_E_MAR_BADFLAGS)
    CASE_HRESULT(VLDTR_E_CT_NOTYPEDEFID)
    CASE_HRESULT(VLDTR_E_FILE_BADFLAGS)
    CASE_HRESULT(VLDTR_E_FILE_NULLHASH)
    CASE_HRESULT(VLDTR_E_MOD_NONAME)
    CASE_HRESULT(VLDTR_E_MOD_NAMEFULLQLFD)
    CASE_HRESULT(VLDTR_E_TD_RTSPCLNOTSPCL)
    CASE_HRESULT(VLDTR_E_TD_EXTENDSIFACE)
    CASE_HRESULT(VLDTR_E_MD_CTORPINVOKE)
    CASE_HRESULT(VLDTR_E_TD_SYSENUMNOTCLASS)
    CASE_HRESULT(VLDTR_E_TD_SYSENUMNOTEXTVTYPE)
    CASE_HRESULT(VLDTR_E_MI_SIGMISMATCH)
    CASE_HRESULT(VLDTR_E_TD_ENUMHASMETHODS)
    CASE_HRESULT(VLDTR_E_TD_ENUMIMPLIFACE)
    CASE_HRESULT(VLDTR_E_TD_ENUMHASPROP)
    CASE_HRESULT(VLDTR_E_TD_ENUMHASEVENT)
    CASE_HRESULT(VLDTR_E_TD_BADMETHODLST)
    CASE_HRESULT(VLDTR_E_TD_BADFIELDLST)
    CASE_HRESULT(VLDTR_E_CN_BADTYPE)
    CASE_HRESULT(VLDTR_E_TD_ENUMNOINSTFLD)
    CASE_HRESULT(VLDTR_E_TD_ENUMMULINSTFLD)
    CASE_HRESULT(VLDTR_E_INTERRUPTED)
    CASE_HRESULT(VLDTR_E_NOTINIT)
    CASE_HRESULT(VLDTR_E_IFACE_NOTIFACE)
    CASE_HRESULT(VLDTR_E_FD_RVAHASNORVA)
    CASE_HRESULT(VLDTR_E_FD_RVAHASZERORVA)
    CASE_HRESULT(VLDTR_E_MD_RVAANDIMPLMAP)
    CASE_HRESULT(VLDTR_E_TD_EXTRAFLAGS)
    CASE_HRESULT(VLDTR_E_TD_EXTENDSITSELF)
    CASE_HRESULT(VLDTR_E_TD_SYSVTNOTEXTOBJ)
    CASE_HRESULT(VLDTR_E_TD_EXTTYPESPEC)
    CASE_HRESULT(VLDTR_E_TD_VTNOSIZE)
    CASE_HRESULT(VLDTR_E_TD_IFACESEALED)
    CASE_HRESULT(VLDTR_E_NC_BADNESTED)
    CASE_HRESULT(VLDTR_E_NC_BADENCLOSER)
    CASE_HRESULT(VLDTR_E_NC_DUP)
    CASE_HRESULT(VLDTR_E_NC_DUPENCLOSER)
    CASE_HRESULT(VLDTR_E_FRVA_ZERORVA)
    CASE_HRESULT(VLDTR_E_FRVA_BADFIELD)
    CASE_HRESULT(VLDTR_E_FRVA_DUPRVA)
    CASE_HRESULT(VLDTR_E_FRVA_DUPFIELD)
    CASE_HRESULT(VLDTR_E_EP_BADTOKEN)
    CASE_HRESULT(VLDTR_E_EP_INSTANCE)
    CASE_HRESULT(VLDTR_E_TD_ENUMFLDBADTYPE)
    CASE_HRESULT(VLDTR_E_MD_BADRVA)
    CASE_HRESULT(VLDTR_E_FD_LITERALNODEFAULT)
    CASE_HRESULT(VLDTR_E_IFACE_METHNOTIMPL)
    CASE_HRESULT(VLDTR_E_CA_BADPARENT)
    CASE_HRESULT(VLDTR_E_CA_BADTYPE)
    CASE_HRESULT(VLDTR_E_CA_NOTCTOR)
    CASE_HRESULT(VLDTR_E_CA_BADSIG)
    CASE_HRESULT(VLDTR_E_CA_NOSIG)
    CASE_HRESULT(VLDTR_E_CA_BADPROLOG)
    CASE_HRESULT(VLDTR_E_MD_BADLOCALSIGTOK)
    CASE_HRESULT(VLDTR_E_MD_BADHEADER)
    CASE_HRESULT(VLDTR_E_EP_TOOMANYARGS)
    CASE_HRESULT(VLDTR_E_EP_BADRET)
    CASE_HRESULT(VLDTR_E_EP_BADARG)
    CASE_HRESULT(VLDTR_E_SIG_BADVOID)
    CASE_HRESULT(CORDBG_E_UNRECOVERABLE_ERROR)
    CASE_HRESULT(CORDBG_E_PROCESS_TERMINATED)
    CASE_HRESULT(CORDBG_E_PROCESS_NOT_SYNCHRONIZED)
    CASE_HRESULT(CORDBG_E_CLASS_NOT_LOADED)
    CASE_HRESULT(CORDBG_E_IL_VAR_NOT_AVAILABLE)
    CASE_HRESULT(CORDBG_E_BAD_REFERENCE_VALUE)
    CASE_HRESULT(CORDBG_E_FIELD_NOT_AVAILABLE)
    CASE_HRESULT(CORDBG_E_NON_NATIVE_FRAME)
    CASE_HRESULT(CORDBG_E_NONCONTINUABLE_EXCEPTION)
    CASE_HRESULT(CORDBG_E_CODE_NOT_AVAILABLE)
    CASE_HRESULT(CORDBG_E_FUNCTION_NOT_IL)
    CASE_HRESULT(CORDBG_S_BAD_START_SEQUENCE_POINT)
    CASE_HRESULT(CORDBG_S_BAD_END_SEQUENCE_POINT)
    CASE_HRESULT(CORDBG_S_INSUFFICIENT_INFO_FOR_SET_IP)
    CASE_HRESULT(CORDBG_E_CANT_SET_IP_INTO_FINALLY)
    CASE_HRESULT(CORDBG_E_CANT_SET_IP_OUT_OF_FINALLY)
    CASE_HRESULT(CORDBG_E_CANT_SET_IP_INTO_CATCH)
    CASE_HRESULT(CORDBG_E_SET_IP_NOT_ALLOWED_ON_NONLEAF_FRAME)
    CASE_HRESULT(CORDBG_E_SET_IP_IMPOSSIBLE)
    CASE_HRESULT(CORDBG_E_FUNC_EVAL_BAD_START_POINT)
    CASE_HRESULT(CORDBG_E_INVALID_OBJECT)
    CASE_HRESULT(CORDBG_E_FUNC_EVAL_NOT_COMPLETE)
    CASE_HRESULT(CORDBG_S_FUNC_EVAL_HAS_NO_RESULT)
    CASE_HRESULT(CORDBG_S_VALUE_POINTS_TO_VOID)
    CASE_HRESULT(CORDBG_E_INPROC_NOT_IMPL)
    CASE_HRESULT(CORDBG_S_FUNC_EVAL_ABORTED)
    CASE_HRESULT(CORDBG_E_STATIC_VAR_NOT_AVAILABLE)
    CASE_HRESULT(CORDBG_E_OBJECT_IS_NOT_COPYABLE_VALUE_CLASS)
    CASE_HRESULT(CORDBG_E_CANT_SETIP_INTO_OR_OUT_OF_FILTER)
    CASE_HRESULT(CORDBG_E_CANT_CHANGE_JIT_SETTING_FOR_ZAP_MODULE)
    CASE_HRESULT(CORDBG_E_REMOTE_CONNECTION_CONN_RESET)
    CASE_HRESULT(CORDBG_E_REMOTE_CONNECTION_KEEP_ALIVE)
    CASE_HRESULT(CORDBG_E_REMOTE_CONNECTION_FATAL_ERROR)
    CASE_HRESULT(CORDBG_E_CANT_SET_TO_JMC)
    CASE_HRESULT(CORDBG_E_BAD_THREAD_STATE)
    CASE_HRESULT(CORDBG_E_DEBUGGER_ALREADY_ATTACHED)
    CASE_HRESULT(CORDBG_E_SUPERFLOUS_CONTINUE)
    CASE_HRESULT(CORDBG_E_SET_VALUE_NOT_ALLOWED_ON_NONLEAF_FRAME)
    CASE_HRESULT(CORDBG_E_ENC_EH_MAX_NESTING_LEVEL_CANT_INCREASE)
    CASE_HRESULT(CORDBG_E_ENC_MODULE_NOT_ENC_ENABLED)
    CASE_HRESULT(CORDBG_E_SET_IP_NOT_ALLOWED_ON_EXCEPTION)
    CASE_HRESULT(CORDBG_E_VARIABLE_IS_ACTUALLY_LITERAL)
    CASE_HRESULT(CORDBG_E_PROCESS_DETACHED)
    CASE_HRESULT(CORDBG_E_ENC_METHOD_SIG_CHANGED)
    CASE_HRESULT(CORDBG_E_ENC_METHOD_NO_LOCAL_SIG)
    CASE_HRESULT(CORDBG_E_ENC_CANT_ADD_FIELD_TO_VALUE_OR_LAYOUT_CLASS)
    CASE_HRESULT(CORDBG_E_ENC_CANT_CHANGE_FIELD)
    CASE_HRESULT(CORDBG_E_ENC_CANT_ADD_NON_PRIVATE_MEMBER)
    CASE_HRESULT(CORDBG_E_FIELD_NOT_STATIC)
    CASE_HRESULT(CORDBG_E_FIELD_NOT_INSTANCE)
    CASE_HRESULT(CORDBG_E_ENC_ZAPPED_WITHOUT_ENC)
    CASE_HRESULT(CORDBG_E_ENC_BAD_METHOD_INFO)
    CASE_HRESULT(CORDBG_E_ENC_JIT_CANT_UPDATE)
    CASE_HRESULT(CORDBG_E_ENC_MISSING_CLASS)
    CASE_HRESULT(CORDBG_E_ENC_INTERNAL_ERROR)
    CASE_HRESULT(CORDBG_E_ENC_HANGING_FIELD)
    CASE_HRESULT(CORDBG_E_MODULE_NOT_LOADED)
    CASE_HRESULT(CORDBG_E_ENC_CANT_CHANGE_SUPERCLASS)
    CASE_HRESULT(CORDBG_E_UNABLE_TO_SET_BREAKPOINT)
    CASE_HRESULT(CORDBG_E_DEBUGGING_NOT_POSSIBLE)
    CASE_HRESULT(CORDBG_E_KERNEL_DEBUGGER_ENABLED)
    CASE_HRESULT(CORDBG_E_KERNEL_DEBUGGER_PRESENT)
    CASE_HRESULT(CORDBG_E_HELPER_THREAD_DEAD)
    CASE_HRESULT(CORDBG_E_INTERFACE_INHERITANCE_CANT_CHANGE)
    CASE_HRESULT(CORDBG_E_INCOMPATIBLE_PROTOCOL)
    CASE_HRESULT(CORDBG_E_TOO_MANY_PROCESSES)
    CASE_HRESULT(CORDBG_E_INTEROP_NOT_SUPPORTED)
    CASE_HRESULT(CORDBG_E_NO_REMAP_BREAKPIONT)
    CASE_HRESULT(CORDBG_E_OBJECT_NEUTERED)
    CASE_HRESULT(CORPROF_E_FUNCTION_NOT_COMPILED)
    CASE_HRESULT(CORPROF_E_DATAINCOMPLETE)
    CASE_HRESULT(CORPROF_E_NOT_REJITABLE_METHODS)
    CASE_HRESULT(CORPROF_E_CANNOT_UPDATE_METHOD)
    CASE_HRESULT(CORPROF_E_FUNCTION_NOT_IL)
    CASE_HRESULT(CORPROF_E_NOT_MANAGED_THREAD)
    CASE_HRESULT(CORPROF_E_CALL_ONLY_FROM_INIT)
    CASE_HRESULT(CORPROF_E_INPROC_NOT_ENABLED)
    CASE_HRESULT(CORPROF_E_JITMAPS_NOT_ENABLED)
    CASE_HRESULT(CORPROF_E_INPROC_ALREADY_BEGUN)
    CASE_HRESULT(CORPROF_E_INPROC_NOT_AVAILABLE)
    CASE_HRESULT(CORPROF_E_NOT_YET_AVAILABLE)
    CASE_HRESULT(SECURITY_E_XML_TO_ASN_ENCODING)
    CASE_HRESULT(SECURITY_E_INCOMPATIBLE_SHARE)
    CASE_HRESULT(SECURITY_E_UNVERIFIABLE)
    CASE_HRESULT(SECURITY_E_INCOMPATIBLE_EVIDENCE)
    CASE_HRESULT(CLDB_E_INTERNALERROR)
    CASE_HRESULT(CORSEC_E_DECODE_SET)
    CASE_HRESULT(CORSEC_E_ENCODE_SET)
    CASE_HRESULT(CORSEC_E_UNSUPPORTED_FORMAT)
    CASE_HRESULT(SN_CRYPTOAPI_CALL_FAILED)
    CASE_HRESULT(SN_NO_SUITABLE_CSP)
    CASE_HRESULT(CORSEC_E_INVALID_ATTR)
    CASE_HRESULT(CORSEC_E_POLICY_EXCEPTION)
    CASE_HRESULT(CORSEC_E_MIN_GRANT_FAIL)
    CASE_HRESULT(CORSEC_E_NO_EXEC_PERM)
    CASE_HRESULT(CORSEC_E_XMLSYNTAX)
    CASE_HRESULT(CORSEC_E_INVALID_STRONGNAME)
    CASE_HRESULT(CORSEC_E_MISSING_STRONGNAME)
    CASE_HRESULT(CORSEC_E_CONTAINER_NOT_FOUND)
    CASE_HRESULT(CORSEC_E_INVALID_IMAGE_FORMAT)
    CASE_HRESULT(CORSEC_E_CRYPTO)
    CASE_HRESULT(CORSEC_E_CRYPTO_UNEX_OPER)
    CASE_HRESULT(CORSECATTR_E_BAD_ATTRIBUTE)
    CASE_HRESULT(CORSECATTR_E_MISSING_CONSTRUCTOR)
    CASE_HRESULT(CORSECATTR_E_FAILED_TO_CREATE_PERM)
    CASE_HRESULT(CORSECATTR_E_BAD_ACTION_ASM)
    CASE_HRESULT(CORSECATTR_E_BAD_ACTION_OTHER)
    CASE_HRESULT(CORSECATTR_E_BAD_PARENT)
    CASE_HRESULT(CORSECATTR_E_TRUNCATED)
    CASE_HRESULT(CORSECATTR_E_BAD_VERSION)
    CASE_HRESULT(CORSECATTR_E_BAD_ACTION)
    CASE_HRESULT(CORSECATTR_E_NO_SELF_REF)
    CASE_HRESULT(CORSECATTR_E_BAD_NONCAS)
    CASE_HRESULT(CORSECATTR_E_ASSEMBLY_LOAD_FAILED)
    CASE_HRESULT(CORSECATTR_E_ASSEMBLY_LOAD_FAILED_EX)
    CASE_HRESULT(CORSECATTR_E_TYPE_LOAD_FAILED)
    CASE_HRESULT(CORSECATTR_E_TYPE_LOAD_FAILED_EX)
    CASE_HRESULT(CORSECATTR_E_ABSTRACT)
    CASE_HRESULT(CORSECATTR_E_UNSUPPORTED_TYPE)
    CASE_HRESULT(CORSECATTR_E_UNSUPPORTED_ENUM_TYPE)
    CASE_HRESULT(CORSECATTR_E_NO_FIELD)
    CASE_HRESULT(CORSECATTR_E_NO_PROPERTY)
    CASE_HRESULT(CORSECATTR_E_EXCEPTION)
    CASE_HRESULT(CORSECATTR_E_EXCEPTION_HR)
    CASE_HRESULT(ISS_E_ISOSTORE)
    CASE_HRESULT(ISS_E_OPEN_STORE_FILE)
    CASE_HRESULT(ISS_E_OPEN_FILE_MAPPING)
    CASE_HRESULT(ISS_E_MAP_VIEW_OF_FILE)
    CASE_HRESULT(ISS_E_GET_FILE_SIZE)
    CASE_HRESULT(ISS_E_CREATE_MUTEX)
    CASE_HRESULT(ISS_E_LOCK_FAILED)
    CASE_HRESULT(ISS_E_FILE_WRITE)
    CASE_HRESULT(ISS_E_SET_FILE_POINTER)
    CASE_HRESULT(ISS_E_CREATE_DIR)
    CASE_HRESULT(ISS_E_STORE_NOT_OPEN)
    CASE_HRESULT(ISS_E_CORRUPTED_STORE_FILE)
    CASE_HRESULT(ISS_E_STORE_VERSION)
    CASE_HRESULT(ISS_E_FILE_NOT_MAPPED)
    CASE_HRESULT(ISS_E_BLOCK_SIZE_TOO_SMALL)
    CASE_HRESULT(ISS_E_ALLOC_TOO_LARGE)
    CASE_HRESULT(ISS_E_USAGE_WILL_EXCEED_QUOTA)
    CASE_HRESULT(ISS_E_TABLE_ROW_NOT_FOUND)
    CASE_HRESULT(ISS_E_DEPRECATE)
    CASE_HRESULT(ISS_E_CALLER)
    CASE_HRESULT(ISS_E_PATH_LENGTH)
    CASE_HRESULT(ISS_E_MACHINE)
    CASE_HRESULT(ISS_E_MACHINE_DACL)
    CASE_HRESULT(ISS_E_ISOSTORE_END)
    CASE_HRESULT(COR_E_APPLICATION)
    CASE_HRESULT(COR_E_ARGUMENTOUTOFRANGE)
    CASE_HRESULT(COR_E_ARITHMETIC)
    CASE_HRESULT(COR_E_ARRAYTYPEMISMATCH)
    CASE_HRESULT(COR_E_CONTEXTMARSHAL)
    CASE_HRESULT(COR_E_TIMEOUT)
    CASE_HRESULT(COR_E_DEVICESNOTSUPPORTED)
    CASE_HRESULT(COR_E_DIVIDEBYZERO)
    CASE_HRESULT(COR_E_EXCEPTION)
    CASE_HRESULT(COR_E_EXECUTIONENGINE)
    CASE_HRESULT(COR_E_FIELDACCESS)
    CASE_HRESULT(COR_E_FORMAT)
    CASE_HRESULT(COR_E_BADIMAGEFORMAT)
    CASE_HRESULT(COR_E_ASSEMBLYEXPECTED)
    CASE_HRESULT(COR_E_TYPEUNLOADED)
    CASE_HRESULT(COR_E_INDEXOUTOFRANGE)
    CASE_HRESULT(COR_E_INVALIDOPERATION)
    CASE_HRESULT(COR_E_INVALIDPROGRAM)
    CASE_HRESULT(COR_E_MEMBERACCESS)
    CASE_HRESULT(COR_E_METHODACCESS)
    CASE_HRESULT(COR_E_MISSINGFIELD)
    CASE_HRESULT(COR_E_MISSINGMANIFESTRESOURCE)
    CASE_HRESULT(COR_E_MISSINGMEMBER)
    CASE_HRESULT(COR_E_MISSINGMETHOD)
    CASE_HRESULT(COR_E_MULTICASTNOTSUPPORTED)
    CASE_HRESULT(COR_E_NOTFINITENUMBER)
    CASE_HRESULT(COR_E_DUPLICATEWAITOBJECT)
    CASE_HRESULT(COR_E_PLATFORMNOTSUPPORTED)
    CASE_HRESULT(COR_E_NOTSUPPORTED)
    CASE_HRESULT(COR_E_OVERFLOW)
    CASE_HRESULT(COR_E_RANK)
    CASE_HRESULT(COR_E_REMOTING)
    CASE_HRESULT(COR_E_SERVER)
    CASE_HRESULT(COR_E_SERVICEDCOMPONENT)
    CASE_HRESULT(COR_E_SECURITY)
    CASE_HRESULT(COR_E_SERIALIZATION)
    CASE_HRESULT(COR_E_STACKOVERFLOW)
    CASE_HRESULT(COR_E_SYNCHRONIZATIONLOCK)
    CASE_HRESULT(COR_E_SYSTEM)
    CASE_HRESULT(COR_E_THREADABORTED)
    CASE_HRESULT(COR_E_THREADINTERRUPTED)
    CASE_HRESULT(COR_E_THREADSTATE)
    CASE_HRESULT(COR_E_THREADSTOP)
    CASE_HRESULT(COR_E_TYPEINITIALIZATION)
    CASE_HRESULT(COR_E_TYPELOAD)
    CASE_HRESULT(COR_E_ENTRYPOINTNOTFOUND)
    CASE_HRESULT(COR_E_DLLNOTFOUND)
    CASE_HRESULT(COR_E_VERIFICATION)
    CASE_HRESULT(COR_E_INVALIDCOMOBJECT)
    CASE_HRESULT(COR_E_MARSHALDIRECTIVE)
    CASE_HRESULT(COR_E_INVALIDOLEVARIANTTYPE)
    CASE_HRESULT(COR_E_SAFEARRAYTYPEMISMATCH)
    CASE_HRESULT(COR_E_SAFEARRAYRANKMISMATCH)
    CASE_HRESULT(COR_E_INVALIDFILTERCRITERIA)
    CASE_HRESULT(COR_E_REFLECTIONTYPELOAD)
    CASE_HRESULT(COR_E_TARGET)
    CASE_HRESULT(COR_E_TARGETINVOCATION)
    CASE_HRESULT(COR_E_CUSTOMATTRIBUTEFORMAT)
    CASE_HRESULT(COR_E_ENDOFSTREAM)
    CASE_HRESULT(COR_E_FILELOAD)
    CASE_HRESULT(COR_E_FILENOTFOUND)
    CASE_HRESULT(COR_E_BAD_PATHNAME)
    CASE_HRESULT(COR_E_IO)
    CASE_HRESULT(COR_E_DIRECTORYNOTFOUND)
    CASE_HRESULT(COR_E_PATHTOOLONG)
    CASE_HRESULT(COR_E_OBJECTDISPOSED)
    CASE_HRESULT(COR_E_NEWER_RUNTIME)
    CASE_HRESULT(CLR_E_SHIM_RUNTIMELOAD)
    CASE_HRESULT(CLR_E_SHIM_RUNTIMEEXPORT)
    CASE_HRESULT(CLR_E_SHIM_INSTALLROOT)
    CASE_HRESULT(CLR_E_SHIM_INSTALLCOMP)
    CASE_HRESULT(VER_E_HRESULT)
    CASE_HRESULT(VER_E_OFFSET)
    CASE_HRESULT(VER_E_OPCODE)
    CASE_HRESULT(VER_E_OPERAND)
    CASE_HRESULT(VER_E_TOKEN)
    CASE_HRESULT(VER_E_EXCEPT)
    CASE_HRESULT(VER_E_STACK_SLOT)
    CASE_HRESULT(VER_E_LOC)
    CASE_HRESULT(VER_E_ARG)
    CASE_HRESULT(VER_E_FOUND)
    CASE_HRESULT(VER_E_EXPECTED)
    CASE_HRESULT(VER_E_UNKNOWN_OPCODE)
    CASE_HRESULT(VER_E_SIG_CALLCONV)
    CASE_HRESULT(VER_E_SIG_ELEMTYPE)
    CASE_HRESULT(VER_E_RET_SIG)
    CASE_HRESULT(VER_E_FIELD_SIG)
    CASE_HRESULT(VER_E_INTERNAL)
    CASE_HRESULT(VER_E_STACK_TOO_LARGE)
    CASE_HRESULT(VER_E_ARRAY_NAME_LONG)
    CASE_HRESULT(VER_E_FALLTHRU)
    CASE_HRESULT(VER_E_TRY_GTEQ_END)
    CASE_HRESULT(VER_E_TRYEND_GT_CS)
    CASE_HRESULT(VER_E_HND_GTEQ_END)
    CASE_HRESULT(VER_E_HNDEND_GT_CS)
    CASE_HRESULT(VER_E_FLT_GTEQ_CS)
    CASE_HRESULT(VER_E_TRY_START)
    CASE_HRESULT(VER_E_HND_START)
    CASE_HRESULT(VER_E_FLT_START)
    CASE_HRESULT(VER_E_TRY_OVERLAP)
    CASE_HRESULT(VER_E_TRY_EQ_HND_FIL)
    CASE_HRESULT(VER_E_TRY_SHARE_FIN_FAL)
    CASE_HRESULT(VER_E_HND_OVERLAP)
    CASE_HRESULT(VER_E_HND_EQ)
    CASE_HRESULT(VER_E_FIL_OVERLAP)
    CASE_HRESULT(VER_E_FIL_EQ)
    CASE_HRESULT(VER_E_FIL_CONT_TRY)
    CASE_HRESULT(VER_E_FIL_CONT_HND)
    CASE_HRESULT(VER_E_FIL_CONT_FIL)
    CASE_HRESULT(VER_E_FIL_GTEQ_CS)
    CASE_HRESULT(VER_E_FIL_START)
    CASE_HRESULT(VER_E_FALLTHRU_EXCEP)
    CASE_HRESULT(VER_E_FALLTHRU_INTO_HND)
    CASE_HRESULT(VER_E_FALLTHRU_INTO_FIL)
    CASE_HRESULT(VER_E_LEAVE)
    CASE_HRESULT(VER_E_RETHROW)
    CASE_HRESULT(VER_E_ENDFINALLY)
    CASE_HRESULT(VER_E_ENDFILTER)
    CASE_HRESULT(VER_E_ENDFILTER_MISSING)
    CASE_HRESULT(VER_E_BR_INTO_TRY)
    CASE_HRESULT(VER_E_BR_INTO_HND)
    CASE_HRESULT(VER_E_BR_INTO_FIL)
    CASE_HRESULT(VER_E_BR_OUTOF_TRY)
    CASE_HRESULT(VER_E_BR_OUTOF_HND)
    CASE_HRESULT(VER_E_BR_OUTOF_FIL)
    CASE_HRESULT(VER_E_BR_OUTOF_FIN)
    CASE_HRESULT(VER_E_RET_FROM_TRY)
    CASE_HRESULT(VER_E_RET_FROM_HND)
    CASE_HRESULT(VER_E_RET_FROM_FIL)
    CASE_HRESULT(VER_E_BAD_JMP_TARGET)
    CASE_HRESULT(VER_E_PATH_LOC)
    CASE_HRESULT(VER_E_PATH_THIS)
    CASE_HRESULT(VER_E_PATH_STACK)
    CASE_HRESULT(VER_E_PATH_STACK_DEPTH)
    CASE_HRESULT(VER_E_THIS)
    CASE_HRESULT(VER_E_THIS_UNINIT_EXCEP)
    CASE_HRESULT(VER_E_THIS_UNINIT_STORE)
    CASE_HRESULT(VER_E_THIS_UNINIT_RET)
    CASE_HRESULT(VER_E_THIS_UNINIT_V_RET)
    CASE_HRESULT(VER_E_THIS_UNINIT_BR)
    CASE_HRESULT(VER_E_LDFTN_CTOR)
    CASE_HRESULT(VER_E_STACK_NOT_EQ)
    CASE_HRESULT(VER_E_STACK_UNEXPECTED)
    CASE_HRESULT(VER_E_STACK_EXCEPTION)
    CASE_HRESULT(VER_E_STACK_OVERFLOW)
    CASE_HRESULT(VER_E_STACK_UNDERFLOW)
    CASE_HRESULT(VER_E_STACK_EMPTY)
    CASE_HRESULT(VER_E_STACK_UNINIT)
    CASE_HRESULT(VER_E_STACK_I_I4_I8)
    CASE_HRESULT(VER_E_STACK_R_R4_R8)
    CASE_HRESULT(VER_E_STACK_NO_R_I8)
    CASE_HRESULT(VER_E_STACK_NUMERIC)
    CASE_HRESULT(VER_E_STACK_OBJREF)
    CASE_HRESULT(VER_E_STACK_P_OBJREF)
    CASE_HRESULT(VER_E_STACK_BYREF)
    CASE_HRESULT(VER_E_STACK_METHOD)
    CASE_HRESULT(VER_E_STACK_ARRAY_SD)
    CASE_HRESULT(VER_E_STACK_VALCLASS)
    CASE_HRESULT(VER_E_STACK_P_VALCLASS)
    CASE_HRESULT(VER_E_STACK_NO_VALCLASS)
    CASE_HRESULT(VER_E_LOC_DEAD)
    CASE_HRESULT(VER_E_LOC_NUM)
    CASE_HRESULT(VER_E_ARG_NUM)
    CASE_HRESULT(VER_E_TOKEN_RESOLVE)
    CASE_HRESULT(VER_E_TOKEN_TYPE)
    CASE_HRESULT(VER_E_TOKEN_TYPE_MEMBER)
    CASE_HRESULT(VER_E_TOKEN_TYPE_FIELD)
    CASE_HRESULT(VER_E_TOKEN_TYPE_SIG)
    CASE_HRESULT(VER_E_UNVERIFIABLE)
    CASE_HRESULT(VER_E_LDSTR_OPERAND)
    CASE_HRESULT(VER_E_RET_PTR_TO_STACK)
    CASE_HRESULT(VER_E_RET_VOID)
    CASE_HRESULT(VER_E_RET_MISSING)
    CASE_HRESULT(VER_E_RET_EMPTY)
    CASE_HRESULT(VER_E_RET_UNINIT)
    CASE_HRESULT(VER_E_ARRAY_ACCESS)
    CASE_HRESULT(VER_E_ARRAY_V_STORE)
    CASE_HRESULT(VER_E_ARRAY_SD)
    CASE_HRESULT(VER_E_ARRAY_SD_PTR)
    CASE_HRESULT(VER_E_ARRAY_FIELD)
    CASE_HRESULT(VER_E_ARGLIST)
    CASE_HRESULT(VER_E_VALCLASS)
    CASE_HRESULT(VER_E_METHOD_ACCESS)
    CASE_HRESULT(VER_E_FIELD_ACCESS)
    CASE_HRESULT(VER_E_DEAD)
    CASE_HRESULT(VER_E_FIELD_STATIC)
    CASE_HRESULT(VER_E_FIELD_NO_STATIC)
    CASE_HRESULT(VER_E_ADDR)
    CASE_HRESULT(VER_E_ADDR_BYREF)
    CASE_HRESULT(VER_E_ADDR_LITERAL)
    CASE_HRESULT(VER_E_INITONLY)
    CASE_HRESULT(VER_E_THROW)
    CASE_HRESULT(VER_E_CALLVIRT_VALCLASS)
    CASE_HRESULT(VER_E_CALL_SIG)
    CASE_HRESULT(VER_E_CALL_STATIC)
    CASE_HRESULT(VER_E_CTOR)
    CASE_HRESULT(VER_E_CTOR_VIRT)
    CASE_HRESULT(VER_E_CTOR_OR_SUPER)
    CASE_HRESULT(VER_E_CTOR_MUL_INIT)
    CASE_HRESULT(VER_E_SIG)
    CASE_HRESULT(VER_E_SIG_ARRAY)
    CASE_HRESULT(VER_E_SIG_ARRAY_PTR)
    CASE_HRESULT(VER_E_SIG_ARRAY_BYREF)
    CASE_HRESULT(VER_E_SIG_ELEM_PTR)
    CASE_HRESULT(VER_E_SIG_VARARG)
    CASE_HRESULT(VER_E_SIG_VOID)
    CASE_HRESULT(VER_E_SIG_BYREF_BYREF)
    CASE_HRESULT(VER_E_CODE_SIZE_ZERO)
    CASE_HRESULT(VER_E_BAD_VARARG)
    CASE_HRESULT(VER_E_TAIL_CALL)
    CASE_HRESULT(VER_E_TAIL_BYREF)
    CASE_HRESULT(VER_E_TAIL_RET)
    CASE_HRESULT(VER_E_TAIL_RET_VOID)
    CASE_HRESULT(VER_E_TAIL_RET_TYPE)
    CASE_HRESULT(VER_E_TAIL_STACK_EMPTY)
    CASE_HRESULT(VER_E_METHOD_END)
    CASE_HRESULT(VER_E_BAD_BRANCH)
    CASE_HRESULT(VER_E_FIN_OVERLAP)
    CASE_HRESULT(VER_E_LEXICAL_NESTING)
    CASE_HRESULT(VER_E_VOLATILE)
    CASE_HRESULT(VER_E_UNALIGNED)
    CASE_HRESULT(VER_E_INNERMOST_FIRST)
    CASE_HRESULT(VER_E_CALLI_VIRTUAL)
    CASE_HRESULT(VER_E_CALL_ABSTRACT)
    CASE_HRESULT(VER_E_STACK_UNEXP_ARRAY)
    CASE_HRESULT(VER_E_NOT_IN_GC_HEAP)
    CASE_HRESULT(VER_E_TRY_N_EMPTY_STACK)
    CASE_HRESULT(VER_E_DLGT_CTOR)
    CASE_HRESULT(VER_E_DLGT_BB)
    CASE_HRESULT(VER_E_DLGT_PATTERN)
    CASE_HRESULT(VER_E_DLGT_LDFTN)
    CASE_HRESULT(VER_E_FTN_ABSTRACT)
    CASE_HRESULT(VER_E_SIG_C_VC)
    CASE_HRESULT(VER_E_SIG_VC_C)
    CASE_HRESULT(VER_E_BOX_PTR_TO_STACK)
    CASE_HRESULT(VER_E_SIG_BYREF_TB_AH)
    CASE_HRESULT(VER_E_SIG_ARRAY_TB_AH)
    CASE_HRESULT(VER_E_ENDFILTER_STACK)
    CASE_HRESULT(VER_E_DLGT_SIG_I)
    CASE_HRESULT(VER_E_DLGT_SIG_O)
    CASE_HRESULT(VER_E_RA_PTR_TO_STACK)
    CASE_HRESULT(VER_E_CATCH_VALUE_TYPE)
    CASE_HRESULT(VER_E_CATCH_BYREF)
    CASE_HRESULT(VER_E_FIL_PRECEED_HND)
    CASE_HRESULT(VER_E_LDVIRTFTN_STATIC)
    CASE_HRESULT(VER_E_CALLVIRT_STATIC)
    CASE_HRESULT(VER_E_INITLOCALS)
    CASE_HRESULT(VER_E_BR_TO_EXCEPTION)
    CASE_HRESULT(VER_E_CALL_CTOR)
    CASE_HRESULT(VER_E_BAD_PE)
    CASE_HRESULT(VER_E_BAD_MD)
    CASE_HRESULT(VER_E_BAD_APPDOMAIN)
    CASE_HRESULT(VER_E_TYPELOAD)
    CASE_HRESULT(VER_E_PE_LOAD)
    CASE_HRESULT(VER_E_WRITE_RVA_STATIC)
    CASE_HRESULT(CORDBG_E_THREAD_NOT_SCHEDULED)
#endif

        default:
            return NULL;
    }
}
#ifdef _PREFAST_
#pragma warning(pop)
#endif


// ---------------------------------------------------------------------------
// HRException class.  Implements exception API for exceptions from HRESULTS
// ---------------------------------------------------------------------------

HRESULT HRException::GetHR()
{
    LIMITED_METHOD_DAC_CONTRACT;
    return m_hr; 
}

// ---------------------------------------------------------------------------
// COMException class. - moved to COMEx.cpp
// ---------------------------------------------------------------------------

// ---------------------------------------------------------------------------
// SEHException class.  Implements exception API for SEH exception info
// ---------------------------------------------------------------------------

HRESULT SEHException::GetHR()
{
    LIMITED_METHOD_DAC_CONTRACT;

    if (IsComPlusException(&m_exception)) // EE exception
        return (HRESULT) m_exception.ExceptionInformation[0];
    else
        return m_exception.ExceptionCode;
}

IErrorInfo *SEHException::GetErrorInfo() 
{
    LIMITED_METHOD_CONTRACT;
    return NULL;
}

void SEHException::GetMessage(SString &string)
{
    WRAPPER_NO_CONTRACT;
    
    if (IsComPlusException(&m_exception)) // EE exception
    {
        GenerateTopLevelHRExceptionMessage(GetHR(), string);
    }
    else
    {
        if (m_exception.ExceptionCode != 0)
        {
            string.Printf("Exception code 0x%.8x", m_exception.ExceptionCode);
        }
        else
        {
            // If we don't have a valid exception code, then give a generic message that's a little nicer than
            // "code 0x00000000".
            string.Printf("Unknown exception");
        }
    }
}

//==============================================================================
// DelegatingException class.  Implements exception API for "foreign" exceptions.
//==============================================================================

DelegatingException::DelegatingException()
 : m_delegatedException((Exception*)DELEGATE_NOT_YET_SET)
{
    LIMITED_METHOD_DAC_CONTRACT;
} // DelegatingException::DelegatingException()

//------------------------------------------------------------------------------
DelegatingException::~DelegatingException()
{
    WRAPPER_NO_CONTRACT;
    
    // If there is a valid delegate pointer (inited and non-NULL), delete it.
    if (IsDelegateValid())
        Delete(m_delegatedException);

    // Avoid confusion.
    m_delegatedException = NULL;
} // DelegatingException::~DelegatingException()

//------------------------------------------------------------------------------
// Retrieve the delegating exception, or get one from the Thread, or get NULL.
Exception* DelegatingException::GetDelegate()
{
    WRAPPER_NO_CONTRACT;
    
    // If we haven't gotten the exception pointer before..
    if (!IsDelegateSet())
    {
        // .. get it now.  NULL in case there isn't one and we take default action.
        m_delegatedException = NULL;
        GetLastThrownObjectExceptionFromThread(reinterpret_cast<void**>(&m_delegatedException));
    }

    return m_delegatedException;
} // Exception* DelegatingException::GetDelegate()

//------------------------------------------------------------------------------
// Virtual overrides
HRESULT DelegatingException::GetHR()
{
    WRAPPER_NO_CONTRACT;
    SUPPORTS_DAC_HOST_ONLY;

    // Retrieve any delegating exception.
    Exception *pDelegate = GetDelegate();

    // If there is a delegate exception, defer to it.  Otherwise,
    //  default to E_FAIL.
    return pDelegate ? pDelegate->GetHR() : E_FAIL;

} // HRESULT DelegatingException::GetHR()

//------------------------------------------------------------------------------
IErrorInfo *DelegatingException::GetErrorInfo()
{
    WRAPPER_NO_CONTRACT;
    
    // Retrieve any delegating exception.
    Exception *pDelegate = GetDelegate();

    // If there is a delegate exception, defer to it.  Otherwise,
    //  default to NULL.
    return pDelegate ? pDelegate->GetErrorInfo() : NULL;

} // IErrorInfo *DelegatingException::GetErrorInfo()

//------------------------------------------------------------------------------
void DelegatingException::GetMessage(SString &result)
{
    WRAPPER_NO_CONTRACT;
    
    // Retrieve any delegating exception.
    Exception *pDelegate = GetDelegate();

    // If there is a delegate exception, defer to it.  Otherwise,
    //  default to a generic message.
    if (pDelegate)
    {
        pDelegate->GetMessage(result);
    }
    else
    {
        // If we don't have a valid exception code, then give a generic message 
        //  that's a little nicer than "code 0x00000000".
        result.Printf("Unknown exception");
    }
} // void DelegatingException::GetMessage()

//------------------------------------------------------------------------------
Exception *DelegatingException::Clone()
{
    WRAPPER_NO_CONTRACT;

    // Clone the base exception, this will also take care of cloning the inner 
    // exception if there is one.
    NewHolder<DelegatingException> retExcep((DelegatingException*)Exception::Clone());
        
    // If there is a valid delegating exception...
    if (IsDelegateValid())
    {   // ... clone it.
        retExcep->m_delegatedException = m_delegatedException->Clone();
    }
    else
    {   // ... but if there is not, just copy -- either NULL or DELEGATE_NOT_YET_SET
        retExcep->m_delegatedException = m_delegatedException;
    }

    retExcep.SuppressRelease();
    return retExcep;
} // virtual Exception *DelegatingException::Clone()

//==============================================================================
//==============================================================================

void DECLSPEC_NORETURN ThrowHR(HRESULT hr)
{
    WRAPPER_NO_CONTRACT;

    STRESS_LOG1(LF_EH, LL_INFO100, "ThrowHR: HR = %x\n", hr);
    
    if (hr == E_OUTOFMEMORY)
        ThrowOutOfMemory();

    // Catchers assume only failing hresults
    _ASSERTE(FAILED(hr));   
    if (hr == S_OK)
        hr = E_FAIL;

    EX_THROW(HRException, (hr));
}

void DECLSPEC_NORETURN ThrowHR(HRESULT hr, SString const &msg)
{
    STATIC_CONTRACT_SO_TOLERANT;
    WRAPPER_NO_CONTRACT;

    STRESS_LOG1(LF_EH, LL_INFO100, "ThrowHR: HR = %x\n", hr);
    
    if (hr == E_OUTOFMEMORY)
        ThrowOutOfMemory();

    // Catchers assume only failing hresults
    _ASSERTE(FAILED(hr));   
    if (hr == S_OK)
        hr = E_FAIL;

    EX_THROW(HRMsgException, (hr, msg));
}

void DECLSPEC_NORETURN ThrowHR(HRESULT hr, UINT uText)
{
    WRAPPER_NO_CONTRACT;
    SUPPORTS_DAC_HOST_ONLY;

    if (hr == E_OUTOFMEMORY)
        ThrowOutOfMemory();

    // Catchers assume only failing hresults
    _ASSERTE(FAILED(hr));   
    if (hr == S_OK)
        hr = E_FAIL;

    SString sExceptionText;

    // We won't check the return value here. If it fails, we'll just
    // throw the HR
    sExceptionText.LoadResource(CCompRC::Error, uText);

    EX_THROW(HRMsgException, (hr, sExceptionText));
}

void DECLSPEC_NORETURN ThrowWin32(DWORD err)
{
    STATIC_CONTRACT_SO_TOLERANT;
    WRAPPER_NO_CONTRACT;
    if (err == ERROR_NOT_ENOUGH_MEMORY)
    {
        ThrowOutOfMemory();
    }
    else
    {
        ThrowHR(HRESULT_FROM_WIN32(err));
    }
}

void DECLSPEC_NORETURN ThrowLastError()
{
    WRAPPER_NO_CONTRACT;
    SUPPORTS_DAC;

    ThrowWin32(GetLastError());
}

void DECLSPEC_NORETURN ThrowOutOfMemory()
{
    CONTRACTL
    {
        THROWS;
        GC_NOTRIGGER;
        SO_TOLERANT;
        SUPPORTS_DAC;
    }
    CONTRACTL_END;
    
#ifndef DACCESS_COMPILE

    // Use volatile store to prevent compiler from optimizing the static variable away
    VolatileStoreWithoutBarrier<HRESULT>(&g_hrFatalError, COR_E_OUTOFMEMORY);

    // Regular CLR builds - throw our pre-created OOM exception object
    PAL_CPP_THROW(Exception *, Exception::GetOOMException());

#else

    // DAC builds - raise a DacError
    DacError(E_OUTOFMEMORY);

    // DacError always throws but isn't marked DECLSPEC_NORETURN so we have to
    // tell the compiler that this code is unreachable. We could mark DacError
    // (and DacNotImpl) as DECLSPEC_NORETURN, but then we've have to update a
    // lot of code where we do something afterwards. Also, due to inlining,
    // we'd sometimes have to change functions which call functions that only
    // call DacNotImpl. I have these changes in a bbpack and some of them look
    // nice, but I'm not sure if it's worth the risk of merge conflicts.
    UNREACHABLE();

#endif
}

#include "corexcep.h"

#ifdef FEATURE_STACK_PROBE
void DECLSPEC_NORETURN ThrowStackOverflow()
{
    CONTRACTL
    {
        // This should be throws... But it isn't because a SO doesn't technically
        // fall into the same THROW/NOTHROW conventions as the rest of the contract
        // infrastructure.
        NOTHROW;

        GC_NOTRIGGER;
        SO_TOLERANT;
        SUPPORTS_DAC; 
    }
    CONTRACTL_END;
	
    //g_hrFatalError=COR_E_STACKOVERFLOW;
    PTR_INT32 p_ghrFatalError = dac_cast<PTR_INT32>(GVAL_ADDR(g_hrFatalError));
    _ASSERTE(p_ghrFatalError != NULL);
    *p_ghrFatalError = COR_E_STACKOVERFLOW;


    RaiseException(EXCEPTION_SOFTSO, 0, 0, NULL);
    UNREACHABLE();
}
#endif

void DECLSPEC_NORETURN ThrowMessage(LPCSTR string, ...)
{
    CONTRACTL
    {
        THROWS;
        GC_NOTRIGGER;
        SO_TOLERANT;
    }
    CONTRACTL_END;
    
    StackSString message;

    va_list args;
    va_start(args, string);
    message.VPrintf(string, args);
    va_end(args);

    EX_THROW(HRMsgException, (E_FAIL, message));
}


//--------------------------------------------------------------------------------
// Helper for EX_THROW_WITH_INNER()
//
// Clones an exception into the current domain. Also handles special cases for
// OOM and other stuff. Making this a function so we don't inline all this logic
// every place we call EX_THROW_WITH_INNER.
//
// If the "inner" is a transient exception such as OOM or ThreadAbort, this function
// will just throw it rather than allow it to be wrapped in another exception.
//--------------------------------------------------------------------------------
Exception *ExThrowWithInnerHelper(Exception *inner)
{
    CONTRACTL
    {
        THROWS;
        GC_NOTRIGGER;
    }
    CONTRACTL_END

    // Yes, NULL is a legal case. Makes it easier to author uniform helpers for
    // both wrapped and normal exceptions.
    if (inner == NULL)
    {
        return NULL;
    }

    if (inner == Exception::GetOOMException())
    {
        // We don't want to do allocations if we're already throwing an OOM!
        PAL_CPP_THROW(Exception*, inner);
    }

    inner = inner->DomainBoundClone();

    // It isn't useful to wrap OOMs and StackOverflows in other exceptions. Just throw them now.
    // 
    if (inner->IsTransient())
    {
        PAL_CPP_THROW(Exception*, inner);
    }
    return inner;
}

#ifdef _DEBUG

#ifdef _MSC_VER
#pragma optimize("", off)
#endif // _MSC_VER

void ExThrowTrap(const char *fcn, const char *file, int line, const char *szType, HRESULT hr, const char *args)
{
    SUPPORTS_DAC;
    return;
}

#ifdef _MSC_VER
#pragma optimize("", on)
#endif // _MSC_VER

#endif




//-------------------------------------------------------------------------------------------
// This routine will generate the most descriptive possible error message for an hresult.
// It will generate at minimum the hex value. It will also try to generate the symbolic name
// (E_POINTER) and the friendly description (from the message tables.)
//
// bNoGeekStuff suppresses hex HR codes. Use this sparingly as most error strings generated by the
// CLR are aimed at developers, not end-users.
//-------------------------------------------------------------------------------------------
void GetHRMsg(HRESULT hr, SString &result, BOOL bNoGeekStuff/* = FALSE*/)
{
    CONTRACTL
    {
        GC_NOTRIGGER;
        THROWS;
    }
    CONTRACTL_END;

    result = W("");     // Make sure this routine isn't an inadvertent data-leak exploit!

    

    SString strDescr;
    BOOL    fHaveDescr = FALSE;

    if (FAILED(hr) && HRESULT_FACILITY(hr) == FACILITY_URT && HRESULT_CODE(hr) < MAX_URT_HRESULT_CODE)
    {
        fHaveDescr = strDescr.LoadResource(CCompRC::Error, MSG_FOR_URT_HR(hr));
    }
    else
    {
        DWORD dwFlags = FORMAT_MESSAGE_FROM_SYSTEM;
        dwFlags |= FORMAT_MESSAGE_MAX_WIDTH_MASK;

#if FEATURE_USE_LCID        
        fHaveDescr = strDescr.FormatMessage(dwFlags, 0, hr, LANG_USER_DEFAULT);
#else
        fHaveDescr = strDescr.FormatMessage(dwFlags, 0, hr, 0);
#endif
    }

    LPCSTR name = Exception::GetHRSymbolicName(hr);

    // If we can't get a resource string, print the hresult regardless.
    if (!fHaveDescr)
    {
        bNoGeekStuff = FALSE;
    }

    if (fHaveDescr)
    {
        result.Append(strDescr);
    }

    
    if (!bNoGeekStuff)
    {
        if (fHaveDescr)
        {
            result.Append(W(" ("));
        }

        SString strExcepFromHR;
        strExcepFromHR.LoadResource(CCompRC::Error, IDS_EE_EXCEPTION_FROM_HRESULT);
        result.Append(strExcepFromHR);
        result.AppendPrintf(W("0x%.8X"), hr);
        if (name != NULL)
        {
            result.AppendPrintf(W(" (%S)"), name);
        }


        if (fHaveDescr)
        {
            result.Append(W(")"));
        }
        
    }
}


//-------------------------------------------------------------------------------------------
// Similar to GetHRMsg but phrased for top-level exception message.
//-------------------------------------------------------------------------------------------
void GenerateTopLevelHRExceptionMessage(HRESULT hresult, SString &result)
{
    CONTRACTL
    {
        GC_NOTRIGGER;
        THROWS;
    }
    CONTRACTL_END;

    result = W("");     // Make sure this routine isn't an inadvertent data-leak exploit!

    GetHRMsg(hresult, result);
}

#if !defined(DACCESS_COMPILE)

void GetCurrentExceptionPointers(PEXCEPTION_POINTERS pExceptionInfo)
{
    WRAPPER_NO_CONTRACT;

    PEXCEPTION_RECORD pRecord = (PEXCEPTION_RECORD)ClrFlsGetValue(TlsIdx_PEXCEPTION_RECORD);
    PCONTEXT pContext = (PCONTEXT)ClrFlsGetValue(TlsIdx_PCONTEXT);

    pExceptionInfo->ContextRecord = pContext;
    pExceptionInfo->ExceptionRecord = pRecord;

#ifdef _DEBUG
    if (pRecord != NULL)
    {
        _ASSERTE ((PVOID)(pRecord) > (PVOID)(&pRecord));
    }
#endif
}
#endif // !defined(DACCESS_COMPILE)

DWORD GetCurrentExceptionCode()
{
    WRAPPER_NO_CONTRACT;
    SUPPORTS_DAC_HOST_ONLY; 

    return (DWORD)(size_t)ClrFlsGetValue(TlsIdx_EXCEPTION_CODE);
}

bool IsCurrentExceptionSO()
{
    WRAPPER_NO_CONTRACT;
    DWORD exceptionCode = GetCurrentExceptionCode();
	return IsSOExceptionCode(exceptionCode);
}

bool IsSOExceptionCode(DWORD exceptionCode)
{
	if (exceptionCode == STATUS_STACK_OVERFLOW 
#ifdef FEATURE_STACK_PROBE
		  || exceptionCode == EXCEPTION_SOFTSO
#endif
	   )
	{
		return TRUE;
	}
	else
		return FALSE;
}


//=========================================================================================== 
// These abstractions hide the difference between legacy desktop CLR's (that don't support
// side-by-side-inproc and rely on a fixed SEH code to identify managed exceptions) and
// new CLR's that support side-by-side inproc.
//
// The new CLR's use a different set of SEH codes to avoid conflicting with the legacy CLR's.
// In addition, to distinguish between EH's raised by different inproc instances of the CLR,
// the module handle of the owning CLR is stored in ExceptionRecord.ExceptionInformation[4].
//
// (Note: all existing SEH's use either only slot [0] or no slots at all. We are leaving
//  slots [1] thru [3] open for future expansion.) 
//===========================================================================================

// Is this exception code one of the special CLR-specific SEH codes that participate in the
// instance-tagging scheme?
BOOL IsInstanceTaggedSEHCode(DWORD dwExceptionCode)
{
   LIMITED_METHOD_DAC_CONTRACT;

    return dwExceptionCode == EXCEPTION_COMPLUS;
}

// This set of overloads generates the NumberParameters and ExceptionInformation[] array to
// pass to RaiseException().
//
// Parameters:
//    exceptionArgs:   a fixed-size array of size INSTANCE_TAGGED_SEH_PARAM_ARRAY_SIZE.
//                     This will get filled in by this function. (The module handle goes
//                     in the last slot if this is a side-by-side-inproc enabled build.)
//
//    exceptionArg1... up to four arguments that go in slots [0]..[3]. These depends
//                     the specific requirements of your exception code.
//
// Returns:
//    The NumberParameters to pass to RaiseException().
//
//    Basically, this is  either INSTANCE_TAGGED_SEH_PARAM_ARRAY_SIZE or the count of your
//    fixed arguments depending on whether this tagged-SEH-enabled build.
//
// This function is not permitted to fail.

// (the existing system can support more overloads up to 4 fixed arguments but we don't need them at this time.)

static DWORD MarkAsThrownByUsWorker(UINT numArgs, /*out*/ ULONG_PTR exceptionArgs[INSTANCE_TAGGED_SEH_PARAM_ARRAY_SIZE], ULONG_PTR arg0 = 0)
{
    STATIC_CONTRACT_NOTHROW;
    STATIC_CONTRACT_GC_NOTRIGGER;
    STATIC_CONTRACT_FORBID_FAULT;


    _ASSERTE(numArgs < INSTANCE_TAGGED_SEH_PARAM_ARRAY_SIZE);
    FillMemory(exceptionArgs, sizeof(ULONG_PTR) * INSTANCE_TAGGED_SEH_PARAM_ARRAY_SIZE, 0);

    exceptionArgs[0] = arg0;

#if !defined(FEATURE_UTILCODE_NO_DEPENDENCIES) && (defined(FEATURE_CORECLR) || !defined(SELF_NO_HOST) || defined(DACCESS_COMPILE))
    exceptionArgs[INSTANCE_TAGGED_SEH_PARAM_ARRAY_SIZE - 1] = (ULONG_PTR) (GetCLRModule());
#endif // !defined(FEATURE_UTILCODE_NO_DEPENDENCIES) && defined(FEATURE_CORECLR) || !defined(SELF_NO_HOST) || defined(DACCESS_COMPILE)
    
    return INSTANCE_TAGGED_SEH_PARAM_ARRAY_SIZE;
}

DWORD MarkAsThrownByUs(/*out*/ ULONG_PTR exceptionArgs[INSTANCE_TAGGED_SEH_PARAM_ARRAY_SIZE])
{
    STATIC_CONTRACT_NOTHROW;
    STATIC_CONTRACT_GC_NOTRIGGER;
    STATIC_CONTRACT_FORBID_FAULT;

    return MarkAsThrownByUsWorker(0, exceptionArgs);
}

DWORD MarkAsThrownByUs(/*out*/ ULONG_PTR exceptionArgs[INSTANCE_TAGGED_SEH_PARAM_ARRAY_SIZE], ULONG_PTR arg0)
{
    STATIC_CONTRACT_NOTHROW;
    STATIC_CONTRACT_GC_NOTRIGGER;
    STATIC_CONTRACT_FORBID_FAULT;

    return MarkAsThrownByUsWorker(1, exceptionArgs, arg0);
}

// Given an exception record, checks if it's exception code matches a specific exception code
// *and* whether it was tagged by the calling instance of the CLR.
//
// If this is a non-tagged-SEH-enabled build, it is blindly assumed to be tagged by the
// calling instance of the CLR. 
BOOL WasThrownByUs(const EXCEPTION_RECORD *pcER, DWORD dwExceptionCode)
{
    STATIC_CONTRACT_NOTHROW;
    STATIC_CONTRACT_GC_NOTRIGGER;
    STATIC_CONTRACT_FORBID_FAULT;
    STATIC_CONTRACT_SO_TOLERANT;
    STATIC_CONTRACT_SUPPORTS_DAC;

    _ASSERTE(IsInstanceTaggedSEHCode(dwExceptionCode));
    _ASSERTE(pcER != NULL);
    if (dwExceptionCode != pcER->ExceptionCode)
    {
        return FALSE;
    }

    if (pcER->NumberParameters != INSTANCE_TAGGED_SEH_PARAM_ARRAY_SIZE)
    {
        return FALSE;
    }
#if!defined(FEATURE_UTILCODE_NO_DEPENDENCIES) && (defined(FEATURE_CORECLR) || !defined(SELF_NO_HOST) || defined(DACCESS_COMPILE))
    if ( ((ULONG_PTR)(GetCLRModule())) != pcER->ExceptionInformation[INSTANCE_TAGGED_SEH_PARAM_ARRAY_SIZE - 1] )
    {
        return FALSE;
    }
    return TRUE;
#else // !(!defined(FEATURE_UTILCODE_NO_DEPENDENCIES) && (defined(FEATURE_CORECLR) || !defined(SELF_NO_HOST) || defined(DACCESS_COMPILE)))
    return FALSE;
#endif // !defined(FEATURE_UTILCODE_NO_DEPENDENCIES) && defined(FEATURE_CORECLR) || !defined(SELF_NO_HOST) || defined(DACCESS_COMPILE)
}



//-----------------------------------------------------------------------------------
// The following group wraps the basic abstracts specifically for EXCEPTION_COMPLUS.
//-----------------------------------------------------------------------------------
BOOL IsComPlusException(const EXCEPTION_RECORD *pcER)
{
    STATIC_CONTRACT_WRAPPER;

    return WasThrownByUs(pcER, EXCEPTION_COMPLUS);
}

VOID RaiseComPlusException()
{
    STATIC_CONTRACT_THROWS;
    STATIC_CONTRACT_GC_NOTRIGGER;
    STATIC_CONTRACT_FORBID_FAULT; 


    ULONG_PTR exceptionArgs[INSTANCE_TAGGED_SEH_PARAM_ARRAY_SIZE];
    DWORD     numParams = MarkAsThrownByUs(exceptionArgs);
    RaiseException(EXCEPTION_COMPLUS, 0, numParams, exceptionArgs);
}

//=========================================================================================== 
//=========================================================================================== 
