﻿using System;
using Xunit;

namespace Humanizer.Tests.Localisation.sr
{
    [UseCulture("sr")]
    public class TimeSpanHumanizeTests
    {

        [Theory]
        [Trait("Translation", "Google")]
        [InlineData(366, "1 година")]
        [InlineData(731, "2 године")]
        [InlineData(1096, "3 године")]
        [InlineData(4018, "11 година")]
        public void Years(int days, string expected)
        {
            Assert.Equal(expected, TimeSpan.FromDays(days).Humanize(maxUnit: Humanizer.Localisation.TimeUnit.Year));
        }

        [Theory]
        [Trait("Translation", "Google")]
        [InlineData(31, "1 месец")]
        [InlineData(61, "2 месеца")]
        [InlineData(92, "3 месеца")]
        [InlineData(335, "11 месеци")]
        public void Months(int days, string expected)
        {
            Assert.Equal(expected, TimeSpan.FromDays(days).Humanize(maxUnit: Humanizer.Localisation.TimeUnit.Year));
        }

        [Theory]
        [InlineData(35, "5 недеља")]
        [InlineData(14, "2 недеље")]
        [InlineData(7, "1 недеља")]
        public void Weeks(int days, string expected)
        {
            var actual = TimeSpan.FromDays(days).Humanize();
            Assert.Equal(expected, actual);
        }

        [Theory]
        [InlineData(6, "6 дана")]
        [InlineData(2, "2 дана")]
        [InlineData(1, "1 дан")]
        public void Days(int days, string expected)
        {
            var actual = TimeSpan.FromDays(days).Humanize();
            Assert.Equal(expected, actual);
        }

        [Theory]
        [InlineData(5, "5 сати")]
        [InlineData(2, "2 сата")]
        [InlineData(1, "1 сат")]
        public void Hours(int hours, string expected)
        {
            var actual = TimeSpan.FromHours(hours).Humanize();
            Assert.Equal(expected, actual);
        }

        [Theory]
        [InlineData(2, "2 минута")]
        [InlineData(1, "1 минут")]
        public void Minutes(int minutes, string expected)
        {
            var actual = TimeSpan.FromMinutes(minutes).Humanize();
            Assert.Equal(expected, actual);
        }

        [Theory]
        [InlineData(2, "2 секунде")]
        [InlineData(1, "1 секунда")]
        public void Seconds(int seconds, string expected)
        {
            var actual = TimeSpan.FromSeconds(seconds).Humanize();
            Assert.Equal(expected, actual);
        }

        [Theory]
        [InlineData(2, "2 милисекунде")]
        [InlineData(1, "1 милисекунда")]
        public void Milliseconds(int ms, string expected)
        {
            var actual = TimeSpan.FromMilliseconds(ms).Humanize();
            Assert.Equal(expected, actual);
        }

        [Fact]
        public void NoTime()
        {
            var noTime = TimeSpan.Zero;
            var actual = noTime.Humanize();
            Assert.Equal("без протеклог времена", actual);
        }
    }
}
