// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.

/*****************************************************************************/
#ifndef HARDWARE_INTRINSIC
#error Define HARDWARE_INTRINSIC before including this file
#endif
/*****************************************************************************/

// clang-format off

#ifdef FEATURE_HW_INTRINSICS
/* Note
    1) Each hardware intrinsic has a unique Intrinsic ID with type of `enum NamedIntrinsic`
    2) All the overloads of an intrinsic in an ISA class share one Intrinsic ID
    3) The intrinsic that generates instructions with a fixed imm8 operand has a `ival` field with "not -1" value, e.g., Sse.CompareEqual(v1,v2) -> cmpps xmm0, xmm1, 0
    4) SIMD intrinsics have a non-zero `SIMD size` field based-on that operate over `Vector128<T>`(16) or `Vector256<T>`(32)
    5) Scalar intrinsics that operate over general purpose registers (e.g., Sse41.Crc32) have `SIMD size` with 0
    6) Each intrinsic has a `NumArg` for number of parameters, and some intrinsics that are overloaded on multiple parameter numbers have this field with -1
    7) Each intrinsic has 10 `instructions` fields that list the instructions should be generated based-on the base type
    8) Each intrinsic has one category with type of `enum HWIntrinsicCategory`, please see the definition of HWIntrinsicCategory for details
    9) Each intrinsic has one or more flags with type of `enum HWIntrinsicFlag`
*/
// ************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                  Intrinsic ID                                      Function name                                      ISA        ival       SIMD size       NumArg                                                                   instructions                                                                                                Category                                         Flags
//                                                                                                                                                                            {TYP_BYTE,      TYP_UBYTE,       TYP_SHORT,     TYP_USHORT,    TYP_INT,       TYP_UINT,      TYP_LONG,      TYP_ULONG,     TYP_FLOAT,     TYP_DOUBLE}
// ************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  SSE Intrinsics
HARDWARE_INTRINSIC(SSE_IsSupported,                                  "get_IsSupported",                                  SSE,        -1,            0,           0,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_IsSupportedProperty,               HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE_Add,                                          "Add",                                              SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_addps,     INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(SSE_AddScalar,                                    "AddScalar",                                        SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_addss,     INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE_And,                                          "And",                                              SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_andps,     INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(SSE_AndNot,                                       "AndNot",                                           SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_andnps,    INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE_CompareEqual,                                 "CompareEqual",                                     SSE,         0,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpps,     INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(SSE_CompareEqualOrderedScalar,                    "CompareEqualOrderedScalar",                        SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_comiss,    INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_MultiIns|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_CompareEqualScalar,                           "CompareEqualScalar",                               SSE,         0,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpss,     INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE_CompareEqualUnorderedScalar,                  "CompareEqualUnorderedScalar",                      SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_ucomiss,   INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_MultiIns|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_CompareGreaterThan,                           "CompareGreaterThan",                               SSE,         6,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpps,     INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE_CompareGreaterThanOrderedScalar,              "CompareGreaterThanOrderedScalar",                  SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_comiss,    INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_MultiIns|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_CompareGreaterThanScalar,                     "CompareGreaterThanScalar",                         SSE,         6,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpss,     INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE_CompareGreaterThanUnorderedScalar,            "CompareGreaterThanUnorderedScalar",                SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_ucomiss,   INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_MultiIns|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_CompareGreaterThanOrEqual,                    "CompareGreaterThanOrEqual",                        SSE,         5,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpps,     INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE_CompareGreaterThanOrEqualOrderedScalar,       "CompareGreaterThanOrEqualOrderedScalar",           SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_comiss,    INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_MultiIns|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_CompareGreaterThanOrEqualScalar,              "CompareGreaterThanOrEqualScalar",                  SSE,         5,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpss,     INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE_CompareGreaterThanOrEqualUnorderedScalar,     "CompareGreaterThanOrEqualUnorderedScalar",         SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_ucomiss,   INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_MultiIns|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_CompareLessThan,                              "CompareLessThan",                                  SSE,         1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpps,     INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE_CompareLessThanOrderedScalar,                 "CompareLessThanOrderedScalar",                     SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_comiss,    INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_MultiIns|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_CompareLessThanScalar,                        "CompareLessThanScalar",                            SSE,         1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpss,     INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE_CompareLessThanUnorderedScalar,               "CompareLessThanUnorderedScalar",                   SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_ucomiss,   INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_MultiIns|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_CompareLessThanOrEqual,                       "CompareLessThanOrEqual",                           SSE,         2,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpps,     INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE_CompareLessThanOrEqualOrderedScalar,          "CompareLessThanOrEqualOrderedScalar",              SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_comiss,    INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_MultiIns|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_CompareLessThanOrEqualScalar,                 "CompareLessThanOrEqualScalar",                     SSE,         2,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpss,     INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE_CompareLessThanOrEqualUnorderedScalar,        "CompareLessThanOrEqualUnorderedScalar",            SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_ucomiss,   INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_MultiIns|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_CompareNotEqual,                              "CompareNotEqual",                                  SSE,         4,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpps,     INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(SSE_CompareNotEqualOrderedScalar,                 "CompareNotEqualOrderedScalar",                     SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_comiss,    INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_MultiIns|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_CompareNotEqualScalar,                        "CompareNotEqualScalar",                            SSE,         4,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpss,     INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE_CompareNotEqualUnorderedScalar,               "CompareNotEqualUnorderedScalar",                   SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_ucomiss,   INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_MultiIns|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_CompareNotGreaterThan,                        "CompareNotGreaterThan",                            SSE,         2,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpps,     INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE_CompareNotGreaterThanScalar,                  "CompareNotGreaterThanScalar",                      SSE,         2,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpss,     INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE_CompareNotGreaterThanOrEqual,                 "CompareNotGreaterThanOrEqual",                     SSE,         1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpps,     INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE_CompareNotGreaterThanOrEqualScalar,           "CompareNotGreaterThanOrEqualScalar",               SSE,         1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpss,     INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE_CompareNotLessThan,                           "CompareNotLessThan",                               SSE,         5,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpps,     INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE_CompareNotLessThanScalar,                     "CompareNotLessThanScalar",                         SSE,         5,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpss,     INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE_CompareNotLessThanOrEqual,                    "CompareNotLessThanOrEqual",                        SSE,         6,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpps,     INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE_CompareNotLessThanOrEqualScalar,              "CompareNotLessThanOrEqualScalar",                  SSE,         6,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpss,     INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE_CompareOrdered,                               "CompareOrdered",                                   SSE,         7,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpps,     INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE_CompareOrderedScalar,                         "CompareOrderedScalar",                             SSE,         7,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpss,     INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE_CompareUnordered,                             "CompareUnordered",                                 SSE,         3,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpps,     INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE_CompareUnorderedScalar,                       "CompareUnorderedScalar",                           SSE,         3,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpss,     INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE_ConvertToInt32,                               "ConvertToInt32",                                   SSE,        -1,           16,           1,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cvtss2si,  INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_ConvertToInt64,                               "ConvertToInt64",                                   SSE,        -1,           16,           1,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cvtss2si,  INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_ConvertToSingle,                              "ConvertToSingle",                                  SSE,        -1,           16,           1,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movss,     INS_invalid},           HW_Category_Helper,                            HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_ConvertScalarToVector128Single,               "ConvertScalarToVector128Single",                   SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cvtsi2ss,  INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_SecondArgMaybe64Bit|HW_Flag_BaseTypeFromFirstArg|HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE_ConvertToInt32WithTruncation,                 "ConvertToInt32WithTruncation",                     SSE,        -1,           16,           1,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cvttss2si, INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_ConvertToInt64WithTruncation,                 "ConvertToInt64WithTruncation",                     SSE,        -1,           16,           1,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cvttss2si, INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_BaseTypeFromFirstArg|HW_Flag_64BitOnly|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_Divide,                                       "Divide",                                           SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_divps,     INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE_DivideScalar,                                 "DivideScalar",                                     SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_divss,     INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE_LoadAlignedVector128,                         "LoadAlignedVector128",                             SSE,        -1,           16,           1,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movaps,    INS_invalid},           HW_Category_MemoryLoad,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_LoadHigh,                                     "LoadHigh",                                         SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movhps,    INS_invalid},           HW_Category_MemoryLoad,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_LoadLow,                                      "LoadLow",                                          SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movlps,    INS_invalid},           HW_Category_MemoryLoad,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_LoadScalarVector128,                          "LoadScalarVector128",                              SSE,        -1,           16,           1,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movss,     INS_invalid},           HW_Category_MemoryLoad,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_LoadVector128,                                "LoadVector128",                                    SSE,        -1,           16,           1,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movups,    INS_invalid},           HW_Category_MemoryLoad,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_Max,                                          "Max",                                              SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_maxps,     INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(SSE_MaxScalar,                                    "MaxScalar",                                        SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_maxss,     INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE_Min,                                          "Min",                                              SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_minps,     INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(SSE_MinScalar,                                    "MinScalar",                                        SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_minss,     INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE_MoveHighToLow,                                "MoveHighToLow",                                    SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movhlps,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoContainment)
HARDWARE_INTRINSIC(SSE_MoveLowToHigh,                                "MoveLowToHigh",                                    SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movlhps,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoContainment)
HARDWARE_INTRINSIC(SSE_MoveMask,                                     "MoveMask",                                         SSE,        -1,           16,           1,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movmskps,  INS_invalid},           HW_Category_Special,                           HW_Flag_NoContainment|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_MoveScalar,                                   "MoveScalar",                                       SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movss,     INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_NoContainment)
HARDWARE_INTRINSIC(SSE_Multiply,                                     "Multiply",                                         SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_mulps,     INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(SSE_MultiplyScalar,                               "MultiplyScalar",                                   SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_mulss,     INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE_Or,                                           "Or",                                               SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_orps,      INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(SSE_Prefetch0,                                    "Prefetch0",                                        SSE,        -1,            0,           1,           {INS_invalid,   INS_prefetcht0,  INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_Special,                           HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_Prefetch1,                                    "Prefetch1",                                        SSE,        -1,            0,           1,           {INS_invalid,   INS_prefetcht1,  INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_Special,                           HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_Prefetch2,                                    "Prefetch2",                                        SSE,        -1,            0,           1,           {INS_invalid,   INS_prefetcht2,  INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_Special,                           HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_PrefetchNonTemporal,                          "PrefetchNonTemporal",                              SSE,        -1,            0,           1,           {INS_invalid,   INS_prefetchnta, INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_Special,                           HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_Reciprocal,                                   "Reciprocal",                                       SSE,        -1,           16,           1,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_rcpps,     INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_ReciprocalScalar,                             "ReciprocalScalar",                                 SSE,        -1,           16,          -1,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_rcpss,     INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE_ReciprocalSqrt,                               "ReciprocalSqrt",                                   SSE,        -1,           16,           1,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_rsqrtps,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_ReciprocalSqrtScalar,                         "ReciprocalSqrtScalar",                             SSE,        -1,           16,          -1,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_rsqrtss,   INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE_SetScalarVector128,                           "SetScalarVector128",                               SSE,        -1,           16,           1,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movss,     INS_invalid},           HW_Category_Helper,                            HW_Flag_MultiIns|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_SetZeroVector128,                             "SetZeroVector128",                                 SSE,        -1,           16,           0,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_xorps,     INS_invalid},           HW_Category_Helper,                            HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_Shuffle,                                      "Shuffle",                                          SSE,        -1,           16,           3,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_shufps,    INS_invalid},           HW_Category_IMM,                               HW_Flag_FullRangeIMM)
HARDWARE_INTRINSIC(SSE_Sqrt,                                         "Sqrt",                                             SSE,        -1,           16,           1,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_sqrtps,    INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_SqrtScalar,                                   "SqrtScalar",                                       SSE,        -1,           16,          -1,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_sqrtss,    INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE_StaticCast,                                   "StaticCast",                                       SSE,        -1,           16,           1,           {INS_movaps,    INS_movaps,      INS_movaps,    INS_movaps,    INS_movaps,    INS_movaps,    INS_movaps,    INS_movaps,    INS_movaps,    INS_movaps},            HW_Category_Helper,                            HW_Flag_TwoTypeGeneric|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_Store,                                        "Store",                                            SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movups,    INS_invalid},           HW_Category_MemoryStore,                       HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_StoreAligned,                                 "StoreAligned",                                     SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movaps,    INS_invalid},           HW_Category_MemoryStore,                       HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_StoreAlignedNonTemporal,                      "StoreAlignedNonTemporal",                          SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movntps,   INS_invalid},           HW_Category_MemoryStore,                       HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_StoreFence,                                   "StoreFence",                                       SSE,        -1,            0,           0,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_Special,                           HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_StoreHigh,                                    "StoreHigh",                                        SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movhps,    INS_invalid},           HW_Category_MemoryStore,                       HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_StoreLow,                                     "StoreLow",                                         SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movlps,    INS_invalid},           HW_Category_MemoryStore,                       HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_StoreScalar,                                  "StoreScalar",                                      SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movss,     INS_invalid},           HW_Category_MemoryStore,                       HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE_Subtract,                                     "Subtract",                                         SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_subps,     INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE_SubtractScalar,                               "SubtractScalar",                                   SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_subss,     INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE_UnpackHigh,                                   "UnpackHigh",                                       SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_unpckhps,  INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE_UnpackLow,                                    "UnpackLow",                                        SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_unpcklps,  INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE_Xor,                                          "Xor",                                              SSE,        -1,           16,           2,           {INS_invalid,   INS_invalid,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_xorps,     INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_Commutative)

// ************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                  Intrinsic ID                                      Function name                                      ISA        ival       SIMD size       NumArg                                                                   instructions                                                                                              Category                                         Flags
//                                                                                                                                                                            {TYP_BYTE,      TYP_UBYTE,     TYP_SHORT,     TYP_USHORT,    TYP_INT,       TYP_UINT,      TYP_LONG,      TYP_ULONG,     TYP_FLOAT,     TYP_DOUBLE}
// ************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  SSE2 Intrinsics
HARDWARE_INTRINSIC(SSE2_IsSupported,                                 "get_IsSupported",                                  SSE2,       -1,            0,          0,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_IsSupportedProperty,               HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE2_Add,                                         "Add",                                              SSE2,       -1,           16,          2,            {INS_paddb,     INS_paddb,     INS_paddw,     INS_paddw,     INS_paddd,     INS_paddd,     INS_paddq,     INS_paddq,     INS_invalid,   INS_addpd},             HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(SSE2_AddSaturate,                                 "AddSaturate",                                      SSE2,       -1,           16,          2,            {INS_paddsb,    INS_paddusb,   INS_paddsw,    INS_paddusw,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(SSE2_AddScalar,                                   "AddScalar",                                        SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_addsd},             HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE2_And,                                         "And",                                              SSE2,       -1,           16,          2,            {INS_pand,      INS_pand,      INS_pand,      INS_pand,      INS_pand,      INS_pand,      INS_pand,      INS_pand,      INS_invalid,   INS_andpd},             HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(SSE2_AndNot,                                      "AndNot",                                           SSE2,       -1,           16,          2,            {INS_pandn,     INS_pandn,     INS_pandn,     INS_pandn,     INS_pandn,     INS_pandn,     INS_pandn,     INS_pandn,     INS_invalid,   INS_andnpd},            HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE2_Average,                                     "Average",                                          SSE2,       -1,           16,          2,            {INS_invalid,   INS_pavgb,     INS_invalid,   INS_pavgw,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(SSE2_CompareEqual,                                "CompareEqual",                                     SSE2,        0,           16,          2,            {INS_pcmpeqb,   INS_pcmpeqb,   INS_pcmpeqw,   INS_pcmpeqw,   INS_pcmpeqd,   INS_pcmpeqd,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmppd},             HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(SSE2_CompareEqualOrderedScalar,                   "CompareEqualOrderedScalar",                        SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_comisd},            HW_Category_SIMDScalar,                        HW_Flag_MultiIns|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_CompareEqualScalar,                          "CompareEqualScalar",                               SSE2,        0,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpsd},             HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE2_CompareEqualUnorderedScalar,                 "CompareEqualUnorderedScalar",                      SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_ucomisd},           HW_Category_SIMDScalar,                        HW_Flag_MultiIns|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_CompareGreaterThan,                          "CompareGreaterThan",                               SSE2,        6,           16,          2,            {INS_pcmpgtb,   INS_invalid,   INS_pcmpgtw,   INS_invalid,   INS_pcmpgtd,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmppd},             HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE2_CompareGreaterThanOrderedScalar,             "CompareGreaterThanOrderedScalar",                  SSE2,       -1,           16,          2,            {INS_pcmpgtb,   INS_invalid,   INS_pcmpgtw,   INS_invalid,   INS_pcmpgtd,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_comisd},            HW_Category_SIMDScalar,                        HW_Flag_MultiIns|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_CompareGreaterThanScalar,                    "CompareGreaterThanScalar",                         SSE2,        6,           16,          2,            {INS_pcmpgtb,   INS_invalid,   INS_pcmpgtw,   INS_invalid,   INS_pcmpgtd,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpsd},             HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE2_CompareGreaterThanUnorderedScalar,           "CompareGreaterThanUnorderedScalar",                SSE2,       -1,           16,          2,            {INS_pcmpgtb,   INS_invalid,   INS_pcmpgtw,   INS_invalid,   INS_pcmpgtd,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_ucomisd},           HW_Category_SIMDScalar,                        HW_Flag_MultiIns|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_CompareGreaterThanOrEqual,                   "CompareGreaterThanOrEqual",                        SSE2,        5,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmppd},             HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE2_CompareGreaterThanOrEqualOrderedScalar,      "CompareGreaterThanOrEqualOrderedScalar",           SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_comisd},            HW_Category_SIMDScalar,                        HW_Flag_MultiIns|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_CompareGreaterThanOrEqualScalar,             "CompareGreaterThanOrEqualScalar",                  SSE2,        5,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpsd},             HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE2_CompareGreaterThanOrEqualUnorderedScalar,    "CompareGreaterThanOrEqualUnorderedScalar",         SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_ucomisd},           HW_Category_SIMDScalar,                        HW_Flag_MultiIns|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_CompareLessThan,                             "CompareLessThan",                                  SSE2,        1,           16,          2,            {INS_pcmpgtb,   INS_invalid,   INS_pcmpgtw,   INS_invalid,   INS_pcmpgtd,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmppd},             HW_Category_Special,                           HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE2_CompareLessThanOrderedScalar,                "CompareLessThanOrderedScalar",                     SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_comisd},            HW_Category_SIMDScalar,                        HW_Flag_MultiIns|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_CompareLessThanScalar,                       "CompareLessThanScalar",                            SSE2,        1,           16,          2,            {INS_pcmpgtb,   INS_invalid,   INS_pcmpgtw,   INS_invalid,   INS_pcmpgtd,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpsd},             HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE2_CompareLessThanUnorderedScalar,              "CompareLessThanUnorderedScalar",                   SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_ucomisd},           HW_Category_SIMDScalar,                        HW_Flag_MultiIns|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_CompareLessThanOrEqual,                      "CompareLessThanOrEqual",                           SSE2,        2,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmppd},             HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE2_CompareLessThanOrEqualOrderedScalar,         "CompareLessThanOrEqualOrderedScalar",              SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_comisd},            HW_Category_SIMDScalar,                        HW_Flag_MultiIns|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_CompareLessThanOrEqualScalar,                "CompareLessThanOrEqualScalar",                     SSE2,        2,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpsd},             HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE2_CompareLessThanOrEqualUnorderedScalar,       "CompareLessThanOrEqualUnorderedScalar",            SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_ucomisd},           HW_Category_SIMDScalar,                        HW_Flag_MultiIns|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_CompareNotEqual,                             "CompareNotEqual",                                  SSE2,        4,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmppd},             HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(SSE2_CompareNotEqualOrderedScalar,                "CompareNotEqualOrderedScalar",                     SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_comisd},            HW_Category_SIMDScalar,                        HW_Flag_MultiIns|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_CompareNotEqualScalar,                       "CompareNotEqualScalar",                            SSE2,        4,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpsd},             HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE2_CompareNotEqualUnorderedScalar,              "CompareNotEqualUnorderedScalar",                   SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_ucomisd},           HW_Category_SIMDScalar,                        HW_Flag_MultiIns|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_CompareNotGreaterThan,                       "CompareNotGreaterThan",                            SSE2,        2,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmppd},             HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE2_CompareNotGreaterThanScalar,                 "CompareNotGreaterThanScalar",                      SSE2,        2,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpsd},             HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE2_CompareNotGreaterThanOrEqual,                "CompareNotGreaterThanOrEqual",                     SSE2,        1,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmppd},             HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE2_CompareNotGreaterThanOrEqualScalar,          "CompareNotGreaterThanOrEqualScalar",               SSE2,        1,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpsd},             HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE2_CompareNotLessThan,                          "CompareNotLessThan",                               SSE2,        5,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmppd},             HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE2_CompareNotLessThanScalar,                    "CompareNotLessThanScalar",                         SSE2,        5,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpsd},             HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE2_CompareNotLessThanOrEqual,                   "CompareNotLessThanOrEqual",                        SSE2,        6,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmppd},             HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE2_CompareNotLessThanOrEqualScalar,             "CompareNotLessThanOrEqualScalar",                  SSE2,        6,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpsd},             HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE2_CompareOrdered,                              "CompareOrdered",                                   SSE2,        7,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmppd},             HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE2_CompareOrderedScalar,                        "CompareOrderedScalar",                             SSE2,        7,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpsd},             HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE2_CompareUnordered,                            "CompareUnordered",                                 SSE2,        3,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmppd},             HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE2_CompareUnorderedScalar,                      "CompareUnorderedScalar",                           SSE2,        3,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpsd},             HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE2_ConvertToDouble,                             "ConvertToDouble",                                  SSE2,       -1,           16,          1,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movsdsse2},         HW_Category_Helper,                            HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_ConvertToInt32,                              "ConvertToInt32",                                   SSE2,       -1,           16,          1,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_mov_xmm2i, INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cvtsd2si},          HW_Category_SIMDScalar,                        HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_ConvertToInt32WithTruncation,                "ConvertToInt32WithTruncation",                     SSE2,       -1,           16,          1,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cvttsd2si},         HW_Category_SIMDScalar,                        HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_ConvertToInt64,                              "ConvertToInt64",                                   SSE2,       -1,           16,          1,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_mov_xmm2i, INS_invalid,   INS_invalid,   INS_cvtsd2si},          HW_Category_SIMDScalar,                        HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_ConvertToInt64WithTruncation,                "ConvertToInt64WithTruncation",                     SSE2,       -1,           16,          1,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cvttsd2si},         HW_Category_SIMDScalar,                        HW_Flag_BaseTypeFromFirstArg|HW_Flag_64BitOnly|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_ConvertToUInt32,                             "ConvertToUInt32",                                  SSE2,       -1,           16,          1,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_mov_xmm2i, INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_ConvertToUInt64,                             "ConvertToUInt64",                                  SSE2,       -1,           16,          1,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_mov_xmm2i, INS_invalid,   INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_ConvertToVector128Double,                    "ConvertToVector128Double",                         SSE2,       -1,           16,          1,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cvtdq2pd,  INS_invalid,   INS_invalid,   INS_invalid,   INS_cvtps2pd,  INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_ConvertScalarToVector128Double,              "ConvertScalarToVector128Double",                   SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cvtsi2sd,  INS_invalid,   INS_cvtsi2sd,  INS_invalid,   INS_cvtss2sd,  INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_SecondArgMaybe64Bit|HW_Flag_SpecialCodeGen|HW_Flag_BaseTypeFromSecondArg)
HARDWARE_INTRINSIC(SSE2_ConvertToVector128Int32,                     "ConvertToVector128Int32",                          SSE2,       -1,           16,          1,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cvtps2dq,  INS_cvtpd2dq},          HW_Category_SimpleSIMD,                        HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_ConvertScalarToVector128Int32,               "ConvertScalarToVector128Int32",                    SSE2,       -1,           16,          1,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_mov_i2xmm, INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_ConvertToVector128Int32WithTruncation,       "ConvertToVector128Int32WithTruncation",            SSE2,       -1,           16,          1,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cvttps2dq, INS_cvttpd2dq},         HW_Category_SimpleSIMD,                        HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_ConvertScalarToVector128Int64,               "ConvertScalarToVector128Int64",                    SSE2,       -1,           16,          1,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_mov_i2xmm, INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_64BitOnly|HW_Flag_SpecialCodeGen|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_ConvertToVector128Single,                    "ConvertToVector128Single",                         SSE2,       -1,           16,          1,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cvtdq2ps,  INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cvtpd2ps},          HW_Category_SimpleSIMD,                        HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_ConvertScalarToVector128Single,              "ConvertScalarToVector128Single",                   SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cvtsd2ss,  INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(SSE2_ConvertScalarToVector128UInt32,              "ConvertScalarToVector128UInt32",                   SSE2,       -1,           16,          1,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_mov_i2xmm, INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_ConvertScalarToVector128UInt64,              "ConvertScalarToVector128UInt64",                   SSE2,       -1,           16,          1,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_mov_i2xmm, INS_invalid,   INS_invalid},           HW_Category_SIMDScalar,                        HW_Flag_64BitOnly|HW_Flag_SpecialCodeGen|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_Divide,                                      "Divide",                                           SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_divpd},             HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE2_DivideScalar,                                "DivideScalar",                                     SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_divsd},             HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE2_Extract,                                     "Extract",                                          SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_pextrw,    INS_pextrw,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_IMM,                               HW_Flag_FullRangeIMM|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_Insert,                                      "Insert",                                           SSE2,       -1,           16,          3,            {INS_invalid,   INS_invalid,   INS_pinsrw,    INS_pinsrw,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_IMM,                               HW_Flag_FullRangeIMM)
HARDWARE_INTRINSIC(SSE2_LoadAlignedVector128,                        "LoadAlignedVector128",                             SSE2,       -1,           16,          1,            {INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_invalid,   INS_movapd},            HW_Category_MemoryLoad,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_LoadFence,                                   "LoadFence",                                        SSE2,       -1,            0,          0,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_Special,                           HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_LoadHigh,                                    "LoadHigh",                                         SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movhpd},            HW_Category_MemoryLoad,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_LoadLow,                                     "LoadLow",                                          SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movlpd},            HW_Category_MemoryLoad,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_LoadScalarVector128,                         "LoadScalarVector128",                              SSE2,       -1,           16,          1,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movd,      INS_movd,      INS_movq,      INS_movq,      INS_invalid,   INS_movsdsse2},         HW_Category_MemoryLoad,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_LoadVector128,                               "LoadVector128",                                    SSE2,       -1,           16,          1,            {INS_movdqu,    INS_movdqu,    INS_movdqu,    INS_movdqu,    INS_movdqu,    INS_movdqu,    INS_movdqu,    INS_movdqu,    INS_invalid,   INS_movupd},            HW_Category_MemoryLoad,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_MaskMove,                                    "MaskMove",                                         SSE2,       -1,           16,          3,            {INS_maskmovdqu,INS_maskmovdqu,INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_MemoryStore,                       HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_Max,                                         "Max",                                              SSE2,       -1,           16,          2,            {INS_invalid,   INS_pmaxub,    INS_pmaxsw,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_maxpd},             HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(SSE2_MemoryFence,                                 "MemoryFence",                                      SSE2,       -1,            0,          0,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_Special,                           HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_MaxScalar,                                   "MaxScalar",                                        SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_maxsd},             HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE2_Min,                                         "Min",                                              SSE2,       -1,           16,          2,            {INS_invalid,   INS_pminub,    INS_pminsw,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_minpd},             HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(SSE2_MinScalar,                                   "MinScalar",                                        SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_minsd},             HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE2_MoveMask,                                    "MoveMask",                                         SSE2,       -1,           16,          1,            {INS_pmovmskb,  INS_pmovmskb,  INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movmskpd},          HW_Category_Special,                           HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_MoveScalar,                                  "MoveScalar",                                       SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movsdsse2},         HW_Category_SIMDScalar,                        HW_Flag_NoContainment)
HARDWARE_INTRINSIC(SSE2_Multiply,                                    "Multiply",                                         SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_pmuludq,   INS_invalid,   INS_invalid,   INS_invalid,   INS_mulpd},             HW_Category_SimpleSIMD,                        HW_Flag_Commutative|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(SSE2_MultiplyHigh,                                "MultiplyHigh",                                     SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_pmulhw,    INS_pmulhuw,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(SSE2_MultiplyHorizontalAdd,                       "MultiplyHorizontalAdd",                            SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_pmaddwd,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_Commutative|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(SSE2_MultiplyLow,                                 "MultiplyLow",                                      SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_pmullw,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(SSE2_MultiplyScalar,                              "MultiplyScalar",                                   SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_mulsd},             HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE2_Or,                                          "Or",                                               SSE2,       -1,           16,          2,            {INS_por,       INS_por,       INS_por,       INS_por,       INS_por,       INS_por,       INS_por,       INS_por,       INS_invalid,   INS_orpd},              HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(SSE2_PackSignedSaturate,                          "PackSignedSaturate",                               SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_packsswb,  INS_invalid,   INS_packssdw,  INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(SSE2_PackUnsignedSaturate,                        "PackUnsignedSaturate",                             SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_packuswb,  INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(SSE2_SetScalarVector128,                          "SetScalarVector128",                               SSE2,       -1,           16,          1,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movsdsse2},         HW_Category_Helper,                            HW_Flag_MultiIns|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_SetZeroVector128,                            "SetZeroVector128",                                 SSE2,       -1,           16,          0,            {INS_pxor,      INS_pxor,      INS_pxor,      INS_pxor,      INS_pxor,      INS_pxor,      INS_pxor,      INS_pxor,      INS_invalid,   INS_xorpd},             HW_Category_Helper,                            HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE2_SumAbsoluteDifferences,                      "SumAbsoluteDifferences",                           SSE2,       -1,           16,          2,            {INS_invalid,   INS_psadbw,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(SSE2_ShiftLeftLogical,                            "ShiftLeftLogical",                                 SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_psllw,     INS_psllw,     INS_pslld,     INS_pslld,     INS_psllq,     INS_psllq,     INS_invalid,   INS_invalid},           HW_Category_IMM,                               HW_Flag_MaybeIMM|HW_Flag_NoJmpTableIMM|HW_Flag_FullRangeIMM)
HARDWARE_INTRINSIC(SSE2_ShiftLeftLogical128BitLane,                  "ShiftLeftLogical128BitLane",                       SSE2,       -1,           16,          2,            {INS_pslldq,    INS_pslldq,    INS_pslldq,    INS_pslldq,    INS_pslldq,    INS_pslldq,    INS_pslldq,    INS_pslldq,    INS_invalid,   INS_invalid},           HW_Category_IMM,                               HW_Flag_FullRangeIMM)
HARDWARE_INTRINSIC(SSE2_ShiftRightArithmetic,                        "ShiftRightArithmetic",                             SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_psraw,     INS_invalid,   INS_psrad,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_IMM,                               HW_Flag_MaybeIMM|HW_Flag_NoJmpTableIMM|HW_Flag_FullRangeIMM)
HARDWARE_INTRINSIC(SSE2_ShiftRightLogical,                           "ShiftRightLogical",                                SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_psrlw,     INS_psrlw,     INS_psrld,     INS_psrld,     INS_psrlq,     INS_psrlq,     INS_invalid,   INS_invalid},           HW_Category_IMM,                               HW_Flag_MaybeIMM|HW_Flag_NoJmpTableIMM|HW_Flag_FullRangeIMM)
HARDWARE_INTRINSIC(SSE2_ShiftRightLogical128BitLane,                 "ShiftRightLogical128BitLane",                      SSE2,       -1,           16,          2,            {INS_psrldq,    INS_psrldq,    INS_psrldq,    INS_psrldq,    INS_psrldq,    INS_psrldq,    INS_psrldq,    INS_psrldq,    INS_invalid,   INS_invalid},           HW_Category_IMM,                               HW_Flag_FullRangeIMM)
HARDWARE_INTRINSIC(SSE2_Shuffle,                                     "Shuffle",                                          SSE2,       -1,           16,         -1,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_pshufd,    INS_pshufd,    INS_invalid,   INS_invalid,   INS_invalid,   INS_shufpd},            HW_Category_IMM,                               HW_Flag_FullRangeIMM)
HARDWARE_INTRINSIC(SSE2_ShuffleHigh,                                 "ShuffleHigh",                                      SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_pshufhw,   INS_pshufhw,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_IMM,                               HW_Flag_FullRangeIMM)
HARDWARE_INTRINSIC(SSE2_ShuffleLow,                                  "ShuffleLow",                                       SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_pshuflw,   INS_pshuflw,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_IMM,                               HW_Flag_FullRangeIMM)
HARDWARE_INTRINSIC(SSE2_Sqrt,                                        "Sqrt",                                             SSE2,       -1,           16,          1,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_sqrtpd},            HW_Category_SimpleSIMD,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_SqrtScalar,                                  "SqrtScalar",                                       SSE2,       -1,           16,          1,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_sqrtsd},            HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE2_Store,                                       "Store",                                            SSE2,       -1,           16,          2,            {INS_movdqu,    INS_movdqu,    INS_movdqu,    INS_movdqu,    INS_movdqu,    INS_movdqu,    INS_movdqu,    INS_movdqu,    INS_invalid,   INS_movupd},            HW_Category_MemoryStore,                       HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_StoreAligned,                                "StoreAligned",                                     SSE2,       -1,           16,          2,            {INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_invalid,   INS_movapd},            HW_Category_MemoryStore,                       HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_StoreAlignedNonTemporal,                     "StoreAlignedNonTemporal",                          SSE2,       -1,           16,          2,            {INS_movntdq,   INS_movntdq,   INS_movntdq,   INS_movntdq,   INS_movntdq,   INS_movntdq,   INS_movntdq,   INS_movntdq,   INS_invalid,   INS_movntpd},           HW_Category_MemoryStore,                       HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_StoreHigh,                                   "StoreHigh",                                        SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movhpd},            HW_Category_MemoryStore,                       HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_StoreLow,                                    "StoreLow",                                         SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movq,      INS_movq,      INS_invalid,   INS_movlpd},            HW_Category_MemoryStore,                       HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_StoreNonTemporal,                            "StoreNonTemporal",                                 SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movnti,    INS_movnti,    INS_movnti,    INS_movnti,    INS_invalid,   INS_invalid},           HW_Category_Scalar,                            HW_Flag_NoRMWSemantics|HW_Flag_SecondArgMaybe64Bit)
HARDWARE_INTRINSIC(SSE2_StoreScalar,                                 "StoreScalar",                                      SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movsdsse2},         HW_Category_MemoryStore,                       HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_Subtract,                                    "Subtract",                                         SSE2,       -1,           16,          2,            {INS_psubb,     INS_psubb,     INS_psubw,     INS_psubw,     INS_psubd,     INS_psubd,     INS_psubq,     INS_psubq,     INS_invalid,   INS_subpd},             HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE2_SubtractSaturate,                            "SubtractSaturate",                                 SSE2,       -1,           16,          2,            {INS_psubsb,    INS_psubusb,   INS_psubsw,    INS_psubusw,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE2_SubtractScalar,                              "SubtractScalar",                                   SSE2,       -1,           16,          2,            {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_subsd},             HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE2_UnpackHigh,                                  "UnpackHigh",                                       SSE2,       -1,           16,          2,            {INS_punpckhbw, INS_punpckhbw, INS_punpckhwd, INS_punpckhwd, INS_punpckhdq, INS_punpckhdq, INS_punpckhqdq,INS_punpckhqdq,INS_invalid,   INS_unpckhpd},          HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE2_UnpackLow,                                   "UnpackLow",                                        SSE2,       -1,           16,          2,            {INS_punpcklbw, INS_punpcklbw, INS_punpcklwd, INS_punpcklwd, INS_punpckldq, INS_punpckldq, INS_punpcklqdq,INS_punpcklqdq,INS_invalid,   INS_unpcklpd},          HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE2_Xor,                                         "Xor",                                              SSE2,       -1,           16,          2,            {INS_pxor,      INS_pxor,      INS_pxor,      INS_pxor,      INS_pxor,      INS_pxor,      INS_pxor,      INS_pxor,      INS_invalid,   INS_xorpd},             HW_Category_SimpleSIMD,                        HW_Flag_Commutative)

// ************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                  Intrinsic ID                                      Function name                                      ISA        ival       SIMD size       NumArg                                                                   instructions                                                                                              Category                                         Flags
//                                                                                                                                                                            {TYP_BYTE,      TYP_UBYTE,     TYP_SHORT,     TYP_USHORT,    TYP_INT,       TYP_UINT,      TYP_LONG,      TYP_ULONG,     TYP_FLOAT,     TYP_DOUBLE}
// ************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  SSE3 Intrinsics
HARDWARE_INTRINSIC(SSE3_IsSupported,                                 "get_IsSupported",                                  SSE3,       -1,           0,            0,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_IsSupportedProperty,               HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE3_AddSubtract,                                 "AddSubtract",                                      SSE3,       -1,           16,           2,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_addsubps,  INS_addsubpd},          HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE3_HorizontalAdd,                               "HorizontalAdd",                                    SSE3,       -1,           16,           2,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_haddps,    INS_haddpd},            HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE3_HorizontalSubtract,                          "HorizontalSubtract",                               SSE3,       -1,           16,           2,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_hsubps,    INS_hsubpd},            HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE3_LoadAndDuplicateToVector128,                 "LoadAndDuplicateToVector128",                      SSE3,       -1,           16,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_lddqu,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movddup},           HW_Category_MemoryLoad,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE3_LoadDquVector128,                            "LoadDquVector128",                                 SSE3,       -1,           16,           1,           {INS_lddqu,     INS_lddqu,     INS_lddqu,     INS_lddqu,     INS_lddqu,     INS_lddqu,     INS_lddqu,     INS_lddqu,     INS_invalid,   INS_invalid},           HW_Category_MemoryLoad,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE3_MoveAndDuplicate,                            "MoveAndDuplicate",                                 SSE3,       -1,           16,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movddup},           HW_Category_SimpleSIMD,                        HW_Flag_NoContainment|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE3_MoveHighAndDuplicate,                        "MoveHighAndDuplicate",                             SSE3,       -1,           16,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movshdup,  INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoContainment|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE3_MoveLowAndDuplicate,                         "MoveLowAndDuplicate",                              SSE3,       -1,           16,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movsldup,  INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoContainment|HW_Flag_NoRMWSemantics)

// ************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                  Intrinsic ID                                      Function name                                      ISA        ival       SIMD size       NumArg                                                                   instructions                                                                                              Category                                         Flags
//                                                                                                                                                                            {TYP_BYTE,      TYP_UBYTE,     TYP_SHORT,     TYP_USHORT,    TYP_INT,       TYP_UINT,      TYP_LONG,      TYP_ULONG,     TYP_FLOAT,     TYP_DOUBLE}
// ************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  SSSE3 Intrinsics
HARDWARE_INTRINSIC(SSSE3_IsSupported,                                "get_IsSupported",                                  SSSE3,      -1,           0,            0,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_IsSupportedProperty,               HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSSE3_Abs,                                        "Abs",                                              SSSE3,      -1,           16,           1,           {INS_invalid,   INS_pabsb,     INS_invalid,   INS_pabsw,     INS_invalid,   INS_pabsd,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSSE3_AlignRight,                                 "AlignRight",                                       SSSE3,      -1,           16,           3,           {INS_palignr,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_IMM,                               HW_Flag_FullRangeIMM)
HARDWARE_INTRINSIC(SSSE3_HorizontalAdd,                              "HorizontalAdd",                                    SSSE3,      -1,           16,           2,           {INS_invalid,   INS_invalid,   INS_phaddw,    INS_invalid,   INS_phaddd,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSSE3_HorizontalAddSaturate,                      "HorizontalAddSaturate",                            SSSE3,      -1,           16,           2,           {INS_invalid,   INS_invalid,   INS_phaddsw,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSSE3_HorizontalSubtract,                         "HorizontalSubtract",                               SSSE3,      -1,           16,           2,           {INS_invalid,   INS_invalid,   INS_phsubw,    INS_invalid,   INS_phsubd,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSSE3_HorizontalSubtractSaturate,                 "HorizontalSubtractSaturate",                       SSSE3,      -1,           16,           2,           {INS_invalid,   INS_invalid,   INS_phsubsw,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSSE3_MultiplyAddAdjacent,                        "MultiplyAddAdjacent",                              SSSE3,      -1,           16,           2,           {INS_invalid,   INS_invalid,   INS_pmaddubsw, INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSSE3_MultiplyHighRoundScale,                     "MultiplyHighRoundScale",                           SSSE3,      -1,           16,           2,           {INS_invalid,   INS_invalid,   INS_pmulhrsw,  INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSSE3_Shuffle,                                    "Shuffle",                                          SSSE3,      -1,           16,           2,           {INS_pshufb,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSSE3_Sign,                                       "Sign",                                             SSSE3,      -1,           16,           2,           {INS_psignb,    INS_invalid,   INS_psignw,    INS_invalid,   INS_psignd,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)

// ************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                  Intrinsic ID                                      Function name                                      ISA        ival       SIMD size       NumArg                                                                   instructions                                                                                              Category                                         Flags
//                                                                                                                                                                            {TYP_BYTE,      TYP_UBYTE,     TYP_SHORT,     TYP_USHORT,     TYP_INT,       TYP_UINT,      TYP_LONG,      TYP_ULONG,     TYP_FLOAT,     TYP_DOUBLE}
// ************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  SSE41 Intrinsics
HARDWARE_INTRINSIC(SSE41_IsSupported,                                "get_IsSupported",                                  SSE41,      -1,           0,            0,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},          HW_Category_IsSupportedProperty,               HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE41_Blend,                                      "Blend",                                            SSE41,      -1,           16,           3,           {INS_invalid,   INS_invalid,   INS_pblendw,   INS_pblendw,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_blendps,   INS_blendpd},          HW_Category_IMM,                               HW_Flag_FullRangeIMM)
HARDWARE_INTRINSIC(SSE41_BlendVariable,                              "BlendVariable",                                    SSE41,      -1,           16,           3,           {INS_pblendvb,  INS_pblendvb,  INS_invalid,   INS_invalid,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_blendvps,  INS_blendvpd},         HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE41_Ceiling,                                    "Ceiling",                                          SSE41,      10,           16,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_roundps,   INS_roundpd},          HW_Category_SimpleSIMD,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE41_CeilingScalar,                              "CeilingScalar",                                    SSE41,      10,           16,          -1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_roundss,   INS_roundsd},          HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE41_CompareEqual,                               "CompareEqual",                                     SSE41,      -1,           16,           2,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,    INS_invalid,   INS_invalid,   INS_pcmpeqq,   INS_pcmpeqq,   INS_invalid,   INS_invalid},          HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(SSE41_ConvertToVector128Int16,                    "ConvertToVector128Int16",                          SSE41,      -1,           16,           1,           {INS_pmovsxbw,  INS_pmovzxbw,  INS_invalid,   INS_invalid,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},          HW_Category_SimpleSIMD,                        HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE41_ConvertToVector128Int32,                    "ConvertToVector128Int32",                          SSE41,      -1,           16,           1,           {INS_pmovsxbd,  INS_pmovzxbd,  INS_pmovsxwd,  INS_pmovzxwd,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},          HW_Category_SimpleSIMD,                        HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE41_ConvertToVector128Int64,                    "ConvertToVector128Int64",                          SSE41,      -1,           16,           1,           {INS_pmovsxbq,  INS_pmovzxbq,  INS_pmovsxwq,  INS_pmovzxwq,   INS_pmovsxdq,  INS_pmovzxdq,  INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},          HW_Category_SimpleSIMD,                        HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE41_DotProduct,                                 "DotProduct",                                       SSE41,      -1,           16,           3,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_dpps,      INS_dppd},             HW_Category_IMM,                               HW_Flag_FullRangeIMM)
HARDWARE_INTRINSIC(SSE41_Extract,                                    "Extract",                                          SSE41,      -1,           16,           2,           {INS_pextrb,    INS_pextrb,    INS_invalid,   INS_invalid,    INS_pextrd,    INS_pextrd,    INS_pextrq,    INS_pextrq,    INS_extractps, INS_invalid},          HW_Category_IMM,                               HW_Flag_FullRangeIMM|HW_Flag_BaseTypeFromFirstArg|HW_Flag_MultiIns|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE41_Floor,                                      "Floor",                                            SSE41,       9,           16,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_roundps,   INS_roundpd},          HW_Category_SimpleSIMD,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE41_FloorScalar,                                "FloorScalar",                                      SSE41,       9,           16,          -1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_roundss,   INS_roundsd},          HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE41_Insert,                                     "Insert",                                           SSE41,      -1,           16,           3,           {INS_pinsrb,    INS_pinsrb,    INS_invalid,   INS_invalid,    INS_pinsrd,    INS_pinsrd,    INS_pinsrq,    INS_pinsrq,    INS_insertps,  INS_invalid},          HW_Category_IMM,                               HW_Flag_FullRangeIMM|HW_Flag_SecondArgMaybe64Bit)
HARDWARE_INTRINSIC(SSE41_LoadAlignedVector128NonTemporal,            "LoadAlignedVector128NonTemporal",                  SSE41,      -1,           16,           1,           {INS_movntdqa,  INS_movntdqa,  INS_movntdqa,  INS_movntdqa,   INS_movntdqa,  INS_movntdqa,  INS_movntdqa,  INS_movntdqa,  INS_invalid,   INS_invalid},          HW_Category_MemoryLoad,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE41_Max,                                        "Max",                                              SSE41,      -1,           16,           2,           {INS_pmaxsb,    INS_invalid,   INS_invalid,   INS_pmaxuw,     INS_pmaxsd,    INS_pmaxud,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},          HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE41_Min,                                        "Min",                                              SSE41,      -1,           16,           2,           {INS_pminsb,    INS_invalid,   INS_invalid,   INS_pminuw,     INS_pminsd,    INS_pminud,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},          HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE41_MinHorizontal,                              "MinHorizontal",                                    SSE41,      -1,           16,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_phminposuw, INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},          HW_Category_SimpleSIMD,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE41_MultipleSumAbsoluteDifferences,             "MultipleSumAbsoluteDifferences",                   SSE41,      -1,           16,           3,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_mpsadbw,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},          HW_Category_IMM,                               HW_Flag_FullRangeIMM)
HARDWARE_INTRINSIC(SSE41_Multiply,                                   "Multiply",                                         SSE41,      -1,           16,           2,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,    INS_invalid,   INS_invalid,   INS_pmuldq,    INS_invalid,   INS_invalid,   INS_invalid},          HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(SSE41_MultiplyLow,                                "MultiplyLow",                                      SSE41,      -1,           16,           2,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,    INS_pmulld,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},          HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE41_PackUnsignedSaturate,                       "PackUnsignedSaturate",                             SSE41,      -1,           16,           2,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_packusdw,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},          HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE41_RoundCurrentDirection,                      "RoundCurrentDirection",                            SSE41,       4,           16,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_roundps,   INS_roundpd},          HW_Category_SimpleSIMD,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE41_RoundCurrentDirectionScalar,                "RoundCurrentDirectionScalar",                      SSE41,       4,           16,          -1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_roundss,   INS_roundsd},          HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE41_RoundToNearestInteger,                      "RoundToNearestInteger",                            SSE41,       8,           16,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_roundps,   INS_roundpd},          HW_Category_SimpleSIMD,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE41_RoundToNearestIntegerScalar,                "RoundToNearestIntegerScalar",                      SSE41,       8,           16,          -1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_roundss,   INS_roundsd},          HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE41_RoundToNegativeInfinity,                    "RoundToNegativeInfinity",                          SSE41,       9,           16,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_roundps,   INS_roundpd},          HW_Category_SimpleSIMD,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE41_RoundToNegativeInfinityScalar,              "RoundToNegativeInfinityScalar",                    SSE41,       9,           16,          -1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_roundss,   INS_roundsd},          HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE41_RoundToPositiveInfinity,                    "RoundToPositiveInfinity",                          SSE41,      10,           16,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_roundps,   INS_roundpd},          HW_Category_SimpleSIMD,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE41_RoundToPositiveInfinityScalar,              "RoundToPositiveInfinityScalar",                    SSE41,      10,           16,          -1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_roundss,   INS_roundsd},          HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE41_RoundToZero,                                "RoundToZero",                                      SSE41,      11,           16,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_roundps,   INS_roundpd},          HW_Category_SimpleSIMD,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE41_RoundToZeroScalar,                          "RoundToZeroScalar",                                SSE41,      11,           16,          -1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_roundss,   INS_roundsd},          HW_Category_SIMDScalar,                        HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE41_TestAllOnes,                                "TestAllOnes",                                      SSE41,      -1,           16,           1,           {INS_ptest,     INS_ptest,     INS_ptest,     INS_ptest,      INS_ptest,     INS_ptest,     INS_ptest,     INS_ptest,     INS_invalid,   INS_invalid},          HW_Category_SimpleSIMD,                        HW_Flag_MultiIns|HW_Flag_NoContainment|HW_Flag_NoRMWSemantics|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(SSE41_TestAllZeros,                               "TestAllZeros",                                     SSE41,      -1,           16,           2,           {INS_ptest,     INS_ptest,     INS_ptest,     INS_ptest,      INS_ptest,     INS_ptest,     INS_ptest,     INS_ptest,     INS_invalid,   INS_invalid},          HW_Category_SimpleSIMD,                        HW_Flag_MultiIns|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(SSE41_TestC,                                      "TestC",                                            SSE41,      -1,           16,           2,           {INS_ptest,     INS_ptest,     INS_ptest,     INS_ptest,      INS_ptest,     INS_ptest,     INS_ptest,     INS_ptest,     INS_invalid,   INS_invalid},          HW_Category_SimpleSIMD,                        HW_Flag_MultiIns|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(SSE41_TestMixOnesZeros,                           "TestMixOnesZeros",                                 SSE41,      -1,           16,           2,           {INS_ptest,     INS_ptest,     INS_ptest,     INS_ptest,      INS_ptest,     INS_ptest,     INS_ptest,     INS_ptest,     INS_invalid,   INS_invalid},          HW_Category_SimpleSIMD,                        HW_Flag_MultiIns|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(SSE41_TestNotZAndNotC,                            "TestNotZAndNotC",                                  SSE41,      -1,           16,           2,           {INS_ptest,     INS_ptest,     INS_ptest,     INS_ptest,      INS_ptest,     INS_ptest,     INS_ptest,     INS_ptest,     INS_invalid,   INS_invalid},          HW_Category_SimpleSIMD,                        HW_Flag_MultiIns|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(SSE41_TestZ,                                      "TestZ",                                            SSE41,      -1,           16,           2,           {INS_ptest,     INS_ptest,     INS_ptest,     INS_ptest,      INS_ptest,     INS_ptest,     INS_ptest,     INS_ptest,     INS_invalid,   INS_invalid},          HW_Category_SimpleSIMD,                        HW_Flag_MultiIns|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg)

//  SSE42 Intrinsics
HARDWARE_INTRINSIC(SSE42_IsSupported,                                "get_IsSupported",                                  SSE42,      -1,           0,            0,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_IsSupportedProperty,               HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE42_Crc32,                                      "Crc32",                                            SSE42,      -1,           0,            2,           {INS_invalid,   INS_crc32,     INS_invalid,   INS_crc32,     INS_invalid,   INS_crc32,     INS_invalid,   INS_crc32,     INS_invalid,   INS_invalid},           HW_Category_Scalar,                            HW_Flag_NoFloatingPointUsed)
HARDWARE_INTRINSIC(SSE42_CompareGreaterThan,                         "CompareGreaterThan",                               SSE42,      -1,           16,           2,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_pcmpgtq,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)

// ************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                  Intrinsic ID                                      Function name                                      ISA        ival       SIMD size       NumArg                                                                   instructions                                                                                              Category                                         Flags
//                                                                                                                                                                            {TYP_BYTE,      TYP_UBYTE,     TYP_SHORT,     TYP_USHORT,    TYP_INT,       TYP_UINT,      TYP_LONG,      TYP_ULONG,     TYP_FLOAT,     TYP_DOUBLE}
// ************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  AVX Intrinsics
HARDWARE_INTRINSIC(AVX_IsSupported,                                  "get_IsSupported",                                  AVX,        -1,            0,           0,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_IsSupportedProperty,               HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX_Add,                                          "Add",                                              AVX,        -1,           32,           2,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_addps,     INS_addpd},             HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(AVX_AddSubtract,                                  "AddSubtract",                                      AVX,        -1,           32,           2,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_addsubps,  INS_addsubpd},          HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX_And,                                          "And",                                              AVX,        -1,           32,           2,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_andps,     INS_andpd},             HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(AVX_AndNot,                                       "AndNot",                                           AVX,        -1,           32,           2,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_andnps,    INS_andnpd},            HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX_Blend,                                        "Blend",                                            AVX,        -1,           32,           3,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_blendps,   INS_blendpd},           HW_Category_IMM,                               HW_Flag_FullRangeIMM)
HARDWARE_INTRINSIC(AVX_BlendVariable,                                "BlendVariable",                                    AVX,        -1,           32,           3,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_vblendvps, INS_vblendvpd},         HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX_Ceiling,                                      "Ceiling",                                          AVX,        10,           32,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_roundps,   INS_roundpd},           HW_Category_SimpleSIMD,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(AVX_BroadcastScalarToVector128,                   "BroadcastScalarToVector128",                       AVX,        -1,           16,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_vbroadcastss, INS_invalid},        HW_Category_MemoryLoad,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX_BroadcastScalarToVector256,                   "BroadcastScalarToVector256",                       AVX,        -1,           32,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_vbroadcastss, INS_vbroadcastsd},   HW_Category_MemoryLoad,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX_BroadcastVector128ToVector256,                "BroadcastVector128ToVector256",                    AVX,        -1,           32,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,INS_vbroadcastf128,INS_vbroadcastf128},   HW_Category_MemoryLoad,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX_Compare,                                      "Compare",                                          AVX,        -1,           32,           3,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpps,     INS_cmppd},             HW_Category_IMM,                               HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX_CompareScalar,                                "CompareScalar",                                    AVX,        -1,           16,           3,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cmpss,     INS_cmpsd},             HW_Category_IMM,                               HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(AVX_ConvertToSingle,                              "ConvertToSingle",                                  AVX,        -1,           32,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movss,     INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(AVX_ConvertToVector128Int32,                      "ConvertToVector128Int32",                          AVX,        -1,           32,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cvtpd2dq,  INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(AVX_ConvertToVector128Single,                     "ConvertToVector128Single",                         AVX,        -1,           32,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cvtpd2ps,  INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(AVX_ConvertToVector256Int32,                      "ConvertToVector256Int32",                          AVX,        -1,           32,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cvtps2dq,  INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(AVX_ConvertToVector256Single,                     "ConvertToVector256Single",                         AVX,        -1,           32,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cvtdq2ps,  INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(AVX_ConvertToVector256Double,                     "ConvertToVector256Double",                         AVX,        -1,           32,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cvtdq2pd,  INS_invalid,   INS_invalid,   INS_invalid,   INS_cvtps2pd,  INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(AVX_ConvertToVector128Int32WithTruncation,        "ConvertToVector128Int32WithTruncation",            AVX,        -1,           32,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cvttpd2dq, INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(AVX_ConvertToVector256Int32WithTruncation,        "ConvertToVector256Int32WithTruncation",            AVX,        -1,           32,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_cvttps2dq, INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(AVX_Divide,                                       "Divide",                                           AVX,        -1,           32,           2,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_divps,     INS_divpd},             HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX_DotProduct,                                   "DotProduct",                                       AVX,        -1,           32,           3,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_dpps,      INS_invalid},           HW_Category_IMM,                               HW_Flag_FullRangeIMM)
HARDWARE_INTRINSIC(AVX_DuplicateEvenIndexed,                         "DuplicateEvenIndexed",                             AVX,        -1,           32,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movsldup,  INS_movddup},           HW_Category_SimpleSIMD,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(AVX_DuplicateOddIndexed,                          "DuplicateOddIndexed",                              AVX,        -1,           32,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movshdup,  INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(AVX_Extract,                                      "Extract",                                          AVX,        -1,           32,           2,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_IMM,                               HW_Flag_BaseTypeFromFirstArg|HW_Flag_FullRangeIMM|HW_Flag_NoCodeGen)
HARDWARE_INTRINSIC(AVX_ExtendToVector256,                            "ExtendToVector256",                                AVX,        -1,           32,           1,           {INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movaps,    INS_movapd},            HW_Category_Helper,                            HW_Flag_OneTypeGeneric|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(AVX_ExtractVector128,                             "ExtractVector128",                                 AVX,        -1,           32,          -1,           {INS_vextractf128,INS_vextractf128,INS_vextractf128,INS_vextractf128,INS_vextractf128,INS_vextractf128,INS_vextractf128,INS_vextractf128,INS_vextractf128, INS_vextractf128},HW_Category_IMM,                  HW_Flag_OneTypeGeneric|HW_Flag_SpecialImport|HW_Flag_SpecialCodeGen|HW_Flag_FullRangeIMM)
HARDWARE_INTRINSIC(AVX_Floor,                                        "Floor",                                            AVX,         9,           32,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_roundps,   INS_roundpd},           HW_Category_SimpleSIMD,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(AVX_GetLowerHalf,                                 "GetLowerHalf",                                     AVX,        -1,           32,           1,           {INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movaps,    INS_movapd},            HW_Category_Helper,                            HW_Flag_OneTypeGeneric|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(AVX_HorizontalAdd,                                "HorizontalAdd",                                    AVX,        -1,           32,           2,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_haddps,    INS_haddpd},            HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX_HorizontalSubtract,                           "HorizontalSubtract",                               AVX,        -1,           32,           2,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_hsubps,    INS_hsubpd},            HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX_Insert,                                       "Insert",                                           AVX,        -1,           32,           3,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_IMM,                               HW_Flag_FullRangeIMM|HW_Flag_NoCodeGen|HW_Flag_SecondArgMaybe64Bit)
HARDWARE_INTRINSIC(AVX_InsertVector128,                              "InsertVector128",                                  AVX,        -1,           32,           3,           {INS_vinsertf128,INS_vinsertf128,INS_vinsertf128,INS_vinsertf128,INS_vinsertf128,INS_vinsertf128,INS_vinsertf128,INS_vinsertf128,INS_vinsertf128, INS_vinsertf128},HW_Category_IMM,                            HW_Flag_FullRangeIMM|HW_Flag_OneTypeGeneric|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX_LoadAlignedVector256,                         "LoadAlignedVector256",                             AVX,        -1,           32,           1,           {INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movaps,    INS_movapd},            HW_Category_MemoryLoad,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(AVX_LoadDquVector256,                             "LoadDquVector256",                                 AVX,        -1,           32,           1,           {INS_lddqu,     INS_lddqu,     INS_lddqu,     INS_lddqu,     INS_lddqu,     INS_lddqu,     INS_lddqu,     INS_lddqu,     INS_invalid,   INS_invalid},           HW_Category_MemoryLoad,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(AVX_LoadVector256,                                "LoadVector256",                                    AVX,        -1,           32,           1,           {INS_movdqu,    INS_movdqu,    INS_movdqu,    INS_movdqu,    INS_movdqu,    INS_movdqu,    INS_movdqu,    INS_movdqu,    INS_movups,    INS_movupd},            HW_Category_MemoryLoad,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(AVX_Max,                                          "Max",                                              AVX,        -1,           32,           2,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_maxps,     INS_maxpd},             HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(AVX_Min,                                          "Min",                                              AVX,        -1,           32,           2,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_minps,     INS_minpd},             HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(AVX_MaskLoad,                                     "MaskLoad",                                         AVX,        -1,           0,            2,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_vmaskmovps,INS_vmaskmovpd},        HW_Category_MemoryLoad,                        HW_Flag_UnfixedSIMDSize)
HARDWARE_INTRINSIC(AVX_MoveMask,                                     "MoveMask",                                         AVX,        -1,           32,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_movmskps,  INS_movmskpd},          HW_Category_SimpleSIMD,                        HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(AVX_Multiply,                                     "Multiply",                                         AVX,        -1,           32,           2,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_mulps,     INS_mulpd},             HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(AVX_Or,                                           "Or",                                               AVX,        -1,           32,           2,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_orps,      INS_orpd},              HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(AVX_Permute,                                      "Permute",                                          AVX,        -1,           0,            2,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_vpermilps, INS_vpermilpd},         HW_Category_IMM,                               HW_Flag_FullRangeIMM|HW_Flag_UnfixedSIMDSize)
HARDWARE_INTRINSIC(AVX_Permute2x128,                                 "Permute2x128",                                     AVX,        -1,           32,           3,           {INS_vperm2f128,INS_vperm2f128,INS_vperm2f128,INS_vperm2f128,INS_vperm2f128,INS_vperm2f128,INS_vperm2f128,INS_vperm2f128,INS_vperm2f128,INS_vperm2f128},        HW_Category_IMM,                               HW_Flag_OneTypeGeneric|HW_Flag_FullRangeIMM)
HARDWARE_INTRINSIC(AVX_PermuteVar,                                   "PermuteVar",                                       AVX,        -1,           0,            2,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_vpermilpsvar,INS_vpermilpdvar},    HW_Category_SimpleSIMD,                        HW_Flag_UnfixedSIMDSize)
HARDWARE_INTRINSIC(AVX_Reciprocal,                                   "Reciprocal",                                       AVX,        -1,           32,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_rcpps,     INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(AVX_ReciprocalSqrt,                               "ReciprocalSqrt",                                   AVX,        -1,           32,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_rsqrtps,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(AVX_RoundCurrentDirection,                        "RoundCurrentDirection",                            AVX,         4,           32,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_roundps,   INS_roundpd},           HW_Category_SimpleSIMD,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(AVX_RoundToNearestInteger,                        "RoundToNearestInteger",                            AVX,         8,           32,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_roundps,   INS_roundpd},           HW_Category_SimpleSIMD,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(AVX_RoundToNegativeInfinity,                      "RoundToNegativeInfinity",                          AVX,         9,           32,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_roundps,   INS_roundpd},           HW_Category_SimpleSIMD,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(AVX_RoundToPositiveInfinity,                      "RoundToPositiveInfinity",                          AVX,        10,           32,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_roundps,   INS_roundpd},           HW_Category_SimpleSIMD,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(AVX_RoundToZero,                                  "RoundToZero",                                      AVX,        11,           32,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_roundps,   INS_roundpd},           HW_Category_SimpleSIMD,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(AVX_SetVector256,                                 "SetVector256",                                     AVX,        -1,           32,          -1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_Helper,                            HW_Flag_NoCodeGen|HW_Flag_SecondArgMaybe64Bit)
HARDWARE_INTRINSIC(AVX_SetHighLow,                                   "SetHighLow",                                       AVX,        -1,           32,           2,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_Helper,                            HW_Flag_NoCodeGen|HW_Flag_OneTypeGeneric)
HARDWARE_INTRINSIC(AVX_SetAllVector256,                              "SetAllVector256",                                  AVX,        -1,           32,           1,           {INS_vpbroadcastb,INS_vpbroadcastb,INS_vpbroadcastw,INS_vpbroadcastw,INS_vpbroadcastd,INS_vpbroadcastd,INS_vpbroadcastq,INS_vpbroadcastq,INS_vbroadcastss,INS_vbroadcastsd},HW_Category_Helper,                HW_Flag_MultiIns|HW_Flag_SpecialImport|HW_Flag_OneTypeGeneric)
HARDWARE_INTRINSIC(AVX_SetZeroVector256,                             "SetZeroVector256",                                 AVX,        -1,           32,           0,           {INS_pxor,      INS_pxor,      INS_pxor,      INS_pxor,      INS_pxor,      INS_pxor,      INS_pxor,      INS_pxor,      INS_xorps,     INS_xorpd},             HW_Category_Helper,                            HW_Flag_OneTypeGeneric|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(AVX_Shuffle,                                      "Shuffle",                                          AVX,        -1,           32,           3,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_shufps,    INS_shufpd},            HW_Category_IMM,                               HW_Flag_NoRMWSemantics|HW_Flag_FullRangeIMM)
HARDWARE_INTRINSIC(AVX_Sqrt,                                         "Sqrt",                                             AVX,        -1,           32,           1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_sqrtps,    INS_sqrtpd},            HW_Category_SimpleSIMD,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(AVX_StaticCast,                                   "StaticCast",                                       AVX,        -1,           32,           1,           {INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movaps,    INS_movapd},            HW_Category_Helper,                            HW_Flag_TwoTypeGeneric|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(AVX_Store,                                        "Store",                                            AVX,        -1,           32,           2,           {INS_movdqu,    INS_movdqu,    INS_movdqu,    INS_movdqu,    INS_movdqu,    INS_movdqu,    INS_movdqu,    INS_movdqu,    INS_movups,    INS_movupd},            HW_Category_MemoryStore,                       HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(AVX_StoreAligned,                                 "StoreAligned",                                     AVX,        -1,           32,           2,           {INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movdqa,    INS_movaps,    INS_movapd},            HW_Category_MemoryStore,                       HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(AVX_StoreAlignedNonTemporal,                      "StoreAlignedNonTemporal",                          AVX,        -1,           32,           2,           {INS_movntdq,   INS_movntdq,   INS_movntdq,   INS_movntdq,   INS_movntdq,   INS_movntdq,   INS_movntdq,   INS_movntdq,   INS_movntps,   INS_movntpd},           HW_Category_MemoryStore,                       HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(AVX_Subtract,                                     "Subtract",                                         AVX,        -1,           32,           2,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_subps,     INS_subpd},             HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX_TestC,                                        "TestC",                                            AVX,        -1,            0,           2,           {INS_ptest,     INS_ptest,     INS_ptest,     INS_ptest,     INS_ptest,     INS_ptest,     INS_ptest,     INS_ptest,     INS_vtestps,   INS_vtestpd},           HW_Category_SimpleSIMD,                        HW_Flag_OneTypeGeneric|HW_Flag_UnfixedSIMDSize|HW_Flag_MultiIns|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(AVX_TestNotZAndNotC,                              "TestNotZAndNotC",                                  AVX,        -1,            0,           2,           {INS_ptest,     INS_ptest,     INS_ptest,     INS_ptest,     INS_ptest,     INS_ptest,     INS_ptest,     INS_ptest,     INS_vtestps,   INS_vtestpd},           HW_Category_SimpleSIMD,                        HW_Flag_OneTypeGeneric|HW_Flag_UnfixedSIMDSize|HW_Flag_MultiIns|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(AVX_TestZ,                                        "TestZ",                                            AVX,        -1,            0,           2,           {INS_ptest,     INS_ptest,     INS_ptest,     INS_ptest,     INS_ptest,     INS_ptest,     INS_ptest,     INS_ptest,     INS_vtestps,   INS_vtestpd},           HW_Category_SimpleSIMD,                        HW_Flag_OneTypeGeneric|HW_Flag_UnfixedSIMDSize|HW_Flag_MultiIns|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(AVX_UnpackHigh,                                   "UnpackHigh",                                       AVX,        -1,           32,           2,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_unpckhps,  INS_unpckhpd},          HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX_UnpackLow,                                    "UnpackLow",                                        AVX,        -1,           32,           2,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_unpcklps,  INS_unpcklpd},          HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX_Xor,                                          "Xor",                                              AVX,        -1,           32,           2,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_xorps,     INS_xorpd},             HW_Category_SimpleSIMD,                        HW_Flag_Commutative)

//  AVX2 Intrinsics
HARDWARE_INTRINSIC(AVX2_IsSupported,                                 "get_IsSupported",                                  AVX2,       -1,           0,            0,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_IsSupportedProperty,               HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX2_Abs,                                         "Abs",                                              AVX2,       -1,           32,           1,           {INS_pabsb,     INS_pabsb,     INS_pabsw,     INS_pabsw,     INS_pabsd,     INS_pabsd,     INS_paddq,     INS_paddq,     INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(AVX2_Add,                                         "Add",                                              AVX2,       -1,           32,           2,           {INS_paddb,     INS_paddb,     INS_paddw,     INS_paddw,     INS_paddd,     INS_paddd,     INS_paddq,     INS_paddq,     INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(AVX2_AddSaturate,                                 "AddSaturate",                                      AVX2,       -1,           32,           2,           {INS_paddsb,    INS_paddusb,   INS_paddsw,    INS_paddusw,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(AVX2_And,                                         "And",                                              AVX2,       -1,           32,           2,           {INS_pand,      INS_pand,      INS_pand,      INS_pand,      INS_pand,      INS_pand,      INS_pand,      INS_pand,      INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(AVX2_AndNot,                                      "AndNot",                                           AVX2,       -1,           32,           2,           {INS_pandn,     INS_pandn,     INS_pandn,     INS_pandn,     INS_pandn,     INS_pandn,     INS_pandn,     INS_pandn,     INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX2_Average,                                     "Average",                                          AVX2,       -1,           32,           2,           {INS_invalid,   INS_pavgb,     INS_invalid,   INS_pavgw,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(AVX2_BlendVariable,                               "BlendVariable",                                    AVX2,       -1,           32,           3,           {INS_vpblendvb, INS_vpblendvb, INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX2_BroadcastScalarToVector128,                  "BroadcastScalarToVector128",                       AVX2,       -1,           16,           1,           {INS_vpbroadcastb,INS_vpbroadcastb,INS_vpbroadcastw,INS_vpbroadcastw,INS_vpbroadcastd,INS_vpbroadcastd,INS_vpbroadcastq,INS_vpbroadcastq,INS_vbroadcastss,INS_movddup},        HW_Category_SimpleSIMD,         HW_Flag_OneTypeGeneric)
HARDWARE_INTRINSIC(AVX2_BroadcastScalarToVector256,                  "BroadcastScalarToVector256",                       AVX2,       -1,           32,           1,           {INS_vpbroadcastb,INS_vpbroadcastb,INS_vpbroadcastw,INS_vpbroadcastw,INS_vpbroadcastd,INS_vpbroadcastd,INS_vpbroadcastq,INS_vpbroadcastq,INS_vbroadcastss,INS_vbroadcastsd},   HW_Category_SimpleSIMD,         HW_Flag_OneTypeGeneric)
HARDWARE_INTRINSIC(AVX2_BroadcastVector128ToVector256,               "BroadcastVector128ToVector256",                    AVX2,       -1,           32,           1,           {INS_vbroadcasti128,INS_vbroadcasti128,INS_vbroadcasti128,INS_vbroadcasti128,INS_vbroadcasti128,INS_vbroadcasti128,INS_vbroadcasti128,INS_vbroadcasti128,INS_invalid,INS_invalid},HW_Category_MemoryLoad,      HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX2_CompareEqual,                                "CompareEqual",                                     AVX2,       -1,           32,           2,           {INS_pcmpeqb,   INS_pcmpeqb,   INS_pcmpeqw,   INS_pcmpeqw,   INS_pcmpeqd,   INS_pcmpeqd,   INS_pcmpeqq,   INS_pcmpeqq,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(AVX2_CompareGreaterThan,                          "CompareGreaterThan",                               AVX2,       -1,           32,           2,           {INS_pcmpgtb,   INS_invalid,   INS_pcmpgtw,   INS_invalid,   INS_pcmpgtd,   INS_invalid,   INS_pcmpgtq,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX2_ExtractVector128,                            "ExtractVector128",                                 AVX2,       -1,           32,          -1,           {INS_vextracti128,INS_vextracti128,INS_vextracti128,INS_vextracti128,INS_vextracti128,INS_vextracti128,INS_vextracti128,INS_vextracti128,INS_invalid, INS_invalid},HW_Category_IMM,                            HW_Flag_SpecialImport|HW_Flag_SpecialCodeGen|HW_Flag_FullRangeIMM)
HARDWARE_INTRINSIC(AVX2_ConvertToVector256Int16,                     "ConvertToVector256Int16",                          AVX2,       -1,           32,           1,           {INS_pmovsxbw,  INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(AVX2_ConvertToVector256UInt16,                    "ConvertToVector256UInt16",                         AVX2,       -1,           32,           1,           {INS_invalid,   INS_pmovzxbw,  INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(AVX2_ConvertToVector256Int32,                     "ConvertToVector256Int32",                          AVX2,       -1,           32,           1,           {INS_pmovsxbd,  INS_invalid,   INS_pmovsxwd,  INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(AVX2_ConvertToVector256UInt32,                    "ConvertToVector256UInt32",                         AVX2,       -1,           32,           1,           {INS_invalid,   INS_pmovzxbd,  INS_invalid,   INS_pmovzxwd,  INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(AVX2_ConvertToVector256Int64,                     "ConvertToVector256Int64",                          AVX2,       -1,           32,           1,           {INS_pmovsxbq,  INS_invalid,   INS_pmovsxwq,  INS_invalid,   INS_pmovsxdq,  INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(AVX2_ConvertToVector256UInt64,                    "ConvertToVector256UInt64",                         AVX2,       -1,           32,           1,           {INS_invalid,   INS_pmovzxbq,  INS_invalid,   INS_pmovzxwq,  INS_invalid,   INS_pmovzxdq,  INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(AVX2_HorizontalAdd,                               "HorizontalAdd",                                    AVX2,       -1,           32,           2,           {INS_invalid,   INS_invalid,   INS_phaddw,    INS_invalid,   INS_phaddd,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX2_HorizontalAddSaturate,                       "HorizontalAddSaturate",                            AVX2,       -1,           32,           2,           {INS_invalid,   INS_invalid,   INS_phaddsw,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX2_HorizontalSubtract,                          "HorizontalSubtract",                               AVX2,       -1,           32,           2,           {INS_invalid,   INS_invalid,   INS_phsubw,    INS_invalid,   INS_phsubd,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX2_HorizontalSubtractSaturate,                  "HorizontalSubtractSaturate",                       AVX2,       -1,           32,           2,           {INS_invalid,   INS_invalid,   INS_phsubsw,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX2_InsertVector128,                             "InsertVector128",                                  AVX2,       -1,           32,           3,           {INS_vinserti128,INS_vinserti128,INS_vinserti128,INS_vinserti128,INS_vinserti128,INS_vinserti128,INS_vinserti128,INS_vinserti128,INS_invalid, INS_invalid},     HW_Category_IMM,                               HW_Flag_FullRangeIMM|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX2_LoadAlignedVector256NonTemporal,             "LoadAlignedVector256NonTemporal",                  AVX2,       -1,           32,           1,           {INS_movntdqa,  INS_movntdqa,  INS_movntdqa,  INS_movntdqa,  INS_movntdqa,  INS_movntdqa,  INS_movntdqa,  INS_movntdqa,  INS_invalid,   INS_invalid},           HW_Category_MemoryLoad,                        HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(AVX2_Max,                                         "Max",                                              AVX2,       -1,           32,           2,           {INS_pmaxsb,    INS_pmaxub,    INS_pmaxsw,    INS_pmaxuw,    INS_pmaxsd,    INS_pmaxud,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(AVX2_Min,                                         "Min",                                              AVX2,       -1,           32,           2,           {INS_pminsb,    INS_pminub,    INS_pminsw,    INS_pminuw,    INS_pminsd,    INS_pminud,    INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(AVX2_MoveMask,                                    "MoveMask",                                         AVX2,       -1,           32,           1,           {INS_pmovmskb,  INS_pmovmskb,  INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},            HW_Category_SimpleSIMD,                        HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(AVX2_Multiply,                                    "Multiply",                                         AVX2,       -1,           32,           2,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_pmuldq,    INS_pmuludq,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(AVX2_Or,                                          "Or",                                               AVX2,       -1,           32,           2,           {INS_por,       INS_por,       INS_por,       INS_por,       INS_por,       INS_por,       INS_por,       INS_por,       INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_Commutative)
HARDWARE_INTRINSIC(AVX2_Permute2x128,                                "Permute2x128",                                     AVX2,       -1,           32,           3,           {INS_vperm2i128,INS_vperm2i128,INS_vperm2i128,INS_vperm2i128,INS_vperm2i128,INS_vperm2i128,INS_vperm2i128,INS_vperm2i128,INS_invalid,   INS_invalid},           HW_Category_IMM,                               HW_Flag_OneTypeGeneric|HW_Flag_FullRangeIMM)
HARDWARE_INTRINSIC(AVX2_ShiftLeftLogical,                            "ShiftLeftLogical",                                 AVX2,       -1,           32,           2,           {INS_invalid,   INS_invalid,   INS_psllw,     INS_psllw,     INS_pslld,     INS_pslld,     INS_psllq,     INS_psllq,     INS_invalid,   INS_invalid},           HW_Category_IMM,                               HW_Flag_MaybeIMM|HW_Flag_NoJmpTableIMM|HW_Flag_FullRangeIMM)
HARDWARE_INTRINSIC(AVX2_ShiftLeftLogical128BitLane,                  "ShiftLeftLogical128BitLane",                       AVX2,       -1,           32,           2,           {INS_pslldq,    INS_pslldq,    INS_pslldq,    INS_pslldq,    INS_pslldq,    INS_pslldq,    INS_pslldq,    INS_pslldq,    INS_invalid,   INS_invalid},           HW_Category_IMM,                               HW_Flag_FullRangeIMM)
HARDWARE_INTRINSIC(AVX2_ShiftLeftLogicalVariable,                    "ShiftLeftLogicalVariable",                         AVX2,       -1,            0,           2,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_vpsllvd,   INS_vpsllvd,   INS_vpsllvq,   INS_vpsllvq,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_UnfixedSIMDSize|HW_Flag_NoContainment)
HARDWARE_INTRINSIC(AVX2_ShiftRightArithmetic,                        "ShiftRightArithmetic",                             AVX2,       -1,           32,           2,           {INS_invalid,   INS_invalid,   INS_psraw,     INS_invalid,   INS_psrad,     INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_IMM,                               HW_Flag_MaybeIMM|HW_Flag_NoJmpTableIMM|HW_Flag_FullRangeIMM)
HARDWARE_INTRINSIC(AVX2_ShiftRightLogical,                           "ShiftRightLogical",                                AVX2,       -1,           32,           2,           {INS_invalid,   INS_invalid,   INS_psrlw,     INS_psrlw,     INS_psrld,     INS_psrld,     INS_psrlq,     INS_psrlq,     INS_invalid,   INS_invalid},           HW_Category_IMM,                               HW_Flag_MaybeIMM|HW_Flag_NoJmpTableIMM|HW_Flag_FullRangeIMM)
HARDWARE_INTRINSIC(AVX2_ShiftRightLogical128BitLane,                 "ShiftRightLogical128BitLane",                      AVX2,       -1,           32,           2,           {INS_psrldq,    INS_psrldq,    INS_psrldq,    INS_psrldq,    INS_psrldq,    INS_psrldq,    INS_psrldq,    INS_psrldq,    INS_invalid,   INS_invalid},           HW_Category_IMM,                               HW_Flag_FullRangeIMM)
HARDWARE_INTRINSIC(AVX2_ShiftRightLogicalVariable,                   "ShiftRightLogicalVariable",                        AVX2,       -1,            0,           2,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_vpsrlvd,   INS_vpsrlvd,   INS_vpsrlvq,   INS_vpsrlvq,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_UnfixedSIMDSize|HW_Flag_NoContainment)
HARDWARE_INTRINSIC(AVX2_Subtract,                                    "Subtract",                                         AVX2,       -1,           32,           2,           {INS_psubb,     INS_psubb,     INS_psubw,     INS_psubw,     INS_psubd,     INS_psubd,     INS_psubq,     INS_psubq,     INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX2_SubtractSaturate,                            "SubtractSaturate",                                 AVX2,       -1,           32,           2,           {INS_psubsb,    INS_psubusb,   INS_psubsw,    INS_psubusw,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX2_UnpackHigh,                                  "UnpackHigh",                                       AVX2,       -1,           32,           2,           {INS_punpckhbw, INS_punpckhbw, INS_punpckhwd, INS_punpckhwd, INS_punpckhdq, INS_punpckhdq, INS_punpckhqdq,INS_punpckhqdq,INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX2_UnpackLow,                                   "UnpackLow",                                        AVX2,       -1,           32,           2,           {INS_punpcklbw, INS_punpcklbw, INS_punpcklwd, INS_punpcklwd, INS_punpckldq, INS_punpckldq, INS_punpcklqdq,INS_punpcklqdq,INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX2_Xor,                                         "Xor",                                              AVX2,       -1,           32,           2,           {INS_pxor,      INS_pxor,      INS_pxor,      INS_pxor,      INS_pxor,      INS_pxor,      INS_pxor,      INS_pxor,      INS_invalid,   INS_invalid},           HW_Category_SimpleSIMD,                        HW_Flag_Commutative)

//  AES Intrinsics
HARDWARE_INTRINSIC(AES_IsSupported,                                  "get_IsSupported",                                  AES,        -1,           0,            0,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_IsSupportedProperty,               HW_Flag_NoFlag)

//  BMI1 Intrinsics
HARDWARE_INTRINSIC(BMI1_IsSupported,                                 "get_IsSupported",                                  BMI1,       -1,           0,            0,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_IsSupportedProperty,               HW_Flag_NoFlag)

//  BMI2 Intrinsics
HARDWARE_INTRINSIC(BMI2_IsSupported,                                 "get_IsSupported",                                  BMI2,       -1,           0,            0,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_IsSupportedProperty,               HW_Flag_NoFlag)

//  FMA Intrinsics
HARDWARE_INTRINSIC(FMA_IsSupported,                                  "get_IsSupported",                                  FMA,        -1,           0,            0,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_IsSupportedProperty,               HW_Flag_NoFlag)

//  LZCNT Intrinsics
HARDWARE_INTRINSIC(LZCNT_IsSupported,                                "get_IsSupported",                                  LZCNT,      -1,           0,            0,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_IsSupportedProperty,               HW_Flag_NoFlag)
HARDWARE_INTRINSIC(LZCNT_LeadingZeroCount,                           "LeadingZeroCount",                                 LZCNT,      -1,           0,            1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_lzcnt,     INS_invalid,   INS_lzcnt,     INS_invalid,   INS_invalid},           HW_Category_Scalar,                            HW_Flag_NoFloatingPointUsed|HW_Flag_NoRMWSemantics)

//  PCLMULQDQ Intrinsics
HARDWARE_INTRINSIC(PCLMULQDQ_IsSupported,                            "get_IsSupported",                                  PCLMULQDQ,  -1,           0,            0,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_IsSupportedProperty,               HW_Flag_NoFlag)

//  POPCNT Intrinsics
HARDWARE_INTRINSIC(POPCNT_IsSupported,                               "get_IsSupported",                                  POPCNT,     -1,           0,            0,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_IsSupportedProperty,               HW_Flag_NoFlag)
HARDWARE_INTRINSIC(POPCNT_PopCount,                                  "PopCount",                                         POPCNT,     -1,           0,            1,           {INS_invalid,   INS_invalid,   INS_invalid,   INS_invalid,   INS_popcnt,    INS_invalid,   INS_popcnt,    INS_invalid,   INS_invalid,   INS_invalid},           HW_Category_Scalar,                            HW_Flag_NoFloatingPointUsed|HW_Flag_NoRMWSemantics)
#endif // FEATURE_HW_INTRINSIC

#undef HARDWARE_INTRINSIC

// clang-format on
