// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.

// Helper functionality to locate inputs and find outputs for
// k-nucleotide benchmark in CoreCLR test harness

using System;
using System.IO;
using System.Text;

namespace BenchmarksGame
{
    class TestHarnessHelpers
    {
        public string InputFile;
        public int[] expectedCountLetter;
        public int[] expectedCountPairs;
        public int[] expectedCountFragments;
        public int[][] expectedFrequencies;

        public TestHarnessHelpers(bool bigInput, [System.Runtime.CompilerServices.CallerFilePath] string csFileName = "")
        {
            if (bigInput)
            {
                InputFile = FindInputFile("knucleotide-input-big.txt", csFileName);
                expectedCountLetter = new int[] { 302923, 301375, 198136, 197566 };
                expectedCountPairs = new int[] { 91779, 91253, 91225, 90837, 60096, 60030, 59889, 59795, 59756, 59713, 59572, 59557, 39203, 39190, 39081, 39023 };
                expectedCountFragments = new int[] { 11765, 3572, 380, 7, 7 };
            }
            else
            {
                InputFile = FindInputFile("knucleotide-input.txt", csFileName);
                expectedCountLetter = new int[] { 1576, 1480, 974, 970 };
                expectedCountPairs = new int[] { 496, 480, 470, 420, 316, 315, 310, 302, 298, 292, 273, 272, 202, 201, 185, 167 };
                expectedCountFragments = new int[] { 54, 24, 4, 0, 0 };
            }
            expectedFrequencies = new int[][] { expectedCountLetter, expectedCountPairs };
        }

        public string FindInputFile(string inputFile, string csFileName)
        {
            string CoreRoot = System.Environment.GetEnvironmentVariable("CORE_ROOT");

            if (CoreRoot == null)
            {
                Console.WriteLine("This benchmark requries CORE_ROOT to be set");
                return null;
            }

            // The convention is that the csproj file has the same name as the cs file.
            string projectName = Path.GetFileNameWithoutExtension(csFileName);
            int slashIndex = projectName.LastIndexOfAny(new char[] { '/', '\\' });
            if (slashIndex != -1)
            {
                // csFileName was generated by the C# compiler, which may have run on
                // a different host system with different path separator than the
                // currently executing host, which dictates GetFileNameWithoutExtension's
                // behavior... so hope that the slash here is a cross-host path separator,
                // and chop of what were likely direcotires.
                projectName = projectName.Substring(slashIndex + 1);
            }

            // Normal testing -- input file will end up next to the assembly
            // and CoreRoot points at the test overlay dir
            string[] pathPartsNormal = new string[] {
                   CoreRoot, "..", "..", "JIT", "Performance",
                    "CodeQuality", "BenchmarksGame", "k-nucleotide", projectName, inputFile
                };

            string inputPathNormal = Path.Combine(pathPartsNormal);

            // Perf testing -- input file will end up next to the assembly
            // and CoreRoot points at this directory
            string[] pathPartsPerf = new string[] { CoreRoot, inputFile };

            string inputPathPerf = Path.Combine(pathPartsPerf);

            string inputPath = null;

            if (File.Exists(inputPathNormal))
            {
                inputPath = inputPathNormal;
            }
            else if (File.Exists(inputPathPerf))
            {
                inputPath = inputPathPerf;
            }

            if (inputPath != null)
            {
                Console.WriteLine("Using input file {0}", inputFile);
            }
            else
            {
                throw new Exception($"Unable to find input file {inputFile}.  Tried {inputPathNormal} and {inputPathPerf}; csFileName was {csFileName}, so projectName was {projectName}.");
            }

            return inputPath;
        }
    }
}
