# Basic workflow

The `editables` project is designed to support *build backends*, allowing them
to declare what they wish to expose as "editable", and returning a list of
support files that need to be included in the wheel generated by the
`build_editable` [backend hook](https://peps.python.org/pep-0660/#build-editable).
Note that the `editables` library does not build wheel files directly - it
returns the content that needs to be added to the wheel, but it is the build
backend's responsibility to actually create the wheel from that data.

## Create a project

The first step is for the backend to create an "editable project". The project
name must follow the normal rules for Python project names from
[PEP 426](https://peps.python.org/pep-0426/#name).

```python
project = EditableProject("myproject")
```

## Specify what to expose

Once the project has been created, the backend can specify which files should be
exposed when the editable install is done. There are two mechanisms currently
implemented for this.

To add a particular directory (typically the project's "src" directory) to
`sys.path` at runtime, simply call the `add_to_path` method

```python
project.add_to_path("src")
```

This will simply write the given directory into the `.pth` file added to the
wheel. See the "Implementation Details" section for more information. Note that
this method requires no runtime support.

To expose a single `.py` file as a module, call the `map` method, giving the
name by which the module can be imported, and the path to the implementation
`.py` file. It *is* possible to give the module a name that is not the same as
the implementation filename, although this is expected to be extremely uncommon.

```python
project.map("module", "src/module.py")
```

To expose a directory as a package, the `map` method is used in precisely the
same way, but with the directory name:

```python
project.map("mypackage", "src/mypackage")
```

Using the `map` method does require a runtime support module.

## Build the wheel

### Files to add

Once all of the content to expose is specified, the backend can start building
the wheel. To determine what files to write to the wheel, the `files` method
should be used. This returns a sequence of pairs, each of which specifies a
filename, and the content to write to that file. Both the name and the content
are strings, and so should be encoded appropriately (i.e., in UTF-8) when
writing to the wheel.

```python
for name, content in my_project.files():
    wheel.add_file(name, content)
```

### Runtime dependencies

If the `map` method is used, the resulting wheel will require that the runtime
support module is installed. To ensure that is the case, dependency metadata
must be added to the wheel. The `dependencies` method provides the required
metadata.

```python
for dep in my_project.dependencies():
    wheel.metadata.dependencies.add(dep)
```

Note that if the backend only uses the `add_to_path` method, no runtime support
is needed, so the `dependencies` method will return an empty list. For safety,
and to protect against future changes, it should still be called, though.
