package integration

import (
	"os"

	. "github.com/containers/podman/v3/test/utils"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
)

var _ = Describe("Podman run passwd", func() {
	var (
		tempdir    string
		err        error
		podmanTest *PodmanTestIntegration
	)

	BeforeEach(func() {
		tempdir, err = CreateTempDirInTempDir()
		if err != nil {
			os.Exit(1)
		}
		podmanTest = PodmanTestCreate(tempdir)
		podmanTest.Setup()
		podmanTest.SeedImages()
	})

	AfterEach(func() {
		podmanTest.Cleanup()
		f := CurrentGinkgoTestDescription()
		processTestResult(f)

	})

	It("podman run no user specified ", func() {
		session := podmanTest.Podman([]string{"run", "--read-only", BB, "mount"})
		session.WaitWithDefaultTimeout()
		Expect(session.ExitCode()).To(Equal(0))
		Expect(session.LineInOutputContains("passwd")).To(BeFalse())
	})
	It("podman run user specified in container", func() {
		session := podmanTest.Podman([]string{"run", "--read-only", "-u", "bin", BB, "mount"})
		session.WaitWithDefaultTimeout()
		Expect(session.ExitCode()).To(Equal(0))
		Expect(session.LineInOutputContains("passwd")).To(BeFalse())
	})

	It("podman run UID specified in container", func() {
		session := podmanTest.Podman([]string{"run", "--read-only", "-u", "2:1", BB, "mount"})
		session.WaitWithDefaultTimeout()
		Expect(session.ExitCode()).To(Equal(0))
		Expect(session.LineInOutputContains("passwd")).To(BeFalse())
	})

	It("podman run UID not specified in container", func() {
		session := podmanTest.Podman([]string{"run", "--read-only", "-u", "20001:1", BB, "mount"})
		session.WaitWithDefaultTimeout()
		Expect(session.ExitCode()).To(Equal(0))
		Expect(session.LineInOutputContains("passwd")).To(BeTrue())
	})

	It("podman can run container without /etc/passwd", func() {
		dockerfile := `FROM alpine
RUN rm -f /etc/passwd /etc/shadow /etc/group
USER 1000`
		imgName := "testimg"
		podmanTest.BuildImage(dockerfile, imgName, "false")
		session := podmanTest.Podman([]string{"run", "--rm", imgName, "ls", "/etc/"})
		session.WaitWithDefaultTimeout()
		Expect(session.ExitCode()).To(Equal(0))
		Expect(session.OutputToString()).To(Not(ContainSubstring("passwd")))
	})

	It("podman run with no user specified does not change --group specified", func() {
		session := podmanTest.Podman([]string{"run", "--read-only", BB, "mount"})
		session.WaitWithDefaultTimeout()
		Expect(session.ExitCode()).To(Equal(0))
		Expect(session.LineInOutputContains("/etc/group")).To(BeFalse())
	})

	It("podman run group specified in container", func() {
		session := podmanTest.Podman([]string{"run", "--read-only", "-u", "root:bin", BB, "mount"})
		session.WaitWithDefaultTimeout()
		Expect(session.ExitCode()).To(Equal(0))
		Expect(session.LineInOutputContains("/etc/group")).To(BeFalse())
	})

	It("podman run non-numeric group not specified in container", func() {
		session := podmanTest.Podman([]string{"run", "--read-only", "-u", "root:doesnotexist", BB, "mount"})
		session.WaitWithDefaultTimeout()
		Expect(session.ExitCode()).To(Not(Equal(0)))
	})

	It("podman run numeric group specified in container", func() {
		session := podmanTest.Podman([]string{"run", "--read-only", "-u", "root:11", BB, "mount"})
		session.WaitWithDefaultTimeout()
		Expect(session.ExitCode()).To(Equal(0))
		Expect(session.LineInOutputContains("/etc/group")).To(BeFalse())
	})

	It("podman run numeric group not specified in container", func() {
		session := podmanTest.Podman([]string{"run", "--read-only", "-u", "20001:20001", BB, "mount"})
		session.WaitWithDefaultTimeout()
		Expect(session.ExitCode()).To(Equal(0))
		Expect(session.LineInOutputContains("/etc/group")).To(BeTrue())
	})

	It("podman run numeric user not specified in container modifies group", func() {
		session := podmanTest.Podman([]string{"run", "--read-only", "-u", "20001", BB, "mount"})
		session.WaitWithDefaultTimeout()
		Expect(session.ExitCode()).To(Equal(0))
		Expect(session.LineInOutputContains("/etc/group")).To(BeTrue())
	})

	It("podman run numeric group from image and no group file", func() {
		dockerfile := `FROM alpine
RUN rm -f /etc/passwd /etc/shadow /etc/group
USER 1000`
		imgName := "testimg"
		podmanTest.BuildImage(dockerfile, imgName, "false")
		session := podmanTest.Podman([]string{"run", "--rm", imgName, "ls", "/etc/"})
		session.WaitWithDefaultTimeout()
		Expect(session.ExitCode()).To(Equal(0))
		Expect(session.OutputToString()).To(Not(ContainSubstring("/etc/group")))
	})
})
