// --- BEGIN COPYRIGHT BLOCK ---
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; version 2 of the License.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation, Inc.,
// 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
//
// (C) 2007 Red Hat, Inc.
// All rights reserved.
// --- END COPYRIGHT BLOCK ---
package org.dogtagpki.legacy.server.policy.constraints;

import java.util.Date;
import java.util.Vector;

import org.dogtagpki.legacy.policy.EPolicyException;
import org.dogtagpki.legacy.policy.IPolicyProcessor;
import org.dogtagpki.legacy.policy.IRevocationPolicy;
import org.dogtagpki.legacy.server.policy.APolicyRule;
import org.mozilla.jss.netscape.security.x509.CertificateValidity;
import org.mozilla.jss.netscape.security.x509.RevocationReason;
import org.mozilla.jss.netscape.security.x509.X509CertImpl;
import org.mozilla.jss.netscape.security.x509.X509CertInfo;

import com.netscape.certsrv.base.EBaseException;
import com.netscape.certsrv.base.IExtendedPluginInfo;
import com.netscape.certsrv.request.PolicyResult;
import com.netscape.cmscore.apps.CMS;
import com.netscape.cmscore.base.ConfigStore;
import com.netscape.cmscore.request.Request;

/**
 * Whether to allow revocation of an expired cert.
 * <P>
 *
 * <PRE>
 * NOTE:  The Policy Framework has been replaced by the Profile Framework.
 * </PRE>
 * <P>
 *
 * @version $Revision$, $Date$
 */
public class RevocationConstraints extends APolicyRule
        implements IRevocationPolicy, IExtendedPluginInfo {

    public static org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(RevocationConstraints.class);

    private static final String PROP_ALLOW_EXPIRED_CERTS = "allowExpiredCerts";
    private static final String PROP_ALLOW_ON_HOLD = "allowOnHold";

    private boolean mAllowExpiredCerts = true;
    private boolean mAllowOnHold = true;

    private final static Vector<String> defConfParams = new Vector<>();
    static {
        defConfParams.addElement(PROP_ALLOW_EXPIRED_CERTS + "=" + true);
        defConfParams.addElement(PROP_ALLOW_ON_HOLD + "=" + true);
    }

    public RevocationConstraints() {
        NAME = "RevocationConstraints";
        DESC = "Whether to allow revocation of expired certs and on-hold.";
    }

    @Override
    public String[] getExtendedPluginInfo() {
        String[] params = {
                PROP_ALLOW_EXPIRED_CERTS + ";boolean;Allow a user to revoke an already-expired certificate",
                PROP_ALLOW_ON_HOLD + ";boolean;Allow a user to set reason to On-Hold",
                IExtendedPluginInfo.HELP_TOKEN +
                        ";configuration-policyrules-revocationconstraints",
                IExtendedPluginInfo.HELP_TEXT +
                        ";Allow administrator to decide policy on whether to allow " +
                        "recovation of expired certificates" +
                        "and set reason to On-Hold"

        };

        return params;

    }

    /**
     * Initializes this policy rule.
     * <P>
     *
     * The entries probably are of the form:
     *
     * ra.Policy.rule.<ruleName>.implName=ValidityConstraints ra.Policy.rule.<ruleName>.enable=true
     * ra.Policy.rule.<ruleName>.allowExpiredCerts=true
     *
     * @param config The config store reference
     */
    @Override
    public void init(IPolicyProcessor owner, ConfigStore config) throws EPolicyException {
        // Get min and max validity in days and onfigure them.
        try {
            mAllowExpiredCerts =
                    config.getBoolean(PROP_ALLOW_EXPIRED_CERTS, true);
            mAllowOnHold =
                    config.getBoolean(PROP_ALLOW_ON_HOLD, true);
        } catch (EBaseException e) {
            // never happen.
        }

        logger.debug("RevocationConstraints: allow expired certs " + mAllowExpiredCerts);
        logger.debug("RevocationConstraints: allow on hold " + mAllowOnHold);
    }

    /**
     * Applies the policy on the given Request.
     * <P>
     *
     * @param req The request on which to apply policy.
     * @return The policy result object.
     */
    @Override
    public PolicyResult apply(Request req) {
        logger.debug("RevocationConstraints: apply begins");
        if (req.getExtDataInInteger(Request.REVOKED_REASON) == null) {
            logger.debug("RevocationConstraints: apply: no revocationReason found in request");
            return PolicyResult.REJECTED;
        }
        RevocationReason rr = RevocationReason.valueOf(
                req.getExtDataInInteger(Request.REVOKED_REASON).intValue());

        if (!mAllowOnHold && (rr != null)) {
            int reason = rr.getCode();

            if (reason == RevocationReason.CERTIFICATE_HOLD.getCode()) {
                String params[] = { getInstanceName() };

                setError(req, CMS.getUserMessage("CMS_POLICY_NO_ON_HOLD_ALLOWED", params), "");
                return PolicyResult.REJECTED;
            }
        }

        if (mAllowExpiredCerts)
            // nothing to check.
            return PolicyResult.ACCEPTED;

        PolicyResult result = PolicyResult.ACCEPTED;

        try {
            // Get the certificates being renwed.
            X509CertImpl[] oldCerts =
                    req.getExtDataInCertArray(Request.OLD_CERTS);

            if (oldCerts == null) {
                setError(req, CMS.getUserMessage("CMS_POLICY_NO_OLD_CERT"),
                        getInstanceName());
                return PolicyResult.REJECTED;
            }

            // check if each cert to be renewed is expired.
            for (int i = 0; i < oldCerts.length; i++) {
                X509CertInfo oldCertInfo = (X509CertInfo)
                        oldCerts[i].get(
                                X509CertImpl.NAME + "." + X509CertImpl.INFO);
                CertificateValidity oldValidity = (CertificateValidity)
                        oldCertInfo.get(X509CertInfo.VALIDITY);
                Date notAfter = (Date)
                        oldValidity.get(CertificateValidity.NOT_AFTER);

                // Is the Certificate still valid?
                Date now = new Date();

                if (notAfter.before(now)) {
                    String params[] = { getInstanceName() };

                    setError(req,
                            CMS.getUserMessage("CMS_POLICY_CANNOT_REVOKE_EXPIRED_CERTS",
                                    params), "");
                    result = PolicyResult.REJECTED;
                    break;
                }
            }

        } catch (Exception e) {
            String params[] = { getInstanceName(), e.toString() };

            setError(req, CMS.getUserMessage("CMS_POLICY_UNEXPECTED_POLICY_ERROR", params), "");
            result = PolicyResult.REJECTED;
        }
        return result;
    }

    /**
     * Return configured parameters for a policy rule instance.
     *
     * @return nvPairs A Vector of name/value pairs.
     */
    @Override
    public Vector<String> getInstanceParams() {
        Vector<String> confParams = new Vector<>();

        confParams.addElement(
                PROP_ALLOW_EXPIRED_CERTS + "=" + mAllowExpiredCerts);
        confParams.addElement(
                PROP_ALLOW_ON_HOLD + "=" + mAllowOnHold);
        return confParams;
    }

    /**
     * Return default parameters for a policy implementation.
     *
     * @return nvPairs A Vector of name/value pairs.
     */
    @Override
    public Vector<String> getDefaultParams() {
        return defConfParams;
    }
}
