/*
 * validate_x86.c - validate event tables + encodings
 *
 * Copyright (c) 2010 Google, Inc
 * Contributed by Stephane Eranian <eranian@gmail.com>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * This file is part of libpfm, a performance monitoring support library for
 * applications on Linux.
 */
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <inttypes.h>
#include <stdarg.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>

#include <perfmon/pfmlib.h>

#define MAX_ENCODING	8
#define SRC_LINE	.line = __LINE__

typedef struct {
	const char *name;
	const char *fstr;
	uint64_t codes[MAX_ENCODING];
	int ret, count;
	int line;
} test_event_t;

static const test_event_t x86_test_events[]={
	{ SRC_LINE,
	  .name = "core::INST_RETIRED:ANY_P",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5300c0ull,
	},
	{ SRC_LINE,
	  .name = "core::INST_RETIRED:ANY_P:ANY_P",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5300c0ull,
	},
	{ SRC_LINE,
	  .name = "core::INST_RETIRED:ANY_P:DEAD",
	  .ret  = PFM_ERR_ATTR, /* cannot know if it is umask or mod */
	  .count = 0,
	  .codes[0] = 0ull,
	},
	{ SRC_LINE,
	  .name = "core::INST_RETIRED:ANY_P:u:u",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5100c0ull,
	},
	{ SRC_LINE,
	  .name = "core::INST_RETIRED:ANY_P:u=0:k=1:u=1",
	  .ret  = PFM_ERR_ATTR_SET,
	  .count = 0,
	  .codes[0] = 0ull,
	},
	{ SRC_LINE,
	  .name = "core::INST_RETIRED:ANY_P:c=1:i",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1d300c0ull,
	},
	{ SRC_LINE,
	  .name = "core::INST_RETIRED:ANY_P:c=1:i=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1d300c0ull,
	},
	{ SRC_LINE,
	  .name = "core::INST_RETIRED:ANY_P:c=2",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x25300c0ull,
	},
	{ SRC_LINE,
	  .name = "core::INST_RETIRED:ANY_P:c=320",
	  .ret  = PFM_ERR_ATTR_VAL,
	  .count = 0,
	  .codes[0] = 0ull,
	},
	{ SRC_LINE,
	  .name = "core::INST_RETIRED:ANY_P:t=1",
	  .ret  = PFM_ERR_ATTR,
	  .count = 0,
	  .codes[0] = 0ull,
	},
	{ SRC_LINE,
	  .name = "core::L2_LINES_IN",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x537024ull,
	},
	{ SRC_LINE,
	  .name = "core::L2_LINES_IN:SELF",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x537024ull,
	  .fstr = "core::L2_LINES_IN:SELF:ANY:k=1:u=1:e=0:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "core::L2_LINES_IN:SELF:BOTH_CORES",
	  .ret  = PFM_ERR_FEATCOMB,
	  .count = 0,
	  .codes[0] = 0ull,
	},
	{ SRC_LINE,
	  .name = "core::L2_LINES_IN:SELF:PREFETCH",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x535024ull,
	},
	{ SRC_LINE,
	  .name = "core::L2_LINES_IN:SELF:PREFETCH:ANY",
	  .ret  = PFM_ERR_FEATCOMB,
	  .count = 0,
	  .codes[0] = 0ull,
	},
	{ SRC_LINE,
	  .name = "core::RS_UOPS_DISPATCHED_NONE",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1d300a0ull,
	},
	{ SRC_LINE,
	  .name = "core::RS_UOPS_DISPATCHED_NONE:c=2",
	  .ret  = PFM_ERR_ATTR_SET,
	  .count = 1,
	  .codes[0] = 0ull,
	},
	{ SRC_LINE,
	  .name = "core::branch_instructions_retired",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5300c4ull,
	  .fstr = "core::BR_INST_RETIRED:ANY:k=1:u=1:e=0:i=0:c=0"
	},
	{ SRC_LINE,
	  .name = "nhm::branch_instructions_retired",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5300c4ull,
	  .fstr = "nhm::BR_INST_RETIRED:ALL_BRANCHES:k=1:u=1:e=0:i=0:c=0:t=0"
	},
	{ SRC_LINE,
	  .name = "wsm::BRANCH_INSTRUCTIONS_RETIRED",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5300c4ull, /* architected encoding, guaranteed to exist */
	  .fstr = "wsm::BR_INST_RETIRED:ALL_BRANCHES:k=1:u=1:e=0:i=0:c=0:t=0"
	},
	{ SRC_LINE,
	  .name = "nhm::ARITH:DIV:k",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1d60114ull,
	  .fstr = "nhm::ARITH:CYCLES_DIV_BUSY:k=1:u=0:e=1:i=1:c=1:t=0",
	},
	{ SRC_LINE,
	  .name = "nhm::ARITH:CYCLES_DIV_BUSY:k=1:u=1:e=1:i=1:c=1:t=0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1d70114ull,
	  .fstr = "nhm::ARITH:CYCLES_DIV_BUSY:k=1:u=1:e=1:i=1:c=1:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm::UOPS_EXECUTED:CORE_STALL_COUNT:u",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1f53fb1ull,
	  .fstr = "wsm::UOPS_EXECUTED:CORE_STALL_CYCLES:k=0:u=1:e=1:i=1:c=1:t=1",
	},
	{ SRC_LINE,
	  .name = "wsm::UOPS_EXECUTED:CORE_STALL_COUNT:u:t=0",
	  .ret  = PFM_ERR_ATTR_SET,
	  .count = 0,
	  .codes[0] = 0ull,
	},
	{ SRC_LINE,
	  .name = "wsm_unc::unc_qmc_writes:full_any:partial_any",
	  .ret  = PFM_ERR_FEATCOMB,
	  .count = 0,
	  .codes[0] = 0ull,
	},
	{ SRC_LINE,
	  .name = "wsm_unc::unc_qmc_writes",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x50072full,
	  .fstr = "wsm_unc::UNC_QMC_WRITES:FULL_ANY:e=0:i=0:c=0:o=0",
	},
	{ SRC_LINE,
	  .name = "wsm_unc::unc_qmc_writes:full_any",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x50072full,
	  .fstr = "wsm_unc::UNC_QMC_WRITES:FULL_ANY:e=0:i=0:c=0:o=0",
	},
	{ SRC_LINE,
	  .name = "wsm_unc::unc_qmc_writes:full_ch0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x50012full,
	  .fstr = "wsm_unc::UNC_QMC_WRITES:FULL_CH0:e=0:i=0:c=0:o=0",
	},
	{ SRC_LINE,
	  .name = "wsm_unc::unc_qmc_writes:partial_any",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x50382full,
	  .fstr = "wsm_unc::UNC_QMC_WRITES:PARTIAL_ANY:e=0:i=0:c=0:o=0",
	},
	{ SRC_LINE,
	  .name = "wsm_unc::unc_qmc_writes:partial_ch0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x50082full,
	  .fstr = "wsm_unc::UNC_QMC_WRITES:PARTIAL_CH0:e=0:i=0:c=0:o=0",
	},
	{ SRC_LINE,
	  .name = "wsm_unc::unc_qmc_writes:partial_ch0:partial_ch1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x50182full,
	  .fstr = "wsm_unc::UNC_QMC_WRITES:PARTIAL_CH0:PARTIAL_CH1:e=0:i=0:c=0:o=0",
	},
	{ SRC_LINE,
	  .name = "amd64_fam10h_barcelona::DISPATCHED_FPU",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x533f00ull,
	  .fstr = "amd64_fam10h_barcelona::DISPATCHED_FPU:ALL:k=1:u=1:e=0:i=0:c=0:h=0:g=0"
	},
	{ SRC_LINE,
	  .name = "amd64_fam10h_barcelona::DISPATCHED_FPU:k:u=0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x523f00ull,
	  .fstr = "amd64_fam10h_barcelona::DISPATCHED_FPU:ALL:k=1:u=0:e=0:i=0:c=0:h=0:g=0"
	},
	{ SRC_LINE,
	  .name = "amd64_fam10h_barcelona::DISPATCHED_FPU:OPS_ADD:OPS_MULTIPLY",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x530300ull,
	  .fstr = "amd64_fam10h_barcelona::DISPATCHED_FPU:OPS_ADD:OPS_MULTIPLY:k=1:u=1:e=0:i=0:c=0:h=0:g=0",
	},
	{ SRC_LINE,
	  .name = "amd64_fam10h_barcelona::L2_CACHE_MISS:ALL:DATA",
	  .ret  = PFM_ERR_FEATCOMB,
	  .count = 0,
	  .codes[0] = 0ull,
	},
	{ SRC_LINE,
	  .name = "amd64_fam10h_barcelona::MEMORY_CONTROLLER_REQUESTS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x10053fff0ull,
	  .fstr = "amd64_fam10h_barcelona::MEMORY_CONTROLLER_REQUESTS:ALL:k=1:u=1:e=0:i=0:c=0:h=0:g=0",
	},
	{ SRC_LINE,
	  .name = "amd64_k8_revb::RETURN_STACK_OVERFLOWS:g=1:u",
	  .ret  = PFM_ERR_ATTR,
	  .count = 0,
	  .codes[0] = 0ull,
	},
	{ SRC_LINE,
	  .name = "amd64_k8_revb::RETURN_STACK_HITS:e=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x570088ull,
	  .fstr = "amd64_k8_revb::RETURN_STACK_HITS:k=1:u=1:e=1:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "amd64_k8_revb::PROBE:ALL",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x533fecull,
	  .fstr = "amd64_k8_revb::PROBE:ALL:k=1:u=1:e=0:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "amd64_k8_revc::PROBE:ALL",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x533fecull,
	  .fstr = "amd64_k8_revc::PROBE:ALL:k=1:u=1:e=0:i=0:c=0",
	},
	{ SRC_LINE,
	  .name = "amd64_k8_revd::PROBE:ALL",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x537fecull,
	  .fstr = "amd64_k8_revd::PROBE:ALL:k=1:u=1:e=0:i=0:c=0"
	},
	{ SRC_LINE,
	  .name = "amd64_k8_reve::PROBE:ALL",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x537fecull,
	  .fstr = "amd64_k8_reve::PROBE:ALL:k=1:u=1:e=0:i=0:c=0"
	},
	{ SRC_LINE,
	  .name = "amd64_k8_revf::PROBE:ALL",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x537fecull,
	  .fstr = "amd64_k8_revf::PROBE:ALL:k=1:u=1:e=0:i=0:c=0"
	},
	{ SRC_LINE,
	  .name = "amd64_k8_revg::PROBE:ALL",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x537fecull,
	  .fstr = "amd64_k8_revg::PROBE:ALL:k=1:u=1:e=0:i=0:c=0"
	},
	{ SRC_LINE,
	  .name = "amd64_fam10h_barcelona::L1_DTLB_MISS_AND_L2_DTLB_HIT:L2_1G_TLB_HIT",
	  .ret  = PFM_ERR_ATTR,
	  .count = 0,
	  .codes[0] = 0ull,
	},
	{ SRC_LINE,
	  .name = "amd64_fam10h_barcelona::L1_DTLB_MISS_AND_L2_DTLB_HIT:ALL",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x530345ull,
	  .fstr = "amd64_fam10h_barcelona::L1_DTLB_MISS_AND_L2_DTLB_HIT:ALL:k=1:u=1:e=0:i=0:c=0:h=0:g=0"
	},
	{ SRC_LINE,
	  .name = "amd64_fam10h_shanghai::L1_DTLB_MISS_AND_L2_DTLB_HIT:ALL",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x530745ull,
	  .fstr = "amd64_fam10h_shanghai::L1_DTLB_MISS_AND_L2_DTLB_HIT:ALL:k=1:u=1:e=0:i=0:c=0:h=0:g=0"
	},
	{ SRC_LINE,
	  .name = "amd64_fam10h_istanbul::L1_DTLB_MISS_AND_L2_DTLB_HIT:ALL",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x530745ull,
	  .fstr = "amd64_fam10h_istanbul::L1_DTLB_MISS_AND_L2_DTLB_HIT:ALL:k=1:u=1:e=0:i=0:c=0:h=0:g=0"
	},
	{ SRC_LINE,
	  .name = "amd64_fam10h_barcelona::READ_REQUEST_TO_L3_CACHE",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x40053f7e0ull,
	  .fstr = "amd64_fam10h_barcelona::READ_REQUEST_TO_L3_CACHE:ANY_READ:ALL_CORES:k=1:u=1:e=0:i=0:c=0:h=0:g=0",
	},
	{ SRC_LINE,
	  .name = "amd64_fam10h_shanghai::READ_REQUEST_TO_L3_CACHE",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x40053f7e0ull,
	  .fstr = "amd64_fam10h_shanghai::READ_REQUEST_TO_L3_CACHE:ANY_READ:ALL_CORES:k=1:u=1:e=0:i=0:c=0:h=0:g=0",
	},
	{ SRC_LINE,
	  .name = "core::RAT_STALLS:ANY:u:c=1,cycles", /* must cut at comma */
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1510fd2ull,
	  .fstr = "core::RAT_STALLS:ANY:k=0:u=1:e=0:i=0:c=1"
	},
	{ SRC_LINE,
	  .name = "wsm::mem_uncore_retired:remote_dram",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x53200f,
	  .fstr = "wsm::MEM_UNCORE_RETIRED:REMOTE_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm_dp::mem_uncore_retired:remote_dram",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x53100f,
	  .fstr = "wsm_dp::MEM_UNCORE_RETIRED:REMOTE_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm::mem_uncore_retired:local_dram",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x53100f,
	  .fstr = "wsm::MEM_UNCORE_RETIRED:LOCAL_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm_dp::mem_uncore_retired:local_dram",
	  .ret  = PFM_ERR_ATTR,
	  .count = 1,
	  .codes[0] = 0,
	},
	{ SRC_LINE,
	  .name = "nhm::mem_uncore_retired:uncacheable",
	  .ret  = PFM_ERR_ATTR,
	  .count = 1,
	  .codes[0] = 0,
	},
	{ SRC_LINE,
	  .name = "nhm::mem_uncore_retired:l3_data_miss_unknown",
	  .ret  = PFM_ERR_ATTR,
	  .count = 1,
	  .codes[0] = 0,
	},
	{ SRC_LINE,
	  .name = "nhm_ex::mem_uncore_retired:uncacheable",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x53800f,
	  .fstr = "nhm_ex::MEM_UNCORE_RETIRED:UNCACHEABLE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "nhm_ex::mem_uncore_retired:l3_data_miss_unknown",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x53010f,
	  .fstr = "nhm_ex::MEM_UNCORE_RETIRED:L3_DATA_MISS_UNKNOWN:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "nhm::mem_uncore_retired:local_dram",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x53200f,
	  .fstr = "nhm::MEM_UNCORE_RETIRED:LOCAL_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "nhm_ex::mem_uncore_retired:local_dram",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x53200f,
	  .fstr = "nhm_ex::MEM_UNCORE_RETIRED:LOCAL_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm::offcore_response_0:k",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5201b7,
	  .codes[1] = 0xffff,
	  .fstr = "wsm::OFFCORE_RESPONSE_0:ANY_REQUEST:ANY_RESPONSE:k=1:u=0:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm::offcore_response_0:local_dram",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x20ff,
	  .fstr = "wsm::OFFCORE_RESPONSE_0:ANY_REQUEST:LOCAL_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm::offcore_response_0:PF_IFETCH",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0xff40,
	  .fstr = "wsm::OFFCORE_RESPONSE_0:PF_IFETCH:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm::offcore_response_0:ANY_DATA:LOCAL_DRAM",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x2033,
	  .fstr = "wsm::OFFCORE_RESPONSE_0:DMND_DATA_RD:DMND_RFO:PF_DATA_RD:PF_RFO:LOCAL_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm::offcore_response_0:DMND_RFO:DMND_DATA_RD:LOCAL_DRAM:REMOTE_DRAM",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x6003,
	  .fstr = "wsm::OFFCORE_RESPONSE_0:DMND_DATA_RD:DMND_RFO:LOCAL_DRAM:REMOTE_DRAM:k=1:u=1:e=0:i=0:c=0:t=0"
	},
	{ SRC_LINE,
	  .name = "wsm::offcore_response_1:local_dram",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x20ff,
	  .fstr = "wsm::OFFCORE_RESPONSE_1:ANY_REQUEST:LOCAL_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm::offcore_response_1:PF_IFETCH",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0xff40,
	  .fstr = "wsm::OFFCORE_RESPONSE_1:PF_IFETCH:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm::offcore_response_1:ANY_DATA:LOCAL_DRAM",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x2033,
	  .fstr = "wsm::OFFCORE_RESPONSE_1:DMND_DATA_RD:DMND_RFO:PF_DATA_RD:PF_RFO:LOCAL_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm::offcore_response_1:DMND_RFO:DMND_DATA_RD:LOCAL_DRAM:REMOTE_DRAM",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x6003,
	  .fstr = "wsm::OFFCORE_RESPONSE_1:DMND_DATA_RD:DMND_RFO:LOCAL_DRAM:REMOTE_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm::offcore_response_0:ANY_LLC_MISS",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0xf8ff,
	  .fstr = "wsm::OFFCORE_RESPONSE_0:ANY_REQUEST:REMOTE_CACHE_HITM:REMOTE_CACHE_FWD:LOCAL_DRAM:REMOTE_DRAM:NON_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm_dp::offcore_response_0:ANY_LLC_MISS",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0xf8ff,
	  .fstr = "wsm_dp::OFFCORE_RESPONSE_0:ANY_REQUEST:REMOTE_CACHE_HITM:LOCAL_DRAM_AND_REMOTE_CACHE_HIT:REMOTE_DRAM:OTHER_LLC_MISS:NON_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm_dp::offcore_response_0:LOCAL_CACHE",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x7ff,
	  .fstr = "wsm_dp::OFFCORE_RESPONSE_0:ANY_REQUEST:UNCORE_HIT:OTHER_CORE_HIT_SNP:OTHER_CORE_HITM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm_dp::offcore_response_0:ANY_CACHE_DRAM",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x7fff,
	  .fstr = "wsm_dp::OFFCORE_RESPONSE_0:ANY_REQUEST:UNCORE_HIT:OTHER_CORE_HIT_SNP:OTHER_CORE_HITM:REMOTE_CACHE_HITM:LOCAL_DRAM_AND_REMOTE_CACHE_HIT:REMOTE_DRAM:OTHER_LLC_MISS:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "nhm::offcore_response_0:k",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5201b7,
	  .codes[1] = 0xffff,
	  .fstr = "nhm::OFFCORE_RESPONSE_0:ANY_REQUEST:ANY_RESPONSE:k=1:u=0:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "nhm::offcore_response_0:local_dram",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x40ff,
	  .fstr = "nhm::OFFCORE_RESPONSE_0:ANY_REQUEST:LOCAL_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "nhm::offcore_response_0:any_llc_miss",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0xf8ff,
	  .fstr = "nhm::OFFCORE_RESPONSE_0:ANY_REQUEST:REMOTE_CACHE_HITM:REMOTE_CACHE_FWD:REMOTE_DRAM:LOCAL_DRAM:NON_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "nhm::offcore_response_0:any_dram",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x60ff,
	  .fstr = "nhm::OFFCORE_RESPONSE_0:ANY_REQUEST:REMOTE_DRAM:LOCAL_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "nhm::offcore_response_0:PF_IFETCH",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0xff40,
	  .fstr = "nhm::OFFCORE_RESPONSE_0:PF_IFETCH:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "nhm::offcore_response_0:ANY_DATA:LOCAL_DRAM",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x4033,
	  .fstr = "nhm::OFFCORE_RESPONSE_0:DMND_DATA_RD:DMND_RFO:PF_DATA_RD:PF_RFO:LOCAL_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "nhm::offcore_response_0:DMND_RFO:DMND_DATA_RD:LOCAL_DRAM:REMOTE_DRAM",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x6003,
	  .fstr = "nhm::OFFCORE_RESPONSE_0:DMND_DATA_RD:DMND_RFO:REMOTE_DRAM:LOCAL_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "amd64_k8_revg::DISPATCHED_FPU:0xff:k",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x52ff00ull,
	  .fstr = "amd64_k8_revg::DISPATCHED_FPU:0xff:k=1:u=0:e=0:i=0:c=0"
	},
	{ SRC_LINE,
	  .name = "amd64_k8_revg::DISPATCHED_FPU:0x4ff",
	  .ret  = PFM_ERR_ATTR,
	  .count = 0,
	},
	{ SRC_LINE,
	  .name = "amd64_fam10h_barcelona::DISPATCHED_FPU:0x4ff:u",
	  .ret  = PFM_ERR_ATTR
	},
	{ SRC_LINE,
	  .name = "amd64_fam10h_barcelona::DISPATCHED_FPU:0xff:u",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x51ff00ull,
	  .fstr = "amd64_fam10h_barcelona::DISPATCHED_FPU:0xff:k=0:u=1:e=0:i=0:c=0:h=0:g=0"
	},
	{ SRC_LINE,
	  .name = "wsm::inst_retired:0xff:k",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x52ffc0,
	  .fstr = "wsm::INST_RETIRED:0xff:k=1:u=0:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "wsm::uops_issued:0xff:stall_cycles",
	  .ret  = PFM_ERR_ATTR,
	  .count = 0,
	},
	{ SRC_LINE,
	  .name = "wsm::uops_issued:0xff:0xf1",
	  .ret  = PFM_ERR_ATTR,
	  .count = 0,
	},
	{ SRC_LINE,
	  .name = "wsm::uops_issued:0xff=",
	  .ret  = PFM_ERR_ATTR_VAL,
	  .count = 0,
	},
	{ SRC_LINE,
	  .name = "wsm::uops_issued:123",
	  .ret  = PFM_ERR_ATTR,
	  .count = 0,
	},
	{ SRC_LINE,
	  .name = "wsm::uops_issued:0xfff",
	  .ret  = PFM_ERR_ATTR,
	  .count = 0,
	},
	{ SRC_LINE,
	  .name = "netburst::global_power_events",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x2600020f,
	  .codes[1] = 0x3d000,
	  .fstr = "netburst::global_power_events:RUNNING:k=1:u=1:e=0:cmpl=0:thr=0",
	},
	{ SRC_LINE,
	  .name = "netburst::global_power_events:RUNNING:u:k",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x2600020f,
	  .codes[1] = 0x3d000,
	  .fstr = "netburst::global_power_events:RUNNING:k=1:u=1:e=0:cmpl=0:thr=0",
	},
	{ SRC_LINE,
	  .name = "netburst::global_power_events:RUNNING:e",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x2600020f,
	  .codes[1] = 0x107d000,
	  .fstr = "netburst::global_power_events:RUNNING:k=1:u=1:e=1:cmpl=0:thr=0",
	},
	{ SRC_LINE,
	  .name = "netburst::global_power_events:RUNNING:cmpl:e:u",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x26000205,
	  .codes[1] = 0x10fd000,
	  .fstr = "netburst::global_power_events:RUNNING:k=0:u=1:e=1:cmpl=1:thr=0",
	},
	{ SRC_LINE,
	  .name = "netburst::global_power_events:RUNNING:cmpl:thr=8:u",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x26000205,
	  .codes[1] = 0x8fd000,
	  .fstr = "netburst::global_power_events:RUNNING:k=0:u=1:e=0:cmpl=1:thr=8",
	},
	{ SRC_LINE,
	  .name = "netburst::global_power_events:RUNNING:cmpl:thr=32:u",
	  .ret  = PFM_ERR_ATTR_VAL,
	  .count = 0,
	},
	{ SRC_LINE,
	  .name = "netburst::instr_completed:nbogus",
	  .ret  = PFM_ERR_NOTFOUND,
	  .count = 0,
	},
	{ SRC_LINE,
	  .name = "netburst_p::instr_completed:nbogus",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0xe00020f,
	  .codes[1] = 0x39000,
	  .fstr = "netburst_p::instr_completed:NBOGUS:k=1:u=1:e=0:cmpl=0:thr=0",
	},
	{ SRC_LINE,
	  .name = "snb::cpl_cycles:ring0_trans:u",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x155015c,
	  .fstr = "snb::CPL_CYCLES:RING0:k=0:u=1:e=1:i=0:c=1:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::cpl_cycles:ring0_trans:e=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x157015cull,
	},
	{ SRC_LINE,
	  .name = "snb::OFFCORE_REQUESTS_OUTSTanding:ALL_DATA_RD_CYCLES",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1530860,
	  .fstr = "snb::OFFCORE_REQUESTS_OUTSTANDING:ALL_DATA_RD:k=1:u=1:e=0:i=0:c=1:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::uops_issued:core_stall_cycles:u:k",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1f3010e,
	  .fstr = "snb::UOPS_ISSUED:ANY:k=1:u=1:e=0:i=1:c=1:t=1",
	},
	{ SRC_LINE,
	  .name = "snb::LLC_REFERences:k:t=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x724f2e,
	  .fstr = "snb::LAST_LEVEL_CACHE_REFERENCES:k=1:u=0:e=0:i=0:c=0:t=1",
	},
	{ SRC_LINE,
	  .name = "snb::ITLB:0x1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5301ae,
	  .fstr = "snb::ITLB:0x1:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_0:DMND_RFO:ANY_RESPONSE",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x10002,
	  .fstr = "snb::OFFCORE_RESPONSE_0:DMND_RFO:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_0:DMND_RFO:ANY_REQUEST",
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x18fff,
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_0:DMND_RFO",
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x10002,
	  .fstr = "snb::OFFCORE_RESPONSE_0:DMND_RFO:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_0:any_response",
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x18fff,
	  .fstr = "snb::OFFCORE_RESPONSE_0:ANY_REQUEST:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_0:NO_SUPP:SNP_NONE:PF_RFO:PF_IFETCH",
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x80020060,
	  .fstr = "snb::OFFCORE_RESPONSE_0:PF_RFO:PF_IFETCH:NO_SUPP:SNP_NONE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_1:DMND_RFO:ANY_RESPONSE",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x10002,
	  .fstr = "snb::OFFCORE_RESPONSE_1:DMND_RFO:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_1:DMND_RFO:ANY_REQUEST",
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x18fff,
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_1:DMND_RFO",
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x10002,
	  .fstr = "snb::OFFCORE_RESPONSE_1:DMND_RFO:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_1:any_response",
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x18fff,
	  .fstr = "snb::OFFCORE_RESPONSE_1:ANY_REQUEST:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_1:NO_SUPP:SNP_NONE:PF_RFO:PF_IFETCH",
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x80020060,
	  .fstr = "snb::OFFCORE_RESPONSE_1:PF_RFO:PF_IFETCH:NO_SUPP:SNP_NONE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_1:ANY_REQUEST:LLC_MISS_LOCAL_DRAM",
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x3f80408fffull,
	  .fstr = "snb::OFFCORE_RESPONSE_1:DMND_DATA_RD:DMND_RFO:DMND_IFETCH:WB:PF_DATA_RD:PF_RFO:PF_IFETCH:PF_LLC_DATA_RD:PF_LLC_RFO:PF_LLC_IFETCH:BUS_LOCKS:STRM_ST:OTHER:MISS_DRAM:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "amd64_fam14h_bobcat::MAB_REQUESTS:DC_BUFFER_0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x530068,
	  .fstr = "amd64_fam14h_bobcat::MAB_REQUESTS:DC_BUFFER_0:k=1:u=1:e=0:i=0:c=0:h=0:g=0",
	},
	{ SRC_LINE,
	  .name = "amd64_fam14h_bobcat::MAB_REQUESTS:DC_BUFFER_0:DC_BUFFER_1",
	  .ret  = PFM_ERR_FEATCOMB,
	},
	{ SRC_LINE,
	  .name = "amd64_fam14h_bobcat::MAB_REQUESTS:DC_BUFFER_0:IC_BUFFER_0",
	  .ret  = PFM_ERR_FEATCOMB,
	},
	{ SRC_LINE,
	  .name = "amd64_fam14h_bobcat::MAB_REQUESTS:ANY_DC_BUFFER",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x530b68,
	  .fstr = "amd64_fam14h_bobcat::MAB_REQUESTS:ANY_DC_BUFFER:k=1:u=1:e=0:i=0:c=0:h=0:g=0",
	},
	{ SRC_LINE,
	  .name = "amd64_fam14h_bobcat::MAB_REQUESTS:ANY_IC_BUFFER",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x530a68,
	  .fstr = "amd64_fam14h_bobcat::MAB_REQUESTS:ANY_IC_BUFFER:k=1:u=1:e=0:i=0:c=0:h=0:g=0",
	},
	{ SRC_LINE,
	  .name = "amd64_fam14h_bobcat::MAB_REQUESTS:ANY_IC_BUFFER:IC_BUFFER_1",
	  .ret  = PFM_ERR_FEATCOMB,
	},

	{ SRC_LINE,
	  .name = "core::INST_RETIRED:ANY_P:e",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "core::INST_RETIRED:ANY_P:e:c=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x15700c0ull,
	},
	{ SRC_LINE,
	  .name = "atom::INST_RETIRED:ANY_P:e",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5700c0ull,
	},
	{ SRC_LINE,
	  .name = "atom::INST_RETIRED:ANY_P:e:c=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x15700c0ull,
	},
	{ SRC_LINE,
	  .name = "nhm::INST_RETIRED:ANY_P:e",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "nhm::INST_RETIRED:ANY_P:e:c=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x15700c0ull,
	},
	{ SRC_LINE,
	  .name = "wsm::INST_RETIRED:ANY_P:e",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "wsm::INST_RETIRED:ANY_P:e:c=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x15700c0ull,
	},
	{ SRC_LINE,
	  .name = "snb::INST_RETIRED:ANY_P:e",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "snb::INST_RETIRED:ANY_P:e:c=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x15700c0ull,
	},
	{ SRC_LINE,
	  .name = "snb::INST_RETIRED:ANY_P:e:c=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x15700c0ull,
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_0:any_request",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7ull,
	  .codes[1]= 0x18fffull,
	  .fstr = "snb::OFFCORE_RESPONSE_0:DMND_DATA_RD:DMND_RFO:DMND_IFETCH:WB:PF_DATA_RD:PF_RFO:PF_IFETCH:PF_LLC_DATA_RD:PF_LLC_RFO:PF_LLC_IFETCH:BUS_LOCKS:STRM_ST:OTHER:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_0:dmnd_data_rd",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7ull,
	  .codes[1]=0x10001ull,
	  .fstr = "snb::OFFCORE_RESPONSE_0:DMND_DATA_RD:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_0:dmnd_data_rd:llc_hite",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7ull,
	  .codes[1]=0x3f80080001ull,
	  .fstr = "snb::OFFCORE_RESPONSE_0:DMND_DATA_RD:LLC_HITE:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_0:dmnd_data_rd:llc_hite:snp_any",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7ull,
	  .codes[1]=0x3f80080001ull,
	  .fstr = "snb::OFFCORE_RESPONSE_0:DMND_DATA_RD:LLC_HITE:SNP_NONE:SNP_NOT_NEEDED:SNP_MISS:SNP_NO_FWD:SNP_FWD:HITM:NON_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_0:dmnd_data_rd:llc_hite:hitm",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7ull,
	  .codes[1]=0x1000080001ull,
	  .fstr = "snb::OFFCORE_RESPONSE_0:DMND_DATA_RD:LLC_HITE:HITM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_0:dmnd_data_rd:any_response",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7ull,
	  .codes[1]=0x10001ull,
	  .fstr = "snb::OFFCORE_RESPONSE_0:DMND_DATA_RD:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_0:dmnd_data_rd:any_response:snp_any",
	  .ret  = PFM_ERR_FEATCOMB,
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_0:dmnd_data_rd:any_response:llc_hitmesf",
	  .ret  = PFM_ERR_FEATCOMB,
	},
	{ SRC_LINE,
	  .name = "snb::offcore_response_0:any_response",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7ull,
	  .codes[1]=0x18fffull,
	  .fstr = "snb::OFFCORE_RESPONSE_0:ANY_REQUEST:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "amd64_fam11h_turion::MAB_REQUESTS:DC_BUFFER_0",
	  .ret  = PFM_ERR_NOTFOUND,
	},
	{ SRC_LINE,
	  .name = "amd64_fam11h_turion::RETIRED_INSTRUCTIONS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5300c0,
	  .fstr = "amd64_fam11h_turion::RETIRED_INSTRUCTIONS:k=1:u=1:e=0:i=0:c=0:h=0:g=0",
	},
	{ SRC_LINE,
	  .name = "amd64_fam11h_turion::RETIRED_UOPS:k",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5200c1,
	  .fstr = "amd64_fam11h_turion::RETIRED_UOPS:k=1:u=0:e=0:i=0:c=0:h=0:g=0",
	},
	{ SRC_LINE,
	  .name = "amd64_fam11h_turion::CPU_CLK_UNHALTED",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x530076,
	  .fstr = "amd64_fam11h_turion::CPU_CLK_UNHALTED:k=1:u=1:e=0:i=0:c=0:h=0:g=0",
	},
	{ SRC_LINE,
	  .name = "amd64_fam11h_turion::RETIRED_UOPS:c=1:i",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1d300c1,
	  .fstr = "amd64_fam11h_turion::RETIRED_UOPS:k=1:u=1:e=0:i=1:c=1:h=0:g=0",
	},
	{ SRC_LINE,
	  .name = "ivb::ARITH:FPU_DIV",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1570114,
	  .fstr = "ivb::ARITH:FPU_DIV_ACTIVE:k=1:u=1:e=1:i=0:c=1:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::INST_RETIRED:ALL",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5301c0,
	  .fstr = "ivb::INST_RETIRED:ALL:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::INST_RETIRED:ALL:k",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5201c0,
	  .fstr = "ivb::INST_RETIRED:ALL:k=1:u=0:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::INST_RETIRED:ALL:u",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5101c0,
	  .fstr = "ivb::INST_RETIRED:ALL:k=0:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::TLB_ACCESS:LOAD_STLB_HIT",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x53045f,
	  .fstr = "ivb::TLB_ACCESS:LOAD_STLB_HIT:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::MOVE_ELIMINATION:INT_NOT_ELIMINATED",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x530158,
	  .fstr = "ivb::MOVE_ELIMINATION:INT_NOT_ELIMINATED:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::RESOURCE_STALLS:SB:RS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x530ca2,
	  .fstr = "ivb::RESOURCE_STALLS:RS:SB:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::RESOURCE_STALLS:ROB:RS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5314a2,
	  .fstr = "ivb::RESOURCE_STALLS:RS:ROB:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::CPU_CLK_UNHALTED:REF_P",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x53013c,
	  .fstr = "ivb::CPU_CLK_UNHALTED:REF_XCLK:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::DTLB_LOAD_MISSES:DEMAND_LD_MISS_CAUSES_A_WALK",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x538108,
	  .fstr = "ivb::DTLB_LOAD_MISSES:MISS_CAUSES_A_WALK:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::offcore_response_0:k",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5201b7,
	  .codes[1] = 0x18fff,
	  .fstr = "ivb::OFFCORE_RESPONSE_0:ANY_REQUEST:ANY_RESPONSE:k=1:u=0:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::offcore_response_0:LLC_MISS_LOCAL",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x3f80408fffull,
	  .fstr = "ivb::OFFCORE_RESPONSE_0:ANY_REQUEST:LLC_MISS_LOCAL:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::DTLB_LOAD_MISSES:STLB_HIT",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x53045f,
	  .fstr = "ivb::DTLB_LOAD_MISSES:STLB_HIT:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::DTLB_LOAD_MISSES:LARGE_WALK_COMPLETED:u",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x518808,
	  .fstr = "ivb::DTLB_LOAD_MISSES:LARGE_WALK_COMPLETED:k=0:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::l2_lines_in:i:i=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xd301f1,
	  .fstr = "snb::L2_LINES_IN:I:k=1:u=1:e=0:i=1:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::l2_lines_in:i=1:i",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xd301f1,
	  .fstr = "snb::L2_LINES_IN:I:k=1:u=1:e=0:i=1:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::l2_lines_in:i:i=0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5301f1,
	  .fstr = "snb::L2_LINES_IN:I:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::l2_lines_in:e:e=0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5304f1,
	  .fstr = "snb::L2_LINES_IN:E:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb::l2_lines_in:e:e=1",
	  .ret  = PFM_ERR_ATTR,
	  .count = 0,
	},
	{ SRC_LINE,
	  .name = "snb::l2_lines_in:e:e=1:c=10",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xa5704f1,
	  .fstr = "snb::L2_LINES_IN:E:k=1:u=1:e=1:i=0:c=10:t=0",
	},
	{ SRC_LINE,
	  .name = "snb_unc_cbo0::unc_clockticks",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5000ff,
	  .fstr = "snb_unc_cbo0::UNC_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "snb_unc_cbo1::unc_clockticks",
	  .ret  = PFM_ERR_NOTFOUND
	},
	{ SRC_LINE,
	  .name = "snb_unc_cbo2::unc_clockticks",
	  .ret  = PFM_ERR_NOTFOUND
	},
	{ SRC_LINE,
	  .name = "snb_unc_cbo3::unc_clockticks",
	  .ret  = PFM_ERR_NOTFOUND
	},
	{ SRC_LINE,
	  .name = "snb_unc_cbo1::UNC_CBO_CACHE_LOOKUP:STATE_MESI:READ_FILTER:c=1:i",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1d01f34,
	  .fstr = "snb_unc_cbo1::UNC_CBO_CACHE_LOOKUP:STATE_MESI:READ_FILTER:e=0:i=1:c=1",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_cbo1::UNC_C_CLOCKTICKS:u",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "snbep_unc_cbo0::UNC_C_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x00,
	  .fstr = "snbep_unc_cbo0::UNC_C_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_cbo0::UNC_C_LLC_LOOKUP:DATA_READ",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x334,
	  .codes[1] = 0x7c0000,
	  .fstr = "snbep_unc_cbo0::UNC_C_LLC_LOOKUP:DATA_READ:STATE_MESIF:e=0:i=0:t=0:tf=0:nf=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_cbo0::UNC_C_LLC_LOOKUP:DATA_READ:nf=1",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "snbep_unc_cbo0::UNC_C_LLC_LOOKUP:DATA_READ:tid=1",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "snbep_unc_cbo0::UNC_C_LLC_LOOKUP:NID:STATE_M",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "snbep_unc_cbo0::UNC_C_LLC_LOOKUP:NID:nf=3",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x4134,
	  .codes[1] = 0x7c0c00,
	  .fstr = "snbep_unc_cbo0::UNC_C_LLC_LOOKUP:NID:STATE_MESIF:e=0:i=0:t=0:tf=0:nf=3",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_cbo0::UNC_C_LLC_LOOKUP:NID:STATE_M:nf=3",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x4134,
	  .codes[1] = 0x200c00,
	  .fstr = "snbep_unc_cbo0::UNC_C_LLC_LOOKUP:NID:STATE_M:e=0:i=0:t=0:tf=0:nf=3",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_cbo0::UNC_C_LLC_LOOKUP:NID:nf=3:tid=1",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "snbep_unc_cbo0::UNC_C_TOR_INSERTS:OPCODE",
	  .ret  = PFM_ERR_UMASK,
	},
	{ SRC_LINE,
	  .name = "snbep_unc_cbo0::UNC_C_TOR_INSERTS:WB",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1035,
	  .fstr = "snbep_unc_cbo0::UNC_C_TOR_INSERTS:WB:e=0:i=0:t=0:tf=0:nf=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_cbo0::UNC_C_TOR_INSERTS:OPCODE:OPC_PCIWILF",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x135,
	  .codes[1] = 0xca000000,
	  .fstr = "snbep_unc_cbo0::UNC_C_TOR_INSERTS:OPCODE:OPC_PCIWILF:e=0:i=0:t=0:tf=0:nf=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_cbo0::UNC_C_TOR_INSERTS:OPCODE:OPC_PCIWILF:nf=1",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "snbep_unc_cbo0::UNC_C_TOR_INSERTS:NID_OPCODE:OPC_PCIRDCUR:nf=1",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x4135,
	  .codes[1] = 0xcf000400,
	  .fstr = "snbep_unc_cbo0::UNC_C_TOR_INSERTS:NID_OPCODE:OPC_PCIRDCUR:e=0:i=0:t=0:tf=0:nf=1",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_cbo0::UNC_C_TOR_INSERTS:OPC_RFO:NID_OPCODE:nf=1",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x4135,
	  .codes[1] = 0xc0000400,
	  .fstr = "snbep_unc_cbo0::UNC_C_TOR_INSERTS:NID_OPCODE:OPC_RFO:e=0:i=0:t=0:tf=0:nf=1",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_ha::UNC_H_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "snbep_unc_ha::UNC_H_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_ha::UNC_H_REQUESTS:READS:t=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1000301,
	  .fstr = "snbep_unc_ha::UNC_H_REQUESTS:READS:e=0:i=0:t=1",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_imc0::UNC_M_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xff,
	  .fstr = "snbep_unc_imc0::UNC_M_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_imc0::UNC_M_CLOCKTICKS:t=1",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "snbep_unc_imc0::UNC_M_CAS_COUNT:RD",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0304,
	  .fstr = "snbep_unc_imc0::UNC_M_CAS_COUNT:RD:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_imc0::UNC_M_POWER_CKE_CYCLES:RANK0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x183,
	  .fstr = "snbep_unc_imc0::UNC_M_POWER_CKE_CYCLES:RANK0:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_imc0::UNC_M_CAS_COUNT:WR",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xc04,
	  .fstr = "snbep_unc_imc0::UNC_M_CAS_COUNT:WR:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_pcu::UNC_P_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x0,
	  .fstr = "snbep_unc_pcu::UNC_P_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_pcu::UNC_P_CLOCKTICKS:t=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1000000,
	  .fstr = "snbep_unc_pcu::UNC_P_CLOCKTICKS:e=0:i=0:t=1",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_pcu::UNC_P_CORE0_TRANSITION_CYCLES:C6",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x200303,
	  .fstr = "snbep_unc_pcu::UNC_P_CORE0_TRANSITION_CYCLES:C6:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_pcu::UNC_P_FREQ_BAND0_CYCLES",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "snbep_unc_pcu::UNC_P_FREQ_BAND1_CYCLES",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "snbep_unc_pcu::UNC_P_FREQ_BAND2_CYCLES",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "snbep_unc_pcu::UNC_P_FREQ_BAND3_CYCLES",
	  .ret  = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "snbep_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:ff=32",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0xb,
	  .codes[1] = 0x20,
	  .fstr = "snbep_unc_pcu::UNC_P_FREQ_BAND0_CYCLES:e=0:i=0:t=0:ff=32",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_pcu::UNC_P_FREQ_BAND1_CYCLES:ff=16",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0xc,
	  .codes[1] = 0x1000,
	  .fstr = "snbep_unc_pcu::UNC_P_FREQ_BAND1_CYCLES:e=0:i=0:t=0:ff=16",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_pcu::UNC_P_FREQ_BAND2_CYCLES:ff=8",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0xd,
	  .codes[1] = 0x80000,
	  .fstr = "snbep_unc_pcu::UNC_P_FREQ_BAND2_CYCLES:e=0:i=0:t=0:ff=8",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_pcu::UNC_P_FREQ_BAND3_CYCLES:ff=40",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0xe,
	  .codes[1] = 0x28000000,
	  .fstr = "snbep_unc_pcu::UNC_P_FREQ_BAND3_CYCLES:e=0:i=0:t=0:ff=40",
	},
	{ SRC_LINE,
	  .name = "SNBEP_UNC_PCU::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x4080,
	  .fstr = "snbep_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0:e=0:i=0:t=0"
	},
	{ SRC_LINE,
	  .name = "SNBEP_UNC_PCU::UNC_P_POWER_STATE_OCCUPANCY:CORES_C3",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x8080,
	  .fstr = "snbep_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C3:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "SNBEP_UNC_PCU::UNC_P_POWER_STATE_OCCUPANCY:CORES_C6",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xc080,
	  .fstr = "snbep_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C6:e=0:i=0:t=0"
	},
	{ SRC_LINE,
	  .name = "SNBEP_UNC_PCU::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0:i",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x40004080,
	  .fstr = "snbep_unc_pcu::UNC_P_POWER_STATE_OCCUPANCY:CORES_C0:e=0:i=1:t=0"
	},
	{ SRC_LINE,
	  .name = "snbep_unc_qpi0::UNC_Q_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x14,
	  .fstr = "snbep_unc_qpi0::UNC_Q_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_qpi0::UNC_Q_RXL_FLITS_G0:DATA",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x201,
	  .fstr = "snbep_unc_qpi0::UNC_Q_RXL_FLITS_G0:DATA:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_qpi0::UNC_Q_RXL_FLITS_G0:IDLE:t=1:i",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1800101,
	  .fstr = "snbep_unc_qpi0::UNC_Q_RXL_FLITS_G0:IDLE:e=0:i=1:t=1",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_qpi0::UNC_Q_TXL_FLITS_G0:DATA",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x200,
	  .fstr = "snbep_unc_qpi0::UNC_Q_TXL_FLITS_G0:DATA:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_qpi0::UNC_Q_RXL_FLITS_G1:HOM",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x200602,
	  .fstr = "snbep_unc_qpi0::UNC_Q_RXL_FLITS_G1:HOM:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_qpi0::UNC_Q_TXL_FLITS_G1:HOM",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x200600,
	  .fstr = "snbep_unc_qpi0::UNC_Q_TXL_FLITS_G1:HOM:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_ubo::UNC_U_LOCK_CYCLES",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x44,
	  .fstr = "snbep_unc_ubo::UNC_U_LOCK_CYCLES:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_r2pcie::UNC_R2_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1,
	  .fstr = "snbep_unc_r2pcie::UNC_R2_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_r2pcie::UNC_R2_RING_AD_USED:ANY",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0xf07,
	  .fstr = "snbep_unc_r2pcie::UNC_R2_RING_AD_USED:ANY:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_r3qpi0::UNC_R3_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1,
	  .fstr = "snbep_unc_r3qpi0::UNC_R3_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_r3qpi0::UNC_R3_TXR_CYCLES_FULL:e=0:i=0:t=0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x25,
	  .fstr = "snbep_unc_r3qpi0::UNC_R3_TXR_CYCLES_FULL:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_r3qpi1::UNC_R3_CLOCKTICKS",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1,
	  .fstr = "snbep_unc_r3qpi1::UNC_R3_CLOCKTICKS:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snbep_unc_r3qpi1::UNC_R3_TXR_CYCLES_FULL:e=0:i=0:t=0",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x25,
	  .fstr = "snbep_unc_r3qpi1::UNC_R3_TXR_CYCLES_FULL:e=0:i=0:t=0",
	},
	{ SRC_LINE,
	  .name = "knc::cpu_clk_unhalted",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x53002a,
	  .fstr = "knc::CPU_CLK_UNHALTED:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "knc::instructions_executed",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x530016,
	  .fstr = "knc::INSTRUCTIONS_EXECUTED:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "knc::vpu_data_read",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x532000,
	  .fstr = "knc::VPU_DATA_READ:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "knc::vpu_data_read:t:c=1:i",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1f32000,
	  .fstr = "knc::VPU_DATA_READ:k=1:u=1:e=0:i=1:c=1:t=1",
	},
	{ SRC_LINE,
	  .name = "snb_ep::cpl_cycles:ring0_trans:u",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x155015c,
	  .fstr = "snb_ep::CPL_CYCLES:RING0:k=0:u=1:e=1:i=0:c=1:t=0",
	},
	{ SRC_LINE,
	  .name = "snb_ep::cpl_cycles:ring0_trans:e=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x157015cull,
	},
	{ SRC_LINE,
	  .name = "snb_ep::OFFCORE_REQUESTS_OUTSTanding:ALL_DATA_RD_CYCLES",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1530860,
	  .fstr = "snb_ep::OFFCORE_REQUESTS_OUTSTANDING:ALL_DATA_RD:k=1:u=1:e=0:i=0:c=1:t=0",
	},
	{ SRC_LINE,
	  .name = "snb_ep::uops_issued:core_stall_cycles:u:k",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x1f3010e,
	  .fstr = "snb_ep::UOPS_ISSUED:ANY:k=1:u=1:e=0:i=1:c=1:t=1",
	},
	{ SRC_LINE,
	  .name = "snb_ep::LLC_REFERences:k:t=1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x724f2e,
	  .fstr = "snb_ep::LAST_LEVEL_CACHE_REFERENCES:k=1:u=0:e=0:i=0:c=0:t=1",
	},
	{ SRC_LINE,
	  .name = "snb_ep::ITLB:0x1",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5301ae,
	  .fstr = "snb_ep::ITLB:0x1:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb_ep::mem_load_uops_llc_miss_retired:local_dram",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5301d3,
	  .fstr = "snb_ep::MEM_LOAD_UOPS_LLC_MISS_RETIRED:LOCAL_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb_ep::mem_load_uops_llc_miss_retired:remote_dram",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5304d3,
	  .fstr = "snb_ep::MEM_LOAD_UOPS_LLC_MISS_RETIRED:REMOTE_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb_ep::offcore_response_0:DMND_RFO:ANY_RESPONSE",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x10002,
	  .fstr = "snb_ep::OFFCORE_RESPONSE_0:DMND_RFO:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb_ep::offcore_response_0:DMND_RFO:ANY_REQUEST",
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x18fff,
	},
	{ SRC_LINE,
	  .name = "snb_ep::offcore_response_0:DMND_RFO",
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x10002,
	  .fstr = "snb_ep::OFFCORE_RESPONSE_0:DMND_RFO:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb_ep::offcore_response_0:any_response",
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x18fff,
	  .fstr = "snb_ep::OFFCORE_RESPONSE_0:ANY_REQUEST:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb_ep::offcore_response_0:NO_SUPP:SNP_NONE:PF_RFO:PF_IFETCH",
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x80020060,
	  .fstr = "snb_ep::OFFCORE_RESPONSE_0:PF_RFO:PF_IFETCH:NO_SUPP:SNP_NONE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb_ep::offcore_response_1:DMND_RFO:ANY_RESPONSE",
	  .ret  = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x10002,
	  .fstr = "snb_ep::OFFCORE_RESPONSE_1:DMND_RFO:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb_ep::offcore_response_1:DMND_RFO:ANY_REQUEST",
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x18fff,
	},
	{ SRC_LINE,
	  .name = "snb_ep::offcore_response_1:DMND_RFO",
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x10002,
	  .fstr = "snb_ep::OFFCORE_RESPONSE_1:DMND_RFO:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb_ep::offcore_response_1:any_response",
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x18fff,
	  .fstr = "snb_ep::OFFCORE_RESPONSE_1:ANY_REQUEST:ANY_RESPONSE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb_ep::offcore_response_1:NO_SUPP:SNP_NONE:PF_RFO:PF_IFETCH",
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x80020060,
	  .fstr = "snb_ep::OFFCORE_RESPONSE_1:PF_RFO:PF_IFETCH:NO_SUPP:SNP_NONE:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb_ep::offcore_response_1:ANY_REQUEST:LLC_MISS_LOCAL_DRAM",
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x3f80408fffull,
	  .fstr = "snb_ep::OFFCORE_RESPONSE_1:DMND_DATA_RD:DMND_RFO:DMND_IFETCH:WB:PF_DATA_RD:PF_RFO:PF_IFETCH:PF_LLC_DATA_RD:PF_LLC_RFO:PF_LLC_IFETCH:BUS_LOCKS:STRM_ST:OTHER:LLC_MISS_LOCAL_DRAM:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb_ep::offcore_response_1:ANY_REQUEST:LLC_MISS_REMOTE_DRAM",
	  .count = 2,
	  .codes[0] = 0x5301bb,
	  .codes[1] = 0x3fff808fffull,
	  .fstr = "snb_ep::OFFCORE_RESPONSE_1:DMND_DATA_RD:DMND_RFO:DMND_IFETCH:WB:PF_DATA_RD:PF_RFO:PF_IFETCH:PF_LLC_DATA_RD:PF_LLC_RFO:PF_LLC_IFETCH:BUS_LOCKS:STRM_ST:OTHER:LLC_MISS_REMOTE_DRAM:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "snb_ep::mem_trans_retired:latency_above_threshold",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0]=0x5301cd,
	  .codes[1] = 3,
	  .fstr = "snb_ep::MEM_TRANS_RETIRED:LATENCY_ABOVE_THRESHOLD:k=1:u=1:e=0:i=0:c=0:t=0:ldlat=3",
	},
	{ SRC_LINE,
	  .name = "snb_ep::mem_trans_retired:latency_above_threshold:ldlat=2",
	  .ret = PFM_ERR_ATTR_VAL,
	},
	{ SRC_LINE,
	  .name = "snb_ep::mem_trans_retired:latency_above_threshold:ldlat=3",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0]=0x5301cd,
	  .codes[1] = 3,
	  .fstr = "snb_ep::MEM_TRANS_RETIRED:LATENCY_ABOVE_THRESHOLD:k=1:u=1:e=0:i=0:c=0:t=0:ldlat=3",
	},
	{ SRC_LINE,
	  .name = "snb_ep::mem_trans_retired:latency_above_threshold:ldlat=1000000",
	  .ret = PFM_ERR_ATTR_VAL,
	},
	{ SRC_LINE,
	  .name = "snb::mem_trans_retired:latency_above_threshold",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301cd,
	  .codes[1] = 3,
	  .fstr = "snb::MEM_TRANS_RETIRED:LATENCY_ABOVE_THRESHOLD:k=1:u=1:e=0:i=0:c=0:t=0:ldlat=3",
	},
	{ SRC_LINE,
	  .name = "snb::mem_trans_retired:latency_above_threshold:ldlat=2",
	  .ret = PFM_ERR_ATTR_VAL,
	},
	{ SRC_LINE,
	  .name = "snb::mem_trans_retired:latency_above_threshold:ldlat=3",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301cd,
	  .codes[1] = 3,
	  .fstr = "snb::MEM_TRANS_RETIRED:LATENCY_ABOVE_THRESHOLD:k=1:u=1:e=0:i=0:c=0:t=0:ldlat=3",
	},
	{ SRC_LINE,
	  .name = "snb::mem_trans_retired:latency_above_threshold:ldlat=1000000",
	  .ret = PFM_ERR_ATTR_VAL,
	},
	{ SRC_LINE,
	  .name = "ivb::mem_trans_retired:latency_above_threshold",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301cd,
	  .codes[1] = 3,
	  .fstr = "ivb::MEM_TRANS_RETIRED:LATENCY_ABOVE_THRESHOLD:k=1:u=1:e=0:i=0:c=0:t=0:ldlat=3",
	},
	{ SRC_LINE,
	  .name = "ivb::mem_trans_retired:latency_above_threshold:ldlat=2",
	  .ret = PFM_ERR_ATTR_VAL,
	},
	{ SRC_LINE,
	  .name = "ivb::mem_trans_retired:latency_above_threshold:ldlat=3:u",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5101cd,
	  .codes[1] = 3,
	  .fstr = "ivb::MEM_TRANS_RETIRED:LATENCY_ABOVE_THRESHOLD:k=0:u=1:e=0:i=0:c=0:t=0:ldlat=3",
	},
	{ SRC_LINE,
	  .name = "ivb::mem_trans_retired:latency_above_threshold:ldlat=1000000",
	  .ret = PFM_ERR_ATTR_VAL,
	},
	{ SRC_LINE,
	  .name = "nhm::mem_inst_retired:latency_above_threshold",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x53100b,
	  .codes[1] = 3,
	  .fstr = "nhm::MEM_INST_RETIRED:LATENCY_ABOVE_THRESHOLD:k=1:u=1:e=0:i=0:c=0:t=0:ldlat=3",
	},
	{ SRC_LINE,
	  .name = "nhm::mem_inst_retired:latency_above_threshold:ldlat=2",
	  .ret = PFM_ERR_ATTR_VAL,
	},
	{ SRC_LINE,
	  .name = "nhm::mem_inst_retired:latency_above_threshold:ldlat=3",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x53100b,
	  .codes[1] = 3,
	  .fstr = "nhm::MEM_INST_RETIRED:LATENCY_ABOVE_THRESHOLD:k=1:u=1:e=0:i=0:c=0:t=0:ldlat=3",
	},
	{ SRC_LINE,
	  .name = "nhm::mem_inst_retired:latency_above_threshold:ldlat=1000000",
	  .ret = PFM_ERR_ATTR_VAL,
	},
	{ SRC_LINE,
	  .name = "wsm::mem_inst_retired:latency_above_threshold",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x53100b,
	  .codes[1] = 3,
	  .fstr = "wsm::MEM_INST_RETIRED:LATENCY_ABOVE_THRESHOLD:k=1:u=1:e=0:i=0:c=0:t=0:ldlat=3",
	},
	{ SRC_LINE,
	  .name = "wsm::mem_inst_retired:latency_above_threshold:ldlat=2",
	  .ret = PFM_ERR_ATTR_VAL,
	},
	{ SRC_LINE,
	  .name = "wsm::mem_inst_retired:latency_above_threshold:ldlat=3",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x53100b,
	  .codes[1] = 3,
	  .fstr = "wsm::MEM_INST_RETIRED:LATENCY_ABOVE_THRESHOLD:k=1:u=1:e=0:i=0:c=0:t=0:ldlat=3",
	},
	{ SRC_LINE,
	  .name = "wsm::mem_inst_retired:latency_above_threshold:ldlat=1000000",
	  .ret = PFM_ERR_ATTR_VAL,
	},
	{ SRC_LINE,
	  .name = "amd64_fam15h_interlagos::LINK_TRANSMIT_BANDWIDTH_LINK_0:NOP_DW_SENT",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5308f6,
	  .fstr = "amd64_fam15h_interlagos::LINK_TRANSMIT_BANDWIDTH_LINK_0:NOP_DW_SENT:SUBLINK_0",
	},
	{ SRC_LINE,
	  .name = "amd64_fam15h_interlagos::LINK_TRANSMIT_BANDWIDTH_LINK_0:ALL",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x533ff6,
	  .fstr = "amd64_fam15h_interlagos::LINK_TRANSMIT_BANDWIDTH_LINK_0:ALL:SUBLINK_0",
	},
	{ SRC_LINE,
	  .name = "amd64_fam15h_interlagos::LINK_TRANSMIT_BANDWIDTH_LINK_0:ALL:SUBLINK_1",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x53bff6,
	  .fstr = "amd64_fam15h_interlagos::LINK_TRANSMIT_BANDWIDTH_LINK_0:ALL:SUBLINK_1",
	},
	{ SRC_LINE,
	  .name = "amd64_fam15h_interlagos::LINK_TRANSMIT_BANDWIDTH_LINK_0:COMMAND_DW_SENT:DATA_DW_SENT",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5303f6,
	  .fstr = "amd64_fam15h_interlagos::LINK_TRANSMIT_BANDWIDTH_LINK_0:COMMAND_DW_SENT:DATA_DW_SENT:SUBLINK_0",
	},
	{ SRC_LINE,
	  .name = "amd64_fam15h_interlagos::DISPATCHED_FPU_OPS:0x4ff:u",
	  .ret  = PFM_ERR_ATTR
	},
	{ SRC_LINE,
	  .name = "amd64_fam15h_interlagos::DISPATCHED_FPU_OPS:0xff:u",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x51ff00ull,
	  .fstr = "amd64_fam15h_interlagos::DISPATCHED_FPU_OPS:0xff:k=0:u=1:e=0:i=0:c=0:h=0:g=0"
	},
	{ SRC_LINE,
	  .name = "ivb_ep::mem_load_uops_llc_miss_retired:local_dram",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5303d3,
	  .fstr = "ivb_ep::MEM_LOAD_UOPS_LLC_MISS_RETIRED:LOCAL_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb_ep::mem_load_uops_llc_miss_retired:remote_dram",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x530cd3,
	  .fstr = "ivb_ep::MEM_LOAD_UOPS_LLC_MISS_RETIRED:REMOTE_DRAM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb_ep::mem_load_uops_llc_miss_retired:remote_hitm",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5310d3,
	  .fstr = "ivb_ep::MEM_LOAD_UOPS_LLC_MISS_RETIRED:REMOTE_HITM:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb_ep::mem_load_uops_llc_miss_retired:remote_fwd",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5320d3,
	  .fstr = "ivb_ep::MEM_LOAD_UOPS_LLC_MISS_RETIRED:REMOTE_FWD:k=1:u=1:e=0:i=0:c=0:t=0",
	},
	{ SRC_LINE,
	  .name = "ivb::mem_load_uops_llc_miss_retired:remote_dram",
	  .ret = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "ivb_ep::offcore_response_0:any_request:LLC_MISS_REMOTE",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301b7,
	  .codes[1] = 0x3fff808fffULL,
	  .fstr = "ivb_ep::OFFCORE_RESPONSE_0:DMND_DATA_RD:DMND_RFO:DMND_IFETCH:WB:PF_DATA_RD:PF_RFO:PF_IFETCH:PF_LLC_DATA_RD:PF_LLC_RFO:PF_LLC_IFETCH:BUS_LOCKS:STRM_ST:OTHER:LLC_MISS_REMOTE_DRAM:SNP_ANY:k=1:u=1:e=0:i=0:c=0:t=0"
	},
	{ SRC_LINE,
	  .name = "hsw::mem_trans_retired:latency_above_threshold:ldlat=3:u",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5101cd,
	  .codes[1] = 3,
	  .fstr = "hsw::MEM_TRANS_RETIRED:LOAD_LATENCY:k=0:u=1:e=0:i=0:c=0:t=0:ldlat=3:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "hsw::mem_trans_retired:latency_above_threshold:ldlat=1000000",
	  .ret = PFM_ERR_ATTR_VAL,
	},
	{ SRC_LINE,
	  .name = "hsw::mem_trans_retired:load_latency",
	  .ret = PFM_SUCCESS,
	  .count = 2,
	  .codes[0] = 0x5301cd,
	  .codes[1] = 3,
	  .fstr = "hsw::MEM_TRANS_RETIRED:LOAD_LATENCY:k=1:u=1:e=0:i=0:c=0:t=0:ldlat=3:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "hsw::mem_trans_retired:load_latency:ldlat=1000000",
	  .ret = PFM_ERR_ATTR_VAL,
	},
	{ SRC_LINE,
	  .name = "hsw::mem_trans_retired:latency_above_threshold:ldlat=2:intx=0:intxcp=0",
	  .ret = PFM_ERR_ATTR_VAL,
	},
	{ SRC_LINE,
	  .name = "hsw::inst_Retired:any_p:intx",
	  .count = 1,
	  .codes[0] = 0x1005300c0,
	  .fstr = "hsw::INST_RETIRED:ANY_P:k=1:u=1:e=0:i=0:c=0:t=0:intx=1:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "hsw::inst_Retired:any_p:intx:intxcp",
	  .count = 1,
	  .codes[0] = 0x3005300c0,
	  .fstr = "hsw::INST_RETIRED:ANY_P:k=1:u=1:e=0:i=0:c=0:t=0:intx=1:intxcp=1",
	},
	{ SRC_LINE,
	  .name = "hsw::inst_Retired:any_p:intx=0:intxcp",
	  .count = 1,
	  .codes[0] = 0x2005300c0,
	  .fstr = "hsw::INST_RETIRED:ANY_P:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=1",
	},
	{ SRC_LINE,
	  .name = "hsw::cycle_activity:cycles_l2_pending",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x15301a3,
	  .fstr = "hsw::CYCLE_ACTIVITY:CYCLES_L2_PENDING:k=1:u=1:e=0:i=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "hsw::cycle_activity:cycles_l2_pending:c=8",
	  .ret = PFM_ERR_ATTR,
	},
	{ SRC_LINE,
	  .name = "hsw::hle_retired:aborted",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5304c8,
	  .fstr = "hsw::HLE_RETIRED:ABORTED:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "hsw::rtm_retired:aborted",
	  .ret = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5304c9,
	  .fstr = "hsw::RTM_RETIRED:ABORTED:k=1:u=1:e=0:i=0:c=0:t=0:intx=0:intxcp=0",
	},
	{ SRC_LINE,
	  .name = "ivb_unc_cbo0::unc_clockticks",
	  .ret  = PFM_SUCCESS,
	  .count = 1,
	  .codes[0] = 0x5000ff,
	  .fstr = "ivb_unc_cbo0::UNC_CLOCKTICKS",
	},
	{ SRC_LINE,
	  .name = "ivb_unc_cbo1::unc_clockticks",
	  .ret  = PFM_ERR_NOTFOUND
	},
};
#define NUM_TEST_EVENTS (int)(sizeof(x86_test_events)/sizeof(test_event_t))

static int check_test_events(FILE *fp)
{
	const test_event_t *e;
	char *fstr;
	uint64_t *codes;
	int count, i, j;
	int ret, errors = 0;

	for (i=0, e = x86_test_events; i < NUM_TEST_EVENTS; i++, e++) {
		codes = NULL;
		count = 0;
		fstr = NULL;
		ret = pfm_get_event_encoding(e->name, PFM_PLM0 | PFM_PLM3, &fstr, NULL, &codes, &count);
		if (ret != e->ret) {
			fprintf(fp,"Line %d, Event%d %s, ret=%s(%d) expected %s(%d)\n", e->line, i, e->name, pfm_strerror(ret), ret, pfm_strerror(e->ret), e->ret);
			errors++;
		} else {
			if (ret != PFM_SUCCESS) {
				if (fstr) {
					fprintf(fp,"Line %d, Event%d %s, expected fstr NULL but it is not\n", e->line, i, e->name);
					errors++;
				}
				if (count != 0) {
					fprintf(fp,"Line %d, Event%d %s, expected count=0 instead of %d\n", e->line, i, e->name, count);
					errors++;
				}
				if (codes) {
					fprintf(fp,"Line %d, Event%d %s, expected codes[] NULL but it is not\n", e->line, i, e->name);
					errors++;
				}
			} else {
				if (count != e->count) {
					fprintf(fp,"Line %d, Event%d %s, count=%d expected %d\n", e->line, i, e->name, count, e->count);
					errors++;
				}
				for (j=0; j < count; j++) {
					if (codes[j] != e->codes[j]) {
						fprintf(fp,"Line %d, Event%d %s, codes[%d]=%#"PRIx64" expected %#"PRIx64"\n", e->line, i, e->name, j, codes[j], e->codes[j]);
						errors++;
					}
				}
				if (e->fstr && strcmp(fstr, e->fstr)) {
					fprintf(fp,"Line %d, Event%d %s, fstr=%s expected %s\n", e->line, i, e->name, fstr, e->fstr);
					errors++;
				}
			}
		}
		if (codes)
			free(codes);
		if (fstr)
			free(fstr);
	}
	printf("\t %d x86 events: %d errors\n", i, errors);
	return errors;
}

int
validate_arch(FILE *fp)
{
	return check_test_events(fp);
}
