/*
 * COPYRIGHT (c) International Business Machines Corp. 2011-2017
 *
 * This program is provided under the terms of the Common Public License,
 * version 1.0 (CPL-1.0). Any use, reproduction or distribution for this
 * software constitutes recipient's acceptance of CPL-1.0 terms which can be
 * found in the file LICENSE file or at
 * https://opensource.org/licenses/cpl1.0.php
 */

#include "pkcs11types.h"

#ifndef _AES_H_
#define _AES_H_
#endif
#if !defined(TRUE)
#define TRUE 1
#endif

#if !defined(FALSE)
#define FALSE 0
#endif

#define MAX_KEY_SIZE  64
#define MAX_TEXT_SIZE 4096
#define MAX_IV_SIZE  64
#define MAX_COUNTER_SIZE 16
#define MAX_AAD_SIZE 512
#define MAX_TAG_SIZE 16
#define AES_BLOCK_SIZE	16
#define AES_COUNTER_VALUE "0123456789012345"
#define AES_IV_VALUE "1234567890123456"
#define AES_KEY_LEN 32

#define AES_IV_SIZE 16
#define AES_COUNTER_SIZE 16
#define MAX_CHUNKS 8


struct aes_test_vector {
    unsigned char key[MAX_KEY_SIZE];
    unsigned char klen;
    unsigned char plaintext[MAX_TEXT_SIZE];
    unsigned char plen;
    unsigned char ciphertext[MAX_TEXT_SIZE];
    unsigned char clen;
    unsigned char iv[MAX_IV_SIZE];       // aes cbc mode
    unsigned char ivlen;        // aes cbc mode
    unsigned char counter[MAX_COUNTER_SIZE];     // aes ctr mode
    unsigned char counterlen;   // aes ctr mode
    unsigned int counterbits;   // aes ctr mode
    unsigned char aad[MAX_AAD_SIZE];    // aes gcm mode
    unsigned int aadlen;        // aes gcm mode
    unsigned int tag[MAX_TAG_SIZE];     // aes gcm mode
    unsigned int taglen;        // aes gcm mode
    int chunks_plain[MAX_CHUNKS];
    int num_chunks_plain;
    int chunks_ciph[MAX_CHUNKS];
    int num_chunks_ciph;
};

struct published_test_suite_info {
    const char *name;
    unsigned int tvcount;
    struct aes_test_vector *tv;
    unsigned int size;
    CK_MECHANISM mech;
};

struct generated_test_suite_info {
    const char *name;
    CK_MECHANISM mech;
    unsigned int size;
};

char aes_iv[] = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f
};

struct CK_AES_CTR_PARAMS aesctr = {
    .ulCounterBits = AES_COUNTER_SIZE,
    .cb = {0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
           0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff},
};

struct CK_GCM_PARAMS aesgcm = {
    .ulIvLen = AES_BLOCK_SIZE,
    .ulIvBits = AES_BLOCK_SIZE * 8,
    .ulAADLen = 0,
    .ulTagBits = 16,
};

struct CK_MECHANISM aes_keygen = {
    .mechanism = CKM_AES_KEY_GEN,
    .ulParameterLen = 0,
    .pParameter = NULL,
};

struct mac_test_vector {
    unsigned char key[MAX_KEY_SIZE];
    unsigned char klen;
    unsigned char msg[MAX_TEXT_SIZE];
    unsigned char mlen;
    unsigned char mac[MAX_KEY_SIZE];
    unsigned char tlen;
    int chunks_msg[MAX_CHUNKS];
    int num_chunks_message;
};

struct published_mac_test_suite_info {
    const char *name;
    unsigned int tvcount;
    struct mac_test_vector *tv;
    CK_MECHANISM mech;
};

/** NIST Special Publication 800-38A
    http://www.csrc.nist.gov/publications/nistpubs/800-38a/sp800-38a.pdf
**/
static struct aes_test_vector aes_ecb_tv[] = {
    {                           // #0
        .key = {0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6,
                0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c},
        .klen = 16,
        .plaintext = {0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
                      0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
                      0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
                      0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
                      0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
                      0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
                      0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
                      0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10},
        .plen = 64,
        .ciphertext = {0x3a, 0xd7, 0x7b, 0xb4, 0x0d, 0x7a, 0x36, 0x60,
                       0xa8, 0x9e, 0xca, 0xf3, 0x24, 0x66, 0xef, 0x97,
                       0xf5, 0xd3, 0xd5, 0x85, 0x03, 0xb9, 0x69, 0x9d,
                       0xe7, 0x85, 0x89, 0x5a, 0x96, 0xfd, 0xba, 0xaf,
                       0x43, 0xb1, 0xcd, 0x7f, 0x59, 0x8e, 0xce, 0x23,
                       0x88, 0x1b, 0x00, 0xe3, 0xed, 0x03, 0x06, 0x88,
                       0x7b, 0x0c, 0x78, 0x5e, 0x27, 0xe8, 0xad, 0x3f,
                       0x82, 0x23, 0x20, 0x71, 0x04, 0x72, 0x5d, 0xd4},
        .clen = 64,
     }, {                       // #1
         .key = {0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52,
                 0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5,
                 0x62, 0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b},
         .klen = 24,
         .plaintext = {0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
                       0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
                       0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
                       0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
                       0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
                       0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
                       0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
                       0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10},
         .plen = 64,
         .num_chunks_plain = 4,
         .chunks_plain = {16, 16, 16, 16},
         .ciphertext = {0xbd, 0x33, 0x4f, 0x1d, 0x6e, 0x45, 0xf2, 0x5f,
                        0xf7, 0x12, 0xa2, 0x14, 0x57, 0x1f, 0xa5, 0xcc,
                        0x97, 0x41, 0x04, 0x84, 0x6d, 0x0a, 0xd3, 0xad,
                        0x77, 0x34, 0xec, 0xb3, 0xec, 0xee, 0x4e, 0xef,
                        0xef, 0x7a, 0xfd, 0x22, 0x70, 0xe2, 0xe6, 0x0a,
                        0xdc, 0xe0, 0xba, 0x2f, 0xac, 0xe6, 0x44, 0x4e,
                        0x9a, 0x4b, 0x41, 0xba, 0x73, 0x8d, 0x6c, 0x72,
                        0xfb, 0x16, 0x69, 0x16, 0x03, 0xc1, 0x8e, 0x0e},
         .clen = 64,
         .num_chunks_ciph = 4,
         .chunks_ciph = {16, 16, 16, 16},
     }, {                   // #2
         .key = {0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe,
                 0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
                 0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7,
                 0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4},
         .klen = 32,
         .plaintext = {0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
                       0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
                       0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
                       0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
                       0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
                       0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
                       0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
                       0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10},
         .plen = 64,
         .num_chunks_plain = 5,
         .chunks_plain = {25, 0, 8, -1, 31},
         .ciphertext = {0xf3, 0xee, 0xd1, 0xbd, 0xb5, 0xd2, 0xa0, 0x3c,
                        0x06, 0x4b, 0x5a, 0x7e, 0x3d, 0xb1, 0x81, 0xf8,
                        0x59, 0x1c, 0xcb, 0x10, 0xd4, 0x10, 0xed, 0x26,
                        0xdc, 0x5b, 0xa7, 0x4a, 0x31, 0x36, 0x28, 0x70,
                        0xb6, 0xed, 0x21, 0xb9, 0x9c, 0xa6, 0xf4, 0xf9,
                        0xf1, 0x53, 0xe7, 0xb1, 0xbe, 0xaf, 0xed, 0x1d,
                        0x23, 0x30, 0x4b, 0x7a, 0x39, 0xf9, 0xf3, 0xff,
                        0x06, 0x7d, 0x8d, 0x8f, 0x9e, 0x24, 0xec, 0xc7},
         .clen = 64,
         .num_chunks_ciph = 5,
         .chunks_ciph = {25, 0, 8, -1, 31},
     },
};

/** NIST Special Publication 800-38A
    http://www.csrc.nist.gov/publications/nistpubs/800-38a/sp800-38a.pdf
**/
static struct aes_test_vector aes_cbc_tv[] = {
    {                           // #0
        .key = {0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6,
                0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c},
        .klen = 16,
        .plaintext = {0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
                      0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
                      0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
                      0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
                      0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
                      0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
                      0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
                      0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10},
        .plen = 64,
        .ciphertext = {0x76, 0x49, 0xab, 0xac, 0x81, 0x19, 0xb2, 0x46,
                       0xce, 0xe9, 0x8e, 0x9b, 0x12, 0xe9, 0x19, 0x7d,
                       0x50, 0x86, 0xcb, 0x9b, 0x50, 0x72, 0x19, 0xee,
                       0x95, 0xdb, 0x11, 0x3a, 0x91, 0x76, 0x78, 0xb2,
                       0x73, 0xbe, 0xd6, 0xb8, 0xe3, 0xc1, 0x74, 0x3b,
                       0x71, 0x16, 0xe6, 0x9e, 0x22, 0x22, 0x95, 0x16,
                       0x3f, 0xf1, 0xca, 0xa1, 0x68, 0x1f, 0xac, 0x09,
                       0x12, 0x0e, 0xca, 0x30, 0x75, 0x86, 0xe1, 0xa7},
        .clen = 64,
        .iv = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
               0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f},
        .ivlen = 16,
    }, {                       // #1
        .key = {0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52,
                0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5,
                0x62, 0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b},
        .klen = 24,
        .plaintext = {0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
                      0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
                      0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
                      0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
                      0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
                      0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
                      0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
                      0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10},
        .plen = 64,
        .num_chunks_plain = 4,
        .chunks_plain = {16, 16, 16, 16},
        .ciphertext = {0x4f, 0x02, 0x1d, 0xb2, 0x43, 0xbc, 0x63, 0x3d,
                       0x71, 0x78, 0x18, 0x3a, 0x9f, 0xa0, 0x71, 0xe8,
                       0xb4, 0xd9, 0xad, 0xa9, 0xad, 0x7d, 0xed, 0xf4,
                       0xe5, 0xe7, 0x38, 0x76, 0x3f, 0x69, 0x14, 0x5a,
                       0x57, 0x1b, 0x24, 0x20, 0x12, 0xfb, 0x7a, 0xe0,
                       0x7f, 0xa9, 0xba, 0xac, 0x3d, 0xf1, 0x02, 0xe0,
                       0x08, 0xb0, 0xe2, 0x79, 0x88, 0x59, 0x88, 0x81,
                       0xd9, 0x20, 0xa9, 0xe6, 0x4f, 0x56, 0x15, 0xcd},
        .clen = 64,
        .iv = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
               0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f},
        .ivlen = 16,
        .num_chunks_ciph = 4,
        .chunks_ciph = {16, 16, 16, 16},
    }, {                   // #2
        .key = {0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe,
                0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
                0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7,
                0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4},
        .klen = 32,
        .plaintext = {0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
                      0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
                      0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
                      0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
                      0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
                      0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
                      0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
                      0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10},
        .plen = 64,
        .num_chunks_plain = 5,
        .chunks_plain = {25, 0, 8, -1, 31},
        .ciphertext = {0xf5, 0x8c, 0x4c, 0x04, 0xd6, 0xe5, 0xf1, 0xba,
                       0x77, 0x9e, 0xab, 0xfb, 0x5f, 0x7b, 0xfb, 0xd6,
                       0x9c, 0xfc, 0x4e, 0x96, 0x7e, 0xdb, 0x80, 0x8d,
                       0x67, 0x9f, 0x77, 0x7b, 0xc6, 0x70, 0x2c, 0x7d,
                       0x39, 0xf2, 0x33, 0x69, 0xa9, 0xd9, 0xba, 0xcf,
                       0xa5, 0x30, 0xe2, 0x63, 0x04, 0x23, 0x14, 0x61,
                       0xb2, 0xeb, 0x05, 0xe2, 0xc3, 0x9b, 0xe9, 0xfc,
                       0xda, 0x6c, 0x19, 0x07, 0x8c, 0x6a, 0x9d, 0x1b},
        .clen = 64,
        .iv = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
               0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f},
        .ivlen = 16,
        .num_chunks_ciph = 5,
        .chunks_ciph = {25, 0, 8, -1, 31},
    },
};

/** NIST Special Publication 800-38A
    http://www.csrc.nist.gov/publications/nistpubs/800-38a/sp800-38a.pdf
**/
static struct aes_test_vector aes_ctr_tv[] = {
    {                           // #0
        .key = {0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6,
                0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c},
        .klen = 16,
        .counter = {0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
                    0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff},
        .counterlen = 16,
        .counterbits = 128,
        .plaintext = {0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
                      0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
                      0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
                      0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
                      0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
                      0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
                      0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
                      0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10},
        .plen = 64,
        .ciphertext = {0x87, 0x4d, 0x61, 0x91, 0xb6, 0x20, 0xe3, 0x26,
                       0x1b, 0xef, 0x68, 0x64, 0x99, 0x0d, 0xb6, 0xce,
                       0x98, 0x06, 0xf6, 0x6b, 0x79, 0x70, 0xfd, 0xff,
                       0x86, 0x17, 0x18, 0x7b, 0xb9, 0xff, 0xfd, 0xff,
                       0x5a, 0xe4, 0xdf, 0x3e, 0xdb, 0xd5, 0xd3, 0x5e,
                       0x5b, 0x4f, 0x09, 0x02, 0x0d, 0xb0, 0x3e, 0xab,
                       0x1e, 0x03, 0x1d, 0xda, 0x2f, 0xbe, 0x03, 0xd1,
                       0x79, 0x21, 0x70, 0xa0, 0xf3, 0x00, 0x9c, 0xee},
        .clen = 64,
    }, {                       // #1
        .key = {0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52,
                0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5,
                0x62, 0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b},
        .klen = 24,
        .counter = {0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
                    0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff},
        .counterlen = 16,
        .counterbits = 128,
        .plaintext = {0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
                      0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
                      0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
                      0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
                      0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
                      0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
                      0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
                      0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10},
        .plen = 64,
        .num_chunks_plain = 4,
        .chunks_plain = {16, 16, 16, 16},
        .ciphertext = {0x1a, 0xbc, 0x93, 0x24, 0x17, 0x52, 0x1c, 0xa2,
                       0x4f, 0x2b, 0x04, 0x59, 0xfe, 0x7e, 0x6e, 0x0b,
                       0x09, 0x03, 0x39, 0xec, 0x0a, 0xa6, 0xfa, 0xef,
                       0xd5, 0xcc, 0xc2, 0xc6, 0xf4, 0xce, 0x8e, 0x94,
                       0x1e, 0x36, 0xb2, 0x6b, 0xd1, 0xeb, 0xc6, 0x70,
                       0xd1, 0xbd, 0x1d, 0x66, 0x56, 0x20, 0xab, 0xf7,
                       0x4f, 0x78, 0xa7, 0xf6, 0xd2, 0x98, 0x09, 0x58,
                       0x5a, 0x97, 0xda, 0xec, 0x58, 0xc6, 0xb0, 0x50},
        .clen = 64,
        .num_chunks_ciph = 4,
        .chunks_ciph = {16, 16, 16, 16},
    }, {                   // #2
        .key = {0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe,
                0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
                0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7,
                0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4},
        .klen = 32,
        .counter = {0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
                    0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff},
        .counterlen = 16,
        .counterbits = 128,
        .plaintext = {0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
                      0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
                      0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
                      0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
                      0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
                      0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
                      0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
                      0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10},
        .plen = 64,
        .num_chunks_plain = 5,
        .chunks_plain = {25, 0, 8, -1, 31},
        .ciphertext = {0x60, 0x1e, 0xc3, 0x13, 0x77, 0x57, 0x89, 0xa5,
                       0xb7, 0xa7, 0xf5, 0x04, 0xbb, 0xf3, 0xd2, 0x28,
                       0xf4, 0x43, 0xe3, 0xca, 0x4d, 0x62, 0xb5, 0x9a,
                       0xca, 0x84, 0xe9, 0x90, 0xca, 0xca, 0xf5, 0xc5,
                       0x2b, 0x09, 0x30, 0xda, 0xa2, 0x3d, 0xe9, 0x4c,
                       0xe8, 0x70, 0x17, 0xba, 0x2d, 0x84, 0x98, 0x8d,
                       0xdf, 0xc9, 0xc5, 0x8d, 0xb6, 0x7a, 0xad, 0xa6,
                       0x13, 0xc2, 0xdd, 0x08, 0x45, 0x79, 0x41, 0xa6},
        .clen = 64,
        .num_chunks_ciph = 5,
        .chunks_ciph = {25, 0, 8, -1, 31},
    },
};

static struct aes_test_vector aes_gcm_tv[] = {
    {                           /* #0 test with chunks */
        .key = {0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c,
                0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08},
        .klen = 16,
        .iv = {0xca, 0xfe, 0xba, 0xbe, 0xfa, 0xce, 0xdb, 0xad,
               0xde, 0xca, 0xf8, 0x88,},
        .ivlen = 12,
        .aad = {0},
        .aadlen = 0,
        .tag = {0x4d, 0x5c, 0x2a, 0xf3, 0x27, 0xcd, 0x64, 0xa6,
                0x2c, 0xf3, 0x5a, 0xbd, 0x2b, 0xa6, 0xfa, 0xb4},
        .taglen = 128,
        .plaintext = {0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
                      0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
                      0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
                      0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
                      0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
                      0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
                      0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
                      0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55},
        .plen = 64,
        .num_chunks_plain = 5,
        .chunks_plain = {20, 5, 7, 1, 31},
        .ciphertext = {0x42, 0x83, 0x1e, 0xc2, 0x21, 0x77, 0x74, 0x24,
                       0x4b, 0x72, 0x21, 0xb7, 0x84, 0xd0, 0xd4, 0x9c,
                       0xe3, 0xaa, 0x21, 0x2f, 0x2c, 0x02, 0xa4, 0xe0,
                       0x35, 0xc1, 0x7e, 0x23, 0x29, 0xac, 0xa1, 0x2e,
                       0x21, 0xd5, 0x14, 0xb2, 0x54, 0x66, 0x93, 0x1c,
                       0x7d, 0x8f, 0x6a, 0x5a, 0xac, 0x84, 0xaa, 0x05,
                       0x1b, 0xa3, 0x0b, 0x39, 0x6a, 0x0a, 0xac, 0x97,
                       0x3d, 0x58, 0xe0, 0x91, 0x47, 0x3f, 0x59, 0x85,
                       0x4d, 0x5c, 0x2a, 0xf3, 0x27, 0xcd, 0x64, 0xa6,
                       0x2c, 0xf3, 0x5a, 0xbd, 0x2b, 0xa6, 0xfa, 0xb4},
        .clen = 80,
        .num_chunks_ciph = 7,
        .chunks_ciph = {20, 5, 7, 1, 31, 10, 6},
    }, {
        /* #1 - NIST -
         * [Keylen = 128]
         * [IVlen  = 96]
         * [PTlen  = 0]
         * [AADlen = 0]
         * [Taglen = 128]
         */
        .key = {0x11, 0x75, 0x4c, 0xd7, 0x2a, 0xec, 0x30, 0x9b,
                0xf5, 0x2f, 0x76, 0x87, 0x21, 0x2e, 0x89, 0x57},
        .klen = 16,
        .iv = {0x3c, 0x81, 0x9d, 0x9a, 0x9b, 0xed, 0x08, 0x76,
               0x15, 0x03, 0x0b, 0x65},
        .ivlen = 12,
        .aad = {0},
        .aadlen = 0,
        .tag = {0x25, 0x03, 0x27, 0xc6, 0x74, 0xaa, 0xf4, 0x77,
                0xae, 0xf2, 0x67, 0x57, 0x48, 0xcf, 0x69, 0x71},
        .taglen = 128,
        .plaintext = {0},
        .plen = 0,
        .ciphertext = {0x25, 0x03, 0x27, 0xc6, 0x74, 0xaa, 0xf4, 0x77,
                       0xae, 0xf2, 0x67, 0x57, 0x48, 0xcf, 0x69, 0x71},
        .clen = 16,
    }, {
        /* #2 - NIST -
         * [Keylen = 128]
         * [IVlen  = 96]
         * [PTlen  = 128]
         * [AADlen = 0]
         * [Taglen = 128]
         */
        .key = {0x99, 0x71, 0x07, 0x10, 0x59, 0xab, 0xc0, 0x09,
                0xe4, 0xf2, 0xbd, 0x69, 0x86, 0x9d, 0xb3, 0x38},
        .klen = 16,
        .iv = {0x07, 0xa9, 0xa9, 0x5e, 0xa3, 0x82, 0x1e, 0x9c,
               0x13, 0xc6, 0x32, 0x51},
        .ivlen = 12,
        .aad = {0},
        .aadlen = 0,
        .tag = {0x78, 0x70, 0xd9, 0x11, 0x7f, 0x54, 0x81, 0x1a,
                0x34, 0x69, 0x70, 0xf1, 0xde, 0x09, 0x0c, 0x41},
        .taglen = 128,
        .plaintext = {0xf5, 0x4b, 0xc3, 0x50, 0x1f, 0xed, 0x4f, 0x6f,
                      0x6d, 0xfb, 0x5e, 0xa8, 0x01, 0x06, 0xdf, 0x0b,
                      0xd8, 0x36, 0xe6, 0x82, 0x62, 0x25, 0xb7, 0x5c,
                      0x02, 0x22, 0xf6, 0xe8, 0x59, 0xb3, 0x59, 0x83},
        .plen = 32,
        .ciphertext = {0x05, 0x56, 0xc1, 0x59, 0xf8, 0x4e, 0xf3, 0x6c,
                       0xb1, 0x60, 0x2b, 0x45, 0x26, 0xb1, 0x20, 0x09,
                       0xc7, 0x75, 0x61, 0x1b, 0xff, 0xb6, 0x4d, 0xc0,
                       0xd9, 0xca, 0x92, 0x97, 0xcd, 0x2c, 0x6a, 0x01,
                       0x78, 0x70, 0xd9, 0x11, 0x7f, 0x54, 0x81, 0x1a,
                       0x34, 0x69, 0x70, 0xf1, 0xde, 0x09, 0x0c, 0x41},
        .clen = 48,
    }, {
        /* #3 - NIST -
         * [Keylen = 128]
         * [IVlen  = 96]
         * [PTlen  = 104]
         * [AADlen = 0]
         * [Taglen = 128]
         */
        .key = {0xfe, 0x9b, 0xb4, 0x7d, 0xeb, 0x3a, 0x61, 0xe4,
                0x23, 0xc2, 0x23, 0x18, 0x41, 0xcf, 0xd1, 0xfb},
        .klen = 16,
        .iv = {0x4d, 0x32, 0x8e, 0xb7, 0x76, 0xf5, 0x00, 0xa2,
               0xf7, 0xfb, 0x47, 0xaa},
        .ivlen = 12,
        .aad = {0},
        .aadlen = 0,
        .tag = {0x43, 0xfd, 0x47, 0x27, 0xfe, 0x5c, 0xdb, 0x4b,
                0x5b, 0x42, 0x81, 0x8d, 0xea, 0x7e, 0xf8, 0xc9},
        .taglen = 128,
        .plaintext = {0xf1, 0xcc, 0x38, 0x18, 0xe4, 0x21, 0x87, 0x6b,
                      0xb6, 0xb8, 0xbb, 0xd6, 0xc9},
        .plen = 13,
        .ciphertext = {0xb8, 0x8c, 0x5c, 0x19, 0x77, 0xb3, 0x5b, 0x51,
                       0x7b, 0x0a, 0xea, 0xe9, 0x67, 0x43, 0xfd, 0x47,
                       0x27, 0xfe, 0x5c, 0xdb, 0x4b, 0x5b, 0x42, 0x81,
                       0x8d, 0xea, 0x7e, 0xf8, 0xc9},
        .clen = 29,
    }, {
        /* #4 - NIST -
         * [Keylen = 128]
         * [IVlen  = 96]
         * [PTlen  = 128]
         * [AADlen = 0]
         * [Taglen = 96]
         */
        .key = {0xf0, 0x0f, 0xdd, 0x01, 0x8c, 0x02, 0xe0, 0x35,
                0x76, 0x00, 0x8b, 0x51, 0x6e, 0xa9, 0x71, 0xad},
        .klen = 16,
        .iv = {0x3b, 0x3e, 0x27, 0x6f, 0x9e, 0x98, 0xb1, 0xec,
               0xb7, 0xce, 0x6d, 0x28},
        .ivlen = 12,
        .aad = {0},
        .aadlen = 0,
        .tag = {0xcb, 0xa0, 0x6b, 0xb4, 0xf6, 0xe0, 0x97, 0x19,
                0x92, 0x50, 0xb0, 0xd1},
        .taglen = 96,
        .plaintext = {0x28, 0x53, 0xe6, 0x6b, 0x7b, 0x1b, 0x3e, 0x1f,
                      0xa3, 0xd1, 0xf3, 0x72, 0x79, 0xac, 0x82, 0xbe},
        .plen = 16,
        .ciphertext = {0x55, 0xd2, 0xda, 0x7a, 0x3f, 0xb7, 0x73, 0xb8,
                       0xa0, 0x73, 0xdb, 0x49, 0x9e, 0x24, 0xbf, 0x62,
                       0xcb, 0xa0, 0x6b, 0xb4, 0xf6, 0xe0, 0x97, 0x19,
                       0x92, 0x50, 0xb0, 0xd1},
        .clen = 28,
    }, {
        /* #5 - NIST -
         * [Keylen = 128]
         * [IVlen  = 96]
         * [PTlen  = 128]
         * [AADlen = 128]
         * [Taglen = 128]
         */
        .key = {0xc9, 0x39, 0xcc, 0x13, 0x39, 0x7c, 0x1d, 0x37,
                0xde, 0x6a, 0xe0, 0xe1, 0xcb, 0x7c, 0x42, 0x3c},
        .klen = 16,
        .iv = {0xb3, 0xd8, 0xcc, 0x01, 0x7c, 0xbb, 0x89, 0xb3,
               0x9e, 0x0f, 0x67, 0xe2},
        .ivlen = 12,
        .aad = {0x24, 0x82, 0x56, 0x02, 0xbd, 0x12, 0xa9, 0x84,
                0xe0, 0x09, 0x2d, 0x3e, 0x44, 0x8e, 0xda, 0x5f},
        .aadlen = 16,
        .tag = {0x00, 0x32, 0xa1, 0xdc, 0x85, 0xf1, 0xc9, 0x78,
                0x69, 0x25, 0xa2, 0xe7, 0x1d, 0x82, 0x72, 0xdd},
        .taglen = 128,
        .plaintext = {0xc3, 0xb3, 0xc4, 0x1f, 0x11, 0x3a, 0x31, 0xb7,
                      0x3d, 0x9a, 0x5c, 0xd4, 0x32, 0x10, 0x30, 0x69},
        .plen = 16,
        .ciphertext = {0x93, 0xfe, 0x7d, 0x9e, 0x9b, 0xfd, 0x10, 0x34,
                       0x8a, 0x56, 0x06, 0xe5, 0xca, 0xfa, 0x73, 0x54,
                       0x00, 0x32, 0xa1, 0xdc, 0x85, 0xf1, 0xc9, 0x78,
                       0x69, 0x25, 0xa2, 0xe7, 0x1d, 0x82, 0x72, 0xdd},
        .clen = 32,
    }, {
        /* #6 - NIST -
         * [Keylen = 128]
         * [IVlen  = 96]
         * [PTlen  = 128]
         * [AADlen = 128]
         * [Taglen = 96]
         */
        .key = {0x56, 0x2a, 0xe8, 0xaa, 0xdb, 0x8d, 0x23, 0xe0,
                0xf2, 0x71, 0xa9, 0x9a, 0x7d, 0x1b, 0xd4, 0xd1},
        .klen = 16,
        .iv = {0xf7, 0xa5, 0xe2, 0x39, 0x94, 0x13, 0xb8, 0x9b,
               0x6a, 0xd3, 0x1a, 0xff},
        .ivlen = 12,
        .aad = {0x2b, 0x96, 0x80, 0xb8, 0x86, 0xb3, 0xef, 0xb7,
                0xc6, 0x35, 0x4b, 0x38, 0xc6, 0x3b, 0x53, 0x73},
        .aadlen = 16,
        .tag = {0xcb, 0xf2, 0x01, 0x51, 0x84, 0xff, 0xfb, 0x82,
                0xf2, 0x65, 0x1c, 0x36},
        .taglen = 96,
        .plaintext = {0xbb, 0xdc, 0x35, 0x04, 0xd8, 0x03, 0x68, 0x2a,
                      0xa0, 0x8a, 0x77, 0x3c, 0xde, 0x5f, 0x23, 0x1a},
        .plen = 16,
        .ciphertext = {0xe2, 0xb7, 0xe5, 0xed, 0x5f, 0xf2, 0x7f, 0xc8,
                       0x66, 0x41, 0x48, 0xf5, 0xa6, 0x28, 0xa4, 0x6d,
                       0xcb, 0xf2, 0x01, 0x51, 0x84, 0xff, 0xfb, 0x82,
                       0xf2, 0x65, 0x1c, 0x36},
        .clen = 28,
    }, {
        /* #7 - NIST -
         * [Keylen = 128]
         * [IVlen  = 96]
         * [PTlen  = 408]
         * [AADlen = 160]
         * [Taglen = 32]
         */
        .key = {0xe6, 0x49, 0x03, 0xa7, 0x7d, 0x2c, 0x8f, 0x54,
                0xe5, 0x74, 0x13, 0x54, 0x89, 0x5f, 0x9f, 0x25},
        .klen = 16,
        .iv = {0x75, 0xbf, 0xc0, 0xf3, 0xc6, 0xac, 0x07, 0x1a,
               0xf0, 0x43, 0x43, 0x18},
        .ivlen = 12,
        .aad = {0x41, 0x6b, 0x40, 0xf1, 0x4b, 0xdb, 0x9f, 0x0a,
                0xce, 0xf9, 0x96, 0xc9, 0x63, 0xd2, 0x3b, 0xcf,
                0x10, 0xb7, 0x25, 0x18},
        .aadlen = 20,
        .tag = {0x5c, 0x52, 0x6f, 0x9d},
        .taglen = 32,
        .plaintext = {0x19, 0x56, 0x1f, 0x57, 0xd5, 0x7d, 0x9a, 0x96,
                      0x1b, 0xbc, 0x6a, 0xc5, 0x63, 0x45, 0x56, 0xd0,
                      0x05, 0xfa, 0x60, 0x10, 0xd9, 0x0b, 0xd2, 0x18,
                      0xc6, 0x27, 0x75, 0x37, 0xa4, 0x3f, 0x8d, 0x3f,
                      0xa8, 0xf2, 0x9a, 0x16, 0xe4, 0xcc, 0x49, 0x5b,
                      0x49, 0xb8, 0xaf, 0x19, 0x5d, 0x91, 0x7c, 0xb7,
                      0x60, 0xc3, 0x4f},
        .plen = 51,
        .ciphertext = {0x89, 0x8a, 0xbb, 0x3d, 0x70, 0x69, 0xc0, 0x59,
                       0x19, 0x04, 0x6f, 0xe4, 0x8c, 0xa9, 0xa4, 0x43,
                       0xa5, 0xd2, 0xbd, 0x2d, 0x28, 0x50, 0x3f, 0xd0,
                       0xa2, 0x71, 0x6b, 0x2e, 0xf5, 0xa1, 0x75, 0xf7,
                       0x48, 0x68, 0xf7, 0x91, 0x7f, 0x55, 0x42, 0x14,
                       0x4b, 0x67, 0x04, 0xdf, 0x8a, 0x42, 0xcc, 0x11,
                       0xc9, 0x65, 0xc3, 0x5c, 0x52, 0x6f, 0x9d},
        .clen = 55,
    }, {
        /* #8 - NIST -
         * [Keylen = 192]
         * [IVlen = 96]
         * [PTlen = 128]
         * [AADlen = 128]
         * [Taglen = 128]
         */
        .key = {0x6f, 0x44, 0xf5, 0x2c, 0x2f, 0x62, 0xda, 0xe4,
                0xe8, 0x68, 0x4b, 0xd2, 0xbc, 0x7d, 0x16, 0xee,
                0x7c, 0x55, 0x73, 0x30, 0x30, 0x5a, 0x79, 0x0d},
        .klen = 24,
        .iv = {0x9a, 0xe3, 0x58, 0x25, 0xd7, 0xc7, 0xed, 0xc9,
               0xa3, 0x9a, 0x07, 0x32},
        .ivlen = 12,
        .aad = {0x1b, 0x42, 0x36, 0xb8, 0x46, 0xfc, 0x2a, 0x0f,
                0x78, 0x28, 0x81, 0xba, 0x48, 0xa0, 0x67, 0xe9},
        .aadlen = 16,
        .tag = {0x1c, 0x19, 0x80, 0x86, 0x45, 0x0a, 0xe1, 0x83,
                0x4d, 0xd6, 0xc2, 0x63, 0x67, 0x96, 0xbc, 0xe2},
        .taglen = 128,
        .plaintext = {0x37, 0x22, 0x2d, 0x30, 0x89, 0x5e, 0xb9, 0x58,
                      0x84, 0xbb, 0xbb, 0xae, 0xe4, 0xd9, 0xca, 0xe1},
        .plen = 16,
        .ciphertext = {0xa5, 0x4b, 0x5d, 0xa3, 0x3f, 0xc1, 0x19, 0x6a,
                       0x8e, 0xf3, 0x1a, 0x53, 0x21, 0xbf, 0xca, 0xeb,
                       0x1c, 0x19, 0x80, 0x86, 0x45, 0x0a, 0xe1, 0x83,
                       0x4d, 0xd6, 0xc2, 0x63, 0x67, 0x96, 0xbc, 0xe2},
        .clen = 32,
    }, {
        /* #9 - NIST -
         * [Keylen = 256]
         * [IVlen = 96]
         * [PTlen = 128]
         * [AADlen = 128]
         * [Taglen = 128]
         */
        .key = {0x92, 0xe1, 0x1d, 0xcd, 0xaa, 0x86, 0x6f, 0x5c,
                0xe7, 0x90, 0xfd, 0x24, 0x50, 0x1f, 0x92, 0x50,
                0x9a, 0xac, 0xf4, 0xcb, 0x8b, 0x13, 0x39, 0xd5,
                0x0c, 0x9c, 0x12, 0x40, 0x93, 0x5d, 0xd0, 0x8b},
        .klen = 32,
        .iv = {0xac, 0x93, 0xa1, 0xa6, 0x14, 0x52, 0x99, 0xbd,
               0xe9, 0x02, 0xf2, 0x1a},
        .ivlen = 12,
        .aad = {0x1e, 0x08, 0x89, 0x01, 0x6f, 0x67, 0x60, 0x1c,
                0x8e, 0xbe, 0xa4, 0x94, 0x3b, 0xc2, 0x3a, 0xd6},
        .aadlen = 16,
        .tag = {0xec, 0xa5, 0xaa, 0x77, 0xd5, 0x1d, 0x4a, 0x0a,
                0x14, 0xd9, 0xc5, 0x1e, 0x1d, 0xa4, 0x74, 0xab},
        .taglen = 128,
        .plaintext = {0x2d, 0x71, 0xbc, 0xfa, 0x91, 0x4e, 0x4a, 0xc0,
                     0x45, 0xb2, 0xaa, 0x60, 0x95, 0x5f,  0xad, 0x24},
        .plen = 16,
        .ciphertext = {0x89, 0x95, 0xae, 0x2e, 0x6d, 0xf3, 0xdb, 0xf9,
                       0x6f, 0xac, 0x7b, 0x71, 0x37, 0xba, 0xe6, 0x7f,
                       0xec, 0xa5, 0xaa, 0x77, 0xd5, 0x1d, 0x4a, 0x0a,
                       0x14, 0xd9, 0xc5, 0x1e, 0x1d, 0xa4, 0x74, 0xab},
        .clen = 32,
    },
};

/** NIST Special Publication 800-38A
    http://www.csrc.nist.gov/publications/nistpubs/800-38a/sp800-38a.pdf
**/
static struct aes_test_vector aes_cfb8_tv[] = {
    {
        .key = {0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6,
                0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c},
        .klen = 16,
        .plaintext = {0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
                      0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
                      0xae, 0x2d},
        .plen = 18,
        .ciphertext = {0x3b, 0x79, 0x42, 0x4c, 0x9c, 0x0d, 0xd4, 0x36,
                       0xba, 0xce, 0x9e, 0x0e, 0xd4, 0x58, 0x6a, 0x4f,
                       0x32, 0xb9},
        .clen = 18,
        .iv = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
               0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f},
        .ivlen = 16,
    }, {
        .key = {0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52,
                0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5,
                0x62, 0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b},
        .klen = 24,
        .plaintext = {0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
                      0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
                      0xae, 0x2d},
        .plen = 18,
        .num_chunks_plain = 2,
        .chunks_plain = {8, 10},
        .ciphertext = {0xcd, 0xa2, 0x52, 0x1e, 0xf0, 0xa9, 0x05, 0xca,
                       0x44, 0xcd, 0x05, 0x7c, 0xbf, 0x0d, 0x47, 0xa0,
                       0x67, 0x8a},
        .clen = 18,
        .iv = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
               0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f},
        .ivlen = 16,
        .num_chunks_ciph = 2,
        .chunks_ciph = {8, 10},
    }, {
        .key = {0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe,
                0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
                0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7,
                0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4},
        .klen = 32,
        .plaintext = {0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
                      0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
                      0xae, 0x2d},
        .plen = 18,
        .num_chunks_plain = 5,
        .chunks_plain = {5, 0, 9, -1, 4},
        .ciphertext = {0xdc, 0x1f, 0x1a, 0x85, 0x20, 0xa6, 0x4d, 0xb5,
                       0x5f, 0xcc, 0x8a, 0xc5, 0x54, 0x84, 0x4e, 0x88,
                       0x97, 0x00},
        .clen = 18,
        .iv = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
               0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f},
        .ivlen = 16,
        .num_chunks_ciph = 5,
        .chunks_ciph = {5, 0, 9, -1, 4},
    }
};

/** NIST Special Publication 800-38A
    http://www.csrc.nist.gov/publications/nistpubs/800-38a/sp800-38a.pdf
**/
static struct aes_test_vector aes_cfb128_tv[] = {
    {                           // #0
        .key = {0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6,
                0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c},
        .klen = 16,
        .plaintext = {0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
                      0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
                      0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
                      0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
                      0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
                      0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
                      0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
                      0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10},
        .plen = 64,
        .ciphertext = {0x3b, 0x3f, 0xd9, 0x2e, 0xb7, 0x2d, 0xad, 0x20,
                       0x33, 0x34, 0x49, 0xf8, 0xe8, 0x3c, 0xfb, 0x4a,
                       0xc8, 0xa6, 0x45, 0x37, 0xa0, 0xb3, 0xa9, 0x3f,
                       0xcd, 0xe3, 0xcd, 0xad, 0x9f, 0x1c, 0xe5, 0x8b,
                       0x26, 0x75, 0x1f, 0x67, 0xa3, 0xcb, 0xb1, 0x40,
                       0xb1, 0x80, 0x8c, 0xf1, 0x87, 0xa4, 0xf4, 0xdf,
                       0xc0, 0x4b, 0x05, 0x35, 0x7c, 0x5d, 0x1c, 0x0e,
                       0xea, 0xc4, 0xc6, 0x6f, 0x9f, 0xf7, 0xf2, 0xe6},
        .clen = 64,
        .iv = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
               0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f},
        .ivlen = 16,
    }, {                       // #2
        .key = {0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52,
                0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5,
                0x62, 0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b},
        .klen = 24,
        .plaintext = {0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
                      0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
                      0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
                      0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
                      0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
                      0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
                      0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
                      0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10},
        .plen = 64,
        .num_chunks_plain = 4,
        .chunks_plain = {16, 16, 16, 16},
        .ciphertext = {0xcd, 0xc8, 0x0d, 0x6f, 0xdd, 0xf1, 0x8c, 0xab,
                       0x34, 0xc2, 0x59, 0x09, 0xc9, 0x9a, 0x41, 0x74,
                       0x67, 0xce, 0x7f, 0x7f, 0x81, 0x17, 0x36, 0x21,
                       0x96, 0x1a, 0x2b, 0x70, 0x17, 0x1d, 0x3d, 0x7a,
                       0x2e, 0x1e, 0x8a, 0x1d, 0xd5, 0x9b, 0x88, 0xb1,
                       0xc8, 0xe6, 0x0f, 0xed, 0x1e, 0xfa, 0xc4, 0xc9,
                       0xc0, 0x5f, 0x9f, 0x9c, 0xa9, 0x83, 0x4f, 0xa0,
                       0x42, 0xae, 0x8f, 0xba, 0x58, 0x4b, 0x09, 0xff},
        .clen = 64,
        .iv = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
               0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f},
        .ivlen = 16,
        .num_chunks_ciph = 4,
        .chunks_ciph = {16, 16, 16, 16},
    }, {                   // #3
        .key = {0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe,
                0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
                0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7,
                0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4},
        .klen = 32,
        .plaintext = {0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
                      0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
                      0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
                      0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
                      0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
                      0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
                      0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
                      0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10},
        .plen = 64,
        .num_chunks_plain = 5,
        .chunks_plain = {25, 0, 8, -1, 31},
        .ciphertext = {0xdc, 0x7e, 0x84, 0xbf, 0xda, 0x79, 0x16, 0x4b,
                       0x7e, 0xcd, 0x84, 0x86, 0x98, 0x5d, 0x38, 0x60,
                       0x39, 0xff, 0xed, 0x14, 0x3b, 0x28, 0xb1, 0xc8,
                       0x32, 0x11, 0x3c, 0x63, 0x31, 0xe5, 0x40, 0x7b,
                       0xdf, 0x10, 0x13, 0x24, 0x15, 0xe5, 0x4b, 0x92,
                       0xa1, 0x3e, 0xd0, 0xa8, 0x26, 0x7a, 0xe2, 0xf9,
                       0x75, 0xa3, 0x85, 0x74, 0x1a, 0xb9, 0xce, 0xf8,
                       0x20, 0x31, 0x62, 0x3d, 0x55, 0xb1, 0xe4, 0x71},
        .clen = 64,
        .iv = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
               0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f},
        .ivlen = 16,
        .num_chunks_ciph = 5,
        .chunks_ciph = {25, 0, 8, -1, 31},
    }
};

/** NIST Special Publication 800-38A
    http://www.csrc.nist.gov/publications/nistpubs/800-38a/sp800-38a.pdf
**/
static struct aes_test_vector aes_ofb_tv[] = {
    {                           // #0
        .key = {0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6,
                0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c},
        .klen = 16,
        .plaintext = {0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
                      0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
                      0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
                      0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
                      0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
                      0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
                      0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
                      0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10},
        .plen = 64,
        .ciphertext = {0x3b, 0x3f, 0xd9, 0x2e, 0xb7, 0x2d, 0xad, 0x20,
                       0x33, 0x34, 0x49, 0xf8, 0xe8, 0x3c, 0xfb, 0x4a,
                       0x77, 0x89, 0x50, 0x8d, 0x16, 0x91, 0x8f, 0x03,
                       0xf5, 0x3c, 0x52, 0xda, 0xc5, 0x4e, 0xd8, 0x25,
                       0x97, 0x40, 0x05, 0x1e, 0x9c, 0x5f, 0xec, 0xf6,
                       0x43, 0x44, 0xf7, 0xa8, 0x22, 0x60, 0xed, 0xcc,
                       0x30, 0x4c, 0x65, 0x28, 0xf6, 0x59, 0xc7, 0x78,
                       0x66, 0xa5, 0x10, 0xd9, 0xc1, 0xd6, 0xae, 0x5e},
        .clen = 64,
        .iv = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
               0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f},
        .ivlen = 16,
    }, {                       // #1
        .key = {0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52,
                0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5,
                0x62, 0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b},
        .klen = 24,
        .plaintext = {0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
                      0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
                      0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
                      0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
                      0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
                      0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
                      0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
                      0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10},
        .plen = 64,
        .num_chunks_plain = 4,
        .chunks_plain = {16, 16, 16, 16},
        .ciphertext = {0xcd, 0xc8, 0x0d, 0x6f, 0xdd, 0xf1, 0x8c, 0xab,
                       0x34, 0xc2, 0x59, 0x09, 0xc9, 0x9a, 0x41, 0x74,
                       0xfc, 0xc2, 0x8b, 0x8d, 0x4c, 0x63, 0x83, 0x7c,
                       0x09, 0xe8, 0x17, 0x00, 0xc1, 0x10, 0x04, 0x01,
                       0x8d, 0x9a, 0x9a, 0xea, 0xc0, 0xf6, 0x59, 0x6f,
                       0x55, 0x9c, 0x6d, 0x4d, 0xaf, 0x59, 0xa5, 0xf2,
                       0x6d, 0x9f, 0x20, 0x08, 0x57, 0xca, 0x6c, 0x3e,
                       0x9c, 0xac, 0x52, 0x4b, 0xd9, 0xac, 0xc9, 0x2a},
        .clen = 64,
        .iv = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
               0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f},
        .ivlen = 16,
        .num_chunks_ciph = 4,
        .chunks_ciph = {16, 16, 16, 16},
    }, {                   //  #2
        .key = {0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe,
                0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
                0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7,
                0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4},
        .klen = 32,
        .plaintext = {0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
                      0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
                      0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
                      0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
                      0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
                      0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
                      0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
                      0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10},
        .plen = 64,
        .num_chunks_plain = 5,
        .chunks_plain = {25, 0, 8, -1, 31},
        .ciphertext = {0xdc, 0x7e, 0x84, 0xbf, 0xda, 0x79, 0x16, 0x4b,
                       0x7e, 0xcd, 0x84, 0x86, 0x98, 0x5d, 0x38, 0x60,
                       0x4f, 0xeb, 0xdc, 0x67, 0x40, 0xd2, 0x0b, 0x3a,
                       0xc8, 0x8f, 0x6a, 0xd8, 0x2a, 0x4f, 0xb0, 0x8d,
                       0x71, 0xab, 0x47, 0xa0, 0x86, 0xe8, 0x6e, 0xed,
                       0xf3, 0x9d, 0x1c, 0x5b, 0xba, 0x97, 0xc4, 0x08,
                       0x01, 0x26, 0x14, 0x1d, 0x67, 0xf3, 0x7b, 0xe8,
                       0x53, 0x8f, 0x5a, 0x8b, 0xe7, 0x40, 0xe4, 0x84},
        .clen = 64,
        .iv = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
               0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f},
        .ivlen = 16,
        .num_chunks_ciph = 5,
        .chunks_ciph = {25, 0, 8, -1, 31},
    }
};

/**
 * NIST Special Publication 800-38B
 * https://csrc.nist.gov/Projects/Cryptographic-Algorithm-Validation-Program/
 *                          CAVP-TESTING-BLOCK-CIPHER-MODES#CMAC
 **/
static struct mac_test_vector aes128_cmac_tv[] = {
    {   // 0 (0)
        .key = { 0xe4,0xb7,0x64,0x5b,0x2f,0x7d,0x63,0xb4,0x67,
                 0x4c,0xd0,0x19,0x70,0xc9,0xd1,0xec },
        .klen = 16,
        .msg = { 0x00 },
        .mlen = 0,
        .mac = { 0xf2,0x66,0xa1,0x87 },
        .tlen = 4,
    }, {     // 1 (1)
        .key = { 0x8e,0xec,0xa0,0xd1,0x46,0xfd,0x09,0xff,0xbb,
                 0xe0,0xd4,0x7e,0xdc,0xdd,0xfc,0xec },
        .klen = 16,
        .msg = { 0x00 },
        .mlen = 0,
        .mac = { 0xc3,0x64,0x2c,0xe5 },
        .tlen = 4,
        .chunks_msg = { 0, 0 },
        .num_chunks_message = 2,
    }, {     // 2 (16)
        .key = { 0xe3,0x80,0x79,0x8d,0x05,0x75,0xd1,0x26,0x9c,
                 0xc8,0xca,0xa2,0x31,0x35,0x44,0x37 },
        .klen = 16,
        .msg = { 0x66,0xea,0xe0,0xea,0xbd,0x62,0x4a,0x7e,0xcc,
                 0x11,0xe9,0xdb,0x46,0x2e,0x2b,0x86 },
        .mlen = 16,
        .mac = { 0x9f,0x19,0x06,0x5a },
        .tlen = 4,
    }, {     // 3 (17)
        .key = { 0xe1,0xdc,0x72,0x4d,0x56,0x21,0xfd,0xde,0x4a,
                 0x8c,0x27,0x60,0x05,0xd6,0x15,0x75 },
        .klen = 16,
        .msg = { 0x87,0x98,0x55,0xff,0x51,0x96,0x86,0x2c,0xc7,
                 0x7e,0x32,0x3f,0x8c,0x76,0xb5,0xb0 },
        .mlen = 16,
        .mac = { 0xd1,0xc2,0xa7,0x0e },
        .tlen = 4,
        .chunks_msg = { 8, 8 },
        .num_chunks_message = 2,
    }, {     // 4 (18)
        .key = { 0x56,0x10,0x52,0xad,0xe5,0x55,0xee,0xda,0x9c,
                 0xb3,0x8b,0x1c,0xca,0x8d,0x11,0x8c },
        .klen = 16,
        .msg = { 0x79,0xfc,0x4d,0x3a,0x8a,0x47,0x38,0x5d,0xb7,
                 0x3e,0x19,0xa5,0x57,0x2a,0x27,0x41 },
        .mlen = 16,
        .mac = { 0x1e,0xc9,0x24,0xeb },
        .tlen = 4,
        .chunks_msg = { 5, 5, 6 },
        .num_chunks_message = 3,
    }, {     // 5 (24)
        .key = { 0x67,0x08,0xc9,0x88,0x7b,0x84,0x70,0x84,0xf1,
                 0x23,0xd3,0xdd,0x9c,0x3a,0x81,0x36 },
        .klen = 16,
        .msg = { 0xa8,0xde,0x55,0x17,0x0c,0x6d,0xc0,0xd8,0x0d,
                 0xe3,0x2f,0x50,0x8b,0xf4,0x9b,0x70 },
        .mlen = 16,
        .mac = { 0xcf,0xef,0x9b,0x78,0x39,0x84,0x1f,0xdb,0xcc,
                 0xbb,0x6c,0x2c,0xf2,0x38,0xf7 },
        .tlen = 15,
    }, {     // 6 (25)
        .key = { 0xe3,0x9c,0x6f,0xf1,0xad,0x22,0x79,0x3d,0xc5,
                 0x25,0xd3,0x4e,0x7d,0x7e,0x7d,0x6d },
        .klen = 16,
        .msg = { 0xf2,0x38,0x77,0xb7,0x74,0x71,0xc8,0x0d,0xd5,
                 0x65,0xec,0xe7,0xb2,0xca,0x0b,0xdd },
        .mlen = 16,
        .mac = { 0x1a,0x5c,0x33,0xd5,0x4e,0x6d,0xe6,0xd9,0xfa,
                 0x61,0xcb,0x96,0x36,0x05,0x3f },
        .tlen = 15,
        .chunks_msg = { 8, 8 },
        .num_chunks_message = 2,
    }, {     // 7 (26)
        .key = { 0x5f,0xca,0xd3,0x8a,0xe7,0x78,0x39,0x49,0x12,
                 0xf8,0xf1,0xb8,0x41,0x3c,0xf7,0x73 },
        .klen = 16,
        .msg = { 0x07,0x18,0x55,0x02,0xbf,0x6d,0x27,0x5c,0x84,
                 0xe3,0xac,0x4f,0x5f,0x77,0xc3,0xd4 },
        .mlen = 16,
        .mac = { 0xfd,0x44,0xfb,0xc0,0xdd,0x97,0x19,0xe8,0xb5,
                 0x69,0xff,0x10,0x42,0x1d,0xf4 },
        .tlen = 15,
        .chunks_msg = { 5, 5, 6 },
        .num_chunks_message = 3,
    }, {     // 8 (32)
        .key = { 0x77,0xa7,0x7f,0xaf,0x29,0x0c,0x1f,0xa3,0x0c,
                 0x68,0x3d,0xf1,0x6b,0xa7,0xa7,0x7b },
        .klen = 16,
        .msg = { 0x02,0x06,0x83,0xe1,0xf0,0x39,0x2f,0x4c,0xac,
                 0x54,0x31,0x8b,0x60,0x29,0x25,0x9e,0x9c,0x55,
                 0x3d,0xbc,0x4b,0x6a,0xd9,0x98,0xe6,0x4d,0x58,
                 0xe4,0xe7,0xdc,0x2e,0x13 },
        .mlen = 32,
        .mac = { 0xfb,0xfe,0xa4,0x1b },
        .tlen = 4,
    }, {     // 9 (33)
        .key = { 0x65,0x33,0x78,0x0f,0xc3,0x28,0xa8,0x8d,0x60,
                 0x52,0x68,0xd6,0x2f,0x29,0x5d,0xc6 },
        .klen = 16,
        .msg = { 0x02,0x74,0x9f,0x4f,0x9a,0xd8,0x2f,0xa7,0xba,
                 0x41,0xd9,0x35,0xa6,0xf1,0xaa,0x63,0x76,0xb3,
                 0x0b,0x87,0x75,0xb6,0x44,0x5a,0xc8,0x9b,0x3e,
                 0xac,0x50,0xcd,0x8d,0x56 },
        .mlen = 32,
        .mac = { 0x0b,0xfa,0x13,0x4a },
        .tlen = 4,
        .chunks_msg = { 16, 16 },
        .num_chunks_message = 2,
    }, {     // 10 (34)
        .key = { 0x49,0x2d,0xac,0xdc,0xb4,0xa3,0x5f,0xc4,0x38,
                 0xa6,0xea,0xa3,0x5e,0x26,0xd2,0xf6 },
        .klen = 16,
        .msg = { 0x99,0x2c,0x3e,0x56,0x13,0xff,0x42,0x0f,0xd9,
                 0x1b,0xe8,0x78,0xd0,0xe1,0xd6,0x2e,0x71,0xf5,
                 0xf1,0x10,0x6f,0x50,0x23,0xf3,0x99,0x91,0x5c,
                 0x98,0x03,0x94,0xd6,0xdb },
        .mlen = 32,
        .mac = { 0xb0,0x58,0x1e,0xd0 },
        .tlen = 4,
        .chunks_msg = { 10, 10, 12 },
        .num_chunks_message = 3,
    }, {     // 11 (40)
        .key = { 0x53,0x4c,0x6f,0x8f,0x88,0xbc,0x35,0x3f,0xae,
                 0xe5,0x26,0x64,0x99,0x5d,0x54,0x57 },
        .klen = 16,
        .msg = { 0x49,0x81,0xc5,0x1f,0xcc,0x09,0x35,0xf6,0x19,
                 0xec,0x6b,0xf8,0x62,0x68,0x3b,0x00,0x25,0xcc,
                 0x48,0x72,0x48,0x39,0xbc,0x1e,0x67,0xaa,0x3c,
                 0x68,0x6d,0x32,0x1b,0xa6 },
        .mlen = 32,
        .mac = { 0x63,0x77,0xc6,0xcf,0xe8,0xdd,0x60,0x5e,0xf0,
                 0xa6,0x2a,0x84,0x5a,0xb3,0xf7 },
        .tlen = 15,
    }, {     // 12 (41)
        .key = { 0xa1,0x7c,0x96,0xfa,0xa6,0x4c,0xca,0xb2,0xc4,
                 0x5d,0x93,0xa0,0x63,0x89,0x36,0x81 },
        .klen = 16,
        .msg = { 0xf2,0x68,0xc7,0xd4,0xdf,0x15,0x9e,0xf4,0xd2,
                 0x84,0xbe,0x92,0x42,0x9b,0x80,0x72,0x6e,0xf1,
                 0x34,0x73,0x4e,0xeb,0xb9,0xcc,0xc9,0x25,0x4c,
                 0x96,0x28,0x13,0x9e,0x8b },
        .mlen = 32,
        .mac = { 0xb7,0x3d,0xfe,0x7c,0x49,0x1b,0x2d,0x69,0x2e,
                 0x17,0x04,0x81,0x97,0x1e,0x4e },
        .tlen = 15,
        .chunks_msg = { 16, 16 },
        .num_chunks_message = 2,
    }, {     // 13 (42)
        .key = { 0xc5,0x58,0x1d,0x40,0xb3,0x31,0xe2,0x40,0x03,
                 0x90,0x1b,0xd6,0xbf,0x24,0x4a,0xca },
        .klen = 16,
        .msg = { 0x5d,0x44,0xba,0xc1,0xa3,0x88,0xeb,0x81,0xb0,
                 0xc3,0x57,0x1f,0x6d,0x05,0x66,0xb6,0x9b,0xde,
                 0xf5,0xff,0x21,0x66,0x4b,0x73,0xa4,0x80,0x4e,
                 0xb0,0x59,0x60,0xf6,0x1e },
        .mlen = 32,
        .mac = { 0xa1,0x44,0x7d,0x0b,0x0b,0x02,0x83,0x24,0x9e,
                 0x76,0x6f,0x83,0x8d,0xb9,0xf4 },
        .tlen = 15,
        .chunks_msg = { 10, 10, 12 },
        .num_chunks_message = 3,
    }, {     // 14 (48)
        .key = { 0x49,0x9d,0xb5,0xa3,0xec,0xc8,0x3d,0x34,0xfd,
                 0x88,0x5f,0xde,0x06,0x93,0x10,0x97 },
        .klen = 16,
        .msg = { 0xf2,0x78,0x35,0x40,0xc9,0xe1,0x70,0x6e,0xe3,
                 0xe7,0xa4,0x3e,0x71,0x83,0x39,0x87,0xbb,0x72,
                 0x44,0x1c,0x1e,0x2e,0xab,0x58,0x50,0x1c,0x8b,
                 0xfa,0xec,0x07,0xd6,0x33,0x2a },
        .mlen = 33,
        .mac = { 0x8e,0x96,0x49,0xdb },
        .tlen = 4,
    }, {     // 15 (49)
        .key = { 0xa5,0x86,0xd9,0x2b,0xc5,0xf9,0x46,0xce,0x58,
                 0x08,0x03,0x22,0x04,0x50,0x83,0xd6 },
        .klen = 16,
        .msg = { 0x12,0x95,0xc5,0xbd,0x2d,0x07,0x6b,0x18,0x72,
                 0x77,0xa0,0x2d,0x57,0x91,0x2d,0x2e,0x9e,0x27,
                 0x71,0x90,0x31,0x82,0x60,0x03,0x51,0xf6,0xcc,
                 0xa7,0xef,0xb9,0xe4,0x7d,0x5a },
        .mlen = 33,
        .mac = { 0x6e,0x55,0x44,0x64 },
        .tlen = 4,
        .chunks_msg = { 16, 17 },
        .num_chunks_message = 2,
    }, {     // 16 (50)
        .key = { 0x52,0x57,0x3c,0x65,0x50,0x88,0x69,0xdb,0x32,
                 0x59,0xcf,0x5e,0xcb,0x62,0x75,0x65 },
        .klen = 16,
        .msg = { 0xb1,0x1a,0x4c,0x21,0x55,0xd3,0x79,0xd1,0xdf,
                 0xa3,0xfc,0xb4,0x32,0xf6,0x5c,0xc5,0xac,0xc4,
                 0x87,0xca,0x4b,0x1d,0x17,0x87,0x14,0x06,0xd3,
                 0x31,0xb5,0x83,0x79,0x25,0x69 },
        .mlen = 33,
        .mac = { 0x2e,0x8b,0x88,0x44 },
        .tlen = 4,
        .chunks_msg = { 11, 10, 12 },
        .num_chunks_message = 3,
    }, {     // 17 (56)
        .key = { 0xa0,0xc3,0x34,0xff,0x35,0x01,0xc9,0x9a,0x9d,
                 0x5f,0x26,0x60,0xf4,0xa2,0xcc,0x5f },
        .klen = 16,
        .msg = { 0xb4,0x69,0x3a,0x2a,0xa1,0x1c,0xf9,0xa5,0x44,
                 0x2f,0x08,0xdf,0xa7,0x18,0x59,0x0f,0xef,0xf8,
                 0xd3,0x8f,0xdf,0x15,0xf8,0xee,0x9d,0x8a,0xc5,
                 0x41,0xb9,0x3d,0xd9,0xb9,0x6b },
        .mlen = 33,
        .mac = { 0x6d,0x4b,0xf5,0x0d,0x3a,0x13,0xa2,0x6d,0x9d,
                 0xc7,0x56,0x6d,0xee,0x12,0x23 },
        .tlen = 15,
    }, {     // 18 (57)
        .key = { 0xd2,0x65,0x33,0xa7,0x60,0xe7,0x6d,0xfa,0xa4,
                 0xa7,0xc9,0x52,0x9d,0xf8,0x60,0x31 },
        .klen = 16,
        .msg = { 0x80,0xf3,0xea,0x04,0xe4,0x2d,0x4c,0xef,0x33,
                 0x4c,0x04,0xc7,0xea,0xae,0x86,0x83,0x2f,0xaf,
                 0x52,0xf2,0xaa,0x28,0x6b,0x5a,0x96,0x36,0x0b,
                 0x60,0x18,0x3d,0x95,0x27,0xcf },
        .mlen = 33,
        .mac = { 0xec,0xee,0x84,0xf8,0x19,0x24,0xee,0x3c,0x20,
                 0x82,0x9e,0x3d,0x97,0xeb,0xf9 },
        .tlen = 15,
        .chunks_msg = { 16, 17 },
        .num_chunks_message = 2,
    }, {     // 19 (58)
        .key = { 0xc2,0xca,0x6a,0xce,0x8b,0x8b,0x19,0x31,0x4c,
                 0xc1,0x43,0x90,0xe2,0x2c,0xbc,0x37 },
        .klen = 16,
        .msg = { 0x8a,0x78,0x2e,0x7b,0x7c,0x2e,0xa7,0x7f,0x63,
                 0xdd,0xaa,0x85,0x63,0xed,0x90,0xd3,0xb5,0x7c,
                 0x8c,0x22,0x42,0xa6,0xdd,0x00,0x71,0x04,0x48,
                 0x73,0x3d,0x72,0xc6,0xdf,0x8a },
        .mlen = 33,
        .mac = { 0x77,0x1a,0x53,0xea,0x36,0x49,0x73,0xcc,0x5c,
                 0x01,0x77,0xdb,0x25,0x0a,0xc7 },
        .tlen = 15,
        .chunks_msg = { 11, 10, 12 },
        .num_chunks_message = 3,
    }, {     // 20 (64)
        .key = { 0x8c,0x9e,0xab,0xe8,0x71,0xc6,0xe9,0x51,0x11,
                 0x94,0xb4,0x8e,0xbf,0x9e,0x9b,0x03 },
        .klen = 16,
        .msg = { 0x05,0x49,0xea,0xec,0xa4,0xf3,0x97,0x24,0x7f,
                 0x1d,0x25,0x96,0x12,0xe6,0x86,0x7e,0x7d,0x78,
                 0x8c,0x71,0xd0,0x3c,0x51,0x36,0x86,0x4a,0xd6,
                 0xd8,0x4f,0x24,0xea,0xf9,0x13,0xa3,0x4e,0x69,
                 0x33 },
        .mlen = 37,
        .mac = { 0x0c,0x1e,0x97,0xd0 },
        .tlen = 4,
    }, {     // 21 (72)
        .key = { 0x69,0x90,0xdb,0x6a,0x5c,0xd8,0xdc,0x14,0x94,
                 0xcd,0x63,0x92,0x21,0x51,0x92,0x0c },
        .klen = 16,
        .msg = { 0x8c,0xed,0xde,0xbd,0x38,0xf0,0x04,0x06,0x74,
                 0x3a,0x67,0x56,0x56,0x5c,0xe7,0x62,0xd3,0x46,
                 0x44,0x35,0xd5,0x0b,0xd6,0x1b,0x8d,0xe5,0x7f,
                 0xbe,0x0b,0x79,0xdf,0x8f,0x0c,0x5c,0xc6,0x67,
                 0x13 },
        .mlen = 37,
        .mac = { 0xe5,0x2c,0xd4,0xfa,0x3c,0xf7,0x2b,0x2f,0x06,
                 0xee,0xce,0x11,0x22,0x42,0xe6 },
        .tlen = 15,
    }
};

/**
 * NIST Special Publication 800-38B
 * https://csrc.nist.gov/Projects/Cryptographic-Algorithm-Validation-Program/
 *                          CAVP-TESTING-BLOCK-CIPHER-MODES#CMAC
 **/
static struct mac_test_vector aes192_cmac_tv[] = {
    {   // 0 (0)
        .key = { 0xaf,0x18,0x9e,0xb9,0x93,0xee,0xd7,0x22,0x5b,
                 0xd3,0xb6,0x1a,0xf2,0xd3,0xa9,0xa8,0x54,0xc8,
                 0xb0,0x1b,0xa3,0x22,0x11,0xf1 },
        .klen = 24,
        .msg = { 0x00 },
        .mlen = 0,
        .mac = { 0x77,0x1b,0x8f,0x67,0x3b,0x2d,0xaf,0x2f,0xf0,
                 0x54 },
        .tlen = 10,
    }, {     // 1 (1)
        .key = { 0x33,0xf0,0x1a,0xa9,0x48,0x38,0xf1,0x5b,0xcd,
                 0x13,0x5e,0x58,0x4b,0x78,0xce,0x67,0x3f,0x83,
                 0x3e,0xa5,0x1a,0x6b,0x59,0x1f },
        .klen = 24,
        .msg = { 0x00 },
        .mlen = 0,
        .mac = { 0x8f,0xee,0x17,0x9b,0x1e,0x67,0x87,0x07,0xde,
                 0x7f },
        .tlen = 10,
        .chunks_msg = { 0, 0 },
        .num_chunks_message = 2,
    }, {     // 2 (8)
        .key = { 0x91,0xa8,0xa9,0x47,0x3c,0xe4,0x8a,0x2e,0x2d,
                 0x59,0x88,0x54,0x98,0xc6,0x7e,0xfc,0x42,0x24,
                 0x8e,0x8f,0x7e,0xce,0xb6,0xe4 },
        .klen = 24,
        .msg = { 0x00 },
        .mlen = 0,
        .mac = { 0x7c,0x4d,0x12,0xc6,0x08,0x76,0x39,0xec,0x7c,
                 0x68,0xb6,0x64 },
        .tlen = 12,
    }, {     // 3 (16)
        .key = { 0x33,0x64,0x16,0x9e,0xe0,0x77,0xca,0xfb,0x9b,
                 0x15,0xcb,0xab,0xaf,0xa4,0xf3,0xf1,0xea,0x13,
                 0x8b,0xbc,0x09,0x75,0x64,0x70 },
        .klen = 24,
        .msg = { 0x00 },
        .mlen = 0,
        .mac = { 0x57,0xaa,0xb6,0x46,0x10,0x5c,0x5e,0x5b,0x80,
                 0x21,0x4b,0x87,0xff,0xfa,0x38,0x47 },
        .tlen = 16,
    }, {     // 4 (24)
        .key = { 0xb1,0x90,0x41,0x30,0x8b,0x76,0x71,0x38,0xee,
                 0x19,0x60,0x22,0x48,0x03,0x01,0x68,0x3d,0x14,
                 0x41,0x93,0x3e,0xc8,0x3a,0xa3 },
        .klen = 24,
        .msg = { 0x86,0x61,0x70,0x12,0xaf,0x7d,0x94,0x78,0x41,
                 0xba,0x40,0x8d,0xf5,0xff,0x8b,0x9a,0x44,0xf2,
                 0x59,0x85,0xcc,0xc5,0xdc,0x44,0xf5,0x52,0x5a,
                 0x97,0xd4,0x6b,0x30,0xf7 },
        .mlen = 32,
        .mac = { 0x8e,0xfc,0x4d,0x25,0xb6,0x92,0xc3,0x4d,0xdb,
                 0xbc },
        .tlen = 10,
    }, {     // 5 (25)
        .key = { 0x7f,0xea,0x56,0x3a,0x86,0x65,0x71,0x82,0x24,
                 0x72,0xda,0xde,0x8a,0x0b,0xec,0x4b,0x98,0x20,
                 0x2d,0x47,0xa3,0x44,0x31,0x29 },
        .klen = 24,
        .msg = { 0xa2,0x06,0xa1,0xeb,0x70,0xa9,0xd2,0x4b,0xb5,
                 0xe7,0x2f,0x31,0x4e,0x7d,0x91,0xde,0x07,0x4f,
                 0x59,0x05,0x56,0x53,0xbd,0xd2,0x4a,0xab,0x5f,
                 0x2b,0xbe,0x11,0x24,0x36 },
        .mlen = 32,
        .mac = { 0x3b,0xfe,0x96,0xf0,0x5e,0x9c,0xf9,0x6a,0x98,
                 0xbd },
        .tlen = 10,
        .chunks_msg = { 16, 16 },
        .num_chunks_message = 2,
    }, {     // 6 (26)
        .key = { 0x3f,0x29,0xd9,0xb2,0xdb,0xf5,0xd3,0xe5,0x1f,
                 0xcf,0x96,0xe8,0x6e,0x1d,0x40,0x75,0x66,0x92,
                 0x02,0x09,0x66,0xa8,0xc5,0xf1 },
        .klen = 24,
        .msg = { 0x5e,0x51,0x36,0xff,0x2c,0x39,0xc2,0xcb,0xb8,
                 0x07,0x46,0xb3,0xdf,0x2e,0x39,0xe2,0x74,0x4c,
                 0x8f,0x11,0x9a,0x02,0x38,0xfb,0x70,0x02,0xe9,
                 0x11,0xf4,0x69,0x1e,0x6e },
        .mlen = 32,
        .mac = { 0x0a,0x89,0x1a,0x79,0x9a,0x35,0x6e,0xc6,0x24,
                 0x1c },
        .tlen = 10,
        .chunks_msg = { 10, 10, 12 },
        .num_chunks_message = 3,
    }, {     // 7 (32)
        .key = { 0x09,0x89,0x1e,0xd1,0x4f,0x44,0x88,0x06,0x9c,
                 0xd6,0xa5,0x74,0x40,0x61,0xe0,0x6f,0x8f,0xf8,
                 0xd1,0xbc,0x87,0xb1,0x04,0x48 },
        .klen = 24,
        .msg = { 0x3c,0x3d,0x7f,0x18,0xa0,0xbd,0x3b,0x6f,0x36,
                 0x6c,0x1b,0x49,0x7c,0x60,0x2e,0x88,0x9c,0xac,
                 0x88,0xfe,0xed,0xd7,0xe6,0x83,0xbd,0x8d,0x95,
                 0x45,0x4e,0x95,0x8f,0x60 },
        .mlen = 32,
        .mac = { 0xa7,0x37,0xae,0x84,0x6c,0x1f,0xec,0xea,0x1c,
                 0xf7,0x7a,0x17 },
        .tlen = 12,
    }, {     // 8 (33)
        .key = { 0xa9,0x2e,0x82,0x0d,0x4a,0xa8,0x3a,0xb3,0x2b,
                 0xc9,0xce,0x53,0x7a,0x95,0x82,0x30,0x38,0x96,
                 0x06,0x48,0x40,0xd2,0x9e,0xe4 },
        .klen = 24,
        .msg = { 0x6f,0x6c,0x7c,0xe5,0xd1,0xc7,0x25,0xf7,0xb0,
                 0x28,0x47,0x03,0x73,0x6a,0x1d,0xe9,0xca,0x8f,
                 0x75,0xcf,0x91,0x10,0xb4,0x15,0xf7,0x70,0x19,
                 0x8b,0xdf,0x17,0x46,0xe2 },
        .mlen = 32,
        .mac = { 0x08,0x70,0x06,0xce,0xa6,0x4c,0x26,0x5c,0xb2,
                 0x33,0x1a,0x93 },
        .tlen = 12,
        .chunks_msg = { 17, 15 },
        .num_chunks_message = 2,
    }, {     // 9 (34)
        .key = { 0x33,0x14,0xe9,0xaf,0x7c,0xe9,0x91,0x7f,0xf7,
                 0xf9,0x22,0xc8,0x67,0x06,0x27,0x7a,0x4e,0x98,
                 0xd2,0x8e,0x11,0x97,0x41,0x3b },
        .klen = 24,
        .msg = { 0xca,0x2b,0x30,0xd7,0x2d,0xf5,0x32,0xef,0xce,
                 0x9c,0xde,0x44,0xff,0xb9,0xbb,0x56,0x11,0xe7,
                 0x46,0xba,0x1a,0x76,0xdc,0x6a,0xf4,0x64,0xd9,
                 0x5f,0x89,0x88,0x02,0x62 },
        .mlen = 32,
        .mac = { 0xa1,0xf0,0xe1,0xa5,0xd2,0x31,0x60,0xf0,0x52,
                 0xb3,0xaa,0x83 },
        .tlen = 12,
        .chunks_msg = { 10, 11, 11 },
        .num_chunks_message = 3,
    }, {     // 10 (40)
        .key = { 0x20,0x51,0xaf,0x34,0x76,0x2e,0xbe,0x55,0x6f,
                 0x72,0xa5,0xc6,0xed,0xc7,0x77,0x1e,0xb9,0x24,
                 0x5f,0xad,0x76,0xf0,0x34,0xbe },
        .klen = 24,
        .msg = { 0xae,0x8e,0x93,0xc9,0xc9,0x91,0xcf,0x89,0x6a,
                 0x49,0x1a,0x89,0x07,0xdf,0x4e,0x4b,0xe5,0x18,
                 0x6a,0xe4,0x96,0xcd,0x34,0x0d,0xc1,0x9b,0x23,
                 0x78,0x21,0xdb,0x7b,0x60 },
        .mlen = 32,
        .mac = { 0x74,0xf7,0x46,0x08,0xc0,0x4f,0x0f,0x4e,0x47,
                 0xfa,0x64,0x04,0x33,0xb6,0xe6,0xfb },
        .tlen = 16,
    }, {     // 11 (41)
        .key = { 0x45,0x13,0xf5,0x5f,0x24,0x3b,0x70,0x54,0x4f,
                 0xd9,0x7f,0xaf,0xe6,0x5a,0x68,0xed,0x82,0xc4,
                 0xac,0xdc,0xc9,0xf7,0x39,0xff },
        .klen = 24,
        .msg = { 0xbb,0x52,0x5c,0xbd,0x4c,0xfe,0x78,0x4d,0x33,
                 0xc4,0xfa,0x8b,0x6e,0x75,0x00,0x6e,0x25,0x69,
                 0x11,0xb0,0x3e,0xc4,0x54,0x2f,0x2c,0xf8,0xff,
                 0x94,0x07,0x8b,0x13,0x32 },
        .mlen = 32,
        .mac = { 0x52,0xd2,0x39,0x39,0xe5,0x98,0xa7,0xe5,0x54,
                 0xd3,0x60,0x4d,0xdd,0xf7,0x90,0xb5 },
        .tlen = 16,
        .chunks_msg = { 16, 16 },
        .num_chunks_message = 2,
    }, {     // 12 (42)
        .key = { 0xcd,0x98,0x06,0x78,0x5d,0xb2,0x97,0x69,0xde,
                 0x6f,0xdf,0x52,0x23,0xe4,0xf2,0x08,0x78,0xbf,
                 0x13,0xcf,0xe3,0x7d,0x1c,0x2a },
        .klen = 24,
        .msg = { 0xdf,0xdd,0xb7,0x19,0xd0,0x03,0x98,0xbf,0x48,
                 0xa6,0xce,0xfd,0x27,0x73,0x63,0x89,0xe6,0x54,
                 0xa9,0x3b,0x85,0x95,0xcd,0x5a,0xc4,0x46,0xaf,
                 0x19,0x96,0xe0,0xf1,0x61 },
        .mlen = 32,
        .mac = { 0x09,0xc4,0xd3,0x37,0xdb,0x14,0x6f,0x92,0x0d,
                 0x93,0x7e,0xd2,0x83,0xcc,0x69,0x44 },
        .tlen = 16,
        .chunks_msg = { 10, 10, 12 },
        .num_chunks_message = 3,
    }, {     // 13 (48)
        .key = { 0x2c,0xc8,0x39,0x87,0x6d,0x39,0x89,0xf3,0x57,
                 0x31,0xbe,0x37,0x1f,0x60,0xde,0x14,0x0e,0x3c,
                 0x91,0x62,0x31,0xec,0x78,0x0e },
        .klen = 24,
        .msg = { 0x2e,0xae,0xf8,0x6b,0x0f,0x60,0x23,0x64,0xf8,
                 0x65,0x10,0xea,0xbc,0x58,0xbc,0x9a,0xd1,0xe6,
                 0xf0,0xa6,0xf6,0xdf,0x0b,0x83,0x18,0x8c,0x01,
                 0xe1,0x77,0x44,0xa4,0xe0,0x05,0x3a,0x22,0x81,
                 0x0e,0x99,0xcf,0x5a,0x1e,0xd3,0x25,0x8f,0x20,
                 0x35,0x09,0xfd },
        .mlen = 48,
        .mac = { 0x80,0x6a,0x7c,0x69,0xee,0xbc,0x60,0x41,0xe6,
                 0xf9 },
        .tlen = 10,
    }, {     // 14 (49)
        .key = { 0x45,0x02,0x8d,0xd9,0x9d,0x0c,0x7f,0x13,0xfd,
                 0x00,0xa2,0xe8,0x05,0x92,0x2e,0xbf,0x4f,0x83,
                 0x30,0x80,0xf5,0x14,0x2f,0xa7 },
        .klen = 24,
        .msg = { 0xf8,0x22,0x92,0x64,0xbd,0xb7,0x65,0x7c,0x05,
                 0x6e,0xd4,0xaf,0x2a,0x07,0x33,0x55,0x33,0x3d,
                 0xe2,0x9b,0xa7,0x48,0xc4,0x28,0xa4,0xa6,0xf2,
                 0x01,0x2b,0x1b,0xb1,0xaa,0xbe,0xf7,0xa2,0x8f,
                 0x2d,0x97,0x39,0xa2,0xad,0x2e,0x70,0x44,0x2a,
                 0xc6,0xbd,0x69 },
        .mlen = 48,
        .mac = { 0x42,0x88,0xca,0x11,0xa1,0x7f,0x43,0x64,0x12,
                 0x5f },
        .tlen = 10,
        .chunks_msg = { 32, 16 },
        .num_chunks_message = 2,
    }, {     // 15 (50)
        .key = { 0x7e,0xff,0xb0,0xee,0x6e,0xaf,0x5d,0xf3,0x7e,
                 0x69,0xae,0xbe,0x27,0x92,0x0c,0x62,0x40,0x74,
                 0x1e,0x39,0xce,0xcf,0x40,0x84 },
        .klen = 24,
        .msg = { 0x37,0x3b,0x1b,0x64,0xbb,0x55,0x16,0xd9,0x6e,
                 0x40,0x86,0x31,0xac,0xf8,0x49,0x66,0xd2,0x76,
                 0x46,0x53,0xa2,0x80,0xf3,0x23,0xe9,0xc5,0x1b,
                 0x0a,0x5e,0x29,0xde,0x33,0xce,0x5e,0xf9,0xf9,
                 0x76,0xb4,0x47,0x59,0xb1,0x32,0x88,0xa7,0xd3,
                 0xe5,0x62,0x81 },
        .mlen = 48,
        .mac = { 0x2e,0x2f,0x77,0x79,0x69,0x8b,0x53,0x06,0xe5,
                 0x67 },
        .tlen = 10,
        .chunks_msg = { 16, 16, 16 },
        .num_chunks_message = 3,
    }, {     // 16 (140)
        .key = { 0xa1,0x2a,0x70,0xeb,0xca,0xad,0x31,0x0f,0x65,
                 0x02,0xfe,0x82,0xc3,0x08,0xcf,0xa8,0xc1,0xf6,
                 0x8b,0x1f,0x9c,0xe5,0x7b,0x5a },
        .klen = 24,
        .msg = { 0xe1,0x9a,0xc0,0x8a,0xeb,0x3c,0xd7,0x8e,0x40,
                 0x54,0x8d,0xb4,0x3c,0x5a,0xca,0x50,0xad,0x78,
                 0x60,0xfe,0x92,0x3e,0xef,0xf0,0x23,0x97,0x06,
                 0x86,0x11,0xfc,0x8a,0x1f,0x0e,0xe7,0x5c,0x30,
                 0x90,0x8b,0x36,0x93,0xd2,0xc2,0xd6,0x21,0x1c,
                 0x47,0x5f,0x82,0x45,0x25,0xfc,0x44,0x76,0x6f,
                 0xc6,0x2b,0x30,0x32,0xe8,0xc8 },
        .mlen = 60,
        .mac = { 0x86,0xc4,0x41,0x50,0xfe,0xd8,0x4d,0x2e,0x18,
                 0x97,0x66,0x61,0x27,0xe3,0xb5,0x6e },
        .tlen = 16,
    }, {     // 17 (141)
        .key = { 0x62,0xb8,0x26,0x3d,0xc0,0x15,0xef,0x87,0x3c,
                 0xd1,0x62,0x72,0xe4,0xda,0x89,0x79,0x9b,0x91,
                 0x0f,0x2b,0x04,0x20,0x44,0x20 },
        .klen = 24,
        .msg = { 0x8b,0x4d,0x85,0xd9,0x95,0x26,0x50,0xae,0xb2,
                 0x39,0x1f,0x70,0x1f,0x1d,0x34,0x2b,0x45,0xe4,
                 0x6f,0x0a,0x33,0xb6,0x67,0x1d,0x4c,0xb8,0xb9,
                 0x59,0x3d,0x32,0xa0,0xe1,0x33,0xf2,0xc6,0x84,
                 0x4a,0xeb,0x5a,0x86,0x48,0x22,0x63,0xa3,0x8b,
                 0xcb,0x54,0x51,0x40,0xae,0xe0,0x46,0xd6,0xd0,
                 0x0f,0x3a,0xb2,0x50,0x92,0xee },
        .mlen = 60,
        .mac = { 0xca,0x55,0x8a,0x0b,0xea,0x31,0xff,0x12,0x99,
                 0xee,0xb9,0x4b,0xc0,0x6c,0xda,0x38 },
        .tlen = 16,
        .chunks_msg = { 30, 30 },
        .num_chunks_message = 2,
    }, {     // 18 (142)
        .key = { 0x3d,0xa1,0xc7,0xcf,0xfa,0xa1,0x67,0x55,0x7b,
                 0x25,0x06,0x34,0xe8,0x05,0x2f,0xa0,0x30,0xc6,
                 0x62,0xe8,0xfb,0xbe,0xab,0xb3 },
        .klen = 24,
        .msg = { 0xd1,0xde,0x1b,0xe5,0xf7,0xa4,0x61,0x91,0xbb,
                 0x3d,0x24,0xd8,0x6f,0xc3,0xea,0xbd,0x5b,0x6f,
                 0xd6,0x65,0x5f,0xb0,0x6f,0xcb,0xda,0x7a,0xa4,
                 0xb5,0xca,0xe1,0x0c,0xe7,0x34,0xe6,0x72,0x96,
                 0xc6,0x91,0x90,0x1f,0xbb,0x40,0x1b,0xf9,0xff,
                 0x3e,0x00,0xe8,0x93,0x97,0xee,0x74,0x34,0x2b,
                 0x14,0x9a,0x37,0x02,0x2c,0xc1 },
        .mlen = 60,
        .mac = { 0xa9,0x22,0x6f,0x20,0xdb,0x94,0x7f,0x83,0x5e,
                 0x85,0x75,0x42,0x1c,0x05,0xa2,0x41 },
        .tlen = 16,
        .chunks_msg = { 20, 20, 20 },
        .num_chunks_message = 3,
    }, {     // 19 (143)
        .key = { 0x01,0x2d,0x3f,0x35,0xfe,0x16,0x2a,0xc6,0x99,
                 0x9a,0xd8,0x4d,0x90,0x57,0xff,0xfb,0x5c,0xac,
                 0x15,0xbc,0x4e,0x78,0x09,0x17 },
        .klen = 24,
        .msg = { 0x94,0x0c,0xec,0x23,0xe8,0x87,0x56,0x9a,0xa8,
                 0x1d,0x54,0x99,0x02,0x50,0x37,0xbe,0x68,0xdb,
                 0x9a,0xc7,0xf2,0x8b,0xe8,0x06,0xfd,0x50,0x1e,
                 0x10,0x19,0x94,0xd2,0xb6,0x22,0xcb,0x75,0x36,
                 0xac,0x4c,0xd5,0xb2,0xa4,0x5f,0xe5,0xe0,0x35,
                 0x97,0xc1,0x04,0xe9,0x62,0xed,0x1b,0x11,0xde,
                 0xdd,0x59,0x66,0xee,0xbb,0xd6 },
        .mlen = 60,
        .mac = { 0xd1,0xf9,0x0c,0x9f,0x65,0x29,0xdb,0x7c,0x7b,
                 0xfa,0x8e,0x79,0x24,0x80,0x59,0x25 },
        .tlen = 16,
        .chunks_msg = { 12, 6, 6, 6, 6, 6, 6, 12 },
        .num_chunks_message = 8,
    }
};

/**
 * NIST Special Publication 800-38B
 * https://csrc.nist.gov/Projects/Cryptographic-Algorithm-Validation-Program/
 *                          CAVP-TESTING-BLOCK-CIPHER-MODES#CMAC
 **/
static struct mac_test_vector aes256_cmac_tv[] = {
    {   // 0 (0)
        .key = { 0x95,0xd8,0xaf,0xb8,0xa4,0xb7,0x24,0x5c,0xe7,
                 0x9f,0x9f,0x9c,0x5d,0xdd,0x40,0xde,0x61,0xb3,
                 0x59,0x05,0xdc,0xb6,0x38,0xf2,0xb8,0x75,0x40,
                 0x4a,0x98,0x5b,0x3f,0x7a },
        .klen = 32,
        .msg = { 0x00 },
        .mlen = 0,
        .mac = { 0x68,0xad,0xfc,0x9b,0x59 },
        .tlen = 5,
    }, {     // 1 (1)
        .key = { 0x96,0xa0,0x85,0xfe,0x88,0xbc,0x98,0xc7,0x63,
                 0xc1,0x06,0x4d,0xa4,0xc9,0xc8,0xb4,0xaa,0x4a,
                 0xd4,0x2a,0xba,0xff,0x40,0x7a,0x46,0x26,0xab,
                 0xde,0x6f,0xec,0xd9,0x15 },
        .klen = 32,
        .msg = { 0x00 },
        .mlen = 0,
        .mac = { 0xd8,0xdc,0x67,0x26,0xd8 },
        .tlen = 5,
        .chunks_msg = { 0, 0 },
        .num_chunks_message = 2,
    }, {     // 2 (8)
        .key = { 0xf0,0xa3,0xe4,0xc2,0x37,0xd8,0x67,0x18,0xd8,
                 0x4c,0x43,0x18,0x5e,0x70,0xf9,0xce,0xf0,0xdc,
                 0x92,0xb3,0x78,0xe3,0xe0,0xdb,0x04,0x6b,0x06,
                 0x71,0x6c,0xfb,0x3b,0x61 },
        .klen = 32,
        .msg = { 0x00 },
        .mlen = 0,
        .mac = { 0x38,0xba,0x46,0x60,0x2f,0x34,0x11,0xa5,0x8b,
                 0x2e },
        .tlen = 10,
    }, {     // 3 (16)
        .key = { 0x91,0x6c,0xd8,0x11,0x9a,0x4d,0x7e,0x82,0x5e,
                 0x01,0xa8,0x6f,0x93,0xb0,0xee,0xe4,0xa4,0x6d,
                 0x29,0x21,0x61,0x15,0xcc,0x9e,0xf6,0x7a,0x78,
                 0x4c,0x19,0xc1,0xca,0x1e },
        .klen = 32,
        .msg = { 0x33,0x10,0xfc,0x5b,0xc9,0x10,0xf4,0xc9,0xb9,
                 0xcf,0x49,0x57,0xac,0x63,0x8a,0xe7 },
        .mlen = 16,
        .mac = { 0x03,0xf0,0x45,0xf8,0x86 },
        .tlen = 5,
    }, {     // 4 (17)
        .key = { 0x00,0x9f,0x47,0xf1,0x80,0xe0,0x85,0x77,0x6b,
                 0xe6,0x64,0x4a,0xea,0xc0,0x07,0x0b,0xe6,0x4c,
                 0x28,0x9f,0x84,0xa7,0xba,0x3d,0xec,0xe7,0xcd,
                 0xc5,0x4f,0x0d,0xb3,0x54 },
        .klen = 32,
        .msg = { 0x41,0xb9,0x85,0x38,0xc7,0x52,0x79,0x5d,0x9d,
                 0x48,0x70,0x2b,0x56,0xf3,0x34,0xd9 },
        .mlen = 16,
        .mac = { 0xb9,0x61,0xae,0xec,0x5d },
        .tlen = 5,
        .chunks_msg = { 8, 8 },
        .num_chunks_message = 2,
    }, {     // 5 (18)
        .key = { 0x7a,0xd4,0x91,0xec,0x57,0x18,0x7d,0x42,0x43,
                 0xc3,0x66,0x03,0xce,0xfe,0x68,0x2c,0x0c,0x56,
                 0x67,0x5c,0x31,0x04,0x48,0x39,0x5e,0x71,0x60,
                 0x0f,0xbb,0xf9,0x2c,0xbb },
        .klen = 32,
        .msg = { 0x0b,0xa2,0x1b,0x26,0x0e,0xe8,0x94,0x14,0x78,
                 0x53,0xa1,0x43,0x28,0xce,0xcf,0x1d },
        .mlen = 16,
        .mac = { 0x1d,0xd6,0xcd,0xc6,0xa7 },
        .tlen = 5,
        .chunks_msg = { 5, 5, 6 },
        .num_chunks_message = 3,
    }, {     // 6 (24)
        .key = { 0x3a,0x75,0xa9,0xd2,0xbd,0xb8,0xc8,0x04,0xba,
                 0x4a,0xb4,0x98,0x35,0x73,0xa6,0xb2,0x53,0x16,
                 0x0d,0xd9,0x0f,0x8e,0xdd,0xfb,0x2f,0xdc,0x2a,
                 0xb1,0x76,0x04,0xf5,0xc5 },
        .klen = 32,
        .msg = { 0x42,0xf3,0x5d,0x5a,0xa5,0x33,0xa7,0xa0,0xa5,
                 0xf7,0x4e,0x14,0x4f,0x2a,0x5f,0x20 },
        .mlen = 16,
        .mac = { 0xf1,0x53,0x2f,0x87,0x32,0xd9,0xf5,0x90,0x30,
                 0x07 },
        .tlen = 10,
    }, {     // 7 (32)
        .key = { 0xdc,0xba,0x2c,0xe0,0x16,0x33,0x93,0x7b,0x1c,
                 0xda,0xb1,0x2b,0x2e,0x83,0x59,0x8a,0x49,0xc5,
                 0x16,0x09,0xef,0xae,0x0f,0x40,0x26,0xb6,0x2d,
                 0x82,0xc3,0xf2,0x80,0xb5 },
        .klen = 32,
        .msg = { 0x9a,0xa3,0xe8,0xad,0x92,0x77,0x7d,0xfe,0xb1,
                 0x21,0xa6,0x46,0xce,0x2e,0x91,0x8d,0x1e,0x12,
                 0xb3,0x07,0x54,0xbc,0x09,0x47,0x0d,0x6d,0xa4,
                 0xaf,0x6c,0xc9,0x64,0x2b,0x01,0x2f,0x04,0x1f,
                 0xf0,0x46,0x56,0x9d,0x4f,0xd8,0xd0,0xdc,0xcf,
                 0xe4,0x48,0xe5 },
        .mlen = 48,
        .mac = { 0x81,0x62,0x82,0xfb,0x33 },
        .tlen = 5,
    }, {     // 8 (33)
        .key = { 0x0b,0x12,0x2a,0xc8,0xf3,0x4e,0xd1,0xfe,0x08,
                 0x2a,0x36,0x25,0xd1,0x57,0x56,0x14,0x54,0x16,
                 0x7a,0xc1,0x45,0xa1,0x0b,0xbf,0x77,0xc6,0xa7,
                 0x05,0x96,0xd5,0x74,0xf1 },
        .klen = 32,
        .msg = { 0x49,0x8b,0x53,0xfd,0xec,0x87,0xed,0xcb,0xf0,
                 0x70,0x97,0xdc,0xcd,0xe9,0x3a,0x08,0x4b,0xad,
                 0x75,0x01,0xa2,0x24,0xe3,0x88,0xdf,0x34,0x9c,
                 0xe1,0x89,0x59,0xfe,0x84,0x85,0xf8,0xad,0x15,
                 0x37,0xf0,0xd8,0x96,0xea,0x73,0xbe,0xdc,0x72,
                 0x14,0x71,0x3f },
        .mlen = 48,
        .mac = { 0xf6,0x2c,0x46,0x32,0x9b },
        .tlen = 5,
        .chunks_msg = { 24, 24 },
        .num_chunks_message = 2,
    }, {     // 9 (34)
        .key = { 0x75,0x11,0xf3,0xa0,0x4e,0x0e,0xa0,0xe7,0xce,
                 0xda,0x9e,0x06,0x66,0x6d,0x15,0x53,0xab,0x58,
                 0x63,0x84,0x0b,0xa7,0x6d,0xb6,0xb8,0x0b,0x37,
                 0xb5,0xe9,0x39,0xb3,0x77 },
        .klen = 32,
        .msg = { 0x30,0x83,0x43,0x08,0x18,0x7b,0x8b,0xe1,0xbe,
                 0x40,0x5d,0x4b,0x27,0x4d,0x97,0xb5,0xd8,0xab,
                 0x71,0x90,0x50,0x64,0x18,0x5f,0xd7,0x3b,0x99,
                 0x0d,0x76,0x9a,0x01,0x84,0x47,0xcc,0xc2,0x7b,
                 0xef,0x6c,0x59,0x8f,0x68,0x1e,0x2e,0x96,0x04,
                 0x7d,0xbc,0x30 },
        .mlen = 48,
        .mac = { 0xc0,0x4f,0xf7,0x16,0x38 },
        .tlen = 5,
        .chunks_msg = { 16, 16, 16 },
        .num_chunks_message = 3,
    }, {     // 10 (66)
        .key = { 0xcf,0x3b,0x0e,0xbe,0xe6,0xb4,0xac,0x11,0xaa,
                 0x76,0x78,0xb2,0xf5,0x45,0x3c,0x13,0x07,0xaf,
                 0x5c,0xda,0x7c,0x34,0x67,0x2a,0x7b,0xaa,0xec,
                 0x25,0x2f,0xe0,0x8f,0xaf },
        .klen = 32,
        .msg = { 0xe4,0xbd,0x45,0xe3,0x1e,0x1d,0x3c,0xf9,0x27,
                 0x60,0x96,0xc1,0x8d,0x2d,0x70 },
        .mlen = 15,
        .mac = { 0x32,0x6b,0x02,0x39,0x04 },
        .tlen = 5,
    }, {     // 11 (67)
        .key = { 0x47,0x97,0x20,0xcd,0xda,0xa5,0xe0,0x08,0xfa,
                 0x19,0x4b,0xb8,0x59,0xec,0x3f,0xce,0x76,0xb2,
                 0xd6,0x96,0xc2,0x29,0xa0,0xe7,0x02,0x12,0x2f,
                 0x9d,0xf4,0x3f,0xd9,0x48 },
        .klen = 32,
        .msg = { 0x3a,0x98,0x7e,0xb8,0x79,0x58,0x08,0x01,0x28,
                 0xf0,0x8d,0xcd,0xf9,0x1e,0x63 },
        .mlen = 15,
        .mac = { 0x1e,0xbe,0xca,0x94,0xae },
        .tlen = 5,
        .chunks_msg = { 1, 14 },
        .num_chunks_message = 2,
    }, {     // 12 (93)
        .key = { 0x23,0xe5,0x42,0x2e,0x8d,0x75,0x60,0xa9,0xe6,
                 0x56,0x42,0xb5,0xe7,0x23,0xa4,0x75,0x36,0xc1,
                 0x67,0x91,0xf3,0xa0,0xcf,0x91,0x8d,0x3d,0xee,
                 0x8a,0xdb,0xec,0x60,0xfd },
        .klen = 32,
        .msg = { 0xb9,0xee,0x14,0x00,0x18,0x6c,0x0c,0x07,0x74,
                 0x40,0x1a,0x81,0x5b,0xcd,0xe3,0x0d,0x3b,0xe1,
                 0xd4,0xf8,0x7f,0x42,0x64,0x6c,0xfb,0x8a,0x99,
                 0xe4,0x8a,0x35,0xce,0xe3,0xf5,0xf9,0xb3,0xe6,
                 0x17,0x56,0x95,0x97,0x3f,0x6d,0xe0,0x43,0xd6,
                 0x15,0xe2,0x8e },
        .mlen = 48,
        .mac = { 0x17,0x75,0x84,0x70,0x19,0xca,0x9b,0x88,0x68,
                 0x3e },
        .tlen = 10,
        .chunks_msg = { 8, 4, 4, 8, 4, 4, 8, 8 },
        .num_chunks_message = 8,
    }, {     // 13 (94)
        .key = { 0x7c,0xfc,0x08,0x6d,0x10,0x65,0x9d,0x7c,0xb9,
                 0x24,0x72,0x08,0x35,0x8d,0xd8,0x2c,0x03,0xb8,
                 0xdb,0xd8,0x23,0x32,0x23,0x23,0x1d,0xf2,0x18,
                 0xe2,0x44,0x8f,0x4a,0x79 },
        .klen = 32,
        .msg = { 0xa3,0xaf,0x8f,0x99,0x70,0x3a,0x60,0x10,0x86,
                 0xc2,0xa1,0xff,0xe5,0x5f,0xde,0x4c,0x2c,0x41,
                 0x53,0xdb,0xff,0x8d,0x66,0x01,0xab,0x68,0x74,
                 0x3c,0x0d,0x50,0xd0,0x21,0xb0,0xb3,0x09,0x95,
                 0x35,0xba,0x6c,0x40,0xf8,0x66,0xca,0x3f,0xf0,
                 0xdf,0x7c,0x19 },
        .mlen = 48,
        .mac = { 0x9d,0x71,0x3a,0x19,0x44,0xb8,0xeb,0x64,0x95,
                 0x84 },
        .tlen = 10,
        .chunks_msg = { 7, 7, 7, 15, 12 },
        .num_chunks_message = 5,
    }
};

/**
 * Derived CBC-MAC test vectors from AES-CBC test vectors
 * https://csrc.nist.gov/projects/cryptographic-algorithm-validation-program/
 * block-ciphers#AES
 **/
static struct mac_test_vector aes128_cbc_mac_tv[] = {
    {   // 0
        .key = { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00 },
        .klen = 16,
        .msg = { 0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00 },
        .mlen = 16,
        .mac = { 0x3a,0xd7,0x8e,0x72,0x6c,0x1e,0xc0,0x2b,0x7e,
                 0xbf,0xe9,0x2b,0x23,0xd9,0xec,0x34 },
        .tlen = 4,
    }, {   // 1
        .key = { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00 },
        .klen = 16,
        .msg = { 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00 },
        .mlen = 16,
        .mac = { 0xf8,0x07,0xc3,0xe7,0x98,0x5f,0xe0,0xf5,0xa5,
                 0x0e,0x2c,0xdb,0x25,0xc5,0x10,0x9e },
        .tlen = 15,
    }, {   // 2
        .key = { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00 },
        .klen = 16,
        .msg = { 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
                 0xff,0xff,0xff,0xff,0xff,0xff,0xff },
        .mlen = 16,
        .mac = { 0x3f,0x5b,0x8c,0xc9,0xea,0x85,0x5a,0x0a,0xfa,
                 0x73,0x47,0xd2,0x3e,0x8d,0x66,0x4e },
        .tlen = 10,
    }, {   // 3
        .key = { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00 },
        .klen = 16,
        .msg = { 0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00 },
        .mlen = 16,
        .mac = { 0x3a,0xd7,0x8e,0x72,0x6c,0x1e,0xc0,0x2b,0x7e,
                 0xbf,0xe9,0x2b,0x23,0xd9,0xec,0x34 },
        .tlen = 4,
        .chunks_msg = { 7, 7, 2 },
        .num_chunks_message = 3,
    }, {   // 4
        .key = { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00 },
        .klen = 16,
        .msg = { 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00 },
        .mlen = 16,
        .mac = { 0xf8,0x07,0xc3,0xe7,0x98,0x5f,0xe0,0xf5,0xa5,
                 0x0e,0x2c,0xdb,0x25,0xc5,0x10,0x9e },
        .tlen = 15,
        .chunks_msg = { 5, 7, 4 },
        .num_chunks_message = 3,
    }, {   // 5
        .key = { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00 },
        .klen = 16,
        .msg = { 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
                 0xff,0xff,0xff,0xff,0xff,0xff,0xff },
        .mlen = 16,
        .mac = { 0x3f,0x5b,0x8c,0xc9,0xea,0x85,0x5a,0x0a,0xfa,
                 0x73,0x47,0xd2,0x3e,0x8d,0x66,0x4e },
        .tlen = 10,
        .chunks_msg = { 1, 2, 3, 4, 6 },
        .num_chunks_message = 5,
    }, {   // 6
        .key = { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00 },
        .klen = 16,
        .msg = { 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
                 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
                 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
                 0xff,0xff,0xff,0xff,0xff },
        .mlen = 32,
        .mac = { 0x7e,0x75,0x51,0x49,0x15,0x3e,0x01,0x75,0xfb,
                 0x9e,0x26,0x6e,0xe8,0x04,0x4d,0x0d },
        .tlen = 10,
        .chunks_msg = { 10, 2, 8, 10, 2 },
        .num_chunks_message = 5,
    }, {   // 7
        .key = { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00 },
        .klen = 16,
        .msg = { 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
                 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
                 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
                 0xff,0xff,0xff },
        .mlen = 30,
        .mac = { 0x3d,0x40,0xa3,0x24,0x33,0x75,0xcc,0x40,0xb6,
                 0xa8,0x63,0xdd,0x05,0x8b,0xe6,0x04 },
        .tlen = 10,
        .chunks_msg = { 10, 2, 8, 10 },
        .num_chunks_message = 4,
    }
};

/**
 * Derived CBC-MAC test vectors from AES-CBC test vectors
 * https://csrc.nist.gov/projects/cryptographic-algorithm-validation-program/
 * block-ciphers#AES
 **/
static struct mac_test_vector aes192_cbc_mac_tv[] = {
    {   // 0
        .key = { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00 },
        .klen = 24,
        .msg = { 0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00 },
        .mlen = 16,
        .mac = { 0x6c,0xd0,0x25,0x13,0xe8,0xd4,0xdc,0x98,0x6b,
                 0x4a,0xfe,0x08,0x7a,0x60,0xbd,0x0c },
        .tlen = 4,
    }, {   // 1
        .key = { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00 },
        .klen = 24,
        .msg = { 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00 },
        .mlen = 16,
        .mac = { 0x93,0xba,0xaf,0xfb,0x35,0xfb,0xe7,0x39,0xc1,
                 0x7c,0x6a,0xc2,0x2e,0xec,0xf1,0x8f },
        .tlen = 15,
    }, {   // 2
        .key = { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00 },
        .klen = 24,
        .msg = { 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
                 0xff,0xff,0xff,0xff,0xff,0xff,0xff },
        .mlen = 16,
        .mac = { 0xb1,0x3d,0xb4,0xda,0x1f,0x71,0x8b,0xc6,0x90,
                 0x47,0x97,0xc8,0x2b,0xcf,0x2d,0x32 },
        .tlen = 10,
    }, {   // 3
        .key = { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00 },
        .klen = 24,
        .msg = { 0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00 },
        .mlen = 16,
        .mac = { 0x6c,0xd0,0x25,0x13,0xe8,0xd4,0xdc,0x98,0x6b,
                 0x4a,0xfe,0x08,0x7a,0x60,0xbd,0x0c },
        .tlen = 4,
        .chunks_msg = { 7, 7, 2 },
        .num_chunks_message = 3,
    }, {   // 4
        .key = { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00 },
        .klen = 24,
        .msg = { 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00 },
        .mlen = 16,
        .mac = { 0x93,0xba,0xaf,0xfb,0x35,0xfb,0xe7,0x39,0xc1,
                 0x7c,0x6a,0xc2,0x2e,0xec,0xf1,0x8f },
        .tlen = 15,
        .chunks_msg = { 5, 7, 4 },
        .num_chunks_message = 3,
    }, {   // 5
        .key = { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00 },
        .klen = 24,
        .msg = { 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
                 0xff,0xff,0xff,0xff,0xff,0xff,0xff },
        .mlen = 16,
        .mac = { 0xb1,0x3d,0xb4,0xda,0x1f,0x71,0x8b,0xc6,0x90,
                 0x47,0x97,0xc8,0x2b,0xcf,0x2d,0x32 },
        .tlen = 10,
        .chunks_msg = { 1, 2, 3, 4, 6 },
        .num_chunks_message = 5,
    }, {   // 6
        .key = { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00 },
        .klen = 24,
        .msg = { 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
                 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
                 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
                 0xff,0xff,0xff,0xff,0xff },
        .mlen = 32,
        .mac = { 0x60,0x4f,0x50,0xbc,0xdc,0x29,0x13,0xce,0xc7,
                 0xb5,0x94,0xcc,0xd3,0x3e,0x6b,0xaa },
        .tlen = 10,
        .chunks_msg = { 10, 2, 8, 10, 2 },
        .num_chunks_message = 5,
    }, {   // 7
        .key = { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00 },
        .klen = 24,
        .msg = { 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
                 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
                 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
                 0xff,0xff,0xff },
        .mlen = 30,
        .mac = { 0x79,0x64,0xcf,0x7d,0x8f,0x2d,0xee,0x93,0x99,
                 0x5f,0x8e,0xb7,0xa7,0x3f,0xb0,0xab },
        .tlen = 10,
        .chunks_msg = { 10, 2, 8, 10 },
        .num_chunks_message = 4,
    }
};

/**
 * Derived CBC-MAC test vectors from AES-CBC test vectors
 * https://csrc.nist.gov/projects/cryptographic-algorithm-validation-program/
 * block-ciphers#AES
 **/
static struct mac_test_vector aes256_cbc_mac_tv[] = {
    {   // 0
        .key = { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00 },
        .klen = 32,
        .msg = { 0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00 },
        .mlen = 16,
        .mac = { 0xdd,0xc6,0xbf,0x79,0x0c,0x15,0x76,0x0d,0x8d,
                 0x9a,0xeb,0x6f,0x9a,0x75,0xfd,0x4e },
        .tlen = 4,
    }, {   // 1
        .key = { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00 },
        .klen = 32,
        .msg = { 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00 },
        .mlen = 16,
        .mac = { 0x9b,0x58,0xdb,0xfd,0x77,0xfe,0x5a,0xca,0x9c,
                 0xfc,0x19,0x0c,0xd1,0xb8,0x2d,0x19 },
        .tlen = 15,
    }, {   // 2
        .key = { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00 },
        .klen = 32,
        .msg = { 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
                 0xff,0xff,0xff,0xff,0xff,0xff,0xff },
        .mlen = 16,
        .mac = { 0xac,0xda,0xce,0x80,0x78,0xa3,0x2b,0x1a,0x18,
                 0x2b,0xfa,0x49,0x87,0xca,0x13,0x47 },
        .tlen = 10,
    }, {   // 3
        .key = { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00 },
        .klen = 32,
        .msg = { 0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00 },
        .mlen = 16,
        .mac = { 0xdd,0xc6,0xbf,0x79,0x0c,0x15,0x76,0x0d,0x8d,
                 0x9a,0xeb,0x6f,0x9a,0x75,0xfd,0x4e },
        .tlen = 4,
        .chunks_msg = { 7, 7, 2 },
        .num_chunks_message = 3,
    }, {   // 4
        .key = { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00 },
        .klen = 32,
        .msg = { 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00 },
        .mlen = 16,
        .mac = { 0x9b,0x58,0xdb,0xfd,0x77,0xfe,0x5a,0xca,0x9c,
                 0xfc,0x19,0x0c,0xd1,0xb8,0x2d,0x19 },
        .tlen = 15,
        .chunks_msg = { 5, 7, 4 },
        .num_chunks_message = 3,
    }, {   // 5
        .key = { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00 },
        .klen = 32,
        .msg = { 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
                 0xff,0xff,0xff,0xff,0xff,0xff,0xff },
        .mlen = 16,
        .mac = { 0xac,0xda,0xce,0x80,0x78,0xa3,0x2b,0x1a,0x18,
                 0x2b,0xfa,0x49,0x87,0xca,0x13,0x47 },
        .tlen = 10,
        .chunks_msg = { 1, 2, 3, 4, 6 },
        .num_chunks_message = 5,
    }, {   // 6
        .key = { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00 },
        .klen = 32,
        .msg = { 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
                 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
                 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
                 0xff,0xff,0xff,0xff,0xff },
        .mlen = 32,
        .mac = { 0xc4,0xa5,0x8e,0xcb,0x0d,0x79,0x41,0xd2,0xa5,
                 0x90,0xc1,0xcb,0x95,0xd3,0x1d,0xfc },
        .tlen = 10,
        .chunks_msg = { 10, 2, 8, 10, 2 },
        .num_chunks_message = 5,
    }, {   // 7
        .key = { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                 0x00,0x00,0x00,0x00,0x00 },
        .klen = 32,
        .msg = { 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
                 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
                 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
                 0xff,0xff,0xff },
        .mlen = 30,
        .mac = { 0xca,0x87,0x7f,0x3d,0x45,0x0b,0x8a,0x09,0x0a,
                 0xad,0xf5,0xb9,0xab,0xe6,0x06,0xaf },
        .tlen = 10,
        .chunks_msg = { 10, 2, 8, 10 },
        .num_chunks_message = 4,
    }
};

#define NUM_OF_PUBLISHED_TESTSUITES	7

struct published_test_suite_info published_test_suites[] = {
    {
        .name = "AES_ECB",
        .tvcount = 3,
        .tv = aes_ecb_tv,
        .size = AES_BLOCK_SIZE,
        .mech = {CKM_AES_ECB, 0, 0},
    }, {
        .name = "AES_CBC",
        .tvcount = 3,
        .tv = aes_cbc_tv,
        .size = AES_BLOCK_SIZE,
        .mech = {CKM_AES_CBC, &aes_iv, AES_IV_SIZE},
    }, {
        .name = "AES_CTR",
        .tvcount = 3,
        .tv = aes_ctr_tv,
        .size = AES_BLOCK_SIZE,
        .mech = {CKM_AES_CTR, &aesctr, sizeof(aesctr)},
    }, {
        .name = "AES_GCM",
        .tvcount = 10,
        .tv = aes_gcm_tv,
        .size = AES_BLOCK_SIZE,
        .mech = {CKM_AES_GCM, &aesgcm, sizeof(aesgcm)},
    }, {
        .name = "AES_CFB8",
        .tvcount = 3,
        .tv = aes_cfb8_tv,
        .size = 1,
        .mech = {CKM_AES_CFB8, &aes_iv, AES_IV_SIZE},
    }, {
        .name = "AES_CFB128",
        .tvcount = 3,
        .tv = aes_cfb128_tv,
        .size = AES_BLOCK_SIZE,
        .mech = {CKM_AES_CFB128, &aes_iv, AES_IV_SIZE},
    }, {
        .name = "AES_OFB",
        .tvcount = 3,
        .tv = aes_ofb_tv,
        .size = AES_BLOCK_SIZE,
        .mech = {CKM_AES_OFB, &aes_iv, AES_IV_SIZE},
    }
};

#define NUM_OF_GENERATED_TESTSUITES 4

struct generated_test_suite_info generated_test_suites[] = {
    {
        .name = "AES_ECB",
        .mech = {CKM_AES_ECB, 0, 0},
    }, {
        .name = "AES_CBC",
        .mech = {CKM_AES_CBC, &aes_iv, AES_IV_SIZE},
    }, {
        .name = "AES_CBC_PAD",
        .mech = {CKM_AES_CBC_PAD, &aes_iv, AES_IV_SIZE},
    }, {
        .name = "AES_CTR",
        .mech = {CKM_AES_CTR, &aesctr, sizeof(aesctr)},
    }
};

#define NUM_OF_GENERATED_ERR_TESTSUITES 2

struct generated_test_suite_info generated_err_test_suites[] = {
    {
        .name = "AES_ECB",
        .mech = {CKM_AES_ECB, 0, 0},
    }, {
        .name = "AES_CBC",
        .mech = {CKM_AES_CBC, &aes_iv, AES_IV_SIZE},
    }
};

#define NUM_OF_PUBLISHED_MAC_TESTSUITES 15

struct published_mac_test_suite_info published_mac_test_suites[] = {
    {
        .name = "AES_CMAC_GENERAL (128)",
        .tvcount = 22,
        .tv = aes128_cmac_tv,
        .mech = {CKM_AES_CMAC_GENERAL, 0, 0},
    }, {
        .name = "AES_CMAC_GENERAL (192)",
        .tvcount = 20,
        .tv = aes192_cmac_tv,
        .mech = {CKM_AES_CMAC_GENERAL, 0, 0},
    }, {
        .name = "AES_CMAC_GENERAL (256)",
        .tvcount = 14,
        .tv = aes256_cmac_tv,
        .mech = {CKM_AES_CMAC_GENERAL, 0, 0},
    }, {
        .name = "AES_CMAC (128)",
        .tvcount = 22,
        .tv = aes128_cmac_tv,
        .mech = {CKM_AES_CMAC, 0, 0},
    }, {
        .name = "AES_CMAC (192)",
        .tvcount = 20,
        .tv = aes192_cmac_tv,
        .mech = {CKM_AES_CMAC, 0, 0},
    }, {
        .name = "AES_CMAC (256)",
        .tvcount = 14,
        .tv = aes256_cmac_tv,
        .mech = {CKM_AES_CMAC, 0, 0},
    }, {
        .name = "IBM-CMAC (128)",
        .tvcount = 22,
        .tv = aes128_cmac_tv,
        .mech = {CKM_IBM_CMAC, 0, 0},
    }, {
        .name = "IBM-CMAC (192)",
        .tvcount = 20,
        .tv = aes192_cmac_tv,
        .mech = {CKM_IBM_CMAC, 0, 0},
    }, {
        .name = "IBM-CMAC (256)",
        .tvcount = 14,
        .tv = aes256_cmac_tv,
        .mech = {CKM_IBM_CMAC, 0, 0},
    }, {
        .name = "AES_MAC_GENERAL (128)",
        .tvcount = 8,
        .tv = aes128_cbc_mac_tv,
        .mech = {CKM_AES_MAC_GENERAL, 0, 0},
    }, {
        .name = "AES_MAC_GENERAL (192)",
        .tvcount = 8,
        .tv = aes192_cbc_mac_tv,
        .mech = {CKM_AES_MAC_GENERAL, 0, 0},
    }, {
        .name = "AES_MAC_GENERAL (256)",
        .tvcount = 8,
        .tv = aes256_cbc_mac_tv,
        .mech = {CKM_AES_MAC_GENERAL, 0, 0},
    }, {
        .name = "AES_MAC (128)",
        .tvcount = 8,
        .tv = aes128_cbc_mac_tv,
        .mech = {CKM_AES_MAC, 0, 0},
    }, {
        .name = "AES_MAC (192)",
        .tvcount = 8,
        .tv = aes192_cbc_mac_tv,
        .mech = {CKM_AES_MAC, 0, 0},
    }, {
        .name = "AES_MAC (256)",
        .tvcount = 8,
        .tv = aes256_cbc_mac_tv,
        .mech = {CKM_AES_MAC, 0, 0},
   },
};
