/*
             Common Public License Version 0.5

             THE ACCOMPANYING PROGRAM IS PROVIDED UNDER THE TERMS OF
             THIS COMMON PUBLIC LICENSE ("AGREEMENT"). ANY USE,
             REPRODUCTION OR DISTRIBUTION OF THE PROGRAM CONSTITUTES
             RECIPIENT'S ACCEPTANCE OF THIS AGREEMENT.

             1. DEFINITIONS

             "Contribution" means: 
                   a) in the case of the initial Contributor, the
                   initial code and documentation distributed under
                   this Agreement, and 

                   b) in the case of each subsequent Contributor:
                   i) changes to the Program, and
                   ii) additions to the Program;

                   where such changes and/or additions to the Program
                   originate from and are distributed by that
                   particular Contributor. A Contribution 'originates'
                   from a Contributor if it was added to the Program
                   by such Contributor itself or anyone acting on such
                   Contributor's behalf. Contributions do not include
                   additions to the Program which: (i) are separate
                   modules of software distributed in conjunction with
                   the Program under their own license agreement, and
                   (ii) are not derivative works of the Program.


             "Contributor" means any person or entity that distributes
             the Program.

             "Licensed Patents " mean patent claims licensable by a
             Contributor which are necessarily infringed by the use or
             sale of its Contribution alone or when combined with the
             Program. 

             "Program" means the Contributions distributed in
             accordance with this Agreement.

             "Recipient" means anyone who receives the Program under
             this Agreement, including all Contributors.

             2. GRANT OF RIGHTS

                   a) Subject to the terms of this Agreement, each
                   Contributor hereby grants Recipient a
                   non-exclusive, worldwide, royalty-free copyright
                   license to reproduce, prepare derivative works of,
                   publicly display, publicly perform, distribute and
                   sublicense the Contribution of such Contributor, if
                   any, and such derivative works, in source code and
                   object code form.

                   b) Subject to the terms of this Agreement, each
                   Contributor hereby grants Recipient a
                   non-exclusive, worldwide, royalty-free patent
                   license under Licensed Patents to make, use, sell,
                   offer to sell, import and otherwise transfer the
                   Contribution of such Contributor, if any, in source
                   code and object code form. This patent license
                   shall apply to the combination of the Contribution
                   and the Program if, at the time the Contribution is
                   added by the Contributor, such addition of the
                   Contribution causes such combination to be covered
                   by the Licensed Patents. The patent license shall
                   not apply to any other combinations which include
                   the Contribution. No hardware per se is licensed
                   hereunder.

                   c) Recipient understands that although each
                   Contributor grants the licenses to its
                   Contributions set forth herein, no assurances are
                   provided by any Contributor that the Program does
                   not infringe the patent or other intellectual
                   property rights of any other entity. Each
                   Contributor disclaims any liability to Recipient
                   for claims brought by any other entity based on
                   infringement of intellectual property rights or
                   otherwise. As a condition to exercising the rights
                   and licenses granted hereunder, each Recipient
                   hereby assumes sole responsibility to secure any
                   other intellectual property rights needed, if any.

                   For example, if a third party patent license is
                   required to allow Recipient to distribute the
                   Program, it is Recipient's responsibility to
                   acquire that license before distributing the
                   Program.

                   d) Each Contributor represents that to its
                   knowledge it has sufficient copyright rights in its
                   Contribution, if any, to grant the copyright
                   license set forth in this Agreement.

             3. REQUIREMENTS

             A Contributor may choose to distribute the Program in
             object code form under its own license agreement, provided
             that:
                   a) it complies with the terms and conditions of
                   this Agreement; and

                   b) its license agreement:
                   i) effectively disclaims on behalf of all
                   Contributors all warranties and conditions, express
                   and implied, including warranties or conditions of
                   title and non-infringement, and implied warranties
                   or conditions of merchantability and fitness for a
                   particular purpose;

                   ii) effectively excludes on behalf of all
                   Contributors all liability for damages, including
                   direct, indirect, special, incidental and
                   consequential damages, such as lost profits;

                   iii) states that any provisions which differ from
                   this Agreement are offered by that Contributor
                   alone and not by any other party; and

                   iv) states that source code for the Program is
                   available from such Contributor, and informs
                   licensees how to obtain it in a reasonable manner
                   on or through a medium customarily used for
                   software exchange.

             When the Program is made available in source code form:
                   a) it must be made available under this Agreement;
                   and
                   b) a copy of this Agreement must be included with
                   each copy of the Program. 

             Contributors may not remove or alter any copyright notices
             contained within the Program.

             Each Contributor must identify itself as the originator of
             its Contribution, if any, in a manner that reasonably
             allows subsequent Recipients to identify the originator of
             the Contribution. 


             4. COMMERCIAL DISTRIBUTION

             Commercial distributors of software may accept certain
             responsibilities with respect to end users, business
             partners and the like. While this license is intended to
             facilitate the commercial use of the Program, the
             Contributor who includes the Program in a commercial
             product offering should do so in a manner which does not
             create potential liability for other Contributors.
             Therefore, if a Contributor includes the Program in a
             commercial product offering, such Contributor ("Commercial
             Contributor") hereby agrees to defend and indemnify every
             other Contributor ("Indemnified Contributor") against any
             losses, damages and costs (collectively "Losses") arising
             from claims, lawsuits and other legal actions brought by a
             third party against the Indemnified Contributor to the
             extent caused by the acts or omissions of such Commercial
             Contributor in connection with its distribution of the
             Program in a commercial product offering. The obligations
             in this section do not apply to any claims or Losses
             relating to any actual or alleged intellectual property
             infringement. In order to qualify, an Indemnified
             Contributor must: a) promptly notify the Commercial
             Contributor in writing of such claim, and b) allow the
             Commercial Contributor to control, and cooperate with the
             Commercial Contributor in, the defense and any related
             settlement negotiations. The Indemnified Contributor may
             participate in any such claim at its own expense.


             For example, a Contributor might include the Program in a
             commercial product offering, Product X. That Contributor
             is then a Commercial Contributor. If that Commercial
             Contributor then makes performance claims, or offers
             warranties related to Product X, those performance claims
             and warranties are such Commercial Contributor's
             responsibility alone. Under this section, the Commercial
             Contributor would have to defend claims against the other
             Contributors related to those performance claims and
             warranties, and if a court requires any other Contributor
             to pay any damages as a result, the Commercial Contributor
             must pay those damages.


             5. NO WARRANTY

             EXCEPT AS EXPRESSLY SET FORTH IN THIS AGREEMENT, THE
             PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT
             WARRANTIES OR CONDITIONS OF ANY KIND, EITHER EXPRESS OR
             IMPLIED INCLUDING, WITHOUT LIMITATION, ANY WARRANTIES OR
             CONDITIONS OF TITLE, NON-INFRINGEMENT, MERCHANTABILITY OR
             FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is solely
             responsible for determining the appropriateness of using
             and distributing the Program and assumes all risks
             associated with its exercise of rights under this
             Agreement, including but not limited to the risks and
             costs of program errors, compliance with applicable laws,
             damage to or loss of data, programs or equipment, and
             unavailability or interruption of operations. 

             6. DISCLAIMER OF LIABILITY
             EXCEPT AS EXPRESSLY SET FORTH IN THIS AGREEMENT, NEITHER
             RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY
             FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
             OR CONSEQUENTIAL DAMAGES (INCLUDING WITHOUT LIMITATION
             LOST PROFITS), HOWEVER CAUSED AND ON ANY THEORY OF
             LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
             (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
             OF THE USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE
             OF ANY RIGHTS GRANTED HEREUNDER, EVEN IF ADVISED OF THE
             POSSIBILITY OF SUCH DAMAGES.

             7. GENERAL

             If any provision of this Agreement is invalid or
             unenforceable under applicable law, it shall not affect
             the validity or enforceability of the remainder of the
             terms of this Agreement, and without further action by the
             parties hereto, such provision shall be reformed to the
             minimum extent necessary to make such provision valid and
             enforceable.


             If Recipient institutes patent litigation against a
             Contributor with respect to a patent applicable to
             software (including a cross-claim or counterclaim in a
             lawsuit), then any patent licenses granted by that
             Contributor to such Recipient under this Agreement shall
             terminate as of the date such litigation is filed. In
             addition, If Recipient institutes patent litigation
             against any entity (including a cross-claim or
             counterclaim in a lawsuit) alleging that the Program
             itself (excluding combinations of the Program with other
             software or hardware) infringes such Recipient's
             patent(s), then such Recipient's rights granted under
             Section 2(b) shall terminate as of the date such
             litigation is filed.

             All Recipient's rights under this Agreement shall
             terminate if it fails to comply with any of the material
             terms or conditions of this Agreement and does not cure
             such failure in a reasonable period of time after becoming
             aware of such noncompliance. If all Recipient's rights
             under this Agreement terminate, Recipient agrees to cease
             use and distribution of the Program as soon as reasonably
             practicable. However, Recipient's obligations under this
             Agreement and any licenses granted by Recipient relating
             to the Program shall continue and survive. 

             Everyone is permitted to copy and distribute copies of
             this Agreement, but in order to avoid inconsistency the
             Agreement is copyrighted and may only be modified in the
             following manner. The Agreement Steward reserves the right
             to publish new versions (including revisions) of this
             Agreement from time to time. No one other than the
             Agreement Steward has the right to modify this Agreement.

             IBM is the initial Agreement Steward. IBM may assign the
             responsibility to serve as the Agreement Steward to a
             suitable separate entity. Each new version of the
             Agreement will be given a distinguishing version number.
             The Program (including Contributions) may always be
             distributed subject to the version of the Agreement under
             which it was received. In addition, after a new version of
             the Agreement is published, Contributor may elect to
             distribute the Program (including its Contributions) under
             the new version. Except as expressly stated in Sections
             2(a) and 2(b) above, Recipient receives no rights or
             licenses to the intellectual property of any Contributor
             under this Agreement, whether expressly, by implication,
             estoppel or otherwise. All rights in the Program not
             expressly granted under this Agreement are reserved.


             This Agreement is governed by the laws of the State of New
             York and the intellectual property laws of the United
             States of America. No party to this Agreement will bring a
             legal action under this Agreement more than one year after
             the cause of action arose. Each party waives its rights to
             a jury trial in any resulting litigation. 



*/

/* (C) COPYRIGHT International Business Machines Corp. 2001,2002          */


// File:  mech_rsa.c
//
// Mechanisms for RSA
//
// Routines contained within:

#include <pthread.h>
#include <stdio.h>

#include <string.h>            // for memcmp() et al
#include <stdlib.h>

#include "pkcs11types.h"
#include "defs.h"
#include "host_defs.h"
#include "h_extern.h"
#include "tok_spec_struct.h"
#include "trace.h"


CK_RV
rsa_get_key_info(OBJECT *key_obj, CK_ULONG *mod_bytes,
		CK_OBJECT_CLASS *keyclass)
{
	CK_RV		rc;
	CK_ATTRIBUTE	*attr;

	rc = template_attribute_find(key_obj->template, CKA_MODULUS, &attr);
	if (rc == FALSE) {
		TRACE_ERROR("Could not find CKA_MODULUS in the template\n");
		return CKR_FUNCTION_FAILED;
	} else
	*mod_bytes = attr->ulValueLen;

	rc = template_attribute_find(key_obj->template, CKA_CLASS, &attr);
	if (rc == FALSE) {
		TRACE_ERROR("Could not find CKA_CLASS in the template\n");
		return CKR_FUNCTION_FAILED;
	}
	else
		*keyclass = *(CK_OBJECT_CLASS *)attr->pValue;

	return CKR_OK;
}


/*
 * Format an encryption block according to PKCS #1: RSA Encryption, Version
 * 1.5.
 */

CK_RV
rsa_format_block( CK_BYTE   * in_data,
                  CK_ULONG    in_data_len,
                  CK_BYTE   * out_data,
                  CK_ULONG    out_data_len,
                  CK_ULONG    type )
{
    CK_ULONG        padding_len, i;
    CK_RV           rc;

    if (!in_data || !in_data_len || !out_data || !out_data_len) {
	TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
	return CKR_FUNCTION_FAILED;
    }

    if (out_data_len < (in_data_len + 11)) {
      TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
      rc = CKR_BUFFER_TOO_SMALL;
      return rc;
    }

    /*
     * The padding string PS shall consist of k-3-||D|| octets.
     */
    padding_len = out_data_len - 3 - in_data_len;

    /*
     * For block types 01 and 02, the padding string is at least eight octets
     * long, which is a security condition for public-key operations that
     * prevents an attacker from recoving data by trying all possible
     * encryption blocks.
     */
    if ((type == 1 || type == 2) && ((padding_len) < 8)) {
        TRACE_ERROR("%s\n", ock_err(ERR_DATA_LEN_RANGE));
        rc = CKR_DATA_LEN_RANGE;
        return rc;
    }

    /*
     * The leading 00 octet.
     */
    out_data[0] = (CK_BYTE)0;

    /*
     * The block type.
     */
    out_data[1] = (CK_BYTE)type;

    switch (type) {
        /*
         * For block type 00, the octets shall have value 00.
         * EB = 00 || 00 || 00 * i || D
         * Where D must begin with a nonzero octet.
         */
        case 0:
            if (in_data[0] == (CK_BYTE)0) {
                TRACE_ERROR("%s\n", ock_err(ERR_DATA_INVALID));
                rc = CKR_DATA_INVALID;
                return rc;
            }

            for (i = 2; i < (padding_len + 2); i++)
                out_data[i] = (CK_BYTE)0;

            break;

        /*
         * For block type 01, they shall have value FF.
         * EB = 00 || 01 || FF * i || 00 || D
         */
        case 1:
            for (i = 2; i < (padding_len + 2); i++)
                out_data[i] = (CK_BYTE)0xff;

            break;

        /*
         * For block type 02, they shall be pseudorandomly generated and
         * nonzero.
         * EB = 00 || 02 || ?? * i || 00 || D
         * Where ?? is nonzero.
         */
        case 2:
            for (i = 2; i < (padding_len + 2); i++) {
                rc = rng_generate(&out_data[i], 1);
                if (rc != CKR_OK) {
                    TRACE_DEVEL("rng_generate failed.\n");
                    return rc;
                }
                if (out_data[i] == (CK_BYTE)0) {
                    /* avoid zeros by explicitly making them all 0xff -
                     * won't hurt entropy that bad, and it's better than
                     * looping over rng_generate */
                    out_data[i] = (CK_BYTE)0xff;
                }
            }
            break;

        default:
            TRACE_ERROR("%s\n", ock_err(ERR_DATA_INVALID));
            rc = CKR_DATA_INVALID;
            return rc;
    }

    out_data[i] = (CK_BYTE)0;
    i++;

    memcpy(&out_data[i], in_data, in_data_len);

    rc = CKR_OK;
    return rc;
}


/*
 * Parse an encryption block according to PKCS #1: RSA Encryption, Version
 * 1.5.
 */

CK_RV
rsa_parse_block( CK_BYTE  * in_data,
                 CK_ULONG   in_data_len,
                 CK_BYTE  * out_data,
                 CK_ULONG * out_data_len,
                 CK_ULONG   type )
{
    CK_ULONG        i;
    CK_RV           rc;

    if (!in_data || !out_data || !out_data_len) {
	TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
	return CKR_FUNCTION_FAILED;
    }

    if (in_data_len <= 11) {
        TRACE_DEVEL("%s\n", ock_err(ERR_FUNCTION_FAILED));
        rc = CKR_FUNCTION_FAILED;
        return rc;
    }

    /*
     * Check for the leading 00 octet.
     */
    if (in_data[0] != (CK_BYTE)0) {
        TRACE_ERROR("%s\n", ock_err(ERR_ENCRYPTED_DATA_INVALID));
        rc = CKR_ENCRYPTED_DATA_INVALID;
        return rc;
    }

    /*
     * Check the block type.
     */
    if (in_data[1] != (CK_BYTE)type) {
        TRACE_ERROR("%s\n", ock_err(ERR_ENCRYPTED_DATA_INVALID));
        rc = CKR_ENCRYPTED_DATA_INVALID;
        return rc;
    }

    /*
     * The block type shall be a single octet indicating the structure of the
     * encryption block. It shall have value 00, 01, or 02. For a private-key
     * operation, the block type shall be 00 or 01. For a public-key
     * operation, it shall be 02.
     *
     * For block type 00, the octets shall have value 00; for block type 01,
     * they shall have value FF; and for block type 02, they shall be
     * pseudorandomly generated and nonzero.
     *
     * For block type 00, the data must begin with a nonzero octet or have
     * known length so that the encryption block can be parsed unambiguously.
     * For block types 01 and 02, the encryption block can be parsed
     * unambiguously since the padding string contains no octets with value 00
     * and the padding string is separated from the data by an octet with
     * value 00.
     */
    switch (type) {
        /*
         * For block type 00, the octets shall have value 00.
         * EB = 00 || 00 || 00 * i || D
         * Where D must begin with a nonzero octet.
         */
        case 0:
            for (i = 2; i <= (in_data_len - 2); i++) {
                if (in_data[i] != (CK_BYTE)0)
                    break;
            }
            break;

        /*
         * For block type 01, they shall have value FF.
         * EB = 00 || 01 || FF * i || 00 || D
         */
        case 1:
            for (i = 2; i <= (in_data_len - 2); i++) {
                if (in_data[i] != (CK_BYTE)0xff) {
                    if (in_data[i] == (CK_BYTE)0) {
                        i++;
                        break;
                    }

                    TRACE_ERROR("%s\n", ock_err(ERR_ENCRYPTED_DATA_INVALID));
                    rc = CKR_ENCRYPTED_DATA_INVALID;
                    return rc;
                }
            }
            break;

        /*
         * For block type 02, they shall be pseudorandomly generated and
         * nonzero.
         * EB = 00 || 02 || ?? * i || 00 || D
         * Where ?? is nonzero.
         */
        case 2:
            for (i = 2; i <= (in_data_len - 2); i++) {
                if (in_data[i] == (CK_BYTE)0) {
                    i++;
                    break;
                }
            }
            break;

        default:
            TRACE_ERROR("%s\n", ock_err(ERR_ENCRYPTED_DATA_INVALID));
            rc = CKR_ENCRYPTED_DATA_INVALID;
            return rc;
    }

    /*
     * For block types 01 and 02, the padding string is at least eight octets
     * long, which is a security condition for public-key operations that
     * prevents an attacker from recoving data by trying all possible
     * encryption blocks.
     */
    if ((type == 1 || type == 2) && ((i - 3) < 8)) {
        TRACE_ERROR("%s\n", ock_err(ERR_ENCRYPTED_DATA_INVALID));
        rc = CKR_ENCRYPTED_DATA_INVALID;
        return rc;
    }

    if (in_data_len <= i) {
        TRACE_ERROR("%s\n", ock_err(ERR_ENCRYPTED_DATA_INVALID));
        rc = CKR_ENCRYPTED_DATA_INVALID;
        return rc;
    }

    if (*out_data_len < (in_data_len - i)) {
      TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
      rc = CKR_BUFFER_TOO_SMALL;
      return rc;
    }

    memcpy(out_data, &in_data[i], in_data_len - i);
    *out_data_len = in_data_len - i;

    rc = CKR_OK;
    return rc;
}

/* helper function for rsa-oaep */
CK_RV get_mgf_mech(CK_RSA_PKCS_MGF_TYPE mgf, CK_MECHANISM_TYPE *mech)
{
	switch(mgf) {
	case CKG_MGF1_SHA1:
		*mech = CKM_SHA_1;
		break;
	case CKG_MGF1_SHA256:
		*mech = CKM_SHA256;
		break;
	case CKG_MGF1_SHA384:
		*mech = CKM_SHA384;
		break;
	case CKG_MGF1_SHA512:
		*mech = CKM_SHA512;
		break;
	default:
		return CKR_MECHANISM_INVALID;
	}

	return CKR_OK;
}

//
//
CK_RV
rsa_pkcs_encrypt( SESSION           *sess,
                  CK_BBOOL           length_only,
                  ENCR_DECR_CONTEXT *ctx,
                  CK_BYTE           *in_data,
                  CK_ULONG           in_data_len,
                  CK_BYTE           *out_data,
                  CK_ULONG          *out_data_len )
{
   OBJECT          *key_obj  = NULL;
   CK_ULONG         modulus_bytes;
   CK_OBJECT_CLASS  keyclass;
   CK_RV            rc;


   rc = object_mgr_find_in_map1( ctx->key, &key_obj );
   if (rc != CKR_OK){
      TRACE_ERROR("Failed to acquire key from specified handle");
      if (rc == CKR_OBJECT_HANDLE_INVALID)
	 return CKR_KEY_HANDLE_INVALID;
      else
	 return rc;
   }

   rc = rsa_get_key_info(key_obj, &modulus_bytes, &keyclass);
   if (rc != CKR_OK) {
      TRACE_DEVEL("rsa_get_key_info failed.\n");
      return rc;
   }

   // check input data length restrictions
   //
   if (in_data_len > (modulus_bytes - 11)){
      TRACE_ERROR("%s\n", ock_err(ERR_DATA_LEN_RANGE));
      return CKR_DATA_LEN_RANGE;
   }

   if (length_only == TRUE) {
      *out_data_len = modulus_bytes;
      return CKR_OK;
   }

   if (*out_data_len < modulus_bytes) {
      *out_data_len = modulus_bytes;
      TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
      return CKR_BUFFER_TOO_SMALL;
   }

   // this had better be a public key
   if (keyclass != CKO_PUBLIC_KEY){
      TRACE_ERROR("This operation requires a public key.\n");
      return CKR_KEY_FUNCTION_NOT_PERMITTED;
   }

   if ( token_specific.t_rsa_encrypt == NULL ) {
     TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
     return CKR_MECHANISM_INVALID;
   }

   rc = token_specific.t_rsa_encrypt(in_data, in_data_len, out_data, out_data_len, key_obj);

   if (rc != CKR_OK)
      TRACE_DEVEL("Token Specific rsa encrypt failed.\n");

   return rc;
}


//
//
CK_RV
rsa_pkcs_decrypt( SESSION           *sess,
                  CK_BBOOL           length_only,
                  ENCR_DECR_CONTEXT *ctx,
                  CK_BYTE           *in_data,
                  CK_ULONG           in_data_len,
                  CK_BYTE           *out_data,
                  CK_ULONG          *out_data_len )
{
   OBJECT          *key_obj  = NULL;
   CK_ULONG         modulus_bytes;
   CK_OBJECT_CLASS  keyclass;
   CK_RV            rc;


   rc = object_mgr_find_in_map1( ctx->key, &key_obj );
   if (rc != CKR_OK) {
      TRACE_ERROR("Failed to acquire key from specified handle");
      if (rc == CKR_OBJECT_HANDLE_INVALID)
	 return CKR_KEY_HANDLE_INVALID;
      else
         return rc;
   }

   rc = rsa_get_key_info(key_obj, &modulus_bytes, &keyclass);
   if (rc != CKR_OK) {
      TRACE_DEVEL("rsa_get_key_info failed.\n");
      return rc;
   }

   // check input data length restrictions
   //
   if (in_data_len != modulus_bytes){
      TRACE_ERROR("%s\n", ock_err(ERR_ENCRYPTED_DATA_LEN_RANGE));
      return CKR_ENCRYPTED_DATA_LEN_RANGE;
   }
   if (length_only == TRUE) {
      // this is not exact but it's the upper bound; otherwise we'll need
      // to do the RSA operation just to get the required length
      //
      *out_data_len = modulus_bytes - 11;
      return CKR_OK;
   }

   if (*out_data_len < (modulus_bytes - 11)) {
      *out_data_len = modulus_bytes - 11;
      TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
      return CKR_BUFFER_TOO_SMALL;
   }

   // this had better be a private key
   if (keyclass != CKO_PRIVATE_KEY){
      TRACE_ERROR("This operation requires a private key.\n");
      return CKR_KEY_FUNCTION_NOT_PERMITTED;
   }

   /* check for token specific call first */
   if( token_specific.t_rsa_decrypt == NULL ) {
     TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
     return CKR_MECHANISM_INVALID;
   }

   rc = token_specific.t_rsa_decrypt(in_data, in_data_len, out_data, out_data_len, key_obj);

   if (rc != CKR_OK) {
      if (rc == CKR_DATA_LEN_RANGE) {
         TRACE_ERROR("%s\n", ock_err(ERR_ENCRYPTED_DATA_LEN_RANGE));
         return CKR_ENCRYPTED_DATA_LEN_RANGE;
      }
      TRACE_DEVEL("Token Specific rsa decrypt failed.\n");
   }

   return rc;
}


CK_RV rsa_oaep_crypt(SESSION *sess, CK_BBOOL length_only,
		     ENCR_DECR_CONTEXT *ctx, CK_BYTE *in_data,
		     CK_ULONG in_data_len, CK_BYTE *out_data,
		     CK_ULONG *out_data_len, CK_BBOOL encrypt)
{
	OBJECT *key_obj = NULL;
	CK_ULONG hlen, modulus_bytes;
	CK_OBJECT_CLASS keyclass;
	CK_BYTE hash[MAX_SHA_HASH_SIZE];
	CK_RV rc;
	CK_RSA_PKCS_OAEP_PARAMS_PTR oaepParms = NULL;

	rc = object_mgr_find_in_map1(ctx->key, &key_obj);
	if (rc != CKR_OK) {
		TRACE_ERROR("Failed to acquire key from specified handle");
		if (rc == CKR_OBJECT_HANDLE_INVALID)
			return CKR_KEY_HANDLE_INVALID;
		else
			return rc;
	}

	rc = rsa_get_key_info(key_obj, &modulus_bytes, &keyclass);
	if (rc != CKR_OK) {
		TRACE_DEVEL("rsa_get_key_info failed.\n");
		return CKR_FUNCTION_FAILED;
	}

	if (length_only == TRUE) {
		*out_data_len = modulus_bytes;
		return CKR_OK;
	}

	if (*out_data_len < modulus_bytes) {
		*out_data_len = modulus_bytes;
		TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
		return CKR_BUFFER_TOO_SMALL;
	}

	/*
	 * To help mitigate timing and fault attacks when decrypting,
	 * check oaep parameters that are passed in right now and compute
	 * the hash of the Label.
	 *
	 * PKCS#11v2.20, section 12.1.7, Step a: if "source" is empty,
	 * then pSourceData and ulSourceDatalen must be NULL, and zero
	 * respectively.
	 */
	oaepParms = (CK_RSA_PKCS_OAEP_PARAMS_PTR)ctx->mech.pParameter;
	if (!(oaepParms->source) && (oaepParms->pSourceData ||
		oaepParms->ulSourceDataLen)) {
		TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_PARAM_INVALID));
		return CKR_MECHANISM_PARAM_INVALID;
	}

	/* verify hashAlg now as well as get hash size. */
	hlen = 0;
	rc = get_sha_size(oaepParms->hashAlg, &hlen);
	if (rc != CKR_OK) {
		TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_PARAM_INVALID));
		return CKR_MECHANISM_PARAM_INVALID;
	}

	/* modulus size should be >= 2*hashsize+2 */
	if (modulus_bytes < (2 * hlen + 2)) {
		TRACE_ERROR("%s\n", ock_err(ERR_KEY_SIZE_RANGE));
		return CKR_KEY_SIZE_RANGE;
	}

	/* hash the label now */
	if (!(oaepParms->pSourceData) || !(oaepParms->ulSourceDataLen))
		rc = compute_sha("", 0, hash, oaepParms->hashAlg);
	else
		rc = compute_sha(oaepParms->pSourceData,
				 oaepParms->ulSourceDataLen, hash,
				 oaepParms->hashAlg);
	
	if (encrypt) {
		if (in_data_len > (modulus_bytes - 2 * hlen - 2)) {
			TRACE_ERROR("%s\n", ock_err(ERR_DATA_LEN_RANGE));
			return CKR_DATA_LEN_RANGE;
		}

		// this had better be a public key
		if (keyclass != CKO_PUBLIC_KEY) {
			TRACE_ERROR("This operation requires a public key.\n");
			return CKR_KEY_FUNCTION_NOT_PERMITTED;
		}

		if (token_specific.t_rsa_oaep_encrypt == NULL) {
			TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
			return CKR_MECHANISM_INVALID;
		}

		rc = token_specific.t_rsa_oaep_encrypt(ctx, in_data,
						       in_data_len, out_data,
						       out_data_len, hash,
						       hlen);
	} else {
		// decrypt
		if (in_data_len != modulus_bytes) {
			TRACE_ERROR("%s\n",
				    ock_err(ERR_ENCRYPTED_DATA_LEN_RANGE));
			return CKR_ENCRYPTED_DATA_LEN_RANGE;
		}

		// this had better be a private key
		if (keyclass != CKO_PRIVATE_KEY) {
			TRACE_ERROR("This operation requires a private key.\n");
			return CKR_KEY_FUNCTION_NOT_PERMITTED;
		}

		if (token_specific.t_rsa_oaep_decrypt == NULL) {
			TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
			return CKR_MECHANISM_INVALID;
		}

		 rc = token_specific.t_rsa_oaep_decrypt(ctx, in_data,
							in_data_len, out_data,
							out_data_len, hash,
							hlen);
	}

	if (rc != CKR_OK)
		TRACE_DEVEL("Token Specific rsa oaep decrypt failed.\n");

	return rc;
}

CK_RV
rsa_pkcs_sign( SESSION             *sess,
               CK_BBOOL             length_only,
               SIGN_VERIFY_CONTEXT *ctx,
               CK_BYTE             *in_data,
               CK_ULONG             in_data_len,
               CK_BYTE             *out_data,
               CK_ULONG            *out_data_len )
{
   OBJECT          *key_obj   = NULL;
   CK_ULONG         modulus_bytes;
   CK_OBJECT_CLASS  keyclass;
   CK_RV            rc;


   if (!sess || !ctx || !out_data_len){
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   rc = object_mgr_find_in_map1( ctx->key, &key_obj );
   if (rc != CKR_OK){
      TRACE_ERROR("Failed to acquire key from specified handle");
      if (rc == CKR_OBJECT_HANDLE_INVALID)
	 return CKR_KEY_HANDLE_INVALID;
      else
         return rc;
   }

   rc = rsa_get_key_info(key_obj, &modulus_bytes, &keyclass);
   if (rc != CKR_OK) {
      TRACE_DEVEL("rsa_get_key_info failed.\n");
      return rc;
   }

   // check input data length restrictions
   //
   if (in_data_len > (modulus_bytes - 11)){
      TRACE_ERROR("%s\n", ock_err(ERR_DATA_LEN_RANGE));
      return CKR_DATA_LEN_RANGE;
   }
   if (length_only == TRUE) {
      *out_data_len = modulus_bytes;
      return CKR_OK;
   }

   if (*out_data_len < modulus_bytes) {
      *out_data_len = modulus_bytes;
      TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
      return CKR_BUFFER_TOO_SMALL;
   }

   // this had better be a private key
   //
   if (keyclass != CKO_PRIVATE_KEY){
      TRACE_ERROR("This operation requires a private key.\n");
      return CKR_KEY_FUNCTION_NOT_PERMITTED;
   }

   /* check for token specific call first */
   if(token_specific.t_rsa_sign == NULL) {
     TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
     return CKR_MECHANISM_INVALID;
   }

   rc = token_specific.t_rsa_sign(in_data, in_data_len, out_data, out_data_len, key_obj);

   if (rc != CKR_OK)
      TRACE_DEVEL("Token Specific rsa sign failed.\n");
   return rc;
}


//
//
CK_RV
rsa_pkcs_verify( SESSION             * sess,
                 SIGN_VERIFY_CONTEXT * ctx,
                 CK_BYTE             * in_data,
                 CK_ULONG              in_data_len,
                 CK_BYTE             * signature,
                 CK_ULONG              sig_len )
{
   OBJECT          *key_obj  = NULL;
   CK_ULONG         modulus_bytes;
   CK_OBJECT_CLASS  keyclass;
   CK_RV            rc;


   rc = object_mgr_find_in_map1( ctx->key, &key_obj );
   if (rc != CKR_OK){
      TRACE_ERROR("Failed to acquire key from specified handle");
      if (rc == CKR_OBJECT_HANDLE_INVALID)
	 return CKR_KEY_HANDLE_INVALID;
      else
         return rc;
   }

   rc = rsa_get_key_info(key_obj, &modulus_bytes, &keyclass);
   if (rc != CKR_OK) {
      TRACE_DEVEL("rsa_get_key_info failed.\n");
      return rc;
   }

   // check input data length restrictions
   //
   if (sig_len != modulus_bytes){
      TRACE_ERROR("%s\n", ock_err(ERR_SIGNATURE_LEN_RANGE));
      return CKR_SIGNATURE_LEN_RANGE;
   }

   // verifying is a public key operation
   //
   if (keyclass != CKO_PUBLIC_KEY) {
      TRACE_ERROR("This operation requires a public key.\n");
      return CKR_KEY_FUNCTION_NOT_PERMITTED;
   }

   /* check for token specific call first */
   if (token_specific.t_rsa_verify == NULL) {
      TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
      return CKR_MECHANISM_INVALID;
   }

   rc = token_specific.t_rsa_verify(in_data, in_data_len, signature, sig_len, key_obj);
   if (rc != CKR_OK)
      TRACE_DEVEL("Token Specific rsa verify failed.\n");

   return rc;
}


//
//
CK_RV
rsa_pkcs_verify_recover( SESSION             * sess,
                         CK_BBOOL              length_only,
                         SIGN_VERIFY_CONTEXT * ctx,
                         CK_BYTE             * signature,
                         CK_ULONG              sig_len,
                         CK_BYTE             * out_data,
                         CK_ULONG            * out_data_len )
{
   OBJECT          *key_obj  = NULL;
   CK_OBJECT_CLASS  keyclass;
   CK_ULONG         modulus_bytes;
   CK_RV            rc;


   if (!sess || !ctx || !out_data_len){
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   rc = object_mgr_find_in_map1( ctx->key, &key_obj );
   if (rc != CKR_OK){
      TRACE_ERROR("Failed to acquire key from specified handle");
      if (rc == CKR_OBJECT_HANDLE_INVALID)
	 return CKR_KEY_HANDLE_INVALID;
      else
         return rc;
   }

   rc = rsa_get_key_info(key_obj, &modulus_bytes, &keyclass);
   if (rc != CKR_OK) {
      TRACE_DEVEL("rsa_get_key_info failed.\n");
      return rc;
   }

   // check input data length restrictions
   //
   if (sig_len != modulus_bytes){
      TRACE_ERROR("%s\n", ock_err(ERR_SIGNATURE_LEN_RANGE));
      return CKR_SIGNATURE_LEN_RANGE;
   }
   if (length_only == TRUE) {
      *out_data_len = modulus_bytes - 11;
      return CKR_OK;
   }

    /* this had better be a public key */
   if (keyclass != CKO_PUBLIC_KEY) {
      TRACE_ERROR("This operation requires a public key.\n");
      return CKR_KEY_FUNCTION_NOT_PERMITTED;
   }

     /* check for token specific call first */
   if (token_specific.t_rsa_verify_recover == NULL) {
      TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
      return CKR_MECHANISM_INVALID;
   }

   rc = token_specific.t_rsa_verify_recover(signature, sig_len, out_data, out_data_len, key_obj);
   if (rc != CKR_OK)
      TRACE_DEVEL("Token Specific rsa verify failed.\n");

   return rc;
}


//
//
CK_RV
rsa_x509_encrypt( SESSION           *sess,
                  CK_BBOOL           length_only,
                  ENCR_DECR_CONTEXT *ctx,
                  CK_BYTE           *in_data,
                  CK_ULONG           in_data_len,
                  CK_BYTE           *out_data,
                  CK_ULONG          *out_data_len )
{
   OBJECT          *key_obj  = NULL;
   CK_OBJECT_CLASS  keyclass;
   CK_ULONG         modulus_bytes;
   CK_RV            rc;


   rc = object_mgr_find_in_map1( ctx->key, &key_obj );
   if (rc != CKR_OK){
      TRACE_ERROR("Failed to acquire key from specified handle");
      if (rc == CKR_OBJECT_HANDLE_INVALID)
	 return CKR_KEY_HANDLE_INVALID;
      else
         return rc;
   }

   rc = rsa_get_key_info(key_obj, &modulus_bytes, &keyclass);
   if (rc != CKR_OK) {
      TRACE_DEVEL("rsa_get_key_info failed.\n");
      return rc;
   }

   // CKM_RSA_X_509 requires input data length to be no bigger than the modulus
   //
   if (in_data_len > modulus_bytes){
      TRACE_ERROR("%s\n", ock_err(ERR_DATA_LEN_RANGE));
      return CKR_DATA_LEN_RANGE;
   }
   if (length_only == TRUE) {
      *out_data_len = modulus_bytes;
      return CKR_OK;
   }

   if (*out_data_len < modulus_bytes) {
      *out_data_len = modulus_bytes;
      TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
      return CKR_BUFFER_TOO_SMALL;
   }

    /* this had better be a public key */
   if (keyclass != CKO_PUBLIC_KEY) {
      TRACE_ERROR("This operation requires a public key.\n");
      return CKR_KEY_FUNCTION_NOT_PERMITTED;
   }

     /* check for token specific call first */
   if (token_specific.t_rsa_x509_encrypt == NULL) {
      TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
      return CKR_MECHANISM_INVALID;
   }

   rc = token_specific.t_rsa_x509_encrypt(in_data, in_data_len, out_data, out_data_len, key_obj);
   if (rc != CKR_OK)
      TRACE_DEVEL("Token Specific rsa x509 encrypt failed.\n");

   return rc;
}


//
//
CK_RV
rsa_x509_decrypt( SESSION           *sess,
                  CK_BBOOL           length_only,
                  ENCR_DECR_CONTEXT *ctx,
                  CK_BYTE           *in_data,
                  CK_ULONG           in_data_len,
                  CK_BYTE           *out_data,
                  CK_ULONG          *out_data_len )
{
   OBJECT          *key_obj  = NULL;
   CK_ULONG         modulus_bytes;
   CK_OBJECT_CLASS  keyclass;
   CK_RV            rc;


   rc = object_mgr_find_in_map1( ctx->key, &key_obj );
   if (rc != CKR_OK){
      TRACE_ERROR("Failed to acquire key from specified handle");
      if (rc == CKR_OBJECT_HANDLE_INVALID)
	 return CKR_KEY_HANDLE_INVALID;
      else
         return rc;
   }

   rc = rsa_get_key_info(key_obj, &modulus_bytes, &keyclass);
   if (rc != CKR_OK) {
      TRACE_DEVEL("rsa_get_key_info failed.\n");
      return rc;
   }

   // check input data length restrictions
   //
   if (in_data_len != modulus_bytes){
      TRACE_ERROR("%s\n", ock_err(ERR_ENCRYPTED_DATA_LEN_RANGE));
      return CKR_ENCRYPTED_DATA_LEN_RANGE;
   }
   if (length_only == TRUE) {
      *out_data_len = modulus_bytes;
      return CKR_OK;
   }

   // Although X.509 prepads with zeros, we don't strip it after
   // decryption (PKCS #11 specifies that X.509 decryption is supposed
   // to produce K bytes of cleartext where K is the modulus length)
   //
   if (*out_data_len < modulus_bytes) {
      *out_data_len = modulus_bytes;
      TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
      return CKR_BUFFER_TOO_SMALL;
   }

    /* this had better be a private key */
   if (keyclass != CKO_PRIVATE_KEY) {
      TRACE_ERROR("This operation requires a private key.\n");
      return CKR_KEY_FUNCTION_NOT_PERMITTED;
   }

    /* check for token specific call first */
   if (token_specific.t_rsa_x509_encrypt == NULL) {
      TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
      return CKR_MECHANISM_INVALID;
   }

   rc = token_specific.t_rsa_x509_decrypt(in_data, in_data_len, out_data, out_data_len, key_obj);
   if (rc != CKR_OK)
      TRACE_ERROR("Token Specific rsa x509 decrypt failed.\n");
   // ckm_rsa_operation is used for all RSA operations so we need to adjust
   // the return code accordingly
   //
   if (rc == CKR_DATA_LEN_RANGE){
      TRACE_ERROR("%s\n", ock_err(ERR_ENCRYPTED_DATA_LEN_RANGE));
      return CKR_ENCRYPTED_DATA_LEN_RANGE;
   }
   return rc;
}


//
//
CK_RV
rsa_x509_sign( SESSION             *sess,
               CK_BBOOL             length_only,
               SIGN_VERIFY_CONTEXT *ctx,
               CK_BYTE             *in_data,
               CK_ULONG             in_data_len,
               CK_BYTE             *out_data,
               CK_ULONG            *out_data_len )
{
   OBJECT          *key_obj   = NULL;
   CK_ULONG         modulus_bytes;
   CK_OBJECT_CLASS keyclass;
   CK_RV            rc;


   if (!sess || !ctx || !out_data_len){
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   rc = object_mgr_find_in_map1( ctx->key, &key_obj );
   if (rc != CKR_OK){
      TRACE_ERROR("Failed to acquire key from specified handle");
      if (rc == CKR_OBJECT_HANDLE_INVALID)
	 return CKR_KEY_HANDLE_INVALID;
      else
         return rc;
   }

   rc = rsa_get_key_info(key_obj, &modulus_bytes, &keyclass);
   if (rc != CKR_OK) {
      TRACE_DEVEL("rsa_get_key_info failed.\n");
      return rc;
   }

   // check input data length restrictions
   //
   if (in_data_len > modulus_bytes){
      TRACE_ERROR("%s\n", ock_err(ERR_DATA_LEN_RANGE));
      return CKR_DATA_LEN_RANGE;
   }
   if (length_only == TRUE) {
      *out_data_len = modulus_bytes;
      return CKR_OK;
   }

   if (*out_data_len < modulus_bytes) {
      *out_data_len = modulus_bytes;
      TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
      return CKR_BUFFER_TOO_SMALL;
   }

    /* this had better be a private key */
   if (keyclass != CKO_PRIVATE_KEY) {
      TRACE_ERROR("This operation requires a private key.\n");
      return CKR_KEY_FUNCTION_NOT_PERMITTED;
   }

    /* check for token specific call first */
   if (token_specific.t_rsa_x509_sign == NULL) {
      TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
      return CKR_MECHANISM_INVALID;
   }
   rc = token_specific.t_rsa_x509_sign(in_data, in_data_len, out_data, out_data_len, key_obj);
   if (rc != CKR_OK)
      TRACE_DEVEL("Token Specific rsa x509 sign failed.\n");

   return rc;
}


//
//
CK_RV
rsa_x509_verify( SESSION             * sess,
                 SIGN_VERIFY_CONTEXT * ctx,
                 CK_BYTE             * in_data,
                 CK_ULONG              in_data_len,
                 CK_BYTE             * signature,
                 CK_ULONG              sig_len )
{
   OBJECT          *key_obj  = NULL;
   CK_OBJECT_CLASS  keyclass;
   CK_ULONG         modulus_bytes;
   CK_RV            rc;


   rc = object_mgr_find_in_map1( ctx->key, &key_obj );
   if (rc != CKR_OK){
      TRACE_ERROR("Failed to acquire key from specified handle");
      if (rc == CKR_OBJECT_HANDLE_INVALID)
	 return CKR_KEY_HANDLE_INVALID;
      else
         return rc;
   }

   rc = rsa_get_key_info(key_obj, &modulus_bytes, &keyclass);
   if (rc != CKR_OK) {
      TRACE_DEVEL("rsa_get_key_info failed.\n");
      return rc;
   }

   // check input data length restrictions
   //
   if (sig_len != modulus_bytes){
      TRACE_ERROR("%s\n", ock_err(ERR_SIGNATURE_LEN_RANGE));
      return CKR_SIGNATURE_LEN_RANGE;
   }

    /* this had better be a public key */
   if (keyclass != CKO_PUBLIC_KEY) {
       TRACE_ERROR("This operation requires a public key.\n");
       return CKR_KEY_FUNCTION_NOT_PERMITTED;
   }

    /* check for token specific call first */
   if (token_specific.t_rsa_x509_verify == NULL) {
      TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
      return CKR_MECHANISM_INVALID;
   }
   // verify is a public key operation --> encrypt
   //
   rc = token_specific.t_rsa_x509_verify(in_data, in_data_len, signature, sig_len, key_obj);
   if (rc != CKR_OK)
      TRACE_ERROR("Token Specific rsa x509 verify failed.\n");

   return rc;
}


//
//
CK_RV
rsa_x509_verify_recover( SESSION             * sess,
                         CK_BBOOL              length_only,
                         SIGN_VERIFY_CONTEXT * ctx,
                         CK_BYTE             * signature,
                         CK_ULONG              sig_len,
                         CK_BYTE             * out_data,
                         CK_ULONG            * out_data_len )
{
   OBJECT          *key_obj  = NULL;
   CK_ULONG         modulus_bytes;
   CK_OBJECT_CLASS  keyclass;
   CK_RV            rc;


   if (!sess || !ctx || !out_data_len){
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   rc = object_mgr_find_in_map1( ctx->key, &key_obj );
   if (rc != CKR_OK){
      TRACE_ERROR("Failed to acquire key from specified handle");
      if (rc == CKR_OBJECT_HANDLE_INVALID)
	 return CKR_KEY_HANDLE_INVALID;
      else
         return rc;
   }

   rc = rsa_get_key_info(key_obj, &modulus_bytes, &keyclass);
   if (rc != CKR_OK) {
      TRACE_DEVEL("rsa_get_key_info failed.\n");
      return rc;
   }

   // check input data length restrictions
   //
   if (sig_len != modulus_bytes){
      TRACE_ERROR("%s\n", ock_err(ERR_SIGNATURE_LEN_RANGE));
      return CKR_SIGNATURE_LEN_RANGE;
   }
   if (length_only == TRUE) {
      *out_data_len = modulus_bytes;
      return CKR_OK;
   }

   // we perform no stripping of prepended zero bytes here
   //
   if (*out_data_len < modulus_bytes) {
      *out_data_len = modulus_bytes;
      TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
      return CKR_BUFFER_TOO_SMALL;
   }

    /* this had better be a public key */
   if (keyclass != CKO_PUBLIC_KEY) {
      TRACE_ERROR("This operation requires a public key.\n");
      return CKR_KEY_FUNCTION_NOT_PERMITTED;
   }

    /* check for token specific call first */
   if (token_specific.t_rsa_x509_verify_recover == NULL) {
      TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
      return CKR_MECHANISM_INVALID;
   }

   // verify is a public key operation --> encrypt
   //
   rc = token_specific.t_rsa_x509_verify_recover(signature, sig_len, out_data, out_data_len, key_obj);
   if (rc != CKR_OK)
      TRACE_ERROR("Token Specific rsa x509 verify recover.\n");

   return rc;
}

CK_RV rsa_pss_sign(SESSION *sess, CK_BBOOL length_only,
		   SIGN_VERIFY_CONTEXT *ctx, CK_BYTE *in_data,
		   CK_ULONG in_data_len, CK_BYTE *out_data,
		   CK_ULONG *out_data_len)
{
	CK_RV rc;
	OBJECT *key_obj = NULL;
	CK_ULONG modulus_bytes, hlen;
	CK_OBJECT_CLASS keyclass;
	CK_RSA_PKCS_PSS_PARAMS_PTR pssParms = NULL;

	if (!sess || !ctx || !out_data_len) {
		TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
		return CKR_FUNCTION_FAILED;
	}

	rc = object_mgr_find_in_map1(ctx->key, &key_obj);
	if (rc != CKR_OK) {
		TRACE_ERROR("Failed to acquire key from specified handle");
		if (rc == CKR_OBJECT_HANDLE_INVALID)
			return CKR_KEY_HANDLE_INVALID;
		else
			return rc;
	}

	/* get modulus and key class */
	rc = rsa_get_key_info(key_obj, &modulus_bytes, &keyclass);
	if (rc != CKR_OK) {
		TRACE_DEVEL("rsa_get_key_info failed.\n");
		return rc;
	}

	if (length_only == TRUE) {
		*out_data_len = modulus_bytes;
		return CKR_OK;
	}

        /* verify hashAlg now as well as get hash size. */
	pssParms = (CK_RSA_PKCS_PSS_PARAMS_PTR)ctx->mech.pParameter;
        hlen = 0;
        rc = get_sha_size(pssParms->hashAlg, &hlen);
        if (rc != CKR_OK) {
                TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_PARAM_INVALID));
                return CKR_MECHANISM_PARAM_INVALID;
        }

	/* pkcs#11v2.2, 12.1.10 states that this mechanism does not
	 * compute a hash value on the message to be signed.
	 * It assumes the input data is the hashed message.
	 */
	if (in_data_len != hlen) {
		TRACE_ERROR("%s\n", ock_err(CKR_DATA_LEN_RANGE));
		return CKR_DATA_LEN_RANGE;
	}

	if (*out_data_len < modulus_bytes) {
		*out_data_len = modulus_bytes;
		TRACE_ERROR("%s\n", ock_err(ERR_BUFFER_TOO_SMALL));
		return CKR_BUFFER_TOO_SMALL;
	}

	/* this had better be a private key */
	if (keyclass != CKO_PRIVATE_KEY) {
		TRACE_ERROR("This operation requires a private key.\n");
		return CKR_KEY_FUNCTION_NOT_PERMITTED;
	}

	if (token_specific.t_rsa_pss_sign == NULL) {
		TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
		return CKR_MECHANISM_INVALID;
	}

	rc = token_specific.t_rsa_pss_sign(ctx, in_data, in_data_len, out_data,
					   out_data_len);
	if (rc != CKR_OK)
		TRACE_DEVEL("Token Specific rsa pss sign failed.\n");

	return rc;
}

CK_RV rsa_pss_verify(SESSION *sess, SIGN_VERIFY_CONTEXT *ctx, CK_BYTE *in_data,
		     CK_ULONG in_data_len, CK_BYTE *signature, CK_ULONG sig_len)
{
	CK_RV rc;
	OBJECT *key_obj = NULL;
	CK_ULONG modulus_bytes;
	CK_OBJECT_CLASS keyclass;

	rc = object_mgr_find_in_map1(ctx->key, &key_obj);
	if (rc != CKR_OK) {
		TRACE_ERROR("Failed to acquire key from specified handle");
		if (rc == CKR_OBJECT_HANDLE_INVALID)
			return CKR_KEY_HANDLE_INVALID;
		else
			return rc;
	}

	/* get modulus and key class */
	rc = rsa_get_key_info(key_obj, &modulus_bytes, &keyclass);
	if (rc != CKR_OK) {
		TRACE_DEVEL("rsa_get_key_info failed.\n");
		return rc;
	}

	/* check input data length restrictions */
	if (sig_len != modulus_bytes) {
		TRACE_ERROR("%s\n", ock_err(ERR_SIGNATURE_LEN_RANGE));
		return CKR_SIGNATURE_LEN_RANGE;
	}

	/* this had better be a public key */
	if (keyclass != CKO_PUBLIC_KEY) {
		TRACE_ERROR("This operation requires a public key.\n");
		return CKR_KEY_FUNCTION_NOT_PERMITTED;
	}

	if (token_specific.t_rsa_pss_verify == NULL) {
		TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
		return CKR_MECHANISM_INVALID;
	}

	rc = token_specific.t_rsa_pss_verify(ctx, in_data, in_data_len,
					     signature, sig_len);
	if (rc != CKR_OK)
		TRACE_ERROR("Token Specific rsa pss verify.\n");

	return rc;
}

CK_RV rsa_hash_pss_sign(SESSION *sess, CK_BBOOL length_only,
			SIGN_VERIFY_CONTEXT *ctx, CK_BYTE *in_data,
			CK_ULONG in_data_len, CK_BYTE *sig, CK_ULONG *sig_len)
{
	CK_ULONG hlen;
	CK_BYTE hash[MAX_SHA_HASH_SIZE];
	DIGEST_CONTEXT digest_ctx;
	SIGN_VERIFY_CONTEXT sign_ctx;
	CK_MECHANISM digest_mech, sign_mech;
	CK_RV rc;

	if (!sess || !ctx || !in_data) {
		TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
		return CKR_FUNCTION_FAILED;
	}

	memset(&digest_ctx, 0x0, sizeof(digest_ctx));
	memset(&sign_ctx, 0x0, sizeof(sign_ctx));

	switch (ctx->mech.mechanism) {
	case CKM_SHA1_RSA_PKCS_PSS:
		digest_mech.mechanism = CKM_SHA_1;
		break;
	case CKM_SHA256_RSA_PKCS_PSS:
		digest_mech.mechanism = CKM_SHA256;
		break;
	case CKM_SHA384_RSA_PKCS_PSS:
		digest_mech.mechanism = CKM_SHA384;
		break;
	case CKM_SHA512_RSA_PKCS_PSS:
		digest_mech.mechanism = CKM_SHA512;
		break;
	default:
		return CKR_MECHANISM_INVALID;
	}

	digest_mech.ulParameterLen = 0;
	digest_mech.pParameter = NULL;

        rc = get_sha_size(digest_mech.mechanism, &hlen);
        if (rc != CKR_OK) {
                TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_PARAM_INVALID));
                return CKR_MECHANISM_PARAM_INVALID;
        }

	rc = digest_mgr_init(sess, &digest_ctx, &digest_mech);
	if (rc != CKR_OK) {
		TRACE_DEVEL("Digest Mgr Init failed.\n");
		return rc;
	}

	rc = digest_mgr_digest(sess, length_only, &digest_ctx, in_data,
			       in_data_len, hash, &hlen);
	if (rc != CKR_OK) {
		TRACE_DEVEL("Digest Mgr Digest failed.\n");
		return rc;
	}

	/* sign the hash */
	sign_mech.mechanism = CKM_RSA_PKCS_PSS;
	sign_mech.ulParameterLen = ctx->mech.ulParameterLen;
	sign_mech.pParameter = ctx->mech.pParameter;

	rc = sign_mgr_init(sess, &sign_ctx, &sign_mech, FALSE, ctx->key);
	if (rc != CKR_OK) {
		TRACE_DEVEL("Sign Mgr Init failed.\n");
		goto done;
	}

	rc = sign_mgr_sign(sess, length_only, &sign_ctx, hash, hlen,
			   sig, sig_len);
	if (rc != CKR_OK)
		TRACE_DEVEL("Sign Mgr Sign failed.\n");

done:
	sign_mgr_cleanup(&sign_ctx);
	return rc;
}

CK_RV rsa_hash_pss_update(SESSION *sess, SIGN_VERIFY_CONTEXT *ctx,
			  CK_BYTE *in_data, CK_ULONG in_data_len)
{
	DIGEST_CONTEXT *digest_ctx = NULL;
	CK_MECHANISM digest_mech;
	CK_RV rc;

	if (!sess || !ctx) {
		TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
		return CKR_FUNCTION_FAILED;
	}

	/* see if digest has already been through init */
	digest_ctx = (DIGEST_CONTEXT *)ctx->context;
	if (digest_ctx->active == FALSE) {
		switch (ctx->mech.mechanism) {
		case CKM_SHA1_RSA_PKCS_PSS:
			digest_mech.mechanism = CKM_SHA_1;
			break;
		case CKM_SHA256_RSA_PKCS_PSS:
			digest_mech.mechanism = CKM_SHA256;
			break;
		case CKM_SHA384_RSA_PKCS_PSS:
			digest_mech.mechanism = CKM_SHA384;
			break;
		case CKM_SHA512_RSA_PKCS_PSS:
			digest_mech.mechanism = CKM_SHA512;
			break;
		default:
			return CKR_MECHANISM_INVALID;
		}

		digest_mech.ulParameterLen = 0;
		digest_mech.pParameter = NULL;

		rc = digest_mgr_init(sess, digest_ctx, &digest_mech);
		if (rc != CKR_OK) {
			TRACE_DEVEL("Digest Mgr Init failed.\n");
			return rc;
		}
	}

	rc = digest_mgr_digest_update(sess, digest_ctx, in_data, in_data_len);
	if (rc != CKR_OK)
		TRACE_DEVEL("Digest Mgr Update failed.\n");

	return rc;
}

CK_RV rsa_hash_pss_sign_final(SESSION *sess, CK_BBOOL length_only,
			      SIGN_VERIFY_CONTEXT *ctx, CK_BYTE *signature,
			      CK_ULONG *sig_len)
{
	CK_ULONG hlen;
	CK_BYTE hash[MAX_SHA_HASH_SIZE];
	DIGEST_CONTEXT *digest_ctx;
	SIGN_VERIFY_CONTEXT sign_ctx;
	CK_MECHANISM sign_mech;
	CK_RV rc;

	if (!sess || !ctx || !sig_len) {
		TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
		return CKR_FUNCTION_FAILED;
	}

	memset(&sign_ctx, 0x0, sizeof(sign_ctx));

	digest_ctx = (DIGEST_CONTEXT *)ctx->context;

        rc = get_sha_size(digest_ctx->mech.mechanism, &hlen);
        if (rc != CKR_OK) {
                TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_PARAM_INVALID));
                return CKR_MECHANISM_PARAM_INVALID;
        }

	rc = digest_mgr_digest_final(sess, length_only, digest_ctx,
				     hash, &hlen);
	if (rc != CKR_OK) {
		TRACE_DEVEL("Digest Mgr Final failed.\n");
		return rc;
	}

	/* sign the hash */
	sign_mech.mechanism = CKM_RSA_PKCS_PSS;
	sign_mech.ulParameterLen = ctx->mech.ulParameterLen;
	sign_mech.pParameter = ctx->mech.pParameter;

	rc = sign_mgr_init(sess, &sign_ctx, &sign_mech, FALSE, ctx->key);
	if (rc != CKR_OK) {
		TRACE_DEVEL("Sign Mgr Init failed.\n");
		goto done;
	}

	rc = sign_mgr_sign(sess, length_only, &sign_ctx, hash, hlen,
			   signature, sig_len);
	if (rc != CKR_OK)
		TRACE_DEVEL("Sign Mgr Sign failed.\n");

done:
	sign_mgr_cleanup(&sign_ctx);
	return rc;
}

CK_RV rsa_hash_pss_verify(SESSION *sess, SIGN_VERIFY_CONTEXT *ctx,
			  CK_BYTE *in_data, CK_ULONG in_data_len,
			  CK_BYTE *signature, CK_ULONG sig_len)
{
	CK_ULONG hlen;
	CK_BYTE hash[MAX_SHA_HASH_SIZE];
	DIGEST_CONTEXT digest_ctx;
	SIGN_VERIFY_CONTEXT verify_ctx;
	CK_MECHANISM digest_mech, verify_mech;
	CK_RV rc;

	if (!sess || !ctx || !in_data) {
		TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
		return CKR_FUNCTION_FAILED;
	}
	memset(&digest_ctx, 0x0, sizeof(digest_ctx));
	memset(&verify_ctx, 0x0, sizeof(verify_ctx));

	switch (ctx->mech.mechanism) {
	case CKM_SHA1_RSA_PKCS_PSS:
		digest_mech.mechanism = CKM_SHA_1;
		break;
	case CKM_SHA256_RSA_PKCS_PSS:
		digest_mech.mechanism = CKM_SHA256;
		break;
	case CKM_SHA384_RSA_PKCS_PSS:
		digest_mech.mechanism = CKM_SHA384;
		break;
	case CKM_SHA512_RSA_PKCS_PSS:
		digest_mech.mechanism = CKM_SHA512;
		break;
	default:
		return CKR_MECHANISM_INVALID;
	}

	digest_mech.ulParameterLen = 0;
	digest_mech.pParameter = NULL;

	rc = get_sha_size(digest_mech.mechanism, &hlen);
	if (rc != CKR_OK) {
		TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_PARAM_INVALID));
		return CKR_MECHANISM_PARAM_INVALID;
	}

	rc = digest_mgr_init(sess, &digest_ctx, &digest_mech);
	if (rc != CKR_OK) {
		TRACE_DEVEL("Digest Mgr Init failed.\n");
		return rc;
	}

	rc = digest_mgr_digest(sess, FALSE, &digest_ctx, in_data,
			       in_data_len, hash, &hlen);
	if (rc != CKR_OK) {
		TRACE_DEVEL("Digest Mgr Digest failed.\n");
		return rc;
	}

	/* sign the hash */
	verify_mech.mechanism = CKM_RSA_PKCS_PSS;
	verify_mech.ulParameterLen = ctx->mech.ulParameterLen;
	verify_mech.pParameter = ctx->mech.pParameter;

	rc = verify_mgr_init(sess, &verify_ctx, &verify_mech, FALSE, ctx->key);
	if (rc != CKR_OK) {
		TRACE_DEVEL("Verify Mgr Init failed.\n");
		goto done;
	}

	rc = verify_mgr_verify(sess, &verify_ctx, hash, hlen, signature,
			       sig_len);
	if (rc != CKR_OK)
		TRACE_DEVEL("Verify Mgr Verify failed.\n");

done:
	verify_mgr_cleanup(&verify_ctx);
	return rc;
}

CK_RV rsa_hash_pss_verify_final(SESSION *sess, SIGN_VERIFY_CONTEXT *ctx,
				CK_BYTE *signature, CK_ULONG sig_len)
{
	CK_ULONG hlen;
	CK_BYTE hash[MAX_SHA_HASH_SIZE];
	DIGEST_CONTEXT *digest_ctx;
	SIGN_VERIFY_CONTEXT verify_ctx;
	CK_MECHANISM verify_mech;
	CK_RV rc;

	if (!sess || !ctx || !signature) {
		TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
		return CKR_FUNCTION_FAILED;
	}

	memset(&verify_ctx, 0x0, sizeof(verify_ctx));

	digest_ctx = (DIGEST_CONTEXT *)ctx->context;

	rc = get_sha_size(digest_ctx->mech.mechanism, &hlen);
	if (rc != CKR_OK) {
		TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_PARAM_INVALID));
		return CKR_MECHANISM_PARAM_INVALID;
	}

	rc = digest_mgr_digest_final(sess, FALSE, digest_ctx, hash, &hlen);
	if (rc != CKR_OK) {
		TRACE_DEVEL("Digest Mgr Final failed.\n");
		return rc;
	}

	/* sign the hash */
	verify_mech.mechanism = CKM_RSA_PKCS_PSS;
	verify_mech.ulParameterLen = ctx->mech.ulParameterLen;
	verify_mech.pParameter = ctx->mech.pParameter;

	rc = verify_mgr_init(sess, &verify_ctx, &verify_mech, FALSE, ctx->key);
	if (rc != CKR_OK) {
		TRACE_DEVEL("Verify Mgr Init failed.\n");
		goto done;
	}

	rc = verify_mgr_verify(sess, &verify_ctx, hash, hlen, signature,
			       sig_len);
	if (rc != CKR_OK)
		TRACE_DEVEL("Verify Mgr Verify failed.\n");

done:
	verify_mgr_cleanup(&verify_ctx);
	return rc;
}

//
//
CK_RV
rsa_hash_pkcs_sign( SESSION              * sess,
                    CK_BBOOL               length_only,
                    SIGN_VERIFY_CONTEXT  * ctx,
                    CK_BYTE              * in_data,
                    CK_ULONG               in_data_len,
                    CK_BYTE              * signature,
                    CK_ULONG             * sig_len )
{
   CK_BYTE            * ber_data  = NULL;
   CK_BYTE            * octet_str = NULL;
   CK_BYTE            * oid       = NULL;
   CK_BYTE            * tmp       = NULL;

   CK_ULONG             buf1[16];  // 64 bytes is more than enough

   CK_BYTE              hash[SHA5_HASH_SIZE];  // must be large enough for the largest hash
   DIGEST_CONTEXT       digest_ctx;
   SIGN_VERIFY_CONTEXT  sign_ctx;
   CK_MECHANISM         digest_mech;
   CK_MECHANISM         sign_mech;
   CK_ULONG             ber_data_len, hash_len, octet_str_len, oid_len;
   CK_RV                rc;

   if (!sess || !ctx || !in_data){
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   memset( &digest_ctx, 0x0, sizeof(digest_ctx) );
   memset( &sign_ctx,   0x0, sizeof(sign_ctx)   );

   if (ctx->mech.mechanism == CKM_MD2_RSA_PKCS) {
      digest_mech.mechanism      = CKM_MD2;
      oid = ber_AlgMd2;
      oid_len = ber_AlgMd2Len;

   }
   else if (ctx->mech.mechanism == CKM_MD5_RSA_PKCS) {
      digest_mech.mechanism      = CKM_MD5;
      oid = ber_AlgMd5;
      oid_len = ber_AlgMd5Len;
   }
   else if (ctx->mech.mechanism == CKM_SHA256_RSA_PKCS) {
      digest_mech.mechanism      = CKM_SHA256;
      oid = ber_AlgSha256;
      oid_len = ber_AlgSha256Len;
   }
   else if (ctx->mech.mechanism == CKM_SHA384_RSA_PKCS) {
      digest_mech.mechanism      = CKM_SHA384;
      oid = ber_AlgSha384;
      oid_len = ber_AlgSha384Len;
   }
   else if (ctx->mech.mechanism == CKM_SHA512_RSA_PKCS) {
      digest_mech.mechanism      = CKM_SHA512;
      oid = ber_AlgSha512;
      oid_len = ber_AlgSha512Len;
   }
   else {
      digest_mech.mechanism      = CKM_SHA_1;
      oid = ber_AlgSha1;
      oid_len = ber_AlgSha1Len;
   }

   digest_mech.ulParameterLen = 0;
   digest_mech.pParameter     = NULL;

   rc = digest_mgr_init( sess, &digest_ctx, &digest_mech );
   if (rc != CKR_OK){
      TRACE_DEVEL("Digest Mgr Init failed.\n");
      return rc;
   }
   hash_len = sizeof(hash);
   rc = digest_mgr_digest( sess, length_only, &digest_ctx, in_data, in_data_len, hash, &hash_len );
   if (rc != CKR_OK){
      TRACE_DEVEL("Digest Mgr Digest failed.\n");
      return rc;
   }
      // build the BER-encodings

    rc = ber_encode_OCTET_STRING( FALSE, &octet_str, &octet_str_len, hash, hash_len );
    if (rc != CKR_OK){
      TRACE_DEVEL("ber_encode_OCTET_STRING failed.\n");
       goto error;
    }
    tmp = (CK_BYTE *)buf1;
    memcpy( tmp,           oid,       oid_len );
    memcpy( tmp + oid_len, octet_str, octet_str_len);

    rc = ber_encode_SEQUENCE( FALSE, &ber_data, &ber_data_len, tmp, (oid_len + octet_str_len) );
    if (rc != CKR_OK){
       TRACE_DEVEL("ber_encode_SEQUENCE failed.\n");
       goto error;
    }
    // sign the BER-encoded data block
   

   sign_mech.mechanism      = CKM_RSA_PKCS;
   sign_mech.ulParameterLen = 0;
   sign_mech.pParameter     = NULL;

   rc = sign_mgr_init( sess, &sign_ctx, &sign_mech, FALSE, ctx->key );
   if (rc != CKR_OK){
      TRACE_DEVEL("Sign Mgr Init failed.\n");
      goto error;
   }
   rc = sign_mgr_sign( sess, length_only, &sign_ctx, ber_data, ber_data_len, signature, sig_len );
   if (rc != CKR_OK)
      TRACE_DEVEL("Sign Mgr Sign failed.\n");

error:
   if (octet_str) free( octet_str );
   if (ber_data)  free( ber_data );
   sign_mgr_cleanup( &sign_ctx );
   return rc;
}


//
//
CK_RV
rsa_hash_pkcs_sign_update( SESSION              * sess,
                           SIGN_VERIFY_CONTEXT  * ctx,
                           CK_BYTE              * in_data,
                           CK_ULONG               in_data_len )
{
   RSA_DIGEST_CONTEXT  * context = NULL;
   CK_MECHANISM          digest_mech;
   CK_RV                 rc;

   if (!sess || !ctx) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   context = (RSA_DIGEST_CONTEXT *)ctx->context;

   if (context->flag == FALSE) {
      if (ctx->mech.mechanism == CKM_MD2_RSA_PKCS)
         digest_mech.mechanism = CKM_MD2;
      else if (ctx->mech.mechanism == CKM_MD5_RSA_PKCS)
         digest_mech.mechanism = CKM_MD5;
      else if (ctx->mech.mechanism == CKM_SHA256_RSA_PKCS)
         digest_mech.mechanism = CKM_SHA256;
      else if (ctx->mech.mechanism == CKM_SHA384_RSA_PKCS)
         digest_mech.mechanism = CKM_SHA384;
      else if (ctx->mech.mechanism == CKM_SHA512_RSA_PKCS)
         digest_mech.mechanism = CKM_SHA512;
      else
         digest_mech.mechanism = CKM_SHA_1;

      digest_mech.ulParameterLen = 0;
      digest_mech.pParameter     = NULL;

      rc = digest_mgr_init( sess, &context->hash_context, &digest_mech );
      if (rc != CKR_OK){
         TRACE_DEVEL("Digest Mgr Init failed.\n");
         return rc;
      }
      context->flag = TRUE;
   }

   rc = digest_mgr_digest_update( sess, &context->hash_context, in_data, in_data_len );
   if (rc != CKR_OK){
      TRACE_DEVEL("Digest Mgr Digest failed.\n");
      return rc;
   }
   return CKR_OK;
}


//
//
CK_RV
rsa_hash_pkcs_verify( SESSION              * sess,
                      SIGN_VERIFY_CONTEXT  * ctx,
                      CK_BYTE              * in_data,
                      CK_ULONG               in_data_len,
                      CK_BYTE              * signature,
                      CK_ULONG               sig_len )
{
   CK_BYTE            * ber_data  = NULL;
   CK_BYTE            * octet_str = NULL;
   CK_BYTE            * oid       = NULL;
   CK_BYTE            * tmp       = NULL;

   CK_ULONG             buf1[16];  // 64 bytes is more than enough
   CK_BYTE              hash[SHA5_HASH_SIZE];
   DIGEST_CONTEXT       digest_ctx;
   SIGN_VERIFY_CONTEXT  verify_ctx;
   CK_MECHANISM         digest_mech;
   CK_MECHANISM         verify_mech;
   CK_ULONG             ber_data_len, hash_len, octet_str_len, oid_len;
   CK_RV                rc;

   if (!sess || !ctx || !in_data){
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   memset( &digest_ctx, 0x0, sizeof(digest_ctx) );
   memset( &verify_ctx, 0x0, sizeof(verify_ctx) );

   if (ctx->mech.mechanism == CKM_MD2_RSA_PKCS) {
      digest_mech.mechanism      = CKM_MD2;
      oid = ber_AlgMd2;
      oid_len = ber_AlgMd2Len;
   }
   else if (ctx->mech.mechanism == CKM_MD5_RSA_PKCS) {
      digest_mech.mechanism      = CKM_MD5;
      oid = ber_AlgMd5;
      oid_len = ber_AlgMd5Len;
   }
   else if (ctx->mech.mechanism == CKM_SHA256_RSA_PKCS) {
      digest_mech.mechanism      = CKM_SHA256;
      oid = ber_AlgSha256;
      oid_len = ber_AlgSha256Len;
   }
   else if (ctx->mech.mechanism == CKM_SHA384_RSA_PKCS) {
      digest_mech.mechanism      = CKM_SHA384;
      oid = ber_AlgSha384;
      oid_len = ber_AlgSha384Len;
   }
   else if (ctx->mech.mechanism == CKM_SHA512_RSA_PKCS) {
      digest_mech.mechanism      = CKM_SHA512;
      oid = ber_AlgSha512;
      oid_len = ber_AlgSha512Len;
   }
   else {
      digest_mech.mechanism      = CKM_SHA_1;
      oid = ber_AlgSha1;
      oid_len = ber_AlgSha1Len;
   }


   digest_mech.ulParameterLen = 0;
   digest_mech.pParameter     = NULL;

   rc = digest_mgr_init( sess, &digest_ctx, &digest_mech );
   if (rc != CKR_OK){
      TRACE_DEVEL("Digest Mgr Init failed.\n");
      return rc;
   }
   hash_len = sizeof(hash);
   rc = digest_mgr_digest( sess, FALSE, &digest_ctx, in_data, in_data_len, hash, &hash_len );
   if (rc != CKR_OK){
      TRACE_DEVEL("Digest Mgr Digest failed.\n");
      return rc;
   }

   // Build the BER encoding
   //
   rc = ber_encode_OCTET_STRING( FALSE, &octet_str, &octet_str_len, hash, hash_len );
   if (rc != CKR_OK){
      TRACE_DEVEL("ber_encode_OCTET_STRING failed.\n");
      goto done;
   }
   tmp = (CK_BYTE *)buf1;
   memcpy( tmp,           oid,       oid_len );
   memcpy( tmp + oid_len, octet_str, octet_str_len );

   rc = ber_encode_SEQUENCE( FALSE, &ber_data, &ber_data_len, tmp, (oid_len + octet_str_len) );
   if (rc != CKR_OK){
      TRACE_DEVEL("ber_encode_SEQUENCE failed.\n");
      goto done;
   }
   // Verify the Signed BER-encoded Data block
   //
   verify_mech.mechanism      = CKM_RSA_PKCS;
   verify_mech.ulParameterLen = 0;
   verify_mech.pParameter     = NULL;

   rc = verify_mgr_init( sess, &verify_ctx, &verify_mech, FALSE, ctx->key );
   if (rc != CKR_OK){
      TRACE_DEVEL("Verify Mgr Init failed.\n");
      goto done;
   }
   rc = verify_mgr_verify( sess, &verify_ctx, ber_data, ber_data_len, signature, sig_len );
   if (rc != CKR_OK)
      TRACE_DEVEL("Verify Mgr Verify failed.\n");
done:
   if (octet_str) free( octet_str );
   if (ber_data)  free( ber_data );
   sign_mgr_cleanup( &verify_ctx );
   return rc;
}

//
//
CK_RV
rsa_hash_pkcs_verify_update( SESSION              * sess,
                             SIGN_VERIFY_CONTEXT  * ctx,
                             CK_BYTE              * in_data,
                             CK_ULONG               in_data_len )
{
   RSA_DIGEST_CONTEXT  * context = NULL;
   CK_MECHANISM          digest_mech;
   CK_RV                 rc;

   if (!sess || !ctx) {
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   context = (RSA_DIGEST_CONTEXT *)ctx->context;

   if (context->flag == FALSE) {
      if (ctx->mech.mechanism == CKM_MD2_RSA_PKCS)
         digest_mech.mechanism = CKM_MD2;
      else if (ctx->mech.mechanism == CKM_MD5_RSA_PKCS)
         digest_mech.mechanism = CKM_MD5;
      else if (ctx->mech.mechanism == CKM_SHA256_RSA_PKCS)
         digest_mech.mechanism = CKM_SHA256;
      else if (ctx->mech.mechanism == CKM_SHA384_RSA_PKCS)
         digest_mech.mechanism = CKM_SHA384;
      else if (ctx->mech.mechanism == CKM_SHA512_RSA_PKCS)
         digest_mech.mechanism = CKM_SHA512;
      else
         digest_mech.mechanism = CKM_SHA_1;

      digest_mech.ulParameterLen = 0;
      digest_mech.pParameter     = NULL;

      rc = digest_mgr_init( sess, &context->hash_context, &digest_mech );
      if (rc != CKR_OK){
         TRACE_DEVEL("Digest Mgr Init failed.\n");
         return rc;
      }
      context->flag = TRUE;
   }

   rc = digest_mgr_digest_update( sess, &context->hash_context, in_data, in_data_len );
   if (rc != CKR_OK){
      TRACE_DEVEL("Digest Mgr Update failed.\n");
      return rc;
   }
   return CKR_OK;
}


//
//
CK_RV
rsa_hash_pkcs_sign_final( SESSION              * sess,
                          CK_BBOOL               length_only,
                          SIGN_VERIFY_CONTEXT  * ctx,
                          CK_BYTE              * signature,
                          CK_ULONG             * sig_len )
{
   CK_BYTE            * ber_data  = NULL;
   CK_BYTE            * octet_str = NULL;
   CK_BYTE            * oid       = NULL;
   CK_BYTE            * tmp       = NULL;

   CK_ULONG              buf1[16];  // 64 bytes is more than enough

   CK_BYTE               hash[SHA5_HASH_SIZE];
   RSA_DIGEST_CONTEXT  * context = NULL;
   CK_ULONG              ber_data_len, hash_len, octet_str_len, oid_len;
   CK_MECHANISM          sign_mech;
   SIGN_VERIFY_CONTEXT   sign_ctx;
   CK_RV                 rc;

   if (!sess || !ctx || !sig_len){
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }

   if (ctx->mech.mechanism == CKM_MD2_RSA_PKCS) {
      oid = ber_AlgMd2;
      oid_len = ber_AlgMd2Len;
   }
   else if (ctx->mech.mechanism == CKM_MD5_RSA_PKCS) {
      oid = ber_AlgMd5;
      oid_len = ber_AlgMd5Len;
   }
   else if (ctx->mech.mechanism == CKM_SHA256_RSA_PKCS) {
      oid = ber_AlgSha256;
      oid_len = ber_AlgSha256Len;
   }
   else if (ctx->mech.mechanism == CKM_SHA384_RSA_PKCS) {
      oid = ber_AlgSha384;
      oid_len = ber_AlgSha384Len;
   }
   else if (ctx->mech.mechanism == CKM_SHA512_RSA_PKCS) {
      oid = ber_AlgSha512;
      oid_len = ber_AlgSha512Len;
   }
   else {
      oid = ber_AlgSha1;
      oid_len = ber_AlgSha1Len;
   }

   memset( &sign_ctx, 0x0, sizeof(sign_ctx));

   context = (RSA_DIGEST_CONTEXT *)ctx->context;

   hash_len = sizeof(hash);
   rc = digest_mgr_digest_final( sess, length_only, &context->hash_context, hash, &hash_len );
   if (rc != CKR_OK){
      TRACE_DEVEL("Digest Mgr Final failed.\n");
      return rc;
   }
   // Build the BER Encoded Data block
   //
   rc = ber_encode_OCTET_STRING( FALSE, &octet_str, &octet_str_len, hash, hash_len );
   if (rc != CKR_OK){
      TRACE_DEVEL("ber_encode_OCTET_STRING failed.\n");
      return rc;
   }
   tmp = (CK_BYTE *)buf1;
   memcpy( tmp,           oid,       oid_len );
   memcpy( tmp + oid_len, octet_str, octet_str_len );

   rc = ber_encode_SEQUENCE( FALSE, &ber_data, &ber_data_len, tmp, (oid_len + octet_str_len) );
   if (rc != CKR_OK){
      TRACE_DEVEL("ber_encode_SEQUENCE failed.\n");
      goto done;
   }
   // sign the BER-encoded data block
   //   

   sign_mech.mechanism      = CKM_RSA_PKCS;
   sign_mech.ulParameterLen = 0;
   sign_mech.pParameter     = NULL;

   rc = sign_mgr_init( sess, &sign_ctx, &sign_mech, FALSE, ctx->key );
   if (rc != CKR_OK){
      TRACE_DEVEL("Sign Mgr Init failed.\n");
      goto done;
   }
   rc = sign_mgr_sign( sess, length_only, &sign_ctx, ber_data, ber_data_len, signature, sig_len );
   if (rc != CKR_OK)
      TRACE_DEVEL("Sign Mgr Sign failed.\n");

   /** Not sure why this check is here */
   if (length_only == TRUE || rc == CKR_BUFFER_TOO_SMALL)
	goto done;

done:
   if (octet_str) free( octet_str );
   if (ber_data)  free( ber_data );
   sign_mgr_cleanup( &sign_ctx );
   return rc;
}


//
//
CK_RV
rsa_hash_pkcs_verify_final( SESSION              * sess,
                            SIGN_VERIFY_CONTEXT  * ctx,
                            CK_BYTE              * signature,
                            CK_ULONG               sig_len )
{
   CK_BYTE            * ber_data  = NULL;
   CK_BYTE            * octet_str = NULL;
   CK_BYTE            * oid       = NULL;
   CK_BYTE            * tmp       = NULL;

   CK_ULONG             buf1[16];   // 64 bytes is more than enough
   CK_BYTE               hash[SHA5_HASH_SIZE];
   RSA_DIGEST_CONTEXT  * context = NULL;
   CK_ULONG              ber_data_len, hash_len, octet_str_len, oid_len;
   CK_MECHANISM          verify_mech;
   SIGN_VERIFY_CONTEXT   verify_ctx;
   CK_RV                 rc;

   if (!sess || !ctx || !signature){
      TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
      return CKR_FUNCTION_FAILED;
   }
   if (ctx->mech.mechanism == CKM_MD2_RSA_PKCS) {
      oid = ber_AlgMd2;
      oid_len = ber_AlgMd2Len;
   }
   else if (ctx->mech.mechanism == CKM_MD5_RSA_PKCS) {
      oid = ber_AlgMd5;
      oid_len = ber_AlgMd5Len;
   }
   else if (ctx->mech.mechanism == CKM_SHA256_RSA_PKCS) {
      oid = ber_AlgSha256;
      oid_len = ber_AlgSha256Len;
   }
   else if (ctx->mech.mechanism == CKM_SHA384_RSA_PKCS) {
      oid = ber_AlgSha384;
      oid_len = ber_AlgSha384Len;
   }
   else if (ctx->mech.mechanism == CKM_SHA512_RSA_PKCS) {
      oid = ber_AlgSha512;
      oid_len = ber_AlgSha512Len;
   }
   else {
      oid = ber_AlgSha1;
      oid_len = ber_AlgSha1Len;
   }

   memset( &verify_ctx, 0x0, sizeof(verify_ctx));

   context = (RSA_DIGEST_CONTEXT *)ctx->context;

   hash_len = sizeof(hash);
   rc = digest_mgr_digest_final( sess, FALSE, &context->hash_context, hash, &hash_len );
   if (rc != CKR_OK){
      TRACE_DEVEL("Digest Mgr Final failed.\n");
      return rc;
   }
   // Build the BER encoding
   //
   rc = ber_encode_OCTET_STRING( FALSE, &octet_str, &octet_str_len, hash, hash_len );
   if (rc != CKR_OK){
      TRACE_DEVEL("ber_encode_OCTET_STRING failed.\n");
      goto done;
   }
   tmp = (CK_BYTE *)buf1;
   memcpy( tmp,           oid,       oid_len );
   memcpy( tmp + oid_len, octet_str, octet_str_len );

   rc = ber_encode_SEQUENCE( FALSE, &ber_data, &ber_data_len, tmp, (oid_len + octet_str_len) );
   if (rc != CKR_OK){
      TRACE_DEVEL("ber_encode_SEQUENCE failed.\n");
      goto done;
   }
   // verify the signed BER-encoded data block
   //

   verify_mech.mechanism      = CKM_RSA_PKCS;
   verify_mech.ulParameterLen = 0;
   verify_mech.pParameter     = NULL;

   rc = verify_mgr_init( sess, &verify_ctx, &verify_mech, FALSE, ctx->key );
   if (rc != CKR_OK){
      TRACE_DEVEL("Verify Mgr Init failed.\n");
      goto done;
   }
   rc = verify_mgr_verify( sess, &verify_ctx, ber_data, ber_data_len, signature, sig_len );
   if (rc != CKR_OK)
      TRACE_DEVEL("Verify Mgr Verify failed.\n");
done:
   if (octet_str) free( octet_str );
   if (ber_data)  free( ber_data );
   verify_mgr_cleanup( &verify_ctx );
   return rc;
}


//
// mechanisms
//



//
//
CK_RV
ckm_rsa_key_pair_gen( TEMPLATE  * publ_tmpl,
                      TEMPLATE  * priv_tmpl )
{
   CK_RV                rc;

   /* check for token specific call first */
   if (token_specific.t_rsa_generate_keypair == NULL) {
      TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
      return CKR_MECHANISM_INVALID;
   }

   rc = token_specific.t_rsa_generate_keypair(publ_tmpl, priv_tmpl);
   if (rc != CKR_OK)
      TRACE_DEVEL("Token specific rsa generate keypair failed.\n");

   return rc;
}

CK_RV mgf1(CK_BYTE *seed, CK_ULONG seedlen, CK_BYTE *mask, CK_ULONG maskLen,
	   CK_RSA_PKCS_MGF_TYPE mgf)
{

	int i, T_len = 0;
	char *seed_buffer;
	unsigned char counter[4];
	CK_BYTE hash[MAX_SHA_HASH_SIZE];
	CK_RV rc = CKR_OK;
	CK_MECHANISM_TYPE mech;
	CK_ULONG hlen;

	if (!mask || !seed)
		return CKR_FUNCTION_FAILED;

	rc = get_mgf_mech(mgf, &mech);
	if (rc != CKR_OK)
		return CKR_FUNCTION_FAILED;

	rc = get_sha_size(mech, &hlen);
	if (rc != CKR_OK)
		return CKR_FUNCTION_FAILED;

	/* do some preparations */
	seed_buffer = malloc(seedlen + 4);
	if (seed_buffer == NULL)
		return CKR_HOST_MEMORY;

	T_len = maskLen;
	for (i = 0; T_len > 0; i++) {
		/* convert i to an octet string of length 4 octets. */
		counter[0] = (unsigned char)((i >> 24) & 0xff);
		counter[1] = (unsigned char)((i >> 16) & 0xff);
		counter[2] = (unsigned char)((i >> 8) & 0xff);
		counter[3] = (unsigned char)(i & 0xff);

		/* concatenate seed and octet string */
		memset(seed_buffer, 0, seedlen + 4);
		memcpy(seed_buffer, seed, seedlen);
		memcpy(seed_buffer + seedlen, counter, 4);

		/* compute hash of concatenated seed and octet string */
		rc = compute_sha(seed_buffer, seedlen + 4, hash, mech);
		if (rc != CKR_OK)
			goto done;

		if (T_len >= hlen)
			memcpy(mask + (i * hlen), hash, hlen);
		else
			/* in the case masklen is not a multiple of the
			 * of the hash length, only copy over remainder
			 */
			memcpy(mask + (i * hlen), hash, T_len);

		T_len -= hlen;
	}

done:
	if (seed_buffer)
		free(seed_buffer);
	return rc;
}

// RSA mechanism - EME-OAEP encoding
//
CK_RV encode_eme_oaep(CK_BYTE *mData, CK_ULONG mLen, CK_BYTE *emData,
                      CK_ULONG modLength, CK_RSA_PKCS_MGF_TYPE mgf,
                      CK_BYTE *hash, CK_ULONG hlen)
{
        int i, ps_len, dbMask_len;
        CK_BYTE *maskedSeed, *maskedDB, *dbMask;
        CK_BYTE seed[MAX_SHA_HASH_SIZE];
        CK_RV rc = CKR_OK;

        if (!mData || !emData) {
		TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
                return CKR_FUNCTION_FAILED;
        }

	/* pkcs1v2.2 Step i:
	 * The encoded messages is a concatenated single octet, 0x00 with
	 * maskedSeed and maskedDB to create encoded message EM.
	 * So lets mark of the places in our output buffer.
	 */
	memset(emData, 0, modLength);
	maskedSeed = emData + 1;
	maskedDB = emData + hlen + 1;

	/* pkcs1v2.2, Step b:
	 * Generate an octet string PS and concatenate to DB.
	 */
        ps_len = modLength - mLen - (2 * hlen) - 2;
        memcpy(maskedDB, hash, hlen);
        memset(maskedDB + hlen, 0, ps_len);

	/* pkcs1v2.2, Step c:
	 * We have already concatenated hash and PS to maskedDB.
	 * Now just concatenate 0x01 and message.
	 */
	maskedDB[hlen + ps_len] = 0x01;
	memcpy(maskedDB + (hlen + ps_len + 1), mData, mLen);

	/* pkcs1v2.2, Step d:
	 * Generate a random seed.
	 */
	rc = rng_generate(seed, hlen);
	if (rc != CKR_OK)
		return rc;
	
	/* pkcs1v2.2, Step e:
	 * Compute dbmask using MGF1.
	 */
	dbMask_len = modLength - hlen - 1;
	dbMask = malloc(sizeof(CK_BYTE) * dbMask_len);
	if (dbMask == NULL) {
		TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
		return CKR_HOST_MEMORY;
	}

	rc = mgf1(seed, hlen, dbMask, dbMask_len, mgf);
	if (rc != CKR_OK)
		goto done;

	/* pkcs1v2.2, Step f:
	 * Compute maskedDB.
	 */
	for (i = 0; i < dbMask_len; i++)
		maskedDB[i] ^= dbMask[i];

	/* pkcs1v2.2, Step g:
	 * Compute seedMask using MGF1.
	 */
	memset(maskedSeed, 0, hlen);
	rc = mgf1(maskedDB, dbMask_len, maskedSeed, hlen, mgf);
	if (rc != CKR_OK)
		goto done;

	/* pkcs1v2.2, Step h:
	 * Compute maskedSeed.
	 */
	for (i = 0; i < hlen; i++)
		maskedSeed[i] ^= seed[i];
done:
	if (dbMask)
		free(dbMask);
	return rc;
}

CK_RV decode_eme_oaep(CK_BYTE *emData, CK_ULONG emLen, CK_BYTE *out_data,
		      CK_ULONG *out_data_len, CK_RSA_PKCS_MGF_TYPE mgf,
		      CK_BYTE *hash, CK_ULONG hlen)
{
	int i, error = 0;;
	CK_RV rc = CKR_OK;
	CK_ULONG dbMask_len, ps_len;
	CK_BYTE *maskedSeed, *maskedDB, *dbMask, *seedMask;

	if (!emData || !out_data) {
		TRACE_ERROR("%s received bad argument(s)\n", __FUNCTION__);
                return CKR_FUNCTION_FAILED;
	}

	/* allocate memory now for later use */
	dbMask_len = *out_data_len - hlen - 1;
	dbMask = malloc(sizeof(CK_BYTE) * dbMask_len);
	seedMask = malloc(sizeof(CK_BYTE) * hlen);
	if ((seedMask == NULL) || (dbMask == NULL)) {
		TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
		rc = CKR_HOST_MEMORY;
		goto done;
	}

        /* pkcs1v2.2, section 7.1.2, Step 3b:
	 * Separate the encoded message EM and process the decrypted message.
	 *
	 * To mitigate fault and timing attacks, just flag errors and
	 * keep going.
	 */
	maskedSeed = emData + 1;
	maskedDB = emData + hlen + 1;

	/* pkcs1v2.2, section 7.1.2, Step 3c:
	 * Compute seedMask using MGF1.
	 */
	if (mgf1(maskedDB, dbMask_len, seedMask, hlen, mgf))
		error++;
	
	/* pkcs1v2.2, section 7.1.2, Step 3d:
	 * Compute seed using MGF1.
	 */
	for (i = 0; i < hlen; i++)
		seedMask[i] ^= maskedSeed[i];

	/* pkcs1v2.2, section 7.1.2, Step 3e:
	 * Compute dbMask using MGF1.
	 */
	if (mgf1(seedMask, hlen, dbMask, dbMask_len, mgf))
		error++;

	/* pkcs1v2.2, section 7.1.2, Step 3f:
	 * Compute db using MGF1.
	 */
	for (i = 0; i < dbMask_len; i++)
		dbMask[i] ^= maskedDB[i];
	
	/* pkcs1v2.2, section 7.1.2, Step 3g:
	 * DB = lHash’ || PS || 0x01 || M .
	 *
	 * If there is no octet with hexadecimal value 0x01 to separate
	 * PS from M, if lHash does not equal lHash’, output “decryption
	 * error” and stop.
	 */
	if (memcmp(dbMask, hash, hlen))
		error++;
	
	ps_len = hlen;
	while ((dbMask[ps_len] == 0x00) && (ps_len < dbMask_len))
		ps_len++;

	if ((ps_len == dbMask_len) || (dbMask[ps_len] != 0x01) || emData[0])
		error++;

	if (error) {
		rc = CKR_FUNCTION_FAILED;
		goto done;
	} else {
		ps_len++;
		*out_data_len = dbMask_len - ps_len;
		memcpy(out_data, dbMask + ps_len, dbMask_len - ps_len);
	}

done:
	if (seedMask)
		free(seedMask);
	if (dbMask)
		free(dbMask);
	return rc;
}

CK_RV emsa_pss_encode(CK_RSA_PKCS_PSS_PARAMS *pssParms, CK_BYTE *in_data,
		      CK_ULONG in_data_len, CK_BYTE *em, CK_ULONG *modbytes)
{
	CK_BYTE *salt, *DB, *H, *buf = NULL;
	CK_ULONG emBits, emLen, buflen, hlen, PSlen;
	CK_RV rc = CKR_OK;
	int i;

	/*
	 * Note: pkcs#11v2.20, Section 12.1.10:
	 * in_data is the hashed message, mHash.
	 *
	 * Note: em is provided by the caller. It should be big enough to
	 * hold k bytes of data, where k is the length in octets of the
	 * modulus n.
	 */

	/* pkcs1v2.2 8.1.1 describes emBits as length in bits of the
	 * modulus - 1. It also says, the octet length of EM will be
	 * one less than k if modBits - 1 is divisible by 8 and equal
	 * to k otherwise. k is the length in octets of the modulus n.
	 */
	emBits = (*modbytes * 8) - 1;
	if ((emBits % 8) == 0)
		emLen = *modbytes - 1;
	else
		emLen = *modbytes;
	
	/* get hash size based on hashAlg */
	if (get_sha_size(pssParms->hashAlg, &hlen))
		return CKR_MECHANISM_INVALID;

	/* allocate a helper buffer to be used for M' and dbmask */
	buflen = emLen - hlen - 1;
	if (buflen < (8 + hlen + pssParms->sLen))
		buflen = 8 + hlen + pssParms->sLen;
	buf = (CK_BYTE *)malloc(buflen);
	if (buf == NULL)
		return CKR_HOST_MEMORY;

	memset(em, 0, emLen);
	memset(buf, 0, buflen);

	/* set some pointers for EM */
	DB = em;
	H = em + (emLen - hlen - 1);

	/* pkcs1v2.2, Step 3: Check length */
	if (emLen < hlen + pssParms->sLen + 2) {
		rc = CKR_FUNCTION_FAILED;
		goto done;
	}

	/* pkcs1v2.2, Step 4: Generate salt */
	salt = buf + (8 + in_data_len);
	rc = rng_generate(salt, pssParms->sLen);
	if (rc != CKR_OK)
		goto done;

	/* pkcs1v2.2, Step 5: set M' */
	memcpy(buf + 8, in_data, in_data_len);

	/* pkcs1v2.2, Step 6: Compute Hash(M') */
	rc = compute_sha(buf, 8 + hlen + pssParms->sLen, H, pssParms->hashAlg);
	if (rc != CKR_OK)
		goto done;

	/* pkcs1v2.2, Step 7 & 8: Generate DB */
	PSlen = emLen - pssParms->sLen - hlen - 2;
	DB[PSlen] = 0x01;
	memcpy(DB + (PSlen + 1), salt, pssParms->sLen);

	/* pkcs1v2.2, Step 9: Generate dbMask
	 * Note: reuse "buf" for dbMask.
	 */
	memset(buf, 0, buflen);
	rc = mgf1(H, hlen, buf, emLen - hlen - 1, pssParms->mgf);
	if (rc != CKR_OK)
		goto done;

	/* pkcs1v2.2, Step 10: Compute maskedDB */
	for (i = 0; i < (emLen - hlen - 1); i++)
		em[i] ^= buf[i];

	/* pkcs1v2.2, Step 11: Set leftmost bits to zero. */
	em[0] &= 0xFF >> (8 * emLen - emBits);

	/* pkcs1v2.2, Step 12: EM = maskedDB || H || 0xbc */
	em[emLen - 1] = 0xbc;
	*modbytes = emLen;

done:
	if (buf)
		free(buf);

	return rc;
}

CK_RV emsa_pss_verify(CK_RSA_PKCS_PSS_PARAMS *pssParms, CK_BYTE *in_data,
		      CK_ULONG in_data_len, CK_BYTE *sig, CK_ULONG modbytes)
{
	int i;
	CK_ULONG buflen, hlen, emBits, emLen, plen;
	CK_BYTE *salt, *H, *M, *buf = NULL;
        CK_BYTE hash[MAX_SHA_HASH_SIZE];
	CK_RV rc = CKR_OK;

	/* pkcs1v2.2 8.1.1 describes emBits as length in bits of the
	 * modulus - 1. It also says, the octet length of EM will be
	 * one less than k if modBits - 1 is divisible by 8 and equal
	 * to k otherwise. k is the length in octets of the modulus n.
	 */
	emBits = (modbytes * 8) - 1;
	if ((emBits % 8) == 0)
		emLen = modbytes - 1;
	else
		emLen = modbytes;

	/* get hash size based on hashAlg */
	if (get_sha_size(pssParms->hashAlg, &hlen))
		return CKR_MECHANISM_INVALID;

	/* set up a big enough helper buffer to be used for M' and DB. */
	buflen = (emLen - hlen - 1) + (8 + hlen + pssParms->sLen);
	buf = (CK_BYTE *)malloc(buflen);
	if (buf == NULL)
		return CKR_HOST_MEMORY;
	memset(buf, 0, buflen);

	/* pkcs1v2.2, Step 4: Check rightmost octet. */
	if (sig[emLen - 1] != 0xbc) {
		rc = CKR_SIGNATURE_INVALID;
		goto done;
	}

	/* pkcs1v2.2, Step 5: Extract maskedDB and H */
	H = sig + (emLen - hlen - 1);
	
	/* pkcs1v2.2, Step 6: Check leftmost bits */
	if (sig[0] & ~(0xFF >> (8 * emLen - emBits))) {
		rc = CKR_SIGNATURE_INVALID;
		goto done;
	}

	/* pkcs1v2.2, Step 7: Compute mgf. */
	rc = mgf1(H, hlen, buf, emLen - hlen - 1, pssParms->mgf);
	if (rc != CKR_OK)
		goto done;

	/* pkcs1v2.2, Step 8: DB = maskedDB ^ dbMask. */
	for (i = 0; i < emLen - hlen - 1; i++)
		buf[i] ^= sig[i];

	/* pkcs1v2.2, Step 9: Set leftmost bits in DB to zero. */
	buf[0] &= 0xFF >> (8 * emLen - emBits);

	/* pkcs1v2.2, Step 10: check DB. */
	i = 0;
	plen = emLen - hlen - pssParms->sLen - 2;
	while ((buf[i] == 0) && ( i < plen))
		i++;

	if ((i != plen) || (buf[i++] != 0x01)) {
		rc = CKR_SIGNATURE_INVALID;
		goto done;
	}

	/* pkcs1v2.2, Step 11: Get the salt from DB. */
	salt = buf + i;

	/* pkcs1v2.2, Step 12: Set M'. Note: Use end of buf. */
	M = buf + (i + pssParms->sLen);
	memset(M, 0, 8);
	memcpy(M + 8, in_data, in_data_len);	// in_data is mHash.
	memcpy(M + (8 + in_data_len), salt, pssParms->sLen);

	/* pkcs1v2.2, Step 13: Compute Hash(M'). */
	rc = compute_sha(M, 8 + hlen + pssParms->sLen, hash, pssParms->hashAlg);
	if (rc != CKR_OK)
		goto done;

	/* pkcs1v2.2, Step 14: H == H'. */
	if (memcmp(hash, H, hlen))
		rc = CKR_SIGNATURE_INVALID;
	else
		rc = CKR_OK;
done:
	if (buf)
		free(buf);

	return rc;
}

CK_RV check_pss_params(CK_MECHANISM *mech, CK_ULONG modlen)
{
	CK_RSA_PKCS_PSS_PARAMS *pssParams;
	CK_MECHANISM_TYPE mgf_mech;
	CK_ULONG hlen;
	CK_RV rc;

	pssParams = (CK_RSA_PKCS_PSS_PARAMS *)mech->pParameter;

	if (mech->ulParameterLen != sizeof(CK_RSA_PKCS_PSS_PARAMS)) {
		TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_PARAM_INVALID));
		return CKR_MECHANISM_PARAM_INVALID;
	}

	/*
	 * If the signature mechanism includes hashing, make sure
	 * pssParams->hashAlg matches.
	 *
	 * Note: pkcs#1v2.2, Section 8.1, It is recommended that the
	 * hash algorithm used to hash the message be the same as the
	 * one used in mgf.
	 */
	rc = get_mgf_mech(pssParams->mgf, &mgf_mech);
	if (rc != CKR_OK) {
		TRACE_DEVEL("MGF mechanism is invalid.\n");
		return rc;
	}

	switch (mech->mechanism) {
	case CKM_SHA1_RSA_PKCS_PSS:
		if ((pssParams->hashAlg != CKM_SHA_1) &&
		    (pssParams->hashAlg != mgf_mech)) {
			TRACE_ERROR("%s\n",
				    ock_err(ERR_MECHANISM_PARAM_INVALID));
			return CKR_MECHANISM_PARAM_INVALID;
		}
		break;

	case CKM_SHA256_RSA_PKCS_PSS:
		if ((pssParams->hashAlg != CKM_SHA256) &&
		    (pssParams->hashAlg != mgf_mech)) {
			TRACE_ERROR("%s\n",
				    ock_err(ERR_MECHANISM_PARAM_INVALID));
			return CKR_MECHANISM_PARAM_INVALID;
		}
		break;

	case CKM_SHA384_RSA_PKCS_PSS:
		if ((pssParams->hashAlg != CKM_SHA384) &&
		    (pssParams->hashAlg != mgf_mech)) {
			TRACE_ERROR("%s\n",
				    ock_err(ERR_MECHANISM_PARAM_INVALID));
			return CKR_MECHANISM_PARAM_INVALID;
		}
		break;

	case CKM_SHA512_RSA_PKCS_PSS:
		if ((pssParams->hashAlg != CKM_SHA512) &&
		    (pssParams->hashAlg != mgf_mech)) {
			TRACE_ERROR("%s\n",
				    ock_err(ERR_MECHANISM_PARAM_INVALID));
			return CKR_MECHANISM_PARAM_INVALID;
		}
		break;

	case CKM_RSA_PKCS_PSS:
		if (pssParams->hashAlg != mgf_mech) {
			TRACE_ERROR("%s\n",
				    ock_err(ERR_MECHANISM_PARAM_INVALID));
			return CKR_MECHANISM_PARAM_INVALID;
		}
		break;

	default:
		TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
		return CKR_MECHANISM_INVALID;
	}

	/* check the salt length,  pkcs11v2.2 Section 12.1.14 */
	rc = get_sha_size(pssParams->hashAlg, &hlen);
	if (rc != CKR_OK) {
		TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_PARAM_INVALID));
		return CKR_MECHANISM_PARAM_INVALID;
	}

	if (!((pssParams->sLen >= 0) &&
	      (pssParams->sLen <= modlen - 2 - hlen))) {
		TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_PARAM_INVALID));
		return CKR_MECHANISM_PARAM_INVALID;
	}

	return CKR_OK;
}
