#include "pkcs11types.h"

#define MAX_KEY_SIZE  80
#define MAX_DATA_SIZE 512
#define MAX_HASH_SIZE 64

CK_ULONG four = 4;

struct DIGEST_TEST_VECTOR {
        CK_BYTE data[512];
        CK_ULONG data_len;
        CK_BYTE hash[512];
        CK_ULONG hash_len;
};

/** SHA test Vectors for Hashing Byte-Oriented Messages
    http://csrc.nist.gov/groups/STM/cavp/documents/shs/shabytetestvectors.zip
    SHA1ShortMsg.rsp

    FIPS PUB 180-1
    http://www.itl.nist.gov/fipspubs/fip180-1.htm
**/
struct DIGEST_TEST_VECTOR sha1_digest_test_vector[] = {
        {
                .data = "",
                .data_len = 0,
                .hash =       { 0xda, 0x39, 0xa3, 0xee, 0x5e, 0x6b, 0x4b, 0x0d, 0x32, 0x55,
                                0xbf, 0xef, 0x95, 0x60, 0x18, 0x90, 0xaf, 0xd8, 0x07, 0x09 },
                .hash_len = 20,
        }, {
                .data = "abc",
                .data_len = 3,
                .hash =       { 0xA9, 0x99, 0x3E, 0x36, 0x47, 0x06, 0x81, 0x6A, 0xBA, 0x3E,
                                0x25, 0x71, 0x78, 0x50, 0xC2, 0x6C, 0x9C, 0xD0, 0xD8, 0x9D },
                .hash_len = 20,
        }, {
                .data = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
                .data_len = 56,
                .hash =       { 0x84, 0x98, 0x3E, 0x44, 0x1C, 0x3B, 0xD2, 0x6E, 0xBA, 0xAE,
                                0x4A, 0xA1, 0xF9, 0x51, 0x29, 0xE5, 0xE5, 0x46, 0x70, 0xF1 },
                .hash_len = 20,
        }
};

/** The MD2 Message-Digest Algorithm
    http://tools.ietf.org/html/rfc1319
    A.5 Test suite
**/
struct DIGEST_TEST_VECTOR md2_digest_test_vector[] = {
        {
                .data = "",
                .data_len = 0,
                .hash =       { 0x83, 0x50, 0xe5, 0xa3, 0xe2, 0x4c, 0x15, 0x3d,
                                0xf2, 0x27, 0x5c, 0x9f, 0x80, 0x69, 0x27, 0x73 },
                .hash_len = 16,
        }, {
                .data = "a",
                .data_len = 1,
                .hash =       { 0x32, 0xec, 0x01, 0xec, 0x4a, 0x6d, 0xac, 0x72,
                                0xc0, 0xab, 0x96, 0xfb, 0x34, 0xc0, 0xb5, 0xd1 },
                .hash_len = 16,
        }, {
                .data = "abc",
                .data_len = 3,
                .hash =       { 0xda, 0x85, 0x3b, 0x0d, 0x3f, 0x88, 0xd9, 0x9b,
                                0x30, 0x28, 0x3a, 0x69, 0xe6, 0xde, 0xd6, 0xbb },
                .hash_len = 16,
        }, {
                .data = "message digest",
                .data_len = 14,
                .hash =       { 0xab, 0x4f, 0x49, 0x6b, 0xfb, 0x2a, 0x53, 0x0b,
                                0x21, 0x9f, 0xf3, 0x30, 0x31, 0xfe, 0x06, 0xb0 },
                .hash_len = 20,
        }, {
                .data = "abcdefghijklmnopqrstuvwxyz",
                .data_len = 26,
                .hash =       { 0x4e, 0x8d, 0xdf, 0xf3, 0x65, 0x02, 0x92, 0xab,
                                0x5a, 0x41, 0x08, 0xc3, 0xaa, 0x47, 0x94, 0x0b },
                .hash_len = 20,
        }, {
                .data = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789",
                .data_len = 62,
                .hash =       { 0xda, 0x33, 0xde, 0xf2, 0xa4, 0x2d, 0xf1, 0x39,
                                0x75, 0x35, 0x28, 0x46, 0xc3, 0x03, 0x38, 0xcd },
                .hash_len = 20,
        }, {
                .data = "12345678901234567890123456789012345678901234567890123456789012345678901234567890",
                .data_len = 80,
                .hash =       { 0xd5, 0x97, 0x6f, 0x79, 0xd8, 0x3d, 0x3a, 0x0d,
                                0xc9, 0x80, 0x6c, 0x3c, 0x66, 0xf3, 0xef, 0xd8 },
                .hash_len = 20,
        }
};

/** The MD5 Message-Digest Algorithm
    http://tools.ietf.org/html/rfc1321
    A.5 Test suite
**/
struct DIGEST_TEST_VECTOR md5_digest_test_vector[] = {
        {
                .data = "",
                .data_len = 0,
                .hash =       { 0xd4, 0x1d, 0x8c, 0xd9, 0x8f, 0x00, 0xb2, 0x04,
                                0xe9, 0x80, 0x09, 0x98, 0xec, 0xf8, 0x42, 0x7e },
                .hash_len = 16,
        }, {
                .data = "a",
                .data_len = 1,
                .hash =       { 0x0c, 0xc1, 0x75, 0xb9, 0xc0, 0xf1, 0xb6, 0xa8,
                                0x31, 0xc3, 0x99, 0xe2, 0x69, 0x77, 0x26, 0x61 },
                .hash_len = 16,
        }, {
                .data = "abc",
                .data_len = 3,
                .hash =       { 0x90, 0x01, 0x50, 0x98, 0x3c, 0xd2, 0x4f, 0xb0,
                                0xd6, 0x96, 0x3f, 0x7d, 0x28, 0xe1, 0x7f, 0x72},
                .hash_len = 16,
        }
};

/** SHA test Vectors for Hashing Byte-Oriented Messages
    http://csrc.nist.gov/groups/STM/cavp/documents/shs/shabytetestvectors.zip
    SHA256ShortMsg.rsp

    http://csrc.nist.gov/groups/ST/toolkit/examples.html
**/
struct DIGEST_TEST_VECTOR sha256_digest_test_vector[] = {
        {
                .data =       { 0x00 },
                .data_len = 0,
                .hash =       { 0xe3, 0xb0, 0xc4, 0x42, 0x98, 0xfc, 0x1c, 0x14,
                                0x9a, 0xfb, 0xf4, 0xc8, 0x99, 0x6f, 0xb9, 0x24,
                                0x27, 0xae, 0x41, 0xe4, 0x64, 0x9b, 0x93, 0x4c,
                                0xa4, 0x95, 0x99, 0x1b, 0x78, 0x52, 0xb8, 0x55 },
                .hash_len = 32,
        }, {
		.data = "abc",
		.data_len = 3,
		.hash =	      {	0xba, 0x78, 0x16, 0xbf, 0x8f, 0x01, 0xcf, 0xea,
				0x41, 0x41, 0x40, 0xde, 0x5d, 0xae, 0x22, 0x23,
				0xb0, 0x03, 0x61, 0xa3, 0x96, 0x17, 0x7a, 0x9c,
				0xb4, 0x10, 0xff, 0x61, 0xf2, 0x00, 0x15, 0xad },
		.hash_len = 32,
        }, {
		.data = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		.data_len = 56,
		.hash =       {	0x24, 0x8d, 0x6a, 0x61, 0xd2, 0x06, 0x38, 0xb8,
				0xe5, 0xc0, 0x26, 0x93, 0x0c, 0x3e, 0x60, 0x39,
				0xa3, 0x3c, 0xe4, 0x59, 0x64, 0xff, 0x21, 0x67,
				0xf6, 0xec, 0xed, 0xd4, 0x19, 0xdb, 0x06, 0xc1 },
		.hash_len = 32,
	}
};

/** SHA test Vectors for Hashing Byte-Oriented Messages
    http://csrc.nist.gov/groups/STM/cavp/documents/shs/shabytetestvectors.zip
    SHA384ShortMsg.rsp

    http://csrc.nist.gov/groups/ST/toolkit/examples.html
**/
struct DIGEST_TEST_VECTOR sha384_digest_test_vector[] = {
        {
                .data =       { 0x00},
                .data_len = 0,
                .hash =       { 0x38, 0xb0, 0x60, 0xa7, 0x51, 0xac, 0x96, 0x38,
                                0x4c, 0xd9, 0x32, 0x7e, 0xb1, 0xb1, 0xe3, 0x6a,
                                0x21, 0xfd, 0xb7, 0x11, 0x14, 0xbe, 0x07, 0x43,
                                0x4c, 0x0c, 0xc7, 0xbf, 0x63, 0xf6, 0xe1, 0xda,
                                0x27, 0x4e, 0xde, 0xbf, 0xe7, 0x6f, 0x65, 0xfb,
                                0xd5, 0x1a, 0xd2, 0xf1, 0x48, 0x98, 0xb9, 0x5b },
                .hash_len = 48,
        },{
		.data = "abc",
		.data_len = 3,
		.hash =       {	0xcb, 0x00, 0x75, 0x3f, 0x45, 0xa3, 0x5e, 0x8b,
				0xb5, 0xa0, 0x3d, 0x69, 0x9a, 0xc6, 0x50, 0x07,
				0x27, 0x2c, 0x32, 0xab, 0x0e, 0xde, 0xd1, 0x63,
				0x1a, 0x8b, 0x60, 0x5a, 0x43, 0xff, 0x5b, 0xed,
				0x80, 0x86, 0x07, 0x2b, 0xa1, 0xe7, 0xcc, 0x23,
				0x58, 0xba, 0xec, 0xa1, 0x34, 0xc8, 0x25, 0xa7 },
		.hash_len = 48,
	},{
		.data = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu",
		.data_len = 112,
		.hash =       {	0x09, 0x33, 0x0c, 0x33, 0xf7, 0x11, 0x47, 0xe8,
				0x3d, 0x19, 0x2f, 0xc7, 0x82, 0xcd, 0x1b, 0x47,
				0x53, 0x11, 0x1b, 0x17, 0x3b, 0x3b, 0x05, 0xd2,
				0x2f, 0xa0, 0x80, 0x86, 0xe3, 0xb0, 0xf7, 0x12,
				0xfc, 0xc7, 0xc7, 0x1a, 0x55, 0x7e, 0x2d, 0xb9,
				0x66, 0xc3, 0xe9, 0xfa, 0x91, 0x74, 0x60, 0x39 },
		.hash_len = 48,
	}
};

/** SHA test Vectors for Hashing Byte-Oriented Messages
    http://csrc.nist.gov/groups/STM/cavp/documents/shs/shabytetestvectors.zip
    SHA512ShortMsg.rsp

    http://csrc.nist.gov/groups/ST/toolkit/examples.html
**/
struct DIGEST_TEST_VECTOR sha512_digest_test_vector[] = {
        {
                .data =       { 0x00 },
                .data_len = 0,
                .hash =       { 0xcf, 0x83, 0xe1, 0x35, 0x7e, 0xef, 0xb8, 0xbd,
                                0xf1, 0x54, 0x28, 0x50, 0xd6, 0x6d, 0x80, 0x07,
                                0xd6, 0x20, 0xe4, 0x05, 0x0b, 0x57, 0x15, 0xdc,
                                0x83, 0xf4, 0xa9, 0x21, 0xd3, 0x6c, 0xe9, 0xce,
                                0x47, 0xd0, 0xd1, 0x3c, 0x5d, 0x85, 0xf2, 0xb0,
                                0xff, 0x83, 0x18, 0xd2, 0x87, 0x7e, 0xec, 0x2f,
                                0x63, 0xb9, 0x31, 0xbd, 0x47, 0x41, 0x7a, 0x81,
                                0xa5, 0x38, 0x32, 0x7a, 0xf9, 0x27, 0xda, 0x3e },
                .hash_len = 64,
	},{
		.data = "abc",
		.data_len = 3,
		.hash =	      {	0xdd, 0xaf, 0x35, 0xa1, 0x93, 0x61, 0x7a, 0xba,
				0xcc, 0x41, 0x73, 0x49, 0xae, 0x20, 0x41, 0x31,
				0x12, 0xe6, 0xfa, 0x4e, 0x89, 0xa9, 0x7e, 0xa2,
				0x0a, 0x9e, 0xee, 0xe6, 0x4b, 0x55, 0xd3, 0x9a,
				0x21, 0x92, 0x99, 0x2a, 0x27, 0x4f, 0xc1, 0xa8,
				0x36, 0xba, 0x3c, 0x23, 0xa3, 0xfe, 0xeb, 0xbd,
				0x45, 0x4d, 0x44, 0x23, 0x64, 0x3c, 0xe8, 0x0e,
				0x2a, 0x9a, 0xc9, 0x4f, 0xa5, 0x4c, 0xa4, 0x9f },
		.hash_len = 64,
	},{
		.data = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu",
		.data_len = 112,
		.hash =       {	0x8e, 0x95, 0x9b, 0x75, 0xda, 0xe3, 0x13, 0xda,
				0x8c, 0xf4, 0xf7, 0x28, 0x14, 0xfc, 0x14, 0x3f,
				0x8f, 0x77, 0x79, 0xc6, 0xeb, 0x9f, 0x7f, 0xa1,
				0x72, 0x99, 0xae, 0xad, 0xb6, 0x88, 0x90, 0x18,
				0x50, 0x1d, 0x28, 0x9e, 0x49, 0x00, 0xf7, 0xe4,
				0x33, 0x1b, 0x99, 0xde, 0xc4, 0xb5, 0x43, 0x3a,
				0xc7, 0xd3, 0x29, 0xee, 0xb6, 0xdd, 0x26, 0x54,
				0x5e, 0x96, 0xe5, 0x5b, 0x87, 0x4b, 0xe9, 0x09 },
		.hash_len = 64,
	}
};

/** The hash function RIPEMD-160
    http://homes.esat.kuleuven.be/~bosselae/ripemd160.html
**/
struct DIGEST_TEST_VECTOR ripemd128_digest_test_vector[] = {
        {
                .data = "",
                .data_len = 0,
                .hash =       { 0xcd, 0xf2, 0x62, 0x13, 0xa1, 0x50, 0xdc, 0x3e,
                                0xcb, 0x61, 0x0f, 0x18, 0xf6, 0xb3, 0x8b, 0x46 },
                .hash_len = 16,
        }
};

/** The hash function RIPEMD-160
    http://homes.esat.kuleuven.be/~bosselae/ripemd160.html
**/
struct DIGEST_TEST_VECTOR ripemd160_digest_test_vector[] = {
        {
                .data = "",
                .data_len = 0,
                .hash =       { 0x9c, 0x11, 0x85, 0xa5, 0xc5, 0xe9, 0xfc, 0x54, 0x61, 0x28,
                                0x08, 0x97, 0x7e, 0xe8, 0xf5, 0x48, 0xb2, 0x25, 0x8d, 0x31 },
                .hash_len = 20,
        }
};

struct digest_test_suite_info {
        const char *name;
        unsigned int tvcount;
        struct DIGEST_TEST_VECTOR *tv;
        CK_MECHANISM mech;
};

#define NUM_DIGEST_TEST_SUITES 8
struct digest_test_suite_info digest_test_suites[] = {
        {
                .name = "SHA-1",
                .tvcount = 3,
                .tv = sha1_digest_test_vector,
                .mech = {CKM_SHA_1, 0, 0},
        }, {
                .name = "SHA-256",
                .tvcount = 3,
                .tv = sha256_digest_test_vector,
                .mech = {CKM_SHA256, 0, 0},
        }, {
                .name = "SHA-384",
                .tvcount = 3,
                .tv = sha384_digest_test_vector,
                .mech = {CKM_SHA384, 0, 0},
        }, {
                .name = "SHA-512",
                .tvcount = 3,
                .tv = sha512_digest_test_vector,
                .mech = {CKM_SHA512, 0, 0},
        }, {
                .name = "MD2",
                .tvcount = 7,
                .tv = md2_digest_test_vector,
                .mech = {CKM_MD2, 0, 0},
        }, {
                .name = "MD5",
                .tvcount = 3,
                .tv = md5_digest_test_vector,
                .mech = {CKM_MD5, 0, 0},
        }, {
                .name = "RIPEMD-128",
                .tvcount = 1,
                .tv = ripemd128_digest_test_vector,
                .mech = {CKM_RIPEMD128, 0, 0},
        }, {
                .name = "RIPEMD-160",
                .tvcount = 1,
                .tv = ripemd160_digest_test_vector,
                .mech = {CKM_RIPEMD160, 0, 0},
        }
};

/** MD5 HMAC and SHA1 HMAC test vectors from RFC2202:
 *  http://www.faqs.org/rfcs/rfc2202.html
 *
 *  Test vectors for SHA256 and SHA512 from RFC4868:
 *  http://tools.ietf.org/html/rfc4868
 **/

struct HMAC_TEST_VECTOR {
	/* test vector inputs */
	CK_BYTE         key[512];
	CK_ULONG        key_len;
	CK_BYTE         data[512];
	CK_ULONG        data_len;
	CK_ULONG        hash_len;
	CK_BYTE         result[512];
};

struct HMAC_TEST_VECTOR sha1_hmac_test_vector[] = {
        {
                .key =        { 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
                                0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b },
                .key_len = 20,
                .data =       { "Hi There" },
                .data_len = 8,
                .hash_len = 20,
                .result =     { 0xb6, 0x17, 0x31, 0x86, 0x55, 0x05, 0x72, 0x64, 0xe2, 0x8b,
                                0xc0, 0xb6, 0xfb, 0x37, 0x8c, 0x8e, 0xf1, 0x46, 0xbe, 0x00 }
	}, {
	        .key =        { 'J', 'e', 'f', 'e' },
                .key_len = 4,
                .data =       { "what do ya want for nothing?" },
                .data_len = 28,
                .hash_len = 20,
                .result =     { 0xef, 0xfc, 0xdf, 0x6a, 0xe5, 0xeb, 0x2f, 0xa2, 0xd2, 0x74,
                                0x16, 0xd5, 0xf1, 0x84, 0xdf, 0x9c, 0x25, 0x9a, 0x7c, 0x79 }
	}, {
	        .key =        { 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa },
                .key_len = 20,
                .data =       { 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD },
                .data_len = 50,
                .hash_len = 20,
                .result =     { 0x12, 0x5d, 0x73, 0x42, 0xb9, 0xac, 0x11, 0xcd, 0x91, 0xa3,
                                0x9a, 0xf4, 0x8a, 0xa1, 0x7b, 0x4f, 0x63, 0xf1, 0x75, 0xd3 }
	}, {
	        .key =        { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a,
                                0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14,
                                0x15, 0x16, 0x17, 0x18, 0x19 },
                .key_len = 25,
                .data =       { 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD },
                .data_len = 50,
                .hash_len = 20,
                .result =     { 0x4c, 0x90, 0x07, 0xf4, 0x02, 0x62, 0x50, 0xc6, 0xbc, 0x84,
                                0x14, 0xf9, 0xbf, 0x50, 0xc8, 0x6c, 0x2d, 0x72, 0x35, 0xda }
	}, {
	        .key =        { 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
                                0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c },
                .key_len = 20,
                .data =       { "Test With Truncation" },
                .data_len = 20,
                .hash_len = 20,
                .result =     { 0x4c, 0x1a, 0x03, 0x42, 0x4b, 0x55, 0xe0, 0x7f, 0xe7, 0xf2,
                                0x7b, 0xe1, 0xd5, 0x8b, 0xb9, 0x32, 0x4a, 0x9a, 0x5a, 0x04 }
	}, {
	        .key =        { 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa },
                .key_len = 80,
                .data =       { "Test Using Larger Than Block-Size Key - Hash Key First" },
                .data_len = 54,
                .hash_len = 20,
                .result =     { 0xaa, 0x4a, 0xe5, 0xe1, 0x52, 0x72, 0xd0, 0x0e, 0x95, 0x70, 0x56,
                                0x37, 0xce, 0x8a, 0x3b, 0x55, 0xed, 0x40, 0x21, 0x12 }
	}, {
	        .key =        { 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa },
                .key_len = 80,
                .data =       { "Test Using Larger Than Block-Size Key and Larger Than One Block-Size Data" },
                .data_len = 73,
                .hash_len = 20,
                .result =     { 0xe8, 0xe9, 0x9d, 0x0f, 0x45, 0x23, 0x7d, 0x78, 0x6d, 0x6b,
                                0xba, 0xa7, 0x96, 0x5c, 0x78, 0x08, 0xbb, 0xff, 0x1a, 0x91 }
	},
};

struct HMAC_TEST_VECTOR sha256_hmac_test_vector[] = {
        {
                .key =        { 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
                                0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
                                0x0b, 0x0b, 0x0b, 0x0b },
                .key_len = 20,
                .data =       { "Hi There" },
                .data_len = 8,
                .hash_len = 32,
                .result =     { 0xb0, 0x34, 0x4c, 0x61, 0xd8, 0xdb, 0x38, 0x53,
                                0x5c, 0xa8, 0xaf, 0xce, 0xaf, 0x0b, 0xf1, 0x2b,
                                0x88, 0x1d, 0xc2, 0x00, 0xc9, 0x83, 0x3d, 0xa7,
                                0x26, 0xe9, 0x37, 0x6c, 0x2e, 0x32, 0xcf, 0xf7 }
	}, {
	        .key =        { 'J', 'e', 'f', 'e' },
                .key_len = 4,
                .data =       { "what do ya want for nothing?" },
                .data_len = 28,
                .hash_len = 32,
                .result =     { 0x5b, 0xdc, 0xc1, 0x46, 0xbf, 0x60, 0x75, 0x4e,
                                0x6a, 0x04, 0x24, 0x26, 0x08, 0x95, 0x75, 0xc7,
                                0x5a, 0x00, 0x3f, 0x08, 0x9d, 0x27, 0x39, 0x83,
                                0x9d, 0xec, 0x58, 0xb9, 0x64, 0xec, 0x38, 0x43 }
	}, {
	        .key =        { 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa },
                .key_len = 20,
                .data =       { 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD },
                .data_len = 50,
                .hash_len = 32,
                .result =     { 0x77, 0x3e, 0xa9, 0x1e, 0x36, 0x80, 0x0e, 0x46,
                                0x85, 0x4d, 0xb8, 0xeb, 0xd0, 0x91, 0x81, 0xa7,
                                0x29, 0x59, 0x09, 0x8b, 0x3e, 0xf8, 0xc1, 0x22,
                                0xd9, 0x63, 0x55, 0x14, 0xce, 0xd5, 0x65, 0xfe }
	}, {
                .key =        { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                                0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
                                0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
                                0x19 },
                .key_len = 25,
                .data =       { 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD },
                .data_len = 50,
                .hash_len = 32,
                .result =     { 0x82, 0x55, 0x8a, 0x38, 0x9a, 0x44, 0x3c, 0x0e,
                                0xa4, 0xcc, 0x81, 0x98, 0x99, 0xf2, 0x08, 0x3a,
                                0x85, 0xf0, 0xfa, 0xa3, 0xe5, 0x78, 0xf8, 0x07,
                                0x7a, 0x2e, 0x3f, 0xf4, 0x67, 0x29, 0x66, 0x5b }
	}, 
};

struct HMAC_TEST_VECTOR sha512_hmac_test_vector[] = {
        {
                .key =        { 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
                                0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
                                0x0b, 0x0b, 0x0b, 0x0b },
                .key_len = 20,
                .data =       { "Hi There" },
                .data_len = 8,
                .hash_len = 64,
                .result =     { 0x87, 0xaa, 0x7c, 0xde, 0xa5, 0xef, 0x61, 0x9d,
                                0x4f, 0xf0, 0xb4, 0x24, 0x1a, 0x1d, 0x6c, 0xb0,
                                0x23, 0x79, 0xf4, 0xe2, 0xce, 0x4e, 0xc2, 0x78,
                                0x7a, 0xd0, 0xb3, 0x05, 0x45, 0xe1, 0x7c, 0xde,
                                0xda, 0xa8, 0x33, 0xb7, 0xd6, 0xb8, 0xa7, 0x02,
                                0x03, 0x8b, 0x27, 0x4e, 0xae, 0xa3, 0xf4, 0xe4,
                                0xbe, 0x9d, 0x91, 0x4e, 0xeb, 0x61, 0xf1, 0x70,
                                0x2e, 0x69, 0x6c, 0x20, 0x3a, 0x12, 0x68, 0x54 }
	}, {
	        .key =        { 'J', 'e', 'f', 'e' },
                .key_len = 4,
                .data =       { "what do ya want for nothing?" },
                .data_len = 28,
                .hash_len = 64,
                .result =     { 0x16, 0x4b, 0x7a, 0x7b, 0xfc, 0xf8, 0x19, 0xe2,
                                0xe3, 0x95, 0xfb, 0xe7, 0x3b, 0x56, 0xe0, 0xa3,
                                0x87, 0xbd, 0x64, 0x22, 0x2e, 0x83, 0x1f, 0xd6,
                                0x10, 0x27, 0x0c, 0xd7, 0xea, 0x25, 0x05, 0x54,
                                0x97, 0x58, 0xbf, 0x75, 0xc0, 0x5a, 0x99, 0x4a,
                                0x6d, 0x03, 0x4f, 0x65, 0xf8, 0xf0, 0xe6, 0xfd,
                                0xca, 0xea, 0xb1, 0xa3, 0x4d, 0x4a, 0x6b, 0x4b,
                                0x63, 0x6e, 0x07, 0x0a, 0x38, 0xbc, 0xe7, 0x37 }
	}, {
	        .key =        { 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa },
                .key_len = 20,
                .data =       { 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD },
                .data_len = 50,
                .hash_len = 64,
                .result =     { 0xfa, 0x73, 0xb0, 0x08, 0x9d, 0x56, 0xa2, 0x84,
                                0xef, 0xb0, 0xf0, 0x75, 0x6c, 0x89, 0x0b, 0xe9,
                                0xb1, 0xb5, 0xdb, 0xdd, 0x8e, 0xe8, 0x1a, 0x36,
                                0x55, 0xf8, 0x3e, 0x33, 0xb2, 0x27, 0x9d, 0x39,
                                0xbf, 0x3e, 0x84, 0x82, 0x79, 0xa7, 0x22, 0xc8,
                                0x06, 0xb4, 0x85, 0xa4, 0x7e, 0x67, 0xc8, 0x07,
                                0xb9, 0x46, 0xa3, 0x37, 0xbe, 0xe8, 0x94, 0x26,
                                0x74, 0x27, 0x88, 0x59, 0xe1, 0x32, 0x92, 0xfb }
	}, {
	        .key =        { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                                0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
                                0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
                                0x19 },
                .key_len = 25,
                .data =       { 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD },
                .data_len = 50,
                .hash_len = 64,
                .result =     { 0xb0, 0xba, 0x46, 0x56, 0x37, 0x45, 0x8c, 0x69,
                                0x90, 0xe5, 0xa8, 0xc5, 0xf6, 0x1d, 0x4a, 0xf7,
                                0xe5, 0x76, 0xd9, 0x7f, 0xf9, 0x4b, 0x87, 0x2d,
                                0xe7, 0x6f, 0x80, 0x50, 0x36, 0x1e, 0xe3, 0xdb,
                                0xa9, 0x1c, 0xa5, 0xc1, 0x1a, 0xa2, 0x5e, 0xb4,
                                0xd6, 0x79, 0x27, 0x5c, 0xc5, 0x78, 0x80, 0x63,
                                0xa5, 0xf1, 0x97, 0x41, 0x12, 0x0c, 0x4f, 0x2d,
                                0xe2, 0xad, 0xeb, 0xeb, 0x10, 0xa2, 0x98, 0xdd }
	},
};

struct HMAC_TEST_VECTOR md5_hmac_test_vector[] = {
        {
                .key =        { 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
                                0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b },
                .key_len = 16,
                .data =       { "Hi There" },
                .data_len = 8,
                .hash_len = 16,
                .result =     { 0x92, 0x94, 0x72, 0x7a, 0x36, 0x38, 0xBB, 0x1C,
                                0x13, 0xf4, 0x8E, 0xf8, 0x15, 0x8b, 0xfc, 0x9d }
	}, {
	        .key =        { 'J', 'e', 'f', 'e' },
                .key_len = 4,
                .data =       { "what do ya want for nothing?" },
                .data_len = 28,
                .hash_len = 16,
                .result =     { 0x75, 0x0c, 0x78, 0x3e, 0x6a, 0xb0, 0xb5, 0x03,
                                0xea, 0xa8, 0x6e, 0x31, 0x0a, 0x5d, 0xb7, 0x38 }
	}, {
	        .key =        { 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa },
                .key_len = 16,
                .data =       { 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD },
                .data_len = 50,
                .hash_len = 16,
                .result =     { 0x56, 0xbe, 0x34, 0x52, 0x1d, 0x14, 0x4c, 0x88,
                                0xdb, 0xb8, 0xc7, 0x33, 0xf0, 0xe8, 0xb3, 0xf6 }
	}, {
	        .key =        { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                                0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
                                0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
                                0x19 },
                .key_len = 25,
                .data =       { 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD },
                .data_len = 50,
                .hash_len = 16,
                .result =     { 0x69, 0x7e, 0xaf, 0x0a, 0xca, 0x3a, 0x3a, 0xea,
                                0x3a, 0x75, 0x16, 0x47, 0x46, 0xff, 0xaa, 0x79 }
	}, {
	        .key =        { 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
                                0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c },
                .key_len = 16,
                .data =       { "Test With Truncation" },
                .data_len = 20,
                .hash_len = 16,
                .result =     { 0x56, 0x46, 0x1e, 0xf2, 0x34, 0x2e, 0xdc, 0x00,
                                0xf9, 0xba, 0xb9, 0x95, 0x69, 0x0e, 0xfd, 0x4c}
	}, {
	        .key =        { 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa },
                .key_len = 80,
                .data =       { "Test Using Larger Than Block-Size Key - Hash Key First" },
                .data_len = 54,
                .hash_len = 16,
                .result =     { 0x6b, 0x1a, 0xb7, 0xfe, 0x4b, 0xd7, 0xbf, 0x8f,
                                0x0b, 0x62, 0xe6, 0xce, 0x61, 0xb9, 0xd0, 0xcd }
	}, {
	        .key =        { 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa },
                .key_len = 80,
                .data =       { "Test Using Larger Than Block-Size Key and Larger Than One Block-Size Data" },
                .data_len = 73,
                .hash_len = 16,
                .result =     { 0x6f, 0x63, 0x0f, 0xad, 0x67, 0xcd, 0xa0, 0xee,
                                0x1f, 0xb1, 0xf5, 0x62, 0xdb, 0x3a, 0xa5, 0x3e }
	},
};

struct HMAC_TEST_VECTOR sha1_hmac_general_test_vector[] = {
        {
                .key =        { 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
                                0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b },
                .key_len = 20,
                .data =       { "Hi There" },
                .data_len = 8,
                .hash_len = 4,
                .result =     { 0xb6, 0x17, 0x31, 0x86, 0x55, 0x05, 0x72, 0x64, 0xe2, 0x8b,
                                0xc0, 0xb6, 0xfb, 0x37, 0x8c, 0x8e, 0xf1, 0x46, 0xbe, 0x00 }
	}, {
	        .key =        { 'J', 'e', 'f', 'e' },
                .key_len = 4,
                .data =       { "what do ya want for nothing?" },
                .data_len = 28,
                .hash_len = 4,
                .result =     { 0xef, 0xfc, 0xdf, 0x6a, 0xe5, 0xeb, 0x2f, 0xa2, 0xd2, 0x74,
                                0x16, 0xd5, 0xf1, 0x84, 0xdf, 0x9c, 0x25, 0x9a, 0x7c, 0x79 }
	}, {
	        .key =        { 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa },
                .key_len = 20,
                .data =       { 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD },
                .data_len = 50,
                .hash_len = 4,
                .result =     { 0x12, 0x5d, 0x73, 0x42, 0xb9, 0xac, 0x11, 0xcd, 0x91, 0xa3,
                                0x9a, 0xf4, 0x8a, 0xa1, 0x7b, 0x4f, 0x63, 0xf1, 0x75, 0xd3 }
	}, {
	        .key =        { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a,
                                0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14,
                                0x15, 0x16, 0x17, 0x18, 0x19 },
                .key_len = 25,
                .data =       { 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD },
                .data_len = 50,
                .hash_len = 4,
                .result =     { 0x4c, 0x90, 0x07, 0xf4, 0x02, 0x62, 0x50, 0xc6, 0xbc, 0x84,
                                0x14, 0xf9, 0xbf, 0x50, 0xc8, 0x6c, 0x2d, 0x72, 0x35, 0xda }
	}, {
	        .key =        { 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
                                0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c },
                .key_len = 20,
                .data =       { "Test With Truncation" },
                .data_len = 20,
                .hash_len = 4,
                .result =     { 0x4c, 0x1a, 0x03, 0x42, 0x4b, 0x55, 0xe0, 0x7f, 0xe7, 0xf2,
                                0x7b, 0xe1, 0xd5, 0x8b, 0xb9, 0x32, 0x4a, 0x9a, 0x5a, 0x04 }
	}, {
	        .key =        { 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa },
                .key_len = 80,
                .data =       { "Test Using Larger Than Block-Size Key - Hash Key First" },
                .data_len = 54,
                .hash_len = 4,
                .result =     { 0xaa, 0x4a, 0xe5, 0xe1, 0x52, 0x72, 0xd0, 0x0e, 0x95, 0x70,
                                0x56, 0x37, 0xce, 0x8a, 0x3b, 0x55, 0xed, 0x40, 0x21, 0x12 }
	}, {
	        .key =        { 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa },
                .key_len = 80,
                .data =       { "Test Using Larger Than Block-Size Key and Larger Than One Block-Size Data" },
                .data_len = 73,
                .hash_len = 4,
                .result =     { 0xe8, 0xe9, 0x9d, 0x0f, 0x45, 0x23, 0x7d, 0x78, 0x6d, 0x6b,
                                0xba, 0xa7, 0x96, 0x5c, 0x78, 0x08, 0xbb, 0xff, 0x1a, 0x91 }
	},
};

struct HMAC_TEST_VECTOR sha256_hmac_general_test_vector[] = {
        {
                .key =        { 'J', 'e', 'f', 'e' },
                .key_len = 4,
                .data =       { "what do ya want for nothing?" },
                .data_len = 28,
                .hash_len = 4,
                .result =     { 0x5b, 0xdc, 0xc1, 0x46, 0xbf, 0x60, 0x75, 0x4e,
                                0x6a, 0x04, 0x24, 0x26, 0x08, 0x95, 0x75, 0xc7,
                                0x5a, 0x00, 0x3f, 0x08, 0x9d, 0x27, 0x39, 0x83,
                                0x9d, 0xec, 0x58, 0xb9, 0x64, 0xec, 0x38, 0x43 }
	}, {
	        .key =        { 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa },
                .key_len = 20,
                .data =       { 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD },
                .data_len = 50,
                .hash_len = 4,
                .result =      { 0x77, 0x3e, 0xa9, 0x1e, 0x36, 0x80, 0x0e, 0x46,
                                0x85, 0x4d, 0xb8, 0xeb, 0xd0, 0x91, 0x81, 0xa7,
                                0x29, 0x59, 0x09, 0x8b, 0x3e, 0xf8, 0xc1, 0x22,
                                0xd9, 0x63, 0x55, 0x14, 0xce, 0xd5, 0x65, 0xfe }
	}, {
	        .key =        { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                                0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
                                0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
                                0x19 },
                .key_len = 25,
                .data =       { 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD },
                .data_len = 50,
                .hash_len = 4,
                .result =     { 0x82, 0x55, 0x8a, 0x38, 0x9a, 0x44, 0x3c, 0x0e,
                                0xa4, 0xcc, 0x81, 0x98, 0x99, 0xf2, 0x08, 0x3a,
                                0x85, 0xf0, 0xfa, 0xa3, 0xe5, 0x78, 0xf8, 0x07,
                                0x7a, 0x2e, 0x3f, 0xf4, 0x67, 0x29, 0x66, 0x5b }
	},
};

struct HMAC_TEST_VECTOR sha512_hmac_general_test_vector[] = {
        {
                .key =        { 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
                                0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
                                0x0b, 0x0b, 0x0b, 0x0b },
	  .key_len = 20,
	  .data =             { "Hi There" },
	  .data_len = 8,
	  .hash_len = 4,
	  .result =           { 0x87, 0xaa, 0x7c, 0xde, 0xa5, 0xef, 0x61, 0x9d,
                                0x4f, 0xf0, 0xb4, 0x24, 0x1a, 0x1d, 0x6c, 0xb0,
                                0x23, 0x79, 0xf4, 0xe2, 0xce, 0x4e, 0xc2, 0x78,
                                0x7a, 0xd0, 0xb3, 0x05, 0x45, 0xe1, 0x7c, 0xde,
                                0xda, 0xa8, 0x33, 0xb7, 0xd6, 0xb8, 0xa7, 0x02,
                                0x03, 0x8b, 0x27, 0x4e, 0xae, 0xa3, 0xf4, 0xe4,
                                0xbe, 0x9d, 0x91, 0x4e, 0xeb, 0x61, 0xf1, 0x70,
                                0x2e, 0x69, 0x6c, 0x20, 0x3a, 0x12, 0x68, 0x54 }
	}, {
	        .key =        { 'J', 'e', 'f', 'e' },
                .key_len = 4,
                .data =       { "what do ya want for nothing?" },
                .data_len = 28,
                .hash_len = 4,
                .result =     { 0x16, 0x4b, 0x7a, 0x7b, 0xfc, 0xf8, 0x19, 0xe2,
                                0xe3, 0x95, 0xfb, 0xe7, 0x3b, 0x56, 0xe0, 0xa3,
                                0x87, 0xbd, 0x64, 0x22, 0x2e, 0x83, 0x1f, 0xd6,
                                0x10, 0x27, 0x0c, 0xd7, 0xea, 0x25, 0x05, 0x54,
                                0x97, 0x58, 0xbf, 0x75, 0xc0, 0x5a, 0x99, 0x4a,
                                0x6d, 0x03, 0x4f, 0x65, 0xf8, 0xf0, 0xe6, 0xfd,
                                0xca, 0xea, 0xb1, 0xa3, 0x4d, 0x4a, 0x6b, 0x4b,
                                0x63, 0x6e, 0x07, 0x0a, 0x38, 0xbc, 0xe7, 0x37 }
	}, {
	        .key =        { 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa },
                .key_len = 20,
                .data =       { 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD },
                .data_len = 50,
                .hash_len = 4,
                .result =     { 0xfa, 0x73, 0xb0, 0x08, 0x9d, 0x56, 0xa2, 0x84,
                                0xef, 0xb0, 0xf0, 0x75, 0x6c, 0x89, 0x0b, 0xe9,
                                0xb1, 0xb5, 0xdb, 0xdd, 0x8e, 0xe8, 0x1a, 0x36,
                                0x55, 0xf8, 0x3e, 0x33, 0xb2, 0x27, 0x9d, 0x39,
                                0xbf, 0x3e, 0x84, 0x82, 0x79, 0xa7, 0x22, 0xc8,
                                0x06, 0xb4, 0x85, 0xa4, 0x7e, 0x67, 0xc8, 0x07,
                                0xb9, 0x46, 0xa3, 0x37, 0xbe, 0xe8, 0x94, 0x26,
                                0x74, 0x27, 0x88, 0x59, 0xe1, 0x32, 0x92, 0xfb }
	}, {
	        .key =        { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                                0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
                                0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
                                0x19 },
	  .key_len = 25,
	  .data =             { 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD },
	  .data_len = 50,
	  .hash_len = 4,
	  .result =           { 0xb0, 0xba, 0x46, 0x56, 0x37, 0x45, 0x8c, 0x69,
                                0x90, 0xe5, 0xa8, 0xc5, 0xf6, 0x1d, 0x4a, 0xf7,
                                0xe5, 0x76, 0xd9, 0x7f, 0xf9, 0x4b, 0x87, 0x2d,
                                0xe7, 0x6f, 0x80, 0x50, 0x36, 0x1e, 0xe3, 0xdb,
                                0xa9, 0x1c, 0xa5, 0xc1, 0x1a, 0xa2, 0x5e, 0xb4,
                                0xd6, 0x79, 0x27, 0x5c, 0xc5, 0x78, 0x80, 0x63,
                                0xa5, 0xf1, 0x97, 0x41, 0x12, 0x0c, 0x4f, 0x2d,
                                0xe2, 0xad, 0xeb, 0xeb, 0x10, 0xa2, 0x98, 0xdd }
	},
};

struct HMAC_TEST_VECTOR md5_hmac_general_test_vector[] = {
	{
	        .key =        { 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
                                0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b },
                .key_len = 16,
                .data =       { "Hi There" },
                .data_len = 8,
                .hash_len = 4,
                .result =     { 0x92, 0x94, 0x72, 0x7a, 0x36, 0x38, 0xBB, 0x1C,
                                0x13, 0xf4, 0x8E, 0xf8, 0x15, 0x8b, 0xfc, 0x9d }
	}, {
	        .key =        { 'J', 'e', 'f', 'e' },
                .key_len = 4,
                .data =       { "what do ya want for nothing?" },
                .data_len = 28,
                .hash_len = 4,
                .result =     { 0x75, 0x0c, 0x78, 0x3e, 0x6a, 0xb0, 0xb5, 0x03,
                                0xea, 0xa8, 0x6e, 0x31, 0x0a, 0x5d, 0xb7, 0x38 }
	}, {
	        .key =        { 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa },
                .key_len = 16,
                .data =       { 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
                                0xDD, 0xDD },
                .data_len = 50,
                .hash_len = 4,
                .result =     { 0x56, 0xbe, 0x34, 0x52, 0x1d, 0x14, 0x4c, 0x88,
                                0xdb, 0xb8, 0xc7, 0x33, 0xf0, 0xe8, 0xb3, 0xf6 }
	}, {
	        .key =        { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                                0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
                                0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
                                0x19 },
                .key_len = 25,
                .data =       { 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
                                0xCD, 0xCD },
                .data_len = 50,
                .hash_len = 4,
                .result =     { 0x69, 0x7e, 0xaf, 0x0a, 0xca, 0x3a, 0x3a, 0xea,
                                0x3a, 0x75, 0x16, 0x47, 0x46, 0xff, 0xaa, 0x79 }
	}, {
	        .key =        { 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
                                0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c },
                .key_len = 16,
                .data =       { "Test With Truncation" },
                .data_len = 20,
                .hash_len = 4,
                .result =     { 0x56, 0x46, 0x1e, 0xf2, 0x34, 0x2e, 0xdc, 0x00,
                                0xf9, 0xba, 0xb9, 0x95, 0x69, 0x0e, 0xfd, 0x4c}
	}, {
	        .key =        { 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa },
                .key_len = 80,
                .data =       { "Test Using Larger Than Block-Size Key - Hash Key First" },
                .data_len = 54,
                .hash_len = 4,
                .result =     { 0x6b, 0x1a, 0xb7, 0xfe, 0x4b, 0xd7, 0xbf, 0x8f,
                                0x0b, 0x62, 0xe6, 0xce, 0x61, 0xb9, 0xd0, 0xcd }
	}, {
	        .key =        { 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
                                0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa },
                .key_len = 80,
                .data =       { "Test Using Larger Than Block-Size Key and Larger Than One Block-Size Data" },
                .data_len = 73,
                .hash_len = 4,
                .result =     { 0x6f, 0x63, 0x0f, 0xad, 0x67, 0xcd, 0xa0, 0xee,
                                0x1f, 0xb1, 0xf5, 0x62, 0xdb, 0x3a, 0xa5, 0x3e }
	},
};

struct HMAC_TEST_SUITE_INFO {
        const char *name;
        unsigned int tvcount;
        struct HMAC_TEST_VECTOR *tv;
        CK_MECHANISM mech;
};

#define NUM_OF_HMAC_TEST_SUITES 8
struct HMAC_TEST_SUITE_INFO hmac_test_suites[] = {
        {
                .name = "SHA-1 HMAC",
                .tvcount = 7,
                .tv = sha1_hmac_test_vector,
                .mech = { CKM_SHA_1_HMAC, 0, 0 },
        }, {
                .name = "SHA-256 HMAC",
                .tvcount = 4,
                .tv = sha256_hmac_test_vector,
                .mech = { CKM_SHA256_HMAC, 0, 0 },
        }, {
                .name = "SHA-512 HMAC",
                .tvcount = 4,
                .tv = sha512_hmac_test_vector,
                .mech = { CKM_SHA512_HMAC, 0, 0 },
        }, {
                .name = "MD5 HMAC",
                .tvcount = 7,
                .tv = md5_hmac_test_vector,
                .mech = { CKM_MD5_HMAC, 0, 0 },
        }, {
                .name = "SHA-1 HMAC General",
                .tvcount = 7,
                .tv = sha1_hmac_general_test_vector,
                .mech = { CKM_SHA_1_HMAC_GENERAL, &four, sizeof(CK_ULONG) },
        }, {
                .name = "SHA-256 HMAC General",
                .tvcount = 3,
                .tv = sha256_hmac_general_test_vector,
                .mech = { CKM_SHA256_HMAC_GENERAL, &four, sizeof(CK_ULONG) },
        }, {
                .name = "SHA-512 HMAC General",
                .tvcount = 4,
                .tv = sha512_hmac_general_test_vector,
                .mech = { CKM_SHA512_HMAC_GENERAL, &four, sizeof(CK_ULONG) },
        }, {
                .name = "MD5 HMAC General",
                .tvcount = 7,
                .tv = md5_hmac_general_test_vector,
                .mech = { CKM_MD5_HMAC_GENERAL, &four, sizeof(CK_ULONG) },
        }
};
