#include "debug_utils-inl.h"
#include "env-inl.h"
#include "node_errors.h"
#include "node_process.h"

#include <time.h>  // tzset(), _tzset()

namespace node {
using v8::Array;
using v8::Boolean;
using v8::Context;
using v8::DontDelete;
using v8::DontEnum;
using v8::EscapableHandleScope;
using v8::HandleScope;
using v8::Integer;
using v8::Isolate;
using v8::Just;
using v8::Local;
using v8::Maybe;
using v8::MaybeLocal;
using v8::Name;
using v8::NamedPropertyHandlerConfiguration;
using v8::NewStringType;
using v8::Nothing;
using v8::Object;
using v8::ObjectTemplate;
using v8::PropertyCallbackInfo;
using v8::PropertyHandlerFlags;
using v8::ReadOnly;
using v8::String;
using v8::Value;

class RealEnvStore final : public KVStore {
 public:
  MaybeLocal<String> Get(Isolate* isolate, Local<String> key) const override;
  Maybe<std::string> Get(const char* key) const override;
  void Set(Isolate* isolate, Local<String> key, Local<String> value) override;
  int32_t Query(Isolate* isolate, Local<String> key) const override;
  int32_t Query(const char* key) const override;
  void Delete(Isolate* isolate, Local<String> key) override;
  Local<Array> Enumerate(Isolate* isolate) const override;
};

class MapKVStore final : public KVStore {
 public:
  MaybeLocal<String> Get(Isolate* isolate, Local<String> key) const override;
  Maybe<std::string> Get(const char* key) const override;
  void Set(Isolate* isolate, Local<String> key, Local<String> value) override;
  int32_t Query(Isolate* isolate, Local<String> key) const override;
  int32_t Query(const char* key) const override;
  void Delete(Isolate* isolate, Local<String> key) override;
  Local<Array> Enumerate(Isolate* isolate) const override;

  std::shared_ptr<KVStore> Clone(Isolate* isolate) const override;

  MapKVStore() = default;
  MapKVStore(const MapKVStore& other) : KVStore(), map_(other.map_) {}

 private:
  mutable Mutex mutex_;
  std::unordered_map<std::string, std::string> map_;
};

namespace per_process {
Mutex env_var_mutex;
std::shared_ptr<KVStore> system_environment = std::make_shared<RealEnvStore>();
}  // namespace per_process

template <typename T>
void DateTimeConfigurationChangeNotification(Isolate* isolate, const T& key) {
  if (key.length() == 2 && key[0] == 'T' && key[1] == 'Z') {
#ifdef __POSIX__
    tzset();
#else
    _tzset();
#endif
    auto constexpr time_zone_detection = Isolate::TimeZoneDetection::kRedetect;
    isolate->DateTimeConfigurationChangeNotification(time_zone_detection);
  }
}

Maybe<std::string> RealEnvStore::Get(const char* key) const {
  Mutex::ScopedLock lock(per_process::env_var_mutex);

  size_t init_sz = 256;
  MaybeStackBuffer<char, 256> val;
  int ret = uv_os_getenv(key, *val, &init_sz);

  if (ret == UV_ENOBUFS) {
    // Buffer is not large enough, reallocate to the updated init_sz
    // and fetch env value again.
    val.AllocateSufficientStorage(init_sz);
    ret = uv_os_getenv(key, *val, &init_sz);
  }

  if (ret >= 0) {  // Env key value fetch success.
    return Just(std::string(*val, init_sz));
  }

  return Nothing<std::string>();
}

MaybeLocal<String> RealEnvStore::Get(Isolate* isolate,
                                     Local<String> property) const {
  node::Utf8Value key(isolate, property);
  Maybe<std::string> value = Get(*key);

  if (value.IsJust()) {
    std::string val = value.FromJust();
    return String::NewFromUtf8(
        isolate, val.data(), NewStringType::kNormal, val.size());
  }

  return MaybeLocal<String>();
}

void RealEnvStore::Set(Isolate* isolate,
                       Local<String> property,
                       Local<String> value) {
  Mutex::ScopedLock lock(per_process::env_var_mutex);

  node::Utf8Value key(isolate, property);
  node::Utf8Value val(isolate, value);

#ifdef _WIN32
  if (key.length() > 0 && key[0] == '=') return;
#endif
  uv_os_setenv(*key, *val);
  DateTimeConfigurationChangeNotification(isolate, key);
}

int32_t RealEnvStore::Query(const char* key) const {
  Mutex::ScopedLock lock(per_process::env_var_mutex);

  char val[2];
  size_t init_sz = sizeof(val);
  int ret = uv_os_getenv(key, val, &init_sz);

  if (ret == UV_ENOENT) {
    return -1;
  }

#ifdef _WIN32
  if (key[0] == '=') {
    return static_cast<int32_t>(ReadOnly) |
           static_cast<int32_t>(DontDelete) |
           static_cast<int32_t>(DontEnum);
  }
#endif

  return 0;
}

int32_t RealEnvStore::Query(Isolate* isolate, Local<String> property) const {
  node::Utf8Value key(isolate, property);
  return Query(*key);
}

void RealEnvStore::Delete(Isolate* isolate, Local<String> property) {
  Mutex::ScopedLock lock(per_process::env_var_mutex);

  node::Utf8Value key(isolate, property);
  uv_os_unsetenv(*key);
  DateTimeConfigurationChangeNotification(isolate, key);
}

Local<Array> RealEnvStore::Enumerate(Isolate* isolate) const {
  Mutex::ScopedLock lock(per_process::env_var_mutex);
  uv_env_item_t* items;
  int count;

  auto cleanup = OnScopeLeave([&]() { uv_os_free_environ(items, count); });
  CHECK_EQ(uv_os_environ(&items, &count), 0);

  MaybeStackBuffer<Local<Value>, 256> env_v(count);
  int env_v_index = 0;
  for (int i = 0; i < count; i++) {
#ifdef _WIN32
    // If the key starts with '=' it is a hidden environment variable.
    // The '\0' check is a workaround for the bug behind
    // https://github.com/libuv/libuv/pull/2473 and can be removed later.
    if (items[i].name[0] == '=' || items[i].name[0] == '\0') continue;
#endif
    MaybeLocal<String> str = String::NewFromUtf8(isolate, items[i].name);
    if (str.IsEmpty()) {
      isolate->ThrowException(ERR_STRING_TOO_LONG(isolate));
      return Local<Array>();
    }
    env_v[env_v_index++] = str.ToLocalChecked();
  }

  return Array::New(isolate, env_v.out(), env_v_index);
}

std::shared_ptr<KVStore> KVStore::Clone(Isolate* isolate) const {
  HandleScope handle_scope(isolate);
  Local<Context> context = isolate->GetCurrentContext();

  std::shared_ptr<KVStore> copy = KVStore::CreateMapKVStore();
  Local<Array> keys = Enumerate(isolate);
  uint32_t keys_length = keys->Length();
  for (uint32_t i = 0; i < keys_length; i++) {
    Local<Value> key = keys->Get(context, i).ToLocalChecked();
    CHECK(key->IsString());
    copy->Set(isolate,
              key.As<String>(),
              Get(isolate, key.As<String>()).ToLocalChecked());
  }
  return copy;
}

Maybe<std::string> MapKVStore::Get(const char* key) const {
  Mutex::ScopedLock lock(mutex_);
  auto it = map_.find(key);
  return it == map_.end() ? Nothing<std::string>() : Just(it->second);
}

MaybeLocal<String> MapKVStore::Get(Isolate* isolate, Local<String> key) const {
  Utf8Value str(isolate, key);
  Maybe<std::string> value = Get(*str);
  if (value.IsNothing()) return Local<String>();
  std::string val = value.FromJust();
  return String::NewFromUtf8(
      isolate, val.data(), NewStringType::kNormal, val.size());
}

void MapKVStore::Set(Isolate* isolate, Local<String> key, Local<String> value) {
  Mutex::ScopedLock lock(mutex_);
  Utf8Value key_str(isolate, key);
  Utf8Value value_str(isolate, value);
  if (*key_str != nullptr && key_str.length() > 0 && *value_str != nullptr) {
    map_[std::string(*key_str, key_str.length())] =
        std::string(*value_str, value_str.length());
  }
}

int32_t MapKVStore::Query(const char* key) const {
  Mutex::ScopedLock lock(mutex_);
  return map_.find(key) == map_.end() ? -1 : 0;
}

int32_t MapKVStore::Query(Isolate* isolate, Local<String> key) const {
  Utf8Value str(isolate, key);
  return Query(*str);
}

void MapKVStore::Delete(Isolate* isolate, Local<String> key) {
  Mutex::ScopedLock lock(mutex_);
  Utf8Value str(isolate, key);
  map_.erase(std::string(*str, str.length()));
}

Local<Array> MapKVStore::Enumerate(Isolate* isolate) const {
  Mutex::ScopedLock lock(mutex_);
  std::vector<Local<Value>> values;
  values.reserve(map_.size());
  for (const auto& pair : map_) {
    values.emplace_back(
        String::NewFromUtf8(isolate, pair.first.data(),
                            NewStringType::kNormal, pair.first.size())
            .ToLocalChecked());
  }
  return Array::New(isolate, values.data(), values.size());
}

std::shared_ptr<KVStore> MapKVStore::Clone(Isolate* isolate) const {
  return std::make_shared<MapKVStore>(*this);
}

std::shared_ptr<KVStore> KVStore::CreateMapKVStore() {
  return std::make_shared<MapKVStore>();
}

Maybe<bool> KVStore::AssignFromObject(Local<Context> context,
                                      Local<Object> entries) {
  Isolate* isolate = context->GetIsolate();
  HandleScope handle_scope(isolate);
  Local<Array> keys;
  if (!entries->GetOwnPropertyNames(context).ToLocal(&keys))
    return Nothing<bool>();
  uint32_t keys_length = keys->Length();
  for (uint32_t i = 0; i < keys_length; i++) {
    Local<Value> key;
    if (!keys->Get(context, i).ToLocal(&key))
      return Nothing<bool>();
    if (!key->IsString()) continue;

    Local<Value> value;
    Local<String> value_string;
    if (!entries->Get(context, key).ToLocal(&value) ||
        !value->ToString(context).ToLocal(&value_string)) {
      return Nothing<bool>();
    }

    Set(isolate, key.As<String>(), value_string);
  }
  return Just(true);
}

static void EnvGetter(Local<Name> property,
                      const PropertyCallbackInfo<Value>& info) {
  Environment* env = Environment::GetCurrent(info);
  CHECK(env->has_run_bootstrapping_code());
  if (property->IsSymbol()) {
    return info.GetReturnValue().SetUndefined();
  }
  CHECK(property->IsString());
  MaybeLocal<String> value_string =
      env->env_vars()->Get(env->isolate(), property.As<String>());
  if (!value_string.IsEmpty()) {
    info.GetReturnValue().Set(value_string.ToLocalChecked());
  }
}

static void EnvSetter(Local<Name> property,
                      Local<Value> value,
                      const PropertyCallbackInfo<Value>& info) {
  Environment* env = Environment::GetCurrent(info);
  CHECK(env->has_run_bootstrapping_code());
  // calling env->EmitProcessEnvWarning() sets a variable indicating that
  // warnings have been emitted. It should be called last after other
  // conditions leading to a warning have been met.
  if (env->options()->pending_deprecation && !value->IsString() &&
      !value->IsNumber() && !value->IsBoolean() &&
      env->EmitProcessEnvWarning()) {
    if (ProcessEmitDeprecationWarning(
            env,
            "Assigning any value other than a string, number, or boolean to a "
            "process.env property is deprecated. Please make sure to convert "
            "the "
            "value to a string before setting process.env with it.",
            "DEP0104")
            .IsNothing())
      return;
  }

  Local<String> key;
  Local<String> value_string;
  if (!property->ToString(env->context()).ToLocal(&key) ||
      !value->ToString(env->context()).ToLocal(&value_string)) {
    return;
  }

  env->env_vars()->Set(env->isolate(), key, value_string);

  // Whether it worked or not, always return value.
  info.GetReturnValue().Set(value);
}

static void EnvQuery(Local<Name> property,
                     const PropertyCallbackInfo<Integer>& info) {
  Environment* env = Environment::GetCurrent(info);
  CHECK(env->has_run_bootstrapping_code());
  if (property->IsString()) {
    int32_t rc = env->env_vars()->Query(env->isolate(), property.As<String>());
    if (rc != -1) info.GetReturnValue().Set(rc);
  }
}

static void EnvDeleter(Local<Name> property,
                       const PropertyCallbackInfo<Boolean>& info) {
  Environment* env = Environment::GetCurrent(info);
  CHECK(env->has_run_bootstrapping_code());
  if (property->IsString()) {
    env->env_vars()->Delete(env->isolate(), property.As<String>());
  }

  // process.env never has non-configurable properties, so always
  // return true like the tc39 delete operator.
  info.GetReturnValue().Set(true);
}

static void EnvEnumerator(const PropertyCallbackInfo<Array>& info) {
  Environment* env = Environment::GetCurrent(info);
  CHECK(env->has_run_bootstrapping_code());

  info.GetReturnValue().Set(
      env->env_vars()->Enumerate(env->isolate()));
}

MaybeLocal<Object> CreateEnvVarProxy(Local<Context> context, Isolate* isolate) {
  EscapableHandleScope scope(isolate);
  Local<ObjectTemplate> env_proxy_template = ObjectTemplate::New(isolate);
  env_proxy_template->SetHandler(NamedPropertyHandlerConfiguration(
      EnvGetter, EnvSetter, EnvQuery, EnvDeleter, EnvEnumerator, Local<Value>(),
      PropertyHandlerFlags::kHasNoSideEffect));
  return scope.EscapeMaybe(env_proxy_template->NewInstance(context));
}
}  // namespace node
