# Upgrade Inhibition
### Process Inhibition
With latest changes on Leapp and with new actors added to the el7toel8 Leapp repository, any actor can inhibit the upgrade process by producing a specific message when a problem is found. The message model to use in this case is [Inhibitor](https://github.com/oamg/leapp-repository/blob/master/repos/system_upgrade/el7toel8/models/reports.py). If there is at least one Inhibitor message produced before the Report phase, the upgrade will be stopped in the Reports phase, in which the messages are being collected. It means that any Inhibitor message produced **after** the Report phase will have **no** inhibiting effect. The details mentioned in the Inhibitor messages will be part of the report available to the user to review.


### Sample Actor
Let’s start with a very simple actor that will verify if system architecture is supported:

```python
import platform

from leapp.actors import Actor
from leapp.tags import ChecksPhaseTag


class CheckSystemArch(Actor):
name = 'check_system_arch'
    description = 'Verify if system has a supported arch.'
    consumes = ()
    produces = ()
    tags = (ChecksPhaseTag,)

    def process(self):
        if platform.machine() != 'x86_64':
            self.log.info("Unsupported arch!")
```

If this actor is executed using `snactor` tool in a system with unsupported architecture, we will see the following output on log:

```sh
$ snactor run CheckSystemArch
2018-09-25 19:08:59.622 INFO     PID: 1996 leapp: Logging has been initialized
2018-09-25 19:08:59.638 INFO     PID: 1996 leapp.repository.sandbox: A new repository 'sandbox' is initialized at /home/leapp/sandbox
2018-09-25 19:08:59.695 INFO     PID: 2021 leapp.actors.check_system_arch: Unsupported arch!
```

If, instead of only adding a message to the log, the actor writer wants to make sure that the upgrade process will be stopped in case of unsupported arch, it only needs to produce a [Inhibitor](https://github.com/oamg/leapp-repository/blob/master/repos/system_upgrade/el7toel8/models/reports.py) message.

```python
import platform

from leapp.actors import Actor
from leapp.models import Inhibitor
from leapp.tags import ChecksPhaseTag


class CheckSystemArch(Actor):
name = 'check_system_arch'
    description = 'Verify if system has a supported arch.'
    consumes = ()
    produces = (Inhibitor,)
    tags = (ChecksPhaseTag,)

    def process(self):
        if platform.machine() != 'x86_64':
            self.produce(Inhibitor(
                summary='Unsupported arch!',
                details='This process can only be executed on x86_64 systems',
                solutions='There is no current solution for this problem'))
```

Running the actor again, it is possible to verify that a new message was generated. We will still use `snactor` tool to run the actor, but passing `--print-output` this time to output all generated messages by the actor:

```sh
$ snactor run --print-output CheckSystemArch
2018-09-25 19:20:32.74  INFO     PID: 2621 leapp: Logging has been initialized
2018-09-25 19:20:32.94  INFO     PID: 2621 leapp.repository.sandbox: A new repository 'sandbox' is initialized at /home/leapp/sandbox
[
  {
    "stamp": "2018-09-25T19:20:32.143709Z",
    "hostname": "leapp",
    "actor": "check_system_arch",
    "topic": "system_info",
    "context": "904b0170-cfe7-4217-81d3-a259550e73c1",
    "phase": "NON-WORKFLOW-EXECUTION",
    "message": {
      "hash": "483ac0e46c99535e893ab30cf0d95774830f8406f1633594c97469cf16773ec3",
      "data": "{\"details\": \"This process can only be executed on x86_64 systems\", \"result\": \"Fail\", \"severity\": \"Error\", \"solutions\": \"There is no current solution for this problem\", \"summary\": \"Unsupported arch!\"}"
    },
    "type": "Inhibitor"
  }
]
```

This is all that an actor needs to do in order to verify if some condition is present on the system and inhibit the upgrade process based on that check.

After all the system checks are executed by different actors, an existing actor named [VerifyCheckResults](https://github.com/oamg/leapp-repository/tree/master/repos/system_upgrade/el7toel8/actors/verifycheckresults) is scheduled to run in the Leapp upgrade workflow. If some [Inhibitor](https://github.com/oamg/leapp-repository/blob/master/repos/system_upgrade/el7toel8/models/reports.py) message was generated by some previous execution of another actor in any previous phase of the workflow, like the sample one we just wrote, the following output will be displayed to the user:

```sh
$ leapp upgrade
(...)
2018-09-25 19:30:49.978 INFO     PID: 2873 leapp.actors.verify_check_results: Generated report at /tmp/leapp-report.txt

============================================================
                        ERRORS
============================================================

2018-09-25 19:30:49.989122 [ERROR] Actor: check_system_arch Message: Error: Fail: Unsupported arch!
2018-09-25 19:30:49.989271 [ERROR] Actor: verify_check_results Message: Ending process due to errors found during checks

============================================================
                     END OF ERRORS
============================================================

[]
```
