#!/bin/bash

# clones and updates an automotive sig repo
# $1: branch to be used
# $2: alternate tmp directory (if you have faster storage)
# $3: kernel source tarball
# $4: package name
# shellcheck disable=SC2164

automotive_branch=$1;
automotive_tmp=$2;
automotive_tarball=$3;
package_name=$4;

redhat=$(dirname "$0")/..;
distgit="git@gitlab.com:CentOS/automotive/rpms/${package_name}"
lookaside_url=https://git.centos.org/sources/upload_sig.cgi
hashtype=sha512

# XXX: Placeholders for RHEL variables
rhdistgit_zstream_flag="no"
rtbz=""

function die
{
	echo "Error: $1" >&2;
	exit 1;
}

# DO NOT USE: This is a place holder until the centos-git-common tools are
# updated to support the new lookaside cache.
# We might want to use centpkg-sig instead of these tools (when it's updated
# to use the new cache). If so we would define it in Makefile.rhpkg and then we
# might be able to use the standard "dist-release" and "dist-git" make targets.
function upload_sources()
{
	# Uploading the tarball only using CentOS tools (uses the old lookaside cache)
	echo "Cloning the centos common repository";
	git clone https://git.centos.org/centos-git-common.git centos-git-common >/dev/null || die "Unable to clone centos tools";

	[ -n "$RH_DIST_GIT_TEST" ] && return

	./centos-git-common/lookaside_upload -f "$automotive_tarball" -n "$package_name" -b "$automotive_branch";
}

# Upload the tarball to the lookaside cache (not using the tools)
function upload()
{
	test -a ~/.centos.cert || die "Client certificate required. See: https://wiki.centos.org/Authentication"

	[ -n "$RH_DIST_GIT_TEST" ] && return

	# Until the centos-git-common project is updated, upload the tarball manually.
	curl ${lookaside_url} \
		--fail \
		--cert ~/.centos.cert \
		--form "name=${package_name}" \
		--form "hash=${hashtype}" \
		--form "${hashtype}sum=${sha}" \
		--form "file=@${automotive_tarball}" \
		--progress-bar | tee /dev/null
}

function update_distgit()
{
	# Cloning the dist-git repo
	echo "Cloning $package_name dist-git repository";
	git clone -b "$automotive_branch" "$distgit" "$tmpdir/$package_name" >/dev/null || die "Unable to clone $distgit";

	# Copy all the new sources (including SPEC) except the big tarball, but use it as a reference
	rm -f "$tmpdir/$package_name/*" &> /dev/null; # Replaces everything, git will know if a file changes
	tarball_name=$(basename -- "$automotive_tarball");
	find "$redhat"/rpm/SOURCES/ \
		\( ! -name "${tarball_name}" \) \
		\( ! -name "*.tar.bz2" \) \
		\( ! -name "*gitignore" \) -type f -print0 | xargs -0 cp -t "$tmpdir/$package_name/" &> /dev/null;

	# Remove EMPTY files auto generated by dist-sources
	grep -lrI "# EMPTY" "$tmpdir/$package_name" | xargs rm -f &> /dev/null;

	# Because centpkg-sig (equivalent of fedpkg, centpkg or rhpkg) requires an
	# update to support the new lookaside the sources file is updated manually.
	echo "SHA512 ($tarball_name) = $sha" > "$tmpdir/$package_name/sources"
}

function create_changelog()
{
	echo "Creating diff for review ($tmpdir/diff) and changelog"
	# diff the result (redhat/cvs/dontdiff). note: diff returns 1 if
	# differences were found
	diff -X "$redhat"/git/dontdiff -upr "$tmpdir/$package_name" "$redhat"/rpm/SOURCES/ > "$tmpdir"/diff;

	# Create the changelog file (no zstream_flag and rtbz).
	# Outputs a ready to use commit message for dist-git.
	"$redhat"/scripts/create_distgit_changelog.sh "$redhat"/rpm/SOURCES/kernel.spec \
		"$rhdistgit_zstream_flag" "$package_name" "$rtbz" >"$tmpdir"/changelog
	# remove BZs (this will need a more formal solution when we move to Jira)
	sed -i "/^Resolves: /d" "$tmpdir"/changelog;
}

sha=$("${hashtype}"sum "$automotive_tarball" | awk "{ print \$1 }")
test -n "$sha" || die "Could not generate the file hash. Does $automotive_tarball exist?"

# Create a directory for staging dist-git changes
date=$(date +"%Y-%m-%d");
tmpdir="$(mktemp -d --tmpdir="$automotive_tmp" AUTOMOTIVE."$date".XXXXXXXX)";
test -d "$tmpdir" || die "Unable to create temporary directory";

# update the dist-git repo
update_distgit || die "Unable to update package repository.";

# upload the sources to the centos lookaside cache
upload || die "Unable to upload the sources"

# generate the diff and changelog
create_changelog

# all done
echo "dist-git updated here: $tmpdir"
