/*
 * @(#)CommandMap.java	1.16 99/12/06
 *
 * Copyright 1997-1999 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the proprietary information of Sun Microsystems, Inc.  
 * Use is subject to license terms.
 * 
 */

package javax.activation;


/**
 * The CommandMap class provides an interface to a registry of
 * command objects available in the system.
 * Developers are expected to either use the CommandMap
 * implementation included with this package (MailcapCommandMap) or
 * develop their own. Note that some of the methods in this class are
 * abstract.
 */
public abstract class CommandMap {
    private static CommandMap defaultCommandMap = null;

    /**
     * Get the default CommandMap.
     * <p>
     *
     * <ul>
     * <li> In cases where a CommandMap instance has been previously set
     *      to some value (via <i>setDefaultCommandMap</i>)
     *  return the CommandMap.
     * <li>
     *  In cases where no CommandMap has been set, the CommandMap
     *       creates an instance of <code>MailcapCommandMap</code> and
     *       set that to the default, returning its value.
     *
     * </ul>
     *
     * @return the CommandMap
     */
    public static CommandMap getDefaultCommandMap() {
	if (defaultCommandMap == null)
	    defaultCommandMap = new MailcapCommandMap();

	return defaultCommandMap;
    }

    /**
     * Set the default CommandMap. Reset the CommandMap to the default by
     * calling this method with <code>null</code>.
     *
     * @param commandMap The new default CommandMap.
     * @exception SecurityException if the caller doesn't have permission
     *					to change the default
     */
    public static void setDefaultCommandMap(CommandMap commandMap) {
	SecurityManager security = System.getSecurityManager();
	if (security != null) {
	    try {
		// if it's ok with the SecurityManager, it's ok with me...
		security.checkSetFactory();
	    } catch (SecurityException ex) {
		// otherwise, we also allow it if this code and the
		// factory come from the same class loader (e.g.,
		// the JAF classes were loaded with the applet classes).
		if (CommandMap.class.getClassLoader() !=
			    commandMap.getClass().getClassLoader())
		    throw ex;
	    }
	}
	defaultCommandMap = commandMap;
    }

    /**
     * Get the preferred command list from a MIME Type. The actual semantics
     * are determined by the implementation of the CommandMap.
     *
     * @param mimeType	the MIME type
     * @return the CommandInfo classes that represent the command Beans.
     */
    abstract public  CommandInfo[] getPreferredCommands(String mimeType);

    /**
     * Get all the available commands for this type. This method
     * should return all the possible commands for this MIME type.
     *
     * @param mimeType	the MIME type
     * @return the CommandInfo objects representing all the commands.
     */
    abstract public CommandInfo[] getAllCommands(String mimeType);

    /**
     * Get the default command corresponding to the MIME type.
     *
     * @param mimeType	the MIME type
     * @param cmdName	the command name
     * @return the CommandInfo corresponding to the command.
     */
    abstract public CommandInfo getCommand(String mimeType, String cmdName);

    /**
     * Locate a DataContentHandler that corresponds to the MIME type.
     * The mechanism and semantics for determining this are determined
     * by the implementation of the particular CommandMap.
     *
     * @param mimeType	the MIME type
     * @return		the DataContentHandler for the MIME type
     */
    abstract public DataContentHandler createDataContentHandler(String
								mimeType);
}
