/*
 * @(#)MailcapFile.java	1.15 02/02/03
 *
 * Copyright 1997-2002 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the proprietary information of Sun Microsystems, Inc.  
 * Use is subject to license terms.
 * 
 */

package com.sun.activation.registries;

import java.io.*;
import java.util.*;

public class MailcapFile {

    private Hashtable type_hash = null;
    private static boolean debug = false;

    static {
	try {
	    debug = Boolean.getBoolean("javax.activation.debug");
	} catch (Throwable t) {
	    // ignore any errors
	}
    }

    /**
     * The constructor that takes a filename as an argument.
     *
     * @param new_fname The file name of the mailcap file.
     */
    public MailcapFile(String new_fname) throws IOException {
	if (debug)
	    System.out.println("new MailcapFile: file " + new_fname);
	FileReader reader = null;
	try {
	    reader = new FileReader(new_fname);
	    type_hash = createMailcapHash(new BufferedReader(reader));
	} finally {
	    if (reader != null) {
		try {
		    reader.close();
		} catch (IOException ex) { }
	    }
	}
    }

    /**
     * The constructor that takes an input stream as an argument.
     *
     * @param is	the input stream
     */
    public MailcapFile(InputStream is) throws IOException {
	if (debug)
	    System.out.println("new MailcapFile: InputStream");
	type_hash = createMailcapHash(
	    new BufferedReader(new InputStreamReader(is, "iso-8859-1")));
    }

    /**
     * Mailcap file default constructor.
     */
    public MailcapFile() {
	if (debug)
	    System.out.println("new MailcapFile: default");
	// initialize the hash table
	type_hash = new Hashtable();
    }

    /**
     * Get the Hashtable of MailcapEntries based on the MIME type.
     *
     * <p>
     * <strong>Semantics:</strong> First check for the litteral mime type,
     * if that fails looks for wildcard <type>/\* and return that. Return the
     * list of all that hit.
     */
    public Hashtable getMailcapList(String mime_type) {
	Hashtable search_result = null;
	Hashtable wildcard_result = null;

	// first try the literal
	search_result = (Hashtable)type_hash.get(mime_type);

	// ok, now try the wildcard
	int separator = mime_type.indexOf('/');
	String type = mime_type.substring(0, separator + 1) + "*";
	wildcard_result = (Hashtable)type_hash.get(type);

	if (wildcard_result != null) { // damn, we have to merge!!!
	    if (search_result != null)
		search_result = mergeResults(search_result, wildcard_result);
	    else
		search_result = wildcard_result;
	}
	return search_result;
    }

    /**
     * Merge the first hash into the second.
     * This merge will only effect the hashtable that is
     * returned, we don't want to touch the one passed in since
     * its integrity must be maintained.
     */
    private Hashtable mergeResults(Hashtable first, Hashtable second) {
	Enumeration verb_enum = second.keys();
	Hashtable clonedHash = (Hashtable)first.clone();

	// iterate through the verbs in the second map
	while (verb_enum.hasMoreElements()) {
	    String verb = (String)verb_enum.nextElement();
	    Vector cmdVector = (Vector)clonedHash.get(verb);
	    if (cmdVector == null) {
		clonedHash.put(verb, second.get(verb));
	    } else {
		// merge the two
		Vector oldV = (Vector)second.get(verb);
		Enumeration v_enum = oldV.elements();
		cmdVector = (Vector)cmdVector.clone();
		clonedHash.put(verb, cmdVector);
		while (v_enum.hasMoreElements()) {
		    cmdVector.addElement(v_enum.nextElement());
		}
	    }
	}
	return clonedHash;
    }

    /**
     * appendToMailcap: Append to this Mailcap DB, use the mailcap
     * format:
     * Comment == "# <i>comment string</i>
     * Entry == "mimetype;        javabeanclass<nl>
     *
     * Example:
     * # this is a comment
     * image/gif       jaf.viewers.ImageViewer
     */
    public void appendToMailcap(String mail_cap) {
	if (debug)
	    System.out.println("appendToMailcap: " + mail_cap);
	try {
	    parse(new StringReader(mail_cap), type_hash);
	} catch (IOException ex) {
	    // can't happen
	}
    }

    /**
     * create a hash table of Mail Type Entry Obj from a string buffer
     */
    private Hashtable createMailcapHash(Reader reader) throws IOException {
	Hashtable new_hash = new Hashtable();

	parse(reader, new_hash);

	return new_hash;
    }

    /**
     * parse file into a hash table of MC Type Entry Obj
     */
    private void parse(Reader reader, Hashtable hash) throws IOException {
	BufferedReader buf_reader = new BufferedReader(reader);
	String line = null;
	String continued = null;

	while ((line = buf_reader.readLine()) != null) {
	    //    System.out.println("parsing line: " + line);

	    line = line.trim();

	    try {
		if (line.charAt(0) == '#')
		    continue;
		if (line.charAt(line.length() - 1) == '\\') {
		    if (continued != null)
			continued += line.substring(0, line.length() - 1);
		    else
			continued = line.substring(0, line.length() - 1);
		} else if (continued != null) {
		    // handle the two strings
		    continued = continued + line;
		    //	System.out.println("parse: " + continued);
		    try {
			parseLine(continued, hash);
		    } catch (MailcapParseException e) {
			//e.printStackTrace();
		    }
		    continued = null;
		}
		else {
		    //	System.out.println("parse: " + line);
		    try {
			parseLine(line, hash);
			// System.out.println("hash.size = " + hash.size());
		    } catch (MailcapParseException e) {
			//e.printStackTrace();
		    }
		}
	    } catch (StringIndexOutOfBoundsException e) {}
	}
    }

    /**
     *	A routine to parse individual entries in a Mailcap file.
     *
     *	Note that this routine does not handle line continuations.
     *	They should have been handled prior to calling this routine.
     */
    protected static void parseLine(String mailcapEntry, Hashtable masterHash)
				throws MailcapParseException, IOException {
	MailcapTokenizer tokenizer = new MailcapTokenizer(mailcapEntry);
	tokenizer.setIsAutoquoting(false);
	String primaryType = "";
	String subType = "*";
	String soFar = "";

	if (debug)
	    System.out.println("parse: " + mailcapEntry);
	//	parse the primary type
	int currentToken = tokenizer.nextToken();
	soFar = soFar.concat(tokenizer.getCurrentTokenValue());
	if (currentToken != MailcapTokenizer.STRING_TOKEN) {
	    reportParseError(MailcapTokenizer.STRING_TOKEN, currentToken,
					tokenizer.getCurrentTokenValue());
	}
	primaryType = tokenizer.getCurrentTokenValue().toLowerCase();

	//	parse the '/' between primary and sub
	//	if it's not present that's ok, we just don't have a subtype
	currentToken = tokenizer.nextToken();
	soFar = soFar.concat(tokenizer.getCurrentTokenValue());
	if ((currentToken != MailcapTokenizer.SLASH_TOKEN) &&
			(currentToken != MailcapTokenizer.SEMICOLON_TOKEN)) {
	    reportParseError(MailcapTokenizer.SLASH_TOKEN,
				MailcapTokenizer.SEMICOLON_TOKEN, currentToken,
				tokenizer.getCurrentTokenValue());
	}

	//	only need to look for a sub type if we got a '/'
	if (currentToken == MailcapTokenizer.SLASH_TOKEN) {
	    //	parse the sub type
	    currentToken = tokenizer.nextToken();
	    soFar = soFar.concat(tokenizer.getCurrentTokenValue());
	    if (currentToken != MailcapTokenizer.STRING_TOKEN) {
		reportParseError(MailcapTokenizer.STRING_TOKEN,
			    currentToken, tokenizer.getCurrentTokenValue());
	    }
	    subType = tokenizer.getCurrentTokenValue().toLowerCase();

	    //	get the next token to simplify the next step
	    currentToken = tokenizer.nextToken();
	    soFar = soFar.concat(tokenizer.getCurrentTokenValue());
	}

	if (debug)
	    System.out.println("  Type: " + primaryType + "/" + subType);
	//	now setup the commands hashtable
	Hashtable commands =
	    (Hashtable)masterHash.get(primaryType + "/" + subType);
	if (commands == null) {
		commands = new Hashtable();
		masterHash.put(primaryType + "/" + subType, commands);
	}

	//	parse the ';' that separates the type from the parameters
	if (currentToken != MailcapTokenizer.SEMICOLON_TOKEN) {
	    reportParseError(MailcapTokenizer.SEMICOLON_TOKEN,
			    currentToken, tokenizer.getCurrentTokenValue());
	}
	//	eat it

	//	parse the required view command
	tokenizer.setIsAutoquoting(true);
	currentToken = tokenizer.nextToken();
	tokenizer.setIsAutoquoting(false);
	soFar = soFar.concat(tokenizer.getCurrentTokenValue());
	if ((currentToken != MailcapTokenizer.STRING_TOKEN) &&
		    (currentToken != MailcapTokenizer.SEMICOLON_TOKEN)) {
	    //	System.out.println("THE LINE SO FAR: " + soFar);
	    reportParseError(MailcapTokenizer.STRING_TOKEN,
			    MailcapTokenizer.SEMICOLON_TOKEN, currentToken,
			    tokenizer.getCurrentTokenValue());
	}
	//	eat it since JAF doesn't really care

	//	only have to get the next token if the current one isn't a ';'
	if (currentToken != MailcapTokenizer.SEMICOLON_TOKEN) {
	    currentToken = tokenizer.nextToken();
	}

	// look for a ';' which will indicate whether
	// a parameter list is present or not
	if (currentToken == MailcapTokenizer.SEMICOLON_TOKEN) {
	    do {
		//	eat the ';'

		//	parse the parameter name
		currentToken = tokenizer.nextToken();
		if (currentToken != MailcapTokenizer.STRING_TOKEN) {
		    reportParseError(MailcapTokenizer.STRING_TOKEN,
			    currentToken, tokenizer.getCurrentTokenValue());
		}
		String paramName =
		    tokenizer.getCurrentTokenValue().toLowerCase();

		//	parse the '=' which separates the name from the value
		currentToken = tokenizer.nextToken();
		if ((currentToken != MailcapTokenizer.EQUALS_TOKEN) &&
		    (currentToken != MailcapTokenizer.SEMICOLON_TOKEN) &&
		    (currentToken != MailcapTokenizer.EOI_TOKEN)) {
		    reportParseError(MailcapTokenizer.EQUALS_TOKEN,
			    MailcapTokenizer.SEMICOLON_TOKEN,
			    MailcapTokenizer.EOI_TOKEN,
			    currentToken, tokenizer.getCurrentTokenValue());
		}

		//	we only have a useful command if it is named
		if (currentToken == MailcapTokenizer.EQUALS_TOKEN) {
		    //	eat it

		    //	parse the parameter value (which is autoquoted)
		    tokenizer.setIsAutoquoting(true);
		    currentToken = tokenizer.nextToken();
		    tokenizer.setIsAutoquoting(false);
		    if (currentToken != MailcapTokenizer.STRING_TOKEN) {
			reportParseError(MailcapTokenizer.STRING_TOKEN,
			currentToken, tokenizer.getCurrentTokenValue());
		    }
		    String paramValue =
				tokenizer.getCurrentTokenValue();

		    // add the class to the list iff it is one we care about
		    if (paramName.startsWith("x-java-")) {
			String commandName = paramName.substring(7);
			//	7 == "x-java-".length

			//	setup the class entry list
			if (debug)
			    System.out.println("    Command: " + commandName +
						", Class: " + paramValue);
			Vector classes =
				(Vector)commands.get(commandName);
			if (classes == null) {
			    classes = new Vector();
			    commands.put(commandName, classes);
			}
			// classes.addElement(paramValue);
			classes.insertElementAt(paramValue, 0);
		    }

		    //	set up the next iteration
		    currentToken = tokenizer.nextToken();
		}
	    } while (currentToken == MailcapTokenizer.SEMICOLON_TOKEN);
	} else if (currentToken != MailcapTokenizer.EOI_TOKEN) {
	    reportParseError(MailcapTokenizer.EOI_TOKEN,
		MailcapTokenizer.SEMICOLON_TOKEN,
		currentToken, tokenizer.getCurrentTokenValue());
	}
     }

     protected static void reportParseError(int expectedToken, int actualToken,
		String actualTokenValue) throws MailcapParseException {
     	throw new MailcapParseException("Encountered a " +
		MailcapTokenizer.nameForToken(actualToken) + " token (" +
		actualTokenValue + ") while expecting a " +
		MailcapTokenizer.nameForToken(expectedToken) + " token.");
     }

     protected static void reportParseError(int expectedToken,
	int otherExpectedToken, int actualToken, String actualTokenValue)
					throws MailcapParseException {
     	throw new MailcapParseException("Encountered a " +
		MailcapTokenizer.nameForToken(actualToken) + " token (" +
		actualTokenValue + ") while expecting a " +
		MailcapTokenizer.nameForToken(expectedToken) + " or a " +
		MailcapTokenizer.nameForToken(otherExpectedToken) + " token.");
     }

     protected static void reportParseError(int expectedToken,
	    int otherExpectedToken, int anotherExpectedToken, int actualToken,
	    String actualTokenValue) throws MailcapParseException {
	if (debug)
	    System.out.println("PARSE ERROR: " + "Encountered a " +
		MailcapTokenizer.nameForToken(actualToken) + " token (" +
		actualTokenValue + ") while expecting a " +
		MailcapTokenizer.nameForToken(expectedToken) + ", a " +
		MailcapTokenizer.nameForToken(otherExpectedToken) + ", or a " +
		MailcapTokenizer.nameForToken(anotherExpectedToken) + " token.");
     	throw new MailcapParseException("Encountered a " +
		MailcapTokenizer.nameForToken(actualToken) + " token (" +
		actualTokenValue + ") while expecting a " +
		MailcapTokenizer.nameForToken(expectedToken) + ", a " +
		MailcapTokenizer.nameForToken(otherExpectedToken) + ", or a " +
		MailcapTokenizer.nameForToken(anotherExpectedToken) + " token.");
     }

     /** for debugging
     public static void	main(String[] args) throws Exception {
     	Hashtable masterHash = new Hashtable();
     	for (int i = 0; i < args.length; ++i) {
	    System.out.println("Entry " + i + ": " + args[i]);
	    parseLine(args[i], masterHash);
     	}

     	Enumeration types = masterHash.keys();
     	while (types.hasMoreElements()) {
	    String key = (String)types.nextElement();
	    System.out.println("MIME Type: " + key);

	    Hashtable commandHash = (Hashtable)masterHash.get(key);
	    Enumeration commands = commandHash.keys();
	    while (commands.hasMoreElements()) {
		String command = (String)commands.nextElement();
		System.out.println("    Command: " + command);

		Vector classes = (Vector)commandHash.get(command);
		for (int i = 0; i < classes.size(); ++i) {
			System.out.println("        Class: " +
					    (String)classes.elementAt(i));
		}
	    }

	    System.out.println("");
	}
    }
    */
}
