// Copyright 2020 The Go Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package ecdsa_test

import (
	"crypto"
	"crypto/ecdsa"
	"crypto/elliptic"
	"crypto/internal/boring"
	"crypto/rand"
	"crypto/x509"
	"testing"
)

func testEqual(t *testing.T, c elliptic.Curve) {
	private, _ := ecdsa.GenerateKey(c, rand.Reader)
	public := &private.PublicKey

	if !public.Equal(public) {
		t.Errorf("public key is not equal to itself: %v", public)
	}
	if !public.Equal(crypto.Signer(private).Public().(*ecdsa.PublicKey)) {
		t.Errorf("private.Public() is not Equal to public: %q", public)
	}
	if !private.Equal(private) {
		t.Errorf("private key is not equal to itself: %v", private)
	}

	enc, err := x509.MarshalPKCS8PrivateKey(private)
	if err != nil {
		t.Fatal(err)
	}
	decoded, err := x509.ParsePKCS8PrivateKey(enc)
	if err != nil {
		t.Fatal(err)
	}
	if !public.Equal(decoded.(crypto.Signer).Public()) {
		t.Errorf("public key is not equal to itself after decoding: %v", public)
	}
	if !private.Equal(decoded) {
		t.Errorf("private key is not equal to itself after decoding: %v", private)
	}

	other, _ := ecdsa.GenerateKey(c, rand.Reader)
	if public.Equal(other.Public()) {
		t.Errorf("different public keys are Equal")
	}
	if private.Equal(other) {
		t.Errorf("different private keys are Equal")
	}

	// Ensure that keys with the same coordinates but on different curves
	// aren't considered Equal.
	differentCurve := &ecdsa.PublicKey{}
	*differentCurve = *public // make a copy of the public key
	if differentCurve.Curve == elliptic.P256() {
		differentCurve.Curve = elliptic.P224()
	} else {
		differentCurve.Curve = elliptic.P256()
	}
	if public.Equal(differentCurve) {
		t.Errorf("public keys with different curves are Equal")
	}
}

func TestEqual(t *testing.T) {
	if !boring.Enabled() {
		t.Run("P224", func(t *testing.T) { testEqual(t, elliptic.P224()) })
	}
	if !boring.Enabled() && testing.Short() {
		return
	}
	t.Run("P256", func(t *testing.T) { testEqual(t, elliptic.P256()) })
	t.Run("P384", func(t *testing.T) { testEqual(t, elliptic.P384()) })
	t.Run("P521", func(t *testing.T) { testEqual(t, elliptic.P521()) })
}
