## Copyright (C) 2012 ABRT team <abrt-devel-list@redhat.com>
## Copyright (C) 2001-2005 Red Hat, Inc.

## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.

## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.

## You should have received a copy of the GNU General Public License
## along with this program; if not, write to the Free Software
## Foundation, Inc., 51 Franklin Street, Suite 500, Boston, MA  02110-1335  USA

import logging
import os
import sys
import traceback

# gnome-abrt
from gnome_abrt import errors
from gnome_abrt.l10n import _

class Controller(object):

    def __init__(self, sources, sigchld_assign=None):
        self.sources = sources
        self.run_event_fn = self._first_event_run
        self._sigchld_assign = sigchld_assign

    def report(self, problem):
        if not problem:
            logging.error("BUG: Controller: Can't report None problem")
            return

        self.run_event_fn("report-gui", problem)

    def delete(self, problem):
        if not problem:
            logging.error("BUG: Controller: Can't delete None problem")
            return

        problem.delete()

    def analyze(self, problem):
        if not problem:
            logging.error("BUG: Controller: Can't open detail of None problem")
            return

        self.run_event_fn("open-gui", problem)

    def _refresh_sources(self):
        for name, src in self.sources:
            try:
                src.refresh()
            except errors.UnavailableSource:
                logging.debug("Cannot refresh problem source {0}: {1}"
                        .format(name, traceback.format_exc()))

    def _first_event_run(self, event, problem):
        if self._sigchld_assign is not None:
            self._sigchld_assign(self._refresh_sources)

        self.run_event_fn = self._run_event_on_problem
        self.run_event_fn(event, problem)

    def _run_event_on_problem(self, event, problem):
        if problem.assure_ownership():
            pid = -1
            try:
                pid = os.fork()
            except OSError as ex:
                logging.debug(traceback.format_exc())
                logging.error(
                        _("Failed to execute the requested action because of"
                          " an error in new process creation"))
                return

            if pid == 0:
                exctbl = "/usr/local/libexec/abrt-handle-event"
                args = [exctbl, "-e", event, "--", problem.problem_id]
                try:
                    os.execvp(exctbl, args)
                #pylint: disable=W0703
                except Exception as ex:
                    logging.exception(ex)
                    sys.exit(1)
