#
# This file is part of pyasn1-modules software.
#
# Created by Russ Housley
# Copyright (c) 2019, Vigil Security, LLC
# License: http://snmplabs.com/pyasn1/license.html
#

import sys

from pyasn1.codec.der.decoder import decode as der_decode
from pyasn1.codec.der.encoder import encode as der_encode

from pyasn1_modules import pem
from pyasn1_modules import rfc5652
from pyasn1_modules import rfc8358

try:
    import unittest2 as unittest
except ImportError:
    import unittest


class P7STestCase(unittest.TestCase):
    pem_text_list = (
"""\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""",
"""\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""",
"""\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""",
)

    def setUp(self):
        self.asn1Spec = rfc5652.ContentInfo()

    def testDerCodec(self):
        oids = [ ]
        for pem_text in self.pem_text_list:
            substrate = pem.readBase64fromText(pem_text)
            asn1Object, rest = der_decode(substrate, asn1Spec=self.asn1Spec)
            assert not rest
            assert asn1Object.prettyPrint()
            assert der_encode(asn1Object) == substrate

            assert asn1Object['contentType'] == rfc5652.id_signedData
            sd, rest = der_decode(asn1Object['content'], asn1Spec=rfc5652.SignedData())
            assert not rest
            assert sd.prettyPrint()
            assert der_encode(sd) == asn1Object['content']

            oids.append(sd['encapContentInfo']['eContentType'])

        assert rfc8358.id_ct_asciiTextWithCRLF in oids
        assert rfc8358.id_ct_pdf in oids
        assert rfc8358.id_ct_xml in oids


suite = unittest.TestLoader().loadTestsFromModule(sys.modules[__name__])

if __name__ == '__main__':
    import sys

    result = unittest.TextTestRunner(verbosity=2).run(suite)
    sys.exit(not result.wasSuccessful())
