#!/bin/bash
# BEGIN_ICS_COPYRIGHT8 ****************************************
#
# Copyright (c) 2015-2020, Intel Corporation
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#     * Redistributions of source code must retain the above copyright notice,
#       this list of conditions and the following disclaimer.
#     * Redistributions in binary form must reproduce the above copyright
#       notice, this list of conditions and the following disclaimer in the
#       documentation and/or other materials provided with the distribution.
#     * Neither the name of Intel Corporation nor the names of its contributors
#       may be used to endorse or promote products derived from this software
#       without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# END_ICS_COPYRIGHT8   ****************************************


# [ICS VERSION STRING: unknown]

# disable the specified set of ports


# optional override of defaults
if [ -f /etc/eth-tools/ethfastfabric.conf ]
then
	. /etc/eth-tools/ethfastfabric.conf
fi

. /usr/lib/eth-tools/ethfastfabric.conf.def

. /usr/lib/eth-tools/ff_funcs

lidmap="$(mktemp --tmpdir lidmapXXXXXX)"
trap "rm -f $lidmap; exit 1" SIGHUP SIGTERM SIGINT
trap "rm -f $lidmap" EXIT

Usage_full()
{
	echo "Usage: opadisableports [-R] [-h hfi] [-p port] [reason] < disable.csv" >&2
	echo "              or" >&2
	echo "       opadisableports --help" >&2
	echo "   --help - produce full help text" >&2
	echo "   -R - do not attempt to get routes for computation of distance" >&2
	echo "        instead just disable switch port with lower LID assuming" >&2
	echo "        that will be closer to this node" >&2
	echo "   -h hfi - hfi, numbered 1..n, 0= -p port will be a system wide port num" >&2
	echo "            (default is 0)" >&2
	echo "   -p port - port, numbered 1..n, 0=1st active (default is 1st active)" >&2
	echo "   reason - optional text description of reason ports are being disabled," >&2
	echo "            will be saved in the reason field of the output file" >&2
	echo  >&2
	echo "The -h and -p options permit a variety of selections:" >&2
	echo "   -h 0       - 1st active port in system (this is the default)" >&2
	echo "   -h 0 -p 0  - 1st active port in system" >&2
	echo "   -h x       - 1st active port on HFI x" >&2
	echo "   -h x -p 0  - 1st active port on HFI x" >&2
	echo "   -h 0 -p y  - port y within system (irrespective of which ports are active)" >&2
	echo "   -h x -p y  - HFI x, port y" >&2

	echo  >&2
	echo "The disable.csv is an input file listing the links to disable.  It is of the" >&2 
	echo "form:" >&2
	echo "  NodeGUID;PortNum;NodeType;NodeDesc;NodeGUID;PortNum;NodeType;NodeDesc;Reason" >&2
	echo "For each listed link, the switch port closer to this node will be disabled." >&2
	echo "The Reason field is optional." >&2
	echo "An input file such as this can be generated by ethextractbadlinks or" >&2
	echo "ethextractsellinks." >&2
	echo >&2
	echo "Information about the links disabled (and the reason) will be saved (in same">&2
	echo "format) to an output file named $CONFIG_DIR/$FF_PRD_NAME/disabled:hfi:port.csv where">&2
	echo "the hfi:port part of the file name is replaced by the HFI number and the port">&2
	echo "number being operated on (such as 1:1 or 2:1).  This CSV file can be used as">&2
        echo "input to opaenableports." >&2
	echo >&2
	echo "for example:" >&2
	echo "   opadisableports 'bad cable' < disable.csv" >&2
	echo "   opadisableports -h 1 -p 1 'dead servers' < disable.csv" >&2
	echo "   ethextractsellinks -F lid:3 | opadisableports 'bad server'" >&2
	echo "   ethextractmissinglinks -T $CONFIG_DIR/$FF_PRD_NAME/topology.0:0.xml | opadisableports" >&2
	exit 0
}

Usage()
{
	echo "Usage: opadisableports [-R] [-h hfi] [-p port] [reason] < disable.csv" >&2
	echo "              or" >&2
	echo "       opadisableports --help" >&2
	echo "   --help - produce full help text" >&2
	echo "   -R - do not attempt to get routes for computation of distance" >&2
	echo "        instead just disable switch port with lower LID assuming" >&2
	echo "        that will be closer to this node" >&2
	echo "   -h hfi - hfi, numbered 1..n, 0= -p port will be a system wide port num" >&2
	echo "            (default is 0)" >&2
	echo "   -p port - port, numbered 1..n, 0=1st active (default is 1st active)" >&2
	echo "   reason - optional text description of reason ports are being disabled," >&2
	echo "            will be saved in the reason field of the output file" >&2
	echo  >&2
	echo "disable.csv is an input file listing the links to disable." >&2
	echo "It is of the form:" >&2
	echo "  NodeGUID;PortNum;NodeType;NodeDesc;NodeGUID;PortNum;NodeType;NodeDesc;Reason" >&2
	echo "For each listed link, the switch port closer to this node will be disabled." >&2
	echo "The Reason field is optional.">&2
	echo "An input file such as this can be generated by ethextractbadlinks or" >&2
	echo "ethextractsellinks." >&2
	echo >&2
	echo "Information about the links disabled (and the reason) will be saved (in same">&2
	echo "format) to an output file named $CONFIG_DIR/$FF_PRD_NAME/disabled:hfi:port.csv where">&2
	echo "the hfi:port part of the file name is replaced by the HFI number and the port">&2
	echo "number being operated on (such as 1:1 or 2:1)." >&2
	echo >&2
	echo "for example:" >&2
	echo "   opadisableports 'bad cable' < disable.csv" >&2
	exit 2
}

if [ x"$1" = "x--help" ]
then
	Usage_full
fi

res=0
reason=
reason_csv=
use_distance=y
hfi_input=0
port_input=0

while getopts Rh:p: param
do
	case $param in
	R)	use_distance=n;;
	p)	port_input="$OPTARG";;
	h)	hfi_input="$OPTARG";;
	?)	Usage;;
	esac
done
shift $((OPTIND -1))
if [ $# -ge 1 ]
then
	reason="$1"
	reason_csv=";$1"	# add field separator
	shift
fi
if [ $# -ge 1 ]
then
	Usage
fi

/usr/sbin/oparesolvehfiport $hfi_input $port_input &>/dev/null
if [ $? -ne 0 -o "$hfi_input" = "" -o "$port_input" = "" ]
then
	echo "opadisableports: Error: Invalid port specification: $hfi_input:$port_input" >&2
	exit 1
fi

hfi=$(/usr/sbin/oparesolvehfiport $hfi_input $port_input -o hfinum)
port=`echo $(/usr/sbin/oparesolvehfiport $hfi_input $port_input | cut -f 2 -d ':')`
port_opts="-h $hfi -p $port"

lookup_lid_by_guid_port()
{
	local nodeguid="$1"
	local portnum="$2"
	local guid dport type desc lid

	grep "^$nodeguid;$portnum;" < $lidmap|while read guid dport type desc lid
	do
		echo -n $lid
	done
}
lookup_guid_by_desc_port()
{
	local nodedesc="$1"
	local portnum="$2"

	local guid dport type desc lid

	grep "^[^;]*;$portnum;[^;]*;$nodedesc;[^;]*" < $lidmap | while read guid dport type desc lid
	do
		echo -n $guid
		return
	done
}

get_distance()
{
	local lid=$1
	local portguid

	if [ "$use_distance" = n ]
	then
		echo "0"
	else
		portguid=$(eval opasaquery $port_opts -o portguid -l $lid)
		eval opasaquery $port_opts -o trace -g $portguid|grep -c "NodeType:"
	fi
}

disable_ports()
{
	suffix=":$hfi:$port"

	# generate lidmap
	/usr/sbin/ethextractifids $port_opts > $lidmap
	if [ ! -s $lidmap ]
	then
		echo "opadisableports: Unable to determine fabric ifids" >&2
		rm -f $lidmap
		return 1
	fi

	disabled=0
	skipped=0
	failed=0
	IFS=';'
	while read nodeguid1 port1 type1 desc1 nodeguid2 port2 type2 desc2 rest
	do
		# For non ISLs pick the switch side, so we can reenable later
		if [ "$type1" = SW -a "$type2" != SW ]
		then
			if [ -z "$nodeguid1" ]
			then
				nodeguid1=$(lookup_guid_by_desc_port "$desc1" 0)
			fi
			lid=$(lookup_lid_by_guid_port $nodeguid1 0)
			distance=$(get_distance $lid)
			echo "$distance;$lid;$nodeguid1;$port1;$type1;$desc1;$nodeguid2;$port2;$type2;$desc2;$rest"
		elif [ "$type1" != SW -a "$type2" = SW ]
		then
			if [ -z "$nodeguid2" ]
			then
				nodeguid2=$(lookup_guid_by_desc_port "$desc2" 0)
			fi
			lid=$(lookup_lid_by_guid_port $nodeguid2 0)
			distance=$(get_distance $lid)
			echo "$distance;$lid;$nodeguid2;$port2;$type2;$desc2;$nodeguid1;$port1;$type1;$desc1;$rest"
		else
			# determine which side of ISL to disable
			if [ -z "$nodeguid1" ]
			then
				nodeguid1=$(lookup_guid_by_desc_port "$desc1" 0)
			fi
			if [ -z "$nodeguid2" ]
			then
				nodeguid2=$(lookup_guid_by_desc_port "$desc2" 0)
			fi
			lid1=$(lookup_lid_by_guid_port $nodeguid1 0)
			lid2=$(lookup_lid_by_guid_port $nodeguid2 0)
			if [ x"$lid1" != x -a x"$lid2" = x ]
			then
				# might be a re-disable case, use the LID we can still resolve
				distance=$(get_distance $lid1)
				echo "$distance;$lid1;$nodeguid1;$port1;$type1;$desc1;$nodeguid2;$port2;$type2;$desc2;$rest"
			elif [ x"$lid1" = x -a x"$lid2" != x ]
			then
				# might be a re-disable case, use the LID we can still resolve
				distance=$(get_distance $lid2)
				echo "$distance;$lid2;$nodeguid2;$port2;$type2;$desc2;$nodeguid1;$port1;$type1;$desc1;$rest"
			elif [ x"$lid1" = x -a x"$lid2" = x ]
			then
				# can't resolve
				lid=
				echo "0;;$nodeguid1;$port1;$type1;$desc1;$nodeguid2;$port2;$type2;$desc2;$rest"
			else
				distance1=$(get_distance $lid1)
				distance2=$(get_distance $lid2)
				if [ $(( $distance1 )) -lt $(( $distance2 )) ]
				then
					# pick side closer to FM
					echo "$distance1;$lid1;$nodeguid1;$port1;$type1;$desc1;$nodeguid2;$port2;$type2;$desc2;$rest"
				elif [ $(( $lid1 )) -eq $(( $lid2 )) ]
				then
					# this implies a switch is connected to itself
					# atypical config, for completeness pick lower port number
					if [ $(( $port1 )) -le $(( $port2 )) ]
					then
						echo "$distance1;$lid1;$nodeguid1;$port1;$type1;$desc1;$nodeguid2;$port2;$type2;$desc2;$rest"
					else
						echo "$distance2;$lid2;$nodeguid2;$port2;$type2;$desc2;$nodeguid1;$port1;$type1;$desc1;$rest"
					fi
				elif [ $(( $distance1 )) -eq $(( $distance2 )) ]
				then
					# two LIDs of an ISL are equal distance?
					# not expected, but for completeness, pick lower LID
					# as a tie breaker and assume it might be closer to FM
					# LID == tested above, but use -le to be paranoid
					if [ $(( $lid1 )) -le $(( $lid2 )) ]
					then
						echo "$distance1;$lid1;$nodeguid1;$port1;$type1;$desc1;$nodeguid2;$port2;$type2;$desc2;$rest"
					else
						echo "$distance2;$lid2;$nodeguid2;$port2;$type2;$desc2;$nodeguid1;$port1;$type1;$desc1;$rest"
					fi
				else
					# pick side closer to FM
					echo "$distance2;$lid2;$nodeguid2;$port2;$type2;$desc2;$nodeguid1;$port1;$type1;$desc1;$rest"
				fi
			fi
		fi
	done | if [ "$use_distance" = y ]
	then
		sort -r -n -t ';' -k 1	# do furthest from this node first
	else
		cat	# do in order of input file
	fi	|
	{
	while read distance lid guid dport type desc lguid lport ltype ldesc rest
	do
		if [ x"$rest" != x ]
		then
			rest=";$rest"	# add field separator
		fi

		if [ x"$lid" = x ]
		then
			echo "Skipping link: $desc:$dport -> $ldesc:$lport"
			skipped=$(( $skipped + 1))
		else
			echo "Disabling link: $desc:$dport -> $ldesc:$lport"
			eval /usr/sbin/opaportconfig $port_opts -l $lid -m $dport disable

			if [ $? = 0 ]
			then
				logger -p user.err "Disabled link: $desc:$dport -> $ldesc:$lport due to: $reason"
				disabled=$(( $disabled + 1))
				if [ ! -e $CONFIG_DIR/$FF_PRD_NAME/disabled$suffix.csv ] || ! grep "^$guid;$dport;" < $CONFIG_DIR/$FF_PRD_NAME/disabled$suffix.csv > /dev/null 2>&1
				then
					# keep a disabled file per local hfi:port
					# same format as our input but 1st port
					# indicates which one was disabled and
					# second port is only for info
					# if fed back into this tool will select
					# same port to disable
					echo "$guid;$dport;$type;$desc;$lguid;$lport;$ltype;$ldesc$rest$reason_csv" >> $CONFIG_DIR/$FF_PRD_NAME/disabled$suffix.csv
				fi
			else
				failed=$(( $failed + 1))
			fi
		fi
	done
	if [ $failed -eq 0 ]
	then
		echo "Disabled: $disabled; Skipped: $skipped"
		return 0
	else
		echo "Disabled: $disabled; Skipped: $skipped; Failed: $failed"
		return 1
	fi
	}
}

disable_ports "$hfi" "$port"
if [ $? -ne 0 ]
then
	res=1
fi

exit $res
