﻿// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE file in the project root for full license information.

using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Resources;
using System.Reflection;
using System.Diagnostics;
using System.Diagnostics.CodeAnalysis;
using System.Globalization;
#if FEATURE_RESGENCACHE
using System.Runtime.Serialization;
using System.Runtime.Serialization.Formatters.Binary;
#endif
using Microsoft.Build.Framework;
using Microsoft.Build.Shared;
#if FEATURE_COM_INTEROP
using Microsoft.Win32;
#endif
using System.CodeDom;
using System.CodeDom.Compiler;
using System.Xml;
using System.Runtime.InteropServices;
#if FEATURE_SYSTEM_CONFIGURATION
using System.Configuration;
#endif
using System.Security;
#if FEATURE_RESX_RESOURCE_READER
using System.ComponentModel.Design;
#endif
#if FEATURE_APPDOMAIN
using System.Runtime.Remoting;
#endif

#if (!STANDALONEBUILD)
using Microsoft.Internal.Performance;
#endif
using System.Runtime.Versioning;

using Microsoft.Build.Utilities;
using System.Xml.Linq;
using Microsoft.Build.Shared.FileSystem;

namespace Microsoft.Build.Tasks
{
    /// <summary>
    /// This class defines the "GenerateResource" MSBuild task, which enables using resource APIs
    /// to transform resource files.
    /// </summary>
    [RequiredRuntime("v2.0")]
    public sealed partial class GenerateResource : TaskExtension
    {

#if !FEATURE_CODEDOM
        private readonly string CSharpLanguageName = "CSharp";
        private readonly string VisualBasicLanguageName = "VisualBasic";
#endif


        #region Fields

#if FEATURE_RESGENCACHE
        // This cache helps us track the linked resource files listed inside of a resx resource file
        private ResGenDependencies _cache;
#endif

        // This is where we store the list of input files/sources
        private ITaskItem[] _sources = null;

        // Indicates whether the resource reader should use the source file's
        // directory to resolve relative file paths.
        private bool _useSourcePath = false;

        // This is needed for the actual items from the project
        private ITaskItem[] _references = null;

        // Any additional inputs to dependency checking.
        private ITaskItem[] _additionalInputs = null;

        // This is the path/name of the dependency cache file
        private ITaskItem _stateFile = null;

        // This list is all of the resource file(s) generated by the task
        private ITaskItem[] _outputResources = null;

        // List of those output resources that were not actually created, due to an error
        private ArrayList _unsuccessfullyCreatedOutFiles = new ArrayList();

        // Storage for names of *all files* written to disk.
        private ArrayList _filesWritten = new ArrayList();

        // StronglyTypedLanguage
        private string _stronglyTypedLanguage = null;

        // StronglyTypedNamespace
        private string _stronglyTypedNamespace = null;

        // StronglyTypedManifestPrefix
        private string _stronglyTypedManifestPrefix = null;

        // StronglyTypedClassName
        private string _stronglyTypedClassName = null;

        // StronglyTypedFileName
        private string _stronglyTypedFileName = null;

        // Whether the STR class should have public members; default is false
        private bool _publicClass = false;

        // Did the CodeDOM succeed when creating any Strongly Typed Resource class?
        private bool _stronglyTypedResourceSuccessfullyCreated = false;

        // When true, a separate AppDomain is always created.
        private bool _neverLockTypeAssemblies = false;

        private bool _foundNewestUncorrelatedInputWriteTime = false;

        private DateTime _newestUncorrelatedInputWriteTime;

        // The targets may pass in the path to the SDKToolsPath. If so this should be used to generate the commandline 
        // for logging purposes.  Also, when ExecuteAsTool is true, it determines where the system goes looking for resgen.exe
        private string _sdkToolsPath;

        // True if the resource generation should be sent out-of-proc to resgen.exe; false otherwise.  Defaults to true
        // because we want to execute out-of-proc when ToolsVersion is < 4.0, and the earlier targets files don't know
        // about this property.
        private bool _executeAsTool = true;

        // Path to resgen.exe
        private string _resgenPath;

        // table of already seen types by their typename
        // note the use of the ordinal comparer that matches the case sensitive Type.GetType usage
        private Dictionary<string, Type> _typeTable = new Dictionary<string, Type>(StringComparer.Ordinal);

        /// <summary>
        /// Table of aliases for types defined in resx / resw files
        /// Ordinal comparer matches ResXResourceReader's use of a HashTable. 
        /// </summary>
        private Dictionary<string, string> _aliases = new Dictionary<string, string>(StringComparer.Ordinal);

        // Our calculation is not quite correct. Using a number substantially less than 32768 in order to
        // be sure we don't exceed it.
        private static int s_maximumCommandLength = 28000;

        // Contains the list of paths from which inputs will not be taken into account during up-to-date check.  
        private ITaskItem[] _excludedInputPaths;

#if FEATURE_APPDOMAIN
        /// <summary>
        /// The task items that we remoted across the appdomain boundary
        /// we use this list to disconnect the task items once we're done.
        /// </summary>
        private List<ITaskItem> _remotedTaskItems;
#endif

        /// <summary>
        /// Satellite input assemblies.
        /// </summary>
        private List<ITaskItem> _satelliteInputs;

        #endregion  // fields

        #region Properties

        /// <summary>
        /// The names of the items to be converted. The extension must be one of the
        /// following: .txt, .resx or .resources.
        /// </summary>
        [Required]
        [Output]
        public ITaskItem[] Sources
        {
            set { _sources = value; }
            get { return _sources; }
        }

        /// <summary>
        /// Indicates whether the resource reader should use the source file's directory to
        /// resolve relative file paths.
        /// </summary>
        public bool UseSourcePath
        {
            set { _useSourcePath = value; }
            get { return _useSourcePath; }
        }

        /// <summary>
        /// Resolves types in ResX files (XML resources) for Strongly Typed Resources
        /// </summary>
        public ITaskItem[] References
        {
            set { _references = value; }
            get { return _references; }
        }

        /// <summary>
        /// Additional inputs to the dependency checking done by this task. For example,
        /// the project and targets files typically should be inputs, so that if they are updated,
        /// all resources are regenerated.
        /// </summary>
        public ITaskItem[] AdditionalInputs
        {
            set { _additionalInputs = value; }
            get { return _additionalInputs; }
        }

        /// <summary>
        /// This is the path/name of the file containing the dependency cache
        /// </summary>
        public ITaskItem StateFile
        {
            set { _stateFile = value; }
            get { return _stateFile; }
        }

        /// <summary>
        /// The name(s) of the resource file to create. If the user does not specify this
        /// attribute, the task will append a .resources extension to each input filename
        /// argument and write the file to the directory that contains the input file.
        /// Includes any output files that were already up to date, but not any output files
        /// that failed to be written due to an error.
        /// </summary>
        [Output]
        public ITaskItem[] OutputResources
        {
            set { _outputResources = value; }
            get { return _outputResources; }
        }

        /// <summary>
        /// Storage for names of *all files* written to disk.  This is part of the implementation
        /// for Clean, and contains the OutputResources items and the StateFile item.
        /// Includes any output files that were already up to date, but not any output files
        /// that failed to be written due to an error. 
        /// </summary>
        [Output]
        public ITaskItem[] FilesWritten
        {
            get
            {
                return (ITaskItem[])_filesWritten.ToArray(typeof(ITaskItem));
            }
        }

        /// <summary>
        /// The language to use when generating the class source for the strongly typed resource.
        /// This parameter must match exactly one of the languages used by the CodeDomProvider.
        /// </summary>
        public string StronglyTypedLanguage
        {
            set
            {
                // Since this string is passed directly into the framework, we don't want to
                // try to validate it -- that might prevent future expansion of supported languages.
                _stronglyTypedLanguage = value;
            }
            get { return _stronglyTypedLanguage; }
        }

        /// <summary>
        /// Specifies the namespace to use for the generated class source for the
        /// strongly typed resource. If left blank, no namespace is used.
        /// </summary>
        public string StronglyTypedNamespace
        {
            set { _stronglyTypedNamespace = value; }
            get { return _stronglyTypedNamespace; }
        }

        /// <summary>
        /// Specifies the resource namespace or manifest prefix to use in the generated 
        /// class source for the strongly typed resource. 
        /// </summary>
        public string StronglyTypedManifestPrefix
        {
            set { _stronglyTypedManifestPrefix = value; }
            get { return _stronglyTypedManifestPrefix; }
        }

        /// <summary>
        /// Specifies the class name for the strongly typed resource class.  If left blank, the base
        /// name of the resource file is used.
        /// </summary>
        [Output]
        public string StronglyTypedClassName
        {
            set { _stronglyTypedClassName = value; }
            get { return _stronglyTypedClassName; }
        }

        /// <summary>
        /// Specifies the filename for the source file.  If left blank, the name of the class is
        /// used as the base filename, with the extension dependent on the language.
        /// </summary>
        [Output]
        public string StronglyTypedFileName
        {
            set { _stronglyTypedFileName = value; }
            get { return _stronglyTypedFileName; }
        }

        /// <summary>
        /// Specifies whether the strongly typed class should be created public (with public methods)
        /// instead of the default internal. Analogous to resgen.exe's /publicClass switch.
        /// </summary>
        public bool PublicClass
        {
            set { _publicClass = value; }
            get { return _publicClass; }
        }

        /// <summary>
        /// Whether this rule is generating .resources files or extracting .ResW files from assemblies.
        /// Requires some additional input filtering.
        /// </summary>
        public bool ExtractResWFiles
        {
            get;
            set;
        }

        /// <summary>
        /// (default = false)
        /// When true, a new AppDomain is always created to evaluate the .resx files.
        /// When false, a new AppDomain is created only when it looks like a user's 
        ///  assembly is referenced by the .resx.
        /// </summary>
        public bool NeverLockTypeAssemblies
        {
            set { _neverLockTypeAssemblies = value; }
            get { return _neverLockTypeAssemblies; }
        }

        /// <summary>
        /// Even though the generate resource task will do the processing in process, a logging message is still generated. This logging message
        /// will include the path to the windows SDK. Since the targets now will pass in the Windows SDK path we should use this for logging.
        /// </summary>
        public string SdkToolsPath
        {
            get { return _sdkToolsPath; }
            set { _sdkToolsPath = value; }
        }

        /// <summary>
        /// Property to allow multitargeting of ResolveComReferences:  If true, tlbimp.exe and 
        /// aximp.exe from the appropriate target framework will be run out-of-proc to generate
        /// the necessary wrapper assemblies.  
        /// </summary>
        public bool ExecuteAsTool
        {
            set { _executeAsTool = value; }
            get { return _executeAsTool; }
        }

        /// <summary>
        /// Array of equals-separated pairs of environment
        /// variables that should be passed to the spawned resgen.exe,
        /// in addition to (or selectively overriding) the regular environment block.
        /// These aren't currently used when resgen is run in-process.
        /// </summary>
        public string[] EnvironmentVariables
        {
            get;
            set;
        }

        /// <summary>
        /// That set of paths from which tracked inputs will be ignored during
        /// Up to date checking
        /// </summary>
        public ITaskItem[] ExcludedInputPaths
        {
            get { return _excludedInputPaths; }
            set { _excludedInputPaths = value; }
        }

        /// <summary>
        /// Property used to set whether tracked incremental build will be used. If true, 
        /// incremental build is turned on; otherwise, a rebuild will be forced.  
        /// </summary>
        public bool MinimalRebuildFromTracking
        {
            get
            {
                // not using tracking anymore
                return false;
            }

            set
            {
                // do nothing 
            }
        }

        /// <summary>
        /// True if we should be tracking file access patterns - necessary for incremental 
        /// build support.
        /// </summary>
        public bool TrackFileAccess
        {
            get
            {
                // not using tracking anymore
                return false;
            }
            set
            {
                // do nothing 
            }
        }

        /// <summary>
        /// Names of the read tracking logs.  
        /// </summary>
        [SuppressMessage("Microsoft.Naming", "CA1702:CompoundWordsShouldBeCasedCorrectly", MessageId = "TLog", Justification = "Has now shipped as public API; plus it's unclear whether 'Tlog' or 'TLog' is the preferred casing")]
        public ITaskItem[] TLogReadFiles
        {
            get
            {
                return Array.Empty<ITaskItem>();
            }
        }

        /// <summary>
        /// Names of the write tracking logs.
        /// </summary>
        [SuppressMessage("Microsoft.Naming", "CA1702:CompoundWordsShouldBeCasedCorrectly", MessageId = "TLog", Justification = "Has now shipped as public API; plus it's unclear whether 'Tlog' or 'TLog' is the preferred casing")]
        public ITaskItem[] TLogWriteFiles
        {
            get
            {
                return Array.Empty<ITaskItem>();
            }
        }

        /// <summary>
        /// Intermediate directory into which the tracking logs from running this task will be placed.  
        /// </summary>
        public string TrackerLogDirectory
        {
            get
            {
                return String.Empty;
            }
            set
            {
                // do nothing
            }
        }

        /// <summary>
        /// Microsoft.Build.Utilities.ExecutableType of ResGen.exe.  Used to determine whether or not 
        /// Tracker.exe needs to be used to spawn ResGen.exe.  If empty, uses a heuristic to determine
        /// a default architecture. 
        /// </summary>
        public string ToolArchitecture
        {
            get
            {
                return String.Empty;
            }

            set
            {
                // do nothing
            }
        }

        /// <summary>
        /// Path to the appropriate .NET Framework location that contains FileTracker.dll.  If set, the user 
        /// takes responsibility for making sure that the bitness of the FileTracker.dll that they pass matches 
        /// the bitness of the ResGen.exe that they intend to use. If not set, the task decides the appropriate
        /// location based on the current .NET Framework version. 
        /// </summary>
        /// <comments>
        /// Should only need to be used in partial or full checked in toolset scenarios. 
        /// </comments>
        public string TrackerFrameworkPath
        {
            get
            {
                return String.Empty;
            }

            set
            {
                // do nothing
            }
        }

        /// <summary>
        /// Path to the appropriate Windows SDK location that contains Tracker.exe.  If set, the user takes 
        /// responsibility for making sure that the bitness of the Tracker.exe that they pass matches the 
        /// bitness of the ResGen.exe that they intend to use. If not set, the task decides the appropriate 
        /// location based on the current Windows SDK. 
        /// </summary>
        /// <comments>
        /// Should only need to be used in partial or full checked in toolset scenarios. 
        /// </comments>
        public string TrackerSdkPath
        {
            get
            {
                return String.Empty;
            }

            set
            {
                // do nothing
            }
        }

        /// <summary>
        /// Where to extract ResW files.  (Could be the intermediate directory.)
        /// </summary>
        public string OutputDirectory
        {
            get;
            set;
        }

        #endregion // properties

        /// <summary>
        /// Simple public constructor.
        /// </summary>
        public GenerateResource()
        {
            // do nothing
        }

#if FEATURE_COM_INTEROP
        /// <summary>
        /// Static constructor checks the registry opt-out for mark-of-the-web rejection.
        /// </summary>
        static GenerateResource()
        {
            try
            {
                object allowUntrustedFiles = Registry.GetValue(@"HKEY_LOCAL_MACHINE\SOFTWARE\Microsoft\.NETFramework\SDK", "AllowProcessOfUntrustedResourceFiles", null);
                if (allowUntrustedFiles is String)
                {
                    allowMOTW = ((string)allowUntrustedFiles).Equals("true", StringComparison.OrdinalIgnoreCase);
                }
            }
            catch { }
        }
#endif

        /// <summary>
        /// Logs a Resgen.exe command line that indicates what parameters were
        /// passed to this task. Since this task is replacing Resgen, and we used
        /// to log the Resgen.exe command line, we need to continue logging an
        /// equivalent command line.
        /// </summary>
        /// <param name="inputFiles"></param>
        /// <param name="outputFiles"></param>
        private void LogResgenCommandLine(List<ITaskItem> inputFiles, List<ITaskItem> outputFiles)
        {
            CommandLineBuilderExtension commandLineBuilder = new CommandLineBuilderExtension();

            // start the command line with the path to Resgen.exe
            commandLineBuilder.AppendFileNameIfNotNull(Path.Combine(_resgenPath, "resgen.exe"));

            GenerateResGenCommandLineWithoutResources(commandLineBuilder);

            if (StronglyTypedLanguage == null)
            {
                // append the resources to compile
                for (int i = 0; i < inputFiles.Count; ++i)
                {
                    if (!ExtractResWFiles)
                    {
                        commandLineBuilder.AppendFileNamesIfNotNull
                        (
                            new string[] { inputFiles[i].ItemSpec, outputFiles[i].ItemSpec },
                            ","
                        );
                    }
                    else
                    {
                        commandLineBuilder.AppendFileNameIfNotNull(inputFiles[i].ItemSpec);
                    }
                }
            }
            else
            {
                // append the resource to compile
                commandLineBuilder.AppendFileNamesIfNotNull(inputFiles.ToArray(), " ");
                commandLineBuilder.AppendFileNamesIfNotNull(outputFiles.ToArray(), " ");

                // append the strongly-typed resource details
                commandLineBuilder.AppendSwitchIfNotNull
                (
                    "/str:",
                    new string[] { StronglyTypedLanguage, StronglyTypedNamespace, StronglyTypedClassName, StronglyTypedFileName },
                    ","
                );
            }

            Log.LogCommandLine(MessageImportance.Low, commandLineBuilder.ToString());
        }

        /// <summary>
        /// Generate the parts of the resgen command line that are don't involve resgen.exe itself or the 
        /// resources to be generated. 
        /// </summary>
        /// <comments>
        /// Expects resGenCommand to be non-null -- otherwise, it doesn't get passed back to the caller, so it's 
        /// useless anyway.
        /// </comments>
        /// <param name="resGenCommand"></param>
        private void GenerateResGenCommandLineWithoutResources(CommandLineBuilderExtension resGenCommand)
        {
            // Throw an internal error, since this method should only ever get called by other aspects of this task, not
            // anything that the user touches. 
            ErrorUtilities.VerifyThrowInternalNull(resGenCommand, "resGenCommand");

            // append the /useSourcePath flag if requested.
            if (UseSourcePath)
            {
                resGenCommand.AppendSwitch("/useSourcePath");
            }

            // append the /publicClass flag if requested
            if (PublicClass)
            {
                resGenCommand.AppendSwitch("/publicClass");
            }

            // append the references, if any
            if (References != null)
            {
                foreach (ITaskItem reference in References)
                {
                    resGenCommand.AppendSwitchIfNotNull("/r:", reference);
                }
            }

            // append /compile switch if not creating strongly typed class
            if (String.IsNullOrEmpty(StronglyTypedLanguage))
            {
                resGenCommand.AppendSwitch("/compile");
            }
        }

        /// <summary>
        /// This is the main entry point for the GenerateResource task.
        /// </summary>
        /// <returns>true, if task executes successfully</returns>
        public override bool Execute()
        {
            bool outOfProcExecutionSucceeded = true;
#if (!STANDALONEBUILD)
            using (new CodeMarkerStartEnd(CodeMarkerEvent.perfMSBuildGenerateResourceBegin, CodeMarkerEvent.perfMSBuildGenerateResourceEnd))
#endif
            {
                // If we're extracting ResW files from assemblies (instead of building resources),
                // our Sources can contain PDB's, pictures, and other non-DLL's.  Prune that list.  
                // .NET Framework assemblies are not included.  However, other Microsoft ones
                // such as MSTestFramework may be included (resolved from GetSDKReferenceFiles).
                if (ExtractResWFiles && Sources != null)
                {
                    _satelliteInputs = new List<ITaskItem>();

                    List<ITaskItem> newSources = new List<ITaskItem>();
                    foreach (ITaskItem item in Sources)
                    {
                        if (item.ItemSpec.EndsWith(".dll", StringComparison.OrdinalIgnoreCase))
                        {
                            if (item.ItemSpec.EndsWith(".resources.dll", StringComparison.OrdinalIgnoreCase))
                            {
                                _satelliteInputs.Add(item);
                            }
                            else
                            {
                                newSources.Add(item);
                            }
                        }
                    }
                    Sources = newSources.ToArray();
                }


                // If there are no sources to process, just return (with success) and report the condition.
                if ((Sources == null) || (Sources.Length == 0))
                {
                    Log.LogMessageFromResources(MessageImportance.Low, "GenerateResource.NoSources");
                    // Indicate we generated nothing
                    OutputResources = null;
                    return true;
                }

                if (!ValidateParameters())
                {
                    // Indicate we generated nothing
                    OutputResources = null;
                    return false;
                }

                // In the case that OutputResources wasn't set, build up the outputs by transforming the Sources
                // However if we are extracting ResW files, we cannot easily tell which files we'll produce up front,
                // without loading each assembly.
                if (!ExtractResWFiles && !CreateOutputResourcesNames())
                {
                    // Indicate we generated nothing
                    OutputResources = null;
                    return false;
                }

                List<ITaskItem> inputsToProcess;
                List<ITaskItem> outputsToProcess;
                List<ITaskItem> cachedOutputFiles;  // For incremental builds, this is the set of already-existing, up to date files.

                GetResourcesToProcess(out inputsToProcess, out outputsToProcess, out cachedOutputFiles);

                if (inputsToProcess.Count == 0 && !Log.HasLoggedErrors)
                {
                    if (cachedOutputFiles.Count > 0)
                    {
                        OutputResources = cachedOutputFiles.ToArray();
                    }

                    Log.LogMessageFromResources("GenerateResource.NothingOutOfDate");
                }
                else
                {
                    if (!ComputePathToResGen())
                    {
                        // unable to compute the path to resgen.exe and that is necessary to 
                        // continue forward, so return now.
                        return false;
                    }

                    // Check for the mark of the web on all possibly-exploitable files
                    // to be processed.
                    bool dangerousResourceFound = false;

                    foreach (ITaskItem source in _sources)
                    {
                        if (IsDangerous(source.ItemSpec))
                        {
                            Log.LogErrorWithCodeFromResources("GenerateResource.MOTW", source.ItemSpec);
                            dangerousResourceFound = true;
                        }
                    }

                    if (dangerousResourceFound)
                    {
                        // Do no further processing
                        return false;
                    }

                    if (ExecuteAsTool)
                    {
                        outOfProcExecutionSucceeded = GenerateResourcesUsingResGen(inputsToProcess, outputsToProcess);
                    }
                    else // Execute in-proc (or in a separate appdomain if necessary)
                    {
                        // Log equivalent command line as this is a convenient way to log all the references, etc
                        // even though we're not actually running resgen.exe
                        LogResgenCommandLine(inputsToProcess, outputsToProcess);

#if FEATURE_APPDOMAIN
                        // Figure out whether a separate AppDomain is required because an assembly would be locked.
                        bool needSeparateAppDomain = NeedSeparateAppDomain();

                        AppDomain appDomain = null;
#endif
                        ProcessResourceFiles process = null;

                        try
                        {
#if FEATURE_APPDOMAIN
                            if (needSeparateAppDomain)
                            {
                                // we're going to be remoting across the appdomain boundary, so
                                // create the list that we'll use to disconnect the taskitems once we're done
                                _remotedTaskItems = new List<ITaskItem>();

                                appDomain = AppDomain.CreateDomain
                                (
                                    "generateResourceAppDomain",
                                    null,
                                    AppDomain.CurrentDomain.SetupInformation
                                );

                                object obj = appDomain.CreateInstanceFromAndUnwrap
                                   (
                                       typeof(ProcessResourceFiles).Module.FullyQualifiedName,
                                       typeof(ProcessResourceFiles).FullName
                                   );

                                Type processType = obj.GetType();
                                ErrorUtilities.VerifyThrow(processType == typeof(ProcessResourceFiles), "Somehow got a wrong and possibly incompatible type for ProcessResourceFiles.");

                                process = (ProcessResourceFiles)obj;

                                RecordItemsForDisconnectIfNecessary(_references);
                                RecordItemsForDisconnectIfNecessary(inputsToProcess);
                                RecordItemsForDisconnectIfNecessary(outputsToProcess);
                            }
                            else
#endif
                            {
                                process = new ProcessResourceFiles();
                            }

                            process.Run(Log, _references, inputsToProcess, _satelliteInputs, outputsToProcess, UseSourcePath,
                                StronglyTypedLanguage, _stronglyTypedNamespace, _stronglyTypedManifestPrefix,
                                StronglyTypedFileName, StronglyTypedClassName, PublicClass,
                                ExtractResWFiles, OutputDirectory);

                            this.StronglyTypedClassName = process.StronglyTypedClassName; // in case a default was chosen
                            this.StronglyTypedFileName = process.StronglyTypedFilename;   // in case a default was chosen
                            _stronglyTypedResourceSuccessfullyCreated = process.StronglyTypedResourceSuccessfullyCreated;
                            if (null != process.UnsuccessfullyCreatedOutFiles)
                            {
                                foreach (string item in process.UnsuccessfullyCreatedOutFiles)
                                {
                                    _unsuccessfullyCreatedOutFiles.Add(item);
                                }
                            }

                            if (ExtractResWFiles)
                            {
                                ITaskItem[] outputResources = process.ExtractedResWFiles.ToArray();
#if FEATURE_APPDOMAIN
                                if (needSeparateAppDomain)
                                {
                                    // Ensure we can unload the other AppDomain, yet still use the
                                    // ITaskItems we got back.  Clone them.
                                    outputResources = CloneValuesInThisAppDomain(outputResources);
                                }
#endif

                                if (cachedOutputFiles.Count > 0)
                                {
                                    OutputResources = new ITaskItem[outputResources.Length + cachedOutputFiles.Count];
                                    outputResources.CopyTo(OutputResources, 0);
                                    cachedOutputFiles.CopyTo(OutputResources, outputResources.Length);
                                }
                                else
                                {
                                    OutputResources = outputResources;
                                }

#if FEATURE_RESGENCACHE
                                // Get portable library cache info (and if needed, marshal it to this AD).
                                List<ResGenDependencies.PortableLibraryFile> portableLibraryCacheInfo = process.PortableLibraryCacheInfo;
                                for (int i = 0; i < portableLibraryCacheInfo.Count; i++)
                                {
                                    _cache.UpdatePortableLibrary(portableLibraryCacheInfo[i]);
                                }
#endif
                            }

                            process = null;
                        }
                        finally
                        {
#if FEATURE_APPDOMAIN
                            if (needSeparateAppDomain && appDomain != null)
                            {
                                Log.MarkAsInactive();

                                AppDomain.Unload(appDomain);
                                process = null;
                                appDomain = null;

                                // if we've been asked to remote these items then
                                // we need to disconnect them from .NET Remoting now we're all done with them
                                if (_remotedTaskItems != null)
                                {
                                    foreach (ITaskItem item in _remotedTaskItems)
                                    {
                                        if (item is MarshalByRefObject)
                                        {
                                            // Tell remoting to forget connections to the taskitem
                                            RemotingServices.Disconnect((MarshalByRefObject)item);
                                        }
                                    }
                                }

                                _remotedTaskItems = null;
                            }
#endif
                        }
                    }
                }

#if FEATURE_RESGENCACHE
                // And now we serialize the cache to save our resgen linked file resolution for later use.
                WriteStateFile();
#endif

                RemoveUnsuccessfullyCreatedResourcesFromOutputResources();

                RecordFilesWritten();
            }

            return !Log.HasLoggedErrors && outOfProcExecutionSucceeded;
        }

#if FEATURE_COM_INTEROP
        private static bool allowMOTW;

        private const string CLSID_InternetSecurityManager = "7b8a2d94-0ac9-11d1-896c-00c04fb6bfc4";

        private const uint ZoneLocalMachine = 0;

        private const uint ZoneIntranet = 1;

        private const uint ZoneTrusted = 2;

        private const uint ZoneInternet = 3;

        private const uint ZoneUntrusted = 4;

        private static IInternetSecurityManager internetSecurityManager = null;

        // Resources can have arbitrarily serialized objects in them which can execute arbitrary code
        // so check to see if we should trust them before analyzing them
        private bool IsDangerous(String filename)
        {
            // If they are opted out, there's no work to do
            if (allowMOTW)
            {
                return false;
            }

            // First check the zone, if they are not an untrusted zone, they aren't dangerous

            if (internetSecurityManager == null)
            {
                Type iismType = Type.GetTypeFromCLSID(new Guid(CLSID_InternetSecurityManager));
                internetSecurityManager = (IInternetSecurityManager)Activator.CreateInstance(iismType);
            }

            Int32 zone = 0;
            internetSecurityManager.MapUrlToZone(Path.GetFullPath(filename), out zone, 0);
            if (zone < ZoneInternet)
            {
                return false;
            }

            // By default all file types that get here are considered dangerous
            bool dangerous = true;

            if (String.Equals(Path.GetExtension(filename), ".resx", StringComparison.OrdinalIgnoreCase) ||
                String.Equals(Path.GetExtension(filename), ".resw", StringComparison.OrdinalIgnoreCase))
            {
                // XML files are only dangerous if there are unrecognized objects in them
                dangerous = false;

                FileStream stream = new FileStream(filename, FileMode.Open, FileAccess.Read, FileShare.Read);
                XmlTextReader reader = new XmlTextReader(stream);
                reader.DtdProcessing = DtdProcessing.Ignore;
                reader.XmlResolver = null;
                try
                {
                    while (reader.Read())
                    {
                        if (reader.NodeType == XmlNodeType.Element)
                        {
                            string s = reader.LocalName;

                            // We only want to parse data nodes,
                            // the mimetype attribute gives the serializer
                            // that's requested.
                            if (reader.LocalName.Equals("data"))
                            {
                                if (reader["mimetype"] != null)
                                {
                                    dangerous = true;
                                }
                            }
                            else if (reader.LocalName.Equals("metadata"))
                            {
                                if (reader["mimetype"] != null)
                                {
                                    dangerous = true;
                                }
                            }
                        }
                    }
                }
                catch
                {
                    // If we hit an error while parsing assume there's a dangerous type in this file.
                    dangerous = true;
                }
                stream.Close();
            }

            return dangerous;
        }
#else
        private bool IsDangerous(String filename)
        {
            return false;
        }
#endif

        /// <summary>
        /// For setting OutputResources and ensuring it can be read after the second AppDomain has been unloaded.
        /// </summary>
        /// <param name="remoteValues">ITaskItems in another AppDomain</param>
        /// <returns></returns>
        private static ITaskItem[] CloneValuesInThisAppDomain(IList<ITaskItem> remoteValues)
        {
            ITaskItem[] clonedOutput = new ITaskItem[remoteValues.Count];
            for (int i = 0; i < remoteValues.Count; i++)
            {
                clonedOutput[i] = new TaskItem(remoteValues[i]);
            }

            return clonedOutput;
        }

#if FEATURE_APPDOMAIN
        /// <summary>
        /// Remember this TaskItem so that we can disconnect it when this Task has finished executing
        /// Only if we're passing TaskItems to another AppDomain is this necessary. This call
        /// Will make that determination for you.
        /// </summary>
        private void RecordItemsForDisconnectIfNecessary(IEnumerable<ITaskItem> items)
        {
            if (_remotedTaskItems != null && items != null)
            {
                // remember that we need to disconnect these items
                _remotedTaskItems.AddRange(items);
            }
        }
#endif

        /// <summary>
        /// Computes the path to ResGen.exe for use in logging and for passing to the 
        /// nested ResGen task.
        /// </summary>
        /// <returns>True if the path is found (or it doesn't matter because we're executing in memory), false otherwise</returns>
        private bool ComputePathToResGen()
        {
#if FEATURE_RESGEN
            _resgenPath = null;

            if (String.IsNullOrEmpty(_sdkToolsPath))
            {
                _resgenPath = ToolLocationHelper.GetPathToDotNetFrameworkSdkFile("resgen.exe", TargetDotNetFrameworkVersion.Version35);

                if (null == _resgenPath && ExecuteAsTool)
                {
                    Log.LogErrorWithCodeFromResources("General.PlatformSDKFileNotFound", "resgen.exe",
                        ToolLocationHelper.GetDotNetFrameworkSdkInstallKeyValue(TargetDotNetFrameworkVersion.Version35),
                        ToolLocationHelper.GetDotNetFrameworkSdkRootRegistryKey(TargetDotNetFrameworkVersion.Version35));
                }
            }
            else
            {
                _resgenPath = SdkToolsPathUtility.GeneratePathToTool(
                    SdkToolsPathUtility.FileInfoExists,
                    Microsoft.Build.Utilities.ProcessorArchitecture.CurrentProcessArchitecture,
                    SdkToolsPath,
                    "resgen.exe",
                    Log,
                    ExecuteAsTool);
            }

            if (null == _resgenPath && !ExecuteAsTool)
            {
                // if Resgen.exe is not installed, just use the filename
                _resgenPath = String.Empty;
                return true;
            }

            // We may be passing this to the ResGen task (wrapper around resgen.exe), in which case
            // we want to pass just the path -- ResGen will attach the "resgen.exe" onto the end 
            // itself.
            if (_resgenPath != null)
            {
                _resgenPath = Path.GetDirectoryName(_resgenPath);
            }

            return _resgenPath != null;
#else
            _resgenPath = string.Empty;
            return true;
#endif
        }

        /// <summary>
        /// Wrapper around the call to the ResGen task that handles setting up the
        /// task to run properly.
        /// </summary>
        /// <param name="inputsToProcess">Array of names of inputs to be processed</param>
        /// <param name="outputsToProcess">Array of output names corresponding to the inputs</param>
        private bool GenerateResourcesUsingResGen(List<ITaskItem> inputsToProcess, List<ITaskItem> outputsToProcess)
        {
#if FEATURE_RESGEN
            bool resGenSucceeded = false;

            if (StronglyTypedLanguage != null)
            {
                resGenSucceeded = GenerateStronglyTypedResourceUsingResGen(inputsToProcess, outputsToProcess);
            }
            else
            {
                resGenSucceeded = TransformResourceFilesUsingResGen(inputsToProcess, outputsToProcess);
            }

            return resGenSucceeded;
#else
            Log.LogError("ResGen.exe not supported on .NET Core MSBuild");
            return false;
#endif
        }


#if FEATURE_RESGEN
        /// <summary>
        /// Given an instance of the ResGen task with everything but the strongly typed 
        /// resource-related parameters filled out, execute the task and return the result
        /// </summary>
        /// <param name="resGen">The task to execute.</param>
        private bool TransformResourceFilesUsingResGen(List<ITaskItem> inputsToProcess, List<ITaskItem> outputsToProcess)
        {
            ErrorUtilities.VerifyThrow(inputsToProcess.Count != 0, "There should be resource files to process");
            ErrorUtilities.VerifyThrow(inputsToProcess.Count == outputsToProcess.Count, "The number of inputs and outputs should be equal");

            bool succeeded = true;

            // We need to do a whole lot of work to make sure that we're not overrunning the command line ... UNLESS
            // we're running ResGen 4.0 or later, which supports response files. 
            if (!_resgenPath.Equals(Path.GetDirectoryName(NativeMethodsShared.GetLongFilePath(ToolLocationHelper.GetPathToDotNetFrameworkSdkFile("resgen.exe", TargetDotNetFrameworkVersion.Version35))), StringComparison.OrdinalIgnoreCase))
            {
                ResGen resGen = CreateResGenTaskWithDefaultParameters();

                resGen.InputFiles = inputsToProcess.ToArray();
                resGen.OutputFiles = outputsToProcess.ToArray();

                ITaskItem[] outputFiles = resGen.OutputFiles;

                succeeded = resGen.Execute();

                if (!succeeded)
                {
                    foreach (ITaskItem outputFile in outputFiles)
                    {
                        if (!FileSystems.Default.FileExists(outputFile.ItemSpec))
                        {
                            _unsuccessfullyCreatedOutFiles.Add(outputFile.ItemSpec);
                        }
                    }
                }
            }
            else
            {
                int initialResourceIndex = 0;
                int numberOfResourcesToAdd = 0;
                bool doneProcessingResources = false;
                CommandLineBuilderExtension resourcelessCommandBuilder = new CommandLineBuilderExtension();
                string resourcelessCommand = null;

                GenerateResGenCommandLineWithoutResources(resourcelessCommandBuilder);

                if (resourcelessCommandBuilder.Length > 0)
                {
                    resourcelessCommand = resourcelessCommandBuilder.ToString();
                }

                while (!doneProcessingResources)
                {
                    numberOfResourcesToAdd = CalculateResourceBatchSize(inputsToProcess, outputsToProcess, resourcelessCommand, initialResourceIndex);
                    ResGen resGen = CreateResGenTaskWithDefaultParameters();

                    resGen.InputFiles = inputsToProcess.GetRange(initialResourceIndex, numberOfResourcesToAdd).ToArray();
                    resGen.OutputFiles = outputsToProcess.GetRange(initialResourceIndex, numberOfResourcesToAdd).ToArray();

                    ITaskItem[] outputFiles = resGen.OutputFiles;

                    bool thisBatchSucceeded = resGen.Execute();

                    // This batch failed, so add the failing resources from this batch to the list of failures
                    if (!thisBatchSucceeded)
                    {
                        foreach (ITaskItem outputFile in outputFiles)
                        {
                            if (!FileSystems.Default.FileExists(outputFile.ItemSpec))
                            {
                                _unsuccessfullyCreatedOutFiles.Add(outputFile.ItemSpec);
                            }
                        }
                    }

                    initialResourceIndex += numberOfResourcesToAdd;
                    doneProcessingResources = initialResourceIndex == inputsToProcess.Count;
                    succeeded = succeeded && thisBatchSucceeded;
                }
            }

            return succeeded;
        }
#endif


        /// <summary>
        /// Given the list of inputs and outputs, returns the number of resources (starting at the provided initial index)
        /// that can fit onto the commandline without exceeding MaximumCommandLength.
        /// </summary>
        private int CalculateResourceBatchSize(List<ITaskItem> inputsToProcess, List<ITaskItem> outputsToProcess, string resourcelessCommand, int initialResourceIndex)
        {
            CommandLineBuilderExtension currentCommand = new CommandLineBuilderExtension();

            if (!String.IsNullOrEmpty(resourcelessCommand))
            {
                currentCommand.AppendTextUnquoted(resourcelessCommand);
            }

            int i = initialResourceIndex;
            while (currentCommand.Length < s_maximumCommandLength && i < inputsToProcess.Count)
            {
                currentCommand.AppendFileNamesIfNotNull
                    (
                        new ITaskItem[] { inputsToProcess[i], outputsToProcess[i] },
                        ","
                    );
                i++;
            }

            int numberOfResourcesToAdd = 0;
            if (currentCommand.Length <= s_maximumCommandLength)
            {
                // We've successfully added all the rest. 
                numberOfResourcesToAdd = i - initialResourceIndex;
            }
            else
            {
                // The last one added tossed us over the edge.
                numberOfResourcesToAdd = i - initialResourceIndex - 1;
            }

            return numberOfResourcesToAdd;
        }


#if FEATURE_RESGEN
        /// <summary>
        /// Given an instance of the ResGen task with everything but the strongly typed 
        /// resource-related parameters filled out, execute the task and return the result
        /// </summary>
        /// <param name="resGen">The task to execute.</param>
        private bool GenerateStronglyTypedResourceUsingResGen(List<ITaskItem> inputsToProcess, List<ITaskItem> outputsToProcess)
        {
            ErrorUtilities.VerifyThrow(inputsToProcess.Count == 1 && outputsToProcess.Count == 1, "For STR, there should only be one input and one output.");

            ResGen resGen = CreateResGenTaskWithDefaultParameters();

            resGen.InputFiles = inputsToProcess.ToArray();
            resGen.OutputFiles = outputsToProcess.ToArray();

            resGen.StronglyTypedLanguage = StronglyTypedLanguage;
            resGen.StronglyTypedNamespace = StronglyTypedNamespace;
            resGen.StronglyTypedClassName = StronglyTypedClassName;
            resGen.StronglyTypedFileName = StronglyTypedFileName;

            // Save the output file name -- ResGen will delete failing files
            ITaskItem outputFile = resGen.OutputFiles[0];

            _stronglyTypedResourceSuccessfullyCreated = resGen.Execute();

            if (!_stronglyTypedResourceSuccessfullyCreated && (resGen.OutputFiles == null || resGen.OutputFiles.Length == 0))
            {
                _unsuccessfullyCreatedOutFiles.Add(outputFile.ItemSpec);
            }

            // now need to set the defaults (if defaults were chosen) so that they can be 
            // consumed by outside users
            StronglyTypedClassName = resGen.StronglyTypedClassName;
            StronglyTypedFileName = resGen.StronglyTypedFileName;

            return _stronglyTypedResourceSuccessfullyCreated;
        }

        /// <summary>
        /// Factoring out the setting of the default parameters to the 
        /// ResGen task. 
        /// </summary>
        /// <param name="resGen"></param>
        private ResGen CreateResGenTaskWithDefaultParameters()
        {
            ResGen resGen = new ResGen();

            resGen.BuildEngine = BuildEngine;
            resGen.SdkToolsPath = _resgenPath;
            resGen.PublicClass = PublicClass;
            resGen.References = References;
            resGen.UseSourcePath = UseSourcePath;
            resGen.EnvironmentVariables = EnvironmentVariables;

            return resGen;
        }
#endif

        /// <summary>
        /// Check for parameter errors.
        /// </summary>
        /// <returns>true if parameters are valid</returns>
        private bool ValidateParameters()
        {
            // make sure that if the output resources were set, they exactly match the number of input sources
            if ((OutputResources != null) && (OutputResources.Length != Sources.Length))
            {
                Log.LogErrorWithCodeFromResources("General.TwoVectorsMustHaveSameLength", Sources.Length, OutputResources.Length, "Sources", "OutputResources");
                return false;
            }

            // Creating an STR is triggered merely by setting the language
            if (_stronglyTypedLanguage != null)
            {
                // Like Resgen.exe, only a single Sources is allowed if you are generating STR.
                // Otherwise, each STR class overwrites the previous one. In theory we could generate separate 
                // STR classes for each input, but then the class name and file name parameters would have to be vectors.
                if (Sources.Length != 1)
                {
                    Log.LogErrorWithCodeFromResources("GenerateResource.STRLanguageButNotExactlyOneSourceFile");
                    return false;
                }
            }
            else
            {
                if (StronglyTypedClassName != null || StronglyTypedNamespace != null || StronglyTypedFileName != null || StronglyTypedManifestPrefix != null)
                {
                    // We have no language to generate a STR, but nevertheless the user passed us a class, 
                    // namespace, and/or filename. Let them know that they probably wanted to pass a language too.
                    Log.LogErrorWithCodeFromResources("GenerateResource.STRClassNamespaceOrFilenameWithoutLanguage");
                    return false;
                }
            }

            if (ExtractResWFiles && ExecuteAsTool)
            {
                // This combination isn't currently supported, because ResGen may produce some not-easily-predictable
                // set of ResW files and we don't have any logic to get that set of files back into GenerateResource
                // at the moment.  This could be solved fixed with some engineering effort.
                Log.LogErrorWithCodeFromResources("GenerateResource.ExecuteAsToolAndExtractResWNotSupported");
                return false;
            }

            return true;
        }

        /// <summary>
        /// Returns true if everything is up to date and we don't need to do any work.
        /// </summary>
        /// <returns></returns>
        private void GetResourcesToProcess(out List<ITaskItem> inputsToProcess, out List<ITaskItem> outputsToProcess, out List<ITaskItem> cachedOutputFiles)
        {
#if FEATURE_RESGENCACHE
            // First we look to see if we have a resgen linked files cache.  If so, then we can use that
            // cache to speed up processing.
            ReadStateFile();
#endif

            bool nothingOutOfDate = true;
            inputsToProcess = new List<ITaskItem>();
            outputsToProcess = new List<ITaskItem>();
            cachedOutputFiles = new List<ITaskItem>();

            // decide what sources we need to build
            for (int i = 0; i < Sources.Length; ++i)
            {
                if (ExtractResWFiles)
                {
#if FEATURE_RESGENCACHE
                    // We can't cheaply predict the output files, since that would require
                    // loading each assembly.  So don't even try guessing what they will be.
                    // However, our cache will sometimes record all the info we need (for incremental builds).
                    string sourceFileName = Sources[i].ItemSpec;
                    ResGenDependencies.PortableLibraryFile library = _cache.TryGetPortableLibraryInfo(sourceFileName);
                    if (library != null && library.AllOutputFilesAreUpToDate())
                    {
                        AppendCachedOutputTaskItems(library, cachedOutputFiles);
                    }
                    else
                    {
#endif
                        inputsToProcess.Add(Sources[i]);
#if FEATURE_RESGENCACHE
                    }
#endif

                    continue;
                }

                // Attributes from input items are forwarded to output items.
                Sources[i].CopyMetadataTo(OutputResources[i]);
                Sources[i].SetMetadata("OutputResource", OutputResources[i].ItemSpec);

                if (!FileSystems.Default.FileExists(Sources[i].ItemSpec))
                {
                    // Error but continue with the files that do exist
                    Log.LogErrorWithCodeFromResources("GenerateResource.ResourceNotFound", Sources[i].ItemSpec);
                    _unsuccessfullyCreatedOutFiles.Add(OutputResources[i].ItemSpec);
                }
                else
                {
                    // check to see if the output resources file (and, if it is a .resx, any linked files) 
                    // is up to date compared to the input file
                    if (ShouldRebuildResgenOutputFile(Sources[i].ItemSpec, OutputResources[i].ItemSpec))
                    {
                        nothingOutOfDate = false;
                        inputsToProcess.Add(Sources[i]);
                        outputsToProcess.Add(OutputResources[i]);
                    }
                }
            }

            // If the STR class file is out of date (for example, it's missing) we don't want to skip
            // resource generation.
            if (_stronglyTypedLanguage != null)
            {
                // We're generating a STR class file, so there must be exactly one input resource file.
                // If that resource file itself is out of date, the STR class file is going to get generated anyway.
                if (nothingOutOfDate && FileSystems.Default.FileExists(Sources[0].ItemSpec))
                {
                    GetStronglyTypedResourceToProcess(ref inputsToProcess, ref outputsToProcess);
                }
            }
        }

#if FEATURE_RESGENCACHE
        /// <summary>
        /// Given a cached portable library that is up to date, create ITaskItems to represent the output of the task, as if we did real work.
        /// </summary>
        /// <param name="library">The portable library cache entry to extract output files & metadata from.</param>
        /// <param name="cachedOutputFiles">List of output files produced from the cache.</param>
        private void AppendCachedOutputTaskItems(ResGenDependencies.PortableLibraryFile library, List<ITaskItem> cachedOutputFiles)
        {
            foreach (string outputFileName in library.OutputFiles)
            {
                ITaskItem item = new TaskItem(outputFileName);
                item.SetMetadata("ResourceIndexName", library.AssemblySimpleName);
                if (library.NeutralResourceLanguage != null)
                {
                    item.SetMetadata("NeutralResourceLanguage", library.NeutralResourceLanguage);
                }

                cachedOutputFiles.Add(item);
            }
        }
#endif

        /// <summary>
        /// Checks if this list contain any duplicates.  Do this so we don't have any races where we have two
        /// threads trying to write to the same file simultaneously.
        /// </summary>
        /// <param name="originalList">A list that may have duplicates</param>
        /// <returns>Were there duplicates?</returns>
        private bool ContainsDuplicates(IList<ITaskItem> originalList)
        {
            HashSet<string> set = new HashSet<string>(StringComparer.OrdinalIgnoreCase);
            foreach (ITaskItem item in originalList)
            {
                try
                {
                    // Get the fully qualified path, to ensure two file names don't end up pointing to the same directory.
                    if (!set.Add(item.GetMetadata("FullPath")))
                    {
                        Log.LogErrorWithCodeFromResources("GenerateResource.DuplicateOutputFilenames", item.ItemSpec);
                        return true;
                    }
                }
                catch (InvalidOperationException e)
                {
                    Log.LogErrorWithCodeFromResources("GenerateResource.InvalidFilename", item.ItemSpec, e.Message);
                    // Returning true causes us to not continue executing.
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// Determines if the given output file is up to date with respect to the
        /// the given input file by comparing timestamps of the two files as well as
        /// (if the source is a .resx) the linked files inside the .resx file itself
        /// </summary>
        /// <param name="sourceFilePath"></param>
        /// <param name="outputFilePath"></param>
        /// <returns></returns>
        private bool ShouldRebuildResgenOutputFile(string sourceFilePath, string outputFilePath)
        {
            // See if any uncorrelated inputs are missing before checking source and output file timestamps.
            // Don't consult the uncorrelated input file times if we haven't already got them:
            // typically, it's the .resx's that are out of date so we want to check those first.
            if (_foundNewestUncorrelatedInputWriteTime && GetNewestUncorrelatedInputWriteTime() == DateTime.MaxValue)
            {
                // An uncorrelated input is missing; need to build
                return true;
            }

            DateTime outputTime = NativeMethodsShared.GetLastWriteFileUtcTime(outputFilePath);

            // Quick check to see if any uncorrelated input is newer in which case we can avoid checking source file timestamp
            if (_foundNewestUncorrelatedInputWriteTime && GetNewestUncorrelatedInputWriteTime() > outputTime)
            {
                // An uncorrelated input is newer, need to build
                return true;
            }

            DateTime sourceTime = NativeMethodsShared.GetLastWriteFileUtcTime(sourceFilePath);

            if (!String.Equals(Path.GetExtension(sourceFilePath), ".resx", StringComparison.OrdinalIgnoreCase) &&
                !String.Equals(Path.GetExtension(sourceFilePath), ".resw", StringComparison.OrdinalIgnoreCase))
            {
                // If source file is NOT a .resx, for example a .restext file, 
                // timestamp checking is simple, because there's no linked files to examine, and no references.
                return NeedToRebuildSourceFile(sourceTime, outputTime);
            }

#if FEATURE_RESGENCACHE
            // OK, we have a .resx file

            // PERF: Regardless of whether the outputFile exists, if the source file is a .resx 
            // go ahead and retrieve it from the cache. This is because we want the cache 
            // to be populated so that incremental builds can be fast.
            // Note that this is a trade-off: clean builds will be slightly slower. However,
            // for clean builds we're about to read in this very same .resx file so reading
            // it now will page it in. The second read should be cheap.
            ResGenDependencies.ResXFile resxFileInfo = null;
            try
            {
                resxFileInfo = _cache.GetResXFileInfo(sourceFilePath);
            }
            catch (Exception e)  // Catching Exception, but rethrowing unless it's a well-known exception.
            {
                if (ExceptionHandling.NotExpectedIoOrXmlException(e))
                {
                    throw;
                }

                // Return true, so that resource processing will display the error
                // No point logging a duplicate error here as well
                return true;
            }

            // if the .resources file is out of date even just with respect to the .resx or 
            // the additional inputs, we don't need to go to the point of checking the linked files. 
            if (NeedToRebuildSourceFile(sourceTime, outputTime))
            {
                return true;
            }

            // The .resources is up to date with respect to the .resx file -
            // we need to compare timestamps for each linked file inside
            // the .resx file itself
            if (resxFileInfo.LinkedFiles != null)
            {
                foreach (string linkedFilePath in resxFileInfo.LinkedFiles)
                {
                    DateTime linkedFileTime = NativeMethodsShared.GetLastWriteFileUtcTime(linkedFilePath);

                    if (linkedFileTime == DateTime.MinValue)
                    {
                        // Linked file is missing - force a build, so that resource generation
                        // will produce a nice error message
                        return true;
                    }

                    if (linkedFileTime > outputTime)
                    {
                        // Linked file is newer, need to build
                        return true;
                    }
                }
            }

            return false;
#else
#if FEATURE_RESX_RESOURCE_READER
            Compile error: if we get Resx reading before binary serialization
            it might be interesting to get caching working some other way.
#endif
            // On .NET Core, we don't have binary serialization to maintain
            // the cache, but conveniently .NET Core assemblies (the only
            // supported target of .NET Core MSBuild) cannot use linked
            // resources. So the only relevant comparison is input/output.
            // See https://github.com/Microsoft/msbuild/issues/1197
            return NeedToRebuildSourceFile(sourceTime, outputTime);
#endif
        }

        /// <summary>
        /// Returns true if the output does not exist, if the provided source is newer than the output, 
        /// or if any of the set of additional inputs is newer than the output.  Otherwise, returns false. 
        /// </summary>
        private bool NeedToRebuildSourceFile(DateTime sourceTime, DateTime outputTime)
        {
            if (sourceTime > outputTime)
            {
                // Source file is newer, need to build
                return true;
            }

            if (GetNewestUncorrelatedInputWriteTime() > outputTime)
            {
                // An uncorrelated input is newer, need to build
                return true;
            }

            return false;
        }

        /// <summary>
        /// Add the strongly typed resource to the set of resources to process if it is out of date. 
        /// </summary>
        private void GetStronglyTypedResourceToProcess(ref List<ITaskItem> inputsToProcess, ref List<ITaskItem> outputsToProcess)
        {
            bool needToRebuildSTR = false;

            // The resource file isn't out of date. So check whether the STR class file is.
            try
            {
                if (StronglyTypedFileName == null)
                {
#if FEATURE_CODEDOM
                    CodeDomProvider provider = null;

                    if (ProcessResourceFiles.TryCreateCodeDomProvider(Log, StronglyTypedLanguage, out provider))
                    {
                        StronglyTypedFileName = ProcessResourceFiles.GenerateDefaultStronglyTypedFilename(provider, OutputResources[0].ItemSpec);
                    }
#else
                    StronglyTypedFileName = TryGenerateDefaultStronglyTypedFilename();
#endif
                }
            }
            catch (Exception e) when (ExceptionHandling.IsIoRelatedException(e))
            {
                Log.LogErrorWithCodeFromResources("GenerateResource.CannotWriteSTRFile", StronglyTypedFileName, e.Message);
                // Now that we've logged an error, we know we're not going to do anything more, so 
                // don't bother to correctly populate the inputs / outputs.
                _unsuccessfullyCreatedOutFiles.Add(OutputResources[0].ItemSpec);
                _stronglyTypedResourceSuccessfullyCreated = false;
                return;
            }

            // Now we have the filename, check if it's up to date
            DateTime sourceTime = NativeMethodsShared.GetLastWriteFileUtcTime(Sources[0].ItemSpec);
            DateTime outputTime = NativeMethodsShared.GetLastWriteFileUtcTime(StronglyTypedFileName);

            if (sourceTime == DateTime.MinValue || outputTime == DateTime.MinValue)
            {
                // Source file is missing - force a build, so that resource generation
                // will produce a nice error message; or output file is missing,
                // need to build it
                needToRebuildSTR = true;
            }
            else if (sourceTime > outputTime)
            {
                // Source file is newer, need to build
                needToRebuildSTR = true;
            }

            if (needToRebuildSTR)
            {
                // We know there's only one input, just make sure it gets processed and that will cause
                // the STR class file to get updated
                if (inputsToProcess.Count == 0)
                {
                    inputsToProcess.Add(Sources[0]);
                    outputsToProcess.Add(OutputResources[0]);
                }
            }
            else
            {
                // If the STR class file is up to date and was skipped then we
                // should consider the file written and add it to FilesWritten output
                _stronglyTypedResourceSuccessfullyCreated = true;
            }
        }

        /// <summary>
        /// Returns the newest last write time among the references and additional inputs.
        /// If any do not exist, returns DateTime.MaxValue so that resource generation produces a nice error.
        /// </summary>
        private DateTime GetNewestUncorrelatedInputWriteTime()
        {
            if (!_foundNewestUncorrelatedInputWriteTime)
            {
                _newestUncorrelatedInputWriteTime = DateTime.MinValue;

                // Check the timestamp of each of the passed-in references to find the newest
                if (this.References != null)
                {
                    foreach (ITaskItem reference in this.References)
                    {
                        DateTime referenceTime = NativeMethodsShared.GetLastWriteFileUtcTime(reference.ItemSpec);

                        if (referenceTime == DateTime.MinValue)
                        {
                            // File does not exist: force a build to produce an error message
                            _foundNewestUncorrelatedInputWriteTime = true;
                            return DateTime.MaxValue;
                        }

                        if (referenceTime > _newestUncorrelatedInputWriteTime)
                        {
                            _newestUncorrelatedInputWriteTime = referenceTime;
                        }
                    }
                }

                // Check the timestamp of each of the additional inputs to see if one's even newer
                if (this.AdditionalInputs != null)
                {
                    foreach (ITaskItem additionalInput in this.AdditionalInputs)
                    {
                        DateTime additionalInputTime = NativeMethodsShared.GetLastWriteFileUtcTime(additionalInput.ItemSpec);

                        if (additionalInputTime == DateTime.MinValue)
                        {
                            // File does not exist: force a build to produce an error message
                            _foundNewestUncorrelatedInputWriteTime = true;
                            return DateTime.MaxValue;
                        }

                        if (additionalInputTime > _newestUncorrelatedInputWriteTime)
                        {
                            _newestUncorrelatedInputWriteTime = additionalInputTime;
                        }
                    }
                }

                _foundNewestUncorrelatedInputWriteTime = true;
            }

            return _newestUncorrelatedInputWriteTime;
        }

#if FEATURE_APPDOMAIN
        /// <summary>
        /// Make the decision about whether a separate AppDomain is needed.
        /// If this algorithm is unsure about whether a separate AppDomain is
        /// needed, it should always err on the side of returning 'true'. This 
        /// is because a separate AppDomain, while slow to create, is always safe.
        /// </summary>
        /// <param name="sources">The list of .resx files.</param>
        /// <returns></returns>
        private bool NeedSeparateAppDomain()
        {
            if (NeverLockTypeAssemblies)
            {
                Log.LogMessageFromResources(MessageImportance.Low, "GenerateResource.SeparateAppDomainBecauseNeverLockTypeAssembliesTrue");
                return true;
            }

            foreach (ITaskItem source in _sources)
            {
                string extension = Path.GetExtension(source.ItemSpec);

                if (String.Compare(extension, ".resources.dll", StringComparison.OrdinalIgnoreCase) == 0 ||
                    String.Compare(extension, ".dll", StringComparison.OrdinalIgnoreCase) == 0 ||
                    String.Compare(extension, ".exe", StringComparison.OrdinalIgnoreCase) == 0)
                {
                    return true;
                }

                if (String.Compare(extension, ".resx", StringComparison.OrdinalIgnoreCase) == 0 ||
                    String.Compare(extension, ".resw", StringComparison.OrdinalIgnoreCase) == 0)
                {
                    XmlReader reader = null;
                    string name = null;

                    try
                    {
                        XmlReaderSettings readerSettings = new XmlReaderSettings();
                        readerSettings.DtdProcessing = DtdProcessing.Ignore;
                        reader = XmlReader.Create(source.ItemSpec, readerSettings);

                        while (reader.Read())
                        {
                            // Look for the <data> section
                            if (reader.NodeType == XmlNodeType.Element)
                            {
                                if (String.Equals(reader.Name, "data", StringComparison.OrdinalIgnoreCase))
                                {
                                    // Is there an attribute called type?
                                    string typeName = reader.GetAttribute("type");
                                    name = reader.GetAttribute("name");

                                    if (typeName != null)
                                    {
                                        Type type;

                                        // It is likely that we've seen this type before
                                        // we'll try our table of previously seen types
                                        // since it is *much* faster to do that than
                                        // call Type.GetType needlessly!
                                        if (!_typeTable.TryGetValue(typeName, out type))
                                        {
                                            string resolvedTypeName = typeName;

                                            // This type name might be an alias, so first resolve that if any.  
                                            int indexOfSeperator = typeName.IndexOf(",", StringComparison.Ordinal);

                                            if (indexOfSeperator != -1)
                                            {
                                                string typeFromTypeName = typeName.Substring(0, indexOfSeperator);
                                                string maybeAliasFromTypeName = typeName.Substring(indexOfSeperator + 1);

                                                if (!String.IsNullOrWhiteSpace(maybeAliasFromTypeName))
                                                {
                                                    maybeAliasFromTypeName = maybeAliasFromTypeName.Trim();

                                                    string fullName = null;
                                                    if (_aliases.TryGetValue(maybeAliasFromTypeName, out fullName))
                                                    {
                                                        resolvedTypeName = typeFromTypeName + ", " + fullName;
                                                    }
                                                }
                                            }

                                            // Can this type be found in the GAC?
                                            type = Type.GetType(resolvedTypeName, throwOnError: false, ignoreCase: false);

                                            // Remember our resolved type
                                            _typeTable[typeName] = type;
                                        }

                                        if (type == null)
                                        {
                                            // If the type could not be found in the GAC, then we're going to need
                                            // to load the referenced assemblies (those passed in through the
                                            // "References" parameter during the building of this .RESX.  Therefore,
                                            // we should create a separate app-domain, so that those assemblies
                                            // can be unlocked when the task is finished.
                                            // The type didn't start with "System." so return true.
                                            Log.LogMessageFromResources
                                            (
                                                MessageImportance.Low,
                                                "GenerateResource.SeparateAppDomainBecauseOfType",
                                                (name == null) ? String.Empty : name,
                                                typeName,
                                                source.ItemSpec,
                                                ((IXmlLineInfo)reader).LineNumber
                                            );

                                            return true;
                                        }

                                        // If there's a type, we don't need to look at any mimetype
                                        continue;
                                    }

                                    // DDB #9825.
                                    // There's no type attribute on this <data> -- if there's a MimeType, it's a serialized
                                    // object of unknown type, and we have to assume it will need a new app domain.
                                    // The possible mimetypes ResXResourceReader understands are:
                                    //  
                                    // application/x-microsoft.net.object.binary.base64
                                    // application/x-microsoft.net.object.bytearray.base64
                                    // application/x-microsoft.net.object.binary.base64
                                    // application/x-microsoft.net.object.soap.base64
                                    // text/microsoft-urt/binary-serialized/base64
                                    // text/microsoft-urt/psuedoml-serialized/base64
                                    // text/microsoft-urt/soap-serialized/base64
                                    //
                                    // Of these, application/x-microsoft.net.object.bytearray.base64 usually has a type attribute
                                    // as well; ResxResourceReader will use that Type, which may not need a new app domain. So
                                    // if there's a type attribute, we don't look at mimetype.
                                    //
                                    // If there is a mimetype and no type, we can't tell the type without deserializing and loading it, 
                                    // so we assume a new appdomain is needed.
                                    //
                                    // Actually, if application/x-microsoft.net.object.bytearray.base64 doesn't have a Type attribute, 
                                    // ResxResourceReader assumes System.String, but for safety we don't assume that here.

                                    string mimeType = reader.GetAttribute("mimetype");

                                    if (mimeType != null)
                                    {
                                        if (NeedSeparateAppDomainBasedOnSerializedType(reader))
                                        {
                                            Log.LogMessageFromResources
                                            (
                                                MessageImportance.Low,
                                                "GenerateResource.SeparateAppDomainBecauseOfMimeType",
                                                (name == null) ? String.Empty : name,
                                                mimeType,
                                                source.ItemSpec,
                                                ((IXmlLineInfo)reader).LineNumber
                                            );

                                            return true;
                                        }
                                    }
                                }
                                else if (String.Equals(reader.Name, "assembly", StringComparison.OrdinalIgnoreCase))
                                {
                                    string alias = reader.GetAttribute("alias");
                                    string fullName = reader.GetAttribute("name");

                                    if (!String.IsNullOrWhiteSpace(alias) && !String.IsNullOrWhiteSpace(fullName))
                                    {
                                        alias = alias.Trim();
                                        fullName = fullName.Trim();

                                        _aliases[alias] = fullName;
                                    }
                                }
                            }
                        }
                    }
                    catch (XmlException e)
                    {
                        Log.LogMessageFromResources
                                    (
                                        MessageImportance.Low,
                                        "GenerateResource.SeparateAppDomainBecauseOfExceptionLineNumber",
                                        source.ItemSpec,
                                        ((IXmlLineInfo)reader).LineNumber,
                                        e.Message
                                    );

                        return true;
                    }
#if FEATURE_RESGENCACHE
                    catch (SerializationException e)
                    {
                        Log.LogMessageFromResources
                                    (
                                        MessageImportance.Low,
                                        "GenerateResource.SeparateAppDomainBecauseOfErrorDeserializingLineNumber",
                                        source.ItemSpec,
                                        (name == null) ? String.Empty : name,
                                        ((IXmlLineInfo)reader).LineNumber,
                                        e.Message
                                    );

                        return true;
                    }
#endif
                    catch (Exception e)
                    {
                        // DDB#9819
                        // Customers have reported the following exceptions coming out of this method's call to GetType():
                        //      System.Runtime.InteropServices.COMException (0x8000000A): The data necessary to complete this operation is not yet available. (Exception from HRESULT: 0x8000000A)
                        //      System.NullReferenceException: Object reference not set to an instance of an object.
                        //      System.InvalidOperationException: Collection was modified; enumeration operation may not execute.
                        // We don't have reproes, but probably the right thing to do is to assume a new app domain is needed on almost any exception.
                        // Any problem loading the type will get logged later when the resource reader tries it.
                        //
                        // XmlException or an IO exception is also possible from an invalid input file.
                        if (ExceptionHandling.IsCriticalException(e))
                            throw;

                        // If there was any problem parsing the .resx then log a message and 
                        // fall back to using a separate AppDomain. 
                        Log.LogMessageFromResources
                                    (
                                        MessageImportance.Low,
                                        "GenerateResource.SeparateAppDomainBecauseOfException",
                                        source.ItemSpec,
                                        e.Message
                                    );

                        // In case we need more information from the customer (given this has been heavily reported
                        // and we don't understand it properly) let the usual debug switch dump the stack.
                        if (Environment.GetEnvironmentVariable("MSBUILDDEBUG") == "1")
                        {
                            Log.LogErrorFromException(e, /* stack */ true, /* inner exceptions */ true, null);
                        }

                        return true;
                    }
                    finally
                    {
                        reader?.Close();
                    }
                }
            }

            return false;
        }

        /// <summary>
        /// Finds the "value" element expected to be the next element read from the supplied reader.
        /// Deserializes the data content in order to figure out whether it implies a new app domain
        /// should be used to process resources.
        /// </summary>
        private bool NeedSeparateAppDomainBasedOnSerializedType(XmlReader reader)
        {
#if FEATURE_RESGENCACHE
            while (reader.Read())
            {
                if (reader.NodeType == XmlNodeType.Element)
                {
                    if (!String.Equals(reader.Name, "value", StringComparison.OrdinalIgnoreCase))
                    {
                        // <data> claimed it was serialized, but didn't have a <value>; 
                        // return true to err on side of caution
                        return true;
                    }

                    // Found "value" element
                    string data = reader.ReadElementContentAsString();

                    bool isSerializedObjectLoadable = DetermineWhetherSerializedObjectLoads(data);

                    // If it's not loadable, it's presumably a user type, so create a new app domain
                    return !isSerializedObjectLoadable;
                }
            }

            // We didn't find any element at all -- the .resx is malformed.
            // Return true to err on the side of caution. Error will appear later.
#endif
            return true;
        }
#endif

#if FEATURE_RESGENCACHE
        /// <summary>
        /// Deserializes a base64 block from a resx in order to figure out if its type is in the GAC.
        /// Because we're not providing any assembly resolution callback, deserialization
        /// will attempt to load the object's type using fusion rules, which essentially means
        /// the GAC. So, if the object is null, it's definitely not in the GAC.
        /// </summary>
        private bool DetermineWhetherSerializedObjectLoads(string data)
        {
            byte[] serializedData = ByteArrayFromBase64WrappedString(data);

            BinaryFormatter binaryFormatter = new BinaryFormatter();

            using (MemoryStream memoryStream = new MemoryStream(serializedData))
            {
                object result = binaryFormatter.Deserialize(memoryStream);

                return (result != null);
            }
        }
#endif

        /// <summary>
        /// Chars that should be ignored in the nicely justified block of base64
        /// </summary>
        private static readonly char[] s_specialChars = new char[] { ' ', '\r', '\n' };

        /// <summary>
        /// Turns the nicely justified block of base64 found in a resx into a byte array.
        /// Copied from fx\src\winforms\managed\system\winforms\control.cs
        /// </summary>
        private static byte[] ByteArrayFromBase64WrappedString(string text)
        {
            if (text.IndexOfAny(s_specialChars) != -1)
            {
                StringBuilder sb = new StringBuilder(text.Length);
                for (int i = 0; i < text.Length; i++)
                {
                    switch (text[i])
                    {
                        case ' ':
                        case '\r':
                        case '\n':
                            break;
                        default:
                            sb.Append(text[i]);
                            break;
                    }
                }
                return Convert.FromBase64String(sb.ToString());
            }
            else
            {
                return Convert.FromBase64String(text);
            }
        }

        /// <summary>
        /// Make sure that OutputResources has 1 file name for each name in Sources.
        /// </summary>
        private bool CreateOutputResourcesNames()
        {
            if (OutputResources == null)
            {
                OutputResources = new ITaskItem[Sources.Length];
                int i = 0;
                try
                {
                    for (i = 0; i < Sources.Length; ++i)
                    {
                        OutputResources[i] = new TaskItem(Path.ChangeExtension(Sources[i].ItemSpec, ".resources"));
                    }
                }
                catch (ArgumentException e)
                {
                    Log.LogErrorWithCodeFromResources("GenerateResource.InvalidFilename", Sources[i].ItemSpec, e.Message);
                    return false;
                }
            }

            // Now check for duplicates.
            if (ContainsDuplicates(OutputResources))
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// Remove any output resources that we didn't successfully create (due to error) from the 
        /// OutputResources list. Keeps the ordering of OutputResources the same.
        /// </summary>
        /// <remarks>
        /// Q: Why didn't we keep a "successfully created" list instead, like in the Copy task does, which
        /// would save us doing the removal algorithm below? 
        /// A: Because we want the ordering of OutputResources to be the same as the ordering passed in.
        /// Some items (the up to date ones) would be added to the successful output list first, and the other items 
        /// are added during processing, so the ordering would change. We could fix that up, but it's better to do 
        /// the fix up only in the rarer error case. If there were no errors, the algorithm below skips.</remarks>
        private void RemoveUnsuccessfullyCreatedResourcesFromOutputResources()
        {
            // Normally, there aren't any unsuccessful conversions.
            if (_unsuccessfullyCreatedOutFiles == null ||
                _unsuccessfullyCreatedOutFiles.Count == 0)
            {
                return;
            }

            ErrorUtilities.VerifyThrow(OutputResources != null && OutputResources.Length != 0, "Should be at least one output resource");

            // We only get here if there was at least one resource generation error.
            ITaskItem[] temp = new ITaskItem[OutputResources.Length - _unsuccessfullyCreatedOutFiles.Count];
            int copied = 0;
            int removed = 0;
            for (int i = 0; i < Sources.Length; i++)
            {
                // Check whether this one is in the bad list.
                if (removed < _unsuccessfullyCreatedOutFiles.Count &&
                    _unsuccessfullyCreatedOutFiles.Contains(OutputResources[i].ItemSpec))
                {
                    removed++;
                    Sources[i].SetMetadata("OutputResource", String.Empty);
                }
                else
                {
                    // Copy it to the okay list.
                    temp[copied] = OutputResources[i];
                    copied++;
                }
            }
            OutputResources = temp;
        }

        /// <summary>
        /// Record the list of file that will be written to disk.
        /// </summary>
        private void RecordFilesWritten()
        {
            // Add any output resources that were successfully created,
            // or would have been if they weren't already up to date (important for Clean)
            if (this.OutputResources != null)
            {
                foreach (ITaskItem item in this.OutputResources)
                {
                    _filesWritten.Add(item);
                }
            }

            // Add any state file
            if (StateFile != null && StateFile.ItemSpec.Length > 0)
            {
                // It's possible the file wasn't actually written (eg the path was invalid)
                // We can't easily tell whether that happened here, and I think it's fine to add it anyway.
                _filesWritten.Add(StateFile);
            }

            // Only add the STR class file if the CodeDOM succeeded, or if it exists and is up to date.
            // Otherwise, we probably didn't write it successfully.
            if (_stronglyTypedResourceSuccessfullyCreated)
            {
                if (StronglyTypedFileName == null)
                {
#if FEATURE_CODEDOM
                    CodeDomProvider provider = null;

                    if (ProcessResourceFiles.TryCreateCodeDomProvider(Log, StronglyTypedLanguage, out provider))
                    {
                        StronglyTypedFileName = ProcessResourceFiles.GenerateDefaultStronglyTypedFilename(
                            provider, OutputResources[0].ItemSpec);
                    }
#else
                    StronglyTypedFileName = TryGenerateDefaultStronglyTypedFilename();
#endif
                }

                _filesWritten.Add(new TaskItem(this.StronglyTypedFileName));
            }
        }

#if !FEATURE_CODEDOM
        private string TryGenerateDefaultStronglyTypedFilename()
        {
            string extension = null;
            if (StronglyTypedLanguage == CSharpLanguageName)
            {
                extension = ".cs";
            }
            else if (StronglyTypedLanguage == VisualBasicLanguageName)
            {
                extension = ".vb";
            }
            if (extension != null)
            {
                return Path.ChangeExtension(OutputResources[0].ItemSpec, extension);
            }
            return null;
        }
#endif

#if FEATURE_RESGENCACHE
        /// <summary>
        /// Read the state file if able.
        /// </summary>
        private void ReadStateFile()
        {
            // First we look to see if we have a resgen linked files cache.  If so, then we can use that
            // cache to speed up processing.  If there's a problem reading the cache file (or it
            // just doesn't exist, then this method will return a brand new cache object.

            // This method eats IO Exceptions
            _cache = ResGenDependencies.DeserializeCache((StateFile == null) ? null : StateFile.ItemSpec, UseSourcePath, Log);
            ErrorUtilities.VerifyThrow(_cache != null, "We did not create a cache!");
        }

        /// <summary>
        /// Write the state file if there is one to be written.
        /// </summary>
        private void WriteStateFile()
        {
            if (_cache.IsDirty)
            {
                // And now we serialize the cache to save our resgen linked file resolution for later use.
                _cache.SerializeCache((StateFile == null) ? null : StateFile.ItemSpec, Log);
            }
        }
#endif
    }

    /// <summary>
    /// This class handles the processing of source resource files into compiled resource files.
    /// Its designed to be called from a separate AppDomain so that any files locked by ResXResourceReader
    /// can be released.
    /// </summary>
    internal sealed class ProcessResourceFiles
#if FEATURE_APPDOMAIN
        : MarshalByRefObject
#endif
    {
        #region fields
        /// <summary>
        /// List of readers used for input.
        /// </summary>
        private List<ReaderInfo> _readers = new List<ReaderInfo>();

        /// <summary>
        /// Logger for any messages or errors
        /// </summary>
        private TaskLoggingHelper _logger = null;

        /// <summary>
        /// Language for the strongly typed resources.
        /// </summary>
        private string _stronglyTypedLanguage;

        /// <summary>
        /// Filename for the strongly typed resources.
        /// Getter provided since the processor may choose a default.
        /// </summary>
        internal string StronglyTypedFilename
        {
            get
            {
                return _stronglyTypedFilename;
            }
        }
        private string _stronglyTypedFilename;

        /// <summary>
        /// Namespace for the strongly typed resources.
        /// </summary>
        private string _stronglyTypedNamespace;

        /// <summary>
        /// ResourceNamespace for the strongly typed resources.
        /// </summary>
        private string _stronglyTypedResourcesNamespace;

        /// <summary>
        /// Class name for the strongly typed resources.
        /// Getter provided since the processor may choose a default.
        /// </summary>
        internal string StronglyTypedClassName
        {
            get
            {
                return _stronglyTypedClassName;
            }
        }
        private string _stronglyTypedClassName;

        /// <summary>
        /// Whether the fields in the STR class should be public, rather than internal
        /// </summary>
        private bool _stronglyTypedClassIsPublic;

#if FEATURE_ASSEMBLY_LOADFROM
        /// <summary>
        /// Class that gets called by the ResxResourceReader to resolve references
        /// to assemblies within the .RESX.
        /// </summary>
        private AssemblyNamesTypeResolutionService _typeResolver = null;

        /// <summary>
        /// Handles assembly resolution events.
        /// </summary>
        private ResolveEventHandler _eventHandler;
#endif

        /// <summary>
        /// The referenced assemblies
        /// </summary>
        private ITaskItem[] _assemblyFiles;

#if FEATURE_ASSEMBLY_LOADFROM
        /// <summary>
        /// The AssemblyNameExtensions for each of the referenced assemblies in "assemblyFiles".
        /// This is populated lazily.
        /// </summary>
        private AssemblyNameExtension[] _assemblyNames;
#endif

        /// <summary>
        /// List of input files to process.
        /// </summary>
        private List<ITaskItem> _inFiles;

        /// <summary>
        /// List of satellite input files to process.
        /// </summary>
        private List<ITaskItem> _satelliteInFiles;

        /// <summary>
        /// List of output files to process.
        /// </summary>
        private List<ITaskItem> _outFiles;

        /// <summary>
        /// Whether we are extracting ResW files from an assembly, instead of creating .resources files.
        /// </summary>
        private bool _extractResWFiles;

        /// <summary>
        /// Where to write extracted ResW files.
        /// </summary>
        private string _resWOutputDirectory;

        internal List<ITaskItem> ExtractedResWFiles
        {
            get
            {
                if (_extractedResWFiles == null)
                {
                    _extractedResWFiles = new List<ITaskItem>();
                }
                return _extractedResWFiles;
            }
        }
        private List<ITaskItem> _extractedResWFiles;

#if FEATURE_RESGENCACHE
        /// <summary>
        /// Record all the information about outputs here to avoid future incremental builds.
        /// </summary>
        internal List<ResGenDependencies.PortableLibraryFile> PortableLibraryCacheInfo
        {
            get { return _portableLibraryCacheInfo; }
        }
        private List<ResGenDependencies.PortableLibraryFile> _portableLibraryCacheInfo;
#endif

        /// <summary>
        /// List of output files that we failed to create due to an error.
        /// See note in RemoveUnsuccessfullyCreatedResourcesFromOutputResources()
        /// </summary>
        internal ArrayList UnsuccessfullyCreatedOutFiles
        {
            get
            {
                if (null == _unsuccessfullyCreatedOutFiles)
                {
                    _unsuccessfullyCreatedOutFiles = new ArrayList();
                }
                return _unsuccessfullyCreatedOutFiles;
            }
        }
        private ArrayList _unsuccessfullyCreatedOutFiles;

        /// <summary>
        /// Whether we successfully created the STR class 
        /// </summary>
        internal bool StronglyTypedResourceSuccessfullyCreated
        {
            get
            {
                return _stronglyTypedResourceSuccessfullyCreated;
            }
        }
        private bool _stronglyTypedResourceSuccessfullyCreated = false;

        /// <summary>
        /// Indicates whether the resource reader should use the source file's
        /// directory to resolve relative file paths.
        /// </summary>
        private bool _useSourcePath = false;

#endregion

        /// <summary>
        /// Process all files.
        /// </summary>
        internal void Run(TaskLoggingHelper log, ITaskItem[] assemblyFilesList, List<ITaskItem> inputs, List<ITaskItem> satelliteInputs, List<ITaskItem> outputs, bool sourcePath,
                          string language, string namespacename, string resourcesNamespace, string filename, string classname, bool publicClass,
                          bool extractingResWFiles, string resWOutputDirectory)
        {
            _logger = log;
            _assemblyFiles = assemblyFilesList;
            _inFiles = inputs;
            _satelliteInFiles = satelliteInputs;
            _outFiles = outputs;
            _useSourcePath = sourcePath;
            _stronglyTypedLanguage = language;
            _stronglyTypedNamespace = namespacename;
            _stronglyTypedResourcesNamespace = resourcesNamespace;
            _stronglyTypedFilename = filename;
            _stronglyTypedClassName = classname;
            _stronglyTypedClassIsPublic = publicClass;
            _readers = new List<ReaderInfo>();
            _extractResWFiles = extractingResWFiles;
            _resWOutputDirectory = resWOutputDirectory;
#if FEATURE_RESGENCACHE
            _portableLibraryCacheInfo = new List<ResGenDependencies.PortableLibraryFile>();
#endif

#if FEATURE_ASSEMBLY_LOADFROM
            // If references were passed in, we will have to give the ResxResourceReader an object
            // by which it can resolve types that are referenced from within the .RESX.
            if ((_assemblyFiles != null) && (_assemblyFiles.Length > 0))
            {
                _typeResolver = new AssemblyNamesTypeResolutionService(_assemblyFiles);
            }
#endif

            try
            {
#if FEATURE_ASSEMBLY_LOADFROM
                // Install assembly resolution event handler.
                _eventHandler = new ResolveEventHandler(ResolveAssembly);
                AppDomain.CurrentDomain.AssemblyResolve += _eventHandler;
#endif

                for (int i = 0; i < _inFiles.Count; ++i)
                {
                    string outputSpec = _extractResWFiles ? resWOutputDirectory : _outFiles[i].ItemSpec;
                    if (!ProcessFile(_inFiles[i].ItemSpec, outputSpec))
                    {
                        // Since we failed, remove items from OutputResources.  Note when extracting ResW
                        // files, we won't have added anything to OutputResources up front though.
                        if (!_extractResWFiles)
                        {
                            UnsuccessfullyCreatedOutFiles.Add(outputSpec);
                        }
                    }
                }
            }
            finally
            {
#if FEATURE_ASSEMBLY_LOADFROM
                // Remove the event handler.
                AppDomain.CurrentDomain.AssemblyResolve -= _eventHandler;
                _eventHandler = null;
#endif
            }
        }

#if FEATURE_ASSEMBLY_LOADFROM
        /// <summary>
        /// Callback to resolve assembly names to assemblies.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="args"></param>
        /// <returns></returns>
        internal Assembly ResolveAssembly(object sender, ResolveEventArgs args)
        {
            AssemblyNameExtension requestedAssemblyName = new AssemblyNameExtension(args.Name);

            if (_assemblyFiles != null)
            {
                // Populate the list of assembly names for all passed-in references if it hasn't 
                // been populated already.
                if (_assemblyNames == null)
                {
                    _assemblyNames = new AssemblyNameExtension[_assemblyFiles.Length];
                    for (int i = 0; i < _assemblyFiles.Length; i++)
                    {
                        ITaskItem assemblyFile = _assemblyFiles[i];
                        _assemblyNames[i] = null;

                        if (assemblyFile.ItemSpec != null && FileSystems.Default.FileExists(assemblyFile.ItemSpec))
                        {
                            string fusionName = assemblyFile.GetMetadata(ItemMetadataNames.fusionName);
                            if (!String.IsNullOrEmpty(fusionName))
                            {
                                _assemblyNames[i] = new AssemblyNameExtension(fusionName);
                            }
                            else
                            {
                                // whoever passed us this reference wasn't polite enough to also 
                                // give us a metadata with the fusion name.  Trying to load up every 
                                // assembly here would take a lot of time, so just stick the assembly 
                                // file name (which we assume generally maps to the simple name) into 
                                // the list instead. If there's a fusion name that matches, we'll get 
                                // that first; otherwise there's a good chance that if the simple name
                                // matches the file name, it's a good match.  
                                _assemblyNames[i] = new AssemblyNameExtension(Path.GetFileNameWithoutExtension(assemblyFile.ItemSpec));
                            }
                        }
                    }
                }

                // Loop through all the references passed in, and see if any of them have an assembly
                // name that exactly matches the requested one.
                for (int i = 0; i < _assemblyNames.Length; i++)
                {
                    AssemblyNameExtension candidateAssemblyName = _assemblyNames[i];

                    if (candidateAssemblyName != null)
                    {
                        if (candidateAssemblyName.CompareTo(requestedAssemblyName) == 0)
                        {
                            return Assembly.UnsafeLoadFrom(_assemblyFiles[i].ItemSpec);
                        }
                    }
                }

                // If none of the referenced assembly names matches exactly, try to find one that
                // has the same base name.  This is here to fix bug where the 
                // serialized data inside the .RESX referred to the assembly just by the base name,
                // omitting the version, culture, publickeytoken information.
                for (int i = 0; i < _assemblyNames.Length; i++)
                {
                    AssemblyNameExtension candidateAssemblyName = _assemblyNames[i];

                    if (candidateAssemblyName != null)
                    {
                        if (String.Compare(requestedAssemblyName.Name, candidateAssemblyName.Name, StringComparison.CurrentCultureIgnoreCase) == 0)
                        {
                            return Assembly.UnsafeLoadFrom(_assemblyFiles[i].ItemSpec);
                        }
                    }
                }
            }

            return null;
        }
#endif

#region Code from ResGen.EXE

        /// <summary>
        /// Read all resources from a file and write to a new file in the chosen format
        /// </summary>
        /// <remarks>Uses the input and output file extensions to determine their format</remarks>
        /// <param name="inFile">Input resources file</param>
        /// <param name="outFile">Output resources file</param>
        /// <returns>True if conversion was successful, otherwise false</returns>
        private bool ProcessFile(string inFile, string outFileOrDir)
        {
            Format inFileFormat = GetFormat(inFile);
            if (inFileFormat == Format.Error)
            {
                // GetFormat would have logged an error.
                return false;
            }
            if (inFileFormat != Format.Assembly) // outFileOrDir is a directory when the input file is an assembly
            {
                Format outFileFormat = GetFormat(outFileOrDir);
                if (outFileFormat == Format.Assembly)
                {
                    _logger.LogErrorFromResources("GenerateResource.CannotWriteAssembly", outFileOrDir);
                    return false;
                }
                else if (outFileFormat == Format.Error)
                {
                    return false;
                }
            }

            if (!_extractResWFiles)
            {
                _logger.LogMessageFromResources("GenerateResource.ProcessingFile", inFile, outFileOrDir);
            }

            // Reset state
            _readers = new List<ReaderInfo>();

            try
            {
                ReadResources(inFile, _useSourcePath, outFileOrDir);
            }
            catch (ArgumentException ae)
            {
                if (ae.InnerException is XmlException)
                {
                    XmlException xe = (XmlException) ae.InnerException;
                    _logger.LogErrorWithCodeFromResources(null, FileUtilities.GetFullPathNoThrow(inFile), xe.LineNumber,
                        xe.LinePosition, 0, 0, "General.InvalidResxFile", xe.Message);
                }
                else
                {
                    _logger.LogErrorWithCodeFromResources(null, FileUtilities.GetFullPathNoThrow(inFile), 0, 0, 0, 0,
                        "General.InvalidResxFile", ae.Message);
                }
                return false;
            }
            catch (TextFileException tfe)
            {
                // Used to pass back error context from ReadTextResources to here.
                _logger.LogErrorWithCodeFromResources(null, tfe.FileName, tfe.LineNumber, tfe.LinePosition, 1, 1,
                    "GenerateResource.MessageTunnel", tfe.Message);
                return false;
            }
            catch (XmlException xe)
            {
                _logger.LogErrorWithCodeFromResources(null, FileUtilities.GetFullPathNoThrow(inFile), xe.LineNumber,
                    xe.LinePosition, 0, 0, "General.InvalidResxFile", xe.Message);
                return false;
            }
            catch (Exception e) when (
#if FEATURE_RESGENCACHE
                                      e is SerializationException ||
#endif
                                      e is TargetInvocationException)
            {
                // DDB #9819
                // SerializationException and TargetInvocationException can occur when trying to deserialize a type from a resource format (typically with other exceptions inside)
                // This is a bug in the type being serialized, so the best we can do is dump diagnostic information and move on to the next input resource file.
                _logger.LogErrorWithCodeFromResources(null, FileUtilities.GetFullPathNoThrow(inFile), 0, 0, 0, 0,
                    "General.InvalidResxFile", e.Message);

                // Log the stack, so the problem with the type in the .resx is diagnosable by the customer
                _logger.LogErrorFromException(e, /* stack */ true, /* inner exceptions */ true,
                    FileUtilities.GetFullPathNoThrow(inFile));
                return false;
            }
            catch (Exception e) when (ExceptionHandling.IsIoRelatedException(e))
            {
                // Regular IO error
                _logger.LogErrorWithCodeFromResources(null, FileUtilities.GetFullPathNoThrow(inFile), 0, 0, 0, 0,
                    "General.InvalidResxFile", e.Message);
                return false;
            }

            string currentOutputFile = null;
            string currentOutputDirectory = null;
            string currentOutputSourceCodeFile = null;
            bool currentOutputDirectoryAlreadyExisted = true;

            try
            {
                if (GetFormat(inFile) == Format.Assembly)
                {
#if FEATURE_RESGENCACHE
                    // Prepare cache data
                    ResGenDependencies.PortableLibraryFile library = new ResGenDependencies.PortableLibraryFile(inFile);
#endif
                    List<string> resWFilesForThisAssembly = new List<string>();

                    foreach (ReaderInfo reader in _readers)
                    {
                        String currentOutputFileNoPath = reader.outputFileName + ".resw";
                        currentOutputFile = null;
                        currentOutputDirectoryAlreadyExisted = true;
                        string priDirectory = Path.Combine(outFileOrDir ?? String.Empty,
                            reader.assemblySimpleName);
                        currentOutputDirectory = Path.Combine(priDirectory,
                            reader.cultureName ?? String.Empty);

                        if (!FileSystems.Default.DirectoryExists(currentOutputDirectory))
                        {
                            currentOutputDirectoryAlreadyExisted = false;
                            Directory.CreateDirectory(currentOutputDirectory);
                        }
                        currentOutputFile = Path.Combine(currentOutputDirectory, currentOutputFileNoPath);

                        // For very long resource names, this directory structure may be too deep.
                        // If so, assume that the name is so long it will already uniquely distinguish itself.
                        // However for shorter names we'd still prefer to use the assembly simple name
                        // in the path to avoid conflicts.
                        currentOutputFile = EnsurePathIsShortEnough(currentOutputFile, currentOutputFileNoPath,
                            outFileOrDir, reader.cultureName);

                        if (currentOutputFile == null)
                        {
                            // We couldn't generate a file name short enough to handle this.  Fail but continue.
                            continue;
                        }

                        // Always write the output file here - other logic prevents us from processing this 
                        // file for incremental builds if everything was up to date.
                        WriteResources(reader, currentOutputFile);

                        string escapedOutputFile = EscapingUtilities.Escape(currentOutputFile);
                        ITaskItem newOutputFile = new TaskItem(escapedOutputFile);
                        resWFilesForThisAssembly.Add(escapedOutputFile);
                        newOutputFile.SetMetadata("ResourceIndexName", reader.assemblySimpleName);
#if FEATURE_RESGENCACHE
                        library.AssemblySimpleName = reader.assemblySimpleName;
#endif
                        if (reader.fromNeutralResources)
                        {
                            newOutputFile.SetMetadata("NeutralResourceLanguage", reader.cultureName);
#if FEATURE_RESGENCACHE
                            library.NeutralResourceLanguage = reader.cultureName;
#endif
                        }
                        ExtractedResWFiles.Add(newOutputFile);
                    }

#if FEATURE_RESGENCACHE
                    library.OutputFiles = resWFilesForThisAssembly.ToArray();
                    _portableLibraryCacheInfo.Add(library);
#endif
                }
                else
                {
                    currentOutputFile = outFileOrDir;
                    ErrorUtilities.VerifyThrow(_readers.Count == 1,
                        "We have no readers, or we have multiple readers & are ignoring subsequent ones.  Num readers: {0}",
                        _readers.Count);
                    WriteResources(_readers[0], outFileOrDir);
                }

                if (_stronglyTypedLanguage != null)
                {
                    try
                    {
                        ErrorUtilities.VerifyThrow(_readers.Count == 1,
                            "We have no readers, or we have multiple readers & are ignoring subsequent ones.  Num readers: {0}",
                            _readers.Count);
                        CreateStronglyTypedResources(_readers[0], outFileOrDir, inFile, out currentOutputSourceCodeFile);
                    }
                    catch (Exception e) when (ExceptionHandling.IsIoRelatedException(e))
                    {
                        // IO Error
                        _logger.LogErrorWithCodeFromResources("GenerateResource.CannotWriteSTRFile",
                            _stronglyTypedFilename, e.Message);

                        if (FileSystems.Default.FileExists(outFileOrDir)
                            && GetFormat(inFile) != Format.Assembly
                            // outFileOrDir is a directory when the input file is an assembly
                            && GetFormat(outFileOrDir) != Format.Assembly)
                            // Never delete an assembly since we don't ever actually write to assemblies.
                        {
                            RemoveCorruptedFile(outFileOrDir);
                        }
                        if (currentOutputSourceCodeFile != null)
                        {
                            RemoveCorruptedFile(currentOutputSourceCodeFile);
                        }
                        return false;
                    }
                }
            }
            catch (IOException io)
            {
                if (currentOutputFile != null)
                {
                    _logger.LogErrorWithCodeFromResources("GenerateResource.CannotWriteOutput",
                        FileUtilities.GetFullPathNoThrow(currentOutputFile), io.Message);
                    if (FileSystems.Default.FileExists(currentOutputFile))
                    {
                        if (GetFormat(currentOutputFile) != Format.Assembly)
                            // Never delete an assembly since we don't ever actually write to assemblies.
                        {
                            RemoveCorruptedFile(currentOutputFile);
                        }
                    }
                }

                if (currentOutputDirectory != null &&
                    currentOutputDirectoryAlreadyExisted == false)
                {
                    // Do not annoy the user by removing an empty directory we did not create.
                    try
                    {
                        Directory.Delete(currentOutputDirectory); // Remove output directory if empty
                    }
                    catch (Exception e)
                    {
                        // Fail silently (we are not even checking if the call to File.Delete succeeded)
                        if (ExceptionHandling.IsCriticalException(e))
                        {
                            throw;
                        }
                    }
                }
                return false;
            }
            catch (Exception e) when (
#if FEATURE_RESGENCACHE
                                      e is SerializationException ||
#endif
                                      e is TargetInvocationException)
            {
                // DDB #9819
                // SerializationException and TargetInvocationException can occur when trying to serialize a type into a resource format (typically with other exceptions inside)
                // This is a bug in the type being serialized, so the best we can do is dump diagnostic information and move on to the next input resource file.
                _logger.LogErrorWithCodeFromResources("GenerateResource.CannotWriteOutput",
                    FileUtilities.GetFullPathNoThrow(inFile), e.Message); // Input file is more useful to log

                // Log the stack, so the problem with the type in the .resx is diagnosable by the customer
                _logger.LogErrorFromException(e, /* stack */ true, /* inner exceptions */ true,
                    FileUtilities.GetFullPathNoThrow(inFile));
                return false;
            }
            catch (Exception e) when (ExceptionHandling.IsIoRelatedException(e))
            {
                // Regular IO error
                _logger.LogErrorWithCodeFromResources("GenerateResource.CannotWriteOutput",
                    FileUtilities.GetFullPathNoThrow(currentOutputFile), e.Message);
                return false;
            }

            return true;
        }

        /// <summary>
        /// For very long resource names, the directory structure we generate may be too deep.
        /// If so, assume that the name is so long it will already uniquely distinguish itself.
        /// However for shorter names we'd still prefer to use the assembly simple name
        /// in the path to avoid conflicts.
        /// </summary>
        /// <param name="currentOutputFile">The current path name</param>
        /// <param name="currentOutputFileNoPath">The current file name without a path.</param>
        /// <param name="outputDirectory">Output directory path</param>
        /// <param name="cultureName">culture for this resource</param>
        /// <returns>The current path or a shorter one.</returns>
        private string EnsurePathIsShortEnough(string currentOutputFile, string currentOutputFileNoPath, string outputDirectory, string cultureName)
        {
            // File names >= 260 characters won't work.  File names of exactly 259 characters are odd though.
            // They seem to work with Notepad and Windows Explorer, but not with MakePri.  They don't work
            // reliably with cmd's dir command either (depending on whether you use absolute or relative paths
            // and whether there are quotes around the name).
            const int EffectiveMaxPath = 258;   // Everything <= EffectiveMaxPath should work well.
            bool success = false;
            try
            {
                currentOutputFile = Path.GetFullPath(currentOutputFile);
                success = currentOutputFile.Length <= EffectiveMaxPath;
            }
            catch (PathTooLongException)
            {
                success = false;
            }

            if (!success)
            {
                string shorterPath = Path.Combine(outputDirectory ?? String.Empty, cultureName ?? String.Empty);
                if (!FileSystems.Default.DirectoryExists(shorterPath))
                {
                    Directory.CreateDirectory(shorterPath);
                }
                currentOutputFile = Path.Combine(shorterPath, currentOutputFileNoPath);

                // Try again
                try
                {
                    currentOutputFile = Path.GetFullPath(currentOutputFile);
                    success = currentOutputFile.Length <= EffectiveMaxPath;
                }
                catch (PathTooLongException)
                {
                    success = false;
                }

                // Can't do anything more without violating correctness.
                if (!success)
                {
                    _logger.LogErrorWithCodeFromResources("GenerateResource.PathTooLong", currentOutputFile);
                    currentOutputFile = null;
                    // We've logged an error message.  This MSBuild task will fail, but can continue processing other input (to find other errors).
                }
            }
            return currentOutputFile;
        }

        /// <summary>
        /// Remove a corrupted file, with error handling and a warning if we fail.
        /// </summary>
        /// <param name="filename">Full path to file to delete</param>
        private void RemoveCorruptedFile(string filename)
        {
            _logger.LogWarningWithCodeFromResources("GenerateResource.CorruptOutput", FileUtilities.GetFullPathNoThrow(filename));
            try
            {
                File.Delete(filename);
            }
            catch (Exception deleteException)
            {
                _logger.LogWarningWithCodeFromResources("GenerateResource.DeleteCorruptOutputFailed", FileUtilities.GetFullPathNoThrow(filename), deleteException.Message);

                if (ExceptionHandling.NotExpectedException(deleteException))
                {
                    throw;
                }
            }
        }

        /// <summary>
        /// Figure out the format of an input resources file from the extension
        /// </summary>
        /// <param name="filename">Input resources file</param>
        /// <returns>Resources format</returns>
        private Format GetFormat(string filename)
        {
            string extension = String.Empty;

            try
            {
                extension = Path.GetExtension(filename);
            }
            catch (ArgumentException ex)
            {
                _logger.LogErrorWithCodeFromResources("GenerateResource.InvalidFilename", filename, ex.Message);
                return Format.Error;
            }

            if (String.Compare(extension, ".txt", StringComparison.OrdinalIgnoreCase) == 0 ||
                String.Compare(extension, ".restext", StringComparison.OrdinalIgnoreCase) == 0)
            {
                return Format.Text;
            }
            else if (String.Compare(extension, ".resx", StringComparison.OrdinalIgnoreCase) == 0 ||
                     String.Compare(extension, ".resw", StringComparison.OrdinalIgnoreCase) == 0)
            {
                return Format.XML;
            }
            else if (String.Compare(extension, ".resources.dll", StringComparison.OrdinalIgnoreCase) == 0 ||
                     String.Compare(extension, ".dll", StringComparison.OrdinalIgnoreCase) == 0 ||
                     String.Compare(extension, ".exe", StringComparison.OrdinalIgnoreCase) == 0)
            {
                return Format.Assembly;
            }
            else if (String.Compare(extension, ".resources", StringComparison.OrdinalIgnoreCase) == 0)
            {
                return Format.Binary;
            }
            else
            {
                _logger.LogErrorWithCodeFromResources("GenerateResource.UnknownFileExtension", Path.GetExtension(filename), filename);
                return Format.Error;
            }
        }

        /// <summary>
        /// Text files are just name/value pairs.  ResText is the same format
        /// with a unique extension to work around some ambiguities with MSBuild
        /// ResX is our existing XML format from V1.
        /// </summary>
        private enum Format
        {
            Text, // .txt or .restext
            XML, // .resx
            Binary, // .resources
            Assembly, // .dll, .exe or .resources.dll
            Error, // anything else
        }

        /// <summary>
        /// Reads the resources out of the specified file and populates the
        /// resources hashtable.
        /// </summary>
        /// <param name="filename">Filename to load</param>
        /// <param name="shouldUseSourcePath">Whether to resolve paths in the 
        /// resources file relative to the resources file location</param>
        /// <param name="outFileOrDir"> Output file or directory. </param>
        private void ReadResources(String filename, bool shouldUseSourcePath, String outFileOrDir)
        {
            Format format = GetFormat(filename);

            if (format == Format.Assembly) // Multiple input .resources files within one assembly
            {
#if FEATURE_ASSEMBLY_LOADFROM
                ReadAssemblyResources(filename, outFileOrDir);
#else
                _logger.LogError("Reading resources from Assembly not supported on .NET Core MSBuild");
#endif
            }
            else
            {
                ReaderInfo reader = new ReaderInfo();
                _readers.Add(reader);
                switch (format)
                {
                    case Format.Text:
                        ReadTextResources(reader, filename);
                        break;

                    case Format.XML:
#if FEATURE_RESX_RESOURCE_READER
                        ResXResourceReader resXReader = null;
                        if (_typeResolver != null)
                        {
                            resXReader = new ResXResourceReader(filename, _typeResolver);
                        }
                        else
                        {
                            resXReader = new ResXResourceReader(filename);
                        }

                        if (shouldUseSourcePath)
                        {
                            String fullPath = Path.GetFullPath(filename);
                            resXReader.BasePath = Path.GetDirectoryName(fullPath);
                        }
                        // ReadResources closes the reader for us
                        ReadResources(reader, resXReader, filename);
                        break;
#else

                        using (var xmlReader = new XmlTextReader(filename))
                        {
                            xmlReader.WhitespaceHandling = WhitespaceHandling.None;
                            XDocument doc = XDocument.Load(xmlReader, LoadOptions.PreserveWhitespace);
                            foreach (XElement dataElem in doc.Element("root").Elements("data"))
                            {
                                string name = dataElem.Attribute("name").Value;
                                string value = dataElem.Element("value").Value;
                                AddResource(reader, name, value, filename);
                            }
                        }
                        break;
#endif
                    case Format.Binary:
#if FEATURE_RESX_RESOURCE_READER
                        ReadResources(reader, new ResourceReader(filename), filename); // closes reader for us
#else
                        _logger.LogError("ResGen.exe not supported on .NET Core MSBuild");
#endif
                        break;

                    default:
                        // We should never get here, we've already checked the format
                        Debug.Fail("Unknown format " + format.ToString());
                        return;
                }
                _logger.LogMessageFromResources(MessageImportance.Low, "GenerateResource.ReadResourceMessage", reader.resources.Count, filename);
            }
        }

#if FEATURE_ASSEMBLY_LOADFROM
        /// <summary>
        /// Reads resources from an assembly.
        /// </summary>
        /// <param name="name"></param>
        /// <param name="outFileOrDir"> Output file or directory. </param>
        /// <remarks> This should not run for Framework assemblies. </remarks>
        internal void ReadAssemblyResources(String name, String outFileOrDir)
        {
            // If something else in the solution failed to build...
            if (!FileSystems.Default.FileExists(name))
            {
                _logger.LogErrorWithCodeFromResources("GenerateResource.MissingFile", name);
                return;
            }

            Assembly a = null;
            bool mainAssembly = false;
            bool failedLoadingCultureInfo = false;
            NeutralResourcesLanguageAttribute neutralResourcesLanguageAttribute = null;
            AssemblyName assemblyName = null;

            try
            {
                a = Assembly.UnsafeLoadFrom(name);
                assemblyName = a.GetName();

                if (_extractResWFiles)
                {
                    var targetFrameworkAttribute = a.GetCustomAttribute<TargetFrameworkAttribute>();
                    if (
                            targetFrameworkAttribute != null &&
                            (
                                targetFrameworkAttribute.FrameworkName.StartsWith("Silverlight,", StringComparison.OrdinalIgnoreCase) ||
                                targetFrameworkAttribute.FrameworkName.StartsWith("WindowsPhone,", StringComparison.OrdinalIgnoreCase)
                            )
                        )
                    {
                        // Skip Silverlight assemblies.
                        _logger.LogMessageFromResources("GenerateResource.SkippingExtractingFromNonSupportedFramework", name, targetFrameworkAttribute.FrameworkName);
                        return;
                    }

                    _logger.LogMessageFromResources("GenerateResource.ExtractingResWFiles", name, outFileOrDir);
                }

                CultureInfo ci = null;
                try
                {
                    ci = assemblyName.CultureInfo;
                }
                catch (ArgumentException e)
                {
                    _logger.LogWarningWithCodeFromResources(null, name, 0, 0, 0, 0, "GenerateResource.CreatingCultureInfoFailed", e.GetType().Name, e.Message, assemblyName.ToString());
                    failedLoadingCultureInfo = true;
                }

                if (!failedLoadingCultureInfo)
                {
                    mainAssembly = ci.Equals(CultureInfo.InvariantCulture);
                    neutralResourcesLanguageAttribute = CheckAssemblyCultureInfo(name, assemblyName, ci, a, mainAssembly);
                }  // if (!failedLoadingCultureInfo)
            }
            catch (BadImageFormatException)
            {
                // If we're extracting ResW files, this task is being run on all DLL's referred to by the project.
                // That may potentially include C++ libraries & immersive (non-portable) class libraries, which don't have resources.
                // We can't easily filter those.  We can simply skip them.
                return;
            }
            catch (Exception e)
            {
                if (ExceptionHandling.IsCriticalException(e))
                    throw;
                _logger.LogErrorWithCodeFromResources("GenerateResource.CannotLoadAssemblyLoadFromFailed", name, e);
            }

            if (a != null)
            {
                String[] resources = a.GetManifestResourceNames();
                CultureInfo satCulture = null;
                String expectedExt = null;
                if (!failedLoadingCultureInfo)
                {
                    satCulture = assemblyName.CultureInfo;
                    if (!satCulture.Equals(CultureInfo.InvariantCulture))
                        expectedExt = '.' + satCulture.Name + ".resources";
                }

                foreach (String resName in resources)
                {
                    if (!resName.EndsWith(".resources", StringComparison.OrdinalIgnoreCase)) // Skip non-.resources assembly blobs
                        continue;

                    if (mainAssembly)
                    {
                        if (CultureInfo.InvariantCulture.CompareInfo.IsSuffix(resName, ".en-US.resources", CompareOptions.IgnoreCase))
                        {
                            _logger.LogErrorFromResources("GenerateResource.ImproperlyBuiltMainAssembly", resName, name);
                            continue;
                        }

                        if (neutralResourcesLanguageAttribute == null)
                        {
                            _logger.LogWarningWithCodeFromResources(null, name, 0, 0, 0, 0, "GenerateResource.MainAssemblyMissingNeutralResourcesLanguage", name);
                            break;
                        }
                    }
                    else if (!failedLoadingCultureInfo && !CultureInfo.InvariantCulture.CompareInfo.IsSuffix(resName, expectedExt, CompareOptions.IgnoreCase))
                    {
                        _logger.LogErrorFromResources("GenerateResource.ImproperlyBuiltSatelliteAssembly", resName, expectedExt, name);
                        continue;
                    }

                    try
                    {
                        Stream s = a.GetManifestResourceStream(resName);
                        using (IResourceReader rr = new ResourceReader(s))
                        {
                            ReaderInfo reader = new ReaderInfo();
                            if (mainAssembly)
                            {
                                reader.fromNeutralResources = true;
                                reader.assemblySimpleName = assemblyName.Name;
                            }
                            else
                            {
                                Debug.Assert(assemblyName.Name.EndsWith(".resources", StringComparison.OrdinalIgnoreCase));
                                reader.assemblySimpleName = assemblyName.Name.Remove(assemblyName.Name.Length - 10);  // Remove .resources from satellite assembly name
                            }
                            reader.outputFileName = resName.Remove(resName.Length - 10); // Remove the .resources extension
                            if (satCulture != null && !String.IsNullOrEmpty(satCulture.Name))
                            {
                                reader.cultureName = satCulture.Name;
                            }
                            else if (neutralResourcesLanguageAttribute != null && !String.IsNullOrEmpty(neutralResourcesLanguageAttribute.CultureName))
                            {
                                reader.cultureName = neutralResourcesLanguageAttribute.CultureName;
                            }

                            if (reader.cultureName != null)
                            {
                                // Remove the culture from the filename
                                if (reader.outputFileName.EndsWith("." + reader.cultureName, StringComparison.OrdinalIgnoreCase))
                                    reader.outputFileName = reader.outputFileName.Remove(reader.outputFileName.Length - (reader.cultureName.Length + 1));
                            }
                            _readers.Add(reader);

                            foreach (DictionaryEntry pair in rr)
                            {
                                AddResource(reader, (string)pair.Key, pair.Value, resName);
                            }
                        }
                    }
                    catch (FileNotFoundException)
                    {
                        _logger.LogErrorWithCodeFromResources(null, name, 0, 0, 0, 0, "GenerateResource.NoResourcesFileInAssembly", resName);
                    }
                }
            }

            var satelliteAssemblies = _satelliteInFiles.Where(ti => ti.GetMetadata("OriginalItemSpec").Equals(name, StringComparison.OrdinalIgnoreCase));

            foreach (var satelliteAssembly in satelliteAssemblies)
            {
                ReadAssemblyResources(satelliteAssembly.ItemSpec, outFileOrDir);
            }
        }

        /// <summary>
        /// Checks the consistency of the CultureInfo and NeutralResourcesLanguageAttribute settings.  
        /// </summary>
        /// <param name="name">Assembly's file name</param>
        /// <param name="assemblyName">AssemblyName of this assembly</param>
        /// <param name="culture">Assembly's CultureInfo</param>
        private NeutralResourcesLanguageAttribute CheckAssemblyCultureInfo(String name, AssemblyName assemblyName, CultureInfo culture, Assembly a, bool mainAssembly)
        {
            NeutralResourcesLanguageAttribute neutralResourcesLanguageAttribute = null;
            if (mainAssembly)
            {
                Object[] attrs = a.GetCustomAttributes(typeof(NeutralResourcesLanguageAttribute), false);
                if (attrs.Length != 0)
                {
                    neutralResourcesLanguageAttribute = (NeutralResourcesLanguageAttribute)attrs[0];
                    bool fallbackToSatellite = neutralResourcesLanguageAttribute.Location == UltimateResourceFallbackLocation.Satellite;
                    if (!fallbackToSatellite && neutralResourcesLanguageAttribute.Location != UltimateResourceFallbackLocation.MainAssembly)
                        _logger.LogWarningWithCodeFromResources(null, name, 0, 0, 0, 0, "GenerateResource.UnrecognizedUltimateResourceFallbackLocation", neutralResourcesLanguageAttribute.Location, name);
                    // This MSBuild task needs to not report an error for main assemblies that don't have managed resources.
                }
            }
            else
            {  // Satellite assembly, or a mal-formed main assembly
                // Additional error checking from ResView.
                if (!assemblyName.Name.EndsWith(".resources", StringComparison.OrdinalIgnoreCase))
                {
                    _logger.LogWarningWithCodeFromResources(null, name, 0, 0, 0, 0, "GenerateResource.SatelliteOrMalformedAssembly", name, culture.Name, assemblyName.Name);
                    return null;
                }
                Type[] types = a.GetTypes();
                if (types.Length > 0)
                {
                    _logger.LogWarningWithCodeFromResources("GenerateResource.SatelliteAssemblyContainsCode", name);
                }

                if (!ContainsProperlyNamedResourcesFiles(a, false))
                    _logger.LogWarningWithCodeFromResources("GenerateResource.SatelliteAssemblyContainsNoResourcesFile", assemblyName.CultureInfo.Name);
            }
            return neutralResourcesLanguageAttribute;
        }

        private static bool ContainsProperlyNamedResourcesFiles(Assembly a, bool mainAssembly)
        {
            String postfix = mainAssembly ? ".resources" : a.GetName().CultureInfo.Name + ".resources";
            foreach (String manifestResourceName in a.GetManifestResourceNames())
                if (manifestResourceName.EndsWith(postfix, StringComparison.OrdinalIgnoreCase))
                    return true;
            return false;
        }
#endif

        /// <summary>
        /// Write resources from the resources ArrayList to the specified output file
        /// </summary>
        /// <param name="filename">Output resources file</param>
        private void WriteResources(ReaderInfo reader, String filename)
        {
            Format format = GetFormat(filename);
            switch (format)
            {
                case Format.Text:
                    WriteTextResources(reader, filename);
                    break;

                case Format.XML:
#if FEATURE_RESX_RESOURCE_READER
                    WriteResources(reader, new ResXResourceWriter(filename)); // closes writer for us
#else
                    _logger.LogError(format.ToString() + " not supported on .NET Core MSBuild");
#endif
                    break;


                case Format.Assembly:
                    _logger.LogErrorFromResources("GenerateResource.CannotWriteAssembly", filename);
                    break;

                case Format.Binary:

                    WriteResources(reader, new ResourceWriter(File.OpenWrite(filename))); // closes writer for us
                    break;


                default:
                    // We should never get here, we've already checked the format
                    Debug.Fail("Unknown format " + format.ToString());
                    break;
            }
        }


        /// <summary>
        /// Create a strongly typed resource class
        /// </summary>
        /// <param name="outFile">Output resource filename, for defaulting the class filename</param>
        /// <param name="inputFileName">Input resource filename, for error messages</param>
        private void CreateStronglyTypedResources(ReaderInfo reader, String outFile, String inputFileName, out String sourceFile)
        {
#if FEATURE_CODEDOM
            CodeDomProvider provider = null;

            if (!TryCreateCodeDomProvider(_logger, _stronglyTypedLanguage, out provider))
            {
                sourceFile = null;
                return;
            }

            // Default the class name if we need to
            if (_stronglyTypedClassName == null)
            {
                _stronglyTypedClassName = Path.GetFileNameWithoutExtension(outFile);
            }

            // Default the filename if we need to
            if (_stronglyTypedFilename == null)
            {
                _stronglyTypedFilename = GenerateDefaultStronglyTypedFilename(provider, outFile);
            }
            sourceFile = this.StronglyTypedFilename;

            _logger.LogMessageFromResources("GenerateResource.CreatingSTR", _stronglyTypedFilename);

            // Generate the STR class
            String[] errors;
            bool generateInternalClass = !_stronglyTypedClassIsPublic;
            //StronglyTypedResourcesNamespace can be null and this is ok.
            // If it is null then the default namespace (=stronglyTypedNamespace) is used.
            CodeCompileUnit ccu = StronglyTypedResourceBuilder.Create(
                    reader.resourcesHashTable,
                    _stronglyTypedClassName,
                    _stronglyTypedNamespace,
                    _stronglyTypedResourcesNamespace,
                    provider,
                    generateInternalClass,
                    out errors
                    );

            CodeGeneratorOptions codeGenOptions = new CodeGeneratorOptions();
            using (TextWriter output = new StreamWriter(_stronglyTypedFilename))
            {
                provider.GenerateCodeFromCompileUnit(ccu, output, codeGenOptions);
            }

            if (errors.Length > 0)
            {
                _logger.LogErrorWithCodeFromResources("GenerateResource.ErrorFromCodeDom", inputFileName);
                foreach (String error in errors)
                {
                    _logger.LogErrorWithCodeFromResources("GenerateResource.CodeDomError", error);
                }
            }
            else
            {
                // No errors, and no exceptions - we presumably did create the STR class file
                // and it should get added to FilesWritten. So set a flag to indicate this.
                _stronglyTypedResourceSuccessfullyCreated = true;
            }
#else
            sourceFile = null;
            _logger.LogError("Generating strongly typed resource files not currently supported on .NET Core MSBuild");
#endif
        }

#if FEATURE_CODEDOM
        /// <summary>
        /// If no strongly typed resource class filename was specified, we come up with a default based on the
        /// input file name and the default language extension. 
        /// </summary>
        /// <comments>
        /// Broken out here so it can be called from GenerateResource class.
        /// </comments>
        /// <param name="provider">A CodeDomProvider for the language</param>
        /// <param name="outputResourcesFile">Name of the output resources file</param>
        /// <returns>Filename for strongly typed resource class</returns>
        public static string GenerateDefaultStronglyTypedFilename(CodeDomProvider provider, string outputResourcesFile)
        {
            return Path.ChangeExtension(outputResourcesFile, provider.FileExtension);
        }

        /// <summary>
        /// Tries to create a CodeDom provider for the specified strongly typed language.  If successful, returns true, 
        /// otherwise returns false. 
        /// </summary>
        /// <comments>
        /// Broken out here so it can be called from GenerateResource class.
        /// Not a true "TryXXX" method, as it still throws if it encounters an exception it doesn't expect.  
        /// </comments>
        /// <param name="stronglyTypedLanguage">The language to create a provider for.</param>
        /// <param name="provider">The provider in question, if one is successfully created.</param>
        /// <returns>True if the provider was successfully created, false otherwise.</returns>
        public static bool TryCreateCodeDomProvider(TaskLoggingHelper logger, string stronglyTypedLanguage, out CodeDomProvider provider)
        {
            provider = null;

            try
            {
                provider = CodeDomProvider.CreateProvider(stronglyTypedLanguage);
            }
#if FEATURE_SYSTEM_CONFIGURATION
            catch (ConfigurationException e)
            {
                logger.LogErrorWithCodeFromResources("GenerateResource.STRCodeDomProviderFailed", stronglyTypedLanguage, e.Message);
                return false;
            }
#endif
            catch (SecurityException e)
            {
                logger.LogErrorWithCodeFromResources("GenerateResource.STRCodeDomProviderFailed", stronglyTypedLanguage, e.Message);
                return false;
            }

            return provider != null;
        }
#endif

#if FEATURE_RESX_RESOURCE_READER
        /// <summary>
        /// Read resources from an XML or binary format file
        /// </summary>
        /// <param name="reader">Appropriate IResourceReader</param>
        /// <param name="fileName">Filename, for error messages</param>
        private void ReadResources(ReaderInfo readerInfo, IResourceReader reader, String fileName)
        {
            using (reader)
            {
                IDictionaryEnumerator resEnum = reader.GetEnumerator();
                while (resEnum.MoveNext())
                {
                    string name = (string)resEnum.Key;
                    object value = resEnum.Value;
                    AddResource(readerInfo, name, value, fileName);
                }
            }
        }
#endif

        /// <summary>
        /// Read resources from a text format file
        /// </summary>
        /// <param name="fileName">Input resources filename</param>
        private void ReadTextResources(ReaderInfo reader, String fileName)
        {
            // Check for byte order marks in the beginning of the input file, but
            // default to UTF-8.
            using (LineNumberStreamReader sr = new LineNumberStreamReader(fileName, new UTF8Encoding(true), true))
            {
                StringBuilder name = new StringBuilder(255);
                StringBuilder value = new StringBuilder(2048);

                int ch = sr.Read();
                while (ch != -1)
                {
                    if (ch == '\n' || ch == '\r')
                    {
                        ch = sr.Read();
                        continue;
                    }

                    // Skip over commented lines or ones starting with whitespace.
                    // Support LocStudio INF format's comment char, ';'
                    if (ch == '#' || ch == '\t' || ch == ' ' || ch == ';')
                    {
                        // comment char (or blank line) - skip line.
                        sr.ReadLine();
                        ch = sr.Read();
                        continue;
                    }
                    // Note that in Beta of version 1 we recommended users should put a [strings] 
                    // section in their file.  Now it's completely unnecessary and can 
                    // only cause bugs.  We will not parse anything using '[' stuff now
                    // and we should give a warning about seeing [strings] stuff.
                    // In V1.1 or V2, we can rip this out completely, I hope.
                    if (ch == '[')
                    {
                        String skip = sr.ReadLine();
                        if (skip.Equals("strings]"))
                            _logger.LogWarningWithCodeFromResources(null, fileName, sr.LineNumber - 1, 1, 0, 0, "GenerateResource.ObsoleteStringsTag");
                        else
                        {
                            throw new TextFileException(_logger.FormatResourceString("GenerateResource.UnexpectedInfBracket", "[" + skip), fileName, sr.LineNumber - 1, 1);
                        }
                        ch = sr.Read();
                        continue;
                    }

                    // Read in name
                    name.Length = 0;
                    while (ch != '=')
                    {
                        if (ch == '\r' || ch == '\n')
                            throw new TextFileException(_logger.FormatResourceString("GenerateResource.NoEqualsInLine", name), fileName, sr.LineNumber, sr.LinePosition);

                        name.Append((char)ch);
                        ch = sr.Read();
                        if (ch == -1)
                            break;
                    }
                    if (name.Length == 0)
                        throw new TextFileException(_logger.FormatResourceString("GenerateResource.NoNameInLine"), fileName, sr.LineNumber, sr.LinePosition);

                    // For the INF file, we must allow a space on both sides of the equals
                    // sign.  Deal with it.
                    if (name[name.Length - 1] == ' ')
                    {
                        name.Length = name.Length - 1;
                    }
                    ch = sr.Read(); // move past =
                    // If it exists, move past the first space after the equals sign.
                    if (ch == ' ')
                        ch = sr.Read();

                    // Read in value
                    value.Length = 0;

                    while (ch != -1)
                    {
                        // Did we read @"\r" or @"\n"?
                        bool quotedNewLine = false;
                        if (ch == '\\')
                        {
                            ch = sr.Read();
                            switch (ch)
                            {
                                case '\\':
                                    // nothing needed
                                    break;
                                case 'n':
                                    ch = '\n';
                                    quotedNewLine = true;
                                    break;
                                case 'r':
                                    ch = '\r';
                                    quotedNewLine = true;
                                    break;
                                case 't':
                                    ch = '\t';
                                    break;
                                case '"':
                                    ch = '\"';
                                    break;
                                case 'u':
                                    char[] hex = new char[4];
                                    int numChars = 4;
                                    int index = 0;
                                    while (numChars > 0)
                                    {
                                        int n = sr.Read(hex, index, numChars);
                                        if (n == 0)
                                            throw new TextFileException(_logger.FormatResourceString("GenerateResource.InvalidEscape", name.ToString(), (char)ch), fileName, sr.LineNumber, sr.LinePosition);
                                        index += n;
                                        numChars -= n;
                                    }
                                    try
                                    {
                                        ch = (char)UInt16.Parse(new String(hex), NumberStyles.HexNumber, CultureInfo.CurrentCulture);
                                    }
                                    catch (FormatException)
                                    {
                                        // We know about this one...
                                        throw new TextFileException(_logger.FormatResourceString("GenerateResource.InvalidHexEscapeValue", name.ToString(), new String(hex)), fileName, sr.LineNumber, sr.LinePosition);
                                    }
                                    catch (OverflowException)
                                    {
                                        // We know about this one, too...
                                        throw new TextFileException(_logger.FormatResourceString("GenerateResource.InvalidHexEscapeValue", name.ToString(), new String(hex)), fileName, sr.LineNumber, sr.LinePosition);
                                    }
                                    quotedNewLine = (ch == '\n' || ch == '\r');
                                    break;

                                default:
                                    throw new TextFileException(_logger.FormatResourceString("GenerateResource.InvalidEscape", name.ToString(), (char)ch), fileName, sr.LineNumber, sr.LinePosition);
                            }
                        }

                        // Consume endline...
                        //   Endline can be \r\n or \n.  But do not treat a 
                        //   quoted newline (ie, @"\r" or @"\n" in text) as a
                        //   real new line.  They aren't the end of a line.
                        if (!quotedNewLine)
                        {
                            if (ch == '\r')
                            {
                                ch = sr.Read();
                                if (ch == -1)
                                {
                                    break;
                                }
                                else if (ch == '\n')
                                {
                                    ch = sr.Read();
                                    break;
                                }
                            }
                            else if (ch == '\n')
                            {
                                ch = sr.Read();
                                break;
                            }
                        }

                        value.Append((char)ch);
                        ch = sr.Read();
                    }

                    // Note that value can be an empty string
                    AddResource(reader, name.ToString(), value.ToString(), fileName, sr.LineNumber, sr.LinePosition);
                }
            }
        }


        /// <summary>
        /// Write resources to an XML or binary format resources file.
        /// </summary>
        /// <remarks>Closes writer automatically</remarks>
        /// <param name="writer">Appropriate IResourceWriter</param>
        private void WriteResources(ReaderInfo reader,
#if FEATURE_RESX_RESOURCE_READER
            IResourceWriter writer)
#else
            ResourceWriter writer)
#endif
        {
            Exception capturedException = null;
            try
            {
                foreach (Entry entry in reader.resources)
                {
                    string key = entry.name;
                    object value = entry.value;
#if FEATURE_RESX_RESOURCE_READER
                    writer.AddResource(key, value);
#else
                    writer.AddResource(key, (string) value);
#endif
                }
            }
            catch (Exception e)
            {
                capturedException = e; // Rethrow this after catching exceptions thrown by Close().
            }
            finally
            {
                if (capturedException == null)
                {
                    writer.Dispose(); // If this throws, exceptions will be caught upstream.
                }
                else
                {
                    // It doesn't hurt to call Close() twice. In the event of a full disk, we *need* to call Close() twice.
                    // In that case, the first time we catch an exception indicating that the XML written to disk is malformed,
                    // specifically an InvalidOperationException: "Token EndElement in state Error would result in an invalid XML document."
                    try { writer.Dispose(); }
                    catch (Exception) { } // We agressively catch all exception types since we already have one we will throw.
                    // The second time we catch the out of disk space exception.
                    try { writer.Dispose(); }
                    catch (Exception) { } // We agressively catch all exception types since we already have one we will throw.
                    throw capturedException; // In the event of a full disk, this is an out of disk space IOException.
                }
            }
        }

        /// <summary>
        /// Write resources to a text format resources file
        /// </summary>
        /// <param name="fileName">Output resources file</param>
        private void WriteTextResources(ReaderInfo reader, String fileName)
        {
            using (StreamWriter writer = FileUtilities.OpenWrite(fileName, false, Encoding.UTF8))
            {
                foreach (Entry entry in reader.resources)
                {
                    String key = entry.name;
                    Object v = entry.value;
                    String value = v as String;
                    if (value == null)
                    {
                        _logger.LogErrorWithCodeFromResources(null, fileName, 0, 0, 0, 0, "GenerateResource.OnlyStringsSupported", key, v.GetType().FullName);
                    }
                    else
                    {
                        // Escape any special characters in the String.
                        value = value.Replace("\\", "\\\\");
                        value = value.Replace("\n", "\\n");
                        value = value.Replace("\r", "\\r");
                        value = value.Replace("\t", "\\t");

                        writer.WriteLine("{0}={1}", key, value);
                    }
                }
            }
        }

        /// <summary>
        /// Add a resource from a text file to the internal data structures
        /// </summary>
        /// <param name="name">Resource name</param>
        /// <param name="value">Resource value</param>
        /// <param name="inputFileName">Input file for messages</param>
        /// <param name="lineNumber">Line number for messages</param>
        /// <param name="linePosition">Column number for messages</param>
        private void AddResource(ReaderInfo reader, string name, object value, String inputFileName, int lineNumber, int linePosition)
        {
            Entry entry = new Entry(name, value);

            if (reader.resourcesHashTable.ContainsKey(name))
            {
                _logger.LogWarningWithCodeFromResources(null, inputFileName, lineNumber, linePosition, 0, 0, "GenerateResource.DuplicateResourceName", name);
                return;
            }

            reader.resources.Add(entry);
            reader.resourcesHashTable.Add(name, value);
        }

        /// <summary>
        /// Add a resource from an XML or binary format file to the internal data structures
        /// </summary>
        /// <param name="name">Resource name</param>
        /// <param name="value">Resource value</param>
        /// <param name="inputFileName">Input file for messages</param>
        private void AddResource(ReaderInfo reader, string name, object value, String inputFileName)
        {
            AddResource(reader, name, value, inputFileName, 0, 0);
        }

        internal sealed class ReaderInfo
        {
            public String outputFileName;
            public String cultureName;
            // We use a list to preserve the resource ordering (primarily for easier testing),
            // but also use a hash table to check for duplicate names.
            public ArrayList resources;
            public Hashtable resourcesHashTable;
            public String assemblySimpleName;  // The main assembly's simple name (ie, no .resources)
            public bool fromNeutralResources;  // Was this from the main assembly (or if the NRLA specified fallback to satellite, that satellite?)

            public ReaderInfo()
            {
                resources = new ArrayList();
                resourcesHashTable = new Hashtable(StringComparer.OrdinalIgnoreCase);
            }
        }

        /// <summary>
        /// Custom StreamReader that provides detailed position information,
        /// used when reading text format resources
        /// </summary>
        internal sealed class LineNumberStreamReader : StreamReader
        {
            // Line numbers start from 1, as well as line position.
            // For better error reporting, set line number to 1 and col to 0.
            private int _lineNumber;
            private int _col;

            internal LineNumberStreamReader(String fileName, Encoding encoding, bool detectEncoding)
                : base(File.Open(fileName, FileMode.Open, FileAccess.Read), encoding, detectEncoding)
            {
                _lineNumber = 1;
                _col = 0;
            }

            internal LineNumberStreamReader(Stream stream)
                : base(stream)
            {
                _lineNumber = 1;
                _col = 0;
            }

            public override int Read()
            {
                int ch = base.Read();
                if (ch != -1)
                {
                    _col++;
                    if (ch == '\n')
                    {
                        _lineNumber++;
                        _col = 0;
                    }
                }
                return ch;
            }

            public override int Read([In, Out] char[] chars, int index, int count)
            {
                int r = base.Read(chars, index, count);
                for (int i = 0; i < r; i++)
                {
                    if (chars[i + index] == '\n')
                    {
                        _lineNumber++;
                        _col = 0;
                    }
                    else
                        _col++;
                }
                return r;
            }

            public override String ReadLine()
            {
                String s = base.ReadLine();
                if (s != null)
                {
                    _lineNumber++;
                    _col = 0;
                }
                return s;
            }

            public override String ReadToEnd()
            {
                throw new NotImplementedException("NYI");
            }

            internal int LineNumber
            {
                get { return _lineNumber; }
            }

            internal int LinePosition
            {
                get { return _col; }
            }
        }

        /// <summary>
        /// For flow of control & passing sufficient error context back 
        /// from ReadTextResources
        /// </summary>
        [Serializable]
        internal sealed class TextFileException : Exception
        {
            private String fileName;
            private int lineNumber;
            private int column;

#if FEATURE_RESGENCACHE
            /// <summary>
            /// Fxcop want to have the correct basic exception constructors implemented
            /// </summary>
            private TextFileException(SerializationInfo info, StreamingContext context)
                : base(info, context)
            {
            }
#endif

            internal TextFileException(String message, String fileName, int lineNumber, int linePosition)
                : base(message)
            {
                this.fileName = fileName;
                this.lineNumber = lineNumber;
                column = linePosition;
            }

            internal String FileName
            {
                get { return fileName; }
            }

            internal int LineNumber
            {
                get { return lineNumber; }
            }

            internal int LinePosition
            {
                get { return column; }
            }
        }

        /// <summary>
        /// Name value resource pair to go in resources list
        /// </summary>
        private class Entry
        {
            public Entry(string name, object value)
            {
                this.name = name;
                this.value = value;
            }

            public string name;
            public object value;
        }
#endregion // Code from ResGen.EXE
    }

#if FEATURE_ASSEMBLY_LOADFROM
    /// <summary>
    /// This implemention of ITypeResolutionService is passed into the ResxResourceReader
    /// class, which calls back into the methods on this class in order to resolve types
    /// and assemblies that are referenced inside of the .RESX.
    /// </summary>
    internal class AssemblyNamesTypeResolutionService : ITypeResolutionService
    {
        private Hashtable _cachedAssemblies;
        private ITaskItem[] _referencePaths;
        private Hashtable _cachedTypes = new Hashtable();

        /// <summary>
        /// Constructor, initialized with the set of resolved reference paths passed
        /// into the GenerateResource task.
        /// </summary>
        /// <param name="referencePaths"></param>
        internal AssemblyNamesTypeResolutionService(ITaskItem[] referencePaths)
        {
            _referencePaths = referencePaths;
        }

        /// <summary>
        /// Not implemented.  Not called by the ResxResourceReader.
        /// </summary>
        /// <param name="name"></param>
        /// <returns></returns>
        public Assembly GetAssembly(AssemblyName name)
        {
            throw new NotSupportedException();
        }

        /// <summary>
        /// Not implemented.  Not called by the ResxResourceReader.
        /// </summary>
        /// <param name="name"></param>
        /// <returns></returns>
        public Assembly GetAssembly(AssemblyName name, bool throwOnError)
        {
            throw new NotSupportedException();
        }

        /// <summary>
        /// Given a path to an assembly, load the assembly if it's not already loaded.
        /// </summary>
        /// <param name="pathToAssembly"></param>
        /// <param name="throwOnError"></param>
        /// <returns></returns>
        private Assembly GetAssemblyByPath(string pathToAssembly, bool throwOnError)
        {
            if (_cachedAssemblies == null)
            {
                _cachedAssemblies = new Hashtable();
            }

            if (!_cachedAssemblies.Contains(pathToAssembly))
            {
                try
                {
                    _cachedAssemblies[pathToAssembly] = Assembly.UnsafeLoadFrom(pathToAssembly);
                }
                catch
                {
                    if (throwOnError)
                    {
                        throw;
                    }
                }
            }

            return (Assembly)_cachedAssemblies[pathToAssembly];
        }

        /// <summary>
        /// Not implemented.  Not called by the ResxResourceReader.
        /// </summary>
        /// <param name="name"></param>
        /// <returns></returns>
        public string GetPathOfAssembly(AssemblyName name)
        {
            throw new NotSupportedException();
        }

        /// <summary>
        /// Returns the type with the specified name.  Searches for the type in all 
        /// of the assemblies passed into the References parameter of the GenerateResource
        /// task.
        /// </summary>
        /// <param name="name"></param>
        /// <returns></returns>
        public Type GetType(string name)
        {
            return GetType(name, true);
        }

        /// <summary>
        /// Returns the type with the specified name.  Searches for the type in all 
        /// of the assemblies passed into the References parameter of the GenerateResource
        /// task.
        /// </summary>
        /// <param name="name"></param>
        /// <param name="throwOnError"></param>
        /// <returns></returns>
        public Type GetType(string name, bool throwOnError)
        {
            return GetType(name, throwOnError, false);
        }

        /// <summary>
        /// Returns the type with the specified name.  Searches for the type in all 
        /// of the assemblies passed into the References parameter of the GenerateResource
        /// task.
        /// </summary>
        /// <param name="name"></param>
        /// <param name="throwOnError"></param>
        /// <param name="ignoreCase"></param>
        /// <returns></returns>
        public Type GetType(string name, bool throwOnError, bool ignoreCase)
        {
            Type resultFromCache = (Type)_cachedTypes[name];

            if (!_cachedTypes.Contains(name))
            {
                // first try to resolve in the GAC
                Type result = Type.GetType(name, false, ignoreCase);

                // did not find it in the GAC, check each assembly 
                if ((result == null) && (_referencePaths != null))
                {
                    foreach (ITaskItem referencePath in _referencePaths)
                    {
                        Assembly a = this.GetAssemblyByPath(referencePath.ItemSpec, throwOnError);
                        if (a != null)
                        {
                            result = a.GetType(name, false, ignoreCase);
                            if (result == null)
                            {
                                int indexOfComma = name.IndexOf(",", StringComparison.Ordinal);
                                if (indexOfComma != -1)
                                {
                                    string shortName = name.Substring(0, indexOfComma);
                                    result = a.GetType(shortName, false, ignoreCase);
                                }
                            }

                            if (result != null)
                            {
                                break;
                            }
                        }
                    }
                }

                if (result == null && throwOnError)
                {
                    ErrorUtilities.VerifyThrowArgument(false, "GenerateResource.CouldNotLoadType", name);
                }

                _cachedTypes[name] = result;
                resultFromCache = result;
            }

            return resultFromCache;
        }

        /// <summary>
        /// Not implemented.  Not called by the ResxResourceReader.
        /// </summary>
        /// <param name="name"></param>
        /// <returns></returns>
        public void ReferenceAssembly(AssemblyName name)
        {
            throw new NotSupportedException();
        }
    }
#endif
}
