/*
 * JBoss, Home of Professional Open Source
 * Copyright 2010, Red Hat, Inc., and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,  
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package javax.enterprise.context;

/**
 * <p>
 * Allows the application to manage the {@linkplain javax.enterprise.context.ConversationScoped conversation context} by marking
 * the current conversation as transient or long-running, specifying a conversation identifier, or setting the conversation
 * timeout.
 * </p>
 * 
 * <p>
 * An instance may be injected:
 * </p>
 * 
 * <pre>
 * &#064;Inject
 * Conversation conversation;
 * </pre>
 * 
 * <p>
 * The conversation timeout is a hint to the container that a conversation should not be destroyed if it has been active within
 * the last given interval in milliseconds.
 * </p>
 * 
 * @see javax.enterprise.context.ConversationScoped &#064;ConversationScoped
 * 
 * @author Pete Muir
 * @author Gavin King
 * 
 */
public interface Conversation {

    /**
     * <p>
     * Mark the current transient conversation long-running. A conversation identifier is generated by the container.
     * </p>
     * 
     * @throws IllegalStateException if the current conversation is already marked long-running.
     */
    public void begin();

    /**
     * <p>
     * Mark the current transient conversation long-running, with a specified identifier.
     * </p>
     * 
     * @param id conversation id
     * @throws IllegalStateException if the current conversation is already marked long-running.
     * @throws IllegalArgumentException if a conversation with the specified identifier already exists.
     */
    public void begin(String id);

    /**
     * <p>
     * Marks the current long-running conversation transient.
     * </p>
     * 
     * @throws IllegalStateException if the current conversation is already marked transient.
     */
    public void end();

    /**
     * <p>
     * Get the identifier of the current long-running conversation.
     * </p>
     * 
     * @return the identifier of the current long-running conversation, or a null value if the current conversation is
     *         transient.
     */
    public String getId();

    /**
     * <p>
     * Get the timeout of the current conversation.
     * </p>
     * 
     * @return the current timeout in milliseconds.
     */
    public long getTimeout();

    /**
     * <p>
     * Set the timeout of the current conversation.
     * </p>
     * 
     * @param milliseconds the new timeout in milliseconds.
     */
    public void setTimeout(long milliseconds);

    /**
     * <p>
     * Determine if the conversation is marked transient or long-running.
     * </p>
     * 
     * @return <tt>true</tt> if the conversation is marked transient, or <tt>false</tt>if it is marked long-running.
     */
    public boolean isTransient();
}