#pragma once

#include <functional>
#include <memory>
#include <optional>
#include <string>
#include <unordered_map>
#include <vector>

namespace bpftrace {

class BPFtrace;

namespace ast {

class Node;
class SemanticAnalyser;
class Pass;

/**
   Result of a pass run
 */
class PassResult
{
public:
  static PassResult Error(const std::string &msg);
  static PassResult Success(Node *root = nullptr);

  // Ok returns whether the pass was successful or not
  bool Ok() const
  {
    return success_;
  };

  // Error returns the error generated by the pass, if any.
  const std::string &Error()
  {
    return error_;
  };
  Node *Root() const
  {
    return root_;
  };

private:
  bool success_;
  std::string error_;
  Node *root_ = nullptr;
};

/**
   Context/config for passes

   Note: Most state should end up in the BPFtrace class instead of here
*/

struct PassContext
{
public:
  PassContext(BPFtrace &b) : b(b){};
  BPFtrace &b;
};

using PassFPtr = std::function<PassResult(Node &, PassContext &)>;

/*
  Base pass
*/
class Pass
{
public:
  Pass() = delete;
  Pass(std::string name, PassFPtr fn) : fn_(fn), name(name){};

  virtual ~Pass() = default;

  PassResult Run(Node &root, PassContext &ctx)
  {
    return fn_(root, ctx);
  };

private:
  PassFPtr fn_;

public:
  std::string name;
};

class PassManager
{
public:
  PassManager() = default;

  void AddPass(Pass p);
  [[nodiscard]] std::unique_ptr<Node> Run(std::unique_ptr<Node> n,
                                          PassContext &ctx);

private:
  std::vector<Pass> passes_;
};

} // namespace ast
} // namespace bpftrace
