/*
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * Copyright 2012-2021 the original author or authors.
 */
package org.assertj.core.internal.paths;

import static org.assertj.core.api.Assertions.assertThat;
import static org.assertj.core.api.Assertions.assertThatExceptionOfType;
import static org.assertj.core.api.Assertions.assertThatNullPointerException;
import static org.assertj.core.api.ThrowableAssert.catchThrowable;
import static org.assertj.core.error.ShouldHaveParent.shouldHaveParent;
import static org.assertj.core.util.FailureMessages.actualIsNull;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import java.nio.file.Path;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

class Paths_assertHasParentRaw_Test extends MockPathsBaseTest {

  private Path expectedParent;

  @Override
  @BeforeEach
  public void init() {
    super.init();
    expectedParent = mock(Path.class);
  }

  @Test
  void should_fail_if_actual_is_null() {
    assertThatExceptionOfType(AssertionError.class).isThrownBy(() -> paths.assertHasParentRaw(info, null, expectedParent))
                                                   .withMessage(actualIsNull());
  }

  @Test
  void should_fail_if_provided_parent_is_null() {
    assertThatNullPointerException().isThrownBy(() -> paths.assertHasParentRaw(info, actual, null))
                                    .withMessage("expected parent path should not be null");
  }

  @Test
  void should_fail_if_actual_has_no_parent() {
    // This is the default, but...
    when(actual.getParent()).thenReturn(null);

    Throwable error = catchThrowable(() -> paths.assertHasParentRaw(info, actual, expectedParent));

    assertThat(error).isInstanceOf(AssertionError.class);
    verify(failures).failure(info, shouldHaveParent(actual, expectedParent));
  }

  @Test
  void should_fail_if_actual_parent_is_not_expected_parent() {
    final Path actualParent = mock(Path.class);
    when(actual.getParent()).thenReturn(actualParent);

    Throwable error = catchThrowable(() -> paths.assertHasParentRaw(info, actual, expectedParent));

    assertThat(error).isInstanceOf(AssertionError.class);
    verify(failures).failure(info, shouldHaveParent(actual, actualParent, expectedParent));
  }

  @Test
  void should_succeed_if_parent_is_expected_parent() {
    when(actual.getParent()).thenReturn(expectedParent);

    paths.assertHasParentRaw(info, actual, expectedParent);
  }
}
