// SPDX-License-Identifier: Apache-2.0

use std::collections::{hash_map::Entry, HashMap, HashSet};

use serde::{Deserialize, Serialize};

use crate::{
    ip::{is_ipv6_addr, sanitize_ip_network},
    ErrorKind, InterfaceType, MergedInterfaces, NmstateError,
};

#[derive(Debug, Clone, PartialEq, Eq, Default, Serialize, Deserialize)]
#[non_exhaustive]
#[serde(deny_unknown_fields)]
/// IP routing status
pub struct Routes {
    #[serde(skip_serializing_if = "Option::is_none")]
    /// Running effected routes containing route from universe or link scope,
    /// and only from these protocols:
    ///  * boot (often used by `iproute` command)
    ///  * static
    ///  * ra
    ///  * dhcp
    ///  * mrouted
    ///  * keepalived
    ///  * babel
    ///
    /// Ignored when applying.
    pub running: Option<Vec<RouteEntry>>,
    #[serde(skip_serializing_if = "Option::is_none")]
    /// Static routes containing route from universe or link scope,
    /// and only from these protocols:
    ///  * boot (often used by `iproute` command)
    ///  * static
    ///
    /// When applying, `None` means preserve current routes.
    /// This property is not overriding but adding specified routes to
    /// existing routes. To delete a route entry, please [RouteEntry.state] as
    /// [RouteState::Absent]. Any property of absent [RouteEntry] set to
    /// `None` means wildcard. For example, this [crate::NetworkState] could
    /// remove all routes next hop to interface eth1(showing in yaml):
    /// ```yaml
    /// routes:
    ///   config:
    ///   - next-hop-interface: eth1
    ///     state: absent
    /// ```
    ///
    /// To change a route entry, you need to delete old one and add new one(can
    /// be in single transaction).
    pub config: Option<Vec<RouteEntry>>,
}

impl Routes {
    pub fn new() -> Self {
        Self::default()
    }

    /// TODO: hide it, internal only
    pub fn is_empty(&self) -> bool {
        self.running.is_none() && self.config.is_none()
    }

    pub fn validate(&self) -> Result<(), NmstateError> {
        // All desire non-absent route should have next hop interface
        if let Some(config_routes) = self.config.as_ref() {
            for route in config_routes.iter().filter(|r| !r.is_absent()) {
                if route.next_hop_iface.is_none() {
                    return Err(NmstateError::new(
                        ErrorKind::NotImplementedError,
                        format!(
                            "Route with empty next hop interface \
                            is not supported: {route:?}"
                        ),
                    ));
                }
            }
        }
        Ok(())
    }
}

#[derive(Debug, Clone, Copy, PartialEq, Eq, Serialize, Deserialize)]
#[serde(rename_all = "kebab-case")]
#[non_exhaustive]
pub enum RouteState {
    /// Mark a route entry as absent to remove it.
    Absent,
}

impl Default for RouteState {
    fn default() -> Self {
        Self::Absent
    }
}

#[derive(Debug, Clone, Default, Serialize, Deserialize)]
#[serde(rename_all = "kebab-case")]
#[non_exhaustive]
#[serde(deny_unknown_fields)]
/// Route entry
pub struct RouteEntry {
    #[serde(skip_serializing_if = "Option::is_none")]
    /// Only used for delete route when applying.
    pub state: Option<RouteState>,
    #[serde(skip_serializing_if = "Option::is_none")]
    /// Route destination address or network
    pub destination: Option<String>,
    #[serde(
        skip_serializing_if = "Option::is_none",
        rename = "next-hop-interface"
    )]
    /// Route next hop interface name.
    /// Serialize and deserialize to/from `next-hop-interface`.
    pub next_hop_iface: Option<String>,
    #[serde(
        skip_serializing_if = "Option::is_none",
        rename = "next-hop-address"
    )]
    /// Route next hop IP address.
    /// Serialize and deserialize to/from `next-hop-address`.
    pub next_hop_addr: Option<String>,
    #[serde(
        skip_serializing_if = "Option::is_none",
        default,
        deserialize_with = "crate::deserializer::option_i64_or_string"
    )]
    /// Route metric. [RouteEntry::USE_DEFAULT_METRIC] for default
    /// setting of network backend.
    pub metric: Option<i64>,
    #[serde(
        skip_serializing_if = "Option::is_none",
        default,
        deserialize_with = "crate::deserializer::option_u32_or_string"
    )]
    /// Route table id. [RouteEntry::USE_DEFAULT_ROUTE_TABLE] for main
    /// route table 254.
    pub table_id: Option<u32>,
}

impl RouteEntry {
    pub const USE_DEFAULT_METRIC: i64 = -1;
    pub const USE_DEFAULT_ROUTE_TABLE: u32 = 0;

    pub fn new() -> Self {
        Self::default()
    }

    pub(crate) fn is_absent(&self) -> bool {
        matches!(self.state, Some(RouteState::Absent))
    }

    pub(crate) fn is_match(&self, other: &Self) -> bool {
        if self.destination.as_ref().is_some()
            && self.destination != other.destination
        {
            return false;
        }
        if self.next_hop_iface.as_ref().is_some()
            && self.next_hop_iface != other.next_hop_iface
        {
            return false;
        }

        if self.next_hop_addr.as_ref().is_some()
            && self.next_hop_addr != other.next_hop_addr
        {
            return false;
        }
        if self.table_id.is_some()
            && self.table_id != Some(RouteEntry::USE_DEFAULT_ROUTE_TABLE)
            && self.table_id != other.table_id
        {
            return false;
        }
        true
    }

    // Return tuple of (no_absent, is_ipv4, table_id, next_hop_iface,
    // destination, next_hop_addr)
    fn sort_key(&self) -> (bool, bool, u32, &str, &str, &str) {
        (
            !matches!(self.state, Some(RouteState::Absent)),
            !self
                .destination
                .as_ref()
                .map(|d| is_ipv6_addr(d.as_str()))
                .unwrap_or_default(),
            self.table_id.unwrap_or(RouteEntry::USE_DEFAULT_ROUTE_TABLE),
            self.next_hop_iface.as_deref().unwrap_or(""),
            self.destination.as_deref().unwrap_or(""),
            self.next_hop_addr.as_deref().unwrap_or(""),
        )
    }

    pub(crate) fn sanitize(&mut self) -> Result<(), NmstateError> {
        if let Some(dst) = self.destination.as_ref() {
            let new_dst = sanitize_ip_network(dst)?;
            if dst != &new_dst {
                log::warn!(
                    "Route destination {} sanitized to {}",
                    dst,
                    new_dst
                );
                self.destination = Some(new_dst);
            }
        }
        if let Some(via) = self.next_hop_addr.as_ref() {
            let new_via = format!("{}", via.parse::<std::net::IpAddr>()?);
            if via != &new_via {
                log::warn!(
                    "Route next-hop-address {} sanitized to {}",
                    via,
                    new_via
                );
                self.next_hop_addr = Some(new_via);
            }
        }
        Ok(())
    }

    pub(crate) fn is_ipv6(&self) -> bool {
        self.destination.as_ref().map(|d| is_ipv6_addr(d.as_str()))
            == Some(true)
    }
}

// For Vec::dedup()
impl PartialEq for RouteEntry {
    fn eq(&self, other: &Self) -> bool {
        self.sort_key() == other.sort_key()
    }
}

// For Vec::sort_unstable()
impl Ord for RouteEntry {
    fn cmp(&self, other: &Self) -> std::cmp::Ordering {
        self.sort_key().cmp(&other.sort_key())
    }
}

// For ord
impl Eq for RouteEntry {}

// For ord
impl PartialOrd for RouteEntry {
    fn partial_cmp(&self, other: &Self) -> Option<std::cmp::Ordering> {
        Some(self.cmp(other))
    }
}

impl std::fmt::Display for RouteEntry {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        let mut props = Vec::new();
        if self.is_absent() {
            props.push("state: absent".to_string());
        }
        if let Some(v) = self.destination.as_ref() {
            props.push(format!("destination: {v}"));
        }
        if let Some(v) = self.next_hop_iface.as_ref() {
            props.push(format!("next-hop-interface: {v}"));
        }
        if let Some(v) = self.next_hop_addr.as_ref() {
            props.push(format!("next-hop-address: {v}"));
        }
        if let Some(v) = self.metric.as_ref() {
            props.push(format!("metric: {v}"));
        }
        if let Some(v) = self.table_id.as_ref() {
            props.push(format!("table-id: {v}"));
        }

        write!(f, "{}", props.join(" "))
    }
}

#[derive(Clone, Debug, Default, PartialEq, Eq)]
pub(crate) struct MergedRoutes {
    pub(crate) indexed: HashMap<String, Vec<RouteEntry>>,
    pub(crate) route_changed_ifaces: Vec<String>,
    pub(crate) desired: Routes,
    pub(crate) current: Routes,
}

impl MergedRoutes {
    pub(crate) fn new(
        desired: Routes,
        current: Routes,
        merged_ifaces: &MergedInterfaces,
    ) -> Result<Self, NmstateError> {
        let mut desired_routes = Vec::new();
        if let Some(rts) = desired.config.as_ref() {
            for rt in rts {
                let mut rt = rt.clone();
                rt.sanitize()?;
                desired_routes.push(rt);
            }
        }

        let mut changed_ifaces: HashSet<&str> = HashSet::new();

        let ifaces_marked_as_absent: Vec<&str> = merged_ifaces
            .kernel_ifaces
            .values()
            .filter(|i| i.merged.is_absent())
            .map(|i| i.merged.name())
            .collect();

        let ifaces_with_ipv4_disabled: Vec<&str> = merged_ifaces
            .kernel_ifaces
            .values()
            .filter(|i| !i.merged.base_iface().is_ipv4_enabled())
            .map(|i| i.merged.name())
            .collect();

        let ifaces_with_ipv6_disabled: Vec<&str> = merged_ifaces
            .kernel_ifaces
            .values()
            .filter(|i| !i.merged.base_iface().is_ipv6_enabled())
            .map(|i| i.merged.name())
            .collect();

        // Interface has route added.
        for rt in desired_routes
            .as_slice()
            .iter()
            .filter(|rt| !rt.is_absent())
        {
            if let Some(via) = rt.next_hop_iface.as_ref() {
                if ifaces_marked_as_absent.contains(&via.as_str()) {
                    return Err(NmstateError::new(
                        ErrorKind::InvalidArgument,
                        format!(
                            "The next hop interface of desired Route '{rt}' \
                            has been marked as absent"
                        ),
                    ));
                }
                if rt.is_ipv6()
                    && ifaces_with_ipv6_disabled.contains(&via.as_str())
                {
                    return Err(NmstateError::new(
                        ErrorKind::InvalidArgument,
                        format!(
                            "The next hop interface of desired Route '{rt}' \
                            has been marked as IPv6 disabled"
                        ),
                    ));
                }
                if (!rt.is_ipv6())
                    && ifaces_with_ipv4_disabled.contains(&via.as_str())
                {
                    return Err(NmstateError::new(
                        ErrorKind::InvalidArgument,
                        format!(
                            "The next hop interface of desired Route '{rt}' \
                            has been marked as IPv4 disabled"
                        ),
                    ));
                }
                changed_ifaces.insert(via.as_str());
            }
        }

        // Interface has route deleted.
        for absent_rt in
            desired_routes.as_slice().iter().filter(|rt| rt.is_absent())
        {
            if let Some(cur_rts) = current.config.as_ref() {
                for rt in cur_rts {
                    if absent_rt.is_match(rt) {
                        if let Some(via) = rt.next_hop_iface.as_ref() {
                            changed_ifaces.insert(via.as_str());
                        }
                    }
                }
            }
        }

        let mut flattend_routes: Vec<RouteEntry> = Vec::new();

        if let Some(cur_rts) = current.config.as_ref() {
            for rt in cur_rts {
                if let Some(via) = rt.next_hop_iface.as_ref() {
                    if !ifaces_marked_as_absent.contains(&via.as_str())
                        && ((rt.is_ipv6()
                            && !ifaces_with_ipv6_disabled
                                .contains(&via.as_str()))
                            || (!rt.is_ipv6()
                                && !ifaces_with_ipv4_disabled
                                    .contains(&via.as_str())))
                    {
                        if desired_routes
                            .as_slice()
                            .iter()
                            .filter(|r| r.is_absent())
                            .any(|absent_rt| absent_rt.is_match(rt))
                        {
                            continue;
                        }

                        flattend_routes.push(rt.clone());
                    }
                }
            }
        }

        // Append desired routes
        for rt in desired_routes
            .as_slice()
            .iter()
            .filter(|rt| !rt.is_absent())
        {
            flattend_routes.push(rt.clone());
        }

        flattend_routes.sort_unstable();
        flattend_routes.dedup();

        let mut indexed: HashMap<String, Vec<RouteEntry>> = HashMap::new();

        for rt in flattend_routes {
            if let Some(via) = rt.next_hop_iface.as_ref() {
                let rts: &mut Vec<RouteEntry> =
                    match indexed.entry(via.to_string()) {
                        Entry::Occupied(o) => o.into_mut(),
                        Entry::Vacant(v) => v.insert(Vec::new()),
                    };
                rts.push(rt);
            }
        }

        let route_changed_ifaces: Vec<String> =
            changed_ifaces.iter().map(|i| i.to_string()).collect();

        Ok(Self {
            indexed,
            desired,
            current,
            route_changed_ifaces,
        })
    }

    pub(crate) fn remove_routes_to_ignored_ifaces(
        &mut self,
        ignored_ifaces: &[(String, InterfaceType)],
    ) {
        let ignored_ifaces: Vec<&str> = ignored_ifaces
            .iter()
            .filter_map(|(n, t)| {
                if !t.is_userspace() {
                    Some(n.as_str())
                } else {
                    None
                }
            })
            .collect();

        for iface in ignored_ifaces.as_slice() {
            self.indexed.remove(&iface.to_string());
        }
        self.route_changed_ifaces
            .retain(|n| !ignored_ifaces.contains(&n.as_str()));
    }

    pub(crate) fn is_changed(&self) -> bool {
        !self.route_changed_ifaces.is_empty()
    }
}
