/*
 * Copyright 2012-2017 Red Hat, Inc.
 *
 * This file is part of Thermostat.
 *
 * Thermostat is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2, or (at your
 * option) any later version.
 *
 * Thermostat is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Thermostat; see the file COPYING.  If not see
 * <http://www.gnu.org/licenses/>.
 *
 * Linking this code with other modules is making a combined work
 * based on this code.  Thus, the terms and conditions of the GNU
 * General Public License cover the whole combination.
 *
 * As a special exception, the copyright holders of this code give
 * you permission to link this code with independent modules to
 * produce an executable, regardless of the license terms of these
 * independent modules, and to copy and distribute the resulting
 * executable under terms of your choice, provided that you also
 * meet, for each linked independent module, the terms and conditions
 * of the license of that module.  An independent module is a module
 * which is not derived from or based on this code.  If you modify
 * this code, you may extend this exception to your version of the
 * library, but you are not obligated to do so.  If you do not wish
 * to do so, delete this exception statement from your version.
 */

package com.redhat.thermostat.vm.compiler.common;

import java.util.concurrent.TimeUnit;

import com.redhat.thermostat.client.core.experimental.Duration;
import com.redhat.thermostat.common.Size;

public class ParsedVmCompilerStat {

    public static enum CompileType {
        NO_COMPILE,
        NORMAL_COMPILE,
        OSR_COMPILE,
        NATIVE_COMPILE,;
    }

    private final VmCompilerStat original;

    public ParsedVmCompilerStat(VmCompilerStat stat) {
        this.original = stat;
    }

    public long getTotalCompiles() {
        return original.getTotalCompiles();
    }

    public long getTotalInvalidates() {
        return original.getTotalInvalidates();
    }

    public long getTotalBailouts() {
        return original.getTotalBailouts();
    }

    public long getCompilationTime() {
        return original.getCompilationTime();
    }

    public CompileType getLastType() {
        return getCompileType(original.getLastType());
    }

    public Size getLastSize() {
        return Size.bytes(original.getLastSize());
    }

    public String getLastMethod() {
        return toStandardMethodName(original.getLastMethod());
    }

    public CompileType getLastFailedType() {
        return getCompileType(original.getLastFailedType());
    }

    public String getLastFailedMethod() {
        return toStandardMethodName(original.getLastFailedMethod());
    }

    private CompileType getCompileType(long typeValue) {
        if (typeValue < 0 || typeValue >= CompileType.values().length) {
            throw new IllegalArgumentException("Unexpected compile-type code: " + typeValue);
        }
        return CompileType.values()[(int) typeValue];
    }

    private String toStandardMethodName(String encodedMethodName) {
        if (encodedMethodName == null || encodedMethodName.trim().isEmpty()) {
            return "<none>";
        }
        String[] parts = encodedMethodName.split(" ");
        String className = parts[0].replace('/', '.');
        String methodName = parts[1];
        return className + "." + methodName + "(...)";

    }

}
