/*
 * Copyright 2012-2017 Red Hat, Inc.
 *
 * This file is part of Thermostat.
 *
 * Thermostat is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2, or (at your
 * option) any later version.
 *
 * Thermostat is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Thermostat; see the file COPYING.  If not see
 * <http://www.gnu.org/licenses/>.
 *
 * Linking this code with other modules is making a combined work
 * based on this code.  Thus, the terms and conditions of the GNU
 * General Public License cover the whole combination.
 *
 * As a special exception, the copyright holders of this code give
 * you permission to link this code with independent modules to
 * produce an executable, regardless of the license terms of these
 * independent modules, and to copy and distribute the resulting
 * executable under terms of your choice, provided that you also
 * meet, for each linked independent module, the terms and conditions
 * of the license of that module.  An independent module is a module
 * which is not derived from or based on this code.  If you modify
 * this code, you may extend this exception to your version of the
 * library, but you are not obligated to do so.  If you do not wish
 * to do so, delete this exception statement from your version.
 */


package com.redhat.thermostat.storage.core;

/**
 * Provides authentication service to the command channel API.
 *
 * The protocol works as follows:
 *
 * <ol>
 * <li> The client asks the SecureStorage for an authentication token using {@link #generateToken(String)}.
 *      This should happen over an authenticated and secured connection, thus authenticating the client
 *      in the storage. The returned AuthToken will carry a client-token (that has been generated by the
 *      client) and an auth-token (generated by the storage). If authentication fails at this stage
 *      (e.g. because the client does not have the necessary privileges/roles), a StorageException is
 *      thrown.
 * </li>
 * <li> The AuthToken (both parts, client and auth token), are sent together with the command request to
 *      the command receiver (usually an agent).
 * </li>
 * <li> The agent takes the tokens, and calls the SecureStorage's {@link #verifyToken(AuthToken)} to
 *      verify the validity of the tokens. Again, this needs to happen through an authenticated and
 *      secured connection, thus authenticating the receiver. The storage verifies that it generated
 *      the same token for an authenticated client before, and replies with true if it succeeds, and
 *      false otherwise.
 */
public interface SecureStorage extends Storage {

    /**
     * Generates a token in the storage that can be used to authenticate cmd
     * channel requests.
     * 
     * @param actionName
     *            A unique name of the type of action to be performed.
     * 
     * @throws StorageException
     *             if authentication fails at this point
     */
    AuthToken generateToken(String actionName) throws StorageException;

    /**
     * Verifies the specified token and action in the storage.
     * 
     * @param actionName
     *            A unique name of the type of action to be performed. This
     *            action name is used during verification. This means if
     *            verification succeeds, the given action name can be trusted
     *            and may be used for authorization checks.
     * @param token The token to be verified.
     * 
     * @return <code>true</code> if authentication succeeded, <code>false</code>
     *         otherwise
     */
    boolean verifyToken(AuthToken token, String actionName);
}

