/*
 * Copyright 2012-2016 Red Hat, Inc.
 *
 * This file is part of Thermostat.
 *
 * Thermostat is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2, or (at your
 * option) any later version.
 *
 * Thermostat is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Thermostat; see the file COPYING.  If not see
 * <http://www.gnu.org/licenses/>.
 *
 * Linking this code with other modules is making a combined work
 * based on this code.  Thus, the terms and conditions of the GNU
 * General Public License cover the whole combination.
 *
 * As a special exception, the copyright holders of this code give
 * you permission to link this code with independent modules to
 * produce an executable, regardless of the license terms of these
 * independent modules, and to copy and distribute the resulting
 * executable under terms of your choice, provided that you also
 * meet, for each linked independent module, the terms and conditions
 * of the license of that module.  An independent module is a module
 * which is not derived from or based on this code.  If you modify
 * this code, you may extend this exception to your version of the
 * library, but you are not obligated to do so.  If you do not wish
 * to do so, delete this exception statement from your version.
 */

package com.redhat.thermostat.common;

import java.util.Collection;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.redhat.thermostat.common.utils.LoggingUtils;

public class ActionNotifier<T extends Enum<?>> {

    private static final Logger logger = LoggingUtils.getLogger(ActionNotifier.class);

    public ActionNotifier(Object source) {
        this.source = source;
        listeners = new CopyOnWriteArrayList<ActionListener<T>>();
    }

    private final Collection<ActionListener<T>> listeners;

    private final Object source;

    public void addActionListener(ActionListener<T> listener) {
        listeners.add(listener);
    }

    public void removeActionListener(ActionListener<T> listener) {
        listeners.remove(listener);
    }

    public void fireAction(T actionId) {
        fireAction(actionId, null);
    }

    public int listenersCount() {
        return listeners.size();
    }
    
    public void fireAction(T actionId, Object payload) {
        ActionEvent<T> action = new ActionEvent<>(source, actionId);
        action.setPayload(payload);
        for (ActionListener<T> listener : listeners) {
            try {
                listener.actionPerformed(action);
            } catch (Exception e) {
                // a listener throwing exception is BAD
                // unfortunately, all we can do is make sure other listeners continue working
                logger.log(Level.WARNING, "a listener threw an unexpected exception", e);
            }
        }
    }
}

