# Copyright 2009-2014 MongoDB, Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Tests for the son module."""

import copy
import pickle
import re
import sys
import unittest
sys.path[0:0] = [""]

from nose.plugins.skip import SkipTest

from bson.py3compat import b
from bson.son import SON


class TestSON(unittest.TestCase):
    def test_ordered_dict(self):
        a1 = SON()
        a1["hello"] = "world"
        a1["mike"] = "awesome"
        a1["hello_"] = "mike"
        self.assertEqual(a1.items(), [("hello", "world"),
                                     ("mike", "awesome"),
                                     ("hello_", "mike")])

        b2 = SON({"hello": "world"})
        self.assertEqual(b2["hello"], "world")
        self.assertRaises(KeyError, lambda: b2["goodbye"])

    def test_equality(self):
        a1 = SON({"hello": "world"})
        b2 = SON((('hello', 'world'), ('mike', 'awesome'), ('hello_', 'mike')))

        self.assertEqual(a1, SON({"hello": "world"}))
        self.assertEqual(b2, SON((('hello', 'world'),
                                  ('mike', 'awesome'),
                                  ('hello_', 'mike'))))
        self.assertEqual(b2, dict((('hello_', 'mike'),
                                   ('mike', 'awesome'),
                                   ('hello', 'world'))))

        self.assertNotEqual(a1, b2)
        self.assertNotEqual(b2, SON((('hello_', 'mike'),
                                     ('mike', 'awesome'),
                                     ('hello', 'world'))))

        # Explicitly test inequality
        self.assertFalse(a1 != SON({"hello": "world"}))
        self.assertFalse(b2 != SON((('hello', 'world'),
                                    ('mike', 'awesome'),
                                    ('hello_', 'mike'))))
        self.assertFalse(b2 != dict((('hello_', 'mike'),
                                     ('mike', 'awesome'),
                                     ('hello', 'world'))))

        # Embedded SON.
        d4 = SON([('blah', {'foo': SON()})])
        self.assertEqual(d4, {'blah': {'foo': {}}})
        self.assertEqual(d4, {'blah': {'foo': SON()}})
        self.assertNotEqual(d4, {'blah': {'foo': []}})

        # Original data unaffected.
        self.assertEqual(SON, d4['blah']['foo'].__class__)

    def test_to_dict(self):
        a1 = SON()
        b2 = SON([("blah", SON())])
        c3 = SON([("blah", [SON()])])
        d4 = SON([("blah", {"foo": SON()})])
        self.assertEqual({}, a1.to_dict())
        self.assertEqual({"blah": {}}, b2.to_dict())
        self.assertEqual({"blah": [{}]}, c3.to_dict())
        self.assertEqual({"blah": {"foo": {}}}, d4.to_dict())
        self.assertEqual(dict, a1.to_dict().__class__)
        self.assertEqual(dict, b2.to_dict()["blah"].__class__)
        self.assertEqual(dict, c3.to_dict()["blah"][0].__class__)
        self.assertEqual(dict, d4.to_dict()["blah"]["foo"].__class__)

        # Original data unaffected.
        self.assertEqual(SON, d4['blah']['foo'].__class__)

    def test_pickle(self):

        simple_son = SON([])
        complex_son = SON([('son', simple_son),
                           ('list', [simple_son, simple_son])])

        for protocol in xrange(pickle.HIGHEST_PROTOCOL + 1):
            pickled = pickle.loads(pickle.dumps(complex_son,
                                                protocol=protocol))
            self.assertEqual(pickled['son'], pickled['list'][0])
            self.assertEqual(pickled['son'], pickled['list'][1])

    def test_pickle_backwards_compatability(self):
        # For a full discussion see http://bugs.python.org/issue6137
        if sys.version.startswith('3.0'):
            raise SkipTest("Python 3.0.x can't unpickle "
                           "objects pickled in Python 2.x.")

        # This string was generated by pickling a SON object in pymongo
        # version 2.1.1
        pickled_with_2_1_1 = b(
            "ccopy_reg\n_reconstructor\np0\n(cbson.son\nSON\np1\n"
            "c__builtin__\ndict\np2\n(dp3\ntp4\nRp5\n(dp6\n"
            "S'_SON__keys'\np7\n(lp8\nsb."
        )
        son_2_1_1 = pickle.loads(pickled_with_2_1_1)
        self.assertEqual(son_2_1_1, SON([]))

    def test_copying(self):
        simple_son = SON([])
        complex_son = SON([('son', simple_son),
                           ('list', [simple_son, simple_son])])
        regex_son = SON([("x", re.compile("^hello.*"))])
        reflexive_son = SON([('son', simple_son)])
        reflexive_son["reflexive"] = reflexive_son

        simple_son1 = copy.copy(simple_son)
        self.assertEqual(simple_son, simple_son1)

        complex_son1 = copy.copy(complex_son)
        self.assertEqual(complex_son, complex_son1)

        regex_son1 = copy.copy(regex_son)
        self.assertEqual(regex_son, regex_son1)

        reflexive_son1 = copy.copy(reflexive_son)
        self.assertEqual(reflexive_son, reflexive_son1)

        # Test deepcopying
        simple_son1 = copy.deepcopy(simple_son)
        self.assertEqual(simple_son, simple_son1)

        regex_son1 = copy.deepcopy(regex_son)
        self.assertEqual(regex_son, regex_son1)

        complex_son1 = copy.deepcopy(complex_son)
        self.assertEqual(complex_son, complex_son1)

        reflexive_son1 = copy.deepcopy(reflexive_son)
        self.assertEqual(reflexive_son.keys(), reflexive_son1.keys())
        self.assertEqual(id(reflexive_son1), id(reflexive_son1["reflexive"]))


if __name__ == "__main__":
    unittest.main()
