/**
 * Copyright (C) 2016 MongoDB Inc.
 *
 * This program is free software: you can redistribute it and/or  modify
 * it under the terms of the GNU Affero General Public License, version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * As a special exception, the copyright holders give permission to link the
 * code of portions of this program with the OpenSSL library under certain
 * conditions as described in each individual source file and distribute
 * linked combinations including the program with the OpenSSL library. You
 * must comply with the GNU Affero General Public License in all respects
 * for all of the code used other than as permitted herein. If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so. If you do not
 * wish to do so, delete this exception statement from your version. If you
 * delete this exception statement from all source files in the program,
 * then also delete it in the license file.
 */

#pragma once

#include "mongo/base/disallow_copying.h"
#include "mongo/client/dbclientinterface.h"
#include "mongo/db/repl/optime.h"
#include "mongo/s/client/shard.h"
#include "mongo/stdx/mutex.h"

namespace mongo {

class ShardLocal : public Shard {
    MONGO_DISALLOW_COPYING(ShardLocal);

public:
    explicit ShardLocal(const ShardId& id);

    ~ShardLocal() = default;

    /**
     * These functions are implemented for the Shard interface's sake. They should not be called on
     * ShardLocal because doing so triggers invariants.
     */
    const ConnectionString getConnString() const override;
    const ConnectionString originalConnString() const override;
    std::shared_ptr<RemoteCommandTargeter> getTargeter() const override;
    void updateReplSetMonitor(const HostAndPort& remoteHost,
                              const Status& remoteCommandStatus) override;

    std::string toString() const override;

    bool isRetriableError(ErrorCodes::Error code, RetryPolicy options) final;

    Status createIndexOnConfig(OperationContext* txn,
                               const NamespaceString& ns,
                               const BSONObj& keys,
                               bool unique) override;

private:
    Shard::HostWithResponse _runCommand(OperationContext* txn,
                                        const ReadPreferenceSetting& unused,
                                        const std::string& dbName,
                                        Milliseconds maxTimeMSOverrideUnused,
                                        const BSONObj& cmdObj) final;

    StatusWith<Shard::QueryResponse> _exhaustiveFindOnConfig(
        OperationContext* txn,
        const ReadPreferenceSetting& readPref,
        const repl::ReadConcernLevel& readConcernLevel,
        const NamespaceString& nss,
        const BSONObj& query,
        const BSONObj& sort,
        boost::optional<long long> limit) final;

    /**
     * Checks if an OpTime was set on the current Client (ie if the current operation performed a
     * write) and if so updates _lastOpTime to the OpTime from the write that was just performed.
     * The 'previousOpTimeOnClient' parameter is the optime that *was* the optime on this client
     * before we ran this command through this ShardLocal.  By the time this method is called,
     * if the optime set on the Client is different than 'previousOpTimeOnClient' then that means
     * the command just run did a write and we should update _lastOpTime to capture the optime of
     * the write we just did.  If the current optime on the client is the same as
     * 'previousOpTimeOnClient' then the command we just ran didn't do a write, and we should leave
     * _lastOpTime alone.
     */
    void _updateLastOpTimeFromClient(OperationContext* txn,
                                     const repl::OpTime& previousOpTimeOnClient);

    repl::OpTime _getLastOpTime();

    // Guards _lastOpTime below.
    stdx::mutex _mutex;

    // Stores the optime that was generated by the last operation to perform a write that was run
    // through _runCommand.  Used in _exhaustiveFindOnConfig for waiting for that optime to be
    // committed so that readConcern majority reads will read the writes that were performed without
    // a w:majority write concern.
    repl::OpTime _lastOpTime{};
};

}  // namespace mongo
