package org.apache.lucene.index;

/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import org.apache.lucene.document.BinaryDocValuesField;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.NumericDocValuesField;
import org.apache.lucene.document.SortedDocValuesField;
import org.apache.lucene.document.SortedNumericDocValuesField;
import org.apache.lucene.document.SortedSetDocValuesField;
import org.apache.lucene.document.StringField;
import org.apache.lucene.store.Directory;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.LuceneTestCase;

/** Tests helper methods in DocValues */
public class TestDocValues extends LuceneTestCase {
  
  /** 
   * If the field doesn't exist, we return empty instances:
   * it can easily happen that a segment just doesn't have any docs with the field.
   */
  public void testEmptyIndex() throws Exception {
    Directory dir = newDirectory();
    IndexWriter iw = new IndexWriter(dir, newIndexWriterConfig(null));
    iw.addDocument(new Document());
    DirectoryReader dr = DirectoryReader.open(iw, true);
    LeafReader r = getOnlySegmentReader(dr);
    
    // ok
    assertNotNull(DocValues.getBinary(r, "bogus"));
    assertNotNull(DocValues.getNumeric(r, "bogus"));
    assertNotNull(DocValues.getSorted(r, "bogus"));
    assertNotNull(DocValues.getSortedSet(r, "bogus"));
    assertNotNull(DocValues.getSortedNumeric(r, "bogus"));
    assertNotNull(DocValues.getDocsWithField(r, "bogus"));
    
    dr.close();
    iw.close();
    dir.close();
  }
  
  /** 
   * field just doesnt have any docvalues at all: exception
   */
  public void testMisconfiguredField() throws Exception {
    Directory dir = newDirectory();
    IndexWriter iw = new IndexWriter(dir, newIndexWriterConfig(null));
    Document doc = new Document();
    doc.add(new StringField("foo", "bar", Field.Store.NO));
    iw.addDocument(doc);
    DirectoryReader dr = DirectoryReader.open(iw, true);
    LeafReader r = getOnlySegmentReader(dr);
   
    // errors
    try {
      DocValues.getBinary(r, "foo");
      fail();
    } catch (IllegalStateException expected) {}
    try {
      DocValues.getNumeric(r, "foo");
      fail();
    } catch (IllegalStateException expected) {}
    try {
      DocValues.getSorted(r, "foo");
      fail();
    } catch (IllegalStateException expected) {}
    try {
      DocValues.getSortedSet(r, "foo");
      fail();
    } catch (IllegalStateException expected) {}
    try {
      DocValues.getSortedNumeric(r, "foo");
      fail();
    } catch (IllegalStateException expected) {}
    try {
      DocValues.getDocsWithField(r, "foo");
      fail();
    } catch (IllegalStateException expected) {}
    
    dr.close();
    iw.close();
    dir.close();
  }
  
  /** 
   * field with numeric docvalues
   */
  public void testNumericField() throws Exception {
    Directory dir = newDirectory();
    IndexWriter iw = new IndexWriter(dir, newIndexWriterConfig(null));
    Document doc = new Document();
    doc.add(new NumericDocValuesField("foo", 3));
    iw.addDocument(doc);
    DirectoryReader dr = DirectoryReader.open(iw, true);
    LeafReader r = getOnlySegmentReader(dr);
    
    // ok
    assertNotNull(DocValues.getNumeric(r, "foo"));
    assertNotNull(DocValues.getSortedNumeric(r, "foo"));
    assertNotNull(DocValues.getDocsWithField(r, "foo"));
    
    // errors
    try {
      DocValues.getBinary(r, "foo");
      fail();
    } catch (IllegalStateException expected) {}
    try {
      DocValues.getSorted(r, "foo");
      fail();
    } catch (IllegalStateException expected) {}
    try {
      DocValues.getSortedSet(r, "foo");
      fail();
    } catch (IllegalStateException expected) {}
    
    dr.close();
    iw.close();
    dir.close();
  }
  
  /** 
   * field with binary docvalues
   */
  public void testBinaryField() throws Exception {
    Directory dir = newDirectory();
    IndexWriter iw = new IndexWriter(dir, newIndexWriterConfig(null));
    Document doc = new Document();
    doc.add(new BinaryDocValuesField("foo", new BytesRef("bar")));
    iw.addDocument(doc);
    DirectoryReader dr = DirectoryReader.open(iw, true);
    LeafReader r = getOnlySegmentReader(dr);
    
    // ok
    assertNotNull(DocValues.getBinary(r, "foo"));
    assertNotNull(DocValues.getDocsWithField(r, "foo"));
    
    // errors
    try {
      DocValues.getNumeric(r, "foo");
      fail();
    } catch (IllegalStateException expected) {}
    try {
      DocValues.getSorted(r, "foo");
      fail();
    } catch (IllegalStateException expected) {}
    try {
      DocValues.getSortedSet(r, "foo");
      fail();
    } catch (IllegalStateException expected) {}
    try {
      DocValues.getSortedNumeric(r, "foo");
      fail();
    } catch (IllegalStateException expected) {}
    
    dr.close();
    iw.close();
    dir.close();
  }
  
  /** 
   * field with sorted docvalues
   */
  public void testSortedField() throws Exception {
    Directory dir = newDirectory();
    IndexWriter iw = new IndexWriter(dir, newIndexWriterConfig(null));
    Document doc = new Document();
    doc.add(new SortedDocValuesField("foo", new BytesRef("bar")));
    iw.addDocument(doc);
    DirectoryReader dr = DirectoryReader.open(iw, true);
    LeafReader r = getOnlySegmentReader(dr);
    
    // ok
    assertNotNull(DocValues.getBinary(r, "foo"));
    assertNotNull(DocValues.getSorted(r, "foo"));
    assertNotNull(DocValues.getSortedSet(r, "foo"));
    assertNotNull(DocValues.getDocsWithField(r, "foo"));
    
    // errors
    try {
      DocValues.getNumeric(r, "foo");
      fail();
    } catch (IllegalStateException expected) {}
    try {
      DocValues.getSortedNumeric(r, "foo");
      fail();
    } catch (IllegalStateException expected) {}
    
    dr.close();
    iw.close();
    dir.close();
  }
  
  /** 
   * field with sortedset docvalues
   */
  public void testSortedSetField() throws Exception {
    Directory dir = newDirectory();
    IndexWriter iw = new IndexWriter(dir, newIndexWriterConfig(null));
    Document doc = new Document();
    doc.add(new SortedSetDocValuesField("foo", new BytesRef("bar")));
    iw.addDocument(doc);
    DirectoryReader dr = DirectoryReader.open(iw, true);
    LeafReader r = getOnlySegmentReader(dr);
    
    // ok
    assertNotNull(DocValues.getSortedSet(r, "foo"));
    assertNotNull(DocValues.getDocsWithField(r, "foo"));
    
    // errors
    try {
      DocValues.getBinary(r, "foo");
      fail();
    } catch (IllegalStateException expected) {}
    try {
      DocValues.getNumeric(r, "foo");
      fail();
    } catch (IllegalStateException expected) {}
    try {
      DocValues.getSorted(r, "foo");
      fail();
    } catch (IllegalStateException expected) {}
    try {
      DocValues.getSortedNumeric(r, "foo");
      fail();
    } catch (IllegalStateException expected) {}
    
    dr.close();
    iw.close();
    dir.close();
  }
  
  /** 
   * field with sortednumeric docvalues
   */
  public void testSortedNumericField() throws Exception {
    Directory dir = newDirectory();
    IndexWriter iw = new IndexWriter(dir, newIndexWriterConfig(null));
    Document doc = new Document();
    doc.add(new SortedNumericDocValuesField("foo", 3));
    iw.addDocument(doc);
    DirectoryReader dr = DirectoryReader.open(iw, true);
    LeafReader r = getOnlySegmentReader(dr);
    
    // ok
    assertNotNull(DocValues.getSortedNumeric(r, "foo"));
    assertNotNull(DocValues.getDocsWithField(r, "foo"));
    
    // errors
    try {
      DocValues.getBinary(r, "foo");
      fail();
    } catch (IllegalStateException expected) {}
    try {
      DocValues.getNumeric(r, "foo");
      fail();
    } catch (IllegalStateException expected) {}
    try {
      DocValues.getSorted(r, "foo");
      fail();
    } catch (IllegalStateException expected) {}
    try {
      DocValues.getSortedSet(r, "foo");
      fail();
    } catch (IllegalStateException expected) {}
    
    dr.close();
    iw.close();
    dir.close();
  }
}
