package org.apache.lucene.analysis.standard;

/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.BaseTokenStreamTestCase;
import org.junit.Ignore;

/**
 * This class was automatically generated by generateJavaUnicodeWordBreakTest.pl
 * from: http://www.unicode.org/Public/6.1.0/ucd/auxiliary/WordBreakTest.txt
 *
 * WordBreakTest.txt indicates the points in the provided character sequences
 * at which conforming implementations must and must not break words.  This
 * class tests for expected token extraction from each of the test sequences
 * in WordBreakTest.txt, where the expected tokens are those character
 * sequences bounded by word breaks and containing at least one character
 * from one of the following character sets:
 *
 *    \p{Script = Han}                (From http://www.unicode.org/Public/6.1.0/ucd/Scripts.txt)
 *    \p{Script = Hiragana}
 *    \p{LineBreak = Complex_Context} (From http://www.unicode.org/Public/6.1.0/ucd/LineBreak.txt)
 *    \p{WordBreak = ALetter}         (From http://www.unicode.org/Public/6.1.0/ucd/auxiliary/WordBreakProperty.txt)
 *    \p{WordBreak = Hebrew_Letter}
 *    \p{WordBreak = Katakana}
 *    \p{WordBreak = Numeric}         (Excludes full-width Arabic digits)
 *    [\uFF10-\uFF19]                (Full-width Arabic digits)
 */
@Ignore
public class WordBreakTestUnicode_6_1_0 extends BaseTokenStreamTestCase {

  public void test(Analyzer analyzer) throws Exception {
    // ÷ 0001 ÷ 0001 ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0001",
                     new String[] {  });

    // ÷ 0001 × 0308 ÷ 0001 ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u0001",
                     new String[] {  });

    // ÷ 0001 ÷ 000D ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\r",
                     new String[] {  });

    // ÷ 0001 × 0308 ÷ 000D ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\r",
                     new String[] {  });

    // ÷ 0001 ÷ 000A ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\n",
                     new String[] {  });

    // ÷ 0001 × 0308 ÷ 000A ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\n",
                     new String[] {  });

    // ÷ 0001 ÷ 000B ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u000B",
                     new String[] {  });

    // ÷ 0001 × 0308 ÷ 000B ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u000B",
                     new String[] {  });

    // ÷ 0001 ÷ 3031 ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u3031",
                     new String[] { "\u3031" });

    // ÷ 0001 × 0308 ÷ 3031 ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u3031",
                     new String[] { "\u3031" });

    // ÷ 0001 ÷ 0041 ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0041",
                     new String[] { "\u0041" });

    // ÷ 0001 × 0308 ÷ 0041 ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u0041",
                     new String[] { "\u0041" });

    // ÷ 0001 ÷ 003A ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u003A",
                     new String[] {  });

    // ÷ 0001 × 0308 ÷ 003A ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u003A",
                     new String[] {  });

    // ÷ 0001 ÷ 002C ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u002C",
                     new String[] {  });

    // ÷ 0001 × 0308 ÷ 002C ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u002C",
                     new String[] {  });

    // ÷ 0001 ÷ 0027 ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0027",
                     new String[] {  });

    // ÷ 0001 × 0308 ÷ 0027 ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u0027",
                     new String[] {  });

    // ÷ 0001 ÷ 0030 ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0030",
                     new String[] { "\u0030" });

    // ÷ 0001 × 0308 ÷ 0030 ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u0030",
                     new String[] { "\u0030" });

    // ÷ 0001 ÷ 005F ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u005F",
                     new String[] {  });

    // ÷ 0001 × 0308 ÷ 005F ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u005F",
                     new String[] {  });

    // ÷ 0001 × 00AD ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u00AD",
                     new String[] {  });

    // ÷ 0001 × 0308 × 00AD ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u00AD",
                     new String[] {  });

    // ÷ 0001 × 0300 ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0300",
                     new String[] {  });

    // ÷ 0001 × 0308 × 0300 ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u0300",
                     new String[] {  });

    // ÷ 0001 ÷ 0061 × 2060 ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0061\u2060",
                     new String[] { "\u0061\u2060" });

    // ÷ 0001 × 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u0061\u2060",
                     new String[] { "\u0061\u2060" });

    // ÷ 0001 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0061\u003A",
                     new String[] { "\u0061" });

    // ÷ 0001 × 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u0061\u003A",
                     new String[] { "\u0061" });

    // ÷ 0001 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0061\u0027",
                     new String[] { "\u0061" });

    // ÷ 0001 × 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u0061\u0027",
                     new String[] { "\u0061" });

    // ÷ 0001 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0061\u0027\u2060",
                     new String[] { "\u0061" });

    // ÷ 0001 × 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u0061\u0027\u2060",
                     new String[] { "\u0061" });

    // ÷ 0001 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0061\u002C",
                     new String[] { "\u0061" });

    // ÷ 0001 × 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u0061\u002C",
                     new String[] { "\u0061" });

    // ÷ 0001 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0031\u003A",
                     new String[] { "\u0031" });

    // ÷ 0001 × 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u0031\u003A",
                     new String[] { "\u0031" });

    // ÷ 0001 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0031\u0027",
                     new String[] { "\u0031" });

    // ÷ 0001 × 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u0031\u0027",
                     new String[] { "\u0031" });

    // ÷ 0001 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0031\u002C",
                     new String[] { "\u0031" });

    // ÷ 0001 × 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u0031\u002C",
                     new String[] { "\u0031" });

    // ÷ 0001 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0031\u002E\u2060",
                     new String[] { "\u0031" });

    // ÷ 0001 × 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u0031\u002E\u2060",
                     new String[] { "\u0031" });

    // ÷ 000D ÷ 0001 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0001",
                     new String[] {  });

    // ÷ 000D ÷ 0308 ÷ 0001 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u0001",
                     new String[] {  });

    // ÷ 000D ÷ 000D ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\r",
                     new String[] {  });

    // ÷ 000D ÷ 0308 ÷ 000D ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\r",
                     new String[] {  });

    // ÷ 000D × 000A ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) × [3.0] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\n",
                     new String[] {  });

    // ÷ 000D ÷ 0308 ÷ 000A ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\n",
                     new String[] {  });

    // ÷ 000D ÷ 000B ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u000B",
                     new String[] {  });

    // ÷ 000D ÷ 0308 ÷ 000B ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u000B",
                     new String[] {  });

    // ÷ 000D ÷ 3031 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u3031",
                     new String[] { "\u3031" });

    // ÷ 000D ÷ 0308 ÷ 3031 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u3031",
                     new String[] { "\u3031" });

    // ÷ 000D ÷ 0041 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0041",
                     new String[] { "\u0041" });

    // ÷ 000D ÷ 0308 ÷ 0041 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u0041",
                     new String[] { "\u0041" });

    // ÷ 000D ÷ 003A ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u003A",
                     new String[] {  });

    // ÷ 000D ÷ 0308 ÷ 003A ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u003A",
                     new String[] {  });

    // ÷ 000D ÷ 002C ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u002C",
                     new String[] {  });

    // ÷ 000D ÷ 0308 ÷ 002C ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u002C",
                     new String[] {  });

    // ÷ 000D ÷ 0027 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0027",
                     new String[] {  });

    // ÷ 000D ÷ 0308 ÷ 0027 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u0027",
                     new String[] {  });

    // ÷ 000D ÷ 0030 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0030",
                     new String[] { "\u0030" });

    // ÷ 000D ÷ 0308 ÷ 0030 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u0030",
                     new String[] { "\u0030" });

    // ÷ 000D ÷ 005F ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u005F",
                     new String[] {  });

    // ÷ 000D ÷ 0308 ÷ 005F ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u005F",
                     new String[] {  });

    // ÷ 000D ÷ 00AD ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u00AD",
                     new String[] {  });

    // ÷ 000D ÷ 0308 × 00AD ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u00AD",
                     new String[] {  });

    // ÷ 000D ÷ 0300 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0300",
                     new String[] {  });

    // ÷ 000D ÷ 0308 × 0300 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u0300",
                     new String[] {  });

    // ÷ 000D ÷ 0061 × 2060 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0061\u2060",
                     new String[] { "\u0061\u2060" });

    // ÷ 000D ÷ 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u0061\u2060",
                     new String[] { "\u0061\u2060" });

    // ÷ 000D ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0061\u003A",
                     new String[] { "\u0061" });

    // ÷ 000D ÷ 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u0061\u003A",
                     new String[] { "\u0061" });

    // ÷ 000D ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0061\u0027",
                     new String[] { "\u0061" });

    // ÷ 000D ÷ 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u0061\u0027",
                     new String[] { "\u0061" });

    // ÷ 000D ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0061\u0027\u2060",
                     new String[] { "\u0061" });

    // ÷ 000D ÷ 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u0061\u0027\u2060",
                     new String[] { "\u0061" });

    // ÷ 000D ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0061\u002C",
                     new String[] { "\u0061" });

    // ÷ 000D ÷ 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u0061\u002C",
                     new String[] { "\u0061" });

    // ÷ 000D ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0031\u003A",
                     new String[] { "\u0031" });

    // ÷ 000D ÷ 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u0031\u003A",
                     new String[] { "\u0031" });

    // ÷ 000D ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0031\u0027",
                     new String[] { "\u0031" });

    // ÷ 000D ÷ 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u0031\u0027",
                     new String[] { "\u0031" });

    // ÷ 000D ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0031\u002C",
                     new String[] { "\u0031" });

    // ÷ 000D ÷ 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u0031\u002C",
                     new String[] { "\u0031" });

    // ÷ 000D ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0031\u002E\u2060",
                     new String[] { "\u0031" });

    // ÷ 000D ÷ 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u0031\u002E\u2060",
                     new String[] { "\u0031" });

    // ÷ 000A ÷ 0001 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0001",
                     new String[] {  });

    // ÷ 000A ÷ 0308 ÷ 0001 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u0001",
                     new String[] {  });

    // ÷ 000A ÷ 000D ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\r",
                     new String[] {  });

    // ÷ 000A ÷ 0308 ÷ 000D ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\r",
                     new String[] {  });

    // ÷ 000A ÷ 000A ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\n",
                     new String[] {  });

    // ÷ 000A ÷ 0308 ÷ 000A ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\n",
                     new String[] {  });

    // ÷ 000A ÷ 000B ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u000B",
                     new String[] {  });

    // ÷ 000A ÷ 0308 ÷ 000B ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u000B",
                     new String[] {  });

    // ÷ 000A ÷ 3031 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u3031",
                     new String[] { "\u3031" });

    // ÷ 000A ÷ 0308 ÷ 3031 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u3031",
                     new String[] { "\u3031" });

    // ÷ 000A ÷ 0041 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0041",
                     new String[] { "\u0041" });

    // ÷ 000A ÷ 0308 ÷ 0041 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u0041",
                     new String[] { "\u0041" });

    // ÷ 000A ÷ 003A ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u003A",
                     new String[] {  });

    // ÷ 000A ÷ 0308 ÷ 003A ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u003A",
                     new String[] {  });

    // ÷ 000A ÷ 002C ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u002C",
                     new String[] {  });

    // ÷ 000A ÷ 0308 ÷ 002C ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u002C",
                     new String[] {  });

    // ÷ 000A ÷ 0027 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0027",
                     new String[] {  });

    // ÷ 000A ÷ 0308 ÷ 0027 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u0027",
                     new String[] {  });

    // ÷ 000A ÷ 0030 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0030",
                     new String[] { "\u0030" });

    // ÷ 000A ÷ 0308 ÷ 0030 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u0030",
                     new String[] { "\u0030" });

    // ÷ 000A ÷ 005F ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u005F",
                     new String[] {  });

    // ÷ 000A ÷ 0308 ÷ 005F ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u005F",
                     new String[] {  });

    // ÷ 000A ÷ 00AD ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u00AD",
                     new String[] {  });

    // ÷ 000A ÷ 0308 × 00AD ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u00AD",
                     new String[] {  });

    // ÷ 000A ÷ 0300 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0300",
                     new String[] {  });

    // ÷ 000A ÷ 0308 × 0300 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u0300",
                     new String[] {  });

    // ÷ 000A ÷ 0061 × 2060 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0061\u2060",
                     new String[] { "\u0061\u2060" });

    // ÷ 000A ÷ 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u0061\u2060",
                     new String[] { "\u0061\u2060" });

    // ÷ 000A ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0061\u003A",
                     new String[] { "\u0061" });

    // ÷ 000A ÷ 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u0061\u003A",
                     new String[] { "\u0061" });

    // ÷ 000A ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0061\u0027",
                     new String[] { "\u0061" });

    // ÷ 000A ÷ 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u0061\u0027",
                     new String[] { "\u0061" });

    // ÷ 000A ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0061\u0027\u2060",
                     new String[] { "\u0061" });

    // ÷ 000A ÷ 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u0061\u0027\u2060",
                     new String[] { "\u0061" });

    // ÷ 000A ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0061\u002C",
                     new String[] { "\u0061" });

    // ÷ 000A ÷ 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u0061\u002C",
                     new String[] { "\u0061" });

    // ÷ 000A ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0031\u003A",
                     new String[] { "\u0031" });

    // ÷ 000A ÷ 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u0031\u003A",
                     new String[] { "\u0031" });

    // ÷ 000A ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0031\u0027",
                     new String[] { "\u0031" });

    // ÷ 000A ÷ 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u0031\u0027",
                     new String[] { "\u0031" });

    // ÷ 000A ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0031\u002C",
                     new String[] { "\u0031" });

    // ÷ 000A ÷ 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u0031\u002C",
                     new String[] { "\u0031" });

    // ÷ 000A ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0031\u002E\u2060",
                     new String[] { "\u0031" });

    // ÷ 000A ÷ 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u0031\u002E\u2060",
                     new String[] { "\u0031" });

    // ÷ 000B ÷ 0001 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0001",
                     new String[] {  });

    // ÷ 000B ÷ 0308 ÷ 0001 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u0001",
                     new String[] {  });

    // ÷ 000B ÷ 000D ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\r",
                     new String[] {  });

    // ÷ 000B ÷ 0308 ÷ 000D ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\r",
                     new String[] {  });

    // ÷ 000B ÷ 000A ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\n",
                     new String[] {  });

    // ÷ 000B ÷ 0308 ÷ 000A ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\n",
                     new String[] {  });

    // ÷ 000B ÷ 000B ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u000B",
                     new String[] {  });

    // ÷ 000B ÷ 0308 ÷ 000B ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u000B",
                     new String[] {  });

    // ÷ 000B ÷ 3031 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u3031",
                     new String[] { "\u3031" });

    // ÷ 000B ÷ 0308 ÷ 3031 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u3031",
                     new String[] { "\u3031" });

    // ÷ 000B ÷ 0041 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0041",
                     new String[] { "\u0041" });

    // ÷ 000B ÷ 0308 ÷ 0041 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u0041",
                     new String[] { "\u0041" });

    // ÷ 000B ÷ 003A ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u003A",
                     new String[] {  });

    // ÷ 000B ÷ 0308 ÷ 003A ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u003A",
                     new String[] {  });

    // ÷ 000B ÷ 002C ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u002C",
                     new String[] {  });

    // ÷ 000B ÷ 0308 ÷ 002C ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u002C",
                     new String[] {  });

    // ÷ 000B ÷ 0027 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0027",
                     new String[] {  });

    // ÷ 000B ÷ 0308 ÷ 0027 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u0027",
                     new String[] {  });

    // ÷ 000B ÷ 0030 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0030",
                     new String[] { "\u0030" });

    // ÷ 000B ÷ 0308 ÷ 0030 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u0030",
                     new String[] { "\u0030" });

    // ÷ 000B ÷ 005F ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u005F",
                     new String[] {  });

    // ÷ 000B ÷ 0308 ÷ 005F ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u005F",
                     new String[] {  });

    // ÷ 000B ÷ 00AD ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u00AD",
                     new String[] {  });

    // ÷ 000B ÷ 0308 × 00AD ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u00AD",
                     new String[] {  });

    // ÷ 000B ÷ 0300 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0300",
                     new String[] {  });

    // ÷ 000B ÷ 0308 × 0300 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u0300",
                     new String[] {  });

    // ÷ 000B ÷ 0061 × 2060 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0061\u2060",
                     new String[] { "\u0061\u2060" });

    // ÷ 000B ÷ 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u0061\u2060",
                     new String[] { "\u0061\u2060" });

    // ÷ 000B ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0061\u003A",
                     new String[] { "\u0061" });

    // ÷ 000B ÷ 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u0061\u003A",
                     new String[] { "\u0061" });

    // ÷ 000B ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0061\u0027",
                     new String[] { "\u0061" });

    // ÷ 000B ÷ 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u0061\u0027",
                     new String[] { "\u0061" });

    // ÷ 000B ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0061\u0027\u2060",
                     new String[] { "\u0061" });

    // ÷ 000B ÷ 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u0061\u0027\u2060",
                     new String[] { "\u0061" });

    // ÷ 000B ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0061\u002C",
                     new String[] { "\u0061" });

    // ÷ 000B ÷ 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u0061\u002C",
                     new String[] { "\u0061" });

    // ÷ 000B ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0031\u003A",
                     new String[] { "\u0031" });

    // ÷ 000B ÷ 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u0031\u003A",
                     new String[] { "\u0031" });

    // ÷ 000B ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0031\u0027",
                     new String[] { "\u0031" });

    // ÷ 000B ÷ 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u0031\u0027",
                     new String[] { "\u0031" });

    // ÷ 000B ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0031\u002C",
                     new String[] { "\u0031" });

    // ÷ 000B ÷ 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u0031\u002C",
                     new String[] { "\u0031" });

    // ÷ 000B ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0031\u002E\u2060",
                     new String[] { "\u0031" });

    // ÷ 000B ÷ 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u0031\u002E\u2060",
                     new String[] { "\u0031" });

    // ÷ 3031 ÷ 0001 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0001",
                     new String[] { "\u3031" });

    // ÷ 3031 × 0308 ÷ 0001 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u0001",
                     new String[] { "\u3031\u0308" });

    // ÷ 3031 ÷ 000D ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\r",
                     new String[] { "\u3031" });

    // ÷ 3031 × 0308 ÷ 000D ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\r",
                     new String[] { "\u3031\u0308" });

    // ÷ 3031 ÷ 000A ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\n",
                     new String[] { "\u3031" });

    // ÷ 3031 × 0308 ÷ 000A ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\n",
                     new String[] { "\u3031\u0308" });

    // ÷ 3031 ÷ 000B ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u000B",
                     new String[] { "\u3031" });

    // ÷ 3031 × 0308 ÷ 000B ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u000B",
                     new String[] { "\u3031\u0308" });

    // ÷ 3031 × 3031 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [13.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u3031",
                     new String[] { "\u3031\u3031" });

    // ÷ 3031 × 0308 × 3031 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING DIAERESIS (Extend_FE) × [13.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u3031",
                     new String[] { "\u3031\u0308\u3031" });

    // ÷ 3031 ÷ 0041 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0041",
                     new String[] { "\u3031", "\u0041" });

    // ÷ 3031 × 0308 ÷ 0041 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u0041",
                     new String[] { "\u3031\u0308", "\u0041" });

    // ÷ 3031 ÷ 003A ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u003A",
                     new String[] { "\u3031" });

    // ÷ 3031 × 0308 ÷ 003A ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u003A",
                     new String[] { "\u3031\u0308" });

    // ÷ 3031 ÷ 002C ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u002C",
                     new String[] { "\u3031" });

    // ÷ 3031 × 0308 ÷ 002C ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u002C",
                     new String[] { "\u3031\u0308" });

    // ÷ 3031 ÷ 0027 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0027",
                     new String[] { "\u3031" });

    // ÷ 3031 × 0308 ÷ 0027 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u0027",
                     new String[] { "\u3031\u0308" });

    // ÷ 3031 ÷ 0030 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0030",
                     new String[] { "\u3031", "\u0030" });

    // ÷ 3031 × 0308 ÷ 0030 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u0030",
                     new String[] { "\u3031\u0308", "\u0030" });

    // ÷ 3031 × 005F ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [13.1] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u005F",
                     new String[] { "\u3031\u005F" });

    // ÷ 3031 × 0308 × 005F ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING DIAERESIS (Extend_FE) × [13.1] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u005F",
                     new String[] { "\u3031\u0308\u005F" });

    // ÷ 3031 × 00AD ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u00AD",
                     new String[] { "\u3031\u00AD" });

    // ÷ 3031 × 0308 × 00AD ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u00AD",
                     new String[] { "\u3031\u0308\u00AD" });

    // ÷ 3031 × 0300 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0300",
                     new String[] { "\u3031\u0300" });

    // ÷ 3031 × 0308 × 0300 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u0300",
                     new String[] { "\u3031\u0308\u0300" });

    // ÷ 3031 ÷ 0061 × 2060 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0061\u2060",
                     new String[] { "\u3031", "\u0061\u2060" });

    // ÷ 3031 × 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u0061\u2060",
                     new String[] { "\u3031\u0308", "\u0061\u2060" });

    // ÷ 3031 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0061\u003A",
                     new String[] { "\u3031", "\u0061" });

    // ÷ 3031 × 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u0061\u003A",
                     new String[] { "\u3031\u0308", "\u0061" });

    // ÷ 3031 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0061\u0027",
                     new String[] { "\u3031", "\u0061" });

    // ÷ 3031 × 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u0061\u0027",
                     new String[] { "\u3031\u0308", "\u0061" });

    // ÷ 3031 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0061\u0027\u2060",
                     new String[] { "\u3031", "\u0061" });

    // ÷ 3031 × 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u0061\u0027\u2060",
                     new String[] { "\u3031\u0308", "\u0061" });

    // ÷ 3031 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0061\u002C",
                     new String[] { "\u3031", "\u0061" });

    // ÷ 3031 × 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u0061\u002C",
                     new String[] { "\u3031\u0308", "\u0061" });

    // ÷ 3031 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0031\u003A",
                     new String[] { "\u3031", "\u0031" });

    // ÷ 3031 × 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u0031\u003A",
                     new String[] { "\u3031\u0308", "\u0031" });

    // ÷ 3031 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0031\u0027",
                     new String[] { "\u3031", "\u0031" });

    // ÷ 3031 × 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u0031\u0027",
                     new String[] { "\u3031\u0308", "\u0031" });

    // ÷ 3031 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0031\u002C",
                     new String[] { "\u3031", "\u0031" });

    // ÷ 3031 × 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u0031\u002C",
                     new String[] { "\u3031\u0308", "\u0031" });

    // ÷ 3031 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0031\u002E\u2060",
                     new String[] { "\u3031", "\u0031" });

    // ÷ 3031 × 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u0031\u002E\u2060",
                     new String[] { "\u3031\u0308", "\u0031" });

    // ÷ 0041 ÷ 0001 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0001",
                     new String[] { "\u0041" });

    // ÷ 0041 × 0308 ÷ 0001 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u0001",
                     new String[] { "\u0041\u0308" });

    // ÷ 0041 ÷ 000D ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\r",
                     new String[] { "\u0041" });

    // ÷ 0041 × 0308 ÷ 000D ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\r",
                     new String[] { "\u0041\u0308" });

    // ÷ 0041 ÷ 000A ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\n",
                     new String[] { "\u0041" });

    // ÷ 0041 × 0308 ÷ 000A ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\n",
                     new String[] { "\u0041\u0308" });

    // ÷ 0041 ÷ 000B ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u000B",
                     new String[] { "\u0041" });

    // ÷ 0041 × 0308 ÷ 000B ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u000B",
                     new String[] { "\u0041\u0308" });

    // ÷ 0041 ÷ 3031 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u3031",
                     new String[] { "\u0041", "\u3031" });

    // ÷ 0041 × 0308 ÷ 3031 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u3031",
                     new String[] { "\u0041\u0308", "\u3031" });

    // ÷ 0041 × 0041 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [5.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0041",
                     new String[] { "\u0041\u0041" });

    // ÷ 0041 × 0308 × 0041 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [5.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u0041",
                     new String[] { "\u0041\u0308\u0041" });

    // ÷ 0041 ÷ 003A ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u003A",
                     new String[] { "\u0041" });

    // ÷ 0041 × 0308 ÷ 003A ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u003A",
                     new String[] { "\u0041\u0308" });

    // ÷ 0041 ÷ 002C ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u002C",
                     new String[] { "\u0041" });

    // ÷ 0041 × 0308 ÷ 002C ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u002C",
                     new String[] { "\u0041\u0308" });

    // ÷ 0041 ÷ 0027 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0027",
                     new String[] { "\u0041" });

    // ÷ 0041 × 0308 ÷ 0027 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u0027",
                     new String[] { "\u0041\u0308" });

    // ÷ 0041 × 0030 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [9.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0030",
                     new String[] { "\u0041\u0030" });

    // ÷ 0041 × 0308 × 0030 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [9.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u0030",
                     new String[] { "\u0041\u0308\u0030" });

    // ÷ 0041 × 005F ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [13.1] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u005F",
                     new String[] { "\u0041\u005F" });

    // ÷ 0041 × 0308 × 005F ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [13.1] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u005F",
                     new String[] { "\u0041\u0308\u005F" });

    // ÷ 0041 × 00AD ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u00AD",
                     new String[] { "\u0041\u00AD" });

    // ÷ 0041 × 0308 × 00AD ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u00AD",
                     new String[] { "\u0041\u0308\u00AD" });

    // ÷ 0041 × 0300 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0300",
                     new String[] { "\u0041\u0300" });

    // ÷ 0041 × 0308 × 0300 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u0300",
                     new String[] { "\u0041\u0308\u0300" });

    // ÷ 0041 × 0061 × 2060 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [5.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0061\u2060",
                     new String[] { "\u0041\u0061\u2060" });

    // ÷ 0041 × 0308 × 0061 × 2060 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [5.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u0061\u2060",
                     new String[] { "\u0041\u0308\u0061\u2060" });

    // ÷ 0041 × 0061 ÷ 003A ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [5.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0061\u003A",
                     new String[] { "\u0041\u0061" });

    // ÷ 0041 × 0308 × 0061 ÷ 003A ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [5.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u0061\u003A",
                     new String[] { "\u0041\u0308\u0061" });

    // ÷ 0041 × 0061 ÷ 0027 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [5.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0061\u0027",
                     new String[] { "\u0041\u0061" });

    // ÷ 0041 × 0308 × 0061 ÷ 0027 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [5.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u0061\u0027",
                     new String[] { "\u0041\u0308\u0061" });

    // ÷ 0041 × 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [5.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0061\u0027\u2060",
                     new String[] { "\u0041\u0061" });

    // ÷ 0041 × 0308 × 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [5.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u0061\u0027\u2060",
                     new String[] { "\u0041\u0308\u0061" });

    // ÷ 0041 × 0061 ÷ 002C ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [5.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0061\u002C",
                     new String[] { "\u0041\u0061" });

    // ÷ 0041 × 0308 × 0061 ÷ 002C ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [5.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u0061\u002C",
                     new String[] { "\u0041\u0308\u0061" });

    // ÷ 0041 × 0031 ÷ 003A ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [9.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0031\u003A",
                     new String[] { "\u0041\u0031" });

    // ÷ 0041 × 0308 × 0031 ÷ 003A ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [9.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u0031\u003A",
                     new String[] { "\u0041\u0308\u0031" });

    // ÷ 0041 × 0031 ÷ 0027 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [9.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0031\u0027",
                     new String[] { "\u0041\u0031" });

    // ÷ 0041 × 0308 × 0031 ÷ 0027 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [9.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u0031\u0027",
                     new String[] { "\u0041\u0308\u0031" });

    // ÷ 0041 × 0031 ÷ 002C ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [9.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0031\u002C",
                     new String[] { "\u0041\u0031" });

    // ÷ 0041 × 0308 × 0031 ÷ 002C ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [9.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u0031\u002C",
                     new String[] { "\u0041\u0308\u0031" });

    // ÷ 0041 × 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [9.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0031\u002E\u2060",
                     new String[] { "\u0041\u0031" });

    // ÷ 0041 × 0308 × 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [9.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u0031\u002E\u2060",
                     new String[] { "\u0041\u0308\u0031" });

    // ÷ 003A ÷ 0001 ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0001",
                     new String[] {  });

    // ÷ 003A × 0308 ÷ 0001 ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u0001",
                     new String[] {  });

    // ÷ 003A ÷ 000D ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\r",
                     new String[] {  });

    // ÷ 003A × 0308 ÷ 000D ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\r",
                     new String[] {  });

    // ÷ 003A ÷ 000A ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\n",
                     new String[] {  });

    // ÷ 003A × 0308 ÷ 000A ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\n",
                     new String[] {  });

    // ÷ 003A ÷ 000B ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u000B",
                     new String[] {  });

    // ÷ 003A × 0308 ÷ 000B ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u000B",
                     new String[] {  });

    // ÷ 003A ÷ 3031 ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u3031",
                     new String[] { "\u3031" });

    // ÷ 003A × 0308 ÷ 3031 ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u3031",
                     new String[] { "\u3031" });

    // ÷ 003A ÷ 0041 ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0041",
                     new String[] { "\u0041" });

    // ÷ 003A × 0308 ÷ 0041 ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u0041",
                     new String[] { "\u0041" });

    // ÷ 003A ÷ 003A ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u003A",
                     new String[] {  });

    // ÷ 003A × 0308 ÷ 003A ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u003A",
                     new String[] {  });

    // ÷ 003A ÷ 002C ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u002C",
                     new String[] {  });

    // ÷ 003A × 0308 ÷ 002C ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u002C",
                     new String[] {  });

    // ÷ 003A ÷ 0027 ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0027",
                     new String[] {  });

    // ÷ 003A × 0308 ÷ 0027 ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u0027",
                     new String[] {  });

    // ÷ 003A ÷ 0030 ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0030",
                     new String[] { "\u0030" });

    // ÷ 003A × 0308 ÷ 0030 ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u0030",
                     new String[] { "\u0030" });

    // ÷ 003A ÷ 005F ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u005F",
                     new String[] {  });

    // ÷ 003A × 0308 ÷ 005F ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u005F",
                     new String[] {  });

    // ÷ 003A × 00AD ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u00AD",
                     new String[] {  });

    // ÷ 003A × 0308 × 00AD ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u00AD",
                     new String[] {  });

    // ÷ 003A × 0300 ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0300",
                     new String[] {  });

    // ÷ 003A × 0308 × 0300 ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u0300",
                     new String[] {  });

    // ÷ 003A ÷ 0061 × 2060 ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0061\u2060",
                     new String[] { "\u0061\u2060" });

    // ÷ 003A × 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u0061\u2060",
                     new String[] { "\u0061\u2060" });

    // ÷ 003A ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0061\u003A",
                     new String[] { "\u0061" });

    // ÷ 003A × 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u0061\u003A",
                     new String[] { "\u0061" });

    // ÷ 003A ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0061\u0027",
                     new String[] { "\u0061" });

    // ÷ 003A × 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u0061\u0027",
                     new String[] { "\u0061" });

    // ÷ 003A ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0061\u0027\u2060",
                     new String[] { "\u0061" });

    // ÷ 003A × 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u0061\u0027\u2060",
                     new String[] { "\u0061" });

    // ÷ 003A ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0061\u002C",
                     new String[] { "\u0061" });

    // ÷ 003A × 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u0061\u002C",
                     new String[] { "\u0061" });

    // ÷ 003A ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0031\u003A",
                     new String[] { "\u0031" });

    // ÷ 003A × 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u0031\u003A",
                     new String[] { "\u0031" });

    // ÷ 003A ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0031\u0027",
                     new String[] { "\u0031" });

    // ÷ 003A × 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u0031\u0027",
                     new String[] { "\u0031" });

    // ÷ 003A ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0031\u002C",
                     new String[] { "\u0031" });

    // ÷ 003A × 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u0031\u002C",
                     new String[] { "\u0031" });

    // ÷ 003A ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0031\u002E\u2060",
                     new String[] { "\u0031" });

    // ÷ 003A × 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u0031\u002E\u2060",
                     new String[] { "\u0031" });

    // ÷ 002C ÷ 0001 ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0001",
                     new String[] {  });

    // ÷ 002C × 0308 ÷ 0001 ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u0001",
                     new String[] {  });

    // ÷ 002C ÷ 000D ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\r",
                     new String[] {  });

    // ÷ 002C × 0308 ÷ 000D ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\r",
                     new String[] {  });

    // ÷ 002C ÷ 000A ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\n",
                     new String[] {  });

    // ÷ 002C × 0308 ÷ 000A ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\n",
                     new String[] {  });

    // ÷ 002C ÷ 000B ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u000B",
                     new String[] {  });

    // ÷ 002C × 0308 ÷ 000B ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u000B",
                     new String[] {  });

    // ÷ 002C ÷ 3031 ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u3031",
                     new String[] { "\u3031" });

    // ÷ 002C × 0308 ÷ 3031 ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u3031",
                     new String[] { "\u3031" });

    // ÷ 002C ÷ 0041 ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0041",
                     new String[] { "\u0041" });

    // ÷ 002C × 0308 ÷ 0041 ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u0041",
                     new String[] { "\u0041" });

    // ÷ 002C ÷ 003A ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u003A",
                     new String[] {  });

    // ÷ 002C × 0308 ÷ 003A ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u003A",
                     new String[] {  });

    // ÷ 002C ÷ 002C ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u002C",
                     new String[] {  });

    // ÷ 002C × 0308 ÷ 002C ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u002C",
                     new String[] {  });

    // ÷ 002C ÷ 0027 ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0027",
                     new String[] {  });

    // ÷ 002C × 0308 ÷ 0027 ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u0027",
                     new String[] {  });

    // ÷ 002C ÷ 0030 ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0030",
                     new String[] { "\u0030" });

    // ÷ 002C × 0308 ÷ 0030 ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u0030",
                     new String[] { "\u0030" });

    // ÷ 002C ÷ 005F ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u005F",
                     new String[] {  });

    // ÷ 002C × 0308 ÷ 005F ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u005F",
                     new String[] {  });

    // ÷ 002C × 00AD ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u00AD",
                     new String[] {  });

    // ÷ 002C × 0308 × 00AD ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u00AD",
                     new String[] {  });

    // ÷ 002C × 0300 ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0300",
                     new String[] {  });

    // ÷ 002C × 0308 × 0300 ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u0300",
                     new String[] {  });

    // ÷ 002C ÷ 0061 × 2060 ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0061\u2060",
                     new String[] { "\u0061\u2060" });

    // ÷ 002C × 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u0061\u2060",
                     new String[] { "\u0061\u2060" });

    // ÷ 002C ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0061\u003A",
                     new String[] { "\u0061" });

    // ÷ 002C × 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u0061\u003A",
                     new String[] { "\u0061" });

    // ÷ 002C ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0061\u0027",
                     new String[] { "\u0061" });

    // ÷ 002C × 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u0061\u0027",
                     new String[] { "\u0061" });

    // ÷ 002C ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0061\u0027\u2060",
                     new String[] { "\u0061" });

    // ÷ 002C × 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u0061\u0027\u2060",
                     new String[] { "\u0061" });

    // ÷ 002C ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0061\u002C",
                     new String[] { "\u0061" });

    // ÷ 002C × 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u0061\u002C",
                     new String[] { "\u0061" });

    // ÷ 002C ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0031\u003A",
                     new String[] { "\u0031" });

    // ÷ 002C × 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u0031\u003A",
                     new String[] { "\u0031" });

    // ÷ 002C ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0031\u0027",
                     new String[] { "\u0031" });

    // ÷ 002C × 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u0031\u0027",
                     new String[] { "\u0031" });

    // ÷ 002C ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0031\u002C",
                     new String[] { "\u0031" });

    // ÷ 002C × 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u0031\u002C",
                     new String[] { "\u0031" });

    // ÷ 002C ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0031\u002E\u2060",
                     new String[] { "\u0031" });

    // ÷ 002C × 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u0031\u002E\u2060",
                     new String[] { "\u0031" });

    // ÷ 0027 ÷ 0001 ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0001",
                     new String[] {  });

    // ÷ 0027 × 0308 ÷ 0001 ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u0001",
                     new String[] {  });

    // ÷ 0027 ÷ 000D ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\r",
                     new String[] {  });

    // ÷ 0027 × 0308 ÷ 000D ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\r",
                     new String[] {  });

    // ÷ 0027 ÷ 000A ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\n",
                     new String[] {  });

    // ÷ 0027 × 0308 ÷ 000A ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\n",
                     new String[] {  });

    // ÷ 0027 ÷ 000B ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u000B",
                     new String[] {  });

    // ÷ 0027 × 0308 ÷ 000B ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u000B",
                     new String[] {  });

    // ÷ 0027 ÷ 3031 ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u3031",
                     new String[] { "\u3031" });

    // ÷ 0027 × 0308 ÷ 3031 ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u3031",
                     new String[] { "\u3031" });

    // ÷ 0027 ÷ 0041 ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0041",
                     new String[] { "\u0041" });

    // ÷ 0027 × 0308 ÷ 0041 ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u0041",
                     new String[] { "\u0041" });

    // ÷ 0027 ÷ 003A ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u003A",
                     new String[] {  });

    // ÷ 0027 × 0308 ÷ 003A ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u003A",
                     new String[] {  });

    // ÷ 0027 ÷ 002C ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u002C",
                     new String[] {  });

    // ÷ 0027 × 0308 ÷ 002C ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u002C",
                     new String[] {  });

    // ÷ 0027 ÷ 0027 ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0027",
                     new String[] {  });

    // ÷ 0027 × 0308 ÷ 0027 ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u0027",
                     new String[] {  });

    // ÷ 0027 ÷ 0030 ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0030",
                     new String[] { "\u0030" });

    // ÷ 0027 × 0308 ÷ 0030 ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u0030",
                     new String[] { "\u0030" });

    // ÷ 0027 ÷ 005F ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u005F",
                     new String[] {  });

    // ÷ 0027 × 0308 ÷ 005F ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u005F",
                     new String[] {  });

    // ÷ 0027 × 00AD ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u00AD",
                     new String[] {  });

    // ÷ 0027 × 0308 × 00AD ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u00AD",
                     new String[] {  });

    // ÷ 0027 × 0300 ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0300",
                     new String[] {  });

    // ÷ 0027 × 0308 × 0300 ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u0300",
                     new String[] {  });

    // ÷ 0027 ÷ 0061 × 2060 ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0061\u2060",
                     new String[] { "\u0061\u2060" });

    // ÷ 0027 × 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u0061\u2060",
                     new String[] { "\u0061\u2060" });

    // ÷ 0027 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0061\u003A",
                     new String[] { "\u0061" });

    // ÷ 0027 × 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u0061\u003A",
                     new String[] { "\u0061" });

    // ÷ 0027 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0061\u0027",
                     new String[] { "\u0061" });

    // ÷ 0027 × 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u0061\u0027",
                     new String[] { "\u0061" });

    // ÷ 0027 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0061\u0027\u2060",
                     new String[] { "\u0061" });

    // ÷ 0027 × 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u0061\u0027\u2060",
                     new String[] { "\u0061" });

    // ÷ 0027 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0061\u002C",
                     new String[] { "\u0061" });

    // ÷ 0027 × 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u0061\u002C",
                     new String[] { "\u0061" });

    // ÷ 0027 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0031\u003A",
                     new String[] { "\u0031" });

    // ÷ 0027 × 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u0031\u003A",
                     new String[] { "\u0031" });

    // ÷ 0027 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0031\u0027",
                     new String[] { "\u0031" });

    // ÷ 0027 × 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u0031\u0027",
                     new String[] { "\u0031" });

    // ÷ 0027 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0031\u002C",
                     new String[] { "\u0031" });

    // ÷ 0027 × 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u0031\u002C",
                     new String[] { "\u0031" });

    // ÷ 0027 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0031\u002E\u2060",
                     new String[] { "\u0031" });

    // ÷ 0027 × 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u0031\u002E\u2060",
                     new String[] { "\u0031" });

    // ÷ 0030 ÷ 0001 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0001",
                     new String[] { "\u0030" });

    // ÷ 0030 × 0308 ÷ 0001 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u0001",
                     new String[] { "\u0030\u0308" });

    // ÷ 0030 ÷ 000D ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\r",
                     new String[] { "\u0030" });

    // ÷ 0030 × 0308 ÷ 000D ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\r",
                     new String[] { "\u0030\u0308" });

    // ÷ 0030 ÷ 000A ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\n",
                     new String[] { "\u0030" });

    // ÷ 0030 × 0308 ÷ 000A ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\n",
                     new String[] { "\u0030\u0308" });

    // ÷ 0030 ÷ 000B ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u000B",
                     new String[] { "\u0030" });

    // ÷ 0030 × 0308 ÷ 000B ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u000B",
                     new String[] { "\u0030\u0308" });

    // ÷ 0030 ÷ 3031 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u3031",
                     new String[] { "\u0030", "\u3031" });

    // ÷ 0030 × 0308 ÷ 3031 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u3031",
                     new String[] { "\u0030\u0308", "\u3031" });

    // ÷ 0030 × 0041 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [10.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0041",
                     new String[] { "\u0030\u0041" });

    // ÷ 0030 × 0308 × 0041 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS (Extend_FE) × [10.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u0041",
                     new String[] { "\u0030\u0308\u0041" });

    // ÷ 0030 ÷ 003A ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u003A",
                     new String[] { "\u0030" });

    // ÷ 0030 × 0308 ÷ 003A ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u003A",
                     new String[] { "\u0030\u0308" });

    // ÷ 0030 ÷ 002C ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u002C",
                     new String[] { "\u0030" });

    // ÷ 0030 × 0308 ÷ 002C ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u002C",
                     new String[] { "\u0030\u0308" });

    // ÷ 0030 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0027",
                     new String[] { "\u0030" });

    // ÷ 0030 × 0308 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u0027",
                     new String[] { "\u0030\u0308" });

    // ÷ 0030 × 0030 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [8.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0030",
                     new String[] { "\u0030\u0030" });

    // ÷ 0030 × 0308 × 0030 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS (Extend_FE) × [8.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u0030",
                     new String[] { "\u0030\u0308\u0030" });

    // ÷ 0030 × 005F ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [13.1] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u005F",
                     new String[] { "\u0030\u005F" });

    // ÷ 0030 × 0308 × 005F ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS (Extend_FE) × [13.1] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u005F",
                     new String[] { "\u0030\u0308\u005F" });

    // ÷ 0030 × 00AD ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u00AD",
                     new String[] { "\u0030\u00AD" });

    // ÷ 0030 × 0308 × 00AD ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u00AD",
                     new String[] { "\u0030\u0308\u00AD" });

    // ÷ 0030 × 0300 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0300",
                     new String[] { "\u0030\u0300" });

    // ÷ 0030 × 0308 × 0300 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u0300",
                     new String[] { "\u0030\u0308\u0300" });

    // ÷ 0030 × 0061 × 2060 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [10.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0061\u2060",
                     new String[] { "\u0030\u0061\u2060" });

    // ÷ 0030 × 0308 × 0061 × 2060 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS (Extend_FE) × [10.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u0061\u2060",
                     new String[] { "\u0030\u0308\u0061\u2060" });

    // ÷ 0030 × 0061 ÷ 003A ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [10.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0061\u003A",
                     new String[] { "\u0030\u0061" });

    // ÷ 0030 × 0308 × 0061 ÷ 003A ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS (Extend_FE) × [10.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u0061\u003A",
                     new String[] { "\u0030\u0308\u0061" });

    // ÷ 0030 × 0061 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [10.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0061\u0027",
                     new String[] { "\u0030\u0061" });

    // ÷ 0030 × 0308 × 0061 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS (Extend_FE) × [10.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u0061\u0027",
                     new String[] { "\u0030\u0308\u0061" });

    // ÷ 0030 × 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [10.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0061\u0027\u2060",
                     new String[] { "\u0030\u0061" });

    // ÷ 0030 × 0308 × 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS (Extend_FE) × [10.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u0061\u0027\u2060",
                     new String[] { "\u0030\u0308\u0061" });

    // ÷ 0030 × 0061 ÷ 002C ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [10.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0061\u002C",
                     new String[] { "\u0030\u0061" });

    // ÷ 0030 × 0308 × 0061 ÷ 002C ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS (Extend_FE) × [10.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u0061\u002C",
                     new String[] { "\u0030\u0308\u0061" });

    // ÷ 0030 × 0031 ÷ 003A ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [8.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0031\u003A",
                     new String[] { "\u0030\u0031" });

    // ÷ 0030 × 0308 × 0031 ÷ 003A ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS (Extend_FE) × [8.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u0031\u003A",
                     new String[] { "\u0030\u0308\u0031" });

    // ÷ 0030 × 0031 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [8.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0031\u0027",
                     new String[] { "\u0030\u0031" });

    // ÷ 0030 × 0308 × 0031 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS (Extend_FE) × [8.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u0031\u0027",
                     new String[] { "\u0030\u0308\u0031" });

    // ÷ 0030 × 0031 ÷ 002C ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [8.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0031\u002C",
                     new String[] { "\u0030\u0031" });

    // ÷ 0030 × 0308 × 0031 ÷ 002C ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS (Extend_FE) × [8.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u0031\u002C",
                     new String[] { "\u0030\u0308\u0031" });

    // ÷ 0030 × 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [8.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0031\u002E\u2060",
                     new String[] { "\u0030\u0031" });

    // ÷ 0030 × 0308 × 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS (Extend_FE) × [8.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u0031\u002E\u2060",
                     new String[] { "\u0030\u0308\u0031" });

    // ÷ 005F ÷ 0001 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0001",
                     new String[] {  });

    // ÷ 005F × 0308 ÷ 0001 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u0001",
                     new String[] {  });

    // ÷ 005F ÷ 000D ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\r",
                     new String[] {  });

    // ÷ 005F × 0308 ÷ 000D ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\r",
                     new String[] {  });

    // ÷ 005F ÷ 000A ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\n",
                     new String[] {  });

    // ÷ 005F × 0308 ÷ 000A ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\n",
                     new String[] {  });

    // ÷ 005F ÷ 000B ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u000B",
                     new String[] {  });

    // ÷ 005F × 0308 ÷ 000B ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u000B",
                     new String[] {  });

    // ÷ 005F × 3031 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [13.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u3031",
                     new String[] { "\u005F\u3031" });

    // ÷ 005F × 0308 × 3031 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) × [13.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u3031",
                     new String[] { "\u005F\u0308\u3031" });

    // ÷ 005F × 0041 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [13.2] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0041",
                     new String[] { "\u005F\u0041" });

    // ÷ 005F × 0308 × 0041 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) × [13.2] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u0041",
                     new String[] { "\u005F\u0308\u0041" });

    // ÷ 005F ÷ 003A ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u003A",
                     new String[] {  });

    // ÷ 005F × 0308 ÷ 003A ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u003A",
                     new String[] {  });

    // ÷ 005F ÷ 002C ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u002C",
                     new String[] {  });

    // ÷ 005F × 0308 ÷ 002C ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u002C",
                     new String[] {  });

    // ÷ 005F ÷ 0027 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0027",
                     new String[] {  });

    // ÷ 005F × 0308 ÷ 0027 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u0027",
                     new String[] {  });

    // ÷ 005F × 0030 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [13.2] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0030",
                     new String[] { "\u005F\u0030" });

    // ÷ 005F × 0308 × 0030 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) × [13.2] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u0030",
                     new String[] { "\u005F\u0308\u0030" });

    // ÷ 005F × 005F ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [13.1] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u005F",
                     new String[] {  });

    // ÷ 005F × 0308 × 005F ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) × [13.1] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u005F",
                     new String[] {  });

    // ÷ 005F × 00AD ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u00AD",
                     new String[] {  });

    // ÷ 005F × 0308 × 00AD ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u00AD",
                     new String[] {  });

    // ÷ 005F × 0300 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0300",
                     new String[] {  });

    // ÷ 005F × 0308 × 0300 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u0300",
                     new String[] {  });

    // ÷ 005F × 0061 × 2060 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0061\u2060",
                     new String[] { "\u005F\u0061\u2060" });

    // ÷ 005F × 0308 × 0061 × 2060 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) × [13.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u0061\u2060",
                     new String[] { "\u005F\u0308\u0061\u2060" });

    // ÷ 005F × 0061 ÷ 003A ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0061\u003A",
                     new String[] { "\u005F\u0061" });

    // ÷ 005F × 0308 × 0061 ÷ 003A ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u0061\u003A",
                     new String[] { "\u005F\u0308\u0061" });

    // ÷ 005F × 0061 ÷ 0027 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0061\u0027",
                     new String[] { "\u005F\u0061" });

    // ÷ 005F × 0308 × 0061 ÷ 0027 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u0061\u0027",
                     new String[] { "\u005F\u0308\u0061" });

    // ÷ 005F × 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0061\u0027\u2060",
                     new String[] { "\u005F\u0061" });

    // ÷ 005F × 0308 × 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u0061\u0027\u2060",
                     new String[] { "\u005F\u0308\u0061" });

    // ÷ 005F × 0061 ÷ 002C ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0061\u002C",
                     new String[] { "\u005F\u0061" });

    // ÷ 005F × 0308 × 0061 ÷ 002C ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u0061\u002C",
                     new String[] { "\u005F\u0308\u0061" });

    // ÷ 005F × 0031 ÷ 003A ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0031\u003A",
                     new String[] { "\u005F\u0031" });

    // ÷ 005F × 0308 × 0031 ÷ 003A ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u0031\u003A",
                     new String[] { "\u005F\u0308\u0031" });

    // ÷ 005F × 0031 ÷ 0027 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0031\u0027",
                     new String[] { "\u005F\u0031" });

    // ÷ 005F × 0308 × 0031 ÷ 0027 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u0031\u0027",
                     new String[] { "\u005F\u0308\u0031" });

    // ÷ 005F × 0031 ÷ 002C ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0031\u002C",
                     new String[] { "\u005F\u0031" });

    // ÷ 005F × 0308 × 0031 ÷ 002C ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u0031\u002C",
                     new String[] { "\u005F\u0308\u0031" });

    // ÷ 005F × 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0031\u002E\u2060",
                     new String[] { "\u005F\u0031" });

    // ÷ 005F × 0308 × 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u0031\u002E\u2060",
                     new String[] { "\u005F\u0308\u0031" });

    // ÷ 00AD ÷ 0001 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0001",
                     new String[] {  });

    // ÷ 00AD × 0308 ÷ 0001 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u0001",
                     new String[] {  });

    // ÷ 00AD ÷ 000D ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\r",
                     new String[] {  });

    // ÷ 00AD × 0308 ÷ 000D ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\r",
                     new String[] {  });

    // ÷ 00AD ÷ 000A ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\n",
                     new String[] {  });

    // ÷ 00AD × 0308 ÷ 000A ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\n",
                     new String[] {  });

    // ÷ 00AD ÷ 000B ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u000B",
                     new String[] {  });

    // ÷ 00AD × 0308 ÷ 000B ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u000B",
                     new String[] {  });

    // ÷ 00AD ÷ 3031 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u3031",
                     new String[] { "\u3031" });

    // ÷ 00AD × 0308 ÷ 3031 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u3031",
                     new String[] { "\u3031" });

    // ÷ 00AD ÷ 0041 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0041",
                     new String[] { "\u0041" });

    // ÷ 00AD × 0308 ÷ 0041 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u0041",
                     new String[] { "\u0041" });

    // ÷ 00AD ÷ 003A ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u003A",
                     new String[] {  });

    // ÷ 00AD × 0308 ÷ 003A ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u003A",
                     new String[] {  });

    // ÷ 00AD ÷ 002C ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u002C",
                     new String[] {  });

    // ÷ 00AD × 0308 ÷ 002C ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u002C",
                     new String[] {  });

    // ÷ 00AD ÷ 0027 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0027",
                     new String[] {  });

    // ÷ 00AD × 0308 ÷ 0027 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u0027",
                     new String[] {  });

    // ÷ 00AD ÷ 0030 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0030",
                     new String[] { "\u0030" });

    // ÷ 00AD × 0308 ÷ 0030 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u0030",
                     new String[] { "\u0030" });

    // ÷ 00AD ÷ 005F ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u005F",
                     new String[] {  });

    // ÷ 00AD × 0308 ÷ 005F ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u005F",
                     new String[] {  });

    // ÷ 00AD × 00AD ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u00AD",
                     new String[] {  });

    // ÷ 00AD × 0308 × 00AD ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u00AD",
                     new String[] {  });

    // ÷ 00AD × 0300 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0300",
                     new String[] {  });

    // ÷ 00AD × 0308 × 0300 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u0300",
                     new String[] {  });

    // ÷ 00AD ÷ 0061 × 2060 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0061\u2060",
                     new String[] { "\u0061\u2060" });

    // ÷ 00AD × 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u0061\u2060",
                     new String[] { "\u0061\u2060" });

    // ÷ 00AD ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0061\u003A",
                     new String[] { "\u0061" });

    // ÷ 00AD × 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u0061\u003A",
                     new String[] { "\u0061" });

    // ÷ 00AD ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0061\u0027",
                     new String[] { "\u0061" });

    // ÷ 00AD × 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u0061\u0027",
                     new String[] { "\u0061" });

    // ÷ 00AD ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0061\u0027\u2060",
                     new String[] { "\u0061" });

    // ÷ 00AD × 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u0061\u0027\u2060",
                     new String[] { "\u0061" });

    // ÷ 00AD ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0061\u002C",
                     new String[] { "\u0061" });

    // ÷ 00AD × 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u0061\u002C",
                     new String[] { "\u0061" });

    // ÷ 00AD ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0031\u003A",
                     new String[] { "\u0031" });

    // ÷ 00AD × 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u0031\u003A",
                     new String[] { "\u0031" });

    // ÷ 00AD ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0031\u0027",
                     new String[] { "\u0031" });

    // ÷ 00AD × 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u0031\u0027",
                     new String[] { "\u0031" });

    // ÷ 00AD ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0031\u002C",
                     new String[] { "\u0031" });

    // ÷ 00AD × 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u0031\u002C",
                     new String[] { "\u0031" });

    // ÷ 00AD ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0031\u002E\u2060",
                     new String[] { "\u0031" });

    // ÷ 00AD × 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u0031\u002E\u2060",
                     new String[] { "\u0031" });

    // ÷ 0300 ÷ 0001 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0001",
                     new String[] {  });

    // ÷ 0300 × 0308 ÷ 0001 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u0001",
                     new String[] {  });

    // ÷ 0300 ÷ 000D ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\r",
                     new String[] {  });

    // ÷ 0300 × 0308 ÷ 000D ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\r",
                     new String[] {  });

    // ÷ 0300 ÷ 000A ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\n",
                     new String[] {  });

    // ÷ 0300 × 0308 ÷ 000A ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\n",
                     new String[] {  });

    // ÷ 0300 ÷ 000B ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u000B",
                     new String[] {  });

    // ÷ 0300 × 0308 ÷ 000B ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u000B",
                     new String[] {  });

    // ÷ 0300 ÷ 3031 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u3031",
                     new String[] { "\u3031" });

    // ÷ 0300 × 0308 ÷ 3031 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u3031",
                     new String[] { "\u3031" });

    // ÷ 0300 ÷ 0041 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0041",
                     new String[] { "\u0041" });

    // ÷ 0300 × 0308 ÷ 0041 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u0041",
                     new String[] { "\u0041" });

    // ÷ 0300 ÷ 003A ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u003A",
                     new String[] {  });

    // ÷ 0300 × 0308 ÷ 003A ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u003A",
                     new String[] {  });

    // ÷ 0300 ÷ 002C ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u002C",
                     new String[] {  });

    // ÷ 0300 × 0308 ÷ 002C ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u002C",
                     new String[] {  });

    // ÷ 0300 ÷ 0027 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0027",
                     new String[] {  });

    // ÷ 0300 × 0308 ÷ 0027 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u0027",
                     new String[] {  });

    // ÷ 0300 ÷ 0030 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0030",
                     new String[] { "\u0030" });

    // ÷ 0300 × 0308 ÷ 0030 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u0030",
                     new String[] { "\u0030" });

    // ÷ 0300 ÷ 005F ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u005F",
                     new String[] {  });

    // ÷ 0300 × 0308 ÷ 005F ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u005F",
                     new String[] {  });

    // ÷ 0300 × 00AD ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u00AD",
                     new String[] {  });

    // ÷ 0300 × 0308 × 00AD ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u00AD",
                     new String[] {  });

    // ÷ 0300 × 0300 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0300",
                     new String[] {  });

    // ÷ 0300 × 0308 × 0300 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u0300",
                     new String[] {  });

    // ÷ 0300 ÷ 0061 × 2060 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0061\u2060",
                     new String[] { "\u0061\u2060" });

    // ÷ 0300 × 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u0061\u2060",
                     new String[] { "\u0061\u2060" });

    // ÷ 0300 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0061\u003A",
                     new String[] { "\u0061" });

    // ÷ 0300 × 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u0061\u003A",
                     new String[] { "\u0061" });

    // ÷ 0300 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0061\u0027",
                     new String[] { "\u0061" });

    // ÷ 0300 × 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u0061\u0027",
                     new String[] { "\u0061" });

    // ÷ 0300 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0061\u0027\u2060",
                     new String[] { "\u0061" });

    // ÷ 0300 × 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u0061\u0027\u2060",
                     new String[] { "\u0061" });

    // ÷ 0300 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0061\u002C",
                     new String[] { "\u0061" });

    // ÷ 0300 × 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u0061\u002C",
                     new String[] { "\u0061" });

    // ÷ 0300 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0031\u003A",
                     new String[] { "\u0031" });

    // ÷ 0300 × 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u0031\u003A",
                     new String[] { "\u0031" });

    // ÷ 0300 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0031\u0027",
                     new String[] { "\u0031" });

    // ÷ 0300 × 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u0031\u0027",
                     new String[] { "\u0031" });

    // ÷ 0300 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0031\u002C",
                     new String[] { "\u0031" });

    // ÷ 0300 × 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u0031\u002C",
                     new String[] { "\u0031" });

    // ÷ 0300 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0031\u002E\u2060",
                     new String[] { "\u0031" });

    // ÷ 0300 × 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u0031\u002E\u2060",
                     new String[] { "\u0031" });

    // ÷ 0061 × 2060 ÷ 0001 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0001",
                     new String[] { "\u0061\u2060" });

    // ÷ 0061 × 2060 × 0308 ÷ 0001 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0308\u0001",
                     new String[] { "\u0061\u2060\u0308" });

    // ÷ 0061 × 2060 ÷ 000D ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\r",
                     new String[] { "\u0061\u2060" });

    // ÷ 0061 × 2060 × 0308 ÷ 000D ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0308\r",
                     new String[] { "\u0061\u2060\u0308" });

    // ÷ 0061 × 2060 ÷ 000A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\n",
                     new String[] { "\u0061\u2060" });

    // ÷ 0061 × 2060 × 0308 ÷ 000A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0308\n",
                     new String[] { "\u0061\u2060\u0308" });

    // ÷ 0061 × 2060 ÷ 000B ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u000B",
                     new String[] { "\u0061\u2060" });

    // ÷ 0061 × 2060 × 0308 ÷ 000B ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0308\u000B",
                     new String[] { "\u0061\u2060\u0308" });

    // ÷ 0061 × 2060 ÷ 3031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u3031",
                     new String[] { "\u0061\u2060", "\u3031" });

    // ÷ 0061 × 2060 × 0308 ÷ 3031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0308\u3031",
                     new String[] { "\u0061\u2060\u0308", "\u3031" });

    // ÷ 0061 × 2060 × 0041 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [5.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0041",
                     new String[] { "\u0061\u2060\u0041" });

    // ÷ 0061 × 2060 × 0308 × 0041 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [5.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0308\u0041",
                     new String[] { "\u0061\u2060\u0308\u0041" });

    // ÷ 0061 × 2060 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u003A",
                     new String[] { "\u0061\u2060" });

    // ÷ 0061 × 2060 × 0308 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0308\u003A",
                     new String[] { "\u0061\u2060\u0308" });

    // ÷ 0061 × 2060 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u002C",
                     new String[] { "\u0061\u2060" });

    // ÷ 0061 × 2060 × 0308 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0308\u002C",
                     new String[] { "\u0061\u2060\u0308" });

    // ÷ 0061 × 2060 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0027",
                     new String[] { "\u0061\u2060" });

    // ÷ 0061 × 2060 × 0308 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0308\u0027",
                     new String[] { "\u0061\u2060\u0308" });

    // ÷ 0061 × 2060 × 0030 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [9.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0030",
                     new String[] { "\u0061\u2060\u0030" });

    // ÷ 0061 × 2060 × 0308 × 0030 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [9.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0308\u0030",
                     new String[] { "\u0061\u2060\u0308\u0030" });

    // ÷ 0061 × 2060 × 005F ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [13.1] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u005F",
                     new String[] { "\u0061\u2060\u005F" });

    // ÷ 0061 × 2060 × 0308 × 005F ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [13.1] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0308\u005F",
                     new String[] { "\u0061\u2060\u0308\u005F" });

    // ÷ 0061 × 2060 × 00AD ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u00AD",
                     new String[] { "\u0061\u2060\u00AD" });

    // ÷ 0061 × 2060 × 0308 × 00AD ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0308\u00AD",
                     new String[] { "\u0061\u2060\u0308\u00AD" });

    // ÷ 0061 × 2060 × 0300 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0300",
                     new String[] { "\u0061\u2060\u0300" });

    // ÷ 0061 × 2060 × 0308 × 0300 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0308\u0300",
                     new String[] { "\u0061\u2060\u0308\u0300" });

    // ÷ 0061 × 2060 × 0061 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [5.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0061\u2060",
                     new String[] { "\u0061\u2060\u0061\u2060" });

    // ÷ 0061 × 2060 × 0308 × 0061 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [5.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0308\u0061\u2060",
                     new String[] { "\u0061\u2060\u0308\u0061\u2060" });

    // ÷ 0061 × 2060 × 0061 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [5.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0061\u003A",
                     new String[] { "\u0061\u2060\u0061" });

    // ÷ 0061 × 2060 × 0308 × 0061 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [5.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0308\u0061\u003A",
                     new String[] { "\u0061\u2060\u0308\u0061" });

    // ÷ 0061 × 2060 × 0061 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [5.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0061\u0027",
                     new String[] { "\u0061\u2060\u0061" });

    // ÷ 0061 × 2060 × 0308 × 0061 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [5.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0308\u0061\u0027",
                     new String[] { "\u0061\u2060\u0308\u0061" });

    // ÷ 0061 × 2060 × 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [5.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0061\u0027\u2060",
                     new String[] { "\u0061\u2060\u0061" });

    // ÷ 0061 × 2060 × 0308 × 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [5.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0308\u0061\u0027\u2060",
                     new String[] { "\u0061\u2060\u0308\u0061" });

    // ÷ 0061 × 2060 × 0061 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [5.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0061\u002C",
                     new String[] { "\u0061\u2060\u0061" });

    // ÷ 0061 × 2060 × 0308 × 0061 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [5.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0308\u0061\u002C",
                     new String[] { "\u0061\u2060\u0308\u0061" });

    // ÷ 0061 × 2060 × 0031 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [9.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0031\u003A",
                     new String[] { "\u0061\u2060\u0031" });

    // ÷ 0061 × 2060 × 0308 × 0031 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [9.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0308\u0031\u003A",
                     new String[] { "\u0061\u2060\u0308\u0031" });

    // ÷ 0061 × 2060 × 0031 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [9.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0031\u0027",
                     new String[] { "\u0061\u2060\u0031" });

    // ÷ 0061 × 2060 × 0308 × 0031 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [9.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0308\u0031\u0027",
                     new String[] { "\u0061\u2060\u0308\u0031" });

    // ÷ 0061 × 2060 × 0031 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [9.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0031\u002C",
                     new String[] { "\u0061\u2060\u0031" });

    // ÷ 0061 × 2060 × 0308 × 0031 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [9.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0308\u0031\u002C",
                     new String[] { "\u0061\u2060\u0308\u0031" });

    // ÷ 0061 × 2060 × 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [9.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0031\u002E\u2060",
                     new String[] { "\u0061\u2060\u0031" });

    // ÷ 0061 × 2060 × 0308 × 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [9.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0308\u0031\u002E\u2060",
                     new String[] { "\u0061\u2060\u0308\u0031" });

    // ÷ 0061 ÷ 003A ÷ 0001 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0001",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 003A × 0308 ÷ 0001 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u0001",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 003A ÷ 000D ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\r",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 003A × 0308 ÷ 000D ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\r",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 003A ÷ 000A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\n",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 003A × 0308 ÷ 000A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\n",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 003A ÷ 000B ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u000B",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 003A × 0308 ÷ 000B ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u000B",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 003A ÷ 3031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u3031",
                     new String[] { "\u0061", "\u3031" });

    // ÷ 0061 ÷ 003A × 0308 ÷ 3031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u3031",
                     new String[] { "\u0061", "\u3031" });

    // ÷ 0061 × 003A × 0041 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] COLON (MidLetter) × [7.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0041",
                     new String[] { "\u0061\u003A\u0041" });

    // ÷ 0061 × 003A × 0308 × 0041 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [7.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u0041",
                     new String[] { "\u0061\u003A\u0308\u0041" });

    // ÷ 0061 ÷ 003A ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u003A",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 003A × 0308 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u003A",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 003A ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u002C",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 003A × 0308 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u002C",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 003A ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0027",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 003A × 0308 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u0027",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 003A ÷ 0030 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0030",
                     new String[] { "\u0061", "\u0030" });

    // ÷ 0061 ÷ 003A × 0308 ÷ 0030 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u0030",
                     new String[] { "\u0061", "\u0030" });

    // ÷ 0061 ÷ 003A ÷ 005F ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u005F",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 003A × 0308 ÷ 005F ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u005F",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 003A × 00AD ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u00AD",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 003A × 0308 × 00AD ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u00AD",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 003A × 0300 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0300",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 003A × 0308 × 0300 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u0300",
                     new String[] { "\u0061" });

    // ÷ 0061 × 003A × 0061 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] COLON (MidLetter) × [7.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0061\u2060",
                     new String[] { "\u0061\u003A\u0061\u2060" });

    // ÷ 0061 × 003A × 0308 × 0061 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [7.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u0061\u2060",
                     new String[] { "\u0061\u003A\u0308\u0061\u2060" });

    // ÷ 0061 × 003A × 0061 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] COLON (MidLetter) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0061\u003A",
                     new String[] { "\u0061\u003A\u0061" });

    // ÷ 0061 × 003A × 0308 × 0061 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u0061\u003A",
                     new String[] { "\u0061\u003A\u0308\u0061" });

    // ÷ 0061 × 003A × 0061 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] COLON (MidLetter) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0061\u0027",
                     new String[] { "\u0061\u003A\u0061" });

    // ÷ 0061 × 003A × 0308 × 0061 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u0061\u0027",
                     new String[] { "\u0061\u003A\u0308\u0061" });

    // ÷ 0061 × 003A × 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] COLON (MidLetter) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0061\u0027\u2060",
                     new String[] { "\u0061\u003A\u0061" });

    // ÷ 0061 × 003A × 0308 × 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u0061\u0027\u2060",
                     new String[] { "\u0061\u003A\u0308\u0061" });

    // ÷ 0061 × 003A × 0061 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] COLON (MidLetter) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0061\u002C",
                     new String[] { "\u0061\u003A\u0061" });

    // ÷ 0061 × 003A × 0308 × 0061 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u0061\u002C",
                     new String[] { "\u0061\u003A\u0308\u0061" });

    // ÷ 0061 ÷ 003A ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0031\u003A",
                     new String[] { "\u0061", "\u0031" });

    // ÷ 0061 ÷ 003A × 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u0031\u003A",
                     new String[] { "\u0061", "\u0031" });

    // ÷ 0061 ÷ 003A ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0031\u0027",
                     new String[] { "\u0061", "\u0031" });

    // ÷ 0061 ÷ 003A × 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u0031\u0027",
                     new String[] { "\u0061", "\u0031" });

    // ÷ 0061 ÷ 003A ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0031\u002C",
                     new String[] { "\u0061", "\u0031" });

    // ÷ 0061 ÷ 003A × 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u0031\u002C",
                     new String[] { "\u0061", "\u0031" });

    // ÷ 0061 ÷ 003A ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0031\u002E\u2060",
                     new String[] { "\u0061", "\u0031" });

    // ÷ 0061 ÷ 003A × 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u0031\u002E\u2060",
                     new String[] { "\u0061", "\u0031" });

    // ÷ 0061 ÷ 0027 ÷ 0001 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0001",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 × 0308 ÷ 0001 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u0001",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 ÷ 000D ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\r",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 × 0308 ÷ 000D ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\r",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 ÷ 000A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\n",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 × 0308 ÷ 000A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\n",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 ÷ 000B ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u000B",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 × 0308 ÷ 000B ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u000B",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 ÷ 3031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u3031",
                     new String[] { "\u0061", "\u3031" });

    // ÷ 0061 ÷ 0027 × 0308 ÷ 3031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u3031",
                     new String[] { "\u0061", "\u3031" });

    // ÷ 0061 × 0027 × 0041 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE (MidNumLet) × [7.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0041",
                     new String[] { "\u0061\u0027\u0041" });

    // ÷ 0061 × 0027 × 0308 × 0041 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) × [7.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u0041",
                     new String[] { "\u0061\u0027\u0308\u0041" });

    // ÷ 0061 ÷ 0027 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u003A",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 × 0308 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u003A",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u002C",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 × 0308 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u002C",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0027",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 × 0308 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u0027",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 ÷ 0030 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0030",
                     new String[] { "\u0061", "\u0030" });

    // ÷ 0061 ÷ 0027 × 0308 ÷ 0030 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u0030",
                     new String[] { "\u0061", "\u0030" });

    // ÷ 0061 ÷ 0027 ÷ 005F ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u005F",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 × 0308 ÷ 005F ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u005F",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 × 00AD ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u00AD",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 × 0308 × 00AD ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u00AD",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 × 0300 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0300",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 × 0308 × 0300 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u0300",
                     new String[] { "\u0061" });

    // ÷ 0061 × 0027 × 0061 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE (MidNumLet) × [7.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0061\u2060",
                     new String[] { "\u0061\u0027\u0061\u2060" });

    // ÷ 0061 × 0027 × 0308 × 0061 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) × [7.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u0061\u2060",
                     new String[] { "\u0061\u0027\u0308\u0061\u2060" });

    // ÷ 0061 × 0027 × 0061 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE (MidNumLet) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0061\u003A",
                     new String[] { "\u0061\u0027\u0061" });

    // ÷ 0061 × 0027 × 0308 × 0061 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u0061\u003A",
                     new String[] { "\u0061\u0027\u0308\u0061" });

    // ÷ 0061 × 0027 × 0061 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE (MidNumLet) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0061\u0027",
                     new String[] { "\u0061\u0027\u0061" });

    // ÷ 0061 × 0027 × 0308 × 0061 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u0061\u0027",
                     new String[] { "\u0061\u0027\u0308\u0061" });

    // ÷ 0061 × 0027 × 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE (MidNumLet) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0061\u0027\u2060",
                     new String[] { "\u0061\u0027\u0061" });

    // ÷ 0061 × 0027 × 0308 × 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u0061\u0027\u2060",
                     new String[] { "\u0061\u0027\u0308\u0061" });

    // ÷ 0061 × 0027 × 0061 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE (MidNumLet) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0061\u002C",
                     new String[] { "\u0061\u0027\u0061" });

    // ÷ 0061 × 0027 × 0308 × 0061 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u0061\u002C",
                     new String[] { "\u0061\u0027\u0308\u0061" });

    // ÷ 0061 ÷ 0027 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0031\u003A",
                     new String[] { "\u0061", "\u0031" });

    // ÷ 0061 ÷ 0027 × 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u0031\u003A",
                     new String[] { "\u0061", "\u0031" });

    // ÷ 0061 ÷ 0027 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0031\u0027",
                     new String[] { "\u0061", "\u0031" });

    // ÷ 0061 ÷ 0027 × 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u0031\u0027",
                     new String[] { "\u0061", "\u0031" });

    // ÷ 0061 ÷ 0027 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0031\u002C",
                     new String[] { "\u0061", "\u0031" });

    // ÷ 0061 ÷ 0027 × 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u0031\u002C",
                     new String[] { "\u0061", "\u0031" });

    // ÷ 0061 ÷ 0027 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0031\u002E\u2060",
                     new String[] { "\u0061", "\u0031" });

    // ÷ 0061 ÷ 0027 × 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u0031\u002E\u2060",
                     new String[] { "\u0061", "\u0031" });

    // ÷ 0061 ÷ 0027 × 2060 ÷ 0001 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0001",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 × 2060 × 0308 ÷ 0001 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u0001",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 × 2060 ÷ 000D ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\r",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 × 2060 × 0308 ÷ 000D ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\r",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 × 2060 ÷ 000A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\n",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 × 2060 × 0308 ÷ 000A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\n",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 × 2060 ÷ 000B ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u000B",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 × 2060 × 0308 ÷ 000B ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u000B",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 × 2060 ÷ 3031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u3031",
                     new String[] { "\u0061", "\u3031" });

    // ÷ 0061 ÷ 0027 × 2060 × 0308 ÷ 3031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u3031",
                     new String[] { "\u0061", "\u3031" });

    // ÷ 0061 × 0027 × 2060 × 0041 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [7.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0041",
                     new String[] { "\u0061\u0027\u2060\u0041" });

    // ÷ 0061 × 0027 × 2060 × 0308 × 0041 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [7.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u0041",
                     new String[] { "\u0061\u0027\u2060\u0308\u0041" });

    // ÷ 0061 ÷ 0027 × 2060 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u003A",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 × 2060 × 0308 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u003A",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 × 2060 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u002C",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 × 2060 × 0308 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u002C",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 × 2060 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0027",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 × 2060 × 0308 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u0027",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 × 2060 ÷ 0030 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0030",
                     new String[] { "\u0061", "\u0030" });

    // ÷ 0061 ÷ 0027 × 2060 × 0308 ÷ 0030 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u0030",
                     new String[] { "\u0061", "\u0030" });

    // ÷ 0061 ÷ 0027 × 2060 ÷ 005F ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u005F",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 × 2060 × 0308 ÷ 005F ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u005F",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 × 2060 × 00AD ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u00AD",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 × 2060 × 0308 × 00AD ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u00AD",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 × 2060 × 0300 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0300",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 0027 × 2060 × 0308 × 0300 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u0300",
                     new String[] { "\u0061" });

    // ÷ 0061 × 0027 × 2060 × 0061 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [7.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0061\u2060",
                     new String[] { "\u0061\u0027\u2060\u0061\u2060" });

    // ÷ 0061 × 0027 × 2060 × 0308 × 0061 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [7.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u0061\u2060",
                     new String[] { "\u0061\u0027\u2060\u0308\u0061\u2060" });

    // ÷ 0061 × 0027 × 2060 × 0061 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0061\u003A",
                     new String[] { "\u0061\u0027\u2060\u0061" });

    // ÷ 0061 × 0027 × 2060 × 0308 × 0061 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u0061\u003A",
                     new String[] { "\u0061\u0027\u2060\u0308\u0061" });

    // ÷ 0061 × 0027 × 2060 × 0061 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0061\u0027",
                     new String[] { "\u0061\u0027\u2060\u0061" });

    // ÷ 0061 × 0027 × 2060 × 0308 × 0061 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u0061\u0027",
                     new String[] { "\u0061\u0027\u2060\u0308\u0061" });

    // ÷ 0061 × 0027 × 2060 × 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0061\u0027\u2060",
                     new String[] { "\u0061\u0027\u2060\u0061" });

    // ÷ 0061 × 0027 × 2060 × 0308 × 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u0061\u0027\u2060",
                     new String[] { "\u0061\u0027\u2060\u0308\u0061" });

    // ÷ 0061 × 0027 × 2060 × 0061 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0061\u002C",
                     new String[] { "\u0061\u0027\u2060\u0061" });

    // ÷ 0061 × 0027 × 2060 × 0308 × 0061 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u0061\u002C",
                     new String[] { "\u0061\u0027\u2060\u0308\u0061" });

    // ÷ 0061 ÷ 0027 × 2060 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0031\u003A",
                     new String[] { "\u0061", "\u0031" });

    // ÷ 0061 ÷ 0027 × 2060 × 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u0031\u003A",
                     new String[] { "\u0061", "\u0031" });

    // ÷ 0061 ÷ 0027 × 2060 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0031\u0027",
                     new String[] { "\u0061", "\u0031" });

    // ÷ 0061 ÷ 0027 × 2060 × 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u0031\u0027",
                     new String[] { "\u0061", "\u0031" });

    // ÷ 0061 ÷ 0027 × 2060 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0031\u002C",
                     new String[] { "\u0061", "\u0031" });

    // ÷ 0061 ÷ 0027 × 2060 × 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u0031\u002C",
                     new String[] { "\u0061", "\u0031" });

    // ÷ 0061 ÷ 0027 × 2060 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0031\u002E\u2060",
                     new String[] { "\u0061", "\u0031" });

    // ÷ 0061 ÷ 0027 × 2060 × 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u0031\u002E\u2060",
                     new String[] { "\u0061", "\u0031" });

    // ÷ 0061 ÷ 002C ÷ 0001 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0001",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 002C × 0308 ÷ 0001 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u0001",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 002C ÷ 000D ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\r",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 002C × 0308 ÷ 000D ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\r",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 002C ÷ 000A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\n",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 002C × 0308 ÷ 000A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\n",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 002C ÷ 000B ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u000B",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 002C × 0308 ÷ 000B ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u000B",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 002C ÷ 3031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u3031",
                     new String[] { "\u0061", "\u3031" });

    // ÷ 0061 ÷ 002C × 0308 ÷ 3031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u3031",
                     new String[] { "\u0061", "\u3031" });

    // ÷ 0061 ÷ 002C ÷ 0041 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0041",
                     new String[] { "\u0061", "\u0041" });

    // ÷ 0061 ÷ 002C × 0308 ÷ 0041 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u0041",
                     new String[] { "\u0061", "\u0041" });

    // ÷ 0061 ÷ 002C ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u003A",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 002C × 0308 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u003A",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 002C ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u002C",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 002C × 0308 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u002C",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 002C ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0027",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 002C × 0308 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u0027",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 002C ÷ 0030 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0030",
                     new String[] { "\u0061", "\u0030" });

    // ÷ 0061 ÷ 002C × 0308 ÷ 0030 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u0030",
                     new String[] { "\u0061", "\u0030" });

    // ÷ 0061 ÷ 002C ÷ 005F ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u005F",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 002C × 0308 ÷ 005F ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u005F",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 002C × 00AD ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u00AD",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 002C × 0308 × 00AD ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u00AD",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 002C × 0300 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0300",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 002C × 0308 × 0300 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u0300",
                     new String[] { "\u0061" });

    // ÷ 0061 ÷ 002C ÷ 0061 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0061\u2060",
                     new String[] { "\u0061", "\u0061\u2060" });

    // ÷ 0061 ÷ 002C × 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u0061\u2060",
                     new String[] { "\u0061", "\u0061\u2060" });

    // ÷ 0061 ÷ 002C ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0061\u003A",
                     new String[] { "\u0061", "\u0061" });

    // ÷ 0061 ÷ 002C × 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u0061\u003A",
                     new String[] { "\u0061", "\u0061" });

    // ÷ 0061 ÷ 002C ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0061\u0027",
                     new String[] { "\u0061", "\u0061" });

    // ÷ 0061 ÷ 002C × 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u0061\u0027",
                     new String[] { "\u0061", "\u0061" });

    // ÷ 0061 ÷ 002C ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0061\u0027\u2060",
                     new String[] { "\u0061", "\u0061" });

    // ÷ 0061 ÷ 002C × 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u0061\u0027\u2060",
                     new String[] { "\u0061", "\u0061" });

    // ÷ 0061 ÷ 002C ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0061\u002C",
                     new String[] { "\u0061", "\u0061" });

    // ÷ 0061 ÷ 002C × 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u0061\u002C",
                     new String[] { "\u0061", "\u0061" });

    // ÷ 0061 ÷ 002C ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0031\u003A",
                     new String[] { "\u0061", "\u0031" });

    // ÷ 0061 ÷ 002C × 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u0031\u003A",
                     new String[] { "\u0061", "\u0031" });

    // ÷ 0061 ÷ 002C ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0031\u0027",
                     new String[] { "\u0061", "\u0031" });

    // ÷ 0061 ÷ 002C × 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u0031\u0027",
                     new String[] { "\u0061", "\u0031" });

    // ÷ 0061 ÷ 002C ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0031\u002C",
                     new String[] { "\u0061", "\u0031" });

    // ÷ 0061 ÷ 002C × 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u0031\u002C",
                     new String[] { "\u0061", "\u0031" });

    // ÷ 0061 ÷ 002C ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0031\u002E\u2060",
                     new String[] { "\u0061", "\u0031" });

    // ÷ 0061 ÷ 002C × 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u0031\u002E\u2060",
                     new String[] { "\u0061", "\u0031" });

    // ÷ 0031 ÷ 003A ÷ 0001 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0001",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 003A × 0308 ÷ 0001 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u0001",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 003A ÷ 000D ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\r",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 003A × 0308 ÷ 000D ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\r",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 003A ÷ 000A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\n",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 003A × 0308 ÷ 000A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\n",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 003A ÷ 000B ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u000B",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 003A × 0308 ÷ 000B ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u000B",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 003A ÷ 3031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u3031",
                     new String[] { "\u0031", "\u3031" });

    // ÷ 0031 ÷ 003A × 0308 ÷ 3031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u3031",
                     new String[] { "\u0031", "\u3031" });

    // ÷ 0031 ÷ 003A ÷ 0041 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0041",
                     new String[] { "\u0031", "\u0041" });

    // ÷ 0031 ÷ 003A × 0308 ÷ 0041 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u0041",
                     new String[] { "\u0031", "\u0041" });

    // ÷ 0031 ÷ 003A ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u003A",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 003A × 0308 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u003A",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 003A ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u002C",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 003A × 0308 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u002C",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 003A ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0027",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 003A × 0308 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u0027",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 003A ÷ 0030 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0030",
                     new String[] { "\u0031", "\u0030" });

    // ÷ 0031 ÷ 003A × 0308 ÷ 0030 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u0030",
                     new String[] { "\u0031", "\u0030" });

    // ÷ 0031 ÷ 003A ÷ 005F ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u005F",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 003A × 0308 ÷ 005F ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u005F",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 003A × 00AD ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u00AD",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 003A × 0308 × 00AD ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u00AD",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 003A × 0300 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0300",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 003A × 0308 × 0300 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u0300",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 003A ÷ 0061 × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0061\u2060",
                     new String[] { "\u0031", "\u0061\u2060" });

    // ÷ 0031 ÷ 003A × 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u0061\u2060",
                     new String[] { "\u0031", "\u0061\u2060" });

    // ÷ 0031 ÷ 003A ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0061\u003A",
                     new String[] { "\u0031", "\u0061" });

    // ÷ 0031 ÷ 003A × 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u0061\u003A",
                     new String[] { "\u0031", "\u0061" });

    // ÷ 0031 ÷ 003A ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0061\u0027",
                     new String[] { "\u0031", "\u0061" });

    // ÷ 0031 ÷ 003A × 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u0061\u0027",
                     new String[] { "\u0031", "\u0061" });

    // ÷ 0031 ÷ 003A ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0061\u0027\u2060",
                     new String[] { "\u0031", "\u0061" });

    // ÷ 0031 ÷ 003A × 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u0061\u0027\u2060",
                     new String[] { "\u0031", "\u0061" });

    // ÷ 0031 ÷ 003A ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0061\u002C",
                     new String[] { "\u0031", "\u0061" });

    // ÷ 0031 ÷ 003A × 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u0061\u002C",
                     new String[] { "\u0031", "\u0061" });

    // ÷ 0031 ÷ 003A ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0031\u003A",
                     new String[] { "\u0031", "\u0031" });

    // ÷ 0031 ÷ 003A × 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u0031\u003A",
                     new String[] { "\u0031", "\u0031" });

    // ÷ 0031 ÷ 003A ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0031\u0027",
                     new String[] { "\u0031", "\u0031" });

    // ÷ 0031 ÷ 003A × 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u0031\u0027",
                     new String[] { "\u0031", "\u0031" });

    // ÷ 0031 ÷ 003A ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0031\u002C",
                     new String[] { "\u0031", "\u0031" });

    // ÷ 0031 ÷ 003A × 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u0031\u002C",
                     new String[] { "\u0031", "\u0031" });

    // ÷ 0031 ÷ 003A ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0031\u002E\u2060",
                     new String[] { "\u0031", "\u0031" });

    // ÷ 0031 ÷ 003A × 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u0031\u002E\u2060",
                     new String[] { "\u0031", "\u0031" });

    // ÷ 0031 ÷ 0027 ÷ 0001 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0001",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 0027 × 0308 ÷ 0001 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u0001",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 0027 ÷ 000D ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\r",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 0027 × 0308 ÷ 000D ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\r",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 0027 ÷ 000A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\n",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 0027 × 0308 ÷ 000A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\n",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 0027 ÷ 000B ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u000B",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 0027 × 0308 ÷ 000B ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u000B",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 0027 ÷ 3031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u3031",
                     new String[] { "\u0031", "\u3031" });

    // ÷ 0031 ÷ 0027 × 0308 ÷ 3031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u3031",
                     new String[] { "\u0031", "\u3031" });

    // ÷ 0031 ÷ 0027 ÷ 0041 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0041",
                     new String[] { "\u0031", "\u0041" });

    // ÷ 0031 ÷ 0027 × 0308 ÷ 0041 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u0041",
                     new String[] { "\u0031", "\u0041" });

    // ÷ 0031 ÷ 0027 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u003A",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 0027 × 0308 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u003A",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 0027 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u002C",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 0027 × 0308 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u002C",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 0027 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0027",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 0027 × 0308 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u0027",
                     new String[] { "\u0031" });

    // ÷ 0031 × 0027 × 0030 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] APOSTROPHE (MidNumLet) × [11.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0030",
                     new String[] { "\u0031\u0027\u0030" });

    // ÷ 0031 × 0027 × 0308 × 0030 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) × [11.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u0030",
                     new String[] { "\u0031\u0027\u0308\u0030" });

    // ÷ 0031 ÷ 0027 ÷ 005F ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u005F",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 0027 × 0308 ÷ 005F ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u005F",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 0027 × 00AD ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u00AD",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 0027 × 0308 × 00AD ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u00AD",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 0027 × 0300 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0300",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 0027 × 0308 × 0300 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u0300",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 0027 ÷ 0061 × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0061\u2060",
                     new String[] { "\u0031", "\u0061\u2060" });

    // ÷ 0031 ÷ 0027 × 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u0061\u2060",
                     new String[] { "\u0031", "\u0061\u2060" });

    // ÷ 0031 ÷ 0027 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0061\u003A",
                     new String[] { "\u0031", "\u0061" });

    // ÷ 0031 ÷ 0027 × 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u0061\u003A",
                     new String[] { "\u0031", "\u0061" });

    // ÷ 0031 ÷ 0027 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0061\u0027",
                     new String[] { "\u0031", "\u0061" });

    // ÷ 0031 ÷ 0027 × 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u0061\u0027",
                     new String[] { "\u0031", "\u0061" });

    // ÷ 0031 ÷ 0027 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0061\u0027\u2060",
                     new String[] { "\u0031", "\u0061" });

    // ÷ 0031 ÷ 0027 × 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u0061\u0027\u2060",
                     new String[] { "\u0031", "\u0061" });

    // ÷ 0031 ÷ 0027 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0061\u002C",
                     new String[] { "\u0031", "\u0061" });

    // ÷ 0031 ÷ 0027 × 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u0061\u002C",
                     new String[] { "\u0031", "\u0061" });

    // ÷ 0031 × 0027 × 0031 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] APOSTROPHE (MidNumLet) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0031\u003A",
                     new String[] { "\u0031\u0027\u0031" });

    // ÷ 0031 × 0027 × 0308 × 0031 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u0031\u003A",
                     new String[] { "\u0031\u0027\u0308\u0031" });

    // ÷ 0031 × 0027 × 0031 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] APOSTROPHE (MidNumLet) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0031\u0027",
                     new String[] { "\u0031\u0027\u0031" });

    // ÷ 0031 × 0027 × 0308 × 0031 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u0031\u0027",
                     new String[] { "\u0031\u0027\u0308\u0031" });

    // ÷ 0031 × 0027 × 0031 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] APOSTROPHE (MidNumLet) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0031\u002C",
                     new String[] { "\u0031\u0027\u0031" });

    // ÷ 0031 × 0027 × 0308 × 0031 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u0031\u002C",
                     new String[] { "\u0031\u0027\u0308\u0031" });

    // ÷ 0031 × 0027 × 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] APOSTROPHE (MidNumLet) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0031\u002E\u2060",
                     new String[] { "\u0031\u0027\u0031" });

    // ÷ 0031 × 0027 × 0308 × 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] APOSTROPHE (MidNumLet) × [4.0] COMBINING DIAERESIS (Extend_FE) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u0031\u002E\u2060",
                     new String[] { "\u0031\u0027\u0308\u0031" });

    // ÷ 0031 ÷ 002C ÷ 0001 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0001",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002C × 0308 ÷ 0001 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u0001",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002C ÷ 000D ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\r",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002C × 0308 ÷ 000D ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\r",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002C ÷ 000A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\n",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002C × 0308 ÷ 000A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\n",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002C ÷ 000B ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u000B",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002C × 0308 ÷ 000B ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u000B",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002C ÷ 3031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u3031",
                     new String[] { "\u0031", "\u3031" });

    // ÷ 0031 ÷ 002C × 0308 ÷ 3031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u3031",
                     new String[] { "\u0031", "\u3031" });

    // ÷ 0031 ÷ 002C ÷ 0041 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0041",
                     new String[] { "\u0031", "\u0041" });

    // ÷ 0031 ÷ 002C × 0308 ÷ 0041 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u0041",
                     new String[] { "\u0031", "\u0041" });

    // ÷ 0031 ÷ 002C ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u003A",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002C × 0308 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u003A",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002C ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u002C",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002C × 0308 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u002C",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002C ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0027",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002C × 0308 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u0027",
                     new String[] { "\u0031" });

    // ÷ 0031 × 002C × 0030 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] COMMA (MidNum) × [11.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0030",
                     new String[] { "\u0031\u002C\u0030" });

    // ÷ 0031 × 002C × 0308 × 0030 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) × [11.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u0030",
                     new String[] { "\u0031\u002C\u0308\u0030" });

    // ÷ 0031 ÷ 002C ÷ 005F ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u005F",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002C × 0308 ÷ 005F ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u005F",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002C × 00AD ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u00AD",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002C × 0308 × 00AD ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u00AD",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002C × 0300 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0300",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002C × 0308 × 0300 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u0300",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002C ÷ 0061 × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0061\u2060",
                     new String[] { "\u0031", "\u0061\u2060" });

    // ÷ 0031 ÷ 002C × 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u0061\u2060",
                     new String[] { "\u0031", "\u0061\u2060" });

    // ÷ 0031 ÷ 002C ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0061\u003A",
                     new String[] { "\u0031", "\u0061" });

    // ÷ 0031 ÷ 002C × 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u0061\u003A",
                     new String[] { "\u0031", "\u0061" });

    // ÷ 0031 ÷ 002C ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0061\u0027",
                     new String[] { "\u0031", "\u0061" });

    // ÷ 0031 ÷ 002C × 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u0061\u0027",
                     new String[] { "\u0031", "\u0061" });

    // ÷ 0031 ÷ 002C ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0061\u0027\u2060",
                     new String[] { "\u0031", "\u0061" });

    // ÷ 0031 ÷ 002C × 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u0061\u0027\u2060",
                     new String[] { "\u0031", "\u0061" });

    // ÷ 0031 ÷ 002C ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0061\u002C",
                     new String[] { "\u0031", "\u0061" });

    // ÷ 0031 ÷ 002C × 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u0061\u002C",
                     new String[] { "\u0031", "\u0061" });

    // ÷ 0031 × 002C × 0031 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] COMMA (MidNum) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0031\u003A",
                     new String[] { "\u0031\u002C\u0031" });

    // ÷ 0031 × 002C × 0308 × 0031 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u0031\u003A",
                     new String[] { "\u0031\u002C\u0308\u0031" });

    // ÷ 0031 × 002C × 0031 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] COMMA (MidNum) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0031\u0027",
                     new String[] { "\u0031\u002C\u0031" });

    // ÷ 0031 × 002C × 0308 × 0031 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u0031\u0027",
                     new String[] { "\u0031\u002C\u0308\u0031" });

    // ÷ 0031 × 002C × 0031 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] COMMA (MidNum) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0031\u002C",
                     new String[] { "\u0031\u002C\u0031" });

    // ÷ 0031 × 002C × 0308 × 0031 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u0031\u002C",
                     new String[] { "\u0031\u002C\u0308\u0031" });

    // ÷ 0031 × 002C × 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] COMMA (MidNum) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0031\u002E\u2060",
                     new String[] { "\u0031\u002C\u0031" });

    // ÷ 0031 × 002C × 0308 × 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u0031\u002E\u2060",
                     new String[] { "\u0031\u002C\u0308\u0031" });

    // ÷ 0031 ÷ 002E × 2060 ÷ 0001 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0001",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 0001 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u0001",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002E × 2060 ÷ 000D ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\r",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 000D ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\r",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002E × 2060 ÷ 000A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\n",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 000A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\n",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002E × 2060 ÷ 000B ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u000B",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 000B ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u000B",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002E × 2060 ÷ 3031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u3031",
                     new String[] { "\u0031", "\u3031" });

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 3031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u3031",
                     new String[] { "\u0031", "\u3031" });

    // ÷ 0031 ÷ 002E × 2060 ÷ 0041 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0041",
                     new String[] { "\u0031", "\u0041" });

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 0041 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u0041",
                     new String[] { "\u0031", "\u0041" });

    // ÷ 0031 ÷ 002E × 2060 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u003A",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u003A",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002E × 2060 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u002C",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u002C",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002E × 2060 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0027",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u0027",
                     new String[] { "\u0031" });

    // ÷ 0031 × 002E × 2060 × 0030 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [11.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0030",
                     new String[] { "\u0031\u002E\u2060\u0030" });

    // ÷ 0031 × 002E × 2060 × 0308 × 0030 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [11.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u0030",
                     new String[] { "\u0031\u002E\u2060\u0308\u0030" });

    // ÷ 0031 ÷ 002E × 2060 ÷ 005F ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u005F",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 005F ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u005F",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002E × 2060 × 00AD ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u00AD",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002E × 2060 × 0308 × 00AD ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u00AD",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002E × 2060 × 0300 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0300",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002E × 2060 × 0308 × 0300 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u0300",
                     new String[] { "\u0031" });

    // ÷ 0031 ÷ 002E × 2060 ÷ 0061 × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0061\u2060",
                     new String[] { "\u0031", "\u0061\u2060" });

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u0061\u2060",
                     new String[] { "\u0031", "\u0061\u2060" });

    // ÷ 0031 ÷ 002E × 2060 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0061\u003A",
                     new String[] { "\u0031", "\u0061" });

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u0061\u003A",
                     new String[] { "\u0031", "\u0061" });

    // ÷ 0031 ÷ 002E × 2060 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0061\u0027",
                     new String[] { "\u0031", "\u0061" });

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u0061\u0027",
                     new String[] { "\u0031", "\u0061" });

    // ÷ 0031 ÷ 002E × 2060 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0061\u0027\u2060",
                     new String[] { "\u0031", "\u0061" });

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u0061\u0027\u2060",
                     new String[] { "\u0031", "\u0061" });

    // ÷ 0031 ÷ 002E × 2060 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0061\u002C",
                     new String[] { "\u0031", "\u0061" });

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u0061\u002C",
                     new String[] { "\u0031", "\u0061" });

    // ÷ 0031 × 002E × 2060 × 0031 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0031\u003A",
                     new String[] { "\u0031\u002E\u2060\u0031" });

    // ÷ 0031 × 002E × 2060 × 0308 × 0031 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u0031\u003A",
                     new String[] { "\u0031\u002E\u2060\u0308\u0031" });

    // ÷ 0031 × 002E × 2060 × 0031 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0031\u0027",
                     new String[] { "\u0031\u002E\u2060\u0031" });

    // ÷ 0031 × 002E × 2060 × 0308 × 0031 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u0031\u0027",
                     new String[] { "\u0031\u002E\u2060\u0308\u0031" });

    // ÷ 0031 × 002E × 2060 × 0031 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0031\u002C",
                     new String[] { "\u0031\u002E\u2060\u0031" });

    // ÷ 0031 × 002E × 2060 × 0308 × 0031 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u0031\u002C",
                     new String[] { "\u0031\u002E\u2060\u0308\u0031" });

    // ÷ 0031 × 002E × 2060 × 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0031\u002E\u2060",
                     new String[] { "\u0031\u002E\u2060\u0031" });

    // ÷ 0031 × 002E × 2060 × 0308 × 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u0031\u002E\u2060",
                     new String[] { "\u0031\u002E\u2060\u0308\u0031" });

    // ÷ 0063 × 0061 × 006E × 0027 × 0074 ÷  #  ÷ [0.2] LATIN SMALL LETTER C (ALetter) × [5.0] LATIN SMALL LETTER A (ALetter) × [5.0] LATIN SMALL LETTER N (ALetter) × [6.0] APOSTROPHE (MidNumLet) × [7.0] LATIN SMALL LETTER T (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0063\u0061\u006E\u0027\u0074",
                     new String[] { "\u0063\u0061\u006E\u0027\u0074" });

    // ÷ 0063 × 0061 × 006E × 2019 × 0074 ÷  #  ÷ [0.2] LATIN SMALL LETTER C (ALetter) × [5.0] LATIN SMALL LETTER A (ALetter) × [5.0] LATIN SMALL LETTER N (ALetter) × [6.0] RIGHT SINGLE QUOTATION MARK (MidNumLet) × [7.0] LATIN SMALL LETTER T (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0063\u0061\u006E\u2019\u0074",
                     new String[] { "\u0063\u0061\u006E\u2019\u0074" });

    // ÷ 0061 × 0062 × 00AD × 0062 × 0079 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [5.0] LATIN SMALL LETTER B (ALetter) × [4.0] SOFT HYPHEN (Format_FE) × [5.0] LATIN SMALL LETTER B (ALetter) × [5.0] LATIN SMALL LETTER Y (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0062\u00AD\u0062\u0079",
                     new String[] { "\u0061\u0062\u00AD\u0062\u0079" });

    // ÷ 0061 ÷ 0024 ÷ 002D ÷ 0033 × 0034 × 002C × 0035 × 0036 × 0037 × 002E × 0031 × 0034 ÷ 0025 ÷ 0062 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] DOLLAR SIGN (Other) ÷ [999.0] HYPHEN-MINUS (Other) ÷ [999.0] DIGIT THREE (Numeric) × [8.0] DIGIT FOUR (Numeric) × [12.0] COMMA (MidNum) × [11.0] DIGIT FIVE (Numeric) × [8.0] DIGIT SIX (Numeric) × [8.0] DIGIT SEVEN (Numeric) × [12.0] FULL STOP (MidNumLet) × [11.0] DIGIT ONE (Numeric) × [8.0] DIGIT FOUR (Numeric) ÷ [999.0] PERCENT SIGN (Other) ÷ [999.0] LATIN SMALL LETTER B (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0024\u002D\u0033\u0034\u002C\u0035\u0036\u0037\u002E\u0031\u0034\u0025\u0062",
                     new String[] { "\u0061", "\u0033\u0034\u002C\u0035\u0036\u0037\u002E\u0031\u0034", "\u0062" });

    // ÷ 0033 × 0061 ÷  #  ÷ [0.2] DIGIT THREE (Numeric) × [10.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0033\u0061",
                     new String[] { "\u0033\u0061" });

    // ÷ 2060 ÷ 0063 × 2060 × 0061 × 2060 × 006E × 2060 × 0027 × 2060 × 0074 × 2060 × 2060 ÷  #  ÷ [0.2] WORD JOINER (Format_FE) ÷ [999.0] LATIN SMALL LETTER C (ALetter) × [4.0] WORD JOINER (Format_FE) × [5.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [5.0] LATIN SMALL LETTER N (ALetter) × [4.0] WORD JOINER (Format_FE) × [6.0] APOSTROPHE (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [7.0] LATIN SMALL LETTER T (ALetter) × [4.0] WORD JOINER (Format_FE) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u2060\u0063\u2060\u0061\u2060\u006E\u2060\u0027\u2060\u0074\u2060\u2060",
                     new String[] { "\u0063\u2060\u0061\u2060\u006E\u2060\u0027\u2060\u0074\u2060\u2060" });

    // ÷ 2060 ÷ 0063 × 2060 × 0061 × 2060 × 006E × 2060 × 2019 × 2060 × 0074 × 2060 × 2060 ÷  #  ÷ [0.2] WORD JOINER (Format_FE) ÷ [999.0] LATIN SMALL LETTER C (ALetter) × [4.0] WORD JOINER (Format_FE) × [5.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [5.0] LATIN SMALL LETTER N (ALetter) × [4.0] WORD JOINER (Format_FE) × [6.0] RIGHT SINGLE QUOTATION MARK (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [7.0] LATIN SMALL LETTER T (ALetter) × [4.0] WORD JOINER (Format_FE) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u2060\u0063\u2060\u0061\u2060\u006E\u2060\u2019\u2060\u0074\u2060\u2060",
                     new String[] { "\u0063\u2060\u0061\u2060\u006E\u2060\u2019\u2060\u0074\u2060\u2060" });

    // ÷ 2060 ÷ 0061 × 2060 × 0062 × 2060 × 00AD × 2060 × 0062 × 2060 × 0079 × 2060 × 2060 ÷  #  ÷ [0.2] WORD JOINER (Format_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [5.0] LATIN SMALL LETTER B (ALetter) × [4.0] WORD JOINER (Format_FE) × [4.0] SOFT HYPHEN (Format_FE) × [4.0] WORD JOINER (Format_FE) × [5.0] LATIN SMALL LETTER B (ALetter) × [4.0] WORD JOINER (Format_FE) × [5.0] LATIN SMALL LETTER Y (ALetter) × [4.0] WORD JOINER (Format_FE) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u2060\u0061\u2060\u0062\u2060\u00AD\u2060\u0062\u2060\u0079\u2060\u2060",
                     new String[] { "\u0061\u2060\u0062\u2060\u00AD\u2060\u0062\u2060\u0079\u2060\u2060" });

    // ÷ 2060 ÷ 0061 × 2060 ÷ 0024 × 2060 ÷ 002D × 2060 ÷ 0033 × 2060 × 0034 × 2060 × 002C × 2060 × 0035 × 2060 × 0036 × 2060 × 0037 × 2060 × 002E × 2060 × 0031 × 2060 × 0034 × 2060 ÷ 0025 × 2060 ÷ 0062 × 2060 × 2060 ÷  #  ÷ [0.2] WORD JOINER (Format_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] DOLLAR SIGN (Other) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] HYPHEN-MINUS (Other) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] DIGIT THREE (Numeric) × [4.0] WORD JOINER (Format_FE) × [8.0] DIGIT FOUR (Numeric) × [4.0] WORD JOINER (Format_FE) × [12.0] COMMA (MidNum) × [4.0] WORD JOINER (Format_FE) × [11.0] DIGIT FIVE (Numeric) × [4.0] WORD JOINER (Format_FE) × [8.0] DIGIT SIX (Numeric) × [4.0] WORD JOINER (Format_FE) × [8.0] DIGIT SEVEN (Numeric) × [4.0] WORD JOINER (Format_FE) × [12.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [11.0] DIGIT ONE (Numeric) × [4.0] WORD JOINER (Format_FE) × [8.0] DIGIT FOUR (Numeric) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] PERCENT SIGN (Other) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] LATIN SMALL LETTER B (ALetter) × [4.0] WORD JOINER (Format_FE) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u2060\u0061\u2060\u0024\u2060\u002D\u2060\u0033\u2060\u0034\u2060\u002C\u2060\u0035\u2060\u0036\u2060\u0037\u2060\u002E\u2060\u0031\u2060\u0034\u2060\u0025\u2060\u0062\u2060\u2060",
                     new String[] { "\u0061\u2060", "\u0033\u2060\u0034\u2060\u002C\u2060\u0035\u2060\u0036\u2060\u0037\u2060\u002E\u2060\u0031\u2060\u0034\u2060", "\u0062\u2060\u2060" });

    // ÷ 2060 ÷ 0033 × 2060 × 0061 × 2060 × 2060 ÷  #  ÷ [0.2] WORD JOINER (Format_FE) ÷ [999.0] DIGIT THREE (Numeric) × [4.0] WORD JOINER (Format_FE) × [10.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u2060\u0033\u2060\u0061\u2060\u2060",
                     new String[] { "\u0033\u2060\u0061\u2060\u2060" });

  }
}
