// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.

using System.Diagnostics;
using System.Numerics;
using System.Text;

namespace System.Security.Cryptography
{
    /// <summary>
    /// Reads data encoded via the Distinguished Encoding Rules for Abstract
    /// Syntax Notation 1 (ASN.1) data.
    /// </summary>
    internal class DerSequenceReader
    {
        internal const byte ContextSpecificTagFlag = 0x80;

        internal const byte TagNumberMask = 0x1F;

        private readonly byte[] _data;
        private readonly int _end;
        private int _position;

        internal int ContentLength { get; private set; }

        private DerSequenceReader(bool startAtPayload, byte[] data)
        {
            Debug.Assert(startAtPayload, "This overload is only for bypassing the sequence tag");
            Debug.Assert(data != null, "Data is null");

            _data = data;
            _position = 0;
            _end = data.Length;

            ContentLength = data.Length;
        }

        internal DerSequenceReader(byte[] data)
            : this(data, 0, data.Length)
        {
        }

        internal DerSequenceReader(byte[] data, int offset, int length)
        {
            Debug.Assert(data != null, "Data is null");

            if (offset < 0 || length < 2 || length > data.Length - offset)
                throw new CryptographicException(SR.Cryptography_Der_Invalid_Encoding);

            _data = data;
            _end = offset + length;
            _position = offset;
            EatTag(DerTag.Sequence);
            int contentLength = EatLength();
            Debug.Assert(_end - contentLength >= _position);
            ContentLength = contentLength;

            // If the sequence reports being smaller than the buffer, shrink the end-of-validity.
            _end = _position + contentLength;
        }

        internal static DerSequenceReader CreateForPayload(byte[] payload)
        {
            return new DerSequenceReader(true, payload);
        }

        internal bool HasData
        {
            get { return _position < _end; }
        }

        internal byte PeekTag()
        {
            if (!HasData)
                throw new CryptographicException(SR.Cryptography_Der_Invalid_Encoding);

            byte tag = _data[_position];

            if ((tag & TagNumberMask) == TagNumberMask)
            {
                throw new CryptographicException(SR.Cryptography_Der_Invalid_Encoding);
            }

            return tag;
        }

        internal void SkipValue()
        {
            EatTag((DerTag)PeekTag());
            int contentLength = EatLength();
            _position += contentLength;
        }

        internal int ReadInteger()
        {
            byte[] integerBytes = ReadIntegerBytes();

            // integerBytes is currently Big-Endian, need to reverse it for
            // Little-Endian to pass into BigInteger.
            Array.Reverse(integerBytes);
            BigInteger bigInt = new BigInteger(integerBytes);
            return (int)bigInt;
        }

        internal byte[] ReadIntegerBytes()
        {
            EatTag(DerTag.Integer);

            return ReadContentAsBytes();
        }

        internal byte[] ReadOctetString()
        {
            EatTag(DerTag.OctetString);

            return ReadContentAsBytes();
        }

        internal string ReadOidAsString()
        {
            EatTag(DerTag.ObjectIdentifier);
            int contentLength = EatLength();

            if (contentLength < 1)
                throw new CryptographicException(SR.Cryptography_Der_Invalid_Encoding);

            // Each byte could cause 3 decimal characters to be written, plus a period. Over-allocate
            // and avoid re-alloc.
            StringBuilder builder = new StringBuilder(contentLength * 4);

            // The first byte is ((X * 40) + Y), where X is the first segment and Y the second.
            // ISO/IEC 8825-1:2003 section 8.19.4

            byte firstByte = _data[_position];
            byte first = (byte)(firstByte / 40);
            byte second = (byte)(firstByte % 40);

            builder.Append(first);
            builder.Append('.');
            builder.Append(second);

            // For the rest of the segments, the high bit on the byte is a continuation marker,
            // and data is loaded into a BigInteger 7 bits at a time.
            //
            // When the high bit is 0, the segment ends, so emit a '.' between it and the next one.
            //
            // ISO/IEC 8825-1:2003 section 8.19.2, and the .NET representation of Oid.Value.
            bool needDot = true;
            BigInteger bigInt = new BigInteger(0);

            for (int i = 1; i < contentLength; i++)
            {
                byte current = _data[_position + i];
                byte data = (byte)(current & 0x7F);

                if (needDot)
                {
                    builder.Append('.');
                    needDot = false;
                }

                bigInt <<= 7;
                bigInt += data;

                if (current == data)
                {
                    builder.Append(bigInt);
                    bigInt = 0;
                    needDot = true;
                }
            }

            _position += contentLength;
            return builder.ToString();
        }

        internal Oid ReadOid()
        {
            return new Oid(ReadOidAsString());
        }

        internal DerSequenceReader ReadSequence()
        {
            // DerSequenceReader wants to read its own tag, so don't EatTag here.
            CheckTag(DerTag.Sequence, _data, _position);

            int lengthLength;
            int contentLength = ScanContentLength(_data, _position + 1, _end, out lengthLength);
            int totalLength = 1 + lengthLength + contentLength;

            DerSequenceReader reader = new DerSequenceReader(_data, _position, totalLength);
            _position += totalLength;
            return reader;
        }

        internal string ReadIA5String()
        {
            EatTag(DerTag.IA5String);
            int contentLength = EatLength();

            // IA5 (International Alphabet - 5) is functionally equivalent to 7-bit ASCII.

            string ia5String = System.Text.Encoding.ASCII.GetString(_data, _position, contentLength);
            _position += contentLength;

            return ia5String;
        }

        private byte[] ReadContentAsBytes()
        {
            int contentLength = EatLength();

            byte[] octets = new byte[contentLength];
            Buffer.BlockCopy(_data, _position, octets, 0, contentLength);

            _position += contentLength;
            return octets;
        }

        private void EatTag(DerTag expected)
        {
            if (!HasData)
                throw new CryptographicException(SR.Cryptography_Der_Invalid_Encoding);

            CheckTag(expected, _data, _position);
            _position++;
        }

        private static void CheckTag(DerTag expected, byte[] data, int position)
        {
            if (position >= data.Length)
                throw new CryptographicException(SR.Cryptography_Der_Invalid_Encoding);

            byte actual = data[position];
            byte relevant = (byte)(actual & TagNumberMask);

            // Multi-byte tags are not supported by this implementation.
            if (relevant == TagNumberMask)
            {
                throw new CryptographicException(SR.Cryptography_Der_Invalid_Encoding);
            }

            // Context-specific datatypes cannot be tag-verified
            if ((actual & ContextSpecificTagFlag) != 0)
            {
                return;
            }

            byte expectedByte = (byte)((byte)expected & TagNumberMask);

            if (expectedByte != relevant)
            {
                throw new CryptographicException(
                    SR.Cryptography_Der_Invalid_Encoding
#if DEBUG
                    ,
                    new InvalidOperationException(
                        "Expected tag '0x" + expectedByte.ToString("X2") +
                            "', got '0x" + actual.ToString("X2") +
                            "' at position " + position)
#endif
                    );
            }
        }

        private int EatLength()
        {
            int bytesConsumed;
            int answer = ScanContentLength(_data, _position, _end, out bytesConsumed);

            _position += bytesConsumed;
            return answer;
        }

        private static int ScanContentLength(byte[] data, int offset, int end, out int bytesConsumed)
        {
            Debug.Assert(end <= data.Length);

            if (offset >= end)
                throw new CryptographicException(SR.Cryptography_Der_Invalid_Encoding);

            byte lengthOrLengthLength = data[offset];

            if (lengthOrLengthLength < 0x80)
            {
                bytesConsumed = 1;

                if (lengthOrLengthLength > end - offset - bytesConsumed)
                    throw new CryptographicException(SR.Cryptography_Der_Invalid_Encoding);

                return lengthOrLengthLength;
            }

            int lengthLength = (lengthOrLengthLength & 0x7F);

            if (lengthLength > sizeof(int))
            {
                // .NET Arrays cannot exceed int.MaxValue in length. Since we're bounded by an
                // array we know that this is invalid data.
                throw new CryptographicException(SR.Cryptography_Der_Invalid_Encoding);
            }

            // The one byte which was lengthLength, plus the number of bytes it said to consume.
            bytesConsumed = 1 + lengthLength;

            if (bytesConsumed > end - offset)
                throw new CryptographicException(SR.Cryptography_Der_Invalid_Encoding);

            // CER indefinite length is not supported.
            if (bytesConsumed == 1)
                throw new CryptographicException(SR.Cryptography_Der_Invalid_Encoding);

            int lengthEnd = offset + bytesConsumed;
            int accum = 0;
            
            // data[offset] is lengthLength, so start at data[offset + 1] and stop before
            // data[offset + 1 + lengthLength], aka data[end].
            for (int i = offset + 1; i < lengthEnd; i++)
            {
                accum <<= 8;
                accum |= data[i];
            }

            if (accum < 0)
            {
                // .NET Arrays cannot exceed int.MaxValue in length. Since we're bounded by an
                // array we know that this is invalid data.
                throw new CryptographicException(SR.Cryptography_Der_Invalid_Encoding);
            }

            if (accum > end - offset - bytesConsumed)
                throw new CryptographicException(SR.Cryptography_Der_Invalid_Encoding);

            return accum;
        }

        internal enum DerTag : byte
        {
            Boolean = 0x01,
            Integer = 0x02,
            BitString = 0x03,
            OctetString = 0x04,
            Null = 0x05,
            ObjectIdentifier = 0x06,
            UTF8String = 0x0C,
            Sequence = 0x10,
            Set = 0x11,
            PrintableString = 0x13,
            T61String = 0x14,
            IA5String = 0x16,
            UTCTime = 0x17,
        }
    }
}
