// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.

namespace System.Globalization {
    using System;
    using System.Diagnostics.Contracts;

    ////////////////////////////////////////////////////////////////////////////
    //
    //  Notes about KoreanLunisolarCalendar
    //
    ////////////////////////////////////////////////////////////////////////////
     /*
     **  Calendar support range:
     **      Calendar               Minimum             Maximum
     **      ==========     ==========  ==========
     **      Gregorian              918/02/14          2051/02/10
     **      KoreanLunisolar    918/01/01          2050/13/29
     */

    [Serializable]
    public class KoreanLunisolarCalendar : EastAsianLunisolarCalendar {


        //
        // The era value for the current era.
        //

        public const int GregorianEra = 1;

        //internal static Calendar m_defaultInstance;

        internal const int MIN_LUNISOLAR_YEAR = 918;
        internal const int MAX_LUNISOLAR_YEAR = 2050;

        internal const int MIN_GREGORIAN_YEAR  = 918;
        internal const int MIN_GREGORIAN_MONTH = 2;
        internal const int MIN_GREGORIAN_DAY   = 14;

        internal const int MAX_GREGORIAN_YEAR = 2051;
        internal const int MAX_GREGORIAN_MONTH = 2;
        internal const int MAX_GREGORIAN_DAY = 10;

        internal static DateTime minDate = new DateTime(MIN_GREGORIAN_YEAR, MIN_GREGORIAN_MONTH, MIN_GREGORIAN_DAY);
        internal static DateTime maxDate = new DateTime((new DateTime(MAX_GREGORIAN_YEAR, MAX_GREGORIAN_MONTH, MAX_GREGORIAN_DAY, 23, 59, 59, 999)).Ticks + 9999);

        public override DateTime MinSupportedDateTime {
            get
            {
                return (minDate);
            }
        }



        public override DateTime MaxSupportedDateTime {
            get
            {
                return (maxDate);
            }
        }

        protected override int DaysInYearBeforeMinSupportedYear
        {
            get
            {
                // 917  -- From http://emr.cs.iit.edu/home/reingold/calendar-book/Calendrica.html
                //              using ChineseLunisolar
                return 384;
            }
        }

        static readonly int  [,] yinfo =
        {
/*Y            LM        Lmon    Lday        DaysPerMonth    D1    D2    D3    D4    D5    D6    D7    D8    D9    D10    D11    D12    D13    #Days
918    */{    0    ,    2    ,    14    ,    21936    },/*    29    30    29    30    29    30    29    30    30    29    30    30    0    355
919    */{    0    ,    2    ,    4    ,    17872    },/*    29    30    29    29    29    30    29    30    30    30    29    30    0    354
920    */{    6    ,    1    ,    24    ,    41688    },/*    30    29    30    29    29    29    30    29    30    30    29    30    30    384
921    */{    0    ,    2    ,    11    ,    41648    },/*    30    29    30    29    29    29    30    29    30    29    30    30    0    354
922    */{    0    ,    1    ,    31    ,    43344    },/*    30    29    30    29    30    29    29    30    29    30    29    30    0    354
923    */{    4    ,    1    ,    20    ,    46248    },/*    30    29    30    30    29    30    29    29    30    29    30    29    30    384
924    */{    0    ,    2    ,    8    ,    27936    },/*    29    30    30    29    30    30    29    30    29    29    30    29    0    354
925    */{    12    ,    1    ,    27    ,    44384    },/*    30    29    30    29    30    30    29    30    29    30    30    29    29    384
926    */{    0    ,    2    ,    15    ,    43872    },/*    30    29    30    29    30    29    30    30    29    30    30    29    0    355
927    */{    0    ,    2    ,    5    ,    21936    },/*    29    30    29    30    29    30    29    30    30    29    30    30    0    355
928    */{    8    ,    1    ,    26    ,    17848    },/*    29    30    29    29    29    30    29    30    30    29    30    30    30    384
929    */{    0    ,    2    ,    13    ,    17776    },/*    29    30    29    29    29    30    29    30    29    30    30    30    0    354
930    */{    0    ,    2    ,    2    ,    21168    },/*    29    30    29    30    29    29    30    29    30    29    30    30    0    354
931    */{    5    ,    1    ,    22    ,    26960    },/*    29    30    30    29    30    29    29    30    29    30    29    30    29    383
932    */{    0    ,    2    ,    9    ,    59728    },/*    30    30    30    29    30    29    29    30    29    30    29    30    0    355
933    */{    0    ,    1    ,    29    ,    27296    },/*    29    30    30    29    30    29    30    29    30    29    30    29    0    354
934    */{    1    ,    1    ,    18    ,    44368    },/*    30    29    30    29    30    30    29    30    29    30    29    30    29    384
935    */{    0    ,    2    ,    6    ,    43856    },/*    30    29    30    29    30    29    30    30    29    30    29    30    0    355
936    */{    11    ,    1    ,    27    ,    21344    },/*    29    30    29    30    29    29    30    30    29    30    30    29    29    383
937    */{    0    ,    2    ,    13    ,    51904    },/*    30    30    29    29    30    29    30    29    30    30    29    29    0    354
938    */{    0    ,    2    ,    2    ,    58720    },/*    30    30    30    29    29    30    29    30    29    30    30    29    0    355
939    */{    7    ,    1    ,    23    ,    53928    },/*    30    30    29    30    29    29    30    29    30    29    30    29    30    384
940    */{    0    ,    2    ,    11    ,    53920    },/*    30    30    29    30    29    29    30    29    30    29    30    29    0    354
941    */{    0    ,    1    ,    30    ,    55632    },/*    30    30    29    30    30    29    29    30    29    30    29    30    0    355
942    */{    3    ,    1    ,    20    ,    23208    },/*    29    30    29    30    30    29    30    29    30    29    30    29    30    384
943    */{    0    ,    2    ,    8    ,    22176    },/*    29    30    29    30    29    30    30    29    30    29    30    29    0    354
944    */{    12    ,    1    ,    28    ,    42704    },/*    30    29    30    29    29    30    30    29    30    30    29    30    29    384
945    */{    0    ,    2    ,    15    ,    38352    },/*    30    29    29    30    29    30    29    30    30    30    29    30    0    355
946    */{    0    ,    2    ,    5    ,    19152    },/*    29    30    29    29    30    29    30    29    30    30    29    30    0    354
947    */{    7    ,    1    ,    25    ,    42200    },/*    30    29    30    29    29    30    29    29    30    30    29    30    30    384
948    */{    0    ,    2    ,    13    ,    42192    },/*    30    29    30    29    29    30    29    29    30    30    29    30    0    354
949    */{    0    ,    2    ,    1    ,    45664    },/*    30    29    30    30    29    29    30    29    29    30    30    29    0    354
950    */{    5    ,    1    ,    21    ,    46416    },/*    30    29    30    30    29    30    29    30    29    30    29    30    29    384
951    */{    0    ,    2    ,    9    ,    45936    },/*    30    29    30    30    29    30    29    30    29    30    29    0    0    325
952    */{    0    ,    12    ,    31    ,    43728    },/*    30    29    30    29    30    29    30    29    30    30    29    30    29    384
953    */{    1    ,    1    ,    18    ,    38352    },/*    30    29    29    30    29    30    29    30    30    30    29    30    29    384
954    */{    0    ,    2    ,    6    ,    38320    },/*    30    29    29    30    29    30    29    30    30    29    30    30    0    355
955    */{    9    ,    1    ,    27    ,    19128    },/*    29    30    29    29    30    29    30    29    30    29    30    30    30    384
956    */{    0    ,    2    ,    15    ,    18864    },/*    29    30    29    29    30    29    29    30    30    29    30    30    0    354
957    */{    0    ,    2    ,    3    ,    42160    },/*    30    29    30    29    29    30    29    29    30    29    30    30    0    354
958    */{    7    ,    1    ,    23    ,    43672    },/*    30    29    30    29    30    29    30    29    30    29    29    30    30    384
959    */{    0    ,    2    ,    11    ,    27296    },/*    29    30    30    29    30    29    30    29    30    29    30    29    0    354
960    */{    0    ,    1    ,    31    ,    44368    },/*    30    29    30    29    30    30    29    30    29    30    29    30    0    355
961    */{    3    ,    1    ,    20    ,    19880    },/*    29    30    29    29    30    30    29    30    30    29    30    29    30    384
962    */{    0    ,    2    ,    8    ,    11104    },/*    29    29    30    29    30    29    30    30    29    30    30    29    0    354
963    */{    12    ,    1    ,    28    ,    38256    },/*    30    29    29    30    29    30    29    30    29    30    30    30    29    384
964    */{    0    ,    2    ,    16    ,    41840    },/*    30    29    30    29    29    29    30    30    29    30    30    30    0    355
965    */{    0    ,    2    ,    5    ,    20848    },/*    29    30    29    30    29    29    29    30    29    30    30    30    0    354
966    */{    8    ,    1    ,    25    ,    25776    },/*    29    30    30    29    29    30    29    29    30    29    30    30    29    383
967    */{    0    ,    2    ,    12    ,    54448    },/*    30    30    29    30    29    30    29    29    30    29    30    30    0    355
968    */{    0    ,    2    ,    2    ,    23184    },/*    29    30    29    30    30    29    30    29    30    29    29    30    0    354
969    */{    5    ,    1    ,    21    ,    27472    },/*    29    30    30    29    30    29    30    30    29    30    29    30    29    384
970    */{    0    ,    2    ,    9    ,    22224    },/*    29    30    29    30    29    30    30    29    30    30    29    30    0    355
971    */{    0    ,    1    ,    30    ,    10976    },/*    29    29    30    29    30    29    30    29    30    30    30    29    0    354
972    */{    2    ,    1    ,    19    ,    37744    },/*    30    29    29    30    29    29    30    30    29    30    30    30    29    384
973    */{    0    ,    2    ,    6    ,    41696    },/*    30    29    30    29    29    29    30    29    30    30    30    29    0    354
974    */{    10    ,    1    ,    26    ,    51560    },/*    30    30    29    29    30    29    29    30    29    30    30    29    30    384
975    */{    0    ,    2    ,    14    ,    43344    },/*    30    29    30    29    30    29    29    30    29    30    29    30    0    354
976    */{    0    ,    2    ,    3    ,    54432    },/*    30    30    29    30    29    30    29    29    30    29    30    29    0    354
977    */{    7    ,    1    ,    22    ,    55952    },/*    30    30    29    30    30    29    30    29    30    29    29    30    29    384
978    */{    0    ,    2    ,    10    ,    46496    },/*    30    29    30    30    29    30    29    30    30    29    30    29    0    355
979    */{    0    ,    1    ,    31    ,    22224    },/*    29    30    29    30    29    30    30    29    30    30    29    30    0    355
980    */{    3    ,    1    ,    21    ,    10968    },/*    29    29    30    29    30    29    30    29    30    30    29    30    30    384
981    */{    0    ,    2    ,    8    ,    9680    },/*    29    29    30    29    29    30    29    30    30    30    29    30    0    354
982    */{    12    ,    1    ,    28    ,    37592    },/*    30    29    29    30    29    29    30    29    30    30    29    30    30    384
983    */{    0    ,    2    ,    16    ,    37552    },/*    30    29    29    30    29    29    30    29    30    29    30    30    0    354
984    */{    0    ,    2    ,    5    ,    43344    },/*    30    29    30    29    30    29    29    30    29    30    29    30    0    354
985    */{    9    ,    1    ,    24    ,    46248    },/*    30    29    30    30    29    30    29    29    30    29    30    29    30    384
986    */{    0    ,    2    ,    12    ,    44192    },/*    30    29    30    29    30    30    29    29    30    29    30    29    0    354
987    */{    0    ,    2    ,    1    ,    44368    },/*    30    29    30    29    30    30    29    30    29    30    29    30    0    355
988    */{    5    ,    1    ,    22    ,    21936    },/*    29    30    29    30    29    30    29    30    30    29    30    30    29    384
989    */{    0    ,    2    ,    9    ,    19376    },/*    29    30    29    29    30    29    30    30    30    29    30    30    0    355
990    */{    0    ,    1    ,    30    ,    9648    },/*    29    29    30    29    29    30    29    30    30    29    30    30    0    354
991    */{    2    ,    1    ,    19    ,    37560    },/*    30    29    29    30    29    29    30    29    30    29    30    30    30    384
992    */{    0    ,    2    ,    7    ,    21168    },/*    29    30    29    30    29    29    30    29    30    29    30    30    0    354
993    */{    10    ,    1    ,    26    ,    26968    },/*    29    30    30    29    30    29    29    30    29    30    29    30    30    384
994    */{    0    ,    2    ,    14    ,    22864    },/*    29    30    29    30    30    29    29    30    29    30    29    30    0    354
995    */{    0    ,    2    ,    3    ,    27296    },/*    29    30    30    29    30    29    30    29    30    29    30    29    0    354
996    */{    7    ,    1    ,    23    ,    43856    },/*    30    29    30    29    30    29    30    30    29    30    29    30    29    384
997    */{    0    ,    2    ,    10    ,    43872    },/*    30    29    30    29    30    29    30    30    29    30    30    29    0    355
998    */{    0    ,    1    ,    31    ,    19296    },/*    29    30    29    29    30    29    30    30    29    30    30    29    0    354
999    */{    3    ,    1    ,    20    ,    42352    },/*    30    29    30    29    29    30    29    30    29    30    30    30    29    384
1000    */{    0    ,    2    ,    8    ,    9584    },/*    29    29    30    29    29    30    29    30    29    30    30    30    0    354
1001    */{    12    ,    1    ,    28    ,    21168    },/*    29    30    29    30    29    29    30    29    30    29    30    30    29    383
1002    */{    0    ,    2    ,    15    ,    53920    },/*    30    30    29    30    29    29    30    29    30    29    30    29    0    354
1003    */{    0    ,    2    ,    4    ,    54608    },/*    30    30    29    30    29    30    29    30    29    30    29    30    0    355
1004    */{    9    ,    1    ,    25    ,    23208    },/*    29    30    29    30    30    29    30    29    30    29    30    29    30    384
1005    */{    0    ,    2    ,    12    ,    22176    },/*    29    30    29    30    29    30    30    29    30    29    30    29    0    354
1006    */{    0    ,    2    ,    1    ,    38608    },/*    30    29    29    30    29    30    30    29    30    30    29    30    0    355
1007    */{    5    ,    1    ,    22    ,    19176    },/*    29    30    29    29    30    29    30    29    30    30    30    29    30    384
1008    */{    0    ,    2    ,    10    ,    19152    },/*    29    30    29    29    30    29    30    29    30    30    29    30    0    354
1009    */{    0    ,    1    ,    29    ,    42192    },/*    30    29    30    29    29    30    29    29    30    30    29    30    0    354
1010    */{    2    ,    1    ,    18    ,    53864    },/*    30    30    29    30    29    29    30    29    29    30    30    29    30    384
1011    */{    0    ,    2    ,    6    ,    45664    },/*    30    29    30    30    29    29    30    29    29    30    30    29    0    354
1012    */{    10    ,    1    ,    26    ,    46416    },/*    30    29    30    30    29    30    29    30    29    30    29    30    29    384
1013    */{    0    ,    2    ,    13    ,    44368    },/*    30    29    30    29    30    30    29    30    29    30    29    30    0    355
1014    */{    0    ,    2    ,    3    ,    13728    },/*    29    29    30    30    29    30    29    30    30    29    30    29    0    354
1015    */{    6    ,    1    ,    23    ,    38352    },/*    30    29    29    30    29    30    29    30    30    30    29    30    29    384
1016    */{    0    ,    2    ,    11    ,    38320    },/*    30    29    29    30    29    30    29    30    30    29    30    30    0    355
1017    */{    0    ,    1    ,    31    ,    18864    },/*    29    30    29    29    30    29    29    30    30    29    30    30    0    354
1018    */{    4    ,    1    ,    20    ,    42200    },/*    30    29    30    29    29    30    29    29    30    30    29    30    30    384
1019    */{    0    ,    2    ,    8    ,    42160    },/*    30    29    30    29    29    30    29    29    30    29    30    30    0    354
1020    */{    12    ,    1    ,    28    ,    43608    },/*    30    29    30    29    30    29    30    29    29    30    29    30    30    384
1021    */{    0    ,    2    ,    15    ,    27296    },/*    29    30    30    29    30    29    30    29    30    29    30    29    0    354
1022    */{    0    ,    2    ,    4    ,    44368    },/*    30    29    30    29    30    30    29    30    29    30    29    30    0    355
1023    */{    9    ,    1    ,    25    ,    11688    },/*    29    29    30    29    30    30    29    30    30    29    30    29    30    384
1024    */{    0    ,    2    ,    13    ,    11088    },/*    29    29    30    29    30    29    30    30    29    30    29    30    0    354
1025    */{    0    ,    2    ,    1    ,    38256    },/*    30    29    29    30    29    30    29    30    29    30    30    30    0    355
1026    */{    5    ,    1    ,    22    ,    18800    },/*    29    30    29    29    30    29    29    30    29    30    30    30    29    383
1027    */{    0    ,    2    ,    9    ,    51568    },/*    30    30    29    29    30    29    29    30    29    30    30    30    0    355
1028    */{    0    ,    1    ,    30    ,    25776    },/*    29    30    30    29    29    30    29    29    30    29    30    30    0    354
1029    */{    2    ,    1    ,    18    ,    27216    },/*    29    30    30    29    30    29    30    29    29    30    29    30    29    383
1030    */{    0    ,    2    ,    5    ,    55952    },/*    30    30    29    30    30    29    30    29    30    29    29    30    0    355
1031    */{    10    ,    1    ,    26    ,    27472    },/*    29    30    30    29    30    29    30    30    29    30    29    30    29    384
1032    */{    0    ,    2    ,    14    ,    26320    },/*    29    30    30    29    29    30    30    29    30    30    29    30    0    355
1033    */{    0    ,    2    ,    3    ,    9952    },/*    29    29    30    29    29    30    30    29    30    30    30    29    0    354
1034    */{    6    ,    1    ,    23    ,    37744    },/*    30    29    29    30    29    29    30    30    29    30    30    30    29    384
1035    */{    0    ,    2    ,    11    ,    37600    },/*    30    29    29    30    29    29    30    29    30    30    30    29    0    354
1036    */{    0    ,    1    ,    31    ,    51552    },/*    30    30    29    29    30    29    29    30    29    30    30    29    0    354
1037    */{    4    ,    1    ,    19    ,    54440    },/*    30    30    29    30    29    30    29    29    30    29    30    29    30    384
1038    */{    0    ,    2    ,    7    ,    54432    },/*    30    30    29    30    29    30    29    29    30    29    30    29    0    354
1039    */{    12    ,    1    ,    27    ,    54928    },/*    30    30    29    30    29    30    30    29    30    29    29    30    29    384
1040    */{    0    ,    2    ,    15    ,    46464    },/*    30    29    30    30    29    30    29    30    30    29    29    29    0    354
1041    */{    0    ,    2    ,    3    ,    54960    },/*    30    30    29    30    29    30    30    29    30    29    30    30    0    356
1042    */{    9    ,    1    ,    25    ,    9944    },/*    29    29    30    29    29    30    30    29    30    30    29    30    30    384
1043    */{    0    ,    2    ,    13    ,    9680    },/*    29    29    30    29    29    30    29    30    30    30    29    30    0    354
1044    */{    0    ,    2    ,    2    ,    37552    },/*    30    29    29    30    29    29    30    29    30    29    30    30    0    354
1045    */{    5    ,    1    ,    21    ,    43352    },/*    30    29    30    29    30    29    29    30    29    30    29    30    30    384
1046    */{    0    ,    2    ,    9    ,    43344    },/*    30    29    30    29    30    29    29    30    29    30    29    30    0    354
1047    */{    0    ,    1    ,    29    ,    46240    },/*    30    29    30    30    29    30    29    29    30    29    30    29    0    354
1048    */{    1    ,    1    ,    18    ,    46424    },/*    30    29    30    30    29    30    29    30    29    30    29    30    30    385
1049    */{    0    ,    2    ,    6    ,    11600    },/*    29    29    30    29    30    30    29    30    29    30    29    30    0    354
1050    */{    11    ,    1    ,    26    ,    21936    },/*    29    30    29    30    29    30    29    30    30    29    30    30    29    384
1051    */{    0    ,    2    ,    14    ,    19376    },/*    29    30    29    29    30    29    30    30    30    29    30    30    0    355
1052    */{    0    ,    2    ,    4    ,    9648    },/*    29    29    30    29    29    30    29    30    30    29    30    30    0    354
1053    */{    7    ,    1    ,    23    ,    21176    },/*    29    30    29    30    29    29    30    29    30    29    30    30    30    384
1054    */{    0    ,    2    ,    11    ,    21168    },/*    29    30    29    30    29    29    30    29    30    29    30    30    0    354
1055    */{    0    ,    1    ,    31    ,    26960    },/*    29    30    30    29    30    29    29    30    29    30    29    30    0    354
1056    */{    3    ,    1    ,    20    ,    27304    },/*    29    30    30    29    30    29    30    29    30    29    30    29    30    384
1057    */{    0    ,    2    ,    7    ,    27296    },/*    29    30    30    29    30    29    30    29    30    29    30    29    0    354
1058    */{    12    ,    1    ,    27    ,    43864    },/*    30    29    30    29    30    29    30    30    29    30    29    30    30    385
1059    */{    0    ,    2    ,    16    ,    10064    },/*    29    29    30    29    29    30    30    30    29    30    29    30    0    354
1060    */{    0    ,    2    ,    5    ,    19296    },/*    29    30    29    29    30    29    30    30    29    30    30    29    0    354
1061    */{    8    ,    1    ,    24    ,    42352    },/*    30    29    30    29    29    30    29    30    29    30    30    30    29    384
1062    */{    0    ,    2    ,    12    ,    42336    },/*    30    29    30    29    29    30    29    30    29    30    30    29    0    354
1063    */{    0    ,    2    ,    1    ,    53856    },/*    30    30    29    30    29    29    30    29    29    30    30    29    0    354
1064    */{    5    ,    1    ,    21    ,    59696    },/*    30    30    30    29    30    29    29    30    29    29    30    30    29    384
1065    */{    0    ,    2    ,    8    ,    54608    },/*    30    30    29    30    29    30    29    30    29    30    29    30    0    355
1066    */{    0    ,    1    ,    29    ,    23200    },/*    29    30    29    30    30    29    30    29    30    29    30    29    0    354
1067    */{    1    ,    1    ,    18    ,    43856    },/*    30    29    30    29    30    29    30    30    29    30    29    30    29    384
1068    */{    0    ,    2    ,    6    ,    38608    },/*    30    29    29    30    29    30    30    29    30    30    29    30    0    355
1069    */{    11    ,    1    ,    26    ,    19176    },/*    29    30    29    29    30    29    30    29    30    30    30    29    30    384
1070    */{    0    ,    2    ,    14    ,    18896    },/*    29    30    29    29    30    29    29    30    30    30    29    30    0    354
1071    */{    0    ,    2    ,    3    ,    42192    },/*    30    29    30    29    29    30    29    29    30    30    29    30    0    354
1072    */{    7    ,    1    ,    23    ,    53864    },/*    30    30    29    30    29    29    30    29    29    30    30    29    30    384
1073    */{    0    ,    2    ,    10    ,    43616    },/*    30    29    30    29    30    29    30    29    29    30    30    29    0    354
1074    */{    0    ,    1    ,    30    ,    46416    },/*    30    29    30    30    29    30    29    30    29    30    29    30    0    355
1075    */{    4    ,    1    ,    20    ,    22184    },/*    29    30    29    30    29    30    30    29    30    29    30    29    30    384
1076    */{    0    ,    2    ,    8    ,    13728    },/*    29    29    30    30    29    30    29    30    30    29    30    29    0    354
1077    */{    0    ,    1    ,    27    ,    38352    },/*    30    29    29    30    29    30    29    30    30    30    29    30    0    355
1078    */{    1    ,    1    ,    17    ,    19352    },/*    29    30    29    29    30    29    30    30    30    29    29    30    30    384
1079    */{    0    ,    2    ,    5    ,    17840    },/*    29    30    29    29    29    30    29    30    30    29    30    30    0    354
1080    */{    9    ,    1    ,    25    ,    42168    },/*    30    29    30    29    29    30    29    29    30    29    30    30    30    384
1081    */{    0    ,    2    ,    12    ,    25776    },/*    29    30    30    29    29    30    29    29    30    29    30    30    0    354
1082    */{    0    ,    2    ,    1    ,    43600    },/*    30    29    30    29    30    29    30    29    29    30    29    30    0    354
1083    */{    6    ,    1    ,    21    ,    46408    },/*    30    29    30    30    29    30    29    30    29    30    29    29    30    384
1084    */{    0    ,    2    ,    9    ,    27472    },/*    29    30    30    29    30    29    30    30    29    30    29    30    0    355
1085    */{    0    ,    1    ,    29    ,    11680    },/*    29    29    30    29    30    30    29    30    30    29    30    29    0    354
1086    */{    2    ,    1    ,    18    ,    38320    },/*    30    29    29    30    29    30    29    30    30    29    30    30    29    384
1087    */{    0    ,    2    ,    6    ,    37744    },/*    30    29    29    30    29    29    30    30    29    30    30    30    0    355
1088    */{    12    ,    1    ,    27    ,    18800    },/*    29    30    29    29    30    29    29    30    29    30    30    30    29    383
1089    */{    0    ,    2    ,    13    ,    51568    },/*    30    30    29    29    30    29    29    30    29    30    30    30    0    355
1090    */{    0    ,    2    ,    3    ,    25776    },/*    29    30    30    29    29    30    29    29    30    29    30    30    0    354
1091    */{    8    ,    1    ,    23    ,    27216    },/*    29    30    30    29    30    29    30    29    29    30    29    30    29    383
1092    */{    0    ,    2    ,    10    ,    55888    },/*    30    30    29    30    30    29    30    29    29    30    29    30    0    355
1093    */{    0    ,    1    ,    30    ,    23360    },/*    29    30    29    30    30    29    30    30    29    30    29    29    0    354
1094    */{    4    ,    1    ,    19    ,    43880    },/*    30    29    30    29    30    29    30    30    29    30    30    29    30    385
1095    */{    0    ,    2    ,    8    ,    10976    },/*    29    29    30    29    30    29    30    29    30    30    30    29    0    354
1096    */{    0    ,    1    ,    28    ,    58896    },/*    30    30    30    29    29    30    30    29    29    29    29    30    0    354
1097    */{    2    ,    1    ,    16    ,    51568    },/*    30    30    29    29    30    29    29    30    29    30    30    30    29    384
1098    */{    0    ,    2    ,    4    ,    51552    },/*    30    30    29    29    30    29    29    30    29    30    30    29    0    354
1099    */{    9    ,    1    ,    24    ,    54440    },/*    30    30    29    30    29    30    29    29    30    29    30    29    30    384
1100    */{    0    ,    2    ,    12    ,    21664    },/*    29    30    29    30    29    30    29    29    30    29    30    29    0    353
1101    */{    0    ,    1    ,    31    ,    54864    },/*    30    30    29    30    29    30    30    29    29    30    29    30    0    355
1102    */{    6    ,    1    ,    21    ,    23208    },/*    29    30    29    30    30    29    30    29    30    29    30    29    30    384
1103    */{    0    ,    2    ,    9    ,    21968    },/*    29    30    29    30    29    30    29    30    30    30    29    30    0    355
1104    */{    0    ,    1    ,    30    ,    9936    },/*    29    29    30    29    29    30    30    29    30    30    29    30    0    354
1105    */{    2    ,    1    ,    18    ,    37608    },/*    30    29    29    30    29    29    30    29    30    30    30    29    30    384
1106    */{    0    ,    2    ,    6    ,    37552    },/*    30    29    29    30    29    29    30    29    30    29    30    30    0    354
1107    */{    10    ,    1    ,    26    ,    43352    },/*    30    29    30    29    30    29    29    30    29    30    29    30    30    384
1108    */{    0    ,    2    ,    14    ,    43344    },/*    30    29    30    29    30    29    29    30    29    30    29    30    0    354
1109    */{    0    ,    2    ,    2    ,    46240    },/*    30    29    30    30    29    30    29    29    30    29    30    29    0    354
1110    */{    8    ,    1    ,    22    ,    46416    },/*    30    29    30    30    29    30    29    30    29    30    29    30    29    384
1111    */{    0    ,    2    ,    10    ,    44368    },/*    30    29    30    29    30    30    29    30    29    30    29    30    0    355
1112    */{    0    ,    1    ,    31    ,    21936    },/*    29    30    29    30    29    30    29    30    30    29    30    30    0    355
1113    */{    4    ,    1    ,    20    ,    9656    },/*    29    29    30    29    29    30    29    30    30    29    30    30    30    384
1114    */{    0    ,    2    ,    8    ,    17776    },/*    29    30    29    29    29    30    29    30    29    30    30    30    0    354
1115    */{    0    ,    1    ,    28    ,    21168    },/*    29    30    29    30    29    29    30    29    30    29    30    30    0    354
1116    */{    1    ,    1    ,    17    ,    43352    },/*    30    29    30    29    30    29    29    30    29    30    29    30    30    384
1117    */{    0    ,    2    ,    4    ,    26960    },/*    29    30    30    29    30    29    29    30    29    30    29    30    0    354
1118    */{    9    ,    1    ,    24    ,    29352    },/*    29    30    30    30    29    29    30    29    30    29    30    29    30    384
1119    */{    0    ,    2    ,    12    ,    23200    },/*    29    30    29    30    30    29    30    29    30    29    30    29    0    354
1120    */{    0    ,    2    ,    1    ,    43856    },/*    30    29    30    29    30    29    30    30    29    30    29    30    0    355
1121    */{    5    ,    1    ,    21    ,    19304    },/*    29    30    29    29    30    29    30    30    29    30    30    29    30    384
1122    */{    0    ,    2    ,    9    ,    19296    },/*    29    30    29    29    30    29    30    30    29    30    30    29    0    354
1123    */{    0    ,    1    ,    29    ,    42352    },/*    30    29    30    29    29    30    29    30    29    30    30    30    0    355
1124    */{    3    ,    1    ,    19    ,    21104    },/*    29    30    29    30    29    29    30    29    29    30    30    30    29    383
1125    */{    0    ,    2    ,    5    ,    53856    },/*    30    30    29    30    29    29    30    29    29    30    30    29    0    354
1126    */{    11    ,    1    ,    25    ,    59696    },/*    30    30    30    29    30    29    29    30    29    29    30    30    29    384
1127    */{    0    ,    2    ,    13    ,    54608    },/*    30    30    29    30    29    30    29    30    29    30    29    30    0    355
1128    */{    0    ,    2    ,    3    ,    23200    },/*    29    30    29    30    30    29    30    29    30    29    30    29    0    354
1129    */{    8    ,    1    ,    22    ,    39824    },/*    30    29    29    30    30    29    30    30    30    29    29    30    29    384
1130    */{    0    ,    2    ,    10    ,    38608    },/*    30    29    29    30    29    30    30    29    30    30    29    30    0    355
1131    */{    0    ,    1    ,    31    ,    19168    },/*    29    30    29    29    30    29    30    29    30    30    30    29    0    354
1132    */{    4    ,    1    ,    20    ,    42216    },/*    30    29    30    29    29    30    29    29    30    30    30    29    30    384
1133    */{    0    ,    2    ,    7    ,    42192    },/*    30    29    30    29    29    30    29    29    30    30    29    30    0    354
1134    */{    0    ,    1    ,    27    ,    53840    },/*    30    30    29    30    29    29    30    29    29    30    29    30    0    354
1135    */{    2    ,    1    ,    16    ,    55592    },/*    30    30    29    30    30    29    29    30    29    29    30    29    30    384
1136    */{    0    ,    2    ,    4    ,    46400    },/*    30    29    30    30    29    30    29    30    29    30    29    29    0    354
1137    */{    10    ,    1    ,    23    ,    54952    },/*    30    30    29    30    29    30    30    29    30    29    30    29    30    385
1138    */{    0    ,    2    ,    12    ,    11680    },/*    29    29    30    29    30    30    29    30    30    29    30    29    0    354
1139    */{    0    ,    2    ,    1    ,    38352    },/*    30    29    29    30    29    30    29    30    30    30    29    30    0    355
1140    */{    6    ,    1    ,    22    ,    19160    },/*    29    30    29    29    30    29    30    29    30    30    29    30    30    384
1141    */{    0    ,    2    ,    9    ,    18864    },/*    29    30    29    29    30    29    29    30    30    29    30    30    0    354
1142    */{    0    ,    1    ,    29    ,    42160    },/*    30    29    30    29    29    30    29    29    30    29    30    30    0    354
1143    */{    4    ,    1    ,    18    ,    45656    },/*    30    29    30    30    29    29    30    29    29    30    29    30    30    384
1144    */{    0    ,    2    ,    6    ,    27216    },/*    29    30    30    29    30    29    30    29    29    30    29    30    0    354
1145    */{    11    ,    1    ,    25    ,    46376    },/*    30    29    30    30    29    30    29    30    29    29    30    29    30    384
1146    */{    0    ,    2    ,    13    ,    27456    },/*    29    30    30    29    30    29    30    30    29    30    29    29    0    354
1147    */{    0    ,    2    ,    2    ,    43872    },/*    30    29    30    29    30    29    30    30    29    30    30    29    0    355
1148    */{    8    ,    1    ,    23    ,    38320    },/*    30    29    29    30    29    30    29    30    30    29    30    30    29    384
1149    */{    0    ,    2    ,    10    ,    39280    },/*    30    29    29    30    30    29    29    30    29    30    30    30    0    355
1150    */{    0    ,    1    ,    31    ,    18800    },/*    29    30    29    29    30    29    29    30    29    30    30    30    0    354
1151    */{    4    ,    1    ,    20    ,    25784    },/*    29    30    30    29    29    30    29    29    30    29    30    30    30    384
1152    */{    0    ,    2    ,    8    ,    21680    },/*    29    30    29    30    29    30    29    29    30    29    30    30    0    354
1153    */{    12    ,    1    ,    27    ,    27216    },/*    29    30    30    29    30    29    30    29    29    30    29    30    29    383
1154    */{    0    ,    2    ,    14    ,    55888    },/*    30    30    29    30    30    29    30    29    29    30    29    30    0    355
1155    */{    0    ,    2    ,    4    ,    23232    },/*    29    30    29    30    30    29    30    29    30    30    29    29    0    354
1156    */{    10    ,    1    ,    24    ,    43880    },/*    30    29    30    29    30    29    30    30    29    30    30    29    30    385
1157    */{    0    ,    2    ,    12    ,    9952    },/*    29    29    30    29    29    30    30    29    30    30    30    29    0    354
1158    */{    0    ,    2    ,    1    ,    37600    },/*    30    29    29    30    29    29    30    29    30    30    30    29    0    354
1159    */{    6    ,    1    ,    21    ,    51568    },/*    30    30    29    29    30    29    29    30    29    30    30    30    29    384
1160    */{    0    ,    2    ,    9    ,    51552    },/*    30    30    29    29    30    29    29    30    29    30    30    29    0    354
1161    */{    0    ,    1    ,    28    ,    54432    },/*    30    30    29    30    29    30    29    29    30    29    30    29    0    354
1162    */{    2    ,    1    ,    17    ,    55888    },/*    30    30    29    30    30    29    30    29    29    30    29    30    29    384
1163    */{    0    ,    2    ,    5    ,    54608    },/*    30    30    29    30    29    30    29    30    29    30    29    30    0    355
1164    */{    11    ,    1    ,    26    ,    22184    },/*    29    30    29    30    29    30    30    29    30    29    30    29    30    384
1165    */{    0    ,    2    ,    13    ,    21936    },/*    29    30    29    30    29    30    29    30    30    29    30    30    0    355
1166    */{    0    ,    2    ,    3    ,    9680    },/*    29    29    30    29    29    30    29    30    30    30    29    30    0    354
1167    */{    7    ,    1    ,    23    ,    37608    },/*    30    29    29    30    29    29    30    29    30    30    30    29    30    384
1168    */{    0    ,    2    ,    11    ,    37488    },/*    30    29    29    30    29    29    30    29    29    30    30    30    0    354
1169    */{    0    ,    1    ,    30    ,    43344    },/*    30    29    30    29    30    29    29    30    29    30    29    30    0    354
1170    */{    5    ,    1    ,    19    ,    54440    },/*    30    30    29    30    29    30    29    29    30    29    30    29    30    384
1171    */{    0    ,    2    ,    7    ,    46240    },/*    30    29    30    30    29    30    29    29    30    29    30    29    0    354
1172    */{    0    ,    1    ,    27    ,    46416    },/*    30    29    30    30    29    30    29    30    29    30    29    30    0    355
1173    */{    1    ,    1    ,    16    ,    22184    },/*    29    30    29    30    29    30    30    29    30    29    30    29    30    384
1174    */{    0    ,    2    ,    4    ,    19888    },/*    29    30    29    29    30    30    29    30    30    29    30    30    0    355
1175    */{    9    ,    1    ,    25    ,    9648    },/*    29    29    30    29    29    30    29    30    30    29    30    30    29    383
1176    */{    0    ,    2    ,    12    ,    42352    },/*    30    29    30    29    29    30    29    30    29    30    30    30    0    355
1177    */{    0    ,    2    ,    1    ,    21168    },/*    29    30    29    30    29    29    30    29    30    29    30    30    0    354
1178    */{    6    ,    1    ,    21    ,    43352    },/*    30    29    30    29    30    29    29    30    29    30    29    30    30    384
1179    */{    0    ,    2    ,    9    ,    26960    },/*    29    30    30    29    30    29    29    30    29    30    29    30    0    354
1180    */{    0    ,    1    ,    29    ,    27296    },/*    29    30    30    29    30    29    30    29    30    29    30    29    0    354
1181    */{    3    ,    1    ,    17    ,    44368    },/*    30    29    30    29    30    30    29    30    29    30    29    30    29    384
1182    */{    0    ,    2    ,    5    ,    43856    },/*    30    29    30    29    30    29    30    30    29    30    29    30    0    355
1183    */{    11    ,    1    ,    26    ,    19304    },/*    29    30    29    29    30    29    30    30    29    30    30    29    30    384
1184    */{    0    ,    2    ,    14    ,    19168    },/*    29    30    29    29    30    29    30    29    30    30    30    29    0    354
1185    */{    0    ,    2    ,    2    ,    42352    },/*    30    29    30    29    29    30    29    30    29    30    30    30    0    355
1186    */{    7    ,    1    ,    23    ,    21104    },/*    29    30    29    30    29    29    30    29    29    30    30    30    29    383
1187    */{    0    ,    2    ,    10    ,    53392    },/*    30    30    29    30    29    29    30    29    29    30    30    0    0    325
1188    */{    0    ,    1    ,    1    ,    29848    },/*    29    30    30    30    29    30    29    29    30    29    29    30    30    384
1189    */{    5    ,    1    ,    19    ,    27304    },/*    29    30    30    29    30    29    30    29    30    29    30    29    30    384
1190    */{    0    ,    2    ,    7    ,    23200    },/*    29    30    29    30    30    29    30    29    30    29    30    29    0    354
1191    */{    0    ,    1    ,    27    ,    39760    },/*    30    29    29    30    30    29    30    30    29    30    29    30    0    355
1192    */{    2    ,    1    ,    17    ,    19304    },/*    29    30    29    29    30    29    30    30    29    30    30    29    30    384
1193    */{    0    ,    2    ,    4    ,    19168    },/*    29    30    29    29    30    29    30    29    30    30    30    29    0    354
1194    */{    10    ,    1    ,    24    ,    42216    },/*    30    29    30    29    29    30    29    29    30    30    30    29    30    384
1195    */{    0    ,    2    ,    12    ,    42192    },/*    30    29    30    29    29    30    29    29    30    30    29    30    0    354
1196    */{    0    ,    2    ,    1    ,    53856    },/*    30    30    29    30    29    29    30    29    29    30    30    29    0    354
1197    */{    6    ,    1    ,    20    ,    54568    },/*    30    30    29    30    29    30    29    30    29    29    30    29    30    384
1198    */{    0    ,    2    ,    8    ,    46368    },/*    30    29    30    30    29    30    29    30    29    29    30    29    0    354
1199    */{    0    ,    1    ,    28    ,    54944    },/*    30    30    29    30    29    30    30    29    30    29    30    29    0    355
1200    */{    2    ,    1    ,    18    ,    22224    },/*    29    30    29    30    29    30    30    29    30    30    29    30    29    384
1201    */{    0    ,    2    ,    5    ,    38352    },/*    30    29    29    30    29    30    29    30    30    30    29    30    0    355
1202    */{    12    ,    1    ,    26    ,    18904    },/*    29    30    29    29    30    29    29    30    30    30    29    30    30    384
1203    */{    0    ,    2    ,    14    ,    18864    },/*    29    30    29    29    30    29    29    30    30    29    30    30    0    354
1204    */{    0    ,    2    ,    3    ,    42160    },/*    30    29    30    29    29    30    29    29    30    29    30    30    0    354
1205    */{    8    ,    1    ,    22    ,    43608    },/*    30    29    30    29    30    29    30    29    29    30    29    30    30    384
1206    */{    0    ,    2    ,    10    ,    27216    },/*    29    30    30    29    30    29    30    29    29    30    29    30    0    354
1207    */{    0    ,    1    ,    30    ,    46400    },/*    30    29    30    30    29    30    29    30    29    30    29    29    0    354
1208    */{    4    ,    1    ,    19    ,    46496    },/*    30    29    30    30    29    30    29    30    30    29    30    29    29    384
1209    */{    0    ,    2    ,    6    ,    43872    },/*    30    29    30    29    30    29    30    30    29    30    30    29    0    355
1210    */{    0    ,    1    ,    27    ,    38320    },/*    30    29    29    30    29    30    29    30    30    29    30    30    0    355
1211    */{    2    ,    1    ,    17    ,    18872    },/*    29    30    29    29    30    29    29    30    30    29    30    30    30    384
1212    */{    0    ,    2    ,    5    ,    18800    },/*    29    30    29    29    30    29    29    30    29    30    30    30    0    354
1213    */{    9    ,    1    ,    24    ,    25784    },/*    29    30    30    29    29    30    29    29    30    29    30    30    30    384
1214    */{    0    ,    2    ,    12    ,    21680    },/*    29    30    29    30    29    30    29    29    30    29    30    30    0    354
1215    */{    0    ,    2    ,    1    ,    27216    },/*    29    30    30    29    30    29    30    29    29    30    29    30    0    354
1216    */{    7    ,    1    ,    21    ,    27944    },/*    29    30    30    29    30    30    29    30    29    29    30    29    30    384
1217    */{    0    ,    2    ,    8    ,    23232    },/*    29    30    29    30    30    29    30    29    30    30    29    29    0    354
1218    */{    0    ,    1    ,    28    ,    43872    },/*    30    29    30    29    30    29    30    30    29    30    30    29    0    355
1219    */{    3    ,    1    ,    18    ,    37744    },/*    30    29    29    30    29    29    30    30    29    30    30    30    29    384
1220    */{    0    ,    2    ,    6    ,    37600    },/*    30    29    29    30    29    29    30    29    30    30    30    29    0    354
1221    */{    12    ,    1    ,    25    ,    51568    },/*    30    30    29    29    30    29    29    30    29    30    30    30    29    384
1222    */{    0    ,    2    ,    13    ,    51552    },/*    30    30    29    29    30    29    29    30    29    30    30    29    0    354
1223    */{    0    ,    2    ,    2    ,    54432    },/*    30    30    29    30    29    30    29    29    30    29    30    29    0    354
1224    */{    8    ,    1    ,    22    ,    55888    },/*    30    30    29    30    30    29    30    29    29    30    29    30    29    384
1225    */{    0    ,    2    ,    9    ,    46416    },/*    30    29    30    30    29    30    29    30    29    30    29    30    0    355
1226    */{    0    ,    1    ,    30    ,    22176    },/*    29    30    29    30    29    30    30    29    30    29    30    29    0    354
1227    */{    5    ,    1    ,    19    ,    43736    },/*    30    29    30    29    30    29    30    29    30    30    29    30    30    385
1228    */{    0    ,    2    ,    8    ,    9680    },/*    29    29    30    29    29    30    29    30    30    30    29    30    0    354
1229    */{    0    ,    1    ,    27    ,    37600    },/*    30    29    29    30    29    29    30    29    30    30    30    29    0    354
1230    */{    2    ,    1    ,    16    ,    51544    },/*    30    30    29    29    30    29    29    30    29    30    29    30    30    384
1231    */{    0    ,    2    ,    4    ,    43344    },/*    30    29    30    29    30    29    29    30    29    30    29    30    0    354
1232    */{    9    ,    1    ,    24    ,    54440    },/*    30    30    29    30    29    30    29    29    30    29    30    29    30    384
1233    */{    0    ,    2    ,    11    ,    45728    },/*    30    29    30    30    29    29    30    29    30    29    30    29    0    354
1234    */{    0    ,    1    ,    31    ,    46416    },/*    30    29    30    30    29    30    29    30    29    30    29    30    0    355
1235    */{    7    ,    1    ,    21    ,    22184    },/*    29    30    29    30    29    30    30    29    30    29    30    29    30    384
1236    */{    0    ,    2    ,    9    ,    19872    },/*    29    30    29    29    30    30    29    30    30    29    30    29    0    354
1237    */{    0    ,    1    ,    28    ,    42416    },/*    30    29    30    29    29    30    29    30    30    29    30    30    0    355
1238    */{    4    ,    1    ,    18    ,    21176    },/*    29    30    29    30    29    29    30    29    30    29    30    30    30    384
1239    */{    0    ,    2    ,    6    ,    21168    },/*    29    30    29    30    29    29    30    29    30    29    30    30    0    354
1240    */{    12    ,    1    ,    26    ,    43320    },/*    30    29    30    29    30    29    29    30    29    29    30    30    30    384
1241    */{    0    ,    2    ,    13    ,    26928    },/*    29    30    30    29    30    29    29    30    29    29    30    30    0    354
1242    */{    0    ,    2    ,    2    ,    27296    },/*    29    30    30    29    30    29    30    29    30    29    30    29    0    354
1243    */{    8    ,    1    ,    22    ,    44368    },/*    30    29    30    29    30    30    29    30    29    30    29    30    29    384
1244    */{    0    ,    2    ,    10    ,    44624    },/*    30    29    30    29    30    30    30    29    29    30    29    30    0    355
1245    */{    0    ,    1    ,    30    ,    19296    },/*    29    30    29    29    30    29    30    30    29    30    30    29    0    354
1246    */{    4    ,    1    ,    19    ,    42352    },/*    30    29    30    29    29    30    29    30    29    30    30    30    29    384
1247    */{    0    ,    2    ,    7    ,    42352    },/*    30    29    30    29    29    30    29    30    29    30    30    30    0    355
1248    */{    0    ,    1    ,    28    ,    21104    },/*    29    30    29    30    29    29    30    29    29    30    30    30    0    354
1249    */{    2    ,    1    ,    16    ,    26928    },/*    29    30    30    29    30    29    29    30    29    29    30    30    29    383
1250    */{    0    ,    2    ,    3    ,    58672    },/*    30    30    30    29    29    30    29    30    29    29    30    30    0    355
1251    */{    10    ,    1    ,    24    ,    27800    },/*    29    30    30    29    30    30    29    29    30    29    29    30    30    384
1252    */{    0    ,    2    ,    12    ,    23200    },/*    29    30    29    30    30    29    30    29    30    29    30    29    0    354
1253    */{    0    ,    1    ,    31    ,    23248    },/*    29    30    29    30    30    29    30    29    30    30    29    30    0    355
1254    */{    6    ,    1    ,    21    ,    19304    },/*    29    30    29    29    30    29    30    30    29    30    30    29    30    384
1255    */{    0    ,    2    ,    9    ,    19168    },/*    29    30    29    29    30    29    30    29    30    30    30    29    0    354
1256    */{    0    ,    1    ,    29    ,    42208    },/*    30    29    30    29    29    30    29    29    30    30    30    29    0    354
1257    */{    4    ,    1    ,    17    ,    53864    },/*    30    30    29    30    29    29    30    29    29    30    30    29    30    384
1258    */{    0    ,    2    ,    5    ,    53856    },/*    30    30    29    30    29    29    30    29    29    30    30    29    0    354
1259    */{    11    ,    1    ,    25    ,    54568    },/*    30    30    29    30    29    30    29    30    29    29    30    29    30    384
1260    */{    0    ,    2    ,    13    ,    46368    },/*    30    29    30    30    29    30    29    30    29    29    30    29    0    354
1261    */{    0    ,    2    ,    1    ,    46752    },/*    30    29    30    30    29    30    30    29    30    29    30    29    0    355
1262    */{    9    ,    1    ,    22    ,    22224    },/*    29    30    29    30    29    30    30    29    30    30    29    30    29    384
1263    */{    0    ,    2    ,    10    ,    21872    },/*    29    30    29    30    29    30    29    30    29    30    30    30    0    355
1264    */{    0    ,    1    ,    31    ,    18896    },/*    29    30    29    29    30    29    29    30    30    30    29    30    0    354
1265    */{    5    ,    1    ,    19    ,    42200    },/*    30    29    30    29    29    30    29    29    30    30    29    30    30    384
1266    */{    0    ,    2    ,    7    ,    42160    },/*    30    29    30    29    29    30    29    29    30    29    30    30    0    354
1267    */{    0    ,    1    ,    27    ,    43600    },/*    30    29    30    29    30    29    30    29    29    30    29    30    0    354
1268    */{    1    ,    1    ,    16    ,    46376    },/*    30    29    30    30    29    30    29    30    29    29    30    29    30    384
1269    */{    0    ,    2    ,    3    ,    46368    },/*    30    29    30    30    29    30    29    30    29    29    30    29    0    354
1270    */{    11    ,    1    ,    23    ,    46528    },/*    30    29    30    30    29    30    29    30    30    30    29    29    29    384
1271    */{    0    ,    2    ,    11    ,    43872    },/*    30    29    30    29    30    29    30    30    29    30    30    29    0    355
1272    */{    0    ,    2    ,    1    ,    38320    },/*    30    29    29    30    29    30    29    30    30    29    30    30    0    355
1273    */{    6    ,    1    ,    21    ,    18872    },/*    29    30    29    29    30    29    29    30    30    29    30    30    30    384
1274    */{    0    ,    2    ,    9    ,    18800    },/*    29    30    29    29    30    29    29    30    29    30    30    30    0    354
1275    */{    0    ,    1    ,    29    ,    25776    },/*    29    30    30    29    29    30    29    29    30    29    30    30    0    354
1276    */{    3    ,    1    ,    18    ,    27224    },/*    29    30    30    29    30    29    30    29    29    30    29    30    30    384
1277    */{    0    ,    2    ,    5    ,    27216    },/*    29    30    30    29    30    29    30    29    29    30    29    30    0    354
1278    */{    11    ,    1    ,    25    ,    27432    },/*    29    30    30    29    30    29    30    30    29    29    30    29    30    384
1279    */{    0    ,    2    ,    13    ,    23232    },/*    29    30    29    30    30    29    30    29    30    30    29    29    0    354
1280    */{    0    ,    2    ,    2    ,    43872    },/*    30    29    30    29    30    29    30    30    29    30    30    29    0    355
1281    */{    8    ,    1    ,    22    ,    10984    },/*    29    29    30    29    30    29    30    29    30    30    30    29    30    384
1282    */{    0    ,    2    ,    10    ,    18912    },/*    29    30    29    29    30    29    29    30    30    30    30    29    0    354
1283    */{    0    ,    1    ,    30    ,    42192    },/*    30    29    30    29    29    30    29    29    30    30    29    30    0    354
1284    */{    5    ,    1    ,    19    ,    53848    },/*    30    30    29    30    29    29    30    29    29    30    29    30    30    384
1285    */{    0    ,    2    ,    6    ,    45648    },/*    30    29    30    30    29    29    30    29    29    30    29    30    0    354
1286    */{    0    ,    1    ,    26    ,    46368    },/*    30    29    30    30    29    30    29    30    29    29    30    29    0    354
1287    */{    2    ,    1    ,    15    ,    62096    },/*    30    30    30    30    29    29    30    29    30    29    29    30    29    384
1288    */{    0    ,    2    ,    3    ,    46496    },/*    30    29    30    30    29    30    29    30    30    29    30    29    0    355
1289    */{    10    ,    1    ,    23    ,    38352    },/*    30    29    29    30    29    30    29    30    30    30    29    30    29    384
1290    */{    0    ,    2    ,    11    ,    38320    },/*    30    29    29    30    29    30    29    30    30    29    30    30    0    355
1291    */{    0    ,    2    ,    1    ,    18864    },/*    29    30    29    29    30    29    29    30    30    29    30    30    0    354
1292    */{    6    ,    1    ,    21    ,    42168    },/*    30    29    30    29    29    30    29    29    30    29    30    30    30    384
1293    */{    0    ,    2    ,    8    ,    42160    },/*    30    29    30    29    29    30    29    29    30    29    30    30    0    354
1294    */{    0    ,    1    ,    28    ,    43600    },/*    30    29    30    29    30    29    30    29    29    30    29    30    0    354
1295    */{    4    ,    1    ,    17    ,    46376    },/*    30    29    30    30    29    30    29    30    29    29    30    29    30    384
1296    */{    0    ,    2    ,    5    ,    27968    },/*    29    30    30    29    30    30    29    30    29    30    29    29    0    354
1297    */{    12    ,    1    ,    24    ,    44384    },/*    30    29    30    29    30    30    29    30    29    30    30    29    29    384
1298    */{    0    ,    2    ,    12    ,    43872    },/*    30    29    30    29    30    29    30    30    29    30    30    29    0    355
1299    */{    0    ,    2    ,    2    ,    37744    },/*    30    29    29    30    29    29    30    30    29    30    30    30    0    355
1300    */{    8    ,    1    ,    23    ,    2424    },/*    29    29    29    29    30    29    29    30    29    30    30    30    30    383
1301    */{    0    ,    2    ,    10    ,    18800    },/*    29    30    29    29    30    29    29    30    29    30    30    30    0    354
1302    */{    0    ,    1    ,    30    ,    25776    },/*    29    30    30    29    29    30    29    29    30    29    30    30    0    354
1303    */{    5    ,    1    ,    19    ,    27216    },/*    29    30    30    29    30    29    30    29    29    30    29    30    29    383
1304    */{    0    ,    2    ,    6    ,    55888    },/*    30    30    29    30    30    29    30    29    29    30    29    30    0    355
1305    */{    0    ,    1    ,    26    ,    23200    },/*    29    30    29    30    30    29    30    29    30    29    30    29    0    354
1306    */{    1    ,    1    ,    15    ,    43872    },/*    30    29    30    29    30    29    30    30    29    30    30    29    29    384
1307    */{    0    ,    2    ,    3    ,    42720    },/*    30    29    30    29    29    30    30    29    30    30    30    29    0    355
1308    */{    11    ,    1    ,    24    ,    37608    },/*    30    29    29    30    29    29    30    29    30    30    30    29    30    384
1309    */{    0    ,    2    ,    11    ,    37600    },/*    30    29    29    30    29    29    30    29    30    30    30    29    0    354
1310    */{    0    ,    1    ,    31    ,    51552    },/*    30    30    29    29    30    29    29    30    29    30    30    29    0    354
1311    */{    7    ,    1    ,    20    ,    54440    },/*    30    30    29    30    29    30    29    29    30    29    30    29    30    384
1312    */{    0    ,    2    ,    8    ,    54432    },/*    30    30    29    30    29    30    29    29    30    29    30    29    0    354
1313    */{    0    ,    1    ,    27    ,    54608    },/*    30    30    29    30    29    30    29    30    29    30    29    30    0    355
1314    */{    3    ,    1    ,    17    ,    23208    },/*    29    30    29    30    30    29    30    29    30    29    30    29    30    384
1315    */{    0    ,    2    ,    5    ,    22176    },/*    29    30    29    30    29    30    30    29    30    29    30    29    0    354
1316    */{    0    ,    1    ,    25    ,    42704    },/*    30    29    30    29    29    30    30    29    30    30    29    30    0    355
1317    */{    1    ,    1    ,    14    ,    37608    },/*    30    29    29    30    29    29    30    29    30    30    30    29    30    384
1318    */{    0    ,    2    ,    2    ,    37552    },/*    30    29    29    30    29    29    30    29    30    29    30    30    0    354
1319    */{    8    ,    1    ,    22    ,    42328    },/*    30    29    30    29    29    30    29    30    29    30    29    30    30    384
1320    */{    0    ,    2    ,    10    ,    43344    },/*    30    29    30    29    30    29    29    30    29    30    29    30    0    354
1321    */{    0    ,    1    ,    29    ,    45728    },/*    30    29    30    30    29    29    30    29    30    29    30    29    0    354
1322    */{    5    ,    1    ,    18    ,    46416    },/*    30    29    30    30    29    30    29    30    29    30    29    30    29    384
1323    */{    0    ,    2    ,    6    ,    44368    },/*    30    29    30    29    30    30    29    30    29    30    29    30    0    355
1324    */{    0    ,    1    ,    27    ,    19872    },/*    29    30    29    29    30    30    29    30    30    29    30    29    0    354
1325    */{    1    ,    1    ,    15    ,    42448    },/*    30    29    30    29    29    30    29    30    30    30    29    30    29    384
1326    */{    0    ,    2    ,    3    ,    42352    },/*    30    29    30    29    29    30    29    30    29    30    30    30    0    355
1327    */{    9    ,    1    ,    24    ,    21176    },/*    29    30    29    30    29    29    30    29    30    29    30    30    30    384
1328    */{    0    ,    2    ,    12    ,    21104    },/*    29    30    29    30    29    29    30    29    29    30    30    30    0    354
1329    */{    0    ,    1    ,    31    ,    26928    },/*    29    30    30    29    30    29    29    30    29    29    30    30    0    354
1330    */{    7    ,    1    ,    20    ,    27288    },/*    29    30    30    29    30    29    30    29    30    29    29    30    30    384
1331    */{    0    ,    2    ,    8    ,    27296    },/*    29    30    30    29    30    29    30    29    30    29    30    29    0    354
1332    */{    0    ,    1    ,    28    ,    43856    },/*    30    29    30    29    30    29    30    30    29    30    29    30    0    355
1333    */{    3    ,    1    ,    17    ,    19368    },/*    29    30    29    29    30    29    30    30    30    29    30    29    30    384
1334    */{    0    ,    2    ,    5    ,    19296    },/*    29    30    29    29    30    29    30    30    29    30    30    29    0    354
1335    */{    12    ,    1    ,    25    ,    42608    },/*    30    29    30    29    29    30    30    29    29    30    30    30    29    384
1336    */{    0    ,    2    ,    13    ,    41696    },/*    30    29    30    29    29    29    30    29    30    30    30    29    0    354
1337    */{    0    ,    2    ,    1    ,    53600    },/*    30    30    29    30    29    29    29    30    29    30    30    29    0    354
1338    */{    8    ,    1    ,    21    ,    59696    },/*    30    30    30    29    30    29    29    30    29    29    30    30    29    384
1339    */{    0    ,    2    ,    9    ,    54432    },/*    30    30    29    30    29    30    29    29    30    29    30    29    0    354
1340    */{    0    ,    1    ,    29    ,    55968    },/*    30    30    29    30    30    29    30    29    30    29    30    29    0    355
1341    */{    5    ,    1    ,    18    ,    23376    },/*    29    30    29    30    30    29    30    30    29    30    29    30    29    384
1342    */{    0    ,    2    ,    6    ,    22224    },/*    29    30    29    30    29    30    30    29    30    30    29    30    0    355
1343    */{    0    ,    1    ,    27    ,    19168    },/*    29    30    29    29    30    29    30    29    30    30    30    29    0    354
1344    */{    2    ,    1    ,    16    ,    41704    },/*    30    29    30    29    29    29    30    29    30    30    30    29    30    384
1345    */{    0    ,    2    ,    3    ,    41680    },/*    30    29    30    29    29    29    30    29    30    30    29    30    0    354
1346    */{    10    ,    1    ,    23    ,    53592    },/*    30    30    29    30    29    29    29    30    29    30    29    30    30    384
1347    */{    0    ,    2    ,    11    ,    43600    },/*    30    29    30    29    30    29    30    29    29    30    29    30    0    354
1348    */{    0    ,    1    ,    31    ,    46368    },/*    30    29    30    30    29    30    29    30    29    29    30    29    0    354
1349    */{    7    ,    1    ,    19    ,    54944    },/*    30    30    29    30    29    30    30    29    30    29    30    29    29    384
1350    */{    0    ,    2    ,    7    ,    44448    },/*    30    29    30    29    30    30    29    30    30    29    30    29    0    355
1351    */{    0    ,    1    ,    28    ,    21968    },/*    29    30    29    30    29    30    29    30    30    30    29    30    0    355
1352    */{    3    ,    1    ,    18    ,    18904    },/*    29    30    29    29    30    29    29    30    30    30    29    30    30    384
1353    */{    0    ,    2    ,    5    ,    17840    },/*    29    30    29    29    29    30    29    30    30    29    30    30    0    354
1354    */{    0    ,    1    ,    25    ,    41648    },/*    30    29    30    29    29    29    30    29    30    29    30    30    0    354
1355    */{    1    ,    1    ,    14    ,    53592    },/*    30    30    29    30    29    29    29    30    29    30    29    30    30    384
1356    */{    0    ,    2    ,    2    ,    43600    },/*    30    29    30    29    30    29    30    29    29    30    29    30    0    354
1357    */{    9    ,    1    ,    21    ,    46376    },/*    30    29    30    30    29    30    29    30    29    29    30    29    30    384
1358    */{    0    ,    2    ,    9    ,    27424    },/*    29    30    30    29    30    29    30    30    29    29    30    29    0    354
1359    */{    0    ,    1    ,    29    ,    44384    },/*    30    29    30    29    30    30    29    30    29    30    30    29    0    355
1360    */{    5    ,    1    ,    19    ,    21936    },/*    29    30    29    30    29    30    29    30    30    29    30    30    29    384
1361    */{    0    ,    2    ,    6    ,    37744    },/*    30    29    29    30    29    29    30    30    29    30    30    30    0    355
1362    */{    0    ,    1    ,    27    ,    17776    },/*    29    30    29    29    29    30    29    30    29    30    30    30    0    354
1363    */{    3    ,    1    ,    16    ,    41656    },/*    30    29    30    29    29    29    30    29    30    29    30    30    30    384
1364    */{    0    ,    2    ,    4    ,    21168    },/*    29    30    29    30    29    29    30    29    30    29    30    30    0    354
1365    */{    10    ,    1    ,    23    ,    43600    },/*    30    29    30    29    30    29    30    29    29    30    29    30    29    383
1366    */{    0    ,    2    ,    10    ,    55632    },/*    30    30    29    30    30    29    29    30    29    30    29    30    0    355
1367    */{    0    ,    1    ,    31    ,    23200    },/*    29    30    29    30    30    29    30    29    30    29    30    29    0    354
1368    */{    7    ,    1    ,    20    ,    43872    },/*    30    29    30    29    30    29    30    30    29    30    30    29    29    384
1369    */{    0    ,    2    ,    7    ,    42720    },/*    30    29    30    29    29    30    30    29    30    30    30    29    0    355
1370    */{    0    ,    1    ,    28    ,    21216    },/*    29    30    29    30    29    29    30    29    30    30    30    29    0    354
1371    */{    3    ,    1    ,    17    ,    50544    },/*    30    30    29    29    29    30    29    30    29    30    30    30    29    384
1372    */{    0    ,    2    ,    5    ,    42336    },/*    30    29    30    29    29    30    29    30    29    30    30    29    0    354
1373    */{    11    ,    1    ,    24    ,    53928    },/*    30    30    29    30    29    29    30    29    30    29    30    29    30    384
1374    */{    0    ,    2    ,    12    ,    53920    },/*    30    30    29    30    29    29    30    29    30    29    30    29    0    354
1375    */{    0    ,    2    ,    1    ,    54608    },/*    30    30    29    30    29    30    29    30    29    30    29    30    0    355
1376    */{    9    ,    1    ,    22    ,    23208    },/*    29    30    29    30    30    29    30    29    30    29    30    29    30    384
1377    */{    0    ,    2    ,    9    ,    22176    },/*    29    30    29    30    29    30    30    29    30    29    30    29    0    354
1378    */{    0    ,    1    ,    29    ,    42704    },/*    30    29    30    29    29    30    30    29    30    30    29    30    0    355
1379    */{    5    ,    1    ,    19    ,    21224    },/*    29    30    29    30    29    29    30    29    30    30    30    29    30    384
1380    */{    0    ,    2    ,    7    ,    21168    },/*    29    30    29    30    29    29    30    29    30    29    30    30    0    354
1381    */{    0    ,    1    ,    26    ,    43216    },/*    30    29    30    29    30    29    29    29    30    30    29    30    0    354
1382    */{    2    ,    1    ,    15    ,    53928    },/*    30    30    29    30    29    29    30    29    30    29    30    29    30    384
1383    */{    0    ,    2    ,    3    ,    45728    },/*    30    29    30    30    29    29    30    29    30    29    30    29    0    354
1384    */{    10    ,    1    ,    23    ,    46416    },/*    30    29    30    30    29    30    29    30    29    30    29    30    29    384
1385    */{    0    ,    2    ,    10    ,    44368    },/*    30    29    30    29    30    30    29    30    29    30    29    30    0    355
1386    */{    0    ,    1    ,    31    ,    19872    },/*    29    30    29    29    30    30    29    30    30    29    30    29    0    354
1387    */{    6    ,    1    ,    20    ,    42448    },/*    30    29    30    29    29    30    29    30    30    30    29    30    29    384
1388    */{    0    ,    2    ,    8    ,    42352    },/*    30    29    30    29    29    30    29    30    29    30    30    30    0    355
1389    */{    0    ,    1    ,    28    ,    20912    },/*    29    30    29    30    29    29    29    30    30    29    30    30    0    354
1390    */{    4    ,    1    ,    17    ,    43192    },/*    30    29    30    29    30    29    29    29    30    29    30    30    30    384
1391    */{    0    ,    2    ,    5    ,    25904    },/*    29    30    30    29    29    30    29    30    29    29    30    30    0    354
1392    */{    12    ,    1    ,    25    ,    27288    },/*    29    30    30    29    30    29    30    29    30    29    29    30    30    384
1393    */{    0    ,    2    ,    12    ,    23200    },/*    29    30    29    30    30    29    30    29    30    29    30    29    0    354
1394    */{    0    ,    2    ,    1    ,    43856    },/*    30    29    30    29    30    29    30    30    29    30    29    30    0    355
1395    */{    9    ,    1    ,    22    ,    11176    },/*    29    29    30    29    30    29    30    30    30    29    30    29    30    384
1396    */{    0    ,    2    ,    10    ,    11104    },/*    29    29    30    29    30    29    30    30    29    30    30    29    0    354
1397    */{    0    ,    1    ,    29    ,    50032    },/*    30    30    29    29    29    29    30    30    29    30    30    30    0    355
1398    */{    5    ,    1    ,    19    ,    20848    },/*    29    30    29    30    29    29    29    30    29    30    30    30    29    383
1399    */{    0    ,    2    ,    6    ,    51552    },/*    30    30    29    29    30    29    29    30    29    30    30    29    0    354
1400    */{    0    ,    1    ,    26    ,    42160    },/*    30    29    30    29    29    30    29    29    30    29    30    30    0    354
1401    */{    3    ,    1    ,    15    ,    27280    },/*    29    30    30    29    30    29    30    29    30    29    29    30    29    383
1402    */{    0    ,    2    ,    2    ,    55968    },/*    30    30    29    30    30    29    30    29    30    29    30    29    0    355
1403    */{    11    ,    1    ,    23    ,    23376    },/*    29    30    29    30    30    29    30    30    29    30    29    30    29    384
1404    */{    0    ,    2    ,    11    ,    22224    },/*    29    30    29    30    29    30    30    29    30    30    29    30    0    355
1405    */{    0    ,    1    ,    31    ,    10976    },/*    29    29    30    29    30    29    30    29    30    30    30    29    0    354
1406    */{    7    ,    1    ,    20    ,    41704    },/*    30    29    30    29    29    29    30    29    30    30    30    29    30    384
1407    */{    0    ,    2    ,    8    ,    41680    },/*    30    29    30    29    29    29    30    29    30    30    29    30    0    354
1408    */{    0    ,    1    ,    28    ,    53584    },/*    30    30    29    30    29    29    29    30    29    30    29    30    0    354
1409    */{    4    ,    1    ,    16    ,    54440    },/*    30    30    29    30    29    30    29    29    30    29    30    29    30    384
1410    */{    0    ,    2    ,    4    ,    46368    },/*    30    29    30    30    29    30    29    30    29    29    30    29    0    354
1411    */{    12    ,    1    ,    24    ,    46736    },/*    30    29    30    30    29    30    30    29    30    29    29    30    29    384
1412    */{    0    ,    2    ,    12    ,    44448    },/*    30    29    30    29    30    30    29    30    30    29    30    29    0    355
1413    */{    0    ,    2    ,    1    ,    21968    },/*    29    30    29    30    29    30    29    30    30    30    29    30    0    355
1414    */{    9    ,    1    ,    22    ,    9688    },/*    29    29    30    29    29    30    29    30    30    30    29    30    30    384
1415    */{    0    ,    2    ,    10    ,    17840    },/*    29    30    29    29    29    30    29    30    30    29    30    30    0    354
1416    */{    0    ,    1    ,    30    ,    41648    },/*    30    29    30    29    29    29    30    29    30    29    30    30    0    354
1417    */{    5    ,    1    ,    18    ,    43352    },/*    30    29    30    29    30    29    29    30    29    30    29    30    30    384
1418    */{    0    ,    2    ,    6    ,    43344    },/*    30    29    30    29    30    29    29    30    29    30    29    30    0    354
1419    */{    0    ,    1    ,    26    ,    46368    },/*    30    29    30    30    29    30    29    30    29    29    30    29    0    354
1420    */{    1    ,    1    ,    15    ,    46416    },/*    30    29    30    30    29    30    29    30    29    30    29    30    29    384
1421    */{    0    ,    2    ,    2    ,    43872    },/*    30    29    30    29    30    29    30    30    29    30    30    29    0    355
1422    */{    12    ,    1    ,    23    ,    21936    },/*    29    30    29    30    29    30    29    30    30    29    30    30    29    384
1423    */{    0    ,    2    ,    11    ,    19312    },/*    29    30    29    29    30    29    30    30    29    30    30    30    0    355
1424    */{    0    ,    2    ,    1    ,    17776    },/*    29    30    29    29    29    30    29    30    29    30    30    30    0    354
1425    */{    7    ,    1    ,    20    ,    21176    },/*    29    30    29    30    29    29    30    29    30    29    30    30    30    384
1426    */{    0    ,    2    ,    8    ,    21168    },/*    29    30    29    30    29    29    30    29    30    29    30    30    0    354
1427    */{    0    ,    1    ,    28    ,    26960    },/*    29    30    30    29    30    29    29    30    29    30    29    30    0    354
1428    */{    4    ,    1    ,    17    ,    27816    },/*    29    30    30    29    30    30    29    29    30    29    30    29    30    384
1429    */{    0    ,    2    ,    4    ,    23200    },/*    29    30    29    30    30    29    30    29    30    29    30    29    0    354
1430    */{    12    ,    1    ,    24    ,    39760    },/*    30    29    29    30    30    29    30    30    29    30    29    30    29    384
1431    */{    0    ,    2    ,    12    ,    42720    },/*    30    29    30    29    29    30    30    29    30    30    30    29    0    355
1432    */{    0    ,    2    ,    2    ,    19168    },/*    29    30    29    29    30    29    30    29    30    30    30    29    0    354
1433    */{    8    ,    1    ,    21    ,    42352    },/*    30    29    30    29    29    30    29    30    29    30    30    30    29    384
1434    */{    0    ,    2    ,    9    ,    42336    },/*    30    29    30    29    29    30    29    30    29    30    30    29    0    354
1435    */{    0    ,    1    ,    29    ,    53920    },/*    30    30    29    30    29    29    30    29    30    29    30    29    0    354
1436    */{    6    ,    1    ,    18    ,    59728    },/*    30    30    30    29    30    29    29    30    29    30    29    30    29    384
1437    */{    0    ,    2    ,    5    ,    54608    },/*    30    30    29    30    29    30    29    30    29    30    29    30    0    355
1438    */{    0    ,    1    ,    26    ,    22176    },/*    29    30    29    30    29    30    30    29    30    29    30    29    0    354
1439    */{    2    ,    1    ,    15    ,    43728    },/*    30    29    30    29    30    29    30    29    30    30    29    30    29    384
1440    */{    0    ,    2    ,    3    ,    38368    },/*    30    29    29    30    29    30    29    30    30    30    30    29    0    355
1441    */{    11    ,    1    ,    23    ,    19176    },/*    29    30    29    29    30    29    30    29    30    30    30    29    30    384
1442    */{    0    ,    2    ,    11    ,    18864    },/*    29    30    29    29    30    29    29    30    30    29    30    30    0    354
1443    */{    0    ,    1    ,    31    ,    42192    },/*    30    29    30    29    29    30    29    29    30    30    29    30    0    354
1444    */{    7    ,    1    ,    20    ,    53872    },/*    30    30    29    30    29    29    30    29    29    30    30    30    29    384
1445    */{    0    ,    2    ,    7    ,    45728    },/*    30    29    30    30    29    29    30    29    30    29    30    29    0    354
1446    */{    0    ,    1    ,    27    ,    46416    },/*    30    29    30    30    29    30    29    30    29    30    29    30    0    355
1447    */{    4    ,    1    ,    17    ,    22184    },/*    29    30    29    30    29    30    30    29    30    29    30    29    30    384
1448    */{    0    ,    2    ,    5    ,    11680    },/*    29    29    30    29    30    30    29    30    30    29    30    29    0    354
1449    */{    0    ,    1    ,    24    ,    38320    },/*    30    29    29    30    29    30    29    30    30    29    30    30    0    355
1450    */{    1    ,    1    ,    14    ,    19128    },/*    29    30    29    29    30    29    30    29    30    29    30    30    30    384
1451    */{    0    ,    2    ,    2    ,    18864    },/*    29    30    29    29    30    29    29    30    30    29    30    30    0    354
1452    */{    9    ,    1    ,    22    ,    42168    },/*    30    29    30    29    29    30    29    29    30    29    30    30    30    384
1453    */{    0    ,    2    ,    9    ,    25776    },/*    29    30    30    29    29    30    29    29    30    29    30    30    0    354
1454    */{    0    ,    1    ,    29    ,    27280    },/*    29    30    30    29    30    29    30    29    30    29    29    30    0    354
1455    */{    6    ,    1    ,    18    ,    44368    },/*    30    29    30    29    30    30    29    30    29    30    29    30    29    384
1456    */{    0    ,    2    ,    6    ,    27472    },/*    29    30    30    29    30    29    30    30    29    30    29    30    0    355
1457    */{    0    ,    1    ,    26    ,    11104    },/*    29    29    30    29    30    29    30    30    29    30    30    29    0    354
1458    */{    2    ,    1    ,    15    ,    38256    },/*    30    29    29    30    29    30    29    30    29    30    30    30    29    384
1459    */{    0    ,    2    ,    3    ,    37744    },/*    30    29    29    30    29    29    30    30    29    30    30    30    0    355
1460    */{    11    ,    1    ,    24    ,    18800    },/*    29    30    29    29    30    29    29    30    29    30    30    30    29    383
1461    */{    0    ,    2    ,    10    ,    51552    },/*    30    30    29    29    30    29    29    30    29    30    30    29    0    354
1462    */{    0    ,    1    ,    30    ,    58544    },/*    30    30    30    29    29    30    29    29    30    29    30    30    0    355
1463    */{    7    ,    1    ,    20    ,    27280    },/*    29    30    30    29    30    29    30    29    30    29    29    30    29    383
1464    */{    0    ,    2    ,    7    ,    55968    },/*    30    30    29    30    30    29    30    29    30    29    30    29    0    355
1465    */{    0    ,    1    ,    27    ,    23248    },/*    29    30    29    30    30    29    30    29    30    30    29    30    0    355
1466    */{    3    ,    1    ,    17    ,    11112    },/*    29    29    30    29    30    29    30    30    29    30    30    29    30    384
1467    */{    0    ,    2    ,    5    ,    9952    },/*    29    29    30    29    29    30    30    29    30    30    30    29    0    354
1468    */{    0    ,    1    ,    25    ,    37600    },/*    30    29    29    30    29    29    30    29    30    30    30    29    0    354
1469    */{    2    ,    1    ,    13    ,    51560    },/*    30    30    29    29    30    29    29    30    29    30    30    29    30    384
1470    */{    0    ,    2    ,    1    ,    51536    },/*    30    30    29    29    30    29    29    30    29    30    29    30    0    354
1471    */{    9    ,    1    ,    21    ,    54440    },/*    30    30    29    30    29    30    29    29    30    29    30    29    30    384
1472    */{    0    ,    2    ,    9    ,    46240    },/*    30    29    30    30    29    30    29    29    30    29    30    29    0    354
1473    */{    0    ,    1    ,    28    ,    46736    },/*    30    29    30    30    29    30    30    29    30    29    29    30    0    355
1474    */{    6    ,    1    ,    18    ,    22224    },/*    29    30    29    30    29    30    30    29    30    30    29    30    29    384
1475    */{    0    ,    2    ,    6    ,    21936    },/*    29    30    29    30    29    30    29    30    30    29    30    30    0    355
1476    */{    0    ,    1    ,    27    ,    9680    },/*    29    29    30    29    29    30    29    30    30    30    29    30    0    354
1477    */{    2    ,    1    ,    15    ,    37592    },/*    30    29    29    30    29    29    30    29    30    30    29    30    30    384
1478    */{    0    ,    2    ,    3    ,    37552    },/*    30    29    29    30    29    29    30    29    30    29    30    30    0    354
1479    */{    10    ,    1    ,    23    ,    43352    },/*    30    29    30    29    30    29    29    30    29    30    29    30    30    384
1480    */{    0    ,    2    ,    11    ,    26960    },/*    29    30    30    29    30    29    29    30    29    30    29    30    0    354
1481    */{    0    ,    1    ,    30    ,    29856    },/*    29    30    30    30    29    30    29    29    30    29    30    29    0    354
1482    */{    8    ,    1    ,    19    ,    46416    },/*    30    29    30    30    29    30    29    30    29    30    29    30    29    384
1483    */{    0    ,    2    ,    7    ,    43872    },/*    30    29    30    29    30    29    30    30    29    30    30    29    0    355
1484    */{    0    ,    1    ,    28    ,    21424    },/*    29    30    29    30    29    29    30    30    30    29    30    30    0    355
1485    */{    4    ,    1    ,    17    ,    9656    },/*    29    29    30    29    29    30    29    30    30    29    30    30    30    384
1486    */{    0    ,    2    ,    5    ,    9584    },/*    29    29    30    29    29    30    29    30    29    30    30    30    0    354
1487    */{    0    ,    1    ,    25    ,    21168    },/*    29    30    29    30    29    29    30    29    30    29    30    30    0    354
1488    */{    1    ,    1    ,    14    ,    43352    },/*    30    29    30    29    30    29    29    30    29    30    29    30    30    384
1489    */{    0    ,    2    ,    1    ,    26960    },/*    29    30    30    29    30    29    29    30    29    30    29    30    0    354
1490    */{    9    ,    1    ,    21    ,    27304    },/*    29    30    30    29    30    29    30    29    30    29    30    29    30    384
1491    */{    0    ,    2    ,    9    ,    23200    },/*    29    30    29    30    30    29    30    29    30    29    30    29    0    354
1492    */{    0    ,    1    ,    29    ,    43856    },/*    30    29    30    29    30    29    30    30    29    30    29    30    0    355
1493    */{    5    ,    1    ,    18    ,    19304    },/*    29    30    29    29    30    29    30    30    29    30    30    29    30    384
1494    */{    0    ,    2    ,    6    ,    19168    },/*    29    30    29    29    30    29    30    29    30    30    30    29    0    354
1495    */{    0    ,    1    ,    26    ,    42352    },/*    30    29    30    29    29    30    29    30    29    30    30    30    0    355
1496    */{    3    ,    1    ,    16    ,    21104    },/*    29    30    29    30    29    29    30    29    29    30    30    30    29    383
1497    */{    0    ,    2    ,    2    ,    53920    },/*    30    30    29    30    29    29    30    29    30    29    30    29    0    354
1498    */{    11    ,    1    ,    22    ,    55632    },/*    30    30    29    30    30    29    29    30    29    30    29    30    29    384
1499    */{    0    ,    2    ,    10    ,    46416    },/*    30    29    30    30    29    30    29    30    29    30    29    30    0    355
1500    */{    0    ,    1    ,    31    ,    5792    },/*    29    29    29    30    29    30    30    29    30    29    30    29    0    353
1501    */{    7    ,    1    ,    19    ,    38608    },/*    30    29    29    30    29    30    30    29    30    30    29    30    29    384
1502    */{    0    ,    2    ,    7    ,    38352    },/*    30    29    29    30    29    30    29    30    30    30    29    30    0    355
1503    */{    0    ,    1    ,    28    ,    19168    },/*    29    30    29    29    30    29    30    29    30    30    30    29    0    354
1504    */{    4    ,    1    ,    17    ,    42200    },/*    30    29    30    29    29    30    29    29    30    30    29    30    30    384
1505    */{    0    ,    2    ,    4    ,    42192    },/*    30    29    30    29    29    30    29    29    30    30    29    30    0    354
1506    */{    0    ,    1    ,    24    ,    53840    },/*    30    30    29    30    29    29    30    29    29    30    29    30    0    354
1507    */{    1    ,    1    ,    13    ,    54608    },/*    30    30    29    30    29    30    29    30    29    30    29    30    29    384
1508    */{    0    ,    2    ,    1    ,    46416    },/*    30    29    30    30    29    30    29    30    29    30    29    30    0    355
1509    */{    9    ,    1    ,    21    ,    22184    },/*    29    30    29    30    29    30    30    29    30    29    30    29    30    384
1510    */{    0    ,    2    ,    9    ,    11680    },/*    29    29    30    29    30    30    29    30    30    29    30    29    0    354
1511    */{    0    ,    1    ,    29    ,    38320    },/*    30    29    29    30    29    30    29    30    30    29    30    30    0    355
1512    */{    5    ,    1    ,    19    ,    18872    },/*    29    30    29    29    30    29    29    30    30    29    30    30    30    384
1513    */{    0    ,    2    ,    6    ,    18800    },/*    29    30    29    29    30    29    29    30    29    30    30    30    0    354
1514    */{    0    ,    1    ,    26    ,    42160    },/*    30    29    30    29    29    30    29    29    30    29    30    30    0    354
1515    */{    4    ,    1    ,    15    ,    45656    },/*    30    29    30    30    29    29    30    29    29    30    29    30    30    384
1516    */{    0    ,    2    ,    3    ,    27280    },/*    29    30    30    29    30    29    30    29    30    29    29    30    0    354
1517    */{    12    ,    1    ,    22    ,    44368    },/*    30    29    30    29    30    30    29    30    29    30    29    30    29    384
1518    */{    0    ,    2    ,    10    ,    23376    },/*    29    30    29    30    30    29    30    30    29    30    29    30    0    355
1519    */{    0    ,    1    ,    31    ,    11104    },/*    29    29    30    29    30    29    30    30    29    30    30    29    0    354
1520    */{    8    ,    1    ,    20    ,    38256    },/*    30    29    29    30    29    30    29    30    29    30    30    30    29    384
1521    */{    0    ,    2    ,    7    ,    37616    },/*    30    29    29    30    29    29    30    29    30    30    30    30    0    355
1522    */{    0    ,    1    ,    28    ,    18800    },/*    29    30    29    29    30    29    29    30    29    30    30    30    0    354
1523    */{    4    ,    1    ,    17    ,    25776    },/*    29    30    30    29    29    30    29    29    30    29    30    30    29    383
1524    */{    0    ,    2    ,    4    ,    54432    },/*    30    30    29    30    29    30    29    29    30    29    30    29    0    354
1525    */{    12    ,    1    ,    23    ,    59984    },/*    30    30    30    29    30    29    30    29    29    30    29    30    29    384
1526    */{    0    ,    2    ,    11    ,    54928    },/*    30    30    29    30    29    30    30    29    30    29    29    30    0    355
1527    */{    0    ,    2    ,    1    ,    22224    },/*    29    30    29    30    29    30    30    29    30    30    29    30    0    355
1528    */{    10    ,    1    ,    22    ,    11112    },/*    29    29    30    29    30    29    30    30    29    30    30    29    30    384
1529    */{    0    ,    2    ,    9    ,    9952    },/*    29    29    30    29    29    30    30    29    30    30    30    29    0    354
1530    */{    0    ,    1    ,    29    ,    21216    },/*    29    30    29    30    29    29    30    29    30    30    30    29    0    354
1531    */{    6    ,    1    ,    18    ,    51560    },/*    30    30    29    29    30    29    29    30    29    30    30    29    30    384
1532    */{    0    ,    2    ,    6    ,    51536    },/*    30    30    29    29    30    29    29    30    29    30    29    30    0    354
1533    */{    0    ,    1    ,    25    ,    54432    },/*    30    30    29    30    29    30    29    29    30    29    30    29    0    354
1534    */{    2    ,    1    ,    14    ,    55888    },/*    30    30    29    30    30    29    30    29    29    30    29    30    29    384
1535    */{    0    ,    2    ,    2    ,    46480    },/*    30    29    30    30    29    30    29    30    30    29    29    30    0    355
1536    */{    12    ,    1    ,    23    ,    22224    },/*    29    30    29    30    29    30    30    29    30    30    29    30    29    384
1537    */{    0    ,    2    ,    10    ,    21936    },/*    29    30    29    30    29    30    29    30    30    29    30    30    0    355
1538    */{    0    ,    1    ,    31    ,    9680    },/*    29    29    30    29    29    30    29    30    30    30    29    30    0    354
1539    */{    7    ,    1    ,    20    ,    37592    },/*    30    29    29    30    29    29    30    29    30    30    29    30    30    384
1540    */{    0    ,    2    ,    8    ,    37552    },/*    30    29    29    30    29    29    30    29    30    29    30    30    0    354
1541    */{    0    ,    1    ,    27    ,    43344    },/*    30    29    30    29    30    29    29    30    29    30    29    30    0    354
1542    */{    5    ,    1    ,    16    ,    46248    },/*    30    29    30    30    29    30    29    29    30    29    30    29    30    384
1543    */{    0    ,    2    ,    4    ,    27808    },/*    29    30    30    29    30    30    29    29    30    29    30    29    0    354
1544    */{    0    ,    1    ,    24    ,    44368    },/*    30    29    30    29    30    30    29    30    29    30    29    30    0    355
1545    */{    1    ,    1    ,    13    ,    21936    },/*    29    30    29    30    29    30    29    30    30    29    30    30    29    384
1546    */{    0    ,    2    ,    1    ,    19376    },/*    29    30    29    29    30    29    30    30    30    29    30    30    0    355
1547    */{    9    ,    1    ,    22    ,    9656    },/*    29    29    30    29    29    30    29    30    30    29    30    30    30    384
1548    */{    0    ,    2    ,    10    ,    9584    },/*    29    29    30    29    29    30    29    30    29    30    30    30    0    354
1549    */{    0    ,    1    ,    29    ,    21168    },/*    29    30    29    30    29    29    30    29    30    29    30    30    0    354
1550    */{    6    ,    1    ,    18    ,    43344    },/*    30    29    30    29    30    29    29    30    29    30    29    30    29    383
1551    */{    0    ,    2    ,    5    ,    59728    },/*    30    30    30    29    30    29    29    30    29    30    29    30    0    355
1552    */{    0    ,    1    ,    26    ,    27296    },/*    29    30    30    29    30    29    30    29    30    29    30    29    0    354
1553    */{    3    ,    1    ,    14    ,    44368    },/*    30    29    30    29    30    30    29    30    29    30    29    30    29    384
1554    */{    0    ,    2    ,    2    ,    43856    },/*    30    29    30    29    30    29    30    30    29    30    29    30    0    355
1555    */{    11    ,    1    ,    23    ,    19304    },/*    29    30    29    29    30    29    30    30    29    30    30    29    30    384
1556    */{    0    ,    2    ,    11    ,    19168    },/*    29    30    29    29    30    29    30    29    30    30    30    29    0    354
1557    */{    0    ,    1    ,    30    ,    42352    },/*    30    29    30    29    29    30    29    30    29    30    30    30    0    355
1558    */{    7    ,    1    ,    20    ,    21096    },/*    29    30    29    30    29    29    30    29    29    30    30    29    30    383
1559    */{    0    ,    2    ,    7    ,    53856    },/*    30    30    29    30    29    29    30    29    29    30    30    29    0    354
1560    */{    0    ,    1    ,    27    ,    55632    },/*    30    30    29    30    30    29    29    30    29    30    29    30    0    355
1561    */{    5    ,    1    ,    16    ,    23208    },/*    29    30    29    30    30    29    30    29    30    29    30    29    30    384
1562    */{    0    ,    2    ,    4    ,    22176    },/*    29    30    29    30    29    30    30    29    30    29    30    29    0    354
1563    */{    0    ,    1    ,    24    ,    38608    },/*    30    29    29    30    29    30    30    29    30    30    29    30    0    355
1564    */{    2    ,    1    ,    14    ,    19176    },/*    29    30    29    29    30    29    30    29    30    30    30    29    30    384
1565    */{    0    ,    2    ,    1    ,    19168    },/*    29    30    29    29    30    29    30    29    30    30    30    29    0    354
1566    */{    10    ,    1    ,    21    ,    42200    },/*    30    29    30    29    29    30    29    29    30    30    29    30    30    384
1567    */{    0    ,    2    ,    9    ,    42192    },/*    30    29    30    29    29    30    29    29    30    30    29    30    0    354
1568    */{    0    ,    1    ,    29    ,    53840    },/*    30    30    29    30    29    29    30    29    29    30    29    30    0    354
1569    */{    6    ,    1    ,    17    ,    54600    },/*    30    30    29    30    29    30    29    30    29    30    29    29    30    384
1570    */{    0    ,    2    ,    5    ,    46416    },/*    30    29    30    30    29    30    29    30    29    30    29    30    0    355
1571    */{    0    ,    1    ,    26    ,    13728    },/*    29    29    30    30    29    30    29    30    30    29    30    29    0    354
1572    */{    2    ,    1    ,    15    ,    38352    },/*    30    29    29    30    29    30    29    30    30    30    29    30    29    384
1573    */{    0    ,    2    ,    2    ,    38320    },/*    30    29    29    30    29    30    29    30    30    29    30    30    0    355
1574    */{    12    ,    1    ,    23    ,    18872    },/*    29    30    29    29    30    29    29    30    30    29    30    30    30    384
1575    */{    0    ,    2    ,    11    ,    18800    },/*    29    30    29    29    30    29    29    30    29    30    30    30    0    354
1576    */{    0    ,    1    ,    31    ,    42160    },/*    30    29    30    29    29    30    29    29    30    29    30    30    0    354
1577    */{    8    ,    1    ,    19    ,    45656    },/*    30    29    30    30    29    29    30    29    29    30    29    30    30    384
1578    */{    0    ,    2    ,    7    ,    27216    },/*    29    30    30    29    30    29    30    29    29    30    29    30    0    354
1579    */{    0    ,    1    ,    27    ,    27968    },/*    29    30    30    29    30    30    29    30    29    30    29    29    0    354
1580    */{    4    ,    1    ,    16    ,    44456    },/*    30    29    30    29    30    30    29    30    30    29    30    29    30    385
1581    */{    0    ,    2    ,    4    ,    11104    },/*    29    29    30    29    30    29    30    30    29    30    30    29    0    354
1582    */{    0    ,    1    ,    24    ,    39024    },/*    30    29    29    30    29    29    30    30    39    30    30    30    0    365
1583    */{    2    ,    1    ,    24    ,    18808    },/*    29    30    29    29    30    29    29    30    29    30    30    30    30    384
1584    */{    0    ,    2    ,    12    ,    18800    },/*    29    30    29    29    30    29    29    30    29    30    30    30    0    354
1585    */{    9    ,    1    ,    31    ,    25776    },/*    29    30    30    29    29    30    29    29    30    29    30    30    29    383
1586    */{    0    ,    2    ,    18    ,    54440    },/*    30    30    29    30    29    30    29    29    30    29    30    29    30    384
1587    */{    0    ,    3    ,    9    ,    53968    },/*    30    30    29    30    29    30    29    29    30    29    30    0    0    325
1588    */{    6    ,    1    ,    28    ,    27464    },/*    29    30    30    29    30    29    30    30    29    30    29    29    30    384
1589    */{    0    ,    2    ,    15    ,    22224    },/*    29    30    29    30    29    30    30    29    30    30    29    30    0    355
1590    */{    0    ,    2    ,    5    ,    11104    },/*    29    29    30    29    30    29    30    30    29    30    30    29    0    354
1591    */{    3    ,    1    ,    25    ,    37616    },/*    30    29    29    30    29    29    30    29    30    30    30    30    29    384
1592    */{    0    ,    2    ,    13    ,    37600    },/*    30    29    29    30    29    29    30    29    30    30    30    29    0    354
1593    */{    11    ,    2    ,    1    ,    51560    },/*    30    30    29    29    30    29    29    30    29    30    30    29    30    384
1594    */{    0    ,    2    ,    20    ,    43344    },/*    30    29    30    29    30    29    29    30    29    30    29    30    0    354
1595    */{    0    ,    2    ,    9    ,    54432    },/*    30    30    29    30    29    30    29    29    30    29    30    29    0    354
1596    */{    8    ,    1    ,    29    ,    55888    },/*    30    30    29    30    30    29    30    29    29    30    29    30    29    384
1597    */{    0    ,    2    ,    16    ,    46288    },/*    30    29    30    30    29    30    29    29    30    30    29    30    0    355
1598    */{    0    ,    2    ,    6    ,    22192    },/*    29    30    29    30    29    30    30    29    30    29    30    30    0    355
1599    */{    4    ,    1    ,    27    ,    9944    },/*    29    29    30    29    29    30    30    29    30    30    29    30    30    384
1600    */{    0    ,    2    ,    15    ,    9680    },/*    29    29    30    29    29    30    29    30    30    30    29    30    0    354
1601    */{    0    ,    2    ,    3    ,    37584    },/*    30    29    29    30    29    29    30    29    30    30    29    30    0    354
1602    */{    2    ,    1    ,    23    ,    51608    },/*    30    30    29    29    30    29    29    30    30    29    29    30    30    384
1603    */{    0    ,    2    ,    11    ,    43344    },/*    30    29    30    29    30    29    29    30    29    30    29    30    0    354
1604    */{    9    ,    1    ,    31    ,    46248    },/*    30    29    30    30    29    30    29    29    30    29    30    29    30    384
1605    */{    0    ,    2    ,    18    ,    27296    },/*    29    30    30    29    30    29    30    29    30    29    30    29    0    354
1606    */{    0    ,    2    ,    7    ,    44368    },/*    30    29    30    29    30    30    29    30    29    30    29    30    0    355
1607    */{    6    ,    1    ,    28    ,    21928    },/*    29    30    29    30    29    30    29    30    30    29    30    29    30    384
1608    */{    0    ,    2    ,    16    ,    19376    },/*    29    30    29    29    30    29    30    30    30    29    30    30    0    355
1609    */{    0    ,    2    ,    5    ,    9648    },/*    29    29    30    29    29    30    29    30    30    29    30    30    0    354
1610    */{    3    ,    1    ,    25    ,    21176    },/*    29    30    29    30    29    29    30    29    30    29    30    30    30    384
1611    */{    0    ,    2    ,    13    ,    21168    },/*    29    30    29    30    29    29    30    29    30    29    30    30    0    354
1612    */{    11    ,    2    ,    2    ,    43344    },/*    30    29    30    29    30    29    29    30    29    30    29    30    29    383
1613    */{    0    ,    2    ,    19    ,    59728    },/*    30    30    30    29    30    29    29    30    29    30    29    30    0    355
1614    */{    0    ,    2    ,    9    ,    27296    },/*    29    30    30    29    30    29    30    29    30    29    30    29    0    354
1615    */{    8    ,    1    ,    29    ,    44368    },/*    30    29    30    29    30    30    29    30    29    30    29    30    29    384
1616    */{    0    ,    2    ,    17    ,    39760    },/*    30    29    29    30    30    29    30    30    29    30    29    30    0    355
1617    */{    0    ,    2    ,    6    ,    19296    },/*    29    30    29    29    30    29    30    30    29    30    30    29    0    354
1618    */{    4    ,    1    ,    26    ,    42352    },/*    30    29    30    29    29    30    29    30    29    30    30    30    29    384
1619    */{    0    ,    2    ,    14    ,    42224    },/*    30    29    30    29    29    30    29    29    30    30    30    30    0    355
1620    */{    0    ,    2    ,    4    ,    21088    },/*    29    30    29    30    29    29    30    29    29    30    30    29    0    353
1621    */{    2    ,    1    ,    22    ,    59696    },/*    30    30    30    29    30    29    29    30    29    29    30    30    29    384
1622    */{    0    ,    2    ,    10    ,    54608    },/*    30    30    29    30    29    30    29    30    29    30    29    30    0    355
1623    */{    10    ,    1    ,    31    ,    23208    },/*    29    30    29    30    30    29    30    29    30    29    30    29    30    384
1624    */{    0    ,    2    ,    19    ,    22176    },/*    29    30    29    30    29    30    30    29    30    29    30    29    0    354
1625    */{    0    ,    2    ,    7    ,    38608    },/*    30    29    29    30    29    30    30    29    30    30    29    30    0    355
1626    */{    6    ,    1    ,    28    ,    19176    },/*    29    30    29    29    30    29    30    29    30    30    30    29    30    384
1627    */{    0    ,    2    ,    16    ,    18912    },/*    29    30    29    29    30    29    29    30    30    30    30    29    0    354
1628    */{    0    ,    2    ,    5    ,    42192    },/*    30    29    30    29    29    30    29    29    30    30    29    30    0    354
1629    */{    4    ,    1    ,    24    ,    53864    },/*    30    30    29    30    29    29    30    29    29    30    30    29    30    384
1630    */{    0    ,    2    ,    12    ,    53840    },/*    30    30    29    30    29    29    30    29    29    30    29    30    0    354
1631    */{    11    ,    2    ,    1    ,    54568    },/*    30    30    29    30    29    30    29    30    29    29    30    29    30    384
1632    */{    0    ,    2    ,    20    ,    46400    },/*    30    29    30    30    29    30    29    30    29    30    29    29    0    354
1633    */{    0    ,    2    ,    8    ,    46496    },/*    30    29    30    30    29    30    29    30    30    29    30    29    0    355
1634    */{    8    ,    1    ,    29    ,    38352    },/*    30    29    29    30    29    30    29    30    30    30    29    30    29    384
1635    */{    0    ,    2    ,    17    ,    38320    },/*    30    29    29    30    29    30    29    30    30    29    30    30    0    355
1636    */{    0    ,    2    ,    7    ,    18864    },/*    29    30    29    29    30    29    29    30    30    29    30    30    0    354
1637    */{    4    ,    1    ,    26    ,    42168    },/*    30    29    30    29    29    30    29    29    30    29    30    30    30    384
1638    */{    0    ,    2    ,    14    ,    42160    },/*    30    29    30    29    29    30    29    29    30    29    30    30    0    354
1639    */{    0    ,    2    ,    3    ,    43600    },/*    30    29    30    29    30    29    30    29    29    30    29    30    0    354
1640    */{    1    ,    1    ,    23    ,    46376    },/*    30    29    30    30    29    30    29    30    29    29    30    29    30    384
1641    */{    0    ,    2    ,    10    ,    27968    },/*    29    30    30    29    30    30    29    30    29    30    29    29    0    354
1642    */{    11    ,    1    ,    30    ,    44456    },/*    30    29    30    29    30    30    29    30    30    29    30    29    30    385
1643    */{    0    ,    2    ,    19    ,    11104    },/*    29    29    30    29    30    29    30    30    29    30    30    29    0    354
1644    */{    0    ,    2    ,    8    ,    37744    },/*    30    29    29    30    29    29    30    30    29    30    30    30    0    355
1645    */{    6    ,    1    ,    28    ,    18808    },/*    29    30    29    29    30    29    29    30    29    30    30    30    30    384
1646    */{    0    ,    2    ,    16    ,    18800    },/*    29    30    29    29    30    29    29    30    29    30    30    30    0    354
1647    */{    0    ,    2    ,    5    ,    25776    },/*    29    30    30    29    29    30    29    29    30    29    30    30    0    354
1648    */{    3    ,    1    ,    24    ,    27216    },/*    29    30    30    29    30    29    30    29    29    30    29    30    29    383
1649    */{    0    ,    2    ,    11    ,    55888    },/*    30    30    29    30    30    29    30    29    29    30    29    30    0    355
1650    */{    11    ,    2    ,    1    ,    27464    },/*    29    30    30    29    30    29    30    30    29    30    29    29    30    384
1651    */{    0    ,    2    ,    20    ,    22224    },/*    29    30    29    30    29    30    30    29    30    30    29    30    0    355
1652    */{    0    ,    2    ,    10    ,    11168    },/*    29    29    30    29    30    29    30    30    30    29    30    29    0    354
1653    */{    7    ,    1    ,    29    ,    37616    },/*    30    29    29    30    29    29    30    29    30    30    30    30    29    384
1654    */{    0    ,    2    ,    17    ,    37600    },/*    30    29    29    30    29    29    30    29    30    30    30    29    0    354
1655    */{    0    ,    2    ,    6    ,    51552    },/*    30    30    29    29    30    29    29    30    29    30    30    29    0    354
1656    */{    5    ,    1    ,    26    ,    54440    },/*    30    30    29    30    29    30    29    29    30    29    30    29    30    384
1657    */{    0    ,    2    ,    13    ,    54432    },/*    30    30    29    30    29    30    29    29    30    29    30    29    0    354
1658    */{    0    ,    2    ,    2    ,    55888    },/*    30    30    29    30    30    29    30    29    29    30    29    30    0    355
1659    */{    3    ,    1    ,    23    ,    39592    },/*    30    29    29    30    30    29    30    29    30    29    30    29    30    384
1660    */{    0    ,    2    ,    11    ,    22176    },/*    29    30    29    30    29    30    30    29    30    29    30    29    0    354
1661    */{    7    ,    1    ,    30    ,    42704    },/*    30    29    30    29    29    30    30    29    30    30    29    30    29    384
1662    */{    0    ,    2    ,    18    ,    42448    },/*    30    29    30    29    29    30    29    30    30    30    29    30    0    355
1663    */{    0    ,    2    ,    8    ,    37584    },/*    30    29    29    30    29    29    30    29    30    30    29    30    0    354
1664    */{    6    ,    1    ,    28    ,    43352    },/*    30    29    30    29    30    29    29    30    29    30    29    30    30    384
1665    */{    0    ,    2    ,    15    ,    43344    },/*    30    29    30    29    30    29    29    30    29    30    29    30    0    354
1666    */{    0    ,    2    ,    4    ,    46240    },/*    30    29    30    30    29    30    29    29    30    29    30    29    0    354
1667    */{    4    ,    1    ,    24    ,    46416    },/*    30    29    30    30    29    30    29    30    29    30    29    30    29    384
1668    */{    0    ,    2    ,    12    ,    44368    },/*    30    29    30    29    30    30    29    30    29    30    29    30    0    355
1669    */{    0    ,    2    ,    1    ,    21920    },/*    29    30    29    30    29    30    29    30    30    29    30    29    0    354
1670    */{    2    ,    1    ,    21    ,    42448    },/*    30    29    30    29    29    30    29    30    30    30    29    30    29    384
1671    */{    0    ,    2    ,    9    ,    42416    },/*    30    29    30    29    29    30    29    30    30    29    30    30    0    355
1672    */{    7    ,    1    ,    30    ,    21176    },/*    29    30    29    30    29    29    30    29    30    29    30    30    30    384
1673    */{    0    ,    2    ,    17    ,    21168    },/*    29    30    29    30    29    29    30    29    30    29    30    30    0    354
1674    */{    0    ,    2    ,    6    ,    26928    },/*    29    30    30    29    30    29    29    30    29    29    30    30    0    354
1675    */{    5    ,    1    ,    26    ,    29864    },/*    29    30    30    30    29    30    29    29    30    29    30    29    30    384
1676    */{    0    ,    2    ,    14    ,    27296    },/*    29    30    30    29    30    29    30    29    30    29    30    29    0    354
1677    */{    0    ,    2    ,    2    ,    44432    },/*    30    29    30    29    30    30    29    30    30    29    29    30    0    355
1678    */{    3    ,    1    ,    23    ,    19880    },/*    29    30    29    29    30    30    29    30    30    29    30    29    30    384
1679    */{    0    ,    2    ,    11    ,    19296    },/*    29    30    29    29    30    29    30    30    29    30    30    29    0    354
1680    */{    8    ,    1    ,    31    ,    42352    },/*    30    29    30    29    29    30    29    30    29    30    30    30    29    384
1681    */{    0    ,    2    ,    18    ,    42208    },/*    30    29    30    29    29    30    29    29    30    30    30    29    0    354
1682    */{    0    ,    2    ,    7    ,    53856    },/*    30    30    29    30    29    29    30    29    29    30    30    29    0    354
1683    */{    6    ,    1    ,    27    ,    59696    },/*    30    30    30    29    30    29    29    30    29    29    30    30    29    384
1684    */{    0    ,    2    ,    15    ,    54560    },/*    30    30    29    30    29    30    29    30    29    29    30    29    0    354
1685    */{    0    ,    2    ,    3    ,    55968    },/*    30    30    29    30    30    29    30    29    30    29    30    29    0    355
1686    */{    4    ,    1    ,    24    ,    27472    },/*    29    30    30    29    30    29    30    30    29    30    29    30    29    384
1687    */{    0    ,    2    ,    12    ,    22224    },/*    29    30    29    30    29    30    30    29    30    30    29    30    0    355
1688    */{    0    ,    2    ,    2    ,    19168    },/*    29    30    29    29    30    29    30    29    30    30    30    29    0    354
1689    */{    3    ,    1    ,    21    ,    42216    },/*    30    29    30    29    29    30    29    29    30    30    30    29    30    384
1690    */{    0    ,    2    ,    9    ,    42192    },/*    30    29    30    29    29    30    29    29    30    30    29    30    0    354
1691    */{    7    ,    1    ,    29    ,    53848    },/*    30    30    29    30    29    29    30    29    29    30    29    30    30    384
1692    */{    0    ,    2    ,    17    ,    45136    },/*    30    29    29    32    29    29    29    29    29    30    29    30    0    354
1693    */{    0    ,    2    ,    5    ,    54560    },/*    30    30    29    30    29    30    29    30    29    29    30    29    0    354
1694    */{    5    ,    1    ,    25    ,    54944    },/*    30    30    29    30    29    30    30    29    30    29    30    29    29    384
1695    */{    0    ,    2    ,    13    ,    46496    },/*    30    29    30    30    29    30    29    30    30    29    30    29    0    355
1696    */{    0    ,    2    ,    3    ,    21968    },/*    29    30    29    30    29    30    29    30    30    30    29    30    0    355
1697    */{    3    ,    1    ,    23    ,    19160    },/*    29    30    29    29    30    29    30    29    30    30    29    30    30    384
1698    */{    0    ,    2    ,    11    ,    18896    },/*    29    30    29    29    30    29    29    30    30    30    29    30    0    354
1699    */{    7    ,    1    ,    31    ,    42168    },/*    30    29    30    29    29    30    29    29    30    29    30    30    30    384
1700    */{    0    ,    2    ,    19    ,    42160    },/*    30    29    30    29    29    30    29    29    30    29    30    30    0    354
1701    */{    0    ,    2    ,    8    ,    43600    },/*    30    29    30    29    30    29    30    29    29    30    29    30    0    354
1702    */{    6    ,    1    ,    28    ,    46376    },/*    30    29    30    30    29    30    29    30    29    29    30    29    30    384
1703    */{    0    ,    2    ,    16    ,    27936    },/*    29    30    30    29    30    30    29    30    29    29    30    29    0    354
1704    */{    0    ,    2    ,    5    ,    44448    },/*    30    29    30    29    30    30    29    30    30    29    30    29    0    355
1705    */{    4    ,    1    ,    25    ,    21936    },/*    29    30    29    30    29    30    29    30    30    29    30    30    29    384
1706    */{    0    ,    2    ,    13    ,    37744    },/*    30    29    29    30    29    29    30    30    29    30    30    30    0    355
1707    */{    0    ,    2    ,    3    ,    18800    },/*    29    30    29    29    30    29    29    30    29    30    30    30    0    354
1708    */{    3    ,    1    ,    23    ,    25784    },/*    29    30    30    29    29    30    29    29    30    29    30    30    30    384
1709    */{    0    ,    2    ,    10    ,    42192    },/*    30    29    30    29    29    30    29    29    30    30    29    30    0    354
1710    */{    7    ,    1    ,    30    ,    27216    },/*    29    30    30    29    30    29    30    29    29    30    29    30    29    383
1711    */{    0    ,    2    ,    17    ,    55888    },/*    30    30    29    30    30    29    30    29    29    30    29    30    0    355
1712    */{    0    ,    2    ,    7    ,    23200    },/*    29    30    29    30    30    29    30    29    30    29    30    29    0    354
1713    */{    5    ,    1    ,    26    ,    43872    },/*    30    29    30    29    30    29    30    30    29    30    30    29    29    384
1714    */{    0    ,    2    ,    14    ,    43744    },/*    30    29    30    29    30    29    30    29    30    30    30    29    0    355
1715    */{    0    ,    2    ,    4    ,    37600    },/*    30    29    29    30    29    29    30    29    30    30    30    29    0    354
1716    */{    3    ,    1    ,    24    ,    51568    },/*    30    30    29    29    30    29    29    30    29    30    30    30    29    384
1717    */{    0    ,    2    ,    11    ,    51552    },/*    30    30    29    29    30    29    29    30    29    30    30    29    0    354
1718    */{    8    ,    1    ,    31    ,    54440    },/*    30    30    29    30    29    30    29    29    30    29    30    29    30    384
1719    */{    0    ,    2    ,    19    ,    54432    },/*    30    30    29    30    29    30    29    29    30    29    30    29    0    354
1720    */{    0    ,    2    ,    8    ,    54608    },/*    30    30    29    30    29    30    29    30    29    30    29    30    0    355
1721    */{    6    ,    1    ,    28    ,    23208    },/*    29    30    29    30    30    29    30    29    30    29    30    29    30    384
1722    */{    0    ,    2    ,    16    ,    22176    },/*    29    30    29    30    29    30    30    29    30    29    30    29    0    354
1723    */{    0    ,    2    ,    5    ,    42704    },/*    30    29    30    29    29    30    30    29    30    30    29    30    0    355
1724    */{    4    ,    1    ,    26    ,    21224    },/*    29    30    29    30    29    29    30    29    30    30    30    29    30    384
1725    */{    0    ,    2    ,    13    ,    21200    },/*    29    30    29    30    29    29    30    29    30    30    29    30    0    354
1726    */{    0    ,    2    ,    2    ,    43344    },/*    30    29    30    29    30    29    29    30    29    30    29    30    0    354
1727    */{    3    ,    1    ,    22    ,    58536    },/*    30    30    30    29    29    30    29    29    30    29    30    29    30    384
1728    */{    0    ,    2    ,    10    ,    46240    },/*    30    29    30    30    29    30    29    29    30    29    30    29    0    354
1729    */{    7    ,    1    ,    29    ,    46416    },/*    30    29    30    30    29    30    29    30    29    30    29    30    29    384
1730    */{    0    ,    2    ,    17    ,    40272    },/*    30    29    29    30    30    30    29    30    29    30    29    30    0    355
1731    */{    0    ,    2    ,    7    ,    21920    },/*    29    30    29    30    29    30    29    30    30    29    30    29    0    354
1732    */{    5    ,    1    ,    27    ,    42448    },/*    30    29    30    29    29    30    29    30    30    30    29    30    29    384
1733    */{    0    ,    2    ,    14    ,    42416    },/*    30    29    30    29    29    30    29    30    30    29    30    30    0    355
1734    */{    0    ,    2    ,    4    ,    21168    },/*    29    30    29    30    29    29    30    29    30    29    30    30    0    354
1735    */{    4    ,    1    ,    24    ,    43192    },/*    30    29    30    29    30    29    29    29    30    29    30    30    30    384
1736    */{    0    ,    2    ,    12    ,    26928    },/*    29    30    30    29    30    29    29    30    29    29    30    30    0    354
1737    */{    9    ,    1    ,    31    ,    27288    },/*    29    30    30    29    30    29    30    29    30    29    29    30    30    384
1738    */{    0    ,    2    ,    19    ,    27296    },/*    29    30    30    29    30    29    30    29    30    29    30    29    0    354
1739    */{    0    ,    2    ,    8    ,    43856    },/*    30    29    30    29    30    29    30    30    29    30    29    30    0    355
1740    */{    6    ,    1    ,    29    ,    19880    },/*    29    30    29    29    30    30    29    30    30    29    30    29    30    384
1741    */{    0    ,    2    ,    16    ,    19296    },/*    29    30    29    29    30    29    30    30    29    30    30    29    0    354
1742    */{    0    ,    2    ,    5    ,    42352    },/*    30    29    30    29    29    30    29    30    29    30    30    30    0    355
1743    */{    4    ,    1    ,    26    ,    20848    },/*    29    30    29    30    29    29    29    30    29    30    30    30    29    383
1744    */{    0    ,    2    ,    13    ,    53600    },/*    30    30    29    30    29    29    29    30    29    30    30    29    0    354
1745    */{    0    ,    2    ,    1    ,    59696    },/*    30    30    30    29    30    29    29    30    29    29    30    30    0    355
1746    */{    3    ,    1    ,    22    ,    27280    },/*    29    30    30    29    30    29    30    29    30    29    29    30    29    383
1747    */{    0    ,    2    ,    9    ,    55968    },/*    30    30    29    30    30    29    30    29    30    29    30    29    0    355
1748    */{    7    ,    1    ,    30    ,    23376    },/*    29    30    29    30    30    29    30    30    29    30    29    30    29    384
1749    */{    0    ,    2    ,    17    ,    22224    },/*    29    30    29    30    29    30    30    29    30    30    29    30    0    355
1750    */{    0    ,    2    ,    7    ,    19168    },/*    29    30    29    29    30    29    30    29    30    30    30    29    0    354
1751    */{    5    ,    1    ,    27    ,    42200    },/*    30    29    30    29    29    30    29    29    30    30    29    30    30    384
1752    */{    0    ,    2    ,    15    ,    41680    },/*    30    29    30    29    29    29    30    29    30    30    29    30    0    354
1753    */{    0    ,    2    ,    3    ,    53592    },/*    30    30    29    30    29    29    29    30    29    30    29    30    30    384
1754    */{    4    ,    2    ,    22    ,    43600    },/*    30    29    30    29    30    29    30    29    29    30    29    30    0    354
1755    */{    0    ,    2    ,    11    ,    46368    },/*    30    29    30    30    29    30    29    30    29    29    30    29    0    354
1756    */{    9    ,    1    ,    31    ,    54928    },/*    30    30    29    30    29    30    30    29    30    29    29    30    29    384
1757    */{    0    ,    2    ,    18    ,    44448    },/*    30    29    30    29    30    30    29    30    30    29    30    29    0    355
1758    */{    0    ,    2    ,    8    ,    21968    },/*    29    30    29    30    29    30    29    30    30    30    29    30    0    355
1759    */{    6    ,    1    ,    29    ,    10968    },/*    29    29    30    29    30    29    30    29    30    30    29    30    30    384
1760    */{    0    ,    2    ,    17    ,    17840    },/*    29    30    29    29    29    30    29    30    30    29    30    30    0    354
1761    */{    0    ,    2    ,    5    ,    41648    },/*    30    29    30    29    29    29    30    29    30    29    30    30    0    354
1762    */{    5    ,    1    ,    25    ,    45400    },/*    30    29    30    30    29    29    29    30    29    30    29    30    30    384
1763    */{    0    ,    2    ,    13    ,    43344    },/*    30    29    30    29    30    29    29    30    29    30    29    30    0    354
1764    */{    0    ,    2    ,    2    ,    46368    },/*    30    29    30    30    29    30    29    30    29    29    30    29    0    354
1765    */{    2    ,    1    ,    21    ,    46480    },/*    30    29    30    30    29    30    29    30    30    29    29    30    29    384
1766    */{    0    ,    2    ,    9    ,    44384    },/*    30    29    30    29    30    30    29    30    29    30    30    29    0    355
1767    */{    7    ,    1    ,    30    ,    21936    },/*    29    30    29    30    29    30    29    30    30    29    30    30    29    384
1768    */{    0    ,    2    ,    18    ,    21360    },/*    29    30    29    30    29    29    30    30    29    30    30    30    0    355
1769    */{    0    ,    2    ,    7    ,    17776    },/*    29    30    29    29    29    30    29    30    29    30    30    30    0    354
1770    */{    5    ,    1    ,    27    ,    25272    },/*    29    30    30    29    29    29    30    29    30    29    30    30    30    384
1771    */{    0    ,    2    ,    15    ,    21168    },/*    29    30    29    30    29    29    30    29    30    29    30    30    0    354
1772    */{    0    ,    2    ,    4    ,    26960    },/*    29    30    30    29    30    29    29    30    29    30    29    30    0    354
1773    */{    3    ,    1    ,    23    ,    27816    },/*    29    30    30    29    30    30    29    29    30    29    30    29    30    384
1774    */{    0    ,    2    ,    11    ,    23200    },/*    29    30    29    30    30    29    30    29    30    29    30    29    0    354
1775    */{    10    ,    1    ,    31    ,    43856    },/*    30    29    30    29    30    29    30    30    29    30    29    30    29    384
1776    */{    0    ,    2    ,    19    ,    42704    },/*    30    29    30    29    29    30    30    29    30    30    29    30    0    355
1777    */{    0    ,    2    ,    8    ,    19168    },/*    29    30    29    29    30    29    30    29    30    30    30    29    0    354
1778    */{    6    ,    1    ,    28    ,    38256    },/*    30    29    29    30    29    30    29    30    29    30    30    30    29    384
1779    */{    0    ,    2    ,    16    ,    42336    },/*    30    29    30    29    29    30    29    30    29    30    30    29    0    354
1780    */{    0    ,    2    ,    5    ,    53920    },/*    30    30    29    30    29    29    30    29    30    29    30    29    0    354
1781    */{    5    ,    1    ,    24    ,    59728    },/*    30    30    30    29    30    29    29    30    29    30    29    30    29    384
1782    */{    0    ,    2    ,    12    ,    54608    },/*    30    30    29    30    29    30    29    30    29    30    29    30    0    355
1783    */{    0    ,    2    ,    2    ,    23200    },/*    29    30    29    30    30    29    30    29    30    29    30    29    0    354
1784    */{    3    ,    1    ,    22    ,    43856    },/*    30    29    30    29    30    29    30    30    29    30    29    30    29    384
1785    */{    0    ,    2    ,    9    ,    42704    },/*    30    29    30    29    29    30    30    29    30    30    29    30    0    355
1786    */{    7    ,    1    ,    30    ,    19176    },/*    29    30    29    29    30    29    30    29    30    30    30    29    30    384
1787    */{    0    ,    2    ,    18    ,    19120    },/*    29    30    29    29    30    29    30    29    30    29    30    30    0    354
1788    */{    0    ,    2    ,    7    ,    43216    },/*    30    29    30    29    30    29    29    29    30    30    29    30    0    354
1789    */{    5    ,    1    ,    26    ,    53928    },/*    30    30    29    30    29    29    30    29    30    29    30    29    30    384
1790    */{    0    ,    2    ,    14    ,    45728    },/*    30    29    30    30    29    29    30    29    30    29    30    29    0    354
1791    */{    0    ,    2    ,    3    ,    46416    },/*    30    29    30    30    29    30    29    30    29    30    29    30    0    355
1792    */{    4    ,    1    ,    24    ,    22184    },/*    29    30    29    30    29    30    30    29    30    29    30    29    30    384
1793    */{    0    ,    2    ,    11    ,    19872    },/*    29    30    29    29    30    30    29    30    30    29    30    29    0    354
1794    */{    0    ,    1    ,    31    ,    38352    },/*    30    29    29    30    29    30    29    30    30    30    29    30    0    355
1795    */{    2    ,    1    ,    21    ,    19128    },/*    29    30    29    29    30    29    30    29    30    29    30    30    30    384
1796    */{    0    ,    2    ,    9    ,    18864    },/*    29    30    29    29    30    29    29    30    30    29    30    30    0    354
1797    */{    6    ,    1    ,    28    ,    43192    },/*    30    29    30    29    30    29    29    29    30    29    30    30    30    384
1798    */{    0    ,    2    ,    16    ,    25776    },/*    29    30    30    29    29    30    29    29    30    29    30    30    0    354
1799    */{    0    ,    2    ,    5    ,    27280    },/*    29    30    30    29    30    29    30    29    30    29    29    30    0    354
1800    */{    4    ,    1    ,    25    ,    46416    },/*    30    29    30    30    29    30    29    30    29    30    29    30    29    384
1801    */{    0    ,    2    ,    13    ,    27472    },/*    29    30    30    29    30    29    30    30    29    30    29    30    0    355
1802    */{    0    ,    2    ,    3    ,    11168    },/*    29    29    30    29    30    29    30    30    30    29    30    29    0    354
1803    */{    2    ,    1    ,    23    ,    38320    },/*    30    29    29    30    29    30    29    30    30    29    30    30    29    384
1804    */{    0    ,    2    ,    11    ,    37744    },/*    30    29    29    30    29    29    30    30    29    30    30    30    0    355
1805    */{    6    ,    1    ,    31    ,    20848    },/*    29    30    29    30    29    29    29    30    29    30    30    30    29    383
1806    */{    0    ,    2    ,    18    ,    53600    },/*    30    30    29    30    29    29    29    30    29    30    30    29    0    354
1807    */{    0    ,    2    ,    7    ,    58544    },/*    30    30    30    29    29    30    29    29    30    29    30    30    0    355
1808    */{    5    ,    1    ,    28    ,    27280    },/*    29    30    30    29    30    29    30    29    30    29    29    30    29    383
1809    */{    0    ,    2    ,    14    ,    55952    },/*    30    30    29    30    30    29    30    29    30    29    29    30    0    355
1810    */{    0    ,    2    ,    4    ,    23376    },/*    29    30    29    30    30    29    30    30    29    30    29    30    0    355
1811    */{    3    ,    1    ,    25    ,    11112    },/*    29    29    30    29    30    29    30    30    29    30    30    29    30    384
1812    */{    0    ,    2    ,    13    ,    10976    },/*    29    29    30    29    30    29    30    29    30    30    30    29    0    354
1813    */{    0    ,    2    ,    1    ,    41696    },/*    30    29    30    29    29    29    30    29    30    30    30    29    0    354
1814    */{    2    ,    1    ,    21    ,    53608    },/*    30    30    29    30    29    29    29    30    29    30    30    29    30    384
1815    */{    0    ,    2    ,    9    ,    51536    },/*    30    30    29    29    30    29    29    30    29    30    29    30    0    354
1816    */{    6    ,    1    ,    29    ,    54440    },/*    30    30    29    30    29    30    29    29    30    29    30    29    30    384
1817    */{    0    ,    2    ,    16    ,    46368    },/*    30    29    30    30    29    30    29    30    29    29    30    29    0    354
1818    */{    0    ,    2    ,    5    ,    46736    },/*    30    29    30    30    29    30    30    29    30    29    29    30    0    355
1819    */{    4    ,    1    ,    26    ,    22224    },/*    29    30    29    30    29    30    30    29    30    30    29    30    29    384
1820    */{    0    ,    2    ,    14    ,    21968    },/*    29    30    29    30    29    30    29    30    30    30    29    30    0    355
1821    */{    0    ,    2    ,    3    ,    9680    },/*    29    29    30    29    29    30    29    30    30    30    29    30    0    354
1822    */{    3    ,    1    ,    23    ,    41688    },/*    30    29    30    29    29    29    30    29    30    30    29    30    30    384
1823    */{    0    ,    2    ,    11    ,    41648    },/*    30    29    30    29    29    29    30    29    30    29    30    30    0    354
1824    */{    7    ,    1    ,    31    ,    43352    },/*    30    29    30    29    30    29    29    30    29    30    29    30    30    384
1825    */{    0    ,    2    ,    18    ,    43344    },/*    30    29    30    29    30    29    29    30    29    30    29    30    0    354
1826    */{    0    ,    2    ,    7    ,    46240    },/*    30    29    30    30    29    30    29    29    30    29    30    29    0    354
1827    */{    5    ,    1    ,    27    ,    46416    },/*    30    29    30    30    29    30    29    30    29    30    29    30    29    384
1828    */{    0    ,    2    ,    15    ,    44368    },/*    30    29    30    29    30    30    29    30    29    30    29    30    0    355
1829    */{    0    ,    2    ,    4    ,    21936    },/*    29    30    29    30    29    30    29    30    30    29    30    30    0    355
1830    */{    4    ,    1    ,    25    ,    9656    },/*    29    29    30    29    29    30    29    30    30    29    30    30    30    384
1831    */{    0    ,    2    ,    13    ,    9584    },/*    29    29    30    29    29    30    29    30    29    30    30    30    0    354
1832    */{    9    ,    2    ,    2    ,    21176    },/*    29    30    29    30    29    29    30    29    30    29    30    30    30    384
1833    */{    0    ,    2    ,    20    ,    21168    },/*    29    30    29    30    29    29    30    29    30    29    30    30    0    354
1834    */{    0    ,    2    ,    9    ,    26960    },/*    29    30    30    29    30    29    29    30    29    30    29    30    0    354
1835    */{    6    ,    1    ,    29    ,    27816    },/*    29    30    30    29    30    30    29    29    30    29    30    29    30    384
1836    */{    0    ,    2    ,    17    ,    23200    },/*    29    30    29    30    30    29    30    29    30    29    30    29    0    354
1837    */{    0    ,    2    ,    5    ,    43856    },/*    30    29    30    29    30    29    30    30    29    30    29    30    0    355
1838    */{    4    ,    1    ,    26    ,    21352    },/*    29    30    29    30    29    29    30    30    29    30    30    29    30    384
1839    */{    0    ,    2    ,    14    ,    19168    },/*    29    30    29    29    30    29    30    29    30    30    30    29    0    354
1840    */{    0    ,    2    ,    3    ,    42352    },/*    30    29    30    29    29    30    29    30    29    30    30    30    0    355
1841    */{    3    ,    1    ,    23    ,    21168    },/*    29    30    29    30    29    29    30    29    30    29    30    30    29    383
1842    */{    0    ,    2    ,    10    ,    53920    },/*    30    30    29    30    29    29    30    29    30    29    30    29    0    354
1843    */{    7    ,    1    ,    30    ,    59728    },/*    30    30    30    29    30    29    29    30    29    30    29    30    29    384
1844    */{    0    ,    2    ,    18    ,    54608    },/*    30    30    29    30    29    30    29    30    29    30    29    30    0    355
1845    */{    0    ,    2    ,    7    ,    23200    },/*    29    30    29    30    30    29    30    29    30    29    30    29    0    354
1846    */{    5    ,    1    ,    27    ,    43728    },/*    30    29    30    29    30    29    30    29    30    30    29    30    29    384
1847    */{    0    ,    2    ,    15    ,    38352    },/*    30    29    29    30    29    30    29    30    30    30    29    30    0    355
1848    */{    0    ,    2    ,    5    ,    19168    },/*    29    30    29    29    30    29    30    29    30    30    30    29    0    354
1849    */{    4    ,    1    ,    24    ,    42328    },/*    30    29    30    29    29    30    29    30    29    30    29    30    30    384
1850    */{    0    ,    2    ,    12    ,    42192    },/*    30    29    30    29    29    30    29    29    30    30    29    30    0    354
1851    */{    8    ,    2    ,    1    ,    53848    },/*    30    30    29    30    29    29    30    29    29    30    29    30    30    384
1852    */{    0    ,    2    ,    20    ,    45712    },/*    30    29    30    30    29    29    30    29    30    29    29    30    0    354
1853    */{    0    ,    2    ,    8    ,    46416    },/*    30    29    30    30    29    30    29    30    29    30    29    30    0    355
1854    */{    7    ,    1    ,    29    ,    22184    },/*    29    30    29    30    29    30    30    29    30    29    30    29    30    384
1855    */{    0    ,    2    ,    17    ,    11680    },/*    29    29    30    29    30    30    29    30    30    29    30    29    0    354
1856    */{    0    ,    2    ,    6    ,    38352    },/*    30    29    29    30    29    30    29    30    30    30    29    30    0    355
1857    */{    5    ,    1    ,    26    ,    19128    },/*    29    30    29    29    30    29    30    29    30    29    30    30    30    384
1858    */{    0    ,    2    ,    14    ,    18864    },/*    29    30    29    29    30    29    29    30    30    29    30    30    0    354
1859    */{    0    ,    2    ,    3    ,    42160    },/*    30    29    30    29    29    30    29    29    30    29    30    30    0    354
1860    */{    3    ,    1    ,    23    ,    45656    },/*    30    29    30    30    29    29    30    29    29    30    29    30    30    384
1861    */{    0    ,    2    ,    10    ,    27280    },/*    29    30    30    29    30    29    30    29    30    29    29    30    0    354
1862    */{    8    ,    1    ,    30    ,    44360    },/*    30    29    30    29    30    30    29    30    29    30    29    29    30    384
1863    */{    0    ,    2    ,    18    ,    27472    },/*    29    30    30    29    30    29    30    30    29    30    29    30    0    355
1864    */{    0    ,    2    ,    8    ,    11104    },/*    29    29    30    29    30    29    30    30    29    30    30    29    0    354
1865    */{    5    ,    1    ,    27    ,    38320    },/*    30    29    29    30    29    30    29    30    30    29    30    30    29    384
1866    */{    0    ,    2    ,    15    ,    37744    },/*    30    29    29    30    29    29    30    30    29    30    30    30    0    355
1867    */{    0    ,    2    ,    5    ,    18800    },/*    29    30    29    29    30    29    29    30    29    30    30    30    0    354
1868    */{    4    ,    1    ,    25    ,    25776    },/*    29    30    30    29    29    30    29    29    30    29    30    30    29    383
1869    */{    0    ,    2    ,    11    ,    58528    },/*    30    30    30    29    29    30    29    29    30    29    30    29    0    354
1870    */{    10    ,    1    ,    31    ,    59984    },/*    30    30    30    29    30    29    30    29    29    30    29    30    29    384
1871    */{    0    ,    2    ,    19    ,    55952    },/*    30    30    29    30    30    29    30    29    30    29    29    30    0    355
1872    */{    0    ,    2    ,    9    ,    23248    },/*    29    30    29    30    30    29    30    29    30    30    29    30    0    355
1873    */{    6    ,    1    ,    29    ,    11112    },/*    29    29    30    29    30    29    30    30    29    30    30    29    30    384
1874    */{    0    ,    2    ,    17    ,    10976    },/*    29    29    30    29    30    29    30    29    30    30    30    29    0    354
1875    */{    0    ,    2    ,    6    ,    37600    },/*    30    29    29    30    29    29    30    29    30    30    30    29    0    354
1876    */{    5    ,    1    ,    26    ,    51560    },/*    30    30    29    29    30    29    29    30    29    30    30    29    30    384
1877    */{    0    ,    2    ,    13    ,    51536    },/*    30    30    29    29    30    29    29    30    29    30    29    30    0    354
1878    */{    0    ,    2    ,    2    ,    54432    },/*    30    30    29    30    29    30    29    29    30    29    30    29    0    354
1879    */{    3    ,    1    ,    22    ,    55888    },/*    30    30    29    30    30    29    30    29    29    30    29    30    29    384
1880    */{    0    ,    2    ,    10    ,    46736    },/*    30    29    30    30    29    30    30    29    30    29    29    30    0    355
1881    */{    7    ,    1    ,    30    ,    22224    },/*    29    30    29    30    29    30    30    29    30    30    29    30    29    384
1882    */{    0    ,    2    ,    18    ,    21936    },/*    29    30    29    30    29    30    29    30    30    29    30    30    0    355
1883    */{    0    ,    2    ,    8    ,    9680    },/*    29    29    30    29    29    30    29    30    30    30    29    30    0    354
1884    */{    5    ,    1    ,    28    ,    37592    },/*    30    29    29    30    29    29    30    29    30    30    29    30    30    384
1885    */{    0    ,    2    ,    15    ,    37552    },/*    30    29    29    30    29    29    30    29    30    29    30    30    0    354
1886    */{    0    ,    2    ,    4    ,    43344    },/*    30    29    30    29    30    29    29    30    29    30    29    30    0    354
1887    */{    4    ,    1    ,    24    ,    54440    },/*    30    30    29    30    29    30    29    29    30    29    30    29    30    384
1888    */{    0    ,    2    ,    12    ,    46240    },/*    30    29    30    30    29    30    29    29    30    29    30    29    0    354
1889    */{    0    ,    1    ,    31    ,    46416    },/*    30    29    30    30    29    30    29    30    29    30    29    30    0    355
1890    */{    2    ,    1    ,    21    ,    22184    },/*    29    30    29    30    29    30    30    29    30    29    30    29    30    384
1891    */{    0    ,    2    ,    9    ,    21936    },/*    29    30    29    30    29    30    29    30    30    29    30    30    0    355
1892    */{    6    ,    1    ,    30    ,    9656    },/*    29    29    30    29    29    30    29    30    30    29    30    30    30    384
1893    */{    0    ,    2    ,    17    ,    9584    },/*    29    29    30    29    29    30    29    30    29    30    30    30    0    354
1894    */{    0    ,    2    ,    6    ,    21168    },/*    29    30    29    30    29    29    30    29    30    29    30    30    0    354
1895    */{    5    ,    1    ,    26    ,    43344    },/*    30    29    30    29    30    29    29    30    29    30    29    30    29    383
1896    */{    0    ,    2    ,    13    ,    59728    },/*    30    30    30    29    30    29    29    30    29    30    29    30    0    355
1897    */{    0    ,    2    ,    2    ,    27296    },/*    29    30    30    29    30    29    30    29    30    29    30    29    0    354
1898    */{    3    ,    1    ,    22    ,    44368    },/*    30    29    30    29    30    30    29    30    29    30    29    30    29    384
1899    */{    0    ,    2    ,    10    ,    43856    },/*    30    29    30    29    30    29    30    30    29    30    29    30    0    355
1900    */{    8    ,    1    ,    31    ,    19304    },/*    29    30    29    29    30    29    30    30    29    30    30    29    30    384
1901    */{    0    ,    2    ,    19    ,    19168    },/*    29    30    29    29    30    29    30    29    30    30    30    29    0    354
1902    */{    0    ,    2    ,    8    ,    42352    },/*    30    29    30    29    29    30    29    30    29    30    30    30    0    355
1903    */{    5    ,    1    ,    29    ,    21096    },/*    29    30    29    30    29    29    30    29    29    30    30    29    30    383
1904    */{    0    ,    2    ,    16    ,    53856    },/*    30    30    29    30    29    29    30    29    29    30    30    29    0    354
1905    */{    0    ,    2    ,    4    ,    55632    },/*    30    30    29    30    30    29    29    30    29    30    29    30    0    355
1906    */{    4    ,    1    ,    25    ,    27304    },/*    29    30    30    29    30    29    30    29    30    29    30    29    30    384
1907    */{    0    ,    2    ,    13    ,    22176    },/*    29    30    29    30    29    30    30    29    30    29    30    29    0    354
1908    */{    0    ,    2    ,    2    ,    39632    },/*    30    29    29    30    30    29    30    29    30    30    29    30    0    355
1909    */{    2    ,    1    ,    22    ,    19176    },/*    29    30    29    29    30    29    30    29    30    30    30    29    30    384
1910    */{    0    ,    2    ,    10    ,    19168    },/*    29    30    29    29    30    29    30    29    30    30    30    29    0    354
1911    */{    6    ,    1    ,    30    ,    42200    },/*    30    29    30    29    29    30    29    29    30    30    29    30    30    384
1912    */{    0    ,    2    ,    18    ,    42192    },/*    30    29    30    29    29    30    29    29    30    30    29    30    0    354
1913    */{    0    ,    2    ,    6    ,    53840    },/*    30    30    29    30    29    29    30    29    29    30    29    30    0    354
1914    */{    5    ,    1    ,    26    ,    55624    },/*    30    30    29    30    30    29    29    30    29    30    29    29    30    384
1915    */{    0    ,    2    ,    14    ,    46416    },/*    30    29    30    30    29    30    29    30    29    30    29    30    0    355
1916    */{    0    ,    2    ,    4    ,    22176    },/*    29    30    29    30    29    30    30    29    30    29    30    29    0    354
1917    */{    2    ,    1    ,    23    ,    38608    },/*    30    29    29    30    29    30    30    29    30    30    29    30    29    384
1918    */{    0    ,    2    ,    11    ,    38352    },/*    30    29    29    30    29    30    29    30    30    30    29    30    0    355
1919    */{    7    ,    2    ,    1    ,    19160    },/*    29    30    29    29    30    29    30    29    30    30    29    30    30    384
1920    */{    0    ,    2    ,    20    ,    18864    },/*    29    30    29    29    30    29    29    30    30    29    30    30    0    354
1921    */{    0    ,    2    ,    8    ,    42160    },/*    30    29    30    29    29    30    29    29    30    29    30    30    0    354
1922    */{    5    ,    1    ,    28    ,    45656    },/*    30    29    30    30    29    29    30    29    29    30    29    30    30    384
1923    */{    0    ,    2    ,    16    ,    27280    },/*    29    30    30    29    30    29    30    29    30    29    29    30    0    354
1924    */{    0    ,    2    ,    5    ,    44352    },/*    30    29    30    29    30    30    29    30    29    30    29    29    0    354
1925    */{    4    ,    1    ,    24    ,    46504    },/*    30    29    30    30    29    30    29    30    30    29    30    29    30    385
1926    */{    0    ,    2    ,    13    ,    11104    },/*    29    29    30    29    30    29    30    30    29    30    30    29    0    354
1927    */{    0    ,    2    ,    2    ,    38320    },/*    30    29    29    30    29    30    29    30    30    29    30    30    0    355
1928    */{    2    ,    1    ,    23    ,    18872    },/*    29    30    29    29    30    29    29    30    30    29    30    30    30    384
1929    */{    0    ,    2    ,    10    ,    18800    },/*    29    30    29    29    30    29    29    30    29    30    30    30    0    354
1930    */{    6    ,    1    ,    30    ,    25776    },/*    29    30    30    29    29    30    29    29    30    29    30    30    29    383
1931    */{    0    ,    2    ,    17    ,    58528    },/*    30    30    30    29    29    30    29    29    30    29    30    29    0    354
1932    */{    0    ,    2    ,    6    ,    59984    },/*    30    30    30    29    30    29    30    29    29    30    29    30    0    355
1933    */{    5    ,    1    ,    26    ,    27976    },/*    29    30    30    29    30    30    29    30    29    30    29    29    30    384
1934    */{    0    ,    2    ,    14    ,    23376    },/*    29    30    29    30    30    29    30    30    29    30    29    30    0    355
1935    */{    0    ,    2    ,    4    ,    11104    },/*    29    29    30    29    30    29    30    30    29    30    30    29    0    354
1936    */{    3    ,    1    ,    24    ,    38256    },/*    30    29    29    30    29    30    29    30    29    30    30    30    29    384
1937    */{    0    ,    2    ,    11    ,    37600    },/*    30    29    29    30    29    29    30    29    30    30    30    29    0    354
1938    */{    7    ,    1    ,    31    ,    51560    },/*    30    30    29    29    30    29    29    30    29    30    30    29    30    384
1939    */{    0    ,    2    ,    19    ,    51536    },/*    30    30    29    29    30    29    29    30    29    30    29    30    0    354
1940    */{    0    ,    2    ,    8    ,    54432    },/*    30    30    29    30    29    30    29    29    30    29    30    29    0    354
1941    */{    6    ,    1    ,    27    ,    55888    },/*    30    30    29    30    30    29    30    29    29    30    29    30    29    384
1942    */{    0    ,    2    ,    15    ,    46736    },/*    30    29    30    30    29    30    30    29    30    29    29    30    0    355
1943    */{    0    ,    2    ,    5    ,    22224    },/*    29    30    29    30    29    30    30    29    30    30    29    30    0    355
1944    */{    4    ,    1    ,    26    ,    10968    },/*    29    29    30    29    30    29    30    29    30    30    29    30    30    384
1945    */{    0    ,    2    ,    13    ,    9680    },/*    29    29    30    29    29    30    29    30    30    30    29    30    0    354
1946    */{    0    ,    2    ,    2    ,    37584    },/*    30    29    29    30    29    29    30    29    30    30    29    30    0    354
1947    */{    2    ,    1    ,    22    ,    51544    },/*    30    30    29    29    30    29    29    30    29    30    29    30    30    384
1948    */{    0    ,    2    ,    10    ,    43344    },/*    30    29    30    29    30    29    29    30    29    30    29    30    0    354
1949    */{    7    ,    1    ,    29    ,    54440    },/*    30    30    29    30    29    30    29    29    30    29    30    29    30    384
1950    */{    0    ,    2    ,    17    ,    46240    },/*    30    29    30    30    29    30    29    29    30    29    30    29    0    354
1951    */{    0    ,    2    ,    6    ,    46416    },/*    30    29    30    30    29    30    29    30    29    30    29    30    0    355
1952    */{    5    ,    1    ,    27    ,    22184    },/*    29    30    29    30    29    30    30    29    30    29    30    29    30    384
1953    */{    0    ,    2    ,    14    ,    19888    },/*    29    30    29    29    30    30    29    30    30    29    30    30    0    355
1954    */{    0    ,    2    ,    4    ,    9648    },/*    29    29    30    29    29    30    29    30    30    29    30    30    0    354
1955    */{    3    ,    1    ,    24    ,    37560    },/*    30    29    29    30    29    29    30    29    30    29    30    30    30    384
1956    */{    0    ,    2    ,    12    ,    21168    },/*    29    30    29    30    29    29    30    29    30    29    30    30    0    354
1957    */{    8    ,    1    ,    31    ,    43352    },/*    30    29    30    29    30    29    29    30    29    30    29    30    30    384
1958    */{    0    ,    2    ,    19    ,    26960    },/*    29    30    30    29    30    29    29    30    29    30    29    30    0    354
1959    */{    0    ,    2    ,    8    ,    27296    },/*    29    30    30    29    30    29    30    29    30    29    30    29    0    354
1960    */{    6    ,    1    ,    28    ,    44368    },/*    30    29    30    29    30    30    29    30    29    30    29    30    29    384
1961    */{    0    ,    2    ,    15    ,    43856    },/*    30    29    30    29    30    29    30    30    29    30    29    30    0    355
1962    */{    0    ,    2    ,    5    ,    19296    },/*    29    30    29    29    30    29    30    30    29    30    30    29    0    354
1963    */{    4    ,    1    ,    25    ,    42352    },/*    30    29    30    29    29    30    29    30    29    30    30    30    29    384
1964    */{    0    ,    2    ,    13    ,    42352    },/*    30    29    30    29    29    30    29    30    29    30    30    30    0    355
1965    */{    0    ,    2    ,    2    ,    21104    },/*    29    30    29    30    29    29    30    29    29    30    30    30    0    354
1966    */{    3    ,    1    ,    22    ,    26928    },/*    29    30    30    29    30    29    29    30    29    29    30    30    29    383
1967    */{    0    ,    2    ,    9    ,    55632    },/*    30    30    29    30    30    29    29    30    29    30    29    30    0    355
1968    */{    7    ,    1    ,    30    ,    27304    },/*    29    30    30    29    30    29    30    29    30    29    30    29    30    384
1969    */{    0    ,    2    ,    17    ,    22176    },/*    29    30    29    30    29    30    30    29    30    29    30    29    0    354
1970    */{    0    ,    2    ,    6    ,    39632    },/*    30    29    29    30    30    29    30    29    30    30    29    30    0    355
1971    */{    5    ,    1    ,    27    ,    19176    },/*    29    30    29    29    30    29    30    29    30    30    30    29    30    384
1972    */{    0    ,    2    ,    15    ,    19168    },/*    29    30    29    29    30    29    30    29    30    30    30    29    0    354
1973    */{    0    ,    2    ,    3    ,    42208    },/*    30    29    30    29    29    30    29    29    30    30    30    29    0    354
1974    */{    4    ,    1    ,    23    ,    53864    },/*    30    30    29    30    29    29    30    29    29    30    30    29    30    384
1975    */{    0    ,    2    ,    11    ,    53840    },/*    30    30    29    30    29    29    30    29    29    30    29    30    0    354
1976    */{    8    ,    1    ,    31    ,    54600    },/*    30    30    29    30    29    30    29    30    29    30    29    29    30    384
1977    */{    0    ,    2    ,    18    ,    46400    },/*    30    29    30    30    29    30    29    30    29    30    29    29    0    354
1978    */{    0    ,    2    ,    7    ,    54944    },/*    30    30    29    30    29    30    30    29    30    29    30    29    0    355
1979    */{    6    ,    1    ,    28    ,    38608    },/*    30    29    29    30    29    30    30    29    30    30    29    30    29    384
1980    */{    0    ,    2    ,    16    ,    38320    },/*    30    29    29    30    29    30    29    30    30    29    30    30    0    355
1981    */{    0    ,    2    ,    5    ,    18864    },/*    29    30    29    29    30    29    29    30    30    29    30    30    0    354
1982    */{    4    ,    1    ,    25    ,    42200    },/*    30    29    30    29    29    30    29    29    30    30    29    30    30    384
1983    */{    0    ,    2    ,    13    ,    42160    },/*    30    29    30    29    29    30    29    29    30    29    30    30    0    354
1984    */{    10    ,    2    ,    2    ,    45656    },/*    30    29    30    30    29    29    30    29    29    30    29    30    30    384
1985    */{    0    ,    2    ,    20    ,    27216    },/*    29    30    30    29    30    29    30    29    29    30    29    30    0    354
1986    */{    0    ,    2    ,    9    ,    27968    },/*    29    30    30    29    30    30    29    30    29    30    29    29    0    354
1987    */{    6    ,    1    ,    29    ,    46504    },/*    30    29    30    30    29    30    29    30    30    29    30    29    30    385
1988    */{    0    ,    2    ,    18    ,    11104    },/*    29    29    30    29    30    29    30    30    29    30    30    29    0    354
1989    */{    0    ,    2    ,    6    ,    38320    },/*    30    29    29    30    29    30    29    30    30    29    30    30    0    355
1990    */{    5    ,    1    ,    27    ,    18872    },/*    29    30    29    29    30    29    29    30    30    29    30    30    30    384
1991    */{    0    ,    2    ,    15    ,    18800    },/*    29    30    29    29    30    29    29    30    29    30    30    30    0    354
1992    */{    0    ,    2    ,    4    ,    25776    },/*    29    30    30    29    29    30    29    29    30    29    30    30    0    354
1993    */{    3    ,    1    ,    23    ,    27216    },/*    29    30    30    29    30    29    30    29    29    30    29    30    29    383
1994    */{    0    ,    2    ,    10    ,    59984    },/*    30    30    30    29    30    29    30    29    29    30    29    30    0    355
1995    */{    8    ,    1    ,    31    ,    27976    },/*    29    30    30    29    30    30    29    30    29    30    29    29    30    384
1996    */{    0    ,    2    ,    19    ,    23248    },/*    29    30    29    30    30    29    30    29    30    30    29    30    0    355
1997    */{    0    ,    2    ,    8    ,    11104    },/*    29    29    30    29    30    29    30    30    29    30    30    29    0    354
1998    */{    5    ,    1    ,    28    ,    37744    },/*    30    29    29    30    29    29    30    30    29    30    30    30    29    384
1999    */{    0    ,    2    ,    16    ,    37600    },/*    30    29    29    30    29    29    30    29    30    30    30    29    0    354
2000    */{    0    ,    2    ,    5    ,    51552    },/*    30    30    29    29    30    29    29    30    29    30    30    29    0    354
2001    */{    4    ,    1    ,    24    ,    58536    },/*    30    30    30    29    29    30    29    29    30    29    30    29    30    384
2002    */{    0    ,    2    ,    12    ,    54432    },/*    30    30    29    30    29    30    29    29    30    29    30    29    0    354
2003    */{    0    ,    2    ,    1    ,    55888    },/*    30    30    29    30    30    29    30    29    29    30    29    30    0    355
2004    */{    2    ,    1    ,    22    ,    23208    },/*    29    30    29    30    30    29    30    29    30    29    30    29    30    384
2005    */{    0    ,    2    ,    9    ,    22208    },/*    29    30    29    30    29    30    30    29    30    30    29    29    0    354
2006    */{    7    ,    1    ,    29    ,    43736    },/*    30    29    30    29    30    29    30    29    30    30    29    30    30    385
2007    */{    0    ,    2    ,    18    ,    9680    },/*    29    29    30    29    29    30    29    30    30    30    29    30    0    354
2008    */{    0    ,    2    ,    7    ,    37584    },/*    30    29    29    30    29    29    30    29    30    30    29    30    0    354
2009    */{    5    ,    1    ,    26    ,    51544    },/*    30    30    29    29    30    29    29    30    29    30    29    30    30    384
2010    */{    0    ,    2    ,    14    ,    43344    },/*    30    29    30    29    30    29    29    30    29    30    29    30    0    354
2011    */{    0    ,    2    ,    3    ,    46240    },/*    30    29    30    30    29    30    29    29    30    29    30    29    0    354
2012    */{    3    ,    1    ,    23    ,    47696    },/*    30    29    30    30    30    29    30    29    29    30    29    30    29    384
2013    */{    0    ,    2    ,    10    ,    46416    },/*    30    29    30    30    29    30    29    30    29    30    29    30    0    355
2014    */{    9    ,    1    ,    31    ,    21928    },/*    29    30    29    30    29    30    29    30    30    29    30    29    30    384
2015    */{    0    ,    2    ,    19    ,    19360    },/*    29    30    29    29    30    29    30    30    30    29    30    29    0    354
2016    */{    0    ,    2    ,    8    ,    42416    },/*    30    29    30    29    29    30    29    30    30    29    30    30    0    355
2017    */{    5    ,    1    ,    28    ,    21176    },/*    29    30    29    30    29    29    30    29    30    29    30    30    30    384
2018    */{    0    ,    2    ,    16    ,    21168    },/*    29    30    29    30    29    29    30    29    30    29    30    30    0    354
2019    */{    0    ,    2    ,    5    ,    43344    },/*    30    29    30    29    30    29    29    30    29    30    29    30    0    354
2020    */{    4    ,    1    ,    25    ,    46248    },/*    30    29    30    30    29    30    29    29    30    29    30    29    30    384
2021    */{    0    ,    2    ,    12    ,    27296    },/*    29    30    30    29    30    29    30    29    30    29    30    29    0    354
2022    */{    0    ,    2    ,    1    ,    44368    },/*    30    29    30    29    30    30    29    30    29    30    29    30    0    355
2023    */{    2    ,    1    ,    22    ,    21928    },/*    29    30    29    30    29    30    29    30    30    29    30    29    30    384
2024    */{    0    ,    2    ,    10    ,    19296    },/*    29    30    29    29    30    29    30    30    29    30    30    29    0    354
2025    */{    6    ,    1    ,    29    ,    42352    },/*    30    29    30    29    29    30    29    30    29    30    30    30    29    384
2026    */{    0    ,    2    ,    17    ,    42352    },/*    30    29    30    29    29    30    29    30    29    30    30    30    0    355
2027    */{    0    ,    2    ,    7    ,    21104    },/*    29    30    29    30    29    29    30    29    29    30    30    30    0    354
2028    */{    5    ,    1    ,    27    ,    26928    },/*    29    30    30    29    30    29    29    30    29    29    30    30    29    383
2029    */{    0    ,    2    ,    13    ,    55600    },/*    30    30    29    30    30    29    29    30    29    29    30    30    0    355
2030    */{    0    ,    2    ,    3    ,    23200    },/*    29    30    29    30    30    29    30    29    30    29    30    29    0    354
2031    */{    3    ,    1    ,    23    ,    43856    },/*    30    29    30    29    30    29    30    30    29    30    29    30    29    384
2032    */{    0    ,    2    ,    11    ,    38608    },/*    30    29    29    30    29    30    30    29    30    30    29    30    0    355
2033    */{    11    ,    1    ,    31    ,    19176    },/*    29    30    29    29    30    29    30    29    30    30    30    29    30    384
2034    */{    0    ,    2    ,    19    ,    19168    },/*    29    30    29    29    30    29    30    29    30    30    30    29    0    354
2035    */{    0    ,    2    ,    8    ,    42192    },/*    30    29    30    29    29    30    29    29    30    30    29    30    0    354
2036    */{    6    ,    1    ,    28    ,    53864    },/*    30    30    29    30    29    29    30    29    29    30    30    29    30    384
2037    */{    0    ,    2    ,    15    ,    53840    },/*    30    30    29    30    29    29    30    29    29    30    29    30    0    354
2038    */{    0    ,    2    ,    4    ,    54560    },/*    30    30    29    30    29    30    29    30    29    29    30    29    0    354
2039    */{    5    ,    1    ,    24    ,    55968    },/*    30    30    29    30    30    29    30    29    30    29    30    29    29    384
2040    */{    0    ,    2    ,    12    ,    46752    },/*    30    29    30    30    29    30    30    29    30    29    30    29    0    355
2041    */{    0    ,    2    ,    1    ,    38608    },/*    30    29    29    30    29    30    30    29    30    30    29    30    0    355
2042    */{    2    ,    1    ,    22    ,    19160    },/*    29    30    29    29    30    29    30    29    30    30    29    30    30    384
2043    */{    0    ,    2    ,    10    ,    18864    },/*    29    30    29    29    30    29    29    30    30    29    30    30    0    354
2044    */{    7    ,    1    ,    30    ,    42168    },/*    30    29    30    29    29    30    29    29    30    29    30    30    30    384
2045    */{    0    ,    2    ,    17    ,    42160    },/*    30    29    30    29    29    30    29    29    30    29    30    30    0    354
2046    */{    0    ,    2    ,    6    ,    45648    },/*    30    29    30    30    29    29    30    29    29    30    29    30    0    354
2047    */{    5    ,    1    ,    26    ,    46376    },/*    30    29    30    30    29    30    29    30    29    29    30    29    30    384
2048    */{    0    ,    2    ,    14    ,    27968    },/*    29    30    30    29    30    30    29    30    29    30    29    29    0    354
2049    */{    0    ,    2    ,    2    ,    44448    },/*    30    29    30    29    30    30    29    30    30    29    30    29    0    355
2050    */{    3    ,    1    ,    23    ,    38320    },/*    30    29    29    30    29    30    29    30    30    29    30    30    29    384
        */};


        internal override int MinCalendarYear {
            get
            {
                return (MIN_LUNISOLAR_YEAR);
            }
        }

        internal override int MaxCalendarYear {
            get
            {
                return (MAX_LUNISOLAR_YEAR);
            }
        }

        internal override DateTime MinDate {
            get
            {
                return (minDate);
            }
        }

        internal override DateTime MaxDate {
            get
            {
                return (maxDate);
            }
        }

        internal override EraInfo[] CalEraInfo {
            get
            {
                return null;
            }
        }

        internal override int  GetYearInfo(int LunarYear, int Index) {
            if ((LunarYear < MIN_LUNISOLAR_YEAR) || (LunarYear > MAX_LUNISOLAR_YEAR)) {
                throw new ArgumentOutOfRangeException(
                            "year",
                            String.Format(
                                CultureInfo.CurrentCulture,
                                Environment.GetResourceString("ArgumentOutOfRange_Range"),
                                MIN_LUNISOLAR_YEAR,
                                MAX_LUNISOLAR_YEAR ));
            }
            Contract.EndContractBlock();
        return yinfo[LunarYear - MIN_LUNISOLAR_YEAR, Index];
    }

    internal override int GetYear(int year, DateTime time) 
    {
            return year;
    }

    internal override int GetGregorianYear(int year, int era) 
    {
            if (era != CurrentEra && era != GregorianEra) 
                throw new ArgumentOutOfRangeException("era", Environment.GetResourceString("ArgumentOutOfRange_InvalidEraValue"));

            if (year < MIN_LUNISOLAR_YEAR || year > MAX_LUNISOLAR_YEAR) 
            {
                throw new ArgumentOutOfRangeException(
                            "year",
                            String.Format(
                                CultureInfo.CurrentCulture,
                                Environment.GetResourceString("ArgumentOutOfRange_Range"), MIN_LUNISOLAR_YEAR, MAX_LUNISOLAR_YEAR));
            }
            Contract.EndContractBlock();

            return year;
    }

        /*=================================GetDefaultInstance==========================
        **Action: Internal method to provide a default intance of KoreanLunisolarCalendar.  Used by NLS+ implementation
        **       and other calendars.
        **Returns:
        **Arguments:
        **Exceptions:
        ============================================================================*/
        /*
        internal static Calendar GetDefaultInstance()
        {
            if (m_defaultInstance == null) {
                m_defaultInstance = new KoreanLunisolarCalendar();
            }
            return (m_defaultInstance);
        }
        */

        // Construct an instance of KoreanLunisolar calendar.

        public KoreanLunisolarCalendar() {
        }



        public override int GetEra(DateTime time) {
            CheckTicksRange(time.Ticks);
            return (GregorianEra);
        }

        internal override int BaseCalendarID {
            get {
                return (CAL_KOREA);
            }
        }

        internal override int ID {
            get {
                return (CAL_KOREANLUNISOLAR);
            }
        }



        public override int[] Eras {
            get {
                return (new int[] {GregorianEra});
            }
        }
    }
}
