// Copyright (c) .NET Foundation. All rights reserved.
// Licensed under the Apache License, Version 2.0.
// See THIRD-PARTY-NOTICES.TXT in the project root for license information.

using System.Diagnostics;
using System.Numerics;

namespace System.Net.Http.HPack
{
    internal class IntegerDecoder
    {
        private int _i;
        private int _m;

        public int Value { get; private set; }

        /// <summary>
        /// Decodes the first byte of the integer.
        /// </summary>
        /// <param name="b">
        /// The first byte of the variable-length encoded integer.
        /// </param>
        /// <param name="prefixLength">
        /// The number of lower bits in this prefix byte that the
        /// integer has been encoded into. Must be between 1 and 8.
        /// Upper bits must be zero.
        /// </param>
        /// <returns>
        /// If the integer has been fully decoded, true.
        /// Otherwise, false -- <see cref="Decode(byte)"/> must be called on subsequent bytes.
        /// </returns>
        /// <remarks>
        /// The term "prefix" can be confusing. From the HPACK spec:
        /// An integer is represented in two parts: a prefix that fills the current octet and an
        /// optional list of octets that are used if the integer value does not fit within the prefix.
        /// </remarks>
        public bool StartDecode(byte b, int prefixLength)
        {
            Debug.Assert(prefixLength >= 1 && prefixLength <= 8);
            Debug.Assert((b & ~((1 << prefixLength) - 1)) == 0, "bits other than prefix data must be set to 0.");

            if (b < ((1 << prefixLength) - 1))
            {
                Value = b;
                return true;
            }
            else
            {
                _i = b;
                _m = 0;
                return false;
            }
        }

        /// <summary>
        /// Decodes subsequent bytes of an integer.
        /// </summary>
        /// <returns>If the integer has been fully decoded, true. Otherwise, false -- <see cref="Decode(byte)"/> must be called on subsequent bytes.</returns>
        public bool Decode(byte b)
        {
            // Check if shifting b by _m would result in > 31 bits.
            // No masking is required: if the 8th bit is set, it indicates there is a
            // bit set in a future byte, so it is fine to check that here as if it were
            // bit 0 on the next byte.
            // This is a simplified form of:
            //   int additionalBitsRequired = 32 - BitOperations.LeadingZeroCount((uint)b);
            //   if (_m + additionalBitsRequired > 31)
            if (BitOperations.LeadingZeroCount((uint)b) <= _m)
            {
                throw new HPackDecodingException(SR.net_http_hpack_bad_integer);
            }

            _i = _i + ((b & 127) << _m);

            // If the addition overflowed, the result will be negative.
            if (_i < 0)
            {
                throw new HPackDecodingException(SR.net_http_hpack_bad_integer);
            }

            _m = _m + 7;

            if ((b & 128) == 0)
            {
                if (b == 0 && _m / 7 > 1)
                {
                    // Do not accept overlong encodings.
                    throw new HPackDecodingException(SR.net_http_hpack_bad_integer);
                }

                Value = _i;
                return true;
            }

            return false;
        }
    }
}
