using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.TemplateEngine.Abstractions;
using Microsoft.TemplateEngine.Abstractions.TemplateUpdates;
using Microsoft.TemplateEngine.Cli.UnitTests.CliMocks;
using Microsoft.TemplateEngine.Edge.TemplateUpdates;
using Microsoft.TemplateEngine.Mocks;
using Microsoft.TemplateEngine.TestHelper;
using Microsoft.TemplateSearch.Common;
using Xunit;

namespace Microsoft.TemplateEngine.Cli.UnitTests
{
    public class TemplateSearcherTests : TestBase
    {
        [Fact(DisplayName = nameof(TwoSourcesAreBothSearched))]
        public void TwoSourcesAreBothSearched()
        {
            EngineEnvironmentSettings.SettingsLoader.Components.Register(typeof(MockTemplateSearchSource));
            EngineEnvironmentSettings.SettingsLoader.Components.Register(typeof(MockTemplateSearchSource));

            IList<ITemplateSearchSource> searchSources = EngineEnvironmentSettings.SettingsLoader.Components.OfType<ITemplateSearchSource>().ToList();

            Assert.Equal(2, searchSources.Count);
        }

        [Fact(DisplayName = nameof(SourcesCorrectlySearchOnName))]
        public void SourcesCorrectlySearchOnName()
        {
            MockTemplateSearchSource.ClearResultsForAllSources();
            IReadOnlyDictionary<string, Guid> sourceNameToIdMap = MockTemplateSearchSource.SetupMultipleSources(EngineEnvironmentSettings, GetMockNameSearchResults());

            const string templateName = "foo";

            TemplateSearcher searcher = new TemplateSearcher(EngineEnvironmentSettings, "C#", MockTemplateSearchHelpers.DefaultMatchFilter);
            List<IInstallUnitDescriptor> existingInstalls = new List<IInstallUnitDescriptor>();
            SearchResults searchResults = searcher.SearchForTemplatesAsync(existingInstalls, templateName).Result;
            Assert.True(searchResults.AnySources);
            Assert.Equal(1, searchResults.MatchesBySource.Count);
            Assert.Equal("source one", searchResults.MatchesBySource[0].SourceDisplayName);
            Assert.Equal(1, searchResults.MatchesBySource[0].PacksWithMatches.Count);
            Assert.True(searchResults.MatchesBySource[0].PacksWithMatches.ContainsKey(_fooPackInfo));

            Assert.Single(searchResults.MatchesBySource[0].PacksWithMatches[_fooPackInfo].TemplateMatches.Where(x => string.Equals(x.Info.Name, "MockFooTemplateOne")));
            Assert.Single(searchResults.MatchesBySource[0].PacksWithMatches[_fooPackInfo].TemplateMatches.Where(x => string.Equals(x.Info.Name, "MockFooTemplateTwo")));
        }

        [Fact(DisplayName = nameof(SearcherCorrectlyFiltersSpecifiedPack))]
        public void SearcherCorrectlyFiltersSpecifiedPack()
        {
            const string templateName = "foo";

            TemplateSearcher searcher = new TemplateSearcher(EngineEnvironmentSettings, "C#", MockTemplateSearchHelpers.DefaultMatchFilter);

            IReadOnlyList<IInstallUnitDescriptor> packsToIgnore = new List<IInstallUnitDescriptor>()
            {
                _fooPackInstallDescriptor
            };

            SearchResults searchResults = searcher.SearchForTemplatesAsync(packsToIgnore, templateName).Result;
            Assert.Equal(0, searchResults.MatchesBySource.Count);
        }

        private static readonly PackInfo _fooPackInfo = new PackInfo("fooPack", "1.0.0");
        private static readonly PackInfo _barPackInfo = new PackInfo("barPack", "2.0.0");
        private static readonly PackInfo _redPackInfo = new PackInfo("redPack", "1.1");
        private static readonly PackInfo _bluePackInfo = new PackInfo("bluePack", "2.1");
        private static readonly PackInfo _greenPackInfo = new PackInfo("greenPack", "3.0.0");

        private static readonly IInstallUnitDescriptor _fooPackInstallDescriptor = new NupkgInstallUnitDescriptor(Guid.NewGuid(), Guid.NewGuid(), _fooPackInfo.Name, _fooPackInfo.Version, string.Empty);

        private static IReadOnlyDictionary<string, IReadOnlyList<ITemplateNameSearchResult>> GetMockNameSearchResults()
        {
            Dictionary<string, IReadOnlyList<ITemplateNameSearchResult>> dataForSources = new Dictionary<string, IReadOnlyList<ITemplateNameSearchResult>>();

            List<TemplateNameSearchResult> sourceOneResults = new List<TemplateNameSearchResult>();

            ITemplateInfo sourceOneTemplateOne = new MockTemplateInfo()
            {
                Identity = "Mock.Foo.1",
                Description = "Mock Foo template one",
                Name = "MockFooTemplateOne",
                ShortName = "foo1",
            };
            TemplateNameSearchResult sourceOneResultOne = new TemplateNameSearchResult(sourceOneTemplateOne, _fooPackInfo);
            sourceOneResults.Add(sourceOneResultOne);

            ITemplateInfo sourceOneTemplateTwo = new MockTemplateInfo()
            {
                Identity = "Mock.Foo.2",
                Description = "Mock Foo template two",
                Name = "MockFooTemplateTwo",
                ShortName = "foo2"
            };
            TemplateNameSearchResult sourceOneResultTwo = new TemplateNameSearchResult(sourceOneTemplateTwo, _fooPackInfo);
            sourceOneResults.Add(sourceOneResultTwo);

            ITemplateInfo sourceOneTemplateThree = new MockTemplateInfo()
            {
                Identity = "Mock.Bar.1",
                Description = "Mock Bar template one",
                Name = "MockBarTemplateOne",
                ShortName = "bar1"
            };
            TemplateNameSearchResult sourceOneResultThree = new TemplateNameSearchResult(sourceOneTemplateThree, _barPackInfo);
            sourceOneResults.Add(sourceOneResultThree);

            dataForSources["source one"] = sourceOneResults;

            List<TemplateNameSearchResult> sourceTwoResults = new List<TemplateNameSearchResult>();

            ITemplateInfo sourceTwoTemplateOne = new MockTemplateInfo()
            {
                Identity = "Mock.Red.1",
                Description = "Mock red template",
                Name = "MockRedTemplate",
                ShortName = "red",
            };
            TemplateNameSearchResult sourceTwoResultOne = new TemplateNameSearchResult(sourceTwoTemplateOne, _redPackInfo);
            sourceTwoResults.Add(sourceTwoResultOne);

            ITemplateInfo sourceTwoTemplateTwo = new MockTemplateInfo()
            {
                Identity = "Mock.Blue.1",
                Description = "Mock blue template",
                Name = "MockBlueTemplate",
                ShortName = "blue"
            };
            TemplateNameSearchResult sourceTwoResultTwo = new TemplateNameSearchResult(sourceTwoTemplateTwo, _bluePackInfo);
            sourceTwoResults.Add(sourceTwoResultTwo);

            ITemplateInfo sourceTwoTemplateThree = new MockTemplateInfo()
            {
                Identity = "Mock.Green.1",
                Description = "Mock green template",
                Name = "MockGreenTemplate",
                ShortName = "green"
            };
            TemplateNameSearchResult sourceTwoResultThree = new TemplateNameSearchResult(sourceTwoTemplateThree, _greenPackInfo);
            sourceTwoResults.Add(sourceTwoResultThree);

            dataForSources["source two"] = sourceTwoResults;

            return dataForSources;
        }
    }
}
