﻿// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.

using System.Reflection;
using Xunit;
using Xunit.Sdk;

namespace System.Text.RegularExpressions.Tests
{
    public class RegexParserTests
    {
        private static readonly Type s_parseExceptionType;
        private static readonly FieldInfo s_parseErrorField;

        static RegexParserTests()
        {
            s_parseExceptionType = typeof(Regex).Assembly.GetType("System.Text.RegularExpressions.RegexParseException", true);
            s_parseErrorField = s_parseExceptionType.GetField("_error", BindingFlags.NonPublic | BindingFlags.Instance);
        }

        [Theory]
        // Basic
        [InlineData("", RegexOptions.None, null)]
        [InlineData(" ", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("  ", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?#)", RegexOptions.None, null)]
        [InlineData("(?# )", RegexOptions.None, null)]
        [InlineData("(?#", RegexOptions.None, RegexParseError.UnterminatedComment)]
        [InlineData("(?# ", RegexOptions.None, RegexParseError.UnterminatedComment)]
        [InlineData("(?#)(?#)", RegexOptions.None, null)]
        [InlineData("(?#)(?#)", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?#) (?#)", RegexOptions.None, null)]
        [InlineData("(?#) (?#)", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData(@"[a\p{Lu}(?#)b]", RegexOptions.None, null)]
        [InlineData(@"[a\0(?#)b]", RegexOptions.None, null)]
        [InlineData(@"[a\a(?#)b]", RegexOptions.None, null)]
        [InlineData(@"[a\x00(?#)b]", RegexOptions.None, null)]
        [InlineData(@"[a\u0000(?#)b]", RegexOptions.None, null)]
        [InlineData(@"[a\](?#)b]", RegexOptions.None, null)]
        [InlineData("(?", RegexOptions.None, RegexParseError.UnrecognizedGrouping)]
        [InlineData("(?", RegexOptions.IgnorePatternWhitespace, RegexParseError.UnrecognizedGrouping)]
        [InlineData("(? ", RegexOptions.None, RegexParseError.UnrecognizedGrouping)]
        [InlineData("(? ", RegexOptions.IgnorePatternWhitespace, RegexParseError.UnrecognizedGrouping)]
        [InlineData("(?i)", RegexOptions.None, null)]
        [InlineData("(?im)", RegexOptions.None, null)]
        [InlineData("(?im-x)", RegexOptions.None, null)]
        [InlineData("(?im-x+n)", RegexOptions.None, null)]
        [InlineData("(?i) ", RegexOptions.None, null)]
        [InlineData("(?x) ", RegexOptions.None, null)]
        [InlineData(" (?x) ", RegexOptions.None, null)]
        [InlineData(" (?-x) ", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData(" ( (?-x) ) ", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData(" (?-x:) ", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData(" (?-x: ) ", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData(" (?-x: (?+x: ) ) ", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?-x", RegexOptions.IgnorePatternWhitespace, RegexParseError.UnrecognizedGrouping)]
        [InlineData("(?-x ", RegexOptions.IgnorePatternWhitespace, RegexParseError.UnrecognizedGrouping)]
        [InlineData("(?-x :", RegexOptions.IgnorePatternWhitespace, RegexParseError.UnrecognizedGrouping)]
        [InlineData("(?-x )", RegexOptions.IgnorePatternWhitespace, RegexParseError.UnrecognizedGrouping)]
        [InlineData("(?-x :)", RegexOptions.IgnorePatternWhitespace, RegexParseError.UnrecognizedGrouping)]
        [InlineData(")", RegexOptions.None, RegexParseError.TooManyParentheses)]
        [InlineData("a", RegexOptions.None, null)]
        [InlineData("ab", RegexOptions.None, null)]
        [InlineData("a*", RegexOptions.None, null)]
        [InlineData("a*?", RegexOptions.None, null)]
        [InlineData("a+", RegexOptions.None, null)]
        [InlineData("a+?", RegexOptions.None, null)]
        [InlineData("a?", RegexOptions.None, null)]
        [InlineData("a??", RegexOptions.None, null)]
        [InlineData("()", RegexOptions.None, null)]
        [InlineData("(a)", RegexOptions.None, null)]
        [InlineData("(", RegexOptions.None, RegexParseError.NotEnoughParentheses)]
        [InlineData("(a", RegexOptions.None, RegexParseError.NotEnoughParentheses)]
        [InlineData("|", RegexOptions.None, null)]
        [InlineData(" |", RegexOptions.None, null)]
        [InlineData("| ", RegexOptions.None, null)]
        [InlineData(" | ", RegexOptions.None, null)]
        [InlineData("|", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData(" |", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("| ", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData(" | ", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("||", RegexOptions.None, null)]
        [InlineData("(|)", RegexOptions.None, null)]
        [InlineData("a{0}", RegexOptions.None, null)]
        [InlineData("a{0,}", RegexOptions.None, null)]
        [InlineData("a{0,1}", RegexOptions.None, null)]
        [InlineData("a{1,0}", RegexOptions.None, RegexParseError.IllegalRange)]
        [InlineData("a{0}?", RegexOptions.None, null)]
        [InlineData("a{0,}?", RegexOptions.None, null)]
        [InlineData("a{0,1}?", RegexOptions.None, null)]
        [InlineData("a{", RegexOptions.None, null)]
        [InlineData("a{0", RegexOptions.None, null)]
        [InlineData("a{0,", RegexOptions.None, null)]
        [InlineData("a{0,1", RegexOptions.None, null)]
        [InlineData("a{0 }", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("a{0, }", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("a{0 ,}", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("a{0 ,1}", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("a{0, 1}", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("a{0,1 }", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("a* ?", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("a* ?", RegexOptions.None, null)]
        [InlineData("*", RegexOptions.None, RegexParseError.QuantifyAfterNothing)]
        [InlineData("(*)", RegexOptions.None, RegexParseError.QuantifyAfterNothing)]
        [InlineData("a**", RegexOptions.None, RegexParseError.NestedQuantify)]
        [InlineData("+", RegexOptions.None, RegexParseError.QuantifyAfterNothing)]
        [InlineData("(+)", RegexOptions.None, RegexParseError.QuantifyAfterNothing)]
        [InlineData("a*+", RegexOptions.None, RegexParseError.NestedQuantify)]
        [InlineData("?", RegexOptions.None, RegexParseError.QuantifyAfterNothing)]
        [InlineData("(?)", RegexOptions.None, RegexParseError.QuantifyAfterNothing)]
        [InlineData("a*??", RegexOptions.None, RegexParseError.NestedQuantify)]
        [InlineData("{0}", RegexOptions.None, RegexParseError.QuantifyAfterNothing)]
        [InlineData("({0})", RegexOptions.None, RegexParseError.QuantifyAfterNothing)]
        [InlineData("a*{0}", RegexOptions.None, RegexParseError.NestedQuantify)]
        [InlineData("{0", RegexOptions.None, null)]
        [InlineData("({0)", RegexOptions.None, null)]
        [InlineData("a*{0", RegexOptions.None, null)]
        [InlineData(@"\w", RegexOptions.None, null)]
        [InlineData(@"\b\B\A\G\Z\z\w\W\s\W\s\S\d\D", RegexOptions.None, null)]
        [InlineData(@"\c", RegexOptions.None, RegexParseError.MissingControl)]
        [InlineData(@"\c<", RegexOptions.None, RegexParseError.UnrecognizedControl)]
        [InlineData(@"\ca", RegexOptions.None, null)]
        [InlineData(@"\cA", RegexOptions.None, null)]
        [InlineData(@"\c A", RegexOptions.None, RegexParseError.UnrecognizedControl)]
        [InlineData(@"\c(a)", RegexOptions.None, RegexParseError.UnrecognizedControl)]
        [InlineData(@"\c>", RegexOptions.None, RegexParseError.UnrecognizedControl)]
        [InlineData(@"\c?", RegexOptions.None, RegexParseError.UnrecognizedControl)]
        [InlineData(@"\c@", RegexOptions.None, null)]
        [InlineData(@"\c^", RegexOptions.None, null)]
        [InlineData(@"\c_", RegexOptions.None, null)]
        [InlineData(@"\c`", RegexOptions.None, RegexParseError.UnrecognizedControl)]
        [InlineData(@"\c{", RegexOptions.None, RegexParseError.UnrecognizedControl)]
        [InlineData(@"\cz", RegexOptions.None, null)]
        [InlineData(@"\cZ", RegexOptions.None, null)]
        [InlineData(@"\m", RegexOptions.None, RegexParseError.UnrecognizedEscape)]
        [InlineData(@"\x", RegexOptions.None, RegexParseError.TooFewHex)]
        [InlineData(@"\x ", RegexOptions.None, RegexParseError.TooFewHex)]
        [InlineData(@"\x0", RegexOptions.None, RegexParseError.TooFewHex)]
        [InlineData(@"\x0 ", RegexOptions.None, RegexParseError.TooFewHex)]
        [InlineData(@"\x00", RegexOptions.None, null)]
        [InlineData(@"\x00 ", RegexOptions.None, null)]
        [InlineData(@"\x000", RegexOptions.None, null)]
        [InlineData(@"\xff", RegexOptions.None, null)]
        [InlineData(@"\xFF", RegexOptions.None, null)]
        [InlineData(@"\xfF", RegexOptions.None, null)]
        [InlineData(@"\xfff", RegexOptions.None, null)]
        [InlineData(@"\xgg", RegexOptions.None, RegexParseError.TooFewHex)]
        [InlineData(@"\m ", RegexOptions.None, RegexParseError.UnrecognizedEscape)]
        [InlineData(@"\u", RegexOptions.None, RegexParseError.TooFewHex)]
        [InlineData(@"\u0", RegexOptions.None, RegexParseError.TooFewHex)]
        [InlineData(@"\u00", RegexOptions.None, RegexParseError.TooFewHex)]
        [InlineData(@"\u000", RegexOptions.None, RegexParseError.TooFewHex)]
        [InlineData(@"\u0000", RegexOptions.None, null)]
        [InlineData(@"\u0000 ", RegexOptions.None, null)]
        [InlineData(@"\u ", RegexOptions.None, RegexParseError.TooFewHex)]
        [InlineData(@"\u0 ", RegexOptions.None, RegexParseError.TooFewHex)]
        [InlineData(@"\ugggg", RegexOptions.None, RegexParseError.TooFewHex)]
        [InlineData(@"\0", RegexOptions.None, null)]
        [InlineData(@"\0 ", RegexOptions.None, null)]
        [InlineData(@"\00", RegexOptions.None, null)]
        [InlineData(@"\00 ", RegexOptions.None, null)]
        [InlineData(@"\000", RegexOptions.None, null)]
        [InlineData(@"\000 ", RegexOptions.None, null)]
        [InlineData(@"\0000", RegexOptions.None, null)]
        [InlineData(@"\0000 ", RegexOptions.None, null)]
        [InlineData(@"\7", RegexOptions.None, RegexParseError.UndefinedBackref)]
        [InlineData(@"\78", RegexOptions.None, null)]
        [InlineData(@"\8", RegexOptions.None, RegexParseError.UndefinedBackref)]
        [InlineData(@"\40", RegexOptions.ECMAScript, null)]
        [InlineData(@"\401", RegexOptions.ECMAScript, null)]
        [InlineData(@"\37", RegexOptions.ECMAScript, null)]
        [InlineData(@"\371", RegexOptions.ECMAScript, null)]
        [InlineData(@"\0000", RegexOptions.ECMAScript, null)]
        [InlineData(@"\k", RegexOptions.None, RegexParseError.MalformedNameRef)]
        [InlineData(@"\k ", RegexOptions.None, RegexParseError.MalformedNameRef)]
        [InlineData(@"\k<", RegexOptions.None, RegexParseError.MalformedNameRef)]
        [InlineData(@"\k< ", RegexOptions.None, RegexParseError.UnrecognizedEscape)]
        [InlineData(@"\k<0", RegexOptions.None, RegexParseError.UnrecognizedEscape)]
        [InlineData(@"\k<0 ", RegexOptions.None, RegexParseError.UnrecognizedEscape)]
        [InlineData(@"\k<0>", RegexOptions.None, null)]
        [InlineData(@"\k<0> ", RegexOptions.None, null)]
        [InlineData(@"\k<00> ", RegexOptions.None, null)]
        [InlineData(@"\k<a> ", RegexOptions.None, RegexParseError.UndefinedNameRef)]
        [InlineData(@"(?<a>)\k<a> ", RegexOptions.None, null)]
        [InlineData(@"\k", RegexOptions.ECMAScript, RegexParseError.MalformedNameRef)]
        [InlineData(@"\k ", RegexOptions.ECMAScript, RegexParseError.MalformedNameRef)]
        [InlineData(@"\k<", RegexOptions.ECMAScript, RegexParseError.MalformedNameRef)]
        [InlineData(@"\k< ", RegexOptions.ECMAScript, null)]
        [InlineData(@"\k<0", RegexOptions.ECMAScript, null)]
        [InlineData(@"\k<0 ", RegexOptions.ECMAScript, null)]
        [InlineData(@"\k<0>", RegexOptions.ECMAScript, null)]
        [InlineData(@"\k<0> ", RegexOptions.ECMAScript, null)]
        [InlineData(@"\k'", RegexOptions.None, RegexParseError.MalformedNameRef)]
        [InlineData(@"\k' ", RegexOptions.None, RegexParseError.UnrecognizedEscape)]
        [InlineData(@"\k'0", RegexOptions.None, RegexParseError.UnrecognizedEscape)]
        [InlineData(@"\k'0 ", RegexOptions.None, RegexParseError.UnrecognizedEscape)]
        [InlineData(@"\k'0'", RegexOptions.None, null)]
        [InlineData(@"\k'0' ", RegexOptions.None, null)]
        [InlineData(@"\k'00' ", RegexOptions.None, null)]
        [InlineData(@"\k'a' ", RegexOptions.None, RegexParseError.UndefinedNameRef)]
        [InlineData(@"(?<a>)\k'a' ", RegexOptions.None, null)]
        [InlineData(@"\k<0' ", RegexOptions.None, RegexParseError.UnrecognizedEscape)]
        [InlineData(@"\k'0> ", RegexOptions.None, RegexParseError.UnrecognizedEscape)]
        [InlineData(@"\", RegexOptions.None, RegexParseError.IllegalEndEscape)]
        [InlineData(@"\ ", RegexOptions.None, null)]
        [InlineData(@"\<", RegexOptions.None, null)]
        [InlineData(@"\< ", RegexOptions.None, null)]
        [InlineData(@"\<0", RegexOptions.None, null)]
        [InlineData(@"\<0 ", RegexOptions.None, null)]
        [InlineData(@"\<0>", RegexOptions.None, null)]
        [InlineData(@"\<0> ", RegexOptions.None, null)]
        [InlineData(@"\<00> ", RegexOptions.None, null)]
        [InlineData(@"\<a> ", RegexOptions.None, RegexParseError.UndefinedNameRef)]
        [InlineData(@"(?<a>)\<a> ", RegexOptions.None, null)]
        [InlineData(@"\", RegexOptions.ECMAScript, RegexParseError.IllegalEndEscape)]
        [InlineData(@"\ ", RegexOptions.ECMAScript, null)]
        [InlineData(@"\<", RegexOptions.ECMAScript, null)]
        [InlineData(@"\< ", RegexOptions.ECMAScript, null)]
        [InlineData(@"\<0", RegexOptions.ECMAScript, null)]
        [InlineData(@"\<0 ", RegexOptions.ECMAScript, null)]
        [InlineData(@"\<0>", RegexOptions.ECMAScript, null)]
        [InlineData(@"\<0> ", RegexOptions.ECMAScript, null)]
        [InlineData(@"\'", RegexOptions.None, null)]
        [InlineData(@"\' ", RegexOptions.None, null)]
        [InlineData(@"\'0", RegexOptions.None, null)]
        [InlineData(@"\'0 ", RegexOptions.None, null)]
        [InlineData(@"\'0'", RegexOptions.None, null)]
        [InlineData(@"\'0' ", RegexOptions.None, null)]
        [InlineData(@"\'00' ", RegexOptions.None, null)]
        [InlineData(@"\'a' ", RegexOptions.None, RegexParseError.UndefinedNameRef)]
        [InlineData(@"(?<a>)\'a' ", RegexOptions.None, null)]
        [InlineData(@"\<0' ", RegexOptions.None, null)]
        [InlineData(@"\'0> ", RegexOptions.None, null)]
        [InlineData("\\p{Cc}", RegexOptions.None, null)]
        [InlineData("\\p{ Cc }", RegexOptions.None, RegexParseError.IncompleteSlashP)]
        [InlineData("\\p{ Cc }", RegexOptions.IgnorePatternWhitespace, RegexParseError.IncompleteSlashP)]
        [InlineData("\\p {Cc}", RegexOptions.IgnorePatternWhitespace, RegexParseError.MalformedSlashP)]
        [InlineData("\\p{xxx}", RegexOptions.None, RegexParseError.UnknownUnicodeProperty)]
        [InlineData("\\p", RegexOptions.None, RegexParseError.IncompleteSlashP)]
        [InlineData("\\p{", RegexOptions.None, RegexParseError.IncompleteSlashP)]
        [InlineData("\\p{}", RegexOptions.None, RegexParseError.IncompleteSlashP)]
        [InlineData("\\p{} ", RegexOptions.None, RegexParseError.UnknownUnicodeProperty)]
        [InlineData("\\p {} ", RegexOptions.None, RegexParseError.MalformedSlashP)]
        [InlineData("\\p{Cc ", RegexOptions.None, RegexParseError.IncompleteSlashP)]
        [InlineData("\\p{IsArabicPresentationForms-A}", RegexOptions.None, null)]
        [InlineData("(?:)", RegexOptions.None, null)]
        [InlineData("(?:a)", RegexOptions.None, null)]
        [InlineData("(?:", RegexOptions.None, RegexParseError.NotEnoughParentheses)]
        [InlineData("(?: ", RegexOptions.IgnorePatternWhitespace, RegexParseError.NotEnoughParentheses)]
        [InlineData("(?=)", RegexOptions.None, null)]
        [InlineData("(?=a)", RegexOptions.None, null)]
        [InlineData("(?=", RegexOptions.None, RegexParseError.NotEnoughParentheses)]
        [InlineData("(?= ", RegexOptions.IgnorePatternWhitespace, RegexParseError.NotEnoughParentheses)]
        [InlineData("(?!)", RegexOptions.None, null)]
        [InlineData("(?!a)", RegexOptions.None, null)]
        [InlineData("(?!", RegexOptions.None, RegexParseError.NotEnoughParentheses)]
        [InlineData("(?! ", RegexOptions.IgnorePatternWhitespace, RegexParseError.NotEnoughParentheses)]
        [InlineData("(?>)", RegexOptions.None, null)]
        [InlineData("(?>a)", RegexOptions.None, null)]
        [InlineData("(?>", RegexOptions.None, RegexParseError.NotEnoughParentheses)]
        [InlineData("(?> ", RegexOptions.IgnorePatternWhitespace, RegexParseError.NotEnoughParentheses)]
        [InlineData("(?<=)", RegexOptions.None, null)]
        [InlineData("(?<=a)", RegexOptions.None, null)]
        [InlineData("(?<=", RegexOptions.None, RegexParseError.NotEnoughParentheses)]
        [InlineData("(?<= ", RegexOptions.IgnorePatternWhitespace, RegexParseError.NotEnoughParentheses)]
        [InlineData("(?<!)", RegexOptions.None, null)]
        [InlineData("(?<!a)", RegexOptions.None, null)]
        [InlineData("(?<!", RegexOptions.None, RegexParseError.NotEnoughParentheses)]
        [InlineData("(?<! ", RegexOptions.IgnorePatternWhitespace, RegexParseError.NotEnoughParentheses)]
        [InlineData("(?<", RegexOptions.IgnorePatternWhitespace, RegexParseError.UnrecognizedGrouping)]
        [InlineData("(?<>", RegexOptions.IgnorePatternWhitespace, RegexParseError.InvalidGroupName)]
        [InlineData("(?<a", RegexOptions.IgnorePatternWhitespace, RegexParseError.UnrecognizedGrouping)]
        [InlineData("(?<a>", RegexOptions.IgnorePatternWhitespace, RegexParseError.NotEnoughParentheses)]
        [InlineData("(?<a>a", RegexOptions.IgnorePatternWhitespace, RegexParseError.NotEnoughParentheses)]
        [InlineData("(?<a>a)", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?<a >a)", RegexOptions.None, RegexParseError.InvalidGroupName)]
        [InlineData("(?<a >a)", RegexOptions.IgnorePatternWhitespace, RegexParseError.InvalidGroupName)]
        [InlineData("(?< a>a)", RegexOptions.None, RegexParseError.InvalidGroupName)]
        [InlineData("(?< a>a)", RegexOptions.IgnorePatternWhitespace, RegexParseError.InvalidGroupName)]
        [InlineData("(?< a >a)", RegexOptions.None, RegexParseError.InvalidGroupName)]
        [InlineData("(?< a >a)", RegexOptions.IgnorePatternWhitespace, RegexParseError.InvalidGroupName)]
        [InlineData("(?<ab>a)", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?<0>a)", RegexOptions.IgnorePatternWhitespace, RegexParseError.CapnumNotZero)]
        [InlineData("(?<1>a)", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?<10>a)", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?<1>)", RegexOptions.None, null)]
        [InlineData("(?<1> )", RegexOptions.None, null)]
        [InlineData("(?<1> )", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?'a')", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?(", RegexOptions.IgnorePatternWhitespace, RegexParseError.NotEnoughParentheses)]
        [InlineData("(?(0", RegexOptions.IgnorePatternWhitespace, RegexParseError.MalformedReference)]
        [InlineData("(?(0)", RegexOptions.IgnorePatternWhitespace, RegexParseError.NotEnoughParentheses)]
        [InlineData("(?(0))", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?(0)a)", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?(0)a|)", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?(0)a|b)", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?(0)a|b|)", RegexOptions.IgnorePatternWhitespace, RegexParseError.TooManyAlternates)]
        [InlineData("(?(0)a|b|c)", RegexOptions.IgnorePatternWhitespace, RegexParseError.TooManyAlternates)]
        [InlineData("(?(0 )", RegexOptions.IgnorePatternWhitespace, RegexParseError.MalformedReference)]
        [InlineData("(?(1))", RegexOptions.IgnorePatternWhitespace, RegexParseError.UndefinedReference)]
        [InlineData("(?(00))", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?(a))", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?<a>)(?(a))", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?<a>)(?(a ))", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?<a>)(?( a))", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?(()a()))", RegexOptions.None, null)]
        [InlineData("(?((?<x>)a(?<y>)))", RegexOptions.None, null)]
        [InlineData("(?(?'", RegexOptions.IgnorePatternWhitespace, RegexParseError.AlternationCantCapture)]
        [InlineData("(?(?'x'))", RegexOptions.IgnorePatternWhitespace, RegexParseError.AlternationCantCapture)]
        [InlineData("(?(?#", RegexOptions.IgnorePatternWhitespace, RegexParseError.UnterminatedComment)]
        [InlineData("(?(?#)", RegexOptions.IgnorePatternWhitespace, RegexParseError.AlternationCantHaveComment)]
        [InlineData("(?(?#))", RegexOptions.IgnorePatternWhitespace, RegexParseError.AlternationCantHaveComment)]
        [InlineData("(?(?<", RegexOptions.IgnorePatternWhitespace, RegexParseError.UnrecognizedGrouping)]
        [InlineData("(?(?<a", RegexOptions.IgnorePatternWhitespace, RegexParseError.AlternationCantCapture)]
        [InlineData("(?(?<a>", RegexOptions.IgnorePatternWhitespace, RegexParseError.AlternationCantCapture)]
        [InlineData("(?(?<a>)", RegexOptions.IgnorePatternWhitespace, RegexParseError.AlternationCantCapture)]
        [InlineData("(?(?<a>))", RegexOptions.IgnorePatternWhitespace, RegexParseError.AlternationCantCapture)]
        [InlineData("(?(?<=))", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?(?<!))", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData(@"\1", RegexOptions.None, RegexParseError.UndefinedBackref)]
        [InlineData(@"\1 ", RegexOptions.None, RegexParseError.UndefinedBackref)]
        [InlineData(@"()\1", RegexOptions.None, null)]
        [InlineData(@"()\1 ", RegexOptions.None, null)]
        [InlineData(@"()\10 ", RegexOptions.None, null)]
        [InlineData(@"\1", RegexOptions.ECMAScript, null)]
        [InlineData(@"\1 ", RegexOptions.ECMAScript, null)]
        [InlineData(@"()\1", RegexOptions.ECMAScript, null)]
        [InlineData(@"()\1 ", RegexOptions.ECMAScript, null)]
        [InlineData(@"()\10 ", RegexOptions.ECMAScript, null)]
        [InlineData(@"()()()()()()()()()()\10 ", RegexOptions.ECMAScript, null)]
        [InlineData(@"[", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        [InlineData(@"[ ", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        [InlineData(@"[]", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        [InlineData(@"[] ", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        [InlineData(@"[a]", RegexOptions.None, null)]
        [InlineData(@"[a] ", RegexOptions.None, null)]
        [InlineData(@"[a-", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        [InlineData(@"[a- ", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        [InlineData(@"[a-]", RegexOptions.None, null)]
        [InlineData(@"[a-] ", RegexOptions.None, null)]
        [InlineData(@"[a-b]", RegexOptions.None, null)]
        [InlineData(@"[a-b] ", RegexOptions.None, null)]
        [InlineData(@"[a-[b]] ", RegexOptions.None, null)]
        [InlineData(@"[a-b-[c]] ", RegexOptions.None, null)]
        [InlineData(@"[a-[b]-c] ", RegexOptions.None, RegexParseError.SubtractionMustBeLast)]
        [InlineData(@"[[a]-b] ", RegexOptions.None, null)]
        [InlineData(@"[[a]-[b]] ", RegexOptions.None, null)]
        [InlineData(@"[\w-a] ", RegexOptions.None, null)]
        [InlineData(@"[a-\w] ", RegexOptions.None, RegexParseError.BadClassInCharRange)]
        [InlineData(@"[\p{llll}-a] ", RegexOptions.None, RegexParseError.UnknownUnicodeProperty)]
        [InlineData(@"[\p{Lu}-a] ", RegexOptions.None, null)]
        [InlineData(@"[a-\p{Lu}] ", RegexOptions.None, RegexParseError.BadClassInCharRange)]
        [InlineData(@"[a-[:Ll:]] ", RegexOptions.None, null)]
        [InlineData(@"[a-[:Ll]] ", RegexOptions.None, null)]
        [InlineData(@"[a-[:", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        [InlineData(@"[a-[:L", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        [InlineData(@"[a-[:L:", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        [InlineData(@"[a-[:L:]", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        [InlineData(@"[\-]", RegexOptions.None, null)]
        [InlineData(@"[a-b-c] ", RegexOptions.None, null)]
        [InlineData(@"[-b-c] ", RegexOptions.None, null)]
        [InlineData(@"[-[b] ", RegexOptions.None, null)]
        [InlineData(@"[-[b]] ", RegexOptions.None, null)]
        [InlineData(@"[--b ", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        [InlineData(@"[--b] ", RegexOptions.None, null)]
        [InlineData(@"[--[b ", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        [InlineData(@"[--[b] ", RegexOptions.None, RegexParseError.SubtractionMustBeLast)]
        [InlineData(@"[--[b]] ", RegexOptions.None, null)]
        [InlineData(@"[a--[b ", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        [InlineData(@"[,--[a] ", RegexOptions.None, null)]
        [InlineData(@"[,--[a]] ", RegexOptions.None, null)]
        [InlineData(@"[\s-a]", RegexOptions.None, null)]
        [InlineData(@"[\p{Lu}-a]", RegexOptions.None, null)]
        [InlineData(@"[\c<-\c>]", RegexOptions.None, RegexParseError.UnrecognizedControl)]
        [InlineData(@"[\c>-\c<]", RegexOptions.None, RegexParseError.UnrecognizedControl)]
        [InlineData(@"[\c>-a]", RegexOptions.None, RegexParseError.UnrecognizedControl)]
        [InlineData(@"[a-\c>]", RegexOptions.None, RegexParseError.UnrecognizedControl)]
        [InlineData(@"[a--]", RegexOptions.None, RegexParseError.ReversedCharRange)]
        [InlineData(@"[--a]", RegexOptions.None, null)]
        [InlineData(@"[a-\-]", RegexOptions.None, null)]
        [InlineData(@"[\--a]", RegexOptions.None, null)]
        [InlineData(@"[\0-\1]", RegexOptions.None, null)]
        [InlineData(@"[\1-\0]", RegexOptions.None, RegexParseError.ReversedCharRange)]
        [InlineData(@"[\0-\01]", RegexOptions.None, null)]
        [InlineData(@"[\01-\0]", RegexOptions.None, RegexParseError.ReversedCharRange)]
        [InlineData(@"[[:x:]-a]", RegexOptions.None, null)]
        [InlineData(@"[a-[:x:]]", RegexOptions.None, null)]
        [InlineData(@"[\0-\ca]", RegexOptions.None, null)]
        [InlineData(@"[\ca-\0]", RegexOptions.None, RegexParseError.ReversedCharRange)]
        [InlineData(@"[\ca-\cA]", RegexOptions.None, null)]
        [InlineData(@"[\cA-\ca]", RegexOptions.None, null)]
        [InlineData(@"[\u0-\u1]", RegexOptions.None, RegexParseError.TooFewHex)]
        [InlineData(@"[\u1-\u0]", RegexOptions.None, RegexParseError.TooFewHex)]
        [InlineData(@"[\u0000-\u0000]", RegexOptions.None, null)]
        [InlineData(@"[\u0000-\u0001]", RegexOptions.None, null)]
        [InlineData(@"[\u0001-\u0000]", RegexOptions.None, RegexParseError.ReversedCharRange)]
        [InlineData(@"[\u0001-a]", RegexOptions.None, null)]
        [InlineData(@"[a-\u0001]", RegexOptions.None, RegexParseError.ReversedCharRange)]
        [InlineData(@"[a-a]", RegexOptions.None, null)]
        [InlineData(@"[a-A]", RegexOptions.None, RegexParseError.ReversedCharRange)]
        [InlineData(@"[A-a]", RegexOptions.None, null)]
        [InlineData(@"[a-a]", RegexOptions.IgnoreCase, null)]
        [InlineData(@"[a-A]", RegexOptions.IgnoreCase, RegexParseError.ReversedCharRange)]
        [InlineData(@"[A-a]", RegexOptions.IgnoreCase, null)]
        [InlineData(@"[a-\x61]", RegexOptions.None, null)]
        [InlineData(@"[\x61-a]", RegexOptions.None, null)]
        [InlineData(@"[a-\x60]", RegexOptions.None, RegexParseError.ReversedCharRange)]
        [InlineData(@"[\x62-a]", RegexOptions.None, RegexParseError.ReversedCharRange)]
        [InlineData(@"[a-\x62]", RegexOptions.None, null)]
        [InlineData(@"[\3-\cc]", RegexOptions.None, null)]
        [InlineData(@"[\cc-\3]", RegexOptions.None, null)]
        [InlineData(@"[\2-\cc]", RegexOptions.None, null)]
        [InlineData(@"[\cc-\2]", RegexOptions.None, RegexParseError.ReversedCharRange)]
        [InlineData(@"[\4-\cc]", RegexOptions.None, RegexParseError.ReversedCharRange)]
        [InlineData(@"[\cc-\4]", RegexOptions.None, null)]
        [InlineData(@"[\ca-\cb]", RegexOptions.None, null)]
        [InlineData(@"[\ca-\cB]", RegexOptions.None, null)]
        [InlineData(@"[\cA-\cb]", RegexOptions.None, null)]
        [InlineData(@"[\cA-\cB]", RegexOptions.None, null)]
        [InlineData(@"[\cb-\ca]", RegexOptions.None, RegexParseError.ReversedCharRange)]
        [InlineData(@"[\cb-\cA]", RegexOptions.None, RegexParseError.ReversedCharRange)]
        [InlineData(@"[\cB-\ca]", RegexOptions.None, RegexParseError.ReversedCharRange)]
        [InlineData(@"[\cB-\cA]", RegexOptions.None, RegexParseError.ReversedCharRange)]
        [InlineData(@"[\--#]", RegexOptions.None, null)]
        [InlineData(@"[a-\-b]", RegexOptions.None, null)]
        [InlineData(@"[a-\-\-b]", RegexOptions.None, null)]
        [InlineData(@"[b-\-a]", RegexOptions.None, RegexParseError.ReversedCharRange)]
        [InlineData(@"[b-\-\-a]", RegexOptions.None, RegexParseError.ReversedCharRange)]
        [InlineData(@"[a-\-\D]", RegexOptions.None, RegexParseError.BadClassInCharRange)]
        [InlineData(@"[a-\-\-\D]", RegexOptions.None, RegexParseError.BadClassInCharRange)]
        [InlineData(@"[a -\-\b]", RegexOptions.None, RegexParseError.ReversedCharRange)]
        [InlineData(@"()\2", RegexOptions.None, RegexParseError.UndefinedBackref)]
        [InlineData(@"()()\2", RegexOptions.None, null)]
        [InlineData(@"()\1", RegexOptions.ExplicitCapture, RegexParseError.UndefinedBackref)]
        [InlineData(@"()\2", RegexOptions.ExplicitCapture, RegexParseError.UndefinedBackref)]
        [InlineData(@"()()\2", RegexOptions.ExplicitCapture, RegexParseError.UndefinedBackref)]
        [InlineData(@"()()(?n)\1\2", RegexOptions.None, null)]
        [InlineData(@"()(?n)()\1\2", RegexOptions.None, RegexParseError.UndefinedBackref)]
        [InlineData(@"(?n)()()\1\2", RegexOptions.None, RegexParseError.UndefinedBackref)]
        [InlineData(@"()()(?n)\1\2", RegexOptions.ExplicitCapture, RegexParseError.UndefinedBackref)]
        [InlineData(@"()(?n)()\1\2", RegexOptions.ExplicitCapture, RegexParseError.UndefinedBackref)]
        [InlineData(@"(?n)()()\1\2", RegexOptions.ExplicitCapture, RegexParseError.UndefinedBackref)]
        [InlineData(@"()()(?-n)\1\2", RegexOptions.None, null)]
        [InlineData(@"()(?-n)()\1\2", RegexOptions.None, null)]
        [InlineData(@"(?-n)()()\1\2", RegexOptions.None, null)]
        [InlineData(@"()()(?-n)\1\2", RegexOptions.ExplicitCapture, RegexParseError.UndefinedBackref)]
        [InlineData(@"()(?-n)()\1\2", RegexOptions.ExplicitCapture, RegexParseError.UndefinedBackref)]
        [InlineData(@"(?-n)()()\1\2", RegexOptions.ExplicitCapture, null)]
        [InlineData(@"()()(?n:\1\2)", RegexOptions.None, null)]
        [InlineData(@"()()(?n:\1\2)", RegexOptions.ExplicitCapture, RegexParseError.UndefinedBackref)]
        [InlineData(@"()()(?-n:\1\2)", RegexOptions.None, null)]
        [InlineData(@"()()(?-n:\1\2)", RegexOptions.ExplicitCapture, RegexParseError.UndefinedBackref)]
        [InlineData(@"(?n:)()()\1\2", RegexOptions.None, null)]
        [InlineData(@"(?n:)()()\1\2", RegexOptions.ExplicitCapture, RegexParseError.UndefinedBackref)]
        [InlineData(@"(?-n:)()()\1\2", RegexOptions.None, null)]
        [InlineData(@"(?-n:)()()\1\2", RegexOptions.ExplicitCapture, RegexParseError.UndefinedBackref)]
        [InlineData(@"(?n)(?-n)()()\1\2", RegexOptions.None, null)]
        [InlineData(@"(?n)(?-n)()()\1\2", RegexOptions.ExplicitCapture, null)]
        [InlineData(@"(?-n)(?n)()()\1\2", RegexOptions.None, RegexParseError.UndefinedBackref)]
        [InlineData(@"(?-n)(?n)()()\1\2", RegexOptions.ExplicitCapture, RegexParseError.UndefinedBackref)]
        // References
        [InlineData(@"[aeiou]", RegexOptions.None, null)]
        [InlineData(@"(?<duplicateWord>\w+)\s\k<duplicateWord>\W(?<nextWord>\w+)", RegexOptions.None, null)]
        [InlineData(@"((?<One>abc)\d+)?(?<Two>xyz)(.*)", RegexOptions.None, null)]
        [InlineData(@"(\w+)\s(\1)", RegexOptions.None, null)]
        [InlineData(@"\Bqu\w+", RegexOptions.None, null)]
        [InlineData(@"\bare\w*\b", RegexOptions.None, null)]
        [InlineData(@"\G(\w+\s?\w*),?", RegexOptions.None, null)]
        [InlineData(@"\D+(?<digit>\d+)\D+(?<digit>\d+)?", RegexOptions.None, null)]
        [InlineData(@"(\s\d{4}(-(\d{4}&#124;present))?,?)+", RegexOptions.None, null)]
        [InlineData(@"^((\w+(\s?)){2,}),\s(\w+\s\w+),(\s\d{4}(-(\d{4}|present))?,?)+", RegexOptions.None, null)]
        [InlineData(@"^[0-9-[2468]]+$", RegexOptions.None, null)]
        [InlineData(@"[a-z-[0-9]]", RegexOptions.None, null)]
        [InlineData(@"[\p{IsBasicLatin}-[\x00-\x7F]]", RegexOptions.None, null)]
        [InlineData(@"[\u0000-\uFFFF-[\s\p{P}\p{IsGreek}\x85]]", RegexOptions.None, null)]
        [InlineData(@"[a-z-[d-w-[m-o]]]", RegexOptions.None, null)]
        [InlineData(@"((\w+(\s?)){2,}", RegexOptions.None, RegexParseError.NotEnoughParentheses)]
        [InlineData(@"[a-z-[djp]]", RegexOptions.None, null)]
        [InlineData(@"(\w)\1+.\b", RegexOptions.None, null)]
        [InlineData(@"\d{4}\b", RegexOptions.None, null)]
        [InlineData(@"\d{1,2},", RegexOptions.None, null)]
        [InlineData(@"(?<!(Saturday|Sunday) )\b\w+ \d{1,2}, \d{4}\b", RegexOptions.None, null)]
        [InlineData(@"(?<=\b20)\d{2}\b", RegexOptions.None, null)]
        [InlineData(@"\b\w+\b(?!\p{P})", RegexOptions.None, null)]
        [InlineData(@"\b(?!un)\w+\b", RegexOptions.None, null)]
        [InlineData(@"\b(?ix: d \w+)\s", RegexOptions.None, null)]
        [InlineData(@"(?:\w+)", RegexOptions.None, null)]
        [InlineData(@"(?:\b(?:\w+)\W*)+", RegexOptions.None, null)]
        [InlineData(@"(?:\b(?:\w+)\W*)+\.", RegexOptions.None, null)]
        [InlineData(@"[^<>]*", RegexOptions.None, null)]
        [InlineData(@"\b\w+(?=\sis\b)", RegexOptions.None, null)]
        [InlineData(@"[a-z-[m]]", RegexOptions.None, null)]
        [InlineData(@"^\D\d{1,5}\D*$", RegexOptions.None, null)]
        [InlineData(@"[^0-9]", RegexOptions.None, null)]
        [InlineData(@"(\p{IsGreek}+(\s)?)+", RegexOptions.None, null)]
        [InlineData(@"\b(\p{IsGreek}+(\s)?)+\p{Pd}\s(\p{IsBasicLatin}+(\s)?)+", RegexOptions.None, null)]
        [InlineData(@"\b.*[.?!;:](\s|\z)", RegexOptions.None, null)]
        [InlineData(@"^.+", RegexOptions.None, null)]
        [InlineData(@"[^o]", RegexOptions.None, null)]
        [InlineData(@"\bth[^o]\w+\b", RegexOptions.None, null)]
        [InlineData(@"(\P{Sc})+", RegexOptions.None, null)]
        [InlineData(@"[^\p{P}\d]", RegexOptions.None, null)]
        [InlineData(@"\b[A-Z]\w*\b", RegexOptions.None, null)]
        [InlineData(@"\S+?", RegexOptions.None, null)]
        [InlineData(@"y\s", RegexOptions.None, null)]
        [InlineData(@"gr[ae]y\s\S+?[\s\p{P}]", RegexOptions.None, null)]
        [InlineData(@"[\s\p{P}]", RegexOptions.None, null)]
        [InlineData(@"[\p{P}\d]", RegexOptions.None, null)]
        [InlineData(@"[^aeiou]", RegexOptions.None, null)]
        [InlineData(@"(\w)\1", RegexOptions.None, null)]
        [InlineData(@"[^\p{Ll}\p{Lu}\p{Lt}\p{Lo}\p{Nd}\p{Pc}\p{Lm}] ", RegexOptions.None, null)]
        [InlineData(@"[^a-zA-Z_0-9]", RegexOptions.None, null)]
        [InlineData(@"\P{Nd}", RegexOptions.None, null)]
        [InlineData(@"(\(?\d{3}\)?[\s-])?", RegexOptions.None, null)]
        [InlineData(@"^(\(?\d{3}\)?[\s-])?\d{3}-\d{4}$", RegexOptions.None, null)]
        [InlineData(@"[0-9]", RegexOptions.None, null)]
        [InlineData(@"\p{Nd}", RegexOptions.None, null)]
        [InlineData(@"\b(\S+)\s?", RegexOptions.None, null)]
        [InlineData(@"[^ \f\n\r\t\v]", RegexOptions.None, null)]
        [InlineData(@"[^\f\n\r\t\v\x85\p{Z}]", RegexOptions.None, null)]
        [InlineData(@"(\s|$)", RegexOptions.None, null)]
        [InlineData(@"\b\w+(e)?s(\s|$)", RegexOptions.None, null)]
        [InlineData(@"[ \f\n\r\t\v]", RegexOptions.None, null)]
        [InlineData(@"(\W){1,2}", RegexOptions.None, null)]
        [InlineData(@"(\w+)", RegexOptions.None, null)]
        [InlineData(@"\b", RegexOptions.None, null)]
        [InlineData(@"\b(\w+)(\W){1,2}", RegexOptions.None, null)]
        [InlineData(@"(?>(\w)\1+).\b", RegexOptions.None, null)]
        [InlineData(@"(\b(\w+)\W+)+", RegexOptions.None, null)]
        [InlineData(@"\p{Sc}*(\s?\d+[.,]?\d*)\p{Sc}*", RegexOptions.None, null)]
        [InlineData(@"p{Sc}*(?<amount>\s?\d+[.,]?\d*)\p{Sc}*", RegexOptions.None, null)]
        [InlineData(@"^(\w+\s?)+$", RegexOptions.None, null)]
        [InlineData(@"(?ix) d \w+ \s", RegexOptions.None, null)]
        [InlineData(@"\bthe\w*\b", RegexOptions.None, null)]
        [InlineData(@"\b(?i:t)he\w*\b", RegexOptions.None, null)]
        [InlineData(@"^(\w+)\s(\d+)$", RegexOptions.None, null)]
        [InlineData(@"^(\w+)\s(\d+)\r*$", RegexOptions.Multiline, null)]
        [InlineData(@"(?m)^(\w+)\s(\d+)\r*$", RegexOptions.Multiline, null)]
        [InlineData(@"(?s)^.+", RegexOptions.None, null)]
        [InlineData(@"\b(\d{2}-)*(?(1)\d{7}|\d{3}-\d{2}-\d{4})\b", RegexOptions.None, null)]
        [InlineData(@"\b\(?((\w+),?\s?)+[\.!?]\)?", RegexOptions.None, null)]
        [InlineData(@"(?n)\b\(?((?>\w+),?\s?)+[\.!?]\)?", RegexOptions.None, null)]
        [InlineData(@"\b\(?(?n:(?>\w+),?\s?)+[\.!?]\)?", RegexOptions.None, null)]
        [InlineData(@"\b\(?((?>\w+),?\s?)+[\.!?]\)?", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData(@"(?x)\b \(? ( (?>\w+) ,?\s? )+  [\.!?] \)? # Matches an entire sentence.", RegexOptions.None, null)]
        [InlineData(@"\bb\w+\s", RegexOptions.RightToLeft, null)]
        [InlineData(@"(?<=\d{1,2}\s)\w+,?\s\d{4}", RegexOptions.RightToLeft, null)]
        [InlineData(@"\b(\w+\s*)+", RegexOptions.ECMAScript, null)]
        [InlineData(@"((a+)(\1) ?)+", RegexOptions.ECMAScript, null)]
        [InlineData(@"\b(D\w+)\s(d\w+)\b", RegexOptions.None, null)]
        [InlineData(@"\b(D\w+)(?ixn) \s (d\w+) \b", RegexOptions.None, null)]
        [InlineData(@"\b((?# case-sensitive comparison)D\w+)\s((?#case-insensitive comparison)d\w+)\b", RegexOptions.None, null)]
        [InlineData(@"\b\(?((?>\w+),?\s?)+[\.!?]\)?", RegexOptions.None, null)]
        [InlineData(@"\b(?<n2>\d{2}-)*(?(n2)\d{7}|\d{3}-\d{2}-\d{4})\b", RegexOptions.None, null)]
        [InlineData(@"\b(\d{2}-\d{7}|\d{3}-\d{2}-\d{4})\b", RegexOptions.None, null)]
        [InlineData(@"\bgr(a|e)y\b", RegexOptions.None, null)]
        [InlineData(@"\b91*9*\b", RegexOptions.None, null)]
        [InlineData(@"\ban+\w*?\b", RegexOptions.None, null)]
        [InlineData(@"\ban?\b", RegexOptions.None, null)]
        [InlineData(@"\b\d+\,\d{3}\b", RegexOptions.None, null)]
        [InlineData(@"\b\d{2,}\b\D+", RegexOptions.None, null)]
        [InlineData(@"(00\s){2,4}", RegexOptions.None, null)]
        [InlineData(@"\b\w*?oo\w*?\b", RegexOptions.None, null)]
        [InlineData(@"\b\w+?\b", RegexOptions.None, null)]
        [InlineData(@"^\s*(System.)??Console.Write(Line)??\(??", RegexOptions.None, null)]
        [InlineData(@"(System.)??", RegexOptions.None, null)]
        [InlineData(@"\b(\w{3,}?\.){2}?\w{3,}?\b", RegexOptions.None, null)]
        [InlineData(@"\b[A-Z](\w*?\s*?){1,10}[.!?]", RegexOptions.None, null)]
        [InlineData(@"b.*([0-9]{4})\b", RegexOptions.None, null)]
        [InlineData(@"\b.*?([0-9]{4})\b", RegexOptions.None, null)]
        [InlineData(@"(a?)*", RegexOptions.None, null)]
        [InlineData(@"(a\1|(?(1)\1)){0,2}", RegexOptions.None, null)]
        [InlineData(@"(a\1|(?(1)\1)){2}", RegexOptions.None, null)]
        [InlineData(@"(?<char>\w)\k<char>", RegexOptions.None, null)]
        [InlineData(@"(?<2>\w)\k<2>", RegexOptions.None, null)]
        [InlineData(@"(?<1>a)(?<1>\1b)*", RegexOptions.None, null)]
        [InlineData(@"\b(\p{Lu}{2})(\d{2})?(\p{Lu}{2})\b", RegexOptions.None, null)]
        [InlineData(@"\bgr[ae]y\b", RegexOptions.None, null)]
        [InlineData(@"\b((?# case sensitive comparison)D\w+)\s(?ixn)((?#case insensitive comparison)d\w+)\b", RegexOptions.None, null)]
        [InlineData(@"\{\d+(,-*\d+)*(\:\w{1,4}?)*\}(?x) # Looks for a composite format item.", RegexOptions.None, null)]
        // Negative tests
        [InlineData(@"cat([a-\d]*)dog", RegexOptions.None, RegexParseError.BadClassInCharRange)]
        [InlineData(@"\k<1", RegexOptions.None, RegexParseError.UnrecognizedEscape)]
        [InlineData(@"(?')", RegexOptions.None, RegexParseError.InvalidGroupName)]
        [InlineData(@"(?<)", RegexOptions.None, RegexParseError.InvalidGroupName)]
        [InlineData(@"(?imn )", RegexOptions.None, RegexParseError.UnrecognizedGrouping)]
        [InlineData(@"(?imn", RegexOptions.None, RegexParseError.UnrecognizedGrouping)]
        [InlineData(@"(?'cat'", RegexOptions.None, RegexParseError.NotEnoughParentheses)]
        [InlineData(@"(?'", RegexOptions.None, RegexParseError.UnrecognizedGrouping)]
        [InlineData(@"[^", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        [InlineData(@"[cat", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        [InlineData(@"[^cat", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        [InlineData(@"\p{", RegexOptions.None, RegexParseError.IncompleteSlashP)]
        [InlineData(@"\p{cat", RegexOptions.None, RegexParseError.IncompleteSlashP)]
        [InlineData(@"\k<cat", RegexOptions.None, RegexParseError.UnrecognizedEscape)]
        [InlineData(@"\p{cat}", RegexOptions.None, RegexParseError.UnknownUnicodeProperty)]
        [InlineData(@"\P{cat", RegexOptions.None, RegexParseError.IncompleteSlashP)]
        [InlineData(@"\P{cat}", RegexOptions.None, RegexParseError.UnknownUnicodeProperty)]
        [InlineData(@"(?<cat>", RegexOptions.None, RegexParseError.NotEnoughParentheses)]
        [InlineData(@"\P{", RegexOptions.None, RegexParseError.IncompleteSlashP)]
        [InlineData(@"\k<>", RegexOptions.None, RegexParseError.UnrecognizedEscape)]
        [InlineData(@"(?(", RegexOptions.None, RegexParseError.NotEnoughParentheses)]
        [InlineData(@"(?()|", RegexOptions.None, RegexParseError.NotEnoughParentheses)]
        [InlineData(@"?(a|b)", RegexOptions.None, RegexParseError.QuantifyAfterNothing)]
        [InlineData(@"?((a)", RegexOptions.None, RegexParseError.QuantifyAfterNothing)]
        [InlineData(@"?((a)a", RegexOptions.None, RegexParseError.QuantifyAfterNothing)]
        [InlineData(@"?((a)a|", RegexOptions.None, RegexParseError.QuantifyAfterNothing)]
        [InlineData(@"?((a)a|b", RegexOptions.None, RegexParseError.QuantifyAfterNothing)]
        [InlineData(@"?(a)", RegexOptions.None, RegexParseError.QuantifyAfterNothing)]
        [InlineData(@"[a", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        [InlineData(@"?(a:b)", RegexOptions.None, RegexParseError.QuantifyAfterNothing)]
        [InlineData(@"(?(?", RegexOptions.None, RegexParseError.UnrecognizedGrouping)]
        [InlineData(@"(?(cat", RegexOptions.None, RegexParseError.NotEnoughParentheses)]
        [InlineData(@"(?(cat)|", RegexOptions.None, RegexParseError.NotEnoughParentheses)]
        [InlineData(@"foo(?<0>bar)", RegexOptions.None, RegexParseError.CapnumNotZero)]
        [InlineData(@"foo(?'0'bar)", RegexOptions.None, RegexParseError.CapnumNotZero)]
        [InlineData(@"foo(?<1bar)", RegexOptions.None, RegexParseError.InvalidGroupName)]
        [InlineData(@"foo(?'1bar)", RegexOptions.None, RegexParseError.InvalidGroupName)]
        [InlineData(@"\p{klsak", RegexOptions.None, RegexParseError.IncompleteSlashP)]
        [InlineData(@"(?c:cat)", RegexOptions.None, RegexParseError.UnrecognizedGrouping)]
        [InlineData(@"(??e:cat)", RegexOptions.None, RegexParseError.UnrecognizedGrouping)]
        [InlineData(@"[a-f-[]]+", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        [InlineData(@"[A-[]+", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        [InlineData(@"(?(?e))", RegexOptions.None, RegexParseError.UnrecognizedGrouping)]
        [InlineData(@"(?(?a)", RegexOptions.None, RegexParseError.UnrecognizedGrouping)]
        [InlineData(@"(?r:cat)", RegexOptions.None, RegexParseError.UnrecognizedGrouping)]
        [InlineData(@"\x2", RegexOptions.None, RegexParseError.TooFewHex)]
        [InlineData(@"(cat) (?#cat)    \s+ (?#followed by 1 or more whitespace", RegexOptions.IgnorePatternWhitespace, RegexParseError.UnterminatedComment)]
        [InlineData(@"cat(?(?afdcat)dog)", RegexOptions.None, RegexParseError.UnrecognizedGrouping)]
        [InlineData(@"cat(?(?<cat>cat)dog)", RegexOptions.None, RegexParseError.AlternationCantCapture)]
        [InlineData(@"cat(?(?'cat'cat)dog)", RegexOptions.None, RegexParseError.AlternationCantCapture)]
        [InlineData(@"cat(?(?#COMMENT)cat)", RegexOptions.None, RegexParseError.AlternationCantHaveComment)]
        [InlineData(@"cat(?<>dog)", RegexOptions.None, RegexParseError.InvalidGroupName)]
        [InlineData(@"cat(?<dog<>)_*>dog)", RegexOptions.None, RegexParseError.InvalidGroupName)]
        [InlineData(@"cat(?<dog >)_*>dog)", RegexOptions.None, RegexParseError.InvalidGroupName)]
        [InlineData(@"cat(?<dog!>)_*>dog)", RegexOptions.None, RegexParseError.InvalidGroupName)]
        [InlineData(@"cat(?<dog)_*>dog)", RegexOptions.None, RegexParseError.InvalidGroupName)]
        [InlineData(@"cat(?<1dog>dog)", RegexOptions.None, RegexParseError.InvalidGroupName)]
        [InlineData(@"cat(?<0>dog)", RegexOptions.None, RegexParseError.CapnumNotZero)]
        [InlineData(@"([5-\D]*)dog", RegexOptions.None, RegexParseError.BadClassInCharRange)]
        [InlineData(@"cat([6-\s]*)dog", RegexOptions.None, RegexParseError.BadClassInCharRange)]
        [InlineData(@"cat([c-\S]*)", RegexOptions.None, RegexParseError.BadClassInCharRange)]
        [InlineData(@"cat([7-\w]*)", RegexOptions.None, RegexParseError.BadClassInCharRange)]
        [InlineData(@"cat([a-\W]*)dog", RegexOptions.None, RegexParseError.BadClassInCharRange)]
        [InlineData(@"([f-\p{Lu}]\w*)\s([\p{Lu}]\w*)", RegexOptions.None, RegexParseError.BadClassInCharRange)]
        [InlineData(@"(cat) (?#cat)    \s+ (?#followed by 1 or more whitespace", RegexOptions.None, RegexParseError.UnterminatedComment)]
        [InlineData(@"([1-\P{Ll}][\p{Ll}]*)\s([\P{Ll}][\p{Ll}]*)", RegexOptions.None, RegexParseError.BadClassInCharRange)]
        [InlineData(@"[\P]", RegexOptions.None, RegexParseError.IncompleteSlashP)]
        [InlineData(@"([\pcat])", RegexOptions.None, RegexParseError.MalformedSlashP)]
        [InlineData(@"([\Pcat])", RegexOptions.None, RegexParseError.MalformedSlashP)]
        [InlineData(@"(\p{", RegexOptions.None, RegexParseError.IncompleteSlashP)]
        [InlineData(@"(\p{Ll", RegexOptions.None, RegexParseError.IncompleteSlashP)]
        [InlineData(@"(cat)([\o]*)(dog)", RegexOptions.None, RegexParseError.UnrecognizedEscape)]
        [InlineData(@"[\p]", RegexOptions.None, RegexParseError.IncompleteSlashP)]
        [InlineData(@"(?<cat>cat)\s+(?<dog>dog)\kcat", RegexOptions.None, RegexParseError.MalformedNameRef)]
        [InlineData(@"(?<cat>cat)\s+(?<dog>dog)\k<cat2>", RegexOptions.None, RegexParseError.UndefinedNameRef)]
        [InlineData(@"(?<cat>cat)\s+(?<dog>dog)\k<8>cat", RegexOptions.None, RegexParseError.UndefinedBackref)]
        [InlineData(@"^[abcd]{1}?*$", RegexOptions.None, RegexParseError.NestedQuantify)]
        [InlineData(@"^[abcd]*+$", RegexOptions.None, RegexParseError.NestedQuantify)]
        [InlineData(@"^[abcd]+*$", RegexOptions.None, RegexParseError.NestedQuantify)]
        [InlineData(@"^[abcd]?*$", RegexOptions.None, RegexParseError.NestedQuantify)]
        [InlineData(@"^[abcd]*?+$", RegexOptions.None, RegexParseError.NestedQuantify)]
        [InlineData(@"^[abcd]+?*$", RegexOptions.None, RegexParseError.NestedQuantify)]
        [InlineData(@"^[abcd]{1,}?*$", RegexOptions.None, RegexParseError.NestedQuantify)]
        [InlineData(@"^[abcd]??*$", RegexOptions.None, RegexParseError.NestedQuantify)]
        [InlineData(@"^[abcd]+{0,5}$", RegexOptions.None, RegexParseError.NestedQuantify)]
        [InlineData(@"^[abcd]?{0,5}$", RegexOptions.None, RegexParseError.NestedQuantify)]
        [InlineData(@"\ua", RegexOptions.None, RegexParseError.TooFewHex)]
        [InlineData(@"^[abcd]*{0,5}$", RegexOptions.None, RegexParseError.NestedQuantify)]
        [InlineData(@"^[abcd]{0,16}?*$", RegexOptions.None, RegexParseError.NestedQuantify)]
        [InlineData(@"^[abcd]{1,}*$", RegexOptions.None, RegexParseError.NestedQuantify)]
        [InlineData(@"(?<cat>cat)\s+(?<dog>dog)\k<8>cat", RegexOptions.ECMAScript, RegexParseError.UndefinedBackref)]
        [InlineData(@"(?<cat>cat)\s+(?<dog>dog)\k8", RegexOptions.None, RegexParseError.MalformedNameRef)]
        [InlineData(@"(?<cat>cat)\s+(?<dog>dog)\k8", RegexOptions.ECMAScript, RegexParseError.MalformedNameRef)]
        [InlineData(@"(cat)(\7)", RegexOptions.None, RegexParseError.UndefinedBackref)]
        [InlineData(@"(cat)\s+(?<2147483648>dog)", RegexOptions.None, RegexParseError.CaptureGroupOutOfRange)]
        [InlineData(@"(cat)\s+(?<21474836481097>dog)", RegexOptions.None, RegexParseError.CaptureGroupOutOfRange)]
        [InlineData(@"^[abcd]{1}*$", RegexOptions.None, RegexParseError.NestedQuantify)]
        [InlineData(@"(cat)(\c*)(dog)", RegexOptions.None, RegexParseError.UnrecognizedControl)]
        [InlineData(@"(cat)(\c *)(dog)", RegexOptions.None, RegexParseError.UnrecognizedControl)]
        [InlineData(@"(cat)(\c?*)(dog)", RegexOptions.None, RegexParseError.UnrecognizedControl)]
        [InlineData(@"(cat)(\c`*)(dog)", RegexOptions.None, RegexParseError.UnrecognizedControl)]
        [InlineData(@"(cat)(\c\|*)(dog)", RegexOptions.None, RegexParseError.QuantifyAfterNothing)]
        [InlineData(@"^[abcd]{0,16}*$", RegexOptions.None, RegexParseError.NestedQuantify)]
        [InlineData(@"(cat)\c", RegexOptions.None, RegexParseError.MissingControl)]
        // Deep recursion
        [InlineData(@"@""((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((
(((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((
(((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((
(((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((
(((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((
(((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((
(((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((
(((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((
(((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((
(((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((
(((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((
(((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((""", RegexOptions.None, RegexParseError.NotEnoughParentheses)]
        // Scan control
        [InlineData("(cat)(\\c\0*)(dog)", RegexOptions.None, RegexParseError.UnrecognizedControl)]
        [InlineData(@"(cat)(\c\[*)(dog)", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        // Nested quantifiers
        [InlineData("^[abcd]{0,16}*$", RegexOptions.None, RegexParseError.NestedQuantify)]
        [InlineData("^[abcd]{1,}*$", RegexOptions.None, RegexParseError.NestedQuantify)]
        [InlineData("^[abcd]{1}*$", RegexOptions.None, RegexParseError.NestedQuantify)]
        [InlineData("^[abcd]{0,16}?*$", RegexOptions.None, RegexParseError.NestedQuantify)]
        [InlineData("^[abcd]{1,}?*$", RegexOptions.None, RegexParseError.NestedQuantify)]
        [InlineData("^[abcd]{1}?*$", RegexOptions.None, RegexParseError.NestedQuantify)]
        [InlineData("^[abcd]*+$", RegexOptions.None, RegexParseError.NestedQuantify)]
        [InlineData("^[abcd]+*$", RegexOptions.None, RegexParseError.NestedQuantify)]
        [InlineData("^[abcd]?*$", RegexOptions.None, RegexParseError.NestedQuantify)]
        [InlineData("^[abcd]*?+$", RegexOptions.None, RegexParseError.NestedQuantify)]
        [InlineData("^[abcd]+?*$", RegexOptions.None, RegexParseError.NestedQuantify)]
        [InlineData("^[abcd]??*$", RegexOptions.None, RegexParseError.NestedQuantify)]
        [InlineData("^[abcd]*{0,5}$", RegexOptions.None, RegexParseError.NestedQuantify)]
        [InlineData("^[abcd]+{0,5}$", RegexOptions.None, RegexParseError.NestedQuantify)]
        [InlineData("^[abcd]?{0,5}$", RegexOptions.None, RegexParseError.NestedQuantify)]
        // Invalid character class
        [InlineData("[", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        [InlineData("[]", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        [InlineData("[a", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        [InlineData("[^", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        [InlineData("[cat", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        [InlineData("[^cat", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        [InlineData("[a-", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        // Invalid grouping constructs
        [InlineData("(?<", RegexOptions.None, RegexParseError.UnrecognizedGrouping)]
        [InlineData("(?<cat>", RegexOptions.None, RegexParseError.NotEnoughParentheses)]
        [InlineData("(?'cat'", RegexOptions.None, RegexParseError.NotEnoughParentheses)]
        [InlineData("(?imn", RegexOptions.None, RegexParseError.UnrecognizedGrouping)]
        [InlineData("(?imn )", RegexOptions.None, RegexParseError.UnrecognizedGrouping)]
        [InlineData("(?>-", RegexOptions.None, RegexParseError.NotEnoughParentheses)]
        [InlineData("(?<)", RegexOptions.None, RegexParseError.InvalidGroupName)]
        [InlineData("(?')", RegexOptions.None, RegexParseError.InvalidGroupName)]
        // Invalid alternation constructs
        [InlineData("(?(", RegexOptions.None, RegexParseError.NotEnoughParentheses)]
        [InlineData("(?()|", RegexOptions.None, RegexParseError.NotEnoughParentheses)]
        [InlineData("(?(cat", RegexOptions.None, RegexParseError.NotEnoughParentheses)]
        [InlineData("(?(cat)|", RegexOptions.None, RegexParseError.NotEnoughParentheses)]
        // Regex with 0 numeric names
        [InlineData("foo(?<0>bar)", RegexOptions.None, RegexParseError.CapnumNotZero)]
        [InlineData("foo(?'0'bar)", RegexOptions.None, RegexParseError.CapnumNotZero)]
        // Regex without closing >
        [InlineData("foo(?<1bar)", RegexOptions.None, RegexParseError.InvalidGroupName)]
        [InlineData("foo(?'1bar)", RegexOptions.None, RegexParseError.InvalidGroupName)]
        // Misc
        [InlineData("(?r:cat)", RegexOptions.None, RegexParseError.UnrecognizedGrouping)]
        [InlineData("(?c:cat)", RegexOptions.None, RegexParseError.UnrecognizedGrouping)]
        [InlineData("(??e:cat)", RegexOptions.None, RegexParseError.UnrecognizedGrouping)]
        // Character class subtraction
        [InlineData("[a-f-[]]+", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        //// Not character class substraction
        [InlineData("[A-[]+", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        // Invalid testgroup
        [InlineData("(?(?e))", RegexOptions.None, RegexParseError.UnrecognizedGrouping)]
        [InlineData("(?(?a)", RegexOptions.None, RegexParseError.UnrecognizedGrouping)]
        [InlineData("(?(?", RegexOptions.None, RegexParseError.UnrecognizedGrouping)]
        [InlineData("?(a)", RegexOptions.None, RegexParseError.QuantifyAfterNothing)]
        [InlineData("?(a|b)", RegexOptions.None, RegexParseError.QuantifyAfterNothing)]
        [InlineData("?((a)", RegexOptions.None, RegexParseError.QuantifyAfterNothing)]
        [InlineData("?((a)a", RegexOptions.None, RegexParseError.QuantifyAfterNothing)]
        [InlineData("?((a)a|", RegexOptions.None, RegexParseError.QuantifyAfterNothing)]
        [InlineData("?((a)a|b", RegexOptions.None, RegexParseError.QuantifyAfterNothing)]
        // Testgroup with options
        [InlineData("())", RegexOptions.None, RegexParseError.TooManyParentheses)]
        [InlineData("[a-z-[aeiuo]", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        [InlineData("[a-z-[aeiuo", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        [InlineData("[a-z-[b]", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        [InlineData("[a-z-[b", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        [InlineData("[b-a]", RegexOptions.None, RegexParseError.ReversedCharRange)]
        [InlineData(@"[a-c]{2,1}", RegexOptions.None, RegexParseError.IllegalRange)]
        [InlineData(@"\d{2147483648}", RegexOptions.None, RegexParseError.CaptureGroupOutOfRange)]
        [InlineData("[a-z-[b][", RegexOptions.None, RegexParseError.UnterminatedBracket)]
        [InlineData("(?()|||||)", RegexOptions.None, RegexParseError.TooManyAlternates)]
        public void Parse(string pattern, RegexOptions options, object errorObj)
        {
            RegexParseError? error = (RegexParseError?)errorObj;

            // Parse the main tree and if parsing fails check if the supplied error matches.
            ParseTree(pattern, options, error);

            // Assert that only ArgumentException might be thrown during parsing.
            ParseSubTrees(pattern, options);
        }

        [Theory]
        // OutOfMemoryException
        [InlineData("a{2147483647}", RegexOptions.None, null)]
        [InlineData("a{2147483647,}", RegexOptions.None, null)]
        [InlineData(@"(?(?N))", RegexOptions.None, RegexParseError.UnrecognizedGrouping)]
        [InlineData(@"(?(?i))", RegexOptions.None, RegexParseError.UnrecognizedGrouping)]
        [InlineData(@"(?(?I))", RegexOptions.None, RegexParseError.UnrecognizedGrouping)]
        [InlineData(@"(?(?M))", RegexOptions.None, RegexParseError.UnrecognizedGrouping)]
        [InlineData(@"(?(?s))", RegexOptions.None, RegexParseError.UnrecognizedGrouping)]
        [InlineData(@"(?(?S))", RegexOptions.None, RegexParseError.UnrecognizedGrouping)]
        [InlineData(@"(?(?x))", RegexOptions.None, RegexParseError.UnrecognizedGrouping)]
        [InlineData(@"(?(?X))", RegexOptions.None, RegexParseError.UnrecognizedGrouping)]
        [InlineData(@"(?(?n))", RegexOptions.None, RegexParseError.UnrecognizedGrouping)]
        [InlineData(" (?(?n))", RegexOptions.None, RegexParseError.UnrecognizedGrouping)]
        [InlineData(@"(?(?m))", RegexOptions.None, RegexParseError.UnrecognizedGrouping)]
        // IndexOutOfRangeException
        [InlineData("(?<-", RegexOptions.None, RegexParseError.UnrecognizedGrouping)]
        [InlineData("(?<-", RegexOptions.IgnorePatternWhitespace, RegexParseError.UnrecognizedGrouping)]
        [InlineData(@"^[^<>]*(((?'Open'<)[^<>]*)+((?'Close-Open'>)[^<>]*)+)*(?(Open)(?!))$", RegexOptions.None, null)]
        [InlineData(@"((?'Close-Open'>)[^<>]*)+", RegexOptions.None, RegexParseError.UndefinedNameRef)]
        [InlineData(@"(((?'Open'<)[^<>]*)+((?'Close-Open'>)[^<>]*)+)*", RegexOptions.None, null)]
        [InlineData(@"(?'Close-Open'>)", RegexOptions.None, RegexParseError.UndefinedNameRef)]
        [InlineData("(?<a-00>)", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?<a>)()(?<-0>)(?<-1>)(?<-2>)(?<-3>)", RegexOptions.IgnorePatternWhitespace, RegexParseError.UndefinedBackref)]
        [InlineData("()(?<a>)(?<-0>)(?<-1>)(?<-2>)(?<-3>)", RegexOptions.IgnorePatternWhitespace, RegexParseError.UndefinedBackref)]
        [InlineData("()()(?<-0>)(?<-1>)(?<-2>)(?<-3>)", RegexOptions.IgnorePatternWhitespace, RegexParseError.UndefinedBackref)]
        [InlineData("(?<a>)(?<b>)(?<-1>)(?<-2>)", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?<-4>)(?<4>)", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?<4>)(?<-4>)", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?<a>)(?<-a>)", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?<-a>)(?<a>)", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?<a-0>", RegexOptions.IgnorePatternWhitespace, RegexParseError.NotEnoughParentheses)]
        [InlineData("(?<a-0>)", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?<a-0 >)", RegexOptions.IgnorePatternWhitespace, RegexParseError.InvalidGroupName)]
        [InlineData("(?<a- 0 >)", RegexOptions.IgnorePatternWhitespace, RegexParseError.InvalidGroupName)]
        [InlineData("(?<a- 0>)", RegexOptions.IgnorePatternWhitespace, RegexParseError.InvalidGroupName)]
        [InlineData("(?<-1>)", RegexOptions.IgnorePatternWhitespace, RegexParseError.UndefinedBackref)]
        [InlineData("()(?<-1>)", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?<-1>)()", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?<-00>)", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?<a-", RegexOptions.IgnorePatternWhitespace, RegexParseError.UnrecognizedGrouping)]
        [InlineData("(?<a-0", RegexOptions.IgnorePatternWhitespace, RegexParseError.UnrecognizedGrouping)]
        [InlineData("(?<a-0)", RegexOptions.IgnorePatternWhitespace, RegexParseError.InvalidGroupName)]
        [InlineData("(?<a>)(?<b>)(?<-0>)(?<-1>)(?<-2>)(?<-3>)", RegexOptions.IgnorePatternWhitespace, RegexParseError.UndefinedBackref)]
        [InlineData("(?<-0>)(?<-1>)(?<-2>)(?<-3>)()()", RegexOptions.IgnorePatternWhitespace, RegexParseError.UndefinedBackref)]
        [InlineData("(?<-0>)(?<-1>)(?<-2>)(?<-3>)()(?", RegexOptions.IgnorePatternWhitespace, RegexParseError.UndefinedBackref)]
        [InlineData("(?<-0>)(?<-1>)(?<-2>)(?<-3>)()(?<a>)", RegexOptions.IgnorePatternWhitespace, RegexParseError.UndefinedBackref)]
        [InlineData("(?<-0>)(?<-1>)(?<-2>)(?<-3>)(?<a>)()", RegexOptions.IgnorePatternWhitespace, RegexParseError.UndefinedBackref)]
        [InlineData("(?<-0>)(?<-1>)(?<-2>)(?<-3>)(?<a>)(?", RegexOptions.IgnorePatternWhitespace, RegexParseError.UndefinedBackref)]
        [InlineData("(?<-0>)(?<-1>)(?<-2>)(?<-3>)(?<a>)(?<b>)", RegexOptions.IgnorePatternWhitespace, RegexParseError.UndefinedBackref)]
        [InlineData("(?<a-0>)(?<b-a>)", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?<a-0>)(?<-a>)", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?<a-a>)", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?<-0>)", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?<-0 >)", RegexOptions.IgnorePatternWhitespace, RegexParseError.InvalidGroupName)]
        [InlineData("(?<- 0 >)", RegexOptions.IgnorePatternWhitespace, RegexParseError.InvalidGroupName)]
        [InlineData("(?<- 0>)", RegexOptions.IgnorePatternWhitespace, RegexParseError.InvalidGroupName)]
        [InlineData("(?<a-0')", RegexOptions.IgnorePatternWhitespace, RegexParseError.InvalidGroupName)]
        [InlineData("(?'a-0>)", RegexOptions.IgnorePatternWhitespace, RegexParseError.InvalidGroupName)]
        [InlineData("(?'-0')", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?'a-0')", RegexOptions.IgnorePatternWhitespace, null)]
        [InlineData("(?<-0", RegexOptions.IgnorePatternWhitespace, RegexParseError.UnrecognizedGrouping)]
        [InlineData("(?<-0)", RegexOptions.IgnorePatternWhitespace, RegexParseError.InvalidGroupName)]
        [InlineData("(?<-0>", RegexOptions.IgnorePatternWhitespace, RegexParseError.NotEnoughParentheses)]
        [InlineData(@"(?<cat>cat)\w+(?<dog-()*!@>dog)", RegexOptions.None, RegexParseError.InvalidGroupName)]
        [InlineData(@"(?<cat>cat)\w+(?<dog-catdog>dog)", RegexOptions.None, RegexParseError.UndefinedNameRef)]
        [InlineData(@"(?<cat>cat)\w+(?<dog-1uosn>dog)", RegexOptions.None, RegexParseError.InvalidGroupName)]
        [InlineData(@"(?<cat>cat)\w+(?<dog-16>dog)", RegexOptions.None, RegexParseError.UndefinedBackref)]
        [InlineData(@"cat(?<->dog)", RegexOptions.None, RegexParseError.InvalidGroupName)]
        [InlineData("a{2147483648}", RegexOptions.None, RegexParseError.CaptureGroupOutOfRange)]
        [InlineData("a{2147483648,}", RegexOptions.None, RegexParseError.CaptureGroupOutOfRange)]
        [InlineData("a{0,2147483647}", RegexOptions.None, null)]
        [InlineData("a{0,2147483648}", RegexOptions.None, RegexParseError.CaptureGroupOutOfRange)]
        // Surrogate pair which is parsed as [char,char-char,char] as we operate on UTF-16 code units.
        [InlineData("[\uD82F\uDCA0-\uD82F\uDCA3]", RegexOptions.IgnoreCase, RegexParseError.ReversedCharRange)]
        public void Parse_NotNetFramework(string pattern, RegexOptions options, object error)
        {
            Parse(pattern, options, error);
        }

        private static void ParseSubTrees(string pattern, RegexOptions options)
        {
            // Trim the input from the right and make sure tree invariants hold
            for (int i = pattern.Length - 1; i > 0; i--)
            {
                ParseSubTree(pattern.Substring(0, i), options);
            }

            // Trim the input from the left and make sure tree invariants hold
            for (int i = 1; i < pattern.Length; i++)
            {
                ParseSubTree(pattern.Substring(i), options);
            }

            // Skip middles
            for (int i = 1; i < pattern.Length; i++)
            {
                ParseSubTree(pattern.Substring(0, i) + pattern.Substring(i + 1), options);
            }
        }

        private static void ParseTree(string pattern, RegexOptions options, RegexParseError? error)
        {
            if (error != null)
            {
                Throws(error.Value, () => new Regex(pattern, options));
                return;
            }

            // Nothing to assert here without having access to internals.
            new Regex(pattern, options);
        }

        private static void ParseSubTree(string pattern, RegexOptions options)
        {
            try
            {
                new Regex(pattern, options);
            }
            catch (ArgumentException)
            {
                // We are fine with ArgumentExceptions being thrown during sub expression parsing.
            }
        }

        /// <summary>
        /// Checks if action throws either a RegexParseException or an ArgumentException depending on the
        /// environment and the supplied error.
        /// </summary>
        /// <param name="error">The expected parse error</param>
        /// <param name="action">The action to invoke.</param>
        private static void Throws(RegexParseError error, Action action)
        {
            try
            {
                action();
            }
            catch (Exception e)
            {
                // We use reflection to check if the exception is an internal RegexParseException
                // and extract its error property and compare with the given one.
                if (e.GetType() == s_parseExceptionType)
                {
                    RegexParseError regexParseError = (RegexParseError)s_parseErrorField.GetValue(e);

                    // Success if provided error matches.
                    if (error == regexParseError)
                        return;

                    throw new XunitException($"Expected RegexParseException with error: ({error}) -> Actual error: {regexParseError})");
                }

                throw new XunitException($"Expected RegexParseException -> Actual: ({e.GetType()})");
            }

            throw new XunitException($"Expected RegexParseException with error: ({error}) -> Actual: No exception thrown");
        }
    }
}
