// #Conformance #Printing 

#if TESTS_AS_APP
module Core_printf
#endif

#light

open Printf

let failures = ref []

let report_failure (s : string) = 
    stderr.Write" NO: "
    stderr.WriteLine s
    failures := !failures @ [s]

// change this to true to run every test case
// leave as false to randomly execute a subset of cases (this is a very expensive test area)
let runEveryTest = false
if runEveryTest then stdout.WriteLine "Running every test. Set \"runEveryTest\" to false to reduce runtime"
else stdout.WriteLine "Running a random subset of tests.  Set \"runEveryTest\" to true to run all tests"

let rnd = System.Random()

let test t (s1:Lazy<string>) s2 = 
  if runEveryTest || (rnd.Next() % 10) = 0 then
      let s1 = s1.Force()
      if s1 <> s2 then 
        report_failure ("test "+t+": expected \n\t'"+s2+"' but produced \n\t'"+s1+"'")
      else
        stdout.WriteLine ("test "+t+": correctly produced '"+s1+"'")   

let verify actual expected = test expected actual expected

let adjust1 obj n1 = unbox ((unbox obj) n1)

(*
let _ = test "percent00" (lazy(sprintf "%%")) "%"
let _ = test "percent01" (lazy(sprintf " %%%% ")) " %% "
let _ = test "percent02" (lazy(sprintf "%.2f%.2%" 2.)) "2.00%"
let _ = test "percent03" (lazy(sprintf "   %2%%%%+-%")) "   %%%"
let _ = test "percent04" (lazy(sprintf "%.*%" 2)) "%"
let _ = test "percent05" (lazy(sprintf "%*% %*d" 20 10 5)) "%          5"
let _ = test "percent06" (lazy(sprintf "%*% %*d" 20 8 5)) "%        5"
let _ = test "percent07" (lazy(sprintf "%-+.*%%*d%*.*%" 55 0 8 77 88)) "%8%"
let _ = test "percent08" (lazy(sprintf "%%d")) "%d"
let _ = test "percent09" (lazy(sprintf "% *% %d" 10 6)) "% 6"
*)

let _ = test "cewoui2a" (lazy(sprintf "%o" 0)) "0"
let _ = test "cewoui2b" (lazy(sprintf "%o" 0)) "0"
let _ = test "cewoui2c" (lazy(sprintf "%o" 5)) "5"
let _ = test "cewoui2q" (lazy(sprintf "%o" 8)) "10"
let _ = test "cewoui2w" (lazy(sprintf "%o" 15)) "17"
let _ = test "cewoui2e" (lazy(sprintf "%o" System.Int32.MinValue)) "20000000000"
let _ = test "cewoui2r" (lazy(sprintf "%o" 0xffffffff)) "37777777777"
let _ = test "cewoui2t" (lazy(sprintf "%o" (System.Int32.MinValue+1))) "20000000001"
let _ = test "cewoui2y" (lazy(sprintf "%o" System.Int32.MaxValue)) "17777777777"

let _ = test "cewoui2u" (lazy(sprintf "%o" (-1))) "37777777777"

let f = sprintf "%o"

let _ = test "cewoui2a" (lazy(f 0)) "0"
let _ = test "cewoui2s" (lazy(f 0)) "0"
let _ = test "cewoui2d" (lazy(f 5)) "5"
let _ = test "cewoui2f" (lazy(f 8)) "10"
let _ = test "cewoui2g" (lazy(f 15)) "17"
let _ = test "cewoui2h" (lazy(f System.Int32.MinValue)) "20000000000"
let _ = test "cewoui2j" (lazy(f 0xffffffff)) "37777777777"
let _ = test "cewoui2lk" (lazy(f (System.Int32.MinValue+1))) "20000000001"
let _ = test "cewoui2l" (lazy(f System.Int32.MaxValue)) "17777777777"

let _ = test "cewoui212" (lazy(f (-1))) "37777777777"

// Test nothing comes out until all arguments have been applied
let _ = test "csd3oui2!" (lazy(let buf = new System.Text.StringBuilder() in ignore (bprintf buf "%x%x" 0); buf.ToString())) ""
let _ = test "csd3oui2!" (lazy(let buf = new System.Text.StringBuilder() in ignore (bprintf buf "%x%x" 0 1); buf.ToString())) "01"
let _ = test "csd3oui2!" (lazy(let buf = new System.Text.StringBuilder() in ignore (bprintf buf "%s"); buf.ToString())) ""
let _ = test "csd3oui2!" (lazy(let buf = new System.Text.StringBuilder() in ignore (bprintf buf "%s" "abc"); buf.ToString())) "abc"

let _ = test "cewoui2!" (lazy(sprintf "%x" 0)) "0"
let _ = test "cewoui26" (lazy(sprintf "%x" 5)) "5"
let _ = test "cewoui2f" (lazy(sprintf "%x" 8)) "8"
let _ = test "cewoui29" (lazy(sprintf "%x" 15)) "f"
let _ = test "cewoui2Z" (lazy(sprintf "%x" System.Int32.MinValue)) "80000000"
let _ = test "cewoui2X" (lazy(sprintf "%x" 0xffffffff)) "ffffffff"
let _ = test "cewoui2A" (lazy(sprintf "%x" (System.Int32.MinValue+1))) "80000001"
let _ = test "cewoui2Q" (lazy(sprintf "%x" System.Int32.MaxValue)) "7fffffff"

let fx = sprintf "%x"
let _ = test "cewoui2W" (lazy(fx 0)) "0"
let _ = test "cewoui2E" (lazy(fx 5)) "5"
let _ = test "cewoui2R" (lazy(fx 8)) "8"
let _ = test "cewoui2T" (lazy(fx 15)) "f"
let _ = test "cewoui2Y" (lazy(fx System.Int32.MinValue)) "80000000"
let _ = test "cewoui2U" (lazy(fx 0xffffffff)) "ffffffff"
let _ = test "cewoui2I" (lazy(fx (System.Int32.MinValue+1))) "80000001"
let _ = test "cewoui2O" (lazy(fx System.Int32.MaxValue)) "7fffffff"

let _ = test "cewoui2Z" (lazy(sprintf "%X" 0)) "0"
let _ = test "cewoui2X" (lazy(sprintf "%X" 5)) "5"
let _ = test "cewoui2C" (lazy(sprintf "%X" 8)) "8"
let _ = test "cewoui2V" (lazy(sprintf "%X" 15)) "F"
let _ = test "cewoui2v" (lazy(sprintf "%X" System.Int32.MinValue)) "80000000"
let _ = test "cewoui2B" (lazy(sprintf "%X" 0xffffffff)) "FFFFFFFF"
let _ = test "cewoui2N" (lazy(sprintf "%X" (System.Int32.MinValue+1))) "80000001"
let _ = test "cewoui2M" (lazy(sprintf "%X" System.Int32.MaxValue)) "7FFFFFFF"

let _ = test "cewou44a" (lazy(sprintf "%u" 0)) "0"
let _ = test "cewou44b" (lazy(sprintf "%u" 5)) "5"
let _ = test "cewou44c" (lazy(sprintf "%u" 8)) "8"
let _ = test "cewou44d" (lazy(sprintf "%u" 15)) "15"
let _ = test "cewou44e" (lazy(sprintf "%u" System.Int32.MinValue)) "2147483648"
let _ = test "cewou44f" (lazy(sprintf "%u" 0xffffffff)) "4294967295"
let _ = test "cewou44g" (lazy(sprintf "%u" (System.Int32.MinValue+1))) "2147483649"
let _ = test "cewou44h" (lazy(sprintf "%u" System.Int32.MaxValue)) "2147483647"

let _ = test "cewou45a" (lazy(sprintf "%d" 0ul)) "0"
let _ = test "cewou45b" (lazy(sprintf "%d" 5ul)) "5"
let _ = test "cewou45c" (lazy(sprintf "%d" 8ul)) "8"
let _ = test "cewou45d" (lazy(sprintf "%d" 15ul)) "15"
let _ = test "cewou45e" (lazy(sprintf "%d" 2147483648ul)) "2147483648"
let _ = test "cewou45f" (lazy(sprintf "%d" 4294967295ul)) "4294967295"
let _ = test "cewou45g" (lazy(sprintf "%d" 2147483649ul)) "2147483649"
let _ = test "cewou45h" (lazy(sprintf "%d" 2147483647ul)) "2147483647"

let _ = test "cewou46a" (lazy(sprintf "%d" 0ul)) "0"
let _ = test "cewou46b" (lazy(sprintf "%d" 5ul)) "5"
let _ = test "cewou46c" (lazy(sprintf "%d" 8ul)) "8"
let _ = test "cewou46d" (lazy(sprintf "%d" 15ul)) "15"
let _ = test "cewou46e" (lazy(sprintf "%d" 2147483648ul)) "2147483648"
let _ = test "cewou46f" (lazy(sprintf "%d" 4294967295ul)) "4294967295"
let _ = test "cewou46g" (lazy(sprintf "%d" 2147483649ul)) "2147483649"
let _ = test "cewou46h" (lazy(sprintf "%d" 2147483647ul)) "2147483647"

let _ = test "ceew903" (lazy(sprintf "%u" System.Int64.MaxValue)) "9223372036854775807"
let _ = test "ceew903" (lazy(sprintf "%u" System.Int64.MinValue)) "9223372036854775808"
let _ = test "ceew903" (lazy(sprintf "%d" System.Int64.MaxValue)) "9223372036854775807"
let _ = test "ceew903" (lazy(sprintf "%d" System.Int64.MinValue)) "-9223372036854775808"

let _ = test "ceew903" (lazy(sprintf "%u" System.Int64.MaxValue)) "9223372036854775807"
let _ = test "ceew903" (lazy(sprintf "%u" System.Int64.MinValue)) "9223372036854775808"
let _ = test "ceew903" (lazy(sprintf "%d" System.Int64.MaxValue)) "9223372036854775807"
let _ = test "ceew903" (lazy(sprintf "%d" System.Int64.MinValue)) "-9223372036854775808"

let _ = test "cewou47a" (lazy(sprintf "%d" 0n)) "0"
let _ = test "cewou47b" (lazy(sprintf "%d" 5n)) "5"
let _ = test "cewou47c" (lazy(sprintf "%d" 8n)) "8"
let _ = test "cewou47d" (lazy(sprintf "%d" 15n)) "15"
let _ = test "cewou47e" (lazy(sprintf "%u" 2147483648n)) "2147483648"
let _ = test "cewou47f" (lazy(sprintf "%u" 4294967295n)) "4294967295"
let _ = test "cewou47g" (lazy(sprintf "%u" 2147483649n)) "2147483649"
let _ = test "cewou47h" (lazy(sprintf "%u" 2147483647n)) "2147483647"

let _ = test "cewou47a" (lazy(sprintf "%d" 0n)) "0"
let _ = test "cewou47b" (lazy(sprintf "%d" 5n)) "5"
let _ = test "cewou47c" (lazy(sprintf "%d" 8n)) "8"
let _ = test "cewou47d" (lazy(sprintf "%d" 15n)) "15"
let _ = test "cewou47e" (lazy(sprintf "%u" 2147483648n)) "2147483648"
let _ = test "cewou47f" (lazy(sprintf "%u" 4294967295n)) "4294967295"
let _ = test "cewou47g" (lazy(sprintf "%u" 2147483649n)) "2147483649"
let _ = test "cewou47h" (lazy(sprintf "%u" 2147483647n)) "2147483647"

let _ = test "cewou48a" (lazy(sprintf "%d" 0un)) "0"
let _ = test "cewou48b" (lazy(sprintf "%d" 5un)) "5"
let _ = test "cewou48c" (lazy(sprintf "%d" 8un)) "8"
let _ = test "cewou48d" (lazy(sprintf "%d" 15un)) "15"
let _ = test "cewou48e" (lazy(sprintf "%u" 2147483648un)) "2147483648"
let _ = test "cewou48f" (lazy(sprintf "%u" 4294967295un)) "4294967295"
let _ = test "cewou48g" (lazy(sprintf "%u" 2147483649un)) "2147483649"
let _ = test "cewou48h" (lazy(sprintf "%u" 2147483647un)) "2147483647"

let _ = test "cewou59a" (lazy(sprintf "%d" 0un)) "0"
let _ = test "cewou59b" (lazy(sprintf "%d" 5un)) "5"
let _ = test "cewou59c" (lazy(sprintf "%d" 8un)) "8"
let _ = test "cewou59d" (lazy(sprintf "%d" 15un)) "15"
let _ = test "cewou59e" (lazy(sprintf "%u" 2147483648un)) "2147483648"
let _ = test "cewou59f" (lazy(sprintf "%u" 4294967295un)) "4294967295"
let _ = test "cewou59g" (lazy(sprintf "%u" 2147483649un)) "2147483649"
let _ = test "cewou59h" (lazy(sprintf "%u" 2147483647un)) "2147483647"

let _ = test "cewou49a" (lazy(sprintf "%d" 0)) "0"
let _ = test "cewou49b" (lazy(sprintf "%d" 5)) "5"
let _ = test "cewou49c" (lazy(sprintf "%+d" 5)) "+5"
let _ = test "cewou49d" (lazy(sprintf "% d" 5)) " 5"
let _ = test "cewou49e" (lazy(sprintf "%+4d" 5)) "  +5"
let _ = test "cewou49f" (lazy(sprintf "%-+4d" 5)) "+5  "
let _ = test "cewou49g" (lazy(sprintf "%-4d" 5)) "5   "
let _ = test "cewou49h" (lazy(sprintf "%- 4d" 5)) " 5  "
let _ = test "cewou49i" (lazy(sprintf "%- d" 5)) " 5"
let _ = test "cewou49j" (lazy(sprintf "% d" 5)) " 5"
let _ = test "weioj31" (lazy(sprintf "%*d" 3 5)) "  5"
let _ = test "weioj31" (lazy(sprintf "%3d" 5)) "  5"
let _ = test "weioj32" (lazy(sprintf "%1d" 5)) "5"
let _ = test "weioj32" (lazy(sprintf "%*d" 1 5)) "5"
let _ = test "weioj33" (lazy(sprintf "%2d" 500)) "500"
let _ = test "weioj33" (lazy(sprintf "%*d" 2 500)) "500"
let _ = test "weioj34" (lazy(sprintf "%3d" 500)) "500"
let _ = test "weioj34" (lazy(sprintf "%*d" 3 500)) "500"
let _ = test "weioj35" (lazy(sprintf "%d" 501)) "501"
let _ = test "weioj36" (lazy(sprintf "%2d" (-4))) "-4"
let _ = test "weioj36" (lazy(sprintf "%*d" 2 (-4))) "-4"
let _ = test "weioj37" (lazy(sprintf "%1d" (-4))) "-4"
let _ = test "weioj37" (lazy(sprintf "%*d" 1 (-4))) "-4"
let _ = test "weioj38" (lazy(sprintf "%d" (-401))) "-401"
let _ = test "weioj39" (lazy(sprintf "%d" 2147483647)) "2147483647"
let _ = test "weioj3a" (lazy(sprintf "%d" (-2147483647))) "-2147483647"
let _ = test "weioj3s" (lazy(sprintf "%d" (-2147483648))) "-2147483648"


let _ = test "weioj3d" (lazy(sprintf "print test %O with suffix" 1)) "print test 1 with suffix"
let _ = test "weioj3f" (lazy(sprintf "print test %O %O with suffix" 1 "xyz")) "print test 1 xyz with suffix"
let _ = test "weioj3g" (lazy(sprintf "print test %M with suffix" (System.Convert.ToDecimal(3)))) "print test 3 with suffix"
let _ = test "weioj3h" (lazy(sprintf "print test %M with suffix" (System.Convert.ToDecimal(3.02)))) "print test 3.02 with suffix"

let _ = test "weioj3j" (lazy(sprintf "%O" 3I)) "3"


let _ = test "weiodasj3" (lazy(sprintf "%f" 0.0)) "0.000000"
let _ = test "weiogwej3" (lazy(sprintf "%10f" 0.0)) "  0.000000"
let _ = test "weiogwej3" (lazy(sprintf "%*f" 10 0.0)) "  0.000000"
let _ = test "weiobtj3" (lazy(sprintf "%7f" 0.0)) "0.000000"
let _ = test "weiobtj3" (lazy(sprintf "%*f" 7 0.0)) "0.000000"
let _ = test "weiorwej3" (lazy(sprintf "%7.1f" 0.0)) "    0.0"
let _ = test "weiorwej3" (lazy(sprintf "%*.1f" 7 0.0)) "    0.0"
let _ = test "weiorwej3" (lazy(sprintf "%7.*f" 1 0.0)) "    0.0"
let _ = test "weiorwej3" (lazy(sprintf "%*.*f" 7 1 0.0)) "    0.0"
let _ = test "weivewoj3" (lazy(sprintf "%7.2f" 0.0)) "   0.00"
let _ = test "weivewoj3" (lazy(sprintf "%*.2f" 7 0.0)) "   0.00"
let _ = test "weivewoj3" (lazy(sprintf "%*.*f" 7 2 0.0)) "   0.00"
let _ = test "weivewoj3" (lazy(sprintf "%7.*f" 2 0.0)) "   0.00"
let _ = test "weiqfoj3" (lazy(sprintf "%7.0f" 0.0)) "      0"
let _ = test "weiqfoj3" (lazy(sprintf "%*.0f" 7 0.0)) "      0"
let _ = test "weiqfoj3" (lazy(sprintf "%7.*f" 0 0.0)) "      0"
let _ = test "weiqfoj3" (lazy(sprintf "%*.*f" 7 0 0.0)) "      0"
let _ = test "weieroj3" (lazy(sprintf "%10.2e" 1.0)) " 1.00e+000"
let _ = test "weieroj3" (lazy(sprintf "%*.2e" 10 1.0)) " 1.00e+000"
let _ = test "weio34j3" (lazy(sprintf "%10.2E" 1.0)) " 1.00E+000"
let _ = test "weio34j3" (lazy(sprintf "%10.*E" 2 1.0)) " 1.00E+000"
let _ = test "weiberoj3" (lazy(sprintf "%10.3E" 1.0)) "1.000E+000"
let _ = test "weiberoj3" (lazy(sprintf "%10.*E" 3 1.0)) "1.000E+000"
let _ = test "weiqfwoj3" (lazy(sprintf "%10g" 1.0)) "         1"
let _ = test "weiqfwoj3" (lazy(sprintf "%*g" 10 1.0)) "         1"
let _ = test "weiof33j3" (lazy(sprintf "%10g" 1.01)) "      1.01"
let _ = test "weiof33j3" (lazy(sprintf "%*g" 10 1.01)) "      1.01"
let _ = test "wei54goj3" (lazy(sprintf "%-10g" 1.01)) "1.01      "
let _ = test "wei54goj3" (lazy(sprintf "%-*g" 10 1.01)) "1.01      "
let _ = test "weioqf3j3" (lazy(sprintf "%g" 1.01)) "1.01"
(* NEG: let _ = test "weioqf3j3" (lazy(sprintf "%g" 1)) "1.01" *)


let _ = test "wekodasj3" (lazy(sprintf "%f" 0.0f)) "0.000000"
let _ = test "wekogwej3" (lazy(sprintf "%10f" 0.0f)) "  0.000000"
let _ = test "wekobtj3" (lazy(sprintf "%7f" 0.0f)) "0.000000"
let _ = test "wekorwej3" (lazy(sprintf "%7.1f" 0.0f)) "    0.0"
let _ = test "wekvewoj3" (lazy(sprintf "%7.2f" 0.0f)) "   0.00"
let _ = test "wekqfoj3" (lazy(sprintf "%7.0f" 0.0f)) "      0"
let _ = test "wekeroj3" (lazy(sprintf "%10.2e" 1.0f)) " 1.00e+000"
let _ = test "weko34j3" (lazy(sprintf "%10.2E" 1.0f)) " 1.00E+000"
let _ = test "wekberoj3" (lazy(sprintf "%10.3E" 1.0f)) "1.000E+000"
let _ = test "wekqfwoj3" (lazy(sprintf "%10g" 1.0f)) "         1"
let _ = test "wekof33j3" (lazy(sprintf "%10g" 1.01f)) "      1.01"
let _ = test "wek54goj3" (lazy(sprintf "%-10g" 1.01f)) "1.01      "
let _ = test "wekoqf3j3" (lazy(sprintf "%g" 1.01f)) "1.01"


let _ = test "weioj3Q" (lazy(sprintf "%a" (fun () -> string) 10)) "10"
let _ = test "weioj3W" (lazy(sprintf "%a%a" (fun () s -> s+s) "a" (fun () s -> s+s) "b")) "aabb"
(* NEG: let _ = test "weioj3" (lazy(sprintf "%a" (fun () -> string_of_int) "a")) "10" *)

let _ = test "weioj3ff" (lazy(try failwithf "%a%a" (fun () s -> s+s) "a" (fun () s -> s+s) "b" with Failure s -> s)) "aabb"
let _ = test "weioj3ffdd" (lazy(string (try if true then failwithf "%s" "abc" else 1 with Failure "abc" -> 2))) "2"
let _ = test "weioj3ffd2" (lazy(try if true then failwithf "%s" "abc" else "d"with Failure "abc" -> "e")) "e"

let _ = test "weioj3" (lazy(sprintf "%t" (fun () -> "10"))) "10"
  
let bug600 = sprintf "%d"
let _ = test "bug600a1" (lazy(bug600 2)) "2" 
let _ = test "bug600b1" (lazy(bug600 2)) "2" (* not 22! *)

let bug600b = sprintf "%s"
let _ = test "bug600a2" (lazy(bug600b "2")) "2" 
let _ = test "bug600b2" (lazy(bug600b "2")) "2" (* not 22! *)

let bug600c = sprintf "%x"
let _ = test "bug600a3" (lazy(bug600c 2)) "2" 
let _ = test "bug600b3" (lazy(bug600c 2)) "2" (* not 22! *)

let _ = test "ckwoih" (lazy(sprintf "%x" 0xFFy)) ("ff")
let _ = test "ckwoih" (lazy(sprintf "%x" 0xFFFFs)) ("ffff")
let _ = test "ckwoih" (lazy(sprintf "%x" 0xFFFFFFFF)) ("ffffffff")
let _ = test "ckwoih" (lazy(sprintf "%x" 0xFFFFFFFFFFFFFFFFL)) ("ffffffffffffffff")
let _ = test "ckwoih" (lazy(sprintf "%x" 0xFFFFFFFFn)) ("ffffffff")

let _ = test "ckwoih" (lazy(sprintf "%x" 0xFFuy)) ("ff")
let _ = test "ckwoih" (lazy(sprintf "%x" 0xFFFFus)) ("ffff")
let _ = test "ckwoih" (lazy(sprintf "%x" 0xFFFFFFFFu)) ("ffffffff")
let _ = test "ckwoih" (lazy(sprintf "%x" 0xFFFFFFFFFFFFFFFFUL)) ("ffffffffffffffff")
let _ = test "ckwoih" (lazy(sprintf "%x" 0xFFFFFFFFun)) ("ffffffff")

// Check one with a suffix
module CheckDisplayAttributes1 =
    [<StructuredFormatDisplay("{StructuredDisplay}N")>]
    type Foo() = 
       member x.StructuredDisplay = 3

    test "cenwoiwe1" (lazy(sprintf "%A" (Foo()))) "3N"

// Check one with a prefix
module CheckDisplayAttributes2 =

    [<StructuredFormatDisplay("N{StructuredDisplay}")>]
    type Foo() = 
       member x.StructuredDisplay = 3

    test "cenwoiwe2" (lazy(sprintf "%A" (Foo()))) "N3"
    
// Check one with a prefix returning a string
module CheckDisplayAttributes3 =

    [<StructuredFormatDisplay("N{StructuredDisplay}")>]
    type Foo() = 
       member x.StructuredDisplay = "3"

    test "cenwoiwe3" (lazy(sprintf "%A" (Foo()))) "N3"

// Check one returning a string
module CheckDisplayAttributes4 =

    [<StructuredFormatDisplay("{StructuredDisplay}")>]
    type Foo() = 
       member x.StructuredDisplay = "3"

    test "cenwoiwe4" (lazy(sprintf "%A" (Foo()))) "3"
    
// Check one with an internal property
module CheckDisplayAttributes5 =

    [<StructuredFormatDisplay("{StructuredDisplay}")>]
    type Foo() = 
       member internal x.StructuredDisplay = "3"

    test "cenwoiwe5" (lazy(sprintf "%A" (Foo()))) "3"
    
// Check one with spaces
module CheckDisplayAttributes6 =

    [<StructuredFormatDisplay("  {StructuredDisplay}  ")>]
    type Foo() = 
       member internal x.StructuredDisplay = "3"

    test "cenwoiwe6" (lazy(sprintf "%A" (Foo()))) "  3  "
    
// Check an ill-formed StructuredFormatDisplay string is not shown
module CheckDisplayAttributes7 =

    [<StructuredFormatDisplay("{StructuredDisplay")>]
    type Foo() = 
       member internal x.StructuredDisplay = "3"
       override x.ToString() = "2"

    test "cenwoiwe7" (lazy(sprintf "%A" (Foo()))) "2"
    
// Check one returning a list
module CheckDisplayAttributes8 =

    [<StructuredFormatDisplay("{StructuredDisplay}")>]
    type Foo() = 
       member internal x.StructuredDisplay = [1;2]
       override x.ToString() = "2"

    test "cenwoiwe8" (lazy(sprintf "%A" (Foo()))) "[1; 2]"

// Check one returning two strings
module CheckDisplayAttributes9 =

    [<StructuredFormatDisplay("{Hello} {World}")>]
    type Foo() = 
       member internal x.Hello = "Hello"
       member internal x.World = "World"

    test "cenwoiwe9" (lazy(sprintf "%A" (Foo()))) "Hello World"

// Check one returning an int and a list
module CheckDisplayAttributes10 =

    [<StructuredFormatDisplay("{Hello}: {StructuredDisplay}")>]
    type Foo() = 
       member internal x.StructuredDisplay = [1;2]
       member internal x.Hello = "Hello"
       override x.ToString() = "2"

    test "cenwoiwe10" (lazy(sprintf "%A" (Foo()))) "Hello: [1; 2]"

// Check one returning an int and a string with no spaces
module CheckDisplayAttributes11 =

    [<StructuredFormatDisplay("{Val}{Hello}")>]
    type Foo() = 
       member internal x.Val = 42
       member internal x.Hello = "Hello"

    test "cenwoiwe11" (lazy(sprintf "%A" (Foo()))) "42Hello"

// Check one with an unmatched opening bracket
module CheckDisplayAttributes12 =

    [<StructuredFormatDisplay("{Val{Hello}")>]
    type Foo() = 
       member internal x.Val = 42
       member internal x.Hello = "Hello"
       override x.ToString() = "x"

    // this should produce an error
    test "cenwoiwe12" (lazy(sprintf "%A" (Foo()))) "<StructuredFormatDisplay exception: Method 'Test+CheckDisplayAttributes12+Foo.Val{Hello' not found.>"

// Check one with an unmatched closing bracket
module CheckDisplayAttributes13 =

    [<StructuredFormatDisplay("{Val}Hello}")>]
    type Foo() = 
       member internal x.Val = 42
       member internal x.Hello = "Hello"
       override x.ToString() = "x"

    test "cenwoiwe13" (lazy(sprintf "%A" (Foo()))) "x"

// Check one with an unmatched trailing open bracket
module CheckDisplayAttributes14 =

    [<StructuredFormatDisplay("{Val}{Hello")>]
    type Foo() = 
       member internal x.Val = 42
       member internal x.Hello = "Hello"
       override x.ToString() = "x"

    test "cenwoiwe14" (lazy(sprintf "%A" (Foo()))) "x"

// Check one with unbounded recursion
module CheckDisplayAttributes15 =

    [<StructuredFormatDisplay("{X} {X}")>]
    type Foo() = 
       member internal x.X = Foo()

    test "cenwoiwe15" (lazy(sprintf "%A" (Foo()))) "... ... ... ... ... ... ... ..."

// Check escaped brackets with no other members
module CheckDisplayAttributes16 =

    [<StructuredFormatDisplay("{\{\}}")>]
    type Foo() = 
      member __.``{}`` = "abc"

    test "cenwoiwe16" (lazy(sprintf "%A" (Foo()))) "abc"

// Check escaped brackets with other members
module CheckDisplayAttributes17 =

    [<StructuredFormatDisplay("{One\} \{Two}")>]
    type Foo() =
      member __.``One} {Two`` = "abc"
      member __.One = 123
      member __.Two = 456

    test "cenwoiwe17" (lazy(sprintf "%A" (Foo()))) "abc"

// Check escaped brackets with all brackets escaped
module CheckDisplayAttributes18 =

    [<StructuredFormatDisplay("\{One\} \{Two\}")>]
    type Foo() =
      member __.``One} {Two`` = "abc"
      member __.One = 123
      member __.Two = 456
      override x.ToString() = "x"

    test "cenwoiwe18" (lazy(sprintf "%A" (Foo()))) "{One} {Two}"

// Check escaped brackets with opening bracket escaped, invalidating property reference
module CheckDisplayAttributes19 =

    [<StructuredFormatDisplay("\{One\} \{Two}")>]
    type Foo() =
      member __.``One} {Two`` = "abc"
      member __.One = 123
      member __.Two = 456
      override x.ToString() = "x"

    test "cenwoiwe19" (lazy(sprintf "%A" (Foo()))) "x"

// Check escaped brackets with closing bracket escaped, invalidating property reference
module CheckDisplayAttributes20 =

    [<StructuredFormatDisplay("{One\} \{Two\}")>]
    type Foo() =
      member __.``One} {Two`` = "abc"
      member __.One = 123
      member __.Two = 456
      override x.ToString() = "x"

    test "cenwoiwe20" (lazy(sprintf "%A" (Foo()))) "x"

// Check escaped brackets display properly
module CheckDisplayAttributes21 =

    [<StructuredFormatDisplay("\{{One}\}")>]
    type Foo() =
      member __.``One} {Two`` = "abc"
      member __.One = 123
      member __.Two = 456
      override x.ToString() = "x"

    test "cenwoiwe21" (lazy(sprintf "%A" (Foo()))) "{123}"

// Check one with an two matched pairs and a trailing closing bracket
module CheckDisplayAttributes22 =

    [<StructuredFormatDisplay("{Val}{Hello} }")>]
    type Foo() = 
       member internal x.Val = 42
       member internal x.Hello = "Hello"
       override x.ToString() = "x"

    test "cenwoiwe22" (lazy(sprintf "%A" (Foo()))) "x"

// Check one with an two matched pairs and an unmatched closing bracket in-between
module CheckDisplayAttributes23 =

    [<StructuredFormatDisplay("{Val} } {Hello}")>]
    type Foo() = 
       member internal x.Val = 42
       member internal x.Hello = "Hello"
       override x.ToString() = "x"

    test "cenwoiwe23" (lazy(sprintf "%A" (Foo()))) "x"

let func0()=
    test "test1" (lazy(sprintf "%b" true)) "true"
    test "test2" (lazy(sprintf "%5b" true)) " true"
    test "test3" (lazy(sprintf "%1b" true)) "true"
    test "test4" (lazy(sprintf "%*b" 7 true)) "   true"
    test "test5" (lazy(sprintf "%-b" true)) "true"
    test "test6" (lazy(sprintf "%-5b" true)) "true "
    test "test7" (lazy(sprintf "%-1b" true)) "true"
    test "test8" (lazy(sprintf "%-*b" 7 true)) "true   "
    test "test9" (lazy(sprintf "%b" false)) "false"
    test "test10" (lazy(sprintf "%5b" false)) "false"
    test "test11" (lazy(sprintf "%1b" false)) "false"
    test "test12" (lazy(sprintf "%*b" 7 false)) "  false"
    test "test13" (lazy(sprintf "%-b" false)) "false"
    test "test14" (lazy(sprintf "%-5b" false)) "false"
    test "test15" (lazy(sprintf "%-1b" false)) "false"
    test "test16" (lazy(sprintf "%-*b" 7 false)) "false  "
    test "test17" (lazy(sprintf "%c" 'a')) "a"
    test "test18" (lazy(sprintf "%5c" 'a')) "    a"
    test "test19" (lazy(sprintf "%1c" 'a')) "a"
    test "test20" (lazy(sprintf "%*c" 7 'a')) "      a"
    test "test21" (lazy(sprintf "%-c" 'a')) "a"
    test "test22" (lazy(sprintf "%-5c" 'a')) "a    "
    test "test23" (lazy(sprintf "%-1c" 'a')) "a"
    test "test24" (lazy(sprintf "%-*c" 7 'a')) "a      "
    test "test25" (lazy(sprintf "%c" 'X')) "X"
    test "test26" (lazy(sprintf "%5c" 'X')) "    X"
    test "test27" (lazy(sprintf "%1c" 'X')) "X"
    test "test28" (lazy(sprintf "%*c" 7 'X')) "      X"
    test "test29" (lazy(sprintf "%-c" 'X')) "X"
    test "test30" (lazy(sprintf "%-5c" 'X')) "X    "
    test "test31" (lazy(sprintf "%-1c" 'X')) "X"
    test "test32" (lazy(sprintf "%-*c" 7 'X')) "X      "
    test "test33" (lazy(sprintf "%c" '\n')) "\n"
    test "test34" (lazy(sprintf "%5c" '\n')) "    \n"
    test "test35" (lazy(sprintf "%1c" '\n')) "\n"
    test "test36" (lazy(sprintf "%*c" 7 '\n')) "      \n"
    test "test37" (lazy(sprintf "%-c" '\n')) "\n"
    test "test38" (lazy(sprintf "%-5c" '\n')) "\n    "
    test "test39" (lazy(sprintf "%-1c" '\n')) "\n"
    test "test40" (lazy(sprintf "%-*c" 7 '\n')) "\n      "
    test "test41" (lazy(sprintf "%s" "test")) "test"
    test "test42" (lazy(sprintf "%5s" "test")) " test"
    test "test43" (lazy(sprintf "%1s" "test")) "test"
    test "test44" (lazy(sprintf "%*s" 7 "test")) "   test"
    test "test45" (lazy(sprintf "%-s" "test")) "test"
    test "test46" (lazy(sprintf "%-5s" "test")) "test "
    test "test47" (lazy(sprintf "%-1s" "test")) "test"
    test "test48" (lazy(sprintf "%-*s" 7 "test")) "test   "
    test "test49" (lazy(sprintf "%s" "value")) "value"
    test "test50" (lazy(sprintf "%5s" "value")) "value"
    test "test51" (lazy(sprintf "%1s" "value")) "value"
    test "test52" (lazy(sprintf "%*s" 7 "value")) "  value"
    test "test53" (lazy(sprintf "%-s" "value")) "value"
    test "test54" (lazy(sprintf "%-5s" "value")) "value"
    test "test55" (lazy(sprintf "%-1s" "value")) "value"
    test "test56" (lazy(sprintf "%-*s" 7 "value")) "value  "
    test "test57" (lazy(sprintf "%d" 14)) "14"
    test "test58" (lazy(sprintf "%5d" 14)) "   14"
    test "test59" (lazy(sprintf "%1d" 14)) "14"
    test "test60" (lazy(sprintf "%*d" 7 14)) "     14"
    test "test61" (lazy(sprintf "%-d" 14)) "14"
    test "test62" (lazy(sprintf "%-5d" 14)) "14   "
    test "test63" (lazy(sprintf "%-1d" 14)) "14"
    test "test64" (lazy(sprintf "%-*d" 7 14)) "14     "
    test "test65" (lazy(sprintf "%0d" 14)) "14"
    test "test66" (lazy(sprintf "%05d" 14)) "00014"
    test "test67" (lazy(sprintf "%01d" 14)) "14"
    test "test68" (lazy(sprintf "%0*d" 7 14)) "0000014"
    test "test69" (lazy(sprintf "%-0d" 14)) "14"
    test "test70" (lazy(sprintf "%-05d" 14)) "14   "
    test "test71" (lazy(sprintf "%-01d" 14)) "14"
    test "test72" (lazy(sprintf "%-0*d" 7 14)) "14     "
    test "test73" (lazy(sprintf "%+d" 14)) "+14"
    test "test74" (lazy(sprintf "%+5d" 14)) "  +14"
    test "test75" (lazy(sprintf "%+1d" 14)) "+14"
    test "test76" (lazy(sprintf "%+*d" 7 14)) "    +14"
    test "test77" (lazy(sprintf "%-+d" 14)) "+14"
    test "test78" (lazy(sprintf "%-+5d" 14)) "+14  "
    test "test79" (lazy(sprintf "%-+1d" 14)) "+14"
    test "test80" (lazy(sprintf "%-+*d" 7 14)) "+14    "
    test "test81" (lazy(sprintf "%+0d" 14)) "+14"
    test "test82" (lazy(sprintf "%+05d" 14)) "+0014"
    test "test83" (lazy(sprintf "%+01d" 14)) "+14"
    test "test84" (lazy(sprintf "%+0*d" 7 14)) "+000014"
    test "test85" (lazy(sprintf "%-+0d" 14)) "+14"
    test "test86" (lazy(sprintf "%-+05d" 14)) "+14  "
    test "test87" (lazy(sprintf "%-+01d" 14)) "+14"
    test "test88" (lazy(sprintf "%-+0*d" 7 14)) "+14    "
    test "test89" (lazy(sprintf "% d" 14)) " 14"
    test "test90" (lazy(sprintf "% 5d" 14)) "   14"
    test "test91" (lazy(sprintf "% 1d" 14)) " 14"
    test "test92" (lazy(sprintf "% *d" 7 14)) "     14"
    test "test93" (lazy(sprintf "%- d" 14)) " 14"
    test "test94" (lazy(sprintf "%- 5d" 14)) " 14  "
    test "test95" (lazy(sprintf "%- 1d" 14)) " 14"
    test "test96" (lazy(sprintf "%- *d" 7 14)) " 14    "
    test "test97" (lazy(sprintf "% 0d" 14)) " 14"
    test "test98" (lazy(sprintf "% 05d" 14)) " 0014"
    test "test99" (lazy(sprintf "% 01d" 14)) " 14"
    test "test100" (lazy(sprintf "% 0*d" 7 14)) " 000014"
    test "test101" (lazy(sprintf "%- 0d" 14)) " 14"
    test "test102" (lazy(sprintf "%- 05d" 14)) " 14  "
    test "test103" (lazy(sprintf "%- 01d" 14)) " 14"
    test "test104" (lazy(sprintf "%- 0*d" 7 14)) " 14    "
    test "test105" (lazy(sprintf "%d" -10)) "-10"
    test "test106" (lazy(sprintf "%5d" -10)) "  -10"
    test "test107" (lazy(sprintf "%1d" -10)) "-10"
    test "test108" (lazy(sprintf "%*d" 7 -10)) "    -10"
    test "test109" (lazy(sprintf "%-d" -10)) "-10"
    test "test110" (lazy(sprintf "%-5d" -10)) "-10  "
    test "test111" (lazy(sprintf "%-1d" -10)) "-10"
    test "test112" (lazy(sprintf "%-*d" 7 -10)) "-10    "
    test "test113" (lazy(sprintf "%0d" -10)) "-10"
    test "test114" (lazy(sprintf "%05d" -10)) "-0010"
    test "test115" (lazy(sprintf "%01d" -10)) "-10"
    test "test116" (lazy(sprintf "%0*d" 7 -10)) "-000010"
    test "test117" (lazy(sprintf "%-0d" -10)) "-10"
    test "test118" (lazy(sprintf "%-05d" -10)) "-10  "
    test "test119" (lazy(sprintf "%-01d" -10)) "-10"
    test "test120" (lazy(sprintf "%-0*d" 7 -10)) "-10    "
    test "test121" (lazy(sprintf "%+d" -10)) "-10"
    test "test122" (lazy(sprintf "%+5d" -10)) "  -10"
    test "test123" (lazy(sprintf "%+1d" -10)) "-10"
    test "test124" (lazy(sprintf "%+*d" 7 -10)) "    -10"
    test "test125" (lazy(sprintf "%-+d" -10)) "-10"
    test "test126" (lazy(sprintf "%-+5d" -10)) "-10  "
    test "test127" (lazy(sprintf "%-+1d" -10)) "-10"
    test "test128" (lazy(sprintf "%-+*d" 7 -10)) "-10    "
    test "test129" (lazy(sprintf "%+0d" -10)) "-10"
    test "test130" (lazy(sprintf "%+05d" -10)) "-0010"
    test "test131" (lazy(sprintf "%+01d" -10)) "-10"
    test "test132" (lazy(sprintf "%+0*d" 7 -10)) "-000010"
    test "test133" (lazy(sprintf "%-+0d" -10)) "-10"
    test "test134" (lazy(sprintf "%-+05d" -10)) "-10  "
    test "test135" (lazy(sprintf "%-+01d" -10)) "-10"
    test "test136" (lazy(sprintf "%-+0*d" 7 -10)) "-10    "
    test "test137" (lazy(sprintf "% d" -10)) "-10"
    test "test138" (lazy(sprintf "% 5d" -10)) "  -10"
    test "test139" (lazy(sprintf "% 1d" -10)) "-10"
    test "test140" (lazy(sprintf "% *d" 7 -10)) "    -10"
    test "test141" (lazy(sprintf "%- d" -10)) "-10"
    test "test142" (lazy(sprintf "%- 5d" -10)) "-10  "
    test "test143" (lazy(sprintf "%- 1d" -10)) "-10"
    test "test144" (lazy(sprintf "%- *d" 7 -10)) "-10    "
    test "test145" (lazy(sprintf "% 0d" -10)) "-10"
    test "test146" (lazy(sprintf "% 05d" -10)) "-0010"
    test "test147" (lazy(sprintf "% 01d" -10)) "-10"
    test "test148" (lazy(sprintf "% 0*d" 7 -10)) "-000010"
    test "test149" (lazy(sprintf "%- 0d" -10)) "-10"
    test "test150" (lazy(sprintf "%- 05d" -10)) "-10  "
    test "test151" (lazy(sprintf "%- 01d" -10)) "-10"
    test "test152" (lazy(sprintf "%- 0*d" 7 -10)) "-10    "
    test "test153" (lazy(sprintf "%d" 55s)) "55"
    test "test154" (lazy(sprintf "%5d" 55s)) "   55"
    test "test155" (lazy(sprintf "%1d" 55s)) "55"
    test "test156" (lazy(sprintf "%*d" 7 55s)) "     55"
    test "test157" (lazy(sprintf "%-d" 55s)) "55"
    test "test158" (lazy(sprintf "%-5d" 55s)) "55   "
    test "test159" (lazy(sprintf "%-1d" 55s)) "55"
    test "test160" (lazy(sprintf "%-*d" 7 55s)) "55     "
    test "test161" (lazy(sprintf "%0d" 55s)) "55"
    test "test162" (lazy(sprintf "%05d" 55s)) "00055"
    test "test163" (lazy(sprintf "%01d" 55s)) "55"
    test "test164" (lazy(sprintf "%0*d" 7 55s)) "0000055"
    test "test165" (lazy(sprintf "%-0d" 55s)) "55"
    test "test166" (lazy(sprintf "%-05d" 55s)) "55   "
    test "test167" (lazy(sprintf "%-01d" 55s)) "55"
    test "test168" (lazy(sprintf "%-0*d" 7 55s)) "55     "
    test "test169" (lazy(sprintf "%+d" 55s)) "+55"
    test "test170" (lazy(sprintf "%+5d" 55s)) "  +55"
    test "test171" (lazy(sprintf "%+1d" 55s)) "+55"
    test "test172" (lazy(sprintf "%+*d" 7 55s)) "    +55"
    test "test173" (lazy(sprintf "%-+d" 55s)) "+55"
    test "test174" (lazy(sprintf "%-+5d" 55s)) "+55  "
    test "test175" (lazy(sprintf "%-+1d" 55s)) "+55"
    test "test176" (lazy(sprintf "%-+*d" 7 55s)) "+55    "
    test "test177" (lazy(sprintf "%+0d" 55s)) "+55"
    test "test178" (lazy(sprintf "%+05d" 55s)) "+0055"
    test "test179" (lazy(sprintf "%+01d" 55s)) "+55"
    test "test180" (lazy(sprintf "%+0*d" 7 55s)) "+000055"
    test "test181" (lazy(sprintf "%-+0d" 55s)) "+55"
    test "test182" (lazy(sprintf "%-+05d" 55s)) "+55  "
    test "test183" (lazy(sprintf "%-+01d" 55s)) "+55"
    test "test184" (lazy(sprintf "%-+0*d" 7 55s)) "+55    "
    test "test185" (lazy(sprintf "% d" 55s)) " 55"
    test "test186" (lazy(sprintf "% 5d" 55s)) "   55"
    test "test187" (lazy(sprintf "% 1d" 55s)) " 55"
    test "test188" (lazy(sprintf "% *d" 7 55s)) "     55"
    test "test189" (lazy(sprintf "%- d" 55s)) " 55"
    test "test190" (lazy(sprintf "%- 5d" 55s)) " 55  "
    test "test191" (lazy(sprintf "%- 1d" 55s)) " 55"
    test "test192" (lazy(sprintf "%- *d" 7 55s)) " 55    "
    test "test193" (lazy(sprintf "% 0d" 55s)) " 55"
    test "test194" (lazy(sprintf "% 05d" 55s)) " 0055"
    test "test195" (lazy(sprintf "% 01d" 55s)) " 55"
    test "test196" (lazy(sprintf "% 0*d" 7 55s)) " 000055"
    test "test197" (lazy(sprintf "%- 0d" 55s)) " 55"
    test "test198" (lazy(sprintf "%- 05d" 55s)) " 55  "
    test "test199" (lazy(sprintf "%- 01d" 55s)) " 55"
    test "test200" (lazy(sprintf "%- 0*d" 7 55s)) " 55    "
    test "test201" (lazy(sprintf "%d" -88s)) "-88"
    test "test202" (lazy(sprintf "%5d" -88s)) "  -88"
    test "test203" (lazy(sprintf "%1d" -88s)) "-88"
    test "test204" (lazy(sprintf "%*d" 7 -88s)) "    -88"
    test "test205" (lazy(sprintf "%-d" -88s)) "-88"
    test "test206" (lazy(sprintf "%-5d" -88s)) "-88  "
    test "test207" (lazy(sprintf "%-1d" -88s)) "-88"
    test "test208" (lazy(sprintf "%-*d" 7 -88s)) "-88    "
    test "test209" (lazy(sprintf "%0d" -88s)) "-88"
    test "test210" (lazy(sprintf "%05d" -88s)) "-0088"
    test "test211" (lazy(sprintf "%01d" -88s)) "-88"
    test "test212" (lazy(sprintf "%0*d" 7 -88s)) "-000088"
    test "test213" (lazy(sprintf "%-0d" -88s)) "-88"
    test "test214" (lazy(sprintf "%-05d" -88s)) "-88  "
    test "test215" (lazy(sprintf "%-01d" -88s)) "-88"
    test "test216" (lazy(sprintf "%-0*d" 7 -88s)) "-88    "
    test "test217" (lazy(sprintf "%+d" -88s)) "-88"
    test "test218" (lazy(sprintf "%+5d" -88s)) "  -88"
    test "test219" (lazy(sprintf "%+1d" -88s)) "-88"
    test "test220" (lazy(sprintf "%+*d" 7 -88s)) "    -88"
    test "test221" (lazy(sprintf "%-+d" -88s)) "-88"
    test "test222" (lazy(sprintf "%-+5d" -88s)) "-88  "
    test "test223" (lazy(sprintf "%-+1d" -88s)) "-88"
    test "test224" (lazy(sprintf "%-+*d" 7 -88s)) "-88    "
    test "test225" (lazy(sprintf "%+0d" -88s)) "-88"
    test "test226" (lazy(sprintf "%+05d" -88s)) "-0088"
    test "test227" (lazy(sprintf "%+01d" -88s)) "-88"
    test "test228" (lazy(sprintf "%+0*d" 7 -88s)) "-000088"
    test "test229" (lazy(sprintf "%-+0d" -88s)) "-88"
    test "test230" (lazy(sprintf "%-+05d" -88s)) "-88  "
    test "test231" (lazy(sprintf "%-+01d" -88s)) "-88"
    test "test232" (lazy(sprintf "%-+0*d" 7 -88s)) "-88    "
    test "test233" (lazy(sprintf "% d" -88s)) "-88"
    test "test234" (lazy(sprintf "% 5d" -88s)) "  -88"
    test "test235" (lazy(sprintf "% 1d" -88s)) "-88"
    test "test236" (lazy(sprintf "% *d" 7 -88s)) "    -88"
    test "test237" (lazy(sprintf "%- d" -88s)) "-88"
    test "test238" (lazy(sprintf "%- 5d" -88s)) "-88  "
    test "test239" (lazy(sprintf "%- 1d" -88s)) "-88"
    test "test240" (lazy(sprintf "%- *d" 7 -88s)) "-88    "
    test "test241" (lazy(sprintf "% 0d" -88s)) "-88"
    test "test242" (lazy(sprintf "% 05d" -88s)) "-0088"
    test "test243" (lazy(sprintf "% 01d" -88s)) "-88"
    test "test244" (lazy(sprintf "% 0*d" 7 -88s)) "-000088"
    test "test245" (lazy(sprintf "%- 0d" -88s)) "-88"
    test "test246" (lazy(sprintf "%- 05d" -88s)) "-88  "
    test "test247" (lazy(sprintf "%- 01d" -88s)) "-88"
    test "test248" (lazy(sprintf "%- 0*d" 7 -88s)) "-88    "
    test "test249" (lazy(sprintf "%d" 104us)) "104"
    test "test250" (lazy(sprintf "%5d" 104us)) "  104"
    test "test251" (lazy(sprintf "%1d" 104us)) "104"
    test "test252" (lazy(sprintf "%*d" 7 104us)) "    104"
    test "test253" (lazy(sprintf "%-d" 104us)) "104"
    test "test254" (lazy(sprintf "%-5d" 104us)) "104  "
    test "test255" (lazy(sprintf "%-1d" 104us)) "104"
    test "test256" (lazy(sprintf "%-*d" 7 104us)) "104    "
    test "test257" (lazy(sprintf "%0d" 104us)) "104"
    test "test258" (lazy(sprintf "%05d" 104us)) "00104"
    test "test259" (lazy(sprintf "%01d" 104us)) "104"
    test "test260" (lazy(sprintf "%0*d" 7 104us)) "0000104"
    test "test261" (lazy(sprintf "%-0d" 104us)) "104"
    test "test262" (lazy(sprintf "%-05d" 104us)) "104  "
    test "test263" (lazy(sprintf "%-01d" 104us)) "104"
    test "test264" (lazy(sprintf "%-0*d" 7 104us)) "104    "
    test "test265" (lazy(sprintf "%+d" 104us)) "+104"
    test "test266" (lazy(sprintf "%+5d" 104us)) " +104"
    test "test267" (lazy(sprintf "%+1d" 104us)) "+104"
    test "test268" (lazy(sprintf "%+*d" 7 104us)) "   +104"
    test "test269" (lazy(sprintf "%-+d" 104us)) "+104"
    test "test270" (lazy(sprintf "%-+5d" 104us)) "+104 "
    test "test271" (lazy(sprintf "%-+1d" 104us)) "+104"
    test "test272" (lazy(sprintf "%-+*d" 7 104us)) "+104   "
    test "test273" (lazy(sprintf "%+0d" 104us)) "+104"
    test "test274" (lazy(sprintf "%+05d" 104us)) "+0104"
    test "test275" (lazy(sprintf "%+01d" 104us)) "+104"
    test "test276" (lazy(sprintf "%+0*d" 7 104us)) "+000104"
    test "test277" (lazy(sprintf "%-+0d" 104us)) "+104"
    test "test278" (lazy(sprintf "%-+05d" 104us)) "+104 "
    test "test279" (lazy(sprintf "%-+01d" 104us)) "+104"
    test "test280" (lazy(sprintf "%-+0*d" 7 104us)) "+104   "
    test "test281" (lazy(sprintf "% d" 104us)) " 104"
    test "test282" (lazy(sprintf "% 5d" 104us)) "  104"
    test "test283" (lazy(sprintf "% 1d" 104us)) " 104"
    test "test284" (lazy(sprintf "% *d" 7 104us)) "    104"
    test "test285" (lazy(sprintf "%- d" 104us)) " 104"
    test "test286" (lazy(sprintf "%- 5d" 104us)) " 104 "
    test "test287" (lazy(sprintf "%- 1d" 104us)) " 104"
    test "test288" (lazy(sprintf "%- *d" 7 104us)) " 104   "
    test "test289" (lazy(sprintf "% 0d" 104us)) " 104"
    test "test290" (lazy(sprintf "% 05d" 104us)) " 0104"
    test "test291" (lazy(sprintf "% 01d" 104us)) " 104"
    test "test292" (lazy(sprintf "% 0*d" 7 104us)) " 000104"
    test "test293" (lazy(sprintf "%- 0d" 104us)) " 104"
    test "test294" (lazy(sprintf "%- 05d" 104us)) " 104 "
    test "test295" (lazy(sprintf "%- 01d" 104us)) " 104"
    test "test296" (lazy(sprintf "%- 0*d" 7 104us)) " 104   "
    test "test297" (lazy(sprintf "%d" 12y)) "12"
    test "test298" (lazy(sprintf "%5d" 12y)) "   12"
    test "test299" (lazy(sprintf "%1d" 12y)) "12"
    test "test300" (lazy(sprintf "%*d" 7 12y)) "     12"
    test "test301" (lazy(sprintf "%-d" 12y)) "12"
    test "test302" (lazy(sprintf "%-5d" 12y)) "12   "
    test "test303" (lazy(sprintf "%-1d" 12y)) "12"
    test "test304" (lazy(sprintf "%-*d" 7 12y)) "12     "
    test "test305" (lazy(sprintf "%0d" 12y)) "12"
    test "test306" (lazy(sprintf "%05d" 12y)) "00012"
    test "test307" (lazy(sprintf "%01d" 12y)) "12"
    test "test308" (lazy(sprintf "%0*d" 7 12y)) "0000012"
    test "test309" (lazy(sprintf "%-0d" 12y)) "12"
    test "test310" (lazy(sprintf "%-05d" 12y)) "12   "
    test "test311" (lazy(sprintf "%-01d" 12y)) "12"
    test "test312" (lazy(sprintf "%-0*d" 7 12y)) "12     "
    test "test313" (lazy(sprintf "%+d" 12y)) "+12"
    test "test314" (lazy(sprintf "%+5d" 12y)) "  +12"
    test "test315" (lazy(sprintf "%+1d" 12y)) "+12"
    test "test316" (lazy(sprintf "%+*d" 7 12y)) "    +12"
    test "test317" (lazy(sprintf "%-+d" 12y)) "+12"
    test "test318" (lazy(sprintf "%-+5d" 12y)) "+12  "
    test "test319" (lazy(sprintf "%-+1d" 12y)) "+12"
    test "test320" (lazy(sprintf "%-+*d" 7 12y)) "+12    "
    test "test321" (lazy(sprintf "%+0d" 12y)) "+12"
    test "test322" (lazy(sprintf "%+05d" 12y)) "+0012"
    test "test323" (lazy(sprintf "%+01d" 12y)) "+12"
    test "test324" (lazy(sprintf "%+0*d" 7 12y)) "+000012"
    test "test325" (lazy(sprintf "%-+0d" 12y)) "+12"
    test "test326" (lazy(sprintf "%-+05d" 12y)) "+12  "
    test "test327" (lazy(sprintf "%-+01d" 12y)) "+12"
    test "test328" (lazy(sprintf "%-+0*d" 7 12y)) "+12    "
    test "test329" (lazy(sprintf "% d" 12y)) " 12"
    test "test330" (lazy(sprintf "% 5d" 12y)) "   12"
    test "test331" (lazy(sprintf "% 1d" 12y)) " 12"
    test "test332" (lazy(sprintf "% *d" 7 12y)) "     12"
    test "test333" (lazy(sprintf "%- d" 12y)) " 12"
    test "test334" (lazy(sprintf "%- 5d" 12y)) " 12  "
    test "test335" (lazy(sprintf "%- 1d" 12y)) " 12"
    test "test336" (lazy(sprintf "%- *d" 7 12y)) " 12    "
    test "test337" (lazy(sprintf "% 0d" 12y)) " 12"
    test "test338" (lazy(sprintf "% 05d" 12y)) " 0012"
    test "test339" (lazy(sprintf "% 01d" 12y)) " 12"
    test "test340" (lazy(sprintf "% 0*d" 7 12y)) " 000012"
    test "test341" (lazy(sprintf "%- 0d" 12y)) " 12"
    test "test342" (lazy(sprintf "%- 05d" 12y)) " 12  "
    test "test343" (lazy(sprintf "%- 01d" 12y)) " 12"
    test "test344" (lazy(sprintf "%- 0*d" 7 12y)) " 12    "
    test "test345" (lazy(sprintf "%d" -94y)) "-94"
    test "test346" (lazy(sprintf "%5d" -94y)) "  -94"
    test "test347" (lazy(sprintf "%1d" -94y)) "-94"
    test "test348" (lazy(sprintf "%*d" 7 -94y)) "    -94"
    test "test349" (lazy(sprintf "%-d" -94y)) "-94"
    test "test350" (lazy(sprintf "%-5d" -94y)) "-94  "
    test "test351" (lazy(sprintf "%-1d" -94y)) "-94"
    test "test352" (lazy(sprintf "%-*d" 7 -94y)) "-94    "
    test "test353" (lazy(sprintf "%0d" -94y)) "-94"
    test "test354" (lazy(sprintf "%05d" -94y)) "-0094"
    test "test355" (lazy(sprintf "%01d" -94y)) "-94"
    test "test356" (lazy(sprintf "%0*d" 7 -94y)) "-000094"
    test "test357" (lazy(sprintf "%-0d" -94y)) "-94"
    test "test358" (lazy(sprintf "%-05d" -94y)) "-94  "
    test "test359" (lazy(sprintf "%-01d" -94y)) "-94"
    test "test360" (lazy(sprintf "%-0*d" 7 -94y)) "-94    "
    test "test361" (lazy(sprintf "%+d" -94y)) "-94"
    test "test362" (lazy(sprintf "%+5d" -94y)) "  -94"
    test "test363" (lazy(sprintf "%+1d" -94y)) "-94"
    test "test364" (lazy(sprintf "%+*d" 7 -94y)) "    -94"
    test "test365" (lazy(sprintf "%-+d" -94y)) "-94"
    test "test366" (lazy(sprintf "%-+5d" -94y)) "-94  "
    test "test367" (lazy(sprintf "%-+1d" -94y)) "-94"
    test "test368" (lazy(sprintf "%-+*d" 7 -94y)) "-94    "
    test "test369" (lazy(sprintf "%+0d" -94y)) "-94"
    test "test370" (lazy(sprintf "%+05d" -94y)) "-0094"
    test "test371" (lazy(sprintf "%+01d" -94y)) "-94"
    test "test372" (lazy(sprintf "%+0*d" 7 -94y)) "-000094"
    test "test373" (lazy(sprintf "%-+0d" -94y)) "-94"
    test "test374" (lazy(sprintf "%-+05d" -94y)) "-94  "
    test "test375" (lazy(sprintf "%-+01d" -94y)) "-94"
    test "test376" (lazy(sprintf "%-+0*d" 7 -94y)) "-94    "
    test "test377" (lazy(sprintf "% d" -94y)) "-94"
    test "test378" (lazy(sprintf "% 5d" -94y)) "  -94"
    test "test379" (lazy(sprintf "% 1d" -94y)) "-94"
    test "test380" (lazy(sprintf "% *d" 7 -94y)) "    -94"
    test "test381" (lazy(sprintf "%- d" -94y)) "-94"
    test "test382" (lazy(sprintf "%- 5d" -94y)) "-94  "
    test "test383" (lazy(sprintf "%- 1d" -94y)) "-94"
    test "test384" (lazy(sprintf "%- *d" 7 -94y)) "-94    "
    test "test385" (lazy(sprintf "% 0d" -94y)) "-94"
    test "test386" (lazy(sprintf "% 05d" -94y)) "-0094"
    test "test387" (lazy(sprintf "% 01d" -94y)) "-94"
    test "test388" (lazy(sprintf "% 0*d" 7 -94y)) "-000094"
    test "test389" (lazy(sprintf "%- 0d" -94y)) "-94"
    test "test390" (lazy(sprintf "%- 05d" -94y)) "-94  "
    test "test391" (lazy(sprintf "%- 01d" -94y)) "-94"
    test "test392" (lazy(sprintf "%- 0*d" 7 -94y)) "-94    "
    test "test393" (lazy(sprintf "%d" 88uy)) "88"
    test "test394" (lazy(sprintf "%5d" 88uy)) "   88"
    test "test395" (lazy(sprintf "%1d" 88uy)) "88"
    test "test396" (lazy(sprintf "%*d" 7 88uy)) "     88"
    test "test397" (lazy(sprintf "%-d" 88uy)) "88"
    test "test398" (lazy(sprintf "%-5d" 88uy)) "88   "
    test "test399" (lazy(sprintf "%-1d" 88uy)) "88"
    test "test400" (lazy(sprintf "%-*d" 7 88uy)) "88     "
    test "test401" (lazy(sprintf "%0d" 88uy)) "88"
    test "test402" (lazy(sprintf "%05d" 88uy)) "00088"
    test "test403" (lazy(sprintf "%01d" 88uy)) "88"
    test "test404" (lazy(sprintf "%0*d" 7 88uy)) "0000088"
    test "test405" (lazy(sprintf "%-0d" 88uy)) "88"
    test "test406" (lazy(sprintf "%-05d" 88uy)) "88   "
    test "test407" (lazy(sprintf "%-01d" 88uy)) "88"
    test "test408" (lazy(sprintf "%-0*d" 7 88uy)) "88     "
    test "test409" (lazy(sprintf "%+d" 88uy)) "+88"
    test "test410" (lazy(sprintf "%+5d" 88uy)) "  +88"
    test "test411" (lazy(sprintf "%+1d" 88uy)) "+88"
    test "test412" (lazy(sprintf "%+*d" 7 88uy)) "    +88"
    test "test413" (lazy(sprintf "%-+d" 88uy)) "+88"
    test "test414" (lazy(sprintf "%-+5d" 88uy)) "+88  "
    test "test415" (lazy(sprintf "%-+1d" 88uy)) "+88"
    test "test416" (lazy(sprintf "%-+*d" 7 88uy)) "+88    "
    test "test417" (lazy(sprintf "%+0d" 88uy)) "+88"
    test "test418" (lazy(sprintf "%+05d" 88uy)) "+0088"
    test "test419" (lazy(sprintf "%+01d" 88uy)) "+88"
    test "test420" (lazy(sprintf "%+0*d" 7 88uy)) "+000088"
    test "test421" (lazy(sprintf "%-+0d" 88uy)) "+88"
    test "test422" (lazy(sprintf "%-+05d" 88uy)) "+88  "
    test "test423" (lazy(sprintf "%-+01d" 88uy)) "+88"
    test "test424" (lazy(sprintf "%-+0*d" 7 88uy)) "+88    "
    test "test425" (lazy(sprintf "% d" 88uy)) " 88"
    test "test426" (lazy(sprintf "% 5d" 88uy)) "   88"
    test "test427" (lazy(sprintf "% 1d" 88uy)) " 88"
    test "test428" (lazy(sprintf "% *d" 7 88uy)) "     88"
    test "test429" (lazy(sprintf "%- d" 88uy)) " 88"
    test "test430" (lazy(sprintf "%- 5d" 88uy)) " 88  "
    test "test431" (lazy(sprintf "%- 1d" 88uy)) " 88"
    test "test432" (lazy(sprintf "%- *d" 7 88uy)) " 88    "
    test "test433" (lazy(sprintf "% 0d" 88uy)) " 88"
    test "test434" (lazy(sprintf "% 05d" 88uy)) " 0088"
    test "test435" (lazy(sprintf "% 01d" 88uy)) " 88"
    test "test436" (lazy(sprintf "% 0*d" 7 88uy)) " 000088"
    test "test437" (lazy(sprintf "%- 0d" 88uy)) " 88"
    test "test438" (lazy(sprintf "%- 05d" 88uy)) " 88  "
    test "test439" (lazy(sprintf "%- 01d" 88uy)) " 88"
    test "test440" (lazy(sprintf "%- 0*d" 7 88uy)) " 88    "
    test "test441" (lazy(sprintf "%d" 999L)) "999"
    test "test442" (lazy(sprintf "%5d" 999L)) "  999"
    test "test443" (lazy(sprintf "%1d" 999L)) "999"
    test "test444" (lazy(sprintf "%*d" 7 999L)) "    999"
    test "test445" (lazy(sprintf "%-d" 999L)) "999"
    test "test446" (lazy(sprintf "%-5d" 999L)) "999  "
    test "test447" (lazy(sprintf "%-1d" 999L)) "999"
    test "test448" (lazy(sprintf "%-*d" 7 999L)) "999    "
    test "test449" (lazy(sprintf "%0d" 999L)) "999"
    test "test450" (lazy(sprintf "%05d" 999L)) "00999"
    test "test451" (lazy(sprintf "%01d" 999L)) "999"
    test "test452" (lazy(sprintf "%0*d" 7 999L)) "0000999"
    test "test453" (lazy(sprintf "%-0d" 999L)) "999"
    test "test454" (lazy(sprintf "%-05d" 999L)) "999  "
    test "test455" (lazy(sprintf "%-01d" 999L)) "999"
    test "test456" (lazy(sprintf "%-0*d" 7 999L)) "999    "
    test "test457" (lazy(sprintf "%+d" 999L)) "+999"
    test "test458" (lazy(sprintf "%+5d" 999L)) " +999"
    test "test459" (lazy(sprintf "%+1d" 999L)) "+999"
    test "test460" (lazy(sprintf "%+*d" 7 999L)) "   +999"
    test "test461" (lazy(sprintf "%-+d" 999L)) "+999"
    test "test462" (lazy(sprintf "%-+5d" 999L)) "+999 "
    test "test463" (lazy(sprintf "%-+1d" 999L)) "+999"
    test "test464" (lazy(sprintf "%-+*d" 7 999L)) "+999   "
    test "test465" (lazy(sprintf "%+0d" 999L)) "+999"
    test "test466" (lazy(sprintf "%+05d" 999L)) "+0999"
    test "test467" (lazy(sprintf "%+01d" 999L)) "+999"
    test "test468" (lazy(sprintf "%+0*d" 7 999L)) "+000999"
    test "test469" (lazy(sprintf "%-+0d" 999L)) "+999"
    test "test470" (lazy(sprintf "%-+05d" 999L)) "+999 "
    test "test471" (lazy(sprintf "%-+01d" 999L)) "+999"
    test "test472" (lazy(sprintf "%-+0*d" 7 999L)) "+999   "
    test "test473" (lazy(sprintf "% d" 999L)) " 999"
    test "test474" (lazy(sprintf "% 5d" 999L)) "  999"
    test "test475" (lazy(sprintf "% 1d" 999L)) " 999"
    test "test476" (lazy(sprintf "% *d" 7 999L)) "    999"
    test "test477" (lazy(sprintf "%- d" 999L)) " 999"
    test "test478" (lazy(sprintf "%- 5d" 999L)) " 999 "
    test "test479" (lazy(sprintf "%- 1d" 999L)) " 999"
    test "test480" (lazy(sprintf "%- *d" 7 999L)) " 999   "
    test "test481" (lazy(sprintf "% 0d" 999L)) " 999"
    test "test482" (lazy(sprintf "% 05d" 999L)) " 0999"
    test "test483" (lazy(sprintf "% 01d" 999L)) " 999"
    test "test484" (lazy(sprintf "% 0*d" 7 999L)) " 000999"
    test "test485" (lazy(sprintf "%- 0d" 999L)) " 999"
    test "test486" (lazy(sprintf "%- 05d" 999L)) " 999 "
    test "test487" (lazy(sprintf "%- 01d" 999L)) " 999"
    test "test488" (lazy(sprintf "%- 0*d" 7 999L)) " 999   "
    test "test489" (lazy(sprintf "%d" -321L)) "-321"
    test "test490" (lazy(sprintf "%5d" -321L)) " -321"
    test "test491" (lazy(sprintf "%1d" -321L)) "-321"
    test "test492" (lazy(sprintf "%*d" 7 -321L)) "   -321"
    test "test493" (lazy(sprintf "%-d" -321L)) "-321"
    test "test494" (lazy(sprintf "%-5d" -321L)) "-321 "
    test "test495" (lazy(sprintf "%-1d" -321L)) "-321"
    test "test496" (lazy(sprintf "%-*d" 7 -321L)) "-321   "
    test "test497" (lazy(sprintf "%0d" -321L)) "-321"
    test "test498" (lazy(sprintf "%05d" -321L)) "-0321"
    test "test499" (lazy(sprintf "%01d" -321L)) "-321"
    test "test500" (lazy(sprintf "%0*d" 7 -321L)) "-000321"
    test "test501" (lazy(sprintf "%-0d" -321L)) "-321"
    test "test502" (lazy(sprintf "%-05d" -321L)) "-321 "
    test "test503" (lazy(sprintf "%-01d" -321L)) "-321"
    test "test504" (lazy(sprintf "%-0*d" 7 -321L)) "-321   "
    test "test505" (lazy(sprintf "%+d" -321L)) "-321"
    test "test506" (lazy(sprintf "%+5d" -321L)) " -321"
    test "test507" (lazy(sprintf "%+1d" -321L)) "-321"
    test "test508" (lazy(sprintf "%+*d" 7 -321L)) "   -321"
    test "test509" (lazy(sprintf "%-+d" -321L)) "-321"
    test "test510" (lazy(sprintf "%-+5d" -321L)) "-321 "
    test "test511" (lazy(sprintf "%-+1d" -321L)) "-321"
    test "test512" (lazy(sprintf "%-+*d" 7 -321L)) "-321   "
    test "test513" (lazy(sprintf "%+0d" -321L)) "-321"
    test "test514" (lazy(sprintf "%+05d" -321L)) "-0321"
    test "test515" (lazy(sprintf "%+01d" -321L)) "-321"
    test "test516" (lazy(sprintf "%+0*d" 7 -321L)) "-000321"
    test "test517" (lazy(sprintf "%-+0d" -321L)) "-321"
    test "test518" (lazy(sprintf "%-+05d" -321L)) "-321 "
    test "test519" (lazy(sprintf "%-+01d" -321L)) "-321"
    test "test520" (lazy(sprintf "%-+0*d" 7 -321L)) "-321   "
    test "test521" (lazy(sprintf "% d" -321L)) "-321"
    test "test522" (lazy(sprintf "% 5d" -321L)) " -321"
    test "test523" (lazy(sprintf "% 1d" -321L)) "-321"
    test "test524" (lazy(sprintf "% *d" 7 -321L)) "   -321"
    test "test525" (lazy(sprintf "%- d" -321L)) "-321"
    test "test526" (lazy(sprintf "%- 5d" -321L)) "-321 "
    test "test527" (lazy(sprintf "%- 1d" -321L)) "-321"
    test "test528" (lazy(sprintf "%- *d" 7 -321L)) "-321   "
    test "test529" (lazy(sprintf "% 0d" -321L)) "-321"
    test "test530" (lazy(sprintf "% 05d" -321L)) "-0321"
    test "test531" (lazy(sprintf "% 01d" -321L)) "-321"
    test "test532" (lazy(sprintf "% 0*d" 7 -321L)) "-000321"
    test "test533" (lazy(sprintf "%- 0d" -321L)) "-321"
    test "test534" (lazy(sprintf "%- 05d" -321L)) "-321 "
    test "test535" (lazy(sprintf "%- 01d" -321L)) "-321"
    test "test536" (lazy(sprintf "%- 0*d" 7 -321L)) "-321   "
    test "test537" (lazy(sprintf "%d" 50000UL)) "50000"
    test "test538" (lazy(sprintf "%5d" 50000UL)) "50000"
    test "test539" (lazy(sprintf "%1d" 50000UL)) "50000"
    test "test540" (lazy(sprintf "%*d" 7 50000UL)) "  50000"
    test "test541" (lazy(sprintf "%-d" 50000UL)) "50000"
    test "test542" (lazy(sprintf "%-5d" 50000UL)) "50000"
    test "test543" (lazy(sprintf "%-1d" 50000UL)) "50000"
    test "test544" (lazy(sprintf "%-*d" 7 50000UL)) "50000  "
    test "test545" (lazy(sprintf "%0d" 50000UL)) "50000"
    test "test546" (lazy(sprintf "%05d" 50000UL)) "50000"
    test "test547" (lazy(sprintf "%01d" 50000UL)) "50000"
    test "test548" (lazy(sprintf "%0*d" 7 50000UL)) "0050000"
    test "test549" (lazy(sprintf "%-0d" 50000UL)) "50000"
    test "test550" (lazy(sprintf "%-05d" 50000UL)) "50000"
    test "test551" (lazy(sprintf "%-01d" 50000UL)) "50000"
    test "test552" (lazy(sprintf "%-0*d" 7 50000UL)) "50000  "
    test "test553" (lazy(sprintf "%+d" 50000UL)) "+50000"
    test "test554" (lazy(sprintf "%+5d" 50000UL)) "+50000"
    test "test555" (lazy(sprintf "%+1d" 50000UL)) "+50000"
    test "test556" (lazy(sprintf "%+*d" 7 50000UL)) " +50000"
    test "test557" (lazy(sprintf "%-+d" 50000UL)) "+50000"
    test "test558" (lazy(sprintf "%-+5d" 50000UL)) "+50000"
    test "test559" (lazy(sprintf "%-+1d" 50000UL)) "+50000"
    test "test560" (lazy(sprintf "%-+*d" 7 50000UL)) "+50000 "
    test "test561" (lazy(sprintf "%+0d" 50000UL)) "+50000"
    test "test562" (lazy(sprintf "%+05d" 50000UL)) "+50000"
    test "test563" (lazy(sprintf "%+01d" 50000UL)) "+50000"
    test "test564" (lazy(sprintf "%+0*d" 7 50000UL)) "+050000"
    test "test565" (lazy(sprintf "%-+0d" 50000UL)) "+50000"
    test "test566" (lazy(sprintf "%-+05d" 50000UL)) "+50000"
    test "test567" (lazy(sprintf "%-+01d" 50000UL)) "+50000"
    test "test568" (lazy(sprintf "%-+0*d" 7 50000UL)) "+50000 "
    test "test569" (lazy(sprintf "% d" 50000UL)) " 50000"
    test "test570" (lazy(sprintf "% 5d" 50000UL)) " 50000"
    test "test571" (lazy(sprintf "% 1d" 50000UL)) " 50000"
    test "test572" (lazy(sprintf "% *d" 7 50000UL)) "  50000"
    test "test573" (lazy(sprintf "%- d" 50000UL)) " 50000"
    test "test574" (lazy(sprintf "%- 5d" 50000UL)) " 50000"
    test "test575" (lazy(sprintf "%- 1d" 50000UL)) " 50000"
    test "test576" (lazy(sprintf "%- *d" 7 50000UL)) " 50000 "
    test "test577" (lazy(sprintf "% 0d" 50000UL)) " 50000"
    test "test578" (lazy(sprintf "% 05d" 50000UL)) " 50000"
    test "test579" (lazy(sprintf "% 01d" 50000UL)) " 50000"
    test "test580" (lazy(sprintf "% 0*d" 7 50000UL)) " 050000"
    test "test581" (lazy(sprintf "%- 0d" 50000UL)) " 50000"
    test "test582" (lazy(sprintf "%- 05d" 50000UL)) " 50000"
    test "test583" (lazy(sprintf "%- 01d" 50000UL)) " 50000"
    test "test584" (lazy(sprintf "%- 0*d" 7 50000UL)) " 50000 "
    test "test585" (lazy(sprintf "%d" System.Int32.MaxValue)) "2147483647"
    test "test586" (lazy(sprintf "%5d" System.Int32.MaxValue)) "2147483647"
    test "test587" (lazy(sprintf "%1d" System.Int32.MaxValue)) "2147483647"
    test "test588" (lazy(sprintf "%*d" 7 System.Int32.MaxValue)) "2147483647"
    test "test589" (lazy(sprintf "%-d" System.Int32.MaxValue)) "2147483647"
    test "test590" (lazy(sprintf "%-5d" System.Int32.MaxValue)) "2147483647"
    test "test591" (lazy(sprintf "%-1d" System.Int32.MaxValue)) "2147483647"
    test "test592" (lazy(sprintf "%-*d" 7 System.Int32.MaxValue)) "2147483647"
    test "test593" (lazy(sprintf "%0d" System.Int32.MaxValue)) "2147483647"
    test "test594" (lazy(sprintf "%05d" System.Int32.MaxValue)) "2147483647"
    test "test595" (lazy(sprintf "%01d" System.Int32.MaxValue)) "2147483647"
    test "test596" (lazy(sprintf "%0*d" 7 System.Int32.MaxValue)) "2147483647"
    test "test597" (lazy(sprintf "%-0d" System.Int32.MaxValue)) "2147483647"
    test "test598" (lazy(sprintf "%-05d" System.Int32.MaxValue)) "2147483647"
    test "test599" (lazy(sprintf "%-01d" System.Int32.MaxValue)) "2147483647"
    test "test600" (lazy(sprintf "%-0*d" 7 System.Int32.MaxValue)) "2147483647"
    test "test601" (lazy(sprintf "%+d" System.Int32.MaxValue)) "+2147483647"
    test "test602" (lazy(sprintf "%+5d" System.Int32.MaxValue)) "+2147483647"
    test "test603" (lazy(sprintf "%+1d" System.Int32.MaxValue)) "+2147483647"
    test "test604" (lazy(sprintf "%+*d" 7 System.Int32.MaxValue)) "+2147483647"
    test "test605" (lazy(sprintf "%-+d" System.Int32.MaxValue)) "+2147483647"
    test "test606" (lazy(sprintf "%-+5d" System.Int32.MaxValue)) "+2147483647"
    test "test607" (lazy(sprintf "%-+1d" System.Int32.MaxValue)) "+2147483647"
    test "test608" (lazy(sprintf "%-+*d" 7 System.Int32.MaxValue)) "+2147483647"
    test "test609" (lazy(sprintf "%+0d" System.Int32.MaxValue)) "+2147483647"
    test "test610" (lazy(sprintf "%+05d" System.Int32.MaxValue)) "+2147483647"
    test "test611" (lazy(sprintf "%+01d" System.Int32.MaxValue)) "+2147483647"
    test "test612" (lazy(sprintf "%+0*d" 7 System.Int32.MaxValue)) "+2147483647"
    test "test613" (lazy(sprintf "%-+0d" System.Int32.MaxValue)) "+2147483647"
    test "test614" (lazy(sprintf "%-+05d" System.Int32.MaxValue)) "+2147483647"
    test "test615" (lazy(sprintf "%-+01d" System.Int32.MaxValue)) "+2147483647"
    test "test616" (lazy(sprintf "%-+0*d" 7 System.Int32.MaxValue)) "+2147483647"
    test "test617" (lazy(sprintf "% d" System.Int32.MaxValue)) " 2147483647"
    test "test618" (lazy(sprintf "% 5d" System.Int32.MaxValue)) " 2147483647"
    test "test619" (lazy(sprintf "% 1d" System.Int32.MaxValue)) " 2147483647"
    test "test620" (lazy(sprintf "% *d" 7 System.Int32.MaxValue)) " 2147483647"
    test "test621" (lazy(sprintf "%- d" System.Int32.MaxValue)) " 2147483647"
    test "test622" (lazy(sprintf "%- 5d" System.Int32.MaxValue)) " 2147483647"
    test "test623" (lazy(sprintf "%- 1d" System.Int32.MaxValue)) " 2147483647"
    test "test624" (lazy(sprintf "%- *d" 7 System.Int32.MaxValue)) " 2147483647"
    test "test625" (lazy(sprintf "% 0d" System.Int32.MaxValue)) " 2147483647"
    test "test626" (lazy(sprintf "% 05d" System.Int32.MaxValue)) " 2147483647"
    test "test627" (lazy(sprintf "% 01d" System.Int32.MaxValue)) " 2147483647"
    test "test628" (lazy(sprintf "% 0*d" 7 System.Int32.MaxValue)) " 2147483647"
    test "test629" (lazy(sprintf "%- 0d" System.Int32.MaxValue)) " 2147483647"
    test "test630" (lazy(sprintf "%- 05d" System.Int32.MaxValue)) " 2147483647"
    test "test631" (lazy(sprintf "%- 01d" System.Int32.MaxValue)) " 2147483647"
    test "test632" (lazy(sprintf "%- 0*d" 7 System.Int32.MaxValue)) " 2147483647"
    test "test633" (lazy(sprintf "%d" System.Int64.MaxValue)) "9223372036854775807"
    test "test634" (lazy(sprintf "%5d" System.Int64.MaxValue)) "9223372036854775807"
    test "test635" (lazy(sprintf "%1d" System.Int64.MaxValue)) "9223372036854775807"
    test "test636" (lazy(sprintf "%*d" 7 System.Int64.MaxValue)) "9223372036854775807"
    test "test637" (lazy(sprintf "%-d" System.Int64.MaxValue)) "9223372036854775807"
    test "test638" (lazy(sprintf "%-5d" System.Int64.MaxValue)) "9223372036854775807"
    test "test639" (lazy(sprintf "%-1d" System.Int64.MaxValue)) "9223372036854775807"
    test "test640" (lazy(sprintf "%-*d" 7 System.Int64.MaxValue)) "9223372036854775807"
    test "test641" (lazy(sprintf "%0d" System.Int64.MaxValue)) "9223372036854775807"
    test "test642" (lazy(sprintf "%05d" System.Int64.MaxValue)) "9223372036854775807"
    test "test643" (lazy(sprintf "%01d" System.Int64.MaxValue)) "9223372036854775807"
    test "test644" (lazy(sprintf "%0*d" 7 System.Int64.MaxValue)) "9223372036854775807"
    test "test645" (lazy(sprintf "%-0d" System.Int64.MaxValue)) "9223372036854775807"
    test "test646" (lazy(sprintf "%-05d" System.Int64.MaxValue)) "9223372036854775807"
    test "test647" (lazy(sprintf "%-01d" System.Int64.MaxValue)) "9223372036854775807"
    test "test648" (lazy(sprintf "%-0*d" 7 System.Int64.MaxValue)) "9223372036854775807"
    test "test649" (lazy(sprintf "%+d" System.Int64.MaxValue)) "+9223372036854775807"
    test "test650" (lazy(sprintf "%+5d" System.Int64.MaxValue)) "+9223372036854775807"
    test "test651" (lazy(sprintf "%+1d" System.Int64.MaxValue)) "+9223372036854775807"
    test "test652" (lazy(sprintf "%+*d" 7 System.Int64.MaxValue)) "+9223372036854775807"
    test "test653" (lazy(sprintf "%-+d" System.Int64.MaxValue)) "+9223372036854775807"
    test "test654" (lazy(sprintf "%-+5d" System.Int64.MaxValue)) "+9223372036854775807"
    test "test655" (lazy(sprintf "%-+1d" System.Int64.MaxValue)) "+9223372036854775807"
    test "test656" (lazy(sprintf "%-+*d" 7 System.Int64.MaxValue)) "+9223372036854775807"
    test "test657" (lazy(sprintf "%+0d" System.Int64.MaxValue)) "+9223372036854775807"
    test "test658" (lazy(sprintf "%+05d" System.Int64.MaxValue)) "+9223372036854775807"
    test "test659" (lazy(sprintf "%+01d" System.Int64.MaxValue)) "+9223372036854775807"
    test "test660" (lazy(sprintf "%+0*d" 7 System.Int64.MaxValue)) "+9223372036854775807"
    test "test661" (lazy(sprintf "%-+0d" System.Int64.MaxValue)) "+9223372036854775807"
    test "test662" (lazy(sprintf "%-+05d" System.Int64.MaxValue)) "+9223372036854775807"
    test "test663" (lazy(sprintf "%-+01d" System.Int64.MaxValue)) "+9223372036854775807"
    test "test664" (lazy(sprintf "%-+0*d" 7 System.Int64.MaxValue)) "+9223372036854775807"
    test "test665" (lazy(sprintf "% d" System.Int64.MaxValue)) " 9223372036854775807"
    test "test666" (lazy(sprintf "% 5d" System.Int64.MaxValue)) " 9223372036854775807"
    test "test667" (lazy(sprintf "% 1d" System.Int64.MaxValue)) " 9223372036854775807"
    test "test668" (lazy(sprintf "% *d" 7 System.Int64.MaxValue)) " 9223372036854775807"
    test "test669" (lazy(sprintf "%- d" System.Int64.MaxValue)) " 9223372036854775807"
    test "test670" (lazy(sprintf "%- 5d" System.Int64.MaxValue)) " 9223372036854775807"
    test "test671" (lazy(sprintf "%- 1d" System.Int64.MaxValue)) " 9223372036854775807"
    test "test672" (lazy(sprintf "%- *d" 7 System.Int64.MaxValue)) " 9223372036854775807"
    test "test673" (lazy(sprintf "% 0d" System.Int64.MaxValue)) " 9223372036854775807"
    test "test674" (lazy(sprintf "% 05d" System.Int64.MaxValue)) " 9223372036854775807"
    test "test675" (lazy(sprintf "% 01d" System.Int64.MaxValue)) " 9223372036854775807"
    test "test676" (lazy(sprintf "% 0*d" 7 System.Int64.MaxValue)) " 9223372036854775807"
    test "test677" (lazy(sprintf "%- 0d" System.Int64.MaxValue)) " 9223372036854775807"
    test "test678" (lazy(sprintf "%- 05d" System.Int64.MaxValue)) " 9223372036854775807"
    test "test679" (lazy(sprintf "%- 01d" System.Int64.MaxValue)) " 9223372036854775807"
    test "test680" (lazy(sprintf "%- 0*d" 7 System.Int64.MaxValue)) " 9223372036854775807"
    test "test681" (lazy(sprintf "%d" System.Int32.MinValue)) "-2147483648"
    test "test682" (lazy(sprintf "%5d" System.Int32.MinValue)) "-2147483648"
    test "test683" (lazy(sprintf "%1d" System.Int32.MinValue)) "-2147483648"
    test "test684" (lazy(sprintf "%*d" 7 System.Int32.MinValue)) "-2147483648"
    test "test685" (lazy(sprintf "%-d" System.Int32.MinValue)) "-2147483648"
    test "test686" (lazy(sprintf "%-5d" System.Int32.MinValue)) "-2147483648"
    test "test687" (lazy(sprintf "%-1d" System.Int32.MinValue)) "-2147483648"
    test "test688" (lazy(sprintf "%-*d" 7 System.Int32.MinValue)) "-2147483648"
    test "test689" (lazy(sprintf "%0d" System.Int32.MinValue)) "-2147483648"
    test "test690" (lazy(sprintf "%05d" System.Int32.MinValue)) "-2147483648"
    test "test691" (lazy(sprintf "%01d" System.Int32.MinValue)) "-2147483648"
    test "test692" (lazy(sprintf "%0*d" 7 System.Int32.MinValue)) "-2147483648"
    test "test693" (lazy(sprintf "%-0d" System.Int32.MinValue)) "-2147483648"
    test "test694" (lazy(sprintf "%-05d" System.Int32.MinValue)) "-2147483648"
    test "test695" (lazy(sprintf "%-01d" System.Int32.MinValue)) "-2147483648"
    test "test696" (lazy(sprintf "%-0*d" 7 System.Int32.MinValue)) "-2147483648"
    test "test697" (lazy(sprintf "%+d" System.Int32.MinValue)) "-2147483648"
    test "test698" (lazy(sprintf "%+5d" System.Int32.MinValue)) "-2147483648"
    test "test699" (lazy(sprintf "%+1d" System.Int32.MinValue)) "-2147483648"
    test "test700" (lazy(sprintf "%+*d" 7 System.Int32.MinValue)) "-2147483648"
    test "test701" (lazy(sprintf "%-+d" System.Int32.MinValue)) "-2147483648"
    test "test702" (lazy(sprintf "%-+5d" System.Int32.MinValue)) "-2147483648"
    test "test703" (lazy(sprintf "%-+1d" System.Int32.MinValue)) "-2147483648"
    test "test704" (lazy(sprintf "%-+*d" 7 System.Int32.MinValue)) "-2147483648"
    test "test705" (lazy(sprintf "%+0d" System.Int32.MinValue)) "-2147483648"
    test "test706" (lazy(sprintf "%+05d" System.Int32.MinValue)) "-2147483648"
    test "test707" (lazy(sprintf "%+01d" System.Int32.MinValue)) "-2147483648"
    test "test708" (lazy(sprintf "%+0*d" 7 System.Int32.MinValue)) "-2147483648"
    test "test709" (lazy(sprintf "%-+0d" System.Int32.MinValue)) "-2147483648"
    test "test710" (lazy(sprintf "%-+05d" System.Int32.MinValue)) "-2147483648"
    test "test711" (lazy(sprintf "%-+01d" System.Int32.MinValue)) "-2147483648"
    test "test712" (lazy(sprintf "%-+0*d" 7 System.Int32.MinValue)) "-2147483648"
    test "test713" (lazy(sprintf "% d" System.Int32.MinValue)) "-2147483648"
    test "test714" (lazy(sprintf "% 5d" System.Int32.MinValue)) "-2147483648"
    test "test715" (lazy(sprintf "% 1d" System.Int32.MinValue)) "-2147483648"
    test "test716" (lazy(sprintf "% *d" 7 System.Int32.MinValue)) "-2147483648"
    test "test717" (lazy(sprintf "%- d" System.Int32.MinValue)) "-2147483648"
    test "test718" (lazy(sprintf "%- 5d" System.Int32.MinValue)) "-2147483648"
    test "test719" (lazy(sprintf "%- 1d" System.Int32.MinValue)) "-2147483648"
    test "test720" (lazy(sprintf "%- *d" 7 System.Int32.MinValue)) "-2147483648"
    test "test721" (lazy(sprintf "% 0d" System.Int32.MinValue)) "-2147483648"
    test "test722" (lazy(sprintf "% 05d" System.Int32.MinValue)) "-2147483648"
    test "test723" (lazy(sprintf "% 01d" System.Int32.MinValue)) "-2147483648"
    test "test724" (lazy(sprintf "% 0*d" 7 System.Int32.MinValue)) "-2147483648"
    test "test725" (lazy(sprintf "%- 0d" System.Int32.MinValue)) "-2147483648"
    test "test726" (lazy(sprintf "%- 05d" System.Int32.MinValue)) "-2147483648"
    test "test727" (lazy(sprintf "%- 01d" System.Int32.MinValue)) "-2147483648"
    test "test728" (lazy(sprintf "%- 0*d" 7 System.Int32.MinValue)) "-2147483648"
    test "test729" (lazy(sprintf "%d" System.Int64.MinValue)) "-9223372036854775808"
    test "test730" (lazy(sprintf "%5d" System.Int64.MinValue)) "-9223372036854775808"
    test "test731" (lazy(sprintf "%1d" System.Int64.MinValue)) "-9223372036854775808"
    test "test732" (lazy(sprintf "%*d" 7 System.Int64.MinValue)) "-9223372036854775808"
    test "test733" (lazy(sprintf "%-d" System.Int64.MinValue)) "-9223372036854775808"
    test "test734" (lazy(sprintf "%-5d" System.Int64.MinValue)) "-9223372036854775808"
    test "test735" (lazy(sprintf "%-1d" System.Int64.MinValue)) "-9223372036854775808"
    test "test736" (lazy(sprintf "%-*d" 7 System.Int64.MinValue)) "-9223372036854775808"
    test "test737" (lazy(sprintf "%0d" System.Int64.MinValue)) "-9223372036854775808"
    test "test738" (lazy(sprintf "%05d" System.Int64.MinValue)) "-9223372036854775808"
    test "test739" (lazy(sprintf "%01d" System.Int64.MinValue)) "-9223372036854775808"
    test "test740" (lazy(sprintf "%0*d" 7 System.Int64.MinValue)) "-9223372036854775808"
    test "test741" (lazy(sprintf "%-0d" System.Int64.MinValue)) "-9223372036854775808"
    test "test742" (lazy(sprintf "%-05d" System.Int64.MinValue)) "-9223372036854775808"
    test "test743" (lazy(sprintf "%-01d" System.Int64.MinValue)) "-9223372036854775808"
    test "test744" (lazy(sprintf "%-0*d" 7 System.Int64.MinValue)) "-9223372036854775808"
    test "test745" (lazy(sprintf "%+d" System.Int64.MinValue)) "-9223372036854775808"
    test "test746" (lazy(sprintf "%+5d" System.Int64.MinValue)) "-9223372036854775808"
    test "test747" (lazy(sprintf "%+1d" System.Int64.MinValue)) "-9223372036854775808"
    test "test748" (lazy(sprintf "%+*d" 7 System.Int64.MinValue)) "-9223372036854775808"
    test "test749" (lazy(sprintf "%-+d" System.Int64.MinValue)) "-9223372036854775808"
    test "test750" (lazy(sprintf "%-+5d" System.Int64.MinValue)) "-9223372036854775808"
    test "test751" (lazy(sprintf "%-+1d" System.Int64.MinValue)) "-9223372036854775808"
    test "test752" (lazy(sprintf "%-+*d" 7 System.Int64.MinValue)) "-9223372036854775808"
    test "test753" (lazy(sprintf "%+0d" System.Int64.MinValue)) "-9223372036854775808"
    test "test754" (lazy(sprintf "%+05d" System.Int64.MinValue)) "-9223372036854775808"
    test "test755" (lazy(sprintf "%+01d" System.Int64.MinValue)) "-9223372036854775808"
    test "test756" (lazy(sprintf "%+0*d" 7 System.Int64.MinValue)) "-9223372036854775808"
    test "test757" (lazy(sprintf "%-+0d" System.Int64.MinValue)) "-9223372036854775808"
    test "test758" (lazy(sprintf "%-+05d" System.Int64.MinValue)) "-9223372036854775808"
    test "test759" (lazy(sprintf "%-+01d" System.Int64.MinValue)) "-9223372036854775808"
    test "test760" (lazy(sprintf "%-+0*d" 7 System.Int64.MinValue)) "-9223372036854775808"
    test "test761" (lazy(sprintf "% d" System.Int64.MinValue)) "-9223372036854775808"
    test "test762" (lazy(sprintf "% 5d" System.Int64.MinValue)) "-9223372036854775808"
    test "test763" (lazy(sprintf "% 1d" System.Int64.MinValue)) "-9223372036854775808"
    test "test764" (lazy(sprintf "% *d" 7 System.Int64.MinValue)) "-9223372036854775808"
    test "test765" (lazy(sprintf "%- d" System.Int64.MinValue)) "-9223372036854775808"
    test "test766" (lazy(sprintf "%- 5d" System.Int64.MinValue)) "-9223372036854775808"
    test "test767" (lazy(sprintf "%- 1d" System.Int64.MinValue)) "-9223372036854775808"
    test "test768" (lazy(sprintf "%- *d" 7 System.Int64.MinValue)) "-9223372036854775808"
    test "test769" (lazy(sprintf "% 0d" System.Int64.MinValue)) "-9223372036854775808"
    test "test770" (lazy(sprintf "% 05d" System.Int64.MinValue)) "-9223372036854775808"
    test "test771" (lazy(sprintf "% 01d" System.Int64.MinValue)) "-9223372036854775808"
    test "test772" (lazy(sprintf "% 0*d" 7 System.Int64.MinValue)) "-9223372036854775808"
    test "test773" (lazy(sprintf "%- 0d" System.Int64.MinValue)) "-9223372036854775808"
    test "test774" (lazy(sprintf "%- 05d" System.Int64.MinValue)) "-9223372036854775808"
    test "test775" (lazy(sprintf "%- 01d" System.Int64.MinValue)) "-9223372036854775808"
    test "test776" (lazy(sprintf "%- 0*d" 7 System.Int64.MinValue)) "-9223372036854775808"
    test "test777" (lazy(sprintf "%d" 55n)) "55"
    test "test778" (lazy(sprintf "%5d" 55n)) "   55"
    test "test779" (lazy(sprintf "%1d" 55n)) "55"
    test "test780" (lazy(sprintf "%*d" 7 55n)) "     55"
    test "test781" (lazy(sprintf "%-d" 55n)) "55"
    test "test782" (lazy(sprintf "%-5d" 55n)) "55   "
    test "test783" (lazy(sprintf "%-1d" 55n)) "55"
    test "test784" (lazy(sprintf "%-*d" 7 55n)) "55     "
    test "test785" (lazy(sprintf "%0d" 55n)) "55"
    test "test786" (lazy(sprintf "%05d" 55n)) "00055"
    test "test787" (lazy(sprintf "%01d" 55n)) "55"
    test "test788" (lazy(sprintf "%0*d" 7 55n)) "0000055"
    test "test789" (lazy(sprintf "%-0d" 55n)) "55"
    test "test790" (lazy(sprintf "%-05d" 55n)) "55   "
    test "test791" (lazy(sprintf "%-01d" 55n)) "55"
    test "test792" (lazy(sprintf "%-0*d" 7 55n)) "55     "
    test "test793" (lazy(sprintf "%+d" 55n)) "+55"
    test "test794" (lazy(sprintf "%+5d" 55n)) "  +55"
    test "test795" (lazy(sprintf "%+1d" 55n)) "+55"
    test "test796" (lazy(sprintf "%+*d" 7 55n)) "    +55"
    test "test797" (lazy(sprintf "%-+d" 55n)) "+55"
    test "test798" (lazy(sprintf "%-+5d" 55n)) "+55  "
    test "test799" (lazy(sprintf "%-+1d" 55n)) "+55"
    test "test800" (lazy(sprintf "%-+*d" 7 55n)) "+55    "
    test "test801" (lazy(sprintf "%+0d" 55n)) "+55"
    test "test802" (lazy(sprintf "%+05d" 55n)) "+0055"
    test "test803" (lazy(sprintf "%+01d" 55n)) "+55"
    test "test804" (lazy(sprintf "%+0*d" 7 55n)) "+000055"
    test "test805" (lazy(sprintf "%-+0d" 55n)) "+55"
    test "test806" (lazy(sprintf "%-+05d" 55n)) "+55  "
    test "test807" (lazy(sprintf "%-+01d" 55n)) "+55"
    test "test808" (lazy(sprintf "%-+0*d" 7 55n)) "+55    "
    test "test809" (lazy(sprintf "% d" 55n)) " 55"
    test "test810" (lazy(sprintf "% 5d" 55n)) "   55"
    test "test811" (lazy(sprintf "% 1d" 55n)) " 55"
    test "test812" (lazy(sprintf "% *d" 7 55n)) "     55"
    test "test813" (lazy(sprintf "%- d" 55n)) " 55"
    test "test814" (lazy(sprintf "%- 5d" 55n)) " 55  "
    test "test815" (lazy(sprintf "%- 1d" 55n)) " 55"
    test "test816" (lazy(sprintf "%- *d" 7 55n)) " 55    "
    test "test817" (lazy(sprintf "% 0d" 55n)) " 55"
    test "test818" (lazy(sprintf "% 05d" 55n)) " 0055"
    test "test819" (lazy(sprintf "% 01d" 55n)) " 55"
    test "test820" (lazy(sprintf "% 0*d" 7 55n)) " 000055"
    test "test821" (lazy(sprintf "%- 0d" 55n)) " 55"
    test "test822" (lazy(sprintf "%- 05d" 55n)) " 55  "
    test "test823" (lazy(sprintf "%- 01d" 55n)) " 55"
    test "test824" (lazy(sprintf "%- 0*d" 7 55n)) " 55    "
    test "test825" (lazy(sprintf "%d" 999un)) "999"
    test "test826" (lazy(sprintf "%5d" 999un)) "  999"
    test "test827" (lazy(sprintf "%1d" 999un)) "999"
    test "test828" (lazy(sprintf "%*d" 7 999un)) "    999"
    test "test829" (lazy(sprintf "%-d" 999un)) "999"
    test "test830" (lazy(sprintf "%-5d" 999un)) "999  "
    test "test831" (lazy(sprintf "%-1d" 999un)) "999"
    test "test832" (lazy(sprintf "%-*d" 7 999un)) "999    "
    test "test833" (lazy(sprintf "%0d" 999un)) "999"
    test "test834" (lazy(sprintf "%05d" 999un)) "00999"
    test "test835" (lazy(sprintf "%01d" 999un)) "999"
    test "test836" (lazy(sprintf "%0*d" 7 999un)) "0000999"
    test "test837" (lazy(sprintf "%-0d" 999un)) "999"
    test "test838" (lazy(sprintf "%-05d" 999un)) "999  "
    test "test839" (lazy(sprintf "%-01d" 999un)) "999"
    test "test840" (lazy(sprintf "%-0*d" 7 999un)) "999    "
    test "test841" (lazy(sprintf "%+d" 999un)) "+999"
    test "test842" (lazy(sprintf "%+5d" 999un)) " +999"
    test "test843" (lazy(sprintf "%+1d" 999un)) "+999"
    test "test844" (lazy(sprintf "%+*d" 7 999un)) "   +999"
    test "test845" (lazy(sprintf "%-+d" 999un)) "+999"
    test "test846" (lazy(sprintf "%-+5d" 999un)) "+999 "
    test "test847" (lazy(sprintf "%-+1d" 999un)) "+999"
    test "test848" (lazy(sprintf "%-+*d" 7 999un)) "+999   "
    test "test849" (lazy(sprintf "%+0d" 999un)) "+999"
    test "test850" (lazy(sprintf "%+05d" 999un)) "+0999"
    test "test851" (lazy(sprintf "%+01d" 999un)) "+999"
    test "test852" (lazy(sprintf "%+0*d" 7 999un)) "+000999"
    test "test853" (lazy(sprintf "%-+0d" 999un)) "+999"
    test "test854" (lazy(sprintf "%-+05d" 999un)) "+999 "
    test "test855" (lazy(sprintf "%-+01d" 999un)) "+999"
    test "test856" (lazy(sprintf "%-+0*d" 7 999un)) "+999   "
    test "test857" (lazy(sprintf "% d" 999un)) " 999"
    test "test858" (lazy(sprintf "% 5d" 999un)) "  999"
    test "test859" (lazy(sprintf "% 1d" 999un)) " 999"
    test "test860" (lazy(sprintf "% *d" 7 999un)) "    999"
    test "test861" (lazy(sprintf "%- d" 999un)) " 999"
    test "test862" (lazy(sprintf "%- 5d" 999un)) " 999 "
    test "test863" (lazy(sprintf "%- 1d" 999un)) " 999"
    test "test864" (lazy(sprintf "%- *d" 7 999un)) " 999   "
    test "test865" (lazy(sprintf "% 0d" 999un)) " 999"
    test "test866" (lazy(sprintf "% 05d" 999un)) " 0999"
    test "test867" (lazy(sprintf "% 01d" 999un)) " 999"
    test "test868" (lazy(sprintf "% 0*d" 7 999un)) " 000999"
    test "test869" (lazy(sprintf "%- 0d" 999un)) " 999"
    test "test870" (lazy(sprintf "%- 05d" 999un)) " 999 "
    test "test871" (lazy(sprintf "%- 01d" 999un)) " 999"
    test "test872" (lazy(sprintf "%- 0*d" 7 999un)) " 999   "
    test "test873" (lazy(sprintf "%i" 14)) "14"
    test "test874" (lazy(sprintf "%5i" 14)) "   14"
    test "test875" (lazy(sprintf "%1i" 14)) "14"
    test "test876" (lazy(sprintf "%*i" 7 14)) "     14"
    test "test877" (lazy(sprintf "%-i" 14)) "14"
    test "test878" (lazy(sprintf "%-5i" 14)) "14   "
    test "test879" (lazy(sprintf "%-1i" 14)) "14"
    test "test880" (lazy(sprintf "%-*i" 7 14)) "14     "
    test "test881" (lazy(sprintf "%0i" 14)) "14"
    test "test882" (lazy(sprintf "%05i" 14)) "00014"
    test "test883" (lazy(sprintf "%01i" 14)) "14"
    test "test884" (lazy(sprintf "%0*i" 7 14)) "0000014"
    test "test885" (lazy(sprintf "%-0i" 14)) "14"
    test "test886" (lazy(sprintf "%-05i" 14)) "14   "
    test "test887" (lazy(sprintf "%-01i" 14)) "14"
    test "test888" (lazy(sprintf "%-0*i" 7 14)) "14     "
    test "test889" (lazy(sprintf "%+i" 14)) "+14"
    test "test890" (lazy(sprintf "%+5i" 14)) "  +14"
    test "test891" (lazy(sprintf "%+1i" 14)) "+14"
    test "test892" (lazy(sprintf "%+*i" 7 14)) "    +14"
    test "test893" (lazy(sprintf "%-+i" 14)) "+14"
    test "test894" (lazy(sprintf "%-+5i" 14)) "+14  "
    test "test895" (lazy(sprintf "%-+1i" 14)) "+14"
    test "test896" (lazy(sprintf "%-+*i" 7 14)) "+14    "
    test "test897" (lazy(sprintf "%+0i" 14)) "+14"
    test "test898" (lazy(sprintf "%+05i" 14)) "+0014"
    test "test899" (lazy(sprintf "%+01i" 14)) "+14"
    test "test900" (lazy(sprintf "%+0*i" 7 14)) "+000014"
    test "test901" (lazy(sprintf "%-+0i" 14)) "+14"
    test "test902" (lazy(sprintf "%-+05i" 14)) "+14  "
    test "test903" (lazy(sprintf "%-+01i" 14)) "+14"
    test "test904" (lazy(sprintf "%-+0*i" 7 14)) "+14    "
    test "test905" (lazy(sprintf "% i" 14)) " 14"
    test "test906" (lazy(sprintf "% 5i" 14)) "   14"
    test "test907" (lazy(sprintf "% 1i" 14)) " 14"
    test "test908" (lazy(sprintf "% *i" 7 14)) "     14"
    test "test909" (lazy(sprintf "%- i" 14)) " 14"
    test "test910" (lazy(sprintf "%- 5i" 14)) " 14  "
    test "test911" (lazy(sprintf "%- 1i" 14)) " 14"
    test "test912" (lazy(sprintf "%- *i" 7 14)) " 14    "
    test "test913" (lazy(sprintf "% 0i" 14)) " 14"
    test "test914" (lazy(sprintf "% 05i" 14)) " 0014"
    test "test915" (lazy(sprintf "% 01i" 14)) " 14"
    test "test916" (lazy(sprintf "% 0*i" 7 14)) " 000014"
    test "test917" (lazy(sprintf "%- 0i" 14)) " 14"
    test "test918" (lazy(sprintf "%- 05i" 14)) " 14  "
    test "test919" (lazy(sprintf "%- 01i" 14)) " 14"
    test "test920" (lazy(sprintf "%- 0*i" 7 14)) " 14    "
    test "test921" (lazy(sprintf "%i" -10)) "-10"
    test "test922" (lazy(sprintf "%5i" -10)) "  -10"
    test "test923" (lazy(sprintf "%1i" -10)) "-10"
    test "test924" (lazy(sprintf "%*i" 7 -10)) "    -10"
    test "test925" (lazy(sprintf "%-i" -10)) "-10"
    test "test926" (lazy(sprintf "%-5i" -10)) "-10  "
    test "test927" (lazy(sprintf "%-1i" -10)) "-10"
    test "test928" (lazy(sprintf "%-*i" 7 -10)) "-10    "
    test "test929" (lazy(sprintf "%0i" -10)) "-10"
    test "test930" (lazy(sprintf "%05i" -10)) "-0010"
    test "test931" (lazy(sprintf "%01i" -10)) "-10"
    test "test932" (lazy(sprintf "%0*i" 7 -10)) "-000010"
    test "test933" (lazy(sprintf "%-0i" -10)) "-10"
    test "test934" (lazy(sprintf "%-05i" -10)) "-10  "
    test "test935" (lazy(sprintf "%-01i" -10)) "-10"
    test "test936" (lazy(sprintf "%-0*i" 7 -10)) "-10    "
    test "test937" (lazy(sprintf "%+i" -10)) "-10"
    test "test938" (lazy(sprintf "%+5i" -10)) "  -10"
    test "test939" (lazy(sprintf "%+1i" -10)) "-10"
    test "test940" (lazy(sprintf "%+*i" 7 -10)) "    -10"
    test "test941" (lazy(sprintf "%-+i" -10)) "-10"
    test "test942" (lazy(sprintf "%-+5i" -10)) "-10  "
    test "test943" (lazy(sprintf "%-+1i" -10)) "-10"
    test "test944" (lazy(sprintf "%-+*i" 7 -10)) "-10    "
    test "test945" (lazy(sprintf "%+0i" -10)) "-10"
    test "test946" (lazy(sprintf "%+05i" -10)) "-0010"
    test "test947" (lazy(sprintf "%+01i" -10)) "-10"
    test "test948" (lazy(sprintf "%+0*i" 7 -10)) "-000010"
    test "test949" (lazy(sprintf "%-+0i" -10)) "-10"
    test "test950" (lazy(sprintf "%-+05i" -10)) "-10  "
    test "test951" (lazy(sprintf "%-+01i" -10)) "-10"
    test "test952" (lazy(sprintf "%-+0*i" 7 -10)) "-10    "
    test "test953" (lazy(sprintf "% i" -10)) "-10"
    test "test954" (lazy(sprintf "% 5i" -10)) "  -10"
    test "test955" (lazy(sprintf "% 1i" -10)) "-10"
    test "test956" (lazy(sprintf "% *i" 7 -10)) "    -10"
    test "test957" (lazy(sprintf "%- i" -10)) "-10"
    test "test958" (lazy(sprintf "%- 5i" -10)) "-10  "
    test "test959" (lazy(sprintf "%- 1i" -10)) "-10"
    test "test960" (lazy(sprintf "%- *i" 7 -10)) "-10    "
    test "test961" (lazy(sprintf "% 0i" -10)) "-10"
    test "test962" (lazy(sprintf "% 05i" -10)) "-0010"
    test "test963" (lazy(sprintf "% 01i" -10)) "-10"
    test "test964" (lazy(sprintf "% 0*i" 7 -10)) "-000010"
    test "test965" (lazy(sprintf "%- 0i" -10)) "-10"
    test "test966" (lazy(sprintf "%- 05i" -10)) "-10  "
    test "test967" (lazy(sprintf "%- 01i" -10)) "-10"
    test "test968" (lazy(sprintf "%- 0*i" 7 -10)) "-10    "
    test "test969" (lazy(sprintf "%i" 55s)) "55"
    test "test970" (lazy(sprintf "%5i" 55s)) "   55"
    test "test971" (lazy(sprintf "%1i" 55s)) "55"
    test "test972" (lazy(sprintf "%*i" 7 55s)) "     55"
    test "test973" (lazy(sprintf "%-i" 55s)) "55"
    test "test974" (lazy(sprintf "%-5i" 55s)) "55   "
    test "test975" (lazy(sprintf "%-1i" 55s)) "55"
    test "test976" (lazy(sprintf "%-*i" 7 55s)) "55     "
    test "test977" (lazy(sprintf "%0i" 55s)) "55"
    test "test978" (lazy(sprintf "%05i" 55s)) "00055"
    test "test979" (lazy(sprintf "%01i" 55s)) "55"
    test "test980" (lazy(sprintf "%0*i" 7 55s)) "0000055"
    test "test981" (lazy(sprintf "%-0i" 55s)) "55"
    test "test982" (lazy(sprintf "%-05i" 55s)) "55   "
    test "test983" (lazy(sprintf "%-01i" 55s)) "55"
    test "test984" (lazy(sprintf "%-0*i" 7 55s)) "55     "
    test "test985" (lazy(sprintf "%+i" 55s)) "+55"
    test "test986" (lazy(sprintf "%+5i" 55s)) "  +55"
    test "test987" (lazy(sprintf "%+1i" 55s)) "+55"
    test "test988" (lazy(sprintf "%+*i" 7 55s)) "    +55"
    test "test989" (lazy(sprintf "%-+i" 55s)) "+55"
    test "test990" (lazy(sprintf "%-+5i" 55s)) "+55  "
    test "test991" (lazy(sprintf "%-+1i" 55s)) "+55"
    test "test992" (lazy(sprintf "%-+*i" 7 55s)) "+55    "
    test "test993" (lazy(sprintf "%+0i" 55s)) "+55"
    test "test994" (lazy(sprintf "%+05i" 55s)) "+0055"
    test "test995" (lazy(sprintf "%+01i" 55s)) "+55"
    test "test996" (lazy(sprintf "%+0*i" 7 55s)) "+000055"
    test "test997" (lazy(sprintf "%-+0i" 55s)) "+55"
    test "test998" (lazy(sprintf "%-+05i" 55s)) "+55  "
    test "test999" (lazy(sprintf "%-+01i" 55s)) "+55"
    test "test1000" (lazy(sprintf "%-+0*i" 7 55s)) "+55    "
let func1000()=
    test "test1001" (lazy(sprintf "% i" 55s)) " 55"
    test "test1002" (lazy(sprintf "% 5i" 55s)) "   55"
    test "test1003" (lazy(sprintf "% 1i" 55s)) " 55"
    test "test1004" (lazy(sprintf "% *i" 7 55s)) "     55"
    test "test1005" (lazy(sprintf "%- i" 55s)) " 55"
    test "test1006" (lazy(sprintf "%- 5i" 55s)) " 55  "
    test "test1007" (lazy(sprintf "%- 1i" 55s)) " 55"
    test "test1008" (lazy(sprintf "%- *i" 7 55s)) " 55    "
    test "test1009" (lazy(sprintf "% 0i" 55s)) " 55"
    test "test1010" (lazy(sprintf "% 05i" 55s)) " 0055"
    test "test1011" (lazy(sprintf "% 01i" 55s)) " 55"
    test "test1012" (lazy(sprintf "% 0*i" 7 55s)) " 000055"
    test "test1013" (lazy(sprintf "%- 0i" 55s)) " 55"
    test "test1014" (lazy(sprintf "%- 05i" 55s)) " 55  "
    test "test1015" (lazy(sprintf "%- 01i" 55s)) " 55"
    test "test1016" (lazy(sprintf "%- 0*i" 7 55s)) " 55    "
    test "test1017" (lazy(sprintf "%i" -88s)) "-88"
    test "test1018" (lazy(sprintf "%5i" -88s)) "  -88"
    test "test1019" (lazy(sprintf "%1i" -88s)) "-88"
    test "test1020" (lazy(sprintf "%*i" 7 -88s)) "    -88"
    test "test1021" (lazy(sprintf "%-i" -88s)) "-88"
    test "test1022" (lazy(sprintf "%-5i" -88s)) "-88  "
    test "test1023" (lazy(sprintf "%-1i" -88s)) "-88"
    test "test1024" (lazy(sprintf "%-*i" 7 -88s)) "-88    "
    test "test1025" (lazy(sprintf "%0i" -88s)) "-88"
    test "test1026" (lazy(sprintf "%05i" -88s)) "-0088"
    test "test1027" (lazy(sprintf "%01i" -88s)) "-88"
    test "test1028" (lazy(sprintf "%0*i" 7 -88s)) "-000088"
    test "test1029" (lazy(sprintf "%-0i" -88s)) "-88"
    test "test1030" (lazy(sprintf "%-05i" -88s)) "-88  "
    test "test1031" (lazy(sprintf "%-01i" -88s)) "-88"
    test "test1032" (lazy(sprintf "%-0*i" 7 -88s)) "-88    "
    test "test1033" (lazy(sprintf "%+i" -88s)) "-88"
    test "test1034" (lazy(sprintf "%+5i" -88s)) "  -88"
    test "test1035" (lazy(sprintf "%+1i" -88s)) "-88"
    test "test1036" (lazy(sprintf "%+*i" 7 -88s)) "    -88"
    test "test1037" (lazy(sprintf "%-+i" -88s)) "-88"
    test "test1038" (lazy(sprintf "%-+5i" -88s)) "-88  "
    test "test1039" (lazy(sprintf "%-+1i" -88s)) "-88"
    test "test1040" (lazy(sprintf "%-+*i" 7 -88s)) "-88    "
    test "test1041" (lazy(sprintf "%+0i" -88s)) "-88"
    test "test1042" (lazy(sprintf "%+05i" -88s)) "-0088"
    test "test1043" (lazy(sprintf "%+01i" -88s)) "-88"
    test "test1044" (lazy(sprintf "%+0*i" 7 -88s)) "-000088"
    test "test1045" (lazy(sprintf "%-+0i" -88s)) "-88"
    test "test1046" (lazy(sprintf "%-+05i" -88s)) "-88  "
    test "test1047" (lazy(sprintf "%-+01i" -88s)) "-88"
    test "test1048" (lazy(sprintf "%-+0*i" 7 -88s)) "-88    "
    test "test1049" (lazy(sprintf "% i" -88s)) "-88"
    test "test1050" (lazy(sprintf "% 5i" -88s)) "  -88"
    test "test1051" (lazy(sprintf "% 1i" -88s)) "-88"
    test "test1052" (lazy(sprintf "% *i" 7 -88s)) "    -88"
    test "test1053" (lazy(sprintf "%- i" -88s)) "-88"
    test "test1054" (lazy(sprintf "%- 5i" -88s)) "-88  "
    test "test1055" (lazy(sprintf "%- 1i" -88s)) "-88"
    test "test1056" (lazy(sprintf "%- *i" 7 -88s)) "-88    "
    test "test1057" (lazy(sprintf "% 0i" -88s)) "-88"
    test "test1058" (lazy(sprintf "% 05i" -88s)) "-0088"
    test "test1059" (lazy(sprintf "% 01i" -88s)) "-88"
    test "test1060" (lazy(sprintf "% 0*i" 7 -88s)) "-000088"
    test "test1061" (lazy(sprintf "%- 0i" -88s)) "-88"
    test "test1062" (lazy(sprintf "%- 05i" -88s)) "-88  "
    test "test1063" (lazy(sprintf "%- 01i" -88s)) "-88"
    test "test1064" (lazy(sprintf "%- 0*i" 7 -88s)) "-88    "
    test "test1065" (lazy(sprintf "%i" 104us)) "104"
    test "test1066" (lazy(sprintf "%5i" 104us)) "  104"
    test "test1067" (lazy(sprintf "%1i" 104us)) "104"
    test "test1068" (lazy(sprintf "%*i" 7 104us)) "    104"
    test "test1069" (lazy(sprintf "%-i" 104us)) "104"
    test "test1070" (lazy(sprintf "%-5i" 104us)) "104  "
    test "test1071" (lazy(sprintf "%-1i" 104us)) "104"
    test "test1072" (lazy(sprintf "%-*i" 7 104us)) "104    "
    test "test1073" (lazy(sprintf "%0i" 104us)) "104"
    test "test1074" (lazy(sprintf "%05i" 104us)) "00104"
    test "test1075" (lazy(sprintf "%01i" 104us)) "104"
    test "test1076" (lazy(sprintf "%0*i" 7 104us)) "0000104"
    test "test1077" (lazy(sprintf "%-0i" 104us)) "104"
    test "test1078" (lazy(sprintf "%-05i" 104us)) "104  "
    test "test1079" (lazy(sprintf "%-01i" 104us)) "104"
    test "test1080" (lazy(sprintf "%-0*i" 7 104us)) "104    "
    test "test1081" (lazy(sprintf "%+i" 104us)) "+104"
    test "test1082" (lazy(sprintf "%+5i" 104us)) " +104"
    test "test1083" (lazy(sprintf "%+1i" 104us)) "+104"
    test "test1084" (lazy(sprintf "%+*i" 7 104us)) "   +104"
    test "test1085" (lazy(sprintf "%-+i" 104us)) "+104"
    test "test1086" (lazy(sprintf "%-+5i" 104us)) "+104 "
    test "test1087" (lazy(sprintf "%-+1i" 104us)) "+104"
    test "test1088" (lazy(sprintf "%-+*i" 7 104us)) "+104   "
    test "test1089" (lazy(sprintf "%+0i" 104us)) "+104"
    test "test1090" (lazy(sprintf "%+05i" 104us)) "+0104"
    test "test1091" (lazy(sprintf "%+01i" 104us)) "+104"
    test "test1092" (lazy(sprintf "%+0*i" 7 104us)) "+000104"
    test "test1093" (lazy(sprintf "%-+0i" 104us)) "+104"
    test "test1094" (lazy(sprintf "%-+05i" 104us)) "+104 "
    test "test1095" (lazy(sprintf "%-+01i" 104us)) "+104"
    test "test1096" (lazy(sprintf "%-+0*i" 7 104us)) "+104   "
    test "test1097" (lazy(sprintf "% i" 104us)) " 104"
    test "test1098" (lazy(sprintf "% 5i" 104us)) "  104"
    test "test1099" (lazy(sprintf "% 1i" 104us)) " 104"
    test "test1100" (lazy(sprintf "% *i" 7 104us)) "    104"
    test "test1101" (lazy(sprintf "%- i" 104us)) " 104"
    test "test1102" (lazy(sprintf "%- 5i" 104us)) " 104 "
    test "test1103" (lazy(sprintf "%- 1i" 104us)) " 104"
    test "test1104" (lazy(sprintf "%- *i" 7 104us)) " 104   "
    test "test1105" (lazy(sprintf "% 0i" 104us)) " 104"
    test "test1106" (lazy(sprintf "% 05i" 104us)) " 0104"
    test "test1107" (lazy(sprintf "% 01i" 104us)) " 104"
    test "test1108" (lazy(sprintf "% 0*i" 7 104us)) " 000104"
    test "test1109" (lazy(sprintf "%- 0i" 104us)) " 104"
    test "test1110" (lazy(sprintf "%- 05i" 104us)) " 104 "
    test "test1111" (lazy(sprintf "%- 01i" 104us)) " 104"
    test "test1112" (lazy(sprintf "%- 0*i" 7 104us)) " 104   "
    test "test1113" (lazy(sprintf "%i" 12y)) "12"
    test "test1114" (lazy(sprintf "%5i" 12y)) "   12"
    test "test1115" (lazy(sprintf "%1i" 12y)) "12"
    test "test1116" (lazy(sprintf "%*i" 7 12y)) "     12"
    test "test1117" (lazy(sprintf "%-i" 12y)) "12"
    test "test1118" (lazy(sprintf "%-5i" 12y)) "12   "
    test "test1119" (lazy(sprintf "%-1i" 12y)) "12"
    test "test1120" (lazy(sprintf "%-*i" 7 12y)) "12     "
    test "test1121" (lazy(sprintf "%0i" 12y)) "12"
    test "test1122" (lazy(sprintf "%05i" 12y)) "00012"
    test "test1123" (lazy(sprintf "%01i" 12y)) "12"
    test "test1124" (lazy(sprintf "%0*i" 7 12y)) "0000012"
    test "test1125" (lazy(sprintf "%-0i" 12y)) "12"
    test "test1126" (lazy(sprintf "%-05i" 12y)) "12   "
    test "test1127" (lazy(sprintf "%-01i" 12y)) "12"
    test "test1128" (lazy(sprintf "%-0*i" 7 12y)) "12     "
    test "test1129" (lazy(sprintf "%+i" 12y)) "+12"
    test "test1130" (lazy(sprintf "%+5i" 12y)) "  +12"
    test "test1131" (lazy(sprintf "%+1i" 12y)) "+12"
    test "test1132" (lazy(sprintf "%+*i" 7 12y)) "    +12"
    test "test1133" (lazy(sprintf "%-+i" 12y)) "+12"
    test "test1134" (lazy(sprintf "%-+5i" 12y)) "+12  "
    test "test1135" (lazy(sprintf "%-+1i" 12y)) "+12"
    test "test1136" (lazy(sprintf "%-+*i" 7 12y)) "+12    "
    test "test1137" (lazy(sprintf "%+0i" 12y)) "+12"
    test "test1138" (lazy(sprintf "%+05i" 12y)) "+0012"
    test "test1139" (lazy(sprintf "%+01i" 12y)) "+12"
    test "test1140" (lazy(sprintf "%+0*i" 7 12y)) "+000012"
    test "test1141" (lazy(sprintf "%-+0i" 12y)) "+12"
    test "test1142" (lazy(sprintf "%-+05i" 12y)) "+12  "
    test "test1143" (lazy(sprintf "%-+01i" 12y)) "+12"
    test "test1144" (lazy(sprintf "%-+0*i" 7 12y)) "+12    "
    test "test1145" (lazy(sprintf "% i" 12y)) " 12"
    test "test1146" (lazy(sprintf "% 5i" 12y)) "   12"
    test "test1147" (lazy(sprintf "% 1i" 12y)) " 12"
    test "test1148" (lazy(sprintf "% *i" 7 12y)) "     12"
    test "test1149" (lazy(sprintf "%- i" 12y)) " 12"
    test "test1150" (lazy(sprintf "%- 5i" 12y)) " 12  "
    test "test1151" (lazy(sprintf "%- 1i" 12y)) " 12"
    test "test1152" (lazy(sprintf "%- *i" 7 12y)) " 12    "
    test "test1153" (lazy(sprintf "% 0i" 12y)) " 12"
    test "test1154" (lazy(sprintf "% 05i" 12y)) " 0012"
    test "test1155" (lazy(sprintf "% 01i" 12y)) " 12"
    test "test1156" (lazy(sprintf "% 0*i" 7 12y)) " 000012"
    test "test1157" (lazy(sprintf "%- 0i" 12y)) " 12"
    test "test1158" (lazy(sprintf "%- 05i" 12y)) " 12  "
    test "test1159" (lazy(sprintf "%- 01i" 12y)) " 12"
    test "test1160" (lazy(sprintf "%- 0*i" 7 12y)) " 12    "
    test "test1161" (lazy(sprintf "%i" -94y)) "-94"
    test "test1162" (lazy(sprintf "%5i" -94y)) "  -94"
    test "test1163" (lazy(sprintf "%1i" -94y)) "-94"
    test "test1164" (lazy(sprintf "%*i" 7 -94y)) "    -94"
    test "test1165" (lazy(sprintf "%-i" -94y)) "-94"
    test "test1166" (lazy(sprintf "%-5i" -94y)) "-94  "
    test "test1167" (lazy(sprintf "%-1i" -94y)) "-94"
    test "test1168" (lazy(sprintf "%-*i" 7 -94y)) "-94    "
    test "test1169" (lazy(sprintf "%0i" -94y)) "-94"
    test "test1170" (lazy(sprintf "%05i" -94y)) "-0094"
    test "test1171" (lazy(sprintf "%01i" -94y)) "-94"
    test "test1172" (lazy(sprintf "%0*i" 7 -94y)) "-000094"
    test "test1173" (lazy(sprintf "%-0i" -94y)) "-94"
    test "test1174" (lazy(sprintf "%-05i" -94y)) "-94  "
    test "test1175" (lazy(sprintf "%-01i" -94y)) "-94"
    test "test1176" (lazy(sprintf "%-0*i" 7 -94y)) "-94    "
    test "test1177" (lazy(sprintf "%+i" -94y)) "-94"
    test "test1178" (lazy(sprintf "%+5i" -94y)) "  -94"
    test "test1179" (lazy(sprintf "%+1i" -94y)) "-94"
    test "test1180" (lazy(sprintf "%+*i" 7 -94y)) "    -94"
    test "test1181" (lazy(sprintf "%-+i" -94y)) "-94"
    test "test1182" (lazy(sprintf "%-+5i" -94y)) "-94  "
    test "test1183" (lazy(sprintf "%-+1i" -94y)) "-94"
    test "test1184" (lazy(sprintf "%-+*i" 7 -94y)) "-94    "
    test "test1185" (lazy(sprintf "%+0i" -94y)) "-94"
    test "test1186" (lazy(sprintf "%+05i" -94y)) "-0094"
    test "test1187" (lazy(sprintf "%+01i" -94y)) "-94"
    test "test1188" (lazy(sprintf "%+0*i" 7 -94y)) "-000094"
    test "test1189" (lazy(sprintf "%-+0i" -94y)) "-94"
    test "test1190" (lazy(sprintf "%-+05i" -94y)) "-94  "
    test "test1191" (lazy(sprintf "%-+01i" -94y)) "-94"
    test "test1192" (lazy(sprintf "%-+0*i" 7 -94y)) "-94    "
    test "test1193" (lazy(sprintf "% i" -94y)) "-94"
    test "test1194" (lazy(sprintf "% 5i" -94y)) "  -94"
    test "test1195" (lazy(sprintf "% 1i" -94y)) "-94"
    test "test1196" (lazy(sprintf "% *i" 7 -94y)) "    -94"
    test "test1197" (lazy(sprintf "%- i" -94y)) "-94"
    test "test1198" (lazy(sprintf "%- 5i" -94y)) "-94  "
    test "test1199" (lazy(sprintf "%- 1i" -94y)) "-94"
    test "test1200" (lazy(sprintf "%- *i" 7 -94y)) "-94    "
    test "test1201" (lazy(sprintf "% 0i" -94y)) "-94"
    test "test1202" (lazy(sprintf "% 05i" -94y)) "-0094"
    test "test1203" (lazy(sprintf "% 01i" -94y)) "-94"
    test "test1204" (lazy(sprintf "% 0*i" 7 -94y)) "-000094"
    test "test1205" (lazy(sprintf "%- 0i" -94y)) "-94"
    test "test1206" (lazy(sprintf "%- 05i" -94y)) "-94  "
    test "test1207" (lazy(sprintf "%- 01i" -94y)) "-94"
    test "test1208" (lazy(sprintf "%- 0*i" 7 -94y)) "-94    "
    test "test1209" (lazy(sprintf "%i" 88uy)) "88"
    test "test1210" (lazy(sprintf "%5i" 88uy)) "   88"
    test "test1211" (lazy(sprintf "%1i" 88uy)) "88"
    test "test1212" (lazy(sprintf "%*i" 7 88uy)) "     88"
    test "test1213" (lazy(sprintf "%-i" 88uy)) "88"
    test "test1214" (lazy(sprintf "%-5i" 88uy)) "88   "
    test "test1215" (lazy(sprintf "%-1i" 88uy)) "88"
    test "test1216" (lazy(sprintf "%-*i" 7 88uy)) "88     "
    test "test1217" (lazy(sprintf "%0i" 88uy)) "88"
    test "test1218" (lazy(sprintf "%05i" 88uy)) "00088"
    test "test1219" (lazy(sprintf "%01i" 88uy)) "88"
    test "test1220" (lazy(sprintf "%0*i" 7 88uy)) "0000088"
    test "test1221" (lazy(sprintf "%-0i" 88uy)) "88"
    test "test1222" (lazy(sprintf "%-05i" 88uy)) "88   "
    test "test1223" (lazy(sprintf "%-01i" 88uy)) "88"
    test "test1224" (lazy(sprintf "%-0*i" 7 88uy)) "88     "
    test "test1225" (lazy(sprintf "%+i" 88uy)) "+88"
    test "test1226" (lazy(sprintf "%+5i" 88uy)) "  +88"
    test "test1227" (lazy(sprintf "%+1i" 88uy)) "+88"
    test "test1228" (lazy(sprintf "%+*i" 7 88uy)) "    +88"
    test "test1229" (lazy(sprintf "%-+i" 88uy)) "+88"
    test "test1230" (lazy(sprintf "%-+5i" 88uy)) "+88  "
    test "test1231" (lazy(sprintf "%-+1i" 88uy)) "+88"
    test "test1232" (lazy(sprintf "%-+*i" 7 88uy)) "+88    "
    test "test1233" (lazy(sprintf "%+0i" 88uy)) "+88"
    test "test1234" (lazy(sprintf "%+05i" 88uy)) "+0088"
    test "test1235" (lazy(sprintf "%+01i" 88uy)) "+88"
    test "test1236" (lazy(sprintf "%+0*i" 7 88uy)) "+000088"
    test "test1237" (lazy(sprintf "%-+0i" 88uy)) "+88"
    test "test1238" (lazy(sprintf "%-+05i" 88uy)) "+88  "
    test "test1239" (lazy(sprintf "%-+01i" 88uy)) "+88"
    test "test1240" (lazy(sprintf "%-+0*i" 7 88uy)) "+88    "
    test "test1241" (lazy(sprintf "% i" 88uy)) " 88"
    test "test1242" (lazy(sprintf "% 5i" 88uy)) "   88"
    test "test1243" (lazy(sprintf "% 1i" 88uy)) " 88"
    test "test1244" (lazy(sprintf "% *i" 7 88uy)) "     88"
    test "test1245" (lazy(sprintf "%- i" 88uy)) " 88"
    test "test1246" (lazy(sprintf "%- 5i" 88uy)) " 88  "
    test "test1247" (lazy(sprintf "%- 1i" 88uy)) " 88"
    test "test1248" (lazy(sprintf "%- *i" 7 88uy)) " 88    "
    test "test1249" (lazy(sprintf "% 0i" 88uy)) " 88"
    test "test1250" (lazy(sprintf "% 05i" 88uy)) " 0088"
    test "test1251" (lazy(sprintf "% 01i" 88uy)) " 88"
    test "test1252" (lazy(sprintf "% 0*i" 7 88uy)) " 000088"
    test "test1253" (lazy(sprintf "%- 0i" 88uy)) " 88"
    test "test1254" (lazy(sprintf "%- 05i" 88uy)) " 88  "
    test "test1255" (lazy(sprintf "%- 01i" 88uy)) " 88"
    test "test1256" (lazy(sprintf "%- 0*i" 7 88uy)) " 88    "
    test "test1257" (lazy(sprintf "%i" 999L)) "999"
    test "test1258" (lazy(sprintf "%5i" 999L)) "  999"
    test "test1259" (lazy(sprintf "%1i" 999L)) "999"
    test "test1260" (lazy(sprintf "%*i" 7 999L)) "    999"
    test "test1261" (lazy(sprintf "%-i" 999L)) "999"
    test "test1262" (lazy(sprintf "%-5i" 999L)) "999  "
    test "test1263" (lazy(sprintf "%-1i" 999L)) "999"
    test "test1264" (lazy(sprintf "%-*i" 7 999L)) "999    "
    test "test1265" (lazy(sprintf "%0i" 999L)) "999"
    test "test1266" (lazy(sprintf "%05i" 999L)) "00999"
    test "test1267" (lazy(sprintf "%01i" 999L)) "999"
    test "test1268" (lazy(sprintf "%0*i" 7 999L)) "0000999"
    test "test1269" (lazy(sprintf "%-0i" 999L)) "999"
    test "test1270" (lazy(sprintf "%-05i" 999L)) "999  "
    test "test1271" (lazy(sprintf "%-01i" 999L)) "999"
    test "test1272" (lazy(sprintf "%-0*i" 7 999L)) "999    "
    test "test1273" (lazy(sprintf "%+i" 999L)) "+999"
    test "test1274" (lazy(sprintf "%+5i" 999L)) " +999"
    test "test1275" (lazy(sprintf "%+1i" 999L)) "+999"
    test "test1276" (lazy(sprintf "%+*i" 7 999L)) "   +999"
    test "test1277" (lazy(sprintf "%-+i" 999L)) "+999"
    test "test1278" (lazy(sprintf "%-+5i" 999L)) "+999 "
    test "test1279" (lazy(sprintf "%-+1i" 999L)) "+999"
    test "test1280" (lazy(sprintf "%-+*i" 7 999L)) "+999   "
    test "test1281" (lazy(sprintf "%+0i" 999L)) "+999"
    test "test1282" (lazy(sprintf "%+05i" 999L)) "+0999"
    test "test1283" (lazy(sprintf "%+01i" 999L)) "+999"
    test "test1284" (lazy(sprintf "%+0*i" 7 999L)) "+000999"
    test "test1285" (lazy(sprintf "%-+0i" 999L)) "+999"
    test "test1286" (lazy(sprintf "%-+05i" 999L)) "+999 "
    test "test1287" (lazy(sprintf "%-+01i" 999L)) "+999"
    test "test1288" (lazy(sprintf "%-+0*i" 7 999L)) "+999   "
    test "test1289" (lazy(sprintf "% i" 999L)) " 999"
    test "test1290" (lazy(sprintf "% 5i" 999L)) "  999"
    test "test1291" (lazy(sprintf "% 1i" 999L)) " 999"
    test "test1292" (lazy(sprintf "% *i" 7 999L)) "    999"
    test "test1293" (lazy(sprintf "%- i" 999L)) " 999"
    test "test1294" (lazy(sprintf "%- 5i" 999L)) " 999 "
    test "test1295" (lazy(sprintf "%- 1i" 999L)) " 999"
    test "test1296" (lazy(sprintf "%- *i" 7 999L)) " 999   "
    test "test1297" (lazy(sprintf "% 0i" 999L)) " 999"
    test "test1298" (lazy(sprintf "% 05i" 999L)) " 0999"
    test "test1299" (lazy(sprintf "% 01i" 999L)) " 999"
    test "test1300" (lazy(sprintf "% 0*i" 7 999L)) " 000999"
    test "test1301" (lazy(sprintf "%- 0i" 999L)) " 999"
    test "test1302" (lazy(sprintf "%- 05i" 999L)) " 999 "
    test "test1303" (lazy(sprintf "%- 01i" 999L)) " 999"
    test "test1304" (lazy(sprintf "%- 0*i" 7 999L)) " 999   "
    test "test1305" (lazy(sprintf "%i" -321L)) "-321"
    test "test1306" (lazy(sprintf "%5i" -321L)) " -321"
    test "test1307" (lazy(sprintf "%1i" -321L)) "-321"
    test "test1308" (lazy(sprintf "%*i" 7 -321L)) "   -321"
    test "test1309" (lazy(sprintf "%-i" -321L)) "-321"
    test "test1310" (lazy(sprintf "%-5i" -321L)) "-321 "
    test "test1311" (lazy(sprintf "%-1i" -321L)) "-321"
    test "test1312" (lazy(sprintf "%-*i" 7 -321L)) "-321   "
    test "test1313" (lazy(sprintf "%0i" -321L)) "-321"
    test "test1314" (lazy(sprintf "%05i" -321L)) "-0321"
    test "test1315" (lazy(sprintf "%01i" -321L)) "-321"
    test "test1316" (lazy(sprintf "%0*i" 7 -321L)) "-000321"
    test "test1317" (lazy(sprintf "%-0i" -321L)) "-321"
    test "test1318" (lazy(sprintf "%-05i" -321L)) "-321 "
    test "test1319" (lazy(sprintf "%-01i" -321L)) "-321"
    test "test1320" (lazy(sprintf "%-0*i" 7 -321L)) "-321   "
    test "test1321" (lazy(sprintf "%+i" -321L)) "-321"
    test "test1322" (lazy(sprintf "%+5i" -321L)) " -321"
    test "test1323" (lazy(sprintf "%+1i" -321L)) "-321"
    test "test1324" (lazy(sprintf "%+*i" 7 -321L)) "   -321"
    test "test1325" (lazy(sprintf "%-+i" -321L)) "-321"
    test "test1326" (lazy(sprintf "%-+5i" -321L)) "-321 "
    test "test1327" (lazy(sprintf "%-+1i" -321L)) "-321"
    test "test1328" (lazy(sprintf "%-+*i" 7 -321L)) "-321   "
    test "test1329" (lazy(sprintf "%+0i" -321L)) "-321"
    test "test1330" (lazy(sprintf "%+05i" -321L)) "-0321"
    test "test1331" (lazy(sprintf "%+01i" -321L)) "-321"
    test "test1332" (lazy(sprintf "%+0*i" 7 -321L)) "-000321"
    test "test1333" (lazy(sprintf "%-+0i" -321L)) "-321"
    test "test1334" (lazy(sprintf "%-+05i" -321L)) "-321 "
    test "test1335" (lazy(sprintf "%-+01i" -321L)) "-321"
    test "test1336" (lazy(sprintf "%-+0*i" 7 -321L)) "-321   "
    test "test1337" (lazy(sprintf "% i" -321L)) "-321"
    test "test1338" (lazy(sprintf "% 5i" -321L)) " -321"
    test "test1339" (lazy(sprintf "% 1i" -321L)) "-321"
    test "test1340" (lazy(sprintf "% *i" 7 -321L)) "   -321"
    test "test1341" (lazy(sprintf "%- i" -321L)) "-321"
    test "test1342" (lazy(sprintf "%- 5i" -321L)) "-321 "
    test "test1343" (lazy(sprintf "%- 1i" -321L)) "-321"
    test "test1344" (lazy(sprintf "%- *i" 7 -321L)) "-321   "
    test "test1345" (lazy(sprintf "% 0i" -321L)) "-321"
    test "test1346" (lazy(sprintf "% 05i" -321L)) "-0321"
    test "test1347" (lazy(sprintf "% 01i" -321L)) "-321"
    test "test1348" (lazy(sprintf "% 0*i" 7 -321L)) "-000321"
    test "test1349" (lazy(sprintf "%- 0i" -321L)) "-321"
    test "test1350" (lazy(sprintf "%- 05i" -321L)) "-321 "
    test "test1351" (lazy(sprintf "%- 01i" -321L)) "-321"
    test "test1352" (lazy(sprintf "%- 0*i" 7 -321L)) "-321   "
    test "test1353" (lazy(sprintf "%i" 50000UL)) "50000"
    test "test1354" (lazy(sprintf "%5i" 50000UL)) "50000"
    test "test1355" (lazy(sprintf "%1i" 50000UL)) "50000"
    test "test1356" (lazy(sprintf "%*i" 7 50000UL)) "  50000"
    test "test1357" (lazy(sprintf "%-i" 50000UL)) "50000"
    test "test1358" (lazy(sprintf "%-5i" 50000UL)) "50000"
    test "test1359" (lazy(sprintf "%-1i" 50000UL)) "50000"
    test "test1360" (lazy(sprintf "%-*i" 7 50000UL)) "50000  "
    test "test1361" (lazy(sprintf "%0i" 50000UL)) "50000"
    test "test1362" (lazy(sprintf "%05i" 50000UL)) "50000"
    test "test1363" (lazy(sprintf "%01i" 50000UL)) "50000"
    test "test1364" (lazy(sprintf "%0*i" 7 50000UL)) "0050000"
    test "test1365" (lazy(sprintf "%-0i" 50000UL)) "50000"
    test "test1366" (lazy(sprintf "%-05i" 50000UL)) "50000"
    test "test1367" (lazy(sprintf "%-01i" 50000UL)) "50000"
    test "test1368" (lazy(sprintf "%-0*i" 7 50000UL)) "50000  "
    test "test1369" (lazy(sprintf "%+i" 50000UL)) "+50000"
    test "test1370" (lazy(sprintf "%+5i" 50000UL)) "+50000"
    test "test1371" (lazy(sprintf "%+1i" 50000UL)) "+50000"
    test "test1372" (lazy(sprintf "%+*i" 7 50000UL)) " +50000"
    test "test1373" (lazy(sprintf "%-+i" 50000UL)) "+50000"
    test "test1374" (lazy(sprintf "%-+5i" 50000UL)) "+50000"
    test "test1375" (lazy(sprintf "%-+1i" 50000UL)) "+50000"
    test "test1376" (lazy(sprintf "%-+*i" 7 50000UL)) "+50000 "
    test "test1377" (lazy(sprintf "%+0i" 50000UL)) "+50000"
    test "test1378" (lazy(sprintf "%+05i" 50000UL)) "+50000"
    test "test1379" (lazy(sprintf "%+01i" 50000UL)) "+50000"
    test "test1380" (lazy(sprintf "%+0*i" 7 50000UL)) "+050000"
    test "test1381" (lazy(sprintf "%-+0i" 50000UL)) "+50000"
    test "test1382" (lazy(sprintf "%-+05i" 50000UL)) "+50000"
    test "test1383" (lazy(sprintf "%-+01i" 50000UL)) "+50000"
    test "test1384" (lazy(sprintf "%-+0*i" 7 50000UL)) "+50000 "
    test "test1385" (lazy(sprintf "% i" 50000UL)) " 50000"
    test "test1386" (lazy(sprintf "% 5i" 50000UL)) " 50000"
    test "test1387" (lazy(sprintf "% 1i" 50000UL)) " 50000"
    test "test1388" (lazy(sprintf "% *i" 7 50000UL)) "  50000"
    test "test1389" (lazy(sprintf "%- i" 50000UL)) " 50000"
    test "test1390" (lazy(sprintf "%- 5i" 50000UL)) " 50000"
    test "test1391" (lazy(sprintf "%- 1i" 50000UL)) " 50000"
    test "test1392" (lazy(sprintf "%- *i" 7 50000UL)) " 50000 "
    test "test1393" (lazy(sprintf "% 0i" 50000UL)) " 50000"
    test "test1394" (lazy(sprintf "% 05i" 50000UL)) " 50000"
    test "test1395" (lazy(sprintf "% 01i" 50000UL)) " 50000"
    test "test1396" (lazy(sprintf "% 0*i" 7 50000UL)) " 050000"
    test "test1397" (lazy(sprintf "%- 0i" 50000UL)) " 50000"
    test "test1398" (lazy(sprintf "%- 05i" 50000UL)) " 50000"
    test "test1399" (lazy(sprintf "%- 01i" 50000UL)) " 50000"
    test "test1400" (lazy(sprintf "%- 0*i" 7 50000UL)) " 50000 "
    test "test1401" (lazy(sprintf "%i" System.Int32.MaxValue)) "2147483647"
    test "test1402" (lazy(sprintf "%5i" System.Int32.MaxValue)) "2147483647"
    test "test1403" (lazy(sprintf "%1i" System.Int32.MaxValue)) "2147483647"
    test "test1404" (lazy(sprintf "%*i" 7 System.Int32.MaxValue)) "2147483647"
    test "test1405" (lazy(sprintf "%-i" System.Int32.MaxValue)) "2147483647"
    test "test1406" (lazy(sprintf "%-5i" System.Int32.MaxValue)) "2147483647"
    test "test1407" (lazy(sprintf "%-1i" System.Int32.MaxValue)) "2147483647"
    test "test1408" (lazy(sprintf "%-*i" 7 System.Int32.MaxValue)) "2147483647"
    test "test1409" (lazy(sprintf "%0i" System.Int32.MaxValue)) "2147483647"
    test "test1410" (lazy(sprintf "%05i" System.Int32.MaxValue)) "2147483647"
    test "test1411" (lazy(sprintf "%01i" System.Int32.MaxValue)) "2147483647"
    test "test1412" (lazy(sprintf "%0*i" 7 System.Int32.MaxValue)) "2147483647"
    test "test1413" (lazy(sprintf "%-0i" System.Int32.MaxValue)) "2147483647"
    test "test1414" (lazy(sprintf "%-05i" System.Int32.MaxValue)) "2147483647"
    test "test1415" (lazy(sprintf "%-01i" System.Int32.MaxValue)) "2147483647"
    test "test1416" (lazy(sprintf "%-0*i" 7 System.Int32.MaxValue)) "2147483647"
    test "test1417" (lazy(sprintf "%+i" System.Int32.MaxValue)) "+2147483647"
    test "test1418" (lazy(sprintf "%+5i" System.Int32.MaxValue)) "+2147483647"
    test "test1419" (lazy(sprintf "%+1i" System.Int32.MaxValue)) "+2147483647"
    test "test1420" (lazy(sprintf "%+*i" 7 System.Int32.MaxValue)) "+2147483647"
    test "test1421" (lazy(sprintf "%-+i" System.Int32.MaxValue)) "+2147483647"
    test "test1422" (lazy(sprintf "%-+5i" System.Int32.MaxValue)) "+2147483647"
    test "test1423" (lazy(sprintf "%-+1i" System.Int32.MaxValue)) "+2147483647"
    test "test1424" (lazy(sprintf "%-+*i" 7 System.Int32.MaxValue)) "+2147483647"
    test "test1425" (lazy(sprintf "%+0i" System.Int32.MaxValue)) "+2147483647"
    test "test1426" (lazy(sprintf "%+05i" System.Int32.MaxValue)) "+2147483647"
    test "test1427" (lazy(sprintf "%+01i" System.Int32.MaxValue)) "+2147483647"
    test "test1428" (lazy(sprintf "%+0*i" 7 System.Int32.MaxValue)) "+2147483647"
    test "test1429" (lazy(sprintf "%-+0i" System.Int32.MaxValue)) "+2147483647"
    test "test1430" (lazy(sprintf "%-+05i" System.Int32.MaxValue)) "+2147483647"
    test "test1431" (lazy(sprintf "%-+01i" System.Int32.MaxValue)) "+2147483647"
    test "test1432" (lazy(sprintf "%-+0*i" 7 System.Int32.MaxValue)) "+2147483647"
    test "test1433" (lazy(sprintf "% i" System.Int32.MaxValue)) " 2147483647"
    test "test1434" (lazy(sprintf "% 5i" System.Int32.MaxValue)) " 2147483647"
    test "test1435" (lazy(sprintf "% 1i" System.Int32.MaxValue)) " 2147483647"
    test "test1436" (lazy(sprintf "% *i" 7 System.Int32.MaxValue)) " 2147483647"
    test "test1437" (lazy(sprintf "%- i" System.Int32.MaxValue)) " 2147483647"
    test "test1438" (lazy(sprintf "%- 5i" System.Int32.MaxValue)) " 2147483647"
    test "test1439" (lazy(sprintf "%- 1i" System.Int32.MaxValue)) " 2147483647"
    test "test1440" (lazy(sprintf "%- *i" 7 System.Int32.MaxValue)) " 2147483647"
    test "test1441" (lazy(sprintf "% 0i" System.Int32.MaxValue)) " 2147483647"
    test "test1442" (lazy(sprintf "% 05i" System.Int32.MaxValue)) " 2147483647"
    test "test1443" (lazy(sprintf "% 01i" System.Int32.MaxValue)) " 2147483647"
    test "test1444" (lazy(sprintf "% 0*i" 7 System.Int32.MaxValue)) " 2147483647"
    test "test1445" (lazy(sprintf "%- 0i" System.Int32.MaxValue)) " 2147483647"
    test "test1446" (lazy(sprintf "%- 05i" System.Int32.MaxValue)) " 2147483647"
    test "test1447" (lazy(sprintf "%- 01i" System.Int32.MaxValue)) " 2147483647"
    test "test1448" (lazy(sprintf "%- 0*i" 7 System.Int32.MaxValue)) " 2147483647"
    test "test1449" (lazy(sprintf "%i" System.Int64.MaxValue)) "9223372036854775807"
    test "test1450" (lazy(sprintf "%5i" System.Int64.MaxValue)) "9223372036854775807"
    test "test1451" (lazy(sprintf "%1i" System.Int64.MaxValue)) "9223372036854775807"
    test "test1452" (lazy(sprintf "%*i" 7 System.Int64.MaxValue)) "9223372036854775807"
    test "test1453" (lazy(sprintf "%-i" System.Int64.MaxValue)) "9223372036854775807"
    test "test1454" (lazy(sprintf "%-5i" System.Int64.MaxValue)) "9223372036854775807"
    test "test1455" (lazy(sprintf "%-1i" System.Int64.MaxValue)) "9223372036854775807"
    test "test1456" (lazy(sprintf "%-*i" 7 System.Int64.MaxValue)) "9223372036854775807"
    test "test1457" (lazy(sprintf "%0i" System.Int64.MaxValue)) "9223372036854775807"
    test "test1458" (lazy(sprintf "%05i" System.Int64.MaxValue)) "9223372036854775807"
    test "test1459" (lazy(sprintf "%01i" System.Int64.MaxValue)) "9223372036854775807"
    test "test1460" (lazy(sprintf "%0*i" 7 System.Int64.MaxValue)) "9223372036854775807"
    test "test1461" (lazy(sprintf "%-0i" System.Int64.MaxValue)) "9223372036854775807"
    test "test1462" (lazy(sprintf "%-05i" System.Int64.MaxValue)) "9223372036854775807"
    test "test1463" (lazy(sprintf "%-01i" System.Int64.MaxValue)) "9223372036854775807"
    test "test1464" (lazy(sprintf "%-0*i" 7 System.Int64.MaxValue)) "9223372036854775807"
    test "test1465" (lazy(sprintf "%+i" System.Int64.MaxValue)) "+9223372036854775807"
    test "test1466" (lazy(sprintf "%+5i" System.Int64.MaxValue)) "+9223372036854775807"
    test "test1467" (lazy(sprintf "%+1i" System.Int64.MaxValue)) "+9223372036854775807"
    test "test1468" (lazy(sprintf "%+*i" 7 System.Int64.MaxValue)) "+9223372036854775807"
    test "test1469" (lazy(sprintf "%-+i" System.Int64.MaxValue)) "+9223372036854775807"
    test "test1470" (lazy(sprintf "%-+5i" System.Int64.MaxValue)) "+9223372036854775807"
    test "test1471" (lazy(sprintf "%-+1i" System.Int64.MaxValue)) "+9223372036854775807"
    test "test1472" (lazy(sprintf "%-+*i" 7 System.Int64.MaxValue)) "+9223372036854775807"
    test "test1473" (lazy(sprintf "%+0i" System.Int64.MaxValue)) "+9223372036854775807"
    test "test1474" (lazy(sprintf "%+05i" System.Int64.MaxValue)) "+9223372036854775807"
    test "test1475" (lazy(sprintf "%+01i" System.Int64.MaxValue)) "+9223372036854775807"
    test "test1476" (lazy(sprintf "%+0*i" 7 System.Int64.MaxValue)) "+9223372036854775807"
    test "test1477" (lazy(sprintf "%-+0i" System.Int64.MaxValue)) "+9223372036854775807"
    test "test1478" (lazy(sprintf "%-+05i" System.Int64.MaxValue)) "+9223372036854775807"
    test "test1479" (lazy(sprintf "%-+01i" System.Int64.MaxValue)) "+9223372036854775807"
    test "test1480" (lazy(sprintf "%-+0*i" 7 System.Int64.MaxValue)) "+9223372036854775807"
    test "test1481" (lazy(sprintf "% i" System.Int64.MaxValue)) " 9223372036854775807"
    test "test1482" (lazy(sprintf "% 5i" System.Int64.MaxValue)) " 9223372036854775807"
    test "test1483" (lazy(sprintf "% 1i" System.Int64.MaxValue)) " 9223372036854775807"
    test "test1484" (lazy(sprintf "% *i" 7 System.Int64.MaxValue)) " 9223372036854775807"
    test "test1485" (lazy(sprintf "%- i" System.Int64.MaxValue)) " 9223372036854775807"
    test "test1486" (lazy(sprintf "%- 5i" System.Int64.MaxValue)) " 9223372036854775807"
    test "test1487" (lazy(sprintf "%- 1i" System.Int64.MaxValue)) " 9223372036854775807"
    test "test1488" (lazy(sprintf "%- *i" 7 System.Int64.MaxValue)) " 9223372036854775807"
    test "test1489" (lazy(sprintf "% 0i" System.Int64.MaxValue)) " 9223372036854775807"
    test "test1490" (lazy(sprintf "% 05i" System.Int64.MaxValue)) " 9223372036854775807"
    test "test1491" (lazy(sprintf "% 01i" System.Int64.MaxValue)) " 9223372036854775807"
    test "test1492" (lazy(sprintf "% 0*i" 7 System.Int64.MaxValue)) " 9223372036854775807"
    test "test1493" (lazy(sprintf "%- 0i" System.Int64.MaxValue)) " 9223372036854775807"
    test "test1494" (lazy(sprintf "%- 05i" System.Int64.MaxValue)) " 9223372036854775807"
    test "test1495" (lazy(sprintf "%- 01i" System.Int64.MaxValue)) " 9223372036854775807"
    test "test1496" (lazy(sprintf "%- 0*i" 7 System.Int64.MaxValue)) " 9223372036854775807"
    test "test1497" (lazy(sprintf "%i" System.Int32.MinValue)) "-2147483648"
    test "test1498" (lazy(sprintf "%5i" System.Int32.MinValue)) "-2147483648"
    test "test1499" (lazy(sprintf "%1i" System.Int32.MinValue)) "-2147483648"
    test "test1500" (lazy(sprintf "%*i" 7 System.Int32.MinValue)) "-2147483648"
    test "test1501" (lazy(sprintf "%-i" System.Int32.MinValue)) "-2147483648"
    test "test1502" (lazy(sprintf "%-5i" System.Int32.MinValue)) "-2147483648"
    test "test1503" (lazy(sprintf "%-1i" System.Int32.MinValue)) "-2147483648"
    test "test1504" (lazy(sprintf "%-*i" 7 System.Int32.MinValue)) "-2147483648"
    test "test1505" (lazy(sprintf "%0i" System.Int32.MinValue)) "-2147483648"
    test "test1506" (lazy(sprintf "%05i" System.Int32.MinValue)) "-2147483648"
    test "test1507" (lazy(sprintf "%01i" System.Int32.MinValue)) "-2147483648"
    test "test1508" (lazy(sprintf "%0*i" 7 System.Int32.MinValue)) "-2147483648"
    test "test1509" (lazy(sprintf "%-0i" System.Int32.MinValue)) "-2147483648"
    test "test1510" (lazy(sprintf "%-05i" System.Int32.MinValue)) "-2147483648"
    test "test1511" (lazy(sprintf "%-01i" System.Int32.MinValue)) "-2147483648"
    test "test1512" (lazy(sprintf "%-0*i" 7 System.Int32.MinValue)) "-2147483648"
    test "test1513" (lazy(sprintf "%+i" System.Int32.MinValue)) "-2147483648"
    test "test1514" (lazy(sprintf "%+5i" System.Int32.MinValue)) "-2147483648"
    test "test1515" (lazy(sprintf "%+1i" System.Int32.MinValue)) "-2147483648"
    test "test1516" (lazy(sprintf "%+*i" 7 System.Int32.MinValue)) "-2147483648"
    test "test1517" (lazy(sprintf "%-+i" System.Int32.MinValue)) "-2147483648"
    test "test1518" (lazy(sprintf "%-+5i" System.Int32.MinValue)) "-2147483648"
    test "test1519" (lazy(sprintf "%-+1i" System.Int32.MinValue)) "-2147483648"
    test "test1520" (lazy(sprintf "%-+*i" 7 System.Int32.MinValue)) "-2147483648"
    test "test1521" (lazy(sprintf "%+0i" System.Int32.MinValue)) "-2147483648"
    test "test1522" (lazy(sprintf "%+05i" System.Int32.MinValue)) "-2147483648"
    test "test1523" (lazy(sprintf "%+01i" System.Int32.MinValue)) "-2147483648"
    test "test1524" (lazy(sprintf "%+0*i" 7 System.Int32.MinValue)) "-2147483648"
    test "test1525" (lazy(sprintf "%-+0i" System.Int32.MinValue)) "-2147483648"
    test "test1526" (lazy(sprintf "%-+05i" System.Int32.MinValue)) "-2147483648"
    test "test1527" (lazy(sprintf "%-+01i" System.Int32.MinValue)) "-2147483648"
    test "test1528" (lazy(sprintf "%-+0*i" 7 System.Int32.MinValue)) "-2147483648"
    test "test1529" (lazy(sprintf "% i" System.Int32.MinValue)) "-2147483648"
    test "test1530" (lazy(sprintf "% 5i" System.Int32.MinValue)) "-2147483648"
    test "test1531" (lazy(sprintf "% 1i" System.Int32.MinValue)) "-2147483648"
    test "test1532" (lazy(sprintf "% *i" 7 System.Int32.MinValue)) "-2147483648"
    test "test1533" (lazy(sprintf "%- i" System.Int32.MinValue)) "-2147483648"
    test "test1534" (lazy(sprintf "%- 5i" System.Int32.MinValue)) "-2147483648"
    test "test1535" (lazy(sprintf "%- 1i" System.Int32.MinValue)) "-2147483648"
    test "test1536" (lazy(sprintf "%- *i" 7 System.Int32.MinValue)) "-2147483648"
    test "test1537" (lazy(sprintf "% 0i" System.Int32.MinValue)) "-2147483648"
    test "test1538" (lazy(sprintf "% 05i" System.Int32.MinValue)) "-2147483648"
    test "test1539" (lazy(sprintf "% 01i" System.Int32.MinValue)) "-2147483648"
    test "test1540" (lazy(sprintf "% 0*i" 7 System.Int32.MinValue)) "-2147483648"
    test "test1541" (lazy(sprintf "%- 0i" System.Int32.MinValue)) "-2147483648"
    test "test1542" (lazy(sprintf "%- 05i" System.Int32.MinValue)) "-2147483648"
    test "test1543" (lazy(sprintf "%- 01i" System.Int32.MinValue)) "-2147483648"
    test "test1544" (lazy(sprintf "%- 0*i" 7 System.Int32.MinValue)) "-2147483648"
    test "test1545" (lazy(sprintf "%i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1546" (lazy(sprintf "%5i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1547" (lazy(sprintf "%1i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1548" (lazy(sprintf "%*i" 7 System.Int64.MinValue)) "-9223372036854775808"
    test "test1549" (lazy(sprintf "%-i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1550" (lazy(sprintf "%-5i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1551" (lazy(sprintf "%-1i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1552" (lazy(sprintf "%-*i" 7 System.Int64.MinValue)) "-9223372036854775808"
    test "test1553" (lazy(sprintf "%0i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1554" (lazy(sprintf "%05i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1555" (lazy(sprintf "%01i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1556" (lazy(sprintf "%0*i" 7 System.Int64.MinValue)) "-9223372036854775808"
    test "test1557" (lazy(sprintf "%-0i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1558" (lazy(sprintf "%-05i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1559" (lazy(sprintf "%-01i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1560" (lazy(sprintf "%-0*i" 7 System.Int64.MinValue)) "-9223372036854775808"
    test "test1561" (lazy(sprintf "%+i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1562" (lazy(sprintf "%+5i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1563" (lazy(sprintf "%+1i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1564" (lazy(sprintf "%+*i" 7 System.Int64.MinValue)) "-9223372036854775808"
    test "test1565" (lazy(sprintf "%-+i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1566" (lazy(sprintf "%-+5i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1567" (lazy(sprintf "%-+1i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1568" (lazy(sprintf "%-+*i" 7 System.Int64.MinValue)) "-9223372036854775808"
    test "test1569" (lazy(sprintf "%+0i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1570" (lazy(sprintf "%+05i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1571" (lazy(sprintf "%+01i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1572" (lazy(sprintf "%+0*i" 7 System.Int64.MinValue)) "-9223372036854775808"
    test "test1573" (lazy(sprintf "%-+0i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1574" (lazy(sprintf "%-+05i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1575" (lazy(sprintf "%-+01i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1576" (lazy(sprintf "%-+0*i" 7 System.Int64.MinValue)) "-9223372036854775808"
    test "test1577" (lazy(sprintf "% i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1578" (lazy(sprintf "% 5i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1579" (lazy(sprintf "% 1i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1580" (lazy(sprintf "% *i" 7 System.Int64.MinValue)) "-9223372036854775808"
    test "test1581" (lazy(sprintf "%- i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1582" (lazy(sprintf "%- 5i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1583" (lazy(sprintf "%- 1i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1584" (lazy(sprintf "%- *i" 7 System.Int64.MinValue)) "-9223372036854775808"
    test "test1585" (lazy(sprintf "% 0i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1586" (lazy(sprintf "% 05i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1587" (lazy(sprintf "% 01i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1588" (lazy(sprintf "% 0*i" 7 System.Int64.MinValue)) "-9223372036854775808"
    test "test1589" (lazy(sprintf "%- 0i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1590" (lazy(sprintf "%- 05i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1591" (lazy(sprintf "%- 01i" System.Int64.MinValue)) "-9223372036854775808"
    test "test1592" (lazy(sprintf "%- 0*i" 7 System.Int64.MinValue)) "-9223372036854775808"
    test "test1593" (lazy(sprintf "%i" 55n)) "55"
    test "test1594" (lazy(sprintf "%5i" 55n)) "   55"
    test "test1595" (lazy(sprintf "%1i" 55n)) "55"
    test "test1596" (lazy(sprintf "%*i" 7 55n)) "     55"
    test "test1597" (lazy(sprintf "%-i" 55n)) "55"
    test "test1598" (lazy(sprintf "%-5i" 55n)) "55   "
    test "test1599" (lazy(sprintf "%-1i" 55n)) "55"
    test "test1600" (lazy(sprintf "%-*i" 7 55n)) "55     "
    test "test1601" (lazy(sprintf "%0i" 55n)) "55"
    test "test1602" (lazy(sprintf "%05i" 55n)) "00055"
    test "test1603" (lazy(sprintf "%01i" 55n)) "55"
    test "test1604" (lazy(sprintf "%0*i" 7 55n)) "0000055"
    test "test1605" (lazy(sprintf "%-0i" 55n)) "55"
    test "test1606" (lazy(sprintf "%-05i" 55n)) "55   "
    test "test1607" (lazy(sprintf "%-01i" 55n)) "55"
    test "test1608" (lazy(sprintf "%-0*i" 7 55n)) "55     "
    test "test1609" (lazy(sprintf "%+i" 55n)) "+55"
    test "test1610" (lazy(sprintf "%+5i" 55n)) "  +55"
    test "test1611" (lazy(sprintf "%+1i" 55n)) "+55"
    test "test1612" (lazy(sprintf "%+*i" 7 55n)) "    +55"
    test "test1613" (lazy(sprintf "%-+i" 55n)) "+55"
    test "test1614" (lazy(sprintf "%-+5i" 55n)) "+55  "
    test "test1615" (lazy(sprintf "%-+1i" 55n)) "+55"
    test "test1616" (lazy(sprintf "%-+*i" 7 55n)) "+55    "
    test "test1617" (lazy(sprintf "%+0i" 55n)) "+55"
    test "test1618" (lazy(sprintf "%+05i" 55n)) "+0055"
    test "test1619" (lazy(sprintf "%+01i" 55n)) "+55"
    test "test1620" (lazy(sprintf "%+0*i" 7 55n)) "+000055"
    test "test1621" (lazy(sprintf "%-+0i" 55n)) "+55"
    test "test1622" (lazy(sprintf "%-+05i" 55n)) "+55  "
    test "test1623" (lazy(sprintf "%-+01i" 55n)) "+55"
    test "test1624" (lazy(sprintf "%-+0*i" 7 55n)) "+55    "
    test "test1625" (lazy(sprintf "% i" 55n)) " 55"
    test "test1626" (lazy(sprintf "% 5i" 55n)) "   55"
    test "test1627" (lazy(sprintf "% 1i" 55n)) " 55"
    test "test1628" (lazy(sprintf "% *i" 7 55n)) "     55"
    test "test1629" (lazy(sprintf "%- i" 55n)) " 55"
    test "test1630" (lazy(sprintf "%- 5i" 55n)) " 55  "
    test "test1631" (lazy(sprintf "%- 1i" 55n)) " 55"
    test "test1632" (lazy(sprintf "%- *i" 7 55n)) " 55    "
    test "test1633" (lazy(sprintf "% 0i" 55n)) " 55"
    test "test1634" (lazy(sprintf "% 05i" 55n)) " 0055"
    test "test1635" (lazy(sprintf "% 01i" 55n)) " 55"
    test "test1636" (lazy(sprintf "% 0*i" 7 55n)) " 000055"
    test "test1637" (lazy(sprintf "%- 0i" 55n)) " 55"
    test "test1638" (lazy(sprintf "%- 05i" 55n)) " 55  "
    test "test1639" (lazy(sprintf "%- 01i" 55n)) " 55"
    test "test1640" (lazy(sprintf "%- 0*i" 7 55n)) " 55    "
    test "test1641" (lazy(sprintf "%i" 999un)) "999"
    test "test1642" (lazy(sprintf "%5i" 999un)) "  999"
    test "test1643" (lazy(sprintf "%1i" 999un)) "999"
    test "test1644" (lazy(sprintf "%*i" 7 999un)) "    999"
    test "test1645" (lazy(sprintf "%-i" 999un)) "999"
    test "test1646" (lazy(sprintf "%-5i" 999un)) "999  "
    test "test1647" (lazy(sprintf "%-1i" 999un)) "999"
    test "test1648" (lazy(sprintf "%-*i" 7 999un)) "999    "
    test "test1649" (lazy(sprintf "%0i" 999un)) "999"
    test "test1650" (lazy(sprintf "%05i" 999un)) "00999"
    test "test1651" (lazy(sprintf "%01i" 999un)) "999"
    test "test1652" (lazy(sprintf "%0*i" 7 999un)) "0000999"
    test "test1653" (lazy(sprintf "%-0i" 999un)) "999"
    test "test1654" (lazy(sprintf "%-05i" 999un)) "999  "
    test "test1655" (lazy(sprintf "%-01i" 999un)) "999"
    test "test1656" (lazy(sprintf "%-0*i" 7 999un)) "999    "
    test "test1657" (lazy(sprintf "%+i" 999un)) "+999"
    test "test1658" (lazy(sprintf "%+5i" 999un)) " +999"
    test "test1659" (lazy(sprintf "%+1i" 999un)) "+999"
    test "test1660" (lazy(sprintf "%+*i" 7 999un)) "   +999"
    test "test1661" (lazy(sprintf "%-+i" 999un)) "+999"
    test "test1662" (lazy(sprintf "%-+5i" 999un)) "+999 "
    test "test1663" (lazy(sprintf "%-+1i" 999un)) "+999"
    test "test1664" (lazy(sprintf "%-+*i" 7 999un)) "+999   "
    test "test1665" (lazy(sprintf "%+0i" 999un)) "+999"
    test "test1666" (lazy(sprintf "%+05i" 999un)) "+0999"
    test "test1667" (lazy(sprintf "%+01i" 999un)) "+999"
    test "test1668" (lazy(sprintf "%+0*i" 7 999un)) "+000999"
    test "test1669" (lazy(sprintf "%-+0i" 999un)) "+999"
    test "test1670" (lazy(sprintf "%-+05i" 999un)) "+999 "
    test "test1671" (lazy(sprintf "%-+01i" 999un)) "+999"
    test "test1672" (lazy(sprintf "%-+0*i" 7 999un)) "+999   "
    test "test1673" (lazy(sprintf "% i" 999un)) " 999"
    test "test1674" (lazy(sprintf "% 5i" 999un)) "  999"
    test "test1675" (lazy(sprintf "% 1i" 999un)) " 999"
    test "test1676" (lazy(sprintf "% *i" 7 999un)) "    999"
    test "test1677" (lazy(sprintf "%- i" 999un)) " 999"
    test "test1678" (lazy(sprintf "%- 5i" 999un)) " 999 "
    test "test1679" (lazy(sprintf "%- 1i" 999un)) " 999"
    test "test1680" (lazy(sprintf "%- *i" 7 999un)) " 999   "
    test "test1681" (lazy(sprintf "% 0i" 999un)) " 999"
    test "test1682" (lazy(sprintf "% 05i" 999un)) " 0999"
    test "test1683" (lazy(sprintf "% 01i" 999un)) " 999"
    test "test1684" (lazy(sprintf "% 0*i" 7 999un)) " 000999"
    test "test1685" (lazy(sprintf "%- 0i" 999un)) " 999"
    test "test1686" (lazy(sprintf "%- 05i" 999un)) " 999 "
    test "test1687" (lazy(sprintf "%- 01i" 999un)) " 999"
    test "test1688" (lazy(sprintf "%- 0*i" 7 999un)) " 999   "
    test "test1689" (lazy(sprintf "%u" 14)) "14"
    test "test1690" (lazy(sprintf "%5u" 14)) "   14"
    test "test1691" (lazy(sprintf "%1u" 14)) "14"
    test "test1692" (lazy(sprintf "%*u" 7 14)) "     14"
    test "test1693" (lazy(sprintf "%-u" 14)) "14"
    test "test1694" (lazy(sprintf "%-5u" 14)) "14   "
    test "test1695" (lazy(sprintf "%-1u" 14)) "14"
    test "test1696" (lazy(sprintf "%-*u" 7 14)) "14     "
    test "test1697" (lazy(sprintf "%0u" 14)) "14"
    test "test1698" (lazy(sprintf "%05u" 14)) "00014"
    test "test1699" (lazy(sprintf "%01u" 14)) "14"
    test "test1700" (lazy(sprintf "%0*u" 7 14)) "0000014"
    test "test1701" (lazy(sprintf "%-0u" 14)) "14"
    test "test1702" (lazy(sprintf "%-05u" 14)) "14   "
    test "test1703" (lazy(sprintf "%-01u" 14)) "14"
    test "test1704" (lazy(sprintf "%-0*u" 7 14)) "14     "
    test "test1705" (lazy(sprintf "%+u" 14)) "+14"
    test "test1706" (lazy(sprintf "%+5u" 14)) "  +14"
    test "test1707" (lazy(sprintf "%+1u" 14)) "+14"
    test "test1708" (lazy(sprintf "%+*u" 7 14)) "    +14"
    test "test1709" (lazy(sprintf "%-+u" 14)) "+14"
    test "test1710" (lazy(sprintf "%-+5u" 14)) "+14  "
    test "test1711" (lazy(sprintf "%-+1u" 14)) "+14"
    test "test1712" (lazy(sprintf "%-+*u" 7 14)) "+14    "
    test "test1713" (lazy(sprintf "%+0u" 14)) "+14"
    test "test1714" (lazy(sprintf "%+05u" 14)) "+0014"
    test "test1715" (lazy(sprintf "%+01u" 14)) "+14"
    test "test1716" (lazy(sprintf "%+0*u" 7 14)) "+000014"
    test "test1717" (lazy(sprintf "%-+0u" 14)) "+14"
    test "test1718" (lazy(sprintf "%-+05u" 14)) "+14  "
    test "test1719" (lazy(sprintf "%-+01u" 14)) "+14"
    test "test1720" (lazy(sprintf "%-+0*u" 7 14)) "+14    "
    test "test1721" (lazy(sprintf "% u" 14)) " 14"
    test "test1722" (lazy(sprintf "% 5u" 14)) "   14"
    test "test1723" (lazy(sprintf "% 1u" 14)) " 14"
    test "test1724" (lazy(sprintf "% *u" 7 14)) "     14"
    test "test1725" (lazy(sprintf "%- u" 14)) " 14"
    test "test1726" (lazy(sprintf "%- 5u" 14)) " 14  "
    test "test1727" (lazy(sprintf "%- 1u" 14)) " 14"
    test "test1728" (lazy(sprintf "%- *u" 7 14)) " 14    "
    test "test1729" (lazy(sprintf "% 0u" 14)) " 14"
    test "test1730" (lazy(sprintf "% 05u" 14)) " 0014"
    test "test1731" (lazy(sprintf "% 01u" 14)) " 14"
    test "test1732" (lazy(sprintf "% 0*u" 7 14)) " 000014"
    test "test1733" (lazy(sprintf "%- 0u" 14)) " 14"
    test "test1734" (lazy(sprintf "%- 05u" 14)) " 14  "
    test "test1735" (lazy(sprintf "%- 01u" 14)) " 14"
    test "test1736" (lazy(sprintf "%- 0*u" 7 14)) " 14    "
    test "test1737" (lazy(sprintf "%u" -10)) "4294967286"
    test "test1738" (lazy(sprintf "%5u" -10)) "4294967286"
    test "test1739" (lazy(sprintf "%1u" -10)) "4294967286"
    test "test1740" (lazy(sprintf "%*u" 7 -10)) "4294967286"
    test "test1741" (lazy(sprintf "%-u" -10)) "4294967286"
    test "test1742" (lazy(sprintf "%-5u" -10)) "4294967286"
    test "test1743" (lazy(sprintf "%-1u" -10)) "4294967286"
    test "test1744" (lazy(sprintf "%-*u" 7 -10)) "4294967286"
    test "test1745" (lazy(sprintf "%0u" -10)) "4294967286"
    test "test1746" (lazy(sprintf "%05u" -10)) "4294967286"
    test "test1747" (lazy(sprintf "%01u" -10)) "4294967286"
    test "test1748" (lazy(sprintf "%0*u" 7 -10)) "4294967286"
    test "test1749" (lazy(sprintf "%-0u" -10)) "4294967286"
    test "test1750" (lazy(sprintf "%-05u" -10)) "4294967286"
    test "test1751" (lazy(sprintf "%-01u" -10)) "4294967286"
    test "test1752" (lazy(sprintf "%-0*u" 7 -10)) "4294967286"
    test "test1753" (lazy(sprintf "%+u" -10)) "+4294967286"
    test "test1754" (lazy(sprintf "%+5u" -10)) "+4294967286"
    test "test1755" (lazy(sprintf "%+1u" -10)) "+4294967286"
    test "test1756" (lazy(sprintf "%+*u" 7 -10)) "+4294967286"
    test "test1757" (lazy(sprintf "%-+u" -10)) "+4294967286"
    test "test1758" (lazy(sprintf "%-+5u" -10)) "+4294967286"
    test "test1759" (lazy(sprintf "%-+1u" -10)) "+4294967286"
    test "test1760" (lazy(sprintf "%-+*u" 7 -10)) "+4294967286"
    test "test1761" (lazy(sprintf "%+0u" -10)) "+4294967286"
    test "test1762" (lazy(sprintf "%+05u" -10)) "+4294967286"
    test "test1763" (lazy(sprintf "%+01u" -10)) "+4294967286"
    test "test1764" (lazy(sprintf "%+0*u" 7 -10)) "+4294967286"
    test "test1765" (lazy(sprintf "%-+0u" -10)) "+4294967286"
    test "test1766" (lazy(sprintf "%-+05u" -10)) "+4294967286"
    test "test1767" (lazy(sprintf "%-+01u" -10)) "+4294967286"
    test "test1768" (lazy(sprintf "%-+0*u" 7 -10)) "+4294967286"
    test "test1769" (lazy(sprintf "% u" -10)) " 4294967286"
    test "test1770" (lazy(sprintf "% 5u" -10)) " 4294967286"
    test "test1771" (lazy(sprintf "% 1u" -10)) " 4294967286"
    test "test1772" (lazy(sprintf "% *u" 7 -10)) " 4294967286"
    test "test1773" (lazy(sprintf "%- u" -10)) " 4294967286"
    test "test1774" (lazy(sprintf "%- 5u" -10)) " 4294967286"
    test "test1775" (lazy(sprintf "%- 1u" -10)) " 4294967286"
    test "test1776" (lazy(sprintf "%- *u" 7 -10)) " 4294967286"
    test "test1777" (lazy(sprintf "% 0u" -10)) " 4294967286"
    test "test1778" (lazy(sprintf "% 05u" -10)) " 4294967286"
    test "test1779" (lazy(sprintf "% 01u" -10)) " 4294967286"
    test "test1780" (lazy(sprintf "% 0*u" 7 -10)) " 4294967286"
    test "test1781" (lazy(sprintf "%- 0u" -10)) " 4294967286"
    test "test1782" (lazy(sprintf "%- 05u" -10)) " 4294967286"
    test "test1783" (lazy(sprintf "%- 01u" -10)) " 4294967286"
    test "test1784" (lazy(sprintf "%- 0*u" 7 -10)) " 4294967286"
    test "test1785" (lazy(sprintf "%u" 55s)) "55"
    test "test1786" (lazy(sprintf "%5u" 55s)) "   55"
    test "test1787" (lazy(sprintf "%1u" 55s)) "55"
    test "test1788" (lazy(sprintf "%*u" 7 55s)) "     55"
    test "test1789" (lazy(sprintf "%-u" 55s)) "55"
    test "test1790" (lazy(sprintf "%-5u" 55s)) "55   "
    test "test1791" (lazy(sprintf "%-1u" 55s)) "55"
    test "test1792" (lazy(sprintf "%-*u" 7 55s)) "55     "
    test "test1793" (lazy(sprintf "%0u" 55s)) "55"
    test "test1794" (lazy(sprintf "%05u" 55s)) "00055"
    test "test1795" (lazy(sprintf "%01u" 55s)) "55"
    test "test1796" (lazy(sprintf "%0*u" 7 55s)) "0000055"
    test "test1797" (lazy(sprintf "%-0u" 55s)) "55"
    test "test1798" (lazy(sprintf "%-05u" 55s)) "55   "
    test "test1799" (lazy(sprintf "%-01u" 55s)) "55"
    test "test1800" (lazy(sprintf "%-0*u" 7 55s)) "55     "
    test "test1801" (lazy(sprintf "%+u" 55s)) "+55"
    test "test1802" (lazy(sprintf "%+5u" 55s)) "  +55"
    test "test1803" (lazy(sprintf "%+1u" 55s)) "+55"
    test "test1804" (lazy(sprintf "%+*u" 7 55s)) "    +55"
    test "test1805" (lazy(sprintf "%-+u" 55s)) "+55"
    test "test1806" (lazy(sprintf "%-+5u" 55s)) "+55  "
    test "test1807" (lazy(sprintf "%-+1u" 55s)) "+55"
    test "test1808" (lazy(sprintf "%-+*u" 7 55s)) "+55    "
    test "test1809" (lazy(sprintf "%+0u" 55s)) "+55"
    test "test1810" (lazy(sprintf "%+05u" 55s)) "+0055"
    test "test1811" (lazy(sprintf "%+01u" 55s)) "+55"
    test "test1812" (lazy(sprintf "%+0*u" 7 55s)) "+000055"
    test "test1813" (lazy(sprintf "%-+0u" 55s)) "+55"
    test "test1814" (lazy(sprintf "%-+05u" 55s)) "+55  "
    test "test1815" (lazy(sprintf "%-+01u" 55s)) "+55"
    test "test1816" (lazy(sprintf "%-+0*u" 7 55s)) "+55    "
    test "test1817" (lazy(sprintf "% u" 55s)) " 55"
    test "test1818" (lazy(sprintf "% 5u" 55s)) "   55"
    test "test1819" (lazy(sprintf "% 1u" 55s)) " 55"
    test "test1820" (lazy(sprintf "% *u" 7 55s)) "     55"
    test "test1821" (lazy(sprintf "%- u" 55s)) " 55"
    test "test1822" (lazy(sprintf "%- 5u" 55s)) " 55  "
    test "test1823" (lazy(sprintf "%- 1u" 55s)) " 55"
    test "test1824" (lazy(sprintf "%- *u" 7 55s)) " 55    "
    test "test1825" (lazy(sprintf "% 0u" 55s)) " 55"
    test "test1826" (lazy(sprintf "% 05u" 55s)) " 0055"
    test "test1827" (lazy(sprintf "% 01u" 55s)) " 55"
    test "test1828" (lazy(sprintf "% 0*u" 7 55s)) " 000055"
    test "test1829" (lazy(sprintf "%- 0u" 55s)) " 55"
    test "test1830" (lazy(sprintf "%- 05u" 55s)) " 55  "
    test "test1831" (lazy(sprintf "%- 01u" 55s)) " 55"
    test "test1832" (lazy(sprintf "%- 0*u" 7 55s)) " 55    "
    test "test1833" (lazy(sprintf "%u" -88s)) "65448"
    test "test1834" (lazy(sprintf "%5u" -88s)) "65448"
    test "test1835" (lazy(sprintf "%1u" -88s)) "65448"
    test "test1836" (lazy(sprintf "%*u" 7 -88s)) "  65448"
    test "test1837" (lazy(sprintf "%-u" -88s)) "65448"
    test "test1838" (lazy(sprintf "%-5u" -88s)) "65448"
    test "test1839" (lazy(sprintf "%-1u" -88s)) "65448"
    test "test1840" (lazy(sprintf "%-*u" 7 -88s)) "65448  "
    test "test1841" (lazy(sprintf "%0u" -88s)) "65448"
    test "test1842" (lazy(sprintf "%05u" -88s)) "65448"
    test "test1843" (lazy(sprintf "%01u" -88s)) "65448"
    test "test1844" (lazy(sprintf "%0*u" 7 -88s)) "0065448"
    test "test1845" (lazy(sprintf "%-0u" -88s)) "65448"
    test "test1846" (lazy(sprintf "%-05u" -88s)) "65448"
    test "test1847" (lazy(sprintf "%-01u" -88s)) "65448"
    test "test1848" (lazy(sprintf "%-0*u" 7 -88s)) "65448  "
    test "test1849" (lazy(sprintf "%+u" -88s)) "+65448"
    test "test1850" (lazy(sprintf "%+5u" -88s)) "+65448"
    test "test1851" (lazy(sprintf "%+1u" -88s)) "+65448"
    test "test1852" (lazy(sprintf "%+*u" 7 -88s)) " +65448"
    test "test1853" (lazy(sprintf "%-+u" -88s)) "+65448"
    test "test1854" (lazy(sprintf "%-+5u" -88s)) "+65448"
    test "test1855" (lazy(sprintf "%-+1u" -88s)) "+65448"
    test "test1856" (lazy(sprintf "%-+*u" 7 -88s)) "+65448 "
    test "test1857" (lazy(sprintf "%+0u" -88s)) "+65448"
    test "test1858" (lazy(sprintf "%+05u" -88s)) "+65448"
    test "test1859" (lazy(sprintf "%+01u" -88s)) "+65448"
    test "test1860" (lazy(sprintf "%+0*u" 7 -88s)) "+065448"
    test "test1861" (lazy(sprintf "%-+0u" -88s)) "+65448"
    test "test1862" (lazy(sprintf "%-+05u" -88s)) "+65448"
    test "test1863" (lazy(sprintf "%-+01u" -88s)) "+65448"
    test "test1864" (lazy(sprintf "%-+0*u" 7 -88s)) "+65448 "
    test "test1865" (lazy(sprintf "% u" -88s)) " 65448"
    test "test1866" (lazy(sprintf "% 5u" -88s)) " 65448"
    test "test1867" (lazy(sprintf "% 1u" -88s)) " 65448"
    test "test1868" (lazy(sprintf "% *u" 7 -88s)) "  65448"
    test "test1869" (lazy(sprintf "%- u" -88s)) " 65448"
    test "test1870" (lazy(sprintf "%- 5u" -88s)) " 65448"
    test "test1871" (lazy(sprintf "%- 1u" -88s)) " 65448"
    test "test1872" (lazy(sprintf "%- *u" 7 -88s)) " 65448 "
    test "test1873" (lazy(sprintf "% 0u" -88s)) " 65448"
    test "test1874" (lazy(sprintf "% 05u" -88s)) " 65448"
    test "test1875" (lazy(sprintf "% 01u" -88s)) " 65448"
    test "test1876" (lazy(sprintf "% 0*u" 7 -88s)) " 065448"
    test "test1877" (lazy(sprintf "%- 0u" -88s)) " 65448"
    test "test1878" (lazy(sprintf "%- 05u" -88s)) " 65448"
    test "test1879" (lazy(sprintf "%- 01u" -88s)) " 65448"
    test "test1880" (lazy(sprintf "%- 0*u" 7 -88s)) " 65448 "
    test "test1881" (lazy(sprintf "%u" 104us)) "104"
    test "test1882" (lazy(sprintf "%5u" 104us)) "  104"
    test "test1883" (lazy(sprintf "%1u" 104us)) "104"
    test "test1884" (lazy(sprintf "%*u" 7 104us)) "    104"
    test "test1885" (lazy(sprintf "%-u" 104us)) "104"
    test "test1886" (lazy(sprintf "%-5u" 104us)) "104  "
    test "test1887" (lazy(sprintf "%-1u" 104us)) "104"
    test "test1888" (lazy(sprintf "%-*u" 7 104us)) "104    "
    test "test1889" (lazy(sprintf "%0u" 104us)) "104"
    test "test1890" (lazy(sprintf "%05u" 104us)) "00104"
    test "test1891" (lazy(sprintf "%01u" 104us)) "104"
    test "test1892" (lazy(sprintf "%0*u" 7 104us)) "0000104"
    test "test1893" (lazy(sprintf "%-0u" 104us)) "104"
    test "test1894" (lazy(sprintf "%-05u" 104us)) "104  "
    test "test1895" (lazy(sprintf "%-01u" 104us)) "104"
    test "test1896" (lazy(sprintf "%-0*u" 7 104us)) "104    "
    test "test1897" (lazy(sprintf "%+u" 104us)) "+104"
    test "test1898" (lazy(sprintf "%+5u" 104us)) " +104"
    test "test1899" (lazy(sprintf "%+1u" 104us)) "+104"
    test "test1900" (lazy(sprintf "%+*u" 7 104us)) "   +104"
    test "test1901" (lazy(sprintf "%-+u" 104us)) "+104"
    test "test1902" (lazy(sprintf "%-+5u" 104us)) "+104 "
    test "test1903" (lazy(sprintf "%-+1u" 104us)) "+104"
    test "test1904" (lazy(sprintf "%-+*u" 7 104us)) "+104   "
    test "test1905" (lazy(sprintf "%+0u" 104us)) "+104"
    test "test1906" (lazy(sprintf "%+05u" 104us)) "+0104"
    test "test1907" (lazy(sprintf "%+01u" 104us)) "+104"
    test "test1908" (lazy(sprintf "%+0*u" 7 104us)) "+000104"
    test "test1909" (lazy(sprintf "%-+0u" 104us)) "+104"
    test "test1910" (lazy(sprintf "%-+05u" 104us)) "+104 "
    test "test1911" (lazy(sprintf "%-+01u" 104us)) "+104"
    test "test1912" (lazy(sprintf "%-+0*u" 7 104us)) "+104   "
    test "test1913" (lazy(sprintf "% u" 104us)) " 104"
    test "test1914" (lazy(sprintf "% 5u" 104us)) "  104"
    test "test1915" (lazy(sprintf "% 1u" 104us)) " 104"
    test "test1916" (lazy(sprintf "% *u" 7 104us)) "    104"
    test "test1917" (lazy(sprintf "%- u" 104us)) " 104"
    test "test1918" (lazy(sprintf "%- 5u" 104us)) " 104 "
    test "test1919" (lazy(sprintf "%- 1u" 104us)) " 104"
    test "test1920" (lazy(sprintf "%- *u" 7 104us)) " 104   "
    test "test1921" (lazy(sprintf "% 0u" 104us)) " 104"
    test "test1922" (lazy(sprintf "% 05u" 104us)) " 0104"
    test "test1923" (lazy(sprintf "% 01u" 104us)) " 104"
    test "test1924" (lazy(sprintf "% 0*u" 7 104us)) " 000104"
    test "test1925" (lazy(sprintf "%- 0u" 104us)) " 104"
    test "test1926" (lazy(sprintf "%- 05u" 104us)) " 104 "
    test "test1927" (lazy(sprintf "%- 01u" 104us)) " 104"
    test "test1928" (lazy(sprintf "%- 0*u" 7 104us)) " 104   "
    test "test1929" (lazy(sprintf "%u" 12y)) "12"
    test "test1930" (lazy(sprintf "%5u" 12y)) "   12"
    test "test1931" (lazy(sprintf "%1u" 12y)) "12"
    test "test1932" (lazy(sprintf "%*u" 7 12y)) "     12"
    test "test1933" (lazy(sprintf "%-u" 12y)) "12"
    test "test1934" (lazy(sprintf "%-5u" 12y)) "12   "
    test "test1935" (lazy(sprintf "%-1u" 12y)) "12"
    test "test1936" (lazy(sprintf "%-*u" 7 12y)) "12     "
    test "test1937" (lazy(sprintf "%0u" 12y)) "12"
    test "test1938" (lazy(sprintf "%05u" 12y)) "00012"
    test "test1939" (lazy(sprintf "%01u" 12y)) "12"
    test "test1940" (lazy(sprintf "%0*u" 7 12y)) "0000012"
    test "test1941" (lazy(sprintf "%-0u" 12y)) "12"
    test "test1942" (lazy(sprintf "%-05u" 12y)) "12   "
    test "test1943" (lazy(sprintf "%-01u" 12y)) "12"
    test "test1944" (lazy(sprintf "%-0*u" 7 12y)) "12     "
    test "test1945" (lazy(sprintf "%+u" 12y)) "+12"
    test "test1946" (lazy(sprintf "%+5u" 12y)) "  +12"
    test "test1947" (lazy(sprintf "%+1u" 12y)) "+12"
    test "test1948" (lazy(sprintf "%+*u" 7 12y)) "    +12"
    test "test1949" (lazy(sprintf "%-+u" 12y)) "+12"
    test "test1950" (lazy(sprintf "%-+5u" 12y)) "+12  "
    test "test1951" (lazy(sprintf "%-+1u" 12y)) "+12"
    test "test1952" (lazy(sprintf "%-+*u" 7 12y)) "+12    "
    test "test1953" (lazy(sprintf "%+0u" 12y)) "+12"
    test "test1954" (lazy(sprintf "%+05u" 12y)) "+0012"
    test "test1955" (lazy(sprintf "%+01u" 12y)) "+12"
    test "test1956" (lazy(sprintf "%+0*u" 7 12y)) "+000012"
    test "test1957" (lazy(sprintf "%-+0u" 12y)) "+12"
    test "test1958" (lazy(sprintf "%-+05u" 12y)) "+12  "
    test "test1959" (lazy(sprintf "%-+01u" 12y)) "+12"
    test "test1960" (lazy(sprintf "%-+0*u" 7 12y)) "+12    "
    test "test1961" (lazy(sprintf "% u" 12y)) " 12"
    test "test1962" (lazy(sprintf "% 5u" 12y)) "   12"
    test "test1963" (lazy(sprintf "% 1u" 12y)) " 12"
    test "test1964" (lazy(sprintf "% *u" 7 12y)) "     12"
    test "test1965" (lazy(sprintf "%- u" 12y)) " 12"
    test "test1966" (lazy(sprintf "%- 5u" 12y)) " 12  "
    test "test1967" (lazy(sprintf "%- 1u" 12y)) " 12"
    test "test1968" (lazy(sprintf "%- *u" 7 12y)) " 12    "
    test "test1969" (lazy(sprintf "% 0u" 12y)) " 12"
    test "test1970" (lazy(sprintf "% 05u" 12y)) " 0012"
    test "test1971" (lazy(sprintf "% 01u" 12y)) " 12"
    test "test1972" (lazy(sprintf "% 0*u" 7 12y)) " 000012"
    test "test1973" (lazy(sprintf "%- 0u" 12y)) " 12"
    test "test1974" (lazy(sprintf "%- 05u" 12y)) " 12  "
    test "test1975" (lazy(sprintf "%- 01u" 12y)) " 12"
    test "test1976" (lazy(sprintf "%- 0*u" 7 12y)) " 12    "
    test "test1977" (lazy(sprintf "%u" -94y)) "162"
    test "test1978" (lazy(sprintf "%5u" -94y)) "  162"
    test "test1979" (lazy(sprintf "%1u" -94y)) "162"
    test "test1980" (lazy(sprintf "%*u" 7 -94y)) "    162"
    test "test1981" (lazy(sprintf "%-u" -94y)) "162"
    test "test1982" (lazy(sprintf "%-5u" -94y)) "162  "
    test "test1983" (lazy(sprintf "%-1u" -94y)) "162"
    test "test1984" (lazy(sprintf "%-*u" 7 -94y)) "162    "
    test "test1985" (lazy(sprintf "%0u" -94y)) "162"
    test "test1986" (lazy(sprintf "%05u" -94y)) "00162"
    test "test1987" (lazy(sprintf "%01u" -94y)) "162"
    test "test1988" (lazy(sprintf "%0*u" 7 -94y)) "0000162"
    test "test1989" (lazy(sprintf "%-0u" -94y)) "162"
    test "test1990" (lazy(sprintf "%-05u" -94y)) "162  "
    test "test1991" (lazy(sprintf "%-01u" -94y)) "162"
    test "test1992" (lazy(sprintf "%-0*u" 7 -94y)) "162    "
    test "test1993" (lazy(sprintf "%+u" -94y)) "+162"
    test "test1994" (lazy(sprintf "%+5u" -94y)) " +162"
    test "test1995" (lazy(sprintf "%+1u" -94y)) "+162"
    test "test1996" (lazy(sprintf "%+*u" 7 -94y)) "   +162"
    test "test1997" (lazy(sprintf "%-+u" -94y)) "+162"
    test "test1998" (lazy(sprintf "%-+5u" -94y)) "+162 "
    test "test1999" (lazy(sprintf "%-+1u" -94y)) "+162"
    test "test2000" (lazy(sprintf "%-+*u" 7 -94y)) "+162   "
let func2000()=
    test "test2001" (lazy(sprintf "%+0u" -94y)) "+162"
    test "test2002" (lazy(sprintf "%+05u" -94y)) "+0162"
    test "test2003" (lazy(sprintf "%+01u" -94y)) "+162"
    test "test2004" (lazy(sprintf "%+0*u" 7 -94y)) "+000162"
    test "test2005" (lazy(sprintf "%-+0u" -94y)) "+162"
    test "test2006" (lazy(sprintf "%-+05u" -94y)) "+162 "
    test "test2007" (lazy(sprintf "%-+01u" -94y)) "+162"
    test "test2008" (lazy(sprintf "%-+0*u" 7 -94y)) "+162   "
    test "test2009" (lazy(sprintf "% u" -94y)) " 162"
    test "test2010" (lazy(sprintf "% 5u" -94y)) "  162"
    test "test2011" (lazy(sprintf "% 1u" -94y)) " 162"
    test "test2012" (lazy(sprintf "% *u" 7 -94y)) "    162"
    test "test2013" (lazy(sprintf "%- u" -94y)) " 162"
    test "test2014" (lazy(sprintf "%- 5u" -94y)) " 162 "
    test "test2015" (lazy(sprintf "%- 1u" -94y)) " 162"
    test "test2016" (lazy(sprintf "%- *u" 7 -94y)) " 162   "
    test "test2017" (lazy(sprintf "% 0u" -94y)) " 162"
    test "test2018" (lazy(sprintf "% 05u" -94y)) " 0162"
    test "test2019" (lazy(sprintf "% 01u" -94y)) " 162"
    test "test2020" (lazy(sprintf "% 0*u" 7 -94y)) " 000162"
    test "test2021" (lazy(sprintf "%- 0u" -94y)) " 162"
    test "test2022" (lazy(sprintf "%- 05u" -94y)) " 162 "
    test "test2023" (lazy(sprintf "%- 01u" -94y)) " 162"
    test "test2024" (lazy(sprintf "%- 0*u" 7 -94y)) " 162   "
    test "test2025" (lazy(sprintf "%u" 88uy)) "88"
    test "test2026" (lazy(sprintf "%5u" 88uy)) "   88"
    test "test2027" (lazy(sprintf "%1u" 88uy)) "88"
    test "test2028" (lazy(sprintf "%*u" 7 88uy)) "     88"
    test "test2029" (lazy(sprintf "%-u" 88uy)) "88"
    test "test2030" (lazy(sprintf "%-5u" 88uy)) "88   "
    test "test2031" (lazy(sprintf "%-1u" 88uy)) "88"
    test "test2032" (lazy(sprintf "%-*u" 7 88uy)) "88     "
    test "test2033" (lazy(sprintf "%0u" 88uy)) "88"
    test "test2034" (lazy(sprintf "%05u" 88uy)) "00088"
    test "test2035" (lazy(sprintf "%01u" 88uy)) "88"
    test "test2036" (lazy(sprintf "%0*u" 7 88uy)) "0000088"
    test "test2037" (lazy(sprintf "%-0u" 88uy)) "88"
    test "test2038" (lazy(sprintf "%-05u" 88uy)) "88   "
    test "test2039" (lazy(sprintf "%-01u" 88uy)) "88"
    test "test2040" (lazy(sprintf "%-0*u" 7 88uy)) "88     "
    test "test2041" (lazy(sprintf "%+u" 88uy)) "+88"
    test "test2042" (lazy(sprintf "%+5u" 88uy)) "  +88"
    test "test2043" (lazy(sprintf "%+1u" 88uy)) "+88"
    test "test2044" (lazy(sprintf "%+*u" 7 88uy)) "    +88"
    test "test2045" (lazy(sprintf "%-+u" 88uy)) "+88"
    test "test2046" (lazy(sprintf "%-+5u" 88uy)) "+88  "
    test "test2047" (lazy(sprintf "%-+1u" 88uy)) "+88"
    test "test2048" (lazy(sprintf "%-+*u" 7 88uy)) "+88    "
    test "test2049" (lazy(sprintf "%+0u" 88uy)) "+88"
    test "test2050" (lazy(sprintf "%+05u" 88uy)) "+0088"
    test "test2051" (lazy(sprintf "%+01u" 88uy)) "+88"
    test "test2052" (lazy(sprintf "%+0*u" 7 88uy)) "+000088"
    test "test2053" (lazy(sprintf "%-+0u" 88uy)) "+88"
    test "test2054" (lazy(sprintf "%-+05u" 88uy)) "+88  "
    test "test2055" (lazy(sprintf "%-+01u" 88uy)) "+88"
    test "test2056" (lazy(sprintf "%-+0*u" 7 88uy)) "+88    "
    test "test2057" (lazy(sprintf "% u" 88uy)) " 88"
    test "test2058" (lazy(sprintf "% 5u" 88uy)) "   88"
    test "test2059" (lazy(sprintf "% 1u" 88uy)) " 88"
    test "test2060" (lazy(sprintf "% *u" 7 88uy)) "     88"
    test "test2061" (lazy(sprintf "%- u" 88uy)) " 88"
    test "test2062" (lazy(sprintf "%- 5u" 88uy)) " 88  "
    test "test2063" (lazy(sprintf "%- 1u" 88uy)) " 88"
    test "test2064" (lazy(sprintf "%- *u" 7 88uy)) " 88    "
    test "test2065" (lazy(sprintf "% 0u" 88uy)) " 88"
    test "test2066" (lazy(sprintf "% 05u" 88uy)) " 0088"
    test "test2067" (lazy(sprintf "% 01u" 88uy)) " 88"
    test "test2068" (lazy(sprintf "% 0*u" 7 88uy)) " 000088"
    test "test2069" (lazy(sprintf "%- 0u" 88uy)) " 88"
    test "test2070" (lazy(sprintf "%- 05u" 88uy)) " 88  "
    test "test2071" (lazy(sprintf "%- 01u" 88uy)) " 88"
    test "test2072" (lazy(sprintf "%- 0*u" 7 88uy)) " 88    "
    test "test2073" (lazy(sprintf "%u" 999L)) "999"
    test "test2074" (lazy(sprintf "%5u" 999L)) "  999"
    test "test2075" (lazy(sprintf "%1u" 999L)) "999"
    test "test2076" (lazy(sprintf "%*u" 7 999L)) "    999"
    test "test2077" (lazy(sprintf "%-u" 999L)) "999"
    test "test2078" (lazy(sprintf "%-5u" 999L)) "999  "
    test "test2079" (lazy(sprintf "%-1u" 999L)) "999"
    test "test2080" (lazy(sprintf "%-*u" 7 999L)) "999    "
    test "test2081" (lazy(sprintf "%0u" 999L)) "999"
    test "test2082" (lazy(sprintf "%05u" 999L)) "00999"
    test "test2083" (lazy(sprintf "%01u" 999L)) "999"
    test "test2084" (lazy(sprintf "%0*u" 7 999L)) "0000999"
    test "test2085" (lazy(sprintf "%-0u" 999L)) "999"
    test "test2086" (lazy(sprintf "%-05u" 999L)) "999  "
    test "test2087" (lazy(sprintf "%-01u" 999L)) "999"
    test "test2088" (lazy(sprintf "%-0*u" 7 999L)) "999    "
    test "test2089" (lazy(sprintf "%+u" 999L)) "+999"
    test "test2090" (lazy(sprintf "%+5u" 999L)) " +999"
    test "test2091" (lazy(sprintf "%+1u" 999L)) "+999"
    test "test2092" (lazy(sprintf "%+*u" 7 999L)) "   +999"
    test "test2093" (lazy(sprintf "%-+u" 999L)) "+999"
    test "test2094" (lazy(sprintf "%-+5u" 999L)) "+999 "
    test "test2095" (lazy(sprintf "%-+1u" 999L)) "+999"
    test "test2096" (lazy(sprintf "%-+*u" 7 999L)) "+999   "
    test "test2097" (lazy(sprintf "%+0u" 999L)) "+999"
    test "test2098" (lazy(sprintf "%+05u" 999L)) "+0999"
    test "test2099" (lazy(sprintf "%+01u" 999L)) "+999"
    test "test2100" (lazy(sprintf "%+0*u" 7 999L)) "+000999"
    test "test2101" (lazy(sprintf "%-+0u" 999L)) "+999"
    test "test2102" (lazy(sprintf "%-+05u" 999L)) "+999 "
    test "test2103" (lazy(sprintf "%-+01u" 999L)) "+999"
    test "test2104" (lazy(sprintf "%-+0*u" 7 999L)) "+999   "
    test "test2105" (lazy(sprintf "% u" 999L)) " 999"
    test "test2106" (lazy(sprintf "% 5u" 999L)) "  999"
    test "test2107" (lazy(sprintf "% 1u" 999L)) " 999"
    test "test2108" (lazy(sprintf "% *u" 7 999L)) "    999"
    test "test2109" (lazy(sprintf "%- u" 999L)) " 999"
    test "test2110" (lazy(sprintf "%- 5u" 999L)) " 999 "
    test "test2111" (lazy(sprintf "%- 1u" 999L)) " 999"
    test "test2112" (lazy(sprintf "%- *u" 7 999L)) " 999   "
    test "test2113" (lazy(sprintf "% 0u" 999L)) " 999"
    test "test2114" (lazy(sprintf "% 05u" 999L)) " 0999"
    test "test2115" (lazy(sprintf "% 01u" 999L)) " 999"
    test "test2116" (lazy(sprintf "% 0*u" 7 999L)) " 000999"
    test "test2117" (lazy(sprintf "%- 0u" 999L)) " 999"
    test "test2118" (lazy(sprintf "%- 05u" 999L)) " 999 "
    test "test2119" (lazy(sprintf "%- 01u" 999L)) " 999"
    test "test2120" (lazy(sprintf "%- 0*u" 7 999L)) " 999   "
    test "test2121" (lazy(sprintf "%u" -321L)) "18446744073709551295"
    test "test2122" (lazy(sprintf "%5u" -321L)) "18446744073709551295"
    test "test2123" (lazy(sprintf "%1u" -321L)) "18446744073709551295"
    test "test2124" (lazy(sprintf "%*u" 7 -321L)) "18446744073709551295"
    test "test2125" (lazy(sprintf "%-u" -321L)) "18446744073709551295"
    test "test2126" (lazy(sprintf "%-5u" -321L)) "18446744073709551295"
    test "test2127" (lazy(sprintf "%-1u" -321L)) "18446744073709551295"
    test "test2128" (lazy(sprintf "%-*u" 7 -321L)) "18446744073709551295"
    test "test2129" (lazy(sprintf "%0u" -321L)) "18446744073709551295"
    test "test2130" (lazy(sprintf "%05u" -321L)) "18446744073709551295"
    test "test2131" (lazy(sprintf "%01u" -321L)) "18446744073709551295"
    test "test2132" (lazy(sprintf "%0*u" 7 -321L)) "18446744073709551295"
    test "test2133" (lazy(sprintf "%-0u" -321L)) "18446744073709551295"
    test "test2134" (lazy(sprintf "%-05u" -321L)) "18446744073709551295"
    test "test2135" (lazy(sprintf "%-01u" -321L)) "18446744073709551295"
    test "test2136" (lazy(sprintf "%-0*u" 7 -321L)) "18446744073709551295"
    test "test2137" (lazy(sprintf "%+u" -321L)) "+18446744073709551295"
    test "test2138" (lazy(sprintf "%+5u" -321L)) "+18446744073709551295"
    test "test2139" (lazy(sprintf "%+1u" -321L)) "+18446744073709551295"
    test "test2140" (lazy(sprintf "%+*u" 7 -321L)) "+18446744073709551295"
    test "test2141" (lazy(sprintf "%-+u" -321L)) "+18446744073709551295"
    test "test2142" (lazy(sprintf "%-+5u" -321L)) "+18446744073709551295"
    test "test2143" (lazy(sprintf "%-+1u" -321L)) "+18446744073709551295"
    test "test2144" (lazy(sprintf "%-+*u" 7 -321L)) "+18446744073709551295"
    test "test2145" (lazy(sprintf "%+0u" -321L)) "+18446744073709551295"
    test "test2146" (lazy(sprintf "%+05u" -321L)) "+18446744073709551295"
    test "test2147" (lazy(sprintf "%+01u" -321L)) "+18446744073709551295"
    test "test2148" (lazy(sprintf "%+0*u" 7 -321L)) "+18446744073709551295"
    test "test2149" (lazy(sprintf "%-+0u" -321L)) "+18446744073709551295"
    test "test2150" (lazy(sprintf "%-+05u" -321L)) "+18446744073709551295"
    test "test2151" (lazy(sprintf "%-+01u" -321L)) "+18446744073709551295"
    test "test2152" (lazy(sprintf "%-+0*u" 7 -321L)) "+18446744073709551295"
    test "test2153" (lazy(sprintf "% u" -321L)) " 18446744073709551295"
    test "test2154" (lazy(sprintf "% 5u" -321L)) " 18446744073709551295"
    test "test2155" (lazy(sprintf "% 1u" -321L)) " 18446744073709551295"
    test "test2156" (lazy(sprintf "% *u" 7 -321L)) " 18446744073709551295"
    test "test2157" (lazy(sprintf "%- u" -321L)) " 18446744073709551295"
    test "test2158" (lazy(sprintf "%- 5u" -321L)) " 18446744073709551295"
    test "test2159" (lazy(sprintf "%- 1u" -321L)) " 18446744073709551295"
    test "test2160" (lazy(sprintf "%- *u" 7 -321L)) " 18446744073709551295"
    test "test2161" (lazy(sprintf "% 0u" -321L)) " 18446744073709551295"
    test "test2162" (lazy(sprintf "% 05u" -321L)) " 18446744073709551295"
    test "test2163" (lazy(sprintf "% 01u" -321L)) " 18446744073709551295"
    test "test2164" (lazy(sprintf "% 0*u" 7 -321L)) " 18446744073709551295"
    test "test2165" (lazy(sprintf "%- 0u" -321L)) " 18446744073709551295"
    test "test2166" (lazy(sprintf "%- 05u" -321L)) " 18446744073709551295"
    test "test2167" (lazy(sprintf "%- 01u" -321L)) " 18446744073709551295"
    test "test2168" (lazy(sprintf "%- 0*u" 7 -321L)) " 18446744073709551295"
    test "test2169" (lazy(sprintf "%u" 50000UL)) "50000"
    test "test2170" (lazy(sprintf "%5u" 50000UL)) "50000"
    test "test2171" (lazy(sprintf "%1u" 50000UL)) "50000"
    test "test2172" (lazy(sprintf "%*u" 7 50000UL)) "  50000"
    test "test2173" (lazy(sprintf "%-u" 50000UL)) "50000"
    test "test2174" (lazy(sprintf "%-5u" 50000UL)) "50000"
    test "test2175" (lazy(sprintf "%-1u" 50000UL)) "50000"
    test "test2176" (lazy(sprintf "%-*u" 7 50000UL)) "50000  "
    test "test2177" (lazy(sprintf "%0u" 50000UL)) "50000"
    test "test2178" (lazy(sprintf "%05u" 50000UL)) "50000"
    test "test2179" (lazy(sprintf "%01u" 50000UL)) "50000"
    test "test2180" (lazy(sprintf "%0*u" 7 50000UL)) "0050000"
    test "test2181" (lazy(sprintf "%-0u" 50000UL)) "50000"
    test "test2182" (lazy(sprintf "%-05u" 50000UL)) "50000"
    test "test2183" (lazy(sprintf "%-01u" 50000UL)) "50000"
    test "test2184" (lazy(sprintf "%-0*u" 7 50000UL)) "50000  "
    test "test2185" (lazy(sprintf "%+u" 50000UL)) "+50000"
    test "test2186" (lazy(sprintf "%+5u" 50000UL)) "+50000"
    test "test2187" (lazy(sprintf "%+1u" 50000UL)) "+50000"
    test "test2188" (lazy(sprintf "%+*u" 7 50000UL)) " +50000"
    test "test2189" (lazy(sprintf "%-+u" 50000UL)) "+50000"
    test "test2190" (lazy(sprintf "%-+5u" 50000UL)) "+50000"
    test "test2191" (lazy(sprintf "%-+1u" 50000UL)) "+50000"
    test "test2192" (lazy(sprintf "%-+*u" 7 50000UL)) "+50000 "
    test "test2193" (lazy(sprintf "%+0u" 50000UL)) "+50000"
    test "test2194" (lazy(sprintf "%+05u" 50000UL)) "+50000"
    test "test2195" (lazy(sprintf "%+01u" 50000UL)) "+50000"
    test "test2196" (lazy(sprintf "%+0*u" 7 50000UL)) "+050000"
    test "test2197" (lazy(sprintf "%-+0u" 50000UL)) "+50000"
    test "test2198" (lazy(sprintf "%-+05u" 50000UL)) "+50000"
    test "test2199" (lazy(sprintf "%-+01u" 50000UL)) "+50000"
    test "test2200" (lazy(sprintf "%-+0*u" 7 50000UL)) "+50000 "
    test "test2201" (lazy(sprintf "% u" 50000UL)) " 50000"
    test "test2202" (lazy(sprintf "% 5u" 50000UL)) " 50000"
    test "test2203" (lazy(sprintf "% 1u" 50000UL)) " 50000"
    test "test2204" (lazy(sprintf "% *u" 7 50000UL)) "  50000"
    test "test2205" (lazy(sprintf "%- u" 50000UL)) " 50000"
    test "test2206" (lazy(sprintf "%- 5u" 50000UL)) " 50000"
    test "test2207" (lazy(sprintf "%- 1u" 50000UL)) " 50000"
    test "test2208" (lazy(sprintf "%- *u" 7 50000UL)) " 50000 "
    test "test2209" (lazy(sprintf "% 0u" 50000UL)) " 50000"
    test "test2210" (lazy(sprintf "% 05u" 50000UL)) " 50000"
    test "test2211" (lazy(sprintf "% 01u" 50000UL)) " 50000"
    test "test2212" (lazy(sprintf "% 0*u" 7 50000UL)) " 050000"
    test "test2213" (lazy(sprintf "%- 0u" 50000UL)) " 50000"
    test "test2214" (lazy(sprintf "%- 05u" 50000UL)) " 50000"
    test "test2215" (lazy(sprintf "%- 01u" 50000UL)) " 50000"
    test "test2216" (lazy(sprintf "%- 0*u" 7 50000UL)) " 50000 "
    test "test2217" (lazy(sprintf "%u" System.Int32.MaxValue)) "2147483647"
    test "test2218" (lazy(sprintf "%5u" System.Int32.MaxValue)) "2147483647"
    test "test2219" (lazy(sprintf "%1u" System.Int32.MaxValue)) "2147483647"
    test "test2220" (lazy(sprintf "%*u" 7 System.Int32.MaxValue)) "2147483647"
    test "test2221" (lazy(sprintf "%-u" System.Int32.MaxValue)) "2147483647"
    test "test2222" (lazy(sprintf "%-5u" System.Int32.MaxValue)) "2147483647"
    test "test2223" (lazy(sprintf "%-1u" System.Int32.MaxValue)) "2147483647"
    test "test2224" (lazy(sprintf "%-*u" 7 System.Int32.MaxValue)) "2147483647"
    test "test2225" (lazy(sprintf "%0u" System.Int32.MaxValue)) "2147483647"
    test "test2226" (lazy(sprintf "%05u" System.Int32.MaxValue)) "2147483647"
    test "test2227" (lazy(sprintf "%01u" System.Int32.MaxValue)) "2147483647"
    test "test2228" (lazy(sprintf "%0*u" 7 System.Int32.MaxValue)) "2147483647"
    test "test2229" (lazy(sprintf "%-0u" System.Int32.MaxValue)) "2147483647"
    test "test2230" (lazy(sprintf "%-05u" System.Int32.MaxValue)) "2147483647"
    test "test2231" (lazy(sprintf "%-01u" System.Int32.MaxValue)) "2147483647"
    test "test2232" (lazy(sprintf "%-0*u" 7 System.Int32.MaxValue)) "2147483647"
    test "test2233" (lazy(sprintf "%+u" System.Int32.MaxValue)) "+2147483647"
    test "test2234" (lazy(sprintf "%+5u" System.Int32.MaxValue)) "+2147483647"
    test "test2235" (lazy(sprintf "%+1u" System.Int32.MaxValue)) "+2147483647"
    test "test2236" (lazy(sprintf "%+*u" 7 System.Int32.MaxValue)) "+2147483647"
    test "test2237" (lazy(sprintf "%-+u" System.Int32.MaxValue)) "+2147483647"
    test "test2238" (lazy(sprintf "%-+5u" System.Int32.MaxValue)) "+2147483647"
    test "test2239" (lazy(sprintf "%-+1u" System.Int32.MaxValue)) "+2147483647"
    test "test2240" (lazy(sprintf "%-+*u" 7 System.Int32.MaxValue)) "+2147483647"
    test "test2241" (lazy(sprintf "%+0u" System.Int32.MaxValue)) "+2147483647"
    test "test2242" (lazy(sprintf "%+05u" System.Int32.MaxValue)) "+2147483647"
    test "test2243" (lazy(sprintf "%+01u" System.Int32.MaxValue)) "+2147483647"
    test "test2244" (lazy(sprintf "%+0*u" 7 System.Int32.MaxValue)) "+2147483647"
    test "test2245" (lazy(sprintf "%-+0u" System.Int32.MaxValue)) "+2147483647"
    test "test2246" (lazy(sprintf "%-+05u" System.Int32.MaxValue)) "+2147483647"
    test "test2247" (lazy(sprintf "%-+01u" System.Int32.MaxValue)) "+2147483647"
    test "test2248" (lazy(sprintf "%-+0*u" 7 System.Int32.MaxValue)) "+2147483647"
    test "test2249" (lazy(sprintf "% u" System.Int32.MaxValue)) " 2147483647"
    test "test2250" (lazy(sprintf "% 5u" System.Int32.MaxValue)) " 2147483647"
    test "test2251" (lazy(sprintf "% 1u" System.Int32.MaxValue)) " 2147483647"
    test "test2252" (lazy(sprintf "% *u" 7 System.Int32.MaxValue)) " 2147483647"
    test "test2253" (lazy(sprintf "%- u" System.Int32.MaxValue)) " 2147483647"
    test "test2254" (lazy(sprintf "%- 5u" System.Int32.MaxValue)) " 2147483647"
    test "test2255" (lazy(sprintf "%- 1u" System.Int32.MaxValue)) " 2147483647"
    test "test2256" (lazy(sprintf "%- *u" 7 System.Int32.MaxValue)) " 2147483647"
    test "test2257" (lazy(sprintf "% 0u" System.Int32.MaxValue)) " 2147483647"
    test "test2258" (lazy(sprintf "% 05u" System.Int32.MaxValue)) " 2147483647"
    test "test2259" (lazy(sprintf "% 01u" System.Int32.MaxValue)) " 2147483647"
    test "test2260" (lazy(sprintf "% 0*u" 7 System.Int32.MaxValue)) " 2147483647"
    test "test2261" (lazy(sprintf "%- 0u" System.Int32.MaxValue)) " 2147483647"
    test "test2262" (lazy(sprintf "%- 05u" System.Int32.MaxValue)) " 2147483647"
    test "test2263" (lazy(sprintf "%- 01u" System.Int32.MaxValue)) " 2147483647"
    test "test2264" (lazy(sprintf "%- 0*u" 7 System.Int32.MaxValue)) " 2147483647"
    test "test2265" (lazy(sprintf "%u" System.Int64.MaxValue)) "9223372036854775807"
    test "test2266" (lazy(sprintf "%5u" System.Int64.MaxValue)) "9223372036854775807"
    test "test2267" (lazy(sprintf "%1u" System.Int64.MaxValue)) "9223372036854775807"
    test "test2268" (lazy(sprintf "%*u" 7 System.Int64.MaxValue)) "9223372036854775807"
    test "test2269" (lazy(sprintf "%-u" System.Int64.MaxValue)) "9223372036854775807"
    test "test2270" (lazy(sprintf "%-5u" System.Int64.MaxValue)) "9223372036854775807"
    test "test2271" (lazy(sprintf "%-1u" System.Int64.MaxValue)) "9223372036854775807"
    test "test2272" (lazy(sprintf "%-*u" 7 System.Int64.MaxValue)) "9223372036854775807"
    test "test2273" (lazy(sprintf "%0u" System.Int64.MaxValue)) "9223372036854775807"
    test "test2274" (lazy(sprintf "%05u" System.Int64.MaxValue)) "9223372036854775807"
    test "test2275" (lazy(sprintf "%01u" System.Int64.MaxValue)) "9223372036854775807"
    test "test2276" (lazy(sprintf "%0*u" 7 System.Int64.MaxValue)) "9223372036854775807"
    test "test2277" (lazy(sprintf "%-0u" System.Int64.MaxValue)) "9223372036854775807"
    test "test2278" (lazy(sprintf "%-05u" System.Int64.MaxValue)) "9223372036854775807"
    test "test2279" (lazy(sprintf "%-01u" System.Int64.MaxValue)) "9223372036854775807"
    test "test2280" (lazy(sprintf "%-0*u" 7 System.Int64.MaxValue)) "9223372036854775807"
    test "test2281" (lazy(sprintf "%+u" System.Int64.MaxValue)) "+9223372036854775807"
    test "test2282" (lazy(sprintf "%+5u" System.Int64.MaxValue)) "+9223372036854775807"
    test "test2283" (lazy(sprintf "%+1u" System.Int64.MaxValue)) "+9223372036854775807"
    test "test2284" (lazy(sprintf "%+*u" 7 System.Int64.MaxValue)) "+9223372036854775807"
    test "test2285" (lazy(sprintf "%-+u" System.Int64.MaxValue)) "+9223372036854775807"
    test "test2286" (lazy(sprintf "%-+5u" System.Int64.MaxValue)) "+9223372036854775807"
    test "test2287" (lazy(sprintf "%-+1u" System.Int64.MaxValue)) "+9223372036854775807"
    test "test2288" (lazy(sprintf "%-+*u" 7 System.Int64.MaxValue)) "+9223372036854775807"
    test "test2289" (lazy(sprintf "%+0u" System.Int64.MaxValue)) "+9223372036854775807"
    test "test2290" (lazy(sprintf "%+05u" System.Int64.MaxValue)) "+9223372036854775807"
    test "test2291" (lazy(sprintf "%+01u" System.Int64.MaxValue)) "+9223372036854775807"
    test "test2292" (lazy(sprintf "%+0*u" 7 System.Int64.MaxValue)) "+9223372036854775807"
    test "test2293" (lazy(sprintf "%-+0u" System.Int64.MaxValue)) "+9223372036854775807"
    test "test2294" (lazy(sprintf "%-+05u" System.Int64.MaxValue)) "+9223372036854775807"
    test "test2295" (lazy(sprintf "%-+01u" System.Int64.MaxValue)) "+9223372036854775807"
    test "test2296" (lazy(sprintf "%-+0*u" 7 System.Int64.MaxValue)) "+9223372036854775807"
    test "test2297" (lazy(sprintf "% u" System.Int64.MaxValue)) " 9223372036854775807"
    test "test2298" (lazy(sprintf "% 5u" System.Int64.MaxValue)) " 9223372036854775807"
    test "test2299" (lazy(sprintf "% 1u" System.Int64.MaxValue)) " 9223372036854775807"
    test "test2300" (lazy(sprintf "% *u" 7 System.Int64.MaxValue)) " 9223372036854775807"
    test "test2301" (lazy(sprintf "%- u" System.Int64.MaxValue)) " 9223372036854775807"
    test "test2302" (lazy(sprintf "%- 5u" System.Int64.MaxValue)) " 9223372036854775807"
    test "test2303" (lazy(sprintf "%- 1u" System.Int64.MaxValue)) " 9223372036854775807"
    test "test2304" (lazy(sprintf "%- *u" 7 System.Int64.MaxValue)) " 9223372036854775807"
    test "test2305" (lazy(sprintf "% 0u" System.Int64.MaxValue)) " 9223372036854775807"
    test "test2306" (lazy(sprintf "% 05u" System.Int64.MaxValue)) " 9223372036854775807"
    test "test2307" (lazy(sprintf "% 01u" System.Int64.MaxValue)) " 9223372036854775807"
    test "test2308" (lazy(sprintf "% 0*u" 7 System.Int64.MaxValue)) " 9223372036854775807"
    test "test2309" (lazy(sprintf "%- 0u" System.Int64.MaxValue)) " 9223372036854775807"
    test "test2310" (lazy(sprintf "%- 05u" System.Int64.MaxValue)) " 9223372036854775807"
    test "test2311" (lazy(sprintf "%- 01u" System.Int64.MaxValue)) " 9223372036854775807"
    test "test2312" (lazy(sprintf "%- 0*u" 7 System.Int64.MaxValue)) " 9223372036854775807"
    test "test2313" (lazy(sprintf "%u" System.Int32.MinValue)) "2147483648"
    test "test2314" (lazy(sprintf "%5u" System.Int32.MinValue)) "2147483648"
    test "test2315" (lazy(sprintf "%1u" System.Int32.MinValue)) "2147483648"
    test "test2316" (lazy(sprintf "%*u" 7 System.Int32.MinValue)) "2147483648"
    test "test2317" (lazy(sprintf "%-u" System.Int32.MinValue)) "2147483648"
    test "test2318" (lazy(sprintf "%-5u" System.Int32.MinValue)) "2147483648"
    test "test2319" (lazy(sprintf "%-1u" System.Int32.MinValue)) "2147483648"
    test "test2320" (lazy(sprintf "%-*u" 7 System.Int32.MinValue)) "2147483648"
    test "test2321" (lazy(sprintf "%0u" System.Int32.MinValue)) "2147483648"
    test "test2322" (lazy(sprintf "%05u" System.Int32.MinValue)) "2147483648"
    test "test2323" (lazy(sprintf "%01u" System.Int32.MinValue)) "2147483648"
    test "test2324" (lazy(sprintf "%0*u" 7 System.Int32.MinValue)) "2147483648"
    test "test2325" (lazy(sprintf "%-0u" System.Int32.MinValue)) "2147483648"
    test "test2326" (lazy(sprintf "%-05u" System.Int32.MinValue)) "2147483648"
    test "test2327" (lazy(sprintf "%-01u" System.Int32.MinValue)) "2147483648"
    test "test2328" (lazy(sprintf "%-0*u" 7 System.Int32.MinValue)) "2147483648"
    test "test2329" (lazy(sprintf "%+u" System.Int32.MinValue)) "+2147483648"
    test "test2330" (lazy(sprintf "%+5u" System.Int32.MinValue)) "+2147483648"
    test "test2331" (lazy(sprintf "%+1u" System.Int32.MinValue)) "+2147483648"
    test "test2332" (lazy(sprintf "%+*u" 7 System.Int32.MinValue)) "+2147483648"
    test "test2333" (lazy(sprintf "%-+u" System.Int32.MinValue)) "+2147483648"
    test "test2334" (lazy(sprintf "%-+5u" System.Int32.MinValue)) "+2147483648"
    test "test2335" (lazy(sprintf "%-+1u" System.Int32.MinValue)) "+2147483648"
    test "test2336" (lazy(sprintf "%-+*u" 7 System.Int32.MinValue)) "+2147483648"
    test "test2337" (lazy(sprintf "%+0u" System.Int32.MinValue)) "+2147483648"
    test "test2338" (lazy(sprintf "%+05u" System.Int32.MinValue)) "+2147483648"
    test "test2339" (lazy(sprintf "%+01u" System.Int32.MinValue)) "+2147483648"
    test "test2340" (lazy(sprintf "%+0*u" 7 System.Int32.MinValue)) "+2147483648"
    test "test2341" (lazy(sprintf "%-+0u" System.Int32.MinValue)) "+2147483648"
    test "test2342" (lazy(sprintf "%-+05u" System.Int32.MinValue)) "+2147483648"
    test "test2343" (lazy(sprintf "%-+01u" System.Int32.MinValue)) "+2147483648"
    test "test2344" (lazy(sprintf "%-+0*u" 7 System.Int32.MinValue)) "+2147483648"
    test "test2345" (lazy(sprintf "% u" System.Int32.MinValue)) " 2147483648"
    test "test2346" (lazy(sprintf "% 5u" System.Int32.MinValue)) " 2147483648"
    test "test2347" (lazy(sprintf "% 1u" System.Int32.MinValue)) " 2147483648"
    test "test2348" (lazy(sprintf "% *u" 7 System.Int32.MinValue)) " 2147483648"
    test "test2349" (lazy(sprintf "%- u" System.Int32.MinValue)) " 2147483648"
    test "test2350" (lazy(sprintf "%- 5u" System.Int32.MinValue)) " 2147483648"
    test "test2351" (lazy(sprintf "%- 1u" System.Int32.MinValue)) " 2147483648"
    test "test2352" (lazy(sprintf "%- *u" 7 System.Int32.MinValue)) " 2147483648"
    test "test2353" (lazy(sprintf "% 0u" System.Int32.MinValue)) " 2147483648"
    test "test2354" (lazy(sprintf "% 05u" System.Int32.MinValue)) " 2147483648"
    test "test2355" (lazy(sprintf "% 01u" System.Int32.MinValue)) " 2147483648"
    test "test2356" (lazy(sprintf "% 0*u" 7 System.Int32.MinValue)) " 2147483648"
    test "test2357" (lazy(sprintf "%- 0u" System.Int32.MinValue)) " 2147483648"
    test "test2358" (lazy(sprintf "%- 05u" System.Int32.MinValue)) " 2147483648"
    test "test2359" (lazy(sprintf "%- 01u" System.Int32.MinValue)) " 2147483648"
    test "test2360" (lazy(sprintf "%- 0*u" 7 System.Int32.MinValue)) " 2147483648"
    test "test2361" (lazy(sprintf "%u" System.Int64.MinValue)) "9223372036854775808"
    test "test2362" (lazy(sprintf "%5u" System.Int64.MinValue)) "9223372036854775808"
    test "test2363" (lazy(sprintf "%1u" System.Int64.MinValue)) "9223372036854775808"
    test "test2364" (lazy(sprintf "%*u" 7 System.Int64.MinValue)) "9223372036854775808"
    test "test2365" (lazy(sprintf "%-u" System.Int64.MinValue)) "9223372036854775808"
    test "test2366" (lazy(sprintf "%-5u" System.Int64.MinValue)) "9223372036854775808"
    test "test2367" (lazy(sprintf "%-1u" System.Int64.MinValue)) "9223372036854775808"
    test "test2368" (lazy(sprintf "%-*u" 7 System.Int64.MinValue)) "9223372036854775808"
    test "test2369" (lazy(sprintf "%0u" System.Int64.MinValue)) "9223372036854775808"
    test "test2370" (lazy(sprintf "%05u" System.Int64.MinValue)) "9223372036854775808"
    test "test2371" (lazy(sprintf "%01u" System.Int64.MinValue)) "9223372036854775808"
    test "test2372" (lazy(sprintf "%0*u" 7 System.Int64.MinValue)) "9223372036854775808"
    test "test2373" (lazy(sprintf "%-0u" System.Int64.MinValue)) "9223372036854775808"
    test "test2374" (lazy(sprintf "%-05u" System.Int64.MinValue)) "9223372036854775808"
    test "test2375" (lazy(sprintf "%-01u" System.Int64.MinValue)) "9223372036854775808"
    test "test2376" (lazy(sprintf "%-0*u" 7 System.Int64.MinValue)) "9223372036854775808"
    test "test2377" (lazy(sprintf "%+u" System.Int64.MinValue)) "+9223372036854775808"
    test "test2378" (lazy(sprintf "%+5u" System.Int64.MinValue)) "+9223372036854775808"
    test "test2379" (lazy(sprintf "%+1u" System.Int64.MinValue)) "+9223372036854775808"
    test "test2380" (lazy(sprintf "%+*u" 7 System.Int64.MinValue)) "+9223372036854775808"
    test "test2381" (lazy(sprintf "%-+u" System.Int64.MinValue)) "+9223372036854775808"
    test "test2382" (lazy(sprintf "%-+5u" System.Int64.MinValue)) "+9223372036854775808"
    test "test2383" (lazy(sprintf "%-+1u" System.Int64.MinValue)) "+9223372036854775808"
    test "test2384" (lazy(sprintf "%-+*u" 7 System.Int64.MinValue)) "+9223372036854775808"
    test "test2385" (lazy(sprintf "%+0u" System.Int64.MinValue)) "+9223372036854775808"
    test "test2386" (lazy(sprintf "%+05u" System.Int64.MinValue)) "+9223372036854775808"
    test "test2387" (lazy(sprintf "%+01u" System.Int64.MinValue)) "+9223372036854775808"
    test "test2388" (lazy(sprintf "%+0*u" 7 System.Int64.MinValue)) "+9223372036854775808"
    test "test2389" (lazy(sprintf "%-+0u" System.Int64.MinValue)) "+9223372036854775808"
    test "test2390" (lazy(sprintf "%-+05u" System.Int64.MinValue)) "+9223372036854775808"
    test "test2391" (lazy(sprintf "%-+01u" System.Int64.MinValue)) "+9223372036854775808"
    test "test2392" (lazy(sprintf "%-+0*u" 7 System.Int64.MinValue)) "+9223372036854775808"
    test "test2393" (lazy(sprintf "% u" System.Int64.MinValue)) " 9223372036854775808"
    test "test2394" (lazy(sprintf "% 5u" System.Int64.MinValue)) " 9223372036854775808"
    test "test2395" (lazy(sprintf "% 1u" System.Int64.MinValue)) " 9223372036854775808"
    test "test2396" (lazy(sprintf "% *u" 7 System.Int64.MinValue)) " 9223372036854775808"
    test "test2397" (lazy(sprintf "%- u" System.Int64.MinValue)) " 9223372036854775808"
    test "test2398" (lazy(sprintf "%- 5u" System.Int64.MinValue)) " 9223372036854775808"
    test "test2399" (lazy(sprintf "%- 1u" System.Int64.MinValue)) " 9223372036854775808"
    test "test2400" (lazy(sprintf "%- *u" 7 System.Int64.MinValue)) " 9223372036854775808"
    test "test2401" (lazy(sprintf "% 0u" System.Int64.MinValue)) " 9223372036854775808"
    test "test2402" (lazy(sprintf "% 05u" System.Int64.MinValue)) " 9223372036854775808"
    test "test2403" (lazy(sprintf "% 01u" System.Int64.MinValue)) " 9223372036854775808"
    test "test2404" (lazy(sprintf "% 0*u" 7 System.Int64.MinValue)) " 9223372036854775808"
    test "test2405" (lazy(sprintf "%- 0u" System.Int64.MinValue)) " 9223372036854775808"
    test "test2406" (lazy(sprintf "%- 05u" System.Int64.MinValue)) " 9223372036854775808"
    test "test2407" (lazy(sprintf "%- 01u" System.Int64.MinValue)) " 9223372036854775808"
    test "test2408" (lazy(sprintf "%- 0*u" 7 System.Int64.MinValue)) " 9223372036854775808"
    test "test2409" (lazy(sprintf "%u" 55n)) "55"
    test "test2410" (lazy(sprintf "%5u" 55n)) "   55"
    test "test2411" (lazy(sprintf "%1u" 55n)) "55"
    test "test2412" (lazy(sprintf "%*u" 7 55n)) "     55"
    test "test2413" (lazy(sprintf "%-u" 55n)) "55"
    test "test2414" (lazy(sprintf "%-5u" 55n)) "55   "
    test "test2415" (lazy(sprintf "%-1u" 55n)) "55"
    test "test2416" (lazy(sprintf "%-*u" 7 55n)) "55     "
    test "test2417" (lazy(sprintf "%0u" 55n)) "55"
    test "test2418" (lazy(sprintf "%05u" 55n)) "00055"
    test "test2419" (lazy(sprintf "%01u" 55n)) "55"
    test "test2420" (lazy(sprintf "%0*u" 7 55n)) "0000055"
    test "test2421" (lazy(sprintf "%-0u" 55n)) "55"
    test "test2422" (lazy(sprintf "%-05u" 55n)) "55   "
    test "test2423" (lazy(sprintf "%-01u" 55n)) "55"
    test "test2424" (lazy(sprintf "%-0*u" 7 55n)) "55     "
    test "test2425" (lazy(sprintf "%+u" 55n)) "+55"
    test "test2426" (lazy(sprintf "%+5u" 55n)) "  +55"
    test "test2427" (lazy(sprintf "%+1u" 55n)) "+55"
    test "test2428" (lazy(sprintf "%+*u" 7 55n)) "    +55"
    test "test2429" (lazy(sprintf "%-+u" 55n)) "+55"
    test "test2430" (lazy(sprintf "%-+5u" 55n)) "+55  "
    test "test2431" (lazy(sprintf "%-+1u" 55n)) "+55"
    test "test2432" (lazy(sprintf "%-+*u" 7 55n)) "+55    "
    test "test2433" (lazy(sprintf "%+0u" 55n)) "+55"
    test "test2434" (lazy(sprintf "%+05u" 55n)) "+0055"
    test "test2435" (lazy(sprintf "%+01u" 55n)) "+55"
    test "test2436" (lazy(sprintf "%+0*u" 7 55n)) "+000055"
    test "test2437" (lazy(sprintf "%-+0u" 55n)) "+55"
    test "test2438" (lazy(sprintf "%-+05u" 55n)) "+55  "
    test "test2439" (lazy(sprintf "%-+01u" 55n)) "+55"
    test "test2440" (lazy(sprintf "%-+0*u" 7 55n)) "+55    "
    test "test2441" (lazy(sprintf "% u" 55n)) " 55"
    test "test2442" (lazy(sprintf "% 5u" 55n)) "   55"
    test "test2443" (lazy(sprintf "% 1u" 55n)) " 55"
    test "test2444" (lazy(sprintf "% *u" 7 55n)) "     55"
    test "test2445" (lazy(sprintf "%- u" 55n)) " 55"
    test "test2446" (lazy(sprintf "%- 5u" 55n)) " 55  "
    test "test2447" (lazy(sprintf "%- 1u" 55n)) " 55"
    test "test2448" (lazy(sprintf "%- *u" 7 55n)) " 55    "
    test "test2449" (lazy(sprintf "% 0u" 55n)) " 55"
    test "test2450" (lazy(sprintf "% 05u" 55n)) " 0055"
    test "test2451" (lazy(sprintf "% 01u" 55n)) " 55"
    test "test2452" (lazy(sprintf "% 0*u" 7 55n)) " 000055"
    test "test2453" (lazy(sprintf "%- 0u" 55n)) " 55"
    test "test2454" (lazy(sprintf "%- 05u" 55n)) " 55  "
    test "test2455" (lazy(sprintf "%- 01u" 55n)) " 55"
    test "test2456" (lazy(sprintf "%- 0*u" 7 55n)) " 55    "
    test "test2457" (lazy(sprintf "%u" 999un)) "999"
    test "test2458" (lazy(sprintf "%5u" 999un)) "  999"
    test "test2459" (lazy(sprintf "%1u" 999un)) "999"
    test "test2460" (lazy(sprintf "%*u" 7 999un)) "    999"
    test "test2461" (lazy(sprintf "%-u" 999un)) "999"
    test "test2462" (lazy(sprintf "%-5u" 999un)) "999  "
    test "test2463" (lazy(sprintf "%-1u" 999un)) "999"
    test "test2464" (lazy(sprintf "%-*u" 7 999un)) "999    "
    test "test2465" (lazy(sprintf "%0u" 999un)) "999"
    test "test2466" (lazy(sprintf "%05u" 999un)) "00999"
    test "test2467" (lazy(sprintf "%01u" 999un)) "999"
    test "test2468" (lazy(sprintf "%0*u" 7 999un)) "0000999"
    test "test2469" (lazy(sprintf "%-0u" 999un)) "999"
    test "test2470" (lazy(sprintf "%-05u" 999un)) "999  "
    test "test2471" (lazy(sprintf "%-01u" 999un)) "999"
    test "test2472" (lazy(sprintf "%-0*u" 7 999un)) "999    "
    test "test2473" (lazy(sprintf "%+u" 999un)) "+999"
    test "test2474" (lazy(sprintf "%+5u" 999un)) " +999"
    test "test2475" (lazy(sprintf "%+1u" 999un)) "+999"
    test "test2476" (lazy(sprintf "%+*u" 7 999un)) "   +999"
    test "test2477" (lazy(sprintf "%-+u" 999un)) "+999"
    test "test2478" (lazy(sprintf "%-+5u" 999un)) "+999 "
    test "test2479" (lazy(sprintf "%-+1u" 999un)) "+999"
    test "test2480" (lazy(sprintf "%-+*u" 7 999un)) "+999   "
    test "test2481" (lazy(sprintf "%+0u" 999un)) "+999"
    test "test2482" (lazy(sprintf "%+05u" 999un)) "+0999"
    test "test2483" (lazy(sprintf "%+01u" 999un)) "+999"
    test "test2484" (lazy(sprintf "%+0*u" 7 999un)) "+000999"
    test "test2485" (lazy(sprintf "%-+0u" 999un)) "+999"
    test "test2486" (lazy(sprintf "%-+05u" 999un)) "+999 "
    test "test2487" (lazy(sprintf "%-+01u" 999un)) "+999"
    test "test2488" (lazy(sprintf "%-+0*u" 7 999un)) "+999   "
    test "test2489" (lazy(sprintf "% u" 999un)) " 999"
    test "test2490" (lazy(sprintf "% 5u" 999un)) "  999"
    test "test2491" (lazy(sprintf "% 1u" 999un)) " 999"
    test "test2492" (lazy(sprintf "% *u" 7 999un)) "    999"
    test "test2493" (lazy(sprintf "%- u" 999un)) " 999"
    test "test2494" (lazy(sprintf "%- 5u" 999un)) " 999 "
    test "test2495" (lazy(sprintf "%- 1u" 999un)) " 999"
    test "test2496" (lazy(sprintf "%- *u" 7 999un)) " 999   "
    test "test2497" (lazy(sprintf "% 0u" 999un)) " 999"
    test "test2498" (lazy(sprintf "% 05u" 999un)) " 0999"
    test "test2499" (lazy(sprintf "% 01u" 999un)) " 999"
    test "test2500" (lazy(sprintf "% 0*u" 7 999un)) " 000999"
    test "test2501" (lazy(sprintf "%- 0u" 999un)) " 999"
    test "test2502" (lazy(sprintf "%- 05u" 999un)) " 999 "
    test "test2503" (lazy(sprintf "%- 01u" 999un)) " 999"
    test "test2504" (lazy(sprintf "%- 0*u" 7 999un)) " 999   "
    test "test2505" (lazy(sprintf "%x" 14)) "e"
    test "test2506" (lazy(sprintf "%5x" 14)) "    e"
    test "test2507" (lazy(sprintf "%1x" 14)) "e"
    test "test2508" (lazy(sprintf "%*x" 7 14)) "      e"
    test "test2509" (lazy(sprintf "%-x" 14)) "e"
    test "test2510" (lazy(sprintf "%-5x" 14)) "e    "
    test "test2511" (lazy(sprintf "%-1x" 14)) "e"
    test "test2512" (lazy(sprintf "%-*x" 7 14)) "e      "
    test "test2513" (lazy(sprintf "%0x" 14)) "e"
    test "test2514" (lazy(sprintf "%05x" 14)) "0000e"
    test "test2515" (lazy(sprintf "%01x" 14)) "e"
    test "test2516" (lazy(sprintf "%0*x" 7 14)) "000000e"
    test "test2517" (lazy(sprintf "%-0x" 14)) "e"
    test "test2518" (lazy(sprintf "%-05x" 14)) "e    "
    test "test2519" (lazy(sprintf "%-01x" 14)) "e"
    test "test2520" (lazy(sprintf "%-0*x" 7 14)) "e      "
    test "test2521" (lazy(sprintf "%+x" 14)) "+e"
    test "test2522" (lazy(sprintf "%+5x" 14)) "   +e"
    test "test2523" (lazy(sprintf "%+1x" 14)) "+e"
    test "test2524" (lazy(sprintf "%+*x" 7 14)) "     +e"
    test "test2525" (lazy(sprintf "%-+x" 14)) "+e"
    test "test2526" (lazy(sprintf "%-+5x" 14)) "+e   "
    test "test2527" (lazy(sprintf "%-+1x" 14)) "+e"
    test "test2528" (lazy(sprintf "%-+*x" 7 14)) "+e     "
    test "test2529" (lazy(sprintf "%+0x" 14)) "+e"
    test "test2530" (lazy(sprintf "%+05x" 14)) "+000e"
    test "test2531" (lazy(sprintf "%+01x" 14)) "+e"
    test "test2532" (lazy(sprintf "%+0*x" 7 14)) "+00000e"
    test "test2533" (lazy(sprintf "%-+0x" 14)) "+e"
    test "test2534" (lazy(sprintf "%-+05x" 14)) "+e   "
    test "test2535" (lazy(sprintf "%-+01x" 14)) "+e"
    test "test2536" (lazy(sprintf "%-+0*x" 7 14)) "+e     "
    test "test2537" (lazy(sprintf "% x" 14)) " e"
    test "test2538" (lazy(sprintf "% 5x" 14)) "    e"
    test "test2539" (lazy(sprintf "% 1x" 14)) " e"
    test "test2540" (lazy(sprintf "% *x" 7 14)) "      e"
    test "test2541" (lazy(sprintf "%- x" 14)) " e"
    test "test2542" (lazy(sprintf "%- 5x" 14)) " e   "
    test "test2543" (lazy(sprintf "%- 1x" 14)) " e"
    test "test2544" (lazy(sprintf "%- *x" 7 14)) " e     "
    test "test2545" (lazy(sprintf "% 0x" 14)) " e"
    test "test2546" (lazy(sprintf "% 05x" 14)) " 000e"
    test "test2547" (lazy(sprintf "% 01x" 14)) " e"
    test "test2548" (lazy(sprintf "% 0*x" 7 14)) " 00000e"
    test "test2549" (lazy(sprintf "%- 0x" 14)) " e"
    test "test2550" (lazy(sprintf "%- 05x" 14)) " e   "
    test "test2551" (lazy(sprintf "%- 01x" 14)) " e"
    test "test2552" (lazy(sprintf "%- 0*x" 7 14)) " e     "
    test "test2553" (lazy(sprintf "%x" -10)) "fffffff6"
    test "test2554" (lazy(sprintf "%5x" -10)) "fffffff6"
    test "test2555" (lazy(sprintf "%1x" -10)) "fffffff6"
    test "test2556" (lazy(sprintf "%*x" 7 -10)) "fffffff6"
    test "test2557" (lazy(sprintf "%-x" -10)) "fffffff6"
    test "test2558" (lazy(sprintf "%-5x" -10)) "fffffff6"
    test "test2559" (lazy(sprintf "%-1x" -10)) "fffffff6"
    test "test2560" (lazy(sprintf "%-*x" 7 -10)) "fffffff6"
    test "test2561" (lazy(sprintf "%0x" -10)) "fffffff6"
    test "test2562" (lazy(sprintf "%05x" -10)) "fffffff6"
    test "test2563" (lazy(sprintf "%01x" -10)) "fffffff6"
    test "test2564" (lazy(sprintf "%0*x" 7 -10)) "fffffff6"
    test "test2565" (lazy(sprintf "%-0x" -10)) "fffffff6"
    test "test2566" (lazy(sprintf "%-05x" -10)) "fffffff6"
    test "test2567" (lazy(sprintf "%-01x" -10)) "fffffff6"
    test "test2568" (lazy(sprintf "%-0*x" 7 -10)) "fffffff6"
    test "test2569" (lazy(sprintf "%+x" -10)) "+fffffff6"
    test "test2570" (lazy(sprintf "%+5x" -10)) "+fffffff6"
    test "test2571" (lazy(sprintf "%+1x" -10)) "+fffffff6"
    test "test2572" (lazy(sprintf "%+*x" 7 -10)) "+fffffff6"
    test "test2573" (lazy(sprintf "%-+x" -10)) "+fffffff6"
    test "test2574" (lazy(sprintf "%-+5x" -10)) "+fffffff6"
    test "test2575" (lazy(sprintf "%-+1x" -10)) "+fffffff6"
    test "test2576" (lazy(sprintf "%-+*x" 7 -10)) "+fffffff6"
    test "test2577" (lazy(sprintf "%+0x" -10)) "+fffffff6"
    test "test2578" (lazy(sprintf "%+05x" -10)) "+fffffff6"
    test "test2579" (lazy(sprintf "%+01x" -10)) "+fffffff6"
    test "test2580" (lazy(sprintf "%+0*x" 7 -10)) "+fffffff6"
    test "test2581" (lazy(sprintf "%-+0x" -10)) "+fffffff6"
    test "test2582" (lazy(sprintf "%-+05x" -10)) "+fffffff6"
    test "test2583" (lazy(sprintf "%-+01x" -10)) "+fffffff6"
    test "test2584" (lazy(sprintf "%-+0*x" 7 -10)) "+fffffff6"
    test "test2585" (lazy(sprintf "% x" -10)) " fffffff6"
    test "test2586" (lazy(sprintf "% 5x" -10)) " fffffff6"
    test "test2587" (lazy(sprintf "% 1x" -10)) " fffffff6"
    test "test2588" (lazy(sprintf "% *x" 7 -10)) " fffffff6"
    test "test2589" (lazy(sprintf "%- x" -10)) " fffffff6"
    test "test2590" (lazy(sprintf "%- 5x" -10)) " fffffff6"
    test "test2591" (lazy(sprintf "%- 1x" -10)) " fffffff6"
    test "test2592" (lazy(sprintf "%- *x" 7 -10)) " fffffff6"
    test "test2593" (lazy(sprintf "% 0x" -10)) " fffffff6"
    test "test2594" (lazy(sprintf "% 05x" -10)) " fffffff6"
    test "test2595" (lazy(sprintf "% 01x" -10)) " fffffff6"
    test "test2596" (lazy(sprintf "% 0*x" 7 -10)) " fffffff6"
    test "test2597" (lazy(sprintf "%- 0x" -10)) " fffffff6"
    test "test2598" (lazy(sprintf "%- 05x" -10)) " fffffff6"
    test "test2599" (lazy(sprintf "%- 01x" -10)) " fffffff6"
    test "test2600" (lazy(sprintf "%- 0*x" 7 -10)) " fffffff6"
    test "test2601" (lazy(sprintf "%x" 55s)) "37"
    test "test2602" (lazy(sprintf "%5x" 55s)) "   37"
    test "test2603" (lazy(sprintf "%1x" 55s)) "37"
    test "test2604" (lazy(sprintf "%*x" 7 55s)) "     37"
    test "test2605" (lazy(sprintf "%-x" 55s)) "37"
    test "test2606" (lazy(sprintf "%-5x" 55s)) "37   "
    test "test2607" (lazy(sprintf "%-1x" 55s)) "37"
    test "test2608" (lazy(sprintf "%-*x" 7 55s)) "37     "
    test "test2609" (lazy(sprintf "%0x" 55s)) "37"
    test "test2610" (lazy(sprintf "%05x" 55s)) "00037"
    test "test2611" (lazy(sprintf "%01x" 55s)) "37"
    test "test2612" (lazy(sprintf "%0*x" 7 55s)) "0000037"
    test "test2613" (lazy(sprintf "%-0x" 55s)) "37"
    test "test2614" (lazy(sprintf "%-05x" 55s)) "37   "
    test "test2615" (lazy(sprintf "%-01x" 55s)) "37"
    test "test2616" (lazy(sprintf "%-0*x" 7 55s)) "37     "
    test "test2617" (lazy(sprintf "%+x" 55s)) "+37"
    test "test2618" (lazy(sprintf "%+5x" 55s)) "  +37"
    test "test2619" (lazy(sprintf "%+1x" 55s)) "+37"
    test "test2620" (lazy(sprintf "%+*x" 7 55s)) "    +37"
    test "test2621" (lazy(sprintf "%-+x" 55s)) "+37"
    test "test2622" (lazy(sprintf "%-+5x" 55s)) "+37  "
    test "test2623" (lazy(sprintf "%-+1x" 55s)) "+37"
    test "test2624" (lazy(sprintf "%-+*x" 7 55s)) "+37    "
    test "test2625" (lazy(sprintf "%+0x" 55s)) "+37"
    test "test2626" (lazy(sprintf "%+05x" 55s)) "+0037"
    test "test2627" (lazy(sprintf "%+01x" 55s)) "+37"
    test "test2628" (lazy(sprintf "%+0*x" 7 55s)) "+000037"
    test "test2629" (lazy(sprintf "%-+0x" 55s)) "+37"
    test "test2630" (lazy(sprintf "%-+05x" 55s)) "+37  "
    test "test2631" (lazy(sprintf "%-+01x" 55s)) "+37"
    test "test2632" (lazy(sprintf "%-+0*x" 7 55s)) "+37    "
    test "test2633" (lazy(sprintf "% x" 55s)) " 37"
    test "test2634" (lazy(sprintf "% 5x" 55s)) "   37"
    test "test2635" (lazy(sprintf "% 1x" 55s)) " 37"
    test "test2636" (lazy(sprintf "% *x" 7 55s)) "     37"
    test "test2637" (lazy(sprintf "%- x" 55s)) " 37"
    test "test2638" (lazy(sprintf "%- 5x" 55s)) " 37  "
    test "test2639" (lazy(sprintf "%- 1x" 55s)) " 37"
    test "test2640" (lazy(sprintf "%- *x" 7 55s)) " 37    "
    test "test2641" (lazy(sprintf "% 0x" 55s)) " 37"
    test "test2642" (lazy(sprintf "% 05x" 55s)) " 0037"
    test "test2643" (lazy(sprintf "% 01x" 55s)) " 37"
    test "test2644" (lazy(sprintf "% 0*x" 7 55s)) " 000037"
    test "test2645" (lazy(sprintf "%- 0x" 55s)) " 37"
    test "test2646" (lazy(sprintf "%- 05x" 55s)) " 37  "
    test "test2647" (lazy(sprintf "%- 01x" 55s)) " 37"
    test "test2648" (lazy(sprintf "%- 0*x" 7 55s)) " 37    "
    test "test2649" (lazy(sprintf "%x" -88s)) "ffa8"
    test "test2650" (lazy(sprintf "%5x" -88s)) " ffa8"
    test "test2651" (lazy(sprintf "%1x" -88s)) "ffa8"
    test "test2652" (lazy(sprintf "%*x" 7 -88s)) "   ffa8"
    test "test2653" (lazy(sprintf "%-x" -88s)) "ffa8"
    test "test2654" (lazy(sprintf "%-5x" -88s)) "ffa8 "
    test "test2655" (lazy(sprintf "%-1x" -88s)) "ffa8"
    test "test2656" (lazy(sprintf "%-*x" 7 -88s)) "ffa8   "
    test "test2657" (lazy(sprintf "%0x" -88s)) "ffa8"
    test "test2658" (lazy(sprintf "%05x" -88s)) "0ffa8"
    test "test2659" (lazy(sprintf "%01x" -88s)) "ffa8"
    test "test2660" (lazy(sprintf "%0*x" 7 -88s)) "000ffa8"
    test "test2661" (lazy(sprintf "%-0x" -88s)) "ffa8"
    test "test2662" (lazy(sprintf "%-05x" -88s)) "ffa8 "
    test "test2663" (lazy(sprintf "%-01x" -88s)) "ffa8"
    test "test2664" (lazy(sprintf "%-0*x" 7 -88s)) "ffa8   "
    test "test2665" (lazy(sprintf "%+x" -88s)) "+ffa8"
    test "test2666" (lazy(sprintf "%+5x" -88s)) "+ffa8"
    test "test2667" (lazy(sprintf "%+1x" -88s)) "+ffa8"
    test "test2668" (lazy(sprintf "%+*x" 7 -88s)) "  +ffa8"
    test "test2669" (lazy(sprintf "%-+x" -88s)) "+ffa8"
    test "test2670" (lazy(sprintf "%-+5x" -88s)) "+ffa8"
    test "test2671" (lazy(sprintf "%-+1x" -88s)) "+ffa8"
    test "test2672" (lazy(sprintf "%-+*x" 7 -88s)) "+ffa8  "
    test "test2673" (lazy(sprintf "%+0x" -88s)) "+ffa8"
    test "test2674" (lazy(sprintf "%+05x" -88s)) "+ffa8"
    test "test2675" (lazy(sprintf "%+01x" -88s)) "+ffa8"
    test "test2676" (lazy(sprintf "%+0*x" 7 -88s)) "+00ffa8"
    test "test2677" (lazy(sprintf "%-+0x" -88s)) "+ffa8"
    test "test2678" (lazy(sprintf "%-+05x" -88s)) "+ffa8"
    test "test2679" (lazy(sprintf "%-+01x" -88s)) "+ffa8"
    test "test2680" (lazy(sprintf "%-+0*x" 7 -88s)) "+ffa8  "
    test "test2681" (lazy(sprintf "% x" -88s)) " ffa8"
    test "test2682" (lazy(sprintf "% 5x" -88s)) " ffa8"
    test "test2683" (lazy(sprintf "% 1x" -88s)) " ffa8"
    test "test2684" (lazy(sprintf "% *x" 7 -88s)) "   ffa8"
    test "test2685" (lazy(sprintf "%- x" -88s)) " ffa8"
    test "test2686" (lazy(sprintf "%- 5x" -88s)) " ffa8"
    test "test2687" (lazy(sprintf "%- 1x" -88s)) " ffa8"
    test "test2688" (lazy(sprintf "%- *x" 7 -88s)) " ffa8  "
    test "test2689" (lazy(sprintf "% 0x" -88s)) " ffa8"
    test "test2690" (lazy(sprintf "% 05x" -88s)) " ffa8"
    test "test2691" (lazy(sprintf "% 01x" -88s)) " ffa8"
    test "test2692" (lazy(sprintf "% 0*x" 7 -88s)) " 00ffa8"
    test "test2693" (lazy(sprintf "%- 0x" -88s)) " ffa8"
    test "test2694" (lazy(sprintf "%- 05x" -88s)) " ffa8"
    test "test2695" (lazy(sprintf "%- 01x" -88s)) " ffa8"
    test "test2696" (lazy(sprintf "%- 0*x" 7 -88s)) " ffa8  "
    test "test2697" (lazy(sprintf "%x" 104us)) "68"
    test "test2698" (lazy(sprintf "%5x" 104us)) "   68"
    test "test2699" (lazy(sprintf "%1x" 104us)) "68"
    test "test2700" (lazy(sprintf "%*x" 7 104us)) "     68"
    test "test2701" (lazy(sprintf "%-x" 104us)) "68"
    test "test2702" (lazy(sprintf "%-5x" 104us)) "68   "
    test "test2703" (lazy(sprintf "%-1x" 104us)) "68"
    test "test2704" (lazy(sprintf "%-*x" 7 104us)) "68     "
    test "test2705" (lazy(sprintf "%0x" 104us)) "68"
    test "test2706" (lazy(sprintf "%05x" 104us)) "00068"
    test "test2707" (lazy(sprintf "%01x" 104us)) "68"
    test "test2708" (lazy(sprintf "%0*x" 7 104us)) "0000068"
    test "test2709" (lazy(sprintf "%-0x" 104us)) "68"
    test "test2710" (lazy(sprintf "%-05x" 104us)) "68   "
    test "test2711" (lazy(sprintf "%-01x" 104us)) "68"
    test "test2712" (lazy(sprintf "%-0*x" 7 104us)) "68     "
    test "test2713" (lazy(sprintf "%+x" 104us)) "+68"
    test "test2714" (lazy(sprintf "%+5x" 104us)) "  +68"
    test "test2715" (lazy(sprintf "%+1x" 104us)) "+68"
    test "test2716" (lazy(sprintf "%+*x" 7 104us)) "    +68"
    test "test2717" (lazy(sprintf "%-+x" 104us)) "+68"
    test "test2718" (lazy(sprintf "%-+5x" 104us)) "+68  "
    test "test2719" (lazy(sprintf "%-+1x" 104us)) "+68"
    test "test2720" (lazy(sprintf "%-+*x" 7 104us)) "+68    "
    test "test2721" (lazy(sprintf "%+0x" 104us)) "+68"
    test "test2722" (lazy(sprintf "%+05x" 104us)) "+0068"
    test "test2723" (lazy(sprintf "%+01x" 104us)) "+68"
    test "test2724" (lazy(sprintf "%+0*x" 7 104us)) "+000068"
    test "test2725" (lazy(sprintf "%-+0x" 104us)) "+68"
    test "test2726" (lazy(sprintf "%-+05x" 104us)) "+68  "
    test "test2727" (lazy(sprintf "%-+01x" 104us)) "+68"
    test "test2728" (lazy(sprintf "%-+0*x" 7 104us)) "+68    "
    test "test2729" (lazy(sprintf "% x" 104us)) " 68"
    test "test2730" (lazy(sprintf "% 5x" 104us)) "   68"
    test "test2731" (lazy(sprintf "% 1x" 104us)) " 68"
    test "test2732" (lazy(sprintf "% *x" 7 104us)) "     68"
    test "test2733" (lazy(sprintf "%- x" 104us)) " 68"
    test "test2734" (lazy(sprintf "%- 5x" 104us)) " 68  "
    test "test2735" (lazy(sprintf "%- 1x" 104us)) " 68"
    test "test2736" (lazy(sprintf "%- *x" 7 104us)) " 68    "
    test "test2737" (lazy(sprintf "% 0x" 104us)) " 68"
    test "test2738" (lazy(sprintf "% 05x" 104us)) " 0068"
    test "test2739" (lazy(sprintf "% 01x" 104us)) " 68"
    test "test2740" (lazy(sprintf "% 0*x" 7 104us)) " 000068"
    test "test2741" (lazy(sprintf "%- 0x" 104us)) " 68"
    test "test2742" (lazy(sprintf "%- 05x" 104us)) " 68  "
    test "test2743" (lazy(sprintf "%- 01x" 104us)) " 68"
    test "test2744" (lazy(sprintf "%- 0*x" 7 104us)) " 68    "
    test "test2745" (lazy(sprintf "%x" 12y)) "c"
    test "test2746" (lazy(sprintf "%5x" 12y)) "    c"
    test "test2747" (lazy(sprintf "%1x" 12y)) "c"
    test "test2748" (lazy(sprintf "%*x" 7 12y)) "      c"
    test "test2749" (lazy(sprintf "%-x" 12y)) "c"
    test "test2750" (lazy(sprintf "%-5x" 12y)) "c    "
    test "test2751" (lazy(sprintf "%-1x" 12y)) "c"
    test "test2752" (lazy(sprintf "%-*x" 7 12y)) "c      "
    test "test2753" (lazy(sprintf "%0x" 12y)) "c"
    test "test2754" (lazy(sprintf "%05x" 12y)) "0000c"
    test "test2755" (lazy(sprintf "%01x" 12y)) "c"
    test "test2756" (lazy(sprintf "%0*x" 7 12y)) "000000c"
    test "test2757" (lazy(sprintf "%-0x" 12y)) "c"
    test "test2758" (lazy(sprintf "%-05x" 12y)) "c    "
    test "test2759" (lazy(sprintf "%-01x" 12y)) "c"
    test "test2760" (lazy(sprintf "%-0*x" 7 12y)) "c      "
    test "test2761" (lazy(sprintf "%+x" 12y)) "+c"
    test "test2762" (lazy(sprintf "%+5x" 12y)) "   +c"
    test "test2763" (lazy(sprintf "%+1x" 12y)) "+c"
    test "test2764" (lazy(sprintf "%+*x" 7 12y)) "     +c"
    test "test2765" (lazy(sprintf "%-+x" 12y)) "+c"
    test "test2766" (lazy(sprintf "%-+5x" 12y)) "+c   "
    test "test2767" (lazy(sprintf "%-+1x" 12y)) "+c"
    test "test2768" (lazy(sprintf "%-+*x" 7 12y)) "+c     "
    test "test2769" (lazy(sprintf "%+0x" 12y)) "+c"
    test "test2770" (lazy(sprintf "%+05x" 12y)) "+000c"
    test "test2771" (lazy(sprintf "%+01x" 12y)) "+c"
    test "test2772" (lazy(sprintf "%+0*x" 7 12y)) "+00000c"
    test "test2773" (lazy(sprintf "%-+0x" 12y)) "+c"
    test "test2774" (lazy(sprintf "%-+05x" 12y)) "+c   "
    test "test2775" (lazy(sprintf "%-+01x" 12y)) "+c"
    test "test2776" (lazy(sprintf "%-+0*x" 7 12y)) "+c     "
    test "test2777" (lazy(sprintf "% x" 12y)) " c"
    test "test2778" (lazy(sprintf "% 5x" 12y)) "    c"
    test "test2779" (lazy(sprintf "% 1x" 12y)) " c"
    test "test2780" (lazy(sprintf "% *x" 7 12y)) "      c"
    test "test2781" (lazy(sprintf "%- x" 12y)) " c"
    test "test2782" (lazy(sprintf "%- 5x" 12y)) " c   "
    test "test2783" (lazy(sprintf "%- 1x" 12y)) " c"
    test "test2784" (lazy(sprintf "%- *x" 7 12y)) " c     "
    test "test2785" (lazy(sprintf "% 0x" 12y)) " c"
    test "test2786" (lazy(sprintf "% 05x" 12y)) " 000c"
    test "test2787" (lazy(sprintf "% 01x" 12y)) " c"
    test "test2788" (lazy(sprintf "% 0*x" 7 12y)) " 00000c"
    test "test2789" (lazy(sprintf "%- 0x" 12y)) " c"
    test "test2790" (lazy(sprintf "%- 05x" 12y)) " c   "
    test "test2791" (lazy(sprintf "%- 01x" 12y)) " c"
    test "test2792" (lazy(sprintf "%- 0*x" 7 12y)) " c     "
    test "test2793" (lazy(sprintf "%x" -94y)) "a2"
    test "test2794" (lazy(sprintf "%5x" -94y)) "   a2"
    test "test2795" (lazy(sprintf "%1x" -94y)) "a2"
    test "test2796" (lazy(sprintf "%*x" 7 -94y)) "     a2"
    test "test2797" (lazy(sprintf "%-x" -94y)) "a2"
    test "test2798" (lazy(sprintf "%-5x" -94y)) "a2   "
    test "test2799" (lazy(sprintf "%-1x" -94y)) "a2"
    test "test2800" (lazy(sprintf "%-*x" 7 -94y)) "a2     "
    test "test2801" (lazy(sprintf "%0x" -94y)) "a2"
    test "test2802" (lazy(sprintf "%05x" -94y)) "000a2"
    test "test2803" (lazy(sprintf "%01x" -94y)) "a2"
    test "test2804" (lazy(sprintf "%0*x" 7 -94y)) "00000a2"
    test "test2805" (lazy(sprintf "%-0x" -94y)) "a2"
    test "test2806" (lazy(sprintf "%-05x" -94y)) "a2   "
    test "test2807" (lazy(sprintf "%-01x" -94y)) "a2"
    test "test2808" (lazy(sprintf "%-0*x" 7 -94y)) "a2     "
    test "test2809" (lazy(sprintf "%+x" -94y)) "+a2"
    test "test2810" (lazy(sprintf "%+5x" -94y)) "  +a2"
    test "test2811" (lazy(sprintf "%+1x" -94y)) "+a2"
    test "test2812" (lazy(sprintf "%+*x" 7 -94y)) "    +a2"
    test "test2813" (lazy(sprintf "%-+x" -94y)) "+a2"
    test "test2814" (lazy(sprintf "%-+5x" -94y)) "+a2  "
    test "test2815" (lazy(sprintf "%-+1x" -94y)) "+a2"
    test "test2816" (lazy(sprintf "%-+*x" 7 -94y)) "+a2    "
    test "test2817" (lazy(sprintf "%+0x" -94y)) "+a2"
    test "test2818" (lazy(sprintf "%+05x" -94y)) "+00a2"
    test "test2819" (lazy(sprintf "%+01x" -94y)) "+a2"
    test "test2820" (lazy(sprintf "%+0*x" 7 -94y)) "+0000a2"
    test "test2821" (lazy(sprintf "%-+0x" -94y)) "+a2"
    test "test2822" (lazy(sprintf "%-+05x" -94y)) "+a2  "
    test "test2823" (lazy(sprintf "%-+01x" -94y)) "+a2"
    test "test2824" (lazy(sprintf "%-+0*x" 7 -94y)) "+a2    "
    test "test2825" (lazy(sprintf "% x" -94y)) " a2"
    test "test2826" (lazy(sprintf "% 5x" -94y)) "   a2"
    test "test2827" (lazy(sprintf "% 1x" -94y)) " a2"
    test "test2828" (lazy(sprintf "% *x" 7 -94y)) "     a2"
    test "test2829" (lazy(sprintf "%- x" -94y)) " a2"
    test "test2830" (lazy(sprintf "%- 5x" -94y)) " a2  "
    test "test2831" (lazy(sprintf "%- 1x" -94y)) " a2"
    test "test2832" (lazy(sprintf "%- *x" 7 -94y)) " a2    "
    test "test2833" (lazy(sprintf "% 0x" -94y)) " a2"
    test "test2834" (lazy(sprintf "% 05x" -94y)) " 00a2"
    test "test2835" (lazy(sprintf "% 01x" -94y)) " a2"
    test "test2836" (lazy(sprintf "% 0*x" 7 -94y)) " 0000a2"
    test "test2837" (lazy(sprintf "%- 0x" -94y)) " a2"
    test "test2838" (lazy(sprintf "%- 05x" -94y)) " a2  "
    test "test2839" (lazy(sprintf "%- 01x" -94y)) " a2"
    test "test2840" (lazy(sprintf "%- 0*x" 7 -94y)) " a2    "
    test "test2841" (lazy(sprintf "%x" 88uy)) "58"
    test "test2842" (lazy(sprintf "%5x" 88uy)) "   58"
    test "test2843" (lazy(sprintf "%1x" 88uy)) "58"
    test "test2844" (lazy(sprintf "%*x" 7 88uy)) "     58"
    test "test2845" (lazy(sprintf "%-x" 88uy)) "58"
    test "test2846" (lazy(sprintf "%-5x" 88uy)) "58   "
    test "test2847" (lazy(sprintf "%-1x" 88uy)) "58"
    test "test2848" (lazy(sprintf "%-*x" 7 88uy)) "58     "
    test "test2849" (lazy(sprintf "%0x" 88uy)) "58"
    test "test2850" (lazy(sprintf "%05x" 88uy)) "00058"
    test "test2851" (lazy(sprintf "%01x" 88uy)) "58"
    test "test2852" (lazy(sprintf "%0*x" 7 88uy)) "0000058"
    test "test2853" (lazy(sprintf "%-0x" 88uy)) "58"
    test "test2854" (lazy(sprintf "%-05x" 88uy)) "58   "
    test "test2855" (lazy(sprintf "%-01x" 88uy)) "58"
    test "test2856" (lazy(sprintf "%-0*x" 7 88uy)) "58     "
    test "test2857" (lazy(sprintf "%+x" 88uy)) "+58"
    test "test2858" (lazy(sprintf "%+5x" 88uy)) "  +58"
    test "test2859" (lazy(sprintf "%+1x" 88uy)) "+58"
    test "test2860" (lazy(sprintf "%+*x" 7 88uy)) "    +58"
    test "test2861" (lazy(sprintf "%-+x" 88uy)) "+58"
    test "test2862" (lazy(sprintf "%-+5x" 88uy)) "+58  "
    test "test2863" (lazy(sprintf "%-+1x" 88uy)) "+58"
    test "test2864" (lazy(sprintf "%-+*x" 7 88uy)) "+58    "
    test "test2865" (lazy(sprintf "%+0x" 88uy)) "+58"
    test "test2866" (lazy(sprintf "%+05x" 88uy)) "+0058"
    test "test2867" (lazy(sprintf "%+01x" 88uy)) "+58"
    test "test2868" (lazy(sprintf "%+0*x" 7 88uy)) "+000058"
    test "test2869" (lazy(sprintf "%-+0x" 88uy)) "+58"
    test "test2870" (lazy(sprintf "%-+05x" 88uy)) "+58  "
    test "test2871" (lazy(sprintf "%-+01x" 88uy)) "+58"
    test "test2872" (lazy(sprintf "%-+0*x" 7 88uy)) "+58    "
    test "test2873" (lazy(sprintf "% x" 88uy)) " 58"
    test "test2874" (lazy(sprintf "% 5x" 88uy)) "   58"
    test "test2875" (lazy(sprintf "% 1x" 88uy)) " 58"
    test "test2876" (lazy(sprintf "% *x" 7 88uy)) "     58"
    test "test2877" (lazy(sprintf "%- x" 88uy)) " 58"
    test "test2878" (lazy(sprintf "%- 5x" 88uy)) " 58  "
    test "test2879" (lazy(sprintf "%- 1x" 88uy)) " 58"
    test "test2880" (lazy(sprintf "%- *x" 7 88uy)) " 58    "
    test "test2881" (lazy(sprintf "% 0x" 88uy)) " 58"
    test "test2882" (lazy(sprintf "% 05x" 88uy)) " 0058"
    test "test2883" (lazy(sprintf "% 01x" 88uy)) " 58"
    test "test2884" (lazy(sprintf "% 0*x" 7 88uy)) " 000058"
    test "test2885" (lazy(sprintf "%- 0x" 88uy)) " 58"
    test "test2886" (lazy(sprintf "%- 05x" 88uy)) " 58  "
    test "test2887" (lazy(sprintf "%- 01x" 88uy)) " 58"
    test "test2888" (lazy(sprintf "%- 0*x" 7 88uy)) " 58    "
    test "test2889" (lazy(sprintf "%x" 999L)) "3e7"
    test "test2890" (lazy(sprintf "%5x" 999L)) "  3e7"
    test "test2891" (lazy(sprintf "%1x" 999L)) "3e7"
    test "test2892" (lazy(sprintf "%*x" 7 999L)) "    3e7"
    test "test2893" (lazy(sprintf "%-x" 999L)) "3e7"
    test "test2894" (lazy(sprintf "%-5x" 999L)) "3e7  "
    test "test2895" (lazy(sprintf "%-1x" 999L)) "3e7"
    test "test2896" (lazy(sprintf "%-*x" 7 999L)) "3e7    "
    test "test2897" (lazy(sprintf "%0x" 999L)) "3e7"
    test "test2898" (lazy(sprintf "%05x" 999L)) "003e7"
    test "test2899" (lazy(sprintf "%01x" 999L)) "3e7"
    test "test2900" (lazy(sprintf "%0*x" 7 999L)) "00003e7"
    test "test2901" (lazy(sprintf "%-0x" 999L)) "3e7"
    test "test2902" (lazy(sprintf "%-05x" 999L)) "3e7  "
    test "test2903" (lazy(sprintf "%-01x" 999L)) "3e7"
    test "test2904" (lazy(sprintf "%-0*x" 7 999L)) "3e7    "
    test "test2905" (lazy(sprintf "%+x" 999L)) "+3e7"
    test "test2906" (lazy(sprintf "%+5x" 999L)) " +3e7"
    test "test2907" (lazy(sprintf "%+1x" 999L)) "+3e7"
    test "test2908" (lazy(sprintf "%+*x" 7 999L)) "   +3e7"
    test "test2909" (lazy(sprintf "%-+x" 999L)) "+3e7"
    test "test2910" (lazy(sprintf "%-+5x" 999L)) "+3e7 "
    test "test2911" (lazy(sprintf "%-+1x" 999L)) "+3e7"
    test "test2912" (lazy(sprintf "%-+*x" 7 999L)) "+3e7   "
    test "test2913" (lazy(sprintf "%+0x" 999L)) "+3e7"
    test "test2914" (lazy(sprintf "%+05x" 999L)) "+03e7"
    test "test2915" (lazy(sprintf "%+01x" 999L)) "+3e7"
    test "test2916" (lazy(sprintf "%+0*x" 7 999L)) "+0003e7"
    test "test2917" (lazy(sprintf "%-+0x" 999L)) "+3e7"
    test "test2918" (lazy(sprintf "%-+05x" 999L)) "+3e7 "
    test "test2919" (lazy(sprintf "%-+01x" 999L)) "+3e7"
    test "test2920" (lazy(sprintf "%-+0*x" 7 999L)) "+3e7   "
    test "test2921" (lazy(sprintf "% x" 999L)) " 3e7"
    test "test2922" (lazy(sprintf "% 5x" 999L)) "  3e7"
    test "test2923" (lazy(sprintf "% 1x" 999L)) " 3e7"
    test "test2924" (lazy(sprintf "% *x" 7 999L)) "    3e7"
    test "test2925" (lazy(sprintf "%- x" 999L)) " 3e7"
    test "test2926" (lazy(sprintf "%- 5x" 999L)) " 3e7 "
    test "test2927" (lazy(sprintf "%- 1x" 999L)) " 3e7"
    test "test2928" (lazy(sprintf "%- *x" 7 999L)) " 3e7   "
    test "test2929" (lazy(sprintf "% 0x" 999L)) " 3e7"
    test "test2930" (lazy(sprintf "% 05x" 999L)) " 03e7"
    test "test2931" (lazy(sprintf "% 01x" 999L)) " 3e7"
    test "test2932" (lazy(sprintf "% 0*x" 7 999L)) " 0003e7"
    test "test2933" (lazy(sprintf "%- 0x" 999L)) " 3e7"
    test "test2934" (lazy(sprintf "%- 05x" 999L)) " 3e7 "
    test "test2935" (lazy(sprintf "%- 01x" 999L)) " 3e7"
    test "test2936" (lazy(sprintf "%- 0*x" 7 999L)) " 3e7   "
    test "test2937" (lazy(sprintf "%x" -321L)) "fffffffffffffebf"
    test "test2938" (lazy(sprintf "%5x" -321L)) "fffffffffffffebf"
    test "test2939" (lazy(sprintf "%1x" -321L)) "fffffffffffffebf"
    test "test2940" (lazy(sprintf "%*x" 7 -321L)) "fffffffffffffebf"
    test "test2941" (lazy(sprintf "%-x" -321L)) "fffffffffffffebf"
    test "test2942" (lazy(sprintf "%-5x" -321L)) "fffffffffffffebf"
    test "test2943" (lazy(sprintf "%-1x" -321L)) "fffffffffffffebf"
    test "test2944" (lazy(sprintf "%-*x" 7 -321L)) "fffffffffffffebf"
    test "test2945" (lazy(sprintf "%0x" -321L)) "fffffffffffffebf"
    test "test2946" (lazy(sprintf "%05x" -321L)) "fffffffffffffebf"
    test "test2947" (lazy(sprintf "%01x" -321L)) "fffffffffffffebf"
    test "test2948" (lazy(sprintf "%0*x" 7 -321L)) "fffffffffffffebf"
    test "test2949" (lazy(sprintf "%-0x" -321L)) "fffffffffffffebf"
    test "test2950" (lazy(sprintf "%-05x" -321L)) "fffffffffffffebf"
    test "test2951" (lazy(sprintf "%-01x" -321L)) "fffffffffffffebf"
    test "test2952" (lazy(sprintf "%-0*x" 7 -321L)) "fffffffffffffebf"
    test "test2953" (lazy(sprintf "%+x" -321L)) "+fffffffffffffebf"
    test "test2954" (lazy(sprintf "%+5x" -321L)) "+fffffffffffffebf"
    test "test2955" (lazy(sprintf "%+1x" -321L)) "+fffffffffffffebf"
    test "test2956" (lazy(sprintf "%+*x" 7 -321L)) "+fffffffffffffebf"
    test "test2957" (lazy(sprintf "%-+x" -321L)) "+fffffffffffffebf"
    test "test2958" (lazy(sprintf "%-+5x" -321L)) "+fffffffffffffebf"
    test "test2959" (lazy(sprintf "%-+1x" -321L)) "+fffffffffffffebf"
    test "test2960" (lazy(sprintf "%-+*x" 7 -321L)) "+fffffffffffffebf"
    test "test2961" (lazy(sprintf "%+0x" -321L)) "+fffffffffffffebf"
    test "test2962" (lazy(sprintf "%+05x" -321L)) "+fffffffffffffebf"
    test "test2963" (lazy(sprintf "%+01x" -321L)) "+fffffffffffffebf"
    test "test2964" (lazy(sprintf "%+0*x" 7 -321L)) "+fffffffffffffebf"
    test "test2965" (lazy(sprintf "%-+0x" -321L)) "+fffffffffffffebf"
    test "test2966" (lazy(sprintf "%-+05x" -321L)) "+fffffffffffffebf"
    test "test2967" (lazy(sprintf "%-+01x" -321L)) "+fffffffffffffebf"
    test "test2968" (lazy(sprintf "%-+0*x" 7 -321L)) "+fffffffffffffebf"
    test "test2969" (lazy(sprintf "% x" -321L)) " fffffffffffffebf"
    test "test2970" (lazy(sprintf "% 5x" -321L)) " fffffffffffffebf"
    test "test2971" (lazy(sprintf "% 1x" -321L)) " fffffffffffffebf"
    test "test2972" (lazy(sprintf "% *x" 7 -321L)) " fffffffffffffebf"
    test "test2973" (lazy(sprintf "%- x" -321L)) " fffffffffffffebf"
    test "test2974" (lazy(sprintf "%- 5x" -321L)) " fffffffffffffebf"
    test "test2975" (lazy(sprintf "%- 1x" -321L)) " fffffffffffffebf"
    test "test2976" (lazy(sprintf "%- *x" 7 -321L)) " fffffffffffffebf"
    test "test2977" (lazy(sprintf "% 0x" -321L)) " fffffffffffffebf"
    test "test2978" (lazy(sprintf "% 05x" -321L)) " fffffffffffffebf"
    test "test2979" (lazy(sprintf "% 01x" -321L)) " fffffffffffffebf"
    test "test2980" (lazy(sprintf "% 0*x" 7 -321L)) " fffffffffffffebf"
    test "test2981" (lazy(sprintf "%- 0x" -321L)) " fffffffffffffebf"
    test "test2982" (lazy(sprintf "%- 05x" -321L)) " fffffffffffffebf"
    test "test2983" (lazy(sprintf "%- 01x" -321L)) " fffffffffffffebf"
    test "test2984" (lazy(sprintf "%- 0*x" 7 -321L)) " fffffffffffffebf"
    test "test2985" (lazy(sprintf "%x" 50000UL)) "c350"
    test "test2986" (lazy(sprintf "%5x" 50000UL)) " c350"
    test "test2987" (lazy(sprintf "%1x" 50000UL)) "c350"
    test "test2988" (lazy(sprintf "%*x" 7 50000UL)) "   c350"
    test "test2989" (lazy(sprintf "%-x" 50000UL)) "c350"
    test "test2990" (lazy(sprintf "%-5x" 50000UL)) "c350 "
    test "test2991" (lazy(sprintf "%-1x" 50000UL)) "c350"
    test "test2992" (lazy(sprintf "%-*x" 7 50000UL)) "c350   "
    test "test2993" (lazy(sprintf "%0x" 50000UL)) "c350"
    test "test2994" (lazy(sprintf "%05x" 50000UL)) "0c350"
    test "test2995" (lazy(sprintf "%01x" 50000UL)) "c350"
    test "test2996" (lazy(sprintf "%0*x" 7 50000UL)) "000c350"
    test "test2997" (lazy(sprintf "%-0x" 50000UL)) "c350"
    test "test2998" (lazy(sprintf "%-05x" 50000UL)) "c350 "
    test "test2999" (lazy(sprintf "%-01x" 50000UL)) "c350"
    test "test3000" (lazy(sprintf "%-0*x" 7 50000UL)) "c350   "
#if COMPILED
let func3000()=
    test "test3001" (lazy(sprintf "%+x" 50000UL)) "+c350"
    test "test3002" (lazy(sprintf "%+5x" 50000UL)) "+c350"
    test "test3003" (lazy(sprintf "%+1x" 50000UL)) "+c350"
    test "test3004" (lazy(sprintf "%+*x" 7 50000UL)) "  +c350"
    test "test3005" (lazy(sprintf "%-+x" 50000UL)) "+c350"
    test "test3006" (lazy(sprintf "%-+5x" 50000UL)) "+c350"
    test "test3007" (lazy(sprintf "%-+1x" 50000UL)) "+c350"
    test "test3008" (lazy(sprintf "%-+*x" 7 50000UL)) "+c350  "
    test "test3009" (lazy(sprintf "%+0x" 50000UL)) "+c350"
    test "test3010" (lazy(sprintf "%+05x" 50000UL)) "+c350"
    test "test3011" (lazy(sprintf "%+01x" 50000UL)) "+c350"
    test "test3012" (lazy(sprintf "%+0*x" 7 50000UL)) "+00c350"
    test "test3013" (lazy(sprintf "%-+0x" 50000UL)) "+c350"
    test "test3014" (lazy(sprintf "%-+05x" 50000UL)) "+c350"
    test "test3015" (lazy(sprintf "%-+01x" 50000UL)) "+c350"
    test "test3016" (lazy(sprintf "%-+0*x" 7 50000UL)) "+c350  "
    test "test3017" (lazy(sprintf "% x" 50000UL)) " c350"
    test "test3018" (lazy(sprintf "% 5x" 50000UL)) " c350"
    test "test3019" (lazy(sprintf "% 1x" 50000UL)) " c350"
    test "test3020" (lazy(sprintf "% *x" 7 50000UL)) "   c350"
    test "test3021" (lazy(sprintf "%- x" 50000UL)) " c350"
    test "test3022" (lazy(sprintf "%- 5x" 50000UL)) " c350"
    test "test3023" (lazy(sprintf "%- 1x" 50000UL)) " c350"
    test "test3024" (lazy(sprintf "%- *x" 7 50000UL)) " c350  "
    test "test3025" (lazy(sprintf "% 0x" 50000UL)) " c350"
    test "test3026" (lazy(sprintf "% 05x" 50000UL)) " c350"
    test "test3027" (lazy(sprintf "% 01x" 50000UL)) " c350"
    test "test3028" (lazy(sprintf "% 0*x" 7 50000UL)) " 00c350"
    test "test3029" (lazy(sprintf "%- 0x" 50000UL)) " c350"
    test "test3030" (lazy(sprintf "%- 05x" 50000UL)) " c350"
    test "test3031" (lazy(sprintf "%- 01x" 50000UL)) " c350"
    test "test3032" (lazy(sprintf "%- 0*x" 7 50000UL)) " c350  "
    test "test3033" (lazy(sprintf "%x" System.Int32.MaxValue)) "7fffffff"
    test "test3034" (lazy(sprintf "%5x" System.Int32.MaxValue)) "7fffffff"
    test "test3035" (lazy(sprintf "%1x" System.Int32.MaxValue)) "7fffffff"
    test "test3036" (lazy(sprintf "%*x" 7 System.Int32.MaxValue)) "7fffffff"
    test "test3037" (lazy(sprintf "%-x" System.Int32.MaxValue)) "7fffffff"
    test "test3038" (lazy(sprintf "%-5x" System.Int32.MaxValue)) "7fffffff"
    test "test3039" (lazy(sprintf "%-1x" System.Int32.MaxValue)) "7fffffff"
    test "test3040" (lazy(sprintf "%-*x" 7 System.Int32.MaxValue)) "7fffffff"
    test "test3041" (lazy(sprintf "%0x" System.Int32.MaxValue)) "7fffffff"
    test "test3042" (lazy(sprintf "%05x" System.Int32.MaxValue)) "7fffffff"
    test "test3043" (lazy(sprintf "%01x" System.Int32.MaxValue)) "7fffffff"
    test "test3044" (lazy(sprintf "%0*x" 7 System.Int32.MaxValue)) "7fffffff"
    test "test3045" (lazy(sprintf "%-0x" System.Int32.MaxValue)) "7fffffff"
    test "test3046" (lazy(sprintf "%-05x" System.Int32.MaxValue)) "7fffffff"
    test "test3047" (lazy(sprintf "%-01x" System.Int32.MaxValue)) "7fffffff"
    test "test3048" (lazy(sprintf "%-0*x" 7 System.Int32.MaxValue)) "7fffffff"
    test "test3049" (lazy(sprintf "%+x" System.Int32.MaxValue)) "+7fffffff"
    test "test3050" (lazy(sprintf "%+5x" System.Int32.MaxValue)) "+7fffffff"
    test "test3051" (lazy(sprintf "%+1x" System.Int32.MaxValue)) "+7fffffff"
    test "test3052" (lazy(sprintf "%+*x" 7 System.Int32.MaxValue)) "+7fffffff"
    test "test3053" (lazy(sprintf "%-+x" System.Int32.MaxValue)) "+7fffffff"
    test "test3054" (lazy(sprintf "%-+5x" System.Int32.MaxValue)) "+7fffffff"
    test "test3055" (lazy(sprintf "%-+1x" System.Int32.MaxValue)) "+7fffffff"
    test "test3056" (lazy(sprintf "%-+*x" 7 System.Int32.MaxValue)) "+7fffffff"
    test "test3057" (lazy(sprintf "%+0x" System.Int32.MaxValue)) "+7fffffff"
    test "test3058" (lazy(sprintf "%+05x" System.Int32.MaxValue)) "+7fffffff"
    test "test3059" (lazy(sprintf "%+01x" System.Int32.MaxValue)) "+7fffffff"
    test "test3060" (lazy(sprintf "%+0*x" 7 System.Int32.MaxValue)) "+7fffffff"
    test "test3061" (lazy(sprintf "%-+0x" System.Int32.MaxValue)) "+7fffffff"
    test "test3062" (lazy(sprintf "%-+05x" System.Int32.MaxValue)) "+7fffffff"
    test "test3063" (lazy(sprintf "%-+01x" System.Int32.MaxValue)) "+7fffffff"
    test "test3064" (lazy(sprintf "%-+0*x" 7 System.Int32.MaxValue)) "+7fffffff"
    test "test3065" (lazy(sprintf "% x" System.Int32.MaxValue)) " 7fffffff"
    test "test3066" (lazy(sprintf "% 5x" System.Int32.MaxValue)) " 7fffffff"
    test "test3067" (lazy(sprintf "% 1x" System.Int32.MaxValue)) " 7fffffff"
    test "test3068" (lazy(sprintf "% *x" 7 System.Int32.MaxValue)) " 7fffffff"
    test "test3069" (lazy(sprintf "%- x" System.Int32.MaxValue)) " 7fffffff"
    test "test3070" (lazy(sprintf "%- 5x" System.Int32.MaxValue)) " 7fffffff"
    test "test3071" (lazy(sprintf "%- 1x" System.Int32.MaxValue)) " 7fffffff"
    test "test3072" (lazy(sprintf "%- *x" 7 System.Int32.MaxValue)) " 7fffffff"
    test "test3073" (lazy(sprintf "% 0x" System.Int32.MaxValue)) " 7fffffff"
    test "test3074" (lazy(sprintf "% 05x" System.Int32.MaxValue)) " 7fffffff"
    test "test3075" (lazy(sprintf "% 01x" System.Int32.MaxValue)) " 7fffffff"
    test "test3076" (lazy(sprintf "% 0*x" 7 System.Int32.MaxValue)) " 7fffffff"
    test "test3077" (lazy(sprintf "%- 0x" System.Int32.MaxValue)) " 7fffffff"
    test "test3078" (lazy(sprintf "%- 05x" System.Int32.MaxValue)) " 7fffffff"
    test "test3079" (lazy(sprintf "%- 01x" System.Int32.MaxValue)) " 7fffffff"
    test "test3080" (lazy(sprintf "%- 0*x" 7 System.Int32.MaxValue)) " 7fffffff"
    test "test3081" (lazy(sprintf "%x" System.Int64.MaxValue)) "7fffffffffffffff"
    test "test3082" (lazy(sprintf "%5x" System.Int64.MaxValue)) "7fffffffffffffff"
    test "test3083" (lazy(sprintf "%1x" System.Int64.MaxValue)) "7fffffffffffffff"
    test "test3084" (lazy(sprintf "%*x" 7 System.Int64.MaxValue)) "7fffffffffffffff"
    test "test3085" (lazy(sprintf "%-x" System.Int64.MaxValue)) "7fffffffffffffff"
    test "test3086" (lazy(sprintf "%-5x" System.Int64.MaxValue)) "7fffffffffffffff"
    test "test3087" (lazy(sprintf "%-1x" System.Int64.MaxValue)) "7fffffffffffffff"
    test "test3088" (lazy(sprintf "%-*x" 7 System.Int64.MaxValue)) "7fffffffffffffff"
    test "test3089" (lazy(sprintf "%0x" System.Int64.MaxValue)) "7fffffffffffffff"
    test "test3090" (lazy(sprintf "%05x" System.Int64.MaxValue)) "7fffffffffffffff"
    test "test3091" (lazy(sprintf "%01x" System.Int64.MaxValue)) "7fffffffffffffff"
    test "test3092" (lazy(sprintf "%0*x" 7 System.Int64.MaxValue)) "7fffffffffffffff"
    test "test3093" (lazy(sprintf "%-0x" System.Int64.MaxValue)) "7fffffffffffffff"
    test "test3094" (lazy(sprintf "%-05x" System.Int64.MaxValue)) "7fffffffffffffff"
    test "test3095" (lazy(sprintf "%-01x" System.Int64.MaxValue)) "7fffffffffffffff"
    test "test3096" (lazy(sprintf "%-0*x" 7 System.Int64.MaxValue)) "7fffffffffffffff"
    test "test3097" (lazy(sprintf "%+x" System.Int64.MaxValue)) "+7fffffffffffffff"
    test "test3098" (lazy(sprintf "%+5x" System.Int64.MaxValue)) "+7fffffffffffffff"
    test "test3099" (lazy(sprintf "%+1x" System.Int64.MaxValue)) "+7fffffffffffffff"
    test "test3100" (lazy(sprintf "%+*x" 7 System.Int64.MaxValue)) "+7fffffffffffffff"
    test "test3101" (lazy(sprintf "%-+x" System.Int64.MaxValue)) "+7fffffffffffffff"
    test "test3102" (lazy(sprintf "%-+5x" System.Int64.MaxValue)) "+7fffffffffffffff"
    test "test3103" (lazy(sprintf "%-+1x" System.Int64.MaxValue)) "+7fffffffffffffff"
    test "test3104" (lazy(sprintf "%-+*x" 7 System.Int64.MaxValue)) "+7fffffffffffffff"
    test "test3105" (lazy(sprintf "%+0x" System.Int64.MaxValue)) "+7fffffffffffffff"
    test "test3106" (lazy(sprintf "%+05x" System.Int64.MaxValue)) "+7fffffffffffffff"
    test "test3107" (lazy(sprintf "%+01x" System.Int64.MaxValue)) "+7fffffffffffffff"
    test "test3108" (lazy(sprintf "%+0*x" 7 System.Int64.MaxValue)) "+7fffffffffffffff"
    test "test3109" (lazy(sprintf "%-+0x" System.Int64.MaxValue)) "+7fffffffffffffff"
    test "test3110" (lazy(sprintf "%-+05x" System.Int64.MaxValue)) "+7fffffffffffffff"
    test "test3111" (lazy(sprintf "%-+01x" System.Int64.MaxValue)) "+7fffffffffffffff"
    test "test3112" (lazy(sprintf "%-+0*x" 7 System.Int64.MaxValue)) "+7fffffffffffffff"
    test "test3113" (lazy(sprintf "% x" System.Int64.MaxValue)) " 7fffffffffffffff"
    test "test3114" (lazy(sprintf "% 5x" System.Int64.MaxValue)) " 7fffffffffffffff"
    test "test3115" (lazy(sprintf "% 1x" System.Int64.MaxValue)) " 7fffffffffffffff"
    test "test3116" (lazy(sprintf "% *x" 7 System.Int64.MaxValue)) " 7fffffffffffffff"
    test "test3117" (lazy(sprintf "%- x" System.Int64.MaxValue)) " 7fffffffffffffff"
    test "test3118" (lazy(sprintf "%- 5x" System.Int64.MaxValue)) " 7fffffffffffffff"
    test "test3119" (lazy(sprintf "%- 1x" System.Int64.MaxValue)) " 7fffffffffffffff"
    test "test3120" (lazy(sprintf "%- *x" 7 System.Int64.MaxValue)) " 7fffffffffffffff"
    test "test3121" (lazy(sprintf "% 0x" System.Int64.MaxValue)) " 7fffffffffffffff"
    test "test3122" (lazy(sprintf "% 05x" System.Int64.MaxValue)) " 7fffffffffffffff"
    test "test3123" (lazy(sprintf "% 01x" System.Int64.MaxValue)) " 7fffffffffffffff"
    test "test3124" (lazy(sprintf "% 0*x" 7 System.Int64.MaxValue)) " 7fffffffffffffff"
    test "test3125" (lazy(sprintf "%- 0x" System.Int64.MaxValue)) " 7fffffffffffffff"
    test "test3126" (lazy(sprintf "%- 05x" System.Int64.MaxValue)) " 7fffffffffffffff"
    test "test3127" (lazy(sprintf "%- 01x" System.Int64.MaxValue)) " 7fffffffffffffff"
    test "test3128" (lazy(sprintf "%- 0*x" 7 System.Int64.MaxValue)) " 7fffffffffffffff"
    test "test3129" (lazy(sprintf "%x" System.Int32.MinValue)) "80000000"
    test "test3130" (lazy(sprintf "%5x" System.Int32.MinValue)) "80000000"
    test "test3131" (lazy(sprintf "%1x" System.Int32.MinValue)) "80000000"
    test "test3132" (lazy(sprintf "%*x" 7 System.Int32.MinValue)) "80000000"
    test "test3133" (lazy(sprintf "%-x" System.Int32.MinValue)) "80000000"
    test "test3134" (lazy(sprintf "%-5x" System.Int32.MinValue)) "80000000"
    test "test3135" (lazy(sprintf "%-1x" System.Int32.MinValue)) "80000000"
    test "test3136" (lazy(sprintf "%-*x" 7 System.Int32.MinValue)) "80000000"
    test "test3137" (lazy(sprintf "%0x" System.Int32.MinValue)) "80000000"
    test "test3138" (lazy(sprintf "%05x" System.Int32.MinValue)) "80000000"
    test "test3139" (lazy(sprintf "%01x" System.Int32.MinValue)) "80000000"
    test "test3140" (lazy(sprintf "%0*x" 7 System.Int32.MinValue)) "80000000"
    test "test3141" (lazy(sprintf "%-0x" System.Int32.MinValue)) "80000000"
    test "test3142" (lazy(sprintf "%-05x" System.Int32.MinValue)) "80000000"
    test "test3143" (lazy(sprintf "%-01x" System.Int32.MinValue)) "80000000"
    test "test3144" (lazy(sprintf "%-0*x" 7 System.Int32.MinValue)) "80000000"
    test "test3145" (lazy(sprintf "%+x" System.Int32.MinValue)) "+80000000"
    test "test3146" (lazy(sprintf "%+5x" System.Int32.MinValue)) "+80000000"
    test "test3147" (lazy(sprintf "%+1x" System.Int32.MinValue)) "+80000000"
    test "test3148" (lazy(sprintf "%+*x" 7 System.Int32.MinValue)) "+80000000"
    test "test3149" (lazy(sprintf "%-+x" System.Int32.MinValue)) "+80000000"
    test "test3150" (lazy(sprintf "%-+5x" System.Int32.MinValue)) "+80000000"
    test "test3151" (lazy(sprintf "%-+1x" System.Int32.MinValue)) "+80000000"
    test "test3152" (lazy(sprintf "%-+*x" 7 System.Int32.MinValue)) "+80000000"
    test "test3153" (lazy(sprintf "%+0x" System.Int32.MinValue)) "+80000000"
    test "test3154" (lazy(sprintf "%+05x" System.Int32.MinValue)) "+80000000"
    test "test3155" (lazy(sprintf "%+01x" System.Int32.MinValue)) "+80000000"
    test "test3156" (lazy(sprintf "%+0*x" 7 System.Int32.MinValue)) "+80000000"
    test "test3157" (lazy(sprintf "%-+0x" System.Int32.MinValue)) "+80000000"
    test "test3158" (lazy(sprintf "%-+05x" System.Int32.MinValue)) "+80000000"
    test "test3159" (lazy(sprintf "%-+01x" System.Int32.MinValue)) "+80000000"
    test "test3160" (lazy(sprintf "%-+0*x" 7 System.Int32.MinValue)) "+80000000"
    test "test3161" (lazy(sprintf "% x" System.Int32.MinValue)) " 80000000"
    test "test3162" (lazy(sprintf "% 5x" System.Int32.MinValue)) " 80000000"
    test "test3163" (lazy(sprintf "% 1x" System.Int32.MinValue)) " 80000000"
    test "test3164" (lazy(sprintf "% *x" 7 System.Int32.MinValue)) " 80000000"
    test "test3165" (lazy(sprintf "%- x" System.Int32.MinValue)) " 80000000"
    test "test3166" (lazy(sprintf "%- 5x" System.Int32.MinValue)) " 80000000"
    test "test3167" (lazy(sprintf "%- 1x" System.Int32.MinValue)) " 80000000"
    test "test3168" (lazy(sprintf "%- *x" 7 System.Int32.MinValue)) " 80000000"
    test "test3169" (lazy(sprintf "% 0x" System.Int32.MinValue)) " 80000000"
    test "test3170" (lazy(sprintf "% 05x" System.Int32.MinValue)) " 80000000"
    test "test3171" (lazy(sprintf "% 01x" System.Int32.MinValue)) " 80000000"
    test "test3172" (lazy(sprintf "% 0*x" 7 System.Int32.MinValue)) " 80000000"
    test "test3173" (lazy(sprintf "%- 0x" System.Int32.MinValue)) " 80000000"
    test "test3174" (lazy(sprintf "%- 05x" System.Int32.MinValue)) " 80000000"
    test "test3175" (lazy(sprintf "%- 01x" System.Int32.MinValue)) " 80000000"
    test "test3176" (lazy(sprintf "%- 0*x" 7 System.Int32.MinValue)) " 80000000"
    test "test3177" (lazy(sprintf "%x" System.Int64.MinValue)) "8000000000000000"
    test "test3178" (lazy(sprintf "%5x" System.Int64.MinValue)) "8000000000000000"
    test "test3179" (lazy(sprintf "%1x" System.Int64.MinValue)) "8000000000000000"
    test "test3180" (lazy(sprintf "%*x" 7 System.Int64.MinValue)) "8000000000000000"
    test "test3181" (lazy(sprintf "%-x" System.Int64.MinValue)) "8000000000000000"
    test "test3182" (lazy(sprintf "%-5x" System.Int64.MinValue)) "8000000000000000"
    test "test3183" (lazy(sprintf "%-1x" System.Int64.MinValue)) "8000000000000000"
    test "test3184" (lazy(sprintf "%-*x" 7 System.Int64.MinValue)) "8000000000000000"
    test "test3185" (lazy(sprintf "%0x" System.Int64.MinValue)) "8000000000000000"
    test "test3186" (lazy(sprintf "%05x" System.Int64.MinValue)) "8000000000000000"
    test "test3187" (lazy(sprintf "%01x" System.Int64.MinValue)) "8000000000000000"
    test "test3188" (lazy(sprintf "%0*x" 7 System.Int64.MinValue)) "8000000000000000"
    test "test3189" (lazy(sprintf "%-0x" System.Int64.MinValue)) "8000000000000000"
    test "test3190" (lazy(sprintf "%-05x" System.Int64.MinValue)) "8000000000000000"
    test "test3191" (lazy(sprintf "%-01x" System.Int64.MinValue)) "8000000000000000"
    test "test3192" (lazy(sprintf "%-0*x" 7 System.Int64.MinValue)) "8000000000000000"
    test "test3193" (lazy(sprintf "%+x" System.Int64.MinValue)) "+8000000000000000"
    test "test3194" (lazy(sprintf "%+5x" System.Int64.MinValue)) "+8000000000000000"
    test "test3195" (lazy(sprintf "%+1x" System.Int64.MinValue)) "+8000000000000000"
    test "test3196" (lazy(sprintf "%+*x" 7 System.Int64.MinValue)) "+8000000000000000"
    test "test3197" (lazy(sprintf "%-+x" System.Int64.MinValue)) "+8000000000000000"
    test "test3198" (lazy(sprintf "%-+5x" System.Int64.MinValue)) "+8000000000000000"
    test "test3199" (lazy(sprintf "%-+1x" System.Int64.MinValue)) "+8000000000000000"
    test "test3200" (lazy(sprintf "%-+*x" 7 System.Int64.MinValue)) "+8000000000000000"
    test "test3201" (lazy(sprintf "%+0x" System.Int64.MinValue)) "+8000000000000000"
    test "test3202" (lazy(sprintf "%+05x" System.Int64.MinValue)) "+8000000000000000"
    test "test3203" (lazy(sprintf "%+01x" System.Int64.MinValue)) "+8000000000000000"
    test "test3204" (lazy(sprintf "%+0*x" 7 System.Int64.MinValue)) "+8000000000000000"
    test "test3205" (lazy(sprintf "%-+0x" System.Int64.MinValue)) "+8000000000000000"
    test "test3206" (lazy(sprintf "%-+05x" System.Int64.MinValue)) "+8000000000000000"
    test "test3207" (lazy(sprintf "%-+01x" System.Int64.MinValue)) "+8000000000000000"
    test "test3208" (lazy(sprintf "%-+0*x" 7 System.Int64.MinValue)) "+8000000000000000"
    test "test3209" (lazy(sprintf "% x" System.Int64.MinValue)) " 8000000000000000"
    test "test3210" (lazy(sprintf "% 5x" System.Int64.MinValue)) " 8000000000000000"
    test "test3211" (lazy(sprintf "% 1x" System.Int64.MinValue)) " 8000000000000000"
    test "test3212" (lazy(sprintf "% *x" 7 System.Int64.MinValue)) " 8000000000000000"
    test "test3213" (lazy(sprintf "%- x" System.Int64.MinValue)) " 8000000000000000"
    test "test3214" (lazy(sprintf "%- 5x" System.Int64.MinValue)) " 8000000000000000"
    test "test3215" (lazy(sprintf "%- 1x" System.Int64.MinValue)) " 8000000000000000"
    test "test3216" (lazy(sprintf "%- *x" 7 System.Int64.MinValue)) " 8000000000000000"
    test "test3217" (lazy(sprintf "% 0x" System.Int64.MinValue)) " 8000000000000000"
    test "test3218" (lazy(sprintf "% 05x" System.Int64.MinValue)) " 8000000000000000"
    test "test3219" (lazy(sprintf "% 01x" System.Int64.MinValue)) " 8000000000000000"
    test "test3220" (lazy(sprintf "% 0*x" 7 System.Int64.MinValue)) " 8000000000000000"
    test "test3221" (lazy(sprintf "%- 0x" System.Int64.MinValue)) " 8000000000000000"
    test "test3222" (lazy(sprintf "%- 05x" System.Int64.MinValue)) " 8000000000000000"
    test "test3223" (lazy(sprintf "%- 01x" System.Int64.MinValue)) " 8000000000000000"
    test "test3224" (lazy(sprintf "%- 0*x" 7 System.Int64.MinValue)) " 8000000000000000"
    test "test3225" (lazy(sprintf "%x" 55n)) "37"
    test "test3226" (lazy(sprintf "%5x" 55n)) "   37"
    test "test3227" (lazy(sprintf "%1x" 55n)) "37"
    test "test3228" (lazy(sprintf "%*x" 7 55n)) "     37"
    test "test3229" (lazy(sprintf "%-x" 55n)) "37"
    test "test3230" (lazy(sprintf "%-5x" 55n)) "37   "
    test "test3231" (lazy(sprintf "%-1x" 55n)) "37"
    test "test3232" (lazy(sprintf "%-*x" 7 55n)) "37     "
    test "test3233" (lazy(sprintf "%0x" 55n)) "37"
    test "test3234" (lazy(sprintf "%05x" 55n)) "00037"
    test "test3235" (lazy(sprintf "%01x" 55n)) "37"
    test "test3236" (lazy(sprintf "%0*x" 7 55n)) "0000037"
    test "test3237" (lazy(sprintf "%-0x" 55n)) "37"
    test "test3238" (lazy(sprintf "%-05x" 55n)) "37   "
    test "test3239" (lazy(sprintf "%-01x" 55n)) "37"
    test "test3240" (lazy(sprintf "%-0*x" 7 55n)) "37     "
    test "test3241" (lazy(sprintf "%+x" 55n)) "+37"
    test "test3242" (lazy(sprintf "%+5x" 55n)) "  +37"
    test "test3243" (lazy(sprintf "%+1x" 55n)) "+37"
    test "test3244" (lazy(sprintf "%+*x" 7 55n)) "    +37"
    test "test3245" (lazy(sprintf "%-+x" 55n)) "+37"
    test "test3246" (lazy(sprintf "%-+5x" 55n)) "+37  "
    test "test3247" (lazy(sprintf "%-+1x" 55n)) "+37"
    test "test3248" (lazy(sprintf "%-+*x" 7 55n)) "+37    "
    test "test3249" (lazy(sprintf "%+0x" 55n)) "+37"
    test "test3250" (lazy(sprintf "%+05x" 55n)) "+0037"
    test "test3251" (lazy(sprintf "%+01x" 55n)) "+37"
    test "test3252" (lazy(sprintf "%+0*x" 7 55n)) "+000037"
    test "test3253" (lazy(sprintf "%-+0x" 55n)) "+37"
    test "test3254" (lazy(sprintf "%-+05x" 55n)) "+37  "
    test "test3255" (lazy(sprintf "%-+01x" 55n)) "+37"
    test "test3256" (lazy(sprintf "%-+0*x" 7 55n)) "+37    "
    test "test3257" (lazy(sprintf "% x" 55n)) " 37"
    test "test3258" (lazy(sprintf "% 5x" 55n)) "   37"
    test "test3259" (lazy(sprintf "% 1x" 55n)) " 37"
    test "test3260" (lazy(sprintf "% *x" 7 55n)) "     37"
    test "test3261" (lazy(sprintf "%- x" 55n)) " 37"
    test "test3262" (lazy(sprintf "%- 5x" 55n)) " 37  "
    test "test3263" (lazy(sprintf "%- 1x" 55n)) " 37"
    test "test3264" (lazy(sprintf "%- *x" 7 55n)) " 37    "
    test "test3265" (lazy(sprintf "% 0x" 55n)) " 37"
    test "test3266" (lazy(sprintf "% 05x" 55n)) " 0037"
    test "test3267" (lazy(sprintf "% 01x" 55n)) " 37"
    test "test3268" (lazy(sprintf "% 0*x" 7 55n)) " 000037"
    test "test3269" (lazy(sprintf "%- 0x" 55n)) " 37"
    test "test3270" (lazy(sprintf "%- 05x" 55n)) " 37  "
    test "test3271" (lazy(sprintf "%- 01x" 55n)) " 37"
    test "test3272" (lazy(sprintf "%- 0*x" 7 55n)) " 37    "
    test "test3273" (lazy(sprintf "%x" 999un)) "3e7"
    test "test3274" (lazy(sprintf "%5x" 999un)) "  3e7"
    test "test3275" (lazy(sprintf "%1x" 999un)) "3e7"
    test "test3276" (lazy(sprintf "%*x" 7 999un)) "    3e7"
    test "test3277" (lazy(sprintf "%-x" 999un)) "3e7"
    test "test3278" (lazy(sprintf "%-5x" 999un)) "3e7  "
    test "test3279" (lazy(sprintf "%-1x" 999un)) "3e7"
    test "test3280" (lazy(sprintf "%-*x" 7 999un)) "3e7    "
    test "test3281" (lazy(sprintf "%0x" 999un)) "3e7"
    test "test3282" (lazy(sprintf "%05x" 999un)) "003e7"
    test "test3283" (lazy(sprintf "%01x" 999un)) "3e7"
    test "test3284" (lazy(sprintf "%0*x" 7 999un)) "00003e7"
    test "test3285" (lazy(sprintf "%-0x" 999un)) "3e7"
    test "test3286" (lazy(sprintf "%-05x" 999un)) "3e7  "
    test "test3287" (lazy(sprintf "%-01x" 999un)) "3e7"
    test "test3288" (lazy(sprintf "%-0*x" 7 999un)) "3e7    "
    test "test3289" (lazy(sprintf "%+x" 999un)) "+3e7"
    test "test3290" (lazy(sprintf "%+5x" 999un)) " +3e7"
    test "test3291" (lazy(sprintf "%+1x" 999un)) "+3e7"
    test "test3292" (lazy(sprintf "%+*x" 7 999un)) "   +3e7"
    test "test3293" (lazy(sprintf "%-+x" 999un)) "+3e7"
    test "test3294" (lazy(sprintf "%-+5x" 999un)) "+3e7 "
    test "test3295" (lazy(sprintf "%-+1x" 999un)) "+3e7"
    test "test3296" (lazy(sprintf "%-+*x" 7 999un)) "+3e7   "
    test "test3297" (lazy(sprintf "%+0x" 999un)) "+3e7"
    test "test3298" (lazy(sprintf "%+05x" 999un)) "+03e7"
    test "test3299" (lazy(sprintf "%+01x" 999un)) "+3e7"
    test "test3300" (lazy(sprintf "%+0*x" 7 999un)) "+0003e7"
    test "test3301" (lazy(sprintf "%-+0x" 999un)) "+3e7"
    test "test3302" (lazy(sprintf "%-+05x" 999un)) "+3e7 "
    test "test3303" (lazy(sprintf "%-+01x" 999un)) "+3e7"
    test "test3304" (lazy(sprintf "%-+0*x" 7 999un)) "+3e7   "
    test "test3305" (lazy(sprintf "% x" 999un)) " 3e7"
    test "test3306" (lazy(sprintf "% 5x" 999un)) "  3e7"
    test "test3307" (lazy(sprintf "% 1x" 999un)) " 3e7"
    test "test3308" (lazy(sprintf "% *x" 7 999un)) "    3e7"
    test "test3309" (lazy(sprintf "%- x" 999un)) " 3e7"
    test "test3310" (lazy(sprintf "%- 5x" 999un)) " 3e7 "
    test "test3311" (lazy(sprintf "%- 1x" 999un)) " 3e7"
    test "test3312" (lazy(sprintf "%- *x" 7 999un)) " 3e7   "
    test "test3313" (lazy(sprintf "% 0x" 999un)) " 3e7"
    test "test3314" (lazy(sprintf "% 05x" 999un)) " 03e7"
    test "test3315" (lazy(sprintf "% 01x" 999un)) " 3e7"
    test "test3316" (lazy(sprintf "% 0*x" 7 999un)) " 0003e7"
    test "test3317" (lazy(sprintf "%- 0x" 999un)) " 3e7"
    test "test3318" (lazy(sprintf "%- 05x" 999un)) " 3e7 "
    test "test3319" (lazy(sprintf "%- 01x" 999un)) " 3e7"
    test "test3320" (lazy(sprintf "%- 0*x" 7 999un)) " 3e7   "
    test "test3321" (lazy(sprintf "%X" 14)) "E"
    test "test3322" (lazy(sprintf "%5X" 14)) "    E"
    test "test3323" (lazy(sprintf "%1X" 14)) "E"
    test "test3324" (lazy(sprintf "%*X" 7 14)) "      E"
    test "test3325" (lazy(sprintf "%-X" 14)) "E"
    test "test3326" (lazy(sprintf "%-5X" 14)) "E    "
    test "test3327" (lazy(sprintf "%-1X" 14)) "E"
    test "test3328" (lazy(sprintf "%-*X" 7 14)) "E      "
    test "test3329" (lazy(sprintf "%0X" 14)) "E"
    test "test3330" (lazy(sprintf "%05X" 14)) "0000E"
    test "test3331" (lazy(sprintf "%01X" 14)) "E"
    test "test3332" (lazy(sprintf "%0*X" 7 14)) "000000E"
    test "test3333" (lazy(sprintf "%-0X" 14)) "E"
    test "test3334" (lazy(sprintf "%-05X" 14)) "E    "
    test "test3335" (lazy(sprintf "%-01X" 14)) "E"
    test "test3336" (lazy(sprintf "%-0*X" 7 14)) "E      "
    test "test3337" (lazy(sprintf "%+X" 14)) "+E"
    test "test3338" (lazy(sprintf "%+5X" 14)) "   +E"
    test "test3339" (lazy(sprintf "%+1X" 14)) "+E"
    test "test3340" (lazy(sprintf "%+*X" 7 14)) "     +E"
    test "test3341" (lazy(sprintf "%-+X" 14)) "+E"
    test "test3342" (lazy(sprintf "%-+5X" 14)) "+E   "
    test "test3343" (lazy(sprintf "%-+1X" 14)) "+E"
    test "test3344" (lazy(sprintf "%-+*X" 7 14)) "+E     "
    test "test3345" (lazy(sprintf "%+0X" 14)) "+E"
    test "test3346" (lazy(sprintf "%+05X" 14)) "+000E"
    test "test3347" (lazy(sprintf "%+01X" 14)) "+E"
    test "test3348" (lazy(sprintf "%+0*X" 7 14)) "+00000E"
    test "test3349" (lazy(sprintf "%-+0X" 14)) "+E"
    test "test3350" (lazy(sprintf "%-+05X" 14)) "+E   "
    test "test3351" (lazy(sprintf "%-+01X" 14)) "+E"
    test "test3352" (lazy(sprintf "%-+0*X" 7 14)) "+E     "
    test "test3353" (lazy(sprintf "% X" 14)) " E"
    test "test3354" (lazy(sprintf "% 5X" 14)) "    E"
    test "test3355" (lazy(sprintf "% 1X" 14)) " E"
    test "test3356" (lazy(sprintf "% *X" 7 14)) "      E"
    test "test3357" (lazy(sprintf "%- X" 14)) " E"
    test "test3358" (lazy(sprintf "%- 5X" 14)) " E   "
    test "test3359" (lazy(sprintf "%- 1X" 14)) " E"
    test "test3360" (lazy(sprintf "%- *X" 7 14)) " E     "
    test "test3361" (lazy(sprintf "% 0X" 14)) " E"
    test "test3362" (lazy(sprintf "% 05X" 14)) " 000E"
    test "test3363" (lazy(sprintf "% 01X" 14)) " E"
    test "test3364" (lazy(sprintf "% 0*X" 7 14)) " 00000E"
    test "test3365" (lazy(sprintf "%- 0X" 14)) " E"
    test "test3366" (lazy(sprintf "%- 05X" 14)) " E   "
    test "test3367" (lazy(sprintf "%- 01X" 14)) " E"
    test "test3368" (lazy(sprintf "%- 0*X" 7 14)) " E     "
    test "test3369" (lazy(sprintf "%X" -10)) "FFFFFFF6"
    test "test3370" (lazy(sprintf "%5X" -10)) "FFFFFFF6"
    test "test3371" (lazy(sprintf "%1X" -10)) "FFFFFFF6"
    test "test3372" (lazy(sprintf "%*X" 7 -10)) "FFFFFFF6"
    test "test3373" (lazy(sprintf "%-X" -10)) "FFFFFFF6"
    test "test3374" (lazy(sprintf "%-5X" -10)) "FFFFFFF6"
    test "test3375" (lazy(sprintf "%-1X" -10)) "FFFFFFF6"
    test "test3376" (lazy(sprintf "%-*X" 7 -10)) "FFFFFFF6"
    test "test3377" (lazy(sprintf "%0X" -10)) "FFFFFFF6"
    test "test3378" (lazy(sprintf "%05X" -10)) "FFFFFFF6"
    test "test3379" (lazy(sprintf "%01X" -10)) "FFFFFFF6"
    test "test3380" (lazy(sprintf "%0*X" 7 -10)) "FFFFFFF6"
    test "test3381" (lazy(sprintf "%-0X" -10)) "FFFFFFF6"
    test "test3382" (lazy(sprintf "%-05X" -10)) "FFFFFFF6"
    test "test3383" (lazy(sprintf "%-01X" -10)) "FFFFFFF6"
    test "test3384" (lazy(sprintf "%-0*X" 7 -10)) "FFFFFFF6"
    test "test3385" (lazy(sprintf "%+X" -10)) "+FFFFFFF6"
    test "test3386" (lazy(sprintf "%+5X" -10)) "+FFFFFFF6"
    test "test3387" (lazy(sprintf "%+1X" -10)) "+FFFFFFF6"
    test "test3388" (lazy(sprintf "%+*X" 7 -10)) "+FFFFFFF6"
    test "test3389" (lazy(sprintf "%-+X" -10)) "+FFFFFFF6"
    test "test3390" (lazy(sprintf "%-+5X" -10)) "+FFFFFFF6"
    test "test3391" (lazy(sprintf "%-+1X" -10)) "+FFFFFFF6"
    test "test3392" (lazy(sprintf "%-+*X" 7 -10)) "+FFFFFFF6"
    test "test3393" (lazy(sprintf "%+0X" -10)) "+FFFFFFF6"
    test "test3394" (lazy(sprintf "%+05X" -10)) "+FFFFFFF6"
    test "test3395" (lazy(sprintf "%+01X" -10)) "+FFFFFFF6"
    test "test3396" (lazy(sprintf "%+0*X" 7 -10)) "+FFFFFFF6"
    test "test3397" (lazy(sprintf "%-+0X" -10)) "+FFFFFFF6"
    test "test3398" (lazy(sprintf "%-+05X" -10)) "+FFFFFFF6"
    test "test3399" (lazy(sprintf "%-+01X" -10)) "+FFFFFFF6"
    test "test3400" (lazy(sprintf "%-+0*X" 7 -10)) "+FFFFFFF6"
    test "test3401" (lazy(sprintf "% X" -10)) " FFFFFFF6"
    test "test3402" (lazy(sprintf "% 5X" -10)) " FFFFFFF6"
    test "test3403" (lazy(sprintf "% 1X" -10)) " FFFFFFF6"
    test "test3404" (lazy(sprintf "% *X" 7 -10)) " FFFFFFF6"
    test "test3405" (lazy(sprintf "%- X" -10)) " FFFFFFF6"
    test "test3406" (lazy(sprintf "%- 5X" -10)) " FFFFFFF6"
    test "test3407" (lazy(sprintf "%- 1X" -10)) " FFFFFFF6"
    test "test3408" (lazy(sprintf "%- *X" 7 -10)) " FFFFFFF6"
    test "test3409" (lazy(sprintf "% 0X" -10)) " FFFFFFF6"
    test "test3410" (lazy(sprintf "% 05X" -10)) " FFFFFFF6"
    test "test3411" (lazy(sprintf "% 01X" -10)) " FFFFFFF6"
    test "test3412" (lazy(sprintf "% 0*X" 7 -10)) " FFFFFFF6"
    test "test3413" (lazy(sprintf "%- 0X" -10)) " FFFFFFF6"
    test "test3414" (lazy(sprintf "%- 05X" -10)) " FFFFFFF6"
    test "test3415" (lazy(sprintf "%- 01X" -10)) " FFFFFFF6"
    test "test3416" (lazy(sprintf "%- 0*X" 7 -10)) " FFFFFFF6"
    test "test3417" (lazy(sprintf "%X" 55s)) "37"
    test "test3418" (lazy(sprintf "%5X" 55s)) "   37"
    test "test3419" (lazy(sprintf "%1X" 55s)) "37"
    test "test3420" (lazy(sprintf "%*X" 7 55s)) "     37"
    test "test3421" (lazy(sprintf "%-X" 55s)) "37"
    test "test3422" (lazy(sprintf "%-5X" 55s)) "37   "
    test "test3423" (lazy(sprintf "%-1X" 55s)) "37"
    test "test3424" (lazy(sprintf "%-*X" 7 55s)) "37     "
    test "test3425" (lazy(sprintf "%0X" 55s)) "37"
    test "test3426" (lazy(sprintf "%05X" 55s)) "00037"
    test "test3427" (lazy(sprintf "%01X" 55s)) "37"
    test "test3428" (lazy(sprintf "%0*X" 7 55s)) "0000037"
    test "test3429" (lazy(sprintf "%-0X" 55s)) "37"
    test "test3430" (lazy(sprintf "%-05X" 55s)) "37   "
    test "test3431" (lazy(sprintf "%-01X" 55s)) "37"
    test "test3432" (lazy(sprintf "%-0*X" 7 55s)) "37     "
    test "test3433" (lazy(sprintf "%+X" 55s)) "+37"
    test "test3434" (lazy(sprintf "%+5X" 55s)) "  +37"
    test "test3435" (lazy(sprintf "%+1X" 55s)) "+37"
    test "test3436" (lazy(sprintf "%+*X" 7 55s)) "    +37"
    test "test3437" (lazy(sprintf "%-+X" 55s)) "+37"
    test "test3438" (lazy(sprintf "%-+5X" 55s)) "+37  "
    test "test3439" (lazy(sprintf "%-+1X" 55s)) "+37"
    test "test3440" (lazy(sprintf "%-+*X" 7 55s)) "+37    "
    test "test3441" (lazy(sprintf "%+0X" 55s)) "+37"
    test "test3442" (lazy(sprintf "%+05X" 55s)) "+0037"
    test "test3443" (lazy(sprintf "%+01X" 55s)) "+37"
    test "test3444" (lazy(sprintf "%+0*X" 7 55s)) "+000037"
    test "test3445" (lazy(sprintf "%-+0X" 55s)) "+37"
    test "test3446" (lazy(sprintf "%-+05X" 55s)) "+37  "
    test "test3447" (lazy(sprintf "%-+01X" 55s)) "+37"
    test "test3448" (lazy(sprintf "%-+0*X" 7 55s)) "+37    "
    test "test3449" (lazy(sprintf "% X" 55s)) " 37"
    test "test3450" (lazy(sprintf "% 5X" 55s)) "   37"
    test "test3451" (lazy(sprintf "% 1X" 55s)) " 37"
    test "test3452" (lazy(sprintf "% *X" 7 55s)) "     37"
    test "test3453" (lazy(sprintf "%- X" 55s)) " 37"
    test "test3454" (lazy(sprintf "%- 5X" 55s)) " 37  "
    test "test3455" (lazy(sprintf "%- 1X" 55s)) " 37"
    test "test3456" (lazy(sprintf "%- *X" 7 55s)) " 37    "
    test "test3457" (lazy(sprintf "% 0X" 55s)) " 37"
    test "test3458" (lazy(sprintf "% 05X" 55s)) " 0037"
    test "test3459" (lazy(sprintf "% 01X" 55s)) " 37"
    test "test3460" (lazy(sprintf "% 0*X" 7 55s)) " 000037"
    test "test3461" (lazy(sprintf "%- 0X" 55s)) " 37"
    test "test3462" (lazy(sprintf "%- 05X" 55s)) " 37  "
    test "test3463" (lazy(sprintf "%- 01X" 55s)) " 37"
    test "test3464" (lazy(sprintf "%- 0*X" 7 55s)) " 37    "
    test "test3465" (lazy(sprintf "%X" -88s)) "FFA8"
    test "test3466" (lazy(sprintf "%5X" -88s)) " FFA8"
    test "test3467" (lazy(sprintf "%1X" -88s)) "FFA8"
    test "test3468" (lazy(sprintf "%*X" 7 -88s)) "   FFA8"
    test "test3469" (lazy(sprintf "%-X" -88s)) "FFA8"
    test "test3470" (lazy(sprintf "%-5X" -88s)) "FFA8 "
    test "test3471" (lazy(sprintf "%-1X" -88s)) "FFA8"
    test "test3472" (lazy(sprintf "%-*X" 7 -88s)) "FFA8   "
    test "test3473" (lazy(sprintf "%0X" -88s)) "FFA8"
    test "test3474" (lazy(sprintf "%05X" -88s)) "0FFA8"
    test "test3475" (lazy(sprintf "%01X" -88s)) "FFA8"
    test "test3476" (lazy(sprintf "%0*X" 7 -88s)) "000FFA8"
    test "test3477" (lazy(sprintf "%-0X" -88s)) "FFA8"
    test "test3478" (lazy(sprintf "%-05X" -88s)) "FFA8 "
    test "test3479" (lazy(sprintf "%-01X" -88s)) "FFA8"
    test "test3480" (lazy(sprintf "%-0*X" 7 -88s)) "FFA8   "
    test "test3481" (lazy(sprintf "%+X" -88s)) "+FFA8"
    test "test3482" (lazy(sprintf "%+5X" -88s)) "+FFA8"
    test "test3483" (lazy(sprintf "%+1X" -88s)) "+FFA8"
    test "test3484" (lazy(sprintf "%+*X" 7 -88s)) "  +FFA8"
    test "test3485" (lazy(sprintf "%-+X" -88s)) "+FFA8"
    test "test3486" (lazy(sprintf "%-+5X" -88s)) "+FFA8"
    test "test3487" (lazy(sprintf "%-+1X" -88s)) "+FFA8"
    test "test3488" (lazy(sprintf "%-+*X" 7 -88s)) "+FFA8  "
    test "test3489" (lazy(sprintf "%+0X" -88s)) "+FFA8"
    test "test3490" (lazy(sprintf "%+05X" -88s)) "+FFA8"
    test "test3491" (lazy(sprintf "%+01X" -88s)) "+FFA8"
    test "test3492" (lazy(sprintf "%+0*X" 7 -88s)) "+00FFA8"
    test "test3493" (lazy(sprintf "%-+0X" -88s)) "+FFA8"
    test "test3494" (lazy(sprintf "%-+05X" -88s)) "+FFA8"
    test "test3495" (lazy(sprintf "%-+01X" -88s)) "+FFA8"
    test "test3496" (lazy(sprintf "%-+0*X" 7 -88s)) "+FFA8  "
    test "test3497" (lazy(sprintf "% X" -88s)) " FFA8"
    test "test3498" (lazy(sprintf "% 5X" -88s)) " FFA8"
    test "test3499" (lazy(sprintf "% 1X" -88s)) " FFA8"
    test "test3500" (lazy(sprintf "% *X" 7 -88s)) "   FFA8"
    test "test3501" (lazy(sprintf "%- X" -88s)) " FFA8"
    test "test3502" (lazy(sprintf "%- 5X" -88s)) " FFA8"
    test "test3503" (lazy(sprintf "%- 1X" -88s)) " FFA8"
    test "test3504" (lazy(sprintf "%- *X" 7 -88s)) " FFA8  "
    test "test3505" (lazy(sprintf "% 0X" -88s)) " FFA8"
    test "test3506" (lazy(sprintf "% 05X" -88s)) " FFA8"
    test "test3507" (lazy(sprintf "% 01X" -88s)) " FFA8"
    test "test3508" (lazy(sprintf "% 0*X" 7 -88s)) " 00FFA8"
    test "test3509" (lazy(sprintf "%- 0X" -88s)) " FFA8"
    test "test3510" (lazy(sprintf "%- 05X" -88s)) " FFA8"
    test "test3511" (lazy(sprintf "%- 01X" -88s)) " FFA8"
    test "test3512" (lazy(sprintf "%- 0*X" 7 -88s)) " FFA8  "
    test "test3513" (lazy(sprintf "%X" 104us)) "68"
    test "test3514" (lazy(sprintf "%5X" 104us)) "   68"
    test "test3515" (lazy(sprintf "%1X" 104us)) "68"
    test "test3516" (lazy(sprintf "%*X" 7 104us)) "     68"
    test "test3517" (lazy(sprintf "%-X" 104us)) "68"
    test "test3518" (lazy(sprintf "%-5X" 104us)) "68   "
    test "test3519" (lazy(sprintf "%-1X" 104us)) "68"
    test "test3520" (lazy(sprintf "%-*X" 7 104us)) "68     "
    test "test3521" (lazy(sprintf "%0X" 104us)) "68"
    test "test3522" (lazy(sprintf "%05X" 104us)) "00068"
    test "test3523" (lazy(sprintf "%01X" 104us)) "68"
    test "test3524" (lazy(sprintf "%0*X" 7 104us)) "0000068"
    test "test3525" (lazy(sprintf "%-0X" 104us)) "68"
    test "test3526" (lazy(sprintf "%-05X" 104us)) "68   "
    test "test3527" (lazy(sprintf "%-01X" 104us)) "68"
    test "test3528" (lazy(sprintf "%-0*X" 7 104us)) "68     "
    test "test3529" (lazy(sprintf "%+X" 104us)) "+68"
    test "test3530" (lazy(sprintf "%+5X" 104us)) "  +68"
    test "test3531" (lazy(sprintf "%+1X" 104us)) "+68"
    test "test3532" (lazy(sprintf "%+*X" 7 104us)) "    +68"
    test "test3533" (lazy(sprintf "%-+X" 104us)) "+68"
    test "test3534" (lazy(sprintf "%-+5X" 104us)) "+68  "
    test "test3535" (lazy(sprintf "%-+1X" 104us)) "+68"
    test "test3536" (lazy(sprintf "%-+*X" 7 104us)) "+68    "
    test "test3537" (lazy(sprintf "%+0X" 104us)) "+68"
    test "test3538" (lazy(sprintf "%+05X" 104us)) "+0068"
    test "test3539" (lazy(sprintf "%+01X" 104us)) "+68"
    test "test3540" (lazy(sprintf "%+0*X" 7 104us)) "+000068"
    test "test3541" (lazy(sprintf "%-+0X" 104us)) "+68"
    test "test3542" (lazy(sprintf "%-+05X" 104us)) "+68  "
    test "test3543" (lazy(sprintf "%-+01X" 104us)) "+68"
    test "test3544" (lazy(sprintf "%-+0*X" 7 104us)) "+68    "
    test "test3545" (lazy(sprintf "% X" 104us)) " 68"
    test "test3546" (lazy(sprintf "% 5X" 104us)) "   68"
    test "test3547" (lazy(sprintf "% 1X" 104us)) " 68"
    test "test3548" (lazy(sprintf "% *X" 7 104us)) "     68"
    test "test3549" (lazy(sprintf "%- X" 104us)) " 68"
    test "test3550" (lazy(sprintf "%- 5X" 104us)) " 68  "
    test "test3551" (lazy(sprintf "%- 1X" 104us)) " 68"
    test "test3552" (lazy(sprintf "%- *X" 7 104us)) " 68    "
    test "test3553" (lazy(sprintf "% 0X" 104us)) " 68"
    test "test3554" (lazy(sprintf "% 05X" 104us)) " 0068"
    test "test3555" (lazy(sprintf "% 01X" 104us)) " 68"
    test "test3556" (lazy(sprintf "% 0*X" 7 104us)) " 000068"
    test "test3557" (lazy(sprintf "%- 0X" 104us)) " 68"
    test "test3558" (lazy(sprintf "%- 05X" 104us)) " 68  "
    test "test3559" (lazy(sprintf "%- 01X" 104us)) " 68"
    test "test3560" (lazy(sprintf "%- 0*X" 7 104us)) " 68    "
    test "test3561" (lazy(sprintf "%X" 12y)) "C"
    test "test3562" (lazy(sprintf "%5X" 12y)) "    C"
    test "test3563" (lazy(sprintf "%1X" 12y)) "C"
    test "test3564" (lazy(sprintf "%*X" 7 12y)) "      C"
    test "test3565" (lazy(sprintf "%-X" 12y)) "C"
    test "test3566" (lazy(sprintf "%-5X" 12y)) "C    "
    test "test3567" (lazy(sprintf "%-1X" 12y)) "C"
    test "test3568" (lazy(sprintf "%-*X" 7 12y)) "C      "
    test "test3569" (lazy(sprintf "%0X" 12y)) "C"
    test "test3570" (lazy(sprintf "%05X" 12y)) "0000C"
    test "test3571" (lazy(sprintf "%01X" 12y)) "C"
    test "test3572" (lazy(sprintf "%0*X" 7 12y)) "000000C"
    test "test3573" (lazy(sprintf "%-0X" 12y)) "C"
    test "test3574" (lazy(sprintf "%-05X" 12y)) "C    "
    test "test3575" (lazy(sprintf "%-01X" 12y)) "C"
    test "test3576" (lazy(sprintf "%-0*X" 7 12y)) "C      "
    test "test3577" (lazy(sprintf "%+X" 12y)) "+C"
    test "test3578" (lazy(sprintf "%+5X" 12y)) "   +C"
    test "test3579" (lazy(sprintf "%+1X" 12y)) "+C"
    test "test3580" (lazy(sprintf "%+*X" 7 12y)) "     +C"
    test "test3581" (lazy(sprintf "%-+X" 12y)) "+C"
    test "test3582" (lazy(sprintf "%-+5X" 12y)) "+C   "
    test "test3583" (lazy(sprintf "%-+1X" 12y)) "+C"
    test "test3584" (lazy(sprintf "%-+*X" 7 12y)) "+C     "
    test "test3585" (lazy(sprintf "%+0X" 12y)) "+C"
    test "test3586" (lazy(sprintf "%+05X" 12y)) "+000C"
    test "test3587" (lazy(sprintf "%+01X" 12y)) "+C"
    test "test3588" (lazy(sprintf "%+0*X" 7 12y)) "+00000C"
    test "test3589" (lazy(sprintf "%-+0X" 12y)) "+C"
    test "test3590" (lazy(sprintf "%-+05X" 12y)) "+C   "
    test "test3591" (lazy(sprintf "%-+01X" 12y)) "+C"
    test "test3592" (lazy(sprintf "%-+0*X" 7 12y)) "+C     "
    test "test3593" (lazy(sprintf "% X" 12y)) " C"
    test "test3594" (lazy(sprintf "% 5X" 12y)) "    C"
    test "test3595" (lazy(sprintf "% 1X" 12y)) " C"
    test "test3596" (lazy(sprintf "% *X" 7 12y)) "      C"
    test "test3597" (lazy(sprintf "%- X" 12y)) " C"
    test "test3598" (lazy(sprintf "%- 5X" 12y)) " C   "
    test "test3599" (lazy(sprintf "%- 1X" 12y)) " C"
    test "test3600" (lazy(sprintf "%- *X" 7 12y)) " C     "
    test "test3601" (lazy(sprintf "% 0X" 12y)) " C"
    test "test3602" (lazy(sprintf "% 05X" 12y)) " 000C"
    test "test3603" (lazy(sprintf "% 01X" 12y)) " C"
    test "test3604" (lazy(sprintf "% 0*X" 7 12y)) " 00000C"
    test "test3605" (lazy(sprintf "%- 0X" 12y)) " C"
    test "test3606" (lazy(sprintf "%- 05X" 12y)) " C   "
    test "test3607" (lazy(sprintf "%- 01X" 12y)) " C"
    test "test3608" (lazy(sprintf "%- 0*X" 7 12y)) " C     "
    test "test3609" (lazy(sprintf "%X" -94y)) "A2"
    test "test3610" (lazy(sprintf "%5X" -94y)) "   A2"
    test "test3611" (lazy(sprintf "%1X" -94y)) "A2"
    test "test3612" (lazy(sprintf "%*X" 7 -94y)) "     A2"
    test "test3613" (lazy(sprintf "%-X" -94y)) "A2"
    test "test3614" (lazy(sprintf "%-5X" -94y)) "A2   "
    test "test3615" (lazy(sprintf "%-1X" -94y)) "A2"
    test "test3616" (lazy(sprintf "%-*X" 7 -94y)) "A2     "
    test "test3617" (lazy(sprintf "%0X" -94y)) "A2"
    test "test3618" (lazy(sprintf "%05X" -94y)) "000A2"
    test "test3619" (lazy(sprintf "%01X" -94y)) "A2"
    test "test3620" (lazy(sprintf "%0*X" 7 -94y)) "00000A2"
    test "test3621" (lazy(sprintf "%-0X" -94y)) "A2"
    test "test3622" (lazy(sprintf "%-05X" -94y)) "A2   "
    test "test3623" (lazy(sprintf "%-01X" -94y)) "A2"
    test "test3624" (lazy(sprintf "%-0*X" 7 -94y)) "A2     "
    test "test3625" (lazy(sprintf "%+X" -94y)) "+A2"
    test "test3626" (lazy(sprintf "%+5X" -94y)) "  +A2"
    test "test3627" (lazy(sprintf "%+1X" -94y)) "+A2"
    test "test3628" (lazy(sprintf "%+*X" 7 -94y)) "    +A2"
    test "test3629" (lazy(sprintf "%-+X" -94y)) "+A2"
    test "test3630" (lazy(sprintf "%-+5X" -94y)) "+A2  "
    test "test3631" (lazy(sprintf "%-+1X" -94y)) "+A2"
    test "test3632" (lazy(sprintf "%-+*X" 7 -94y)) "+A2    "
    test "test3633" (lazy(sprintf "%+0X" -94y)) "+A2"
    test "test3634" (lazy(sprintf "%+05X" -94y)) "+00A2"
    test "test3635" (lazy(sprintf "%+01X" -94y)) "+A2"
    test "test3636" (lazy(sprintf "%+0*X" 7 -94y)) "+0000A2"
    test "test3637" (lazy(sprintf "%-+0X" -94y)) "+A2"
    test "test3638" (lazy(sprintf "%-+05X" -94y)) "+A2  "
    test "test3639" (lazy(sprintf "%-+01X" -94y)) "+A2"
    test "test3640" (lazy(sprintf "%-+0*X" 7 -94y)) "+A2    "
    test "test3641" (lazy(sprintf "% X" -94y)) " A2"
    test "test3642" (lazy(sprintf "% 5X" -94y)) "   A2"
    test "test3643" (lazy(sprintf "% 1X" -94y)) " A2"
    test "test3644" (lazy(sprintf "% *X" 7 -94y)) "     A2"
    test "test3645" (lazy(sprintf "%- X" -94y)) " A2"
    test "test3646" (lazy(sprintf "%- 5X" -94y)) " A2  "
    test "test3647" (lazy(sprintf "%- 1X" -94y)) " A2"
    test "test3648" (lazy(sprintf "%- *X" 7 -94y)) " A2    "
    test "test3649" (lazy(sprintf "% 0X" -94y)) " A2"
    test "test3650" (lazy(sprintf "% 05X" -94y)) " 00A2"
    test "test3651" (lazy(sprintf "% 01X" -94y)) " A2"
    test "test3652" (lazy(sprintf "% 0*X" 7 -94y)) " 0000A2"
    test "test3653" (lazy(sprintf "%- 0X" -94y)) " A2"
    test "test3654" (lazy(sprintf "%- 05X" -94y)) " A2  "
    test "test3655" (lazy(sprintf "%- 01X" -94y)) " A2"
    test "test3656" (lazy(sprintf "%- 0*X" 7 -94y)) " A2    "
    test "test3657" (lazy(sprintf "%X" 88uy)) "58"
    test "test3658" (lazy(sprintf "%5X" 88uy)) "   58"
    test "test3659" (lazy(sprintf "%1X" 88uy)) "58"
    test "test3660" (lazy(sprintf "%*X" 7 88uy)) "     58"
    test "test3661" (lazy(sprintf "%-X" 88uy)) "58"
    test "test3662" (lazy(sprintf "%-5X" 88uy)) "58   "
    test "test3663" (lazy(sprintf "%-1X" 88uy)) "58"
    test "test3664" (lazy(sprintf "%-*X" 7 88uy)) "58     "
    test "test3665" (lazy(sprintf "%0X" 88uy)) "58"
    test "test3666" (lazy(sprintf "%05X" 88uy)) "00058"
    test "test3667" (lazy(sprintf "%01X" 88uy)) "58"
    test "test3668" (lazy(sprintf "%0*X" 7 88uy)) "0000058"
    test "test3669" (lazy(sprintf "%-0X" 88uy)) "58"
    test "test3670" (lazy(sprintf "%-05X" 88uy)) "58   "
    test "test3671" (lazy(sprintf "%-01X" 88uy)) "58"
    test "test3672" (lazy(sprintf "%-0*X" 7 88uy)) "58     "
    test "test3673" (lazy(sprintf "%+X" 88uy)) "+58"
    test "test3674" (lazy(sprintf "%+5X" 88uy)) "  +58"
    test "test3675" (lazy(sprintf "%+1X" 88uy)) "+58"
    test "test3676" (lazy(sprintf "%+*X" 7 88uy)) "    +58"
    test "test3677" (lazy(sprintf "%-+X" 88uy)) "+58"
    test "test3678" (lazy(sprintf "%-+5X" 88uy)) "+58  "
    test "test3679" (lazy(sprintf "%-+1X" 88uy)) "+58"
    test "test3680" (lazy(sprintf "%-+*X" 7 88uy)) "+58    "
    test "test3681" (lazy(sprintf "%+0X" 88uy)) "+58"
    test "test3682" (lazy(sprintf "%+05X" 88uy)) "+0058"
    test "test3683" (lazy(sprintf "%+01X" 88uy)) "+58"
    test "test3684" (lazy(sprintf "%+0*X" 7 88uy)) "+000058"
    test "test3685" (lazy(sprintf "%-+0X" 88uy)) "+58"
    test "test3686" (lazy(sprintf "%-+05X" 88uy)) "+58  "
    test "test3687" (lazy(sprintf "%-+01X" 88uy)) "+58"
    test "test3688" (lazy(sprintf "%-+0*X" 7 88uy)) "+58    "
    test "test3689" (lazy(sprintf "% X" 88uy)) " 58"
    test "test3690" (lazy(sprintf "% 5X" 88uy)) "   58"
    test "test3691" (lazy(sprintf "% 1X" 88uy)) " 58"
    test "test3692" (lazy(sprintf "% *X" 7 88uy)) "     58"
    test "test3693" (lazy(sprintf "%- X" 88uy)) " 58"
    test "test3694" (lazy(sprintf "%- 5X" 88uy)) " 58  "
    test "test3695" (lazy(sprintf "%- 1X" 88uy)) " 58"
    test "test3696" (lazy(sprintf "%- *X" 7 88uy)) " 58    "
    test "test3697" (lazy(sprintf "% 0X" 88uy)) " 58"
    test "test3698" (lazy(sprintf "% 05X" 88uy)) " 0058"
    test "test3699" (lazy(sprintf "% 01X" 88uy)) " 58"
    test "test3700" (lazy(sprintf "% 0*X" 7 88uy)) " 000058"
    test "test3701" (lazy(sprintf "%- 0X" 88uy)) " 58"
    test "test3702" (lazy(sprintf "%- 05X" 88uy)) " 58  "
    test "test3703" (lazy(sprintf "%- 01X" 88uy)) " 58"
    test "test3704" (lazy(sprintf "%- 0*X" 7 88uy)) " 58    "
    test "test3705" (lazy(sprintf "%X" 999L)) "3E7"
    test "test3706" (lazy(sprintf "%5X" 999L)) "  3E7"
    test "test3707" (lazy(sprintf "%1X" 999L)) "3E7"
    test "test3708" (lazy(sprintf "%*X" 7 999L)) "    3E7"
    test "test3709" (lazy(sprintf "%-X" 999L)) "3E7"
    test "test3710" (lazy(sprintf "%-5X" 999L)) "3E7  "
    test "test3711" (lazy(sprintf "%-1X" 999L)) "3E7"
    test "test3712" (lazy(sprintf "%-*X" 7 999L)) "3E7    "
    test "test3713" (lazy(sprintf "%0X" 999L)) "3E7"
    test "test3714" (lazy(sprintf "%05X" 999L)) "003E7"
    test "test3715" (lazy(sprintf "%01X" 999L)) "3E7"
    test "test3716" (lazy(sprintf "%0*X" 7 999L)) "00003E7"
    test "test3717" (lazy(sprintf "%-0X" 999L)) "3E7"
    test "test3718" (lazy(sprintf "%-05X" 999L)) "3E7  "
    test "test3719" (lazy(sprintf "%-01X" 999L)) "3E7"
    test "test3720" (lazy(sprintf "%-0*X" 7 999L)) "3E7    "
    test "test3721" (lazy(sprintf "%+X" 999L)) "+3E7"
    test "test3722" (lazy(sprintf "%+5X" 999L)) " +3E7"
    test "test3723" (lazy(sprintf "%+1X" 999L)) "+3E7"
    test "test3724" (lazy(sprintf "%+*X" 7 999L)) "   +3E7"
    test "test3725" (lazy(sprintf "%-+X" 999L)) "+3E7"
    test "test3726" (lazy(sprintf "%-+5X" 999L)) "+3E7 "
    test "test3727" (lazy(sprintf "%-+1X" 999L)) "+3E7"
    test "test3728" (lazy(sprintf "%-+*X" 7 999L)) "+3E7   "
    test "test3729" (lazy(sprintf "%+0X" 999L)) "+3E7"
    test "test3730" (lazy(sprintf "%+05X" 999L)) "+03E7"
    test "test3731" (lazy(sprintf "%+01X" 999L)) "+3E7"
    test "test3732" (lazy(sprintf "%+0*X" 7 999L)) "+0003E7"
    test "test3733" (lazy(sprintf "%-+0X" 999L)) "+3E7"
    test "test3734" (lazy(sprintf "%-+05X" 999L)) "+3E7 "
    test "test3735" (lazy(sprintf "%-+01X" 999L)) "+3E7"
    test "test3736" (lazy(sprintf "%-+0*X" 7 999L)) "+3E7   "
    test "test3737" (lazy(sprintf "% X" 999L)) " 3E7"
    test "test3738" (lazy(sprintf "% 5X" 999L)) "  3E7"
    test "test3739" (lazy(sprintf "% 1X" 999L)) " 3E7"
    test "test3740" (lazy(sprintf "% *X" 7 999L)) "    3E7"
    test "test3741" (lazy(sprintf "%- X" 999L)) " 3E7"
    test "test3742" (lazy(sprintf "%- 5X" 999L)) " 3E7 "
    test "test3743" (lazy(sprintf "%- 1X" 999L)) " 3E7"
    test "test3744" (lazy(sprintf "%- *X" 7 999L)) " 3E7   "
    test "test3745" (lazy(sprintf "% 0X" 999L)) " 3E7"
    test "test3746" (lazy(sprintf "% 05X" 999L)) " 03E7"
    test "test3747" (lazy(sprintf "% 01X" 999L)) " 3E7"
    test "test3748" (lazy(sprintf "% 0*X" 7 999L)) " 0003E7"
    test "test3749" (lazy(sprintf "%- 0X" 999L)) " 3E7"
    test "test3750" (lazy(sprintf "%- 05X" 999L)) " 3E7 "
    test "test3751" (lazy(sprintf "%- 01X" 999L)) " 3E7"
    test "test3752" (lazy(sprintf "%- 0*X" 7 999L)) " 3E7   "
    test "test3753" (lazy(sprintf "%X" -321L)) "FFFFFFFFFFFFFEBF"
    test "test3754" (lazy(sprintf "%5X" -321L)) "FFFFFFFFFFFFFEBF"
    test "test3755" (lazy(sprintf "%1X" -321L)) "FFFFFFFFFFFFFEBF"
    test "test3756" (lazy(sprintf "%*X" 7 -321L)) "FFFFFFFFFFFFFEBF"
    test "test3757" (lazy(sprintf "%-X" -321L)) "FFFFFFFFFFFFFEBF"
    test "test3758" (lazy(sprintf "%-5X" -321L)) "FFFFFFFFFFFFFEBF"
    test "test3759" (lazy(sprintf "%-1X" -321L)) "FFFFFFFFFFFFFEBF"
    test "test3760" (lazy(sprintf "%-*X" 7 -321L)) "FFFFFFFFFFFFFEBF"
    test "test3761" (lazy(sprintf "%0X" -321L)) "FFFFFFFFFFFFFEBF"
    test "test3762" (lazy(sprintf "%05X" -321L)) "FFFFFFFFFFFFFEBF"
    test "test3763" (lazy(sprintf "%01X" -321L)) "FFFFFFFFFFFFFEBF"
    test "test3764" (lazy(sprintf "%0*X" 7 -321L)) "FFFFFFFFFFFFFEBF"
    test "test3765" (lazy(sprintf "%-0X" -321L)) "FFFFFFFFFFFFFEBF"
    test "test3766" (lazy(sprintf "%-05X" -321L)) "FFFFFFFFFFFFFEBF"
    test "test3767" (lazy(sprintf "%-01X" -321L)) "FFFFFFFFFFFFFEBF"
    test "test3768" (lazy(sprintf "%-0*X" 7 -321L)) "FFFFFFFFFFFFFEBF"
    test "test3769" (lazy(sprintf "%+X" -321L)) "+FFFFFFFFFFFFFEBF"
    test "test3770" (lazy(sprintf "%+5X" -321L)) "+FFFFFFFFFFFFFEBF"
    test "test3771" (lazy(sprintf "%+1X" -321L)) "+FFFFFFFFFFFFFEBF"
    test "test3772" (lazy(sprintf "%+*X" 7 -321L)) "+FFFFFFFFFFFFFEBF"
    test "test3773" (lazy(sprintf "%-+X" -321L)) "+FFFFFFFFFFFFFEBF"
    test "test3774" (lazy(sprintf "%-+5X" -321L)) "+FFFFFFFFFFFFFEBF"
    test "test3775" (lazy(sprintf "%-+1X" -321L)) "+FFFFFFFFFFFFFEBF"
    test "test3776" (lazy(sprintf "%-+*X" 7 -321L)) "+FFFFFFFFFFFFFEBF"
    test "test3777" (lazy(sprintf "%+0X" -321L)) "+FFFFFFFFFFFFFEBF"
    test "test3778" (lazy(sprintf "%+05X" -321L)) "+FFFFFFFFFFFFFEBF"
    test "test3779" (lazy(sprintf "%+01X" -321L)) "+FFFFFFFFFFFFFEBF"
    test "test3780" (lazy(sprintf "%+0*X" 7 -321L)) "+FFFFFFFFFFFFFEBF"
    test "test3781" (lazy(sprintf "%-+0X" -321L)) "+FFFFFFFFFFFFFEBF"
    test "test3782" (lazy(sprintf "%-+05X" -321L)) "+FFFFFFFFFFFFFEBF"
    test "test3783" (lazy(sprintf "%-+01X" -321L)) "+FFFFFFFFFFFFFEBF"
    test "test3784" (lazy(sprintf "%-+0*X" 7 -321L)) "+FFFFFFFFFFFFFEBF"
    test "test3785" (lazy(sprintf "% X" -321L)) " FFFFFFFFFFFFFEBF"
    test "test3786" (lazy(sprintf "% 5X" -321L)) " FFFFFFFFFFFFFEBF"
    test "test3787" (lazy(sprintf "% 1X" -321L)) " FFFFFFFFFFFFFEBF"
    test "test3788" (lazy(sprintf "% *X" 7 -321L)) " FFFFFFFFFFFFFEBF"
    test "test3789" (lazy(sprintf "%- X" -321L)) " FFFFFFFFFFFFFEBF"
    test "test3790" (lazy(sprintf "%- 5X" -321L)) " FFFFFFFFFFFFFEBF"
    test "test3791" (lazy(sprintf "%- 1X" -321L)) " FFFFFFFFFFFFFEBF"
    test "test3792" (lazy(sprintf "%- *X" 7 -321L)) " FFFFFFFFFFFFFEBF"
    test "test3793" (lazy(sprintf "% 0X" -321L)) " FFFFFFFFFFFFFEBF"
    test "test3794" (lazy(sprintf "% 05X" -321L)) " FFFFFFFFFFFFFEBF"
    test "test3795" (lazy(sprintf "% 01X" -321L)) " FFFFFFFFFFFFFEBF"
    test "test3796" (lazy(sprintf "% 0*X" 7 -321L)) " FFFFFFFFFFFFFEBF"
    test "test3797" (lazy(sprintf "%- 0X" -321L)) " FFFFFFFFFFFFFEBF"
    test "test3798" (lazy(sprintf "%- 05X" -321L)) " FFFFFFFFFFFFFEBF"
    test "test3799" (lazy(sprintf "%- 01X" -321L)) " FFFFFFFFFFFFFEBF"
    test "test3800" (lazy(sprintf "%- 0*X" 7 -321L)) " FFFFFFFFFFFFFEBF"
    test "test3801" (lazy(sprintf "%X" 50000UL)) "C350"
    test "test3802" (lazy(sprintf "%5X" 50000UL)) " C350"
    test "test3803" (lazy(sprintf "%1X" 50000UL)) "C350"
    test "test3804" (lazy(sprintf "%*X" 7 50000UL)) "   C350"
    test "test3805" (lazy(sprintf "%-X" 50000UL)) "C350"
    test "test3806" (lazy(sprintf "%-5X" 50000UL)) "C350 "
    test "test3807" (lazy(sprintf "%-1X" 50000UL)) "C350"
    test "test3808" (lazy(sprintf "%-*X" 7 50000UL)) "C350   "
    test "test3809" (lazy(sprintf "%0X" 50000UL)) "C350"
    test "test3810" (lazy(sprintf "%05X" 50000UL)) "0C350"
    test "test3811" (lazy(sprintf "%01X" 50000UL)) "C350"
    test "test3812" (lazy(sprintf "%0*X" 7 50000UL)) "000C350"
    test "test3813" (lazy(sprintf "%-0X" 50000UL)) "C350"
    test "test3814" (lazy(sprintf "%-05X" 50000UL)) "C350 "
    test "test3815" (lazy(sprintf "%-01X" 50000UL)) "C350"
    test "test3816" (lazy(sprintf "%-0*X" 7 50000UL)) "C350   "
    test "test3817" (lazy(sprintf "%+X" 50000UL)) "+C350"
    test "test3818" (lazy(sprintf "%+5X" 50000UL)) "+C350"
    test "test3819" (lazy(sprintf "%+1X" 50000UL)) "+C350"
    test "test3820" (lazy(sprintf "%+*X" 7 50000UL)) "  +C350"
    test "test3821" (lazy(sprintf "%-+X" 50000UL)) "+C350"
    test "test3822" (lazy(sprintf "%-+5X" 50000UL)) "+C350"
    test "test3823" (lazy(sprintf "%-+1X" 50000UL)) "+C350"
    test "test3824" (lazy(sprintf "%-+*X" 7 50000UL)) "+C350  "
    test "test3825" (lazy(sprintf "%+0X" 50000UL)) "+C350"
    test "test3826" (lazy(sprintf "%+05X" 50000UL)) "+C350"
    test "test3827" (lazy(sprintf "%+01X" 50000UL)) "+C350"
    test "test3828" (lazy(sprintf "%+0*X" 7 50000UL)) "+00C350"
    test "test3829" (lazy(sprintf "%-+0X" 50000UL)) "+C350"
    test "test3830" (lazy(sprintf "%-+05X" 50000UL)) "+C350"
    test "test3831" (lazy(sprintf "%-+01X" 50000UL)) "+C350"
    test "test3832" (lazy(sprintf "%-+0*X" 7 50000UL)) "+C350  "
    test "test3833" (lazy(sprintf "% X" 50000UL)) " C350"
    test "test3834" (lazy(sprintf "% 5X" 50000UL)) " C350"
    test "test3835" (lazy(sprintf "% 1X" 50000UL)) " C350"
    test "test3836" (lazy(sprintf "% *X" 7 50000UL)) "   C350"
    test "test3837" (lazy(sprintf "%- X" 50000UL)) " C350"
    test "test3838" (lazy(sprintf "%- 5X" 50000UL)) " C350"
    test "test3839" (lazy(sprintf "%- 1X" 50000UL)) " C350"
    test "test3840" (lazy(sprintf "%- *X" 7 50000UL)) " C350  "
    test "test3841" (lazy(sprintf "% 0X" 50000UL)) " C350"
    test "test3842" (lazy(sprintf "% 05X" 50000UL)) " C350"
    test "test3843" (lazy(sprintf "% 01X" 50000UL)) " C350"
    test "test3844" (lazy(sprintf "% 0*X" 7 50000UL)) " 00C350"
    test "test3845" (lazy(sprintf "%- 0X" 50000UL)) " C350"
    test "test3846" (lazy(sprintf "%- 05X" 50000UL)) " C350"
    test "test3847" (lazy(sprintf "%- 01X" 50000UL)) " C350"
    test "test3848" (lazy(sprintf "%- 0*X" 7 50000UL)) " C350  "
    test "test3849" (lazy(sprintf "%X" System.Int32.MaxValue)) "7FFFFFFF"
    test "test3850" (lazy(sprintf "%5X" System.Int32.MaxValue)) "7FFFFFFF"
    test "test3851" (lazy(sprintf "%1X" System.Int32.MaxValue)) "7FFFFFFF"
    test "test3852" (lazy(sprintf "%*X" 7 System.Int32.MaxValue)) "7FFFFFFF"
    test "test3853" (lazy(sprintf "%-X" System.Int32.MaxValue)) "7FFFFFFF"
    test "test3854" (lazy(sprintf "%-5X" System.Int32.MaxValue)) "7FFFFFFF"
    test "test3855" (lazy(sprintf "%-1X" System.Int32.MaxValue)) "7FFFFFFF"
    test "test3856" (lazy(sprintf "%-*X" 7 System.Int32.MaxValue)) "7FFFFFFF"
    test "test3857" (lazy(sprintf "%0X" System.Int32.MaxValue)) "7FFFFFFF"
    test "test3858" (lazy(sprintf "%05X" System.Int32.MaxValue)) "7FFFFFFF"
    test "test3859" (lazy(sprintf "%01X" System.Int32.MaxValue)) "7FFFFFFF"
    test "test3860" (lazy(sprintf "%0*X" 7 System.Int32.MaxValue)) "7FFFFFFF"
    test "test3861" (lazy(sprintf "%-0X" System.Int32.MaxValue)) "7FFFFFFF"
    test "test3862" (lazy(sprintf "%-05X" System.Int32.MaxValue)) "7FFFFFFF"
    test "test3863" (lazy(sprintf "%-01X" System.Int32.MaxValue)) "7FFFFFFF"
    test "test3864" (lazy(sprintf "%-0*X" 7 System.Int32.MaxValue)) "7FFFFFFF"
    test "test3865" (lazy(sprintf "%+X" System.Int32.MaxValue)) "+7FFFFFFF"
    test "test3866" (lazy(sprintf "%+5X" System.Int32.MaxValue)) "+7FFFFFFF"
    test "test3867" (lazy(sprintf "%+1X" System.Int32.MaxValue)) "+7FFFFFFF"
    test "test3868" (lazy(sprintf "%+*X" 7 System.Int32.MaxValue)) "+7FFFFFFF"
    test "test3869" (lazy(sprintf "%-+X" System.Int32.MaxValue)) "+7FFFFFFF"
    test "test3870" (lazy(sprintf "%-+5X" System.Int32.MaxValue)) "+7FFFFFFF"
    test "test3871" (lazy(sprintf "%-+1X" System.Int32.MaxValue)) "+7FFFFFFF"
    test "test3872" (lazy(sprintf "%-+*X" 7 System.Int32.MaxValue)) "+7FFFFFFF"
    test "test3873" (lazy(sprintf "%+0X" System.Int32.MaxValue)) "+7FFFFFFF"
    test "test3874" (lazy(sprintf "%+05X" System.Int32.MaxValue)) "+7FFFFFFF"
    test "test3875" (lazy(sprintf "%+01X" System.Int32.MaxValue)) "+7FFFFFFF"
    test "test3876" (lazy(sprintf "%+0*X" 7 System.Int32.MaxValue)) "+7FFFFFFF"
    test "test3877" (lazy(sprintf "%-+0X" System.Int32.MaxValue)) "+7FFFFFFF"
    test "test3878" (lazy(sprintf "%-+05X" System.Int32.MaxValue)) "+7FFFFFFF"
    test "test3879" (lazy(sprintf "%-+01X" System.Int32.MaxValue)) "+7FFFFFFF"
    test "test3880" (lazy(sprintf "%-+0*X" 7 System.Int32.MaxValue)) "+7FFFFFFF"
    test "test3881" (lazy(sprintf "% X" System.Int32.MaxValue)) " 7FFFFFFF"
    test "test3882" (lazy(sprintf "% 5X" System.Int32.MaxValue)) " 7FFFFFFF"
    test "test3883" (lazy(sprintf "% 1X" System.Int32.MaxValue)) " 7FFFFFFF"
    test "test3884" (lazy(sprintf "% *X" 7 System.Int32.MaxValue)) " 7FFFFFFF"
    test "test3885" (lazy(sprintf "%- X" System.Int32.MaxValue)) " 7FFFFFFF"
    test "test3886" (lazy(sprintf "%- 5X" System.Int32.MaxValue)) " 7FFFFFFF"
    test "test3887" (lazy(sprintf "%- 1X" System.Int32.MaxValue)) " 7FFFFFFF"
    test "test3888" (lazy(sprintf "%- *X" 7 System.Int32.MaxValue)) " 7FFFFFFF"
    test "test3889" (lazy(sprintf "% 0X" System.Int32.MaxValue)) " 7FFFFFFF"
    test "test3890" (lazy(sprintf "% 05X" System.Int32.MaxValue)) " 7FFFFFFF"
    test "test3891" (lazy(sprintf "% 01X" System.Int32.MaxValue)) " 7FFFFFFF"
    test "test3892" (lazy(sprintf "% 0*X" 7 System.Int32.MaxValue)) " 7FFFFFFF"
    test "test3893" (lazy(sprintf "%- 0X" System.Int32.MaxValue)) " 7FFFFFFF"
    test "test3894" (lazy(sprintf "%- 05X" System.Int32.MaxValue)) " 7FFFFFFF"
    test "test3895" (lazy(sprintf "%- 01X" System.Int32.MaxValue)) " 7FFFFFFF"
    test "test3896" (lazy(sprintf "%- 0*X" 7 System.Int32.MaxValue)) " 7FFFFFFF"
    test "test3897" (lazy(sprintf "%X" System.Int64.MaxValue)) "7FFFFFFFFFFFFFFF"
    test "test3898" (lazy(sprintf "%5X" System.Int64.MaxValue)) "7FFFFFFFFFFFFFFF"
    test "test3899" (lazy(sprintf "%1X" System.Int64.MaxValue)) "7FFFFFFFFFFFFFFF"
    test "test3900" (lazy(sprintf "%*X" 7 System.Int64.MaxValue)) "7FFFFFFFFFFFFFFF"
    test "test3901" (lazy(sprintf "%-X" System.Int64.MaxValue)) "7FFFFFFFFFFFFFFF"
    test "test3902" (lazy(sprintf "%-5X" System.Int64.MaxValue)) "7FFFFFFFFFFFFFFF"
    test "test3903" (lazy(sprintf "%-1X" System.Int64.MaxValue)) "7FFFFFFFFFFFFFFF"
    test "test3904" (lazy(sprintf "%-*X" 7 System.Int64.MaxValue)) "7FFFFFFFFFFFFFFF"
    test "test3905" (lazy(sprintf "%0X" System.Int64.MaxValue)) "7FFFFFFFFFFFFFFF"
    test "test3906" (lazy(sprintf "%05X" System.Int64.MaxValue)) "7FFFFFFFFFFFFFFF"
    test "test3907" (lazy(sprintf "%01X" System.Int64.MaxValue)) "7FFFFFFFFFFFFFFF"
    test "test3908" (lazy(sprintf "%0*X" 7 System.Int64.MaxValue)) "7FFFFFFFFFFFFFFF"
    test "test3909" (lazy(sprintf "%-0X" System.Int64.MaxValue)) "7FFFFFFFFFFFFFFF"
    test "test3910" (lazy(sprintf "%-05X" System.Int64.MaxValue)) "7FFFFFFFFFFFFFFF"
    test "test3911" (lazy(sprintf "%-01X" System.Int64.MaxValue)) "7FFFFFFFFFFFFFFF"
    test "test3912" (lazy(sprintf "%-0*X" 7 System.Int64.MaxValue)) "7FFFFFFFFFFFFFFF"
    test "test3913" (lazy(sprintf "%+X" System.Int64.MaxValue)) "+7FFFFFFFFFFFFFFF"
    test "test3914" (lazy(sprintf "%+5X" System.Int64.MaxValue)) "+7FFFFFFFFFFFFFFF"
    test "test3915" (lazy(sprintf "%+1X" System.Int64.MaxValue)) "+7FFFFFFFFFFFFFFF"
    test "test3916" (lazy(sprintf "%+*X" 7 System.Int64.MaxValue)) "+7FFFFFFFFFFFFFFF"
    test "test3917" (lazy(sprintf "%-+X" System.Int64.MaxValue)) "+7FFFFFFFFFFFFFFF"
    test "test3918" (lazy(sprintf "%-+5X" System.Int64.MaxValue)) "+7FFFFFFFFFFFFFFF"
    test "test3919" (lazy(sprintf "%-+1X" System.Int64.MaxValue)) "+7FFFFFFFFFFFFFFF"
    test "test3920" (lazy(sprintf "%-+*X" 7 System.Int64.MaxValue)) "+7FFFFFFFFFFFFFFF"
    test "test3921" (lazy(sprintf "%+0X" System.Int64.MaxValue)) "+7FFFFFFFFFFFFFFF"
    test "test3922" (lazy(sprintf "%+05X" System.Int64.MaxValue)) "+7FFFFFFFFFFFFFFF"
    test "test3923" (lazy(sprintf "%+01X" System.Int64.MaxValue)) "+7FFFFFFFFFFFFFFF"
    test "test3924" (lazy(sprintf "%+0*X" 7 System.Int64.MaxValue)) "+7FFFFFFFFFFFFFFF"
    test "test3925" (lazy(sprintf "%-+0X" System.Int64.MaxValue)) "+7FFFFFFFFFFFFFFF"
    test "test3926" (lazy(sprintf "%-+05X" System.Int64.MaxValue)) "+7FFFFFFFFFFFFFFF"
    test "test3927" (lazy(sprintf "%-+01X" System.Int64.MaxValue)) "+7FFFFFFFFFFFFFFF"
    test "test3928" (lazy(sprintf "%-+0*X" 7 System.Int64.MaxValue)) "+7FFFFFFFFFFFFFFF"
    test "test3929" (lazy(sprintf "% X" System.Int64.MaxValue)) " 7FFFFFFFFFFFFFFF"
    test "test3930" (lazy(sprintf "% 5X" System.Int64.MaxValue)) " 7FFFFFFFFFFFFFFF"
    test "test3931" (lazy(sprintf "% 1X" System.Int64.MaxValue)) " 7FFFFFFFFFFFFFFF"
    test "test3932" (lazy(sprintf "% *X" 7 System.Int64.MaxValue)) " 7FFFFFFFFFFFFFFF"
    test "test3933" (lazy(sprintf "%- X" System.Int64.MaxValue)) " 7FFFFFFFFFFFFFFF"
    test "test3934" (lazy(sprintf "%- 5X" System.Int64.MaxValue)) " 7FFFFFFFFFFFFFFF"
    test "test3935" (lazy(sprintf "%- 1X" System.Int64.MaxValue)) " 7FFFFFFFFFFFFFFF"
    test "test3936" (lazy(sprintf "%- *X" 7 System.Int64.MaxValue)) " 7FFFFFFFFFFFFFFF"
    test "test3937" (lazy(sprintf "% 0X" System.Int64.MaxValue)) " 7FFFFFFFFFFFFFFF"
    test "test3938" (lazy(sprintf "% 05X" System.Int64.MaxValue)) " 7FFFFFFFFFFFFFFF"
    test "test3939" (lazy(sprintf "% 01X" System.Int64.MaxValue)) " 7FFFFFFFFFFFFFFF"
    test "test3940" (lazy(sprintf "% 0*X" 7 System.Int64.MaxValue)) " 7FFFFFFFFFFFFFFF"
    test "test3941" (lazy(sprintf "%- 0X" System.Int64.MaxValue)) " 7FFFFFFFFFFFFFFF"
    test "test3942" (lazy(sprintf "%- 05X" System.Int64.MaxValue)) " 7FFFFFFFFFFFFFFF"
    test "test3943" (lazy(sprintf "%- 01X" System.Int64.MaxValue)) " 7FFFFFFFFFFFFFFF"
    test "test3944" (lazy(sprintf "%- 0*X" 7 System.Int64.MaxValue)) " 7FFFFFFFFFFFFFFF"
    test "test3945" (lazy(sprintf "%X" System.Int32.MinValue)) "80000000"
    test "test3946" (lazy(sprintf "%5X" System.Int32.MinValue)) "80000000"
    test "test3947" (lazy(sprintf "%1X" System.Int32.MinValue)) "80000000"
    test "test3948" (lazy(sprintf "%*X" 7 System.Int32.MinValue)) "80000000"
    test "test3949" (lazy(sprintf "%-X" System.Int32.MinValue)) "80000000"
    test "test3950" (lazy(sprintf "%-5X" System.Int32.MinValue)) "80000000"
    test "test3951" (lazy(sprintf "%-1X" System.Int32.MinValue)) "80000000"
    test "test3952" (lazy(sprintf "%-*X" 7 System.Int32.MinValue)) "80000000"
    test "test3953" (lazy(sprintf "%0X" System.Int32.MinValue)) "80000000"
    test "test3954" (lazy(sprintf "%05X" System.Int32.MinValue)) "80000000"
    test "test3955" (lazy(sprintf "%01X" System.Int32.MinValue)) "80000000"
    test "test3956" (lazy(sprintf "%0*X" 7 System.Int32.MinValue)) "80000000"
    test "test3957" (lazy(sprintf "%-0X" System.Int32.MinValue)) "80000000"
    test "test3958" (lazy(sprintf "%-05X" System.Int32.MinValue)) "80000000"
    test "test3959" (lazy(sprintf "%-01X" System.Int32.MinValue)) "80000000"
    test "test3960" (lazy(sprintf "%-0*X" 7 System.Int32.MinValue)) "80000000"
    test "test3961" (lazy(sprintf "%+X" System.Int32.MinValue)) "+80000000"
    test "test3962" (lazy(sprintf "%+5X" System.Int32.MinValue)) "+80000000"
    test "test3963" (lazy(sprintf "%+1X" System.Int32.MinValue)) "+80000000"
    test "test3964" (lazy(sprintf "%+*X" 7 System.Int32.MinValue)) "+80000000"
    test "test3965" (lazy(sprintf "%-+X" System.Int32.MinValue)) "+80000000"
    test "test3966" (lazy(sprintf "%-+5X" System.Int32.MinValue)) "+80000000"
    test "test3967" (lazy(sprintf "%-+1X" System.Int32.MinValue)) "+80000000"
    test "test3968" (lazy(sprintf "%-+*X" 7 System.Int32.MinValue)) "+80000000"
    test "test3969" (lazy(sprintf "%+0X" System.Int32.MinValue)) "+80000000"
    test "test3970" (lazy(sprintf "%+05X" System.Int32.MinValue)) "+80000000"
    test "test3971" (lazy(sprintf "%+01X" System.Int32.MinValue)) "+80000000"
    test "test3972" (lazy(sprintf "%+0*X" 7 System.Int32.MinValue)) "+80000000"
    test "test3973" (lazy(sprintf "%-+0X" System.Int32.MinValue)) "+80000000"
    test "test3974" (lazy(sprintf "%-+05X" System.Int32.MinValue)) "+80000000"
    test "test3975" (lazy(sprintf "%-+01X" System.Int32.MinValue)) "+80000000"
    test "test3976" (lazy(sprintf "%-+0*X" 7 System.Int32.MinValue)) "+80000000"
    test "test3977" (lazy(sprintf "% X" System.Int32.MinValue)) " 80000000"
    test "test3978" (lazy(sprintf "% 5X" System.Int32.MinValue)) " 80000000"
    test "test3979" (lazy(sprintf "% 1X" System.Int32.MinValue)) " 80000000"
    test "test3980" (lazy(sprintf "% *X" 7 System.Int32.MinValue)) " 80000000"
    test "test3981" (lazy(sprintf "%- X" System.Int32.MinValue)) " 80000000"
    test "test3982" (lazy(sprintf "%- 5X" System.Int32.MinValue)) " 80000000"
    test "test3983" (lazy(sprintf "%- 1X" System.Int32.MinValue)) " 80000000"
    test "test3984" (lazy(sprintf "%- *X" 7 System.Int32.MinValue)) " 80000000"
    test "test3985" (lazy(sprintf "% 0X" System.Int32.MinValue)) " 80000000"
    test "test3986" (lazy(sprintf "% 05X" System.Int32.MinValue)) " 80000000"
    test "test3987" (lazy(sprintf "% 01X" System.Int32.MinValue)) " 80000000"
    test "test3988" (lazy(sprintf "% 0*X" 7 System.Int32.MinValue)) " 80000000"
    test "test3989" (lazy(sprintf "%- 0X" System.Int32.MinValue)) " 80000000"
    test "test3990" (lazy(sprintf "%- 05X" System.Int32.MinValue)) " 80000000"
    test "test3991" (lazy(sprintf "%- 01X" System.Int32.MinValue)) " 80000000"
    test "test3992" (lazy(sprintf "%- 0*X" 7 System.Int32.MinValue)) " 80000000"
    test "test3993" (lazy(sprintf "%X" System.Int64.MinValue)) "8000000000000000"
    test "test3994" (lazy(sprintf "%5X" System.Int64.MinValue)) "8000000000000000"
    test "test3995" (lazy(sprintf "%1X" System.Int64.MinValue)) "8000000000000000"
    test "test3996" (lazy(sprintf "%*X" 7 System.Int64.MinValue)) "8000000000000000"
    test "test3997" (lazy(sprintf "%-X" System.Int64.MinValue)) "8000000000000000"
    test "test3998" (lazy(sprintf "%-5X" System.Int64.MinValue)) "8000000000000000"
    test "test3999" (lazy(sprintf "%-1X" System.Int64.MinValue)) "8000000000000000"
    test "test4000" (lazy(sprintf "%-*X" 7 System.Int64.MinValue)) "8000000000000000"
let func4000()=
    test "test4001" (lazy(sprintf "%0X" System.Int64.MinValue)) "8000000000000000"
    test "test4002" (lazy(sprintf "%05X" System.Int64.MinValue)) "8000000000000000"
    test "test4003" (lazy(sprintf "%01X" System.Int64.MinValue)) "8000000000000000"
    test "test4004" (lazy(sprintf "%0*X" 7 System.Int64.MinValue)) "8000000000000000"
    test "test4005" (lazy(sprintf "%-0X" System.Int64.MinValue)) "8000000000000000"
    test "test4006" (lazy(sprintf "%-05X" System.Int64.MinValue)) "8000000000000000"
    test "test4007" (lazy(sprintf "%-01X" System.Int64.MinValue)) "8000000000000000"
    test "test4008" (lazy(sprintf "%-0*X" 7 System.Int64.MinValue)) "8000000000000000"
    test "test4009" (lazy(sprintf "%+X" System.Int64.MinValue)) "+8000000000000000"
    test "test4010" (lazy(sprintf "%+5X" System.Int64.MinValue)) "+8000000000000000"
    test "test4011" (lazy(sprintf "%+1X" System.Int64.MinValue)) "+8000000000000000"
    test "test4012" (lazy(sprintf "%+*X" 7 System.Int64.MinValue)) "+8000000000000000"
    test "test4013" (lazy(sprintf "%-+X" System.Int64.MinValue)) "+8000000000000000"
    test "test4014" (lazy(sprintf "%-+5X" System.Int64.MinValue)) "+8000000000000000"
    test "test4015" (lazy(sprintf "%-+1X" System.Int64.MinValue)) "+8000000000000000"
    test "test4016" (lazy(sprintf "%-+*X" 7 System.Int64.MinValue)) "+8000000000000000"
    test "test4017" (lazy(sprintf "%+0X" System.Int64.MinValue)) "+8000000000000000"
    test "test4018" (lazy(sprintf "%+05X" System.Int64.MinValue)) "+8000000000000000"
    test "test4019" (lazy(sprintf "%+01X" System.Int64.MinValue)) "+8000000000000000"
    test "test4020" (lazy(sprintf "%+0*X" 7 System.Int64.MinValue)) "+8000000000000000"
    test "test4021" (lazy(sprintf "%-+0X" System.Int64.MinValue)) "+8000000000000000"
    test "test4022" (lazy(sprintf "%-+05X" System.Int64.MinValue)) "+8000000000000000"
    test "test4023" (lazy(sprintf "%-+01X" System.Int64.MinValue)) "+8000000000000000"
    test "test4024" (lazy(sprintf "%-+0*X" 7 System.Int64.MinValue)) "+8000000000000000"
    test "test4025" (lazy(sprintf "% X" System.Int64.MinValue)) " 8000000000000000"
    test "test4026" (lazy(sprintf "% 5X" System.Int64.MinValue)) " 8000000000000000"
    test "test4027" (lazy(sprintf "% 1X" System.Int64.MinValue)) " 8000000000000000"
    test "test4028" (lazy(sprintf "% *X" 7 System.Int64.MinValue)) " 8000000000000000"
    test "test4029" (lazy(sprintf "%- X" System.Int64.MinValue)) " 8000000000000000"
    test "test4030" (lazy(sprintf "%- 5X" System.Int64.MinValue)) " 8000000000000000"
    test "test4031" (lazy(sprintf "%- 1X" System.Int64.MinValue)) " 8000000000000000"
    test "test4032" (lazy(sprintf "%- *X" 7 System.Int64.MinValue)) " 8000000000000000"
    test "test4033" (lazy(sprintf "% 0X" System.Int64.MinValue)) " 8000000000000000"
    test "test4034" (lazy(sprintf "% 05X" System.Int64.MinValue)) " 8000000000000000"
    test "test4035" (lazy(sprintf "% 01X" System.Int64.MinValue)) " 8000000000000000"
    test "test4036" (lazy(sprintf "% 0*X" 7 System.Int64.MinValue)) " 8000000000000000"
    test "test4037" (lazy(sprintf "%- 0X" System.Int64.MinValue)) " 8000000000000000"
    test "test4038" (lazy(sprintf "%- 05X" System.Int64.MinValue)) " 8000000000000000"
    test "test4039" (lazy(sprintf "%- 01X" System.Int64.MinValue)) " 8000000000000000"
    test "test4040" (lazy(sprintf "%- 0*X" 7 System.Int64.MinValue)) " 8000000000000000"
    test "test4041" (lazy(sprintf "%X" 55n)) "37"
    test "test4042" (lazy(sprintf "%5X" 55n)) "   37"
    test "test4043" (lazy(sprintf "%1X" 55n)) "37"
    test "test4044" (lazy(sprintf "%*X" 7 55n)) "     37"
    test "test4045" (lazy(sprintf "%-X" 55n)) "37"
    test "test4046" (lazy(sprintf "%-5X" 55n)) "37   "
    test "test4047" (lazy(sprintf "%-1X" 55n)) "37"
    test "test4048" (lazy(sprintf "%-*X" 7 55n)) "37     "
    test "test4049" (lazy(sprintf "%0X" 55n)) "37"
    test "test4050" (lazy(sprintf "%05X" 55n)) "00037"
    test "test4051" (lazy(sprintf "%01X" 55n)) "37"
    test "test4052" (lazy(sprintf "%0*X" 7 55n)) "0000037"
    test "test4053" (lazy(sprintf "%-0X" 55n)) "37"
    test "test4054" (lazy(sprintf "%-05X" 55n)) "37   "
    test "test4055" (lazy(sprintf "%-01X" 55n)) "37"
    test "test4056" (lazy(sprintf "%-0*X" 7 55n)) "37     "
    test "test4057" (lazy(sprintf "%+X" 55n)) "+37"
    test "test4058" (lazy(sprintf "%+5X" 55n)) "  +37"
    test "test4059" (lazy(sprintf "%+1X" 55n)) "+37"
    test "test4060" (lazy(sprintf "%+*X" 7 55n)) "    +37"
    test "test4061" (lazy(sprintf "%-+X" 55n)) "+37"
    test "test4062" (lazy(sprintf "%-+5X" 55n)) "+37  "
    test "test4063" (lazy(sprintf "%-+1X" 55n)) "+37"
    test "test4064" (lazy(sprintf "%-+*X" 7 55n)) "+37    "
    test "test4065" (lazy(sprintf "%+0X" 55n)) "+37"
    test "test4066" (lazy(sprintf "%+05X" 55n)) "+0037"
    test "test4067" (lazy(sprintf "%+01X" 55n)) "+37"
    test "test4068" (lazy(sprintf "%+0*X" 7 55n)) "+000037"
    test "test4069" (lazy(sprintf "%-+0X" 55n)) "+37"
    test "test4070" (lazy(sprintf "%-+05X" 55n)) "+37  "
    test "test4071" (lazy(sprintf "%-+01X" 55n)) "+37"
    test "test4072" (lazy(sprintf "%-+0*X" 7 55n)) "+37    "
    test "test4073" (lazy(sprintf "% X" 55n)) " 37"
    test "test4074" (lazy(sprintf "% 5X" 55n)) "   37"
    test "test4075" (lazy(sprintf "% 1X" 55n)) " 37"
    test "test4076" (lazy(sprintf "% *X" 7 55n)) "     37"
    test "test4077" (lazy(sprintf "%- X" 55n)) " 37"
    test "test4078" (lazy(sprintf "%- 5X" 55n)) " 37  "
    test "test4079" (lazy(sprintf "%- 1X" 55n)) " 37"
    test "test4080" (lazy(sprintf "%- *X" 7 55n)) " 37    "
    test "test4081" (lazy(sprintf "% 0X" 55n)) " 37"
    test "test4082" (lazy(sprintf "% 05X" 55n)) " 0037"
    test "test4083" (lazy(sprintf "% 01X" 55n)) " 37"
    test "test4084" (lazy(sprintf "% 0*X" 7 55n)) " 000037"
    test "test4085" (lazy(sprintf "%- 0X" 55n)) " 37"
    test "test4086" (lazy(sprintf "%- 05X" 55n)) " 37  "
    test "test4087" (lazy(sprintf "%- 01X" 55n)) " 37"
    test "test4088" (lazy(sprintf "%- 0*X" 7 55n)) " 37    "
    test "test4089" (lazy(sprintf "%X" 999un)) "3E7"
    test "test4090" (lazy(sprintf "%5X" 999un)) "  3E7"
    test "test4091" (lazy(sprintf "%1X" 999un)) "3E7"
    test "test4092" (lazy(sprintf "%*X" 7 999un)) "    3E7"
    test "test4093" (lazy(sprintf "%-X" 999un)) "3E7"
    test "test4094" (lazy(sprintf "%-5X" 999un)) "3E7  "
    test "test4095" (lazy(sprintf "%-1X" 999un)) "3E7"
    test "test4096" (lazy(sprintf "%-*X" 7 999un)) "3E7    "
    test "test4097" (lazy(sprintf "%0X" 999un)) "3E7"
    test "test4098" (lazy(sprintf "%05X" 999un)) "003E7"
    test "test4099" (lazy(sprintf "%01X" 999un)) "3E7"
    test "test4100" (lazy(sprintf "%0*X" 7 999un)) "00003E7"
    test "test4101" (lazy(sprintf "%-0X" 999un)) "3E7"
    test "test4102" (lazy(sprintf "%-05X" 999un)) "3E7  "
    test "test4103" (lazy(sprintf "%-01X" 999un)) "3E7"
    test "test4104" (lazy(sprintf "%-0*X" 7 999un)) "3E7    "
    test "test4105" (lazy(sprintf "%+X" 999un)) "+3E7"
    test "test4106" (lazy(sprintf "%+5X" 999un)) " +3E7"
    test "test4107" (lazy(sprintf "%+1X" 999un)) "+3E7"
    test "test4108" (lazy(sprintf "%+*X" 7 999un)) "   +3E7"
    test "test4109" (lazy(sprintf "%-+X" 999un)) "+3E7"
    test "test4110" (lazy(sprintf "%-+5X" 999un)) "+3E7 "
    test "test4111" (lazy(sprintf "%-+1X" 999un)) "+3E7"
    test "test4112" (lazy(sprintf "%-+*X" 7 999un)) "+3E7   "
    test "test4113" (lazy(sprintf "%+0X" 999un)) "+3E7"
    test "test4114" (lazy(sprintf "%+05X" 999un)) "+03E7"
    test "test4115" (lazy(sprintf "%+01X" 999un)) "+3E7"
    test "test4116" (lazy(sprintf "%+0*X" 7 999un)) "+0003E7"
    test "test4117" (lazy(sprintf "%-+0X" 999un)) "+3E7"
    test "test4118" (lazy(sprintf "%-+05X" 999un)) "+3E7 "
    test "test4119" (lazy(sprintf "%-+01X" 999un)) "+3E7"
    test "test4120" (lazy(sprintf "%-+0*X" 7 999un)) "+3E7   "
    test "test4121" (lazy(sprintf "% X" 999un)) " 3E7"
    test "test4122" (lazy(sprintf "% 5X" 999un)) "  3E7"
    test "test4123" (lazy(sprintf "% 1X" 999un)) " 3E7"
    test "test4124" (lazy(sprintf "% *X" 7 999un)) "    3E7"
    test "test4125" (lazy(sprintf "%- X" 999un)) " 3E7"
    test "test4126" (lazy(sprintf "%- 5X" 999un)) " 3E7 "
    test "test4127" (lazy(sprintf "%- 1X" 999un)) " 3E7"
    test "test4128" (lazy(sprintf "%- *X" 7 999un)) " 3E7   "
    test "test4129" (lazy(sprintf "% 0X" 999un)) " 3E7"
    test "test4130" (lazy(sprintf "% 05X" 999un)) " 03E7"
    test "test4131" (lazy(sprintf "% 01X" 999un)) " 3E7"
    test "test4132" (lazy(sprintf "% 0*X" 7 999un)) " 0003E7"
    test "test4133" (lazy(sprintf "%- 0X" 999un)) " 3E7"
    test "test4134" (lazy(sprintf "%- 05X" 999un)) " 3E7 "
    test "test4135" (lazy(sprintf "%- 01X" 999un)) " 3E7"
    test "test4136" (lazy(sprintf "%- 0*X" 7 999un)) " 3E7   "
    test "test4137" (lazy(sprintf "%o" 14)) "16"
    test "test4138" (lazy(sprintf "%5o" 14)) "   16"
    test "test4139" (lazy(sprintf "%1o" 14)) "16"
    test "test4140" (lazy(sprintf "%*o" 7 14)) "     16"
    test "test4141" (lazy(sprintf "%-o" 14)) "16"
    test "test4142" (lazy(sprintf "%-5o" 14)) "16   "
    test "test4143" (lazy(sprintf "%-1o" 14)) "16"
    test "test4144" (lazy(sprintf "%-*o" 7 14)) "16     "
    test "test4145" (lazy(sprintf "%0o" 14)) "16"
    test "test4146" (lazy(sprintf "%05o" 14)) "00016"
    test "test4147" (lazy(sprintf "%01o" 14)) "16"
    test "test4148" (lazy(sprintf "%0*o" 7 14)) "0000016"
    test "test4149" (lazy(sprintf "%-0o" 14)) "16"
    test "test4150" (lazy(sprintf "%-05o" 14)) "16   "
    test "test4151" (lazy(sprintf "%-01o" 14)) "16"
    test "test4152" (lazy(sprintf "%-0*o" 7 14)) "16     "
    test "test4153" (lazy(sprintf "%+o" 14)) "+16"
    test "test4154" (lazy(sprintf "%+5o" 14)) "  +16"
    test "test4155" (lazy(sprintf "%+1o" 14)) "+16"
    test "test4156" (lazy(sprintf "%+*o" 7 14)) "    +16"
    test "test4157" (lazy(sprintf "%-+o" 14)) "+16"
    test "test4158" (lazy(sprintf "%-+5o" 14)) "+16  "
    test "test4159" (lazy(sprintf "%-+1o" 14)) "+16"
    test "test4160" (lazy(sprintf "%-+*o" 7 14)) "+16    "
    test "test4161" (lazy(sprintf "%+0o" 14)) "+16"
    test "test4162" (lazy(sprintf "%+05o" 14)) "+0016"
    test "test4163" (lazy(sprintf "%+01o" 14)) "+16"
    test "test4164" (lazy(sprintf "%+0*o" 7 14)) "+000016"
    test "test4165" (lazy(sprintf "%-+0o" 14)) "+16"
    test "test4166" (lazy(sprintf "%-+05o" 14)) "+16  "
    test "test4167" (lazy(sprintf "%-+01o" 14)) "+16"
    test "test4168" (lazy(sprintf "%-+0*o" 7 14)) "+16    "
    test "test4169" (lazy(sprintf "% o" 14)) " 16"
    test "test4170" (lazy(sprintf "% 5o" 14)) "   16"
    test "test4171" (lazy(sprintf "% 1o" 14)) " 16"
    test "test4172" (lazy(sprintf "% *o" 7 14)) "     16"
    test "test4173" (lazy(sprintf "%- o" 14)) " 16"
    test "test4174" (lazy(sprintf "%- 5o" 14)) " 16  "
    test "test4175" (lazy(sprintf "%- 1o" 14)) " 16"
    test "test4176" (lazy(sprintf "%- *o" 7 14)) " 16    "
    test "test4177" (lazy(sprintf "% 0o" 14)) " 16"
    test "test4178" (lazy(sprintf "% 05o" 14)) " 0016"
    test "test4179" (lazy(sprintf "% 01o" 14)) " 16"
    test "test4180" (lazy(sprintf "% 0*o" 7 14)) " 000016"
    test "test4181" (lazy(sprintf "%- 0o" 14)) " 16"
    test "test4182" (lazy(sprintf "%- 05o" 14)) " 16  "
    test "test4183" (lazy(sprintf "%- 01o" 14)) " 16"
    test "test4184" (lazy(sprintf "%- 0*o" 7 14)) " 16    "
    test "test4185" (lazy(sprintf "%o" -10)) "37777777766"
    test "test4186" (lazy(sprintf "%5o" -10)) "37777777766"
    test "test4187" (lazy(sprintf "%1o" -10)) "37777777766"
    test "test4188" (lazy(sprintf "%*o" 7 -10)) "37777777766"
    test "test4189" (lazy(sprintf "%-o" -10)) "37777777766"
    test "test4190" (lazy(sprintf "%-5o" -10)) "37777777766"
    test "test4191" (lazy(sprintf "%-1o" -10)) "37777777766"
    test "test4192" (lazy(sprintf "%-*o" 7 -10)) "37777777766"
    test "test4193" (lazy(sprintf "%0o" -10)) "37777777766"
    test "test4194" (lazy(sprintf "%05o" -10)) "37777777766"
    test "test4195" (lazy(sprintf "%01o" -10)) "37777777766"
    test "test4196" (lazy(sprintf "%0*o" 7 -10)) "37777777766"
    test "test4197" (lazy(sprintf "%-0o" -10)) "37777777766"
    test "test4198" (lazy(sprintf "%-05o" -10)) "37777777766"
    test "test4199" (lazy(sprintf "%-01o" -10)) "37777777766"
    test "test4200" (lazy(sprintf "%-0*o" 7 -10)) "37777777766"
    test "test4201" (lazy(sprintf "%+o" -10)) "+37777777766"
    test "test4202" (lazy(sprintf "%+5o" -10)) "+37777777766"
    test "test4203" (lazy(sprintf "%+1o" -10)) "+37777777766"
    test "test4204" (lazy(sprintf "%+*o" 7 -10)) "+37777777766"
    test "test4205" (lazy(sprintf "%-+o" -10)) "+37777777766"
    test "test4206" (lazy(sprintf "%-+5o" -10)) "+37777777766"
    test "test4207" (lazy(sprintf "%-+1o" -10)) "+37777777766"
    test "test4208" (lazy(sprintf "%-+*o" 7 -10)) "+37777777766"
    test "test4209" (lazy(sprintf "%+0o" -10)) "+37777777766"
    test "test4210" (lazy(sprintf "%+05o" -10)) "+37777777766"
    test "test4211" (lazy(sprintf "%+01o" -10)) "+37777777766"
    test "test4212" (lazy(sprintf "%+0*o" 7 -10)) "+37777777766"
    test "test4213" (lazy(sprintf "%-+0o" -10)) "+37777777766"
    test "test4214" (lazy(sprintf "%-+05o" -10)) "+37777777766"
    test "test4215" (lazy(sprintf "%-+01o" -10)) "+37777777766"
    test "test4216" (lazy(sprintf "%-+0*o" 7 -10)) "+37777777766"
    test "test4217" (lazy(sprintf "% o" -10)) " 37777777766"
    test "test4218" (lazy(sprintf "% 5o" -10)) " 37777777766"
    test "test4219" (lazy(sprintf "% 1o" -10)) " 37777777766"
    test "test4220" (lazy(sprintf "% *o" 7 -10)) " 37777777766"
    test "test4221" (lazy(sprintf "%- o" -10)) " 37777777766"
    test "test4222" (lazy(sprintf "%- 5o" -10)) " 37777777766"
    test "test4223" (lazy(sprintf "%- 1o" -10)) " 37777777766"
    test "test4224" (lazy(sprintf "%- *o" 7 -10)) " 37777777766"
    test "test4225" (lazy(sprintf "% 0o" -10)) " 37777777766"
    test "test4226" (lazy(sprintf "% 05o" -10)) " 37777777766"
    test "test4227" (lazy(sprintf "% 01o" -10)) " 37777777766"
    test "test4228" (lazy(sprintf "% 0*o" 7 -10)) " 37777777766"
    test "test4229" (lazy(sprintf "%- 0o" -10)) " 37777777766"
    test "test4230" (lazy(sprintf "%- 05o" -10)) " 37777777766"
    test "test4231" (lazy(sprintf "%- 01o" -10)) " 37777777766"
    test "test4232" (lazy(sprintf "%- 0*o" 7 -10)) " 37777777766"
    test "test4233" (lazy(sprintf "%o" 55s)) "67"
    test "test4234" (lazy(sprintf "%5o" 55s)) "   67"
    test "test4235" (lazy(sprintf "%1o" 55s)) "67"
    test "test4236" (lazy(sprintf "%*o" 7 55s)) "     67"
    test "test4237" (lazy(sprintf "%-o" 55s)) "67"
    test "test4238" (lazy(sprintf "%-5o" 55s)) "67   "
    test "test4239" (lazy(sprintf "%-1o" 55s)) "67"
    test "test4240" (lazy(sprintf "%-*o" 7 55s)) "67     "
    test "test4241" (lazy(sprintf "%0o" 55s)) "67"
    test "test4242" (lazy(sprintf "%05o" 55s)) "00067"
    test "test4243" (lazy(sprintf "%01o" 55s)) "67"
    test "test4244" (lazy(sprintf "%0*o" 7 55s)) "0000067"
    test "test4245" (lazy(sprintf "%-0o" 55s)) "67"
    test "test4246" (lazy(sprintf "%-05o" 55s)) "67   "
    test "test4247" (lazy(sprintf "%-01o" 55s)) "67"
    test "test4248" (lazy(sprintf "%-0*o" 7 55s)) "67     "
    test "test4249" (lazy(sprintf "%+o" 55s)) "+67"
    test "test4250" (lazy(sprintf "%+5o" 55s)) "  +67"
    test "test4251" (lazy(sprintf "%+1o" 55s)) "+67"
    test "test4252" (lazy(sprintf "%+*o" 7 55s)) "    +67"
    test "test4253" (lazy(sprintf "%-+o" 55s)) "+67"
    test "test4254" (lazy(sprintf "%-+5o" 55s)) "+67  "
    test "test4255" (lazy(sprintf "%-+1o" 55s)) "+67"
    test "test4256" (lazy(sprintf "%-+*o" 7 55s)) "+67    "
    test "test4257" (lazy(sprintf "%+0o" 55s)) "+67"
    test "test4258" (lazy(sprintf "%+05o" 55s)) "+0067"
    test "test4259" (lazy(sprintf "%+01o" 55s)) "+67"
    test "test4260" (lazy(sprintf "%+0*o" 7 55s)) "+000067"
    test "test4261" (lazy(sprintf "%-+0o" 55s)) "+67"
    test "test4262" (lazy(sprintf "%-+05o" 55s)) "+67  "
    test "test4263" (lazy(sprintf "%-+01o" 55s)) "+67"
    test "test4264" (lazy(sprintf "%-+0*o" 7 55s)) "+67    "
    test "test4265" (lazy(sprintf "% o" 55s)) " 67"
    test "test4266" (lazy(sprintf "% 5o" 55s)) "   67"
    test "test4267" (lazy(sprintf "% 1o" 55s)) " 67"
    test "test4268" (lazy(sprintf "% *o" 7 55s)) "     67"
    test "test4269" (lazy(sprintf "%- o" 55s)) " 67"
    test "test4270" (lazy(sprintf "%- 5o" 55s)) " 67  "
    test "test4271" (lazy(sprintf "%- 1o" 55s)) " 67"
    test "test4272" (lazy(sprintf "%- *o" 7 55s)) " 67    "
    test "test4273" (lazy(sprintf "% 0o" 55s)) " 67"
    test "test4274" (lazy(sprintf "% 05o" 55s)) " 0067"
    test "test4275" (lazy(sprintf "% 01o" 55s)) " 67"
    test "test4276" (lazy(sprintf "% 0*o" 7 55s)) " 000067"
    test "test4277" (lazy(sprintf "%- 0o" 55s)) " 67"
    test "test4278" (lazy(sprintf "%- 05o" 55s)) " 67  "
    test "test4279" (lazy(sprintf "%- 01o" 55s)) " 67"
    test "test4280" (lazy(sprintf "%- 0*o" 7 55s)) " 67    "
    test "test4281" (lazy(sprintf "%o" -88s)) "177650"
    test "test4282" (lazy(sprintf "%5o" -88s)) "177650"
    test "test4283" (lazy(sprintf "%1o" -88s)) "177650"
    test "test4284" (lazy(sprintf "%*o" 7 -88s)) " 177650"
    test "test4285" (lazy(sprintf "%-o" -88s)) "177650"
    test "test4286" (lazy(sprintf "%-5o" -88s)) "177650"
    test "test4287" (lazy(sprintf "%-1o" -88s)) "177650"
    test "test4288" (lazy(sprintf "%-*o" 7 -88s)) "177650 "
    test "test4289" (lazy(sprintf "%0o" -88s)) "177650"
    test "test4290" (lazy(sprintf "%05o" -88s)) "177650"
    test "test4291" (lazy(sprintf "%01o" -88s)) "177650"
    test "test4292" (lazy(sprintf "%0*o" 7 -88s)) "0177650"
    test "test4293" (lazy(sprintf "%-0o" -88s)) "177650"
    test "test4294" (lazy(sprintf "%-05o" -88s)) "177650"
    test "test4295" (lazy(sprintf "%-01o" -88s)) "177650"
    test "test4296" (lazy(sprintf "%-0*o" 7 -88s)) "177650 "
    test "test4297" (lazy(sprintf "%+o" -88s)) "+177650"
    test "test4298" (lazy(sprintf "%+5o" -88s)) "+177650"
    test "test4299" (lazy(sprintf "%+1o" -88s)) "+177650"
    test "test4300" (lazy(sprintf "%+*o" 7 -88s)) "+177650"
    test "test4301" (lazy(sprintf "%-+o" -88s)) "+177650"
    test "test4302" (lazy(sprintf "%-+5o" -88s)) "+177650"
    test "test4303" (lazy(sprintf "%-+1o" -88s)) "+177650"
    test "test4304" (lazy(sprintf "%-+*o" 7 -88s)) "+177650"
    test "test4305" (lazy(sprintf "%+0o" -88s)) "+177650"
    test "test4306" (lazy(sprintf "%+05o" -88s)) "+177650"
    test "test4307" (lazy(sprintf "%+01o" -88s)) "+177650"
    test "test4308" (lazy(sprintf "%+0*o" 7 -88s)) "+177650"
    test "test4309" (lazy(sprintf "%-+0o" -88s)) "+177650"
    test "test4310" (lazy(sprintf "%-+05o" -88s)) "+177650"
    test "test4311" (lazy(sprintf "%-+01o" -88s)) "+177650"
    test "test4312" (lazy(sprintf "%-+0*o" 7 -88s)) "+177650"
    test "test4313" (lazy(sprintf "% o" -88s)) " 177650"
    test "test4314" (lazy(sprintf "% 5o" -88s)) " 177650"
    test "test4315" (lazy(sprintf "% 1o" -88s)) " 177650"
    test "test4316" (lazy(sprintf "% *o" 7 -88s)) " 177650"
    test "test4317" (lazy(sprintf "%- o" -88s)) " 177650"
    test "test4318" (lazy(sprintf "%- 5o" -88s)) " 177650"
    test "test4319" (lazy(sprintf "%- 1o" -88s)) " 177650"
    test "test4320" (lazy(sprintf "%- *o" 7 -88s)) " 177650"
    test "test4321" (lazy(sprintf "% 0o" -88s)) " 177650"
    test "test4322" (lazy(sprintf "% 05o" -88s)) " 177650"
    test "test4323" (lazy(sprintf "% 01o" -88s)) " 177650"
    test "test4324" (lazy(sprintf "% 0*o" 7 -88s)) " 177650"
    test "test4325" (lazy(sprintf "%- 0o" -88s)) " 177650"
    test "test4326" (lazy(sprintf "%- 05o" -88s)) " 177650"
    test "test4327" (lazy(sprintf "%- 01o" -88s)) " 177650"
    test "test4328" (lazy(sprintf "%- 0*o" 7 -88s)) " 177650"
    test "test4329" (lazy(sprintf "%o" 104us)) "150"
    test "test4330" (lazy(sprintf "%5o" 104us)) "  150"
    test "test4331" (lazy(sprintf "%1o" 104us)) "150"
    test "test4332" (lazy(sprintf "%*o" 7 104us)) "    150"
    test "test4333" (lazy(sprintf "%-o" 104us)) "150"
    test "test4334" (lazy(sprintf "%-5o" 104us)) "150  "
    test "test4335" (lazy(sprintf "%-1o" 104us)) "150"
    test "test4336" (lazy(sprintf "%-*o" 7 104us)) "150    "
    test "test4337" (lazy(sprintf "%0o" 104us)) "150"
    test "test4338" (lazy(sprintf "%05o" 104us)) "00150"
    test "test4339" (lazy(sprintf "%01o" 104us)) "150"
    test "test4340" (lazy(sprintf "%0*o" 7 104us)) "0000150"
    test "test4341" (lazy(sprintf "%-0o" 104us)) "150"
    test "test4342" (lazy(sprintf "%-05o" 104us)) "150  "
    test "test4343" (lazy(sprintf "%-01o" 104us)) "150"
    test "test4344" (lazy(sprintf "%-0*o" 7 104us)) "150    "
    test "test4345" (lazy(sprintf "%+o" 104us)) "+150"
    test "test4346" (lazy(sprintf "%+5o" 104us)) " +150"
    test "test4347" (lazy(sprintf "%+1o" 104us)) "+150"
    test "test4348" (lazy(sprintf "%+*o" 7 104us)) "   +150"
    test "test4349" (lazy(sprintf "%-+o" 104us)) "+150"
    test "test4350" (lazy(sprintf "%-+5o" 104us)) "+150 "
    test "test4351" (lazy(sprintf "%-+1o" 104us)) "+150"
    test "test4352" (lazy(sprintf "%-+*o" 7 104us)) "+150   "
    test "test4353" (lazy(sprintf "%+0o" 104us)) "+150"
    test "test4354" (lazy(sprintf "%+05o" 104us)) "+0150"
    test "test4355" (lazy(sprintf "%+01o" 104us)) "+150"
    test "test4356" (lazy(sprintf "%+0*o" 7 104us)) "+000150"
    test "test4357" (lazy(sprintf "%-+0o" 104us)) "+150"
    test "test4358" (lazy(sprintf "%-+05o" 104us)) "+150 "
    test "test4359" (lazy(sprintf "%-+01o" 104us)) "+150"
    test "test4360" (lazy(sprintf "%-+0*o" 7 104us)) "+150   "
    test "test4361" (lazy(sprintf "% o" 104us)) " 150"
    test "test4362" (lazy(sprintf "% 5o" 104us)) "  150"
    test "test4363" (lazy(sprintf "% 1o" 104us)) " 150"
    test "test4364" (lazy(sprintf "% *o" 7 104us)) "    150"
    test "test4365" (lazy(sprintf "%- o" 104us)) " 150"
    test "test4366" (lazy(sprintf "%- 5o" 104us)) " 150 "
    test "test4367" (lazy(sprintf "%- 1o" 104us)) " 150"
    test "test4368" (lazy(sprintf "%- *o" 7 104us)) " 150   "
    test "test4369" (lazy(sprintf "% 0o" 104us)) " 150"
    test "test4370" (lazy(sprintf "% 05o" 104us)) " 0150"
    test "test4371" (lazy(sprintf "% 01o" 104us)) " 150"
    test "test4372" (lazy(sprintf "% 0*o" 7 104us)) " 000150"
    test "test4373" (lazy(sprintf "%- 0o" 104us)) " 150"
    test "test4374" (lazy(sprintf "%- 05o" 104us)) " 150 "
    test "test4375" (lazy(sprintf "%- 01o" 104us)) " 150"
    test "test4376" (lazy(sprintf "%- 0*o" 7 104us)) " 150   "
    test "test4377" (lazy(sprintf "%o" 12y)) "14"
    test "test4378" (lazy(sprintf "%5o" 12y)) "   14"
    test "test4379" (lazy(sprintf "%1o" 12y)) "14"
    test "test4380" (lazy(sprintf "%*o" 7 12y)) "     14"
    test "test4381" (lazy(sprintf "%-o" 12y)) "14"
    test "test4382" (lazy(sprintf "%-5o" 12y)) "14   "
    test "test4383" (lazy(sprintf "%-1o" 12y)) "14"
    test "test4384" (lazy(sprintf "%-*o" 7 12y)) "14     "
    test "test4385" (lazy(sprintf "%0o" 12y)) "14"
    test "test4386" (lazy(sprintf "%05o" 12y)) "00014"
    test "test4387" (lazy(sprintf "%01o" 12y)) "14"
    test "test4388" (lazy(sprintf "%0*o" 7 12y)) "0000014"
    test "test4389" (lazy(sprintf "%-0o" 12y)) "14"
    test "test4390" (lazy(sprintf "%-05o" 12y)) "14   "
    test "test4391" (lazy(sprintf "%-01o" 12y)) "14"
    test "test4392" (lazy(sprintf "%-0*o" 7 12y)) "14     "
    test "test4393" (lazy(sprintf "%+o" 12y)) "+14"
    test "test4394" (lazy(sprintf "%+5o" 12y)) "  +14"
    test "test4395" (lazy(sprintf "%+1o" 12y)) "+14"
    test "test4396" (lazy(sprintf "%+*o" 7 12y)) "    +14"
    test "test4397" (lazy(sprintf "%-+o" 12y)) "+14"
    test "test4398" (lazy(sprintf "%-+5o" 12y)) "+14  "
    test "test4399" (lazy(sprintf "%-+1o" 12y)) "+14"
    test "test4400" (lazy(sprintf "%-+*o" 7 12y)) "+14    "
    test "test4401" (lazy(sprintf "%+0o" 12y)) "+14"
    test "test4402" (lazy(sprintf "%+05o" 12y)) "+0014"
    test "test4403" (lazy(sprintf "%+01o" 12y)) "+14"
    test "test4404" (lazy(sprintf "%+0*o" 7 12y)) "+000014"
    test "test4405" (lazy(sprintf "%-+0o" 12y)) "+14"
    test "test4406" (lazy(sprintf "%-+05o" 12y)) "+14  "
    test "test4407" (lazy(sprintf "%-+01o" 12y)) "+14"
    test "test4408" (lazy(sprintf "%-+0*o" 7 12y)) "+14    "
    test "test4409" (lazy(sprintf "% o" 12y)) " 14"
    test "test4410" (lazy(sprintf "% 5o" 12y)) "   14"
    test "test4411" (lazy(sprintf "% 1o" 12y)) " 14"
    test "test4412" (lazy(sprintf "% *o" 7 12y)) "     14"
    test "test4413" (lazy(sprintf "%- o" 12y)) " 14"
    test "test4414" (lazy(sprintf "%- 5o" 12y)) " 14  "
    test "test4415" (lazy(sprintf "%- 1o" 12y)) " 14"
    test "test4416" (lazy(sprintf "%- *o" 7 12y)) " 14    "
    test "test4417" (lazy(sprintf "% 0o" 12y)) " 14"
    test "test4418" (lazy(sprintf "% 05o" 12y)) " 0014"
    test "test4419" (lazy(sprintf "% 01o" 12y)) " 14"
    test "test4420" (lazy(sprintf "% 0*o" 7 12y)) " 000014"
    test "test4421" (lazy(sprintf "%- 0o" 12y)) " 14"
    test "test4422" (lazy(sprintf "%- 05o" 12y)) " 14  "
    test "test4423" (lazy(sprintf "%- 01o" 12y)) " 14"
    test "test4424" (lazy(sprintf "%- 0*o" 7 12y)) " 14    "
    test "test4425" (lazy(sprintf "%o" -94y)) "242"
    test "test4426" (lazy(sprintf "%5o" -94y)) "  242"
    test "test4427" (lazy(sprintf "%1o" -94y)) "242"
    test "test4428" (lazy(sprintf "%*o" 7 -94y)) "    242"
    test "test4429" (lazy(sprintf "%-o" -94y)) "242"
    test "test4430" (lazy(sprintf "%-5o" -94y)) "242  "
    test "test4431" (lazy(sprintf "%-1o" -94y)) "242"
    test "test4432" (lazy(sprintf "%-*o" 7 -94y)) "242    "
    test "test4433" (lazy(sprintf "%0o" -94y)) "242"
    test "test4434" (lazy(sprintf "%05o" -94y)) "00242"
    test "test4435" (lazy(sprintf "%01o" -94y)) "242"
    test "test4436" (lazy(sprintf "%0*o" 7 -94y)) "0000242"
    test "test4437" (lazy(sprintf "%-0o" -94y)) "242"
    test "test4438" (lazy(sprintf "%-05o" -94y)) "242  "
    test "test4439" (lazy(sprintf "%-01o" -94y)) "242"
    test "test4440" (lazy(sprintf "%-0*o" 7 -94y)) "242    "
    test "test4441" (lazy(sprintf "%+o" -94y)) "+242"
    test "test4442" (lazy(sprintf "%+5o" -94y)) " +242"
    test "test4443" (lazy(sprintf "%+1o" -94y)) "+242"
    test "test4444" (lazy(sprintf "%+*o" 7 -94y)) "   +242"
    test "test4445" (lazy(sprintf "%-+o" -94y)) "+242"
    test "test4446" (lazy(sprintf "%-+5o" -94y)) "+242 "
    test "test4447" (lazy(sprintf "%-+1o" -94y)) "+242"
    test "test4448" (lazy(sprintf "%-+*o" 7 -94y)) "+242   "
    test "test4449" (lazy(sprintf "%+0o" -94y)) "+242"
    test "test4450" (lazy(sprintf "%+05o" -94y)) "+0242"
    test "test4451" (lazy(sprintf "%+01o" -94y)) "+242"
    test "test4452" (lazy(sprintf "%+0*o" 7 -94y)) "+000242"
    test "test4453" (lazy(sprintf "%-+0o" -94y)) "+242"
    test "test4454" (lazy(sprintf "%-+05o" -94y)) "+242 "
    test "test4455" (lazy(sprintf "%-+01o" -94y)) "+242"
    test "test4456" (lazy(sprintf "%-+0*o" 7 -94y)) "+242   "
    test "test4457" (lazy(sprintf "% o" -94y)) " 242"
    test "test4458" (lazy(sprintf "% 5o" -94y)) "  242"
    test "test4459" (lazy(sprintf "% 1o" -94y)) " 242"
    test "test4460" (lazy(sprintf "% *o" 7 -94y)) "    242"
    test "test4461" (lazy(sprintf "%- o" -94y)) " 242"
    test "test4462" (lazy(sprintf "%- 5o" -94y)) " 242 "
    test "test4463" (lazy(sprintf "%- 1o" -94y)) " 242"
    test "test4464" (lazy(sprintf "%- *o" 7 -94y)) " 242   "
    test "test4465" (lazy(sprintf "% 0o" -94y)) " 242"
    test "test4466" (lazy(sprintf "% 05o" -94y)) " 0242"
    test "test4467" (lazy(sprintf "% 01o" -94y)) " 242"
    test "test4468" (lazy(sprintf "% 0*o" 7 -94y)) " 000242"
    test "test4469" (lazy(sprintf "%- 0o" -94y)) " 242"
    test "test4470" (lazy(sprintf "%- 05o" -94y)) " 242 "
    test "test4471" (lazy(sprintf "%- 01o" -94y)) " 242"
    test "test4472" (lazy(sprintf "%- 0*o" 7 -94y)) " 242   "
    test "test4473" (lazy(sprintf "%o" 88uy)) "130"
    test "test4474" (lazy(sprintf "%5o" 88uy)) "  130"
    test "test4475" (lazy(sprintf "%1o" 88uy)) "130"
    test "test4476" (lazy(sprintf "%*o" 7 88uy)) "    130"
    test "test4477" (lazy(sprintf "%-o" 88uy)) "130"
    test "test4478" (lazy(sprintf "%-5o" 88uy)) "130  "
    test "test4479" (lazy(sprintf "%-1o" 88uy)) "130"
    test "test4480" (lazy(sprintf "%-*o" 7 88uy)) "130    "
    test "test4481" (lazy(sprintf "%0o" 88uy)) "130"
    test "test4482" (lazy(sprintf "%05o" 88uy)) "00130"
    test "test4483" (lazy(sprintf "%01o" 88uy)) "130"
    test "test4484" (lazy(sprintf "%0*o" 7 88uy)) "0000130"
    test "test4485" (lazy(sprintf "%-0o" 88uy)) "130"
    test "test4486" (lazy(sprintf "%-05o" 88uy)) "130  "
    test "test4487" (lazy(sprintf "%-01o" 88uy)) "130"
    test "test4488" (lazy(sprintf "%-0*o" 7 88uy)) "130    "
    test "test4489" (lazy(sprintf "%+o" 88uy)) "+130"
    test "test4490" (lazy(sprintf "%+5o" 88uy)) " +130"
    test "test4491" (lazy(sprintf "%+1o" 88uy)) "+130"
    test "test4492" (lazy(sprintf "%+*o" 7 88uy)) "   +130"
    test "test4493" (lazy(sprintf "%-+o" 88uy)) "+130"
    test "test4494" (lazy(sprintf "%-+5o" 88uy)) "+130 "
    test "test4495" (lazy(sprintf "%-+1o" 88uy)) "+130"
    test "test4496" (lazy(sprintf "%-+*o" 7 88uy)) "+130   "
    test "test4497" (lazy(sprintf "%+0o" 88uy)) "+130"
    test "test4498" (lazy(sprintf "%+05o" 88uy)) "+0130"
    test "test4499" (lazy(sprintf "%+01o" 88uy)) "+130"
    test "test4500" (lazy(sprintf "%+0*o" 7 88uy)) "+000130"
    test "test4501" (lazy(sprintf "%-+0o" 88uy)) "+130"
    test "test4502" (lazy(sprintf "%-+05o" 88uy)) "+130 "
    test "test4503" (lazy(sprintf "%-+01o" 88uy)) "+130"
    test "test4504" (lazy(sprintf "%-+0*o" 7 88uy)) "+130   "
    test "test4505" (lazy(sprintf "% o" 88uy)) " 130"
    test "test4506" (lazy(sprintf "% 5o" 88uy)) "  130"
    test "test4507" (lazy(sprintf "% 1o" 88uy)) " 130"
    test "test4508" (lazy(sprintf "% *o" 7 88uy)) "    130"
    test "test4509" (lazy(sprintf "%- o" 88uy)) " 130"
    test "test4510" (lazy(sprintf "%- 5o" 88uy)) " 130 "
    test "test4511" (lazy(sprintf "%- 1o" 88uy)) " 130"
    test "test4512" (lazy(sprintf "%- *o" 7 88uy)) " 130   "
    test "test4513" (lazy(sprintf "% 0o" 88uy)) " 130"
    test "test4514" (lazy(sprintf "% 05o" 88uy)) " 0130"
    test "test4515" (lazy(sprintf "% 01o" 88uy)) " 130"
    test "test4516" (lazy(sprintf "% 0*o" 7 88uy)) " 000130"
    test "test4517" (lazy(sprintf "%- 0o" 88uy)) " 130"
    test "test4518" (lazy(sprintf "%- 05o" 88uy)) " 130 "
    test "test4519" (lazy(sprintf "%- 01o" 88uy)) " 130"
    test "test4520" (lazy(sprintf "%- 0*o" 7 88uy)) " 130   "
    test "test4521" (lazy(sprintf "%o" 999L)) "1747"
    test "test4522" (lazy(sprintf "%5o" 999L)) " 1747"
    test "test4523" (lazy(sprintf "%1o" 999L)) "1747"
    test "test4524" (lazy(sprintf "%*o" 7 999L)) "   1747"
    test "test4525" (lazy(sprintf "%-o" 999L)) "1747"
    test "test4526" (lazy(sprintf "%-5o" 999L)) "1747 "
    test "test4527" (lazy(sprintf "%-1o" 999L)) "1747"
    test "test4528" (lazy(sprintf "%-*o" 7 999L)) "1747   "
    test "test4529" (lazy(sprintf "%0o" 999L)) "1747"
    test "test4530" (lazy(sprintf "%05o" 999L)) "01747"
    test "test4531" (lazy(sprintf "%01o" 999L)) "1747"
    test "test4532" (lazy(sprintf "%0*o" 7 999L)) "0001747"
    test "test4533" (lazy(sprintf "%-0o" 999L)) "1747"
    test "test4534" (lazy(sprintf "%-05o" 999L)) "1747 "
    test "test4535" (lazy(sprintf "%-01o" 999L)) "1747"
    test "test4536" (lazy(sprintf "%-0*o" 7 999L)) "1747   "
    test "test4537" (lazy(sprintf "%+o" 999L)) "+1747"
    test "test4538" (lazy(sprintf "%+5o" 999L)) "+1747"
    test "test4539" (lazy(sprintf "%+1o" 999L)) "+1747"
    test "test4540" (lazy(sprintf "%+*o" 7 999L)) "  +1747"
    test "test4541" (lazy(sprintf "%-+o" 999L)) "+1747"
    test "test4542" (lazy(sprintf "%-+5o" 999L)) "+1747"
    test "test4543" (lazy(sprintf "%-+1o" 999L)) "+1747"
    test "test4544" (lazy(sprintf "%-+*o" 7 999L)) "+1747  "
    test "test4545" (lazy(sprintf "%+0o" 999L)) "+1747"
    test "test4546" (lazy(sprintf "%+05o" 999L)) "+1747"
    test "test4547" (lazy(sprintf "%+01o" 999L)) "+1747"
    test "test4548" (lazy(sprintf "%+0*o" 7 999L)) "+001747"
    test "test4549" (lazy(sprintf "%-+0o" 999L)) "+1747"
    test "test4550" (lazy(sprintf "%-+05o" 999L)) "+1747"
    test "test4551" (lazy(sprintf "%-+01o" 999L)) "+1747"
    test "test4552" (lazy(sprintf "%-+0*o" 7 999L)) "+1747  "
    test "test4553" (lazy(sprintf "% o" 999L)) " 1747"
    test "test4554" (lazy(sprintf "% 5o" 999L)) " 1747"
    test "test4555" (lazy(sprintf "% 1o" 999L)) " 1747"
    test "test4556" (lazy(sprintf "% *o" 7 999L)) "   1747"
    test "test4557" (lazy(sprintf "%- o" 999L)) " 1747"
    test "test4558" (lazy(sprintf "%- 5o" 999L)) " 1747"
    test "test4559" (lazy(sprintf "%- 1o" 999L)) " 1747"
    test "test4560" (lazy(sprintf "%- *o" 7 999L)) " 1747  "
    test "test4561" (lazy(sprintf "% 0o" 999L)) " 1747"
    test "test4562" (lazy(sprintf "% 05o" 999L)) " 1747"
    test "test4563" (lazy(sprintf "% 01o" 999L)) " 1747"
    test "test4564" (lazy(sprintf "% 0*o" 7 999L)) " 001747"
    test "test4565" (lazy(sprintf "%- 0o" 999L)) " 1747"
    test "test4566" (lazy(sprintf "%- 05o" 999L)) " 1747"
    test "test4567" (lazy(sprintf "%- 01o" 999L)) " 1747"
    test "test4568" (lazy(sprintf "%- 0*o" 7 999L)) " 1747  "
    test "test4569" (lazy(sprintf "%o" -321L)) "1777777777777777777277"
    test "test4570" (lazy(sprintf "%5o" -321L)) "1777777777777777777277"
    test "test4571" (lazy(sprintf "%1o" -321L)) "1777777777777777777277"
    test "test4572" (lazy(sprintf "%*o" 7 -321L)) "1777777777777777777277"
    test "test4573" (lazy(sprintf "%-o" -321L)) "1777777777777777777277"
    test "test4574" (lazy(sprintf "%-5o" -321L)) "1777777777777777777277"
    test "test4575" (lazy(sprintf "%-1o" -321L)) "1777777777777777777277"
    test "test4576" (lazy(sprintf "%-*o" 7 -321L)) "1777777777777777777277"
    test "test4577" (lazy(sprintf "%0o" -321L)) "1777777777777777777277"
    test "test4578" (lazy(sprintf "%05o" -321L)) "1777777777777777777277"
    test "test4579" (lazy(sprintf "%01o" -321L)) "1777777777777777777277"
    test "test4580" (lazy(sprintf "%0*o" 7 -321L)) "1777777777777777777277"
    test "test4581" (lazy(sprintf "%-0o" -321L)) "1777777777777777777277"
    test "test4582" (lazy(sprintf "%-05o" -321L)) "1777777777777777777277"
    test "test4583" (lazy(sprintf "%-01o" -321L)) "1777777777777777777277"
    test "test4584" (lazy(sprintf "%-0*o" 7 -321L)) "1777777777777777777277"
    test "test4585" (lazy(sprintf "%+o" -321L)) "+1777777777777777777277"
    test "test4586" (lazy(sprintf "%+5o" -321L)) "+1777777777777777777277"
    test "test4587" (lazy(sprintf "%+1o" -321L)) "+1777777777777777777277"
    test "test4588" (lazy(sprintf "%+*o" 7 -321L)) "+1777777777777777777277"
    test "test4589" (lazy(sprintf "%-+o" -321L)) "+1777777777777777777277"
    test "test4590" (lazy(sprintf "%-+5o" -321L)) "+1777777777777777777277"
    test "test4591" (lazy(sprintf "%-+1o" -321L)) "+1777777777777777777277"
    test "test4592" (lazy(sprintf "%-+*o" 7 -321L)) "+1777777777777777777277"
    test "test4593" (lazy(sprintf "%+0o" -321L)) "+1777777777777777777277"
    test "test4594" (lazy(sprintf "%+05o" -321L)) "+1777777777777777777277"
    test "test4595" (lazy(sprintf "%+01o" -321L)) "+1777777777777777777277"
    test "test4596" (lazy(sprintf "%+0*o" 7 -321L)) "+1777777777777777777277"
    test "test4597" (lazy(sprintf "%-+0o" -321L)) "+1777777777777777777277"
    test "test4598" (lazy(sprintf "%-+05o" -321L)) "+1777777777777777777277"
    test "test4599" (lazy(sprintf "%-+01o" -321L)) "+1777777777777777777277"
    test "test4600" (lazy(sprintf "%-+0*o" 7 -321L)) "+1777777777777777777277"
    test "test4601" (lazy(sprintf "% o" -321L)) " 1777777777777777777277"
    test "test4602" (lazy(sprintf "% 5o" -321L)) " 1777777777777777777277"
    test "test4603" (lazy(sprintf "% 1o" -321L)) " 1777777777777777777277"
    test "test4604" (lazy(sprintf "% *o" 7 -321L)) " 1777777777777777777277"
    test "test4605" (lazy(sprintf "%- o" -321L)) " 1777777777777777777277"
    test "test4606" (lazy(sprintf "%- 5o" -321L)) " 1777777777777777777277"
    test "test4607" (lazy(sprintf "%- 1o" -321L)) " 1777777777777777777277"
    test "test4608" (lazy(sprintf "%- *o" 7 -321L)) " 1777777777777777777277"
    test "test4609" (lazy(sprintf "% 0o" -321L)) " 1777777777777777777277"
    test "test4610" (lazy(sprintf "% 05o" -321L)) " 1777777777777777777277"
    test "test4611" (lazy(sprintf "% 01o" -321L)) " 1777777777777777777277"
    test "test4612" (lazy(sprintf "% 0*o" 7 -321L)) " 1777777777777777777277"
    test "test4613" (lazy(sprintf "%- 0o" -321L)) " 1777777777777777777277"
    test "test4614" (lazy(sprintf "%- 05o" -321L)) " 1777777777777777777277"
    test "test4615" (lazy(sprintf "%- 01o" -321L)) " 1777777777777777777277"
    test "test4616" (lazy(sprintf "%- 0*o" 7 -321L)) " 1777777777777777777277"
    test "test4617" (lazy(sprintf "%o" 50000UL)) "141520"
    test "test4618" (lazy(sprintf "%5o" 50000UL)) "141520"
    test "test4619" (lazy(sprintf "%1o" 50000UL)) "141520"
    test "test4620" (lazy(sprintf "%*o" 7 50000UL)) " 141520"
    test "test4621" (lazy(sprintf "%-o" 50000UL)) "141520"
    test "test4622" (lazy(sprintf "%-5o" 50000UL)) "141520"
    test "test4623" (lazy(sprintf "%-1o" 50000UL)) "141520"
    test "test4624" (lazy(sprintf "%-*o" 7 50000UL)) "141520 "
    test "test4625" (lazy(sprintf "%0o" 50000UL)) "141520"
    test "test4626" (lazy(sprintf "%05o" 50000UL)) "141520"
    test "test4627" (lazy(sprintf "%01o" 50000UL)) "141520"
    test "test4628" (lazy(sprintf "%0*o" 7 50000UL)) "0141520"
    test "test4629" (lazy(sprintf "%-0o" 50000UL)) "141520"
    test "test4630" (lazy(sprintf "%-05o" 50000UL)) "141520"
    test "test4631" (lazy(sprintf "%-01o" 50000UL)) "141520"
    test "test4632" (lazy(sprintf "%-0*o" 7 50000UL)) "141520 "
    test "test4633" (lazy(sprintf "%+o" 50000UL)) "+141520"
    test "test4634" (lazy(sprintf "%+5o" 50000UL)) "+141520"
    test "test4635" (lazy(sprintf "%+1o" 50000UL)) "+141520"
    test "test4636" (lazy(sprintf "%+*o" 7 50000UL)) "+141520"
    test "test4637" (lazy(sprintf "%-+o" 50000UL)) "+141520"
    test "test4638" (lazy(sprintf "%-+5o" 50000UL)) "+141520"
    test "test4639" (lazy(sprintf "%-+1o" 50000UL)) "+141520"
    test "test4640" (lazy(sprintf "%-+*o" 7 50000UL)) "+141520"
    test "test4641" (lazy(sprintf "%+0o" 50000UL)) "+141520"
    test "test4642" (lazy(sprintf "%+05o" 50000UL)) "+141520"
    test "test4643" (lazy(sprintf "%+01o" 50000UL)) "+141520"
    test "test4644" (lazy(sprintf "%+0*o" 7 50000UL)) "+141520"
    test "test4645" (lazy(sprintf "%-+0o" 50000UL)) "+141520"
    test "test4646" (lazy(sprintf "%-+05o" 50000UL)) "+141520"
    test "test4647" (lazy(sprintf "%-+01o" 50000UL)) "+141520"
    test "test4648" (lazy(sprintf "%-+0*o" 7 50000UL)) "+141520"
    test "test4649" (lazy(sprintf "% o" 50000UL)) " 141520"
    test "test4650" (lazy(sprintf "% 5o" 50000UL)) " 141520"
    test "test4651" (lazy(sprintf "% 1o" 50000UL)) " 141520"
    test "test4652" (lazy(sprintf "% *o" 7 50000UL)) " 141520"
    test "test4653" (lazy(sprintf "%- o" 50000UL)) " 141520"
    test "test4654" (lazy(sprintf "%- 5o" 50000UL)) " 141520"
    test "test4655" (lazy(sprintf "%- 1o" 50000UL)) " 141520"
    test "test4656" (lazy(sprintf "%- *o" 7 50000UL)) " 141520"
    test "test4657" (lazy(sprintf "% 0o" 50000UL)) " 141520"
    test "test4658" (lazy(sprintf "% 05o" 50000UL)) " 141520"
    test "test4659" (lazy(sprintf "% 01o" 50000UL)) " 141520"
    test "test4660" (lazy(sprintf "% 0*o" 7 50000UL)) " 141520"
    test "test4661" (lazy(sprintf "%- 0o" 50000UL)) " 141520"
    test "test4662" (lazy(sprintf "%- 05o" 50000UL)) " 141520"
    test "test4663" (lazy(sprintf "%- 01o" 50000UL)) " 141520"
    test "test4664" (lazy(sprintf "%- 0*o" 7 50000UL)) " 141520"
    test "test4665" (lazy(sprintf "%o" System.Int32.MaxValue)) "17777777777"
    test "test4666" (lazy(sprintf "%5o" System.Int32.MaxValue)) "17777777777"
    test "test4667" (lazy(sprintf "%1o" System.Int32.MaxValue)) "17777777777"
    test "test4668" (lazy(sprintf "%*o" 7 System.Int32.MaxValue)) "17777777777"
    test "test4669" (lazy(sprintf "%-o" System.Int32.MaxValue)) "17777777777"
    test "test4670" (lazy(sprintf "%-5o" System.Int32.MaxValue)) "17777777777"
    test "test4671" (lazy(sprintf "%-1o" System.Int32.MaxValue)) "17777777777"
    test "test4672" (lazy(sprintf "%-*o" 7 System.Int32.MaxValue)) "17777777777"
    test "test4673" (lazy(sprintf "%0o" System.Int32.MaxValue)) "17777777777"
    test "test4674" (lazy(sprintf "%05o" System.Int32.MaxValue)) "17777777777"
    test "test4675" (lazy(sprintf "%01o" System.Int32.MaxValue)) "17777777777"
    test "test4676" (lazy(sprintf "%0*o" 7 System.Int32.MaxValue)) "17777777777"
    test "test4677" (lazy(sprintf "%-0o" System.Int32.MaxValue)) "17777777777"
    test "test4678" (lazy(sprintf "%-05o" System.Int32.MaxValue)) "17777777777"
    test "test4679" (lazy(sprintf "%-01o" System.Int32.MaxValue)) "17777777777"
    test "test4680" (lazy(sprintf "%-0*o" 7 System.Int32.MaxValue)) "17777777777"
    test "test4681" (lazy(sprintf "%+o" System.Int32.MaxValue)) "+17777777777"
    test "test4682" (lazy(sprintf "%+5o" System.Int32.MaxValue)) "+17777777777"
    test "test4683" (lazy(sprintf "%+1o" System.Int32.MaxValue)) "+17777777777"
    test "test4684" (lazy(sprintf "%+*o" 7 System.Int32.MaxValue)) "+17777777777"
    test "test4685" (lazy(sprintf "%-+o" System.Int32.MaxValue)) "+17777777777"
    test "test4686" (lazy(sprintf "%-+5o" System.Int32.MaxValue)) "+17777777777"
    test "test4687" (lazy(sprintf "%-+1o" System.Int32.MaxValue)) "+17777777777"
    test "test4688" (lazy(sprintf "%-+*o" 7 System.Int32.MaxValue)) "+17777777777"
    test "test4689" (lazy(sprintf "%+0o" System.Int32.MaxValue)) "+17777777777"
    test "test4690" (lazy(sprintf "%+05o" System.Int32.MaxValue)) "+17777777777"
    test "test4691" (lazy(sprintf "%+01o" System.Int32.MaxValue)) "+17777777777"
    test "test4692" (lazy(sprintf "%+0*o" 7 System.Int32.MaxValue)) "+17777777777"
    test "test4693" (lazy(sprintf "%-+0o" System.Int32.MaxValue)) "+17777777777"
    test "test4694" (lazy(sprintf "%-+05o" System.Int32.MaxValue)) "+17777777777"
    test "test4695" (lazy(sprintf "%-+01o" System.Int32.MaxValue)) "+17777777777"
    test "test4696" (lazy(sprintf "%-+0*o" 7 System.Int32.MaxValue)) "+17777777777"
    test "test4697" (lazy(sprintf "% o" System.Int32.MaxValue)) " 17777777777"
    test "test4698" (lazy(sprintf "% 5o" System.Int32.MaxValue)) " 17777777777"
    test "test4699" (lazy(sprintf "% 1o" System.Int32.MaxValue)) " 17777777777"
    test "test4700" (lazy(sprintf "% *o" 7 System.Int32.MaxValue)) " 17777777777"
    test "test4701" (lazy(sprintf "%- o" System.Int32.MaxValue)) " 17777777777"
    test "test4702" (lazy(sprintf "%- 5o" System.Int32.MaxValue)) " 17777777777"
    test "test4703" (lazy(sprintf "%- 1o" System.Int32.MaxValue)) " 17777777777"
    test "test4704" (lazy(sprintf "%- *o" 7 System.Int32.MaxValue)) " 17777777777"
    test "test4705" (lazy(sprintf "% 0o" System.Int32.MaxValue)) " 17777777777"
    test "test4706" (lazy(sprintf "% 05o" System.Int32.MaxValue)) " 17777777777"
    test "test4707" (lazy(sprintf "% 01o" System.Int32.MaxValue)) " 17777777777"
    test "test4708" (lazy(sprintf "% 0*o" 7 System.Int32.MaxValue)) " 17777777777"
    test "test4709" (lazy(sprintf "%- 0o" System.Int32.MaxValue)) " 17777777777"
    test "test4710" (lazy(sprintf "%- 05o" System.Int32.MaxValue)) " 17777777777"
    test "test4711" (lazy(sprintf "%- 01o" System.Int32.MaxValue)) " 17777777777"
    test "test4712" (lazy(sprintf "%- 0*o" 7 System.Int32.MaxValue)) " 17777777777"
    test "test4713" (lazy(sprintf "%o" System.Int64.MaxValue)) "777777777777777777777"
    test "test4714" (lazy(sprintf "%5o" System.Int64.MaxValue)) "777777777777777777777"
    test "test4715" (lazy(sprintf "%1o" System.Int64.MaxValue)) "777777777777777777777"
    test "test4716" (lazy(sprintf "%*o" 7 System.Int64.MaxValue)) "777777777777777777777"
    test "test4717" (lazy(sprintf "%-o" System.Int64.MaxValue)) "777777777777777777777"
    test "test4718" (lazy(sprintf "%-5o" System.Int64.MaxValue)) "777777777777777777777"
    test "test4719" (lazy(sprintf "%-1o" System.Int64.MaxValue)) "777777777777777777777"
    test "test4720" (lazy(sprintf "%-*o" 7 System.Int64.MaxValue)) "777777777777777777777"
    test "test4721" (lazy(sprintf "%0o" System.Int64.MaxValue)) "777777777777777777777"
    test "test4722" (lazy(sprintf "%05o" System.Int64.MaxValue)) "777777777777777777777"
    test "test4723" (lazy(sprintf "%01o" System.Int64.MaxValue)) "777777777777777777777"
    test "test4724" (lazy(sprintf "%0*o" 7 System.Int64.MaxValue)) "777777777777777777777"
    test "test4725" (lazy(sprintf "%-0o" System.Int64.MaxValue)) "777777777777777777777"
    test "test4726" (lazy(sprintf "%-05o" System.Int64.MaxValue)) "777777777777777777777"
    test "test4727" (lazy(sprintf "%-01o" System.Int64.MaxValue)) "777777777777777777777"
    test "test4728" (lazy(sprintf "%-0*o" 7 System.Int64.MaxValue)) "777777777777777777777"
    test "test4729" (lazy(sprintf "%+o" System.Int64.MaxValue)) "+777777777777777777777"
    test "test4730" (lazy(sprintf "%+5o" System.Int64.MaxValue)) "+777777777777777777777"
    test "test4731" (lazy(sprintf "%+1o" System.Int64.MaxValue)) "+777777777777777777777"
    test "test4732" (lazy(sprintf "%+*o" 7 System.Int64.MaxValue)) "+777777777777777777777"
    test "test4733" (lazy(sprintf "%-+o" System.Int64.MaxValue)) "+777777777777777777777"
    test "test4734" (lazy(sprintf "%-+5o" System.Int64.MaxValue)) "+777777777777777777777"
    test "test4735" (lazy(sprintf "%-+1o" System.Int64.MaxValue)) "+777777777777777777777"
    test "test4736" (lazy(sprintf "%-+*o" 7 System.Int64.MaxValue)) "+777777777777777777777"
    test "test4737" (lazy(sprintf "%+0o" System.Int64.MaxValue)) "+777777777777777777777"
    test "test4738" (lazy(sprintf "%+05o" System.Int64.MaxValue)) "+777777777777777777777"
    test "test4739" (lazy(sprintf "%+01o" System.Int64.MaxValue)) "+777777777777777777777"
    test "test4740" (lazy(sprintf "%+0*o" 7 System.Int64.MaxValue)) "+777777777777777777777"
    test "test4741" (lazy(sprintf "%-+0o" System.Int64.MaxValue)) "+777777777777777777777"
    test "test4742" (lazy(sprintf "%-+05o" System.Int64.MaxValue)) "+777777777777777777777"
    test "test4743" (lazy(sprintf "%-+01o" System.Int64.MaxValue)) "+777777777777777777777"
    test "test4744" (lazy(sprintf "%-+0*o" 7 System.Int64.MaxValue)) "+777777777777777777777"
    test "test4745" (lazy(sprintf "% o" System.Int64.MaxValue)) " 777777777777777777777"
    test "test4746" (lazy(sprintf "% 5o" System.Int64.MaxValue)) " 777777777777777777777"
    test "test4747" (lazy(sprintf "% 1o" System.Int64.MaxValue)) " 777777777777777777777"
    test "test4748" (lazy(sprintf "% *o" 7 System.Int64.MaxValue)) " 777777777777777777777"
    test "test4749" (lazy(sprintf "%- o" System.Int64.MaxValue)) " 777777777777777777777"
    test "test4750" (lazy(sprintf "%- 5o" System.Int64.MaxValue)) " 777777777777777777777"
    test "test4751" (lazy(sprintf "%- 1o" System.Int64.MaxValue)) " 777777777777777777777"
    test "test4752" (lazy(sprintf "%- *o" 7 System.Int64.MaxValue)) " 777777777777777777777"
    test "test4753" (lazy(sprintf "% 0o" System.Int64.MaxValue)) " 777777777777777777777"
    test "test4754" (lazy(sprintf "% 05o" System.Int64.MaxValue)) " 777777777777777777777"
    test "test4755" (lazy(sprintf "% 01o" System.Int64.MaxValue)) " 777777777777777777777"
    test "test4756" (lazy(sprintf "% 0*o" 7 System.Int64.MaxValue)) " 777777777777777777777"
    test "test4757" (lazy(sprintf "%- 0o" System.Int64.MaxValue)) " 777777777777777777777"
    test "test4758" (lazy(sprintf "%- 05o" System.Int64.MaxValue)) " 777777777777777777777"
    test "test4759" (lazy(sprintf "%- 01o" System.Int64.MaxValue)) " 777777777777777777777"
    test "test4760" (lazy(sprintf "%- 0*o" 7 System.Int64.MaxValue)) " 777777777777777777777"
    test "test4761" (lazy(sprintf "%o" System.Int32.MinValue)) "20000000000"
    test "test4762" (lazy(sprintf "%5o" System.Int32.MinValue)) "20000000000"
    test "test4763" (lazy(sprintf "%1o" System.Int32.MinValue)) "20000000000"
    test "test4764" (lazy(sprintf "%*o" 7 System.Int32.MinValue)) "20000000000"
    test "test4765" (lazy(sprintf "%-o" System.Int32.MinValue)) "20000000000"
    test "test4766" (lazy(sprintf "%-5o" System.Int32.MinValue)) "20000000000"
    test "test4767" (lazy(sprintf "%-1o" System.Int32.MinValue)) "20000000000"
    test "test4768" (lazy(sprintf "%-*o" 7 System.Int32.MinValue)) "20000000000"
    test "test4769" (lazy(sprintf "%0o" System.Int32.MinValue)) "20000000000"
    test "test4770" (lazy(sprintf "%05o" System.Int32.MinValue)) "20000000000"
    test "test4771" (lazy(sprintf "%01o" System.Int32.MinValue)) "20000000000"
    test "test4772" (lazy(sprintf "%0*o" 7 System.Int32.MinValue)) "20000000000"
    test "test4773" (lazy(sprintf "%-0o" System.Int32.MinValue)) "20000000000"
    test "test4774" (lazy(sprintf "%-05o" System.Int32.MinValue)) "20000000000"
    test "test4775" (lazy(sprintf "%-01o" System.Int32.MinValue)) "20000000000"
    test "test4776" (lazy(sprintf "%-0*o" 7 System.Int32.MinValue)) "20000000000"
    test "test4777" (lazy(sprintf "%+o" System.Int32.MinValue)) "+20000000000"
    test "test4778" (lazy(sprintf "%+5o" System.Int32.MinValue)) "+20000000000"
    test "test4779" (lazy(sprintf "%+1o" System.Int32.MinValue)) "+20000000000"
    test "test4780" (lazy(sprintf "%+*o" 7 System.Int32.MinValue)) "+20000000000"
    test "test4781" (lazy(sprintf "%-+o" System.Int32.MinValue)) "+20000000000"
    test "test4782" (lazy(sprintf "%-+5o" System.Int32.MinValue)) "+20000000000"
    test "test4783" (lazy(sprintf "%-+1o" System.Int32.MinValue)) "+20000000000"
    test "test4784" (lazy(sprintf "%-+*o" 7 System.Int32.MinValue)) "+20000000000"
    test "test4785" (lazy(sprintf "%+0o" System.Int32.MinValue)) "+20000000000"
    test "test4786" (lazy(sprintf "%+05o" System.Int32.MinValue)) "+20000000000"
    test "test4787" (lazy(sprintf "%+01o" System.Int32.MinValue)) "+20000000000"
    test "test4788" (lazy(sprintf "%+0*o" 7 System.Int32.MinValue)) "+20000000000"
    test "test4789" (lazy(sprintf "%-+0o" System.Int32.MinValue)) "+20000000000"
    test "test4790" (lazy(sprintf "%-+05o" System.Int32.MinValue)) "+20000000000"
    test "test4791" (lazy(sprintf "%-+01o" System.Int32.MinValue)) "+20000000000"
    test "test4792" (lazy(sprintf "%-+0*o" 7 System.Int32.MinValue)) "+20000000000"
    test "test4793" (lazy(sprintf "% o" System.Int32.MinValue)) " 20000000000"
    test "test4794" (lazy(sprintf "% 5o" System.Int32.MinValue)) " 20000000000"
    test "test4795" (lazy(sprintf "% 1o" System.Int32.MinValue)) " 20000000000"
    test "test4796" (lazy(sprintf "% *o" 7 System.Int32.MinValue)) " 20000000000"
    test "test4797" (lazy(sprintf "%- o" System.Int32.MinValue)) " 20000000000"
    test "test4798" (lazy(sprintf "%- 5o" System.Int32.MinValue)) " 20000000000"
    test "test4799" (lazy(sprintf "%- 1o" System.Int32.MinValue)) " 20000000000"
    test "test4800" (lazy(sprintf "%- *o" 7 System.Int32.MinValue)) " 20000000000"
    test "test4801" (lazy(sprintf "% 0o" System.Int32.MinValue)) " 20000000000"
    test "test4802" (lazy(sprintf "% 05o" System.Int32.MinValue)) " 20000000000"
    test "test4803" (lazy(sprintf "% 01o" System.Int32.MinValue)) " 20000000000"
    test "test4804" (lazy(sprintf "% 0*o" 7 System.Int32.MinValue)) " 20000000000"
    test "test4805" (lazy(sprintf "%- 0o" System.Int32.MinValue)) " 20000000000"
    test "test4806" (lazy(sprintf "%- 05o" System.Int32.MinValue)) " 20000000000"
    test "test4807" (lazy(sprintf "%- 01o" System.Int32.MinValue)) " 20000000000"
    test "test4808" (lazy(sprintf "%- 0*o" 7 System.Int32.MinValue)) " 20000000000"
    test "test4809" (lazy(sprintf "%o" System.Int64.MinValue)) "1000000000000000000000"
    test "test4810" (lazy(sprintf "%5o" System.Int64.MinValue)) "1000000000000000000000"
    test "test4811" (lazy(sprintf "%1o" System.Int64.MinValue)) "1000000000000000000000"
    test "test4812" (lazy(sprintf "%*o" 7 System.Int64.MinValue)) "1000000000000000000000"
    test "test4813" (lazy(sprintf "%-o" System.Int64.MinValue)) "1000000000000000000000"
    test "test4814" (lazy(sprintf "%-5o" System.Int64.MinValue)) "1000000000000000000000"
    test "test4815" (lazy(sprintf "%-1o" System.Int64.MinValue)) "1000000000000000000000"
    test "test4816" (lazy(sprintf "%-*o" 7 System.Int64.MinValue)) "1000000000000000000000"
    test "test4817" (lazy(sprintf "%0o" System.Int64.MinValue)) "1000000000000000000000"
    test "test4818" (lazy(sprintf "%05o" System.Int64.MinValue)) "1000000000000000000000"
    test "test4819" (lazy(sprintf "%01o" System.Int64.MinValue)) "1000000000000000000000"
    test "test4820" (lazy(sprintf "%0*o" 7 System.Int64.MinValue)) "1000000000000000000000"
    test "test4821" (lazy(sprintf "%-0o" System.Int64.MinValue)) "1000000000000000000000"
    test "test4822" (lazy(sprintf "%-05o" System.Int64.MinValue)) "1000000000000000000000"
    test "test4823" (lazy(sprintf "%-01o" System.Int64.MinValue)) "1000000000000000000000"
    test "test4824" (lazy(sprintf "%-0*o" 7 System.Int64.MinValue)) "1000000000000000000000"
    test "test4825" (lazy(sprintf "%+o" System.Int64.MinValue)) "+1000000000000000000000"
    test "test4826" (lazy(sprintf "%+5o" System.Int64.MinValue)) "+1000000000000000000000"
    test "test4827" (lazy(sprintf "%+1o" System.Int64.MinValue)) "+1000000000000000000000"
    test "test4828" (lazy(sprintf "%+*o" 7 System.Int64.MinValue)) "+1000000000000000000000"
    test "test4829" (lazy(sprintf "%-+o" System.Int64.MinValue)) "+1000000000000000000000"
    test "test4830" (lazy(sprintf "%-+5o" System.Int64.MinValue)) "+1000000000000000000000"
    test "test4831" (lazy(sprintf "%-+1o" System.Int64.MinValue)) "+1000000000000000000000"
    test "test4832" (lazy(sprintf "%-+*o" 7 System.Int64.MinValue)) "+1000000000000000000000"
    test "test4833" (lazy(sprintf "%+0o" System.Int64.MinValue)) "+1000000000000000000000"
    test "test4834" (lazy(sprintf "%+05o" System.Int64.MinValue)) "+1000000000000000000000"
    test "test4835" (lazy(sprintf "%+01o" System.Int64.MinValue)) "+1000000000000000000000"
    test "test4836" (lazy(sprintf "%+0*o" 7 System.Int64.MinValue)) "+1000000000000000000000"
    test "test4837" (lazy(sprintf "%-+0o" System.Int64.MinValue)) "+1000000000000000000000"
    test "test4838" (lazy(sprintf "%-+05o" System.Int64.MinValue)) "+1000000000000000000000"
    test "test4839" (lazy(sprintf "%-+01o" System.Int64.MinValue)) "+1000000000000000000000"
    test "test4840" (lazy(sprintf "%-+0*o" 7 System.Int64.MinValue)) "+1000000000000000000000"
    test "test4841" (lazy(sprintf "% o" System.Int64.MinValue)) " 1000000000000000000000"
    test "test4842" (lazy(sprintf "% 5o" System.Int64.MinValue)) " 1000000000000000000000"
    test "test4843" (lazy(sprintf "% 1o" System.Int64.MinValue)) " 1000000000000000000000"
    test "test4844" (lazy(sprintf "% *o" 7 System.Int64.MinValue)) " 1000000000000000000000"
    test "test4845" (lazy(sprintf "%- o" System.Int64.MinValue)) " 1000000000000000000000"
    test "test4846" (lazy(sprintf "%- 5o" System.Int64.MinValue)) " 1000000000000000000000"
    test "test4847" (lazy(sprintf "%- 1o" System.Int64.MinValue)) " 1000000000000000000000"
    test "test4848" (lazy(sprintf "%- *o" 7 System.Int64.MinValue)) " 1000000000000000000000"
    test "test4849" (lazy(sprintf "% 0o" System.Int64.MinValue)) " 1000000000000000000000"
    test "test4850" (lazy(sprintf "% 05o" System.Int64.MinValue)) " 1000000000000000000000"
    test "test4851" (lazy(sprintf "% 01o" System.Int64.MinValue)) " 1000000000000000000000"
    test "test4852" (lazy(sprintf "% 0*o" 7 System.Int64.MinValue)) " 1000000000000000000000"
    test "test4853" (lazy(sprintf "%- 0o" System.Int64.MinValue)) " 1000000000000000000000"
    test "test4854" (lazy(sprintf "%- 05o" System.Int64.MinValue)) " 1000000000000000000000"
    test "test4855" (lazy(sprintf "%- 01o" System.Int64.MinValue)) " 1000000000000000000000"
    test "test4856" (lazy(sprintf "%- 0*o" 7 System.Int64.MinValue)) " 1000000000000000000000"
    test "test4857" (lazy(sprintf "%o" 55n)) "67"
    test "test4858" (lazy(sprintf "%5o" 55n)) "   67"
    test "test4859" (lazy(sprintf "%1o" 55n)) "67"
    test "test4860" (lazy(sprintf "%*o" 7 55n)) "     67"
    test "test4861" (lazy(sprintf "%-o" 55n)) "67"
    test "test4862" (lazy(sprintf "%-5o" 55n)) "67   "
    test "test4863" (lazy(sprintf "%-1o" 55n)) "67"
    test "test4864" (lazy(sprintf "%-*o" 7 55n)) "67     "
    test "test4865" (lazy(sprintf "%0o" 55n)) "67"
    test "test4866" (lazy(sprintf "%05o" 55n)) "00067"
    test "test4867" (lazy(sprintf "%01o" 55n)) "67"
    test "test4868" (lazy(sprintf "%0*o" 7 55n)) "0000067"
    test "test4869" (lazy(sprintf "%-0o" 55n)) "67"
    test "test4870" (lazy(sprintf "%-05o" 55n)) "67   "
    test "test4871" (lazy(sprintf "%-01o" 55n)) "67"
    test "test4872" (lazy(sprintf "%-0*o" 7 55n)) "67     "
    test "test4873" (lazy(sprintf "%+o" 55n)) "+67"
    test "test4874" (lazy(sprintf "%+5o" 55n)) "  +67"
    test "test4875" (lazy(sprintf "%+1o" 55n)) "+67"
    test "test4876" (lazy(sprintf "%+*o" 7 55n)) "    +67"
    test "test4877" (lazy(sprintf "%-+o" 55n)) "+67"
    test "test4878" (lazy(sprintf "%-+5o" 55n)) "+67  "
    test "test4879" (lazy(sprintf "%-+1o" 55n)) "+67"
    test "test4880" (lazy(sprintf "%-+*o" 7 55n)) "+67    "
    test "test4881" (lazy(sprintf "%+0o" 55n)) "+67"
    test "test4882" (lazy(sprintf "%+05o" 55n)) "+0067"
    test "test4883" (lazy(sprintf "%+01o" 55n)) "+67"
    test "test4884" (lazy(sprintf "%+0*o" 7 55n)) "+000067"
    test "test4885" (lazy(sprintf "%-+0o" 55n)) "+67"
    test "test4886" (lazy(sprintf "%-+05o" 55n)) "+67  "
    test "test4887" (lazy(sprintf "%-+01o" 55n)) "+67"
    test "test4888" (lazy(sprintf "%-+0*o" 7 55n)) "+67    "
    test "test4889" (lazy(sprintf "% o" 55n)) " 67"
    test "test4890" (lazy(sprintf "% 5o" 55n)) "   67"
    test "test4891" (lazy(sprintf "% 1o" 55n)) " 67"
    test "test4892" (lazy(sprintf "% *o" 7 55n)) "     67"
    test "test4893" (lazy(sprintf "%- o" 55n)) " 67"
    test "test4894" (lazy(sprintf "%- 5o" 55n)) " 67  "
    test "test4895" (lazy(sprintf "%- 1o" 55n)) " 67"
    test "test4896" (lazy(sprintf "%- *o" 7 55n)) " 67    "
    test "test4897" (lazy(sprintf "% 0o" 55n)) " 67"
    test "test4898" (lazy(sprintf "% 05o" 55n)) " 0067"
    test "test4899" (lazy(sprintf "% 01o" 55n)) " 67"
    test "test4900" (lazy(sprintf "% 0*o" 7 55n)) " 000067"
    test "test4901" (lazy(sprintf "%- 0o" 55n)) " 67"
    test "test4902" (lazy(sprintf "%- 05o" 55n)) " 67  "
    test "test4903" (lazy(sprintf "%- 01o" 55n)) " 67"
    test "test4904" (lazy(sprintf "%- 0*o" 7 55n)) " 67    "
    test "test4905" (lazy(sprintf "%o" 999un)) "1747"
    test "test4906" (lazy(sprintf "%5o" 999un)) " 1747"
    test "test4907" (lazy(sprintf "%1o" 999un)) "1747"
    test "test4908" (lazy(sprintf "%*o" 7 999un)) "   1747"
    test "test4909" (lazy(sprintf "%-o" 999un)) "1747"
    test "test4910" (lazy(sprintf "%-5o" 999un)) "1747 "
    test "test4911" (lazy(sprintf "%-1o" 999un)) "1747"
    test "test4912" (lazy(sprintf "%-*o" 7 999un)) "1747   "
    test "test4913" (lazy(sprintf "%0o" 999un)) "1747"
    test "test4914" (lazy(sprintf "%05o" 999un)) "01747"
    test "test4915" (lazy(sprintf "%01o" 999un)) "1747"
    test "test4916" (lazy(sprintf "%0*o" 7 999un)) "0001747"
    test "test4917" (lazy(sprintf "%-0o" 999un)) "1747"
    test "test4918" (lazy(sprintf "%-05o" 999un)) "1747 "
    test "test4919" (lazy(sprintf "%-01o" 999un)) "1747"
    test "test4920" (lazy(sprintf "%-0*o" 7 999un)) "1747   "
    test "test4921" (lazy(sprintf "%+o" 999un)) "+1747"
    test "test4922" (lazy(sprintf "%+5o" 999un)) "+1747"
    test "test4923" (lazy(sprintf "%+1o" 999un)) "+1747"
    test "test4924" (lazy(sprintf "%+*o" 7 999un)) "  +1747"
    test "test4925" (lazy(sprintf "%-+o" 999un)) "+1747"
    test "test4926" (lazy(sprintf "%-+5o" 999un)) "+1747"
    test "test4927" (lazy(sprintf "%-+1o" 999un)) "+1747"
    test "test4928" (lazy(sprintf "%-+*o" 7 999un)) "+1747  "
    test "test4929" (lazy(sprintf "%+0o" 999un)) "+1747"
    test "test4930" (lazy(sprintf "%+05o" 999un)) "+1747"
    test "test4931" (lazy(sprintf "%+01o" 999un)) "+1747"
    test "test4932" (lazy(sprintf "%+0*o" 7 999un)) "+001747"
    test "test4933" (lazy(sprintf "%-+0o" 999un)) "+1747"
    test "test4934" (lazy(sprintf "%-+05o" 999un)) "+1747"
    test "test4935" (lazy(sprintf "%-+01o" 999un)) "+1747"
    test "test4936" (lazy(sprintf "%-+0*o" 7 999un)) "+1747  "
    test "test4937" (lazy(sprintf "% o" 999un)) " 1747"
    test "test4938" (lazy(sprintf "% 5o" 999un)) " 1747"
    test "test4939" (lazy(sprintf "% 1o" 999un)) " 1747"
    test "test4940" (lazy(sprintf "% *o" 7 999un)) "   1747"
    test "test4941" (lazy(sprintf "%- o" 999un)) " 1747"
    test "test4942" (lazy(sprintf "%- 5o" 999un)) " 1747"
    test "test4943" (lazy(sprintf "%- 1o" 999un)) " 1747"
    test "test4944" (lazy(sprintf "%- *o" 7 999un)) " 1747  "
    test "test4945" (lazy(sprintf "% 0o" 999un)) " 1747"
    test "test4946" (lazy(sprintf "% 05o" 999un)) " 1747"
    test "test4947" (lazy(sprintf "% 01o" 999un)) " 1747"
    test "test4948" (lazy(sprintf "% 0*o" 7 999un)) " 001747"
    test "test4949" (lazy(sprintf "%- 0o" 999un)) " 1747"
    test "test4950" (lazy(sprintf "%- 05o" 999un)) " 1747"
    test "test4951" (lazy(sprintf "%- 01o" 999un)) " 1747"
    test "test4952" (lazy(sprintf "%- 0*o" 7 999un)) " 1747  "
    test "test4953" (lazy(sprintf "%e" 5.0)) "5.000000e+000"
    test "test4954" (lazy(sprintf "%5e" 5.0)) "5.000000e+000"
    test "test4955" (lazy(sprintf "%1e" 5.0)) "5.000000e+000"
    test "test4956" (lazy(sprintf "%*e" 7 5.0)) "5.000000e+000"
    test "test4957" (lazy(sprintf "%.5e" 5.0)) "5.00000e+000"
    test "test4958" (lazy(sprintf "%.*e" 4 5.0)) "5.0000e+000"
    test "test4959" (lazy(sprintf "%*.*e" 5 4 5.0)) "5.0000e+000"
    test "test4960" (lazy(sprintf "%-e" 5.0)) "5.000000e+000"
    test "test4961" (lazy(sprintf "%-5e" 5.0)) "5.000000e+000"
    test "test4962" (lazy(sprintf "%-1e" 5.0)) "5.000000e+000"
    test "test4963" (lazy(sprintf "%-*e" 7 5.0)) "5.000000e+000"
    test "test4964" (lazy(sprintf "%-.5e" 5.0)) "5.00000e+000"
    test "test4965" (lazy(sprintf "%-.*e" 4 5.0)) "5.0000e+000"
    test "test4966" (lazy(sprintf "%-*.*e" 5 4 5.0)) "5.0000e+000"
    test "test4967" (lazy(sprintf "%0e" 5.0)) "5.000000e+000"
    test "test4968" (lazy(sprintf "%05e" 5.0)) "5.000000e+000"
    test "test4969" (lazy(sprintf "%01e" 5.0)) "5.000000e+000"
    test "test4970" (lazy(sprintf "%0*e" 7 5.0)) "5.000000e+000"
    test "test4971" (lazy(sprintf "%0.5e" 5.0)) "5.00000e+000"
    test "test4972" (lazy(sprintf "%0.*e" 4 5.0)) "5.0000e+000"
    test "test4973" (lazy(sprintf "%0*.*e" 5 4 5.0)) "5.0000e+000"
    test "test4974" (lazy(sprintf "%-0e" 5.0)) "5.000000e+000"
    test "test4975" (lazy(sprintf "%-05e" 5.0)) "5.000000e+000"
    test "test4976" (lazy(sprintf "%-01e" 5.0)) "5.000000e+000"
    test "test4977" (lazy(sprintf "%-0*e" 7 5.0)) "5.000000e+000"
    test "test4978" (lazy(sprintf "%-0.5e" 5.0)) "5.00000e+000"
    test "test4979" (lazy(sprintf "%-0.*e" 4 5.0)) "5.0000e+000"
    test "test4980" (lazy(sprintf "%-0*.*e" 5 4 5.0)) "5.0000e+000"
    test "test4981" (lazy(sprintf "%+e" 5.0)) "+5.000000e+000"
    test "test4982" (lazy(sprintf "%+5e" 5.0)) "+5.000000e+000"
    test "test4983" (lazy(sprintf "%+1e" 5.0)) "+5.000000e+000"
    test "test4984" (lazy(sprintf "%+*e" 7 5.0)) "+5.000000e+000"
    test "test4985" (lazy(sprintf "%+.5e" 5.0)) "+5.00000e+000"
    test "test4986" (lazy(sprintf "%+.*e" 4 5.0)) "+5.0000e+000"
    test "test4987" (lazy(sprintf "%+*.*e" 5 4 5.0)) "+5.0000e+000"
    test "test4988" (lazy(sprintf "%-+e" 5.0)) "+5.000000e+000"
    test "test4989" (lazy(sprintf "%-+5e" 5.0)) "+5.000000e+000"
    test "test4990" (lazy(sprintf "%-+1e" 5.0)) "+5.000000e+000"
    test "test4991" (lazy(sprintf "%-+*e" 7 5.0)) "+5.000000e+000"
    test "test4992" (lazy(sprintf "%-+.5e" 5.0)) "+5.00000e+000"
    test "test4993" (lazy(sprintf "%-+.*e" 4 5.0)) "+5.0000e+000"
    test "test4994" (lazy(sprintf "%-+*.*e" 5 4 5.0)) "+5.0000e+000"
    test "test4995" (lazy(sprintf "%+0e" 5.0)) "+5.000000e+000"
    test "test4996" (lazy(sprintf "%+05e" 5.0)) "+5.000000e+000"
    test "test4997" (lazy(sprintf "%+01e" 5.0)) "+5.000000e+000"
    test "test4998" (lazy(sprintf "%+0*e" 7 5.0)) "+5.000000e+000"
    test "test4999" (lazy(sprintf "%+0.5e" 5.0)) "+5.00000e+000"
    test "test5000" (lazy(sprintf "%+0.*e" 4 5.0)) "+5.0000e+000"
let func5000()=
    test "test5001" (lazy(sprintf "%+0*.*e" 5 4 5.0)) "+5.0000e+000"
    test "test5002" (lazy(sprintf "%-+0e" 5.0)) "+5.000000e+000"
    test "test5003" (lazy(sprintf "%-+05e" 5.0)) "+5.000000e+000"
    test "test5004" (lazy(sprintf "%-+01e" 5.0)) "+5.000000e+000"
    test "test5005" (lazy(sprintf "%-+0*e" 7 5.0)) "+5.000000e+000"
    test "test5006" (lazy(sprintf "%-+0.5e" 5.0)) "+5.00000e+000"
    test "test5007" (lazy(sprintf "%-+0.*e" 4 5.0)) "+5.0000e+000"
    test "test5008" (lazy(sprintf "%-+0*.*e" 5 4 5.0)) "+5.0000e+000"
    test "test5009" (lazy(sprintf "% e" 5.0)) " 5.000000e+000"
    test "test5010" (lazy(sprintf "% 5e" 5.0)) " 5.000000e+000"
    test "test5011" (lazy(sprintf "% 1e" 5.0)) " 5.000000e+000"
    test "test5012" (lazy(sprintf "% *e" 7 5.0)) " 5.000000e+000"
    test "test5013" (lazy(sprintf "% .5e" 5.0)) " 5.00000e+000"
    test "test5014" (lazy(sprintf "% .*e" 4 5.0)) " 5.0000e+000"
    test "test5015" (lazy(sprintf "% *.*e" 5 4 5.0)) " 5.0000e+000"
    test "test5016" (lazy(sprintf "%- e" 5.0)) " 5.000000e+000"
    test "test5017" (lazy(sprintf "%- 5e" 5.0)) " 5.000000e+000"
    test "test5018" (lazy(sprintf "%- 1e" 5.0)) " 5.000000e+000"
    test "test5019" (lazy(sprintf "%- *e" 7 5.0)) " 5.000000e+000"
    test "test5020" (lazy(sprintf "%- .5e" 5.0)) " 5.00000e+000"
    test "test5021" (lazy(sprintf "%- .*e" 4 5.0)) " 5.0000e+000"
    test "test5022" (lazy(sprintf "%- *.*e" 5 4 5.0)) " 5.0000e+000"
    test "test5023" (lazy(sprintf "% 0e" 5.0)) " 5.000000e+000"
    test "test5024" (lazy(sprintf "% 05e" 5.0)) " 5.000000e+000"
    test "test5025" (lazy(sprintf "% 01e" 5.0)) " 5.000000e+000"
    test "test5026" (lazy(sprintf "% 0*e" 7 5.0)) " 5.000000e+000"
    test "test5027" (lazy(sprintf "% 0.5e" 5.0)) " 5.00000e+000"
    test "test5028" (lazy(sprintf "% 0.*e" 4 5.0)) " 5.0000e+000"
    test "test5029" (lazy(sprintf "% 0*.*e" 5 4 5.0)) " 5.0000e+000"
    test "test5030" (lazy(sprintf "%- 0e" 5.0)) " 5.000000e+000"
    test "test5031" (lazy(sprintf "%- 05e" 5.0)) " 5.000000e+000"
    test "test5032" (lazy(sprintf "%- 01e" 5.0)) " 5.000000e+000"
    test "test5033" (lazy(sprintf "%- 0*e" 7 5.0)) " 5.000000e+000"
    test "test5034" (lazy(sprintf "%- 0.5e" 5.0)) " 5.00000e+000"
    test "test5035" (lazy(sprintf "%- 0.*e" 4 5.0)) " 5.0000e+000"
    test "test5036" (lazy(sprintf "%- 0*.*e" 5 4 5.0)) " 5.0000e+000"
    test "test5037" (lazy(sprintf "%e" -10.0)) "-1.000000e+001"
    test "test5038" (lazy(sprintf "%5e" -10.0)) "-1.000000e+001"
    test "test5039" (lazy(sprintf "%1e" -10.0)) "-1.000000e+001"
    test "test5040" (lazy(sprintf "%*e" 7 -10.0)) "-1.000000e+001"
    test "test5041" (lazy(sprintf "%.5e" -10.0)) "-1.00000e+001"
    test "test5042" (lazy(sprintf "%.*e" 4 -10.0)) "-1.0000e+001"
    test "test5043" (lazy(sprintf "%*.*e" 5 4 -10.0)) "-1.0000e+001"
    test "test5044" (lazy(sprintf "%-e" -10.0)) "-1.000000e+001"
    test "test5045" (lazy(sprintf "%-5e" -10.0)) "-1.000000e+001"
    test "test5046" (lazy(sprintf "%-1e" -10.0)) "-1.000000e+001"
    test "test5047" (lazy(sprintf "%-*e" 7 -10.0)) "-1.000000e+001"
    test "test5048" (lazy(sprintf "%-.5e" -10.0)) "-1.00000e+001"
    test "test5049" (lazy(sprintf "%-.*e" 4 -10.0)) "-1.0000e+001"
    test "test5050" (lazy(sprintf "%-*.*e" 5 4 -10.0)) "-1.0000e+001"
    test "test5051" (lazy(sprintf "%0e" -10.0)) "-1.000000e+001"
    test "test5052" (lazy(sprintf "%05e" -10.0)) "-1.000000e+001"
    test "test5053" (lazy(sprintf "%01e" -10.0)) "-1.000000e+001"
    test "test5054" (lazy(sprintf "%0*e" 7 -10.0)) "-1.000000e+001"
    test "test5055" (lazy(sprintf "%0.5e" -10.0)) "-1.00000e+001"
    test "test5056" (lazy(sprintf "%0.*e" 4 -10.0)) "-1.0000e+001"
    test "test5057" (lazy(sprintf "%0*.*e" 5 4 -10.0)) "-1.0000e+001"
    test "test5058" (lazy(sprintf "%-0e" -10.0)) "-1.000000e+001"
    test "test5059" (lazy(sprintf "%-05e" -10.0)) "-1.000000e+001"
    test "test5060" (lazy(sprintf "%-01e" -10.0)) "-1.000000e+001"
    test "test5061" (lazy(sprintf "%-0*e" 7 -10.0)) "-1.000000e+001"
    test "test5062" (lazy(sprintf "%-0.5e" -10.0)) "-1.00000e+001"
    test "test5063" (lazy(sprintf "%-0.*e" 4 -10.0)) "-1.0000e+001"
    test "test5064" (lazy(sprintf "%-0*.*e" 5 4 -10.0)) "-1.0000e+001"
    test "test5065" (lazy(sprintf "%+e" -10.0)) "-1.000000e+001"
    test "test5066" (lazy(sprintf "%+5e" -10.0)) "-1.000000e+001"
    test "test5067" (lazy(sprintf "%+1e" -10.0)) "-1.000000e+001"
    test "test5068" (lazy(sprintf "%+*e" 7 -10.0)) "-1.000000e+001"
    test "test5069" (lazy(sprintf "%+.5e" -10.0)) "-1.00000e+001"
    test "test5070" (lazy(sprintf "%+.*e" 4 -10.0)) "-1.0000e+001"
    test "test5071" (lazy(sprintf "%+*.*e" 5 4 -10.0)) "-1.0000e+001"
    test "test5072" (lazy(sprintf "%-+e" -10.0)) "-1.000000e+001"
    test "test5073" (lazy(sprintf "%-+5e" -10.0)) "-1.000000e+001"
    test "test5074" (lazy(sprintf "%-+1e" -10.0)) "-1.000000e+001"
    test "test5075" (lazy(sprintf "%-+*e" 7 -10.0)) "-1.000000e+001"
    test "test5076" (lazy(sprintf "%-+.5e" -10.0)) "-1.00000e+001"
    test "test5077" (lazy(sprintf "%-+.*e" 4 -10.0)) "-1.0000e+001"
    test "test5078" (lazy(sprintf "%-+*.*e" 5 4 -10.0)) "-1.0000e+001"
    test "test5079" (lazy(sprintf "%+0e" -10.0)) "-1.000000e+001"
    test "test5080" (lazy(sprintf "%+05e" -10.0)) "-1.000000e+001"
    test "test5081" (lazy(sprintf "%+01e" -10.0)) "-1.000000e+001"
    test "test5082" (lazy(sprintf "%+0*e" 7 -10.0)) "-1.000000e+001"
    test "test5083" (lazy(sprintf "%+0.5e" -10.0)) "-1.00000e+001"
    test "test5084" (lazy(sprintf "%+0.*e" 4 -10.0)) "-1.0000e+001"
    test "test5085" (lazy(sprintf "%+0*.*e" 5 4 -10.0)) "-1.0000e+001"
    test "test5086" (lazy(sprintf "%-+0e" -10.0)) "-1.000000e+001"
    test "test5087" (lazy(sprintf "%-+05e" -10.0)) "-1.000000e+001"
    test "test5088" (lazy(sprintf "%-+01e" -10.0)) "-1.000000e+001"
    test "test5089" (lazy(sprintf "%-+0*e" 7 -10.0)) "-1.000000e+001"
    test "test5090" (lazy(sprintf "%-+0.5e" -10.0)) "-1.00000e+001"
    test "test5091" (lazy(sprintf "%-+0.*e" 4 -10.0)) "-1.0000e+001"
    test "test5092" (lazy(sprintf "%-+0*.*e" 5 4 -10.0)) "-1.0000e+001"
    test "test5093" (lazy(sprintf "% e" -10.0)) "-1.000000e+001"
    test "test5094" (lazy(sprintf "% 5e" -10.0)) "-1.000000e+001"
    test "test5095" (lazy(sprintf "% 1e" -10.0)) "-1.000000e+001"
    test "test5096" (lazy(sprintf "% *e" 7 -10.0)) "-1.000000e+001"
    test "test5097" (lazy(sprintf "% .5e" -10.0)) "-1.00000e+001"
    test "test5098" (lazy(sprintf "% .*e" 4 -10.0)) "-1.0000e+001"
    test "test5099" (lazy(sprintf "% *.*e" 5 4 -10.0)) "-1.0000e+001"
    test "test5100" (lazy(sprintf "%- e" -10.0)) "-1.000000e+001"
    test "test5101" (lazy(sprintf "%- 5e" -10.0)) "-1.000000e+001"
    test "test5102" (lazy(sprintf "%- 1e" -10.0)) "-1.000000e+001"
    test "test5103" (lazy(sprintf "%- *e" 7 -10.0)) "-1.000000e+001"
    test "test5104" (lazy(sprintf "%- .5e" -10.0)) "-1.00000e+001"
    test "test5105" (lazy(sprintf "%- .*e" 4 -10.0)) "-1.0000e+001"
    test "test5106" (lazy(sprintf "%- *.*e" 5 4 -10.0)) "-1.0000e+001"
    test "test5107" (lazy(sprintf "% 0e" -10.0)) "-1.000000e+001"
    test "test5108" (lazy(sprintf "% 05e" -10.0)) "-1.000000e+001"
    test "test5109" (lazy(sprintf "% 01e" -10.0)) "-1.000000e+001"
    test "test5110" (lazy(sprintf "% 0*e" 7 -10.0)) "-1.000000e+001"
    test "test5111" (lazy(sprintf "% 0.5e" -10.0)) "-1.00000e+001"
    test "test5112" (lazy(sprintf "% 0.*e" 4 -10.0)) "-1.0000e+001"
    test "test5113" (lazy(sprintf "% 0*.*e" 5 4 -10.0)) "-1.0000e+001"
    test "test5114" (lazy(sprintf "%- 0e" -10.0)) "-1.000000e+001"
    test "test5115" (lazy(sprintf "%- 05e" -10.0)) "-1.000000e+001"
    test "test5116" (lazy(sprintf "%- 01e" -10.0)) "-1.000000e+001"
    test "test5117" (lazy(sprintf "%- 0*e" 7 -10.0)) "-1.000000e+001"
    test "test5118" (lazy(sprintf "%- 0.5e" -10.0)) "-1.00000e+001"
    test "test5119" (lazy(sprintf "%- 0.*e" 4 -10.0)) "-1.0000e+001"
    test "test5120" (lazy(sprintf "%- 0*.*e" 5 4 -10.0)) "-1.0000e+001"
    test "test5121" (lazy(sprintf "%e" 5.0f)) "5.000000e+000"
    test "test5122" (lazy(sprintf "%5e" 5.0f)) "5.000000e+000"
    test "test5123" (lazy(sprintf "%1e" 5.0f)) "5.000000e+000"
    test "test5124" (lazy(sprintf "%*e" 7 5.0f)) "5.000000e+000"
    test "test5125" (lazy(sprintf "%.5e" 5.0f)) "5.00000e+000"
    test "test5126" (lazy(sprintf "%.*e" 4 5.0f)) "5.0000e+000"
    test "test5127" (lazy(sprintf "%*.*e" 5 4 5.0f)) "5.0000e+000"
    test "test5128" (lazy(sprintf "%-e" 5.0f)) "5.000000e+000"
    test "test5129" (lazy(sprintf "%-5e" 5.0f)) "5.000000e+000"
    test "test5130" (lazy(sprintf "%-1e" 5.0f)) "5.000000e+000"
    test "test5131" (lazy(sprintf "%-*e" 7 5.0f)) "5.000000e+000"
    test "test5132" (lazy(sprintf "%-.5e" 5.0f)) "5.00000e+000"
    test "test5133" (lazy(sprintf "%-.*e" 4 5.0f)) "5.0000e+000"
    test "test5134" (lazy(sprintf "%-*.*e" 5 4 5.0f)) "5.0000e+000"
    test "test5135" (lazy(sprintf "%0e" 5.0f)) "5.000000e+000"
    test "test5136" (lazy(sprintf "%05e" 5.0f)) "5.000000e+000"
    test "test5137" (lazy(sprintf "%01e" 5.0f)) "5.000000e+000"
    test "test5138" (lazy(sprintf "%0*e" 7 5.0f)) "5.000000e+000"
    test "test5139" (lazy(sprintf "%0.5e" 5.0f)) "5.00000e+000"
    test "test5140" (lazy(sprintf "%0.*e" 4 5.0f)) "5.0000e+000"
    test "test5141" (lazy(sprintf "%0*.*e" 5 4 5.0f)) "5.0000e+000"
    test "test5142" (lazy(sprintf "%-0e" 5.0f)) "5.000000e+000"
    test "test5143" (lazy(sprintf "%-05e" 5.0f)) "5.000000e+000"
    test "test5144" (lazy(sprintf "%-01e" 5.0f)) "5.000000e+000"
    test "test5145" (lazy(sprintf "%-0*e" 7 5.0f)) "5.000000e+000"
    test "test5146" (lazy(sprintf "%-0.5e" 5.0f)) "5.00000e+000"
    test "test5147" (lazy(sprintf "%-0.*e" 4 5.0f)) "5.0000e+000"
    test "test5148" (lazy(sprintf "%-0*.*e" 5 4 5.0f)) "5.0000e+000"
    test "test5149" (lazy(sprintf "%+e" 5.0f)) "+5.000000e+000"
    test "test5150" (lazy(sprintf "%+5e" 5.0f)) "+5.000000e+000"
    test "test5151" (lazy(sprintf "%+1e" 5.0f)) "+5.000000e+000"
    test "test5152" (lazy(sprintf "%+*e" 7 5.0f)) "+5.000000e+000"
    test "test5153" (lazy(sprintf "%+.5e" 5.0f)) "+5.00000e+000"
    test "test5154" (lazy(sprintf "%+.*e" 4 5.0f)) "+5.0000e+000"
    test "test5155" (lazy(sprintf "%+*.*e" 5 4 5.0f)) "+5.0000e+000"
    test "test5156" (lazy(sprintf "%-+e" 5.0f)) "+5.000000e+000"
    test "test5157" (lazy(sprintf "%-+5e" 5.0f)) "+5.000000e+000"
    test "test5158" (lazy(sprintf "%-+1e" 5.0f)) "+5.000000e+000"
    test "test5159" (lazy(sprintf "%-+*e" 7 5.0f)) "+5.000000e+000"
    test "test5160" (lazy(sprintf "%-+.5e" 5.0f)) "+5.00000e+000"
    test "test5161" (lazy(sprintf "%-+.*e" 4 5.0f)) "+5.0000e+000"
    test "test5162" (lazy(sprintf "%-+*.*e" 5 4 5.0f)) "+5.0000e+000"
    test "test5163" (lazy(sprintf "%+0e" 5.0f)) "+5.000000e+000"
    test "test5164" (lazy(sprintf "%+05e" 5.0f)) "+5.000000e+000"
    test "test5165" (lazy(sprintf "%+01e" 5.0f)) "+5.000000e+000"
    test "test5166" (lazy(sprintf "%+0*e" 7 5.0f)) "+5.000000e+000"
    test "test5167" (lazy(sprintf "%+0.5e" 5.0f)) "+5.00000e+000"
    test "test5168" (lazy(sprintf "%+0.*e" 4 5.0f)) "+5.0000e+000"
    test "test5169" (lazy(sprintf "%+0*.*e" 5 4 5.0f)) "+5.0000e+000"
    test "test5170" (lazy(sprintf "%-+0e" 5.0f)) "+5.000000e+000"
    test "test5171" (lazy(sprintf "%-+05e" 5.0f)) "+5.000000e+000"
    test "test5172" (lazy(sprintf "%-+01e" 5.0f)) "+5.000000e+000"
    test "test5173" (lazy(sprintf "%-+0*e" 7 5.0f)) "+5.000000e+000"
    test "test5174" (lazy(sprintf "%-+0.5e" 5.0f)) "+5.00000e+000"
    test "test5175" (lazy(sprintf "%-+0.*e" 4 5.0f)) "+5.0000e+000"
    test "test5176" (lazy(sprintf "%-+0*.*e" 5 4 5.0f)) "+5.0000e+000"
    test "test5177" (lazy(sprintf "% e" 5.0f)) " 5.000000e+000"
    test "test5178" (lazy(sprintf "% 5e" 5.0f)) " 5.000000e+000"
    test "test5179" (lazy(sprintf "% 1e" 5.0f)) " 5.000000e+000"
    test "test5180" (lazy(sprintf "% *e" 7 5.0f)) " 5.000000e+000"
    test "test5181" (lazy(sprintf "% .5e" 5.0f)) " 5.00000e+000"
    test "test5182" (lazy(sprintf "% .*e" 4 5.0f)) " 5.0000e+000"
    test "test5183" (lazy(sprintf "% *.*e" 5 4 5.0f)) " 5.0000e+000"
    test "test5184" (lazy(sprintf "%- e" 5.0f)) " 5.000000e+000"
    test "test5185" (lazy(sprintf "%- 5e" 5.0f)) " 5.000000e+000"
    test "test5186" (lazy(sprintf "%- 1e" 5.0f)) " 5.000000e+000"
    test "test5187" (lazy(sprintf "%- *e" 7 5.0f)) " 5.000000e+000"
    test "test5188" (lazy(sprintf "%- .5e" 5.0f)) " 5.00000e+000"
    test "test5189" (lazy(sprintf "%- .*e" 4 5.0f)) " 5.0000e+000"
    test "test5190" (lazy(sprintf "%- *.*e" 5 4 5.0f)) " 5.0000e+000"
    test "test5191" (lazy(sprintf "% 0e" 5.0f)) " 5.000000e+000"
    test "test5192" (lazy(sprintf "% 05e" 5.0f)) " 5.000000e+000"
    test "test5193" (lazy(sprintf "% 01e" 5.0f)) " 5.000000e+000"
    test "test5194" (lazy(sprintf "% 0*e" 7 5.0f)) " 5.000000e+000"
    test "test5195" (lazy(sprintf "% 0.5e" 5.0f)) " 5.00000e+000"
    test "test5196" (lazy(sprintf "% 0.*e" 4 5.0f)) " 5.0000e+000"
    test "test5197" (lazy(sprintf "% 0*.*e" 5 4 5.0f)) " 5.0000e+000"
    test "test5198" (lazy(sprintf "%- 0e" 5.0f)) " 5.000000e+000"
    test "test5199" (lazy(sprintf "%- 05e" 5.0f)) " 5.000000e+000"
    test "test5200" (lazy(sprintf "%- 01e" 5.0f)) " 5.000000e+000"
    test "test5201" (lazy(sprintf "%- 0*e" 7 5.0f)) " 5.000000e+000"
    test "test5202" (lazy(sprintf "%- 0.5e" 5.0f)) " 5.00000e+000"
    test "test5203" (lazy(sprintf "%- 0.*e" 4 5.0f)) " 5.0000e+000"
    test "test5204" (lazy(sprintf "%- 0*.*e" 5 4 5.0f)) " 5.0000e+000"
    test "test5205" (lazy(sprintf "%e" -10.0f)) "-1.000000e+001"
    test "test5206" (lazy(sprintf "%5e" -10.0f)) "-1.000000e+001"
    test "test5207" (lazy(sprintf "%1e" -10.0f)) "-1.000000e+001"
    test "test5208" (lazy(sprintf "%*e" 7 -10.0f)) "-1.000000e+001"
    test "test5209" (lazy(sprintf "%.5e" -10.0f)) "-1.00000e+001"
    test "test5210" (lazy(sprintf "%.*e" 4 -10.0f)) "-1.0000e+001"
    test "test5211" (lazy(sprintf "%*.*e" 5 4 -10.0f)) "-1.0000e+001"
    test "test5212" (lazy(sprintf "%-e" -10.0f)) "-1.000000e+001"
    test "test5213" (lazy(sprintf "%-5e" -10.0f)) "-1.000000e+001"
    test "test5214" (lazy(sprintf "%-1e" -10.0f)) "-1.000000e+001"
    test "test5215" (lazy(sprintf "%-*e" 7 -10.0f)) "-1.000000e+001"
    test "test5216" (lazy(sprintf "%-.5e" -10.0f)) "-1.00000e+001"
    test "test5217" (lazy(sprintf "%-.*e" 4 -10.0f)) "-1.0000e+001"
    test "test5218" (lazy(sprintf "%-*.*e" 5 4 -10.0f)) "-1.0000e+001"
    test "test5219" (lazy(sprintf "%0e" -10.0f)) "-1.000000e+001"
    test "test5220" (lazy(sprintf "%05e" -10.0f)) "-1.000000e+001"
    test "test5221" (lazy(sprintf "%01e" -10.0f)) "-1.000000e+001"
    test "test5222" (lazy(sprintf "%0*e" 7 -10.0f)) "-1.000000e+001"
    test "test5223" (lazy(sprintf "%0.5e" -10.0f)) "-1.00000e+001"
    test "test5224" (lazy(sprintf "%0.*e" 4 -10.0f)) "-1.0000e+001"
    test "test5225" (lazy(sprintf "%0*.*e" 5 4 -10.0f)) "-1.0000e+001"
    test "test5226" (lazy(sprintf "%-0e" -10.0f)) "-1.000000e+001"
    test "test5227" (lazy(sprintf "%-05e" -10.0f)) "-1.000000e+001"
    test "test5228" (lazy(sprintf "%-01e" -10.0f)) "-1.000000e+001"
    test "test5229" (lazy(sprintf "%-0*e" 7 -10.0f)) "-1.000000e+001"
    test "test5230" (lazy(sprintf "%-0.5e" -10.0f)) "-1.00000e+001"
    test "test5231" (lazy(sprintf "%-0.*e" 4 -10.0f)) "-1.0000e+001"
    test "test5232" (lazy(sprintf "%-0*.*e" 5 4 -10.0f)) "-1.0000e+001"
    test "test5233" (lazy(sprintf "%+e" -10.0f)) "-1.000000e+001"
    test "test5234" (lazy(sprintf "%+5e" -10.0f)) "-1.000000e+001"
    test "test5235" (lazy(sprintf "%+1e" -10.0f)) "-1.000000e+001"
    test "test5236" (lazy(sprintf "%+*e" 7 -10.0f)) "-1.000000e+001"
    test "test5237" (lazy(sprintf "%+.5e" -10.0f)) "-1.00000e+001"
    test "test5238" (lazy(sprintf "%+.*e" 4 -10.0f)) "-1.0000e+001"
    test "test5239" (lazy(sprintf "%+*.*e" 5 4 -10.0f)) "-1.0000e+001"
    test "test5240" (lazy(sprintf "%-+e" -10.0f)) "-1.000000e+001"
    test "test5241" (lazy(sprintf "%-+5e" -10.0f)) "-1.000000e+001"
    test "test5242" (lazy(sprintf "%-+1e" -10.0f)) "-1.000000e+001"
    test "test5243" (lazy(sprintf "%-+*e" 7 -10.0f)) "-1.000000e+001"
    test "test5244" (lazy(sprintf "%-+.5e" -10.0f)) "-1.00000e+001"
    test "test5245" (lazy(sprintf "%-+.*e" 4 -10.0f)) "-1.0000e+001"
    test "test5246" (lazy(sprintf "%-+*.*e" 5 4 -10.0f)) "-1.0000e+001"
    test "test5247" (lazy(sprintf "%+0e" -10.0f)) "-1.000000e+001"
    test "test5248" (lazy(sprintf "%+05e" -10.0f)) "-1.000000e+001"
    test "test5249" (lazy(sprintf "%+01e" -10.0f)) "-1.000000e+001"
    test "test5250" (lazy(sprintf "%+0*e" 7 -10.0f)) "-1.000000e+001"
    test "test5251" (lazy(sprintf "%+0.5e" -10.0f)) "-1.00000e+001"
    test "test5252" (lazy(sprintf "%+0.*e" 4 -10.0f)) "-1.0000e+001"
    test "test5253" (lazy(sprintf "%+0*.*e" 5 4 -10.0f)) "-1.0000e+001"
    test "test5254" (lazy(sprintf "%-+0e" -10.0f)) "-1.000000e+001"
    test "test5255" (lazy(sprintf "%-+05e" -10.0f)) "-1.000000e+001"
    test "test5256" (lazy(sprintf "%-+01e" -10.0f)) "-1.000000e+001"
    test "test5257" (lazy(sprintf "%-+0*e" 7 -10.0f)) "-1.000000e+001"
    test "test5258" (lazy(sprintf "%-+0.5e" -10.0f)) "-1.00000e+001"
    test "test5259" (lazy(sprintf "%-+0.*e" 4 -10.0f)) "-1.0000e+001"
    test "test5260" (lazy(sprintf "%-+0*.*e" 5 4 -10.0f)) "-1.0000e+001"
    test "test5261" (lazy(sprintf "% e" -10.0f)) "-1.000000e+001"
    test "test5262" (lazy(sprintf "% 5e" -10.0f)) "-1.000000e+001"
    test "test5263" (lazy(sprintf "% 1e" -10.0f)) "-1.000000e+001"
    test "test5264" (lazy(sprintf "% *e" 7 -10.0f)) "-1.000000e+001"
    test "test5265" (lazy(sprintf "% .5e" -10.0f)) "-1.00000e+001"
    test "test5266" (lazy(sprintf "% .*e" 4 -10.0f)) "-1.0000e+001"
    test "test5267" (lazy(sprintf "% *.*e" 5 4 -10.0f)) "-1.0000e+001"
    test "test5268" (lazy(sprintf "%- e" -10.0f)) "-1.000000e+001"
    test "test5269" (lazy(sprintf "%- 5e" -10.0f)) "-1.000000e+001"
    test "test5270" (lazy(sprintf "%- 1e" -10.0f)) "-1.000000e+001"
    test "test5271" (lazy(sprintf "%- *e" 7 -10.0f)) "-1.000000e+001"
    test "test5272" (lazy(sprintf "%- .5e" -10.0f)) "-1.00000e+001"
    test "test5273" (lazy(sprintf "%- .*e" 4 -10.0f)) "-1.0000e+001"
    test "test5274" (lazy(sprintf "%- *.*e" 5 4 -10.0f)) "-1.0000e+001"
    test "test5275" (lazy(sprintf "% 0e" -10.0f)) "-1.000000e+001"
    test "test5276" (lazy(sprintf "% 05e" -10.0f)) "-1.000000e+001"
    test "test5277" (lazy(sprintf "% 01e" -10.0f)) "-1.000000e+001"
    test "test5278" (lazy(sprintf "% 0*e" 7 -10.0f)) "-1.000000e+001"
    test "test5279" (lazy(sprintf "% 0.5e" -10.0f)) "-1.00000e+001"
    test "test5280" (lazy(sprintf "% 0.*e" 4 -10.0f)) "-1.0000e+001"
    test "test5281" (lazy(sprintf "% 0*.*e" 5 4 -10.0f)) "-1.0000e+001"
    test "test5282" (lazy(sprintf "%- 0e" -10.0f)) "-1.000000e+001"
    test "test5283" (lazy(sprintf "%- 05e" -10.0f)) "-1.000000e+001"
    test "test5284" (lazy(sprintf "%- 01e" -10.0f)) "-1.000000e+001"
    test "test5285" (lazy(sprintf "%- 0*e" 7 -10.0f)) "-1.000000e+001"
    test "test5286" (lazy(sprintf "%- 0.5e" -10.0f)) "-1.00000e+001"
    test "test5287" (lazy(sprintf "%- 0.*e" 4 -10.0f)) "-1.0000e+001"
    test "test5288" (lazy(sprintf "%- 0*.*e" 5 4 -10.0f)) "-1.0000e+001"
    test "test5289" (lazy(sprintf "%e" 5.0M)) "5.000000e+000"
    test "test5290" (lazy(sprintf "%5e" 5.0M)) "5.000000e+000"
    test "test5291" (lazy(sprintf "%1e" 5.0M)) "5.000000e+000"
    test "test5292" (lazy(sprintf "%*e" 7 5.0M)) "5.000000e+000"
    test "test5293" (lazy(sprintf "%.5e" 5.0M)) "5.00000e+000"
    test "test5294" (lazy(sprintf "%.*e" 4 5.0M)) "5.0000e+000"
    test "test5295" (lazy(sprintf "%*.*e" 5 4 5.0M)) "5.0000e+000"
    test "test5296" (lazy(sprintf "%-e" 5.0M)) "5.000000e+000"
    test "test5297" (lazy(sprintf "%-5e" 5.0M)) "5.000000e+000"
    test "test5298" (lazy(sprintf "%-1e" 5.0M)) "5.000000e+000"
    test "test5299" (lazy(sprintf "%-*e" 7 5.0M)) "5.000000e+000"
    test "test5300" (lazy(sprintf "%-.5e" 5.0M)) "5.00000e+000"
    test "test5301" (lazy(sprintf "%-.*e" 4 5.0M)) "5.0000e+000"
    test "test5302" (lazy(sprintf "%-*.*e" 5 4 5.0M)) "5.0000e+000"
    test "test5303" (lazy(sprintf "%0e" 5.0M)) "5.000000e+000"
    test "test5304" (lazy(sprintf "%05e" 5.0M)) "5.000000e+000"
    test "test5305" (lazy(sprintf "%01e" 5.0M)) "5.000000e+000"
    test "test5306" (lazy(sprintf "%0*e" 7 5.0M)) "5.000000e+000"
    test "test5307" (lazy(sprintf "%0.5e" 5.0M)) "5.00000e+000"
    test "test5308" (lazy(sprintf "%0.*e" 4 5.0M)) "5.0000e+000"
    test "test5309" (lazy(sprintf "%0*.*e" 5 4 5.0M)) "5.0000e+000"
    test "test5310" (lazy(sprintf "%-0e" 5.0M)) "5.000000e+000"
    test "test5311" (lazy(sprintf "%-05e" 5.0M)) "5.000000e+000"
    test "test5312" (lazy(sprintf "%-01e" 5.0M)) "5.000000e+000"
    test "test5313" (lazy(sprintf "%-0*e" 7 5.0M)) "5.000000e+000"
    test "test5314" (lazy(sprintf "%-0.5e" 5.0M)) "5.00000e+000"
    test "test5315" (lazy(sprintf "%-0.*e" 4 5.0M)) "5.0000e+000"
    test "test5316" (lazy(sprintf "%-0*.*e" 5 4 5.0M)) "5.0000e+000"
    test "test5317" (lazy(sprintf "%+e" 5.0M)) "+5.000000e+000"
    test "test5318" (lazy(sprintf "%+5e" 5.0M)) "+5.000000e+000"
    test "test5319" (lazy(sprintf "%+1e" 5.0M)) "+5.000000e+000"
    test "test5320" (lazy(sprintf "%+*e" 7 5.0M)) "+5.000000e+000"
    test "test5321" (lazy(sprintf "%+.5e" 5.0M)) "+5.00000e+000"
    test "test5322" (lazy(sprintf "%+.*e" 4 5.0M)) "+5.0000e+000"
    test "test5323" (lazy(sprintf "%+*.*e" 5 4 5.0M)) "+5.0000e+000"
    test "test5324" (lazy(sprintf "%-+e" 5.0M)) "+5.000000e+000"
    test "test5325" (lazy(sprintf "%-+5e" 5.0M)) "+5.000000e+000"
    test "test5326" (lazy(sprintf "%-+1e" 5.0M)) "+5.000000e+000"
    test "test5327" (lazy(sprintf "%-+*e" 7 5.0M)) "+5.000000e+000"
    test "test5328" (lazy(sprintf "%-+.5e" 5.0M)) "+5.00000e+000"
    test "test5329" (lazy(sprintf "%-+.*e" 4 5.0M)) "+5.0000e+000"
    test "test5330" (lazy(sprintf "%-+*.*e" 5 4 5.0M)) "+5.0000e+000"
    test "test5331" (lazy(sprintf "%+0e" 5.0M)) "+5.000000e+000"
    test "test5332" (lazy(sprintf "%+05e" 5.0M)) "+5.000000e+000"
    test "test5333" (lazy(sprintf "%+01e" 5.0M)) "+5.000000e+000"
    test "test5334" (lazy(sprintf "%+0*e" 7 5.0M)) "+5.000000e+000"
    test "test5335" (lazy(sprintf "%+0.5e" 5.0M)) "+5.00000e+000"
    test "test5336" (lazy(sprintf "%+0.*e" 4 5.0M)) "+5.0000e+000"
    test "test5337" (lazy(sprintf "%+0*.*e" 5 4 5.0M)) "+5.0000e+000"
    test "test5338" (lazy(sprintf "%-+0e" 5.0M)) "+5.000000e+000"
    test "test5339" (lazy(sprintf "%-+05e" 5.0M)) "+5.000000e+000"
    test "test5340" (lazy(sprintf "%-+01e" 5.0M)) "+5.000000e+000"
    test "test5341" (lazy(sprintf "%-+0*e" 7 5.0M)) "+5.000000e+000"
    test "test5342" (lazy(sprintf "%-+0.5e" 5.0M)) "+5.00000e+000"
    test "test5343" (lazy(sprintf "%-+0.*e" 4 5.0M)) "+5.0000e+000"
    test "test5344" (lazy(sprintf "%-+0*.*e" 5 4 5.0M)) "+5.0000e+000"
    test "test5345" (lazy(sprintf "% e" 5.0M)) " 5.000000e+000"
    test "test5346" (lazy(sprintf "% 5e" 5.0M)) " 5.000000e+000"
    test "test5347" (lazy(sprintf "% 1e" 5.0M)) " 5.000000e+000"
    test "test5348" (lazy(sprintf "% *e" 7 5.0M)) " 5.000000e+000"
    test "test5349" (lazy(sprintf "% .5e" 5.0M)) " 5.00000e+000"
    test "test5350" (lazy(sprintf "% .*e" 4 5.0M)) " 5.0000e+000"
    test "test5351" (lazy(sprintf "% *.*e" 5 4 5.0M)) " 5.0000e+000"
    test "test5352" (lazy(sprintf "%- e" 5.0M)) " 5.000000e+000"
    test "test5353" (lazy(sprintf "%- 5e" 5.0M)) " 5.000000e+000"
    test "test5354" (lazy(sprintf "%- 1e" 5.0M)) " 5.000000e+000"
    test "test5355" (lazy(sprintf "%- *e" 7 5.0M)) " 5.000000e+000"
    test "test5356" (lazy(sprintf "%- .5e" 5.0M)) " 5.00000e+000"
    test "test5357" (lazy(sprintf "%- .*e" 4 5.0M)) " 5.0000e+000"
    test "test5358" (lazy(sprintf "%- *.*e" 5 4 5.0M)) " 5.0000e+000"
    test "test5359" (lazy(sprintf "% 0e" 5.0M)) " 5.000000e+000"
    test "test5360" (lazy(sprintf "% 05e" 5.0M)) " 5.000000e+000"
    test "test5361" (lazy(sprintf "% 01e" 5.0M)) " 5.000000e+000"
    test "test5362" (lazy(sprintf "% 0*e" 7 5.0M)) " 5.000000e+000"
    test "test5363" (lazy(sprintf "% 0.5e" 5.0M)) " 5.00000e+000"
    test "test5364" (lazy(sprintf "% 0.*e" 4 5.0M)) " 5.0000e+000"
    test "test5365" (lazy(sprintf "% 0*.*e" 5 4 5.0M)) " 5.0000e+000"
    test "test5366" (lazy(sprintf "%- 0e" 5.0M)) " 5.000000e+000"
    test "test5367" (lazy(sprintf "%- 05e" 5.0M)) " 5.000000e+000"
    test "test5368" (lazy(sprintf "%- 01e" 5.0M)) " 5.000000e+000"
    test "test5369" (lazy(sprintf "%- 0*e" 7 5.0M)) " 5.000000e+000"
    test "test5370" (lazy(sprintf "%- 0.5e" 5.0M)) " 5.00000e+000"
    test "test5371" (lazy(sprintf "%- 0.*e" 4 5.0M)) " 5.0000e+000"
    test "test5372" (lazy(sprintf "%- 0*.*e" 5 4 5.0M)) " 5.0000e+000"
    test "test5373" (lazy(sprintf "%e" -10.0M)) "-1.000000e+001"
    test "test5374" (lazy(sprintf "%5e" -10.0M)) "-1.000000e+001"
    test "test5375" (lazy(sprintf "%1e" -10.0M)) "-1.000000e+001"
    test "test5376" (lazy(sprintf "%*e" 7 -10.0M)) "-1.000000e+001"
    test "test5377" (lazy(sprintf "%.5e" -10.0M)) "-1.00000e+001"
    test "test5378" (lazy(sprintf "%.*e" 4 -10.0M)) "-1.0000e+001"
    test "test5379" (lazy(sprintf "%*.*e" 5 4 -10.0M)) "-1.0000e+001"
    test "test5380" (lazy(sprintf "%-e" -10.0M)) "-1.000000e+001"
    test "test5381" (lazy(sprintf "%-5e" -10.0M)) "-1.000000e+001"
    test "test5382" (lazy(sprintf "%-1e" -10.0M)) "-1.000000e+001"
    test "test5383" (lazy(sprintf "%-*e" 7 -10.0M)) "-1.000000e+001"
    test "test5384" (lazy(sprintf "%-.5e" -10.0M)) "-1.00000e+001"
    test "test5385" (lazy(sprintf "%-.*e" 4 -10.0M)) "-1.0000e+001"
    test "test5386" (lazy(sprintf "%-*.*e" 5 4 -10.0M)) "-1.0000e+001"
    test "test5387" (lazy(sprintf "%0e" -10.0M)) "-1.000000e+001"
    test "test5388" (lazy(sprintf "%05e" -10.0M)) "-1.000000e+001"
    test "test5389" (lazy(sprintf "%01e" -10.0M)) "-1.000000e+001"
    test "test5390" (lazy(sprintf "%0*e" 7 -10.0M)) "-1.000000e+001"
    test "test5391" (lazy(sprintf "%0.5e" -10.0M)) "-1.00000e+001"
    test "test5392" (lazy(sprintf "%0.*e" 4 -10.0M)) "-1.0000e+001"
    test "test5393" (lazy(sprintf "%0*.*e" 5 4 -10.0M)) "-1.0000e+001"
    test "test5394" (lazy(sprintf "%-0e" -10.0M)) "-1.000000e+001"
    test "test5395" (lazy(sprintf "%-05e" -10.0M)) "-1.000000e+001"
    test "test5396" (lazy(sprintf "%-01e" -10.0M)) "-1.000000e+001"
    test "test5397" (lazy(sprintf "%-0*e" 7 -10.0M)) "-1.000000e+001"
    test "test5398" (lazy(sprintf "%-0.5e" -10.0M)) "-1.00000e+001"
    test "test5399" (lazy(sprintf "%-0.*e" 4 -10.0M)) "-1.0000e+001"
    test "test5400" (lazy(sprintf "%-0*.*e" 5 4 -10.0M)) "-1.0000e+001"
    test "test5401" (lazy(sprintf "%+e" -10.0M)) "-1.000000e+001"
    test "test5402" (lazy(sprintf "%+5e" -10.0M)) "-1.000000e+001"
    test "test5403" (lazy(sprintf "%+1e" -10.0M)) "-1.000000e+001"
    test "test5404" (lazy(sprintf "%+*e" 7 -10.0M)) "-1.000000e+001"
    test "test5405" (lazy(sprintf "%+.5e" -10.0M)) "-1.00000e+001"
    test "test5406" (lazy(sprintf "%+.*e" 4 -10.0M)) "-1.0000e+001"
    test "test5407" (lazy(sprintf "%+*.*e" 5 4 -10.0M)) "-1.0000e+001"
    test "test5408" (lazy(sprintf "%-+e" -10.0M)) "-1.000000e+001"
    test "test5409" (lazy(sprintf "%-+5e" -10.0M)) "-1.000000e+001"
    test "test5410" (lazy(sprintf "%-+1e" -10.0M)) "-1.000000e+001"
    test "test5411" (lazy(sprintf "%-+*e" 7 -10.0M)) "-1.000000e+001"
    test "test5412" (lazy(sprintf "%-+.5e" -10.0M)) "-1.00000e+001"
    test "test5413" (lazy(sprintf "%-+.*e" 4 -10.0M)) "-1.0000e+001"
    test "test5414" (lazy(sprintf "%-+*.*e" 5 4 -10.0M)) "-1.0000e+001"
    test "test5415" (lazy(sprintf "%+0e" -10.0M)) "-1.000000e+001"
    test "test5416" (lazy(sprintf "%+05e" -10.0M)) "-1.000000e+001"
    test "test5417" (lazy(sprintf "%+01e" -10.0M)) "-1.000000e+001"
    test "test5418" (lazy(sprintf "%+0*e" 7 -10.0M)) "-1.000000e+001"
    test "test5419" (lazy(sprintf "%+0.5e" -10.0M)) "-1.00000e+001"
    test "test5420" (lazy(sprintf "%+0.*e" 4 -10.0M)) "-1.0000e+001"
    test "test5421" (lazy(sprintf "%+0*.*e" 5 4 -10.0M)) "-1.0000e+001"
    test "test5422" (lazy(sprintf "%-+0e" -10.0M)) "-1.000000e+001"
    test "test5423" (lazy(sprintf "%-+05e" -10.0M)) "-1.000000e+001"
    test "test5424" (lazy(sprintf "%-+01e" -10.0M)) "-1.000000e+001"
    test "test5425" (lazy(sprintf "%-+0*e" 7 -10.0M)) "-1.000000e+001"
    test "test5426" (lazy(sprintf "%-+0.5e" -10.0M)) "-1.00000e+001"
    test "test5427" (lazy(sprintf "%-+0.*e" 4 -10.0M)) "-1.0000e+001"
    test "test5428" (lazy(sprintf "%-+0*.*e" 5 4 -10.0M)) "-1.0000e+001"
    test "test5429" (lazy(sprintf "% e" -10.0M)) "-1.000000e+001"
    test "test5430" (lazy(sprintf "% 5e" -10.0M)) "-1.000000e+001"
    test "test5431" (lazy(sprintf "% 1e" -10.0M)) "-1.000000e+001"
    test "test5432" (lazy(sprintf "% *e" 7 -10.0M)) "-1.000000e+001"
    test "test5433" (lazy(sprintf "% .5e" -10.0M)) "-1.00000e+001"
    test "test5434" (lazy(sprintf "% .*e" 4 -10.0M)) "-1.0000e+001"
    test "test5435" (lazy(sprintf "% *.*e" 5 4 -10.0M)) "-1.0000e+001"
    test "test5436" (lazy(sprintf "%- e" -10.0M)) "-1.000000e+001"
    test "test5437" (lazy(sprintf "%- 5e" -10.0M)) "-1.000000e+001"
    test "test5438" (lazy(sprintf "%- 1e" -10.0M)) "-1.000000e+001"
    test "test5439" (lazy(sprintf "%- *e" 7 -10.0M)) "-1.000000e+001"
    test "test5440" (lazy(sprintf "%- .5e" -10.0M)) "-1.00000e+001"
    test "test5441" (lazy(sprintf "%- .*e" 4 -10.0M)) "-1.0000e+001"
    test "test5442" (lazy(sprintf "%- *.*e" 5 4 -10.0M)) "-1.0000e+001"
    test "test5443" (lazy(sprintf "% 0e" -10.0M)) "-1.000000e+001"
    test "test5444" (lazy(sprintf "% 05e" -10.0M)) "-1.000000e+001"
    test "test5445" (lazy(sprintf "% 01e" -10.0M)) "-1.000000e+001"
    test "test5446" (lazy(sprintf "% 0*e" 7 -10.0M)) "-1.000000e+001"
    test "test5447" (lazy(sprintf "% 0.5e" -10.0M)) "-1.00000e+001"
    test "test5448" (lazy(sprintf "% 0.*e" 4 -10.0M)) "-1.0000e+001"
    test "test5449" (lazy(sprintf "% 0*.*e" 5 4 -10.0M)) "-1.0000e+001"
    test "test5450" (lazy(sprintf "%- 0e" -10.0M)) "-1.000000e+001"
    test "test5451" (lazy(sprintf "%- 05e" -10.0M)) "-1.000000e+001"
    test "test5452" (lazy(sprintf "%- 01e" -10.0M)) "-1.000000e+001"
    test "test5453" (lazy(sprintf "%- 0*e" 7 -10.0M)) "-1.000000e+001"
    test "test5454" (lazy(sprintf "%- 0.5e" -10.0M)) "-1.00000e+001"
    test "test5455" (lazy(sprintf "%- 0.*e" 4 -10.0M)) "-1.0000e+001"
    test "test5456" (lazy(sprintf "%- 0*.*e" 5 4 -10.0M)) "-1.0000e+001"
    test "test5457" (lazy(sprintf "%E" 5.0)) "5.000000E+000"
    test "test5458" (lazy(sprintf "%5E" 5.0)) "5.000000E+000"
    test "test5459" (lazy(sprintf "%1E" 5.0)) "5.000000E+000"
    test "test5460" (lazy(sprintf "%*E" 7 5.0)) "5.000000E+000"
    test "test5461" (lazy(sprintf "%.5E" 5.0)) "5.00000E+000"
    test "test5462" (lazy(sprintf "%.*E" 4 5.0)) "5.0000E+000"
    test "test5463" (lazy(sprintf "%*.*E" 5 4 5.0)) "5.0000E+000"
    test "test5464" (lazy(sprintf "%-E" 5.0)) "5.000000E+000"
    test "test5465" (lazy(sprintf "%-5E" 5.0)) "5.000000E+000"
    test "test5466" (lazy(sprintf "%-1E" 5.0)) "5.000000E+000"
    test "test5467" (lazy(sprintf "%-*E" 7 5.0)) "5.000000E+000"
    test "test5468" (lazy(sprintf "%-.5E" 5.0)) "5.00000E+000"
    test "test5469" (lazy(sprintf "%-.*E" 4 5.0)) "5.0000E+000"
    test "test5470" (lazy(sprintf "%-*.*E" 5 4 5.0)) "5.0000E+000"
    test "test5471" (lazy(sprintf "%0E" 5.0)) "5.000000E+000"
    test "test5472" (lazy(sprintf "%05E" 5.0)) "5.000000E+000"
    test "test5473" (lazy(sprintf "%01E" 5.0)) "5.000000E+000"
    test "test5474" (lazy(sprintf "%0*E" 7 5.0)) "5.000000E+000"
    test "test5475" (lazy(sprintf "%0.5E" 5.0)) "5.00000E+000"
    test "test5476" (lazy(sprintf "%0.*E" 4 5.0)) "5.0000E+000"
    test "test5477" (lazy(sprintf "%0*.*E" 5 4 5.0)) "5.0000E+000"
    test "test5478" (lazy(sprintf "%-0E" 5.0)) "5.000000E+000"
    test "test5479" (lazy(sprintf "%-05E" 5.0)) "5.000000E+000"
    test "test5480" (lazy(sprintf "%-01E" 5.0)) "5.000000E+000"
    test "test5481" (lazy(sprintf "%-0*E" 7 5.0)) "5.000000E+000"
    test "test5482" (lazy(sprintf "%-0.5E" 5.0)) "5.00000E+000"
    test "test5483" (lazy(sprintf "%-0.*E" 4 5.0)) "5.0000E+000"
    test "test5484" (lazy(sprintf "%-0*.*E" 5 4 5.0)) "5.0000E+000"
    test "test5485" (lazy(sprintf "%+E" 5.0)) "+5.000000E+000"
    test "test5486" (lazy(sprintf "%+5E" 5.0)) "+5.000000E+000"
    test "test5487" (lazy(sprintf "%+1E" 5.0)) "+5.000000E+000"
    test "test5488" (lazy(sprintf "%+*E" 7 5.0)) "+5.000000E+000"
    test "test5489" (lazy(sprintf "%+.5E" 5.0)) "+5.00000E+000"
    test "test5490" (lazy(sprintf "%+.*E" 4 5.0)) "+5.0000E+000"
    test "test5491" (lazy(sprintf "%+*.*E" 5 4 5.0)) "+5.0000E+000"
    test "test5492" (lazy(sprintf "%-+E" 5.0)) "+5.000000E+000"
    test "test5493" (lazy(sprintf "%-+5E" 5.0)) "+5.000000E+000"
    test "test5494" (lazy(sprintf "%-+1E" 5.0)) "+5.000000E+000"
    test "test5495" (lazy(sprintf "%-+*E" 7 5.0)) "+5.000000E+000"
    test "test5496" (lazy(sprintf "%-+.5E" 5.0)) "+5.00000E+000"
    test "test5497" (lazy(sprintf "%-+.*E" 4 5.0)) "+5.0000E+000"
    test "test5498" (lazy(sprintf "%-+*.*E" 5 4 5.0)) "+5.0000E+000"
    test "test5499" (lazy(sprintf "%+0E" 5.0)) "+5.000000E+000"
    test "test5500" (lazy(sprintf "%+05E" 5.0)) "+5.000000E+000"
    test "test5501" (lazy(sprintf "%+01E" 5.0)) "+5.000000E+000"
    test "test5502" (lazy(sprintf "%+0*E" 7 5.0)) "+5.000000E+000"
    test "test5503" (lazy(sprintf "%+0.5E" 5.0)) "+5.00000E+000"
    test "test5504" (lazy(sprintf "%+0.*E" 4 5.0)) "+5.0000E+000"
    test "test5505" (lazy(sprintf "%+0*.*E" 5 4 5.0)) "+5.0000E+000"
    test "test5506" (lazy(sprintf "%-+0E" 5.0)) "+5.000000E+000"
    test "test5507" (lazy(sprintf "%-+05E" 5.0)) "+5.000000E+000"
    test "test5508" (lazy(sprintf "%-+01E" 5.0)) "+5.000000E+000"
    test "test5509" (lazy(sprintf "%-+0*E" 7 5.0)) "+5.000000E+000"
    test "test5510" (lazy(sprintf "%-+0.5E" 5.0)) "+5.00000E+000"
    test "test5511" (lazy(sprintf "%-+0.*E" 4 5.0)) "+5.0000E+000"
    test "test5512" (lazy(sprintf "%-+0*.*E" 5 4 5.0)) "+5.0000E+000"
    test "test5513" (lazy(sprintf "% E" 5.0)) " 5.000000E+000"
    test "test5514" (lazy(sprintf "% 5E" 5.0)) " 5.000000E+000"
    test "test5515" (lazy(sprintf "% 1E" 5.0)) " 5.000000E+000"
    test "test5516" (lazy(sprintf "% *E" 7 5.0)) " 5.000000E+000"
    test "test5517" (lazy(sprintf "% .5E" 5.0)) " 5.00000E+000"
    test "test5518" (lazy(sprintf "% .*E" 4 5.0)) " 5.0000E+000"
    test "test5519" (lazy(sprintf "% *.*E" 5 4 5.0)) " 5.0000E+000"
    test "test5520" (lazy(sprintf "%- E" 5.0)) " 5.000000E+000"
    test "test5521" (lazy(sprintf "%- 5E" 5.0)) " 5.000000E+000"
    test "test5522" (lazy(sprintf "%- 1E" 5.0)) " 5.000000E+000"
    test "test5523" (lazy(sprintf "%- *E" 7 5.0)) " 5.000000E+000"
    test "test5524" (lazy(sprintf "%- .5E" 5.0)) " 5.00000E+000"
    test "test5525" (lazy(sprintf "%- .*E" 4 5.0)) " 5.0000E+000"
    test "test5526" (lazy(sprintf "%- *.*E" 5 4 5.0)) " 5.0000E+000"
    test "test5527" (lazy(sprintf "% 0E" 5.0)) " 5.000000E+000"
    test "test5528" (lazy(sprintf "% 05E" 5.0)) " 5.000000E+000"
    test "test5529" (lazy(sprintf "% 01E" 5.0)) " 5.000000E+000"
    test "test5530" (lazy(sprintf "% 0*E" 7 5.0)) " 5.000000E+000"
    test "test5531" (lazy(sprintf "% 0.5E" 5.0)) " 5.00000E+000"
    test "test5532" (lazy(sprintf "% 0.*E" 4 5.0)) " 5.0000E+000"
    test "test5533" (lazy(sprintf "% 0*.*E" 5 4 5.0)) " 5.0000E+000"
    test "test5534" (lazy(sprintf "%- 0E" 5.0)) " 5.000000E+000"
    test "test5535" (lazy(sprintf "%- 05E" 5.0)) " 5.000000E+000"
    test "test5536" (lazy(sprintf "%- 01E" 5.0)) " 5.000000E+000"
    test "test5537" (lazy(sprintf "%- 0*E" 7 5.0)) " 5.000000E+000"
    test "test5538" (lazy(sprintf "%- 0.5E" 5.0)) " 5.00000E+000"
    test "test5539" (lazy(sprintf "%- 0.*E" 4 5.0)) " 5.0000E+000"
    test "test5540" (lazy(sprintf "%- 0*.*E" 5 4 5.0)) " 5.0000E+000"
    test "test5541" (lazy(sprintf "%E" -10.0)) "-1.000000E+001"
    test "test5542" (lazy(sprintf "%5E" -10.0)) "-1.000000E+001"
    test "test5543" (lazy(sprintf "%1E" -10.0)) "-1.000000E+001"
    test "test5544" (lazy(sprintf "%*E" 7 -10.0)) "-1.000000E+001"
    test "test5545" (lazy(sprintf "%.5E" -10.0)) "-1.00000E+001"
    test "test5546" (lazy(sprintf "%.*E" 4 -10.0)) "-1.0000E+001"
    test "test5547" (lazy(sprintf "%*.*E" 5 4 -10.0)) "-1.0000E+001"
    test "test5548" (lazy(sprintf "%-E" -10.0)) "-1.000000E+001"
    test "test5549" (lazy(sprintf "%-5E" -10.0)) "-1.000000E+001"
    test "test5550" (lazy(sprintf "%-1E" -10.0)) "-1.000000E+001"
    test "test5551" (lazy(sprintf "%-*E" 7 -10.0)) "-1.000000E+001"
    test "test5552" (lazy(sprintf "%-.5E" -10.0)) "-1.00000E+001"
    test "test5553" (lazy(sprintf "%-.*E" 4 -10.0)) "-1.0000E+001"
    test "test5554" (lazy(sprintf "%-*.*E" 5 4 -10.0)) "-1.0000E+001"
    test "test5555" (lazy(sprintf "%0E" -10.0)) "-1.000000E+001"
    test "test5556" (lazy(sprintf "%05E" -10.0)) "-1.000000E+001"
    test "test5557" (lazy(sprintf "%01E" -10.0)) "-1.000000E+001"
    test "test5558" (lazy(sprintf "%0*E" 7 -10.0)) "-1.000000E+001"
    test "test5559" (lazy(sprintf "%0.5E" -10.0)) "-1.00000E+001"
    test "test5560" (lazy(sprintf "%0.*E" 4 -10.0)) "-1.0000E+001"
    test "test5561" (lazy(sprintf "%0*.*E" 5 4 -10.0)) "-1.0000E+001"
    test "test5562" (lazy(sprintf "%-0E" -10.0)) "-1.000000E+001"
    test "test5563" (lazy(sprintf "%-05E" -10.0)) "-1.000000E+001"
    test "test5564" (lazy(sprintf "%-01E" -10.0)) "-1.000000E+001"
    test "test5565" (lazy(sprintf "%-0*E" 7 -10.0)) "-1.000000E+001"
    test "test5566" (lazy(sprintf "%-0.5E" -10.0)) "-1.00000E+001"
    test "test5567" (lazy(sprintf "%-0.*E" 4 -10.0)) "-1.0000E+001"
    test "test5568" (lazy(sprintf "%-0*.*E" 5 4 -10.0)) "-1.0000E+001"
    test "test5569" (lazy(sprintf "%+E" -10.0)) "-1.000000E+001"
    test "test5570" (lazy(sprintf "%+5E" -10.0)) "-1.000000E+001"
    test "test5571" (lazy(sprintf "%+1E" -10.0)) "-1.000000E+001"
    test "test5572" (lazy(sprintf "%+*E" 7 -10.0)) "-1.000000E+001"
    test "test5573" (lazy(sprintf "%+.5E" -10.0)) "-1.00000E+001"
    test "test5574" (lazy(sprintf "%+.*E" 4 -10.0)) "-1.0000E+001"
    test "test5575" (lazy(sprintf "%+*.*E" 5 4 -10.0)) "-1.0000E+001"
    test "test5576" (lazy(sprintf "%-+E" -10.0)) "-1.000000E+001"
    test "test5577" (lazy(sprintf "%-+5E" -10.0)) "-1.000000E+001"
    test "test5578" (lazy(sprintf "%-+1E" -10.0)) "-1.000000E+001"
    test "test5579" (lazy(sprintf "%-+*E" 7 -10.0)) "-1.000000E+001"
    test "test5580" (lazy(sprintf "%-+.5E" -10.0)) "-1.00000E+001"
    test "test5581" (lazy(sprintf "%-+.*E" 4 -10.0)) "-1.0000E+001"
    test "test5582" (lazy(sprintf "%-+*.*E" 5 4 -10.0)) "-1.0000E+001"
    test "test5583" (lazy(sprintf "%+0E" -10.0)) "-1.000000E+001"
    test "test5584" (lazy(sprintf "%+05E" -10.0)) "-1.000000E+001"
    test "test5585" (lazy(sprintf "%+01E" -10.0)) "-1.000000E+001"
    test "test5586" (lazy(sprintf "%+0*E" 7 -10.0)) "-1.000000E+001"
    test "test5587" (lazy(sprintf "%+0.5E" -10.0)) "-1.00000E+001"
    test "test5588" (lazy(sprintf "%+0.*E" 4 -10.0)) "-1.0000E+001"
    test "test5589" (lazy(sprintf "%+0*.*E" 5 4 -10.0)) "-1.0000E+001"
    test "test5590" (lazy(sprintf "%-+0E" -10.0)) "-1.000000E+001"
    test "test5591" (lazy(sprintf "%-+05E" -10.0)) "-1.000000E+001"
    test "test5592" (lazy(sprintf "%-+01E" -10.0)) "-1.000000E+001"
    test "test5593" (lazy(sprintf "%-+0*E" 7 -10.0)) "-1.000000E+001"
    test "test5594" (lazy(sprintf "%-+0.5E" -10.0)) "-1.00000E+001"
    test "test5595" (lazy(sprintf "%-+0.*E" 4 -10.0)) "-1.0000E+001"
    test "test5596" (lazy(sprintf "%-+0*.*E" 5 4 -10.0)) "-1.0000E+001"
    test "test5597" (lazy(sprintf "% E" -10.0)) "-1.000000E+001"
    test "test5598" (lazy(sprintf "% 5E" -10.0)) "-1.000000E+001"
    test "test5599" (lazy(sprintf "% 1E" -10.0)) "-1.000000E+001"
    test "test5600" (lazy(sprintf "% *E" 7 -10.0)) "-1.000000E+001"
    test "test5601" (lazy(sprintf "% .5E" -10.0)) "-1.00000E+001"
    test "test5602" (lazy(sprintf "% .*E" 4 -10.0)) "-1.0000E+001"
    test "test5603" (lazy(sprintf "% *.*E" 5 4 -10.0)) "-1.0000E+001"
    test "test5604" (lazy(sprintf "%- E" -10.0)) "-1.000000E+001"
    test "test5605" (lazy(sprintf "%- 5E" -10.0)) "-1.000000E+001"
    test "test5606" (lazy(sprintf "%- 1E" -10.0)) "-1.000000E+001"
    test "test5607" (lazy(sprintf "%- *E" 7 -10.0)) "-1.000000E+001"
    test "test5608" (lazy(sprintf "%- .5E" -10.0)) "-1.00000E+001"
    test "test5609" (lazy(sprintf "%- .*E" 4 -10.0)) "-1.0000E+001"
    test "test5610" (lazy(sprintf "%- *.*E" 5 4 -10.0)) "-1.0000E+001"
    test "test5611" (lazy(sprintf "% 0E" -10.0)) "-1.000000E+001"
    test "test5612" (lazy(sprintf "% 05E" -10.0)) "-1.000000E+001"
    test "test5613" (lazy(sprintf "% 01E" -10.0)) "-1.000000E+001"
    test "test5614" (lazy(sprintf "% 0*E" 7 -10.0)) "-1.000000E+001"
    test "test5615" (lazy(sprintf "% 0.5E" -10.0)) "-1.00000E+001"
    test "test5616" (lazy(sprintf "% 0.*E" 4 -10.0)) "-1.0000E+001"
    test "test5617" (lazy(sprintf "% 0*.*E" 5 4 -10.0)) "-1.0000E+001"
    test "test5618" (lazy(sprintf "%- 0E" -10.0)) "-1.000000E+001"
    test "test5619" (lazy(sprintf "%- 05E" -10.0)) "-1.000000E+001"
    test "test5620" (lazy(sprintf "%- 01E" -10.0)) "-1.000000E+001"
    test "test5621" (lazy(sprintf "%- 0*E" 7 -10.0)) "-1.000000E+001"
    test "test5622" (lazy(sprintf "%- 0.5E" -10.0)) "-1.00000E+001"
    test "test5623" (lazy(sprintf "%- 0.*E" 4 -10.0)) "-1.0000E+001"
    test "test5624" (lazy(sprintf "%- 0*.*E" 5 4 -10.0)) "-1.0000E+001"
    test "test5625" (lazy(sprintf "%E" 5.0f)) "5.000000E+000"
    test "test5626" (lazy(sprintf "%5E" 5.0f)) "5.000000E+000"
    test "test5627" (lazy(sprintf "%1E" 5.0f)) "5.000000E+000"
    test "test5628" (lazy(sprintf "%*E" 7 5.0f)) "5.000000E+000"
    test "test5629" (lazy(sprintf "%.5E" 5.0f)) "5.00000E+000"
    test "test5630" (lazy(sprintf "%.*E" 4 5.0f)) "5.0000E+000"
    test "test5631" (lazy(sprintf "%*.*E" 5 4 5.0f)) "5.0000E+000"
    test "test5632" (lazy(sprintf "%-E" 5.0f)) "5.000000E+000"
    test "test5633" (lazy(sprintf "%-5E" 5.0f)) "5.000000E+000"
    test "test5634" (lazy(sprintf "%-1E" 5.0f)) "5.000000E+000"
    test "test5635" (lazy(sprintf "%-*E" 7 5.0f)) "5.000000E+000"
    test "test5636" (lazy(sprintf "%-.5E" 5.0f)) "5.00000E+000"
    test "test5637" (lazy(sprintf "%-.*E" 4 5.0f)) "5.0000E+000"
    test "test5638" (lazy(sprintf "%-*.*E" 5 4 5.0f)) "5.0000E+000"
    test "test5639" (lazy(sprintf "%0E" 5.0f)) "5.000000E+000"
    test "test5640" (lazy(sprintf "%05E" 5.0f)) "5.000000E+000"
    test "test5641" (lazy(sprintf "%01E" 5.0f)) "5.000000E+000"
    test "test5642" (lazy(sprintf "%0*E" 7 5.0f)) "5.000000E+000"
    test "test5643" (lazy(sprintf "%0.5E" 5.0f)) "5.00000E+000"
    test "test5644" (lazy(sprintf "%0.*E" 4 5.0f)) "5.0000E+000"
    test "test5645" (lazy(sprintf "%0*.*E" 5 4 5.0f)) "5.0000E+000"
    test "test5646" (lazy(sprintf "%-0E" 5.0f)) "5.000000E+000"
    test "test5647" (lazy(sprintf "%-05E" 5.0f)) "5.000000E+000"
    test "test5648" (lazy(sprintf "%-01E" 5.0f)) "5.000000E+000"
    test "test5649" (lazy(sprintf "%-0*E" 7 5.0f)) "5.000000E+000"
    test "test5650" (lazy(sprintf "%-0.5E" 5.0f)) "5.00000E+000"
    test "test5651" (lazy(sprintf "%-0.*E" 4 5.0f)) "5.0000E+000"
    test "test5652" (lazy(sprintf "%-0*.*E" 5 4 5.0f)) "5.0000E+000"
    test "test5653" (lazy(sprintf "%+E" 5.0f)) "+5.000000E+000"
    test "test5654" (lazy(sprintf "%+5E" 5.0f)) "+5.000000E+000"
    test "test5655" (lazy(sprintf "%+1E" 5.0f)) "+5.000000E+000"
    test "test5656" (lazy(sprintf "%+*E" 7 5.0f)) "+5.000000E+000"
    test "test5657" (lazy(sprintf "%+.5E" 5.0f)) "+5.00000E+000"
    test "test5658" (lazy(sprintf "%+.*E" 4 5.0f)) "+5.0000E+000"
    test "test5659" (lazy(sprintf "%+*.*E" 5 4 5.0f)) "+5.0000E+000"
    test "test5660" (lazy(sprintf "%-+E" 5.0f)) "+5.000000E+000"
    test "test5661" (lazy(sprintf "%-+5E" 5.0f)) "+5.000000E+000"
    test "test5662" (lazy(sprintf "%-+1E" 5.0f)) "+5.000000E+000"
    test "test5663" (lazy(sprintf "%-+*E" 7 5.0f)) "+5.000000E+000"
    test "test5664" (lazy(sprintf "%-+.5E" 5.0f)) "+5.00000E+000"
    test "test5665" (lazy(sprintf "%-+.*E" 4 5.0f)) "+5.0000E+000"
    test "test5666" (lazy(sprintf "%-+*.*E" 5 4 5.0f)) "+5.0000E+000"
    test "test5667" (lazy(sprintf "%+0E" 5.0f)) "+5.000000E+000"
    test "test5668" (lazy(sprintf "%+05E" 5.0f)) "+5.000000E+000"
    test "test5669" (lazy(sprintf "%+01E" 5.0f)) "+5.000000E+000"
    test "test5670" (lazy(sprintf "%+0*E" 7 5.0f)) "+5.000000E+000"
    test "test5671" (lazy(sprintf "%+0.5E" 5.0f)) "+5.00000E+000"
    test "test5672" (lazy(sprintf "%+0.*E" 4 5.0f)) "+5.0000E+000"
    test "test5673" (lazy(sprintf "%+0*.*E" 5 4 5.0f)) "+5.0000E+000"
    test "test5674" (lazy(sprintf "%-+0E" 5.0f)) "+5.000000E+000"
    test "test5675" (lazy(sprintf "%-+05E" 5.0f)) "+5.000000E+000"
    test "test5676" (lazy(sprintf "%-+01E" 5.0f)) "+5.000000E+000"
    test "test5677" (lazy(sprintf "%-+0*E" 7 5.0f)) "+5.000000E+000"
    test "test5678" (lazy(sprintf "%-+0.5E" 5.0f)) "+5.00000E+000"
    test "test5679" (lazy(sprintf "%-+0.*E" 4 5.0f)) "+5.0000E+000"
    test "test5680" (lazy(sprintf "%-+0*.*E" 5 4 5.0f)) "+5.0000E+000"
    test "test5681" (lazy(sprintf "% E" 5.0f)) " 5.000000E+000"
    test "test5682" (lazy(sprintf "% 5E" 5.0f)) " 5.000000E+000"
    test "test5683" (lazy(sprintf "% 1E" 5.0f)) " 5.000000E+000"
    test "test5684" (lazy(sprintf "% *E" 7 5.0f)) " 5.000000E+000"
    test "test5685" (lazy(sprintf "% .5E" 5.0f)) " 5.00000E+000"
    test "test5686" (lazy(sprintf "% .*E" 4 5.0f)) " 5.0000E+000"
    test "test5687" (lazy(sprintf "% *.*E" 5 4 5.0f)) " 5.0000E+000"
    test "test5688" (lazy(sprintf "%- E" 5.0f)) " 5.000000E+000"
    test "test5689" (lazy(sprintf "%- 5E" 5.0f)) " 5.000000E+000"
    test "test5690" (lazy(sprintf "%- 1E" 5.0f)) " 5.000000E+000"
    test "test5691" (lazy(sprintf "%- *E" 7 5.0f)) " 5.000000E+000"
    test "test5692" (lazy(sprintf "%- .5E" 5.0f)) " 5.00000E+000"
    test "test5693" (lazy(sprintf "%- .*E" 4 5.0f)) " 5.0000E+000"
    test "test5694" (lazy(sprintf "%- *.*E" 5 4 5.0f)) " 5.0000E+000"
    test "test5695" (lazy(sprintf "% 0E" 5.0f)) " 5.000000E+000"
    test "test5696" (lazy(sprintf "% 05E" 5.0f)) " 5.000000E+000"
    test "test5697" (lazy(sprintf "% 01E" 5.0f)) " 5.000000E+000"
    test "test5698" (lazy(sprintf "% 0*E" 7 5.0f)) " 5.000000E+000"
    test "test5699" (lazy(sprintf "% 0.5E" 5.0f)) " 5.00000E+000"
    test "test5700" (lazy(sprintf "% 0.*E" 4 5.0f)) " 5.0000E+000"
    test "test5701" (lazy(sprintf "% 0*.*E" 5 4 5.0f)) " 5.0000E+000"
    test "test5702" (lazy(sprintf "%- 0E" 5.0f)) " 5.000000E+000"
    test "test5703" (lazy(sprintf "%- 05E" 5.0f)) " 5.000000E+000"
    test "test5704" (lazy(sprintf "%- 01E" 5.0f)) " 5.000000E+000"
    test "test5705" (lazy(sprintf "%- 0*E" 7 5.0f)) " 5.000000E+000"
    test "test5706" (lazy(sprintf "%- 0.5E" 5.0f)) " 5.00000E+000"
    test "test5707" (lazy(sprintf "%- 0.*E" 4 5.0f)) " 5.0000E+000"
    test "test5708" (lazy(sprintf "%- 0*.*E" 5 4 5.0f)) " 5.0000E+000"
    test "test5709" (lazy(sprintf "%E" -10.0f)) "-1.000000E+001"
    test "test5710" (lazy(sprintf "%5E" -10.0f)) "-1.000000E+001"
    test "test5711" (lazy(sprintf "%1E" -10.0f)) "-1.000000E+001"
    test "test5712" (lazy(sprintf "%*E" 7 -10.0f)) "-1.000000E+001"
    test "test5713" (lazy(sprintf "%.5E" -10.0f)) "-1.00000E+001"
    test "test5714" (lazy(sprintf "%.*E" 4 -10.0f)) "-1.0000E+001"
    test "test5715" (lazy(sprintf "%*.*E" 5 4 -10.0f)) "-1.0000E+001"
    test "test5716" (lazy(sprintf "%-E" -10.0f)) "-1.000000E+001"
    test "test5717" (lazy(sprintf "%-5E" -10.0f)) "-1.000000E+001"
    test "test5718" (lazy(sprintf "%-1E" -10.0f)) "-1.000000E+001"
    test "test5719" (lazy(sprintf "%-*E" 7 -10.0f)) "-1.000000E+001"
    test "test5720" (lazy(sprintf "%-.5E" -10.0f)) "-1.00000E+001"
    test "test5721" (lazy(sprintf "%-.*E" 4 -10.0f)) "-1.0000E+001"
    test "test5722" (lazy(sprintf "%-*.*E" 5 4 -10.0f)) "-1.0000E+001"
    test "test5723" (lazy(sprintf "%0E" -10.0f)) "-1.000000E+001"
    test "test5724" (lazy(sprintf "%05E" -10.0f)) "-1.000000E+001"
    test "test5725" (lazy(sprintf "%01E" -10.0f)) "-1.000000E+001"
    test "test5726" (lazy(sprintf "%0*E" 7 -10.0f)) "-1.000000E+001"
    test "test5727" (lazy(sprintf "%0.5E" -10.0f)) "-1.00000E+001"
    test "test5728" (lazy(sprintf "%0.*E" 4 -10.0f)) "-1.0000E+001"
    test "test5729" (lazy(sprintf "%0*.*E" 5 4 -10.0f)) "-1.0000E+001"
    test "test5730" (lazy(sprintf "%-0E" -10.0f)) "-1.000000E+001"
    test "test5731" (lazy(sprintf "%-05E" -10.0f)) "-1.000000E+001"
    test "test5732" (lazy(sprintf "%-01E" -10.0f)) "-1.000000E+001"
    test "test5733" (lazy(sprintf "%-0*E" 7 -10.0f)) "-1.000000E+001"
    test "test5734" (lazy(sprintf "%-0.5E" -10.0f)) "-1.00000E+001"
    test "test5735" (lazy(sprintf "%-0.*E" 4 -10.0f)) "-1.0000E+001"
    test "test5736" (lazy(sprintf "%-0*.*E" 5 4 -10.0f)) "-1.0000E+001"
    test "test5737" (lazy(sprintf "%+E" -10.0f)) "-1.000000E+001"
    test "test5738" (lazy(sprintf "%+5E" -10.0f)) "-1.000000E+001"
    test "test5739" (lazy(sprintf "%+1E" -10.0f)) "-1.000000E+001"
    test "test5740" (lazy(sprintf "%+*E" 7 -10.0f)) "-1.000000E+001"
    test "test5741" (lazy(sprintf "%+.5E" -10.0f)) "-1.00000E+001"
    test "test5742" (lazy(sprintf "%+.*E" 4 -10.0f)) "-1.0000E+001"
    test "test5743" (lazy(sprintf "%+*.*E" 5 4 -10.0f)) "-1.0000E+001"
    test "test5744" (lazy(sprintf "%-+E" -10.0f)) "-1.000000E+001"
    test "test5745" (lazy(sprintf "%-+5E" -10.0f)) "-1.000000E+001"
    test "test5746" (lazy(sprintf "%-+1E" -10.0f)) "-1.000000E+001"
    test "test5747" (lazy(sprintf "%-+*E" 7 -10.0f)) "-1.000000E+001"
    test "test5748" (lazy(sprintf "%-+.5E" -10.0f)) "-1.00000E+001"
    test "test5749" (lazy(sprintf "%-+.*E" 4 -10.0f)) "-1.0000E+001"
    test "test5750" (lazy(sprintf "%-+*.*E" 5 4 -10.0f)) "-1.0000E+001"
    test "test5751" (lazy(sprintf "%+0E" -10.0f)) "-1.000000E+001"
    test "test5752" (lazy(sprintf "%+05E" -10.0f)) "-1.000000E+001"
    test "test5753" (lazy(sprintf "%+01E" -10.0f)) "-1.000000E+001"
    test "test5754" (lazy(sprintf "%+0*E" 7 -10.0f)) "-1.000000E+001"
    test "test5755" (lazy(sprintf "%+0.5E" -10.0f)) "-1.00000E+001"
    test "test5756" (lazy(sprintf "%+0.*E" 4 -10.0f)) "-1.0000E+001"
    test "test5757" (lazy(sprintf "%+0*.*E" 5 4 -10.0f)) "-1.0000E+001"
    test "test5758" (lazy(sprintf "%-+0E" -10.0f)) "-1.000000E+001"
    test "test5759" (lazy(sprintf "%-+05E" -10.0f)) "-1.000000E+001"
    test "test5760" (lazy(sprintf "%-+01E" -10.0f)) "-1.000000E+001"
    test "test5761" (lazy(sprintf "%-+0*E" 7 -10.0f)) "-1.000000E+001"
    test "test5762" (lazy(sprintf "%-+0.5E" -10.0f)) "-1.00000E+001"
    test "test5763" (lazy(sprintf "%-+0.*E" 4 -10.0f)) "-1.0000E+001"
    test "test5764" (lazy(sprintf "%-+0*.*E" 5 4 -10.0f)) "-1.0000E+001"
    test "test5765" (lazy(sprintf "% E" -10.0f)) "-1.000000E+001"
    test "test5766" (lazy(sprintf "% 5E" -10.0f)) "-1.000000E+001"
    test "test5767" (lazy(sprintf "% 1E" -10.0f)) "-1.000000E+001"
    test "test5768" (lazy(sprintf "% *E" 7 -10.0f)) "-1.000000E+001"
    test "test5769" (lazy(sprintf "% .5E" -10.0f)) "-1.00000E+001"
    test "test5770" (lazy(sprintf "% .*E" 4 -10.0f)) "-1.0000E+001"
    test "test5771" (lazy(sprintf "% *.*E" 5 4 -10.0f)) "-1.0000E+001"
    test "test5772" (lazy(sprintf "%- E" -10.0f)) "-1.000000E+001"
    test "test5773" (lazy(sprintf "%- 5E" -10.0f)) "-1.000000E+001"
    test "test5774" (lazy(sprintf "%- 1E" -10.0f)) "-1.000000E+001"
    test "test5775" (lazy(sprintf "%- *E" 7 -10.0f)) "-1.000000E+001"
    test "test5776" (lazy(sprintf "%- .5E" -10.0f)) "-1.00000E+001"
    test "test5777" (lazy(sprintf "%- .*E" 4 -10.0f)) "-1.0000E+001"
    test "test5778" (lazy(sprintf "%- *.*E" 5 4 -10.0f)) "-1.0000E+001"
    test "test5779" (lazy(sprintf "% 0E" -10.0f)) "-1.000000E+001"
    test "test5780" (lazy(sprintf "% 05E" -10.0f)) "-1.000000E+001"
    test "test5781" (lazy(sprintf "% 01E" -10.0f)) "-1.000000E+001"
    test "test5782" (lazy(sprintf "% 0*E" 7 -10.0f)) "-1.000000E+001"
    test "test5783" (lazy(sprintf "% 0.5E" -10.0f)) "-1.00000E+001"
    test "test5784" (lazy(sprintf "% 0.*E" 4 -10.0f)) "-1.0000E+001"
    test "test5785" (lazy(sprintf "% 0*.*E" 5 4 -10.0f)) "-1.0000E+001"
    test "test5786" (lazy(sprintf "%- 0E" -10.0f)) "-1.000000E+001"
    test "test5787" (lazy(sprintf "%- 05E" -10.0f)) "-1.000000E+001"
    test "test5788" (lazy(sprintf "%- 01E" -10.0f)) "-1.000000E+001"
    test "test5789" (lazy(sprintf "%- 0*E" 7 -10.0f)) "-1.000000E+001"
    test "test5790" (lazy(sprintf "%- 0.5E" -10.0f)) "-1.00000E+001"
    test "test5791" (lazy(sprintf "%- 0.*E" 4 -10.0f)) "-1.0000E+001"
    test "test5792" (lazy(sprintf "%- 0*.*E" 5 4 -10.0f)) "-1.0000E+001"
    test "test5793" (lazy(sprintf "%E" 5.0M)) "5.000000E+000"
    test "test5794" (lazy(sprintf "%5E" 5.0M)) "5.000000E+000"
    test "test5795" (lazy(sprintf "%1E" 5.0M)) "5.000000E+000"
    test "test5796" (lazy(sprintf "%*E" 7 5.0M)) "5.000000E+000"
    test "test5797" (lazy(sprintf "%.5E" 5.0M)) "5.00000E+000"
    test "test5798" (lazy(sprintf "%.*E" 4 5.0M)) "5.0000E+000"
    test "test5799" (lazy(sprintf "%*.*E" 5 4 5.0M)) "5.0000E+000"
    test "test5800" (lazy(sprintf "%-E" 5.0M)) "5.000000E+000"
    test "test5801" (lazy(sprintf "%-5E" 5.0M)) "5.000000E+000"
    test "test5802" (lazy(sprintf "%-1E" 5.0M)) "5.000000E+000"
    test "test5803" (lazy(sprintf "%-*E" 7 5.0M)) "5.000000E+000"
    test "test5804" (lazy(sprintf "%-.5E" 5.0M)) "5.00000E+000"
    test "test5805" (lazy(sprintf "%-.*E" 4 5.0M)) "5.0000E+000"
    test "test5806" (lazy(sprintf "%-*.*E" 5 4 5.0M)) "5.0000E+000"
    test "test5807" (lazy(sprintf "%0E" 5.0M)) "5.000000E+000"
    test "test5808" (lazy(sprintf "%05E" 5.0M)) "5.000000E+000"
    test "test5809" (lazy(sprintf "%01E" 5.0M)) "5.000000E+000"
    test "test5810" (lazy(sprintf "%0*E" 7 5.0M)) "5.000000E+000"
    test "test5811" (lazy(sprintf "%0.5E" 5.0M)) "5.00000E+000"
    test "test5812" (lazy(sprintf "%0.*E" 4 5.0M)) "5.0000E+000"
    test "test5813" (lazy(sprintf "%0*.*E" 5 4 5.0M)) "5.0000E+000"
    test "test5814" (lazy(sprintf "%-0E" 5.0M)) "5.000000E+000"
    test "test5815" (lazy(sprintf "%-05E" 5.0M)) "5.000000E+000"
    test "test5816" (lazy(sprintf "%-01E" 5.0M)) "5.000000E+000"
    test "test5817" (lazy(sprintf "%-0*E" 7 5.0M)) "5.000000E+000"
    test "test5818" (lazy(sprintf "%-0.5E" 5.0M)) "5.00000E+000"
    test "test5819" (lazy(sprintf "%-0.*E" 4 5.0M)) "5.0000E+000"
    test "test5820" (lazy(sprintf "%-0*.*E" 5 4 5.0M)) "5.0000E+000"
    test "test5821" (lazy(sprintf "%+E" 5.0M)) "+5.000000E+000"
    test "test5822" (lazy(sprintf "%+5E" 5.0M)) "+5.000000E+000"
    test "test5823" (lazy(sprintf "%+1E" 5.0M)) "+5.000000E+000"
    test "test5824" (lazy(sprintf "%+*E" 7 5.0M)) "+5.000000E+000"
    test "test5825" (lazy(sprintf "%+.5E" 5.0M)) "+5.00000E+000"
    test "test5826" (lazy(sprintf "%+.*E" 4 5.0M)) "+5.0000E+000"
    test "test5827" (lazy(sprintf "%+*.*E" 5 4 5.0M)) "+5.0000E+000"
    test "test5828" (lazy(sprintf "%-+E" 5.0M)) "+5.000000E+000"
    test "test5829" (lazy(sprintf "%-+5E" 5.0M)) "+5.000000E+000"
    test "test5830" (lazy(sprintf "%-+1E" 5.0M)) "+5.000000E+000"
    test "test5831" (lazy(sprintf "%-+*E" 7 5.0M)) "+5.000000E+000"
    test "test5832" (lazy(sprintf "%-+.5E" 5.0M)) "+5.00000E+000"
    test "test5833" (lazy(sprintf "%-+.*E" 4 5.0M)) "+5.0000E+000"
    test "test5834" (lazy(sprintf "%-+*.*E" 5 4 5.0M)) "+5.0000E+000"
    test "test5835" (lazy(sprintf "%+0E" 5.0M)) "+5.000000E+000"
    test "test5836" (lazy(sprintf "%+05E" 5.0M)) "+5.000000E+000"
    test "test5837" (lazy(sprintf "%+01E" 5.0M)) "+5.000000E+000"
    test "test5838" (lazy(sprintf "%+0*E" 7 5.0M)) "+5.000000E+000"
    test "test5839" (lazy(sprintf "%+0.5E" 5.0M)) "+5.00000E+000"
    test "test5840" (lazy(sprintf "%+0.*E" 4 5.0M)) "+5.0000E+000"
    test "test5841" (lazy(sprintf "%+0*.*E" 5 4 5.0M)) "+5.0000E+000"
    test "test5842" (lazy(sprintf "%-+0E" 5.0M)) "+5.000000E+000"
    test "test5843" (lazy(sprintf "%-+05E" 5.0M)) "+5.000000E+000"
    test "test5844" (lazy(sprintf "%-+01E" 5.0M)) "+5.000000E+000"
    test "test5845" (lazy(sprintf "%-+0*E" 7 5.0M)) "+5.000000E+000"
    test "test5846" (lazy(sprintf "%-+0.5E" 5.0M)) "+5.00000E+000"
    test "test5847" (lazy(sprintf "%-+0.*E" 4 5.0M)) "+5.0000E+000"
    test "test5848" (lazy(sprintf "%-+0*.*E" 5 4 5.0M)) "+5.0000E+000"
    test "test5849" (lazy(sprintf "% E" 5.0M)) " 5.000000E+000"
    test "test5850" (lazy(sprintf "% 5E" 5.0M)) " 5.000000E+000"
    test "test5851" (lazy(sprintf "% 1E" 5.0M)) " 5.000000E+000"
    test "test5852" (lazy(sprintf "% *E" 7 5.0M)) " 5.000000E+000"
    test "test5853" (lazy(sprintf "% .5E" 5.0M)) " 5.00000E+000"
    test "test5854" (lazy(sprintf "% .*E" 4 5.0M)) " 5.0000E+000"
    test "test5855" (lazy(sprintf "% *.*E" 5 4 5.0M)) " 5.0000E+000"
    test "test5856" (lazy(sprintf "%- E" 5.0M)) " 5.000000E+000"
    test "test5857" (lazy(sprintf "%- 5E" 5.0M)) " 5.000000E+000"
    test "test5858" (lazy(sprintf "%- 1E" 5.0M)) " 5.000000E+000"
    test "test5859" (lazy(sprintf "%- *E" 7 5.0M)) " 5.000000E+000"
    test "test5860" (lazy(sprintf "%- .5E" 5.0M)) " 5.00000E+000"
    test "test5861" (lazy(sprintf "%- .*E" 4 5.0M)) " 5.0000E+000"
    test "test5862" (lazy(sprintf "%- *.*E" 5 4 5.0M)) " 5.0000E+000"
    test "test5863" (lazy(sprintf "% 0E" 5.0M)) " 5.000000E+000"
    test "test5864" (lazy(sprintf "% 05E" 5.0M)) " 5.000000E+000"
    test "test5865" (lazy(sprintf "% 01E" 5.0M)) " 5.000000E+000"
    test "test5866" (lazy(sprintf "% 0*E" 7 5.0M)) " 5.000000E+000"
    test "test5867" (lazy(sprintf "% 0.5E" 5.0M)) " 5.00000E+000"
    test "test5868" (lazy(sprintf "% 0.*E" 4 5.0M)) " 5.0000E+000"
    test "test5869" (lazy(sprintf "% 0*.*E" 5 4 5.0M)) " 5.0000E+000"
    test "test5870" (lazy(sprintf "%- 0E" 5.0M)) " 5.000000E+000"
    test "test5871" (lazy(sprintf "%- 05E" 5.0M)) " 5.000000E+000"
    test "test5872" (lazy(sprintf "%- 01E" 5.0M)) " 5.000000E+000"
    test "test5873" (lazy(sprintf "%- 0*E" 7 5.0M)) " 5.000000E+000"
    test "test5874" (lazy(sprintf "%- 0.5E" 5.0M)) " 5.00000E+000"
    test "test5875" (lazy(sprintf "%- 0.*E" 4 5.0M)) " 5.0000E+000"
    test "test5876" (lazy(sprintf "%- 0*.*E" 5 4 5.0M)) " 5.0000E+000"
    test "test5877" (lazy(sprintf "%E" -10.0M)) "-1.000000E+001"
    test "test5878" (lazy(sprintf "%5E" -10.0M)) "-1.000000E+001"
    test "test5879" (lazy(sprintf "%1E" -10.0M)) "-1.000000E+001"
    test "test5880" (lazy(sprintf "%*E" 7 -10.0M)) "-1.000000E+001"
    test "test5881" (lazy(sprintf "%.5E" -10.0M)) "-1.00000E+001"
    test "test5882" (lazy(sprintf "%.*E" 4 -10.0M)) "-1.0000E+001"
    test "test5883" (lazy(sprintf "%*.*E" 5 4 -10.0M)) "-1.0000E+001"
    test "test5884" (lazy(sprintf "%-E" -10.0M)) "-1.000000E+001"
    test "test5885" (lazy(sprintf "%-5E" -10.0M)) "-1.000000E+001"
    test "test5886" (lazy(sprintf "%-1E" -10.0M)) "-1.000000E+001"
    test "test5887" (lazy(sprintf "%-*E" 7 -10.0M)) "-1.000000E+001"
    test "test5888" (lazy(sprintf "%-.5E" -10.0M)) "-1.00000E+001"
    test "test5889" (lazy(sprintf "%-.*E" 4 -10.0M)) "-1.0000E+001"
    test "test5890" (lazy(sprintf "%-*.*E" 5 4 -10.0M)) "-1.0000E+001"
    test "test5891" (lazy(sprintf "%0E" -10.0M)) "-1.000000E+001"
    test "test5892" (lazy(sprintf "%05E" -10.0M)) "-1.000000E+001"
    test "test5893" (lazy(sprintf "%01E" -10.0M)) "-1.000000E+001"
    test "test5894" (lazy(sprintf "%0*E" 7 -10.0M)) "-1.000000E+001"
    test "test5895" (lazy(sprintf "%0.5E" -10.0M)) "-1.00000E+001"
    test "test5896" (lazy(sprintf "%0.*E" 4 -10.0M)) "-1.0000E+001"
    test "test5897" (lazy(sprintf "%0*.*E" 5 4 -10.0M)) "-1.0000E+001"
    test "test5898" (lazy(sprintf "%-0E" -10.0M)) "-1.000000E+001"
    test "test5899" (lazy(sprintf "%-05E" -10.0M)) "-1.000000E+001"
    test "test5900" (lazy(sprintf "%-01E" -10.0M)) "-1.000000E+001"
    test "test5901" (lazy(sprintf "%-0*E" 7 -10.0M)) "-1.000000E+001"
    test "test5902" (lazy(sprintf "%-0.5E" -10.0M)) "-1.00000E+001"
    test "test5903" (lazy(sprintf "%-0.*E" 4 -10.0M)) "-1.0000E+001"
    test "test5904" (lazy(sprintf "%-0*.*E" 5 4 -10.0M)) "-1.0000E+001"
    test "test5905" (lazy(sprintf "%+E" -10.0M)) "-1.000000E+001"
    test "test5906" (lazy(sprintf "%+5E" -10.0M)) "-1.000000E+001"
    test "test5907" (lazy(sprintf "%+1E" -10.0M)) "-1.000000E+001"
    test "test5908" (lazy(sprintf "%+*E" 7 -10.0M)) "-1.000000E+001"
    test "test5909" (lazy(sprintf "%+.5E" -10.0M)) "-1.00000E+001"
    test "test5910" (lazy(sprintf "%+.*E" 4 -10.0M)) "-1.0000E+001"
    test "test5911" (lazy(sprintf "%+*.*E" 5 4 -10.0M)) "-1.0000E+001"
    test "test5912" (lazy(sprintf "%-+E" -10.0M)) "-1.000000E+001"
    test "test5913" (lazy(sprintf "%-+5E" -10.0M)) "-1.000000E+001"
    test "test5914" (lazy(sprintf "%-+1E" -10.0M)) "-1.000000E+001"
    test "test5915" (lazy(sprintf "%-+*E" 7 -10.0M)) "-1.000000E+001"
    test "test5916" (lazy(sprintf "%-+.5E" -10.0M)) "-1.00000E+001"
    test "test5917" (lazy(sprintf "%-+.*E" 4 -10.0M)) "-1.0000E+001"
    test "test5918" (lazy(sprintf "%-+*.*E" 5 4 -10.0M)) "-1.0000E+001"
    test "test5919" (lazy(sprintf "%+0E" -10.0M)) "-1.000000E+001"
    test "test5920" (lazy(sprintf "%+05E" -10.0M)) "-1.000000E+001"
    test "test5921" (lazy(sprintf "%+01E" -10.0M)) "-1.000000E+001"
    test "test5922" (lazy(sprintf "%+0*E" 7 -10.0M)) "-1.000000E+001"
    test "test5923" (lazy(sprintf "%+0.5E" -10.0M)) "-1.00000E+001"
    test "test5924" (lazy(sprintf "%+0.*E" 4 -10.0M)) "-1.0000E+001"
    test "test5925" (lazy(sprintf "%+0*.*E" 5 4 -10.0M)) "-1.0000E+001"
    test "test5926" (lazy(sprintf "%-+0E" -10.0M)) "-1.000000E+001"
    test "test5927" (lazy(sprintf "%-+05E" -10.0M)) "-1.000000E+001"
    test "test5928" (lazy(sprintf "%-+01E" -10.0M)) "-1.000000E+001"
    test "test5929" (lazy(sprintf "%-+0*E" 7 -10.0M)) "-1.000000E+001"
    test "test5930" (lazy(sprintf "%-+0.5E" -10.0M)) "-1.00000E+001"
    test "test5931" (lazy(sprintf "%-+0.*E" 4 -10.0M)) "-1.0000E+001"
    test "test5932" (lazy(sprintf "%-+0*.*E" 5 4 -10.0M)) "-1.0000E+001"
    test "test5933" (lazy(sprintf "% E" -10.0M)) "-1.000000E+001"
    test "test5934" (lazy(sprintf "% 5E" -10.0M)) "-1.000000E+001"
    test "test5935" (lazy(sprintf "% 1E" -10.0M)) "-1.000000E+001"
    test "test5936" (lazy(sprintf "% *E" 7 -10.0M)) "-1.000000E+001"
    test "test5937" (lazy(sprintf "% .5E" -10.0M)) "-1.00000E+001"
    test "test5938" (lazy(sprintf "% .*E" 4 -10.0M)) "-1.0000E+001"
    test "test5939" (lazy(sprintf "% *.*E" 5 4 -10.0M)) "-1.0000E+001"
    test "test5940" (lazy(sprintf "%- E" -10.0M)) "-1.000000E+001"
    test "test5941" (lazy(sprintf "%- 5E" -10.0M)) "-1.000000E+001"
    test "test5942" (lazy(sprintf "%- 1E" -10.0M)) "-1.000000E+001"
    test "test5943" (lazy(sprintf "%- *E" 7 -10.0M)) "-1.000000E+001"
    test "test5944" (lazy(sprintf "%- .5E" -10.0M)) "-1.00000E+001"
    test "test5945" (lazy(sprintf "%- .*E" 4 -10.0M)) "-1.0000E+001"
    test "test5946" (lazy(sprintf "%- *.*E" 5 4 -10.0M)) "-1.0000E+001"
    test "test5947" (lazy(sprintf "% 0E" -10.0M)) "-1.000000E+001"
    test "test5948" (lazy(sprintf "% 05E" -10.0M)) "-1.000000E+001"
    test "test5949" (lazy(sprintf "% 01E" -10.0M)) "-1.000000E+001"
    test "test5950" (lazy(sprintf "% 0*E" 7 -10.0M)) "-1.000000E+001"
    test "test5951" (lazy(sprintf "% 0.5E" -10.0M)) "-1.00000E+001"
    test "test5952" (lazy(sprintf "% 0.*E" 4 -10.0M)) "-1.0000E+001"
    test "test5953" (lazy(sprintf "% 0*.*E" 5 4 -10.0M)) "-1.0000E+001"
    test "test5954" (lazy(sprintf "%- 0E" -10.0M)) "-1.000000E+001"
    test "test5955" (lazy(sprintf "%- 05E" -10.0M)) "-1.000000E+001"
    test "test5956" (lazy(sprintf "%- 01E" -10.0M)) "-1.000000E+001"
    test "test5957" (lazy(sprintf "%- 0*E" 7 -10.0M)) "-1.000000E+001"
    test "test5958" (lazy(sprintf "%- 0.5E" -10.0M)) "-1.00000E+001"
    test "test5959" (lazy(sprintf "%- 0.*E" 4 -10.0M)) "-1.0000E+001"
    test "test5960" (lazy(sprintf "%- 0*.*E" 5 4 -10.0M)) "-1.0000E+001"
    test "test5961" (lazy(sprintf "%f" 5.0)) "5.000000"
    test "test5962" (lazy(sprintf "%5f" 5.0)) "5.000000"
    test "test5963" (lazy(sprintf "%1f" 5.0)) "5.000000"
    test "test5964" (lazy(sprintf "%*f" 7 5.0)) "5.000000"
    test "test5965" (lazy(sprintf "%.5f" 5.0)) "5.00000"
    test "test5966" (lazy(sprintf "%.*f" 4 5.0)) "5.0000"
    test "test5967" (lazy(sprintf "%*.*f" 5 4 5.0)) "5.0000"
    test "test5968" (lazy(sprintf "%-f" 5.0)) "5.000000"
    test "test5969" (lazy(sprintf "%-5f" 5.0)) "5.000000"
    test "test5970" (lazy(sprintf "%-1f" 5.0)) "5.000000"
    test "test5971" (lazy(sprintf "%-*f" 7 5.0)) "5.000000"
    test "test5972" (lazy(sprintf "%-.5f" 5.0)) "5.00000"
    test "test5973" (lazy(sprintf "%-.*f" 4 5.0)) "5.0000"
    test "test5974" (lazy(sprintf "%-*.*f" 5 4 5.0)) "5.0000"
    test "test5975" (lazy(sprintf "%0f" 5.0)) "5.000000"
    test "test5976" (lazy(sprintf "%05f" 5.0)) "5.000000"
    test "test5977" (lazy(sprintf "%01f" 5.0)) "5.000000"
    test "test5978" (lazy(sprintf "%0*f" 7 5.0)) "5.000000"
    test "test5979" (lazy(sprintf "%0.5f" 5.0)) "5.00000"
    test "test5980" (lazy(sprintf "%0.*f" 4 5.0)) "5.0000"
    test "test5981" (lazy(sprintf "%0*.*f" 5 4 5.0)) "5.0000"
    test "test5982" (lazy(sprintf "%-0f" 5.0)) "5.000000"
    test "test5983" (lazy(sprintf "%-05f" 5.0)) "5.000000"
    test "test5984" (lazy(sprintf "%-01f" 5.0)) "5.000000"
    test "test5985" (lazy(sprintf "%-0*f" 7 5.0)) "5.000000"
    test "test5986" (lazy(sprintf "%-0.5f" 5.0)) "5.00000"
    test "test5987" (lazy(sprintf "%-0.*f" 4 5.0)) "5.0000"
    test "test5988" (lazy(sprintf "%-0*.*f" 5 4 5.0)) "5.0000"
    test "test5989" (lazy(sprintf "%+f" 5.0)) "+5.000000"
    test "test5990" (lazy(sprintf "%+5f" 5.0)) "+5.000000"
    test "test5991" (lazy(sprintf "%+1f" 5.0)) "+5.000000"
    test "test5992" (lazy(sprintf "%+*f" 7 5.0)) "+5.000000"
    test "test5993" (lazy(sprintf "%+.5f" 5.0)) "+5.00000"
    test "test5994" (lazy(sprintf "%+.*f" 4 5.0)) "+5.0000"
    test "test5995" (lazy(sprintf "%+*.*f" 5 4 5.0)) "+5.0000"
    test "test5996" (lazy(sprintf "%-+f" 5.0)) "+5.000000"
    test "test5997" (lazy(sprintf "%-+5f" 5.0)) "+5.000000"
    test "test5998" (lazy(sprintf "%-+1f" 5.0)) "+5.000000"
    test "test5999" (lazy(sprintf "%-+*f" 7 5.0)) "+5.000000"
    test "test6000" (lazy(sprintf "%-+.5f" 5.0)) "+5.00000"
let func6000()=
    test "test6001" (lazy(sprintf "%-+.*f" 4 5.0)) "+5.0000"
    test "test6002" (lazy(sprintf "%-+*.*f" 5 4 5.0)) "+5.0000"
    test "test6003" (lazy(sprintf "%+0f" 5.0)) "+5.000000"
    test "test6004" (lazy(sprintf "%+05f" 5.0)) "+5.000000"
    test "test6005" (lazy(sprintf "%+01f" 5.0)) "+5.000000"
    test "test6006" (lazy(sprintf "%+0*f" 7 5.0)) "+5.000000"
    test "test6007" (lazy(sprintf "%+0.5f" 5.0)) "+5.00000"
    test "test6008" (lazy(sprintf "%+0.*f" 4 5.0)) "+5.0000"
    test "test6009" (lazy(sprintf "%+0*.*f" 5 4 5.0)) "+5.0000"
    test "test6010" (lazy(sprintf "%-+0f" 5.0)) "+5.000000"
    test "test6011" (lazy(sprintf "%-+05f" 5.0)) "+5.000000"
    test "test6012" (lazy(sprintf "%-+01f" 5.0)) "+5.000000"
    test "test6013" (lazy(sprintf "%-+0*f" 7 5.0)) "+5.000000"
    test "test6014" (lazy(sprintf "%-+0.5f" 5.0)) "+5.00000"
    test "test6015" (lazy(sprintf "%-+0.*f" 4 5.0)) "+5.0000"
    test "test6016" (lazy(sprintf "%-+0*.*f" 5 4 5.0)) "+5.0000"
    test "test6017" (lazy(sprintf "% f" 5.0)) " 5.000000"
    test "test6018" (lazy(sprintf "% 5f" 5.0)) " 5.000000"
    test "test6019" (lazy(sprintf "% 1f" 5.0)) " 5.000000"
    test "test6020" (lazy(sprintf "% *f" 7 5.0)) " 5.000000"
    test "test6021" (lazy(sprintf "% .5f" 5.0)) " 5.00000"
    test "test6022" (lazy(sprintf "% .*f" 4 5.0)) " 5.0000"
    test "test6023" (lazy(sprintf "% *.*f" 5 4 5.0)) " 5.0000"
    test "test6024" (lazy(sprintf "%- f" 5.0)) " 5.000000"
    test "test6025" (lazy(sprintf "%- 5f" 5.0)) " 5.000000"
    test "test6026" (lazy(sprintf "%- 1f" 5.0)) " 5.000000"
    test "test6027" (lazy(sprintf "%- *f" 7 5.0)) " 5.000000"
    test "test6028" (lazy(sprintf "%- .5f" 5.0)) " 5.00000"
    test "test6029" (lazy(sprintf "%- .*f" 4 5.0)) " 5.0000"
    test "test6030" (lazy(sprintf "%- *.*f" 5 4 5.0)) " 5.0000"
    test "test6031" (lazy(sprintf "% 0f" 5.0)) " 5.000000"
    test "test6032" (lazy(sprintf "% 05f" 5.0)) " 5.000000"
    test "test6033" (lazy(sprintf "% 01f" 5.0)) " 5.000000"
    test "test6034" (lazy(sprintf "% 0*f" 7 5.0)) " 5.000000"
    test "test6035" (lazy(sprintf "% 0.5f" 5.0)) " 5.00000"
    test "test6036" (lazy(sprintf "% 0.*f" 4 5.0)) " 5.0000"
    test "test6037" (lazy(sprintf "% 0*.*f" 5 4 5.0)) " 5.0000"
    test "test6038" (lazy(sprintf "%- 0f" 5.0)) " 5.000000"
    test "test6039" (lazy(sprintf "%- 05f" 5.0)) " 5.000000"
    test "test6040" (lazy(sprintf "%- 01f" 5.0)) " 5.000000"
    test "test6041" (lazy(sprintf "%- 0*f" 7 5.0)) " 5.000000"
    test "test6042" (lazy(sprintf "%- 0.5f" 5.0)) " 5.00000"
    test "test6043" (lazy(sprintf "%- 0.*f" 4 5.0)) " 5.0000"
    test "test6044" (lazy(sprintf "%- 0*.*f" 5 4 5.0)) " 5.0000"
    test "test6045" (lazy(sprintf "%f" -10.0)) "-10.000000"
    test "test6046" (lazy(sprintf "%5f" -10.0)) "-10.000000"
    test "test6047" (lazy(sprintf "%1f" -10.0)) "-10.000000"
    test "test6048" (lazy(sprintf "%*f" 7 -10.0)) "-10.000000"
    test "test6049" (lazy(sprintf "%.5f" -10.0)) "-10.00000"
    test "test6050" (lazy(sprintf "%.*f" 4 -10.0)) "-10.0000"
    test "test6051" (lazy(sprintf "%*.*f" 5 4 -10.0)) "-10.0000"
    test "test6052" (lazy(sprintf "%-f" -10.0)) "-10.000000"
    test "test6053" (lazy(sprintf "%-5f" -10.0)) "-10.000000"
    test "test6054" (lazy(sprintf "%-1f" -10.0)) "-10.000000"
    test "test6055" (lazy(sprintf "%-*f" 7 -10.0)) "-10.000000"
    test "test6056" (lazy(sprintf "%-.5f" -10.0)) "-10.00000"
    test "test6057" (lazy(sprintf "%-.*f" 4 -10.0)) "-10.0000"
    test "test6058" (lazy(sprintf "%-*.*f" 5 4 -10.0)) "-10.0000"
    test "test6059" (lazy(sprintf "%0f" -10.0)) "-10.000000"
    test "test6060" (lazy(sprintf "%05f" -10.0)) "-10.000000"
    test "test6061" (lazy(sprintf "%01f" -10.0)) "-10.000000"
    test "test6062" (lazy(sprintf "%0*f" 7 -10.0)) "-10.000000"
    test "test6063" (lazy(sprintf "%0.5f" -10.0)) "-10.00000"
    test "test6064" (lazy(sprintf "%0.*f" 4 -10.0)) "-10.0000"
    test "test6065" (lazy(sprintf "%0*.*f" 5 4 -10.0)) "-10.0000"
    test "test6066" (lazy(sprintf "%-0f" -10.0)) "-10.000000"
    test "test6067" (lazy(sprintf "%-05f" -10.0)) "-10.000000"
    test "test6068" (lazy(sprintf "%-01f" -10.0)) "-10.000000"
    test "test6069" (lazy(sprintf "%-0*f" 7 -10.0)) "-10.000000"
    test "test6070" (lazy(sprintf "%-0.5f" -10.0)) "-10.00000"
    test "test6071" (lazy(sprintf "%-0.*f" 4 -10.0)) "-10.0000"
    test "test6072" (lazy(sprintf "%-0*.*f" 5 4 -10.0)) "-10.0000"
    test "test6073" (lazy(sprintf "%+f" -10.0)) "-10.000000"
    test "test6074" (lazy(sprintf "%+5f" -10.0)) "-10.000000"
    test "test6075" (lazy(sprintf "%+1f" -10.0)) "-10.000000"
    test "test6076" (lazy(sprintf "%+*f" 7 -10.0)) "-10.000000"
    test "test6077" (lazy(sprintf "%+.5f" -10.0)) "-10.00000"
    test "test6078" (lazy(sprintf "%+.*f" 4 -10.0)) "-10.0000"
    test "test6079" (lazy(sprintf "%+*.*f" 5 4 -10.0)) "-10.0000"
    test "test6080" (lazy(sprintf "%-+f" -10.0)) "-10.000000"
    test "test6081" (lazy(sprintf "%-+5f" -10.0)) "-10.000000"
    test "test6082" (lazy(sprintf "%-+1f" -10.0)) "-10.000000"
    test "test6083" (lazy(sprintf "%-+*f" 7 -10.0)) "-10.000000"
    test "test6084" (lazy(sprintf "%-+.5f" -10.0)) "-10.00000"
    test "test6085" (lazy(sprintf "%-+.*f" 4 -10.0)) "-10.0000"
    test "test6086" (lazy(sprintf "%-+*.*f" 5 4 -10.0)) "-10.0000"
    test "test6087" (lazy(sprintf "%+0f" -10.0)) "-10.000000"
    test "test6088" (lazy(sprintf "%+05f" -10.0)) "-10.000000"
    test "test6089" (lazy(sprintf "%+01f" -10.0)) "-10.000000"
    test "test6090" (lazy(sprintf "%+0*f" 7 -10.0)) "-10.000000"
    test "test6091" (lazy(sprintf "%+0.5f" -10.0)) "-10.00000"
    test "test6092" (lazy(sprintf "%+0.*f" 4 -10.0)) "-10.0000"
    test "test6093" (lazy(sprintf "%+0*.*f" 5 4 -10.0)) "-10.0000"
    test "test6094" (lazy(sprintf "%-+0f" -10.0)) "-10.000000"
    test "test6095" (lazy(sprintf "%-+05f" -10.0)) "-10.000000"
    test "test6096" (lazy(sprintf "%-+01f" -10.0)) "-10.000000"
    test "test6097" (lazy(sprintf "%-+0*f" 7 -10.0)) "-10.000000"
    test "test6098" (lazy(sprintf "%-+0.5f" -10.0)) "-10.00000"
    test "test6099" (lazy(sprintf "%-+0.*f" 4 -10.0)) "-10.0000"
    test "test6100" (lazy(sprintf "%-+0*.*f" 5 4 -10.0)) "-10.0000"
    test "test6101" (lazy(sprintf "% f" -10.0)) "-10.000000"
    test "test6102" (lazy(sprintf "% 5f" -10.0)) "-10.000000"
    test "test6103" (lazy(sprintf "% 1f" -10.0)) "-10.000000"
    test "test6104" (lazy(sprintf "% *f" 7 -10.0)) "-10.000000"
    test "test6105" (lazy(sprintf "% .5f" -10.0)) "-10.00000"
    test "test6106" (lazy(sprintf "% .*f" 4 -10.0)) "-10.0000"
    test "test6107" (lazy(sprintf "% *.*f" 5 4 -10.0)) "-10.0000"
    test "test6108" (lazy(sprintf "%- f" -10.0)) "-10.000000"
    test "test6109" (lazy(sprintf "%- 5f" -10.0)) "-10.000000"
    test "test6110" (lazy(sprintf "%- 1f" -10.0)) "-10.000000"
    test "test6111" (lazy(sprintf "%- *f" 7 -10.0)) "-10.000000"
    test "test6112" (lazy(sprintf "%- .5f" -10.0)) "-10.00000"
    test "test6113" (lazy(sprintf "%- .*f" 4 -10.0)) "-10.0000"
    test "test6114" (lazy(sprintf "%- *.*f" 5 4 -10.0)) "-10.0000"
    test "test6115" (lazy(sprintf "% 0f" -10.0)) "-10.000000"
    test "test6116" (lazy(sprintf "% 05f" -10.0)) "-10.000000"
    test "test6117" (lazy(sprintf "% 01f" -10.0)) "-10.000000"
    test "test6118" (lazy(sprintf "% 0*f" 7 -10.0)) "-10.000000"
    test "test6119" (lazy(sprintf "% 0.5f" -10.0)) "-10.00000"
    test "test6120" (lazy(sprintf "% 0.*f" 4 -10.0)) "-10.0000"
    test "test6121" (lazy(sprintf "% 0*.*f" 5 4 -10.0)) "-10.0000"
    test "test6122" (lazy(sprintf "%- 0f" -10.0)) "-10.000000"
    test "test6123" (lazy(sprintf "%- 05f" -10.0)) "-10.000000"
    test "test6124" (lazy(sprintf "%- 01f" -10.0)) "-10.000000"
    test "test6125" (lazy(sprintf "%- 0*f" 7 -10.0)) "-10.000000"
    test "test6126" (lazy(sprintf "%- 0.5f" -10.0)) "-10.00000"
    test "test6127" (lazy(sprintf "%- 0.*f" 4 -10.0)) "-10.0000"
    test "test6128" (lazy(sprintf "%- 0*.*f" 5 4 -10.0)) "-10.0000"
    test "test6129" (lazy(sprintf "%f" 5.0f)) "5.000000"
    test "test6130" (lazy(sprintf "%5f" 5.0f)) "5.000000"
    test "test6131" (lazy(sprintf "%1f" 5.0f)) "5.000000"
    test "test6132" (lazy(sprintf "%*f" 7 5.0f)) "5.000000"
    test "test6133" (lazy(sprintf "%.5f" 5.0f)) "5.00000"
    test "test6134" (lazy(sprintf "%.*f" 4 5.0f)) "5.0000"
    test "test6135" (lazy(sprintf "%*.*f" 5 4 5.0f)) "5.0000"
    test "test6136" (lazy(sprintf "%-f" 5.0f)) "5.000000"
    test "test6137" (lazy(sprintf "%-5f" 5.0f)) "5.000000"
    test "test6138" (lazy(sprintf "%-1f" 5.0f)) "5.000000"
    test "test6139" (lazy(sprintf "%-*f" 7 5.0f)) "5.000000"
    test "test6140" (lazy(sprintf "%-.5f" 5.0f)) "5.00000"
    test "test6141" (lazy(sprintf "%-.*f" 4 5.0f)) "5.0000"
    test "test6142" (lazy(sprintf "%-*.*f" 5 4 5.0f)) "5.0000"
    test "test6143" (lazy(sprintf "%0f" 5.0f)) "5.000000"
    test "test6144" (lazy(sprintf "%05f" 5.0f)) "5.000000"
    test "test6145" (lazy(sprintf "%01f" 5.0f)) "5.000000"
    test "test6146" (lazy(sprintf "%0*f" 7 5.0f)) "5.000000"
    test "test6147" (lazy(sprintf "%0.5f" 5.0f)) "5.00000"
    test "test6148" (lazy(sprintf "%0.*f" 4 5.0f)) "5.0000"
    test "test6149" (lazy(sprintf "%0*.*f" 5 4 5.0f)) "5.0000"
    test "test6150" (lazy(sprintf "%-0f" 5.0f)) "5.000000"
    test "test6151" (lazy(sprintf "%-05f" 5.0f)) "5.000000"
    test "test6152" (lazy(sprintf "%-01f" 5.0f)) "5.000000"
    test "test6153" (lazy(sprintf "%-0*f" 7 5.0f)) "5.000000"
    test "test6154" (lazy(sprintf "%-0.5f" 5.0f)) "5.00000"
    test "test6155" (lazy(sprintf "%-0.*f" 4 5.0f)) "5.0000"
    test "test6156" (lazy(sprintf "%-0*.*f" 5 4 5.0f)) "5.0000"
    test "test6157" (lazy(sprintf "%+f" 5.0f)) "+5.000000"
    test "test6158" (lazy(sprintf "%+5f" 5.0f)) "+5.000000"
    test "test6159" (lazy(sprintf "%+1f" 5.0f)) "+5.000000"
    test "test6160" (lazy(sprintf "%+*f" 7 5.0f)) "+5.000000"
    test "test6161" (lazy(sprintf "%+.5f" 5.0f)) "+5.00000"
    test "test6162" (lazy(sprintf "%+.*f" 4 5.0f)) "+5.0000"
    test "test6163" (lazy(sprintf "%+*.*f" 5 4 5.0f)) "+5.0000"
    test "test6164" (lazy(sprintf "%-+f" 5.0f)) "+5.000000"
    test "test6165" (lazy(sprintf "%-+5f" 5.0f)) "+5.000000"
    test "test6166" (lazy(sprintf "%-+1f" 5.0f)) "+5.000000"
    test "test6167" (lazy(sprintf "%-+*f" 7 5.0f)) "+5.000000"
    test "test6168" (lazy(sprintf "%-+.5f" 5.0f)) "+5.00000"
    test "test6169" (lazy(sprintf "%-+.*f" 4 5.0f)) "+5.0000"
    test "test6170" (lazy(sprintf "%-+*.*f" 5 4 5.0f)) "+5.0000"
    test "test6171" (lazy(sprintf "%+0f" 5.0f)) "+5.000000"
    test "test6172" (lazy(sprintf "%+05f" 5.0f)) "+5.000000"
    test "test6173" (lazy(sprintf "%+01f" 5.0f)) "+5.000000"
    test "test6174" (lazy(sprintf "%+0*f" 7 5.0f)) "+5.000000"
    test "test6175" (lazy(sprintf "%+0.5f" 5.0f)) "+5.00000"
    test "test6176" (lazy(sprintf "%+0.*f" 4 5.0f)) "+5.0000"
    test "test6177" (lazy(sprintf "%+0*.*f" 5 4 5.0f)) "+5.0000"
    test "test6178" (lazy(sprintf "%-+0f" 5.0f)) "+5.000000"
    test "test6179" (lazy(sprintf "%-+05f" 5.0f)) "+5.000000"
    test "test6180" (lazy(sprintf "%-+01f" 5.0f)) "+5.000000"
    test "test6181" (lazy(sprintf "%-+0*f" 7 5.0f)) "+5.000000"
    test "test6182" (lazy(sprintf "%-+0.5f" 5.0f)) "+5.00000"
    test "test6183" (lazy(sprintf "%-+0.*f" 4 5.0f)) "+5.0000"
    test "test6184" (lazy(sprintf "%-+0*.*f" 5 4 5.0f)) "+5.0000"
    test "test6185" (lazy(sprintf "% f" 5.0f)) " 5.000000"
    test "test6186" (lazy(sprintf "% 5f" 5.0f)) " 5.000000"
    test "test6187" (lazy(sprintf "% 1f" 5.0f)) " 5.000000"
    test "test6188" (lazy(sprintf "% *f" 7 5.0f)) " 5.000000"
    test "test6189" (lazy(sprintf "% .5f" 5.0f)) " 5.00000"
    test "test6190" (lazy(sprintf "% .*f" 4 5.0f)) " 5.0000"
    test "test6191" (lazy(sprintf "% *.*f" 5 4 5.0f)) " 5.0000"
    test "test6192" (lazy(sprintf "%- f" 5.0f)) " 5.000000"
    test "test6193" (lazy(sprintf "%- 5f" 5.0f)) " 5.000000"
    test "test6194" (lazy(sprintf "%- 1f" 5.0f)) " 5.000000"
    test "test6195" (lazy(sprintf "%- *f" 7 5.0f)) " 5.000000"
    test "test6196" (lazy(sprintf "%- .5f" 5.0f)) " 5.00000"
    test "test6197" (lazy(sprintf "%- .*f" 4 5.0f)) " 5.0000"
    test "test6198" (lazy(sprintf "%- *.*f" 5 4 5.0f)) " 5.0000"
    test "test6199" (lazy(sprintf "% 0f" 5.0f)) " 5.000000"
    test "test6200" (lazy(sprintf "% 05f" 5.0f)) " 5.000000"
    test "test6201" (lazy(sprintf "% 01f" 5.0f)) " 5.000000"
    test "test6202" (lazy(sprintf "% 0*f" 7 5.0f)) " 5.000000"
    test "test6203" (lazy(sprintf "% 0.5f" 5.0f)) " 5.00000"
    test "test6204" (lazy(sprintf "% 0.*f" 4 5.0f)) " 5.0000"
    test "test6205" (lazy(sprintf "% 0*.*f" 5 4 5.0f)) " 5.0000"
    test "test6206" (lazy(sprintf "%- 0f" 5.0f)) " 5.000000"
    test "test6207" (lazy(sprintf "%- 05f" 5.0f)) " 5.000000"
    test "test6208" (lazy(sprintf "%- 01f" 5.0f)) " 5.000000"
    test "test6209" (lazy(sprintf "%- 0*f" 7 5.0f)) " 5.000000"
    test "test6210" (lazy(sprintf "%- 0.5f" 5.0f)) " 5.00000"
    test "test6211" (lazy(sprintf "%- 0.*f" 4 5.0f)) " 5.0000"
    test "test6212" (lazy(sprintf "%- 0*.*f" 5 4 5.0f)) " 5.0000"
    test "test6213" (lazy(sprintf "%f" -10.0f)) "-10.000000"
    test "test6214" (lazy(sprintf "%5f" -10.0f)) "-10.000000"
    test "test6215" (lazy(sprintf "%1f" -10.0f)) "-10.000000"
    test "test6216" (lazy(sprintf "%*f" 7 -10.0f)) "-10.000000"
    test "test6217" (lazy(sprintf "%.5f" -10.0f)) "-10.00000"
    test "test6218" (lazy(sprintf "%.*f" 4 -10.0f)) "-10.0000"
    test "test6219" (lazy(sprintf "%*.*f" 5 4 -10.0f)) "-10.0000"
    test "test6220" (lazy(sprintf "%-f" -10.0f)) "-10.000000"
    test "test6221" (lazy(sprintf "%-5f" -10.0f)) "-10.000000"
    test "test6222" (lazy(sprintf "%-1f" -10.0f)) "-10.000000"
    test "test6223" (lazy(sprintf "%-*f" 7 -10.0f)) "-10.000000"
    test "test6224" (lazy(sprintf "%-.5f" -10.0f)) "-10.00000"
    test "test6225" (lazy(sprintf "%-.*f" 4 -10.0f)) "-10.0000"
    test "test6226" (lazy(sprintf "%-*.*f" 5 4 -10.0f)) "-10.0000"
    test "test6227" (lazy(sprintf "%0f" -10.0f)) "-10.000000"
    test "test6228" (lazy(sprintf "%05f" -10.0f)) "-10.000000"
    test "test6229" (lazy(sprintf "%01f" -10.0f)) "-10.000000"
    test "test6230" (lazy(sprintf "%0*f" 7 -10.0f)) "-10.000000"
    test "test6231" (lazy(sprintf "%0.5f" -10.0f)) "-10.00000"
    test "test6232" (lazy(sprintf "%0.*f" 4 -10.0f)) "-10.0000"
    test "test6233" (lazy(sprintf "%0*.*f" 5 4 -10.0f)) "-10.0000"
    test "test6234" (lazy(sprintf "%-0f" -10.0f)) "-10.000000"
    test "test6235" (lazy(sprintf "%-05f" -10.0f)) "-10.000000"
    test "test6236" (lazy(sprintf "%-01f" -10.0f)) "-10.000000"
    test "test6237" (lazy(sprintf "%-0*f" 7 -10.0f)) "-10.000000"
    test "test6238" (lazy(sprintf "%-0.5f" -10.0f)) "-10.00000"
    test "test6239" (lazy(sprintf "%-0.*f" 4 -10.0f)) "-10.0000"
    test "test6240" (lazy(sprintf "%-0*.*f" 5 4 -10.0f)) "-10.0000"
    test "test6241" (lazy(sprintf "%+f" -10.0f)) "-10.000000"
    test "test6242" (lazy(sprintf "%+5f" -10.0f)) "-10.000000"
    test "test6243" (lazy(sprintf "%+1f" -10.0f)) "-10.000000"
    test "test6244" (lazy(sprintf "%+*f" 7 -10.0f)) "-10.000000"
    test "test6245" (lazy(sprintf "%+.5f" -10.0f)) "-10.00000"
    test "test6246" (lazy(sprintf "%+.*f" 4 -10.0f)) "-10.0000"
    test "test6247" (lazy(sprintf "%+*.*f" 5 4 -10.0f)) "-10.0000"
    test "test6248" (lazy(sprintf "%-+f" -10.0f)) "-10.000000"
    test "test6249" (lazy(sprintf "%-+5f" -10.0f)) "-10.000000"
    test "test6250" (lazy(sprintf "%-+1f" -10.0f)) "-10.000000"
    test "test6251" (lazy(sprintf "%-+*f" 7 -10.0f)) "-10.000000"
    test "test6252" (lazy(sprintf "%-+.5f" -10.0f)) "-10.00000"
    test "test6253" (lazy(sprintf "%-+.*f" 4 -10.0f)) "-10.0000"
    test "test6254" (lazy(sprintf "%-+*.*f" 5 4 -10.0f)) "-10.0000"
    test "test6255" (lazy(sprintf "%+0f" -10.0f)) "-10.000000"
    test "test6256" (lazy(sprintf "%+05f" -10.0f)) "-10.000000"
    test "test6257" (lazy(sprintf "%+01f" -10.0f)) "-10.000000"
    test "test6258" (lazy(sprintf "%+0*f" 7 -10.0f)) "-10.000000"
    test "test6259" (lazy(sprintf "%+0.5f" -10.0f)) "-10.00000"
    test "test6260" (lazy(sprintf "%+0.*f" 4 -10.0f)) "-10.0000"
    test "test6261" (lazy(sprintf "%+0*.*f" 5 4 -10.0f)) "-10.0000"
    test "test6262" (lazy(sprintf "%-+0f" -10.0f)) "-10.000000"
    test "test6263" (lazy(sprintf "%-+05f" -10.0f)) "-10.000000"
    test "test6264" (lazy(sprintf "%-+01f" -10.0f)) "-10.000000"
    test "test6265" (lazy(sprintf "%-+0*f" 7 -10.0f)) "-10.000000"
    test "test6266" (lazy(sprintf "%-+0.5f" -10.0f)) "-10.00000"
    test "test6267" (lazy(sprintf "%-+0.*f" 4 -10.0f)) "-10.0000"
    test "test6268" (lazy(sprintf "%-+0*.*f" 5 4 -10.0f)) "-10.0000"
    test "test6269" (lazy(sprintf "% f" -10.0f)) "-10.000000"
    test "test6270" (lazy(sprintf "% 5f" -10.0f)) "-10.000000"
    test "test6271" (lazy(sprintf "% 1f" -10.0f)) "-10.000000"
    test "test6272" (lazy(sprintf "% *f" 7 -10.0f)) "-10.000000"
    test "test6273" (lazy(sprintf "% .5f" -10.0f)) "-10.00000"
    test "test6274" (lazy(sprintf "% .*f" 4 -10.0f)) "-10.0000"
    test "test6275" (lazy(sprintf "% *.*f" 5 4 -10.0f)) "-10.0000"
    test "test6276" (lazy(sprintf "%- f" -10.0f)) "-10.000000"
    test "test6277" (lazy(sprintf "%- 5f" -10.0f)) "-10.000000"
    test "test6278" (lazy(sprintf "%- 1f" -10.0f)) "-10.000000"
    test "test6279" (lazy(sprintf "%- *f" 7 -10.0f)) "-10.000000"
    test "test6280" (lazy(sprintf "%- .5f" -10.0f)) "-10.00000"
    test "test6281" (lazy(sprintf "%- .*f" 4 -10.0f)) "-10.0000"
    test "test6282" (lazy(sprintf "%- *.*f" 5 4 -10.0f)) "-10.0000"
    test "test6283" (lazy(sprintf "% 0f" -10.0f)) "-10.000000"
    test "test6284" (lazy(sprintf "% 05f" -10.0f)) "-10.000000"
    test "test6285" (lazy(sprintf "% 01f" -10.0f)) "-10.000000"
    test "test6286" (lazy(sprintf "% 0*f" 7 -10.0f)) "-10.000000"
    test "test6287" (lazy(sprintf "% 0.5f" -10.0f)) "-10.00000"
    test "test6288" (lazy(sprintf "% 0.*f" 4 -10.0f)) "-10.0000"
    test "test6289" (lazy(sprintf "% 0*.*f" 5 4 -10.0f)) "-10.0000"
    test "test6290" (lazy(sprintf "%- 0f" -10.0f)) "-10.000000"
    test "test6291" (lazy(sprintf "%- 05f" -10.0f)) "-10.000000"
    test "test6292" (lazy(sprintf "%- 01f" -10.0f)) "-10.000000"
    test "test6293" (lazy(sprintf "%- 0*f" 7 -10.0f)) "-10.000000"
    test "test6294" (lazy(sprintf "%- 0.5f" -10.0f)) "-10.00000"
    test "test6295" (lazy(sprintf "%- 0.*f" 4 -10.0f)) "-10.0000"
    test "test6296" (lazy(sprintf "%- 0*.*f" 5 4 -10.0f)) "-10.0000"
    test "test6297" (lazy(sprintf "%f" 5.0M)) "5.000000"
    test "test6298" (lazy(sprintf "%5f" 5.0M)) "5.000000"
    test "test6299" (lazy(sprintf "%1f" 5.0M)) "5.000000"
    test "test6300" (lazy(sprintf "%*f" 7 5.0M)) "5.000000"
    test "test6301" (lazy(sprintf "%.5f" 5.0M)) "5.00000"
    test "test6302" (lazy(sprintf "%.*f" 4 5.0M)) "5.0000"
    test "test6303" (lazy(sprintf "%*.*f" 5 4 5.0M)) "5.0000"
    test "test6304" (lazy(sprintf "%-f" 5.0M)) "5.000000"
    test "test6305" (lazy(sprintf "%-5f" 5.0M)) "5.000000"
    test "test6306" (lazy(sprintf "%-1f" 5.0M)) "5.000000"
    test "test6307" (lazy(sprintf "%-*f" 7 5.0M)) "5.000000"
    test "test6308" (lazy(sprintf "%-.5f" 5.0M)) "5.00000"
    test "test6309" (lazy(sprintf "%-.*f" 4 5.0M)) "5.0000"
    test "test6310" (lazy(sprintf "%-*.*f" 5 4 5.0M)) "5.0000"
    test "test6311" (lazy(sprintf "%0f" 5.0M)) "5.000000"
    test "test6312" (lazy(sprintf "%05f" 5.0M)) "5.000000"
    test "test6313" (lazy(sprintf "%01f" 5.0M)) "5.000000"
    test "test6314" (lazy(sprintf "%0*f" 7 5.0M)) "5.000000"
    test "test6315" (lazy(sprintf "%0.5f" 5.0M)) "5.00000"
    test "test6316" (lazy(sprintf "%0.*f" 4 5.0M)) "5.0000"
    test "test6317" (lazy(sprintf "%0*.*f" 5 4 5.0M)) "5.0000"
    test "test6318" (lazy(sprintf "%-0f" 5.0M)) "5.000000"
    test "test6319" (lazy(sprintf "%-05f" 5.0M)) "5.000000"
    test "test6320" (lazy(sprintf "%-01f" 5.0M)) "5.000000"
    test "test6321" (lazy(sprintf "%-0*f" 7 5.0M)) "5.000000"
    test "test6322" (lazy(sprintf "%-0.5f" 5.0M)) "5.00000"
    test "test6323" (lazy(sprintf "%-0.*f" 4 5.0M)) "5.0000"
    test "test6324" (lazy(sprintf "%-0*.*f" 5 4 5.0M)) "5.0000"
    test "test6325" (lazy(sprintf "%+f" 5.0M)) "+5.000000"
    test "test6326" (lazy(sprintf "%+5f" 5.0M)) "+5.000000"
    test "test6327" (lazy(sprintf "%+1f" 5.0M)) "+5.000000"
    test "test6328" (lazy(sprintf "%+*f" 7 5.0M)) "+5.000000"
    test "test6329" (lazy(sprintf "%+.5f" 5.0M)) "+5.00000"
    test "test6330" (lazy(sprintf "%+.*f" 4 5.0M)) "+5.0000"
    test "test6331" (lazy(sprintf "%+*.*f" 5 4 5.0M)) "+5.0000"
    test "test6332" (lazy(sprintf "%-+f" 5.0M)) "+5.000000"
    test "test6333" (lazy(sprintf "%-+5f" 5.0M)) "+5.000000"
    test "test6334" (lazy(sprintf "%-+1f" 5.0M)) "+5.000000"
    test "test6335" (lazy(sprintf "%-+*f" 7 5.0M)) "+5.000000"
    test "test6336" (lazy(sprintf "%-+.5f" 5.0M)) "+5.00000"
    test "test6337" (lazy(sprintf "%-+.*f" 4 5.0M)) "+5.0000"
    test "test6338" (lazy(sprintf "%-+*.*f" 5 4 5.0M)) "+5.0000"
    test "test6339" (lazy(sprintf "%+0f" 5.0M)) "+5.000000"
    test "test6340" (lazy(sprintf "%+05f" 5.0M)) "+5.000000"
    test "test6341" (lazy(sprintf "%+01f" 5.0M)) "+5.000000"
    test "test6342" (lazy(sprintf "%+0*f" 7 5.0M)) "+5.000000"
    test "test6343" (lazy(sprintf "%+0.5f" 5.0M)) "+5.00000"
    test "test6344" (lazy(sprintf "%+0.*f" 4 5.0M)) "+5.0000"
    test "test6345" (lazy(sprintf "%+0*.*f" 5 4 5.0M)) "+5.0000"
    test "test6346" (lazy(sprintf "%-+0f" 5.0M)) "+5.000000"
    test "test6347" (lazy(sprintf "%-+05f" 5.0M)) "+5.000000"
    test "test6348" (lazy(sprintf "%-+01f" 5.0M)) "+5.000000"
    test "test6349" (lazy(sprintf "%-+0*f" 7 5.0M)) "+5.000000"
    test "test6350" (lazy(sprintf "%-+0.5f" 5.0M)) "+5.00000"
    test "test6351" (lazy(sprintf "%-+0.*f" 4 5.0M)) "+5.0000"
    test "test6352" (lazy(sprintf "%-+0*.*f" 5 4 5.0M)) "+5.0000"
    test "test6353" (lazy(sprintf "% f" 5.0M)) " 5.000000"
    test "test6354" (lazy(sprintf "% 5f" 5.0M)) " 5.000000"
    test "test6355" (lazy(sprintf "% 1f" 5.0M)) " 5.000000"
    test "test6356" (lazy(sprintf "% *f" 7 5.0M)) " 5.000000"
    test "test6357" (lazy(sprintf "% .5f" 5.0M)) " 5.00000"
    test "test6358" (lazy(sprintf "% .*f" 4 5.0M)) " 5.0000"
    test "test6359" (lazy(sprintf "% *.*f" 5 4 5.0M)) " 5.0000"
    test "test6360" (lazy(sprintf "%- f" 5.0M)) " 5.000000"
    test "test6361" (lazy(sprintf "%- 5f" 5.0M)) " 5.000000"
    test "test6362" (lazy(sprintf "%- 1f" 5.0M)) " 5.000000"
    test "test6363" (lazy(sprintf "%- *f" 7 5.0M)) " 5.000000"
    test "test6364" (lazy(sprintf "%- .5f" 5.0M)) " 5.00000"
    test "test6365" (lazy(sprintf "%- .*f" 4 5.0M)) " 5.0000"
    test "test6366" (lazy(sprintf "%- *.*f" 5 4 5.0M)) " 5.0000"
    test "test6367" (lazy(sprintf "% 0f" 5.0M)) " 5.000000"
    test "test6368" (lazy(sprintf "% 05f" 5.0M)) " 5.000000"
    test "test6369" (lazy(sprintf "% 01f" 5.0M)) " 5.000000"
    test "test6370" (lazy(sprintf "% 0*f" 7 5.0M)) " 5.000000"
    test "test6371" (lazy(sprintf "% 0.5f" 5.0M)) " 5.00000"
    test "test6372" (lazy(sprintf "% 0.*f" 4 5.0M)) " 5.0000"
    test "test6373" (lazy(sprintf "% 0*.*f" 5 4 5.0M)) " 5.0000"
    test "test6374" (lazy(sprintf "%- 0f" 5.0M)) " 5.000000"
    test "test6375" (lazy(sprintf "%- 05f" 5.0M)) " 5.000000"
    test "test6376" (lazy(sprintf "%- 01f" 5.0M)) " 5.000000"
    test "test6377" (lazy(sprintf "%- 0*f" 7 5.0M)) " 5.000000"
    test "test6378" (lazy(sprintf "%- 0.5f" 5.0M)) " 5.00000"
    test "test6379" (lazy(sprintf "%- 0.*f" 4 5.0M)) " 5.0000"
    test "test6380" (lazy(sprintf "%- 0*.*f" 5 4 5.0M)) " 5.0000"
    test "test6381" (lazy(sprintf "%f" -10.0M)) "-10.000000"
    test "test6382" (lazy(sprintf "%5f" -10.0M)) "-10.000000"
    test "test6383" (lazy(sprintf "%1f" -10.0M)) "-10.000000"
    test "test6384" (lazy(sprintf "%*f" 7 -10.0M)) "-10.000000"
    test "test6385" (lazy(sprintf "%.5f" -10.0M)) "-10.00000"
    test "test6386" (lazy(sprintf "%.*f" 4 -10.0M)) "-10.0000"
    test "test6387" (lazy(sprintf "%*.*f" 5 4 -10.0M)) "-10.0000"
    test "test6388" (lazy(sprintf "%-f" -10.0M)) "-10.000000"
    test "test6389" (lazy(sprintf "%-5f" -10.0M)) "-10.000000"
    test "test6390" (lazy(sprintf "%-1f" -10.0M)) "-10.000000"
    test "test6391" (lazy(sprintf "%-*f" 7 -10.0M)) "-10.000000"
    test "test6392" (lazy(sprintf "%-.5f" -10.0M)) "-10.00000"
    test "test6393" (lazy(sprintf "%-.*f" 4 -10.0M)) "-10.0000"
    test "test6394" (lazy(sprintf "%-*.*f" 5 4 -10.0M)) "-10.0000"
    test "test6395" (lazy(sprintf "%0f" -10.0M)) "-10.000000"
    test "test6396" (lazy(sprintf "%05f" -10.0M)) "-10.000000"
    test "test6397" (lazy(sprintf "%01f" -10.0M)) "-10.000000"
    test "test6398" (lazy(sprintf "%0*f" 7 -10.0M)) "-10.000000"
    test "test6399" (lazy(sprintf "%0.5f" -10.0M)) "-10.00000"
    test "test6400" (lazy(sprintf "%0.*f" 4 -10.0M)) "-10.0000"
    test "test6401" (lazy(sprintf "%0*.*f" 5 4 -10.0M)) "-10.0000"
    test "test6402" (lazy(sprintf "%-0f" -10.0M)) "-10.000000"
    test "test6403" (lazy(sprintf "%-05f" -10.0M)) "-10.000000"
    test "test6404" (lazy(sprintf "%-01f" -10.0M)) "-10.000000"
    test "test6405" (lazy(sprintf "%-0*f" 7 -10.0M)) "-10.000000"
    test "test6406" (lazy(sprintf "%-0.5f" -10.0M)) "-10.00000"
    test "test6407" (lazy(sprintf "%-0.*f" 4 -10.0M)) "-10.0000"
    test "test6408" (lazy(sprintf "%-0*.*f" 5 4 -10.0M)) "-10.0000"
    test "test6409" (lazy(sprintf "%+f" -10.0M)) "-10.000000"
    test "test6410" (lazy(sprintf "%+5f" -10.0M)) "-10.000000"
    test "test6411" (lazy(sprintf "%+1f" -10.0M)) "-10.000000"
    test "test6412" (lazy(sprintf "%+*f" 7 -10.0M)) "-10.000000"
    test "test6413" (lazy(sprintf "%+.5f" -10.0M)) "-10.00000"
    test "test6414" (lazy(sprintf "%+.*f" 4 -10.0M)) "-10.0000"
    test "test6415" (lazy(sprintf "%+*.*f" 5 4 -10.0M)) "-10.0000"
    test "test6416" (lazy(sprintf "%-+f" -10.0M)) "-10.000000"
    test "test6417" (lazy(sprintf "%-+5f" -10.0M)) "-10.000000"
    test "test6418" (lazy(sprintf "%-+1f" -10.0M)) "-10.000000"
    test "test6419" (lazy(sprintf "%-+*f" 7 -10.0M)) "-10.000000"
    test "test6420" (lazy(sprintf "%-+.5f" -10.0M)) "-10.00000"
    test "test6421" (lazy(sprintf "%-+.*f" 4 -10.0M)) "-10.0000"
    test "test6422" (lazy(sprintf "%-+*.*f" 5 4 -10.0M)) "-10.0000"
    test "test6423" (lazy(sprintf "%+0f" -10.0M)) "-10.000000"
    test "test6424" (lazy(sprintf "%+05f" -10.0M)) "-10.000000"
    test "test6425" (lazy(sprintf "%+01f" -10.0M)) "-10.000000"
    test "test6426" (lazy(sprintf "%+0*f" 7 -10.0M)) "-10.000000"
    test "test6427" (lazy(sprintf "%+0.5f" -10.0M)) "-10.00000"
    test "test6428" (lazy(sprintf "%+0.*f" 4 -10.0M)) "-10.0000"
    test "test6429" (lazy(sprintf "%+0*.*f" 5 4 -10.0M)) "-10.0000"
    test "test6430" (lazy(sprintf "%-+0f" -10.0M)) "-10.000000"
    test "test6431" (lazy(sprintf "%-+05f" -10.0M)) "-10.000000"
    test "test6432" (lazy(sprintf "%-+01f" -10.0M)) "-10.000000"
    test "test6433" (lazy(sprintf "%-+0*f" 7 -10.0M)) "-10.000000"
    test "test6434" (lazy(sprintf "%-+0.5f" -10.0M)) "-10.00000"
    test "test6435" (lazy(sprintf "%-+0.*f" 4 -10.0M)) "-10.0000"
    test "test6436" (lazy(sprintf "%-+0*.*f" 5 4 -10.0M)) "-10.0000"
    test "test6437" (lazy(sprintf "% f" -10.0M)) "-10.000000"
    test "test6438" (lazy(sprintf "% 5f" -10.0M)) "-10.000000"
    test "test6439" (lazy(sprintf "% 1f" -10.0M)) "-10.000000"
    test "test6440" (lazy(sprintf "% *f" 7 -10.0M)) "-10.000000"
    test "test6441" (lazy(sprintf "% .5f" -10.0M)) "-10.00000"
    test "test6442" (lazy(sprintf "% .*f" 4 -10.0M)) "-10.0000"
    test "test6443" (lazy(sprintf "% *.*f" 5 4 -10.0M)) "-10.0000"
    test "test6444" (lazy(sprintf "%- f" -10.0M)) "-10.000000"
    test "test6445" (lazy(sprintf "%- 5f" -10.0M)) "-10.000000"
    test "test6446" (lazy(sprintf "%- 1f" -10.0M)) "-10.000000"
    test "test6447" (lazy(sprintf "%- *f" 7 -10.0M)) "-10.000000"
    test "test6448" (lazy(sprintf "%- .5f" -10.0M)) "-10.00000"
    test "test6449" (lazy(sprintf "%- .*f" 4 -10.0M)) "-10.0000"
    test "test6450" (lazy(sprintf "%- *.*f" 5 4 -10.0M)) "-10.0000"
    test "test6451" (lazy(sprintf "% 0f" -10.0M)) "-10.000000"
    test "test6452" (lazy(sprintf "% 05f" -10.0M)) "-10.000000"
    test "test6453" (lazy(sprintf "% 01f" -10.0M)) "-10.000000"
    test "test6454" (lazy(sprintf "% 0*f" 7 -10.0M)) "-10.000000"
    test "test6455" (lazy(sprintf "% 0.5f" -10.0M)) "-10.00000"
    test "test6456" (lazy(sprintf "% 0.*f" 4 -10.0M)) "-10.0000"
    test "test6457" (lazy(sprintf "% 0*.*f" 5 4 -10.0M)) "-10.0000"
    test "test6458" (lazy(sprintf "%- 0f" -10.0M)) "-10.000000"
    test "test6459" (lazy(sprintf "%- 05f" -10.0M)) "-10.000000"
    test "test6460" (lazy(sprintf "%- 01f" -10.0M)) "-10.000000"
    test "test6461" (lazy(sprintf "%- 0*f" 7 -10.0M)) "-10.000000"
    test "test6462" (lazy(sprintf "%- 0.5f" -10.0M)) "-10.00000"
    test "test6463" (lazy(sprintf "%- 0.*f" 4 -10.0M)) "-10.0000"
    test "test6464" (lazy(sprintf "%- 0*.*f" 5 4 -10.0M)) "-10.0000"
    test "test6465" (lazy(sprintf "%F" 5.0)) "5.000000"
    test "test6466" (lazy(sprintf "%5F" 5.0)) "5.000000"
    test "test6467" (lazy(sprintf "%1F" 5.0)) "5.000000"
    test "test6468" (lazy(sprintf "%*F" 7 5.0)) "5.000000"
    test "test6469" (lazy(sprintf "%.5F" 5.0)) "5.00000"
    test "test6470" (lazy(sprintf "%.*F" 4 5.0)) "5.0000"
    test "test6471" (lazy(sprintf "%*.*F" 5 4 5.0)) "5.0000"
    test "test6472" (lazy(sprintf "%-F" 5.0)) "5.000000"
    test "test6473" (lazy(sprintf "%-5F" 5.0)) "5.000000"
    test "test6474" (lazy(sprintf "%-1F" 5.0)) "5.000000"
    test "test6475" (lazy(sprintf "%-*F" 7 5.0)) "5.000000"
    test "test6476" (lazy(sprintf "%-.5F" 5.0)) "5.00000"
    test "test6477" (lazy(sprintf "%-.*F" 4 5.0)) "5.0000"
    test "test6478" (lazy(sprintf "%-*.*F" 5 4 5.0)) "5.0000"
    test "test6479" (lazy(sprintf "%0F" 5.0)) "5.000000"
    test "test6480" (lazy(sprintf "%05F" 5.0)) "5.000000"
    test "test6481" (lazy(sprintf "%01F" 5.0)) "5.000000"
    test "test6482" (lazy(sprintf "%0*F" 7 5.0)) "5.000000"
    test "test6483" (lazy(sprintf "%0.5F" 5.0)) "5.00000"
    test "test6484" (lazy(sprintf "%0.*F" 4 5.0)) "5.0000"
    test "test6485" (lazy(sprintf "%0*.*F" 5 4 5.0)) "5.0000"
    test "test6486" (lazy(sprintf "%-0F" 5.0)) "5.000000"
    test "test6487" (lazy(sprintf "%-05F" 5.0)) "5.000000"
    test "test6488" (lazy(sprintf "%-01F" 5.0)) "5.000000"
    test "test6489" (lazy(sprintf "%-0*F" 7 5.0)) "5.000000"
    test "test6490" (lazy(sprintf "%-0.5F" 5.0)) "5.00000"
    test "test6491" (lazy(sprintf "%-0.*F" 4 5.0)) "5.0000"
    test "test6492" (lazy(sprintf "%-0*.*F" 5 4 5.0)) "5.0000"
    test "test6493" (lazy(sprintf "%+F" 5.0)) "+5.000000"
    test "test6494" (lazy(sprintf "%+5F" 5.0)) "+5.000000"
    test "test6495" (lazy(sprintf "%+1F" 5.0)) "+5.000000"
    test "test6496" (lazy(sprintf "%+*F" 7 5.0)) "+5.000000"
    test "test6497" (lazy(sprintf "%+.5F" 5.0)) "+5.00000"
    test "test6498" (lazy(sprintf "%+.*F" 4 5.0)) "+5.0000"
    test "test6499" (lazy(sprintf "%+*.*F" 5 4 5.0)) "+5.0000"
    test "test6500" (lazy(sprintf "%-+F" 5.0)) "+5.000000"
    test "test6501" (lazy(sprintf "%-+5F" 5.0)) "+5.000000"
    test "test6502" (lazy(sprintf "%-+1F" 5.0)) "+5.000000"
    test "test6503" (lazy(sprintf "%-+*F" 7 5.0)) "+5.000000"
    test "test6504" (lazy(sprintf "%-+.5F" 5.0)) "+5.00000"
    test "test6505" (lazy(sprintf "%-+.*F" 4 5.0)) "+5.0000"
    test "test6506" (lazy(sprintf "%-+*.*F" 5 4 5.0)) "+5.0000"
    test "test6507" (lazy(sprintf "%+0F" 5.0)) "+5.000000"
    test "test6508" (lazy(sprintf "%+05F" 5.0)) "+5.000000"
    test "test6509" (lazy(sprintf "%+01F" 5.0)) "+5.000000"
    test "test6510" (lazy(sprintf "%+0*F" 7 5.0)) "+5.000000"
    test "test6511" (lazy(sprintf "%+0.5F" 5.0)) "+5.00000"
    test "test6512" (lazy(sprintf "%+0.*F" 4 5.0)) "+5.0000"
    test "test6513" (lazy(sprintf "%+0*.*F" 5 4 5.0)) "+5.0000"
    test "test6514" (lazy(sprintf "%-+0F" 5.0)) "+5.000000"
    test "test6515" (lazy(sprintf "%-+05F" 5.0)) "+5.000000"
    test "test6516" (lazy(sprintf "%-+01F" 5.0)) "+5.000000"
    test "test6517" (lazy(sprintf "%-+0*F" 7 5.0)) "+5.000000"
    test "test6518" (lazy(sprintf "%-+0.5F" 5.0)) "+5.00000"
    test "test6519" (lazy(sprintf "%-+0.*F" 4 5.0)) "+5.0000"
    test "test6520" (lazy(sprintf "%-+0*.*F" 5 4 5.0)) "+5.0000"
    test "test6521" (lazy(sprintf "% F" 5.0)) " 5.000000"
    test "test6522" (lazy(sprintf "% 5F" 5.0)) " 5.000000"
    test "test6523" (lazy(sprintf "% 1F" 5.0)) " 5.000000"
    test "test6524" (lazy(sprintf "% *F" 7 5.0)) " 5.000000"
    test "test6525" (lazy(sprintf "% .5F" 5.0)) " 5.00000"
    test "test6526" (lazy(sprintf "% .*F" 4 5.0)) " 5.0000"
    test "test6527" (lazy(sprintf "% *.*F" 5 4 5.0)) " 5.0000"
    test "test6528" (lazy(sprintf "%- F" 5.0)) " 5.000000"
    test "test6529" (lazy(sprintf "%- 5F" 5.0)) " 5.000000"
    test "test6530" (lazy(sprintf "%- 1F" 5.0)) " 5.000000"
    test "test6531" (lazy(sprintf "%- *F" 7 5.0)) " 5.000000"
    test "test6532" (lazy(sprintf "%- .5F" 5.0)) " 5.00000"
    test "test6533" (lazy(sprintf "%- .*F" 4 5.0)) " 5.0000"
    test "test6534" (lazy(sprintf "%- *.*F" 5 4 5.0)) " 5.0000"
    test "test6535" (lazy(sprintf "% 0F" 5.0)) " 5.000000"
    test "test6536" (lazy(sprintf "% 05F" 5.0)) " 5.000000"
    test "test6537" (lazy(sprintf "% 01F" 5.0)) " 5.000000"
    test "test6538" (lazy(sprintf "% 0*F" 7 5.0)) " 5.000000"
    test "test6539" (lazy(sprintf "% 0.5F" 5.0)) " 5.00000"
    test "test6540" (lazy(sprintf "% 0.*F" 4 5.0)) " 5.0000"
    test "test6541" (lazy(sprintf "% 0*.*F" 5 4 5.0)) " 5.0000"
    test "test6542" (lazy(sprintf "%- 0F" 5.0)) " 5.000000"
    test "test6543" (lazy(sprintf "%- 05F" 5.0)) " 5.000000"
    test "test6544" (lazy(sprintf "%- 01F" 5.0)) " 5.000000"
    test "test6545" (lazy(sprintf "%- 0*F" 7 5.0)) " 5.000000"
    test "test6546" (lazy(sprintf "%- 0.5F" 5.0)) " 5.00000"
    test "test6547" (lazy(sprintf "%- 0.*F" 4 5.0)) " 5.0000"
    test "test6548" (lazy(sprintf "%- 0*.*F" 5 4 5.0)) " 5.0000"
    test "test6549" (lazy(sprintf "%F" -10.0)) "-10.000000"
    test "test6550" (lazy(sprintf "%5F" -10.0)) "-10.000000"
    test "test6551" (lazy(sprintf "%1F" -10.0)) "-10.000000"
    test "test6552" (lazy(sprintf "%*F" 7 -10.0)) "-10.000000"
    test "test6553" (lazy(sprintf "%.5F" -10.0)) "-10.00000"
    test "test6554" (lazy(sprintf "%.*F" 4 -10.0)) "-10.0000"
    test "test6555" (lazy(sprintf "%*.*F" 5 4 -10.0)) "-10.0000"
    test "test6556" (lazy(sprintf "%-F" -10.0)) "-10.000000"
    test "test6557" (lazy(sprintf "%-5F" -10.0)) "-10.000000"
    test "test6558" (lazy(sprintf "%-1F" -10.0)) "-10.000000"
    test "test6559" (lazy(sprintf "%-*F" 7 -10.0)) "-10.000000"
    test "test6560" (lazy(sprintf "%-.5F" -10.0)) "-10.00000"
    test "test6561" (lazy(sprintf "%-.*F" 4 -10.0)) "-10.0000"
    test "test6562" (lazy(sprintf "%-*.*F" 5 4 -10.0)) "-10.0000"
    test "test6563" (lazy(sprintf "%0F" -10.0)) "-10.000000"
    test "test6564" (lazy(sprintf "%05F" -10.0)) "-10.000000"
    test "test6565" (lazy(sprintf "%01F" -10.0)) "-10.000000"
    test "test6566" (lazy(sprintf "%0*F" 7 -10.0)) "-10.000000"
    test "test6567" (lazy(sprintf "%0.5F" -10.0)) "-10.00000"
    test "test6568" (lazy(sprintf "%0.*F" 4 -10.0)) "-10.0000"
    test "test6569" (lazy(sprintf "%0*.*F" 5 4 -10.0)) "-10.0000"
    test "test6570" (lazy(sprintf "%-0F" -10.0)) "-10.000000"
    test "test6571" (lazy(sprintf "%-05F" -10.0)) "-10.000000"
    test "test6572" (lazy(sprintf "%-01F" -10.0)) "-10.000000"
    test "test6573" (lazy(sprintf "%-0*F" 7 -10.0)) "-10.000000"
    test "test6574" (lazy(sprintf "%-0.5F" -10.0)) "-10.00000"
    test "test6575" (lazy(sprintf "%-0.*F" 4 -10.0)) "-10.0000"
    test "test6576" (lazy(sprintf "%-0*.*F" 5 4 -10.0)) "-10.0000"
    test "test6577" (lazy(sprintf "%+F" -10.0)) "-10.000000"
    test "test6578" (lazy(sprintf "%+5F" -10.0)) "-10.000000"
    test "test6579" (lazy(sprintf "%+1F" -10.0)) "-10.000000"
    test "test6580" (lazy(sprintf "%+*F" 7 -10.0)) "-10.000000"
    test "test6581" (lazy(sprintf "%+.5F" -10.0)) "-10.00000"
    test "test6582" (lazy(sprintf "%+.*F" 4 -10.0)) "-10.0000"
    test "test6583" (lazy(sprintf "%+*.*F" 5 4 -10.0)) "-10.0000"
    test "test6584" (lazy(sprintf "%-+F" -10.0)) "-10.000000"
    test "test6585" (lazy(sprintf "%-+5F" -10.0)) "-10.000000"
    test "test6586" (lazy(sprintf "%-+1F" -10.0)) "-10.000000"
    test "test6587" (lazy(sprintf "%-+*F" 7 -10.0)) "-10.000000"
    test "test6588" (lazy(sprintf "%-+.5F" -10.0)) "-10.00000"
    test "test6589" (lazy(sprintf "%-+.*F" 4 -10.0)) "-10.0000"
    test "test6590" (lazy(sprintf "%-+*.*F" 5 4 -10.0)) "-10.0000"
    test "test6591" (lazy(sprintf "%+0F" -10.0)) "-10.000000"
    test "test6592" (lazy(sprintf "%+05F" -10.0)) "-10.000000"
    test "test6593" (lazy(sprintf "%+01F" -10.0)) "-10.000000"
    test "test6594" (lazy(sprintf "%+0*F" 7 -10.0)) "-10.000000"
    test "test6595" (lazy(sprintf "%+0.5F" -10.0)) "-10.00000"
    test "test6596" (lazy(sprintf "%+0.*F" 4 -10.0)) "-10.0000"
    test "test6597" (lazy(sprintf "%+0*.*F" 5 4 -10.0)) "-10.0000"
    test "test6598" (lazy(sprintf "%-+0F" -10.0)) "-10.000000"
    test "test6599" (lazy(sprintf "%-+05F" -10.0)) "-10.000000"
    test "test6600" (lazy(sprintf "%-+01F" -10.0)) "-10.000000"
    test "test6601" (lazy(sprintf "%-+0*F" 7 -10.0)) "-10.000000"
    test "test6602" (lazy(sprintf "%-+0.5F" -10.0)) "-10.00000"
    test "test6603" (lazy(sprintf "%-+0.*F" 4 -10.0)) "-10.0000"
    test "test6604" (lazy(sprintf "%-+0*.*F" 5 4 -10.0)) "-10.0000"
    test "test6605" (lazy(sprintf "% F" -10.0)) "-10.000000"
    test "test6606" (lazy(sprintf "% 5F" -10.0)) "-10.000000"
    test "test6607" (lazy(sprintf "% 1F" -10.0)) "-10.000000"
    test "test6608" (lazy(sprintf "% *F" 7 -10.0)) "-10.000000"
    test "test6609" (lazy(sprintf "% .5F" -10.0)) "-10.00000"
    test "test6610" (lazy(sprintf "% .*F" 4 -10.0)) "-10.0000"
    test "test6611" (lazy(sprintf "% *.*F" 5 4 -10.0)) "-10.0000"
    test "test6612" (lazy(sprintf "%- F" -10.0)) "-10.000000"
    test "test6613" (lazy(sprintf "%- 5F" -10.0)) "-10.000000"
    test "test6614" (lazy(sprintf "%- 1F" -10.0)) "-10.000000"
    test "test6615" (lazy(sprintf "%- *F" 7 -10.0)) "-10.000000"
    test "test6616" (lazy(sprintf "%- .5F" -10.0)) "-10.00000"
    test "test6617" (lazy(sprintf "%- .*F" 4 -10.0)) "-10.0000"
    test "test6618" (lazy(sprintf "%- *.*F" 5 4 -10.0)) "-10.0000"
    test "test6619" (lazy(sprintf "% 0F" -10.0)) "-10.000000"
    test "test6620" (lazy(sprintf "% 05F" -10.0)) "-10.000000"
    test "test6621" (lazy(sprintf "% 01F" -10.0)) "-10.000000"
    test "test6622" (lazy(sprintf "% 0*F" 7 -10.0)) "-10.000000"
    test "test6623" (lazy(sprintf "% 0.5F" -10.0)) "-10.00000"
    test "test6624" (lazy(sprintf "% 0.*F" 4 -10.0)) "-10.0000"
    test "test6625" (lazy(sprintf "% 0*.*F" 5 4 -10.0)) "-10.0000"
    test "test6626" (lazy(sprintf "%- 0F" -10.0)) "-10.000000"
    test "test6627" (lazy(sprintf "%- 05F" -10.0)) "-10.000000"
    test "test6628" (lazy(sprintf "%- 01F" -10.0)) "-10.000000"
    test "test6629" (lazy(sprintf "%- 0*F" 7 -10.0)) "-10.000000"
    test "test6630" (lazy(sprintf "%- 0.5F" -10.0)) "-10.00000"
    test "test6631" (lazy(sprintf "%- 0.*F" 4 -10.0)) "-10.0000"
    test "test6632" (lazy(sprintf "%- 0*.*F" 5 4 -10.0)) "-10.0000"
    test "test6633" (lazy(sprintf "%F" 5.0f)) "5.000000"
    test "test6634" (lazy(sprintf "%5F" 5.0f)) "5.000000"
    test "test6635" (lazy(sprintf "%1F" 5.0f)) "5.000000"
    test "test6636" (lazy(sprintf "%*F" 7 5.0f)) "5.000000"
    test "test6637" (lazy(sprintf "%.5F" 5.0f)) "5.00000"
    test "test6638" (lazy(sprintf "%.*F" 4 5.0f)) "5.0000"
    test "test6639" (lazy(sprintf "%*.*F" 5 4 5.0f)) "5.0000"
    test "test6640" (lazy(sprintf "%-F" 5.0f)) "5.000000"
    test "test6641" (lazy(sprintf "%-5F" 5.0f)) "5.000000"
    test "test6642" (lazy(sprintf "%-1F" 5.0f)) "5.000000"
    test "test6643" (lazy(sprintf "%-*F" 7 5.0f)) "5.000000"
    test "test6644" (lazy(sprintf "%-.5F" 5.0f)) "5.00000"
    test "test6645" (lazy(sprintf "%-.*F" 4 5.0f)) "5.0000"
    test "test6646" (lazy(sprintf "%-*.*F" 5 4 5.0f)) "5.0000"
    test "test6647" (lazy(sprintf "%0F" 5.0f)) "5.000000"
    test "test6648" (lazy(sprintf "%05F" 5.0f)) "5.000000"
    test "test6649" (lazy(sprintf "%01F" 5.0f)) "5.000000"
    test "test6650" (lazy(sprintf "%0*F" 7 5.0f)) "5.000000"
    test "test6651" (lazy(sprintf "%0.5F" 5.0f)) "5.00000"
    test "test6652" (lazy(sprintf "%0.*F" 4 5.0f)) "5.0000"
    test "test6653" (lazy(sprintf "%0*.*F" 5 4 5.0f)) "5.0000"
    test "test6654" (lazy(sprintf "%-0F" 5.0f)) "5.000000"
    test "test6655" (lazy(sprintf "%-05F" 5.0f)) "5.000000"
    test "test6656" (lazy(sprintf "%-01F" 5.0f)) "5.000000"
    test "test6657" (lazy(sprintf "%-0*F" 7 5.0f)) "5.000000"
    test "test6658" (lazy(sprintf "%-0.5F" 5.0f)) "5.00000"
    test "test6659" (lazy(sprintf "%-0.*F" 4 5.0f)) "5.0000"
    test "test6660" (lazy(sprintf "%-0*.*F" 5 4 5.0f)) "5.0000"
    test "test6661" (lazy(sprintf "%+F" 5.0f)) "+5.000000"
    test "test6662" (lazy(sprintf "%+5F" 5.0f)) "+5.000000"
    test "test6663" (lazy(sprintf "%+1F" 5.0f)) "+5.000000"
    test "test6664" (lazy(sprintf "%+*F" 7 5.0f)) "+5.000000"
    test "test6665" (lazy(sprintf "%+.5F" 5.0f)) "+5.00000"
    test "test6666" (lazy(sprintf "%+.*F" 4 5.0f)) "+5.0000"
    test "test6667" (lazy(sprintf "%+*.*F" 5 4 5.0f)) "+5.0000"
    test "test6668" (lazy(sprintf "%-+F" 5.0f)) "+5.000000"
    test "test6669" (lazy(sprintf "%-+5F" 5.0f)) "+5.000000"
    test "test6670" (lazy(sprintf "%-+1F" 5.0f)) "+5.000000"
    test "test6671" (lazy(sprintf "%-+*F" 7 5.0f)) "+5.000000"
    test "test6672" (lazy(sprintf "%-+.5F" 5.0f)) "+5.00000"
    test "test6673" (lazy(sprintf "%-+.*F" 4 5.0f)) "+5.0000"
    test "test6674" (lazy(sprintf "%-+*.*F" 5 4 5.0f)) "+5.0000"
    test "test6675" (lazy(sprintf "%+0F" 5.0f)) "+5.000000"
    test "test6676" (lazy(sprintf "%+05F" 5.0f)) "+5.000000"
    test "test6677" (lazy(sprintf "%+01F" 5.0f)) "+5.000000"
    test "test6678" (lazy(sprintf "%+0*F" 7 5.0f)) "+5.000000"
    test "test6679" (lazy(sprintf "%+0.5F" 5.0f)) "+5.00000"
    test "test6680" (lazy(sprintf "%+0.*F" 4 5.0f)) "+5.0000"
    test "test6681" (lazy(sprintf "%+0*.*F" 5 4 5.0f)) "+5.0000"
    test "test6682" (lazy(sprintf "%-+0F" 5.0f)) "+5.000000"
    test "test6683" (lazy(sprintf "%-+05F" 5.0f)) "+5.000000"
    test "test6684" (lazy(sprintf "%-+01F" 5.0f)) "+5.000000"
    test "test6685" (lazy(sprintf "%-+0*F" 7 5.0f)) "+5.000000"
    test "test6686" (lazy(sprintf "%-+0.5F" 5.0f)) "+5.00000"
    test "test6687" (lazy(sprintf "%-+0.*F" 4 5.0f)) "+5.0000"
    test "test6688" (lazy(sprintf "%-+0*.*F" 5 4 5.0f)) "+5.0000"
    test "test6689" (lazy(sprintf "% F" 5.0f)) " 5.000000"
    test "test6690" (lazy(sprintf "% 5F" 5.0f)) " 5.000000"
    test "test6691" (lazy(sprintf "% 1F" 5.0f)) " 5.000000"
    test "test6692" (lazy(sprintf "% *F" 7 5.0f)) " 5.000000"
    test "test6693" (lazy(sprintf "% .5F" 5.0f)) " 5.00000"
    test "test6694" (lazy(sprintf "% .*F" 4 5.0f)) " 5.0000"
    test "test6695" (lazy(sprintf "% *.*F" 5 4 5.0f)) " 5.0000"
    test "test6696" (lazy(sprintf "%- F" 5.0f)) " 5.000000"
    test "test6697" (lazy(sprintf "%- 5F" 5.0f)) " 5.000000"
    test "test6698" (lazy(sprintf "%- 1F" 5.0f)) " 5.000000"
    test "test6699" (lazy(sprintf "%- *F" 7 5.0f)) " 5.000000"
    test "test6700" (lazy(sprintf "%- .5F" 5.0f)) " 5.00000"
    test "test6701" (lazy(sprintf "%- .*F" 4 5.0f)) " 5.0000"
    test "test6702" (lazy(sprintf "%- *.*F" 5 4 5.0f)) " 5.0000"
    test "test6703" (lazy(sprintf "% 0F" 5.0f)) " 5.000000"
    test "test6704" (lazy(sprintf "% 05F" 5.0f)) " 5.000000"
    test "test6705" (lazy(sprintf "% 01F" 5.0f)) " 5.000000"
    test "test6706" (lazy(sprintf "% 0*F" 7 5.0f)) " 5.000000"
    test "test6707" (lazy(sprintf "% 0.5F" 5.0f)) " 5.00000"
    test "test6708" (lazy(sprintf "% 0.*F" 4 5.0f)) " 5.0000"
    test "test6709" (lazy(sprintf "% 0*.*F" 5 4 5.0f)) " 5.0000"
    test "test6710" (lazy(sprintf "%- 0F" 5.0f)) " 5.000000"
    test "test6711" (lazy(sprintf "%- 05F" 5.0f)) " 5.000000"
    test "test6712" (lazy(sprintf "%- 01F" 5.0f)) " 5.000000"
    test "test6713" (lazy(sprintf "%- 0*F" 7 5.0f)) " 5.000000"
    test "test6714" (lazy(sprintf "%- 0.5F" 5.0f)) " 5.00000"
    test "test6715" (lazy(sprintf "%- 0.*F" 4 5.0f)) " 5.0000"
    test "test6716" (lazy(sprintf "%- 0*.*F" 5 4 5.0f)) " 5.0000"
    test "test6717" (lazy(sprintf "%F" -10.0f)) "-10.000000"
    test "test6718" (lazy(sprintf "%5F" -10.0f)) "-10.000000"
    test "test6719" (lazy(sprintf "%1F" -10.0f)) "-10.000000"
    test "test6720" (lazy(sprintf "%*F" 7 -10.0f)) "-10.000000"
    test "test6721" (lazy(sprintf "%.5F" -10.0f)) "-10.00000"
    test "test6722" (lazy(sprintf "%.*F" 4 -10.0f)) "-10.0000"
    test "test6723" (lazy(sprintf "%*.*F" 5 4 -10.0f)) "-10.0000"
    test "test6724" (lazy(sprintf "%-F" -10.0f)) "-10.000000"
    test "test6725" (lazy(sprintf "%-5F" -10.0f)) "-10.000000"
    test "test6726" (lazy(sprintf "%-1F" -10.0f)) "-10.000000"
    test "test6727" (lazy(sprintf "%-*F" 7 -10.0f)) "-10.000000"
    test "test6728" (lazy(sprintf "%-.5F" -10.0f)) "-10.00000"
    test "test6729" (lazy(sprintf "%-.*F" 4 -10.0f)) "-10.0000"
    test "test6730" (lazy(sprintf "%-*.*F" 5 4 -10.0f)) "-10.0000"
    test "test6731" (lazy(sprintf "%0F" -10.0f)) "-10.000000"
    test "test6732" (lazy(sprintf "%05F" -10.0f)) "-10.000000"
    test "test6733" (lazy(sprintf "%01F" -10.0f)) "-10.000000"
    test "test6734" (lazy(sprintf "%0*F" 7 -10.0f)) "-10.000000"
    test "test6735" (lazy(sprintf "%0.5F" -10.0f)) "-10.00000"
    test "test6736" (lazy(sprintf "%0.*F" 4 -10.0f)) "-10.0000"
    test "test6737" (lazy(sprintf "%0*.*F" 5 4 -10.0f)) "-10.0000"
    test "test6738" (lazy(sprintf "%-0F" -10.0f)) "-10.000000"
    test "test6739" (lazy(sprintf "%-05F" -10.0f)) "-10.000000"
    test "test6740" (lazy(sprintf "%-01F" -10.0f)) "-10.000000"
    test "test6741" (lazy(sprintf "%-0*F" 7 -10.0f)) "-10.000000"
    test "test6742" (lazy(sprintf "%-0.5F" -10.0f)) "-10.00000"
    test "test6743" (lazy(sprintf "%-0.*F" 4 -10.0f)) "-10.0000"
    test "test6744" (lazy(sprintf "%-0*.*F" 5 4 -10.0f)) "-10.0000"
    test "test6745" (lazy(sprintf "%+F" -10.0f)) "-10.000000"
    test "test6746" (lazy(sprintf "%+5F" -10.0f)) "-10.000000"
    test "test6747" (lazy(sprintf "%+1F" -10.0f)) "-10.000000"
    test "test6748" (lazy(sprintf "%+*F" 7 -10.0f)) "-10.000000"
    test "test6749" (lazy(sprintf "%+.5F" -10.0f)) "-10.00000"
    test "test6750" (lazy(sprintf "%+.*F" 4 -10.0f)) "-10.0000"
    test "test6751" (lazy(sprintf "%+*.*F" 5 4 -10.0f)) "-10.0000"
    test "test6752" (lazy(sprintf "%-+F" -10.0f)) "-10.000000"
    test "test6753" (lazy(sprintf "%-+5F" -10.0f)) "-10.000000"
    test "test6754" (lazy(sprintf "%-+1F" -10.0f)) "-10.000000"
    test "test6755" (lazy(sprintf "%-+*F" 7 -10.0f)) "-10.000000"
    test "test6756" (lazy(sprintf "%-+.5F" -10.0f)) "-10.00000"
    test "test6757" (lazy(sprintf "%-+.*F" 4 -10.0f)) "-10.0000"
    test "test6758" (lazy(sprintf "%-+*.*F" 5 4 -10.0f)) "-10.0000"
    test "test6759" (lazy(sprintf "%+0F" -10.0f)) "-10.000000"
    test "test6760" (lazy(sprintf "%+05F" -10.0f)) "-10.000000"
    test "test6761" (lazy(sprintf "%+01F" -10.0f)) "-10.000000"
    test "test6762" (lazy(sprintf "%+0*F" 7 -10.0f)) "-10.000000"
    test "test6763" (lazy(sprintf "%+0.5F" -10.0f)) "-10.00000"
    test "test6764" (lazy(sprintf "%+0.*F" 4 -10.0f)) "-10.0000"
    test "test6765" (lazy(sprintf "%+0*.*F" 5 4 -10.0f)) "-10.0000"
    test "test6766" (lazy(sprintf "%-+0F" -10.0f)) "-10.000000"
    test "test6767" (lazy(sprintf "%-+05F" -10.0f)) "-10.000000"
    test "test6768" (lazy(sprintf "%-+01F" -10.0f)) "-10.000000"
    test "test6769" (lazy(sprintf "%-+0*F" 7 -10.0f)) "-10.000000"
    test "test6770" (lazy(sprintf "%-+0.5F" -10.0f)) "-10.00000"
    test "test6771" (lazy(sprintf "%-+0.*F" 4 -10.0f)) "-10.0000"
    test "test6772" (lazy(sprintf "%-+0*.*F" 5 4 -10.0f)) "-10.0000"
    test "test6773" (lazy(sprintf "% F" -10.0f)) "-10.000000"
    test "test6774" (lazy(sprintf "% 5F" -10.0f)) "-10.000000"
    test "test6775" (lazy(sprintf "% 1F" -10.0f)) "-10.000000"
    test "test6776" (lazy(sprintf "% *F" 7 -10.0f)) "-10.000000"
    test "test6777" (lazy(sprintf "% .5F" -10.0f)) "-10.00000"
    test "test6778" (lazy(sprintf "% .*F" 4 -10.0f)) "-10.0000"
    test "test6779" (lazy(sprintf "% *.*F" 5 4 -10.0f)) "-10.0000"
    test "test6780" (lazy(sprintf "%- F" -10.0f)) "-10.000000"
    test "test6781" (lazy(sprintf "%- 5F" -10.0f)) "-10.000000"
    test "test6782" (lazy(sprintf "%- 1F" -10.0f)) "-10.000000"
    test "test6783" (lazy(sprintf "%- *F" 7 -10.0f)) "-10.000000"
    test "test6784" (lazy(sprintf "%- .5F" -10.0f)) "-10.00000"
    test "test6785" (lazy(sprintf "%- .*F" 4 -10.0f)) "-10.0000"
    test "test6786" (lazy(sprintf "%- *.*F" 5 4 -10.0f)) "-10.0000"
    test "test6787" (lazy(sprintf "% 0F" -10.0f)) "-10.000000"
    test "test6788" (lazy(sprintf "% 05F" -10.0f)) "-10.000000"
    test "test6789" (lazy(sprintf "% 01F" -10.0f)) "-10.000000"
    test "test6790" (lazy(sprintf "% 0*F" 7 -10.0f)) "-10.000000"
    test "test6791" (lazy(sprintf "% 0.5F" -10.0f)) "-10.00000"
    test "test6792" (lazy(sprintf "% 0.*F" 4 -10.0f)) "-10.0000"
    test "test6793" (lazy(sprintf "% 0*.*F" 5 4 -10.0f)) "-10.0000"
    test "test6794" (lazy(sprintf "%- 0F" -10.0f)) "-10.000000"
    test "test6795" (lazy(sprintf "%- 05F" -10.0f)) "-10.000000"
    test "test6796" (lazy(sprintf "%- 01F" -10.0f)) "-10.000000"
    test "test6797" (lazy(sprintf "%- 0*F" 7 -10.0f)) "-10.000000"
    test "test6798" (lazy(sprintf "%- 0.5F" -10.0f)) "-10.00000"
    test "test6799" (lazy(sprintf "%- 0.*F" 4 -10.0f)) "-10.0000"
    test "test6800" (lazy(sprintf "%- 0*.*F" 5 4 -10.0f)) "-10.0000"
    test "test6801" (lazy(sprintf "%F" 5.0M)) "5.000000"
    test "test6802" (lazy(sprintf "%5F" 5.0M)) "5.000000"
    test "test6803" (lazy(sprintf "%1F" 5.0M)) "5.000000"
    test "test6804" (lazy(sprintf "%*F" 7 5.0M)) "5.000000"
    test "test6805" (lazy(sprintf "%.5F" 5.0M)) "5.00000"
    test "test6806" (lazy(sprintf "%.*F" 4 5.0M)) "5.0000"
    test "test6807" (lazy(sprintf "%*.*F" 5 4 5.0M)) "5.0000"
    test "test6808" (lazy(sprintf "%-F" 5.0M)) "5.000000"
    test "test6809" (lazy(sprintf "%-5F" 5.0M)) "5.000000"
    test "test6810" (lazy(sprintf "%-1F" 5.0M)) "5.000000"
    test "test6811" (lazy(sprintf "%-*F" 7 5.0M)) "5.000000"
    test "test6812" (lazy(sprintf "%-.5F" 5.0M)) "5.00000"
    test "test6813" (lazy(sprintf "%-.*F" 4 5.0M)) "5.0000"
    test "test6814" (lazy(sprintf "%-*.*F" 5 4 5.0M)) "5.0000"
    test "test6815" (lazy(sprintf "%0F" 5.0M)) "5.000000"
    test "test6816" (lazy(sprintf "%05F" 5.0M)) "5.000000"
    test "test6817" (lazy(sprintf "%01F" 5.0M)) "5.000000"
    test "test6818" (lazy(sprintf "%0*F" 7 5.0M)) "5.000000"
    test "test6819" (lazy(sprintf "%0.5F" 5.0M)) "5.00000"
    test "test6820" (lazy(sprintf "%0.*F" 4 5.0M)) "5.0000"
    test "test6821" (lazy(sprintf "%0*.*F" 5 4 5.0M)) "5.0000"
    test "test6822" (lazy(sprintf "%-0F" 5.0M)) "5.000000"
    test "test6823" (lazy(sprintf "%-05F" 5.0M)) "5.000000"
    test "test6824" (lazy(sprintf "%-01F" 5.0M)) "5.000000"
    test "test6825" (lazy(sprintf "%-0*F" 7 5.0M)) "5.000000"
    test "test6826" (lazy(sprintf "%-0.5F" 5.0M)) "5.00000"
    test "test6827" (lazy(sprintf "%-0.*F" 4 5.0M)) "5.0000"
    test "test6828" (lazy(sprintf "%-0*.*F" 5 4 5.0M)) "5.0000"
    test "test6829" (lazy(sprintf "%+F" 5.0M)) "+5.000000"
    test "test6830" (lazy(sprintf "%+5F" 5.0M)) "+5.000000"
    test "test6831" (lazy(sprintf "%+1F" 5.0M)) "+5.000000"
    test "test6832" (lazy(sprintf "%+*F" 7 5.0M)) "+5.000000"
    test "test6833" (lazy(sprintf "%+.5F" 5.0M)) "+5.00000"
    test "test6834" (lazy(sprintf "%+.*F" 4 5.0M)) "+5.0000"
    test "test6835" (lazy(sprintf "%+*.*F" 5 4 5.0M)) "+5.0000"
    test "test6836" (lazy(sprintf "%-+F" 5.0M)) "+5.000000"
    test "test6837" (lazy(sprintf "%-+5F" 5.0M)) "+5.000000"
    test "test6838" (lazy(sprintf "%-+1F" 5.0M)) "+5.000000"
    test "test6839" (lazy(sprintf "%-+*F" 7 5.0M)) "+5.000000"
    test "test6840" (lazy(sprintf "%-+.5F" 5.0M)) "+5.00000"
    test "test6841" (lazy(sprintf "%-+.*F" 4 5.0M)) "+5.0000"
    test "test6842" (lazy(sprintf "%-+*.*F" 5 4 5.0M)) "+5.0000"
    test "test6843" (lazy(sprintf "%+0F" 5.0M)) "+5.000000"
    test "test6844" (lazy(sprintf "%+05F" 5.0M)) "+5.000000"
    test "test6845" (lazy(sprintf "%+01F" 5.0M)) "+5.000000"
    test "test6846" (lazy(sprintf "%+0*F" 7 5.0M)) "+5.000000"
    test "test6847" (lazy(sprintf "%+0.5F" 5.0M)) "+5.00000"
    test "test6848" (lazy(sprintf "%+0.*F" 4 5.0M)) "+5.0000"
    test "test6849" (lazy(sprintf "%+0*.*F" 5 4 5.0M)) "+5.0000"
    test "test6850" (lazy(sprintf "%-+0F" 5.0M)) "+5.000000"
    test "test6851" (lazy(sprintf "%-+05F" 5.0M)) "+5.000000"
    test "test6852" (lazy(sprintf "%-+01F" 5.0M)) "+5.000000"
    test "test6853" (lazy(sprintf "%-+0*F" 7 5.0M)) "+5.000000"
    test "test6854" (lazy(sprintf "%-+0.5F" 5.0M)) "+5.00000"
    test "test6855" (lazy(sprintf "%-+0.*F" 4 5.0M)) "+5.0000"
    test "test6856" (lazy(sprintf "%-+0*.*F" 5 4 5.0M)) "+5.0000"
    test "test6857" (lazy(sprintf "% F" 5.0M)) " 5.000000"
    test "test6858" (lazy(sprintf "% 5F" 5.0M)) " 5.000000"
    test "test6859" (lazy(sprintf "% 1F" 5.0M)) " 5.000000"
    test "test6860" (lazy(sprintf "% *F" 7 5.0M)) " 5.000000"
    test "test6861" (lazy(sprintf "% .5F" 5.0M)) " 5.00000"
    test "test6862" (lazy(sprintf "% .*F" 4 5.0M)) " 5.0000"
    test "test6863" (lazy(sprintf "% *.*F" 5 4 5.0M)) " 5.0000"
    test "test6864" (lazy(sprintf "%- F" 5.0M)) " 5.000000"
    test "test6865" (lazy(sprintf "%- 5F" 5.0M)) " 5.000000"
    test "test6866" (lazy(sprintf "%- 1F" 5.0M)) " 5.000000"
    test "test6867" (lazy(sprintf "%- *F" 7 5.0M)) " 5.000000"
    test "test6868" (lazy(sprintf "%- .5F" 5.0M)) " 5.00000"
    test "test6869" (lazy(sprintf "%- .*F" 4 5.0M)) " 5.0000"
    test "test6870" (lazy(sprintf "%- *.*F" 5 4 5.0M)) " 5.0000"
    test "test6871" (lazy(sprintf "% 0F" 5.0M)) " 5.000000"
    test "test6872" (lazy(sprintf "% 05F" 5.0M)) " 5.000000"
    test "test6873" (lazy(sprintf "% 01F" 5.0M)) " 5.000000"
    test "test6874" (lazy(sprintf "% 0*F" 7 5.0M)) " 5.000000"
    test "test6875" (lazy(sprintf "% 0.5F" 5.0M)) " 5.00000"
    test "test6876" (lazy(sprintf "% 0.*F" 4 5.0M)) " 5.0000"
    test "test6877" (lazy(sprintf "% 0*.*F" 5 4 5.0M)) " 5.0000"
    test "test6878" (lazy(sprintf "%- 0F" 5.0M)) " 5.000000"
    test "test6879" (lazy(sprintf "%- 05F" 5.0M)) " 5.000000"
    test "test6880" (lazy(sprintf "%- 01F" 5.0M)) " 5.000000"
    test "test6881" (lazy(sprintf "%- 0*F" 7 5.0M)) " 5.000000"
    test "test6882" (lazy(sprintf "%- 0.5F" 5.0M)) " 5.00000"
    test "test6883" (lazy(sprintf "%- 0.*F" 4 5.0M)) " 5.0000"
    test "test6884" (lazy(sprintf "%- 0*.*F" 5 4 5.0M)) " 5.0000"
    test "test6885" (lazy(sprintf "%F" -10.0M)) "-10.000000"
    test "test6886" (lazy(sprintf "%5F" -10.0M)) "-10.000000"
    test "test6887" (lazy(sprintf "%1F" -10.0M)) "-10.000000"
    test "test6888" (lazy(sprintf "%*F" 7 -10.0M)) "-10.000000"
    test "test6889" (lazy(sprintf "%.5F" -10.0M)) "-10.00000"
    test "test6890" (lazy(sprintf "%.*F" 4 -10.0M)) "-10.0000"
    test "test6891" (lazy(sprintf "%*.*F" 5 4 -10.0M)) "-10.0000"
    test "test6892" (lazy(sprintf "%-F" -10.0M)) "-10.000000"
    test "test6893" (lazy(sprintf "%-5F" -10.0M)) "-10.000000"
    test "test6894" (lazy(sprintf "%-1F" -10.0M)) "-10.000000"
    test "test6895" (lazy(sprintf "%-*F" 7 -10.0M)) "-10.000000"
    test "test6896" (lazy(sprintf "%-.5F" -10.0M)) "-10.00000"
    test "test6897" (lazy(sprintf "%-.*F" 4 -10.0M)) "-10.0000"
    test "test6898" (lazy(sprintf "%-*.*F" 5 4 -10.0M)) "-10.0000"
    test "test6899" (lazy(sprintf "%0F" -10.0M)) "-10.000000"
    test "test6900" (lazy(sprintf "%05F" -10.0M)) "-10.000000"
    test "test6901" (lazy(sprintf "%01F" -10.0M)) "-10.000000"
    test "test6902" (lazy(sprintf "%0*F" 7 -10.0M)) "-10.000000"
    test "test6903" (lazy(sprintf "%0.5F" -10.0M)) "-10.00000"
    test "test6904" (lazy(sprintf "%0.*F" 4 -10.0M)) "-10.0000"
    test "test6905" (lazy(sprintf "%0*.*F" 5 4 -10.0M)) "-10.0000"
    test "test6906" (lazy(sprintf "%-0F" -10.0M)) "-10.000000"
    test "test6907" (lazy(sprintf "%-05F" -10.0M)) "-10.000000"
    test "test6908" (lazy(sprintf "%-01F" -10.0M)) "-10.000000"
    test "test6909" (lazy(sprintf "%-0*F" 7 -10.0M)) "-10.000000"
    test "test6910" (lazy(sprintf "%-0.5F" -10.0M)) "-10.00000"
    test "test6911" (lazy(sprintf "%-0.*F" 4 -10.0M)) "-10.0000"
    test "test6912" (lazy(sprintf "%-0*.*F" 5 4 -10.0M)) "-10.0000"
    test "test6913" (lazy(sprintf "%+F" -10.0M)) "-10.000000"
    test "test6914" (lazy(sprintf "%+5F" -10.0M)) "-10.000000"
    test "test6915" (lazy(sprintf "%+1F" -10.0M)) "-10.000000"
    test "test6916" (lazy(sprintf "%+*F" 7 -10.0M)) "-10.000000"
    test "test6917" (lazy(sprintf "%+.5F" -10.0M)) "-10.00000"
    test "test6918" (lazy(sprintf "%+.*F" 4 -10.0M)) "-10.0000"
    test "test6919" (lazy(sprintf "%+*.*F" 5 4 -10.0M)) "-10.0000"
    test "test6920" (lazy(sprintf "%-+F" -10.0M)) "-10.000000"
    test "test6921" (lazy(sprintf "%-+5F" -10.0M)) "-10.000000"
    test "test6922" (lazy(sprintf "%-+1F" -10.0M)) "-10.000000"
    test "test6923" (lazy(sprintf "%-+*F" 7 -10.0M)) "-10.000000"
    test "test6924" (lazy(sprintf "%-+.5F" -10.0M)) "-10.00000"
    test "test6925" (lazy(sprintf "%-+.*F" 4 -10.0M)) "-10.0000"
    test "test6926" (lazy(sprintf "%-+*.*F" 5 4 -10.0M)) "-10.0000"
    test "test6927" (lazy(sprintf "%+0F" -10.0M)) "-10.000000"
    test "test6928" (lazy(sprintf "%+05F" -10.0M)) "-10.000000"
    test "test6929" (lazy(sprintf "%+01F" -10.0M)) "-10.000000"
    test "test6930" (lazy(sprintf "%+0*F" 7 -10.0M)) "-10.000000"
    test "test6931" (lazy(sprintf "%+0.5F" -10.0M)) "-10.00000"
    test "test6932" (lazy(sprintf "%+0.*F" 4 -10.0M)) "-10.0000"
    test "test6933" (lazy(sprintf "%+0*.*F" 5 4 -10.0M)) "-10.0000"
    test "test6934" (lazy(sprintf "%-+0F" -10.0M)) "-10.000000"
    test "test6935" (lazy(sprintf "%-+05F" -10.0M)) "-10.000000"
    test "test6936" (lazy(sprintf "%-+01F" -10.0M)) "-10.000000"
    test "test6937" (lazy(sprintf "%-+0*F" 7 -10.0M)) "-10.000000"
    test "test6938" (lazy(sprintf "%-+0.5F" -10.0M)) "-10.00000"
    test "test6939" (lazy(sprintf "%-+0.*F" 4 -10.0M)) "-10.0000"
    test "test6940" (lazy(sprintf "%-+0*.*F" 5 4 -10.0M)) "-10.0000"
    test "test6941" (lazy(sprintf "% F" -10.0M)) "-10.000000"
    test "test6942" (lazy(sprintf "% 5F" -10.0M)) "-10.000000"
    test "test6943" (lazy(sprintf "% 1F" -10.0M)) "-10.000000"
    test "test6944" (lazy(sprintf "% *F" 7 -10.0M)) "-10.000000"
    test "test6945" (lazy(sprintf "% .5F" -10.0M)) "-10.00000"
    test "test6946" (lazy(sprintf "% .*F" 4 -10.0M)) "-10.0000"
    test "test6947" (lazy(sprintf "% *.*F" 5 4 -10.0M)) "-10.0000"
    test "test6948" (lazy(sprintf "%- F" -10.0M)) "-10.000000"
    test "test6949" (lazy(sprintf "%- 5F" -10.0M)) "-10.000000"
    test "test6950" (lazy(sprintf "%- 1F" -10.0M)) "-10.000000"
    test "test6951" (lazy(sprintf "%- *F" 7 -10.0M)) "-10.000000"
    test "test6952" (lazy(sprintf "%- .5F" -10.0M)) "-10.00000"
    test "test6953" (lazy(sprintf "%- .*F" 4 -10.0M)) "-10.0000"
    test "test6954" (lazy(sprintf "%- *.*F" 5 4 -10.0M)) "-10.0000"
    test "test6955" (lazy(sprintf "% 0F" -10.0M)) "-10.000000"
    test "test6956" (lazy(sprintf "% 05F" -10.0M)) "-10.000000"
    test "test6957" (lazy(sprintf "% 01F" -10.0M)) "-10.000000"
    test "test6958" (lazy(sprintf "% 0*F" 7 -10.0M)) "-10.000000"
    test "test6959" (lazy(sprintf "% 0.5F" -10.0M)) "-10.00000"
    test "test6960" (lazy(sprintf "% 0.*F" 4 -10.0M)) "-10.0000"
    test "test6961" (lazy(sprintf "% 0*.*F" 5 4 -10.0M)) "-10.0000"
    test "test6962" (lazy(sprintf "%- 0F" -10.0M)) "-10.000000"
    test "test6963" (lazy(sprintf "%- 05F" -10.0M)) "-10.000000"
    test "test6964" (lazy(sprintf "%- 01F" -10.0M)) "-10.000000"
    test "test6965" (lazy(sprintf "%- 0*F" 7 -10.0M)) "-10.000000"
    test "test6966" (lazy(sprintf "%- 0.5F" -10.0M)) "-10.00000"
    test "test6967" (lazy(sprintf "%- 0.*F" 4 -10.0M)) "-10.0000"
    test "test6968" (lazy(sprintf "%- 0*.*F" 5 4 -10.0M)) "-10.0000"
    test "test6969" (lazy(sprintf "%g" 5.0)) "5"
    test "test6970" (lazy(sprintf "%5g" 5.0)) "    5"
    test "test6971" (lazy(sprintf "%1g" 5.0)) "5"
    test "test6972" (lazy(sprintf "%*g" 7 5.0)) "      5"
    test "test6973" (lazy(sprintf "%.5g" 5.0)) "5"
    test "test6974" (lazy(sprintf "%.*g" 4 5.0)) "5"
    test "test6975" (lazy(sprintf "%*.*g" 5 4 5.0)) "    5"
    test "test6976" (lazy(sprintf "%-g" 5.0)) "5"
    test "test6977" (lazy(sprintf "%-5g" 5.0)) "5    "
    test "test6978" (lazy(sprintf "%-1g" 5.0)) "5"
    test "test6979" (lazy(sprintf "%-*g" 7 5.0)) "5      "
    test "test6980" (lazy(sprintf "%-.5g" 5.0)) "5"
    test "test6981" (lazy(sprintf "%-.*g" 4 5.0)) "5"
    test "test6982" (lazy(sprintf "%-*.*g" 5 4 5.0)) "5    "
    test "test6983" (lazy(sprintf "%0g" 5.0)) "5"
    test "test6984" (lazy(sprintf "%05g" 5.0)) "00005"
    test "test6985" (lazy(sprintf "%01g" 5.0)) "5"
    test "test6986" (lazy(sprintf "%0*g" 7 5.0)) "0000005"
    test "test6987" (lazy(sprintf "%0.5g" 5.0)) "5"
    test "test6988" (lazy(sprintf "%0.*g" 4 5.0)) "5"
    test "test6989" (lazy(sprintf "%0*.*g" 5 4 5.0)) "00005"
    test "test6990" (lazy(sprintf "%-0g" 5.0)) "5"
    test "test6991" (lazy(sprintf "%-05g" 5.0)) "5    "// "50000"
    test "test6992" (lazy(sprintf "%-01g" 5.0)) "5"
    test "test6993" (lazy(sprintf "%-0*g" 7 5.0)) "5      "//"5000000"
    test "test6994" (lazy(sprintf "%-0.5g" 5.0)) "5"
    test "test6995" (lazy(sprintf "%-0.*g" 4 5.0)) "5"
    test "test6996" (lazy(sprintf "%-0*.*g" 5 4 5.0)) "5    "// "50000"
    test "test6997" (lazy(sprintf "%+g" 5.0)) "+5"
    test "test6998" (lazy(sprintf "%+5g" 5.0)) "   +5"
    test "test6999" (lazy(sprintf "%+1g" 5.0)) "+5"
    test "test7000" (lazy(sprintf "%+*g" 7 5.0)) "     +5"
let func7000()=
    test "test7001" (lazy(sprintf "%+.5g" 5.0)) "+5"
    test "test7002" (lazy(sprintf "%+.*g" 4 5.0)) "+5"
    test "test7003" (lazy(sprintf "%+*.*g" 5 4 5.0)) "   +5"
    test "test7004" (lazy(sprintf "%-+g" 5.0)) "+5"
    test "test7005" (lazy(sprintf "%-+5g" 5.0)) "+5   "
    test "test7006" (lazy(sprintf "%-+1g" 5.0)) "+5"
    test "test7007" (lazy(sprintf "%-+*g" 7 5.0)) "+5     "
    test "test7008" (lazy(sprintf "%-+.5g" 5.0)) "+5"
    test "test7009" (lazy(sprintf "%-+.*g" 4 5.0)) "+5"
    test "test7010" (lazy(sprintf "%-+*.*g" 5 4 5.0)) "+5   "
    test "test7011" (lazy(sprintf "%+0g" 5.0)) "+5"
    test "test7012" (lazy(sprintf "%+05g" 5.0)) "+0005"//"000+5"
    test "test7013" (lazy(sprintf "%+01g" 5.0)) "+5"
    test "test7014" (lazy(sprintf "%+0*g" 7 5.0)) "+000005"// "00000+5"
    test "test7015" (lazy(sprintf "%+0.5g" 5.0)) "+5"
    test "test7016" (lazy(sprintf "%+0.*g" 4 5.0)) "+5"
    test "test7017" (lazy(sprintf "%+0*.*g" 5 4 5.0)) "+0005"// "000+5"
    test "test7018" (lazy(sprintf "%-+0g" 5.0)) "+5"
    test "test7019" (lazy(sprintf "%-+05g" 5.0)) "+5   "//"+5000"
    test "test7020" (lazy(sprintf "%-+01g" 5.0)) "+5"
    test "test7021" (lazy(sprintf "%-+0*g" 7 5.0)) "+5     "// "+500000"
    test "test7022" (lazy(sprintf "%-+0.5g" 5.0)) "+5"
    test "test7023" (lazy(sprintf "%-+0.*g" 4 5.0)) "+5"
    test "test7024" (lazy(sprintf "%-+0*.*g" 5 4 5.0)) "+5   "//"+5000"
    test "test7025" (lazy(sprintf "% g" 5.0)) " 5"
    test "test7026" (lazy(sprintf "% 5g" 5.0)) "    5"
    test "test7027" (lazy(sprintf "% 1g" 5.0)) " 5"
    test "test7028" (lazy(sprintf "% *g" 7 5.0)) "      5"
    test "test7029" (lazy(sprintf "% .5g" 5.0)) " 5"
    test "test7030" (lazy(sprintf "% .*g" 4 5.0)) " 5"
    test "test7031" (lazy(sprintf "% *.*g" 5 4 5.0)) "    5"
    test "test7032" (lazy(sprintf "%- g" 5.0)) " 5"
    test "test7033" (lazy(sprintf "%- 5g" 5.0)) " 5   "
    test "test7034" (lazy(sprintf "%- 1g" 5.0)) " 5"
    test "test7035" (lazy(sprintf "%- *g" 7 5.0)) " 5     "
    test "test7036" (lazy(sprintf "%- .5g" 5.0)) " 5"
    test "test7037" (lazy(sprintf "%- .*g" 4 5.0)) " 5"
    test "test7038" (lazy(sprintf "%- *.*g" 5 4 5.0)) " 5   "
    test "test7039" (lazy(sprintf "% 0g" 5.0)) " 5"
    test "test7040" (lazy(sprintf "% 05g" 5.0)) " 0005"//"000 5"
    test "test7041" (lazy(sprintf "% 01g" 5.0)) " 5"
    test "test7042" (lazy(sprintf "% 0*g" 7 5.0)) " 000005"//"00000 5"
    test "test7043" (lazy(sprintf "% 0.5g" 5.0)) " 5"
    test "test7044" (lazy(sprintf "% 0.*g" 4 5.0)) " 5"
    test "test7045" (lazy(sprintf "% 0*.*g" 5 4 5.0)) " 0005"// "000 5"
    test "test7046" (lazy(sprintf "%- 0g" 5.0)) " 5"
    test "test7047" (lazy(sprintf "%- 05g" 5.0)) " 5   "//" 5000"
    test "test7048" (lazy(sprintf "%- 01g" 5.0)) " 5"
    test "test7049" (lazy(sprintf "%- 0*g" 7 5.0)) " 5     "// " 500000"
    test "test7050" (lazy(sprintf "%- 0.5g" 5.0)) " 5"
    test "test7051" (lazy(sprintf "%- 0.*g" 4 5.0)) " 5"
    test "test7052" (lazy(sprintf "%- 0*.*g" 5 4 5.0)) " 5   "// " 5000"
    test "test7053" (lazy(sprintf "%g" -10.0)) "-10"
    test "test7054" (lazy(sprintf "%5g" -10.0)) "  -10"
    test "test7055" (lazy(sprintf "%1g" -10.0)) "-10"
    test "test7056" (lazy(sprintf "%*g" 7 -10.0)) "    -10"
    test "test7057" (lazy(sprintf "%.5g" -10.0)) "-10"
    test "test7058" (lazy(sprintf "%.*g" 4 -10.0)) "-10"
    test "test7059" (lazy(sprintf "%*.*g" 5 4 -10.0)) "  -10"
    test "test7060" (lazy(sprintf "%-g" -10.0)) "-10"
    test "test7061" (lazy(sprintf "%-5g" -10.0)) "-10  "
    test "test7062" (lazy(sprintf "%-1g" -10.0)) "-10"
    test "test7063" (lazy(sprintf "%-*g" 7 -10.0)) "-10    "
    test "test7064" (lazy(sprintf "%-.5g" -10.0)) "-10"
    test "test7065" (lazy(sprintf "%-.*g" 4 -10.0)) "-10"
    test "test7066" (lazy(sprintf "%-*.*g" 5 4 -10.0)) "-10  "
    test "test7067" (lazy(sprintf "%0g" -10.0)) "-10"
    test "test7068" (lazy(sprintf "%05g" -10.0)) "-0010"// "00-10"
    test "test7069" (lazy(sprintf "%01g" -10.0)) "-10"
    test "test7070" (lazy(sprintf "%0*g" 7 -10.0)) "-000010"// "0000-10"
    test "test7071" (lazy(sprintf "%0.5g" -10.0)) "-10"
    test "test7072" (lazy(sprintf "%0.*g" 4 -10.0)) "-10"
    test "test7073" (lazy(sprintf "%0*.*g" 5 4 -10.0)) "-0010"//  "00-10"
    test "test7074" (lazy(sprintf "%-0g" -10.0)) "-10"
    test "test7075" (lazy(sprintf "%-05g" -10.0)) "-10  "// "-1000"
    test "test7076" (lazy(sprintf "%-01g" -10.0)) "-10"
    test "test7077" (lazy(sprintf "%-0*g" 7 -10.0)) "-10    "//  "-100000"
    test "test7078" (lazy(sprintf "%-0.5g" -10.0)) "-10"
    test "test7079" (lazy(sprintf "%-0.*g" 4 -10.0)) "-10"
    test "test7080" (lazy(sprintf "%-0*.*g" 5 4 -10.0)) "-10  "//   "-1000"
    test "test7081" (lazy(sprintf "%+g" -10.0)) "-10"
    test "test7082" (lazy(sprintf "%+5g" -10.0)) "  -10"
    test "test7083" (lazy(sprintf "%+1g" -10.0)) "-10"
    test "test7084" (lazy(sprintf "%+*g" 7 -10.0)) "    -10"
    test "test7085" (lazy(sprintf "%+.5g" -10.0)) "-10"
    test "test7086" (lazy(sprintf "%+.*g" 4 -10.0)) "-10"
    test "test7087" (lazy(sprintf "%+*.*g" 5 4 -10.0)) "  -10"
    test "test7088" (lazy(sprintf "%-+g" -10.0)) "-10"
    test "test7089" (lazy(sprintf "%-+5g" -10.0)) "-10  "
    test "test7090" (lazy(sprintf "%-+1g" -10.0)) "-10"
    test "test7091" (lazy(sprintf "%-+*g" 7 -10.0)) "-10    "
    test "test7092" (lazy(sprintf "%-+.5g" -10.0)) "-10"
    test "test7093" (lazy(sprintf "%-+.*g" 4 -10.0)) "-10"
    test "test7094" (lazy(sprintf "%-+*.*g" 5 4 -10.0)) "-10  "
    test "test7095" (lazy(sprintf "%+0g" -10.0)) "-10"
    test "test7096" (lazy(sprintf "%+05g" -10.0)) "-0010"// "00-10"
    test "test7097" (lazy(sprintf "%+01g" -10.0)) "-10"
    test "test7098" (lazy(sprintf "%+0*g" 7 -10.0)) "-000010"//  "0000-10"
    test "test7099" (lazy(sprintf "%+0.5g" -10.0)) "-10"
    test "test7100" (lazy(sprintf "%+0.*g" 4 -10.0)) "-10"
    test "test7101" (lazy(sprintf "%+0*.*g" 5 4 -10.0)) "-0010"//  "00-10"
    test "test7102" (lazy(sprintf "%-+0g" -10.0)) "-10"
    test "test7103" (lazy(sprintf "%-+05g" -10.0)) "-10  "// "-1000"
    test "test7104" (lazy(sprintf "%-+01g" -10.0)) "-10"
    test "test7105" (lazy(sprintf "%-+0*g" 7 -10.0)) "-10    "// "-100000"
    test "test7106" (lazy(sprintf "%-+0.5g" -10.0)) "-10"
    test "test7107" (lazy(sprintf "%-+0.*g" 4 -10.0)) "-10"
    test "test7108" (lazy(sprintf "%-+0*.*g" 5 4 -10.0)) "-10  "//  "-1000"
    test "test7109" (lazy(sprintf "% g" -10.0)) "-10"
    test "test7110" (lazy(sprintf "% 5g" -10.0)) "  -10"
    test "test7111" (lazy(sprintf "% 1g" -10.0)) "-10"
    test "test7112" (lazy(sprintf "% *g" 7 -10.0)) "    -10"
    test "test7113" (lazy(sprintf "% .5g" -10.0)) "-10"
    test "test7114" (lazy(sprintf "% .*g" 4 -10.0)) "-10"
    test "test7115" (lazy(sprintf "% *.*g" 5 4 -10.0)) "  -10"
    test "test7116" (lazy(sprintf "%- g" -10.0)) "-10"
    test "test7117" (lazy(sprintf "%- 5g" -10.0)) "-10  "
    test "test7118" (lazy(sprintf "%- 1g" -10.0)) "-10"
    test "test7119" (lazy(sprintf "%- *g" 7 -10.0)) "-10    "
    test "test7120" (lazy(sprintf "%- .5g" -10.0)) "-10"
    test "test7121" (lazy(sprintf "%- .*g" 4 -10.0)) "-10"
    test "test7122" (lazy(sprintf "%- *.*g" 5 4 -10.0)) "-10  "
    test "test7123" (lazy(sprintf "% 0g" -10.0)) "-10"
    test "test7124" (lazy(sprintf "% 05g" -10.0)) "-0010"// "00-10"
    test "test7125" (lazy(sprintf "% 01g" -10.0)) "-10"
    test "test7126" (lazy(sprintf "% 0*g" 7 -10.0)) "-000010"// "0000-10"
    test "test7127" (lazy(sprintf "% 0.5g" -10.0)) "-10"
    test "test7128" (lazy(sprintf "% 0.*g" 4 -10.0)) "-10"
    test "test7129" (lazy(sprintf "% 0*.*g" 5 4 -10.0)) "-0010"//  "00-10"
    test "test7130" (lazy(sprintf "%- 0g" -10.0)) "-10"
    test "test7131" (lazy(sprintf "%- 05g" -10.0)) "-10  "//  "-1000"
    test "test7132" (lazy(sprintf "%- 01g" -10.0)) "-10"
    test "test7133" (lazy(sprintf "%- 0*g" 7 -10.0)) "-10    "// "-100000"
    test "test7134" (lazy(sprintf "%- 0.5g" -10.0)) "-10"
    test "test7135" (lazy(sprintf "%- 0.*g" 4 -10.0)) "-10"
    test "test7136" (lazy(sprintf "%- 0*.*g" 5 4 -10.0)) "-10  "//  "-1000"
    test "test7137" (lazy(sprintf "%g" 5.0f)) "5"
    test "test7138" (lazy(sprintf "%5g" 5.0f)) "    5"
    test "test7139" (lazy(sprintf "%1g" 5.0f)) "5"
    test "test7140" (lazy(sprintf "%*g" 7 5.0f)) "      5"
    test "test7141" (lazy(sprintf "%.5g" 5.0f)) "5"
    test "test7142" (lazy(sprintf "%.*g" 4 5.0f)) "5"
    test "test7143" (lazy(sprintf "%*.*g" 5 4 5.0f)) "    5"
    test "test7144" (lazy(sprintf "%-g" 5.0f)) "5"
    test "test7145" (lazy(sprintf "%-5g" 5.0f)) "5    "
    test "test7146" (lazy(sprintf "%-1g" 5.0f)) "5"
    test "test7147" (lazy(sprintf "%-*g" 7 5.0f)) "5      "
    test "test7148" (lazy(sprintf "%-.5g" 5.0f)) "5"
    test "test7149" (lazy(sprintf "%-.*g" 4 5.0f)) "5"
    test "test7150" (lazy(sprintf "%-*.*g" 5 4 5.0f)) "5    "
    test "test7151" (lazy(sprintf "%0g" 5.0f)) "5"
    test "test7152" (lazy(sprintf "%05g" 5.0f)) "00005"
    test "test7153" (lazy(sprintf "%01g" 5.0f)) "5"
    test "test7154" (lazy(sprintf "%0*g" 7 5.0f)) "0000005"
    test "test7155" (lazy(sprintf "%0.5g" 5.0f)) "5"
    test "test7156" (lazy(sprintf "%0.*g" 4 5.0f)) "5"
    test "test7157" (lazy(sprintf "%0*.*g" 5 4 5.0f)) "00005"
    test "test7158" (lazy(sprintf "%-0g" 5.0f)) "5"
    test "test7159" (lazy(sprintf "%-05g" 5.0f)) "5    "// "50000"
    test "test7160" (lazy(sprintf "%-01g" 5.0f)) "5"
    test "test7161" (lazy(sprintf "%-0*g" 7 5.0f)) "5      "//  "5000000"
    test "test7162" (lazy(sprintf "%-0.5g" 5.0f)) "5"
    test "test7163" (lazy(sprintf "%-0.*g" 4 5.0f)) "5"
    test "test7164" (lazy(sprintf "%-0*.*g" 5 4 5.0f)) "5    "// "50000"
    test "test7165" (lazy(sprintf "%+g" 5.0f)) "+5"
    test "test7166" (lazy(sprintf "%+5g" 5.0f)) "   +5"
    test "test7167" (lazy(sprintf "%+1g" 5.0f)) "+5"
    test "test7168" (lazy(sprintf "%+*g" 7 5.0f)) "     +5"
    test "test7169" (lazy(sprintf "%+.5g" 5.0f)) "+5"
    test "test7170" (lazy(sprintf "%+.*g" 4 5.0f)) "+5"
    test "test7171" (lazy(sprintf "%+*.*g" 5 4 5.0f)) "   +5"
    test "test7172" (lazy(sprintf "%-+g" 5.0f)) "+5"
    test "test7173" (lazy(sprintf "%-+5g" 5.0f)) "+5   "
    test "test7174" (lazy(sprintf "%-+1g" 5.0f)) "+5"
    test "test7175" (lazy(sprintf "%-+*g" 7 5.0f)) "+5     "
    test "test7176" (lazy(sprintf "%-+.5g" 5.0f)) "+5"
    test "test7177" (lazy(sprintf "%-+.*g" 4 5.0f)) "+5"
    test "test7178" (lazy(sprintf "%-+*.*g" 5 4 5.0f)) "+5   "
    test "test7179" (lazy(sprintf "%+0g" 5.0f)) "+5"
    test "test7180" (lazy(sprintf "%+05g" 5.0f)) "+0005"// "000+5"
    test "test7181" (lazy(sprintf "%+01g" 5.0f)) "+5"
    test "test7182" (lazy(sprintf "%+0*g" 7 5.0f)) "+000005"//  "00000+5"
    test "test7183" (lazy(sprintf "%+0.5g" 5.0f)) "+5"
    test "test7184" (lazy(sprintf "%+0.*g" 4 5.0f)) "+5"
    test "test7185" (lazy(sprintf "%+0*.*g" 5 4 5.0f)) "+0005"//  "000+5"
    test "test7186" (lazy(sprintf "%-+0g" 5.0f)) "+5"
    test "test7187" (lazy(sprintf "%-+05g" 5.0f)) "+5   "// "+5000"
    test "test7188" (lazy(sprintf "%-+01g" 5.0f)) "+5"
    test "test7189" (lazy(sprintf "%-+0*g" 7 5.0f)) "+5     "// "+500000"
    test "test7190" (lazy(sprintf "%-+0.5g" 5.0f)) "+5"
    test "test7191" (lazy(sprintf "%-+0.*g" 4 5.0f)) "+5"
    test "test7192" (lazy(sprintf "%-+0*.*g" 5 4 5.0f)) "+5   "// "+5000"
    test "test7193" (lazy(sprintf "% g" 5.0f)) " 5"
    test "test7194" (lazy(sprintf "% 5g" 5.0f)) "    5"
    test "test7195" (lazy(sprintf "% 1g" 5.0f)) " 5"
    test "test7196" (lazy(sprintf "% *g" 7 5.0f)) "      5"
    test "test7197" (lazy(sprintf "% .5g" 5.0f)) " 5"
    test "test7198" (lazy(sprintf "% .*g" 4 5.0f)) " 5"
    test "test7199" (lazy(sprintf "% *.*g" 5 4 5.0f)) "    5"
    test "test7200" (lazy(sprintf "%- g" 5.0f)) " 5"
    test "test7201" (lazy(sprintf "%- 5g" 5.0f)) " 5   "
    test "test7202" (lazy(sprintf "%- 1g" 5.0f)) " 5"
    test "test7203" (lazy(sprintf "%- *g" 7 5.0f)) " 5     "
    test "test7204" (lazy(sprintf "%- .5g" 5.0f)) " 5"
    test "test7205" (lazy(sprintf "%- .*g" 4 5.0f)) " 5"
    test "test7206" (lazy(sprintf "%- *.*g" 5 4 5.0f)) " 5   "
    test "test7207" (lazy(sprintf "% 0g" 5.0f)) " 5"
    test "test7208" (lazy(sprintf "% 05g" 5.0f)) " 0005"// "000 5"
    test "test7209" (lazy(sprintf "% 01g" 5.0f)) " 5"
    test "test7210" (lazy(sprintf "% 0*g" 7 5.0f)) " 000005"//  "00000 5"
    test "test7211" (lazy(sprintf "% 0.5g" 5.0f)) " 5"
    test "test7212" (lazy(sprintf "% 0.*g" 4 5.0f)) " 5"
    test "test7213" (lazy(sprintf "% 0*.*g" 5 4 5.0f)) " 0005"// "000 5"
    test "test7214" (lazy(sprintf "%- 0g" 5.0f)) " 5"
    test "test7215" (lazy(sprintf "%- 05g" 5.0f)) " 5   "//  " 5000"
    test "test7216" (lazy(sprintf "%- 01g" 5.0f)) " 5"
    test "test7217" (lazy(sprintf "%- 0*g" 7 5.0f)) " 5     "// " 500000"
    test "test7218" (lazy(sprintf "%- 0.5g" 5.0f)) " 5"
    test "test7219" (lazy(sprintf "%- 0.*g" 4 5.0f)) " 5"
    test "test7220" (lazy(sprintf "%- 0*.*g" 5 4 5.0f)) " 5   "//  " 5000"
    test "test7221" (lazy(sprintf "%g" -10.0f)) "-10"
    test "test7222" (lazy(sprintf "%5g" -10.0f)) "  -10"
    test "test7223" (lazy(sprintf "%1g" -10.0f)) "-10"
    test "test7224" (lazy(sprintf "%*g" 7 -10.0f)) "    -10"
    test "test7225" (lazy(sprintf "%.5g" -10.0f)) "-10"
    test "test7226" (lazy(sprintf "%.*g" 4 -10.0f)) "-10"
    test "test7227" (lazy(sprintf "%*.*g" 5 4 -10.0f)) "  -10"
    test "test7228" (lazy(sprintf "%-g" -10.0f)) "-10"
    test "test7229" (lazy(sprintf "%-5g" -10.0f)) "-10  "
    test "test7230" (lazy(sprintf "%-1g" -10.0f)) "-10"
    test "test7231" (lazy(sprintf "%-*g" 7 -10.0f)) "-10    "
    test "test7232" (lazy(sprintf "%-.5g" -10.0f)) "-10"
    test "test7233" (lazy(sprintf "%-.*g" 4 -10.0f)) "-10"
    test "test7234" (lazy(sprintf "%-*.*g" 5 4 -10.0f)) "-10  "
    test "test7235" (lazy(sprintf "%0g" -10.0f)) "-10"
    test "test7236" (lazy(sprintf "%05g" -10.0f)) "-0010"//"00-10"
    test "test7237" (lazy(sprintf "%01g" -10.0f)) "-10"
    test "test7238" (lazy(sprintf "%0*g" 7 -10.0f)) "-000010"//  "0000-10"
    test "test7239" (lazy(sprintf "%0.5g" -10.0f)) "-10"
    test "test7240" (lazy(sprintf "%0.*g" 4 -10.0f)) "-10"
    test "test7241" (lazy(sprintf "%0*.*g" 5 4 -10.0f)) "-0010"//  "00-10"
    test "test7242" (lazy(sprintf "%-0g" -10.0f)) "-10"
    test "test7243" (lazy(sprintf "%-05g" -10.0f)) "-10  "// "-1000"
    test "test7244" (lazy(sprintf "%-01g" -10.0f)) "-10"
    test "test7245" (lazy(sprintf "%-0*g" 7 -10.0f)) "-10    "//  "-100000"
    test "test7246" (lazy(sprintf "%-0.5g" -10.0f)) "-10"
    test "test7247" (lazy(sprintf "%-0.*g" 4 -10.0f)) "-10"
    test "test7248" (lazy(sprintf "%-0*.*g" 5 4 -10.0f)) "-10  "// "-1000"
    test "test7249" (lazy(sprintf "%+g" -10.0f)) "-10"
    test "test7250" (lazy(sprintf "%+5g" -10.0f)) "  -10"
    test "test7251" (lazy(sprintf "%+1g" -10.0f)) "-10"
    test "test7252" (lazy(sprintf "%+*g" 7 -10.0f)) "    -10"
    test "test7253" (lazy(sprintf "%+.5g" -10.0f)) "-10"
    test "test7254" (lazy(sprintf "%+.*g" 4 -10.0f)) "-10"
    test "test7255" (lazy(sprintf "%+*.*g" 5 4 -10.0f)) "  -10"
    test "test7256" (lazy(sprintf "%-+g" -10.0f)) "-10"
    test "test7257" (lazy(sprintf "%-+5g" -10.0f)) "-10  "
    test "test7258" (lazy(sprintf "%-+1g" -10.0f)) "-10"
    test "test7259" (lazy(sprintf "%-+*g" 7 -10.0f)) "-10    "
    test "test7260" (lazy(sprintf "%-+.5g" -10.0f)) "-10"
    test "test7261" (lazy(sprintf "%-+.*g" 4 -10.0f)) "-10"
    test "test7262" (lazy(sprintf "%-+*.*g" 5 4 -10.0f)) "-10  "
    test "test7263" (lazy(sprintf "%+0g" -10.0f)) "-10"
    test "test7264" (lazy(sprintf "%+05g" -10.0f)) "-0010"// "00-10"
    test "test7265" (lazy(sprintf "%+01g" -10.0f)) "-10"
    test "test7266" (lazy(sprintf "%+0*g" 7 -10.0f)) "-000010"//  "0000-10"
    test "test7267" (lazy(sprintf "%+0.5g" -10.0f)) "-10"
    test "test7268" (lazy(sprintf "%+0.*g" 4 -10.0f)) "-10"
    test "test7269" (lazy(sprintf "%+0*.*g" 5 4 -10.0f)) "-0010"// "00-10"
    test "test7270" (lazy(sprintf "%-+0g" -10.0f)) "-10"
    test "test7271" (lazy(sprintf "%-+05g" -10.0f)) "-10  " // "-1000"
    test "test7272" (lazy(sprintf "%-+01g" -10.0f)) "-10"
    test "test7273" (lazy(sprintf "%-+0*g" 7 -10.0f)) "-10    "// "-100000"
    test "test7274" (lazy(sprintf "%-+0.5g" -10.0f)) "-10"
    test "test7275" (lazy(sprintf "%-+0.*g" 4 -10.0f)) "-10"
    test "test7276" (lazy(sprintf "%-+0*.*g" 5 4 -10.0f)) "-10  "// "-1000"
    test "test7277" (lazy(sprintf "% g" -10.0f)) "-10"
    test "test7278" (lazy(sprintf "% 5g" -10.0f)) "  -10"
    test "test7279" (lazy(sprintf "% 1g" -10.0f)) "-10"
    test "test7280" (lazy(sprintf "% *g" 7 -10.0f)) "    -10"
    test "test7281" (lazy(sprintf "% .5g" -10.0f)) "-10"
    test "test7282" (lazy(sprintf "% .*g" 4 -10.0f)) "-10"
    test "test7283" (lazy(sprintf "% *.*g" 5 4 -10.0f)) "  -10"
    test "test7284" (lazy(sprintf "%- g" -10.0f)) "-10"
    test "test7285" (lazy(sprintf "%- 5g" -10.0f)) "-10  "
    test "test7286" (lazy(sprintf "%- 1g" -10.0f)) "-10"
    test "test7287" (lazy(sprintf "%- *g" 7 -10.0f)) "-10    "
    test "test7288" (lazy(sprintf "%- .5g" -10.0f)) "-10"
    test "test7289" (lazy(sprintf "%- .*g" 4 -10.0f)) "-10"
    test "test7290" (lazy(sprintf "%- *.*g" 5 4 -10.0f)) "-10  "
    test "test7291" (lazy(sprintf "% 0g" -10.0f)) "-10"
    test "test7292" (lazy(sprintf "% 05g" -10.0f)) "-0010"//  "00-10"
    test "test7293" (lazy(sprintf "% 01g" -10.0f)) "-10"
    test "test7294" (lazy(sprintf "% 0*g" 7 -10.0f)) "-000010"//  "0000-10"
    test "test7295" (lazy(sprintf "% 0.5g" -10.0f)) "-10"
    test "test7296" (lazy(sprintf "% 0.*g" 4 -10.0f)) "-10"
    test "test7297" (lazy(sprintf "% 0*.*g" 5 4 -10.0f)) "-0010"//  "00-10"
    test "test7298" (lazy(sprintf "%- 0g" -10.0f)) "-10"
    test "test7299" (lazy(sprintf "%- 05g" -10.0f)) "-10  "// "-1000"
    test "test7300" (lazy(sprintf "%- 01g" -10.0f)) "-10"
    test "test7301" (lazy(sprintf "%- 0*g" 7 -10.0f)) "-10    "//  "-100000"
    test "test7302" (lazy(sprintf "%- 0.5g" -10.0f)) "-10"
    test "test7303" (lazy(sprintf "%- 0.*g" 4 -10.0f)) "-10"
    test "test7304" (lazy(sprintf "%- 0*.*g" 5 4 -10.0f)) "-10  "//  "-1000"
    test "test7305" (lazy(sprintf "%g" 5.0M)) "5"
    test "test7306" (lazy(sprintf "%5g" 5.0M)) "    5"
    test "test7307" (lazy(sprintf "%1g" 5.0M)) "5"
    test "test7308" (lazy(sprintf "%*g" 7 5.0M)) "      5"
    test "test7309" (lazy(sprintf "%.5g" 5.0M)) "5"
    test "test7310" (lazy(sprintf "%.*g" 4 5.0M)) "5"
    test "test7311" (lazy(sprintf "%*.*g" 5 4 5.0M)) "    5"
    test "test7312" (lazy(sprintf "%-g" 5.0M)) "5"
    test "test7313" (lazy(sprintf "%-5g" 5.0M)) "5    "
    test "test7314" (lazy(sprintf "%-1g" 5.0M)) "5"
    test "test7315" (lazy(sprintf "%-*g" 7 5.0M)) "5      "
    test "test7316" (lazy(sprintf "%-.5g" 5.0M)) "5"
    test "test7317" (lazy(sprintf "%-.*g" 4 5.0M)) "5"
    test "test7318" (lazy(sprintf "%-*.*g" 5 4 5.0M)) "5    "
    test "test7319" (lazy(sprintf "%0g" 5.0M)) "5"
    test "test7320" (lazy(sprintf "%05g" 5.0M)) "00005"
    test "test7321" (lazy(sprintf "%01g" 5.0M)) "5"
    test "test7322" (lazy(sprintf "%0*g" 7 5.0M)) "0000005"
    test "test7323" (lazy(sprintf "%0.5g" 5.0M)) "5"
    test "test7324" (lazy(sprintf "%0.*g" 4 5.0M)) "5"
    test "test7325" (lazy(sprintf "%0*.*g" 5 4 5.0M)) "00005"
    test "test7326" (lazy(sprintf "%-0g" 5.0M)) "5"
    test "test7327" (lazy(sprintf "%-05g" 5.0M)) "5    "// "50000"
    test "test7328" (lazy(sprintf "%-01g" 5.0M)) "5"
    test "test7329" (lazy(sprintf "%-0*g" 7 5.0M)) "5      "// "5000000"
    test "test7330" (lazy(sprintf "%-0.5g" 5.0M)) "5"
    test "test7331" (lazy(sprintf "%-0.*g" 4 5.0M)) "5"
    test "test7332" (lazy(sprintf "%-0*.*g" 5 4 5.0M)) "5    "// "50000"
    test "test7333" (lazy(sprintf "%+g" 5.0M)) "+5"
    test "test7334" (lazy(sprintf "%+5g" 5.0M)) "   +5"
    test "test7335" (lazy(sprintf "%+1g" 5.0M)) "+5"
    test "test7336" (lazy(sprintf "%+*g" 7 5.0M)) "     +5"
    test "test7337" (lazy(sprintf "%+.5g" 5.0M)) "+5"
    test "test7338" (lazy(sprintf "%+.*g" 4 5.0M)) "+5"
    test "test7339" (lazy(sprintf "%+*.*g" 5 4 5.0M)) "   +5"
    test "test7340" (lazy(sprintf "%-+g" 5.0M)) "+5"
    test "test7341" (lazy(sprintf "%-+5g" 5.0M)) "+5   "
    test "test7342" (lazy(sprintf "%-+1g" 5.0M)) "+5"
    test "test7343" (lazy(sprintf "%-+*g" 7 5.0M)) "+5     "
    test "test7344" (lazy(sprintf "%-+.5g" 5.0M)) "+5"
    test "test7345" (lazy(sprintf "%-+.*g" 4 5.0M)) "+5"
    test "test7346" (lazy(sprintf "%-+*.*g" 5 4 5.0M)) "+5   "
    test "test7347" (lazy(sprintf "%+0g" 5.0M)) "+5"
    test "test7348" (lazy(sprintf "%+05g" 5.0M)) "+0005"// "000+5"
    test "test7349" (lazy(sprintf "%+01g" 5.0M)) "+5"
    test "test7350" (lazy(sprintf "%+0*g" 7 5.0M)) "+000005"//  "00000+5"
    test "test7351" (lazy(sprintf "%+0.5g" 5.0M)) "+5"
    test "test7352" (lazy(sprintf "%+0.*g" 4 5.0M)) "+5"
    test "test7353" (lazy(sprintf "%+0*.*g" 5 4 5.0M)) "+0005"//  "000+5"
    test "test7354" (lazy(sprintf "%-+0g" 5.0M)) "+5"
    test "test7355" (lazy(sprintf "%-+05g" 5.0M)) "+5   "// "+5000"
    test "test7356" (lazy(sprintf "%-+01g" 5.0M)) "+5"
    test "test7357" (lazy(sprintf "%-+0*g" 7 5.0M)) "+5     "// "+500000"
    test "test7358" (lazy(sprintf "%-+0.5g" 5.0M)) "+5"
    test "test7359" (lazy(sprintf "%-+0.*g" 4 5.0M)) "+5"
    test "test7360" (lazy(sprintf "%-+0*.*g" 5 4 5.0M)) "+5   "//"+5000"
    test "test7361" (lazy(sprintf "% g" 5.0M)) " 5"
    test "test7362" (lazy(sprintf "% 5g" 5.0M)) "    5"
    test "test7363" (lazy(sprintf "% 1g" 5.0M)) " 5"
    test "test7364" (lazy(sprintf "% *g" 7 5.0M)) "      5"
    test "test7365" (lazy(sprintf "% .5g" 5.0M)) " 5"
    test "test7366" (lazy(sprintf "% .*g" 4 5.0M)) " 5"
    test "test7367" (lazy(sprintf "% *.*g" 5 4 5.0M)) "    5"
    test "test7368" (lazy(sprintf "%- g" 5.0M)) " 5"
    test "test7369" (lazy(sprintf "%- 5g" 5.0M)) " 5   "
    test "test7370" (lazy(sprintf "%- 1g" 5.0M)) " 5"
    test "test7371" (lazy(sprintf "%- *g" 7 5.0M)) " 5     "
    test "test7372" (lazy(sprintf "%- .5g" 5.0M)) " 5"
    test "test7373" (lazy(sprintf "%- .*g" 4 5.0M)) " 5"
    test "test7374" (lazy(sprintf "%- *.*g" 5 4 5.0M)) " 5   "
    test "test7375" (lazy(sprintf "% 0g" 5.0M)) " 5"
    test "test7376" (lazy(sprintf "% 05g" 5.0M)) " 0005"// "000 5"
    test "test7377" (lazy(sprintf "% 01g" 5.0M)) " 5"
    test "test7378" (lazy(sprintf "% 0*g" 7 5.0M)) " 000005"// "00000 5"
    test "test7379" (lazy(sprintf "% 0.5g" 5.0M)) " 5"
    test "test7380" (lazy(sprintf "% 0.*g" 4 5.0M)) " 5"
    test "test7381" (lazy(sprintf "% 0*.*g" 5 4 5.0M)) " 0005"// "000 5"
    test "test7382" (lazy(sprintf "%- 0g" 5.0M)) " 5"
    test "test7383" (lazy(sprintf "%- 05g" 5.0M)) " 5   "// " 5000"
    test "test7384" (lazy(sprintf "%- 01g" 5.0M)) " 5"
    test "test7385" (lazy(sprintf "%- 0*g" 7 5.0M)) " 5     "// " 500000"
    test "test7386" (lazy(sprintf "%- 0.5g" 5.0M)) " 5"
    test "test7387" (lazy(sprintf "%- 0.*g" 4 5.0M)) " 5"
    test "test7388" (lazy(sprintf "%- 0*.*g" 5 4 5.0M)) " 5   "//  " 5000"
    test "test7389" (lazy(sprintf "%g" -10.0M)) "-10"
    test "test7390" (lazy(sprintf "%5g" -10.0M)) "  -10"
    test "test7391" (lazy(sprintf "%1g" -10.0M)) "-10"
    test "test7392" (lazy(sprintf "%*g" 7 -10.0M)) "    -10"
    test "test7393" (lazy(sprintf "%.5g" -10.0M)) "-10"
    test "test7394" (lazy(sprintf "%.*g" 4 -10.0M)) "-10"
    test "test7395" (lazy(sprintf "%*.*g" 5 4 -10.0M)) "  -10"
    test "test7396" (lazy(sprintf "%-g" -10.0M)) "-10"
    test "test7397" (lazy(sprintf "%-5g" -10.0M)) "-10  "
    test "test7398" (lazy(sprintf "%-1g" -10.0M)) "-10"
    test "test7399" (lazy(sprintf "%-*g" 7 -10.0M)) "-10    "
    test "test7400" (lazy(sprintf "%-.5g" -10.0M)) "-10"
    test "test7401" (lazy(sprintf "%-.*g" 4 -10.0M)) "-10"
    test "test7402" (lazy(sprintf "%-*.*g" 5 4 -10.0M)) "-10  "
    test "test7403" (lazy(sprintf "%0g" -10.0M)) "-10"
    test "test7404" (lazy(sprintf "%05g" -10.0M)) "-0010"//  "00-10"
    test "test7405" (lazy(sprintf "%01g" -10.0M)) "-10"
    test "test7406" (lazy(sprintf "%0*g" 7 -10.0M)) "-000010"// "0000-10"
    test "test7407" (lazy(sprintf "%0.5g" -10.0M)) "-10"
    test "test7408" (lazy(sprintf "%0.*g" 4 -10.0M)) "-10"
    test "test7409" (lazy(sprintf "%0*.*g" 5 4 -10.0M)) "-0010"//  "00-10"
    test "test7410" (lazy(sprintf "%-0g" -10.0M)) "-10"
    test "test7411" (lazy(sprintf "%-05g" -10.0M)) "-10  "// "-1000"
    test "test7412" (lazy(sprintf "%-01g" -10.0M)) "-10"
    test "test7413" (lazy(sprintf "%-0*g" 7 -10.0M)) "-10    "//  "-100000"
    test "test7414" (lazy(sprintf "%-0.5g" -10.0M)) "-10"
    test "test7415" (lazy(sprintf "%-0.*g" 4 -10.0M)) "-10"
    test "test7416" (lazy(sprintf "%-0*.*g" 5 4 -10.0M)) "-10  "// "-1000"
    test "test7417" (lazy(sprintf "%+g" -10.0M)) "-10"
    test "test7418" (lazy(sprintf "%+5g" -10.0M)) "  -10"
    test "test7419" (lazy(sprintf "%+1g" -10.0M)) "-10"
    test "test7420" (lazy(sprintf "%+*g" 7 -10.0M)) "    -10"
    test "test7421" (lazy(sprintf "%+.5g" -10.0M)) "-10"
    test "test7422" (lazy(sprintf "%+.*g" 4 -10.0M)) "-10"
    test "test7423" (lazy(sprintf "%+*.*g" 5 4 -10.0M)) "  -10"
    test "test7424" (lazy(sprintf "%-+g" -10.0M)) "-10"
    test "test7425" (lazy(sprintf "%-+5g" -10.0M)) "-10  "
    test "test7426" (lazy(sprintf "%-+1g" -10.0M)) "-10"
    test "test7427" (lazy(sprintf "%-+*g" 7 -10.0M)) "-10    "
    test "test7428" (lazy(sprintf "%-+.5g" -10.0M)) "-10"
    test "test7429" (lazy(sprintf "%-+.*g" 4 -10.0M)) "-10"
    test "test7430" (lazy(sprintf "%-+*.*g" 5 4 -10.0M)) "-10  "
    test "test7431" (lazy(sprintf "%+0g" -10.0M)) "-10"
    test "test7432" (lazy(sprintf "%+05g" -10.0M)) "-0010"// "00-10"
    test "test7433" (lazy(sprintf "%+01g" -10.0M)) "-10"
    test "test7434" (lazy(sprintf "%+0*g" 7 -10.0M)) "-000010"// "0000-10"
    test "test7435" (lazy(sprintf "%+0.5g" -10.0M)) "-10"
    test "test7436" (lazy(sprintf "%+0.*g" 4 -10.0M)) "-10"
    test "test7437" (lazy(sprintf "%+0*.*g" 5 4 -10.0M)) "-0010"// "00-10"
    test "test7438" (lazy(sprintf "%-+0g" -10.0M)) "-10"
    test "test7439" (lazy(sprintf "%-+05g" -10.0M)) "-10  "// "-1000"
    test "test7440" (lazy(sprintf "%-+01g" -10.0M)) "-10"
    test "test7441" (lazy(sprintf "%-+0*g" 7 -10.0M)) "-10    "//  "-100000"
    test "test7442" (lazy(sprintf "%-+0.5g" -10.0M)) "-10"
    test "test7443" (lazy(sprintf "%-+0.*g" 4 -10.0M)) "-10"
    test "test7444" (lazy(sprintf "%-+0*.*g" 5 4 -10.0M)) "-10  "//"-1000"
    test "test7445" (lazy(sprintf "% g" -10.0M)) "-10"
    test "test7446" (lazy(sprintf "% 5g" -10.0M)) "  -10"
    test "test7447" (lazy(sprintf "% 1g" -10.0M)) "-10"
    test "test7448" (lazy(sprintf "% *g" 7 -10.0M)) "    -10"
    test "test7449" (lazy(sprintf "% .5g" -10.0M)) "-10"
    test "test7450" (lazy(sprintf "% .*g" 4 -10.0M)) "-10"
    test "test7451" (lazy(sprintf "% *.*g" 5 4 -10.0M)) "  -10"
    test "test7452" (lazy(sprintf "%- g" -10.0M)) "-10"
    test "test7453" (lazy(sprintf "%- 5g" -10.0M)) "-10  "
    test "test7454" (lazy(sprintf "%- 1g" -10.0M)) "-10"
    test "test7455" (lazy(sprintf "%- *g" 7 -10.0M)) "-10    "
    test "test7456" (lazy(sprintf "%- .5g" -10.0M)) "-10"
    test "test7457" (lazy(sprintf "%- .*g" 4 -10.0M)) "-10"
    test "test7458" (lazy(sprintf "%- *.*g" 5 4 -10.0M)) "-10  "
    test "test7459" (lazy(sprintf "% 0g" -10.0M)) "-10"
    test "test7460" (lazy(sprintf "% 05g" -10.0M)) "-0010"// "00-10"
    test "test7461" (lazy(sprintf "% 01g" -10.0M)) "-10"
    test "test7462" (lazy(sprintf "% 0*g" 7 -10.0M)) "-000010"//  "0000-10"
    test "test7463" (lazy(sprintf "% 0.5g" -10.0M)) "-10"
    test "test7464" (lazy(sprintf "% 0.*g" 4 -10.0M)) "-10"
    test "test7465" (lazy(sprintf "% 0*.*g" 5 4 -10.0M)) "-0010"//  "00-10"
    test "test7466" (lazy(sprintf "%- 0g" -10.0M)) "-10"
    test "test7467" (lazy(sprintf "%- 05g" -10.0M)) "-10  "// "-1000"
    test "test7468" (lazy(sprintf "%- 01g" -10.0M)) "-10"
    test "test7469" (lazy(sprintf "%- 0*g" 7 -10.0M)) "-10    "// "-100000"
    test "test7470" (lazy(sprintf "%- 0.5g" -10.0M)) "-10"
    test "test7471" (lazy(sprintf "%- 0.*g" 4 -10.0M)) "-10"
    test "test7472" (lazy(sprintf "%- 0*.*g" 5 4 -10.0M)) "-10  "// "-1000"
    test "test7473" (lazy(sprintf "%G" 5.0)) "5"
    test "test7474" (lazy(sprintf "%5G" 5.0)) "    5"
    test "test7475" (lazy(sprintf "%1G" 5.0)) "5"
    test "test7476" (lazy(sprintf "%*G" 7 5.0)) "      5"
    test "test7477" (lazy(sprintf "%.5G" 5.0)) "5"
    test "test7478" (lazy(sprintf "%.*G" 4 5.0)) "5"
    test "test7479" (lazy(sprintf "%*.*G" 5 4 5.0)) "    5"
    test "test7480" (lazy(sprintf "%-G" 5.0)) "5"
    test "test7481" (lazy(sprintf "%-5G" 5.0)) "5    "
    test "test7482" (lazy(sprintf "%-1G" 5.0)) "5"
    test "test7483" (lazy(sprintf "%-*G" 7 5.0)) "5      "
    test "test7484" (lazy(sprintf "%-.5G" 5.0)) "5"
    test "test7485" (lazy(sprintf "%-.*G" 4 5.0)) "5"
    test "test7486" (lazy(sprintf "%-*.*G" 5 4 5.0)) "5    "
    test "test7487" (lazy(sprintf "%0G" 5.0)) "5"
    test "test7488" (lazy(sprintf "%05G" 5.0)) "00005"
    test "test7489" (lazy(sprintf "%01G" 5.0)) "5"
    test "test7490" (lazy(sprintf "%0*G" 7 5.0)) "0000005"
    test "test7491" (lazy(sprintf "%0.5G" 5.0)) "5"
    test "test7492" (lazy(sprintf "%0.*G" 4 5.0)) "5"
    test "test7493" (lazy(sprintf "%0*.*G" 5 4 5.0)) "00005"
    test "test7494" (lazy(sprintf "%-0G" 5.0)) "5"
    test "test7495" (lazy(sprintf "%-05G" 5.0)) "5    "// "50000"
    test "test7496" (lazy(sprintf "%-01G" 5.0)) "5"
    test "test7497" (lazy(sprintf "%-0*G" 7 5.0)) "5      "//  "5000000"
    test "test7498" (lazy(sprintf "%-0.5G" 5.0)) "5"
    test "test7499" (lazy(sprintf "%-0.*G" 4 5.0)) "5"
    test "test7500" (lazy(sprintf "%-0*.*G" 5 4 5.0)) "5    "// "50000"
    test "test7501" (lazy(sprintf "%+G" 5.0)) "+5"
    test "test7502" (lazy(sprintf "%+5G" 5.0)) "   +5"
    test "test7503" (lazy(sprintf "%+1G" 5.0)) "+5"
    test "test7504" (lazy(sprintf "%+*G" 7 5.0)) "     +5"
    test "test7505" (lazy(sprintf "%+.5G" 5.0)) "+5"
    test "test7506" (lazy(sprintf "%+.*G" 4 5.0)) "+5"
    test "test7507" (lazy(sprintf "%+*.*G" 5 4 5.0)) "   +5"
    test "test7508" (lazy(sprintf "%-+G" 5.0)) "+5"
    test "test7509" (lazy(sprintf "%-+5G" 5.0)) "+5   "
    test "test7510" (lazy(sprintf "%-+1G" 5.0)) "+5"
    test "test7511" (lazy(sprintf "%-+*G" 7 5.0)) "+5     "
    test "test7512" (lazy(sprintf "%-+.5G" 5.0)) "+5"
    test "test7513" (lazy(sprintf "%-+.*G" 4 5.0)) "+5"
    test "test7514" (lazy(sprintf "%-+*.*G" 5 4 5.0)) "+5   "
    test "test7515" (lazy(sprintf "%+0G" 5.0)) "+5"
    test "test7516" (lazy(sprintf "%+05G" 5.0)) "+0005"// "000+5"
    test "test7517" (lazy(sprintf "%+01G" 5.0)) "+5"
    test "test7518" (lazy(sprintf "%+0*G" 7 5.0)) "+000005"//  "00000+5"
    test "test7519" (lazy(sprintf "%+0.5G" 5.0)) "+5"
    test "test7520" (lazy(sprintf "%+0.*G" 4 5.0)) "+5"
    test "test7521" (lazy(sprintf "%+0*.*G" 5 4 5.0)) "+0005"// "000+5"
    test "test7522" (lazy(sprintf "%-+0G" 5.0)) "+5"
    test "test7523" (lazy(sprintf "%-+05G" 5.0)) "+5   "//"+5000"
    test "test7524" (lazy(sprintf "%-+01G" 5.0)) "+5"
    test "test7525" (lazy(sprintf "%-+0*G" 7 5.0)) "+5     "// "+500000"
    test "test7526" (lazy(sprintf "%-+0.5G" 5.0)) "+5"
    test "test7527" (lazy(sprintf "%-+0.*G" 4 5.0)) "+5"
    test "test7528" (lazy(sprintf "%-+0*.*G" 5 4 5.0)) "+5   "// "+5000"
    test "test7529" (lazy(sprintf "% G" 5.0)) " 5"
    test "test7530" (lazy(sprintf "% 5G" 5.0)) "    5"
    test "test7531" (lazy(sprintf "% 1G" 5.0)) " 5"
    test "test7532" (lazy(sprintf "% *G" 7 5.0)) "      5"
    test "test7533" (lazy(sprintf "% .5G" 5.0)) " 5"
    test "test7534" (lazy(sprintf "% .*G" 4 5.0)) " 5"
    test "test7535" (lazy(sprintf "% *.*G" 5 4 5.0)) "    5"
    test "test7536" (lazy(sprintf "%- G" 5.0)) " 5"
    test "test7537" (lazy(sprintf "%- 5G" 5.0)) " 5   "
    test "test7538" (lazy(sprintf "%- 1G" 5.0)) " 5"
    test "test7539" (lazy(sprintf "%- *G" 7 5.0)) " 5     "
    test "test7540" (lazy(sprintf "%- .5G" 5.0)) " 5"
    test "test7541" (lazy(sprintf "%- .*G" 4 5.0)) " 5"
    test "test7542" (lazy(sprintf "%- *.*G" 5 4 5.0)) " 5   "
    test "test7543" (lazy(sprintf "% 0G" 5.0)) " 5"
    test "test7544" (lazy(sprintf "% 05G" 5.0)) " 0005"// "000 5"
    test "test7545" (lazy(sprintf "% 01G" 5.0)) " 5"
    test "test7546" (lazy(sprintf "% 0*G" 7 5.0)) " 000005"// "00000 5"
    test "test7547" (lazy(sprintf "% 0.5G" 5.0)) " 5"
    test "test7548" (lazy(sprintf "% 0.*G" 4 5.0)) " 5"
    test "test7549" (lazy(sprintf "% 0*.*G" 5 4 5.0)) " 0005"//  "000 5"
    test "test7550" (lazy(sprintf "%- 0G" 5.0)) " 5"
    test "test7551" (lazy(sprintf "%- 05G" 5.0)) " 5   "// " 5000"
    test "test7552" (lazy(sprintf "%- 01G" 5.0)) " 5"
    test "test7553" (lazy(sprintf "%- 0*G" 7 5.0)) " 5     "// " 500000"
    test "test7554" (lazy(sprintf "%- 0.5G" 5.0)) " 5"
    test "test7555" (lazy(sprintf "%- 0.*G" 4 5.0)) " 5"
    test "test7556" (lazy(sprintf "%- 0*.*G" 5 4 5.0)) " 5   "//" 5000"
    test "test7557" (lazy(sprintf "%G" -10.0)) "-10"
    test "test7558" (lazy(sprintf "%5G" -10.0)) "  -10"
    test "test7559" (lazy(sprintf "%1G" -10.0)) "-10"
    test "test7560" (lazy(sprintf "%*G" 7 -10.0)) "    -10"
    test "test7561" (lazy(sprintf "%.5G" -10.0)) "-10"
    test "test7562" (lazy(sprintf "%.*G" 4 -10.0)) "-10"
    test "test7563" (lazy(sprintf "%*.*G" 5 4 -10.0)) "  -10"
    test "test7564" (lazy(sprintf "%-G" -10.0)) "-10"
    test "test7565" (lazy(sprintf "%-5G" -10.0)) "-10  "
    test "test7566" (lazy(sprintf "%-1G" -10.0)) "-10"
    test "test7567" (lazy(sprintf "%-*G" 7 -10.0)) "-10    "
    test "test7568" (lazy(sprintf "%-.5G" -10.0)) "-10"
    test "test7569" (lazy(sprintf "%-.*G" 4 -10.0)) "-10"
    test "test7570" (lazy(sprintf "%-*.*G" 5 4 -10.0)) "-10  "
    test "test7571" (lazy(sprintf "%0G" -10.0)) "-10"
    test "test7572" (lazy(sprintf "%05G" -10.0)) "-0010"//  "00-10"
    test "test7573" (lazy(sprintf "%01G" -10.0)) "-10"
    test "test7574" (lazy(sprintf "%0*G" 7 -10.0)) "-000010"//  "0000-10"
    test "test7575" (lazy(sprintf "%0.5G" -10.0)) "-10"
    test "test7576" (lazy(sprintf "%0.*G" 4 -10.0)) "-10"
    test "test7577" (lazy(sprintf "%0*.*G" 5 4 -10.0)) "-0010"//  "00-10"
    test "test7578" (lazy(sprintf "%-0G" -10.0)) "-10"
    test "test7579" (lazy(sprintf "%-05G" -10.0)) "-10  "// "-1000"
    test "test7580" (lazy(sprintf "%-01G" -10.0)) "-10"
    test "test7581" (lazy(sprintf "%-0*G" 7 -10.0)) "-10    "// "-100000"
    test "test7582" (lazy(sprintf "%-0.5G" -10.0)) "-10"
    test "test7583" (lazy(sprintf "%-0.*G" 4 -10.0)) "-10"
    test "test7584" (lazy(sprintf "%-0*.*G" 5 4 -10.0)) "-10  "// "-1000"
    test "test7585" (lazy(sprintf "%+G" -10.0)) "-10"
    test "test7586" (lazy(sprintf "%+5G" -10.0)) "  -10"
    test "test7587" (lazy(sprintf "%+1G" -10.0)) "-10"
    test "test7588" (lazy(sprintf "%+*G" 7 -10.0)) "    -10"
    test "test7589" (lazy(sprintf "%+.5G" -10.0)) "-10"
    test "test7590" (lazy(sprintf "%+.*G" 4 -10.0)) "-10"
    test "test7591" (lazy(sprintf "%+*.*G" 5 4 -10.0)) "  -10"
    test "test7592" (lazy(sprintf "%-+G" -10.0)) "-10"
    test "test7593" (lazy(sprintf "%-+5G" -10.0)) "-10  "
    test "test7594" (lazy(sprintf "%-+1G" -10.0)) "-10"
    test "test7595" (lazy(sprintf "%-+*G" 7 -10.0)) "-10    "
    test "test7596" (lazy(sprintf "%-+.5G" -10.0)) "-10"
    test "test7597" (lazy(sprintf "%-+.*G" 4 -10.0)) "-10"
    test "test7598" (lazy(sprintf "%-+*.*G" 5 4 -10.0)) "-10  "
    test "test7599" (lazy(sprintf "%+0G" -10.0)) "-10"
    test "test7600" (lazy(sprintf "%+05G" -10.0)) "-0010"//"00-10"
    test "test7601" (lazy(sprintf "%+01G" -10.0)) "-10"
    test "test7602" (lazy(sprintf "%+0*G" 7 -10.0)) "-000010"// "0000-10"
    test "test7603" (lazy(sprintf "%+0.5G" -10.0)) "-10"
    test "test7604" (lazy(sprintf "%+0.*G" 4 -10.0)) "-10"
    test "test7605" (lazy(sprintf "%+0*.*G" 5 4 -10.0)) "-0010"//  "00-10"
    test "test7606" (lazy(sprintf "%-+0G" -10.0)) "-10"
    test "test7607" (lazy(sprintf "%-+05G" -10.0)) "-10  "// "-1000"
    test "test7608" (lazy(sprintf "%-+01G" -10.0)) "-10"
    test "test7609" (lazy(sprintf "%-+0*G" 7 -10.0)) "-10    "//  "-100000"
    test "test7610" (lazy(sprintf "%-+0.5G" -10.0)) "-10"
    test "test7611" (lazy(sprintf "%-+0.*G" 4 -10.0)) "-10"
    test "test7612" (lazy(sprintf "%-+0*.*G" 5 4 -10.0)) "-10  "// "-1000"
    test "test7613" (lazy(sprintf "% G" -10.0)) "-10"
    test "test7614" (lazy(sprintf "% 5G" -10.0)) "  -10"
    test "test7615" (lazy(sprintf "% 1G" -10.0)) "-10"
    test "test7616" (lazy(sprintf "% *G" 7 -10.0)) "    -10"
    test "test7617" (lazy(sprintf "% .5G" -10.0)) "-10"
    test "test7618" (lazy(sprintf "% .*G" 4 -10.0)) "-10"
    test "test7619" (lazy(sprintf "% *.*G" 5 4 -10.0)) "  -10"
    test "test7620" (lazy(sprintf "%- G" -10.0)) "-10"
    test "test7621" (lazy(sprintf "%- 5G" -10.0)) "-10  "
    test "test7622" (lazy(sprintf "%- 1G" -10.0)) "-10"
    test "test7623" (lazy(sprintf "%- *G" 7 -10.0)) "-10    "
    test "test7624" (lazy(sprintf "%- .5G" -10.0)) "-10"
    test "test7625" (lazy(sprintf "%- .*G" 4 -10.0)) "-10"
    test "test7626" (lazy(sprintf "%- *.*G" 5 4 -10.0)) "-10  "
    test "test7627" (lazy(sprintf "% 0G" -10.0)) "-10"
    test "test7628" (lazy(sprintf "% 05G" -10.0)) "-0010"// "00-10"
    test "test7629" (lazy(sprintf "% 01G" -10.0)) "-10"
    test "test7630" (lazy(sprintf "% 0*G" 7 -10.0)) "-000010"//  "0000-10"
    test "test7631" (lazy(sprintf "% 0.5G" -10.0)) "-10"
    test "test7632" (lazy(sprintf "% 0.*G" 4 -10.0)) "-10"
    test "test7633" (lazy(sprintf "% 0*.*G" 5 4 -10.0)) "-0010"//  "00-10"
    test "test7634" (lazy(sprintf "%- 0G" -10.0)) "-10"
    test "test7635" (lazy(sprintf "%- 05G" -10.0)) "-10  "//  "-1000"
    test "test7636" (lazy(sprintf "%- 01G" -10.0)) "-10"
    test "test7637" (lazy(sprintf "%- 0*G" 7 -10.0)) "-10    "// "-100000"
    test "test7638" (lazy(sprintf "%- 0.5G" -10.0)) "-10"
    test "test7639" (lazy(sprintf "%- 0.*G" 4 -10.0)) "-10"
    test "test7640" (lazy(sprintf "%- 0*.*G" 5 4 -10.0)) "-10  "// "-1000"
    test "test7641" (lazy(sprintf "%G" 5.0f)) "5"
    test "test7642" (lazy(sprintf "%5G" 5.0f)) "    5"
    test "test7643" (lazy(sprintf "%1G" 5.0f)) "5"
    test "test7644" (lazy(sprintf "%*G" 7 5.0f)) "      5"
    test "test7645" (lazy(sprintf "%.5G" 5.0f)) "5"
    test "test7646" (lazy(sprintf "%.*G" 4 5.0f)) "5"
    test "test7647" (lazy(sprintf "%*.*G" 5 4 5.0f)) "    5"
    test "test7648" (lazy(sprintf "%-G" 5.0f)) "5"
    test "test7649" (lazy(sprintf "%-5G" 5.0f)) "5    "
    test "test7650" (lazy(sprintf "%-1G" 5.0f)) "5"
    test "test7651" (lazy(sprintf "%-*G" 7 5.0f)) "5      "
    test "test7652" (lazy(sprintf "%-.5G" 5.0f)) "5"
    test "test7653" (lazy(sprintf "%-.*G" 4 5.0f)) "5"
    test "test7654" (lazy(sprintf "%-*.*G" 5 4 5.0f)) "5    "
    test "test7655" (lazy(sprintf "%0G" 5.0f)) "5"
    test "test7656" (lazy(sprintf "%05G" 5.0f)) "00005"
    test "test7657" (lazy(sprintf "%01G" 5.0f)) "5"
    test "test7658" (lazy(sprintf "%0*G" 7 5.0f)) "0000005"
    test "test7659" (lazy(sprintf "%0.5G" 5.0f)) "5"
    test "test7660" (lazy(sprintf "%0.*G" 4 5.0f)) "5"
    test "test7661" (lazy(sprintf "%0*.*G" 5 4 5.0f)) "00005"
    test "test7662" (lazy(sprintf "%-0G" 5.0f)) "5"
    test "test7663" (lazy(sprintf "%-05G" 5.0f)) "5    "// "50000"
    test "test7664" (lazy(sprintf "%-01G" 5.0f)) "5"
    test "test7665" (lazy(sprintf "%-0*G" 7 5.0f)) "5      "// "5000000"
    test "test7666" (lazy(sprintf "%-0.5G" 5.0f)) "5"
    test "test7667" (lazy(sprintf "%-0.*G" 4 5.0f)) "5"
    test "test7668" (lazy(sprintf "%-0*.*G" 5 4 5.0f)) "5    "// "50000"
    test "test7669" (lazy(sprintf "%+G" 5.0f)) "+5"
    test "test7670" (lazy(sprintf "%+5G" 5.0f)) "   +5"
    test "test7671" (lazy(sprintf "%+1G" 5.0f)) "+5"
    test "test7672" (lazy(sprintf "%+*G" 7 5.0f)) "     +5"
    test "test7673" (lazy(sprintf "%+.5G" 5.0f)) "+5"
    test "test7674" (lazy(sprintf "%+.*G" 4 5.0f)) "+5"
    test "test7675" (lazy(sprintf "%+*.*G" 5 4 5.0f)) "   +5"
    test "test7676" (lazy(sprintf "%-+G" 5.0f)) "+5"
    test "test7677" (lazy(sprintf "%-+5G" 5.0f)) "+5   "
    test "test7678" (lazy(sprintf "%-+1G" 5.0f)) "+5"
    test "test7679" (lazy(sprintf "%-+*G" 7 5.0f)) "+5     "
    test "test7680" (lazy(sprintf "%-+.5G" 5.0f)) "+5"
    test "test7681" (lazy(sprintf "%-+.*G" 4 5.0f)) "+5"
    test "test7682" (lazy(sprintf "%-+*.*G" 5 4 5.0f)) "+5   "
    test "test7683" (lazy(sprintf "%+0G" 5.0f)) "+5"
    test "test7684" (lazy(sprintf "%+05G" 5.0f)) "+0005"// "000+5"
    test "test7685" (lazy(sprintf "%+01G" 5.0f)) "+5"
    test "test7686" (lazy(sprintf "%+0*G" 7 5.0f)) "+000005"// "00000+5"
    test "test7687" (lazy(sprintf "%+0.5G" 5.0f)) "+5"
    test "test7688" (lazy(sprintf "%+0.*G" 4 5.0f)) "+5"
    test "test7689" (lazy(sprintf "%+0*.*G" 5 4 5.0f)) "+0005"// "000+5"
    test "test7690" (lazy(sprintf "%-+0G" 5.0f)) "+5"
    test "test7691" (lazy(sprintf "%-+05G" 5.0f)) "+5   "// "+5000"
    test "test7692" (lazy(sprintf "%-+01G" 5.0f)) "+5"
    test "test7693" (lazy(sprintf "%-+0*G" 7 5.0f)) "+5     "// "+500000"
    test "test7694" (lazy(sprintf "%-+0.5G" 5.0f)) "+5"
    test "test7695" (lazy(sprintf "%-+0.*G" 4 5.0f)) "+5"
    test "test7696" (lazy(sprintf "%-+0*.*G" 5 4 5.0f)) "+5   "//  "+5000"
    test "test7697" (lazy(sprintf "% G" 5.0f)) " 5"
    test "test7698" (lazy(sprintf "% 5G" 5.0f)) "    5"
    test "test7699" (lazy(sprintf "% 1G" 5.0f)) " 5"
    test "test7700" (lazy(sprintf "% *G" 7 5.0f)) "      5"
    test "test7701" (lazy(sprintf "% .5G" 5.0f)) " 5"
    test "test7702" (lazy(sprintf "% .*G" 4 5.0f)) " 5"
    test "test7703" (lazy(sprintf "% *.*G" 5 4 5.0f)) "    5"
    test "test7704" (lazy(sprintf "%- G" 5.0f)) " 5"
    test "test7705" (lazy(sprintf "%- 5G" 5.0f)) " 5   "
    test "test7706" (lazy(sprintf "%- 1G" 5.0f)) " 5"
    test "test7707" (lazy(sprintf "%- *G" 7 5.0f)) " 5     "
    test "test7708" (lazy(sprintf "%- .5G" 5.0f)) " 5"
    test "test7709" (lazy(sprintf "%- .*G" 4 5.0f)) " 5"
    test "test7710" (lazy(sprintf "%- *.*G" 5 4 5.0f)) " 5   "
    test "test7711" (lazy(sprintf "% 0G" 5.0f)) " 5"
    test "test7712" (lazy(sprintf "% 05G" 5.0f)) " 0005"//"000 5"
    test "test7713" (lazy(sprintf "% 01G" 5.0f)) " 5"
    test "test7714" (lazy(sprintf "% 0*G" 7 5.0f)) " 000005"// "00000 5"
    test "test7715" (lazy(sprintf "% 0.5G" 5.0f)) " 5"
    test "test7716" (lazy(sprintf "% 0.*G" 4 5.0f)) " 5"
    test "test7717" (lazy(sprintf "% 0*.*G" 5 4 5.0f)) " 0005"//  "000 5"
    test "test7718" (lazy(sprintf "%- 0G" 5.0f)) " 5"
    test "test7719" (lazy(sprintf "%- 05G" 5.0f)) " 5   "//  " 5000"
    test "test7720" (lazy(sprintf "%- 01G" 5.0f)) " 5"
    test "test7721" (lazy(sprintf "%- 0*G" 7 5.0f)) " 5     "// " 500000"
    test "test7722" (lazy(sprintf "%- 0.5G" 5.0f)) " 5"
    test "test7723" (lazy(sprintf "%- 0.*G" 4 5.0f)) " 5"
    test "test7724" (lazy(sprintf "%- 0*.*G" 5 4 5.0f)) " 5   "// " 5000"
    test "test7725" (lazy(sprintf "%G" -10.0f)) "-10"
    test "test7726" (lazy(sprintf "%5G" -10.0f)) "  -10"
    test "test7727" (lazy(sprintf "%1G" -10.0f)) "-10"
    test "test7728" (lazy(sprintf "%*G" 7 -10.0f)) "    -10"
    test "test7729" (lazy(sprintf "%.5G" -10.0f)) "-10"
    test "test7730" (lazy(sprintf "%.*G" 4 -10.0f)) "-10"
    test "test7731" (lazy(sprintf "%*.*G" 5 4 -10.0f)) "  -10"
    test "test7732" (lazy(sprintf "%-G" -10.0f)) "-10"
    test "test7733" (lazy(sprintf "%-5G" -10.0f)) "-10  "
    test "test7734" (lazy(sprintf "%-1G" -10.0f)) "-10"
    test "test7735" (lazy(sprintf "%-*G" 7 -10.0f)) "-10    "
    test "test7736" (lazy(sprintf "%-.5G" -10.0f)) "-10"
    test "test7737" (lazy(sprintf "%-.*G" 4 -10.0f)) "-10"
    test "test7738" (lazy(sprintf "%-*.*G" 5 4 -10.0f)) "-10  "
    test "test7739" (lazy(sprintf "%0G" -10.0f)) "-10"
    test "test7740" (lazy(sprintf "%05G" -10.0f)) "-0010"// "00-10"
    test "test7741" (lazy(sprintf "%01G" -10.0f)) "-10"
    test "test7742" (lazy(sprintf "%0*G" 7 -10.0f)) "-000010"// "0000-10"
    test "test7743" (lazy(sprintf "%0.5G" -10.0f)) "-10"
    test "test7744" (lazy(sprintf "%0.*G" 4 -10.0f)) "-10"
    test "test7745" (lazy(sprintf "%0*.*G" 5 4 -10.0f)) "-0010"// "00-10"
    test "test7746" (lazy(sprintf "%-0G" -10.0f)) "-10"
    test "test7747" (lazy(sprintf "%-05G" -10.0f)) "-10  "//"-1000"
    test "test7748" (lazy(sprintf "%-01G" -10.0f)) "-10"
    test "test7749" (lazy(sprintf "%-0*G" 7 -10.0f)) "-10    "// "-100000"
    test "test7750" (lazy(sprintf "%-0.5G" -10.0f)) "-10"
    test "test7751" (lazy(sprintf "%-0.*G" 4 -10.0f)) "-10"
    test "test7752" (lazy(sprintf "%-0*.*G" 5 4 -10.0f)) "-10  "// "-1000"
    test "test7753" (lazy(sprintf "%+G" -10.0f)) "-10"
    test "test7754" (lazy(sprintf "%+5G" -10.0f)) "  -10"
    test "test7755" (lazy(sprintf "%+1G" -10.0f)) "-10"
    test "test7756" (lazy(sprintf "%+*G" 7 -10.0f)) "    -10"
    test "test7757" (lazy(sprintf "%+.5G" -10.0f)) "-10"
    test "test7758" (lazy(sprintf "%+.*G" 4 -10.0f)) "-10"
    test "test7759" (lazy(sprintf "%+*.*G" 5 4 -10.0f)) "  -10"
    test "test7760" (lazy(sprintf "%-+G" -10.0f)) "-10"
    test "test7761" (lazy(sprintf "%-+5G" -10.0f)) "-10  "
    test "test7762" (lazy(sprintf "%-+1G" -10.0f)) "-10"
    test "test7763" (lazy(sprintf "%-+*G" 7 -10.0f)) "-10    "
    test "test7764" (lazy(sprintf "%-+.5G" -10.0f)) "-10"
    test "test7765" (lazy(sprintf "%-+.*G" 4 -10.0f)) "-10"
    test "test7766" (lazy(sprintf "%-+*.*G" 5 4 -10.0f)) "-10  "
    test "test7767" (lazy(sprintf "%+0G" -10.0f)) "-10"
    test "test7768" (lazy(sprintf "%+05G" -10.0f)) "-0010"// "00-10"
    test "test7769" (lazy(sprintf "%+01G" -10.0f)) "-10"
    test "test7770" (lazy(sprintf "%+0*G" 7 -10.0f)) "-000010"// "0000-10"
    test "test7771" (lazy(sprintf "%+0.5G" -10.0f)) "-10"
    test "test7772" (lazy(sprintf "%+0.*G" 4 -10.0f)) "-10"
    test "test7773" (lazy(sprintf "%+0*.*G" 5 4 -10.0f)) "-0010"// "00-10"
    test "test7774" (lazy(sprintf "%-+0G" -10.0f)) "-10"
    test "test7775" (lazy(sprintf "%-+05G" -10.0f)) "-10  "//"-1000"
    test "test7776" (lazy(sprintf "%-+01G" -10.0f)) "-10"
    test "test7777" (lazy(sprintf "%-+0*G" 7 -10.0f)) "-10    "//"-100000"
    test "test7778" (lazy(sprintf "%-+0.5G" -10.0f)) "-10"
    test "test7779" (lazy(sprintf "%-+0.*G" 4 -10.0f)) "-10"
    test "test7780" (lazy(sprintf "%-+0*.*G" 5 4 -10.0f)) "-10  "//  "-1000"
    test "test7781" (lazy(sprintf "% G" -10.0f)) "-10"
    test "test7782" (lazy(sprintf "% 5G" -10.0f)) "  -10"
    test "test7783" (lazy(sprintf "% 1G" -10.0f)) "-10"
    test "test7784" (lazy(sprintf "% *G" 7 -10.0f)) "    -10"
    test "test7785" (lazy(sprintf "% .5G" -10.0f)) "-10"
    test "test7786" (lazy(sprintf "% .*G" 4 -10.0f)) "-10"
    test "test7787" (lazy(sprintf "% *.*G" 5 4 -10.0f)) "  -10"
    test "test7788" (lazy(sprintf "%- G" -10.0f)) "-10"
    test "test7789" (lazy(sprintf "%- 5G" -10.0f)) "-10  "
    test "test7790" (lazy(sprintf "%- 1G" -10.0f)) "-10"
    test "test7791" (lazy(sprintf "%- *G" 7 -10.0f)) "-10    "
    test "test7792" (lazy(sprintf "%- .5G" -10.0f)) "-10"
    test "test7793" (lazy(sprintf "%- .*G" 4 -10.0f)) "-10"
    test "test7794" (lazy(sprintf "%- *.*G" 5 4 -10.0f)) "-10  "
    test "test7795" (lazy(sprintf "% 0G" -10.0f)) "-10"
    test "test7796" (lazy(sprintf "% 05G" -10.0f)) "-0010"// "00-10"
    test "test7797" (lazy(sprintf "% 01G" -10.0f)) "-10"
    test "test7798" (lazy(sprintf "% 0*G" 7 -10.0f)) "-000010"// "0000-10"
    test "test7799" (lazy(sprintf "% 0.5G" -10.0f)) "-10"
    test "test7800" (lazy(sprintf "% 0.*G" 4 -10.0f)) "-10"
    test "test7801" (lazy(sprintf "% 0*.*G" 5 4 -10.0f)) "-0010"// "00-10"
    test "test7802" (lazy(sprintf "%- 0G" -10.0f)) "-10"
    test "test7803" (lazy(sprintf "%- 05G" -10.0f)) "-10  "// "-1000"
    test "test7804" (lazy(sprintf "%- 01G" -10.0f)) "-10"
    test "test7805" (lazy(sprintf "%- 0*G" 7 -10.0f)) "-10    "// "-100000"
    test "test7806" (lazy(sprintf "%- 0.5G" -10.0f)) "-10"
    test "test7807" (lazy(sprintf "%- 0.*G" 4 -10.0f)) "-10"
    test "test7808" (lazy(sprintf "%- 0*.*G" 5 4 -10.0f)) "-10  "//  "-1000"
    test "test7809" (lazy(sprintf "%G" 5.0M)) "5"
    test "test7810" (lazy(sprintf "%5G" 5.0M)) "    5"
    test "test7811" (lazy(sprintf "%1G" 5.0M)) "5"
    test "test7812" (lazy(sprintf "%*G" 7 5.0M)) "      5"
    test "test7813" (lazy(sprintf "%.5G" 5.0M)) "5"
    test "test7814" (lazy(sprintf "%.*G" 4 5.0M)) "5"
    test "test7815" (lazy(sprintf "%*.*G" 5 4 5.0M)) "    5"
    test "test7816" (lazy(sprintf "%-G" 5.0M)) "5"
    test "test7817" (lazy(sprintf "%-5G" 5.0M)) "5    "
    test "test7818" (lazy(sprintf "%-1G" 5.0M)) "5"
    test "test7819" (lazy(sprintf "%-*G" 7 5.0M)) "5      "
    test "test7820" (lazy(sprintf "%-.5G" 5.0M)) "5"
    test "test7821" (lazy(sprintf "%-.*G" 4 5.0M)) "5"
    test "test7822" (lazy(sprintf "%-*.*G" 5 4 5.0M)) "5    "
    test "test7823" (lazy(sprintf "%0G" 5.0M)) "5"
    test "test7824" (lazy(sprintf "%05G" 5.0M)) "00005"
    test "test7825" (lazy(sprintf "%01G" 5.0M)) "5"
    test "test7826" (lazy(sprintf "%0*G" 7 5.0M)) "0000005"
    test "test7827" (lazy(sprintf "%0.5G" 5.0M)) "5"
    test "test7828" (lazy(sprintf "%0.*G" 4 5.0M)) "5"
    test "test7829" (lazy(sprintf "%0*.*G" 5 4 5.0M)) "00005"
    test "test7830" (lazy(sprintf "%-0G" 5.0M)) "5"
    test "test7831" (lazy(sprintf "%-05G" 5.0M)) "5    "// "50000"
    test "test7832" (lazy(sprintf "%-01G" 5.0M)) "5"
    test "test7833" (lazy(sprintf "%-0*G" 7 5.0M)) "5      "// "5000000"
    test "test7834" (lazy(sprintf "%-0.5G" 5.0M)) "5"
    test "test7835" (lazy(sprintf "%-0.*G" 4 5.0M)) "5"
    test "test7836" (lazy(sprintf "%-0*.*G" 5 4 5.0M)) "5    "// "50000"
    test "test7837" (lazy(sprintf "%+G" 5.0M)) "+5"
    test "test7838" (lazy(sprintf "%+5G" 5.0M)) "   +5"
    test "test7839" (lazy(sprintf "%+1G" 5.0M)) "+5"
    test "test7840" (lazy(sprintf "%+*G" 7 5.0M)) "     +5"
    test "test7841" (lazy(sprintf "%+.5G" 5.0M)) "+5"
    test "test7842" (lazy(sprintf "%+.*G" 4 5.0M)) "+5"
    test "test7843" (lazy(sprintf "%+*.*G" 5 4 5.0M)) "   +5"
    test "test7844" (lazy(sprintf "%-+G" 5.0M)) "+5"
    test "test7845" (lazy(sprintf "%-+5G" 5.0M)) "+5   "
    test "test7846" (lazy(sprintf "%-+1G" 5.0M)) "+5"
    test "test7847" (lazy(sprintf "%-+*G" 7 5.0M)) "+5     "
    test "test7848" (lazy(sprintf "%-+.5G" 5.0M)) "+5"
    test "test7849" (lazy(sprintf "%-+.*G" 4 5.0M)) "+5"
    test "test7850" (lazy(sprintf "%-+*.*G" 5 4 5.0M)) "+5   "
    test "test7851" (lazy(sprintf "%+0G" 5.0M)) "+5"
    test "test7852" (lazy(sprintf "%+05G" 5.0M)) "+0005"// "000+5"
    test "test7853" (lazy(sprintf "%+01G" 5.0M)) "+5"
    test "test7854" (lazy(sprintf "%+0*G" 7 5.0M)) "+000005"// "00000+5"
    test "test7855" (lazy(sprintf "%+0.5G" 5.0M)) "+5"
    test "test7856" (lazy(sprintf "%+0.*G" 4 5.0M)) "+5"
    test "test7857" (lazy(sprintf "%+0*.*G" 5 4 5.0M)) "+0005"// "000+5"
    test "test7858" (lazy(sprintf "%-+0G" 5.0M)) "+5"
    test "test7859" (lazy(sprintf "%-+05G" 5.0M)) "+5   "// "+5000"
    test "test7860" (lazy(sprintf "%-+01G" 5.0M)) "+5"
    test "test7861" (lazy(sprintf "%-+0*G" 7 5.0M)) "+5     "// "+500000"
    test "test7862" (lazy(sprintf "%-+0.5G" 5.0M)) "+5"
    test "test7863" (lazy(sprintf "%-+0.*G" 4 5.0M)) "+5"
    test "test7864" (lazy(sprintf "%-+0*.*G" 5 4 5.0M)) "+5   "//  "+5000"
    test "test7865" (lazy(sprintf "% G" 5.0M)) " 5"
    test "test7866" (lazy(sprintf "% 5G" 5.0M)) "    5"
    test "test7867" (lazy(sprintf "% 1G" 5.0M)) " 5"
    test "test7868" (lazy(sprintf "% *G" 7 5.0M)) "      5"
    test "test7869" (lazy(sprintf "% .5G" 5.0M)) " 5"
    test "test7870" (lazy(sprintf "% .*G" 4 5.0M)) " 5"
    test "test7871" (lazy(sprintf "% *.*G" 5 4 5.0M)) "    5"
    test "test7872" (lazy(sprintf "%- G" 5.0M)) " 5"
    test "test7873" (lazy(sprintf "%- 5G" 5.0M)) " 5   "
    test "test7874" (lazy(sprintf "%- 1G" 5.0M)) " 5"
    test "test7875" (lazy(sprintf "%- *G" 7 5.0M)) " 5     "
    test "test7876" (lazy(sprintf "%- .5G" 5.0M)) " 5"
    test "test7877" (lazy(sprintf "%- .*G" 4 5.0M)) " 5"
    test "test7878" (lazy(sprintf "%- *.*G" 5 4 5.0M)) " 5   "
    test "test7879" (lazy(sprintf "% 0G" 5.0M)) " 5"
    test "test7880" (lazy(sprintf "% 05G" 5.0M)) " 0005"// "000 5"
    test "test7881" (lazy(sprintf "% 01G" 5.0M)) " 5"
    test "test7882" (lazy(sprintf "% 0*G" 7 5.0M)) " 000005"// "00000 5"
    test "test7883" (lazy(sprintf "% 0.5G" 5.0M)) " 5"
    test "test7884" (lazy(sprintf "% 0.*G" 4 5.0M)) " 5"
    test "test7885" (lazy(sprintf "% 0*.*G" 5 4 5.0M)) " 0005"// "000 5"
    test "test7886" (lazy(sprintf "%- 0G" 5.0M)) " 5"
    test "test7887" (lazy(sprintf "%- 05G" 5.0M)) " 5   "// " 5000"
    test "test7888" (lazy(sprintf "%- 01G" 5.0M)) " 5"
    test "test7889" (lazy(sprintf "%- 0*G" 7 5.0M)) " 5     "// " 500000"
    test "test7890" (lazy(sprintf "%- 0.5G" 5.0M)) " 5"
    test "test7891" (lazy(sprintf "%- 0.*G" 4 5.0M)) " 5"
    test "test7892" (lazy(sprintf "%- 0*.*G" 5 4 5.0M)) " 5   "// " 5000"
    test "test7893" (lazy(sprintf "%G" -10.0M)) "-10"
    test "test7894" (lazy(sprintf "%5G" -10.0M)) "  -10"
    test "test7895" (lazy(sprintf "%1G" -10.0M)) "-10"
    test "test7896" (lazy(sprintf "%*G" 7 -10.0M)) "    -10"
    test "test7897" (lazy(sprintf "%.5G" -10.0M)) "-10"
    test "test7898" (lazy(sprintf "%.*G" 4 -10.0M)) "-10"
    test "test7899" (lazy(sprintf "%*.*G" 5 4 -10.0M)) "  -10"
    test "test7900" (lazy(sprintf "%-G" -10.0M)) "-10"
    test "test7901" (lazy(sprintf "%-5G" -10.0M)) "-10  "
    test "test7902" (lazy(sprintf "%-1G" -10.0M)) "-10"
    test "test7903" (lazy(sprintf "%-*G" 7 -10.0M)) "-10    "
    test "test7904" (lazy(sprintf "%-.5G" -10.0M)) "-10"
    test "test7905" (lazy(sprintf "%-.*G" 4 -10.0M)) "-10"
    test "test7906" (lazy(sprintf "%-*.*G" 5 4 -10.0M)) "-10  "
    test "test7907" (lazy(sprintf "%0G" -10.0M)) "-10"
    test "test7908" (lazy(sprintf "%05G" -10.0M)) "-0010"// "00-10"
    test "test7909" (lazy(sprintf "%01G" -10.0M)) "-10"
    test "test7910" (lazy(sprintf "%0*G" 7 -10.0M)) "-000010"// "0000-10"
    test "test7911" (lazy(sprintf "%0.5G" -10.0M)) "-10"
    test "test7912" (lazy(sprintf "%0.*G" 4 -10.0M)) "-10"
    test "test7913" (lazy(sprintf "%0*.*G" 5 4 -10.0M)) "-0010"// "00-10"
    test "test7914" (lazy(sprintf "%-0G" -10.0M)) "-10"
    test "test7915" (lazy(sprintf "%-05G" -10.0M)) "-10  "// "-1000"
    test "test7916" (lazy(sprintf "%-01G" -10.0M)) "-10"
    test "test7917" (lazy(sprintf "%-0*G" 7 -10.0M)) "-10    "// "-100000"
    test "test7918" (lazy(sprintf "%-0.5G" -10.0M)) "-10"
    test "test7919" (lazy(sprintf "%-0.*G" 4 -10.0M)) "-10"
    test "test7920" (lazy(sprintf "%-0*.*G" 5 4 -10.0M)) "-10  "// "-1000"
    test "test7921" (lazy(sprintf "%+G" -10.0M)) "-10"
    test "test7922" (lazy(sprintf "%+5G" -10.0M)) "  -10"
    test "test7923" (lazy(sprintf "%+1G" -10.0M)) "-10"
    test "test7924" (lazy(sprintf "%+*G" 7 -10.0M)) "    -10"
    test "test7925" (lazy(sprintf "%+.5G" -10.0M)) "-10"
    test "test7926" (lazy(sprintf "%+.*G" 4 -10.0M)) "-10"
    test "test7927" (lazy(sprintf "%+*.*G" 5 4 -10.0M)) "  -10"
    test "test7928" (lazy(sprintf "%-+G" -10.0M)) "-10"
    test "test7929" (lazy(sprintf "%-+5G" -10.0M)) "-10  "
    test "test7930" (lazy(sprintf "%-+1G" -10.0M)) "-10"
    test "test7931" (lazy(sprintf "%-+*G" 7 -10.0M)) "-10    "
    test "test7932" (lazy(sprintf "%-+.5G" -10.0M)) "-10"
    test "test7933" (lazy(sprintf "%-+.*G" 4 -10.0M)) "-10"
    test "test7934" (lazy(sprintf "%-+*.*G" 5 4 -10.0M)) "-10  "
    test "test7935" (lazy(sprintf "%+0G" -10.0M)) "-10"
    test "test7936" (lazy(sprintf "%+05G" -10.0M)) "-0010"//  "00-10"
    test "test7937" (lazy(sprintf "%+01G" -10.0M)) "-10"
    test "test7938" (lazy(sprintf "%+0*G" 7 -10.0M)) "-000010"/// "0000-10"
    test "test7939" (lazy(sprintf "%+0.5G" -10.0M)) "-10"
    test "test7940" (lazy(sprintf "%+0.*G" 4 -10.0M)) "-10"
    test "test7941" (lazy(sprintf "%+0*.*G" 5 4 -10.0M)) "-0010"// "00-10"
    test "test7942" (lazy(sprintf "%-+0G" -10.0M)) "-10"
    test "test7943" (lazy(sprintf "%-+05G" -10.0M)) "-10  "// "-1000"
    test "test7944" (lazy(sprintf "%-+01G" -10.0M)) "-10"
    test "test7945" (lazy(sprintf "%-+0*G" 7 -10.0M)) "-10    "// "-100000"
    test "test7946" (lazy(sprintf "%-+0.5G" -10.0M)) "-10"
    test "test7947" (lazy(sprintf "%-+0.*G" 4 -10.0M)) "-10"
    test "test7948" (lazy(sprintf "%-+0*.*G" 5 4 -10.0M)) "-10  "// "-1000"
    test "test7949" (lazy(sprintf "% G" -10.0M)) "-10"
    test "test7950" (lazy(sprintf "% 5G" -10.0M)) "  -10"
    test "test7951" (lazy(sprintf "% 1G" -10.0M)) "-10"
    test "test7952" (lazy(sprintf "% *G" 7 -10.0M)) "    -10"
    test "test7953" (lazy(sprintf "% .5G" -10.0M)) "-10"
    test "test7954" (lazy(sprintf "% .*G" 4 -10.0M)) "-10"
    test "test7955" (lazy(sprintf "% *.*G" 5 4 -10.0M)) "  -10"
    test "test7956" (lazy(sprintf "%- G" -10.0M)) "-10"
    test "test7957" (lazy(sprintf "%- 5G" -10.0M)) "-10  "
    test "test7958" (lazy(sprintf "%- 1G" -10.0M)) "-10"
    test "test7959" (lazy(sprintf "%- *G" 7 -10.0M)) "-10    "
    test "test7960" (lazy(sprintf "%- .5G" -10.0M)) "-10"
    test "test7961" (lazy(sprintf "%- .*G" 4 -10.0M)) "-10"
    test "test7962" (lazy(sprintf "%- *.*G" 5 4 -10.0M)) "-10  "
    test "test7963" (lazy(sprintf "% 0G" -10.0M)) "-10"
    test "test7964" (lazy(sprintf "% 05G" -10.0M)) "-0010"// "00-10"
    test "test7965" (lazy(sprintf "% 01G" -10.0M)) "-10"
    test "test7966" (lazy(sprintf "% 0*G" 7 -10.0M)) "-000010"//  "0000-10"
    test "test7967" (lazy(sprintf "% 0.5G" -10.0M)) "-10"
    test "test7968" (lazy(sprintf "% 0.*G" 4 -10.0M)) "-10"
    test "test7969" (lazy(sprintf "% 0*.*G" 5 4 -10.0M)) "-0010"// "00-10"
    test "test7970" (lazy(sprintf "%- 0G" -10.0M)) "-10"
    test "test7971" (lazy(sprintf "%- 05G" -10.0M)) "-10  "// "-1000"
    test "test7972" (lazy(sprintf "%- 01G" -10.0M)) "-10"
    test "test7973" (lazy(sprintf "%- 0*G" 7 -10.0M)) "-10    "// "-100000"
    test "test7974" (lazy(sprintf "%- 0.5G" -10.0M)) "-10"
    test "test7975" (lazy(sprintf "%- 0.*G" 4 -10.0M)) "-10"
    test "test7976" (lazy(sprintf "%- 0*.*G" 5 4 -10.0M)) "-10  "// "-1000"
    test "test7977" (lazy(sprintf "%M" 10M)) "10"
    test "test7978" (lazy(sprintf "%5M" 10M)) "   10"
    test "test7979" (lazy(sprintf "%1M" 10M)) "10"
    test "test7980" (lazy(sprintf "%*M" 7 10M)) "     10"
    test "test7981" (lazy(sprintf "%.5M" 10M)) "10"
    test "test7982" (lazy(sprintf "%.*M" 4 10M)) "10"
    test "test7983" (lazy(sprintf "%*.*M" 5 4 10M)) "   10"
    test "test7984" (lazy(sprintf "%-M" 10M)) "10"
    test "test7985" (lazy(sprintf "%-5M" 10M)) "10   "
    test "test7986" (lazy(sprintf "%-1M" 10M)) "10"
    test "test7987" (lazy(sprintf "%-*M" 7 10M)) "10     "
    test "test7988" (lazy(sprintf "%-.5M" 10M)) "10"
    test "test7989" (lazy(sprintf "%-.*M" 4 10M)) "10"
    test "test7990" (lazy(sprintf "%-*.*M" 5 4 10M)) "10   "
    test "test7991" (lazy(sprintf "%0M" 10M)) "10"
    test "test7992" (lazy(sprintf "%05M" 10M)) "00010"
    test "test7993" (lazy(sprintf "%01M" 10M)) "10"
    test "test7994" (lazy(sprintf "%0*M" 7 10M)) "0000010"
    test "test7995" (lazy(sprintf "%0.5M" 10M)) "10"
    test "test7996" (lazy(sprintf "%0.*M" 4 10M)) "10"
    test "test7997" (lazy(sprintf "%0*.*M" 5 4 10M)) "00010"
    test "test7998" (lazy(sprintf "%-0M" 10M)) "10"
    test "test7999" (lazy(sprintf "%-05M" 10M)) "10   "//"10000"
    test "test8000" (lazy(sprintf "%-01M" 10M)) "10"
let func8000()=
    test "test8001" (lazy(sprintf "%-0*M" 7 10M)) "10     "//"1000000"
    test "test8002" (lazy(sprintf "%-0.5M" 10M)) "10"
    test "test8003" (lazy(sprintf "%-0.*M" 4 10M)) "10"
    test "test8004" (lazy(sprintf "%-0*.*M" 5 4 10M)) "10   "//"10000"
    test "test8005" (lazy(sprintf "%+M" 10M)) "+10"
    test "test8006" (lazy(sprintf "%+5M" 10M)) "  +10"
    test "test8007" (lazy(sprintf "%+1M" 10M)) "+10"
    test "test8008" (lazy(sprintf "%+*M" 7 10M)) "    +10"
    test "test8009" (lazy(sprintf "%+.5M" 10M)) "+10"
    test "test8010" (lazy(sprintf "%+.*M" 4 10M)) "+10"
    test "test8011" (lazy(sprintf "%+*.*M" 5 4 10M)) "  +10"
    test "test8012" (lazy(sprintf "%-+M" 10M)) "+10"
    test "test8013" (lazy(sprintf "%-+5M" 10M)) "+10  "
    test "test8014" (lazy(sprintf "%-+1M" 10M)) "+10"
    test "test8015" (lazy(sprintf "%-+*M" 7 10M)) "+10    "
    test "test8016" (lazy(sprintf "%-+.5M" 10M)) "+10"
    test "test8017" (lazy(sprintf "%-+.*M" 4 10M)) "+10"
    test "test8018" (lazy(sprintf "%-+*.*M" 5 4 10M)) "+10  "
    test "test8019" (lazy(sprintf "%+0M" 10M)) "+10"
    test "test8020" (lazy(sprintf "%+05M" 10M)) "+0010"// "00+10"
    test "test8021" (lazy(sprintf "%+01M" 10M)) "+10"
    test "test8022" (lazy(sprintf "%+0*M" 7 10M)) "+000010"// "0000+10"
    test "test8023" (lazy(sprintf "%+0.5M" 10M)) "+10"
    test "test8024" (lazy(sprintf "%+0.*M" 4 10M)) "+10"
    test "test8025" (lazy(sprintf "%+0*.*M" 5 4 10M)) "+0010"// "00+10"
    test "test8026" (lazy(sprintf "%-+0M" 10M)) "+10"
    test "test8027" (lazy(sprintf "%-+05M" 10M)) "+10  "//"+1000"
    test "test8028" (lazy(sprintf "%-+01M" 10M)) "+10"
    test "test8029" (lazy(sprintf "%-+0*M" 7 10M)) "+10    "// "+100000"
    test "test8030" (lazy(sprintf "%-+0.5M" 10M)) "+10"
    test "test8031" (lazy(sprintf "%-+0.*M" 4 10M)) "+10"
    test "test8032" (lazy(sprintf "%-+0*.*M" 5 4 10M)) "+10  "//"+1000"
    test "test8033" (lazy(sprintf "% M" 10M)) " 10"
    test "test8034" (lazy(sprintf "% 5M" 10M)) "   10"
    test "test8035" (lazy(sprintf "% 1M" 10M)) " 10"
    test "test8036" (lazy(sprintf "% *M" 7 10M)) "     10"
    test "test8037" (lazy(sprintf "% .5M" 10M)) " 10"
    test "test8038" (lazy(sprintf "% .*M" 4 10M)) " 10"
    test "test8039" (lazy(sprintf "% *.*M" 5 4 10M)) "   10"
    test "test8040" (lazy(sprintf "%- M" 10M)) " 10"
    test "test8041" (lazy(sprintf "%- 5M" 10M)) " 10  "
    test "test8042" (lazy(sprintf "%- 1M" 10M)) " 10"
    test "test8043" (lazy(sprintf "%- *M" 7 10M)) " 10    "
    test "test8044" (lazy(sprintf "%- .5M" 10M)) " 10"
    test "test8045" (lazy(sprintf "%- .*M" 4 10M)) " 10"
    test "test8046" (lazy(sprintf "%- *.*M" 5 4 10M)) " 10  "
    test "test8047" (lazy(sprintf "% 0M" 10M)) " 10"
    test "test8048" (lazy(sprintf "% 05M" 10M)) " 0010"// "00 10"
    test "test8049" (lazy(sprintf "% 01M" 10M)) " 10"
    test "test8050" (lazy(sprintf "% 0*M" 7 10M)) " 000010"//  "0000 10"
    test "test8051" (lazy(sprintf "% 0.5M" 10M)) " 10"
    test "test8052" (lazy(sprintf "% 0.*M" 4 10M)) " 10"
    test "test8053" (lazy(sprintf "% 0*.*M" 5 4 10M)) " 0010"// "00 10"
    test "test8054" (lazy(sprintf "%- 0M" 10M)) " 10"
    test "test8055" (lazy(sprintf "%- 05M" 10M)) " 10  "// " 1000"
    test "test8056" (lazy(sprintf "%- 01M" 10M)) " 10"
    test "test8057" (lazy(sprintf "%- 0*M" 7 10M)) " 10    "//" 100000"
    test "test8058" (lazy(sprintf "%- 0.5M" 10M)) " 10"
    test "test8059" (lazy(sprintf "%- 0.*M" 4 10M)) " 10"
    test "test8060" (lazy(sprintf "%- 0*.*M" 5 4 10M)) " 10  "//" 1000"
    test "test8061" (lazy(sprintf "%M" 1.3M)) "1.3"
    test "test8062" (lazy(sprintf "%5M" 1.3M)) "  1.3"
    test "test8063" (lazy(sprintf "%1M" 1.3M)) "1.3"
    test "test8064" (lazy(sprintf "%*M" 7 1.3M)) "    1.3"
    test "test8065" (lazy(sprintf "%.5M" 1.3M)) "1.3"
    test "test8066" (lazy(sprintf "%.*M" 4 1.3M)) "1.3"
    test "test8067" (lazy(sprintf "%*.*M" 5 4 1.3M)) "  1.3"
    test "test8068" (lazy(sprintf "%-M" 1.3M)) "1.3"
    test "test8069" (lazy(sprintf "%-5M" 1.3M)) "1.3  "
    test "test8070" (lazy(sprintf "%-1M" 1.3M)) "1.3"
    test "test8071" (lazy(sprintf "%-*M" 7 1.3M)) "1.3    "
    test "test8072" (lazy(sprintf "%-.5M" 1.3M)) "1.3"
    test "test8073" (lazy(sprintf "%-.*M" 4 1.3M)) "1.3"
    test "test8074" (lazy(sprintf "%-*.*M" 5 4 1.3M)) "1.3  "
    test "test8075" (lazy(sprintf "%0M" 1.3M)) "1.3"
    test "test8076" (lazy(sprintf "%05M" 1.3M)) "001.3"
    test "test8077" (lazy(sprintf "%01M" 1.3M)) "1.3"
    test "test8078" (lazy(sprintf "%0*M" 7 1.3M)) "00001.3"
    test "test8079" (lazy(sprintf "%0.5M" 1.3M)) "1.3"
    test "test8080" (lazy(sprintf "%0.*M" 4 1.3M)) "1.3"
    test "test8081" (lazy(sprintf "%0*.*M" 5 4 1.3M)) "001.3"
    test "test8082" (lazy(sprintf "%-0M" 1.3M)) "1.3"
    test "test8083" (lazy(sprintf "%-05M" 1.3M)) "1.300"
    test "test8084" (lazy(sprintf "%-01M" 1.3M)) "1.3"
    test "test8085" (lazy(sprintf "%-0*M" 7 1.3M)) "1.30000"
    test "test8086" (lazy(sprintf "%-0.5M" 1.3M)) "1.3"
    test "test8087" (lazy(sprintf "%-0.*M" 4 1.3M)) "1.3"
    test "test8088" (lazy(sprintf "%-0*.*M" 5 4 1.3M)) "1.300"
    test "test8089" (lazy(sprintf "%+M" 1.3M)) "+1.3"
    test "test8090" (lazy(sprintf "%+5M" 1.3M)) " +1.3"
    test "test8091" (lazy(sprintf "%+1M" 1.3M)) "+1.3"
    test "test8092" (lazy(sprintf "%+*M" 7 1.3M)) "   +1.3"
    test "test8093" (lazy(sprintf "%+.5M" 1.3M)) "+1.3"
    test "test8094" (lazy(sprintf "%+.*M" 4 1.3M)) "+1.3"
    test "test8095" (lazy(sprintf "%+*.*M" 5 4 1.3M)) " +1.3"
    test "test8096" (lazy(sprintf "%-+M" 1.3M)) "+1.3"
    test "test8097" (lazy(sprintf "%-+5M" 1.3M)) "+1.3 "
    test "test8098" (lazy(sprintf "%-+1M" 1.3M)) "+1.3"
    test "test8099" (lazy(sprintf "%-+*M" 7 1.3M)) "+1.3   "
    test "test8100" (lazy(sprintf "%-+.5M" 1.3M)) "+1.3"
    test "test8101" (lazy(sprintf "%-+.*M" 4 1.3M)) "+1.3"
    test "test8102" (lazy(sprintf "%-+*.*M" 5 4 1.3M)) "+1.3 "
    test "test8103" (lazy(sprintf "%+0M" 1.3M)) "+1.3"
    test "test8104" (lazy(sprintf "%+05M" 1.3M)) "+01.3"//"0+1.3"
    test "test8105" (lazy(sprintf "%+01M" 1.3M)) "+1.3"
    test "test8106" (lazy(sprintf "%+0*M" 7 1.3M)) "+0001.3"//"000+1.3"
    test "test8107" (lazy(sprintf "%+0.5M" 1.3M)) "+1.3"
    test "test8108" (lazy(sprintf "%+0.*M" 4 1.3M)) "+1.3"
    test "test8109" (lazy(sprintf "%+0*.*M" 5 4 1.3M)) "+01.3"//"0+1.3"
    test "test8110" (lazy(sprintf "%-+0M" 1.3M)) "+1.3"
    test "test8111" (lazy(sprintf "%-+05M" 1.3M)) "+1.30"
    test "test8112" (lazy(sprintf "%-+01M" 1.3M)) "+1.3"
    test "test8113" (lazy(sprintf "%-+0*M" 7 1.3M)) "+1.3000"
    test "test8114" (lazy(sprintf "%-+0.5M" 1.3M)) "+1.3"
    test "test8115" (lazy(sprintf "%-+0.*M" 4 1.3M)) "+1.3"
    test "test8116" (lazy(sprintf "%-+0*.*M" 5 4 1.3M)) "+1.30"
    test "test8117" (lazy(sprintf "% M" 1.3M)) " 1.3"
    test "test8118" (lazy(sprintf "% 5M" 1.3M)) "  1.3"
    test "test8119" (lazy(sprintf "% 1M" 1.3M)) " 1.3"
    test "test8120" (lazy(sprintf "% *M" 7 1.3M)) "    1.3"
    test "test8121" (lazy(sprintf "% .5M" 1.3M)) " 1.3"
    test "test8122" (lazy(sprintf "% .*M" 4 1.3M)) " 1.3"
    test "test8123" (lazy(sprintf "% *.*M" 5 4 1.3M)) "  1.3"
    test "test8124" (lazy(sprintf "%- M" 1.3M)) " 1.3"
    test "test8125" (lazy(sprintf "%- 5M" 1.3M)) " 1.3 "
    test "test8126" (lazy(sprintf "%- 1M" 1.3M)) " 1.3"
    test "test8127" (lazy(sprintf "%- *M" 7 1.3M)) " 1.3   "
    test "test8128" (lazy(sprintf "%- .5M" 1.3M)) " 1.3"
    test "test8129" (lazy(sprintf "%- .*M" 4 1.3M)) " 1.3"
    test "test8130" (lazy(sprintf "%- *.*M" 5 4 1.3M)) " 1.3 "
    test "test8131" (lazy(sprintf "% 0M" 1.3M)) " 1.3"
    test "test8132" (lazy(sprintf "% 05M" 1.3M)) " 01.3"//"0 1.3"
    test "test8133" (lazy(sprintf "% 01M" 1.3M)) " 1.3"
    test "test8134" (lazy(sprintf "% 0*M" 7 1.3M)) " 0001.3"// "000 1.3"
    test "test8135" (lazy(sprintf "% 0.5M" 1.3M)) " 1.3"
    test "test8136" (lazy(sprintf "% 0.*M" 4 1.3M)) " 1.3"
    test "test8137" (lazy(sprintf "% 0*.*M" 5 4 1.3M)) " 01.3"// "0 1.3"
    test "test8138" (lazy(sprintf "%- 0M" 1.3M)) " 1.3"
    test "test8139" (lazy(sprintf "%- 05M" 1.3M)) " 1.30"
    test "test8140" (lazy(sprintf "%- 01M" 1.3M)) " 1.3"
    test "test8141" (lazy(sprintf "%- 0*M" 7 1.3M)) " 1.3000"
    test "test8142" (lazy(sprintf "%- 0.5M" 1.3M)) " 1.3"
    test "test8143" (lazy(sprintf "%- 0.*M" 4 1.3M)) " 1.3"
    test "test8144" (lazy(sprintf "%- 0*.*M" 5 4 1.3M)) " 1.30"
    test "test8145" (lazy(sprintf "%M" -15.5M)) "-15.5"
    test "test8146" (lazy(sprintf "%5M" -15.5M)) "-15.5"
    test "test8147" (lazy(sprintf "%1M" -15.5M)) "-15.5"
    test "test8148" (lazy(sprintf "%*M" 7 -15.5M)) "  -15.5"
    test "test8149" (lazy(sprintf "%.5M" -15.5M)) "-15.5"
    test "test8150" (lazy(sprintf "%.*M" 4 -15.5M)) "-15.5"
    test "test8151" (lazy(sprintf "%*.*M" 5 4 -15.5M)) "-15.5"
    test "test8152" (lazy(sprintf "%-M" -15.5M)) "-15.5"
    test "test8153" (lazy(sprintf "%-5M" -15.5M)) "-15.5"
    test "test8154" (lazy(sprintf "%-1M" -15.5M)) "-15.5"
    test "test8155" (lazy(sprintf "%-*M" 7 -15.5M)) "-15.5  "
    test "test8156" (lazy(sprintf "%-.5M" -15.5M)) "-15.5"
    test "test8157" (lazy(sprintf "%-.*M" 4 -15.5M)) "-15.5"
    test "test8158" (lazy(sprintf "%-*.*M" 5 4 -15.5M)) "-15.5"
    test "test8159" (lazy(sprintf "%0M" -15.5M)) "-15.5"
    test "test8160" (lazy(sprintf "%05M" -15.5M)) "-15.5"
    test "test8161" (lazy(sprintf "%01M" -15.5M)) "-15.5"
    test "test8162" (lazy(sprintf "%0*M" 7 -15.5M)) "-0015.5"// "00-15.5"
    test "test8163" (lazy(sprintf "%0.5M" -15.5M)) "-15.5"
    test "test8164" (lazy(sprintf "%0.*M" 4 -15.5M)) "-15.5"
    test "test8165" (lazy(sprintf "%0*.*M" 5 4 -15.5M)) "-15.5"
    test "test8166" (lazy(sprintf "%-0M" -15.5M)) "-15.5"
    test "test8167" (lazy(sprintf "%-05M" -15.5M)) "-15.5"
    test "test8168" (lazy(sprintf "%-01M" -15.5M)) "-15.5"
    test "test8169" (lazy(sprintf "%-0*M" 7 -15.5M)) "-15.500"
    test "test8170" (lazy(sprintf "%-0.5M" -15.5M)) "-15.5"
    test "test8171" (lazy(sprintf "%-0.*M" 4 -15.5M)) "-15.5"
    test "test8172" (lazy(sprintf "%-0*.*M" 5 4 -15.5M)) "-15.5"
    test "test8173" (lazy(sprintf "%+M" -15.5M)) "-15.5"
    test "test8174" (lazy(sprintf "%+5M" -15.5M)) "-15.5"
    test "test8175" (lazy(sprintf "%+1M" -15.5M)) "-15.5"
    test "test8176" (lazy(sprintf "%+*M" 7 -15.5M)) "  -15.5"
    test "test8177" (lazy(sprintf "%+.5M" -15.5M)) "-15.5"
    test "test8178" (lazy(sprintf "%+.*M" 4 -15.5M)) "-15.5"
    test "test8179" (lazy(sprintf "%+*.*M" 5 4 -15.5M)) "-15.5"
    test "test8180" (lazy(sprintf "%-+M" -15.5M)) "-15.5"
    test "test8181" (lazy(sprintf "%-+5M" -15.5M)) "-15.5"
    test "test8182" (lazy(sprintf "%-+1M" -15.5M)) "-15.5"
    test "test8183" (lazy(sprintf "%-+*M" 7 -15.5M)) "-15.5  "
    test "test8184" (lazy(sprintf "%-+.5M" -15.5M)) "-15.5"
    test "test8185" (lazy(sprintf "%-+.*M" 4 -15.5M)) "-15.5"
    test "test8186" (lazy(sprintf "%-+*.*M" 5 4 -15.5M)) "-15.5"
    test "test8187" (lazy(sprintf "%+0M" -15.5M)) "-15.5"
    test "test8188" (lazy(sprintf "%+05M" -15.5M)) "-15.5"
    test "test8189" (lazy(sprintf "%+01M" -15.5M)) "-15.5"
    test "test8190" (lazy(sprintf "%+0*M" 7 -15.5M)) "-0015.5"//"00-15.5"
    test "test8191" (lazy(sprintf "%+0.5M" -15.5M)) "-15.5"
    test "test8192" (lazy(sprintf "%+0.*M" 4 -15.5M)) "-15.5"
    test "test8193" (lazy(sprintf "%+0*.*M" 5 4 -15.5M)) "-15.5"
    test "test8194" (lazy(sprintf "%-+0M" -15.5M)) "-15.5"
    test "test8195" (lazy(sprintf "%-+05M" -15.5M)) "-15.5"
    test "test8196" (lazy(sprintf "%-+01M" -15.5M)) "-15.5"
    test "test8197" (lazy(sprintf "%-+0*M" 7 -15.5M)) "-15.500"
    test "test8198" (lazy(sprintf "%-+0.5M" -15.5M)) "-15.5"
    test "test8199" (lazy(sprintf "%-+0.*M" 4 -15.5M)) "-15.5"
    test "test8200" (lazy(sprintf "%-+0*.*M" 5 4 -15.5M)) "-15.5"
    test "test8201" (lazy(sprintf "% M" -15.5M)) "-15.5"
    test "test8202" (lazy(sprintf "% 5M" -15.5M)) "-15.5"
    test "test8203" (lazy(sprintf "% 1M" -15.5M)) "-15.5"
    test "test8204" (lazy(sprintf "% *M" 7 -15.5M)) "  -15.5"
    test "test8205" (lazy(sprintf "% .5M" -15.5M)) "-15.5"
    test "test8206" (lazy(sprintf "% .*M" 4 -15.5M)) "-15.5"
    test "test8207" (lazy(sprintf "% *.*M" 5 4 -15.5M)) "-15.5"
    test "test8208" (lazy(sprintf "%- M" -15.5M)) "-15.5"
    test "test8209" (lazy(sprintf "%- 5M" -15.5M)) "-15.5"
    test "test8210" (lazy(sprintf "%- 1M" -15.5M)) "-15.5"
    test "test8211" (lazy(sprintf "%- *M" 7 -15.5M)) "-15.5  "
    test "test8212" (lazy(sprintf "%- .5M" -15.5M)) "-15.5"
    test "test8213" (lazy(sprintf "%- .*M" 4 -15.5M)) "-15.5"
    test "test8214" (lazy(sprintf "%- *.*M" 5 4 -15.5M)) "-15.5"
    test "test8215" (lazy(sprintf "% 0M" -15.5M)) "-15.5"
    test "test8216" (lazy(sprintf "% 05M" -15.5M)) "-15.5"
    test "test8217" (lazy(sprintf "% 01M" -15.5M)) "-15.5"
    test "test8218" (lazy(sprintf "% 0*M" 7 -15.5M)) "-0015.5"//"00-15.5"
    test "test8219" (lazy(sprintf "% 0.5M" -15.5M)) "-15.5"
    test "test8220" (lazy(sprintf "% 0.*M" 4 -15.5M)) "-15.5"
    test "test8221" (lazy(sprintf "% 0*.*M" 5 4 -15.5M)) "-15.5"
    test "test8222" (lazy(sprintf "%- 0M" -15.5M)) "-15.5"
    test "test8223" (lazy(sprintf "%- 05M" -15.5M)) "-15.5"
    test "test8224" (lazy(sprintf "%- 01M" -15.5M)) "-15.5"
    test "test8225" (lazy(sprintf "%- 0*M" 7 -15.5M)) "-15.500"
    test "test8226" (lazy(sprintf "%- 0.5M" -15.5M)) "-15.5"
    test "test8227" (lazy(sprintf "%- 0.*M" 4 -15.5M)) "-15.5"
    test "test8228" (lazy(sprintf "%- 0*.*M" 5 4 -15.5M)) "-15.5"
    test "test8229" (lazy(sprintf "%M" -7M)) "-7"
    test "test8230" (lazy(sprintf "%5M" -7M)) "   -7"
    test "test8231" (lazy(sprintf "%1M" -7M)) "-7"
    test "test8232" (lazy(sprintf "%*M" 7 -7M)) "     -7"
    test "test8233" (lazy(sprintf "%.5M" -7M)) "-7"
    test "test8234" (lazy(sprintf "%.*M" 4 -7M)) "-7"
    test "test8235" (lazy(sprintf "%*.*M" 5 4 -7M)) "   -7"
    test "test8236" (lazy(sprintf "%-M" -7M)) "-7"
    test "test8237" (lazy(sprintf "%-5M" -7M)) "-7   "
    test "test8238" (lazy(sprintf "%-1M" -7M)) "-7"
    test "test8239" (lazy(sprintf "%-*M" 7 -7M)) "-7     "
    test "test8240" (lazy(sprintf "%-.5M" -7M)) "-7"
    test "test8241" (lazy(sprintf "%-.*M" 4 -7M)) "-7"
    test "test8242" (lazy(sprintf "%-*.*M" 5 4 -7M)) "-7   "
    test "test8243" (lazy(sprintf "%0M" -7M)) "-7"
    test "test8244" (lazy(sprintf "%05M" -7M)) "-0007"//"000-7"
    test "test8245" (lazy(sprintf "%01M" -7M)) "-7"
    test "test8246" (lazy(sprintf "%0*M" 7 -7M)) "-000007"// "00000-7"
    test "test8247" (lazy(sprintf "%0.5M" -7M)) "-7"
    test "test8248" (lazy(sprintf "%0.*M" 4 -7M)) "-7"
    test "test8249" (lazy(sprintf "%0*.*M" 5 4 -7M)) "-0007"// "000-7"
    test "test8250" (lazy(sprintf "%-0M" -7M)) "-7"
    test "test8251" (lazy(sprintf "%-05M" -7M)) "-7   "// "-7000"
    test "test8252" (lazy(sprintf "%-01M" -7M)) "-7"
    test "test8253" (lazy(sprintf "%-0*M" 7 -7M)) "-7     "//"-700000"
    test "test8254" (lazy(sprintf "%-0.5M" -7M)) "-7"
    test "test8255" (lazy(sprintf "%-0.*M" 4 -7M)) "-7"
    test "test8256" (lazy(sprintf "%-0*.*M" 5 4 -7M)) "-7   "//"-7000"
    test "test8257" (lazy(sprintf "%+M" -7M)) "-7"
    test "test8258" (lazy(sprintf "%+5M" -7M)) "   -7"
    test "test8259" (lazy(sprintf "%+1M" -7M)) "-7"
    test "test8260" (lazy(sprintf "%+*M" 7 -7M)) "     -7"
    test "test8261" (lazy(sprintf "%+.5M" -7M)) "-7"
    test "test8262" (lazy(sprintf "%+.*M" 4 -7M)) "-7"
    test "test8263" (lazy(sprintf "%+*.*M" 5 4 -7M)) "   -7"
    test "test8264" (lazy(sprintf "%-+M" -7M)) "-7"
    test "test8265" (lazy(sprintf "%-+5M" -7M)) "-7   "
    test "test8266" (lazy(sprintf "%-+1M" -7M)) "-7"
    test "test8267" (lazy(sprintf "%-+*M" 7 -7M)) "-7     "
    test "test8268" (lazy(sprintf "%-+.5M" -7M)) "-7"
    test "test8269" (lazy(sprintf "%-+.*M" 4 -7M)) "-7"
    test "test8270" (lazy(sprintf "%-+*.*M" 5 4 -7M)) "-7   "
    test "test8271" (lazy(sprintf "%+0M" -7M)) "-7"
    test "test8272" (lazy(sprintf "%+05M" -7M)) "-0007"// "000-7"
    test "test8273" (lazy(sprintf "%+01M" -7M)) "-7"
    test "test8274" (lazy(sprintf "%+0*M" 7 -7M)) "-000007"//"00000-7"
    test "test8275" (lazy(sprintf "%+0.5M" -7M)) "-7"
    test "test8276" (lazy(sprintf "%+0.*M" 4 -7M)) "-7"
    test "test8277" (lazy(sprintf "%+0*.*M" 5 4 -7M)) "-0007"//"000-7"
    test "test8278" (lazy(sprintf "%-+0M" -7M)) "-7"
    test "test8279" (lazy(sprintf "%-+05M" -7M)) "-7   "// "-7000"
    test "test8280" (lazy(sprintf "%-+01M" -7M)) "-7"
    test "test8281" (lazy(sprintf "%-+0*M" 7 -7M)) "-7     "// "-700000"
    test "test8282" (lazy(sprintf "%-+0.5M" -7M)) "-7"
    test "test8283" (lazy(sprintf "%-+0.*M" 4 -7M)) "-7"
    test "test8284" (lazy(sprintf "%-+0*.*M" 5 4 -7M)) "-7   "//"-7000"
    test "test8285" (lazy(sprintf "% M" -7M)) "-7"
    test "test8286" (lazy(sprintf "% 5M" -7M)) "   -7"
    test "test8287" (lazy(sprintf "% 1M" -7M)) "-7"
    test "test8288" (lazy(sprintf "% *M" 7 -7M)) "     -7"
    test "test8289" (lazy(sprintf "% .5M" -7M)) "-7"
    test "test8290" (lazy(sprintf "% .*M" 4 -7M)) "-7"
    test "test8291" (lazy(sprintf "% *.*M" 5 4 -7M)) "   -7"
    test "test8292" (lazy(sprintf "%- M" -7M)) "-7"
    test "test8293" (lazy(sprintf "%- 5M" -7M)) "-7   "
    test "test8294" (lazy(sprintf "%- 1M" -7M)) "-7"
    test "test8295" (lazy(sprintf "%- *M" 7 -7M)) "-7     "
    test "test8296" (lazy(sprintf "%- .5M" -7M)) "-7"
    test "test8297" (lazy(sprintf "%- .*M" 4 -7M)) "-7"
    test "test8298" (lazy(sprintf "%- *.*M" 5 4 -7M)) "-7   "
    test "test8299" (lazy(sprintf "% 0M" -7M)) "-7"
    test "test8300" (lazy(sprintf "% 05M" -7M)) "-0007"//"000-7"
    test "test8301" (lazy(sprintf "% 01M" -7M)) "-7"
    test "test8302" (lazy(sprintf "% 0*M" 7 -7M)) "-000007"//"00000-7"
    test "test8303" (lazy(sprintf "% 0.5M" -7M)) "-7"
    test "test8304" (lazy(sprintf "% 0.*M" 4 -7M)) "-7"
    test "test8305" (lazy(sprintf "% 0*.*M" 5 4 -7M)) "-0007"// "000-7"
    test "test8306" (lazy(sprintf "%- 0M" -7M)) "-7"
    test "test8307" (lazy(sprintf "%- 05M" -7M)) "-7   "// "-7000"
    test "test8308" (lazy(sprintf "%- 01M" -7M)) "-7"
    test "test8309" (lazy(sprintf "%- 0*M" 7 -7M)) "-7     "//"-700000"
    test "test8310" (lazy(sprintf "%- 0.5M" -7M)) "-7"
    test "test8311" (lazy(sprintf "%- 0.*M" 4 -7M)) "-7"
    test "test8312" (lazy(sprintf "%- 0*.*M" 5 4 -7M)) "-7   "///"-7000"
    test "test8313" (lazy(sprintf "%O" "abc")) "abc"
    test "test8314" (lazy(sprintf "%5O" "abc")) "  abc"
    test "test8315" (lazy(sprintf "%1O" "abc")) "abc"
    test "test8316" (lazy(sprintf "%*O" 7 "abc")) "    abc"
    test "test8317" (lazy(sprintf "%-O" "abc")) "abc"
    test "test8318" (lazy(sprintf "%-5O" "abc")) "abc  "
    test "test8319" (lazy(sprintf "%-1O" "abc")) "abc"
    test "test8320" (lazy(sprintf "%-*O" 7 "abc")) "abc    "
    test "test8321" (lazy(sprintf "%O" 15)) "15"
    test "test8322" (lazy(sprintf "%5O" 15)) "   15"
    test "test8323" (lazy(sprintf "%1O" 15)) "15"
    test "test8324" (lazy(sprintf "%*O" 7 15)) "     15"
    test "test8325" (lazy(sprintf "%-O" 15)) "15"
    test "test8326" (lazy(sprintf "%-5O" 15)) "15   "
    test "test8327" (lazy(sprintf "%-1O" 15)) "15"
    test "test8328" (lazy(sprintf "%-*O" 7 15)) "15     "
    test "test8329" (lazy(sprintf "%O" -10)) "-10"
    test "test8330" (lazy(sprintf "%5O" -10)) "  -10"
    test "test8331" (lazy(sprintf "%1O" -10)) "-10"
    test "test8332" (lazy(sprintf "%*O" 7 -10)) "    -10"
    test "test8333" (lazy(sprintf "%-O" -10)) "-10"
    test "test8334" (lazy(sprintf "%-5O" -10)) "-10  "
    test "test8335" (lazy(sprintf "%-1O" -10)) "-10"
    test "test8336" (lazy(sprintf "%-*O" 7 -10)) "-10    "
    test "test8337" (lazy(sprintf "%O" null)) "<null>"
    test "test8338" (lazy(sprintf "%5O" null)) "<null>"
    test "test8339" (lazy(sprintf "%1O" null)) "<null>"
    test "test8340" (lazy(sprintf "%*O" 7 null)) " <null>"
    test "test8341" (lazy(sprintf "%-O" null)) "<null>"
    test "test8342" (lazy(sprintf "%-5O" null)) "<null>"
    test "test8343" (lazy(sprintf "%-1O" null)) "<null>"
    test "test8344" (lazy(sprintf "%-*O" 7 null)) "<null> "
    test "test8345" (lazy(sprintf "%O" 'P')) "P"
    test "test8346" (lazy(sprintf "%5O" 'P')) "    P"
    test "test8347" (lazy(sprintf "%1O" 'P')) "P"
    test "test8348" (lazy(sprintf "%*O" 7 'P')) "      P"
    test "test8349" (lazy(sprintf "%-O" 'P')) "P"
    test "test8350" (lazy(sprintf "%-5O" 'P')) "P    "
    test "test8351" (lazy(sprintf "%-1O" 'P')) "P"
    test "test8352" (lazy(sprintf "%-*O" 7 'P')) "P      "
    test "test8353" (lazy(sprintf "%O" System.IO.FileShare.None)) "None"
    test "test8354" (lazy(sprintf "%5O" System.IO.FileShare.None)) " None"
    test "test8355" (lazy(sprintf "%1O" System.IO.FileShare.None)) "None"
    test "test8356" (lazy(sprintf "%*O" 7 System.IO.FileShare.None)) "   None"
    test "test8357" (lazy(sprintf "%-O" System.IO.FileShare.None)) "None"
    test "test8358" (lazy(sprintf "%-5O" System.IO.FileShare.None)) "None "
    test "test8359" (lazy(sprintf "%-1O" System.IO.FileShare.None)) "None"
    test "test8360" (lazy(sprintf "%-*O" 7 System.IO.FileShare.None)) "None   "
    test "test8361" (lazy(sprintf "%A" "abc")) "\"abc\""
    test "test8362" (lazy(sprintf "%5A" "abc")) "\"abc\""
    test "test8363" (lazy(sprintf "%1A" "abc")) "\"abc\""
    test "test8364" (lazy(sprintf "%*A" 7 "abc")) "\"abc\""
    test "test8365" (lazy(sprintf "%.5A" "abc")) "\"abc\""
    test "test8366" (lazy(sprintf "%.*A" 4 "abc")) "\"abc\""
    test "test8367" (lazy(sprintf "%*.*A" 5 4 "abc")) "\"abc\""
    test "test8368" (lazy(sprintf "%-A" "abc")) "\"abc\""
    test "test8369" (lazy(sprintf "%-5A" "abc")) "\"abc\""
    test "test8370" (lazy(sprintf "%-1A" "abc")) "\"abc\""
    test "test8371" (lazy(sprintf "%-*A" 7 "abc")) "\"abc\""
    test "test8372" (lazy(sprintf "%-.5A" "abc")) "\"abc\""
    test "test8373" (lazy(sprintf "%-.*A" 4 "abc")) "\"abc\""
    test "test8374" (lazy(sprintf "%-*.*A" 5 4 "abc")) "\"abc\""
    test "test8375" (lazy(sprintf "%+A" "abc")) "\"abc\""
    test "test8376" (lazy(sprintf "%+5A" "abc")) "\"abc\""
    test "test8377" (lazy(sprintf "%+1A" "abc")) "\"abc\""
    test "test8378" (lazy(sprintf "%+*A" 7 "abc")) "\"abc\""
    test "test8379" (lazy(sprintf "%+.5A" "abc")) "\"abc\""
    test "test8380" (lazy(sprintf "%+.*A" 4 "abc")) "\"abc\""
    test "test8381" (lazy(sprintf "%+*.*A" 5 4 "abc")) "\"abc\""
    test "test8382" (lazy(sprintf "%-+A" "abc")) "\"abc\""
    test "test8383" (lazy(sprintf "%-+5A" "abc")) "\"abc\""
    test "test8384" (lazy(sprintf "%-+1A" "abc")) "\"abc\""
    test "test8385" (lazy(sprintf "%-+*A" 7 "abc")) "\"abc\""
    test "test8386" (lazy(sprintf "%-+.5A" "abc")) "\"abc\""
    test "test8387" (lazy(sprintf "%-+.*A" 4 "abc")) "\"abc\""
    test "test8388" (lazy(sprintf "%-+*.*A" 5 4 "abc")) "\"abc\""
    test "test8389" (lazy(sprintf "%A" 15)) "15"
    test "test8390" (lazy(sprintf "%5A" 15)) "15"
    test "test8391" (lazy(sprintf "%1A" 15)) "15"
    test "test8392" (lazy(sprintf "%*A" 7 15)) "15"
    test "test8393" (lazy(sprintf "%.5A" 15)) "15"
    test "test8394" (lazy(sprintf "%.*A" 4 15)) "15"
    test "test8395" (lazy(sprintf "%*.*A" 5 4 15)) "15"
    test "test8396" (lazy(sprintf "%-A" 15)) "15"
    test "test8397" (lazy(sprintf "%-5A" 15)) "15"
    test "test8398" (lazy(sprintf "%-1A" 15)) "15"
    test "test8399" (lazy(sprintf "%-*A" 7 15)) "15"
    test "test8400" (lazy(sprintf "%-.5A" 15)) "15"
    test "test8401" (lazy(sprintf "%-.*A" 4 15)) "15"
    test "test8402" (lazy(sprintf "%-*.*A" 5 4 15)) "15"
    test "test8403" (lazy(sprintf "%+A" 15)) "15"
    test "test8404" (lazy(sprintf "%+5A" 15)) "15"
    test "test8405" (lazy(sprintf "%+1A" 15)) "15"
    test "test8406" (lazy(sprintf "%+*A" 7 15)) "15"
    test "test8407" (lazy(sprintf "%+.5A" 15)) "15"
    test "test8408" (lazy(sprintf "%+.*A" 4 15)) "15"
    test "test8409" (lazy(sprintf "%+*.*A" 5 4 15)) "15"
    test "test8410" (lazy(sprintf "%-+A" 15)) "15"
    test "test8411" (lazy(sprintf "%-+5A" 15)) "15"
    test "test8412" (lazy(sprintf "%-+1A" 15)) "15"
    test "test8413" (lazy(sprintf "%-+*A" 7 15)) "15"
    test "test8414" (lazy(sprintf "%-+.5A" 15)) "15"
    test "test8415" (lazy(sprintf "%-+.*A" 4 15)) "15"
    test "test8416" (lazy(sprintf "%-+*.*A" 5 4 15)) "15"
    test "test8417" (lazy(sprintf "%A" -10)) "-10"
    test "test8418" (lazy(sprintf "%5A" -10)) "-10"
    test "test8419" (lazy(sprintf "%1A" -10)) "-10"
    test "test8420" (lazy(sprintf "%*A" 7 -10)) "-10"
    test "test8421" (lazy(sprintf "%.5A" -10)) "-10"
    test "test8422" (lazy(sprintf "%.*A" 4 -10)) "-10"
    test "test8423" (lazy(sprintf "%*.*A" 5 4 -10)) "-10"
    test "test8424" (lazy(sprintf "%-A" -10)) "-10"
    test "test8425" (lazy(sprintf "%-5A" -10)) "-10"
    test "test8426" (lazy(sprintf "%-1A" -10)) "-10"
    test "test8427" (lazy(sprintf "%-*A" 7 -10)) "-10"
    test "test8428" (lazy(sprintf "%-.5A" -10)) "-10"
    test "test8429" (lazy(sprintf "%-.*A" 4 -10)) "-10"
    test "test8430" (lazy(sprintf "%-*.*A" 5 4 -10)) "-10"
    test "test8431" (lazy(sprintf "%+A" -10)) "-10"
    test "test8432" (lazy(sprintf "%+5A" -10)) "-10"
    test "test8433" (lazy(sprintf "%+1A" -10)) "-10"
    test "test8434" (lazy(sprintf "%+*A" 7 -10)) "-10"
    test "test8435" (lazy(sprintf "%+.5A" -10)) "-10"
    test "test8436" (lazy(sprintf "%+.*A" 4 -10)) "-10"
    test "test8437" (lazy(sprintf "%+*.*A" 5 4 -10)) "-10"
    test "test8438" (lazy(sprintf "%-+A" -10)) "-10"
    test "test8439" (lazy(sprintf "%-+5A" -10)) "-10"
    test "test8440" (lazy(sprintf "%-+1A" -10)) "-10"
    test "test8441" (lazy(sprintf "%-+*A" 7 -10)) "-10"
    test "test8442" (lazy(sprintf "%-+.5A" -10)) "-10"
    test "test8443" (lazy(sprintf "%-+.*A" 4 -10)) "-10"
    test "test8444" (lazy(sprintf "%-+*.*A" 5 4 -10)) "-10"
    test "test8445" (lazy(sprintf "%A" null)) "<null>"
    test "test8446" (lazy(sprintf "%5A" null)) "<null>"
    test "test8447" (lazy(sprintf "%1A" null)) "<null>"
    test "test8448" (lazy(sprintf "%*A" 7 null)) "<null>"
    test "test8449" (lazy(sprintf "%.5A" null)) "<null>"
    test "test8450" (lazy(sprintf "%.*A" 4 null)) "<null>"
    test "test8451" (lazy(sprintf "%*.*A" 5 4 null)) "<null>"
    test "test8452" (lazy(sprintf "%-A" null)) "<null>"
    test "test8453" (lazy(sprintf "%-5A" null)) "<null>"
    test "test8454" (lazy(sprintf "%-1A" null)) "<null>"
    test "test8455" (lazy(sprintf "%-*A" 7 null)) "<null>"
    test "test8456" (lazy(sprintf "%-.5A" null)) "<null>"
    test "test8457" (lazy(sprintf "%-.*A" 4 null)) "<null>"
    test "test8458" (lazy(sprintf "%-*.*A" 5 4 null)) "<null>"
    test "test8459" (lazy(sprintf "%+A" null)) "<null>"
    test "test8460" (lazy(sprintf "%+5A" null)) "<null>"
    test "test8461" (lazy(sprintf "%+1A" null)) "<null>"
    test "test8462" (lazy(sprintf "%+*A" 7 null)) "<null>"
    test "test8463" (lazy(sprintf "%+.5A" null)) "<null>"
    test "test8464" (lazy(sprintf "%+.*A" 4 null)) "<null>"
    test "test8465" (lazy(sprintf "%+*.*A" 5 4 null)) "<null>"
    test "test8466" (lazy(sprintf "%-+A" null)) "<null>"
    test "test8467" (lazy(sprintf "%-+5A" null)) "<null>"
    test "test8468" (lazy(sprintf "%-+1A" null)) "<null>"
    test "test8469" (lazy(sprintf "%-+*A" 7 null)) "<null>"
    test "test8470" (lazy(sprintf "%-+.5A" null)) "<null>"
    test "test8471" (lazy(sprintf "%-+.*A" 4 null)) "<null>"
    test "test8472" (lazy(sprintf "%-+*.*A" 5 4 null)) "<null>"
    test "test8473" (lazy(sprintf "%A" 'P')) "'P'"
    test "test8474" (lazy(sprintf "%5A" 'P')) "'P'"
    test "test8475" (lazy(sprintf "%1A" 'P')) "'P'"
    test "test8476" (lazy(sprintf "%*A" 7 'P')) "'P'"
    test "test8477" (lazy(sprintf "%.5A" 'P')) "'P'"
    test "test8478" (lazy(sprintf "%.*A" 4 'P')) "'P'"
    test "test8479" (lazy(sprintf "%*.*A" 5 4 'P')) "'P'"
    test "test8480" (lazy(sprintf "%-A" 'P')) "'P'"
    test "test8481" (lazy(sprintf "%-5A" 'P')) "'P'"
    test "test8482" (lazy(sprintf "%-1A" 'P')) "'P'"
    test "test8483" (lazy(sprintf "%-*A" 7 'P')) "'P'"
    test "test8484" (lazy(sprintf "%-.5A" 'P')) "'P'"
    test "test8485" (lazy(sprintf "%-.*A" 4 'P')) "'P'"
    test "test8486" (lazy(sprintf "%-*.*A" 5 4 'P')) "'P'"
    test "test8487" (lazy(sprintf "%+A" 'P')) "'P'"
    test "test8488" (lazy(sprintf "%+5A" 'P')) "'P'"
    test "test8489" (lazy(sprintf "%+1A" 'P')) "'P'"
    test "test8490" (lazy(sprintf "%+*A" 7 'P')) "'P'"
    test "test8491" (lazy(sprintf "%+.5A" 'P')) "'P'"
    test "test8492" (lazy(sprintf "%+.*A" 4 'P')) "'P'"
    test "test8493" (lazy(sprintf "%+*.*A" 5 4 'P')) "'P'"
    test "test8494" (lazy(sprintf "%-+A" 'P')) "'P'"
    test "test8495" (lazy(sprintf "%-+5A" 'P')) "'P'"
    test "test8496" (lazy(sprintf "%-+1A" 'P')) "'P'"
    test "test8497" (lazy(sprintf "%-+*A" 7 'P')) "'P'"
    test "test8498" (lazy(sprintf "%-+.5A" 'P')) "'P'"
    test "test8499" (lazy(sprintf "%-+.*A" 4 'P')) "'P'"
    test "test8500" (lazy(sprintf "%-+*.*A" 5 4 'P')) "'P'"
    test "test8501" (lazy(sprintf "%A" System.IO.FileShare.None)) "None"
    test "test8502" (lazy(sprintf "%5A" System.IO.FileShare.None)) "None"
    test "test8503" (lazy(sprintf "%1A" System.IO.FileShare.None)) "None"
    test "test8504" (lazy(sprintf "%*A" 7 System.IO.FileShare.None)) "None"
    test "test8505" (lazy(sprintf "%.5A" System.IO.FileShare.None)) "None"
    test "test8506" (lazy(sprintf "%.*A" 4 System.IO.FileShare.None)) "None"
    test "test8507" (lazy(sprintf "%*.*A" 5 4 System.IO.FileShare.None)) "None"
    test "test8508" (lazy(sprintf "%-A" System.IO.FileShare.None)) "None"
    test "test8509" (lazy(sprintf "%-5A" System.IO.FileShare.None)) "None"
    test "test8510" (lazy(sprintf "%-1A" System.IO.FileShare.None)) "None"
    test "test8511" (lazy(sprintf "%-*A" 7 System.IO.FileShare.None)) "None"
    test "test8512" (lazy(sprintf "%-.5A" System.IO.FileShare.None)) "None"
    test "test8513" (lazy(sprintf "%-.*A" 4 System.IO.FileShare.None)) "None"
    test "test8514" (lazy(sprintf "%-*.*A" 5 4 System.IO.FileShare.None)) "None"
    test "test8515" (lazy(sprintf "%+A" System.IO.FileShare.None)) "None"
    test "test8516" (lazy(sprintf "%+5A" System.IO.FileShare.None)) "None"
    test "test8517" (lazy(sprintf "%+1A" System.IO.FileShare.None)) "None"
    test "test8518" (lazy(sprintf "%+*A" 7 System.IO.FileShare.None)) "None"
    test "test8519" (lazy(sprintf "%+.5A" System.IO.FileShare.None)) "None"
    test "test8520" (lazy(sprintf "%+.*A" 4 System.IO.FileShare.None)) "None"
    test "test8521" (lazy(sprintf "%+*.*A" 5 4 System.IO.FileShare.None)) "None"
    test "test8522" (lazy(sprintf "%-+A" System.IO.FileShare.None)) "None"
    test "test8523" (lazy(sprintf "%-+5A" System.IO.FileShare.None)) "None"
    test "test8524" (lazy(sprintf "%-+1A" System.IO.FileShare.None)) "None"
    test "test8525" (lazy(sprintf "%-+*A" 7 System.IO.FileShare.None)) "None"
    test "test8526" (lazy(sprintf "%-+.5A" System.IO.FileShare.None)) "None"
    test "test8527" (lazy(sprintf "%-+.*A" 4 System.IO.FileShare.None)) "None"
    test "test8528" (lazy(sprintf "%-+*.*A" 5 4 System.IO.FileShare.None)) "None"
    test "test8529" (lazy(sprintf "%a" (fun _ s -> (string s) + "!!!") "abc")) "abc!!!"
    test "test8530" (lazy(sprintf "%a" (fun _ s -> (string s) + "!!!") 15)) "15!!!"
    test "test8531" (lazy(sprintf "%a" (fun _ s -> (string s) + "!!!") -10)) "-10!!!"
    test "test8532" (lazy(sprintf "%a" (fun _ s -> (string s) + "!!!") null)) "!!!"
    test "test8533" (lazy(sprintf "%a" (fun _ s -> (string s) + "!!!") 'P')) "P!!!"
    test "test8534" (lazy(sprintf "%a" (fun _ s -> (string s) + "!!!") System.IO.FileShare.None)) "None!!!"
    test "test8535" (lazy(sprintf "%t" (fun _ -> "???"))) "???"
    test "test8536" (lazy(sprintf "A%dB" 0)) "A0B"
    test "test8537" (lazy(sprintf "A%dB%dC" 0 1)) "A0B1C"
    test "test8538" (lazy(sprintf "A%dB%dC%dD" 0 1 2)) "A0B1C2D"
    test "test8539" (lazy(sprintf "A%dB%dC%dD%dE" 0 1 2 3)) "A0B1C2D3E"
    test "test8540" (lazy(sprintf "A%dB%dC%dD%dE%dF" 0 1 2 3 4)) "A0B1C2D3E4F"
    test "test8541" (lazy(sprintf "A%dB%dC%dD%dE%dF%dG" 0 1 2 3 4 5)) "A0B1C2D3E4F5G"
    test "test8542" (lazy(sprintf "A%dB%dC%dD%dE%dF%dG%dH" 0 1 2 3 4 5 6)) "A0B1C2D3E4F5G6H"
    test "test8543" (lazy(sprintf "A%dB%dC%dD%dE%dF%dG%dH%dI" 0 1 2 3 4 5 6 7)) "A0B1C2D3E4F5G6H7I"
    test "test8544" (lazy(sprintf "A%dB%dC%dD%dE%dF%dG%dH%dI%dJ" 0 1 2 3 4 5 6 7 8)) "A0B1C2D3E4F5G6H7I8J"
    test "test8545" (lazy(sprintf "A%dB%dC%dD%dE%dF%dG%dH%dI%dJ%dK" 0 1 2 3 4 5 6 7 8 9)) "A0B1C2D3E4F5G6H7I8J9K"
    test "test8546" (lazy(sprintf "A%dB%dC%dD%dE%dF%dG%dH%dI%dJ%dK%dL" 0 1 2 3 4 5 6 7 8 9 10)) "A0B1C2D3E4F5G6H7I8J9K10L"
    test "test8547" (lazy(sprintf "A%dB%dC%dD%dE%dF%dG%dH%dI%dJ%dK%dL%dM" 0 1 2 3 4 5 6 7 8 9 10 11)) "A0B1C2D3E4F5G6H7I8J9K10L11M"
    test "test8548" (lazy(sprintf "A%dB%dC%dD%dE%dF%dG%dH%dI%dJ%dK%dL%dM%dN" 0 1 2 3 4 5 6 7 8 9 10 11 12)) "A0B1C2D3E4F5G6H7I8J9K10L11M12N"
    test "test8549" (lazy(sprintf "A%dB%dC%dD%dE%dF%dG%dH%dI%dJ%dK%dL%dM%dN%dO" 0 1 2 3 4 5 6 7 8 9 10 11 12 13)) "A0B1C2D3E4F5G6H7I8J9K10L11M12N13O"
    test "test8550" (lazy(sprintf "A%dB%dC%dD%dE%dF%dG%dH%dI%dJ%dK%dL%dM%dN%dO%dP" 0 1 2 3 4 5 6 7 8 9 10 11 12 13 14)) "A0B1C2D3E4F5G6H7I8J9K10L11M12N13O14P"
    test "test8551" (lazy(sprintf "A%dB%dC%dD%dE%dF%dG%dH%dI%dJ%dK%dL%dM%dN%dO%dP%dQ" 0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15)) "A0B1C2D3E4F5G6H7I8J9K10L11M12N13O14P15Q"
    test "test8552" (lazy(sprintf "A%dB%dC%dD%dE%dF%dG%dH%dI%dJ%dK%dL%dM%dN%dO%dP%dQ%dR" 0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16)) "A0B1C2D3E4F5G6H7I8J9K10L11M12N13O14P15Q16R"
    test "test8553" (lazy(sprintf "A%dB%dC%dD%dE%dF%dG%dH%dI%dJ%dK%dL%dM%dN%dO%dP%dQ%dR%dS" 0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17)) "A0B1C2D3E4F5G6H7I8J9K10L11M12N13O14P15Q16R17S"
    test "test8554" (lazy(sprintf "A%dB%dC%dD%dE%dF%dG%dH%dI%dJ%dK%dL%dM%dN%dO%dP%dQ%dR%dS%dT" 0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18)) "A0B1C2D3E4F5G6H7I8J9K10L11M12N13O14P15Q16R17S18T"
    test "test8555" (lazy(sprintf "A%dB%dC%dD%dE%dF%dG%dH%dI%dJ%dK%dL%dM%dN%dO%dP%dQ%dR%dS%dT%dU" 0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 19)) "A0B1C2D3E4F5G6H7I8J9K10L11M12N13O14P15Q16R17S18T19U"
    test "test8556" (lazy(sprintf "A%aB" (fun _ v -> string v) 0)) "A0B"
    test "test8557" (lazy(sprintf "A%aB%aC" (fun _ v -> string v) 0 (fun _ v -> string v) 1)) "A0B1C"
    test "test8558" (lazy(sprintf "A%aB%aC%aD" (fun _ v -> string v) 0 (fun _ v -> string v) 1 (fun _ v -> string v) 2)) "A0B1C2D"
    test "test8559" (lazy(sprintf "A%aB%aC%aD%aE" (fun _ v -> string v) 0 (fun _ v -> string v) 1 (fun _ v -> string v) 2 (fun _ v -> string v) 3)) "A0B1C2D3E"
    test "test8560" (lazy(sprintf "A%aB%aC%aD%aE%aF" (fun _ v -> string v) 0 (fun _ v -> string v) 1 (fun _ v -> string v) 2 (fun _ v -> string v) 3 (fun _ v -> string v) 4)) "A0B1C2D3E4F"
    test "test8561" (lazy(sprintf "A%aB%aC%aD%aE%aF%aG" (fun _ v -> string v) 0 (fun _ v -> string v) 1 (fun _ v -> string v) 2 (fun _ v -> string v) 3 (fun _ v -> string v) 4 (fun _ v -> string v) 5)) "A0B1C2D3E4F5G"
    test "test8562" (lazy(sprintf "A%aB%aC%aD%aE%aF%aG%aH" (fun _ v -> string v) 0 (fun _ v -> string v) 1 (fun _ v -> string v) 2 (fun _ v -> string v) 3 (fun _ v -> string v) 4 (fun _ v -> string v) 5 (fun _ v -> string v) 6)) "A0B1C2D3E4F5G6H"
    test "test8563" (lazy(sprintf "A%aB%aC%aD%aE%aF%aG%aH%aI" (fun _ v -> string v) 0 (fun _ v -> string v) 1 (fun _ v -> string v) 2 (fun _ v -> string v) 3 (fun _ v -> string v) 4 (fun _ v -> string v) 5 (fun _ v -> string v) 6 (fun _ v -> string v) 7)) "A0B1C2D3E4F5G6H7I"
    test "test8564" (lazy(sprintf "A%aB%aC%aD%aE%aF%aG%aH%aI%aJ" (fun _ v -> string v) 0 (fun _ v -> string v) 1 (fun _ v -> string v) 2 (fun _ v -> string v) 3 (fun _ v -> string v) 4 (fun _ v -> string v) 5 (fun _ v -> string v) 6 (fun _ v -> string v) 7 (fun _ v -> string v) 8)) "A0B1C2D3E4F5G6H7I8J"
    test "test8565" (lazy(sprintf "A%aB%aC%aD%aE%aF%aG%aH%aI%aJ%aK" (fun _ v -> string v) 0 (fun _ v -> string v) 1 (fun _ v -> string v) 2 (fun _ v -> string v) 3 (fun _ v -> string v) 4 (fun _ v -> string v) 5 (fun _ v -> string v) 6 (fun _ v -> string v) 7 (fun _ v -> string v) 8 (fun _ v -> string v) 9)) "A0B1C2D3E4F5G6H7I8J9K"
    test "test8566" (lazy(sprintf "A%aB%aC%aD%aE%aF%aG%aH%aI%aJ%aK%aL" (fun _ v -> string v) 0 (fun _ v -> string v) 1 (fun _ v -> string v) 2 (fun _ v -> string v) 3 (fun _ v -> string v) 4 (fun _ v -> string v) 5 (fun _ v -> string v) 6 (fun _ v -> string v) 7 (fun _ v -> string v) 8 (fun _ v -> string v) 9 (fun _ v -> string v) 10)) "A0B1C2D3E4F5G6H7I8J9K10L"
    test "test8567" (lazy(sprintf "A%aB%aC%aD%aE%aF%aG%aH%aI%aJ%aK%aL%aM" (fun _ v -> string v) 0 (fun _ v -> string v) 1 (fun _ v -> string v) 2 (fun _ v -> string v) 3 (fun _ v -> string v) 4 (fun _ v -> string v) 5 (fun _ v -> string v) 6 (fun _ v -> string v) 7 (fun _ v -> string v) 8 (fun _ v -> string v) 9 (fun _ v -> string v) 10 (fun _ v -> string v) 11)) "A0B1C2D3E4F5G6H7I8J9K10L11M"
    test "test8568" (lazy(sprintf "A%aB%aC%aD%aE%aF%aG%aH%aI%aJ%aK%aL%aM%aN" (fun _ v -> string v) 0 (fun _ v -> string v) 1 (fun _ v -> string v) 2 (fun _ v -> string v) 3 (fun _ v -> string v) 4 (fun _ v -> string v) 5 (fun _ v -> string v) 6 (fun _ v -> string v) 7 (fun _ v -> string v) 8 (fun _ v -> string v) 9 (fun _ v -> string v) 10 (fun _ v -> string v) 11 (fun _ v -> string v) 12)) "A0B1C2D3E4F5G6H7I8J9K10L11M12N"
    test "test8569" (lazy(sprintf "A%aB%aC%aD%aE%aF%aG%aH%aI%aJ%aK%aL%aM%aN%aO" (fun _ v -> string v) 0 (fun _ v -> string v) 1 (fun _ v -> string v) 2 (fun _ v -> string v) 3 (fun _ v -> string v) 4 (fun _ v -> string v) 5 (fun _ v -> string v) 6 (fun _ v -> string v) 7 (fun _ v -> string v) 8 (fun _ v -> string v) 9 (fun _ v -> string v) 10 (fun _ v -> string v) 11 (fun _ v -> string v) 12 (fun _ v -> string v) 13)) "A0B1C2D3E4F5G6H7I8J9K10L11M12N13O"
    test "test8570" (lazy(sprintf "A%aB%aC%aD%aE%aF%aG%aH%aI%aJ%aK%aL%aM%aN%aO%aP" (fun _ v -> string v) 0 (fun _ v -> string v) 1 (fun _ v -> string v) 2 (fun _ v -> string v) 3 (fun _ v -> string v) 4 (fun _ v -> string v) 5 (fun _ v -> string v) 6 (fun _ v -> string v) 7 (fun _ v -> string v) 8 (fun _ v -> string v) 9 (fun _ v -> string v) 10 (fun _ v -> string v) 11 (fun _ v -> string v) 12 (fun _ v -> string v) 13 (fun _ v -> string v) 14)) "A0B1C2D3E4F5G6H7I8J9K10L11M12N13O14P"
    test "test8571" (lazy(sprintf "A%aB%aC%aD%aE%aF%aG%aH%aI%aJ%aK%aL%aM%aN%aO%aP%aQ" (fun _ v -> string v) 0 (fun _ v -> string v) 1 (fun _ v -> string v) 2 (fun _ v -> string v) 3 (fun _ v -> string v) 4 (fun _ v -> string v) 5 (fun _ v -> string v) 6 (fun _ v -> string v) 7 (fun _ v -> string v) 8 (fun _ v -> string v) 9 (fun _ v -> string v) 10 (fun _ v -> string v) 11 (fun _ v -> string v) 12 (fun _ v -> string v) 13 (fun _ v -> string v) 14 (fun _ v -> string v) 15)) "A0B1C2D3E4F5G6H7I8J9K10L11M12N13O14P15Q"
    test "test8572" (lazy(sprintf "A%aB%aC%aD%aE%aF%aG%aH%aI%aJ%aK%aL%aM%aN%aO%aP%aQ%aR" (fun _ v -> string v) 0 (fun _ v -> string v) 1 (fun _ v -> string v) 2 (fun _ v -> string v) 3 (fun _ v -> string v) 4 (fun _ v -> string v) 5 (fun _ v -> string v) 6 (fun _ v -> string v) 7 (fun _ v -> string v) 8 (fun _ v -> string v) 9 (fun _ v -> string v) 10 (fun _ v -> string v) 11 (fun _ v -> string v) 12 (fun _ v -> string v) 13 (fun _ v -> string v) 14 (fun _ v -> string v) 15 (fun _ v -> string v) 16)) "A0B1C2D3E4F5G6H7I8J9K10L11M12N13O14P15Q16R"
    test "test8573" (lazy(sprintf "A%aB%aC%aD%aE%aF%aG%aH%aI%aJ%aK%aL%aM%aN%aO%aP%aQ%aR%aS" (fun _ v -> string v) 0 (fun _ v -> string v) 1 (fun _ v -> string v) 2 (fun _ v -> string v) 3 (fun _ v -> string v) 4 (fun _ v -> string v) 5 (fun _ v -> string v) 6 (fun _ v -> string v) 7 (fun _ v -> string v) 8 (fun _ v -> string v) 9 (fun _ v -> string v) 10 (fun _ v -> string v) 11 (fun _ v -> string v) 12 (fun _ v -> string v) 13 (fun _ v -> string v) 14 (fun _ v -> string v) 15 (fun _ v -> string v) 16 (fun _ v -> string v) 17)) "A0B1C2D3E4F5G6H7I8J9K10L11M12N13O14P15Q16R17S"
    test "test8574" (lazy(sprintf "A%aB%aC%aD%aE%aF%aG%aH%aI%aJ%aK%aL%aM%aN%aO%aP%aQ%aR%aS%aT" (fun _ v -> string v) 0 (fun _ v -> string v) 1 (fun _ v -> string v) 2 (fun _ v -> string v) 3 (fun _ v -> string v) 4 (fun _ v -> string v) 5 (fun _ v -> string v) 6 (fun _ v -> string v) 7 (fun _ v -> string v) 8 (fun _ v -> string v) 9 (fun _ v -> string v) 10 (fun _ v -> string v) 11 (fun _ v -> string v) 12 (fun _ v -> string v) 13 (fun _ v -> string v) 14 (fun _ v -> string v) 15 (fun _ v -> string v) 16 (fun _ v -> string v) 17 (fun _ v -> string v) 18)) "A0B1C2D3E4F5G6H7I8J9K10L11M12N13O14P15Q16R17S18T"
    test "test8575" (lazy(sprintf "A%aB%aC%aD%aE%aF%aG%aH%aI%aJ%aK%aL%aM%aN%aO%aP%aQ%aR%aS%aT%aU" (fun _ v -> string v) 0 (fun _ v -> string v) 1 (fun _ v -> string v) 2 (fun _ v -> string v) 3 (fun _ v -> string v) 4 (fun _ v -> string v) 5 (fun _ v -> string v) 6 (fun _ v -> string v) 7 (fun _ v -> string v) 8 (fun _ v -> string v) 9 (fun _ v -> string v) 10 (fun _ v -> string v) 11 (fun _ v -> string v) 12 (fun _ v -> string v) 13 (fun _ v -> string v) 14 (fun _ v -> string v) 15 (fun _ v -> string v) 16 (fun _ v -> string v) 17 (fun _ v -> string v) 18 (fun _ v -> string v) 19)) "A0B1C2D3E4F5G6H7I8J9K10L11M12N13O14P15Q16R17S18T19U"
    test "test8576" (lazy(sprintf "01-00%d01-01%a11-10%d11-11" 0  (fun _ v -> (string v) + "X") 1  10 )) "01-00001-011X11-101011-11"
    test "test8577" (lazy(sprintf "01-00%d01-01%a11-10%d11-11%a_TAIL" 0  (fun _ v -> (string v) + "X") 1  10 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "01-00001-011X11-101011-11ReadX_TAIL"
    test "test8578" (lazy(sprintf "01-00%d01-01%d01-02%a11-10%d11-11%d11-12" 0 1  (fun _ v -> (string v) + "X") 1  10 11 )) "01-00001-01101-021X11-101011-111111-12"
    test "test8579" (lazy(sprintf "01-00%d01-01%d01-02%a11-10%d11-11%d11-12%a_TAIL" 0 1  (fun _ v -> (string v) + "X") 1  10 11 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "01-00001-01101-021X11-101011-111111-12ReadX_TAIL"
    test "test8580" (lazy(sprintf "01-00%d01-01%d01-02%d01-03%a11-10%d11-11%d11-12%d11-13" 0 1 2  (fun _ v -> (string v) + "X") 1  10 11 12 )) "01-00001-01101-02201-031X11-101011-111111-121211-13"
    test "test8581" (lazy(sprintf "01-00%d01-01%d01-02%d01-03%a11-10%d11-11%d11-12%d11-13%a_TAIL" 0 1 2  (fun _ v -> (string v) + "X") 1  10 11 12 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "01-00001-01101-02201-031X11-101011-111111-121211-13ReadX_TAIL"
    test "test8582" (lazy(sprintf "01-00%d01-01%d01-02%d01-03%d01-04%a11-10%d11-11%d11-12%d11-13%d11-14" 0 1 2 3  (fun _ v -> (string v) + "X") 1  10 11 12 13 )) "01-00001-01101-02201-03301-041X11-101011-111111-121211-131311-14"
    test "test8583" (lazy(sprintf "01-00%d01-01%d01-02%d01-03%d01-04%a11-10%d11-11%d11-12%d11-13%d11-14%a_TAIL" 0 1 2 3  (fun _ v -> (string v) + "X") 1  10 11 12 13 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "01-00001-01101-02201-03301-041X11-101011-111111-121211-131311-14ReadX_TAIL"
    test "test8584" (lazy(sprintf "01-00%d01-01%d01-02%d01-03%d01-04%d01-05%a11-10%d11-11%d11-12%d11-13%d11-14%d11-15" 0 1 2 3 4  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 )) "01-00001-01101-02201-03301-04401-051X11-101011-111111-121211-131311-141411-15"
    test "test8585" (lazy(sprintf "01-00%d01-01%d01-02%d01-03%d01-04%d01-05%a11-10%d11-11%d11-12%d11-13%d11-14%d11-15%a_TAIL" 0 1 2 3 4  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "01-00001-01101-02201-03301-04401-051X11-101011-111111-121211-131311-141411-15ReadX_TAIL"
    test "test8586" (lazy(sprintf "01-00%d01-01%d01-02%d01-03%d01-04%d01-05%d01-06%a11-10%d11-11%d11-12%d11-13%d11-14%d11-15%d11-16" 0 1 2 3 4 5  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 )) "01-00001-01101-02201-03301-04401-05501-061X11-101011-111111-121211-131311-141411-151511-16"
    test "test8587" (lazy(sprintf "01-00%d01-01%d01-02%d01-03%d01-04%d01-05%d01-06%a11-10%d11-11%d11-12%d11-13%d11-14%d11-15%d11-16%a_TAIL" 0 1 2 3 4 5  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "01-00001-01101-02201-03301-04401-05501-061X11-101011-111111-121211-131311-141411-151511-16ReadX_TAIL"
    test "test8588" (lazy(sprintf "01-00%d01-01%d01-02%d01-03%d01-04%d01-05%d01-06%d01-07%a11-10%d11-11%d11-12%d11-13%d11-14%d11-15%d11-16%d11-17" 0 1 2 3 4 5 6  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 )) "01-00001-01101-02201-03301-04401-05501-06601-071X11-101011-111111-121211-131311-141411-151511-161611-17"
    test "test8589" (lazy(sprintf "01-00%d01-01%d01-02%d01-03%d01-04%d01-05%d01-06%d01-07%a11-10%d11-11%d11-12%d11-13%d11-14%d11-15%d11-16%d11-17%a_TAIL" 0 1 2 3 4 5 6  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "01-00001-01101-02201-03301-04401-05501-06601-071X11-101011-111111-121211-131311-141411-151511-161611-17ReadX_TAIL"
    test "test8590" (lazy(sprintf "01-00%d01-01%d01-02%d01-03%d01-04%d01-05%d01-06%d01-07%d01-08%a11-10%d11-11%d11-12%d11-13%d11-14%d11-15%d11-16%d11-17%d11-18" 0 1 2 3 4 5 6 7  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17 )) "01-00001-01101-02201-03301-04401-05501-06601-07701-081X11-101011-111111-121211-131311-141411-151511-161611-171711-18"
    test "test8591" (lazy(sprintf "01-00%d01-01%d01-02%d01-03%d01-04%d01-05%d01-06%d01-07%d01-08%a11-10%d11-11%d11-12%d11-13%d11-14%d11-15%d11-16%d11-17%d11-18%a_TAIL" 0 1 2 3 4 5 6 7  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "01-00001-01101-02201-03301-04401-05501-06601-07701-081X11-101011-111111-121211-131311-141411-151511-161611-171711-18ReadX_TAIL"
    test "test8592" (lazy(sprintf "01-00%d01-01%d01-02%d01-03%d01-04%d01-05%d01-06%d01-07%d01-08%d01-09%a11-10%d11-11%d11-12%d11-13%d11-14%d11-15%d11-16%d11-17%d11-18%d11-19" 0 1 2 3 4 5 6 7 8  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17 18 )) "01-00001-01101-02201-03301-04401-05501-06601-07701-08801-091X11-101011-111111-121211-131311-141411-151511-161611-171711-181811-19"
    test "test8593" (lazy(sprintf "01-00%d01-01%d01-02%d01-03%d01-04%d01-05%d01-06%d01-07%d01-08%d01-09%a11-10%d11-11%d11-12%d11-13%d11-14%d11-15%d11-16%d11-17%d11-18%d11-19%a_TAIL" 0 1 2 3 4 5 6 7 8  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17 18 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "01-00001-01101-02201-03301-04401-05501-06601-07701-08801-091X11-101011-111111-121211-131311-141411-151511-161611-171711-181811-19ReadX_TAIL"
    test "test8594" (lazy(sprintf "02-00%d02-01%a12-10%d12-11%a22-20%d22-21" 0  (fun _ v -> (string v) + "X") 1  10  (fun _ v -> (string v) + "X") 2  20 )) "02-00002-011X12-101012-112X22-202022-21"
    test "test8595" (lazy(sprintf "02-00%d02-01%a12-10%d12-11%a22-20%d22-21%a_TAIL" 0  (fun _ v -> (string v) + "X") 1  10  (fun _ v -> (string v) + "X") 2  20 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "02-00002-011X12-101012-112X22-202022-21ReadX_TAIL"
    test "test8596" (lazy(sprintf "02-00%d02-01%d02-02%a12-10%d12-11%d12-12%a22-20%d22-21%d22-22" 0 1  (fun _ v -> (string v) + "X") 1  10 11  (fun _ v -> (string v) + "X") 2  20 21 )) "02-00002-01102-021X12-101012-111112-122X22-202022-212122-22"
    test "test8597" (lazy(sprintf "02-00%d02-01%d02-02%a12-10%d12-11%d12-12%a22-20%d22-21%d22-22%a_TAIL" 0 1  (fun _ v -> (string v) + "X") 1  10 11  (fun _ v -> (string v) + "X") 2  20 21 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "02-00002-01102-021X12-101012-111112-122X22-202022-212122-22ReadX_TAIL"
    test "test8598" (lazy(sprintf "02-00%d02-01%d02-02%d02-03%a12-10%d12-11%d12-12%d12-13%a22-20%d22-21%d22-22%d22-23" 0 1 2  (fun _ v -> (string v) + "X") 1  10 11 12  (fun _ v -> (string v) + "X") 2  20 21 22 )) "02-00002-01102-02202-031X12-101012-111112-121212-132X22-202022-212122-222222-23"
    test "test8599" (lazy(sprintf "02-00%d02-01%d02-02%d02-03%a12-10%d12-11%d12-12%d12-13%a22-20%d22-21%d22-22%d22-23%a_TAIL" 0 1 2  (fun _ v -> (string v) + "X") 1  10 11 12  (fun _ v -> (string v) + "X") 2  20 21 22 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "02-00002-01102-02202-031X12-101012-111112-121212-132X22-202022-212122-222222-23ReadX_TAIL"
    test "test8600" (lazy(sprintf "02-00%d02-01%d02-02%d02-03%d02-04%a12-10%d12-11%d12-12%d12-13%d12-14%a22-20%d22-21%d22-22%d22-23%d22-24" 0 1 2 3  (fun _ v -> (string v) + "X") 1  10 11 12 13  (fun _ v -> (string v) + "X") 2  20 21 22 23 )) "02-00002-01102-02202-03302-041X12-101012-111112-121212-131312-142X22-202022-212122-222222-232322-24"
    test "test8601" (lazy(sprintf "02-00%d02-01%d02-02%d02-03%d02-04%a12-10%d12-11%d12-12%d12-13%d12-14%a22-20%d22-21%d22-22%d22-23%d22-24%a_TAIL" 0 1 2 3  (fun _ v -> (string v) + "X") 1  10 11 12 13  (fun _ v -> (string v) + "X") 2  20 21 22 23 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "02-00002-01102-02202-03302-041X12-101012-111112-121212-131312-142X22-202022-212122-222222-232322-24ReadX_TAIL"
    test "test8602" (lazy(sprintf "02-00%d02-01%d02-02%d02-03%d02-04%d02-05%a12-10%d12-11%d12-12%d12-13%d12-14%d12-15%a22-20%d22-21%d22-22%d22-23%d22-24%d22-25" 0 1 2 3 4  (fun _ v -> (string v) + "X") 1  10 11 12 13 14  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 )) "02-00002-01102-02202-03302-04402-051X12-101012-111112-121212-131312-141412-152X22-202022-212122-222222-232322-242422-25"
    test "test8603" (lazy(sprintf "02-00%d02-01%d02-02%d02-03%d02-04%d02-05%a12-10%d12-11%d12-12%d12-13%d12-14%d12-15%a22-20%d22-21%d22-22%d22-23%d22-24%d22-25%a_TAIL" 0 1 2 3 4  (fun _ v -> (string v) + "X") 1  10 11 12 13 14  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "02-00002-01102-02202-03302-04402-051X12-101012-111112-121212-131312-141412-152X22-202022-212122-222222-232322-242422-25ReadX_TAIL"
    test "test8604" (lazy(sprintf "02-00%d02-01%d02-02%d02-03%d02-04%d02-05%d02-06%a12-10%d12-11%d12-12%d12-13%d12-14%d12-15%d12-16%a22-20%d22-21%d22-22%d22-23%d22-24%d22-25%d22-26" 0 1 2 3 4 5  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 )) "02-00002-01102-02202-03302-04402-05502-061X12-101012-111112-121212-131312-141412-151512-162X22-202022-212122-222222-232322-242422-252522-26"
    test "test8605" (lazy(sprintf "02-00%d02-01%d02-02%d02-03%d02-04%d02-05%d02-06%a12-10%d12-11%d12-12%d12-13%d12-14%d12-15%d12-16%a22-20%d22-21%d22-22%d22-23%d22-24%d22-25%d22-26%a_TAIL" 0 1 2 3 4 5  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "02-00002-01102-02202-03302-04402-05502-061X12-101012-111112-121212-131312-141412-151512-162X22-202022-212122-222222-232322-242422-252522-26ReadX_TAIL"
    test "test8606" (lazy(sprintf "02-00%d02-01%d02-02%d02-03%d02-04%d02-05%d02-06%d02-07%a12-10%d12-11%d12-12%d12-13%d12-14%d12-15%d12-16%d12-17%a22-20%d22-21%d22-22%d22-23%d22-24%d22-25%d22-26%d22-27" 0 1 2 3 4 5 6  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 )) "02-00002-01102-02202-03302-04402-05502-06602-071X12-101012-111112-121212-131312-141412-151512-161612-172X22-202022-212122-222222-232322-242422-252522-262622-27"
    test "test8607" (lazy(sprintf "02-00%d02-01%d02-02%d02-03%d02-04%d02-05%d02-06%d02-07%a12-10%d12-11%d12-12%d12-13%d12-14%d12-15%d12-16%d12-17%a22-20%d22-21%d22-22%d22-23%d22-24%d22-25%d22-26%d22-27%a_TAIL" 0 1 2 3 4 5 6  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "02-00002-01102-02202-03302-04402-05502-06602-071X12-101012-111112-121212-131312-141412-151512-161612-172X22-202022-212122-222222-232322-242422-252522-262622-27ReadX_TAIL"
    test "test8608" (lazy(sprintf "02-00%d02-01%d02-02%d02-03%d02-04%d02-05%d02-06%d02-07%d02-08%a12-10%d12-11%d12-12%d12-13%d12-14%d12-15%d12-16%d12-17%d12-18%a22-20%d22-21%d22-22%d22-23%d22-24%d22-25%d22-26%d22-27%d22-28" 0 1 2 3 4 5 6 7  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 27 )) "02-00002-01102-02202-03302-04402-05502-06602-07702-081X12-101012-111112-121212-131312-141412-151512-161612-171712-182X22-202022-212122-222222-232322-242422-252522-262622-272722-28"
    test "test8609" (lazy(sprintf "02-00%d02-01%d02-02%d02-03%d02-04%d02-05%d02-06%d02-07%d02-08%a12-10%d12-11%d12-12%d12-13%d12-14%d12-15%d12-16%d12-17%d12-18%a22-20%d22-21%d22-22%d22-23%d22-24%d22-25%d22-26%d22-27%d22-28%a_TAIL" 0 1 2 3 4 5 6 7  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 27 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "02-00002-01102-02202-03302-04402-05502-06602-07702-081X12-101012-111112-121212-131312-141412-151512-161612-171712-182X22-202022-212122-222222-232322-242422-252522-262622-272722-28ReadX_TAIL"
    test "test8610" (lazy(sprintf "02-00%d02-01%d02-02%d02-03%d02-04%d02-05%d02-06%d02-07%d02-08%d02-09%a12-10%d12-11%d12-12%d12-13%d12-14%d12-15%d12-16%d12-17%d12-18%d12-19%a22-20%d22-21%d22-22%d22-23%d22-24%d22-25%d22-26%d22-27%d22-28%d22-29" 0 1 2 3 4 5 6 7 8  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17 18  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 27 28 )) "02-00002-01102-02202-03302-04402-05502-06602-07702-08802-091X12-101012-111112-121212-131312-141412-151512-161612-171712-181812-192X22-202022-212122-222222-232322-242422-252522-262622-272722-282822-29"
    test "test8611" (lazy(sprintf "02-00%d02-01%d02-02%d02-03%d02-04%d02-05%d02-06%d02-07%d02-08%d02-09%a12-10%d12-11%d12-12%d12-13%d12-14%d12-15%d12-16%d12-17%d12-18%d12-19%a22-20%d22-21%d22-22%d22-23%d22-24%d22-25%d22-26%d22-27%d22-28%d22-29%a_TAIL" 0 1 2 3 4 5 6 7 8  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17 18  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 27 28 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "02-00002-01102-02202-03302-04402-05502-06602-07702-08802-091X12-101012-111112-121212-131312-141412-151512-161612-171712-181812-192X22-202022-212122-222222-232322-242422-252522-262622-272722-282822-29ReadX_TAIL"
    test "test8612" (lazy(sprintf "03-00%d03-01%a13-10%d13-11%a23-20%d23-21%a33-30%d33-31" 0  (fun _ v -> (string v) + "X") 1  10  (fun _ v -> (string v) + "X") 2  20  (fun _ v -> (string v) + "X") 3  30 )) "03-00003-011X13-101013-112X23-202023-213X33-303033-31"
    test "test8613" (lazy(sprintf "03-00%d03-01%a13-10%d13-11%a23-20%d23-21%a33-30%d33-31%a_TAIL" 0  (fun _ v -> (string v) + "X") 1  10  (fun _ v -> (string v) + "X") 2  20  (fun _ v -> (string v) + "X") 3  30 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "03-00003-011X13-101013-112X23-202023-213X33-303033-31ReadX_TAIL"
    test "test8614" (lazy(sprintf "03-00%d03-01%d03-02%a13-10%d13-11%d13-12%a23-20%d23-21%d23-22%a33-30%d33-31%d33-32" 0 1  (fun _ v -> (string v) + "X") 1  10 11  (fun _ v -> (string v) + "X") 2  20 21  (fun _ v -> (string v) + "X") 3  30 31 )) "03-00003-01103-021X13-101013-111113-122X23-202023-212123-223X33-303033-313133-32"
    test "test8615" (lazy(sprintf "03-00%d03-01%d03-02%a13-10%d13-11%d13-12%a23-20%d23-21%d23-22%a33-30%d33-31%d33-32%a_TAIL" 0 1  (fun _ v -> (string v) + "X") 1  10 11  (fun _ v -> (string v) + "X") 2  20 21  (fun _ v -> (string v) + "X") 3  30 31 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "03-00003-01103-021X13-101013-111113-122X23-202023-212123-223X33-303033-313133-32ReadX_TAIL"
    test "test8616" (lazy(sprintf "03-00%d03-01%d03-02%d03-03%a13-10%d13-11%d13-12%d13-13%a23-20%d23-21%d23-22%d23-23%a33-30%d33-31%d33-32%d33-33" 0 1 2  (fun _ v -> (string v) + "X") 1  10 11 12  (fun _ v -> (string v) + "X") 2  20 21 22  (fun _ v -> (string v) + "X") 3  30 31 32 )) "03-00003-01103-02203-031X13-101013-111113-121213-132X23-202023-212123-222223-233X33-303033-313133-323233-33"
    test "test8617" (lazy(sprintf "03-00%d03-01%d03-02%d03-03%a13-10%d13-11%d13-12%d13-13%a23-20%d23-21%d23-22%d23-23%a33-30%d33-31%d33-32%d33-33%a_TAIL" 0 1 2  (fun _ v -> (string v) + "X") 1  10 11 12  (fun _ v -> (string v) + "X") 2  20 21 22  (fun _ v -> (string v) + "X") 3  30 31 32 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "03-00003-01103-02203-031X13-101013-111113-121213-132X23-202023-212123-222223-233X33-303033-313133-323233-33ReadX_TAIL"
    test "test8618" (lazy(sprintf "03-00%d03-01%d03-02%d03-03%d03-04%a13-10%d13-11%d13-12%d13-13%d13-14%a23-20%d23-21%d23-22%d23-23%d23-24%a33-30%d33-31%d33-32%d33-33%d33-34" 0 1 2 3  (fun _ v -> (string v) + "X") 1  10 11 12 13  (fun _ v -> (string v) + "X") 2  20 21 22 23  (fun _ v -> (string v) + "X") 3  30 31 32 33 )) "03-00003-01103-02203-03303-041X13-101013-111113-121213-131313-142X23-202023-212123-222223-232323-243X33-303033-313133-323233-333333-34"
    test "test8619" (lazy(sprintf "03-00%d03-01%d03-02%d03-03%d03-04%a13-10%d13-11%d13-12%d13-13%d13-14%a23-20%d23-21%d23-22%d23-23%d23-24%a33-30%d33-31%d33-32%d33-33%d33-34%a_TAIL" 0 1 2 3  (fun _ v -> (string v) + "X") 1  10 11 12 13  (fun _ v -> (string v) + "X") 2  20 21 22 23  (fun _ v -> (string v) + "X") 3  30 31 32 33 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "03-00003-01103-02203-03303-041X13-101013-111113-121213-131313-142X23-202023-212123-222223-232323-243X33-303033-313133-323233-333333-34ReadX_TAIL"
    test "test8620" (lazy(sprintf "03-00%d03-01%d03-02%d03-03%d03-04%d03-05%a13-10%d13-11%d13-12%d13-13%d13-14%d13-15%a23-20%d23-21%d23-22%d23-23%d23-24%d23-25%a33-30%d33-31%d33-32%d33-33%d33-34%d33-35" 0 1 2 3 4  (fun _ v -> (string v) + "X") 1  10 11 12 13 14  (fun _ v -> (string v) + "X") 2  20 21 22 23 24  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 )) "03-00003-01103-02203-03303-04403-051X13-101013-111113-121213-131313-141413-152X23-202023-212123-222223-232323-242423-253X33-303033-313133-323233-333333-343433-35"
    test "test8621" (lazy(sprintf "03-00%d03-01%d03-02%d03-03%d03-04%d03-05%a13-10%d13-11%d13-12%d13-13%d13-14%d13-15%a23-20%d23-21%d23-22%d23-23%d23-24%d23-25%a33-30%d33-31%d33-32%d33-33%d33-34%d33-35%a_TAIL" 0 1 2 3 4  (fun _ v -> (string v) + "X") 1  10 11 12 13 14  (fun _ v -> (string v) + "X") 2  20 21 22 23 24  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "03-00003-01103-02203-03303-04403-051X13-101013-111113-121213-131313-141413-152X23-202023-212123-222223-232323-242423-253X33-303033-313133-323233-333333-343433-35ReadX_TAIL"
    test "test8622" (lazy(sprintf "03-00%d03-01%d03-02%d03-03%d03-04%d03-05%d03-06%a13-10%d13-11%d13-12%d13-13%d13-14%d13-15%d13-16%a23-20%d23-21%d23-22%d23-23%d23-24%d23-25%d23-26%a33-30%d33-31%d33-32%d33-33%d33-34%d33-35%d33-36" 0 1 2 3 4 5  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 )) "03-00003-01103-02203-03303-04403-05503-061X13-101013-111113-121213-131313-141413-151513-162X23-202023-212123-222223-232323-242423-252523-263X33-303033-313133-323233-333333-343433-353533-36"
    test "test8623" (lazy(sprintf "03-00%d03-01%d03-02%d03-03%d03-04%d03-05%d03-06%a13-10%d13-11%d13-12%d13-13%d13-14%d13-15%d13-16%a23-20%d23-21%d23-22%d23-23%d23-24%d23-25%d23-26%a33-30%d33-31%d33-32%d33-33%d33-34%d33-35%d33-36%a_TAIL" 0 1 2 3 4 5  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "03-00003-01103-02203-03303-04403-05503-061X13-101013-111113-121213-131313-141413-151513-162X23-202023-212123-222223-232323-242423-252523-263X33-303033-313133-323233-333333-343433-353533-36ReadX_TAIL"
    test "test8624" (lazy(sprintf "03-00%d03-01%d03-02%d03-03%d03-04%d03-05%d03-06%d03-07%a13-10%d13-11%d13-12%d13-13%d13-14%d13-15%d13-16%d13-17%a23-20%d23-21%d23-22%d23-23%d23-24%d23-25%d23-26%d23-27%a33-30%d33-31%d33-32%d33-33%d33-34%d33-35%d33-36%d33-37" 0 1 2 3 4 5 6  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36 )) "03-00003-01103-02203-03303-04403-05503-06603-071X13-101013-111113-121213-131313-141413-151513-161613-172X23-202023-212123-222223-232323-242423-252523-262623-273X33-303033-313133-323233-333333-343433-353533-363633-37"
    test "test8625" (lazy(sprintf "03-00%d03-01%d03-02%d03-03%d03-04%d03-05%d03-06%d03-07%a13-10%d13-11%d13-12%d13-13%d13-14%d13-15%d13-16%d13-17%a23-20%d23-21%d23-22%d23-23%d23-24%d23-25%d23-26%d23-27%a33-30%d33-31%d33-32%d33-33%d33-34%d33-35%d33-36%d33-37%a_TAIL" 0 1 2 3 4 5 6  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "03-00003-01103-02203-03303-04403-05503-06603-071X13-101013-111113-121213-131313-141413-151513-161613-172X23-202023-212123-222223-232323-242423-252523-262623-273X33-303033-313133-323233-333333-343433-353533-363633-37ReadX_TAIL"
    test "test8626" (lazy(sprintf "03-00%d03-01%d03-02%d03-03%d03-04%d03-05%d03-06%d03-07%d03-08%a13-10%d13-11%d13-12%d13-13%d13-14%d13-15%d13-16%d13-17%d13-18%a23-20%d23-21%d23-22%d23-23%d23-24%d23-25%d23-26%d23-27%d23-28%a33-30%d33-31%d33-32%d33-33%d33-34%d33-35%d33-36%d33-37%d33-38" 0 1 2 3 4 5 6 7  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 27  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36 37 )) "03-00003-01103-02203-03303-04403-05503-06603-07703-081X13-101013-111113-121213-131313-141413-151513-161613-171713-182X23-202023-212123-222223-232323-242423-252523-262623-272723-283X33-303033-313133-323233-333333-343433-353533-363633-373733-38"
    test "test8627" (lazy(sprintf "03-00%d03-01%d03-02%d03-03%d03-04%d03-05%d03-06%d03-07%d03-08%a13-10%d13-11%d13-12%d13-13%d13-14%d13-15%d13-16%d13-17%d13-18%a23-20%d23-21%d23-22%d23-23%d23-24%d23-25%d23-26%d23-27%d23-28%a33-30%d33-31%d33-32%d33-33%d33-34%d33-35%d33-36%d33-37%d33-38%a_TAIL" 0 1 2 3 4 5 6 7  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 27  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36 37 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "03-00003-01103-02203-03303-04403-05503-06603-07703-081X13-101013-111113-121213-131313-141413-151513-161613-171713-182X23-202023-212123-222223-232323-242423-252523-262623-272723-283X33-303033-313133-323233-333333-343433-353533-363633-373733-38ReadX_TAIL"
    test "test8628" (lazy(sprintf "03-00%d03-01%d03-02%d03-03%d03-04%d03-05%d03-06%d03-07%d03-08%d03-09%a13-10%d13-11%d13-12%d13-13%d13-14%d13-15%d13-16%d13-17%d13-18%d13-19%a23-20%d23-21%d23-22%d23-23%d23-24%d23-25%d23-26%d23-27%d23-28%d23-29%a33-30%d33-31%d33-32%d33-33%d33-34%d33-35%d33-36%d33-37%d33-38%d33-39" 0 1 2 3 4 5 6 7 8  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17 18  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 27 28  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36 37 38 )) "03-00003-01103-02203-03303-04403-05503-06603-07703-08803-091X13-101013-111113-121213-131313-141413-151513-161613-171713-181813-192X23-202023-212123-222223-232323-242423-252523-262623-272723-282823-293X33-303033-313133-323233-333333-343433-353533-363633-373733-383833-39"
    test "test8629" (lazy(sprintf "03-00%d03-01%d03-02%d03-03%d03-04%d03-05%d03-06%d03-07%d03-08%d03-09%a13-10%d13-11%d13-12%d13-13%d13-14%d13-15%d13-16%d13-17%d13-18%d13-19%a23-20%d23-21%d23-22%d23-23%d23-24%d23-25%d23-26%d23-27%d23-28%d23-29%a33-30%d33-31%d33-32%d33-33%d33-34%d33-35%d33-36%d33-37%d33-38%d33-39%a_TAIL" 0 1 2 3 4 5 6 7 8  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17 18  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 27 28  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36 37 38 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "03-00003-01103-02203-03303-04403-05503-06603-07703-08803-091X13-101013-111113-121213-131313-141413-151513-161613-171713-181813-192X23-202023-212123-222223-232323-242423-252523-262623-272723-282823-293X33-303033-313133-323233-333333-343433-353533-363633-373733-383833-39ReadX_TAIL"
    test "test8630" (lazy(sprintf "04-00%d04-01%a14-10%d14-11%a24-20%d24-21%a34-30%d34-31%a44-40%d44-41" 0  (fun _ v -> (string v) + "X") 1  10  (fun _ v -> (string v) + "X") 2  20  (fun _ v -> (string v) + "X") 3  30  (fun _ v -> (string v) + "X") 4  40 )) "04-00004-011X14-101014-112X24-202024-213X34-303034-314X44-404044-41"
    test "test8631" (lazy(sprintf "04-00%d04-01%a14-10%d14-11%a24-20%d24-21%a34-30%d34-31%a44-40%d44-41%a_TAIL" 0  (fun _ v -> (string v) + "X") 1  10  (fun _ v -> (string v) + "X") 2  20  (fun _ v -> (string v) + "X") 3  30  (fun _ v -> (string v) + "X") 4  40 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "04-00004-011X14-101014-112X24-202024-213X34-303034-314X44-404044-41ReadX_TAIL"
    test "test8632" (lazy(sprintf "04-00%d04-01%d04-02%a14-10%d14-11%d14-12%a24-20%d24-21%d24-22%a34-30%d34-31%d34-32%a44-40%d44-41%d44-42" 0 1  (fun _ v -> (string v) + "X") 1  10 11  (fun _ v -> (string v) + "X") 2  20 21  (fun _ v -> (string v) + "X") 3  30 31  (fun _ v -> (string v) + "X") 4  40 41 )) "04-00004-01104-021X14-101014-111114-122X24-202024-212124-223X34-303034-313134-324X44-404044-414144-42"
    test "test8633" (lazy(sprintf "04-00%d04-01%d04-02%a14-10%d14-11%d14-12%a24-20%d24-21%d24-22%a34-30%d34-31%d34-32%a44-40%d44-41%d44-42%a_TAIL" 0 1  (fun _ v -> (string v) + "X") 1  10 11  (fun _ v -> (string v) + "X") 2  20 21  (fun _ v -> (string v) + "X") 3  30 31  (fun _ v -> (string v) + "X") 4  40 41 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "04-00004-01104-021X14-101014-111114-122X24-202024-212124-223X34-303034-313134-324X44-404044-414144-42ReadX_TAIL"
    test "test8634" (lazy(sprintf "04-00%d04-01%d04-02%d04-03%a14-10%d14-11%d14-12%d14-13%a24-20%d24-21%d24-22%d24-23%a34-30%d34-31%d34-32%d34-33%a44-40%d44-41%d44-42%d44-43" 0 1 2  (fun _ v -> (string v) + "X") 1  10 11 12  (fun _ v -> (string v) + "X") 2  20 21 22  (fun _ v -> (string v) + "X") 3  30 31 32  (fun _ v -> (string v) + "X") 4  40 41 42 )) "04-00004-01104-02204-031X14-101014-111114-121214-132X24-202024-212124-222224-233X34-303034-313134-323234-334X44-404044-414144-424244-43"
    test "test8635" (lazy(sprintf "04-00%d04-01%d04-02%d04-03%a14-10%d14-11%d14-12%d14-13%a24-20%d24-21%d24-22%d24-23%a34-30%d34-31%d34-32%d34-33%a44-40%d44-41%d44-42%d44-43%a_TAIL" 0 1 2  (fun _ v -> (string v) + "X") 1  10 11 12  (fun _ v -> (string v) + "X") 2  20 21 22  (fun _ v -> (string v) + "X") 3  30 31 32  (fun _ v -> (string v) + "X") 4  40 41 42 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "04-00004-01104-02204-031X14-101014-111114-121214-132X24-202024-212124-222224-233X34-303034-313134-323234-334X44-404044-414144-424244-43ReadX_TAIL"
    test "test8636" (lazy(sprintf "04-00%d04-01%d04-02%d04-03%d04-04%a14-10%d14-11%d14-12%d14-13%d14-14%a24-20%d24-21%d24-22%d24-23%d24-24%a34-30%d34-31%d34-32%d34-33%d34-34%a44-40%d44-41%d44-42%d44-43%d44-44" 0 1 2 3  (fun _ v -> (string v) + "X") 1  10 11 12 13  (fun _ v -> (string v) + "X") 2  20 21 22 23  (fun _ v -> (string v) + "X") 3  30 31 32 33  (fun _ v -> (string v) + "X") 4  40 41 42 43 )) "04-00004-01104-02204-03304-041X14-101014-111114-121214-131314-142X24-202024-212124-222224-232324-243X34-303034-313134-323234-333334-344X44-404044-414144-424244-434344-44"
    test "test8637" (lazy(sprintf "04-00%d04-01%d04-02%d04-03%d04-04%a14-10%d14-11%d14-12%d14-13%d14-14%a24-20%d24-21%d24-22%d24-23%d24-24%a34-30%d34-31%d34-32%d34-33%d34-34%a44-40%d44-41%d44-42%d44-43%d44-44%a_TAIL" 0 1 2 3  (fun _ v -> (string v) + "X") 1  10 11 12 13  (fun _ v -> (string v) + "X") 2  20 21 22 23  (fun _ v -> (string v) + "X") 3  30 31 32 33  (fun _ v -> (string v) + "X") 4  40 41 42 43 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "04-00004-01104-02204-03304-041X14-101014-111114-121214-131314-142X24-202024-212124-222224-232324-243X34-303034-313134-323234-333334-344X44-404044-414144-424244-434344-44ReadX_TAIL"
    test "test8638" (lazy(sprintf "04-00%d04-01%d04-02%d04-03%d04-04%d04-05%a14-10%d14-11%d14-12%d14-13%d14-14%d14-15%a24-20%d24-21%d24-22%d24-23%d24-24%d24-25%a34-30%d34-31%d34-32%d34-33%d34-34%d34-35%a44-40%d44-41%d44-42%d44-43%d44-44%d44-45" 0 1 2 3 4  (fun _ v -> (string v) + "X") 1  10 11 12 13 14  (fun _ v -> (string v) + "X") 2  20 21 22 23 24  (fun _ v -> (string v) + "X") 3  30 31 32 33 34  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 )) "04-00004-01104-02204-03304-04404-051X14-101014-111114-121214-131314-141414-152X24-202024-212124-222224-232324-242424-253X34-303034-313134-323234-333334-343434-354X44-404044-414144-424244-434344-444444-45"
    test "test8639" (lazy(sprintf "04-00%d04-01%d04-02%d04-03%d04-04%d04-05%a14-10%d14-11%d14-12%d14-13%d14-14%d14-15%a24-20%d24-21%d24-22%d24-23%d24-24%d24-25%a34-30%d34-31%d34-32%d34-33%d34-34%d34-35%a44-40%d44-41%d44-42%d44-43%d44-44%d44-45%a_TAIL" 0 1 2 3 4  (fun _ v -> (string v) + "X") 1  10 11 12 13 14  (fun _ v -> (string v) + "X") 2  20 21 22 23 24  (fun _ v -> (string v) + "X") 3  30 31 32 33 34  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "04-00004-01104-02204-03304-04404-051X14-101014-111114-121214-131314-141414-152X24-202024-212124-222224-232324-242424-253X34-303034-313134-323234-333334-343434-354X44-404044-414144-424244-434344-444444-45ReadX_TAIL"
    test "test8640" (lazy(sprintf "04-00%d04-01%d04-02%d04-03%d04-04%d04-05%d04-06%a14-10%d14-11%d14-12%d14-13%d14-14%d14-15%d14-16%a24-20%d24-21%d24-22%d24-23%d24-24%d24-25%d24-26%a34-30%d34-31%d34-32%d34-33%d34-34%d34-35%d34-36%a44-40%d44-41%d44-42%d44-43%d44-44%d44-45%d44-46" 0 1 2 3 4 5  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 )) "04-00004-01104-02204-03304-04404-05504-061X14-101014-111114-121214-131314-141414-151514-162X24-202024-212124-222224-232324-242424-252524-263X34-303034-313134-323234-333334-343434-353534-364X44-404044-414144-424244-434344-444444-454544-46"
    test "test8641" (lazy(sprintf "04-00%d04-01%d04-02%d04-03%d04-04%d04-05%d04-06%a14-10%d14-11%d14-12%d14-13%d14-14%d14-15%d14-16%a24-20%d24-21%d24-22%d24-23%d24-24%d24-25%d24-26%a34-30%d34-31%d34-32%d34-33%d34-34%d34-35%d34-36%a44-40%d44-41%d44-42%d44-43%d44-44%d44-45%d44-46%a_TAIL" 0 1 2 3 4 5  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "04-00004-01104-02204-03304-04404-05504-061X14-101014-111114-121214-131314-141414-151514-162X24-202024-212124-222224-232324-242424-252524-263X34-303034-313134-323234-333334-343434-353534-364X44-404044-414144-424244-434344-444444-454544-46ReadX_TAIL"
    test "test8642" (lazy(sprintf "04-00%d04-01%d04-02%d04-03%d04-04%d04-05%d04-06%d04-07%a14-10%d14-11%d14-12%d14-13%d14-14%d14-15%d14-16%d14-17%a24-20%d24-21%d24-22%d24-23%d24-24%d24-25%d24-26%d24-27%a34-30%d34-31%d34-32%d34-33%d34-34%d34-35%d34-36%d34-37%a44-40%d44-41%d44-42%d44-43%d44-44%d44-45%d44-46%d44-47" 0 1 2 3 4 5 6  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46 )) "04-00004-01104-02204-03304-04404-05504-06604-071X14-101014-111114-121214-131314-141414-151514-161614-172X24-202024-212124-222224-232324-242424-252524-262624-273X34-303034-313134-323234-333334-343434-353534-363634-374X44-404044-414144-424244-434344-444444-454544-464644-47"
    test "test8643" (lazy(sprintf "04-00%d04-01%d04-02%d04-03%d04-04%d04-05%d04-06%d04-07%a14-10%d14-11%d14-12%d14-13%d14-14%d14-15%d14-16%d14-17%a24-20%d24-21%d24-22%d24-23%d24-24%d24-25%d24-26%d24-27%a34-30%d34-31%d34-32%d34-33%d34-34%d34-35%d34-36%d34-37%a44-40%d44-41%d44-42%d44-43%d44-44%d44-45%d44-46%d44-47%a_TAIL" 0 1 2 3 4 5 6  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "04-00004-01104-02204-03304-04404-05504-06604-071X14-101014-111114-121214-131314-141414-151514-161614-172X24-202024-212124-222224-232324-242424-252524-262624-273X34-303034-313134-323234-333334-343434-353534-363634-374X44-404044-414144-424244-434344-444444-454544-464644-47ReadX_TAIL"
    test "test8644" (lazy(sprintf "04-00%d04-01%d04-02%d04-03%d04-04%d04-05%d04-06%d04-07%d04-08%a14-10%d14-11%d14-12%d14-13%d14-14%d14-15%d14-16%d14-17%d14-18%a24-20%d24-21%d24-22%d24-23%d24-24%d24-25%d24-26%d24-27%d24-28%a34-30%d34-31%d34-32%d34-33%d34-34%d34-35%d34-36%d34-37%d34-38%a44-40%d44-41%d44-42%d44-43%d44-44%d44-45%d44-46%d44-47%d44-48" 0 1 2 3 4 5 6 7  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 27  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36 37  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46 47 )) "04-00004-01104-02204-03304-04404-05504-06604-07704-081X14-101014-111114-121214-131314-141414-151514-161614-171714-182X24-202024-212124-222224-232324-242424-252524-262624-272724-283X34-303034-313134-323234-333334-343434-353534-363634-373734-384X44-404044-414144-424244-434344-444444-454544-464644-474744-48"
    test "test8645" (lazy(sprintf "04-00%d04-01%d04-02%d04-03%d04-04%d04-05%d04-06%d04-07%d04-08%a14-10%d14-11%d14-12%d14-13%d14-14%d14-15%d14-16%d14-17%d14-18%a24-20%d24-21%d24-22%d24-23%d24-24%d24-25%d24-26%d24-27%d24-28%a34-30%d34-31%d34-32%d34-33%d34-34%d34-35%d34-36%d34-37%d34-38%a44-40%d44-41%d44-42%d44-43%d44-44%d44-45%d44-46%d44-47%d44-48%a_TAIL" 0 1 2 3 4 5 6 7  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 27  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36 37  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46 47 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "04-00004-01104-02204-03304-04404-05504-06604-07704-081X14-101014-111114-121214-131314-141414-151514-161614-171714-182X24-202024-212124-222224-232324-242424-252524-262624-272724-283X34-303034-313134-323234-333334-343434-353534-363634-373734-384X44-404044-414144-424244-434344-444444-454544-464644-474744-48ReadX_TAIL"
    test "test8646" (lazy(sprintf "04-00%d04-01%d04-02%d04-03%d04-04%d04-05%d04-06%d04-07%d04-08%d04-09%a14-10%d14-11%d14-12%d14-13%d14-14%d14-15%d14-16%d14-17%d14-18%d14-19%a24-20%d24-21%d24-22%d24-23%d24-24%d24-25%d24-26%d24-27%d24-28%d24-29%a34-30%d34-31%d34-32%d34-33%d34-34%d34-35%d34-36%d34-37%d34-38%d34-39%a44-40%d44-41%d44-42%d44-43%d44-44%d44-45%d44-46%d44-47%d44-48%d44-49" 0 1 2 3 4 5 6 7 8  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17 18  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 27 28  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36 37 38  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46 47 48 )) "04-00004-01104-02204-03304-04404-05504-06604-07704-08804-091X14-101014-111114-121214-131314-141414-151514-161614-171714-181814-192X24-202024-212124-222224-232324-242424-252524-262624-272724-282824-293X34-303034-313134-323234-333334-343434-353534-363634-373734-383834-394X44-404044-414144-424244-434344-444444-454544-464644-474744-484844-49"
    test "test8647" (lazy(sprintf "04-00%d04-01%d04-02%d04-03%d04-04%d04-05%d04-06%d04-07%d04-08%d04-09%a14-10%d14-11%d14-12%d14-13%d14-14%d14-15%d14-16%d14-17%d14-18%d14-19%a24-20%d24-21%d24-22%d24-23%d24-24%d24-25%d24-26%d24-27%d24-28%d24-29%a34-30%d34-31%d34-32%d34-33%d34-34%d34-35%d34-36%d34-37%d34-38%d34-39%a44-40%d44-41%d44-42%d44-43%d44-44%d44-45%d44-46%d44-47%d44-48%d44-49%a_TAIL" 0 1 2 3 4 5 6 7 8  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17 18  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 27 28  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36 37 38  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46 47 48 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "04-00004-01104-02204-03304-04404-05504-06604-07704-08804-091X14-101014-111114-121214-131314-141414-151514-161614-171714-181814-192X24-202024-212124-222224-232324-242424-252524-262624-272724-282824-293X34-303034-313134-323234-333334-343434-353534-363634-373734-383834-394X44-404044-414144-424244-434344-444444-454544-464644-474744-484844-49ReadX_TAIL"
    test "test8648" (lazy(sprintf "05-00%d05-01%a15-10%d15-11%a25-20%d25-21%a35-30%d35-31%a45-40%d45-41%a55-50%d55-51" 0  (fun _ v -> (string v) + "X") 1  10  (fun _ v -> (string v) + "X") 2  20  (fun _ v -> (string v) + "X") 3  30  (fun _ v -> (string v) + "X") 4  40  (fun _ v -> (string v) + "X") 5  50 )) "05-00005-011X15-101015-112X25-202025-213X35-303035-314X45-404045-415X55-505055-51"
    test "test8649" (lazy(sprintf "05-00%d05-01%a15-10%d15-11%a25-20%d25-21%a35-30%d35-31%a45-40%d45-41%a55-50%d55-51%a_TAIL" 0  (fun _ v -> (string v) + "X") 1  10  (fun _ v -> (string v) + "X") 2  20  (fun _ v -> (string v) + "X") 3  30  (fun _ v -> (string v) + "X") 4  40  (fun _ v -> (string v) + "X") 5  50 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "05-00005-011X15-101015-112X25-202025-213X35-303035-314X45-404045-415X55-505055-51ReadX_TAIL"
    test "test8650" (lazy(sprintf "05-00%d05-01%d05-02%a15-10%d15-11%d15-12%a25-20%d25-21%d25-22%a35-30%d35-31%d35-32%a45-40%d45-41%d45-42%a55-50%d55-51%d55-52" 0 1  (fun _ v -> (string v) + "X") 1  10 11  (fun _ v -> (string v) + "X") 2  20 21  (fun _ v -> (string v) + "X") 3  30 31  (fun _ v -> (string v) + "X") 4  40 41  (fun _ v -> (string v) + "X") 5  50 51 )) "05-00005-01105-021X15-101015-111115-122X25-202025-212125-223X35-303035-313135-324X45-404045-414145-425X55-505055-515155-52"
    test "test8651" (lazy(sprintf "05-00%d05-01%d05-02%a15-10%d15-11%d15-12%a25-20%d25-21%d25-22%a35-30%d35-31%d35-32%a45-40%d45-41%d45-42%a55-50%d55-51%d55-52%a_TAIL" 0 1  (fun _ v -> (string v) + "X") 1  10 11  (fun _ v -> (string v) + "X") 2  20 21  (fun _ v -> (string v) + "X") 3  30 31  (fun _ v -> (string v) + "X") 4  40 41  (fun _ v -> (string v) + "X") 5  50 51 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "05-00005-01105-021X15-101015-111115-122X25-202025-212125-223X35-303035-313135-324X45-404045-414145-425X55-505055-515155-52ReadX_TAIL"
    test "test8652" (lazy(sprintf "05-00%d05-01%d05-02%d05-03%a15-10%d15-11%d15-12%d15-13%a25-20%d25-21%d25-22%d25-23%a35-30%d35-31%d35-32%d35-33%a45-40%d45-41%d45-42%d45-43%a55-50%d55-51%d55-52%d55-53" 0 1 2  (fun _ v -> (string v) + "X") 1  10 11 12  (fun _ v -> (string v) + "X") 2  20 21 22  (fun _ v -> (string v) + "X") 3  30 31 32  (fun _ v -> (string v) + "X") 4  40 41 42  (fun _ v -> (string v) + "X") 5  50 51 52 )) "05-00005-01105-02205-031X15-101015-111115-121215-132X25-202025-212125-222225-233X35-303035-313135-323235-334X45-404045-414145-424245-435X55-505055-515155-525255-53"
    test "test8653" (lazy(sprintf "05-00%d05-01%d05-02%d05-03%a15-10%d15-11%d15-12%d15-13%a25-20%d25-21%d25-22%d25-23%a35-30%d35-31%d35-32%d35-33%a45-40%d45-41%d45-42%d45-43%a55-50%d55-51%d55-52%d55-53%a_TAIL" 0 1 2  (fun _ v -> (string v) + "X") 1  10 11 12  (fun _ v -> (string v) + "X") 2  20 21 22  (fun _ v -> (string v) + "X") 3  30 31 32  (fun _ v -> (string v) + "X") 4  40 41 42  (fun _ v -> (string v) + "X") 5  50 51 52 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "05-00005-01105-02205-031X15-101015-111115-121215-132X25-202025-212125-222225-233X35-303035-313135-323235-334X45-404045-414145-424245-435X55-505055-515155-525255-53ReadX_TAIL"
    test "test8654" (lazy(sprintf "05-00%d05-01%d05-02%d05-03%d05-04%a15-10%d15-11%d15-12%d15-13%d15-14%a25-20%d25-21%d25-22%d25-23%d25-24%a35-30%d35-31%d35-32%d35-33%d35-34%a45-40%d45-41%d45-42%d45-43%d45-44%a55-50%d55-51%d55-52%d55-53%d55-54" 0 1 2 3  (fun _ v -> (string v) + "X") 1  10 11 12 13  (fun _ v -> (string v) + "X") 2  20 21 22 23  (fun _ v -> (string v) + "X") 3  30 31 32 33  (fun _ v -> (string v) + "X") 4  40 41 42 43  (fun _ v -> (string v) + "X") 5  50 51 52 53 )) "05-00005-01105-02205-03305-041X15-101015-111115-121215-131315-142X25-202025-212125-222225-232325-243X35-303035-313135-323235-333335-344X45-404045-414145-424245-434345-445X55-505055-515155-525255-535355-54"
    test "test8655" (lazy(sprintf "05-00%d05-01%d05-02%d05-03%d05-04%a15-10%d15-11%d15-12%d15-13%d15-14%a25-20%d25-21%d25-22%d25-23%d25-24%a35-30%d35-31%d35-32%d35-33%d35-34%a45-40%d45-41%d45-42%d45-43%d45-44%a55-50%d55-51%d55-52%d55-53%d55-54%a_TAIL" 0 1 2 3  (fun _ v -> (string v) + "X") 1  10 11 12 13  (fun _ v -> (string v) + "X") 2  20 21 22 23  (fun _ v -> (string v) + "X") 3  30 31 32 33  (fun _ v -> (string v) + "X") 4  40 41 42 43  (fun _ v -> (string v) + "X") 5  50 51 52 53 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "05-00005-01105-02205-03305-041X15-101015-111115-121215-131315-142X25-202025-212125-222225-232325-243X35-303035-313135-323235-333335-344X45-404045-414145-424245-434345-445X55-505055-515155-525255-535355-54ReadX_TAIL"
    test "test8656" (lazy(sprintf "05-00%d05-01%d05-02%d05-03%d05-04%d05-05%a15-10%d15-11%d15-12%d15-13%d15-14%d15-15%a25-20%d25-21%d25-22%d25-23%d25-24%d25-25%a35-30%d35-31%d35-32%d35-33%d35-34%d35-35%a45-40%d45-41%d45-42%d45-43%d45-44%d45-45%a55-50%d55-51%d55-52%d55-53%d55-54%d55-55" 0 1 2 3 4  (fun _ v -> (string v) + "X") 1  10 11 12 13 14  (fun _ v -> (string v) + "X") 2  20 21 22 23 24  (fun _ v -> (string v) + "X") 3  30 31 32 33 34  (fun _ v -> (string v) + "X") 4  40 41 42 43 44  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 )) "05-00005-01105-02205-03305-04405-051X15-101015-111115-121215-131315-141415-152X25-202025-212125-222225-232325-242425-253X35-303035-313135-323235-333335-343435-354X45-404045-414145-424245-434345-444445-455X55-505055-515155-525255-535355-545455-55"
    test "test8657" (lazy(sprintf "05-00%d05-01%d05-02%d05-03%d05-04%d05-05%a15-10%d15-11%d15-12%d15-13%d15-14%d15-15%a25-20%d25-21%d25-22%d25-23%d25-24%d25-25%a35-30%d35-31%d35-32%d35-33%d35-34%d35-35%a45-40%d45-41%d45-42%d45-43%d45-44%d45-45%a55-50%d55-51%d55-52%d55-53%d55-54%d55-55%a_TAIL" 0 1 2 3 4  (fun _ v -> (string v) + "X") 1  10 11 12 13 14  (fun _ v -> (string v) + "X") 2  20 21 22 23 24  (fun _ v -> (string v) + "X") 3  30 31 32 33 34  (fun _ v -> (string v) + "X") 4  40 41 42 43 44  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "05-00005-01105-02205-03305-04405-051X15-101015-111115-121215-131315-141415-152X25-202025-212125-222225-232325-242425-253X35-303035-313135-323235-333335-343435-354X45-404045-414145-424245-434345-444445-455X55-505055-515155-525255-535355-545455-55ReadX_TAIL"
    test "test8658" (lazy(sprintf "05-00%d05-01%d05-02%d05-03%d05-04%d05-05%d05-06%a15-10%d15-11%d15-12%d15-13%d15-14%d15-15%d15-16%a25-20%d25-21%d25-22%d25-23%d25-24%d25-25%d25-26%a35-30%d35-31%d35-32%d35-33%d35-34%d35-35%d35-36%a45-40%d45-41%d45-42%d45-43%d45-44%d45-45%d45-46%a55-50%d55-51%d55-52%d55-53%d55-54%d55-55%d55-56" 0 1 2 3 4 5  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55 )) "05-00005-01105-02205-03305-04405-05505-061X15-101015-111115-121215-131315-141415-151515-162X25-202025-212125-222225-232325-242425-252525-263X35-303035-313135-323235-333335-343435-353535-364X45-404045-414145-424245-434345-444445-454545-465X55-505055-515155-525255-535355-545455-555555-56"
    test "test8659" (lazy(sprintf "05-00%d05-01%d05-02%d05-03%d05-04%d05-05%d05-06%a15-10%d15-11%d15-12%d15-13%d15-14%d15-15%d15-16%a25-20%d25-21%d25-22%d25-23%d25-24%d25-25%d25-26%a35-30%d35-31%d35-32%d35-33%d35-34%d35-35%d35-36%a45-40%d45-41%d45-42%d45-43%d45-44%d45-45%d45-46%a55-50%d55-51%d55-52%d55-53%d55-54%d55-55%d55-56%a_TAIL" 0 1 2 3 4 5  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "05-00005-01105-02205-03305-04405-05505-061X15-101015-111115-121215-131315-141415-151515-162X25-202025-212125-222225-232325-242425-252525-263X35-303035-313135-323235-333335-343435-353535-364X45-404045-414145-424245-434345-444445-454545-465X55-505055-515155-525255-535355-545455-555555-56ReadX_TAIL"
    test "test8660" (lazy(sprintf "05-00%d05-01%d05-02%d05-03%d05-04%d05-05%d05-06%d05-07%a15-10%d15-11%d15-12%d15-13%d15-14%d15-15%d15-16%d15-17%a25-20%d25-21%d25-22%d25-23%d25-24%d25-25%d25-26%d25-27%a35-30%d35-31%d35-32%d35-33%d35-34%d35-35%d35-36%d35-37%a45-40%d45-41%d45-42%d45-43%d45-44%d45-45%d45-46%d45-47%a55-50%d55-51%d55-52%d55-53%d55-54%d55-55%d55-56%d55-57" 0 1 2 3 4 5 6  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55 56 )) "05-00005-01105-02205-03305-04405-05505-06605-071X15-101015-111115-121215-131315-141415-151515-161615-172X25-202025-212125-222225-232325-242425-252525-262625-273X35-303035-313135-323235-333335-343435-353535-363635-374X45-404045-414145-424245-434345-444445-454545-464645-475X55-505055-515155-525255-535355-545455-555555-565655-57"
    test "test8661" (lazy(sprintf "05-00%d05-01%d05-02%d05-03%d05-04%d05-05%d05-06%d05-07%a15-10%d15-11%d15-12%d15-13%d15-14%d15-15%d15-16%d15-17%a25-20%d25-21%d25-22%d25-23%d25-24%d25-25%d25-26%d25-27%a35-30%d35-31%d35-32%d35-33%d35-34%d35-35%d35-36%d35-37%a45-40%d45-41%d45-42%d45-43%d45-44%d45-45%d45-46%d45-47%a55-50%d55-51%d55-52%d55-53%d55-54%d55-55%d55-56%d55-57%a_TAIL" 0 1 2 3 4 5 6  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55 56 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "05-00005-01105-02205-03305-04405-05505-06605-071X15-101015-111115-121215-131315-141415-151515-161615-172X25-202025-212125-222225-232325-242425-252525-262625-273X35-303035-313135-323235-333335-343435-353535-363635-374X45-404045-414145-424245-434345-444445-454545-464645-475X55-505055-515155-525255-535355-545455-555555-565655-57ReadX_TAIL"
    test "test8662" (lazy(sprintf "05-00%d05-01%d05-02%d05-03%d05-04%d05-05%d05-06%d05-07%d05-08%a15-10%d15-11%d15-12%d15-13%d15-14%d15-15%d15-16%d15-17%d15-18%a25-20%d25-21%d25-22%d25-23%d25-24%d25-25%d25-26%d25-27%d25-28%a35-30%d35-31%d35-32%d35-33%d35-34%d35-35%d35-36%d35-37%d35-38%a45-40%d45-41%d45-42%d45-43%d45-44%d45-45%d45-46%d45-47%d45-48%a55-50%d55-51%d55-52%d55-53%d55-54%d55-55%d55-56%d55-57%d55-58" 0 1 2 3 4 5 6 7  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 27  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36 37  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46 47  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55 56 57 )) "05-00005-01105-02205-03305-04405-05505-06605-07705-081X15-101015-111115-121215-131315-141415-151515-161615-171715-182X25-202025-212125-222225-232325-242425-252525-262625-272725-283X35-303035-313135-323235-333335-343435-353535-363635-373735-384X45-404045-414145-424245-434345-444445-454545-464645-474745-485X55-505055-515155-525255-535355-545455-555555-565655-575755-58"
    test "test8663" (lazy(sprintf "05-00%d05-01%d05-02%d05-03%d05-04%d05-05%d05-06%d05-07%d05-08%a15-10%d15-11%d15-12%d15-13%d15-14%d15-15%d15-16%d15-17%d15-18%a25-20%d25-21%d25-22%d25-23%d25-24%d25-25%d25-26%d25-27%d25-28%a35-30%d35-31%d35-32%d35-33%d35-34%d35-35%d35-36%d35-37%d35-38%a45-40%d45-41%d45-42%d45-43%d45-44%d45-45%d45-46%d45-47%d45-48%a55-50%d55-51%d55-52%d55-53%d55-54%d55-55%d55-56%d55-57%d55-58%a_TAIL" 0 1 2 3 4 5 6 7  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 27  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36 37  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46 47  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55 56 57 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "05-00005-01105-02205-03305-04405-05505-06605-07705-081X15-101015-111115-121215-131315-141415-151515-161615-171715-182X25-202025-212125-222225-232325-242425-252525-262625-272725-283X35-303035-313135-323235-333335-343435-353535-363635-373735-384X45-404045-414145-424245-434345-444445-454545-464645-474745-485X55-505055-515155-525255-535355-545455-555555-565655-575755-58ReadX_TAIL"
    test "test8664" (lazy(sprintf "05-00%d05-01%d05-02%d05-03%d05-04%d05-05%d05-06%d05-07%d05-08%d05-09%a15-10%d15-11%d15-12%d15-13%d15-14%d15-15%d15-16%d15-17%d15-18%d15-19%a25-20%d25-21%d25-22%d25-23%d25-24%d25-25%d25-26%d25-27%d25-28%d25-29%a35-30%d35-31%d35-32%d35-33%d35-34%d35-35%d35-36%d35-37%d35-38%d35-39%a45-40%d45-41%d45-42%d45-43%d45-44%d45-45%d45-46%d45-47%d45-48%d45-49%a55-50%d55-51%d55-52%d55-53%d55-54%d55-55%d55-56%d55-57%d55-58%d55-59" 0 1 2 3 4 5 6 7 8  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17 18  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 27 28  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36 37 38  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46 47 48  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55 56 57 58 )) "05-00005-01105-02205-03305-04405-05505-06605-07705-08805-091X15-101015-111115-121215-131315-141415-151515-161615-171715-181815-192X25-202025-212125-222225-232325-242425-252525-262625-272725-282825-293X35-303035-313135-323235-333335-343435-353535-363635-373735-383835-394X45-404045-414145-424245-434345-444445-454545-464645-474745-484845-495X55-505055-515155-525255-535355-545455-555555-565655-575755-585855-59"
    test "test8665" (lazy(sprintf "05-00%d05-01%d05-02%d05-03%d05-04%d05-05%d05-06%d05-07%d05-08%d05-09%a15-10%d15-11%d15-12%d15-13%d15-14%d15-15%d15-16%d15-17%d15-18%d15-19%a25-20%d25-21%d25-22%d25-23%d25-24%d25-25%d25-26%d25-27%d25-28%d25-29%a35-30%d35-31%d35-32%d35-33%d35-34%d35-35%d35-36%d35-37%d35-38%d35-39%a45-40%d45-41%d45-42%d45-43%d45-44%d45-45%d45-46%d45-47%d45-48%d45-49%a55-50%d55-51%d55-52%d55-53%d55-54%d55-55%d55-56%d55-57%d55-58%d55-59%a_TAIL" 0 1 2 3 4 5 6 7 8  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17 18  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 27 28  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36 37 38  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46 47 48  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55 56 57 58 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "05-00005-01105-02205-03305-04405-05505-06605-07705-08805-091X15-101015-111115-121215-131315-141415-151515-161615-171715-181815-192X25-202025-212125-222225-232325-242425-252525-262625-272725-282825-293X35-303035-313135-323235-333335-343435-353535-363635-373735-383835-394X45-404045-414145-424245-434345-444445-454545-464645-474745-484845-495X55-505055-515155-525255-535355-545455-555555-565655-575755-585855-59ReadX_TAIL"
    test "test8666" (lazy(sprintf "06-00%d06-01%a16-10%d16-11%a26-20%d26-21%a36-30%d36-31%a46-40%d46-41%a56-50%d56-51%a66-60%d66-61" 0  (fun _ v -> (string v) + "X") 1  10  (fun _ v -> (string v) + "X") 2  20  (fun _ v -> (string v) + "X") 3  30  (fun _ v -> (string v) + "X") 4  40  (fun _ v -> (string v) + "X") 5  50  (fun _ v -> (string v) + "X") 6  60 )) "06-00006-011X16-101016-112X26-202026-213X36-303036-314X46-404046-415X56-505056-516X66-606066-61"
    test "test8667" (lazy(sprintf "06-00%d06-01%a16-10%d16-11%a26-20%d26-21%a36-30%d36-31%a46-40%d46-41%a56-50%d56-51%a66-60%d66-61%a_TAIL" 0  (fun _ v -> (string v) + "X") 1  10  (fun _ v -> (string v) + "X") 2  20  (fun _ v -> (string v) + "X") 3  30  (fun _ v -> (string v) + "X") 4  40  (fun _ v -> (string v) + "X") 5  50  (fun _ v -> (string v) + "X") 6  60 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "06-00006-011X16-101016-112X26-202026-213X36-303036-314X46-404046-415X56-505056-516X66-606066-61ReadX_TAIL"
    test "test8668" (lazy(sprintf "06-00%d06-01%d06-02%a16-10%d16-11%d16-12%a26-20%d26-21%d26-22%a36-30%d36-31%d36-32%a46-40%d46-41%d46-42%a56-50%d56-51%d56-52%a66-60%d66-61%d66-62" 0 1  (fun _ v -> (string v) + "X") 1  10 11  (fun _ v -> (string v) + "X") 2  20 21  (fun _ v -> (string v) + "X") 3  30 31  (fun _ v -> (string v) + "X") 4  40 41  (fun _ v -> (string v) + "X") 5  50 51  (fun _ v -> (string v) + "X") 6  60 61 )) "06-00006-01106-021X16-101016-111116-122X26-202026-212126-223X36-303036-313136-324X46-404046-414146-425X56-505056-515156-526X66-606066-616166-62"
    test "test8669" (lazy(sprintf "06-00%d06-01%d06-02%a16-10%d16-11%d16-12%a26-20%d26-21%d26-22%a36-30%d36-31%d36-32%a46-40%d46-41%d46-42%a56-50%d56-51%d56-52%a66-60%d66-61%d66-62%a_TAIL" 0 1  (fun _ v -> (string v) + "X") 1  10 11  (fun _ v -> (string v) + "X") 2  20 21  (fun _ v -> (string v) + "X") 3  30 31  (fun _ v -> (string v) + "X") 4  40 41  (fun _ v -> (string v) + "X") 5  50 51  (fun _ v -> (string v) + "X") 6  60 61 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "06-00006-01106-021X16-101016-111116-122X26-202026-212126-223X36-303036-313136-324X46-404046-414146-425X56-505056-515156-526X66-606066-616166-62ReadX_TAIL"
    test "test8670" (lazy(sprintf "06-00%d06-01%d06-02%d06-03%a16-10%d16-11%d16-12%d16-13%a26-20%d26-21%d26-22%d26-23%a36-30%d36-31%d36-32%d36-33%a46-40%d46-41%d46-42%d46-43%a56-50%d56-51%d56-52%d56-53%a66-60%d66-61%d66-62%d66-63" 0 1 2  (fun _ v -> (string v) + "X") 1  10 11 12  (fun _ v -> (string v) + "X") 2  20 21 22  (fun _ v -> (string v) + "X") 3  30 31 32  (fun _ v -> (string v) + "X") 4  40 41 42  (fun _ v -> (string v) + "X") 5  50 51 52  (fun _ v -> (string v) + "X") 6  60 61 62 )) "06-00006-01106-02206-031X16-101016-111116-121216-132X26-202026-212126-222226-233X36-303036-313136-323236-334X46-404046-414146-424246-435X56-505056-515156-525256-536X66-606066-616166-626266-63"
    test "test8671" (lazy(sprintf "06-00%d06-01%d06-02%d06-03%a16-10%d16-11%d16-12%d16-13%a26-20%d26-21%d26-22%d26-23%a36-30%d36-31%d36-32%d36-33%a46-40%d46-41%d46-42%d46-43%a56-50%d56-51%d56-52%d56-53%a66-60%d66-61%d66-62%d66-63%a_TAIL" 0 1 2  (fun _ v -> (string v) + "X") 1  10 11 12  (fun _ v -> (string v) + "X") 2  20 21 22  (fun _ v -> (string v) + "X") 3  30 31 32  (fun _ v -> (string v) + "X") 4  40 41 42  (fun _ v -> (string v) + "X") 5  50 51 52  (fun _ v -> (string v) + "X") 6  60 61 62 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "06-00006-01106-02206-031X16-101016-111116-121216-132X26-202026-212126-222226-233X36-303036-313136-323236-334X46-404046-414146-424246-435X56-505056-515156-525256-536X66-606066-616166-626266-63ReadX_TAIL"
    test "test8672" (lazy(sprintf "06-00%d06-01%d06-02%d06-03%d06-04%a16-10%d16-11%d16-12%d16-13%d16-14%a26-20%d26-21%d26-22%d26-23%d26-24%a36-30%d36-31%d36-32%d36-33%d36-34%a46-40%d46-41%d46-42%d46-43%d46-44%a56-50%d56-51%d56-52%d56-53%d56-54%a66-60%d66-61%d66-62%d66-63%d66-64" 0 1 2 3  (fun _ v -> (string v) + "X") 1  10 11 12 13  (fun _ v -> (string v) + "X") 2  20 21 22 23  (fun _ v -> (string v) + "X") 3  30 31 32 33  (fun _ v -> (string v) + "X") 4  40 41 42 43  (fun _ v -> (string v) + "X") 5  50 51 52 53  (fun _ v -> (string v) + "X") 6  60 61 62 63 )) "06-00006-01106-02206-03306-041X16-101016-111116-121216-131316-142X26-202026-212126-222226-232326-243X36-303036-313136-323236-333336-344X46-404046-414146-424246-434346-445X56-505056-515156-525256-535356-546X66-606066-616166-626266-636366-64"
    test "test8673" (lazy(sprintf "06-00%d06-01%d06-02%d06-03%d06-04%a16-10%d16-11%d16-12%d16-13%d16-14%a26-20%d26-21%d26-22%d26-23%d26-24%a36-30%d36-31%d36-32%d36-33%d36-34%a46-40%d46-41%d46-42%d46-43%d46-44%a56-50%d56-51%d56-52%d56-53%d56-54%a66-60%d66-61%d66-62%d66-63%d66-64%a_TAIL" 0 1 2 3  (fun _ v -> (string v) + "X") 1  10 11 12 13  (fun _ v -> (string v) + "X") 2  20 21 22 23  (fun _ v -> (string v) + "X") 3  30 31 32 33  (fun _ v -> (string v) + "X") 4  40 41 42 43  (fun _ v -> (string v) + "X") 5  50 51 52 53  (fun _ v -> (string v) + "X") 6  60 61 62 63 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "06-00006-01106-02206-03306-041X16-101016-111116-121216-131316-142X26-202026-212126-222226-232326-243X36-303036-313136-323236-333336-344X46-404046-414146-424246-434346-445X56-505056-515156-525256-535356-546X66-606066-616166-626266-636366-64ReadX_TAIL"
    test "test8674" (lazy(sprintf "06-00%d06-01%d06-02%d06-03%d06-04%d06-05%a16-10%d16-11%d16-12%d16-13%d16-14%d16-15%a26-20%d26-21%d26-22%d26-23%d26-24%d26-25%a36-30%d36-31%d36-32%d36-33%d36-34%d36-35%a46-40%d46-41%d46-42%d46-43%d46-44%d46-45%a56-50%d56-51%d56-52%d56-53%d56-54%d56-55%a66-60%d66-61%d66-62%d66-63%d66-64%d66-65" 0 1 2 3 4  (fun _ v -> (string v) + "X") 1  10 11 12 13 14  (fun _ v -> (string v) + "X") 2  20 21 22 23 24  (fun _ v -> (string v) + "X") 3  30 31 32 33 34  (fun _ v -> (string v) + "X") 4  40 41 42 43 44  (fun _ v -> (string v) + "X") 5  50 51 52 53 54  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 )) "06-00006-01106-02206-03306-04406-051X16-101016-111116-121216-131316-141416-152X26-202026-212126-222226-232326-242426-253X36-303036-313136-323236-333336-343436-354X46-404046-414146-424246-434346-444446-455X56-505056-515156-525256-535356-545456-556X66-606066-616166-626266-636366-646466-65"
    test "test8675" (lazy(sprintf "06-00%d06-01%d06-02%d06-03%d06-04%d06-05%a16-10%d16-11%d16-12%d16-13%d16-14%d16-15%a26-20%d26-21%d26-22%d26-23%d26-24%d26-25%a36-30%d36-31%d36-32%d36-33%d36-34%d36-35%a46-40%d46-41%d46-42%d46-43%d46-44%d46-45%a56-50%d56-51%d56-52%d56-53%d56-54%d56-55%a66-60%d66-61%d66-62%d66-63%d66-64%d66-65%a_TAIL" 0 1 2 3 4  (fun _ v -> (string v) + "X") 1  10 11 12 13 14  (fun _ v -> (string v) + "X") 2  20 21 22 23 24  (fun _ v -> (string v) + "X") 3  30 31 32 33 34  (fun _ v -> (string v) + "X") 4  40 41 42 43 44  (fun _ v -> (string v) + "X") 5  50 51 52 53 54  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "06-00006-01106-02206-03306-04406-051X16-101016-111116-121216-131316-141416-152X26-202026-212126-222226-232326-242426-253X36-303036-313136-323236-333336-343436-354X46-404046-414146-424246-434346-444446-455X56-505056-515156-525256-535356-545456-556X66-606066-616166-626266-636366-646466-65ReadX_TAIL"
    test "test8676" (lazy(sprintf "06-00%d06-01%d06-02%d06-03%d06-04%d06-05%d06-06%a16-10%d16-11%d16-12%d16-13%d16-14%d16-15%d16-16%a26-20%d26-21%d26-22%d26-23%d26-24%d26-25%d26-26%a36-30%d36-31%d36-32%d36-33%d36-34%d36-35%d36-36%a46-40%d46-41%d46-42%d46-43%d46-44%d46-45%d46-46%a56-50%d56-51%d56-52%d56-53%d56-54%d56-55%d56-56%a66-60%d66-61%d66-62%d66-63%d66-64%d66-65%d66-66" 0 1 2 3 4 5  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 65 )) "06-00006-01106-02206-03306-04406-05506-061X16-101016-111116-121216-131316-141416-151516-162X26-202026-212126-222226-232326-242426-252526-263X36-303036-313136-323236-333336-343436-353536-364X46-404046-414146-424246-434346-444446-454546-465X56-505056-515156-525256-535356-545456-555556-566X66-606066-616166-626266-636366-646466-656566-66"
    test "test8677" (lazy(sprintf "06-00%d06-01%d06-02%d06-03%d06-04%d06-05%d06-06%a16-10%d16-11%d16-12%d16-13%d16-14%d16-15%d16-16%a26-20%d26-21%d26-22%d26-23%d26-24%d26-25%d26-26%a36-30%d36-31%d36-32%d36-33%d36-34%d36-35%d36-36%a46-40%d46-41%d46-42%d46-43%d46-44%d46-45%d46-46%a56-50%d56-51%d56-52%d56-53%d56-54%d56-55%d56-56%a66-60%d66-61%d66-62%d66-63%d66-64%d66-65%d66-66%a_TAIL" 0 1 2 3 4 5  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 65 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "06-00006-01106-02206-03306-04406-05506-061X16-101016-111116-121216-131316-141416-151516-162X26-202026-212126-222226-232326-242426-252526-263X36-303036-313136-323236-333336-343436-353536-364X46-404046-414146-424246-434346-444446-454546-465X56-505056-515156-525256-535356-545456-555556-566X66-606066-616166-626266-636366-646466-656566-66ReadX_TAIL"
    test "test8678" (lazy(sprintf "06-00%d06-01%d06-02%d06-03%d06-04%d06-05%d06-06%d06-07%a16-10%d16-11%d16-12%d16-13%d16-14%d16-15%d16-16%d16-17%a26-20%d26-21%d26-22%d26-23%d26-24%d26-25%d26-26%d26-27%a36-30%d36-31%d36-32%d36-33%d36-34%d36-35%d36-36%d36-37%a46-40%d46-41%d46-42%d46-43%d46-44%d46-45%d46-46%d46-47%a56-50%d56-51%d56-52%d56-53%d56-54%d56-55%d56-56%d56-57%a66-60%d66-61%d66-62%d66-63%d66-64%d66-65%d66-66%d66-67" 0 1 2 3 4 5 6  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55 56  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 65 66 )) "06-00006-01106-02206-03306-04406-05506-06606-071X16-101016-111116-121216-131316-141416-151516-161616-172X26-202026-212126-222226-232326-242426-252526-262626-273X36-303036-313136-323236-333336-343436-353536-363636-374X46-404046-414146-424246-434346-444446-454546-464646-475X56-505056-515156-525256-535356-545456-555556-565656-576X66-606066-616166-626266-636366-646466-656566-666666-67"
    test "test8679" (lazy(sprintf "06-00%d06-01%d06-02%d06-03%d06-04%d06-05%d06-06%d06-07%a16-10%d16-11%d16-12%d16-13%d16-14%d16-15%d16-16%d16-17%a26-20%d26-21%d26-22%d26-23%d26-24%d26-25%d26-26%d26-27%a36-30%d36-31%d36-32%d36-33%d36-34%d36-35%d36-36%d36-37%a46-40%d46-41%d46-42%d46-43%d46-44%d46-45%d46-46%d46-47%a56-50%d56-51%d56-52%d56-53%d56-54%d56-55%d56-56%d56-57%a66-60%d66-61%d66-62%d66-63%d66-64%d66-65%d66-66%d66-67%a_TAIL" 0 1 2 3 4 5 6  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55 56  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 65 66 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "06-00006-01106-02206-03306-04406-05506-06606-071X16-101016-111116-121216-131316-141416-151516-161616-172X26-202026-212126-222226-232326-242426-252526-262626-273X36-303036-313136-323236-333336-343436-353536-363636-374X46-404046-414146-424246-434346-444446-454546-464646-475X56-505056-515156-525256-535356-545456-555556-565656-576X66-606066-616166-626266-636366-646466-656566-666666-67ReadX_TAIL"
    test "test8680" (lazy(sprintf "06-00%d06-01%d06-02%d06-03%d06-04%d06-05%d06-06%d06-07%d06-08%a16-10%d16-11%d16-12%d16-13%d16-14%d16-15%d16-16%d16-17%d16-18%a26-20%d26-21%d26-22%d26-23%d26-24%d26-25%d26-26%d26-27%d26-28%a36-30%d36-31%d36-32%d36-33%d36-34%d36-35%d36-36%d36-37%d36-38%a46-40%d46-41%d46-42%d46-43%d46-44%d46-45%d46-46%d46-47%d46-48%a56-50%d56-51%d56-52%d56-53%d56-54%d56-55%d56-56%d56-57%d56-58%a66-60%d66-61%d66-62%d66-63%d66-64%d66-65%d66-66%d66-67%d66-68" 0 1 2 3 4 5 6 7  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 27  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36 37  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46 47  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55 56 57  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 65 66 67 )) "06-00006-01106-02206-03306-04406-05506-06606-07706-081X16-101016-111116-121216-131316-141416-151516-161616-171716-182X26-202026-212126-222226-232326-242426-252526-262626-272726-283X36-303036-313136-323236-333336-343436-353536-363636-373736-384X46-404046-414146-424246-434346-444446-454546-464646-474746-485X56-505056-515156-525256-535356-545456-555556-565656-575756-586X66-606066-616166-626266-636366-646466-656566-666666-676766-68"
    test "test8681" (lazy(sprintf "06-00%d06-01%d06-02%d06-03%d06-04%d06-05%d06-06%d06-07%d06-08%a16-10%d16-11%d16-12%d16-13%d16-14%d16-15%d16-16%d16-17%d16-18%a26-20%d26-21%d26-22%d26-23%d26-24%d26-25%d26-26%d26-27%d26-28%a36-30%d36-31%d36-32%d36-33%d36-34%d36-35%d36-36%d36-37%d36-38%a46-40%d46-41%d46-42%d46-43%d46-44%d46-45%d46-46%d46-47%d46-48%a56-50%d56-51%d56-52%d56-53%d56-54%d56-55%d56-56%d56-57%d56-58%a66-60%d66-61%d66-62%d66-63%d66-64%d66-65%d66-66%d66-67%d66-68%a_TAIL" 0 1 2 3 4 5 6 7  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 27  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36 37  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46 47  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55 56 57  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 65 66 67 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "06-00006-01106-02206-03306-04406-05506-06606-07706-081X16-101016-111116-121216-131316-141416-151516-161616-171716-182X26-202026-212126-222226-232326-242426-252526-262626-272726-283X36-303036-313136-323236-333336-343436-353536-363636-373736-384X46-404046-414146-424246-434346-444446-454546-464646-474746-485X56-505056-515156-525256-535356-545456-555556-565656-575756-586X66-606066-616166-626266-636366-646466-656566-666666-676766-68ReadX_TAIL"
    test "test8682" (lazy(sprintf "06-00%d06-01%d06-02%d06-03%d06-04%d06-05%d06-06%d06-07%d06-08%d06-09%a16-10%d16-11%d16-12%d16-13%d16-14%d16-15%d16-16%d16-17%d16-18%d16-19%a26-20%d26-21%d26-22%d26-23%d26-24%d26-25%d26-26%d26-27%d26-28%d26-29%a36-30%d36-31%d36-32%d36-33%d36-34%d36-35%d36-36%d36-37%d36-38%d36-39%a46-40%d46-41%d46-42%d46-43%d46-44%d46-45%d46-46%d46-47%d46-48%d46-49%a56-50%d56-51%d56-52%d56-53%d56-54%d56-55%d56-56%d56-57%d56-58%d56-59%a66-60%d66-61%d66-62%d66-63%d66-64%d66-65%d66-66%d66-67%d66-68%d66-69" 0 1 2 3 4 5 6 7 8  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17 18  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 27 28  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36 37 38  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46 47 48  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55 56 57 58  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 65 66 67 68 )) "06-00006-01106-02206-03306-04406-05506-06606-07706-08806-091X16-101016-111116-121216-131316-141416-151516-161616-171716-181816-192X26-202026-212126-222226-232326-242426-252526-262626-272726-282826-293X36-303036-313136-323236-333336-343436-353536-363636-373736-383836-394X46-404046-414146-424246-434346-444446-454546-464646-474746-484846-495X56-505056-515156-525256-535356-545456-555556-565656-575756-585856-596X66-606066-616166-626266-636366-646466-656566-666666-676766-686866-69"
    test "test8683" (lazy(sprintf "06-00%d06-01%d06-02%d06-03%d06-04%d06-05%d06-06%d06-07%d06-08%d06-09%a16-10%d16-11%d16-12%d16-13%d16-14%d16-15%d16-16%d16-17%d16-18%d16-19%a26-20%d26-21%d26-22%d26-23%d26-24%d26-25%d26-26%d26-27%d26-28%d26-29%a36-30%d36-31%d36-32%d36-33%d36-34%d36-35%d36-36%d36-37%d36-38%d36-39%a46-40%d46-41%d46-42%d46-43%d46-44%d46-45%d46-46%d46-47%d46-48%d46-49%a56-50%d56-51%d56-52%d56-53%d56-54%d56-55%d56-56%d56-57%d56-58%d56-59%a66-60%d66-61%d66-62%d66-63%d66-64%d66-65%d66-66%d66-67%d66-68%d66-69%a_TAIL" 0 1 2 3 4 5 6 7 8  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17 18  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 27 28  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36 37 38  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46 47 48  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55 56 57 58  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 65 66 67 68 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "06-00006-01106-02206-03306-04406-05506-06606-07706-08806-091X16-101016-111116-121216-131316-141416-151516-161616-171716-181816-192X26-202026-212126-222226-232326-242426-252526-262626-272726-282826-293X36-303036-313136-323236-333336-343436-353536-363636-373736-383836-394X46-404046-414146-424246-434346-444446-454546-464646-474746-484846-495X56-505056-515156-525256-535356-545456-555556-565656-575756-585856-596X66-606066-616166-626266-636366-646466-656566-666666-676766-686866-69ReadX_TAIL"
    test "test8684" (lazy(sprintf "07-00%d07-01%a17-10%d17-11%a27-20%d27-21%a37-30%d37-31%a47-40%d47-41%a57-50%d57-51%a67-60%d67-61%a77-70%d77-71" 0  (fun _ v -> (string v) + "X") 1  10  (fun _ v -> (string v) + "X") 2  20  (fun _ v -> (string v) + "X") 3  30  (fun _ v -> (string v) + "X") 4  40  (fun _ v -> (string v) + "X") 5  50  (fun _ v -> (string v) + "X") 6  60  (fun _ v -> (string v) + "X") 7  70 )) "07-00007-011X17-101017-112X27-202027-213X37-303037-314X47-404047-415X57-505057-516X67-606067-617X77-707077-71"
    test "test8685" (lazy(sprintf "07-00%d07-01%a17-10%d17-11%a27-20%d27-21%a37-30%d37-31%a47-40%d47-41%a57-50%d57-51%a67-60%d67-61%a77-70%d77-71%a_TAIL" 0  (fun _ v -> (string v) + "X") 1  10  (fun _ v -> (string v) + "X") 2  20  (fun _ v -> (string v) + "X") 3  30  (fun _ v -> (string v) + "X") 4  40  (fun _ v -> (string v) + "X") 5  50  (fun _ v -> (string v) + "X") 6  60  (fun _ v -> (string v) + "X") 7  70 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "07-00007-011X17-101017-112X27-202027-213X37-303037-314X47-404047-415X57-505057-516X67-606067-617X77-707077-71ReadX_TAIL"
    test "test8686" (lazy(sprintf "07-00%d07-01%d07-02%a17-10%d17-11%d17-12%a27-20%d27-21%d27-22%a37-30%d37-31%d37-32%a47-40%d47-41%d47-42%a57-50%d57-51%d57-52%a67-60%d67-61%d67-62%a77-70%d77-71%d77-72" 0 1  (fun _ v -> (string v) + "X") 1  10 11  (fun _ v -> (string v) + "X") 2  20 21  (fun _ v -> (string v) + "X") 3  30 31  (fun _ v -> (string v) + "X") 4  40 41  (fun _ v -> (string v) + "X") 5  50 51  (fun _ v -> (string v) + "X") 6  60 61  (fun _ v -> (string v) + "X") 7  70 71 )) "07-00007-01107-021X17-101017-111117-122X27-202027-212127-223X37-303037-313137-324X47-404047-414147-425X57-505057-515157-526X67-606067-616167-627X77-707077-717177-72"
    test "test8687" (lazy(sprintf "07-00%d07-01%d07-02%a17-10%d17-11%d17-12%a27-20%d27-21%d27-22%a37-30%d37-31%d37-32%a47-40%d47-41%d47-42%a57-50%d57-51%d57-52%a67-60%d67-61%d67-62%a77-70%d77-71%d77-72%a_TAIL" 0 1  (fun _ v -> (string v) + "X") 1  10 11  (fun _ v -> (string v) + "X") 2  20 21  (fun _ v -> (string v) + "X") 3  30 31  (fun _ v -> (string v) + "X") 4  40 41  (fun _ v -> (string v) + "X") 5  50 51  (fun _ v -> (string v) + "X") 6  60 61  (fun _ v -> (string v) + "X") 7  70 71 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "07-00007-01107-021X17-101017-111117-122X27-202027-212127-223X37-303037-313137-324X47-404047-414147-425X57-505057-515157-526X67-606067-616167-627X77-707077-717177-72ReadX_TAIL"
    test "test8688" (lazy(sprintf "07-00%d07-01%d07-02%d07-03%a17-10%d17-11%d17-12%d17-13%a27-20%d27-21%d27-22%d27-23%a37-30%d37-31%d37-32%d37-33%a47-40%d47-41%d47-42%d47-43%a57-50%d57-51%d57-52%d57-53%a67-60%d67-61%d67-62%d67-63%a77-70%d77-71%d77-72%d77-73" 0 1 2  (fun _ v -> (string v) + "X") 1  10 11 12  (fun _ v -> (string v) + "X") 2  20 21 22  (fun _ v -> (string v) + "X") 3  30 31 32  (fun _ v -> (string v) + "X") 4  40 41 42  (fun _ v -> (string v) + "X") 5  50 51 52  (fun _ v -> (string v) + "X") 6  60 61 62  (fun _ v -> (string v) + "X") 7  70 71 72 )) "07-00007-01107-02207-031X17-101017-111117-121217-132X27-202027-212127-222227-233X37-303037-313137-323237-334X47-404047-414147-424247-435X57-505057-515157-525257-536X67-606067-616167-626267-637X77-707077-717177-727277-73"
    test "test8689" (lazy(sprintf "07-00%d07-01%d07-02%d07-03%a17-10%d17-11%d17-12%d17-13%a27-20%d27-21%d27-22%d27-23%a37-30%d37-31%d37-32%d37-33%a47-40%d47-41%d47-42%d47-43%a57-50%d57-51%d57-52%d57-53%a67-60%d67-61%d67-62%d67-63%a77-70%d77-71%d77-72%d77-73%a_TAIL" 0 1 2  (fun _ v -> (string v) + "X") 1  10 11 12  (fun _ v -> (string v) + "X") 2  20 21 22  (fun _ v -> (string v) + "X") 3  30 31 32  (fun _ v -> (string v) + "X") 4  40 41 42  (fun _ v -> (string v) + "X") 5  50 51 52  (fun _ v -> (string v) + "X") 6  60 61 62  (fun _ v -> (string v) + "X") 7  70 71 72 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "07-00007-01107-02207-031X17-101017-111117-121217-132X27-202027-212127-222227-233X37-303037-313137-323237-334X47-404047-414147-424247-435X57-505057-515157-525257-536X67-606067-616167-626267-637X77-707077-717177-727277-73ReadX_TAIL"
    test "test8690" (lazy(sprintf "07-00%d07-01%d07-02%d07-03%d07-04%a17-10%d17-11%d17-12%d17-13%d17-14%a27-20%d27-21%d27-22%d27-23%d27-24%a37-30%d37-31%d37-32%d37-33%d37-34%a47-40%d47-41%d47-42%d47-43%d47-44%a57-50%d57-51%d57-52%d57-53%d57-54%a67-60%d67-61%d67-62%d67-63%d67-64%a77-70%d77-71%d77-72%d77-73%d77-74" 0 1 2 3  (fun _ v -> (string v) + "X") 1  10 11 12 13  (fun _ v -> (string v) + "X") 2  20 21 22 23  (fun _ v -> (string v) + "X") 3  30 31 32 33  (fun _ v -> (string v) + "X") 4  40 41 42 43  (fun _ v -> (string v) + "X") 5  50 51 52 53  (fun _ v -> (string v) + "X") 6  60 61 62 63  (fun _ v -> (string v) + "X") 7  70 71 72 73 )) "07-00007-01107-02207-03307-041X17-101017-111117-121217-131317-142X27-202027-212127-222227-232327-243X37-303037-313137-323237-333337-344X47-404047-414147-424247-434347-445X57-505057-515157-525257-535357-546X67-606067-616167-626267-636367-647X77-707077-717177-727277-737377-74"
    test "test8691" (lazy(sprintf "07-00%d07-01%d07-02%d07-03%d07-04%a17-10%d17-11%d17-12%d17-13%d17-14%a27-20%d27-21%d27-22%d27-23%d27-24%a37-30%d37-31%d37-32%d37-33%d37-34%a47-40%d47-41%d47-42%d47-43%d47-44%a57-50%d57-51%d57-52%d57-53%d57-54%a67-60%d67-61%d67-62%d67-63%d67-64%a77-70%d77-71%d77-72%d77-73%d77-74%a_TAIL" 0 1 2 3  (fun _ v -> (string v) + "X") 1  10 11 12 13  (fun _ v -> (string v) + "X") 2  20 21 22 23  (fun _ v -> (string v) + "X") 3  30 31 32 33  (fun _ v -> (string v) + "X") 4  40 41 42 43  (fun _ v -> (string v) + "X") 5  50 51 52 53  (fun _ v -> (string v) + "X") 6  60 61 62 63  (fun _ v -> (string v) + "X") 7  70 71 72 73 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "07-00007-01107-02207-03307-041X17-101017-111117-121217-131317-142X27-202027-212127-222227-232327-243X37-303037-313137-323237-333337-344X47-404047-414147-424247-434347-445X57-505057-515157-525257-535357-546X67-606067-616167-626267-636367-647X77-707077-717177-727277-737377-74ReadX_TAIL"
    test "test8692" (lazy(sprintf "07-00%d07-01%d07-02%d07-03%d07-04%d07-05%a17-10%d17-11%d17-12%d17-13%d17-14%d17-15%a27-20%d27-21%d27-22%d27-23%d27-24%d27-25%a37-30%d37-31%d37-32%d37-33%d37-34%d37-35%a47-40%d47-41%d47-42%d47-43%d47-44%d47-45%a57-50%d57-51%d57-52%d57-53%d57-54%d57-55%a67-60%d67-61%d67-62%d67-63%d67-64%d67-65%a77-70%d77-71%d77-72%d77-73%d77-74%d77-75" 0 1 2 3 4  (fun _ v -> (string v) + "X") 1  10 11 12 13 14  (fun _ v -> (string v) + "X") 2  20 21 22 23 24  (fun _ v -> (string v) + "X") 3  30 31 32 33 34  (fun _ v -> (string v) + "X") 4  40 41 42 43 44  (fun _ v -> (string v) + "X") 5  50 51 52 53 54  (fun _ v -> (string v) + "X") 6  60 61 62 63 64  (fun _ v -> (string v) + "X") 7  70 71 72 73 74 )) "07-00007-01107-02207-03307-04407-051X17-101017-111117-121217-131317-141417-152X27-202027-212127-222227-232327-242427-253X37-303037-313137-323237-333337-343437-354X47-404047-414147-424247-434347-444447-455X57-505057-515157-525257-535357-545457-556X67-606067-616167-626267-636367-646467-657X77-707077-717177-727277-737377-747477-75"
    test "test8693" (lazy(sprintf "07-00%d07-01%d07-02%d07-03%d07-04%d07-05%a17-10%d17-11%d17-12%d17-13%d17-14%d17-15%a27-20%d27-21%d27-22%d27-23%d27-24%d27-25%a37-30%d37-31%d37-32%d37-33%d37-34%d37-35%a47-40%d47-41%d47-42%d47-43%d47-44%d47-45%a57-50%d57-51%d57-52%d57-53%d57-54%d57-55%a67-60%d67-61%d67-62%d67-63%d67-64%d67-65%a77-70%d77-71%d77-72%d77-73%d77-74%d77-75%a_TAIL" 0 1 2 3 4  (fun _ v -> (string v) + "X") 1  10 11 12 13 14  (fun _ v -> (string v) + "X") 2  20 21 22 23 24  (fun _ v -> (string v) + "X") 3  30 31 32 33 34  (fun _ v -> (string v) + "X") 4  40 41 42 43 44  (fun _ v -> (string v) + "X") 5  50 51 52 53 54  (fun _ v -> (string v) + "X") 6  60 61 62 63 64  (fun _ v -> (string v) + "X") 7  70 71 72 73 74 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "07-00007-01107-02207-03307-04407-051X17-101017-111117-121217-131317-141417-152X27-202027-212127-222227-232327-242427-253X37-303037-313137-323237-333337-343437-354X47-404047-414147-424247-434347-444447-455X57-505057-515157-525257-535357-545457-556X67-606067-616167-626267-636367-646467-657X77-707077-717177-727277-737377-747477-75ReadX_TAIL"
    test "test8694" (lazy(sprintf "07-00%d07-01%d07-02%d07-03%d07-04%d07-05%d07-06%a17-10%d17-11%d17-12%d17-13%d17-14%d17-15%d17-16%a27-20%d27-21%d27-22%d27-23%d27-24%d27-25%d27-26%a37-30%d37-31%d37-32%d37-33%d37-34%d37-35%d37-36%a47-40%d47-41%d47-42%d47-43%d47-44%d47-45%d47-46%a57-50%d57-51%d57-52%d57-53%d57-54%d57-55%d57-56%a67-60%d67-61%d67-62%d67-63%d67-64%d67-65%d67-66%a77-70%d77-71%d77-72%d77-73%d77-74%d77-75%d77-76" 0 1 2 3 4 5  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 65  (fun _ v -> (string v) + "X") 7  70 71 72 73 74 75 )) "07-00007-01107-02207-03307-04407-05507-061X17-101017-111117-121217-131317-141417-151517-162X27-202027-212127-222227-232327-242427-252527-263X37-303037-313137-323237-333337-343437-353537-364X47-404047-414147-424247-434347-444447-454547-465X57-505057-515157-525257-535357-545457-555557-566X67-606067-616167-626267-636367-646467-656567-667X77-707077-717177-727277-737377-747477-757577-76"
    test "test8695" (lazy(sprintf "07-00%d07-01%d07-02%d07-03%d07-04%d07-05%d07-06%a17-10%d17-11%d17-12%d17-13%d17-14%d17-15%d17-16%a27-20%d27-21%d27-22%d27-23%d27-24%d27-25%d27-26%a37-30%d37-31%d37-32%d37-33%d37-34%d37-35%d37-36%a47-40%d47-41%d47-42%d47-43%d47-44%d47-45%d47-46%a57-50%d57-51%d57-52%d57-53%d57-54%d57-55%d57-56%a67-60%d67-61%d67-62%d67-63%d67-64%d67-65%d67-66%a77-70%d77-71%d77-72%d77-73%d77-74%d77-75%d77-76%a_TAIL" 0 1 2 3 4 5  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 65  (fun _ v -> (string v) + "X") 7  70 71 72 73 74 75 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "07-00007-01107-02207-03307-04407-05507-061X17-101017-111117-121217-131317-141417-151517-162X27-202027-212127-222227-232327-242427-252527-263X37-303037-313137-323237-333337-343437-353537-364X47-404047-414147-424247-434347-444447-454547-465X57-505057-515157-525257-535357-545457-555557-566X67-606067-616167-626267-636367-646467-656567-667X77-707077-717177-727277-737377-747477-757577-76ReadX_TAIL"
    test "test8696" (lazy(sprintf "07-00%d07-01%d07-02%d07-03%d07-04%d07-05%d07-06%d07-07%a17-10%d17-11%d17-12%d17-13%d17-14%d17-15%d17-16%d17-17%a27-20%d27-21%d27-22%d27-23%d27-24%d27-25%d27-26%d27-27%a37-30%d37-31%d37-32%d37-33%d37-34%d37-35%d37-36%d37-37%a47-40%d47-41%d47-42%d47-43%d47-44%d47-45%d47-46%d47-47%a57-50%d57-51%d57-52%d57-53%d57-54%d57-55%d57-56%d57-57%a67-60%d67-61%d67-62%d67-63%d67-64%d67-65%d67-66%d67-67%a77-70%d77-71%d77-72%d77-73%d77-74%d77-75%d77-76%d77-77" 0 1 2 3 4 5 6  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55 56  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 65 66  (fun _ v -> (string v) + "X") 7  70 71 72 73 74 75 76 )) "07-00007-01107-02207-03307-04407-05507-06607-071X17-101017-111117-121217-131317-141417-151517-161617-172X27-202027-212127-222227-232327-242427-252527-262627-273X37-303037-313137-323237-333337-343437-353537-363637-374X47-404047-414147-424247-434347-444447-454547-464647-475X57-505057-515157-525257-535357-545457-555557-565657-576X67-606067-616167-626267-636367-646467-656567-666667-677X77-707077-717177-727277-737377-747477-757577-767677-77"
    test "test8697" (lazy(sprintf "07-00%d07-01%d07-02%d07-03%d07-04%d07-05%d07-06%d07-07%a17-10%d17-11%d17-12%d17-13%d17-14%d17-15%d17-16%d17-17%a27-20%d27-21%d27-22%d27-23%d27-24%d27-25%d27-26%d27-27%a37-30%d37-31%d37-32%d37-33%d37-34%d37-35%d37-36%d37-37%a47-40%d47-41%d47-42%d47-43%d47-44%d47-45%d47-46%d47-47%a57-50%d57-51%d57-52%d57-53%d57-54%d57-55%d57-56%d57-57%a67-60%d67-61%d67-62%d67-63%d67-64%d67-65%d67-66%d67-67%a77-70%d77-71%d77-72%d77-73%d77-74%d77-75%d77-76%d77-77%a_TAIL" 0 1 2 3 4 5 6  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55 56  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 65 66  (fun _ v -> (string v) + "X") 7  70 71 72 73 74 75 76 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "07-00007-01107-02207-03307-04407-05507-06607-071X17-101017-111117-121217-131317-141417-151517-161617-172X27-202027-212127-222227-232327-242427-252527-262627-273X37-303037-313137-323237-333337-343437-353537-363637-374X47-404047-414147-424247-434347-444447-454547-464647-475X57-505057-515157-525257-535357-545457-555557-565657-576X67-606067-616167-626267-636367-646467-656567-666667-677X77-707077-717177-727277-737377-747477-757577-767677-77ReadX_TAIL"
    test "test8698" (lazy(sprintf "07-00%d07-01%d07-02%d07-03%d07-04%d07-05%d07-06%d07-07%d07-08%a17-10%d17-11%d17-12%d17-13%d17-14%d17-15%d17-16%d17-17%d17-18%a27-20%d27-21%d27-22%d27-23%d27-24%d27-25%d27-26%d27-27%d27-28%a37-30%d37-31%d37-32%d37-33%d37-34%d37-35%d37-36%d37-37%d37-38%a47-40%d47-41%d47-42%d47-43%d47-44%d47-45%d47-46%d47-47%d47-48%a57-50%d57-51%d57-52%d57-53%d57-54%d57-55%d57-56%d57-57%d57-58%a67-60%d67-61%d67-62%d67-63%d67-64%d67-65%d67-66%d67-67%d67-68%a77-70%d77-71%d77-72%d77-73%d77-74%d77-75%d77-76%d77-77%d77-78" 0 1 2 3 4 5 6 7  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 27  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36 37  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46 47  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55 56 57  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 65 66 67  (fun _ v -> (string v) + "X") 7  70 71 72 73 74 75 76 77 )) "07-00007-01107-02207-03307-04407-05507-06607-07707-081X17-101017-111117-121217-131317-141417-151517-161617-171717-182X27-202027-212127-222227-232327-242427-252527-262627-272727-283X37-303037-313137-323237-333337-343437-353537-363637-373737-384X47-404047-414147-424247-434347-444447-454547-464647-474747-485X57-505057-515157-525257-535357-545457-555557-565657-575757-586X67-606067-616167-626267-636367-646467-656567-666667-676767-687X77-707077-717177-727277-737377-747477-757577-767677-777777-78"
    test "test8699" (lazy(sprintf "07-00%d07-01%d07-02%d07-03%d07-04%d07-05%d07-06%d07-07%d07-08%a17-10%d17-11%d17-12%d17-13%d17-14%d17-15%d17-16%d17-17%d17-18%a27-20%d27-21%d27-22%d27-23%d27-24%d27-25%d27-26%d27-27%d27-28%a37-30%d37-31%d37-32%d37-33%d37-34%d37-35%d37-36%d37-37%d37-38%a47-40%d47-41%d47-42%d47-43%d47-44%d47-45%d47-46%d47-47%d47-48%a57-50%d57-51%d57-52%d57-53%d57-54%d57-55%d57-56%d57-57%d57-58%a67-60%d67-61%d67-62%d67-63%d67-64%d67-65%d67-66%d67-67%d67-68%a77-70%d77-71%d77-72%d77-73%d77-74%d77-75%d77-76%d77-77%d77-78%a_TAIL" 0 1 2 3 4 5 6 7  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 27  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36 37  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46 47  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55 56 57  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 65 66 67  (fun _ v -> (string v) + "X") 7  70 71 72 73 74 75 76 77 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "07-00007-01107-02207-03307-04407-05507-06607-07707-081X17-101017-111117-121217-131317-141417-151517-161617-171717-182X27-202027-212127-222227-232327-242427-252527-262627-272727-283X37-303037-313137-323237-333337-343437-353537-363637-373737-384X47-404047-414147-424247-434347-444447-454547-464647-474747-485X57-505057-515157-525257-535357-545457-555557-565657-575757-586X67-606067-616167-626267-636367-646467-656567-666667-676767-687X77-707077-717177-727277-737377-747477-757577-767677-777777-78ReadX_TAIL"
    test "test8700" (lazy(sprintf "07-00%d07-01%d07-02%d07-03%d07-04%d07-05%d07-06%d07-07%d07-08%d07-09%a17-10%d17-11%d17-12%d17-13%d17-14%d17-15%d17-16%d17-17%d17-18%d17-19%a27-20%d27-21%d27-22%d27-23%d27-24%d27-25%d27-26%d27-27%d27-28%d27-29%a37-30%d37-31%d37-32%d37-33%d37-34%d37-35%d37-36%d37-37%d37-38%d37-39%a47-40%d47-41%d47-42%d47-43%d47-44%d47-45%d47-46%d47-47%d47-48%d47-49%a57-50%d57-51%d57-52%d57-53%d57-54%d57-55%d57-56%d57-57%d57-58%d57-59%a67-60%d67-61%d67-62%d67-63%d67-64%d67-65%d67-66%d67-67%d67-68%d67-69%a77-70%d77-71%d77-72%d77-73%d77-74%d77-75%d77-76%d77-77%d77-78%d77-79" 0 1 2 3 4 5 6 7 8  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17 18  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 27 28  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36 37 38  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46 47 48  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55 56 57 58  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 65 66 67 68  (fun _ v -> (string v) + "X") 7  70 71 72 73 74 75 76 77 78 )) "07-00007-01107-02207-03307-04407-05507-06607-07707-08807-091X17-101017-111117-121217-131317-141417-151517-161617-171717-181817-192X27-202027-212127-222227-232327-242427-252527-262627-272727-282827-293X37-303037-313137-323237-333337-343437-353537-363637-373737-383837-394X47-404047-414147-424247-434347-444447-454547-464647-474747-484847-495X57-505057-515157-525257-535357-545457-555557-565657-575757-585857-596X67-606067-616167-626267-636367-646467-656567-666667-676767-686867-697X77-707077-717177-727277-737377-747477-757577-767677-777777-787877-79"
    test "test8701" (lazy(sprintf "07-00%d07-01%d07-02%d07-03%d07-04%d07-05%d07-06%d07-07%d07-08%d07-09%a17-10%d17-11%d17-12%d17-13%d17-14%d17-15%d17-16%d17-17%d17-18%d17-19%a27-20%d27-21%d27-22%d27-23%d27-24%d27-25%d27-26%d27-27%d27-28%d27-29%a37-30%d37-31%d37-32%d37-33%d37-34%d37-35%d37-36%d37-37%d37-38%d37-39%a47-40%d47-41%d47-42%d47-43%d47-44%d47-45%d47-46%d47-47%d47-48%d47-49%a57-50%d57-51%d57-52%d57-53%d57-54%d57-55%d57-56%d57-57%d57-58%d57-59%a67-60%d67-61%d67-62%d67-63%d67-64%d67-65%d67-66%d67-67%d67-68%d67-69%a77-70%d77-71%d77-72%d77-73%d77-74%d77-75%d77-76%d77-77%d77-78%d77-79%a_TAIL" 0 1 2 3 4 5 6 7 8  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17 18  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 27 28  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36 37 38  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46 47 48  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55 56 57 58  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 65 66 67 68  (fun _ v -> (string v) + "X") 7  70 71 72 73 74 75 76 77 78 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "07-00007-01107-02207-03307-04407-05507-06607-07707-08807-091X17-101017-111117-121217-131317-141417-151517-161617-171717-181817-192X27-202027-212127-222227-232327-242427-252527-262627-272727-282827-293X37-303037-313137-323237-333337-343437-353537-363637-373737-383837-394X47-404047-414147-424247-434347-444447-454547-464647-474747-484847-495X57-505057-515157-525257-535357-545457-555557-565657-575757-585857-596X67-606067-616167-626267-636367-646467-656567-666667-676767-686867-697X77-707077-717177-727277-737377-747477-757577-767677-777777-787877-79ReadX_TAIL"
    test "test8702" (lazy(sprintf "08-00%d08-01%a18-10%d18-11%a28-20%d28-21%a38-30%d38-31%a48-40%d48-41%a58-50%d58-51%a68-60%d68-61%a78-70%d78-71%a88-80%d88-81" 0  (fun _ v -> (string v) + "X") 1  10  (fun _ v -> (string v) + "X") 2  20  (fun _ v -> (string v) + "X") 3  30  (fun _ v -> (string v) + "X") 4  40  (fun _ v -> (string v) + "X") 5  50  (fun _ v -> (string v) + "X") 6  60  (fun _ v -> (string v) + "X") 7  70  (fun _ v -> (string v) + "X") 8  80 )) "08-00008-011X18-101018-112X28-202028-213X38-303038-314X48-404048-415X58-505058-516X68-606068-617X78-707078-718X88-808088-81"
    test "test8703" (lazy(sprintf "08-00%d08-01%a18-10%d18-11%a28-20%d28-21%a38-30%d38-31%a48-40%d48-41%a58-50%d58-51%a68-60%d68-61%a78-70%d78-71%a88-80%d88-81%a_TAIL" 0  (fun _ v -> (string v) + "X") 1  10  (fun _ v -> (string v) + "X") 2  20  (fun _ v -> (string v) + "X") 3  30  (fun _ v -> (string v) + "X") 4  40  (fun _ v -> (string v) + "X") 5  50  (fun _ v -> (string v) + "X") 6  60  (fun _ v -> (string v) + "X") 7  70  (fun _ v -> (string v) + "X") 8  80 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "08-00008-011X18-101018-112X28-202028-213X38-303038-314X48-404048-415X58-505058-516X68-606068-617X78-707078-718X88-808088-81ReadX_TAIL"
    test "test8704" (lazy(sprintf "08-00%d08-01%d08-02%a18-10%d18-11%d18-12%a28-20%d28-21%d28-22%a38-30%d38-31%d38-32%a48-40%d48-41%d48-42%a58-50%d58-51%d58-52%a68-60%d68-61%d68-62%a78-70%d78-71%d78-72%a88-80%d88-81%d88-82" 0 1  (fun _ v -> (string v) + "X") 1  10 11  (fun _ v -> (string v) + "X") 2  20 21  (fun _ v -> (string v) + "X") 3  30 31  (fun _ v -> (string v) + "X") 4  40 41  (fun _ v -> (string v) + "X") 5  50 51  (fun _ v -> (string v) + "X") 6  60 61  (fun _ v -> (string v) + "X") 7  70 71  (fun _ v -> (string v) + "X") 8  80 81 )) "08-00008-01108-021X18-101018-111118-122X28-202028-212128-223X38-303038-313138-324X48-404048-414148-425X58-505058-515158-526X68-606068-616168-627X78-707078-717178-728X88-808088-818188-82"
    test "test8705" (lazy(sprintf "08-00%d08-01%d08-02%a18-10%d18-11%d18-12%a28-20%d28-21%d28-22%a38-30%d38-31%d38-32%a48-40%d48-41%d48-42%a58-50%d58-51%d58-52%a68-60%d68-61%d68-62%a78-70%d78-71%d78-72%a88-80%d88-81%d88-82%a_TAIL" 0 1  (fun _ v -> (string v) + "X") 1  10 11  (fun _ v -> (string v) + "X") 2  20 21  (fun _ v -> (string v) + "X") 3  30 31  (fun _ v -> (string v) + "X") 4  40 41  (fun _ v -> (string v) + "X") 5  50 51  (fun _ v -> (string v) + "X") 6  60 61  (fun _ v -> (string v) + "X") 7  70 71  (fun _ v -> (string v) + "X") 8  80 81 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "08-00008-01108-021X18-101018-111118-122X28-202028-212128-223X38-303038-313138-324X48-404048-414148-425X58-505058-515158-526X68-606068-616168-627X78-707078-717178-728X88-808088-818188-82ReadX_TAIL"
    test "test8706" (lazy(sprintf "08-00%d08-01%d08-02%d08-03%a18-10%d18-11%d18-12%d18-13%a28-20%d28-21%d28-22%d28-23%a38-30%d38-31%d38-32%d38-33%a48-40%d48-41%d48-42%d48-43%a58-50%d58-51%d58-52%d58-53%a68-60%d68-61%d68-62%d68-63%a78-70%d78-71%d78-72%d78-73%a88-80%d88-81%d88-82%d88-83" 0 1 2  (fun _ v -> (string v) + "X") 1  10 11 12  (fun _ v -> (string v) + "X") 2  20 21 22  (fun _ v -> (string v) + "X") 3  30 31 32  (fun _ v -> (string v) + "X") 4  40 41 42  (fun _ v -> (string v) + "X") 5  50 51 52  (fun _ v -> (string v) + "X") 6  60 61 62  (fun _ v -> (string v) + "X") 7  70 71 72  (fun _ v -> (string v) + "X") 8  80 81 82 )) "08-00008-01108-02208-031X18-101018-111118-121218-132X28-202028-212128-222228-233X38-303038-313138-323238-334X48-404048-414148-424248-435X58-505058-515158-525258-536X68-606068-616168-626268-637X78-707078-717178-727278-738X88-808088-818188-828288-83"
    test "test8707" (lazy(sprintf "08-00%d08-01%d08-02%d08-03%a18-10%d18-11%d18-12%d18-13%a28-20%d28-21%d28-22%d28-23%a38-30%d38-31%d38-32%d38-33%a48-40%d48-41%d48-42%d48-43%a58-50%d58-51%d58-52%d58-53%a68-60%d68-61%d68-62%d68-63%a78-70%d78-71%d78-72%d78-73%a88-80%d88-81%d88-82%d88-83%a_TAIL" 0 1 2  (fun _ v -> (string v) + "X") 1  10 11 12  (fun _ v -> (string v) + "X") 2  20 21 22  (fun _ v -> (string v) + "X") 3  30 31 32  (fun _ v -> (string v) + "X") 4  40 41 42  (fun _ v -> (string v) + "X") 5  50 51 52  (fun _ v -> (string v) + "X") 6  60 61 62  (fun _ v -> (string v) + "X") 7  70 71 72  (fun _ v -> (string v) + "X") 8  80 81 82 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "08-00008-01108-02208-031X18-101018-111118-121218-132X28-202028-212128-222228-233X38-303038-313138-323238-334X48-404048-414148-424248-435X58-505058-515158-525258-536X68-606068-616168-626268-637X78-707078-717178-727278-738X88-808088-818188-828288-83ReadX_TAIL"
    test "test8708" (lazy(sprintf "08-00%d08-01%d08-02%d08-03%d08-04%a18-10%d18-11%d18-12%d18-13%d18-14%a28-20%d28-21%d28-22%d28-23%d28-24%a38-30%d38-31%d38-32%d38-33%d38-34%a48-40%d48-41%d48-42%d48-43%d48-44%a58-50%d58-51%d58-52%d58-53%d58-54%a68-60%d68-61%d68-62%d68-63%d68-64%a78-70%d78-71%d78-72%d78-73%d78-74%a88-80%d88-81%d88-82%d88-83%d88-84" 0 1 2 3  (fun _ v -> (string v) + "X") 1  10 11 12 13  (fun _ v -> (string v) + "X") 2  20 21 22 23  (fun _ v -> (string v) + "X") 3  30 31 32 33  (fun _ v -> (string v) + "X") 4  40 41 42 43  (fun _ v -> (string v) + "X") 5  50 51 52 53  (fun _ v -> (string v) + "X") 6  60 61 62 63  (fun _ v -> (string v) + "X") 7  70 71 72 73  (fun _ v -> (string v) + "X") 8  80 81 82 83 )) "08-00008-01108-02208-03308-041X18-101018-111118-121218-131318-142X28-202028-212128-222228-232328-243X38-303038-313138-323238-333338-344X48-404048-414148-424248-434348-445X58-505058-515158-525258-535358-546X68-606068-616168-626268-636368-647X78-707078-717178-727278-737378-748X88-808088-818188-828288-838388-84"
    test "test8709" (lazy(sprintf "08-00%d08-01%d08-02%d08-03%d08-04%a18-10%d18-11%d18-12%d18-13%d18-14%a28-20%d28-21%d28-22%d28-23%d28-24%a38-30%d38-31%d38-32%d38-33%d38-34%a48-40%d48-41%d48-42%d48-43%d48-44%a58-50%d58-51%d58-52%d58-53%d58-54%a68-60%d68-61%d68-62%d68-63%d68-64%a78-70%d78-71%d78-72%d78-73%d78-74%a88-80%d88-81%d88-82%d88-83%d88-84%a_TAIL" 0 1 2 3  (fun _ v -> (string v) + "X") 1  10 11 12 13  (fun _ v -> (string v) + "X") 2  20 21 22 23  (fun _ v -> (string v) + "X") 3  30 31 32 33  (fun _ v -> (string v) + "X") 4  40 41 42 43  (fun _ v -> (string v) + "X") 5  50 51 52 53  (fun _ v -> (string v) + "X") 6  60 61 62 63  (fun _ v -> (string v) + "X") 7  70 71 72 73  (fun _ v -> (string v) + "X") 8  80 81 82 83 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "08-00008-01108-02208-03308-041X18-101018-111118-121218-131318-142X28-202028-212128-222228-232328-243X38-303038-313138-323238-333338-344X48-404048-414148-424248-434348-445X58-505058-515158-525258-535358-546X68-606068-616168-626268-636368-647X78-707078-717178-727278-737378-748X88-808088-818188-828288-838388-84ReadX_TAIL"
    test "test8710" (lazy(sprintf "08-00%d08-01%d08-02%d08-03%d08-04%d08-05%a18-10%d18-11%d18-12%d18-13%d18-14%d18-15%a28-20%d28-21%d28-22%d28-23%d28-24%d28-25%a38-30%d38-31%d38-32%d38-33%d38-34%d38-35%a48-40%d48-41%d48-42%d48-43%d48-44%d48-45%a58-50%d58-51%d58-52%d58-53%d58-54%d58-55%a68-60%d68-61%d68-62%d68-63%d68-64%d68-65%a78-70%d78-71%d78-72%d78-73%d78-74%d78-75%a88-80%d88-81%d88-82%d88-83%d88-84%d88-85" 0 1 2 3 4  (fun _ v -> (string v) + "X") 1  10 11 12 13 14  (fun _ v -> (string v) + "X") 2  20 21 22 23 24  (fun _ v -> (string v) + "X") 3  30 31 32 33 34  (fun _ v -> (string v) + "X") 4  40 41 42 43 44  (fun _ v -> (string v) + "X") 5  50 51 52 53 54  (fun _ v -> (string v) + "X") 6  60 61 62 63 64  (fun _ v -> (string v) + "X") 7  70 71 72 73 74  (fun _ v -> (string v) + "X") 8  80 81 82 83 84 )) "08-00008-01108-02208-03308-04408-051X18-101018-111118-121218-131318-141418-152X28-202028-212128-222228-232328-242428-253X38-303038-313138-323238-333338-343438-354X48-404048-414148-424248-434348-444448-455X58-505058-515158-525258-535358-545458-556X68-606068-616168-626268-636368-646468-657X78-707078-717178-727278-737378-747478-758X88-808088-818188-828288-838388-848488-85"
    test "test8711" (lazy(sprintf "08-00%d08-01%d08-02%d08-03%d08-04%d08-05%a18-10%d18-11%d18-12%d18-13%d18-14%d18-15%a28-20%d28-21%d28-22%d28-23%d28-24%d28-25%a38-30%d38-31%d38-32%d38-33%d38-34%d38-35%a48-40%d48-41%d48-42%d48-43%d48-44%d48-45%a58-50%d58-51%d58-52%d58-53%d58-54%d58-55%a68-60%d68-61%d68-62%d68-63%d68-64%d68-65%a78-70%d78-71%d78-72%d78-73%d78-74%d78-75%a88-80%d88-81%d88-82%d88-83%d88-84%d88-85%a_TAIL" 0 1 2 3 4  (fun _ v -> (string v) + "X") 1  10 11 12 13 14  (fun _ v -> (string v) + "X") 2  20 21 22 23 24  (fun _ v -> (string v) + "X") 3  30 31 32 33 34  (fun _ v -> (string v) + "X") 4  40 41 42 43 44  (fun _ v -> (string v) + "X") 5  50 51 52 53 54  (fun _ v -> (string v) + "X") 6  60 61 62 63 64  (fun _ v -> (string v) + "X") 7  70 71 72 73 74  (fun _ v -> (string v) + "X") 8  80 81 82 83 84 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "08-00008-01108-02208-03308-04408-051X18-101018-111118-121218-131318-141418-152X28-202028-212128-222228-232328-242428-253X38-303038-313138-323238-333338-343438-354X48-404048-414148-424248-434348-444448-455X58-505058-515158-525258-535358-545458-556X68-606068-616168-626268-636368-646468-657X78-707078-717178-727278-737378-747478-758X88-808088-818188-828288-838388-848488-85ReadX_TAIL"
    test "test8712" (lazy(sprintf "08-00%d08-01%d08-02%d08-03%d08-04%d08-05%d08-06%a18-10%d18-11%d18-12%d18-13%d18-14%d18-15%d18-16%a28-20%d28-21%d28-22%d28-23%d28-24%d28-25%d28-26%a38-30%d38-31%d38-32%d38-33%d38-34%d38-35%d38-36%a48-40%d48-41%d48-42%d48-43%d48-44%d48-45%d48-46%a58-50%d58-51%d58-52%d58-53%d58-54%d58-55%d58-56%a68-60%d68-61%d68-62%d68-63%d68-64%d68-65%d68-66%a78-70%d78-71%d78-72%d78-73%d78-74%d78-75%d78-76%a88-80%d88-81%d88-82%d88-83%d88-84%d88-85%d88-86" 0 1 2 3 4 5  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 65  (fun _ v -> (string v) + "X") 7  70 71 72 73 74 75  (fun _ v -> (string v) + "X") 8  80 81 82 83 84 85 )) "08-00008-01108-02208-03308-04408-05508-061X18-101018-111118-121218-131318-141418-151518-162X28-202028-212128-222228-232328-242428-252528-263X38-303038-313138-323238-333338-343438-353538-364X48-404048-414148-424248-434348-444448-454548-465X58-505058-515158-525258-535358-545458-555558-566X68-606068-616168-626268-636368-646468-656568-667X78-707078-717178-727278-737378-747478-757578-768X88-808088-818188-828288-838388-848488-858588-86"
    test "test8713" (lazy(sprintf "08-00%d08-01%d08-02%d08-03%d08-04%d08-05%d08-06%a18-10%d18-11%d18-12%d18-13%d18-14%d18-15%d18-16%a28-20%d28-21%d28-22%d28-23%d28-24%d28-25%d28-26%a38-30%d38-31%d38-32%d38-33%d38-34%d38-35%d38-36%a48-40%d48-41%d48-42%d48-43%d48-44%d48-45%d48-46%a58-50%d58-51%d58-52%d58-53%d58-54%d58-55%d58-56%a68-60%d68-61%d68-62%d68-63%d68-64%d68-65%d68-66%a78-70%d78-71%d78-72%d78-73%d78-74%d78-75%d78-76%a88-80%d88-81%d88-82%d88-83%d88-84%d88-85%d88-86%a_TAIL" 0 1 2 3 4 5  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 65  (fun _ v -> (string v) + "X") 7  70 71 72 73 74 75  (fun _ v -> (string v) + "X") 8  80 81 82 83 84 85 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "08-00008-01108-02208-03308-04408-05508-061X18-101018-111118-121218-131318-141418-151518-162X28-202028-212128-222228-232328-242428-252528-263X38-303038-313138-323238-333338-343438-353538-364X48-404048-414148-424248-434348-444448-454548-465X58-505058-515158-525258-535358-545458-555558-566X68-606068-616168-626268-636368-646468-656568-667X78-707078-717178-727278-737378-747478-757578-768X88-808088-818188-828288-838388-848488-858588-86ReadX_TAIL"
    test "test8714" (lazy(sprintf "08-00%d08-01%d08-02%d08-03%d08-04%d08-05%d08-06%d08-07%a18-10%d18-11%d18-12%d18-13%d18-14%d18-15%d18-16%d18-17%a28-20%d28-21%d28-22%d28-23%d28-24%d28-25%d28-26%d28-27%a38-30%d38-31%d38-32%d38-33%d38-34%d38-35%d38-36%d38-37%a48-40%d48-41%d48-42%d48-43%d48-44%d48-45%d48-46%d48-47%a58-50%d58-51%d58-52%d58-53%d58-54%d58-55%d58-56%d58-57%a68-60%d68-61%d68-62%d68-63%d68-64%d68-65%d68-66%d68-67%a78-70%d78-71%d78-72%d78-73%d78-74%d78-75%d78-76%d78-77%a88-80%d88-81%d88-82%d88-83%d88-84%d88-85%d88-86%d88-87" 0 1 2 3 4 5 6  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55 56  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 65 66  (fun _ v -> (string v) + "X") 7  70 71 72 73 74 75 76  (fun _ v -> (string v) + "X") 8  80 81 82 83 84 85 86 )) "08-00008-01108-02208-03308-04408-05508-06608-071X18-101018-111118-121218-131318-141418-151518-161618-172X28-202028-212128-222228-232328-242428-252528-262628-273X38-303038-313138-323238-333338-343438-353538-363638-374X48-404048-414148-424248-434348-444448-454548-464648-475X58-505058-515158-525258-535358-545458-555558-565658-576X68-606068-616168-626268-636368-646468-656568-666668-677X78-707078-717178-727278-737378-747478-757578-767678-778X88-808088-818188-828288-838388-848488-858588-868688-87"
    test "test8715" (lazy(sprintf "08-00%d08-01%d08-02%d08-03%d08-04%d08-05%d08-06%d08-07%a18-10%d18-11%d18-12%d18-13%d18-14%d18-15%d18-16%d18-17%a28-20%d28-21%d28-22%d28-23%d28-24%d28-25%d28-26%d28-27%a38-30%d38-31%d38-32%d38-33%d38-34%d38-35%d38-36%d38-37%a48-40%d48-41%d48-42%d48-43%d48-44%d48-45%d48-46%d48-47%a58-50%d58-51%d58-52%d58-53%d58-54%d58-55%d58-56%d58-57%a68-60%d68-61%d68-62%d68-63%d68-64%d68-65%d68-66%d68-67%a78-70%d78-71%d78-72%d78-73%d78-74%d78-75%d78-76%d78-77%a88-80%d88-81%d88-82%d88-83%d88-84%d88-85%d88-86%d88-87%a_TAIL" 0 1 2 3 4 5 6  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55 56  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 65 66  (fun _ v -> (string v) + "X") 7  70 71 72 73 74 75 76  (fun _ v -> (string v) + "X") 8  80 81 82 83 84 85 86 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "08-00008-01108-02208-03308-04408-05508-06608-071X18-101018-111118-121218-131318-141418-151518-161618-172X28-202028-212128-222228-232328-242428-252528-262628-273X38-303038-313138-323238-333338-343438-353538-363638-374X48-404048-414148-424248-434348-444448-454548-464648-475X58-505058-515158-525258-535358-545458-555558-565658-576X68-606068-616168-626268-636368-646468-656568-666668-677X78-707078-717178-727278-737378-747478-757578-767678-778X88-808088-818188-828288-838388-848488-858588-868688-87ReadX_TAIL"
    test "test8716" (lazy(sprintf "08-00%d08-01%d08-02%d08-03%d08-04%d08-05%d08-06%d08-07%d08-08%a18-10%d18-11%d18-12%d18-13%d18-14%d18-15%d18-16%d18-17%d18-18%a28-20%d28-21%d28-22%d28-23%d28-24%d28-25%d28-26%d28-27%d28-28%a38-30%d38-31%d38-32%d38-33%d38-34%d38-35%d38-36%d38-37%d38-38%a48-40%d48-41%d48-42%d48-43%d48-44%d48-45%d48-46%d48-47%d48-48%a58-50%d58-51%d58-52%d58-53%d58-54%d58-55%d58-56%d58-57%d58-58%a68-60%d68-61%d68-62%d68-63%d68-64%d68-65%d68-66%d68-67%d68-68%a78-70%d78-71%d78-72%d78-73%d78-74%d78-75%d78-76%d78-77%d78-78%a88-80%d88-81%d88-82%d88-83%d88-84%d88-85%d88-86%d88-87%d88-88" 0 1 2 3 4 5 6 7  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 27  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36 37  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46 47  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55 56 57  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 65 66 67  (fun _ v -> (string v) + "X") 7  70 71 72 73 74 75 76 77  (fun _ v -> (string v) + "X") 8  80 81 82 83 84 85 86 87 )) "08-00008-01108-02208-03308-04408-05508-06608-07708-081X18-101018-111118-121218-131318-141418-151518-161618-171718-182X28-202028-212128-222228-232328-242428-252528-262628-272728-283X38-303038-313138-323238-333338-343438-353538-363638-373738-384X48-404048-414148-424248-434348-444448-454548-464648-474748-485X58-505058-515158-525258-535358-545458-555558-565658-575758-586X68-606068-616168-626268-636368-646468-656568-666668-676768-687X78-707078-717178-727278-737378-747478-757578-767678-777778-788X88-808088-818188-828288-838388-848488-858588-868688-878788-88"
    test "test8717" (lazy(sprintf "08-00%d08-01%d08-02%d08-03%d08-04%d08-05%d08-06%d08-07%d08-08%a18-10%d18-11%d18-12%d18-13%d18-14%d18-15%d18-16%d18-17%d18-18%a28-20%d28-21%d28-22%d28-23%d28-24%d28-25%d28-26%d28-27%d28-28%a38-30%d38-31%d38-32%d38-33%d38-34%d38-35%d38-36%d38-37%d38-38%a48-40%d48-41%d48-42%d48-43%d48-44%d48-45%d48-46%d48-47%d48-48%a58-50%d58-51%d58-52%d58-53%d58-54%d58-55%d58-56%d58-57%d58-58%a68-60%d68-61%d68-62%d68-63%d68-64%d68-65%d68-66%d68-67%d68-68%a78-70%d78-71%d78-72%d78-73%d78-74%d78-75%d78-76%d78-77%d78-78%a88-80%d88-81%d88-82%d88-83%d88-84%d88-85%d88-86%d88-87%d88-88%a_TAIL" 0 1 2 3 4 5 6 7  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 27  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36 37  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46 47  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55 56 57  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 65 66 67  (fun _ v -> (string v) + "X") 7  70 71 72 73 74 75 76 77  (fun _ v -> (string v) + "X") 8  80 81 82 83 84 85 86 87 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "08-00008-01108-02208-03308-04408-05508-06608-07708-081X18-101018-111118-121218-131318-141418-151518-161618-171718-182X28-202028-212128-222228-232328-242428-252528-262628-272728-283X38-303038-313138-323238-333338-343438-353538-363638-373738-384X48-404048-414148-424248-434348-444448-454548-464648-474748-485X58-505058-515158-525258-535358-545458-555558-565658-575758-586X68-606068-616168-626268-636368-646468-656568-666668-676768-687X78-707078-717178-727278-737378-747478-757578-767678-777778-788X88-808088-818188-828288-838388-848488-858588-868688-878788-88ReadX_TAIL"
    test "test8718" (lazy(sprintf "08-00%d08-01%d08-02%d08-03%d08-04%d08-05%d08-06%d08-07%d08-08%d08-09%a18-10%d18-11%d18-12%d18-13%d18-14%d18-15%d18-16%d18-17%d18-18%d18-19%a28-20%d28-21%d28-22%d28-23%d28-24%d28-25%d28-26%d28-27%d28-28%d28-29%a38-30%d38-31%d38-32%d38-33%d38-34%d38-35%d38-36%d38-37%d38-38%d38-39%a48-40%d48-41%d48-42%d48-43%d48-44%d48-45%d48-46%d48-47%d48-48%d48-49%a58-50%d58-51%d58-52%d58-53%d58-54%d58-55%d58-56%d58-57%d58-58%d58-59%a68-60%d68-61%d68-62%d68-63%d68-64%d68-65%d68-66%d68-67%d68-68%d68-69%a78-70%d78-71%d78-72%d78-73%d78-74%d78-75%d78-76%d78-77%d78-78%d78-79%a88-80%d88-81%d88-82%d88-83%d88-84%d88-85%d88-86%d88-87%d88-88%d88-89" 0 1 2 3 4 5 6 7 8  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17 18  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 27 28  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36 37 38  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46 47 48  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55 56 57 58  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 65 66 67 68  (fun _ v -> (string v) + "X") 7  70 71 72 73 74 75 76 77 78  (fun _ v -> (string v) + "X") 8  80 81 82 83 84 85 86 87 88 )) "08-00008-01108-02208-03308-04408-05508-06608-07708-08808-091X18-101018-111118-121218-131318-141418-151518-161618-171718-181818-192X28-202028-212128-222228-232328-242428-252528-262628-272728-282828-293X38-303038-313138-323238-333338-343438-353538-363638-373738-383838-394X48-404048-414148-424248-434348-444448-454548-464648-474748-484848-495X58-505058-515158-525258-535358-545458-555558-565658-575758-585858-596X68-606068-616168-626268-636368-646468-656568-666668-676768-686868-697X78-707078-717178-727278-737378-747478-757578-767678-777778-787878-798X88-808088-818188-828288-838388-848488-858588-868688-878788-888888-89"
    test "test8719" (lazy(sprintf "08-00%d08-01%d08-02%d08-03%d08-04%d08-05%d08-06%d08-07%d08-08%d08-09%a18-10%d18-11%d18-12%d18-13%d18-14%d18-15%d18-16%d18-17%d18-18%d18-19%a28-20%d28-21%d28-22%d28-23%d28-24%d28-25%d28-26%d28-27%d28-28%d28-29%a38-30%d38-31%d38-32%d38-33%d38-34%d38-35%d38-36%d38-37%d38-38%d38-39%a48-40%d48-41%d48-42%d48-43%d48-44%d48-45%d48-46%d48-47%d48-48%d48-49%a58-50%d58-51%d58-52%d58-53%d58-54%d58-55%d58-56%d58-57%d58-58%d58-59%a68-60%d68-61%d68-62%d68-63%d68-64%d68-65%d68-66%d68-67%d68-68%d68-69%a78-70%d78-71%d78-72%d78-73%d78-74%d78-75%d78-76%d78-77%d78-78%d78-79%a88-80%d88-81%d88-82%d88-83%d88-84%d88-85%d88-86%d88-87%d88-88%d88-89%a_TAIL" 0 1 2 3 4 5 6 7 8  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17 18  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 27 28  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36 37 38  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46 47 48  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55 56 57 58  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 65 66 67 68  (fun _ v -> (string v) + "X") 7  70 71 72 73 74 75 76 77 78  (fun _ v -> (string v) + "X") 8  80 81 82 83 84 85 86 87 88 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "08-00008-01108-02208-03308-04408-05508-06608-07708-08808-091X18-101018-111118-121218-131318-141418-151518-161618-171718-181818-192X28-202028-212128-222228-232328-242428-252528-262628-272728-282828-293X38-303038-313138-323238-333338-343438-353538-363638-373738-383838-394X48-404048-414148-424248-434348-444448-454548-464648-474748-484848-495X58-505058-515158-525258-535358-545458-555558-565658-575758-585858-596X68-606068-616168-626268-636368-646468-656568-666668-676768-686868-697X78-707078-717178-727278-737378-747478-757578-767678-777778-787878-798X88-808088-818188-828288-838388-848488-858588-868688-878788-888888-89ReadX_TAIL"
    test "test8720" (lazy(sprintf "09-00%d09-01%a19-10%d19-11%a29-20%d29-21%a39-30%d39-31%a49-40%d49-41%a59-50%d59-51%a69-60%d69-61%a79-70%d79-71%a89-80%d89-81%a99-90%d99-91" 0  (fun _ v -> (string v) + "X") 1  10  (fun _ v -> (string v) + "X") 2  20  (fun _ v -> (string v) + "X") 3  30  (fun _ v -> (string v) + "X") 4  40  (fun _ v -> (string v) + "X") 5  50  (fun _ v -> (string v) + "X") 6  60  (fun _ v -> (string v) + "X") 7  70  (fun _ v -> (string v) + "X") 8  80  (fun _ v -> (string v) + "X") 9  90 )) "09-00009-011X19-101019-112X29-202029-213X39-303039-314X49-404049-415X59-505059-516X69-606069-617X79-707079-718X89-808089-819X99-909099-91"
    test "test8721" (lazy(sprintf "09-00%d09-01%a19-10%d19-11%a29-20%d29-21%a39-30%d39-31%a49-40%d49-41%a59-50%d59-51%a69-60%d69-61%a79-70%d79-71%a89-80%d89-81%a99-90%d99-91%a_TAIL" 0  (fun _ v -> (string v) + "X") 1  10  (fun _ v -> (string v) + "X") 2  20  (fun _ v -> (string v) + "X") 3  30  (fun _ v -> (string v) + "X") 4  40  (fun _ v -> (string v) + "X") 5  50  (fun _ v -> (string v) + "X") 6  60  (fun _ v -> (string v) + "X") 7  70  (fun _ v -> (string v) + "X") 8  80  (fun _ v -> (string v) + "X") 9  90 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "09-00009-011X19-101019-112X29-202029-213X39-303039-314X49-404049-415X59-505059-516X69-606069-617X79-707079-718X89-808089-819X99-909099-91ReadX_TAIL"
    test "test8722" (lazy(sprintf "09-00%d09-01%d09-02%a19-10%d19-11%d19-12%a29-20%d29-21%d29-22%a39-30%d39-31%d39-32%a49-40%d49-41%d49-42%a59-50%d59-51%d59-52%a69-60%d69-61%d69-62%a79-70%d79-71%d79-72%a89-80%d89-81%d89-82%a99-90%d99-91%d99-92" 0 1  (fun _ v -> (string v) + "X") 1  10 11  (fun _ v -> (string v) + "X") 2  20 21  (fun _ v -> (string v) + "X") 3  30 31  (fun _ v -> (string v) + "X") 4  40 41  (fun _ v -> (string v) + "X") 5  50 51  (fun _ v -> (string v) + "X") 6  60 61  (fun _ v -> (string v) + "X") 7  70 71  (fun _ v -> (string v) + "X") 8  80 81  (fun _ v -> (string v) + "X") 9  90 91 )) "09-00009-01109-021X19-101019-111119-122X29-202029-212129-223X39-303039-313139-324X49-404049-414149-425X59-505059-515159-526X69-606069-616169-627X79-707079-717179-728X89-808089-818189-829X99-909099-919199-92"
    test "test8723" (lazy(sprintf "09-00%d09-01%d09-02%a19-10%d19-11%d19-12%a29-20%d29-21%d29-22%a39-30%d39-31%d39-32%a49-40%d49-41%d49-42%a59-50%d59-51%d59-52%a69-60%d69-61%d69-62%a79-70%d79-71%d79-72%a89-80%d89-81%d89-82%a99-90%d99-91%d99-92%a_TAIL" 0 1  (fun _ v -> (string v) + "X") 1  10 11  (fun _ v -> (string v) + "X") 2  20 21  (fun _ v -> (string v) + "X") 3  30 31  (fun _ v -> (string v) + "X") 4  40 41  (fun _ v -> (string v) + "X") 5  50 51  (fun _ v -> (string v) + "X") 6  60 61  (fun _ v -> (string v) + "X") 7  70 71  (fun _ v -> (string v) + "X") 8  80 81  (fun _ v -> (string v) + "X") 9  90 91 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "09-00009-01109-021X19-101019-111119-122X29-202029-212129-223X39-303039-313139-324X49-404049-414149-425X59-505059-515159-526X69-606069-616169-627X79-707079-717179-728X89-808089-818189-829X99-909099-919199-92ReadX_TAIL"
    test "test8724" (lazy(sprintf "09-00%d09-01%d09-02%d09-03%a19-10%d19-11%d19-12%d19-13%a29-20%d29-21%d29-22%d29-23%a39-30%d39-31%d39-32%d39-33%a49-40%d49-41%d49-42%d49-43%a59-50%d59-51%d59-52%d59-53%a69-60%d69-61%d69-62%d69-63%a79-70%d79-71%d79-72%d79-73%a89-80%d89-81%d89-82%d89-83%a99-90%d99-91%d99-92%d99-93" 0 1 2  (fun _ v -> (string v) + "X") 1  10 11 12  (fun _ v -> (string v) + "X") 2  20 21 22  (fun _ v -> (string v) + "X") 3  30 31 32  (fun _ v -> (string v) + "X") 4  40 41 42  (fun _ v -> (string v) + "X") 5  50 51 52  (fun _ v -> (string v) + "X") 6  60 61 62  (fun _ v -> (string v) + "X") 7  70 71 72  (fun _ v -> (string v) + "X") 8  80 81 82  (fun _ v -> (string v) + "X") 9  90 91 92 )) "09-00009-01109-02209-031X19-101019-111119-121219-132X29-202029-212129-222229-233X39-303039-313139-323239-334X49-404049-414149-424249-435X59-505059-515159-525259-536X69-606069-616169-626269-637X79-707079-717179-727279-738X89-808089-818189-828289-839X99-909099-919199-929299-93"
    test "test8725" (lazy(sprintf "09-00%d09-01%d09-02%d09-03%a19-10%d19-11%d19-12%d19-13%a29-20%d29-21%d29-22%d29-23%a39-30%d39-31%d39-32%d39-33%a49-40%d49-41%d49-42%d49-43%a59-50%d59-51%d59-52%d59-53%a69-60%d69-61%d69-62%d69-63%a79-70%d79-71%d79-72%d79-73%a89-80%d89-81%d89-82%d89-83%a99-90%d99-91%d99-92%d99-93%a_TAIL" 0 1 2  (fun _ v -> (string v) + "X") 1  10 11 12  (fun _ v -> (string v) + "X") 2  20 21 22  (fun _ v -> (string v) + "X") 3  30 31 32  (fun _ v -> (string v) + "X") 4  40 41 42  (fun _ v -> (string v) + "X") 5  50 51 52  (fun _ v -> (string v) + "X") 6  60 61 62  (fun _ v -> (string v) + "X") 7  70 71 72  (fun _ v -> (string v) + "X") 8  80 81 82  (fun _ v -> (string v) + "X") 9  90 91 92 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "09-00009-01109-02209-031X19-101019-111119-121219-132X29-202029-212129-222229-233X39-303039-313139-323239-334X49-404049-414149-424249-435X59-505059-515159-525259-536X69-606069-616169-626269-637X79-707079-717179-727279-738X89-808089-818189-828289-839X99-909099-919199-929299-93ReadX_TAIL"
    test "test8726" (lazy(sprintf "09-00%d09-01%d09-02%d09-03%d09-04%a19-10%d19-11%d19-12%d19-13%d19-14%a29-20%d29-21%d29-22%d29-23%d29-24%a39-30%d39-31%d39-32%d39-33%d39-34%a49-40%d49-41%d49-42%d49-43%d49-44%a59-50%d59-51%d59-52%d59-53%d59-54%a69-60%d69-61%d69-62%d69-63%d69-64%a79-70%d79-71%d79-72%d79-73%d79-74%a89-80%d89-81%d89-82%d89-83%d89-84%a99-90%d99-91%d99-92%d99-93%d99-94" 0 1 2 3  (fun _ v -> (string v) + "X") 1  10 11 12 13  (fun _ v -> (string v) + "X") 2  20 21 22 23  (fun _ v -> (string v) + "X") 3  30 31 32 33  (fun _ v -> (string v) + "X") 4  40 41 42 43  (fun _ v -> (string v) + "X") 5  50 51 52 53  (fun _ v -> (string v) + "X") 6  60 61 62 63  (fun _ v -> (string v) + "X") 7  70 71 72 73  (fun _ v -> (string v) + "X") 8  80 81 82 83  (fun _ v -> (string v) + "X") 9  90 91 92 93 )) "09-00009-01109-02209-03309-041X19-101019-111119-121219-131319-142X29-202029-212129-222229-232329-243X39-303039-313139-323239-333339-344X49-404049-414149-424249-434349-445X59-505059-515159-525259-535359-546X69-606069-616169-626269-636369-647X79-707079-717179-727279-737379-748X89-808089-818189-828289-838389-849X99-909099-919199-929299-939399-94"
    test "test8727" (lazy(sprintf "09-00%d09-01%d09-02%d09-03%d09-04%a19-10%d19-11%d19-12%d19-13%d19-14%a29-20%d29-21%d29-22%d29-23%d29-24%a39-30%d39-31%d39-32%d39-33%d39-34%a49-40%d49-41%d49-42%d49-43%d49-44%a59-50%d59-51%d59-52%d59-53%d59-54%a69-60%d69-61%d69-62%d69-63%d69-64%a79-70%d79-71%d79-72%d79-73%d79-74%a89-80%d89-81%d89-82%d89-83%d89-84%a99-90%d99-91%d99-92%d99-93%d99-94%a_TAIL" 0 1 2 3  (fun _ v -> (string v) + "X") 1  10 11 12 13  (fun _ v -> (string v) + "X") 2  20 21 22 23  (fun _ v -> (string v) + "X") 3  30 31 32 33  (fun _ v -> (string v) + "X") 4  40 41 42 43  (fun _ v -> (string v) + "X") 5  50 51 52 53  (fun _ v -> (string v) + "X") 6  60 61 62 63  (fun _ v -> (string v) + "X") 7  70 71 72 73  (fun _ v -> (string v) + "X") 8  80 81 82 83  (fun _ v -> (string v) + "X") 9  90 91 92 93 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "09-00009-01109-02209-03309-041X19-101019-111119-121219-131319-142X29-202029-212129-222229-232329-243X39-303039-313139-323239-333339-344X49-404049-414149-424249-434349-445X59-505059-515159-525259-535359-546X69-606069-616169-626269-636369-647X79-707079-717179-727279-737379-748X89-808089-818189-828289-838389-849X99-909099-919199-929299-939399-94ReadX_TAIL"
    test "test8728" (lazy(sprintf "09-00%d09-01%d09-02%d09-03%d09-04%d09-05%a19-10%d19-11%d19-12%d19-13%d19-14%d19-15%a29-20%d29-21%d29-22%d29-23%d29-24%d29-25%a39-30%d39-31%d39-32%d39-33%d39-34%d39-35%a49-40%d49-41%d49-42%d49-43%d49-44%d49-45%a59-50%d59-51%d59-52%d59-53%d59-54%d59-55%a69-60%d69-61%d69-62%d69-63%d69-64%d69-65%a79-70%d79-71%d79-72%d79-73%d79-74%d79-75%a89-80%d89-81%d89-82%d89-83%d89-84%d89-85%a99-90%d99-91%d99-92%d99-93%d99-94%d99-95" 0 1 2 3 4  (fun _ v -> (string v) + "X") 1  10 11 12 13 14  (fun _ v -> (string v) + "X") 2  20 21 22 23 24  (fun _ v -> (string v) + "X") 3  30 31 32 33 34  (fun _ v -> (string v) + "X") 4  40 41 42 43 44  (fun _ v -> (string v) + "X") 5  50 51 52 53 54  (fun _ v -> (string v) + "X") 6  60 61 62 63 64  (fun _ v -> (string v) + "X") 7  70 71 72 73 74  (fun _ v -> (string v) + "X") 8  80 81 82 83 84  (fun _ v -> (string v) + "X") 9  90 91 92 93 94 )) "09-00009-01109-02209-03309-04409-051X19-101019-111119-121219-131319-141419-152X29-202029-212129-222229-232329-242429-253X39-303039-313139-323239-333339-343439-354X49-404049-414149-424249-434349-444449-455X59-505059-515159-525259-535359-545459-556X69-606069-616169-626269-636369-646469-657X79-707079-717179-727279-737379-747479-758X89-808089-818189-828289-838389-848489-859X99-909099-919199-929299-939399-949499-95"
    test "test8729" (lazy(sprintf "09-00%d09-01%d09-02%d09-03%d09-04%d09-05%a19-10%d19-11%d19-12%d19-13%d19-14%d19-15%a29-20%d29-21%d29-22%d29-23%d29-24%d29-25%a39-30%d39-31%d39-32%d39-33%d39-34%d39-35%a49-40%d49-41%d49-42%d49-43%d49-44%d49-45%a59-50%d59-51%d59-52%d59-53%d59-54%d59-55%a69-60%d69-61%d69-62%d69-63%d69-64%d69-65%a79-70%d79-71%d79-72%d79-73%d79-74%d79-75%a89-80%d89-81%d89-82%d89-83%d89-84%d89-85%a99-90%d99-91%d99-92%d99-93%d99-94%d99-95%a_TAIL" 0 1 2 3 4  (fun _ v -> (string v) + "X") 1  10 11 12 13 14  (fun _ v -> (string v) + "X") 2  20 21 22 23 24  (fun _ v -> (string v) + "X") 3  30 31 32 33 34  (fun _ v -> (string v) + "X") 4  40 41 42 43 44  (fun _ v -> (string v) + "X") 5  50 51 52 53 54  (fun _ v -> (string v) + "X") 6  60 61 62 63 64  (fun _ v -> (string v) + "X") 7  70 71 72 73 74  (fun _ v -> (string v) + "X") 8  80 81 82 83 84  (fun _ v -> (string v) + "X") 9  90 91 92 93 94 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "09-00009-01109-02209-03309-04409-051X19-101019-111119-121219-131319-141419-152X29-202029-212129-222229-232329-242429-253X39-303039-313139-323239-333339-343439-354X49-404049-414149-424249-434349-444449-455X59-505059-515159-525259-535359-545459-556X69-606069-616169-626269-636369-646469-657X79-707079-717179-727279-737379-747479-758X89-808089-818189-828289-838389-848489-859X99-909099-919199-929299-939399-949499-95ReadX_TAIL"
    test "test8730" (lazy(sprintf "09-00%d09-01%d09-02%d09-03%d09-04%d09-05%d09-06%a19-10%d19-11%d19-12%d19-13%d19-14%d19-15%d19-16%a29-20%d29-21%d29-22%d29-23%d29-24%d29-25%d29-26%a39-30%d39-31%d39-32%d39-33%d39-34%d39-35%d39-36%a49-40%d49-41%d49-42%d49-43%d49-44%d49-45%d49-46%a59-50%d59-51%d59-52%d59-53%d59-54%d59-55%d59-56%a69-60%d69-61%d69-62%d69-63%d69-64%d69-65%d69-66%a79-70%d79-71%d79-72%d79-73%d79-74%d79-75%d79-76%a89-80%d89-81%d89-82%d89-83%d89-84%d89-85%d89-86%a99-90%d99-91%d99-92%d99-93%d99-94%d99-95%d99-96" 0 1 2 3 4 5  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 65  (fun _ v -> (string v) + "X") 7  70 71 72 73 74 75  (fun _ v -> (string v) + "X") 8  80 81 82 83 84 85  (fun _ v -> (string v) + "X") 9  90 91 92 93 94 95 )) "09-00009-01109-02209-03309-04409-05509-061X19-101019-111119-121219-131319-141419-151519-162X29-202029-212129-222229-232329-242429-252529-263X39-303039-313139-323239-333339-343439-353539-364X49-404049-414149-424249-434349-444449-454549-465X59-505059-515159-525259-535359-545459-555559-566X69-606069-616169-626269-636369-646469-656569-667X79-707079-717179-727279-737379-747479-757579-768X89-808089-818189-828289-838389-848489-858589-869X99-909099-919199-929299-939399-949499-959599-96"
    test "test8731" (lazy(sprintf "09-00%d09-01%d09-02%d09-03%d09-04%d09-05%d09-06%a19-10%d19-11%d19-12%d19-13%d19-14%d19-15%d19-16%a29-20%d29-21%d29-22%d29-23%d29-24%d29-25%d29-26%a39-30%d39-31%d39-32%d39-33%d39-34%d39-35%d39-36%a49-40%d49-41%d49-42%d49-43%d49-44%d49-45%d49-46%a59-50%d59-51%d59-52%d59-53%d59-54%d59-55%d59-56%a69-60%d69-61%d69-62%d69-63%d69-64%d69-65%d69-66%a79-70%d79-71%d79-72%d79-73%d79-74%d79-75%d79-76%a89-80%d89-81%d89-82%d89-83%d89-84%d89-85%d89-86%a99-90%d99-91%d99-92%d99-93%d99-94%d99-95%d99-96%a_TAIL" 0 1 2 3 4 5  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 65  (fun _ v -> (string v) + "X") 7  70 71 72 73 74 75  (fun _ v -> (string v) + "X") 8  80 81 82 83 84 85  (fun _ v -> (string v) + "X") 9  90 91 92 93 94 95 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "09-00009-01109-02209-03309-04409-05509-061X19-101019-111119-121219-131319-141419-151519-162X29-202029-212129-222229-232329-242429-252529-263X39-303039-313139-323239-333339-343439-353539-364X49-404049-414149-424249-434349-444449-454549-465X59-505059-515159-525259-535359-545459-555559-566X69-606069-616169-626269-636369-646469-656569-667X79-707079-717179-727279-737379-747479-757579-768X89-808089-818189-828289-838389-848489-858589-869X99-909099-919199-929299-939399-949499-959599-96ReadX_TAIL"
    test "test8732" (lazy(sprintf "09-00%d09-01%d09-02%d09-03%d09-04%d09-05%d09-06%d09-07%a19-10%d19-11%d19-12%d19-13%d19-14%d19-15%d19-16%d19-17%a29-20%d29-21%d29-22%d29-23%d29-24%d29-25%d29-26%d29-27%a39-30%d39-31%d39-32%d39-33%d39-34%d39-35%d39-36%d39-37%a49-40%d49-41%d49-42%d49-43%d49-44%d49-45%d49-46%d49-47%a59-50%d59-51%d59-52%d59-53%d59-54%d59-55%d59-56%d59-57%a69-60%d69-61%d69-62%d69-63%d69-64%d69-65%d69-66%d69-67%a79-70%d79-71%d79-72%d79-73%d79-74%d79-75%d79-76%d79-77%a89-80%d89-81%d89-82%d89-83%d89-84%d89-85%d89-86%d89-87%a99-90%d99-91%d99-92%d99-93%d99-94%d99-95%d99-96%d99-97" 0 1 2 3 4 5 6  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55 56  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 65 66  (fun _ v -> (string v) + "X") 7  70 71 72 73 74 75 76  (fun _ v -> (string v) + "X") 8  80 81 82 83 84 85 86  (fun _ v -> (string v) + "X") 9  90 91 92 93 94 95 96 )) "09-00009-01109-02209-03309-04409-05509-06609-071X19-101019-111119-121219-131319-141419-151519-161619-172X29-202029-212129-222229-232329-242429-252529-262629-273X39-303039-313139-323239-333339-343439-353539-363639-374X49-404049-414149-424249-434349-444449-454549-464649-475X59-505059-515159-525259-535359-545459-555559-565659-576X69-606069-616169-626269-636369-646469-656569-666669-677X79-707079-717179-727279-737379-747479-757579-767679-778X89-808089-818189-828289-838389-848489-858589-868689-879X99-909099-919199-929299-939399-949499-959599-969699-97"
    test "test8733" (lazy(sprintf "09-00%d09-01%d09-02%d09-03%d09-04%d09-05%d09-06%d09-07%a19-10%d19-11%d19-12%d19-13%d19-14%d19-15%d19-16%d19-17%a29-20%d29-21%d29-22%d29-23%d29-24%d29-25%d29-26%d29-27%a39-30%d39-31%d39-32%d39-33%d39-34%d39-35%d39-36%d39-37%a49-40%d49-41%d49-42%d49-43%d49-44%d49-45%d49-46%d49-47%a59-50%d59-51%d59-52%d59-53%d59-54%d59-55%d59-56%d59-57%a69-60%d69-61%d69-62%d69-63%d69-64%d69-65%d69-66%d69-67%a79-70%d79-71%d79-72%d79-73%d79-74%d79-75%d79-76%d79-77%a89-80%d89-81%d89-82%d89-83%d89-84%d89-85%d89-86%d89-87%a99-90%d99-91%d99-92%d99-93%d99-94%d99-95%d99-96%d99-97%a_TAIL" 0 1 2 3 4 5 6  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55 56  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 65 66  (fun _ v -> (string v) + "X") 7  70 71 72 73 74 75 76  (fun _ v -> (string v) + "X") 8  80 81 82 83 84 85 86  (fun _ v -> (string v) + "X") 9  90 91 92 93 94 95 96 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "09-00009-01109-02209-03309-04409-05509-06609-071X19-101019-111119-121219-131319-141419-151519-161619-172X29-202029-212129-222229-232329-242429-252529-262629-273X39-303039-313139-323239-333339-343439-353539-363639-374X49-404049-414149-424249-434349-444449-454549-464649-475X59-505059-515159-525259-535359-545459-555559-565659-576X69-606069-616169-626269-636369-646469-656569-666669-677X79-707079-717179-727279-737379-747479-757579-767679-778X89-808089-818189-828289-838389-848489-858589-868689-879X99-909099-919199-929299-939399-949499-959599-969699-97ReadX_TAIL"
    test "test8734" (lazy(sprintf "09-00%d09-01%d09-02%d09-03%d09-04%d09-05%d09-06%d09-07%d09-08%a19-10%d19-11%d19-12%d19-13%d19-14%d19-15%d19-16%d19-17%d19-18%a29-20%d29-21%d29-22%d29-23%d29-24%d29-25%d29-26%d29-27%d29-28%a39-30%d39-31%d39-32%d39-33%d39-34%d39-35%d39-36%d39-37%d39-38%a49-40%d49-41%d49-42%d49-43%d49-44%d49-45%d49-46%d49-47%d49-48%a59-50%d59-51%d59-52%d59-53%d59-54%d59-55%d59-56%d59-57%d59-58%a69-60%d69-61%d69-62%d69-63%d69-64%d69-65%d69-66%d69-67%d69-68%a79-70%d79-71%d79-72%d79-73%d79-74%d79-75%d79-76%d79-77%d79-78%a89-80%d89-81%d89-82%d89-83%d89-84%d89-85%d89-86%d89-87%d89-88%a99-90%d99-91%d99-92%d99-93%d99-94%d99-95%d99-96%d99-97%d99-98" 0 1 2 3 4 5 6 7  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 27  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36 37  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46 47  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55 56 57  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 65 66 67  (fun _ v -> (string v) + "X") 7  70 71 72 73 74 75 76 77  (fun _ v -> (string v) + "X") 8  80 81 82 83 84 85 86 87  (fun _ v -> (string v) + "X") 9  90 91 92 93 94 95 96 97 )) "09-00009-01109-02209-03309-04409-05509-06609-07709-081X19-101019-111119-121219-131319-141419-151519-161619-171719-182X29-202029-212129-222229-232329-242429-252529-262629-272729-283X39-303039-313139-323239-333339-343439-353539-363639-373739-384X49-404049-414149-424249-434349-444449-454549-464649-474749-485X59-505059-515159-525259-535359-545459-555559-565659-575759-586X69-606069-616169-626269-636369-646469-656569-666669-676769-687X79-707079-717179-727279-737379-747479-757579-767679-777779-788X89-808089-818189-828289-838389-848489-858589-868689-878789-889X99-909099-919199-929299-939399-949499-959599-969699-979799-98"
    test "test8735" (lazy(sprintf "09-00%d09-01%d09-02%d09-03%d09-04%d09-05%d09-06%d09-07%d09-08%a19-10%d19-11%d19-12%d19-13%d19-14%d19-15%d19-16%d19-17%d19-18%a29-20%d29-21%d29-22%d29-23%d29-24%d29-25%d29-26%d29-27%d29-28%a39-30%d39-31%d39-32%d39-33%d39-34%d39-35%d39-36%d39-37%d39-38%a49-40%d49-41%d49-42%d49-43%d49-44%d49-45%d49-46%d49-47%d49-48%a59-50%d59-51%d59-52%d59-53%d59-54%d59-55%d59-56%d59-57%d59-58%a69-60%d69-61%d69-62%d69-63%d69-64%d69-65%d69-66%d69-67%d69-68%a79-70%d79-71%d79-72%d79-73%d79-74%d79-75%d79-76%d79-77%d79-78%a89-80%d89-81%d89-82%d89-83%d89-84%d89-85%d89-86%d89-87%d89-88%a99-90%d99-91%d99-92%d99-93%d99-94%d99-95%d99-96%d99-97%d99-98%a_TAIL" 0 1 2 3 4 5 6 7  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 27  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36 37  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46 47  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55 56 57  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 65 66 67  (fun _ v -> (string v) + "X") 7  70 71 72 73 74 75 76 77  (fun _ v -> (string v) + "X") 8  80 81 82 83 84 85 86 87  (fun _ v -> (string v) + "X") 9  90 91 92 93 94 95 96 97 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "09-00009-01109-02209-03309-04409-05509-06609-07709-081X19-101019-111119-121219-131319-141419-151519-161619-171719-182X29-202029-212129-222229-232329-242429-252529-262629-272729-283X39-303039-313139-323239-333339-343439-353539-363639-373739-384X49-404049-414149-424249-434349-444449-454549-464649-474749-485X59-505059-515159-525259-535359-545459-555559-565659-575759-586X69-606069-616169-626269-636369-646469-656569-666669-676769-687X79-707079-717179-727279-737379-747479-757579-767679-777779-788X89-808089-818189-828289-838389-848489-858589-868689-878789-889X99-909099-919199-929299-939399-949499-959599-969699-979799-98ReadX_TAIL"
    test "test8736" (lazy(sprintf "09-00%d09-01%d09-02%d09-03%d09-04%d09-05%d09-06%d09-07%d09-08%d09-09%a19-10%d19-11%d19-12%d19-13%d19-14%d19-15%d19-16%d19-17%d19-18%d19-19%a29-20%d29-21%d29-22%d29-23%d29-24%d29-25%d29-26%d29-27%d29-28%d29-29%a39-30%d39-31%d39-32%d39-33%d39-34%d39-35%d39-36%d39-37%d39-38%d39-39%a49-40%d49-41%d49-42%d49-43%d49-44%d49-45%d49-46%d49-47%d49-48%d49-49%a59-50%d59-51%d59-52%d59-53%d59-54%d59-55%d59-56%d59-57%d59-58%d59-59%a69-60%d69-61%d69-62%d69-63%d69-64%d69-65%d69-66%d69-67%d69-68%d69-69%a79-70%d79-71%d79-72%d79-73%d79-74%d79-75%d79-76%d79-77%d79-78%d79-79%a89-80%d89-81%d89-82%d89-83%d89-84%d89-85%d89-86%d89-87%d89-88%d89-89%a99-90%d99-91%d99-92%d99-93%d99-94%d99-95%d99-96%d99-97%d99-98%d99-99" 0 1 2 3 4 5 6 7 8  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17 18  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 27 28  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36 37 38  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46 47 48  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55 56 57 58  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 65 66 67 68  (fun _ v -> (string v) + "X") 7  70 71 72 73 74 75 76 77 78  (fun _ v -> (string v) + "X") 8  80 81 82 83 84 85 86 87 88  (fun _ v -> (string v) + "X") 9  90 91 92 93 94 95 96 97 98 )) "09-00009-01109-02209-03309-04409-05509-06609-07709-08809-091X19-101019-111119-121219-131319-141419-151519-161619-171719-181819-192X29-202029-212129-222229-232329-242429-252529-262629-272729-282829-293X39-303039-313139-323239-333339-343439-353539-363639-373739-383839-394X49-404049-414149-424249-434349-444449-454549-464649-474749-484849-495X59-505059-515159-525259-535359-545459-555559-565659-575759-585859-596X69-606069-616169-626269-636369-646469-656569-666669-676769-686869-697X79-707079-717179-727279-737379-747479-757579-767679-777779-787879-798X89-808089-818189-828289-838389-848489-858589-868689-878789-888889-899X99-909099-919199-929299-939399-949499-959599-969699-979799-989899-99"
    test "test8737" (lazy(sprintf "09-00%d09-01%d09-02%d09-03%d09-04%d09-05%d09-06%d09-07%d09-08%d09-09%a19-10%d19-11%d19-12%d19-13%d19-14%d19-15%d19-16%d19-17%d19-18%d19-19%a29-20%d29-21%d29-22%d29-23%d29-24%d29-25%d29-26%d29-27%d29-28%d29-29%a39-30%d39-31%d39-32%d39-33%d39-34%d39-35%d39-36%d39-37%d39-38%d39-39%a49-40%d49-41%d49-42%d49-43%d49-44%d49-45%d49-46%d49-47%d49-48%d49-49%a59-50%d59-51%d59-52%d59-53%d59-54%d59-55%d59-56%d59-57%d59-58%d59-59%a69-60%d69-61%d69-62%d69-63%d69-64%d69-65%d69-66%d69-67%d69-68%d69-69%a79-70%d79-71%d79-72%d79-73%d79-74%d79-75%d79-76%d79-77%d79-78%d79-79%a89-80%d89-81%d89-82%d89-83%d89-84%d89-85%d89-86%d89-87%d89-88%d89-89%a99-90%d99-91%d99-92%d99-93%d99-94%d99-95%d99-96%d99-97%d99-98%d99-99%a_TAIL" 0 1 2 3 4 5 6 7 8  (fun _ v -> (string v) + "X") 1  10 11 12 13 14 15 16 17 18  (fun _ v -> (string v) + "X") 2  20 21 22 23 24 25 26 27 28  (fun _ v -> (string v) + "X") 3  30 31 32 33 34 35 36 37 38  (fun _ v -> (string v) + "X") 4  40 41 42 43 44 45 46 47 48  (fun _ v -> (string v) + "X") 5  50 51 52 53 54 55 56 57 58  (fun _ v -> (string v) + "X") 6  60 61 62 63 64 65 66 67 68  (fun _ v -> (string v) + "X") 7  70 71 72 73 74 75 76 77 78  (fun _ v -> (string v) + "X") 8  80 81 82 83 84 85 86 87 88  (fun _ v -> (string v) + "X") 9  90 91 92 93 94 95 96 97 98 (fun _ v -> (string v) + "X") System.IO.FileShare.Read  )) "09-00009-01109-02209-03309-04409-05509-06609-07709-08809-091X19-101019-111119-121219-131319-141419-151519-161619-171719-181819-192X29-202029-212129-222229-232329-242429-252529-262629-272729-282829-293X39-303039-313139-323239-333339-343439-353539-363639-373739-383839-394X49-404049-414149-424249-434349-444449-454549-464649-474749-484849-495X59-505059-515159-525259-535359-545459-555559-565659-575759-585859-596X69-606069-616169-626269-636369-646469-656569-666669-676769-686869-697X79-707079-717179-727279-737379-747479-757579-767679-777779-787879-798X89-808089-818189-828289-838389-848489-858589-868689-878789-888889-899X99-909099-919199-929299-939399-949499-959599-969699-979799-989899-99ReadX_TAIL"
#endif
func0()
func1000()
func2000()
#if COMPILED
func3000()
func4000()
func5000()
func6000()
func7000()
func8000()
#endif

#if TESTS_AS_APP
let RUN() = !failures
#else
let aa =
  match !failures with 
  | [] -> 
      stdout.WriteLine "Test Passed"
      System.IO.File.WriteAllText("test.ok","ok")
      exit 0
  | _ -> 
      stdout.WriteLine "Test Failed"
      exit 1
#endif

