﻿// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT license. See LICENSE file in the project root for full license information.

namespace vstest.console.Internal
{
    using Microsoft.Extensions.FileSystemGlobbing;
    using Microsoft.Extensions.FileSystemGlobbing.Abstractions;
    using Microsoft.VisualStudio.TestPlatform.CommandLine;
    using Microsoft.VisualStudio.TestPlatform.ObjectModel;
    using Microsoft.VisualStudio.TestPlatform.Utilities.Helpers;
    using Microsoft.VisualStudio.TestPlatform.Utilities.Helpers.Interfaces;
    using System;
    using System.Collections.Generic;
    using System.Globalization;
    using System.IO;

    using CommandLineResources = Microsoft.VisualStudio.TestPlatform.CommandLine.Resources.Resources;

    /// <summary>
    /// Class for getting matching files from wild card pattern file name
    /// Microsoft.Extensions.FileSystemGlobbing methods used to get matching file names
    /// </summary>
    public class FilePatternParser
    {
        private Matcher matcher;
        private IFileHelper fileHelper;
        private char[] wildCardCharacters = { '*' };

        public FilePatternParser()
            : this(new Matcher(), new FileHelper())
        {
        }

        internal FilePatternParser(Matcher matcher, IFileHelper fileHelper)
        {
            this.matcher = matcher;
            this.fileHelper = fileHelper;
        }

        /// <summary>
        /// Used to get matching files with pattern
        /// </summary>
        /// <returns>Returns the list of matching files</returns>
        public List<string> GetMatchingFiles(string filePattern)
        {
            var matchingFiles = new List<string>();

            // Convert the relative path to absolute path
            if (!Path.IsPathRooted(filePattern))
            {
                filePattern = Path.Combine(this.fileHelper.GetCurrentDirectory(), filePattern);
            }

            // If there is no wildcard simply add the file to the list of matching files.
            if (filePattern.IndexOfAny(wildCardCharacters) == -1)
            {
                EqtTrace.Info($"FilePatternParser: The given file {filePattern} is a full path.");

                // Check if the file exists.
                if (!this.fileHelper.Exists(filePattern))
                {
                    throw new TestSourceException(
                        string.Format(CultureInfo.CurrentUICulture, CommandLineResources.TestSourceFileNotFound, filePattern));
                }

                matchingFiles.Add(filePattern);

                return matchingFiles;
            }

            // Split the given wildcard into search directory and pattern to be searched.
            var splitPattern = SplitFilePatternOnWildCard(filePattern);
            EqtTrace.Info($"FilePatternParser: Matching file pattern '{splitPattern.Item2}' within directory '{splitPattern.Item1}'");

            this.matcher.AddInclude(splitPattern.Item2);

            // Execute the given pattern in the search directory.
            var matches = this.matcher.Execute(new DirectoryInfoWrapper(new DirectoryInfo(splitPattern.Item1)));

            // Add all the files to the list of matching files.
            foreach (var match in matches.Files)
            {
                matchingFiles.Add(Path.Combine(splitPattern.Item1, match.Path));
            }

            return matchingFiles;
        }

        /// <summary>
        /// Splits full pattern into search directory and pattern.
        /// </summary>
        private Tuple<string, string> SplitFilePatternOnWildCard(string filePattern)
        {
            // Split the pattern based on first wildcard character found.
            var splitOnWildCardIndex = filePattern.IndexOfAny(wildCardCharacters);
            var directorySeparatorIndex = filePattern.Substring(0, splitOnWildCardIndex).LastIndexOf(Path.DirectorySeparatorChar);

            string searchDir = filePattern.Substring(0, directorySeparatorIndex);
            string pattern = filePattern.Substring(directorySeparatorIndex + 1);

            Tuple<string, string> splitPattern = new Tuple<string, string>(searchDir, pattern);
            return splitPattern;
        }
    }
}
