﻿// Copyright (c) Microsoft.  All Rights Reserved.  Licensed under the Apache License, Version 2.0.  See License.txt in the project root for license information.

using System.Threading.Tasks;
using Microsoft.CodeAnalysis.Test.Utilities;
using Xunit;

namespace Microsoft.CodeAnalysis.Editor.CSharp.UnitTests.Diagnostics.PreferFrameworkType
{
    public partial class PreferFrameworkTypeTests : AbstractCSharpDiagnosticProviderBasedUserDiagnosticTest
    {
        private readonly string _fixAllActionId = FeaturesResources.Use_framework_type;

        [Fact]
        [Trait(Traits.Feature, Traits.Features.CodeActionsUseFrameworkType)]
        [Trait(Traits.Feature, Traits.Features.CodeActionsFixAllOccurrences)]
        public async Task TestFixAllInDocument_DeclarationContext()
        {
            var input = @"
<Workspace>
    <Project Language=""C#"" AssemblyName=""Assembly1"" CommonReferences=""true"">
        <Document>
using System;

class Program
{
    static int F(int x, System.Int16 y)
    {
        {|FixAllInDocument:int|} i1 = 0;
        System.Int16 s1 = 0;
        int i2 = 0;
        return i1 + s1 + i2;
    }
}
        </Document>
        <Document>
using System;

class Program2
{
    static System.Int32 F(System.Int32 x, System.Int16 y)
    {
        System.Int32 i1 = 0;
        System.Int16 s1 = 0;
        System.Int32 i2 = 0;
        return i1 + s1 + i2;
    }
}
        </Document>
    </Project>
    <Project Language=""C#"" AssemblyName=""Assembly2"" CommonReferences=""true"">
        <Document>
using System;

class Program2
{
    static System.Int32 F(System.Int32 x, System.Int16 y)
    {
        System.Int32 i1 = 0;
        System.Int16 s1 = 0;
        System.Int32 i2 = 0;
        return i1 + s1 + i2;
    }
}
        </Document>
    </Project>
</Workspace>";

            var expected = @"
<Workspace>
    <Project Language=""C#"" AssemblyName=""Assembly1"" CommonReferences=""true"">
        <Document>
using System;

class Program
{
    static Int32 F(Int32 x, System.Int16 y)
    {
        Int32 i1 = 0;
        System.Int16 s1 = 0;
        Int32 i2 = 0;
        return i1 + s1 + i2;
    }
}
        </Document>
        <Document>
using System;

class Program2
{
    static System.Int32 F(System.Int32 x, System.Int16 y)
    {
        System.Int32 i1 = 0;
        System.Int16 s1 = 0;
        System.Int32 i2 = 0;
        return i1 + s1 + i2;
    }
}
        </Document>
    </Project>
    <Project Language=""C#"" AssemblyName=""Assembly2"" CommonReferences=""true"">
        <Document>
using System;

class Program2
{
    static System.Int32 F(System.Int32 x, System.Int16 y)
    {
        System.Int32 i1 = 0;
        System.Int16 s1 = 0;
        System.Int32 i2 = 0;
        return i1 + s1 + i2;
    }
}
        </Document>
    </Project>
</Workspace>";

            await TestInRegularAndScriptAsync(input, expected, options: FrameworkTypeEverywhere, fixAllActionEquivalenceKey: _fixAllActionId);
        }

        [Fact]
        [Trait(Traits.Feature, Traits.Features.CodeActionsUseFrameworkType)]
        [Trait(Traits.Feature, Traits.Features.CodeActionsFixAllOccurrences)]
        public async Task TestFixAllInProject_DeclarationContext()
        {
            var input = @"
<Workspace>
    <Project Language=""C#"" AssemblyName=""Assembly1"" CommonReferences=""true"">
        <Document>
using System;

class Program
{
    static int F(int x, System.Int16 y)
    {
        {|FixAllInProject:int|} i1 = 0;
        System.Int16 s1 = 0;
        int i2 = 0;
        return i1 + s1 + i2;
    }
}
        </Document>
        <Document>
using System;

class Program2
{
    static int F(int x, System.Int16 y)
    {
        int i1 = 0;
        System.Int16 s1 = 0;
        int i2 = 0;
        return i1 + s1 + i2;
    }
}
        </Document>
    </Project>
    <Project Language=""C#"" AssemblyName=""Assembly2"" CommonReferences=""true"">
        <Document>
using System;

class Program2
{
    static System.Int32 F(System.Int32 x, System.Int16 y)
    {
        System.Int32 i1 = 0;
        System.Int16 s1 = 0;
        System.Int32 i2 = 0;
        return i1 + s1 + i2;
    }
}
        </Document>
    </Project>
</Workspace>";

            var expected = @"
<Workspace>
    <Project Language=""C#"" AssemblyName=""Assembly1"" CommonReferences=""true"">
        <Document>
using System;

class Program
{
    static Int32 F(Int32 x, System.Int16 y)
    {
        Int32 i1 = 0;
        System.Int16 s1 = 0;
        Int32 i2 = 0;
        return i1 + s1 + i2;
    }
}
        </Document>
        <Document>
using System;

class Program2
{
    static Int32 F(Int32 x, System.Int16 y)
    {
        Int32 i1 = 0;
        System.Int16 s1 = 0;
        Int32 i2 = 0;
        return i1 + s1 + i2;
    }
}
        </Document>
    </Project>
    <Project Language=""C#"" AssemblyName=""Assembly2"" CommonReferences=""true"">
        <Document>
using System;

class Program2
{
    static System.Int32 F(System.Int32 x, System.Int16 y)
    {
        System.Int32 i1 = 0;
        System.Int16 s1 = 0;
        System.Int32 i2 = 0;
        return i1 + s1 + i2;
    }
}
        </Document>
    </Project>
</Workspace>";

            await TestInRegularAndScriptAsync(input, expected, options: FrameworkTypeEverywhere, fixAllActionEquivalenceKey: _fixAllActionId);
        }

        [Fact]
        [Trait(Traits.Feature, Traits.Features.CodeActionsUseFrameworkType)]
        [Trait(Traits.Feature, Traits.Features.CodeActionsFixAllOccurrences)]
        public async Task TestFixAllInSolution_DeclarationContext()
        {
            var input = @"
<Workspace>
    <Project Language=""C#"" AssemblyName=""Assembly1"" CommonReferences=""true"">
        <Document>
using System;

class Program
{
    static int F(int x, System.Int16 y)
    {
        {|FixAllInSolution:int|} i1 = 0;
        System.Int16 s1 = 0;
        int i2 = 0;
        return i1 + s1 + i2;
    }
}
        </Document>
        <Document>
using System;

class Program2
{
    static int F(int x, System.Int16 y)
    {
        int i1 = 0;
        System.Int16 s1 = 0;
        int i2 = 0;
        return i1 + s1 + i2;
    }
}
        </Document>
    </Project>
    <Project Language=""C#"" AssemblyName=""Assembly2"" CommonReferences=""true"">
        <Document>
using System;

class Program2
{
    static int F(int x, System.Int16 y)
    {
        int i1 = 0;
        System.Int16 s1 = 0;
        int i2 = 0;
        return i1 + s1 + i2;
    }
}
        </Document>
    </Project>
</Workspace>";

            var expected = @"
<Workspace>
    <Project Language=""C#"" AssemblyName=""Assembly1"" CommonReferences=""true"">
        <Document>
using System;

class Program
{
    static Int32 F(Int32 x, System.Int16 y)
    {
        Int32 i1 = 0;
        System.Int16 s1 = 0;
        Int32 i2 = 0;
        return i1 + s1 + i2;
    }
}
        </Document>
        <Document>
using System;

class Program2
{
    static Int32 F(Int32 x, System.Int16 y)
    {
        Int32 i1 = 0;
        System.Int16 s1 = 0;
        Int32 i2 = 0;
        return i1 + s1 + i2;
    }
}
        </Document>
    </Project>
    <Project Language=""C#"" AssemblyName=""Assembly2"" CommonReferences=""true"">
        <Document>
using System;

class Program2
{
    static Int32 F(Int32 x, System.Int16 y)
    {
        Int32 i1 = 0;
        System.Int16 s1 = 0;
        Int32 i2 = 0;
        return i1 + s1 + i2;
    }
}
        </Document>
    </Project>
</Workspace>";

            await TestInRegularAndScriptAsync(input, expected, options: FrameworkTypeEverywhere, fixAllActionEquivalenceKey: _fixAllActionId);
        }

        [Fact]
        [Trait(Traits.Feature, Traits.Features.CodeActionsUseFrameworkType)]
        [Trait(Traits.Feature, Traits.Features.CodeActionsFixAllOccurrences)]
        public async Task TestFixAllInSolution_MemberAccessContext()
        {
            var input = @"
<Workspace>
    <Project Language=""C#"" AssemblyName=""Assembly1"" CommonReferences=""true"">
        <Document>
using System;
class ProgramA
{
    private int x = 0;
    private int y = 0;
    private int z = 0;

    private System.Int32 F(System.Int32 p1, System.Int16 p2)
    {
        System.Int32 i1 = this.x;
        System.Int16 s1 = this.y;
        System.Int32 i2 = this.z;
        {|FixAllInSolution:int|}.Parse(i1 + s1 + i2);
        int.Parse(i1 + s1 + i2);
        System.Exception ex = null;
        return i1 + s1 + i2;
    }
}
        </Document>
        <Document>
using System;
class ProgramA2
{
    private int x = 0;
    private int y = 0;
    private int z = 0;

    private System.Int32 F(System.Int32 p1, System.Int16 p2)
    {
        System.Int32 i1 = this.x;
        System.Int16 s1 = this.y;
        System.Int32 i2 = this.z;
        int.Parse(i1 + s1 + i2);
        int.Parse(i1 + s1 + i2);
        System.Exception ex = null;
        return i1 + s1 + i2;
    }
}
        </Document>
    </Project>
    <Project Language=""C#"" AssemblyName=""Assembly2"" CommonReferences=""true"">
        <Document>
using System;
class ProgramA3
{
    private int x = 0;
    private int y = 0;
    private int z = 0;

    private System.Int32 F(System.Int32 p1, System.Int16 p2)
    {
        System.Int32 i1 = this.x;
        System.Int16 s1 = this.y;
        System.Int32 i2 = this.z;
        int.Parse(i1 + s1 + i2);
        int.Parse(i1 + s1 + i2);
        System.Exception ex = null;
        return i1 + s1 + i2;
    }
}
        </Document>
    </Project>
</Workspace>";

            var expected = @"
<Workspace>
    <Project Language=""C#"" AssemblyName=""Assembly1"" CommonReferences=""true"">
        <Document>
using System;
class ProgramA
{
    private int x = 0;
    private int y = 0;
    private int z = 0;

    private System.Int32 F(System.Int32 p1, System.Int16 p2)
    {
        System.Int32 i1 = this.x;
        System.Int16 s1 = this.y;
        System.Int32 i2 = this.z;
        Int32.Parse(i1 + s1 + i2);
        Int32.Parse(i1 + s1 + i2);
        System.Exception ex = null;
        return i1 + s1 + i2;
    }
}
        </Document>
        <Document>
using System;
class ProgramA2
{
    private int x = 0;
    private int y = 0;
    private int z = 0;

    private System.Int32 F(System.Int32 p1, System.Int16 p2)
    {
        System.Int32 i1 = this.x;
        System.Int16 s1 = this.y;
        System.Int32 i2 = this.z;
        Int32.Parse(i1 + s1 + i2);
        Int32.Parse(i1 + s1 + i2);
        System.Exception ex = null;
        return i1 + s1 + i2;
    }
}
        </Document>
    </Project>
    <Project Language=""C#"" AssemblyName=""Assembly2"" CommonReferences=""true"">
        <Document>
using System;
class ProgramA3
{
    private int x = 0;
    private int y = 0;
    private int z = 0;

    private System.Int32 F(System.Int32 p1, System.Int16 p2)
    {
        System.Int32 i1 = this.x;
        System.Int16 s1 = this.y;
        System.Int32 i2 = this.z;
        Int32.Parse(i1 + s1 + i2);
        Int32.Parse(i1 + s1 + i2);
        System.Exception ex = null;
        return i1 + s1 + i2;
    }
}
        </Document>
    </Project>
</Workspace>";
            await TestInRegularAndScriptAsync(input, expected, options: FrameworkTypeEverywhere, fixAllActionEquivalenceKey: _fixAllActionId);
        }
    }
}
