using System.Text;
using System.Diagnostics;

namespace System.Xml {

    internal abstract partial class Base64Encoder {

        byte[]  leftOverBytes;
        int     leftOverBytesCount;
        char[]  charsLine;

        internal const int Base64LineSize = 76;  
        internal const int LineSizeInBytes = Base64LineSize/4*3;    

        internal Base64Encoder() {
            charsLine = new char[Base64LineSize];
        }

        internal abstract void WriteChars( char[] chars, int index, int count );

        internal void Encode( byte[] buffer, int index, int count ) {
            if ( buffer == null ) {
                throw new ArgumentNullException( "buffer" );
            }
            if ( index < 0 ) {
                throw new ArgumentOutOfRangeException( "index" );
            }
            if ( count < 0 ) {
                throw new ArgumentOutOfRangeException( "count" );
            }
            if ( count > buffer.Length - index ) {
                throw new ArgumentOutOfRangeException( "count" );
            }

            // encode left-over buffer
            if( leftOverBytesCount > 0 ) {
                int i = leftOverBytesCount;
                while ( i < 3 && count > 0 ) {
                    leftOverBytes[i++] = buffer[index++];
                    count--;
                }

                // the total number of buffer we have is less than 3 -> return
                if ( count == 0 && i < 3 ) {
                    leftOverBytesCount = i;
                    return;
                }

                // encode the left-over buffer and write out
                int leftOverChars = Convert.ToBase64CharArray( leftOverBytes, 0, 3, charsLine, 0 );
                WriteChars( charsLine, 0, leftOverChars );
            }

            // store new left-over buffer
            leftOverBytesCount = count % 3;
            if ( leftOverBytesCount > 0 )  {
                count -= leftOverBytesCount;
                if ( leftOverBytes == null ) {
                    leftOverBytes = new byte[3];
                }
                for( int i = 0; i < leftOverBytesCount; i++ ) {
                    leftOverBytes[i] = buffer[ index + count + i ];
                }
            }

            // encode buffer in 76 character long chunks
            int endIndex = index + count;
            int chunkSize = LineSizeInBytes;
            while( index < endIndex ) {
                if ( index + chunkSize > endIndex ) {
                    chunkSize = endIndex - index;
                }
                int charCount = Convert.ToBase64CharArray( buffer, index, chunkSize, charsLine, 0 );
                WriteChars( charsLine, 0, charCount );
    
                index += chunkSize;
            }
        }

        internal void Flush() {
            if ( leftOverBytesCount > 0 ) {
                int leftOverChars = Convert.ToBase64CharArray( leftOverBytes, 0, leftOverBytesCount, charsLine, 0 );
                WriteChars( charsLine, 0, leftOverChars );
                leftOverBytesCount = 0;
            }
        }
    } 

    //internal partial class XmlRawWriterBase64Encoder : Base64Encoder {

    //    XmlRawWriter rawWriter;
        
    //    internal XmlRawWriterBase64Encoder( XmlRawWriter rawWriter ) {
    //        this.rawWriter = rawWriter;
    //    }

    //    internal override void WriteChars( char[] chars, int index, int count ) {
    //        rawWriter.WriteRaw( chars, index, count );
    //    }
    //}

#if !SILVERLIGHT || FEATURE_NETCORE
    internal partial class XmlTextWriterBase64Encoder : Base64Encoder {

        XmlTextEncoder xmlTextEncoder;
        
        internal XmlTextWriterBase64Encoder( XmlTextEncoder xmlTextEncoder ) {
            this.xmlTextEncoder = xmlTextEncoder;
        }

        internal override void WriteChars( char[] chars, int index, int count ) {
            xmlTextEncoder.WriteRaw( chars, index, count );
        }
    }
#endif

} 
