# -*- coding: utf-8 -*-

#
# Copyright (c) 2013 Red Hat, Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#           http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

from collections import deque
from redhat_support_tool.helpers.confighelper import _
from redhat_support_tool.helpers.constants import Constants
from redhat_support_tool.helpers.launchhelper import LaunchHelper
from redhat_support_tool.plugins import InteractivePlugin, DisplayOption, \
    ObjectDisplayOption, HiddenCommand
from redhat_support_tool.plugins.diagnose import Diagnose
from redhat_support_tool.plugins.open_case import OpenCase
import logging
import os
import pydoc
# pylint: disable=W0402
import string
import tempfile
import redhat_support_tool.symptoms as symptoms
import redhat_support_tool.helpers.apihelper as apihelper

__author__ = 'Dan Varga <dvarga@redhat.com>'
logger = logging.getLogger("redhat_support_tool.plugins.symptom")


class Symptom(InteractivePlugin, HiddenCommand):
    plugin_name = 'symptom'
    ALL = _("Display symptom details")
    _sections = None
    symptomID = None
    symptom = None
    _submenu_opts = None

    def get_intro_text(self):
        return _('\nType the number of the section to view or \'e\' '
                 'to return to the previous menu.')

    def get_prompt_text(self):
        return _('Option: ')

    def get_sub_menu_options(self):
        return self._submenu_opts

    def insert_obj(self, analyzedict):
        '''
        Allow insertion of a package object by launchhelper (when selecting
        from the list generated by list_kerneldebugs.py)
        '''
        self.symptom = analyzedict['symptom']

    # No arguments required here, data passed on the side
    # Let's make sure we have symptoms to do some work on though
    def validate_args(self):
        # Check for required arguments.
        if symptoms.AnalyzerPlugin.symptoms:
            return True
        else:
            raise Exception("No symptoms found, run analyze first")

    def postinit(self):
        self._submenu_opts = deque()
        self._sections = {}
        self._display_symptom_options()

    def non_interactive_action(self):
        doc = ''
        for opt in self._submenu_opts:
            if opt.display_text != self.ALL:
                doc += self._sections[opt]
        try:
            print(doc)
        except Exception as e:
            # There are some truly bizarre errors when you pipe
            # the output from python's 'print' function with sys encoding
            # set to ascii. These errors seem to manifes when you pipe
            # to something like 'more' or 'less'.  You'll get encoding errors.
            # Curiously, you don't see them with 'grep' or even simply piping
            # to terminal.  WTF :(
            logger.log(logging.WARNING, e)
            import sys
            print(doc)

    def interactive_action(self, display_option=None):
        if display_option.display_text == self.ALL:
            doc = ''
            for opt in self._submenu_opts:
                if opt.display_text != self.ALL:
                    doc += self._sections[opt]
            pydoc.pipepager(doc, cmd='less -R')
        else:
            doc = self._sections[display_option]
            pydoc.pipepager(doc, cmd='less -R')

    def _send_to_shadowman(self, display_option=None):
        lh = LaunchHelper(Diagnose)
        lh.run('', display_option)

    def _opencase(self, display_option=None):
        lh = LaunchHelper(OpenCase)
        try:
            # Remove nonprintable characters from the
            # crash output.  Argh!!!!
            filtered_string = [x for x in display_option.stored_obj if x in string.printable]
            msg = None

            # If the filtered_string is too long (comments can only be
            # ~30k when formatting is applied, plus 20k is a large
            # chunk of text) attach it instead.
            if len(filtered_string) > 20000:
                try:
                    # Because we can't rename on the fly attachments,
                    # this filename is going to look 'odd'.
                    fd, temppath = tempfile.mkstemp(
                                                prefix="symptom-",
                                                suffix="-rhst")
                    attachment = os.fdopen(fd, "w")
                    attachment.write(filtered_string)
                    attachment.close()
                    lh.run('--attachment=%s' % (temppath))
                    os.remove(temppath)
                except:
                    print(_('Unable to upload output to Red Hat'
                            ' Customer Portal, reverting to displaying'
                            ' output to console.'))
            else:
                msg = '%s\nThe following comment was added by ' \
                      'Red Hat Support Tool\nVersion: %s\n' \
                      '%s\n\n%s' % \
                      (str(self.ruler * Constants.MAX_RULE),
                       apihelper.USER_AGENT,
                       str(self.ruler * Constants.MAX_RULE),
                       filtered_string)
                lh.run('-d \'%s\'' % msg)

        except Exception as e:
            msg = _('ERROR: %s') % e
            print(msg)
            logger.log(logging.ERROR, msg)

    def _display_symptom_options(self):
        try:
            print(_('\nSymptom summary'))
            print("\n\t" + self.symptom.before_line)
            print("\t" + self.symptom.source_line)
            # Display entire symptom:
            doc = ''
            doc += '\n%s%s%s\n' % (Constants.BOLD,
                                   "Entire symptom",
                                   Constants.END)
            doc += '%s%s%s\n' % (Constants.BOLD,
                                 str(self.ruler * Constants.MAX_RULE),
                                 Constants.END)
            doc += '%s\n' % self.symptom.token_string
            doc = doc.replace('\r\n', os.linesep)  # Set linesep to platform.
            disp_opt = DisplayOption("Show entire symptom",
                                     'interactive_action')
            self._submenu_opts.append(disp_opt)
            self._sections[disp_opt] = doc

            # Send to Shadowman
            disp_opt = ObjectDisplayOption("Diagnose symptom",
                                     '_send_to_shadowman',
                                     (self.symptom.before_line +
                                      self.symptom.token_string))
            doc = ''
            self._submenu_opts.append(disp_opt)
            self._sections[disp_opt] = doc

            # Open a support case
            disp_opt = ObjectDisplayOption(
                               _("Open a support case with symptom signature"),
                               '_opencase',
                               self.symptom.before_line +
                               self.symptom.token_string)
            doc = ''
            self._submenu_opts.append(disp_opt)
            self._sections[disp_opt] = doc
        except:
            msg = _('ERROR: problem parsing the solution.')
            print(msg)
            logger.log(logging.WARNING, msg)
            return False
        return True
