#
# Copyright (c) 2010 Red Hat, Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#           http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

'''
Module that brokers the RESTful methods.
'''
from redhat_support_lib.infrastructure.common import Base, ignore_none_values
from redhat_support_lib.infrastructure.s3_uploader import AWSUploader
from redhat_support_lib.infrastructure.errors import ConnectionError, \
    RequestError, SftpError
from redhat_support_lib.infrastructure.connectionspool import ConnectionsPool
from redhat_support_lib.utils.filterhelper import FilterHelper
from redhat_support_lib.utils.parsehelper import ParseHelper, ReportParseHelper
from redhat_support_lib.utils.searchhelper import SearchHelper
from redhat_support_lib.utils.urlhelper import UrlHelper
from redhat_support_lib.xml import params, report
from urllib import quote
from urlparse import urlparse
from xml.sax.saxutils import escape
import redhat_support_lib.utils.ftphelper as FtpHelper
import redhat_support_lib.utils.confighelper as confighelper
import json
import logging
import mimetypes
import os.path
import socket
import sys
from email.Header import decode_header
from redhat_support_lib.json import paramsjson

logger = logging.getLogger("redhat_support_lib.infrastructure.brokers")

__author__ = 'Keith Robertson <kroberts@redhat.com>'


class KCSSearch(paramsjson.KCSSearch, Base):
    def __init__(self, KCSSearch):
        self.superclass = KCSSearch

    def __new__(cls, KCSSearch):
        if KCSSearch is None:
            return None
        obj = object.__new__(cls)
        obj.__init__(KCSSearch)
        return obj

    @classmethod
    def fromProps(cls, **kwargs):
        kcssearch = paramsjson.KCSSearch(**kwargs)
        return cls(kcssearch)


class solution(paramsjson.Solution, Base):
    def __init__(self, solution):
        self.superclass = solution

    def __new__(cls, solution):
        if solution is None:
            return None
        obj = object.__new__(cls)
        obj.__init__(solution)
        return obj


    @classmethod
    def fromProps(cls,
                  createdBy=None,
                  title=None,
                  summary=None,
                  kcsState=None,
                  resolution=None,
                  **kwargs):

        issue = kwargs.get('issue', None)
        if issue is not None:
            issue = paramsjson.KCSDescription(issue, None)

        resolution = kwargs.get('resolution', None)
        if resolution is not None:
            resolution = paramsjson.KCSDescription(resolution, None)

        environment = kwargs.get('environment', None)
        if environment is not None:
            environment = paramsjson.KCSDescription(environment, None)

        rootCause = kwargs.get('rootCause', None)
        if rootCause is not None:
            rootCause = paramsjson.KCSDescription(rootCause, None)

        diagnosticSteps = kwargs.get('diagnosticSteps', None)
        if diagnosticSteps is not None:
            internalDiagnosticSteps = paramsjson.KCSDescription(diagnosticSteps, None)

        psol = paramsjson.Solution(authorSSOName=createdBy, title=title,
                                   kcsState=kcsState,
                                   resolution=resolution, **kwargs)

        return cls(psol)

    def set_issue(self, issue):
        issue = paramsjson.KCSDescription(issue, None)
        super(solution, self).set_issue(issue)

    def set_resolution(self, resolution):
        resolution = paramsjson.KCSDescription(resolution, None)
        super(solution, self).set_resolution(resolution)

    def set_environment(self, environment):
        environment = paramsjson.KCSDescription(environment, None)
        super(solution, self).set_environment(environment)

    def set_rootCause(self, rootCause):
        rootCause = paramsjson.KCSDescription(rootCause, None)
        super(solution, self).set_rootCause(rootCause)

    def set_diagnosticSteps(self, diagnosticSteps):
        diagnosticSteps = paramsjson.KCSDescription(diagnosticSteps, None)
        super(solution, self).set_diagnosticSteps(diagnosticSteps)


    def update(self):
        '''
        Update this solution. This solution *must* have an ID.

        .. IMPORTANT::
           The solution must already exist on the Red Hat Customer Portal, if
           you are adding a new solution you should use
           :func:`solutions.add` instead

        raises   -- An exception if there was a connection related issue.
        '''
        pass


class solutions(Base):
    def __init__(self):
        """Constructor."""
        # Assign self to superclass for the broker classes that are aggregates of the specific type.
        # Without this assignment you'll get a nice and confusing recursion stacktrace if
        # a wayward programmer attempts to call a non-existent method on this class.
        self.superclass = self

    def add(self, sol):
        '''

        Add a new solution

        :param sol: Solution to be added to Customer Portal
        :type sol: solution
        :rtype: solution
        :returns: The solution ID and view_uri will be set if successful.
        :raises:
            Exception if there was a connection related issue or an
            issue parsing headers
        '''
        pass

    def get(self, solutionID=None):
        '''
        Queries the API for the given solution ID.

        :param solutionID: Solution ID to be queried from the API
        :type solutionID: Integer
        :rtype: solution
        :raises:
            Exception if there was a connection related issue or an
            issue parsing headers
        '''
        url = '/support/v1/solutions'
        doc, headers = self._getProxy().get(url=UrlHelper.append(url, solutionID))
        logger.debug("REST Response:%s" % doc)
        return solution(paramsjson.Solutionfromdict(doc))

    def list(self, keywords=None, **kwargs):
        '''
        Queries the solutions RESTful interface with a given set of keywords.

        :param keywords: Search string
        :type keywords: string
        :param searchopts: search options/query filters passed to the API
        :type searchopts: dict
        :param kwargs:
            Additional options passed to FilterHelper

            Example:

            .. code-block:: python

                solutions.list('RHEV', authorSSOName="anonymous")

        :type kwargs: dict
        :returns: A list of solution objects
        :rtype: list
        :raises:
            Exception if there was a connection related issue or an
            issue parsing headers
        '''
        url = '/support/search/v2/kcs'
        body = {}
        searchopts = kwargs.pop('searchopts', {})
        if searchopts:
            body['start'] = searchopts['offset']
            body['rows'] = (searchopts['limit'])
        body['expression'] = "fq={}".format(quote('documentKind:"Solution"'))
        body['q'] = keywords
        doc, headers = self._getProxy().action(url=url, body=json.dumps(body))
        logger.debug("REST Response:\n%s" % doc)

        try:
            return ParseHelper.toCollection(KCSSearch,
                                            FilterHelper.filter(paramsjson.KCSSearchesfromdict(doc[u'response'][u'docs']),
                                                                kwargs))
        except Exception as e:
            logger.error(e)
            return []


class article(paramsjson.Article, Base):
    def __init__(self, article):
        self.superclass = article

    def __new__(cls, article):
        if article is None: return None
        obj = object.__new__(cls)
        obj.__init__(article)
        return obj


    @classmethod
    def fromProps(cls, createdBy=None, title=None, summary=None, kcsState=None, body=None, **kwargs):

        body = paramsjson.KCSDescription(body, None)

        psol = paramsjson.Article(authorSSOName=createdBy, title=title,
                                  body=body, **kwargs)
        return cls(psol)

# # No such animal as delete.
#    def delete(self):
#        '''
#
#        '''
#        url = '/rs/articles'
#        sol = self._getProxy().delete(url=UrlHelper.append(url, self.get_id()))
#        print "----%s---" % sol


    def update(self):
        '''
        Update this article. This article *must* have an ID.

        raises   -- An exception if there was a connection related issue.
        '''
        pass


class articles(Base):
    def __init__(self):
        """Constructor."""
        # Assign self to superclass for the broker classes that are aggregates of the specific type.
        # Without this assignment you'll get a nice and confusing recursion stacktrace if
        # a wayward programmer attempts to call a non-existent method on this class.
        self.superclass = self

    def add(self, art):
        '''
        Add a new article

        :param art: Article to be added to Customer Portal
        :type art: article
        :rtype: article
        :returns: The article ID and view_uri will be set if successful.
        :raises:
            Exception if there was a connection related issue or an
            issue parsing headers
        '''
        pass

    def get(self, articleID=None):
        '''
        Queries the API for the given article ID.

        :param articleID: Article ID to be queried from the API
        :type articleID: Integer
        :rtype: article
        :raises:
            Exception if there was a connection related issue or an
            issue parsing headers
        '''
        url = '/support/v1/articles'
        doc, headers = self._getProxy().get(url=UrlHelper.append(url, articleID))
        logger.debug("REST Response:%s" % doc)
        return article(paramsjson.Articlefromdict(doc))

    def list(self, keywords=None, **kwargs):
        '''
        Queries the articles RESTful interface with a given set of keywords.

        :param keywords: Search string
        :type keywords: string
        :param searchopts: search options/query filters passed to the API
        :type searchopts: dict
        :param kwargs:
            Additional options passed to FilterHelper

            Example:

            .. code-block:: python

                articles.list('RHEV', authorSSOName="anonymous")

        :returns: A list of solution objects
        :rtype: list
        :raises:
            Exception if there was a connection related issue or an
            issue parsing headers
        '''
        pass


class comment(paramsjson.Comment, Base):
    def __init__(self, comment):
        self.superclass = comment

    def __new__(cls, comment):
        if comment is None: return None
        obj = object.__new__(cls)
        obj.__init__(comment)
        return obj

    @classmethod
    def fromProps(cls, caseNumber=None, text=None,
                  public=True, **kwargs):
        comment = paramsjson.Comment(caseNumber=caseNumber,
                                     commentBody=text,
                                     isPublic=public,
                                     **kwargs)
        return cls(comment)

    def update(self):
        '''
        Update this comment. This comment must have both an comment ID and a case ID.
        raises   -- An exception if there was a connection related issue.
        '''
        pass


class comments(Base):
    def __init__(self):
        """Constructor."""
        # Assign self to superclass for the broker classes that are aggregates of the specific type.
        # Without this assignment you'll get a nice and confusing recursion stacktrace if
        # a wayward programmer attempts to call a non-existent method on this class.
        self.superclass = self

    def add(self, cmnt):
        '''
        Add a new comment

        :param cmnt: The comment to be added
        :type cmnt: comment
        :returns: Comment object with ID and view_uri populated
        :rtype: comment
        :raises:
            Exception if there was a connection related issue or an
            issue parsing headers
        '''

        url = '/support/v1/cases/{caseNumber}/comments'
        url = UrlHelper.replace(url, {'{caseNumber}': cmnt.get_caseNumber()})
        doc, headers = self._getProxy().add(url=url,
                                            body=json.dumps(ignore_none_values(cmnt.superclass.to_dict())))
        cmnt.set_id(doc.get(u'id'))
        cmnt.set_caseNumber(doc.get(u'caseNumber'))
        return cmnt

    def get(self, caseNumber=None, commentID=None):
        '''
        Queries the API for the given solution ID.

        :param caseNumber: Case number to retrieve the comment from.
        :type caseNumber: string
        :param commentID: ID string of the comment to retrieve
        :type commentID: string
        :returns: Comment matching specified case number and comment ID
        :rtype: comment
        :raises:
            Exception if there was a connection related issue or an
            issue parsing headers
        '''
        pass

    def list(self, caseNumber=None, startDate=None, endDate=None, **kwargs):
        '''
        Gets all of the comments for a given case number.  You can then
        search/filter the returned comments using any of the properties
        of a 'comment'

        :param caseNumber: Case number to retrieve the comment from.
        :type caseNumber: string
        :param startDate: Date to list comments from
        :type startDate:
            ISO 8601 formatted string, either YYYY-MM-DDThh:mm:ss or YYYY-MM-DD
        :param endDate: Date to list comments until
        :type endDate:
            ISO 8601 formatted string, either YYYY-MM-DDThh:mm:ss or YYYY-MM-DD
        :param kwargs:
            Additional options passed to FilterHelper to filter results based
            on additional criteria.

            Example:

            .. code-block:: python

                comments.list('00595293', createdBy="AnonymousUser")

        :type kwargs: dict
        :returns: A list of comment objects
        :rtype: list
        :raises:
            Exception if there was a connection related issue or an
            issue parsing headers
        '''
        pass


class attachment(paramsjson.CaseAttachment, Base):
    def __init__(self, attachment):
        self.superclass = attachment

    def __new__(cls, attachment):
        if attachment is None: return None
        obj = object.__new__(cls)
        obj.__init__(attachment)
        return obj

    @classmethod
    def fromProps(cls, caseNumber=None, public=True,
                  fileName=None, description=None, **kwargs):

        attachmnt = paramsjson.CaseAttachment(caseNumber=caseNumber, isPrivate=not public,
                                              fileName=fileName, description=description, **kwargs)

        return cls(attachmnt)


class attachments(Base):
    def __init__(self):
        """Constructor."""
        # Assign self to superclass for the broker classes that are aggregates of the specific type.
        # Without this assignment you'll get a nice and confusing recursion stacktrace if
        # a wayward programmer attempts to call a non-existent method on this class.
        self.superclass = self

    def aws_add(self, caseNumber=None, public=True, fileName=None, fileChunk=None, description=None, useFtp=False):
        ''''
        Add new attachment to the S3 using the REST APIs.
        '''
        file_uploaded = None
        if useFtp:
            user_api = users()
            userobj = user_api.get(confighelper.get_config_helper().username)

            file_uploaded = FtpHelper.ftp_attachment(fileName, caseNumber, fileChunk, is_internal_user=userobj.isInternal)

        else:
            s3_uploader = AWSUploader(case_number=caseNumber, isPrivate=not public, description=description,
                                      filepath=fileName, fileChunk=fileChunk)
            file_uploaded = s3_uploader.upload_file()

        return file_uploaded

    def add(self, caseNumber=None, public=True, fileName=None,
            fileChunk=None, description=None, useFtp=False):
        '''
        Add a new attachment

        :param caseNumber: The case number to add the attachment to
        :type caseNumber: string
        :param public: If the attachment is public, or Red Hat only
        :type public: boolean
        :param fileName: Path to the file to be uploaded
        :type fileName: string
        :param description:
            Description of the attachment uploaded, for example "sosreport from
            host database01"

        :type description: string
        :returns: The attachment object with the attachment UUID populated
        :rtype: attachment
        :raises:
            Exception if there was a connection related issue or an
            issue parsing headers
        '''
        url = '/support/v1/cases/{caseNumber}/attachments'
        amnt = None
        config = confighelper.get_config_helper()

        exceeds_max_size = (not fileChunk and os.path.getsize(fileName) > config.attachment_max_size)
        if useFtp or exceeds_max_size:
            if public is False:
                raise SftpError("Red Hat Secure FTP does not support uploading private attachments.")
            user_api = users()
            userobj = user_api.get(confighelper.get_config_helper().username)
            amnt = FtpHelper.ftp_attachment(fileName, caseNumber, fileChunk, is_internal_user=userobj.isInternal)
            if exceeds_max_size and not useFtp:
                # if useFtp is True, no need to display this comment in the case
                filebaseName = os.path.basename(fileName)
                cmntText = ('[RHST] The file %s exceeds the byte limit to attach a file '
                            'to a case; therefore the file was uploaded to Red Hat Secure FTP as %s_%s'
                            % (filebaseName, caseNumber, filebaseName))
                cmnt = InstanceMaker.makeComment(caseNumber=caseNumber,
                                                 text=cmntText)
                casecomment = comments()
                casecomment.add(cmnt)
        else:
            url = UrlHelper.replace(url, {'{caseNumber}': caseNumber})

            user_agent = config.userAgent

            doc = self._getProxy().upload(SearchHelper.appendQuery(url, {'private':not public}),
                                          fileName,
                                          fileChunk,
                                          description, user_agent)[0]
            amnt = attachment(doc)
            logger.debug("REST Response:%s" % amnt.to_dict())
        return amnt

    def delete(self, caseNumber=None, attachmentUUID=None):
        '''
        Removes the attachment from the case.

        :param caseNumber: The case number to delete the attachment from
        :type caseNumber: string
        :param attachmentUUID: UUID of the attachment to be deleted
        :type attachmentUUID: string (UUID)
        :returns: True if successful
        :rtype: boolean
        :raises:
            Exception if there was a connection related issue, an issue parsing
            headers, or the attachment deletion failed.
        '''
        pass

    def get(self, caseNumber=None, attachmentUUID=None, fileName=None, attachmentLength=None, destDir=None, link=None):
        '''
        Queries the API for the given attachment.

        :param caseNumber: The case number to retrieve the attachment from
        :type caseNumber: string
        :param attachmentUUID: UUID of the attachment to be retrieved
        :type attachmentUUID: string (UUID)
        :param fileName: Path to the file to be retrieved
        :type fileName: string
        :param attachmentLength: Length of the attachment to be retrieved
        :type attachmentLength: integer
        :param destDir: The directory which the attachment should be saved in
        :type destDir: string
        :returns: The full path to the downloaded file
        :rtype: string
        :raises:
            Exception if there was a connection related issue or an
            issue parsing headers
        '''
        url = '/support/v1/cases/{caseNumber}/attachments/{attachmentUUID}'

        # Pre-checks for proper input
        if not caseNumber or not attachmentUUID:
            raise Exception('caseNumber(%s) attachmentUUID(%s) cannot be empty.' %
                            (caseNumber, attachmentUUID))
        # Join the path
        if destDir and not os.path.isdir(destDir):
            raise Exception("destDir(%s) is not a valid directory" % destDir)

        # BZ967496 - ensure fileName doesn't contain directory separators
        if fileName and os.path.dirname(fileName):
            raise Exception("fileName(%s) contains directory info (%s)."
                            "  Use the destDir parameter for specifying directory info." %
                            (fileName, os.path.dirname(fileName)))

        url = UrlHelper.replace(url,
                                {'{caseNumber}': caseNumber,
                                 '{attachmentUUID}': attachmentUUID})

        httpconnection = None
        fh = None
        try:
            try:
                if not link:
                    raise Exception("Failed to get the download link of the attachment")
                logger.debug('Downloading attachment...')
                config = confighelper.get_config_helper()
                conn = ConnectionsPool(url=link,
                                       key_file=config.key_file,
                                       cert_file=config.cert_file,
                                       timeout=config.timeout,
                                       username=config.username,
                                       password=config.password,
                                       proxy_url=config.proxy_url,
                                       proxy_user=config.proxy_user,
                                       proxy_pass=config.proxy_pass,
                                       debug=config.http_debug,
                                       ssl_ca=config.ssl_ca,
                                       noverify=config.no_verify_ssl)
                httpconnection = conn.getConnection()
                response = httpconnection.doRequest(method='GET', url=link, headers=config.userAgent)
                if response.status < 400:
                    if not fileName:
                        contDispArr = response.msg.dict['content-disposition'].split("\"")
                        fileName = decode_header(contDispArr[1])[0][0]
                    # Join the path
                    if destDir:
                        fileName = os.path.join(destDir, fileName)
                    fh = open(fileName, 'wb')
                    buf = response.read(8192)
                    downloadedBytes = len(buf)
                    while len(buf) > 0:
                        if attachmentLength:
                            percent = str(int(downloadedBytes * 100 / attachmentLength))
                            sys.stdout.write("%s%3s%%" % ("\b\b\b\b", percent))
                            sys.stdout.flush()
                        fh.write(buf)
                        buf = response.read(8192)
                        downloadedBytes += len(buf)
                    if attachmentLength:
                        sys.stdout.write("\n")
                        sys.stdout.flush()
                    logger.debug('Successfully downloaded %s.' % fileName)
                else:
                    logger.debug("unable to download %s as %s. Reason: %s" %
                                  (link, fileName, response.reason))
                    raise RequestError(response.status, response.reason, None)
            except socket.error, se:
                logger.debug('Socket error: msg(%s)' % (se))
                raise ConnectionError, str(se)
            except IOError, ioe:
                logger.debug('I/O error: errno(%s) strerror(%s)' % (ioe.errno, ioe.strerror))
                raise
            except Exception, e:
                logger.debug('Unexpected exception: msg(%s)' % (str(e)))
                raise
        finally:
            if httpconnection:
                try:
                    httpconnection.close()
                except Exception:
                    pass
            if fh:
                try:
                    fh.close()
                except Exception:
                    pass

        return fileName

    def list(self, caseNumber=None, startDate=None, endDate=None, **kwargs):
        '''
        Gets all of the attachments for a given case number.  You can then
        search/filter the returned comments using any of the properties
        of a 'attachment'

        :param caseNumber: Case number to list the attachments from
        :type caseNumber: string
        :param startDate: Date to list attachments from
        :type startDate:
            ISO 8601 formatted string, either YYYY-MM-DDThh:mm:ss or YYYY-MM-DD
        :param endDate: Date to list attachments until
        :type endDate:
            ISO 8601 formatted string, either YYYY-MM-DDThh:mm:ss or YYYY-MM-DD
        :param kwargs:
            Additional options passed to FilterHelper to filter results based
            on additional criteria.

            Example:

            .. code-block:: python

                attachments.list('00595293', createdBy="AnonymousUser")

        :type kwargs: dict
        :returns: A list of attachment objects
        :rtype: list
        :raises:
            Exception if there was a connection related issue or an
            issue parsing headers
        '''
        url = '/support/v1/cases/{caseNumber}/attachments'

        url = UrlHelper.replace(url,
                                {'{caseNumber}': caseNumber})

        doc, headers = self._getProxy().get(SearchHelper.appendQuery(url,
                                                                     {'startDate': startDate,
                                                                      'endDate': endDate}))
        logger.debug("REST Response:\n%s" % doc)
        doc = [] if not isinstance(doc, list) else doc
        return ParseHelper.toCollection(attachment,
                                        FilterHelper.filter(paramsjson.CaseAttachmentsfromdict(doc), kwargs))


class entitlement(paramsjson.Entitlement, Base):
    def __init__(self, entitlement):
        self.superclass = entitlement

    def __new__(cls, entitlement):
        if entitlement is None: return None
        obj = object.__new__(cls)
        obj.__init__(entitlement)
        return obj

    @classmethod
    def fromProps(cls, name=None, **kwargs):
        entitlement = paramsjson.Entitlement(name=name, **kwargs)
        return cls(entitlement)


class entitlements(Base):
    def __init__(self):
        """Constructor."""
        # Assign self to superclass for the broker classes that are aggregates of the specific type.
        # Without this assignment you'll get a nice and confusing recursion stacktrace if
        # a wayward programmer attempts to call a non-existent method on this class.
        self.superclass = self

    def list(self, activeOnly=True, product=None, **kwargs):
        '''
        Queries the entitlements RESTful interface with a given set of
        keywords.

        :param activeOnly: Limit results to only active entitlements
        :type activeOnly: boolean
        :param product: Product to limit results to
        :type product: string
        :param kwargs:
            Additional options passed to FilterHelper to filter results based
            on additional criteria.

            Example:

            .. code-block:: python

                entitlements.list(supportLevel='SUPPORTED')

        :type kwargs: dict
        :returns: A list of entitlement objects
        :rtype: list
        :raises:
            Exception if there was a connection related issue or an
            issue parsing headers
        '''
        url = '/support/v1/entitlements'
        # Strata is inverted in several respects, the option for filtering inactive entitlements being one.
        showAll = not activeOnly
        doc, headers = self._getProxy().get(url=SearchHelper.appendQuery(url, {"showAll": showAll, "product": product}))
        doc = [] if not isinstance(doc, list) else doc
        logger.debug("REST Response:\n%s" % doc)
        return ParseHelper.toCollection(entitlement,
                                        FilterHelper.filter(paramsjson.Entitlementsfromdict(doc), kwargs))


class case(paramsjson.Case, Base):
    def __init__(self, case):
        self.superclass = case

    def __new__(cls, case):
        if case is None: return None
        obj = object.__new__(cls)
        obj.__init__(case)
        return obj

    @classmethod
    def fromProps(cls, summary=None, product=None, version=None, **kwargs):
        case = paramsjson.Case(summary=summary, product=product, version=version, **kwargs)
#
        return cls(case)

    def update(self, data):
        '''
        Update this case. This case *must* have an case number.

        :raises:
            Exception if there was a connection related issue or an
            issue parsing headers
        '''
        url = '/support/v1/cases'
        if self.get_caseNumber() is not None:
            self._getProxy().update(url=UrlHelper.append(url, self.get_caseNumber()),
                                    body=json.dumps(data))
        else:
            raise RequestError('ID cannot be None on update')

    def get_comments(self):
        '''
        Retrieve a list of comments related to the case object

        :returns: A list of comment objects
        :rtype: list
        :raises:
            Exception if there was a connection related issue or an
            issue parsing headers
        '''
        try:
            return ParseHelper.toCollection(comment,
                                            paramsjson.Case.get_comments(self))
        except:
            # AttributeErrors happen if the API forgets to include the XML
            # Stanza for comments, send a empty list back.
            return []

    def get_attachments(self):
        '''
        Retrieve a list of attachments, related to the case object

        :returns: A list of attachment objects
        :rtype: list
        :raises:
            Exception if there was a connection related issue or an
            issue parsing headers
        '''
        url = '/support/v1/cases/{caseNumber}/attachments'
        url = UrlHelper.replace(url, {'{caseNumber}': quote(self.get_caseNumber())})
        doc, headers = self._getProxy().get(url)
        logger.debug("REST Response:\n%s" % doc)
        try:
            return ParseHelper.toCollection(attachment,
                                            paramsjson.CaseAttachmentsfromdict(doc))
        except:
            return []

    def get_caseResourceLinks(self):
        '''
        Retrieve a list of recommendations related to the case object

        :returns: A list of recommendation objects
        :rtype: list
        :raises:
            Exception if there was a connection related issue or an
            issue parsing headers
        '''
        try:
            return ParseHelper.toCollection(caseresourcelink,
                        paramsjson.Case.get_caseResourceLinks(self))
        except:
            # AttributeErrors happen if the API forgets to include the XML
            # Stanza for recommendations, send a empty list back.
            return []


    # def get_entitlement(self):
    #     return entitlement(params.case.get_entitlement(self))


class cases(Base):
    def __init__(self):
        """Constructor."""
        # Assign self to superclass for the broker classes that are aggregates of the specific type.
        # Without this assignment you'll get a nice and confusing recursion stacktrace if
        # a wayward programmer attempts to call a non-existent method on this class.
        self.superclass = self

    def add(self, cs):
        '''
        Add a new case

        :param cs: case object to be submitted to the Customer Portal
        :type cs: case
        :returns:
            The same case object with the case number and associated fields
            populated.

        :rtype: case
        :raises:
            Exception if there was a connection related issue or an
            issue parsing headers
        '''

        url = '/support/v1/cases'
        doc, headers = self._getProxy().add(url=url,
                                            body=json.dumps(ignore_none_values(cs.superclass.to_dict())))
        uri = doc[u'location'][0]
        parsed = urlparse(uri)
        cs.set_caseNumber(os.path.basename(parsed[2]))
        cs.set_uri(uri)
        return cs

    def get(self, caseNumber=None):
        '''
        Queries the API for the given case number.

        :param caseNumber: Case Number of case to retrieve from the API
        :type caseNumber: string
        :returns: The case object for the relevant caseNumber provided
        :rtype: case
        :raises:
            Exception if there was a connection related issue or an
            issue parsing headers
        '''
        url = '/support/v1/cases'

        doc, headers = self._getProxy().get(url=UrlHelper.append(url, caseNumber))
        logger.debug("REST Response:%s" % doc)
        return case(paramsjson.Casefromdict(doc))

    def list(self, keywords=None, includeClosed=False, detail=False,
             group=None, startDate=None, endDate=None, **kwargs):
        '''
        Queries the cases RESTful interface with a given set of keywords.

        :param keywords: Keywords to search cases on (space seperated)
        :type keywords: string
        :param detail:
        :type detail: boolean
        :param group: Case group to search for cases in
        :param startDate: Date to start listing cases from
        :type startDate:
            ISO 8601 formatted string, either YYYY-MM-DDThh:mm:ss or YYYY-MM-DD
        :param endDate: Date to list cases until
        :type endDate:
            ISO 8601 formatted string, either YYYY-MM-DDThh:mm:ss or YYYY-MM-DD
        :param searchopts: search options/query filters passed to the API
        :type searchopts: dict
        :param kwargs:
            Additional options passed to FilterHelper to filter results based
            on additional criteria.

            Example:

            .. code-block:: python

                cases.list(status="Closed")

        :type kwargs: dict
        :returns: A list of case objects
        :rtype: list
        :raises:
            Exception if there was a connection related issue or an
            issue parsing headers
        '''
        pass

    def filter(self, case_filter):
        '''
        Filter case results.

        :param case_filter: a filter object to be submitted
        :type cs: caseFilter
        :returns:
             A list of case objects
        :rtype: list
        :raises:
            Exception if there was a connection related issue or an
            issue parsing headers
        '''

        url = '/support/v1/cases/filter'
        doc, headers = self._getProxy().add(url=url,
                                            body=json.dumps(ignore_none_values((case_filter.to_dict()))))
        logger.debug("REST Response:\n%s" % doc)
        try:
            return ParseHelper.toCollection(case, FilterHelper.filter(paramsjson.Casesfromdict(doc[u'cases'])))
        except Exception as e:
            logger.error(e)
            return []


class symptom(params.symptom, Base):
    def __init__(self, symptom):
        self.superclass = symptom

    def __new__(cls, symptom):
        if symptom is None: return None
        obj = object.__new__(cls)
        obj.__init__(symptom)
        return obj

    @classmethod
    def fromProps(cls, caseNumber=None, category=None,
                  data=None, description=None,
                  location=None, summary=None,
                  uri=None, **kwargs):

        symptom = params.symptom(caseNumber=caseNumber, category=category,
                                 data=data, description=description,
                                 location=location, summary=summary,
                                 uri=uri, **kwargs)
        return cls(symptom)


class extractedSymptom(params.extractedSymptom, Base):
    def __init__(self, extractedSymptom):
        self.superclass = extractedSymptom

    def __new__(cls, extractedSymptom):
        if extractedSymptom is None: return None
        obj = object.__new__(cls)
        obj.__init__(extractedSymptom)
        return obj

    @classmethod
    def fromProps(cls, label=None, createdBy=None, createdDate=None, lastModifiedBy=None,
                  lastModifiedDate=None, linked=None, linkedBy=None, linkedAt=None,
                  type_=None, category=None, occurrences=None, verbatim=None, fields=None,
                  beginIndex=None, endIndex=None, summary=None, signature=None, timestamp=None):

        symptom = params.extractedSymptom(label=label, createdBy=createdBy, createdDate=createdDate,
                                          lastModifiedBy=lastModifiedBy, lastModifiedDate=lastModifiedDate,
                                          linked=linked, linkedBy=linkedBy, linkedAt=linkedAt,
                                          type_=type_, category=category, occurrences=occurrences,
                                          verbatim=verbatim, fields=fields, beginIndex=beginIndex,
                                          endIndex=endIndex, summary=summary, signature=signature,
                                          timestamp=timestamp)
        return cls(extractedSymptom)


class symptoms(Base):
    def __init__(self):
        """Constructor."""
        # Assign self to superclass for the broker classes that are aggregates of the specific type.
        # Without this assignment you'll get a nice and confusing recursion stacktrace if
        # a wayward programmer attempts to call a non-existent method on this class.
        self.superclass = self


    def add(self, sym):
        pass

    def extractFromStr(self,
                       content=None):
        '''
        Queries the symptom extractor RESTful interface with a given string.

        :param content:
            The text that you wish to be analyzed by the diagnostics engine

        :type content: string
        :returns: A list of extractedSymptom objects
        :rtype: list
        :raises:
            Exception if there was a connection related issue or an
            issue parsing headers
        '''
        pass

    def extractFromFile(self, fileName=None):
        '''
        Queries the symptom extractor RESTful interface with a given string.

        :param fileName:
            The path of the file that you wish to be analyzed
            by the diagnostics engine

        :type fileName: file
        :returns: A list of extractedSymptom objects
        :rtype: list
        :raises:
            Exception if there was a connection related issue or an
            issue parsing headers

            socket.error or IOError for issues when reading the specified file.
        '''
        pass


class problem(paramsjson.Recommendation, Base):
    def __init__(self, problem):
        self.superclass = problem

    def __new__(cls, problem):
        if problem is None:
            return None
        obj = object.__new__(cls)
        obj.__init__(problem)
        return obj

    @classmethod
    def fromProps(cls, view_uri=None, allTitle=None, id=None, abstract=None):
        problem = paramsjson.Recommendation(viewuri=view_uri, allTitle=allTitle, id=id, abstract=abstract)
        return cls(problem)


class problems(Base):
    def __init__(self):
        """Constructor."""
        # Assign self to superclass for the broker classes that are aggregates of the specific type.
        # Without this assignment you'll get a nice and confusing recursion stacktrace if
        # a wayward programmer attempts to call a non-existent method on this class.
        self.superclass = self

    def diagnoseStr(self,
                    content=None, start=0, rows=50):
        '''
        Queries the problems RESTful interface with a given string.

        :param content:
            The text that you wish to be analyzed by the diagnostics engine

        :type content: string
        :returns: A list of problem objects
        :rtype: list
        :raises:
            Exception if there was a connection related issue or an
            issue parsing headers
        '''
        url = '/support/search/recommendations'

        body = {"summary": content}
        doc, headers = self._getProxy().action(url=SearchHelper.appendQuery(url, {"start": start, "rows": rows}),
                                               body=json.dumps(body))
        logger.debug("REST Response:\n%s" % doc)

        try:
            return ParseHelper.toCollection(problem, paramsjson.Recommendationsfromdict(doc[u'response'][u'docs']))
        except Exception as e:
            logger.error(e)
            return []

    def diagnoseFile(self, fileName=None):
        '''
        THIS IS NOW NOT SUPPORTED BY Hydra
        Queries the problems RESTful interface with a given file.

        :param fileName:
            The path of the file that you wish to be analyzed
            by the diagnostics engine

        :type fileName: file
        :returns: A list of problem objects
        :rtype: list
        :raises:
            Exception if there was a connection related issue or an
            issue parsing headers

            socket.error or IOError for issues when reading the specified file.
        '''

        print("Diagnose file or directory is no longer supported.")
        return []

    def diagnoseCase(self, diagcase=None, fl='*', start=0, rows=50):
        '''
        Queries the problems RESTful interface with a given file.

        :param diagcase:
            The case to be evaluated by the

        :type diagcase: case object
        :returns: A list of recommendation objects
        :rtype: list
        :raises:
            Exception if there was a connection related issue or an
            issue parsing headers

            socket.error or IOError for issues when reading the specified file.
        '''
        url = '/support/search/recommendations'
        try:
            doc, headers = self._getProxy().action(
                url=(SearchHelper.appendQuery(url, {"start": start, "rows": rows}) + '&fl={}'.format(fl)),
                body=json.dumps(ignore_none_values((diagcase.superclass.to_dict())))
                )
        except socket.error, se:
            logger.debug('Socket error: msg(%s)' % (se))
            raise ConnectionError, str(se)
        except IOError, ioe:
            logger.debug('I/O error: errno(%s) strerror(%s)' % (ioe.errno, ioe.strerror))
            raise
        except Exception, e:
            logger.debug('Unexpected exception: msg(%s)' % (str(e)))
            raise
        logger.debug("REST Response:\n%s" % doc)
        try:
            return ParseHelper.toCollection(problem, paramsjson.Recommendationsfromdict(doc[u'response'][u'docs']))
        except Exception as e:
            logger.error(e)
            return []


class product(paramsjson.Product, Base):
    def __init__(self, product):
        self.superclass = product

    def __new__(cls, product):
        if product is None:
            return None
        obj = object.__new__(cls)
        obj.__init__(product)
        return obj

    @classmethod
    def fromProps(cls, productId=None, name=None):
        product = paramsjson.Product(productId, name)
        return cls(product)

    def get_versions(self):
        url = '/support/v2/products/{prodName}/versions'
        url = UrlHelper.replace(url, {'{prodName}': quote(self.get_name())})
        doc, headers = self._getProxy().get(url)
        doc = [] if not isinstance(doc, list) else doc
        return paramsjson.get_version(doc)


class products(Base):
    def __init__(self):
        """Constructor."""
        # Assign self to superclass for the broker classes that are aggregates of the specific type.
        # Without this assignment you'll get a nice and confusing recursion stacktrace if
        # a wayward programmer attempts to call a non-existent method on this class.
        self.superclass = self

    def list(self, **kwargs):
        '''
        Queries the products RESTful interface with a given set of keywords.

        :param kwargs:
            Properties to be passed to FilterHelper to filter results based
            on additional criteria.

            Example:

            .. code-block:: python

                products.list(name="Fuse ESB")

        :type kwargs: dict
        :returns: A list of product objects
        :rtype: list
        :raises:
            Exception if there was a connection related issue or an
            issue parsing headers
        '''
        url = '/support/v2/products'
        doc, headers = self._getProxy().get(url)
        logger.debug("REST Response:\n%s" % doc)
        doc = [] if not isinstance(doc, list) else doc
        return ParseHelper.toCollection(product,
                                        FilterHelper.filter(paramsjson.Productsfromdict(doc), kwargs))


class caseresourcelink(paramsjson.CaseResourceLink, Base):
    def __init__(self, recommendation):
        self.superclass = recommendation

    def __new__(cls, recommendation):
        if recommendation is None:
            return None
        obj = object.__new__(cls)
        obj.__init__(recommendation)
        return obj

    @classmethod
    def fromProps(cls, **kwargs):

        prec = paramsjson.CaseResourceLink(**kwargs)
        return cls(prec)


class group(paramsjson.CaseGroup, Base):
    def __init__(self, group):
        self.superclass = group

    def __new__(cls, group):
        if group is None:
            return None
        obj = object.__new__(cls)
        obj.__init__(group)
        return obj

    @classmethod
    def fromProps(cls, number=None, name=None, uri=None, isPrivate=None):
        group = paramsjson.CaseGroup(groupNum=number, name=name, isPrivate=isPrivate)
        return cls(group)


class groups(Base):
    def __init__(self):
        """Constructor."""
        # Assign self to superclass for the broker classes that are aggregates of the specific type.
        # Without this assignment you'll get a nice and confusing recursion stacktrace if
        # a wayward programmer attempts to call a non-existent method on this class.
        self.superclass = self

    def list(self):
        '''
        Queries the groups RESTful interface for a list of case groups.

        :returns: A list of group objects
        :rtype: list
        :raises:
            Exception if there was a connection related issue or an
            issue parsing headers
        '''
        url = '/support/v1/accounts/casegroups'
        doc, headers = self._getProxy().get(url)
        logger.debug("REST Response:\n%s" % doc)
        doc = [] if not isinstance(doc, list) else doc
        return ParseHelper.toCollection(group,
                                        FilterHelper.filter(paramsjson.CaseGroupsfromdict(doc)))

    def get(self, groupNumber=None):
        '''
        Queries the API for the given group number.
        '''
        url = '/support/v1/accounts/casegroups'
        doc, headers = self._getProxy().get(url=UrlHelper.append(url, groupNumber))
        logger.debug("REST Response:%s" % doc)
        g = group(paramsjson.CaseGroupfromdict(doc))
        return g


class user(paramsjson.Contact, Base):
    def __init__(self, user):
        self.superclass = user

    def __new__(cls, user):
        if user is None:
            return None
        obj = object.__new__(cls)
        obj.__init__(user)
        return obj

    @classmethod
    def fromProps(cls, firstName=None, lastName=None, orgAdmin=None,
                  hasChat=None, sessionId=None, isInternal=None,
                  canAddAttachments=None):
        user = paramsjson.Contact(firstName=firstName, lastName=lastName,
                                  isOrgAdmin=orgAdmin, hasChat=hasChat,
                                  isInternal=isInternal, canAddAttachments=canAddAttachments)
        return cls(user)


class users(Base):
    def __init__(self):
        """Constructor."""
        # Assign self to superclass for the broker classes that are aggregates of the specific type.
        # Without this assignment you'll get a nice and confusing recursion stacktrace if
        # a wayward programmer attempts to call a non-existent method on this class.
        self.superclass = self

    def get(self, userName=None):
        '''
        Queries the API for the given user name.
        '''
        url = '/support/v1/contacts/sso'
        doc, headers = self._getProxy().get("%s/%s" % (url, userName))
        logger.debug("REST Response:%s" % doc)
        return user(paramsjson.Contactfromdict(doc))


class caseFilter(paramsjson.CaseFilter, Base):
    def __init__(self, caseFilter):
        self.superclass = caseFilter

    def __new__(cls, caseFilter):
        if caseFilter is None:
            return None
        obj = object.__new__(cls)
        obj.__init__(caseFilter)
        return obj

    @classmethod
    def fromProps(cls, endDate=None, accountNumber=None,
                  includeClosed=None, groupNumbers=None,
                  includePrivate=None, keyword=None,
                  count=None, start=None, onlyUngrouped=None,
                  ownerSSOName=None, product=None, severity=None,
                  sortField=None, sortOrder=None, startDate=None,
                  status=None, type_=None, associateSSOName=None, view=None):

        filt = paramsjson.CaseFilter(endDate=endDate,
                  accountNumber=accountNumber,
                  includeClosed=includeClosed,
                  groupNumbers=groupNumbers,
                  includePrivate=includePrivate, keyword=keyword,
                  maxResults=count, offset=start, onlyUngrouped=onlyUngrouped,
                  ownerSSOName=ownerSSOName, product=product,
                  severity=severity, sortField=sortField,
                  sortOrder=sortOrder, startDate=startDate,
                  status=status, type=type_, associateSSOName=associateSSOName,
                  view=view)
        return cls(filt)


class values(Base):
    def __init__(self):
        """Constructor."""
        # Assign self to superclass for the broker classes that are aggregates of the specific type.
        # Without this assignment you'll get a nice and confusing recursion stacktrace if
        # a wayward programmer attempts to call a non-existent method on this class.
        self.superclass = self

    def getType(self, **kwargs):
        '''
        Queries the API for available case type values.
        RESTful method: https://api.access.redhat.com/support/v1/cases/metadata/types

        Keyword arguments:
        returns   -- A list of case type values.
        raises   -- An exception if there was a connection related issue.
        '''
        url = '/support/v1/cases/metadata/types'

        doc, headers = self._getProxy().get(url)
        logger.debug("REST Response:\n%s" % doc)
        try:
            type_values = paramsjson.Valuesfromdict(doc).get_values()
            return type_values
        except Exception as e:
            logger.error(e)
            return []

    def getSeverity(self, **kwargs):
        '''
        Queries the API for available case severity values.
        RESTful method: https://api.access.redhat.com/support/v1/cases/metadata/severities

        Keyword arguments:
        returns   -- A list of case severity values.
        raises   -- An exception if there was a connection related issue.
        '''
        url = '/support/v1/cases/metadata/severities'

        doc, headers = self._getProxy().get(url)
        logger.debug("REST Response:\n%s" % doc)
        try:
            severity_values = paramsjson.Valuesfromdict(doc).get_values()
            return severity_values
        except Exception as e:
            logger.error(e)
            return []

    def getStatus(self, **kwargs):
        '''
        Queries the API for available case status values.
        RESTful method: https://api.access.redhat.com/support/v1/cases/metadata/statuses

        Keyword arguments:
        returns   -- A list of case status values.
        raises   -- An exception if there was a connection related issue.
        '''
        url = '/support/v1/cases/metadata/statuses'

        doc, headers = self._getProxy().get(url)
        logger.debug("REST Response:\n%s" % doc)
        try:
            status_values = paramsjson.Valuesfromdict(doc).get_values()
            return status_values
        except Exception as e:
            logger.error(e)
            return []

    def getInternalStatus(self, **kwargs):
        '''
         list of internal status values

        '''
        internal_status_values = ['Unassigned',
                                  'Waiting on Customer',
                                  'Waiting on Collaboration',
                                  'Waiting on Contributor',
                                  'Waiting on Engineering',
                                  'Waiting on PM',
                                  'Waiting on Sales',
                                  'Waiting on QA',
                                  'Waiting on Owner',
                                  'Waiting on Translation',
                                  'Waiting on 3rd Party Vendor',
                                  'Waiting on Collaboration - Native',
                                  'Closed']
        return internal_status_values


class searchResult(params.searchResult, Base):
    def __init__(self, searchResult):
        self.superclass = searchResult

    def __new__(cls, searchResult):
        if searchResult is None: return None
        obj = object.__new__(cls)
        obj.__init__(searchResult)
        return obj

    @classmethod
    def fromProps(cls, **kwargs):

        symptom = params.searchResult(**kwargs)
        return cls(searchResult)

class search(Base):
    def __init__(self):
        """Constructor."""
        # Assign self to superclass for the broker classes that are aggregates of the specific type.
        # Without this assignment you'll get a nice and confusing recursion stacktrace if
        # a wayward programmer attempts to call a non-existent method on this class.
        self.superclass = self

    def search(self, keywords=None, **kwargs):
        '''
        Queries the search RESTful interface with a given set of keywords.

        :param keywords: Search string
        :type keywords: string
        :param searchopts: search options/query filters passed to the API
        :type searchopts: dict
        :param kwargs:
            Additional options passed to FilterHelper

            Example:

            .. code-block:: python

                api.search.search('RHEV', authorSSOName="anonymous")

        :returns: A list of searchResult objects
        :rtype: list
        :raises:
            Exception if there was a connection related issue or an
            issue parsing headers
        '''
        pass

class InstanceMaker(object):
    '''
    Utility class to make single instances of case, solution, article, entitlement, and comment.
    This is useful so that you only need import API and not all of the helper classes used by
    API to get the job done.
    '''
    @classmethod
    def makeCase(self, summary=None, product=None, version=None, **kwargs):
        return case.fromProps(summary=summary, product=product, version=version, **kwargs)

    @classmethod
    def makeSolution(self, createdBy=None, title=None,
                     summary=None, kcsState=None,
                     resolution=None, **kwargs):
        return solution.fromProps(createdBy=createdBy, title=title,
                                  summary=summary, kcsState=kcsState,
                                  resolution=resolution, **kwargs)

    @classmethod
    def makeArticle(self, createdBy=None, title=None,
                    summary=None, kcsState=None,
                    body=None, **kwargs):
        return article.fromProps(createdBy=createdBy, title=title,
                              summary=summary, kcsState=kcsState,
                              body=body, **kwargs)

    @classmethod
    def makeEntitlement(self, name=None, **kwargs):
        return entitlement.fromProps(name=name, **kwargs)

    @classmethod
    def makeComment(self, caseNumber=None, text=None,
                    public=True, **kwargs):

        user_api = users()
        userobj = user_api.get(confighelper.get_config_helper().username)

        if userobj.isInternal or public is False:
            return comment.fromProps(caseNumber=caseNumber,
                                     text=text,
                                     public=public,
                                     **kwargs)
        else:
            return comment.fromProps(caseNumber=caseNumber,
                                     text=text,
                                     public=None, # the public flag should not be present for external users or else we get 400, hence setting as None
                                     **kwargs)
    @classmethod
    def makeAttachment(self, caseNumber=None, public=True,
                       fileName=None, description=None, **kwargs):
        return attachment.fromProps(caseNumber=caseNumber, private=not public,
                                      fileName=fileName, description=description,
                                      **kwargs)
    @classmethod
    def makeSymptom(self, caseNumber=None, category=None,
                    data=None, description=None,
                    location=None, summary=None,
                    uri=None, **kwargs):
        return symptom.fromProps(caseNumber=caseNumber, category=category,
                                 data=data, description=description,
                                 location=location, summary=summary,
                                 uri=uri, **kwargs)

    @classmethod
    def makeCaseFilter(cls, endDate=None, accountNumber=None,
                       includeClosed=None, groupNumbers=None,
                       includePrivate=None, keyword=None,
                       count=None, start=None, onlyUngrouped=None,
                       ownerSSOName=None, product=None, severity=None,
                       sortField=None, sortOrder=None, startDate=None,
                       status=None, type_=None, associateSSOName=None,
                       view=None):
        return caseFilter.fromProps(endDate=endDate,
                                    accountNumber=accountNumber,
                                    includeClosed=includeClosed,
                                    groupNumbers=groupNumbers,
                                    includePrivate=includePrivate,
                                    keyword=keyword,
                                    count=count, start=start,
                                    onlyUngrouped=onlyUngrouped,
                                    ownerSSOName=ownerSSOName,
                                    product=product,
                                    severity=severity,
                                    sortField=sortField,
                                    sortOrder=sortOrder,
                                    startDate=startDate,
                                    status=status,
                                    type_=type_,
                                    associateSSOName=associateSSOName,
                                    view=view)


