#! /usr/bin/env bash

set -eu

declare -A HDRS

# shellcheck disable=SC2034
SRCS_pinctrl_core=(
	"pinctrl/core.h"
	"pinctrl/pinctrl-utils.h"
)
HDRS["pinctrl_core"]="pinctrl:"
# shellcheck disable=SC2034
SRCS_pinctrl_msm=(
	"pinctrl/qcom/pinctrl-msm.c"
	"pinctrl/qcom/pinctrl-msm.h"
)
HDRS["pinctrl_msm"]="pinctrl: msm:"
# shellcheck disable=SC2034
SRCS_pinctrl_sa8775p=(
	"pinctrl/qcom/pinctrl-sa8775p.c"
)
HDRS["pinctrl_sa8775p"]="pinctrl: qcom: sa8775p:"
# shellcheck disable=SC2034
SRCS_qcom_dma_buf_heaps=(
	"dma-buf/heaps/qcom_carveout_heap.c"
	"dma-buf/heaps/qcom_carveout_heap.h"
	"dma-buf/heaps/qcom_cma_heap.c"
	"dma-buf/heaps/qcom_cma_heap.h"
	"dma-buf/heaps/qcom_dma_heap.c"
	"dma-buf/heaps/qcom_dma_heap_priv.h"
	"dma-buf/heaps/qcom_dma_heap_secure_utils.c"
	"dma-buf/heaps/qcom_dma_heap_secure_utils.h"
	"dma-buf/heaps/qcom_dt_parser.c"
	"dma-buf/heaps/qcom_dt_parser.h"
	"dma-buf/heaps/qcom_dynamic_page_pool.c"
	"dma-buf/heaps/qcom_dynamic_page_pool.h"
	"dma-buf/heaps/qcom_secure_system_heap.c"
	"dma-buf/heaps/qcom_secure_system_heap.h"
	"dma-buf/heaps/qcom_sg_ops.c"
	"dma-buf/heaps/qcom_sg_ops.h"
	"dma-buf/heaps/qcom_system_heap.c"
	"dma-buf/heaps/qcom_system_heap.h"
	"include/linux/msm_dma_iommu_mapping.h"
	"include/linux/qcom-dma-mapping.h"
	"include/linux/qcom_dma_heap.h"
	"include/linux/qti-smmu-proxy-callbacks.h"
)
HDRS["qcom_dma_buf_heaps"]="dma-buf: heaps: qcom:"
# shellcheck disable=SC2034
SRCS_qcom_geni_se=(
	"soc/qcom/qcom-geni-se.c"
	"include/linux/soc/qcom/geni-se.h"
)
HDRS["qcom_geni_se"]="soc: qcom:"
# shellcheck disable=SC2034
SRCS_qcom_geni_serial=(
	"tty/serial/qcom_geni_serial.c"
)
HDRS["qcom_geni_serial"]="tty: serial: qcom-geni-serial:"
# shellcheck disable=SC2034
SRCS_qcom_mem_buf=(
	# Include the gunyah headers. It's not enabled so we don't have a
	# module to link to, but mem_buf uses some types defined in the
	# headers.
	"include/linux/gunyah.h"
	"include/linux/gunyah_rsc_mgr.h"
	"include/linux/gunyah/gh_common.h"
	"include/linux/gunyah/gh_rm_drv.h"

	"soc/qcom/mem_buf/mem-buf.c"
	"soc/qcom/mem_buf/mem-buf-dev.c"
	"soc/qcom/mem_buf/mem-buf-dev-gh.c"
	"soc/qcom/mem_buf/mem-buf-dev.h"
	"soc/qcom/mem_buf/mem_buf_dma_buf.c"
	"soc/qcom/mem_buf/mem-buf-gh.c"
	"soc/qcom/mem_buf/mem-buf-gh.h"
	"soc/qcom/mem_buf/mem-buf-ids.c"
	"soc/qcom/mem_buf/mem-buf-ids.h"
	"soc/qcom/mem_buf/mem-buf-msgq.c"
	"soc/qcom/mem_buf/mem-buf-msgq.h"
	"soc/qcom/mem_buf/trace-mem-buf.h"
	"include/linux/mem-buf-exporter.h"
	"include/linux/mem-buf.h"
	"include/uapi/linux/mem-buf.h"
)
HDRS["qcom_mem_buf"]="soc: qcom: mem_buf:"
# shellcheck disable=SC2034
SRCS_qcom_qtee_shmbridge=(
	"firmware/qtee_shmbridge.c"
	"include/linux/qtee_shmbridge.h"
)
HDRS["qcom_qtee_shmbridge"]="firmware: qtee_shmbridge:"
# shellcheck disable=SC2034
SRCS_qcom_secure_buffer=(
	"soc/qcom/secure_buffer.c"
	"soc/qcom/trace_secure_buffer.h"
	"include/soc/qcom/secure_buffer.h"
)
HDRS["qcom_secure_buffer"]="soc: qcom: secure_buffer:"
SRCS_ufs_qcom=(
	"ufs/host/ufs-qcom.c"
	"ufs/host/ufshcd-pltfrm.h"
	"ufs/host/ufs-qcom.h"
)
HDRS["ufs_qcom"]="scsi: ufs: qcom:"
# shellcheck disable=SC2034
SRCS_i2c_qcom_geni=(
	"i2c/busses/i2c-qcom-geni.c"
)
HDRS["i2c_qcom_geni"]="i2c: qcom-geni:"
# shellcheck disable=SC2034
SRCS_spi_geni_qcom=(
	"spi/spi-geni-qcom.c"
)
HDRS["spi_geni_qcom"]="spi: geni-qcom:"
# shellcheck disable=SC2034
SRCS_dma_headers=(
	"dma/virt-dma.h"
	"dma/dmaengine.h"
)
HDRS["dma_headers"]="dmaengine: virt-dma:"
# shellcheck disable=SC2034
SRCS_dma_qcom_gpi=(
	"dma/qcom/gpi.c"
	"include/linux/dma/qcom-gpi-dma.h"
)
HDRS["dma_qcom_gpi"]="dma: qcom: gpi:"

MODULES_DFL=(
	"dma_headers" # Headers only
	"dma_qcom_gpi"
	"i2c_qcom_geni"
	"pinctrl_core" # Headers only
	"pinctrl_msm"
	"pinctrl_sa8775p"
	"qcom_dma_buf_heaps"
	"qcom_geni_se"
	"qcom_geni_serial"
	"qcom_mem_buf"
	"qcom_secure_buffer"
	"qcom_qtee_shmbridge"
	"spi_geni_qcom"
	"ufs_qcom"
)

KGIT_DFL="https://gitlab.com/redhat/centos-stream/src/kernel/centos-stream-10.git"
KTAG_DFL="kernel-6.12.0-145.el10"
ODIR_DFL="$(pwd)"
usage() {
	cat - <<EOF
$0 [-h] [ -k KERNEL_GIT ] [ -K KERNEL_DIR ] [ -t KERNEL_TAG ] [ -C OUT_OF_TREE_REPOSITORY ] [ -m module[,...] ]

Extract the sources files from the kernel repository at the given tag in preparation for the Qualcomm out-of-tree modules.
A commit for each sub-system extracted will be created in the out-of-tree repository.
Defaults:
	Repository: $KGIT_DFL
	Tag: $KTAG_DFL
	Out-of-tree repository: $ODIR_DFL
EOF
}

KGIT="${KGIT_DFL}"
KTAG="${KTAG_DFL}"
ODIR="${ODIR_DFL}"
parse_cli() {
	local cli_args
	if ! cli_args=$(getopt -o 'hk:K:t:C:m:' --long 'help,kernel:,tag:,dir:,modules:' -- "$@"); then
		usage
		exit 1
	fi
	eval set -- "$cli_args"

	while true; do
		case "$1" in
		'-h' | '--help')
			usage
			exit 0
			;;
		'-k' | '--kernel')
			KGIT="$2"
			shift 2
			continue
			;;
		'-K' | '--kdir')
			KDIR="$2"
			shift 2
			continue
			;;
		'-t' | '--tag')
			KTAG="$2"
			shift 2
			continue
			;;
		'-C' | '--dir')
			ODIR="$2"
			shift 2
			continue
			;;
		'-m' | '--modules')
			IFS=', ' read -r -a MODULES <<<"$2"
			shift 2
			continue
			;;
		'--')
			shift
			break
			;;
		*)
			echo 'getopt error!' >&2
			exit 1
			;;
		esac
	done
}

parse_cli "$@"

if [[ ! -d "$ODIR" ]]; then
	echo "Out-of-tree repository does not exists: '$ODIR'" >&2
	exit 1
fi
if [[ ! -d "$ODIR/.git" ]]; then
	echo "Out-of-tree repository is not a git repository: '$ODIR'" >&2
	exit 1
fi

pushd "$ODIR" >/dev/null

if [[ -n "$KDIR" ]]; then
	git -C "$KDIR" checkout "$KTAG"
else
	KDIR=$(mktemp -d)
	git clone --depth 1 --revision "$KTAG" "$KGIT" "$KDIR"
fi

if [[ "${#MODULES[@]}" -eq 0 ]]; then
	MODULES=("${MODULES_DFL[@]}")
fi

for mod in "${MODULES[@]}"; do
	if ! printf "%s\n" "${MODULES_DFL[@]}" | grep -q -- "$mod"; then
		echo "\"$mod\" is not supported" >&2
		continue
	fi
	echo "$mod:"
	declare -n srcs=SRCS_$mod
	srcdir=$(dirname "${srcs[0]}")
	# Only add to Kbuild if there are sources to build.
	if printf "%s\n" "${srcs[@]}" | grep -q -- '\.c'; then
		str="obj-m += $srcdir/"
		grep -Fxqs "$str" Kbuild || echo "$str" >>Kbuild
	fi
	mkdir -p "$srcdir"
	for file in "${srcs[@]}"; do
		if [[ "$file" = include/* ]]; then
			echo "HEADER $file"
			mkdir -p "$(dirname "$file")"
			cp -v "$KDIR/$file" "$file"
		else
			echo "FILE $file"
			set -x
			cp -v "$KDIR/drivers/$file" "$file"
			if [[ "$file" = *.c ]]; then
				echo "KBUILD ${file##.c}.o: $file"
				echo "obj-m += $(basename "${file%%.c}.o")" >>"$srcdir/Kbuild"
				cat "$srcdir/Kbuild"
			fi
			set +x
		fi
	done

	git add "${srcs[@]}"
	git add "./*Kbuild*"
	git commit -s -m "${HDRS["$mod"]} import $KTAG" -m "Upstream-Status: $KGIT"
done

popd >/dev/null
