// SPDX-License-Identifier: GPL-2.0-only
/*
 * Qualcomm PCIe ECAM root host controller driver
 * Copyright (c) 2024 Qualcomm Innovation Center, Inc. All rights reserved.
 */
#define pr_fmt(fmt) KBUILD_MODNAME ": " fmt

#include <linux/irq.h>
#include <linux/irqchip/chained_irq.h>
#include <linux/irqdomain.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/msi.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/of_irq.h>
#include <linux/of_pci.h>
#include <linux/pci.h>
#include <linux/pci-ecam.h>
#include <linux/platform_device.h>
#include <linux/pm_domain.h>
#include <linux/pm_runtime.h>
#include <linux/slab.h>
#include <linux/device.h>
#include <linux/types.h>
#include "pci-host-common.h"

#define PCIE_MSI_CTRL_BASE			(0x820)
#define PCIE_MSI_CTRL_SIZE			(0x68)
#define PCIE_MSI_CTRL_ADDR_OFFS			(0x0)
#define PCIE_MSI_CTRL_UPPER_ADDR_OFFS		(0x4)
#define PCIE_MSI_CTRL_INT_N_EN_OFFS(n)		(0x8 + 0xc * (n))
#define PCIE_MSI_CTRL_INT_N_MASK_OFFS(n)	(0xc + 0xc * (n))
#define PCIE_MSI_CTRL_INT_N_STATUS_OFFS(n)	(0x10 + 0xc * (n))

#define	MSI_DB_ADDR	0xa0000000
#define MSI_IRQ_PER_GRP (32)

#define PCI_DBI_SIZE   0x10000000
#define DBI_BASE_ADDR	0x350
#define DBI_BASE_ADDR_HIGH	0x354

#define ADDR_LOW(n) (n & GENMASK(31, 0))
#define ADDR_HIGH(n) ((n & GENMASK_ULL(63, 32)) >> 32)

/* Error Reporting Parf Registers */
#define PARF_INT_ALL_STATUS			0x224
#define PARF_INT_ALL_CLEAR			0x228
#define PARF_INT_CLEAR				0x21c
#define PARF_INT_STATUS			0x220
#define PARF_INT_ALL_MASK			0x22c
#define PARF_INT_ALL_2_CLEAR			0x504
#define PARF_INT_ALL_2_STATUS			0x500
#define PARF_INT_ALL_3_CLEAR			0x2e14
#define PARF_INT_ALL_3_STATUS			0x2e10
#define PARF_INT_ALL_4_CLEAR			0x2dd8
#define PARF_INT_ALL_4_STATUS			0x2dd0
#define PARF_INT_ALL_5_CLEAR			0x2ddc
#define PARF_INT_ALL_5_STATUS			0x2dd4
#define PARF_CFG_SAFETY_INT_MASK_CTRL		0x2c60


#define PCIE_AER_EXT_CAP_ID			0x01
#define PCI_EXT_CAP_RASDP_ID			0x0b

/* Interrupt Masks */
#define CFGPCIE_INT_ALL_STATUS_MASK		0x3ff3e
#define CFGPCIE_PARF_INT_STATUS_MASK		0x1b
#define CFGPCIE_INTERFACE_TIMER_STATUS_MASK	0xe7b
#define CFGPCIE_INT_ALL_2_STATUS_MASK		GENMASK(24, 0)
#define CFGPCIE_INT_ALL_3_STATUS_MASK		GENMASK(31, 0)
#define CFGPCIE_INT_ALL_4_STATUS_MASK		GENMASK(31, 0)
#define CFGPCIE_INT_ALL_5_STATUS_MASK		GENMASK(31, 0)

/* PCI_INTERRUPT_LINE register field */
#define SERR_EN					BIT(17)

/* DBI_ROOT_CONTROL_ROOT_CAPABILITIES_REG register fields */
#define PCIE_CAP_SYS_ERR_ON_CORR_ERR_EN		BIT(0)
#define PCIE_CAP_SYS_ERR_ON_NON_FATAL_ERR_EN	BIT(1)
#define PCIE_CAP_SYS_ERR_ON_FATAL_ERR_EN	BIT(2)

/* DBI_DEVICE_CONTROL_DEVICE_STATUS register fields */
#define PCIE_CAP_UNSUPPORT_REQ_REP_EN		BIT(3)
#define PCIE_CAP_FATAL_ERR_REPORT_EN		BIT(2)
#define PCIE_CAP_NON_FATAL_ERR_REPORT_EN	BIT(1)
#define PCIE_CAP_CORR_ERR_REPORT_EN		BIT(0)

/* PARF_CFG_SAFETY_INT_MASK_CTRL register fields */
#define CFG_SAFETY_UNCORR_INT_MASK		BIT(0)
#define CFG_SAFETY_CORR_INT_MASK		BIT(1)

/* DBI_ADV_ERR_CAP_CTRL_OFF register fields */
#define ECRC_GEN_EN				BIT(6)
#define ECRC_CHECK_EN				BIT(8)

/* DBI_ROOT_ERR_CMD_OFF register fields */
#define CORR_ERR_REPORTING_EN			BIT(0)
#define NON_FATAL_ERR_REPORTING_EN		BIT(1)
#define FATAL_ERR_REPORTING_EN			BIT(2)

/* DBI_SAFETY_MASK_OFF register fields */
#define SAFETY_INT_MASK			GENMASK(5, 0)

/* DBI_SAFETY_STATUS register fields */
#define PCIE_RASDP_UNCORR_ERR			BIT(0)
#define PCIE_IFACE_TMR_ERR			BIT(1)
#define PCIE_CDM_CHK_ERR			BIT(2)
#define PCIE_AER_UNCORR_ERR			BIT(3)
#define PCIE_AER_CORR_ERR			BIT(4)
#define PCIE_RASDP_CORR_ERR			BIT(5)

#define PCIE_MISC_CONTROL_1_OFF		0x8BC
#define PCIE_DBI_RO_WR_EN		BIT(0)

/*
 * Error Reporting DBI register
 */
#define DBI_DEVICE_CONTROL_DEVICE_STATUS	0x78
#define DBI_ROOT_CONTROL_ROOT_CAPABILITIES_REG	0x8c
#define DBI_INTERFACE_TIMER_STATUS		0x938
#define DBI_SAFETY_MASK_OFF			0x960
#define DBI_SAFETY_STATUS			0x964

#define DBI_ADV_ERR_CAP_CTRL_OFF		0x18
#define DBI_ROOT_ERR_CMD_OFF			0x12c

/*
 * RAS-DP register
 */
#define PCIE_RASDP_ERROR_MODE_EN_REG		0x3B4
#define RASDP_ERROR_MODE_EN			BIT(0)

/*
 * Interface Timer register
 */
#define PCIE_INTERFACE_TIMER_CONTROL		0x930
#define INTERFACE_TIMER_EN			BIT(0)
#define INTERFACE_TIMER_AER_EN			BIT(1)

#define PCIE_PL_CHK_REG_CONTROL_STATUS		0xB20
#define PCIE_PL_CHK_REG_CHK_REG_START		BIT(0)
#define PCIE_PL_CHK_REG_CHK_REG_CONTINUOUS	BIT(1)
#define PCIE_PL_CHK_REG_CHK_REG_COMPARISON_ERROR	BIT(16)
#define PCIE_PL_CHK_REG_CHK_REG_LOGIC_ERROR	BIT(17)
#define PCIE_PL_CHK_REG_CHK_REG_COMPLETE	BIT(18)

enum qcom_pcie_fault_code {
	RASDP_UNCORR_ERROR,	/* RASDP uncorrectable error */
	RASDP_CORR_ERROR,	/* RASDP correctable error */
	CDM_REG_CHK_ERROR,	/* CDM register check error */
	INTERFACE_TIMER_ERROR,	/* PCIe local bus interface timer error */
	PCIE_SPURIOUS_INT,	/* Spurious Interrupt received */
	MAX_PCIE_SAFETY_FAULT	/* Maximum PCIe fault source code supported */
};

static const char * const pcie_fault_string[] = {
	"RASDP_Uncorr_Error",
	"RASDP_Corr_Error",
	"CDM_Reg_Chk_Error",
	"Interface_Timer_Error",
	"PCIe_Spurious_Interrupt",
	"TOTAL_PCIE_FAULTS",
};

/**
 * struct qcom_msi_irq - MSI IRQ information
 * @client:	pointer to MSI client struct
 * @grp:	group the irq belongs to
 * @grp_index:	index in group
 * @hwirq:	hwirq number
 * @virq:	virq number
 * @pos:	position in MSI bitmap
 */
struct qcom_msi_irq {
	struct qcom_msi_client *client;
	struct qcom_msi_grp *grp;
	unsigned int grp_index;
	unsigned int hwirq;
	unsigned int virq;
	u32 pos;
};

/**
 * struct qcom_msi_grp - MSI group information
 * @int_en_reg:		memory-mapped interrupt enable register address
 * @int_mask_reg:	memory-mapped interrupt mask register address
 * @int_status_reg:	memory-mapped interrupt status register address
 * @mask:		tracks masked/unmasked MSI
 * @irqs:		structure to MSI IRQ information
 */
struct qcom_msi_grp {
	void __iomem *int_en_reg;
	void __iomem *int_mask_reg;
	void __iomem *int_status_reg;
	u32 mask;
	struct qcom_msi_irq irqs[MSI_IRQ_PER_GRP];
};

/**
 * struct qcom_msi - PCIe controller based MSI controller information
 * @clients:		list for tracking clients
 * @dev:		platform device node
 * @nr_hwirqs:		total number of hardware IRQs
 * @nr_virqs:		total number of virqs
 * @nr_grps:		total number of groups
 * @grps:		pointer to all groups information
 * @bitmap:		tracks used/unused MSI
 * @mutex:		for modifying MSI client list and bitmap
 * @inner_domain:	parent domain; gen irq related
 * @msi_domain:		child domain; pcie related
 * @msi_db_addr:	MSI doorbell address
 * @cfg_lock:		lock for configuring MSI controller registers
 * @pcie_msi_cfg:	memory-mapped MSI controller register space
 */
struct qcom_msi {
	struct list_head clients;
	struct device *dev;
	int nr_hwirqs;
	int nr_virqs;
	int nr_grps;
	struct qcom_msi_grp *grps;
	unsigned long *bitmap;
	struct mutex mutex;
	struct irq_domain *inner_domain;
	struct irq_domain *msi_domain;
	phys_addr_t msi_db_addr;
	spinlock_t cfg_lock;
	void __iomem *pcie_msi_cfg;
};

/**
 * struct qcom_msi_client - structure for each client of MSI controller
 * @node:		list to track number of MSI clients
 * @msi:		client specific MSI controller based resource pointer
 * @dev:		client's dev of pci_dev
 * @nr_irqs:		number of irqs allocated for client
 * @msi_addr:		MSI doorbell address
 */
struct qcom_msi_client {
	struct list_head node;
	struct qcom_msi *msi;
	struct device *dev;
	unsigned int nr_irqs;
	phys_addr_t msi_addr;
};

/**
 * struct qcom_pcie - PCIe root port information
 * @dev:	pointer to the device node
 * @parf:	parf address in memory map
 * @dbi_base:	dbi address in memory map
 * @global_irq	multiplexed irq for error interrupts
 * @safety_lock	spinlock for exclusion btw safety errors
 * pcie_fault	Count of each error type
 * pcie_fault_total	Total count of all errors occured on a RP
 * @msi: pointer to the qcom_msi structure
 */
struct qcom_pcie {
	struct device *dev;
	void __iomem *parf;
	void __iomem *dbi_base;
	unsigned int global_irq;
	spinlock_t safety_lock;
	u32 pcie_fault[MAX_PCIE_SAFETY_FAULT];
	u32 pcie_fault_total;
	struct qcom_msi *msi;
};

static struct qcom_pcie *qcom_pcie_lookup(struct device *dev)
{
	return pci_host_bridge_priv(dev_get_drvdata(dev));
}

static void qcom_msi_handler(struct irq_desc *desc)
{
	struct irq_chip *chip = irq_desc_get_chip(desc);
	struct qcom_msi_grp *msi_grp;
	u32 status;
	int i;

	chained_irq_enter(chip, desc);

	msi_grp = irq_desc_get_handler_data(desc);
	status = readl_relaxed(msi_grp->int_status_reg);
	status ^= (msi_grp->mask & status);
	writel(status, msi_grp->int_status_reg);

	for (i = 0; status; i++, status >>= 1)
		if (status & 0x1)
			generic_handle_irq(msi_grp->irqs[i].virq);

	chained_irq_exit(chip, desc);
}

static void qcom_msi_mask_irq(struct irq_data *data)
{
	struct irq_data *parent_data;
	struct qcom_msi_irq *msi_irq;
	struct qcom_msi_grp *msi_grp;
	struct qcom_msi *msi;
	unsigned long flags;

	parent_data = data->parent_data;
	if (!parent_data)
		return;

	msi_irq = irq_data_get_irq_chip_data(parent_data);
	msi = msi_irq->client->msi;
	msi_grp = msi_irq->grp;

	spin_lock_irqsave(&msi->cfg_lock, flags);
	pci_msi_mask_irq(data);
	msi_grp->mask |= BIT(msi_irq->grp_index);
	writel(msi_grp->mask, msi_grp->int_mask_reg);
	spin_unlock_irqrestore(&msi->cfg_lock, flags);
}

static void qcom_msi_unmask_irq(struct irq_data *data)
{
	struct irq_data *parent_data;
	struct qcom_msi_irq *msi_irq;
	struct qcom_msi_grp *msi_grp;
	struct qcom_msi *msi;
	unsigned long flags;

	parent_data = data->parent_data;
	if (!parent_data)
		return;

	msi_irq = irq_data_get_irq_chip_data(parent_data);
	msi = msi_irq->client->msi;
	msi_grp = msi_irq->grp;

	spin_lock_irqsave(&msi->cfg_lock, flags);
	msi_grp->mask &= ~BIT(msi_irq->grp_index);
	writel(msi_grp->mask, msi_grp->int_mask_reg);
	pci_msi_unmask_irq(data);
	spin_unlock_irqrestore(&msi->cfg_lock, flags);
}

static struct irq_chip qcom_msi_irq_chip = {
	.name		= "qcom_pci_msi",
	.irq_enable	= qcom_msi_unmask_irq,
	.irq_disable	= qcom_msi_mask_irq,
	.irq_mask	= qcom_msi_mask_irq,
	.irq_unmask	= qcom_msi_unmask_irq,
};

static int qcom_msi_domain_prepare(struct irq_domain *domain, struct device *dev,
				int nvec, msi_alloc_info_t *arg)
{
	struct qcom_msi *msi = domain->parent->host_data;
	struct qcom_msi_client *client;

	client = kzalloc(sizeof(*client), GFP_KERNEL);
	if (!client)
		return -ENOMEM;

	client->msi = msi;
	client->dev = dev;
	client->msi_addr = msi->msi_db_addr;
	mutex_lock(&msi->mutex);
	list_add_tail(&client->node, &msi->clients);
	mutex_unlock(&msi->mutex);

	/* zero out struct for pcie msi framework */
	memset(arg, 0, sizeof(*arg));
	return 0;
}

static struct msi_domain_ops qcom_msi_domain_ops = {
	.msi_prepare	= qcom_msi_domain_prepare,
};

static struct msi_domain_info qcom_msi_domain_info = {
	.flags	= (MSI_FLAG_USE_DEF_DOM_OPS | MSI_FLAG_USE_DEF_CHIP_OPS |
			MSI_FLAG_MULTI_PCI_MSI | MSI_FLAG_PCI_MSIX),
	.ops	= &qcom_msi_domain_ops,
	.chip	= &qcom_msi_irq_chip,
};

static int qcom_msi_irq_set_affinity(struct irq_data *data,
				const struct cpumask *mask, bool force)
{
	struct irq_data *parent_data = irq_get_irq_data(irqd_to_hwirq(data));
	int ret = 0;

	if (!parent_data)
		return -ENODEV;

	/* set affinity for MSI HW IRQ */
	if (parent_data->chip->irq_set_affinity)
		ret = parent_data->chip->irq_set_affinity(parent_data, mask, force);

	return ret;
}

static void qcom_msi_irq_compose_msi_msg(struct irq_data *data, struct msi_msg *msg)
{
	struct irq_data *parent_data = irq_get_irq_data(irqd_to_hwirq(data));
	struct qcom_msi_irq *msi_irq = irq_data_get_irq_chip_data(data);
	struct qcom_msi_client *client = msi_irq->client;

	if (!parent_data)
		return;

	msg->address_lo = lower_32_bits(client->msi_addr);
	msg->address_hi = upper_32_bits(client->msi_addr);
	msg->data = msi_irq->pos;
}

static struct irq_chip qcom_msi_bottom_irq_chip = {
	.name			= "qcom_msi",
	.irq_set_affinity	= qcom_msi_irq_set_affinity,
	.irq_compose_msi_msg	= qcom_msi_irq_compose_msi_msg,
};

static int qcom_msi_irq_domain_alloc(struct irq_domain *domain, unsigned int virq,
				unsigned int nr_irqs, void *args)
{
	struct device *dev = ((msi_alloc_info_t *)args)->desc->dev;
	struct qcom_msi_client *tmp, *client = NULL;
	struct qcom_msi *msi = domain->host_data;
	int i, ret = 0;
	int pos;

	mutex_lock(&msi->mutex);
	list_for_each_entry(tmp, &msi->clients, node) {
		if (tmp->dev == dev) {
			client = tmp;
			break;
		}
	}

	if (!client) {
		dev_err(msi->dev, "failed to find MSI client dev\n");
		ret = -ENODEV;
		goto out;
	}

	pos = bitmap_find_next_zero_area(msi->bitmap, msi->nr_virqs, 0,
					nr_irqs, nr_irqs - 1);
	if (pos > msi->nr_virqs) {
		ret = -ENOSPC;
		goto out;
	}

	bitmap_set(msi->bitmap, pos, nr_irqs);
	for (i = 0; i < nr_irqs; i++) {
		u32 grp = pos / MSI_IRQ_PER_GRP;
		u32 index = pos % MSI_IRQ_PER_GRP;
		struct qcom_msi_irq *msi_irq = &msi->grps[grp].irqs[index];

		msi_irq->virq = virq + i;
		msi_irq->client = client;
		irq_domain_set_info(domain, msi_irq->virq,
				msi_irq->hwirq,
				&qcom_msi_bottom_irq_chip, msi_irq,
				handle_simple_irq, NULL, NULL);
		client->nr_irqs++;
		pos++;
	}
out:
	mutex_unlock(&msi->mutex);
	return ret;
}

static void qcom_msi_irq_domain_free(struct irq_domain *domain, unsigned int virq,
				unsigned int nr_irqs)
{
	struct irq_data *data = irq_domain_get_irq_data(domain, virq);
	struct qcom_msi_client *client;
	struct qcom_msi_irq *msi_irq;
	struct qcom_msi *msi;

	if (!data)
		return;

	msi_irq = irq_data_get_irq_chip_data(data);
	client  = msi_irq->client;
	msi = client->msi;

	mutex_lock(&msi->mutex);
	bitmap_clear(msi->bitmap, msi_irq->pos, nr_irqs);

	client->nr_irqs -= nr_irqs;
	if (!client->nr_irqs) {
		list_del(&client->node);
		kfree(client);
	}
	mutex_unlock(&msi->mutex);

	irq_domain_free_irqs_parent(domain, virq, nr_irqs);
}

static const struct irq_domain_ops msi_domain_ops = {
	.alloc	= qcom_msi_irq_domain_alloc,
	.free	= qcom_msi_irq_domain_free,
};

static int qcom_msi_alloc_domains(struct qcom_msi *msi)
{
	msi->inner_domain = irq_domain_add_linear(NULL, msi->nr_virqs,
						&msi_domain_ops, msi);
	if (!msi->inner_domain) {
		dev_err(msi->dev, "failed to create IRQ inner domain\n");
		return -ENOMEM;
	}

	msi->msi_domain = pci_msi_create_irq_domain(of_node_to_fwnode(msi->dev->of_node),
					&qcom_msi_domain_info, msi->inner_domain);
	if (!msi->msi_domain) {
		dev_err(msi->dev, "failed to create MSI domain\n");
		irq_domain_remove(msi->inner_domain);
		return -ENOMEM;
	}

	return 0;
}

static void qcom_msi_irq_free(struct qcom_msi *msi, int n)
{
	while (--n >= 0) {
		unsigned int irq = msi->grps[n].irqs[0].hwirq;

		irq_set_chained_handler_and_data(irq, NULL, NULL);
		irq_dispose_mapping(irq);
	}
}

static int qcom_msi_irq_setup(struct qcom_msi *msi)
{
	struct qcom_msi_grp *msi_grp;
	struct qcom_msi_irq *msi_irq;
	int i, index;
	unsigned int irq;

	/* setup each MSI group. nr_hwirqs == nr_grps */
	for (i = 0; i < msi->nr_hwirqs; i++) {
		irq = irq_of_parse_and_map(msi->dev->of_node, i);
		if (!irq) {
			dev_err(msi->dev, "MSI: failed to parse/map interrupt\n");
			qcom_msi_irq_free(msi, i);
			return -ENODEV;
		}

		msi_grp = &msi->grps[i];
		msi_grp->int_en_reg = msi->pcie_msi_cfg +
				PCIE_MSI_CTRL_INT_N_EN_OFFS(i);
		msi_grp->int_mask_reg = msi->pcie_msi_cfg +
				PCIE_MSI_CTRL_INT_N_MASK_OFFS(i);
		msi_grp->int_status_reg = msi->pcie_msi_cfg +
				PCIE_MSI_CTRL_INT_N_STATUS_OFFS(i);

		for (index = 0; index < MSI_IRQ_PER_GRP; index++) {
			msi_irq = &msi_grp->irqs[index];

			msi_irq->grp = msi_grp;
			msi_irq->grp_index = index;
			msi_irq->pos = (i * MSI_IRQ_PER_GRP) + index;
			msi_irq->hwirq = irq;
		}

		irq_set_chained_handler_and_data(irq, qcom_msi_handler, msi_grp);
	}

	return 0;
}

static void qcom_msi_config(struct irq_domain *domain)
{
	struct qcom_msi *msi;
	int i;

	msi = domain->parent->host_data;

	/* program termination address */
	writel(msi->msi_db_addr, msi->pcie_msi_cfg + PCIE_MSI_CTRL_ADDR_OFFS);
	writel(0, msi->pcie_msi_cfg + PCIE_MSI_CTRL_UPPER_ADDR_OFFS);

	/* restore mask and enable all interrupts for each group */
	for (i = 0; i < msi->nr_grps; i++) {
		struct qcom_msi_grp *msi_grp = &msi->grps[i];

		writel(msi_grp->mask, msi_grp->int_mask_reg);
		writel(~0, msi_grp->int_en_reg);
	}
}

static void qcom_msi_deinit(struct qcom_msi *msi)
{
	qcom_msi_irq_free(msi, msi->nr_hwirqs);
	irq_domain_remove(msi->msi_domain);
	irq_domain_remove(msi->inner_domain);
}

static struct qcom_msi *qcom_msi_init(struct qcom_pcie *pcie)
{
	struct qcom_msi *msi;
	struct of_phandle_args irq;
	struct device *dev = pcie->dev;
	int ret;
	int nr = 0;

	msi = devm_kzalloc(dev, sizeof(*msi), GFP_KERNEL);
	if (!msi)
		return ERR_PTR(-ENOMEM);

	msi->dev = dev;
	mutex_init(&msi->mutex);
	spin_lock_init(&msi->cfg_lock);
	INIT_LIST_HEAD(&msi->clients);

	msi->msi_db_addr = MSI_DB_ADDR;
	while (of_irq_parse_one(dev->of_node, nr, &irq) == 0)
		nr++;
	msi->nr_hwirqs = nr - 1;
	if (!msi->nr_hwirqs) {
		dev_err(msi->dev, "no hwirqs found\n");
		return ERR_PTR(-ENODEV);
	}

	dev_dbg(msi->dev, "hwirq:%d pcie_msi_cfg:%p\n", msi->nr_hwirqs, pcie->dbi_base);
	msi->pcie_msi_cfg = pcie->dbi_base + PCIE_MSI_CTRL_BASE;

	msi->nr_virqs = msi->nr_hwirqs * MSI_IRQ_PER_GRP;
	msi->nr_grps = msi->nr_hwirqs;
	msi->grps = devm_kcalloc(dev, msi->nr_grps, sizeof(*msi->grps), GFP_KERNEL);
	if (!msi->grps)
		return ERR_PTR(-ENOMEM);

	msi->bitmap = devm_kcalloc(dev, BITS_TO_LONGS(msi->nr_virqs),
				sizeof(*msi->bitmap), GFP_KERNEL);
	if (!msi->bitmap)
		return ERR_PTR(-ENOMEM);

	ret = qcom_msi_alloc_domains(msi);
	if (ret)
		return ERR_PTR(ret);

	ret = qcom_msi_irq_setup(msi);
	if (ret) {
		qcom_msi_deinit(msi);
		return ERR_PTR(ret);
	}

	qcom_msi_config(msi->msi_domain);
	return msi;
}

static void qcom_pcie_enable_error_reporting(struct qcom_pcie *pcie)
{
	u32 val, mask;

	/* Clear all the interrupts before we enable it */
	writel(0, pcie->dbi_base + DBI_SAFETY_STATUS);
	writel(0, pcie->dbi_base + DBI_INTERFACE_TIMER_STATUS);

	/* Enable interrupts which are aggregated using GLOBAL_INT */
	writel(CFGPCIE_INT_ALL_STATUS_MASK, pcie->parf + PARF_INT_ALL_CLEAR);
	writel(CFGPCIE_PARF_INT_STATUS_MASK, pcie->parf + PARF_INT_CLEAR);
	writel(CFGPCIE_INT_ALL_2_STATUS_MASK, pcie->parf + PARF_INT_ALL_2_CLEAR);
	writel(CFGPCIE_INT_ALL_3_STATUS_MASK, pcie->parf + PARF_INT_ALL_3_CLEAR);
	writel(CFGPCIE_INT_ALL_4_STATUS_MASK, pcie->parf + PARF_INT_ALL_4_CLEAR);
	writel(CFGPCIE_INT_ALL_5_STATUS_MASK, pcie->parf + PARF_INT_ALL_5_CLEAR);

	val = readl(pcie->dbi_base + PCIE_MISC_CONTROL_1_OFF);
	val |= PCIE_DBI_RO_WR_EN;
	writel(val, pcie->dbi_base + PCIE_MISC_CONTROL_1_OFF);

	val = readl(pcie->dbi_base + PCI_INTERRUPT_LINE);
	val |= SERR_EN;
	writel(val, pcie->dbi_base + PCI_INTERRUPT_LINE);

	val = readl(pcie->dbi_base + DBI_DEVICE_CONTROL_DEVICE_STATUS);
	val |= (PCIE_CAP_CORR_ERR_REPORT_EN | PCIE_CAP_NON_FATAL_ERR_REPORT_EN |
			PCIE_CAP_FATAL_ERR_REPORT_EN | PCIE_CAP_UNSUPPORT_REQ_REP_EN);
	writel(val, pcie->dbi_base + DBI_DEVICE_CONTROL_DEVICE_STATUS);

	val = readl(pcie->dbi_base + DBI_ROOT_CONTROL_ROOT_CAPABILITIES_REG);
	val |= (PCIE_CAP_SYS_ERR_ON_CORR_ERR_EN | PCIE_CAP_SYS_ERR_ON_NON_FATAL_ERR_EN |
			PCIE_CAP_SYS_ERR_ON_FATAL_ERR_EN);
	writel(val, pcie->dbi_base + DBI_ROOT_CONTROL_ROOT_CAPABILITIES_REG);

	val = readl(pcie->dbi_base + PCIE_MISC_CONTROL_1_OFF);
	val &= ~PCIE_DBI_RO_WR_EN;
	writel(val, pcie->dbi_base + PCIE_MISC_CONTROL_1_OFF);

	/* Enable RAS-DP Interrupts */
	val = readl(pcie->dbi_base + PCIE_RASDP_ERROR_MODE_EN_REG);
	val |= RASDP_ERROR_MODE_EN;
	writel(val, pcie->dbi_base + PCIE_RASDP_ERROR_MODE_EN_REG);

	/* Enable CDM Check */
	val = readl(pcie->dbi_base + PCIE_PL_CHK_REG_CONTROL_STATUS);
	/* Enable continuous CMD register check mode */
	val |= PCIE_PL_CHK_REG_CHK_REG_CONTINUOUS;
	/* Start the CDM register check */
	val |= PCIE_PL_CHK_REG_CHK_REG_START;
	/* Enable comparison CDM register check mode */
	val |= PCIE_PL_CHK_REG_CHK_REG_COMPARISON_ERROR;
	/* Enable logic CDM register check mode */
	val |= PCIE_PL_CHK_REG_CHK_REG_LOGIC_ERROR;
	writel(val, pcie->dbi_base + PCIE_PL_CHK_REG_CONTROL_STATUS);

	/* Interface Timer Enable */
	val = readl(pcie->dbi_base + PCIE_INTERFACE_TIMER_CONTROL);
	val |=  (INTERFACE_TIMER_EN | INTERFACE_TIMER_AER_EN);
	writel(val, pcie->dbi_base + PCIE_INTERFACE_TIMER_CONTROL);

	/* Enable safety correctable and uncorrectable error reporting */
	val = readl(pcie->parf + PARF_CFG_SAFETY_INT_MASK_CTRL);
	val |= (CFG_SAFETY_UNCORR_INT_MASK | CFG_SAFETY_CORR_INT_MASK);
	writel(val, pcie->parf + PARF_CFG_SAFETY_INT_MASK_CTRL);

	/* Enable CRC check and generation */
	val = readl(pcie->dbi_base + DBI_ADV_ERR_CAP_CTRL_OFF);
	val |= (ECRC_GEN_EN | ECRC_CHECK_EN);
	writel(val, pcie->dbi_base + DBI_ADV_ERR_CAP_CTRL_OFF);

	/* Enable AER */
	val = readl(pcie->dbi_base + DBI_ROOT_ERR_CMD_OFF);
	val |= (CORR_ERR_REPORTING_EN | NON_FATAL_ERR_REPORTING_EN
		| FATAL_ERR_REPORTING_EN);
	writel(val, pcie->dbi_base + DBI_ROOT_ERR_CMD_OFF);

	/* Enable interrupts */
	val = readl(pcie->dbi_base + DBI_SAFETY_MASK_OFF);
	val &= ~(SAFETY_INT_MASK);
	writel(val, pcie->dbi_base + DBI_SAFETY_MASK_OFF);

	/* Disable Legacy Interrupts */
	val = readl(pcie->parf + PARF_INT_ALL_MASK);
	/* Mask the first 22 bits */
	mask = ~0 << 22;
	val = val & mask;
	writel(val, pcie->parf + PARF_INT_ALL_MASK);
}

static void qcom_pcie_check_spurious_int(struct qcom_pcie *pcie)
{
	u32 *pcie_fault = pcie->pcie_fault;
	struct device *dev = pcie->dev;
	struct kobject *kobj_ref = &dev->kobj;
	u32 val;

	val = readl(pcie->dbi_base + DBI_INTERFACE_TIMER_STATUS);
	if (val & CFGPCIE_INTERFACE_TIMER_STATUS_MASK)
		return;

	val = readl(pcie->parf + PARF_INT_ALL_STATUS);
	if (val & CFGPCIE_INT_ALL_STATUS_MASK)
		return;

	val = readl(pcie->parf + PARF_INT_STATUS);
	if (val & CFGPCIE_PARF_INT_STATUS_MASK)
		return;

	val = readl(pcie->parf + PARF_INT_ALL_2_STATUS);
	if (val & CFGPCIE_INT_ALL_2_STATUS_MASK)
		return;

	val = readl(pcie->parf + PARF_INT_ALL_3_STATUS);
	if (val & CFGPCIE_INT_ALL_3_STATUS_MASK)
		return;

	val = readl(pcie->parf + PARF_INT_ALL_4_STATUS);
	if (val & CFGPCIE_INT_ALL_4_STATUS_MASK)
		return;

	val = readl(pcie->parf + PARF_INT_ALL_5_STATUS);
	if (val & CFGPCIE_INT_ALL_5_STATUS_MASK)
		return;

	dev_err(pcie->dev, "PCIe Spurious Interrupt");
	pcie_fault[PCIE_SPURIOUS_INT]++;
	pcie->pcie_fault_total++;
	sysfs_notify(kobj_ref, NULL, "qcom_pcie_error_report");
}

static irqreturn_t qcom_pcie_global_irq_thread(int irq, void *data)
{
	struct qcom_pcie *pcie = data;
	u32 *pcie_fault = pcie->pcie_fault;
	struct device *dev = pcie->dev;
	struct kobject *kobj_ref = &dev->kobj;
	unsigned long irqsave_flags;
	u32 val, int_status;

	spin_lock_irqsave(&pcie->safety_lock, irqsave_flags);

	int_status = readl(pcie->dbi_base + DBI_SAFETY_STATUS);
	writel(0, pcie->dbi_base + DBI_SAFETY_STATUS);

	if (int_status) {
		dev_err(pcie->dev, "global interrupt fired status: %u", int_status);

		if (int_status & PCIE_RASDP_UNCORR_ERR) {
			dev_err(pcie->dev, "RASDP uncorrectable error triggered");
			pcie_fault[RASDP_UNCORR_ERROR]++;
			pcie->pcie_fault_total++;
			sysfs_notify(kobj_ref, NULL, "qcom_pcie_error_report");

			/*
			 * rasdp_uncorr_err ends up triggering a
			 * pcie_uncorr error continuously. So masking
			 * pcie_uncorr interrupts .
			 */
			val = readl(pcie->dbi_base + DBI_SAFETY_MASK_OFF);
			val |= PCIE_AER_UNCORR_ERR;
			writel(val, pcie->dbi_base + DBI_SAFETY_MASK_OFF);
		}

		if (int_status & PCIE_CDM_CHK_ERR) {
			dev_err(pcie->dev, "CDM error triggered");
			val = readl(pcie->dbi_base + PCIE_PL_CHK_REG_CONTROL_STATUS);

			if (val & PCIE_PL_CHK_REG_CHK_REG_COMPARISON_ERROR) {
				pcie_fault[CDM_REG_CHK_ERROR]++;
				pcie->pcie_fault_total++;
				sysfs_notify(kobj_ref, NULL, "qcom_pcie_error_report");

				/*
				 * cdm_chk_err injection results in a continuous
				 * interrupt storm on certain targets, so masking it.
				 */
				val = readl(pcie->dbi_base + DBI_SAFETY_MASK_OFF);
				val |= (PCIE_CDM_CHK_ERR | PCIE_AER_UNCORR_ERR);
				writel(val, pcie->dbi_base + DBI_SAFETY_MASK_OFF);
			}
		}

		if (int_status & PCIE_IFACE_TMR_ERR) {
			dev_err(pcie->dev, "Iface Timeout error triggered");
			pcie_fault[INTERFACE_TIMER_ERROR]++;
			pcie->pcie_fault_total++;
			sysfs_notify(kobj_ref, NULL, "qcom_pcie_error_report");

			/*
			 * interface_timer_err injection results in a continuous
			 * interrupt storm on certain targets, so masking it.
			 */
			val = readl(pcie->dbi_base + DBI_SAFETY_MASK_OFF);
			val |= (PCIE_IFACE_TMR_ERR | PCIE_AER_UNCORR_ERR);
			writel(val, pcie->dbi_base + DBI_SAFETY_MASK_OFF);
		}

		if (int_status & PCIE_RASDP_CORR_ERR) {
			dev_err(pcie->dev, "RASDP correctable error triggered");
			pcie_fault[RASDP_CORR_ERROR]++;
			pcie->pcie_fault_total++;
			sysfs_notify(kobj_ref, NULL, "qcom_pcie_error_report");
		}
	} else {
		qcom_pcie_check_spurious_int(pcie);
	}

	spin_unlock_irqrestore(&pcie->safety_lock, irqsave_flags);
	return IRQ_HANDLED;
}

static ssize_t qcom_pcie_error_report_show(struct device *dev,
		struct device_attribute *attr,
		char *buf)
{
	unsigned int i;
	struct qcom_pcie *pcie = qcom_pcie_lookup(dev);
	u32 *pcie_fault = pcie->pcie_fault;
	size_t len = 0;

	for (i = 0; i < MAX_PCIE_SAFETY_FAULT; i++) {
		if (pcie_fault_string[i])
			len += sysfs_emit_at(buf, len, "%s: %u\n",
					pcie_fault_string[i],
					pcie_fault[i]);
	}

	len += sysfs_emit_at(buf, len, "%s: %u\n",
					pcie_fault_string[i],
					pcie->pcie_fault_total);

	return len;
}
static DEVICE_ATTR_RO(qcom_pcie_error_report);

static struct attribute *qcom_pcie_attrs[] = {
	&dev_attr_qcom_pcie_error_report.attr,
	NULL,
};

static const struct attribute_group qcom_pcie_attribute_group = {
	.attrs = qcom_pcie_attrs,
	.name = "qcom_pcie"
};

static int qcom_pcie_ecam_suspend_noirq(struct device *dev)
{
	if(atomic_read(&dev->power.usage_count) > 1)
		pm_runtime_put_noidle(dev);

	return pm_runtime_put_sync(dev);
}

static int qcom_pcie_ecam_resume_noirq(struct device *dev)
{
	struct qcom_msi *msi;
	struct qcom_pcie *pcie = qcom_pcie_lookup(dev);

	msi = pcie->msi;
	qcom_msi_config(msi->msi_domain);
	pm_runtime_get_noresume(dev);

	return pm_runtime_get_sync(dev);
}

static const struct pci_ecam_ops qcom_pcie_ecam_ops = {
	.pci_ops	= {
		.map_bus	= pci_ecam_map_bus,
		.read		= pci_generic_config_read,
		.write		= pci_generic_config_write,
	}
};

static int qcom_pcie_ecam_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct qcom_msi *msi;
	int ret = 0;
	u64 addr;
	struct pci_host_bridge *bridge;
	struct qcom_pcie *pcie;

	bridge = devm_pci_alloc_host_bridge(dev, sizeof(*pcie));
	if (!bridge)
		return -ENOMEM;

	pcie = pci_host_bridge_priv(bridge);
	pcie->dev = dev;

	pcie->parf = devm_platform_ioremap_resource_byname(pdev, "parf");
	if (IS_ERR(pcie->parf)) {
		ret = PTR_ERR(pcie->parf);
		dev_err(dev, "Failed to get parf address\n");
		return ret;
	}

	if (of_property_read_reg(dev->of_node, 0, &addr, NULL) < 0) {
		dev_err(dev, "Failed to get dbi address\n");
		return -ENODEV;
	}

	pcie->dbi_base = devm_ioremap(dev, addr, PCI_DBI_SIZE);;
	if (!pcie->dbi_base) {
		dev_err(dev, "Failed to map dbi address\n");
		return -ENOMEM;
	}

	pcie->global_irq = platform_get_irq_byname(pdev, "global_int");
	if (pcie->global_irq < 0) {
		ret = pcie->global_irq;
		dev_err(dev, "Failed to get Global IRQ\n");
	}

	ret = devm_pm_runtime_enable(dev);
	if (ret)
		return ret;
	ret = pm_runtime_resume_and_get(dev);
	if (ret < 0) {
		dev_err(dev, "fail to enable pcie controller: %d\n", ret);
		return ret;
	}

	writel_relaxed(ADDR_LOW(addr), pcie->parf + DBI_BASE_ADDR);
	writel_relaxed(ADDR_HIGH(addr), pcie->parf + DBI_BASE_ADDR_HIGH);

	msi = qcom_msi_init(pcie);
	if (IS_ERR(msi)) {
		ret = PTR_ERR(msi);
		goto out_put_pm;
	}

	ret = pci_host_common_init(pdev, bridge, &qcom_pcie_ecam_ops);
	if (ret) {
		dev_err(dev, "pci_host_common_init() failed:%d\n", ret);
		goto out_msi_deinit;
	}

	pcie->msi = msi;

	ret = sysfs_create_group(&pdev->dev.kobj, &qcom_pcie_attribute_group);
	if (ret)
		goto out_pci_remove;

	qcom_pcie_enable_error_reporting(pcie);

	ret = devm_request_threaded_irq(dev, pcie->global_irq, NULL,
					qcom_pcie_global_irq_thread,
					IRQF_ONESHOT,
					"global_irq", pcie);
	if (ret) {
		dev_err(dev, "Failed to request Global IRQ\n");
		goto out_sysfs_remove;
	}

	return 0;

out_sysfs_remove:
	sysfs_remove_group(&pdev->dev.kobj, &qcom_pcie_attribute_group);
out_pci_remove:
	pci_host_common_remove(pdev);
out_msi_deinit:
	qcom_msi_deinit(msi);
out_put_pm:
	pm_runtime_put_sync(dev);

	return ret;
}

static void qcom_pcie_ecam_remove(struct platform_device *pdev)
{
	struct qcom_pcie *pcie = qcom_pcie_lookup(&pdev->dev);

	sysfs_remove_group(&pdev->dev.kobj, &qcom_pcie_attribute_group);
	pci_host_common_remove(pdev);
	qcom_msi_deinit(pcie->msi);
	pm_runtime_put_sync(&pdev->dev);
}

static const struct dev_pm_ops qcom_pcie_ecam_pm_ops = {
	SET_SYSTEM_SLEEP_PM_OPS(qcom_pcie_ecam_suspend_noirq,
				qcom_pcie_ecam_resume_noirq)
};

static const struct of_device_id qcom_pcie_ecam_of_match[] = {
	{ .compatible	= "qcom,pcie-ecam-rc" },
	{ },
};
MODULE_DEVICE_TABLE(of, qcom_pcie_ecam_of_match);

static struct platform_driver qcom_pcie_ecam_driver = {
	.probe	= qcom_pcie_ecam_probe,
	.remove = qcom_pcie_ecam_remove,
	.driver	= {
		.name			= "qcom-pcie-ecam-rc",
		.suppress_bind_attrs	= true,
		.of_match_table		= qcom_pcie_ecam_of_match,
		.probe_type		= PROBE_PREFER_ASYNCHRONOUS,
		.pm			= &qcom_pcie_ecam_pm_ops,
	},
};
module_platform_driver(qcom_pcie_ecam_driver);

MODULE_DESCRIPTION("Qualcomm PCIe ECAM root complex driver");
MODULE_LICENSE("GPL");
