package generate

import (
	"testing"

	"github.com/containers/podman/v3/pkg/domain/entities"
	"github.com/stretchr/testify/assert"
)

func TestValidateRestartPolicyContainer(t *testing.T) {
	type containerInfo struct {
		restart string
	}
	tests := []struct {
		name          string
		containerInfo containerInfo
		wantErr       bool
	}{
		{"good-on", containerInfo{restart: "no"}, false},
		{"good-on-success", containerInfo{restart: "on-success"}, false},
		{"good-on-failure", containerInfo{restart: "on-failure"}, false},
		{"good-on-abnormal", containerInfo{restart: "on-abnormal"}, false},
		{"good-on-watchdog", containerInfo{restart: "on-watchdog"}, false},
		{"good-on-abort", containerInfo{restart: "on-abort"}, false},
		{"good-always", containerInfo{restart: "always"}, false},
		{"fail", containerInfo{restart: "foobar"}, true},
		{"failblank", containerInfo{restart: ""}, true},
	}
	for _, tt := range tests {
		test := tt
		t.Run(tt.name, func(t *testing.T) {
			if err := validateRestartPolicy(test.containerInfo.restart); (err != nil) != test.wantErr {
				t.Errorf("ValidateRestartPolicy() error = %v, wantErr %v", err, test.wantErr)
			}
		})
	}
}

func TestCreateContainerSystemdUnit(t *testing.T) {
	serviceInfo := `# container-639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401.service
`
	headerInfo := `# autogenerated by Podman CI
`
	goodIDContent := `
[Unit]
Description=Podman container-639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401.service
Documentation=man:podman-generate-systemd(1)
Wants=network.target
After=network-online.target

[Service]
Environment=PODMAN_SYSTEMD_UNIT=%n
Restart=always
TimeoutStopSec=82
ExecStart=/usr/bin/podman start 639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401
ExecStop=/usr/bin/podman stop -t 22 639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401
ExecStopPost=/usr/bin/podman stop -t 22 639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401
PIDFile=/run/containers/storage/overlay-containers/639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401/userdata/conmon.pid
Type=forking

[Install]
WantedBy=multi-user.target default.target
`
	goodID := serviceInfo + headerInfo + goodIDContent
	goodIDNoHeaderInfo := serviceInfo + goodIDContent

	goodName := `# container-foobar.service
# autogenerated by Podman CI

[Unit]
Description=Podman container-foobar.service
Documentation=man:podman-generate-systemd(1)
Wants=network.target
After=network-online.target

[Service]
Environment=PODMAN_SYSTEMD_UNIT=%n
Restart=always
TimeoutStopSec=70
ExecStart=/usr/bin/podman start foobar
ExecStop=/usr/bin/podman stop -t 10 foobar
ExecStopPost=/usr/bin/podman stop -t 10 foobar
PIDFile=/run/containers/storage/overlay-containers/639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401/userdata/conmon.pid
Type=forking

[Install]
WantedBy=multi-user.target default.target
`

	goodNameBoundTo := `# container-foobar.service
# autogenerated by Podman CI

[Unit]
Description=Podman container-foobar.service
Documentation=man:podman-generate-systemd(1)
Wants=network.target
After=network-online.target
BindsTo=a.service b.service c.service pod.service
After=a.service b.service c.service pod.service

[Service]
Environment=PODMAN_SYSTEMD_UNIT=%n
Restart=always
TimeoutStopSec=70
ExecStart=/usr/bin/podman start foobar
ExecStop=/usr/bin/podman stop -t 10 foobar
ExecStopPost=/usr/bin/podman stop -t 10 foobar
PIDFile=/run/containers/storage/overlay-containers/639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401/userdata/conmon.pid
Type=forking

[Install]
WantedBy=multi-user.target default.target
`

	goodWithNameAndGeneric := `# jadda-jadda.service
# autogenerated by Podman CI

[Unit]
Description=Podman jadda-jadda.service
Documentation=man:podman-generate-systemd(1)
Wants=network.target
After=network-online.target

[Service]
Environment=PODMAN_SYSTEMD_UNIT=%n
Restart=always
TimeoutStopSec=70
ExecStartPre=/bin/rm -f %t/jadda-jadda.pid %t/jadda-jadda.ctr-id
ExecStart=/usr/bin/podman container run --conmon-pidfile %t/jadda-jadda.pid --cidfile %t/jadda-jadda.ctr-id --cgroups=no-conmon -d --replace --name jadda-jadda --hostname hello-world awesome-image:latest command arg1 ... argN "foo=arg \"with \" space"
ExecStop=/usr/bin/podman container stop --ignore --cidfile %t/jadda-jadda.ctr-id -t 10
ExecStopPost=/usr/bin/podman container rm --ignore -f --cidfile %t/jadda-jadda.ctr-id
PIDFile=%t/jadda-jadda.pid
Type=forking

[Install]
WantedBy=multi-user.target default.target
`

	goodWithExplicitShortDetachParam := `# jadda-jadda.service
# autogenerated by Podman CI

[Unit]
Description=Podman jadda-jadda.service
Documentation=man:podman-generate-systemd(1)
Wants=network.target
After=network-online.target

[Service]
Environment=PODMAN_SYSTEMD_UNIT=%n
Restart=always
TimeoutStopSec=70
ExecStartPre=/bin/rm -f %t/jadda-jadda.pid %t/jadda-jadda.ctr-id
ExecStart=/usr/bin/podman run --conmon-pidfile %t/jadda-jadda.pid --cidfile %t/jadda-jadda.ctr-id --cgroups=no-conmon --replace -d --name jadda-jadda --hostname hello-world awesome-image:latest command arg1 ... argN
ExecStop=/usr/bin/podman stop --ignore --cidfile %t/jadda-jadda.ctr-id -t 10
ExecStopPost=/usr/bin/podman rm --ignore -f --cidfile %t/jadda-jadda.ctr-id
PIDFile=%t/jadda-jadda.pid
Type=forking

[Install]
WantedBy=multi-user.target default.target
`

	goodNameNewWithPodFile := `# jadda-jadda.service
# autogenerated by Podman CI

[Unit]
Description=Podman jadda-jadda.service
Documentation=man:podman-generate-systemd(1)
Wants=network.target
After=network-online.target

[Service]
Environment=PODMAN_SYSTEMD_UNIT=%n
Restart=always
TimeoutStopSec=70
ExecStartPre=/bin/rm -f %t/jadda-jadda.pid %t/jadda-jadda.ctr-id
ExecStart=/usr/bin/podman run --conmon-pidfile %t/jadda-jadda.pid --cidfile %t/jadda-jadda.ctr-id --cgroups=no-conmon --pod-id-file %t/pod-foobar.pod-id-file --replace -d --name jadda-jadda --hostname hello-world awesome-image:latest command arg1 ... argN
ExecStop=/usr/bin/podman stop --ignore --cidfile %t/jadda-jadda.ctr-id -t 10
ExecStopPost=/usr/bin/podman rm --ignore -f --cidfile %t/jadda-jadda.ctr-id
PIDFile=%t/jadda-jadda.pid
Type=forking

[Install]
WantedBy=multi-user.target default.target
`

	goodNameNewDetach := `# jadda-jadda.service
# autogenerated by Podman CI

[Unit]
Description=Podman jadda-jadda.service
Documentation=man:podman-generate-systemd(1)
Wants=network.target
After=network-online.target

[Service]
Environment=PODMAN_SYSTEMD_UNIT=%n
Restart=always
TimeoutStopSec=70
ExecStartPre=/bin/rm -f %t/jadda-jadda.pid %t/jadda-jadda.ctr-id
ExecStart=/usr/bin/podman run --conmon-pidfile %t/jadda-jadda.pid --cidfile %t/jadda-jadda.ctr-id --cgroups=no-conmon --replace --detach --name jadda-jadda --hostname hello-world awesome-image:latest command arg1 ... argN
ExecStop=/usr/bin/podman stop --ignore --cidfile %t/jadda-jadda.ctr-id -t 10
ExecStopPost=/usr/bin/podman rm --ignore -f --cidfile %t/jadda-jadda.ctr-id
PIDFile=%t/jadda-jadda.pid
Type=forking

[Install]
WantedBy=multi-user.target default.target
`

	goodIDNew := `# container-639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401.service
# autogenerated by Podman CI

[Unit]
Description=Podman container-639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401.service
Documentation=man:podman-generate-systemd(1)
Wants=network.target
After=network-online.target

[Service]
Environment=PODMAN_SYSTEMD_UNIT=%n
Restart=always
TimeoutStopSec=70
ExecStartPre=/bin/rm -f %t/container-639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401.pid %t/container-639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401.ctr-id
ExecStart=/usr/bin/podman run --conmon-pidfile %t/container-639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401.pid --cidfile %t/container-639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401.ctr-id --cgroups=no-conmon -d awesome-image:latest
ExecStop=/usr/bin/podman stop --ignore --cidfile %t/container-639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401.ctr-id -t 10
ExecStopPost=/usr/bin/podman rm --ignore -f --cidfile %t/container-639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401.ctr-id
PIDFile=%t/container-639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401.pid
Type=forking

[Install]
WantedBy=multi-user.target default.target
`

	genGoodNewDetach := func(detachparam string) string {
		goodNewDetach := `# jadda-jadda.service
# autogenerated by Podman CI

[Unit]
Description=Podman jadda-jadda.service
Documentation=man:podman-generate-systemd(1)
Wants=network.target
After=network-online.target

[Service]
Environment=PODMAN_SYSTEMD_UNIT=%n
Restart=always
TimeoutStopSec=102
ExecStartPre=/bin/rm -f %t/jadda-jadda.pid %t/jadda-jadda.ctr-id
ExecStart=/usr/bin/podman run --conmon-pidfile %t/jadda-jadda.pid --cidfile %t/jadda-jadda.ctr-id --cgroups=no-conmon ` +
			detachparam +
			` awesome-image:latest
ExecStop=/usr/bin/podman stop --ignore --cidfile %t/jadda-jadda.ctr-id -t 42
ExecStopPost=/usr/bin/podman rm --ignore -f --cidfile %t/jadda-jadda.ctr-id
PIDFile=%t/jadda-jadda.pid
Type=forking

[Install]
WantedBy=multi-user.target default.target
`
		return goodNewDetach
	}

	goodNameNewDetachFalseWithCmd := `# jadda-jadda.service
# autogenerated by Podman CI

[Unit]
Description=Podman jadda-jadda.service
Documentation=man:podman-generate-systemd(1)
Wants=network.target
After=network-online.target

[Service]
Environment=PODMAN_SYSTEMD_UNIT=%n
Restart=always
TimeoutStopSec=102
ExecStartPre=/bin/rm -f %t/jadda-jadda.pid %t/jadda-jadda.ctr-id
ExecStart=/usr/bin/podman run --conmon-pidfile %t/jadda-jadda.pid --cidfile %t/jadda-jadda.ctr-id --cgroups=no-conmon -d --replace --name test -p 80:80 awesome-image:latest somecmd --detach=false
ExecStop=/usr/bin/podman stop --ignore --cidfile %t/jadda-jadda.ctr-id -t 42
ExecStopPost=/usr/bin/podman rm --ignore -f --cidfile %t/jadda-jadda.ctr-id
PIDFile=%t/jadda-jadda.pid
Type=forking

[Install]
WantedBy=multi-user.target default.target
`

	goodNewRootFlags := `# jadda-jadda.service
# autogenerated by Podman CI

[Unit]
Description=Podman jadda-jadda.service
Documentation=man:podman-generate-systemd(1)
Wants=network.target
After=network-online.target

[Service]
Environment=PODMAN_SYSTEMD_UNIT=%n
Restart=always
TimeoutStopSec=102
ExecStartPre=/bin/rm -f %t/jadda-jadda.pid %t/jadda-jadda.ctr-id
ExecStart=/usr/bin/podman --events-backend none --runroot /root run --conmon-pidfile %t/jadda-jadda.pid --cidfile %t/jadda-jadda.ctr-id --cgroups=no-conmon -d awesome-image:latest
ExecStop=/usr/bin/podman --events-backend none --runroot /root stop --ignore --cidfile %t/jadda-jadda.ctr-id -t 42
ExecStopPost=/usr/bin/podman --events-backend none --runroot /root rm --ignore -f --cidfile %t/jadda-jadda.ctr-id
PIDFile=%t/jadda-jadda.pid
Type=forking

[Install]
WantedBy=multi-user.target default.target
`

	goodContainerCreate := `# jadda-jadda.service
# autogenerated by Podman CI

[Unit]
Description=Podman jadda-jadda.service
Documentation=man:podman-generate-systemd(1)
Wants=network.target
After=network-online.target

[Service]
Environment=PODMAN_SYSTEMD_UNIT=%n
Restart=always
TimeoutStopSec=70
ExecStartPre=/bin/rm -f %t/jadda-jadda.pid %t/jadda-jadda.ctr-id
ExecStart=/usr/bin/podman container run --conmon-pidfile %t/jadda-jadda.pid --cidfile %t/jadda-jadda.ctr-id --cgroups=no-conmon -d awesome-image:latest
ExecStop=/usr/bin/podman container stop --ignore --cidfile %t/jadda-jadda.ctr-id -t 10
ExecStopPost=/usr/bin/podman container rm --ignore -f --cidfile %t/jadda-jadda.ctr-id
PIDFile=%t/jadda-jadda.pid
Type=forking

[Install]
WantedBy=multi-user.target default.target
`

	goodNewWithJournaldTag := `# jadda-jadda.service
# autogenerated by Podman CI

[Unit]
Description=Podman jadda-jadda.service
Documentation=man:podman-generate-systemd(1)
Wants=network.target
After=network-online.target

[Service]
Environment=PODMAN_SYSTEMD_UNIT=%n
Restart=always
TimeoutStopSec=70
ExecStartPre=/bin/rm -f %t/jadda-jadda.pid %t/jadda-jadda.ctr-id
ExecStart=/usr/bin/podman run --conmon-pidfile %t/jadda-jadda.pid --cidfile %t/jadda-jadda.ctr-id --cgroups=no-conmon -d --replace --name test --log-driver=journald --log-opt=tag={{.Name}} awesome-image:latest
ExecStop=/usr/bin/podman stop --ignore --cidfile %t/jadda-jadda.ctr-id -t 10
ExecStopPost=/usr/bin/podman rm --ignore -f --cidfile %t/jadda-jadda.ctr-id
PIDFile=%t/jadda-jadda.pid
Type=forking

[Install]
WantedBy=multi-user.target default.target
`

	goodNewWithSpecialChars := `# jadda-jadda.service
# autogenerated by Podman CI

[Unit]
Description=Podman jadda-jadda.service
Documentation=man:podman-generate-systemd(1)
Wants=network.target
After=network-online.target

[Service]
Environment=PODMAN_SYSTEMD_UNIT=%n
Restart=always
TimeoutStopSec=70
ExecStartPre=/bin/rm -f %t/jadda-jadda.pid %t/jadda-jadda.ctr-id
ExecStart=/usr/bin/podman run --conmon-pidfile %t/jadda-jadda.pid --cidfile %t/jadda-jadda.ctr-id --cgroups=no-conmon -d --replace --name test awesome-image:latest sh -c "kill $$$$ && echo %%\\"
ExecStop=/usr/bin/podman stop --ignore --cidfile %t/jadda-jadda.ctr-id -t 10
ExecStopPost=/usr/bin/podman rm --ignore -f --cidfile %t/jadda-jadda.ctr-id
PIDFile=%t/jadda-jadda.pid
Type=forking

[Install]
WantedBy=multi-user.target default.target
`
	tests := []struct {
		name     string
		info     containerInfo
		want     string
		new      bool
		noHeader bool
		wantErr  bool
	}{

		{"good with id",
			containerInfo{
				Executable:        "/usr/bin/podman",
				ServiceName:       "container-639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401",
				ContainerNameOrID: "639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401",
				RestartPolicy:     "always",
				PIDFile:           "/run/containers/storage/overlay-containers/639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401/userdata/conmon.pid",
				StopTimeout:       22,
				PodmanVersion:     "CI",
				EnvVariable:       EnvVariable,
			},
			goodID,
			false,
			false,
			false,
		},
		{"good with noHeader",
			containerInfo{
				Executable:        "/usr/bin/podman",
				ServiceName:       "container-639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401",
				ContainerNameOrID: "639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401",
				RestartPolicy:     "always",
				PIDFile:           "/run/containers/storage/overlay-containers/639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401/userdata/conmon.pid",
				StopTimeout:       22,
				PodmanVersion:     "CI",
				EnvVariable:       EnvVariable,
			},
			goodIDNoHeaderInfo,
			false,
			true,
			false,
		},
		{"good with name",
			containerInfo{
				Executable:        "/usr/bin/podman",
				ServiceName:       "container-foobar",
				ContainerNameOrID: "foobar",
				RestartPolicy:     "always",
				PIDFile:           "/run/containers/storage/overlay-containers/639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401/userdata/conmon.pid",
				StopTimeout:       10,
				PodmanVersion:     "CI",
				EnvVariable:       EnvVariable,
			},
			goodName,
			false,
			false,
			false,
		},
		{"good with name and bound to",
			containerInfo{
				Executable:        "/usr/bin/podman",
				ServiceName:       "container-foobar",
				ContainerNameOrID: "foobar",
				RestartPolicy:     "always",
				PIDFile:           "/run/containers/storage/overlay-containers/639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401/userdata/conmon.pid",
				StopTimeout:       10,
				PodmanVersion:     "CI",
				BoundToServices:   []string{"pod", "a", "b", "c"},
				EnvVariable:       EnvVariable,
			},
			goodNameBoundTo,
			false,
			false,
			false,
		},
		{"bad restart policy",
			containerInfo{
				Executable:    "/usr/bin/podman",
				ServiceName:   "639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401",
				RestartPolicy: "never",
				PIDFile:       "/run/containers/storage/overlay-containers/639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401/userdata/conmon.pid",
				StopTimeout:   10,
				PodmanVersion: "CI",
				EnvVariable:   EnvVariable,
			},
			"",
			false,
			false,
			true,
		},
		{"good with name and generic",
			containerInfo{
				Executable:        "/usr/bin/podman",
				ServiceName:       "jadda-jadda",
				ContainerNameOrID: "jadda-jadda",
				RestartPolicy:     "always",
				PIDFile:           "/run/containers/storage/overlay-containers/639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401/userdata/conmon.pid",
				StopTimeout:       10,
				PodmanVersion:     "CI",
				CreateCommand:     []string{"I'll get stripped", "container", "run", "--name", "jadda-jadda", "--hostname", "hello-world", "awesome-image:latest", "command", "arg1", "...", "argN", "foo=arg \"with \" space"},
				EnvVariable:       EnvVariable,
			},
			goodWithNameAndGeneric,
			true,
			false,
			false,
		},
		{"good with explicit short detach param",
			containerInfo{
				Executable:        "/usr/bin/podman",
				ServiceName:       "jadda-jadda",
				ContainerNameOrID: "jadda-jadda",
				RestartPolicy:     "always",
				PIDFile:           "/run/containers/storage/overlay-containers/639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401/userdata/conmon.pid",
				StopTimeout:       10,
				PodmanVersion:     "CI",
				CreateCommand:     []string{"I'll get stripped", "run", "-d", "--name", "jadda-jadda", "--hostname", "hello-world", "awesome-image:latest", "command", "arg1", "...", "argN"},
				EnvVariable:       EnvVariable,
			},
			goodWithExplicitShortDetachParam,
			true,
			false,
			false,
		},
		{"good with explicit short detach param and podInfo",
			containerInfo{
				Executable:        "/usr/bin/podman",
				ServiceName:       "jadda-jadda",
				ContainerNameOrID: "jadda-jadda",
				RestartPolicy:     "always",
				PIDFile:           "/run/containers/storage/overlay-containers/639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401/userdata/conmon.pid",
				StopTimeout:       10,
				PodmanVersion:     "CI",
				CreateCommand:     []string{"I'll get stripped", "run", "-d", "--name", "jadda-jadda", "--hostname", "hello-world", "awesome-image:latest", "command", "arg1", "...", "argN"},
				EnvVariable:       EnvVariable,
				Pod: &podInfo{
					PodIDFile: "%t/pod-foobar.pod-id-file",
				},
			},
			goodNameNewWithPodFile,
			true,
			false,
			false,
		},
		{"good with explicit full detach param",
			containerInfo{
				Executable:        "/usr/bin/podman",
				ServiceName:       "jadda-jadda",
				ContainerNameOrID: "jadda-jadda",
				RestartPolicy:     "always",
				PIDFile:           "/run/containers/storage/overlay-containers/639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401/userdata/conmon.pid",
				StopTimeout:       10,
				PodmanVersion:     "CI",
				CreateCommand:     []string{"I'll get stripped", "run", "--detach", "--name", "jadda-jadda", "--hostname", "hello-world", "awesome-image:latest", "command", "arg1", "...", "argN"},
				EnvVariable:       EnvVariable,
			},
			goodNameNewDetach,
			true,
			false,
			false,
		},
		{"good with id and no param",
			containerInfo{
				Executable:        "/usr/bin/podman",
				ServiceName:       "container-639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401",
				ContainerNameOrID: "639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401",
				RestartPolicy:     "always",
				PIDFile:           "/run/containers/storage/overlay-containers/639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401/userdata/conmon.pid",
				StopTimeout:       10,
				PodmanVersion:     "CI",
				CreateCommand:     []string{"I'll get stripped", "run", "awesome-image:latest"},
				EnvVariable:       EnvVariable,
			},
			goodIDNew,
			true,
			false,
			false,
		},
		{"good with explicit detach=true param",
			containerInfo{
				Executable:        "/usr/bin/podman",
				ServiceName:       "jadda-jadda",
				ContainerNameOrID: "jadda-jadda",
				RestartPolicy:     "always",
				PIDFile:           "/var/run/containers/storage/overlay-containers/639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401/userdata/conmon.pid",
				StopTimeout:       42,
				PodmanVersion:     "CI",
				CreateCommand:     []string{"I'll get stripped", "run", "--detach=true", "awesome-image:latest"},
				EnvVariable:       EnvVariable,
			},
			genGoodNewDetach("--detach=true"),
			true,
			false,
			false,
		},
		{"good with explicit detach=false param",
			containerInfo{
				Executable:        "/usr/bin/podman",
				ServiceName:       "jadda-jadda",
				ContainerNameOrID: "jadda-jadda",
				RestartPolicy:     "always",
				PIDFile:           "/var/run/containers/storage/overlay-containers/639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401/userdata/conmon.pid",
				StopTimeout:       42,
				PodmanVersion:     "CI",
				CreateCommand:     []string{"I'll get stripped", "run", "--detach=false", "awesome-image:latest"},
				EnvVariable:       EnvVariable,
			},
			genGoodNewDetach("-d"),
			true,
			false,
			false,
		},
		{"good with explicit detach=false param",
			containerInfo{
				Executable:        "/usr/bin/podman",
				ServiceName:       "jadda-jadda",
				ContainerNameOrID: "jadda-jadda",
				RestartPolicy:     "always",
				PIDFile:           "/var/run/containers/storage/overlay-containers/639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401/userdata/conmon.pid",
				StopTimeout:       42,
				PodmanVersion:     "CI",
				CreateCommand:     []string{"I'll get stripped", "run", "--name", "test", "-p", "80:80", "--detach=false", "awesome-image:latest", "somecmd", "--detach=false"},
				EnvVariable:       EnvVariable,
			},
			goodNameNewDetachFalseWithCmd,
			true,
			false,
			false,
		},
		{"good with multiple detach=false params",
			containerInfo{
				Executable:        "/usr/bin/podman",
				ServiceName:       "jadda-jadda",
				ContainerNameOrID: "jadda-jadda",
				RestartPolicy:     "always",
				PIDFile:           "/var/run/containers/storage/overlay-containers/639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401/userdata/conmon.pid",
				StopTimeout:       42,
				PodmanVersion:     "CI",
				CreateCommand:     []string{"I'll get stripped", "run", "--name", "test", "-p", "80:80", "--detach=false", "--detach=false", "awesome-image:latest", "somecmd", "--detach=false"},
				EnvVariable:       EnvVariable,
			},
			goodNameNewDetachFalseWithCmd,
			true,
			false,
			false,
		},
		{"good with multiple shorthand params detach first",
			containerInfo{
				Executable:        "/usr/bin/podman",
				ServiceName:       "jadda-jadda",
				ContainerNameOrID: "jadda-jadda",
				RestartPolicy:     "always",
				PIDFile:           "/var/run/containers/storage/overlay-containers/639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401/userdata/conmon.pid",
				StopTimeout:       42,
				PodmanVersion:     "CI",
				CreateCommand:     []string{"I'll get stripped", "run", "-dti", "awesome-image:latest"},
				EnvVariable:       EnvVariable,
			},
			genGoodNewDetach("-dti"),
			true,
			false,
			false,
		},
		{"good with multiple shorthand params detach last",
			containerInfo{
				Executable:        "/usr/bin/podman",
				ServiceName:       "jadda-jadda",
				ContainerNameOrID: "jadda-jadda",
				RestartPolicy:     "always",
				PIDFile:           "/var/run/containers/storage/overlay-containers/639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401/userdata/conmon.pid",
				StopTimeout:       42,
				PodmanVersion:     "CI",
				CreateCommand:     []string{"I'll get stripped", "run", "-tid", "awesome-image:latest"},
				EnvVariable:       EnvVariable,
			},
			genGoodNewDetach("-tid"),
			true,
			false,
			false,
		},
		{"good with root flags",
			containerInfo{
				Executable:        "/usr/bin/podman",
				ServiceName:       "jadda-jadda",
				ContainerNameOrID: "jadda-jadda",
				RestartPolicy:     "always",
				PIDFile:           "/var/run/containers/storage/overlay-containers/639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401/userdata/conmon.pid",
				StopTimeout:       42,
				PodmanVersion:     "CI",
				CreateCommand:     []string{"I'll get stripped", "--events-backend", "none", "--runroot", "/root", "run", "awesome-image:latest"},
				EnvVariable:       EnvVariable,
			},
			goodNewRootFlags,
			true,
			false,
			false,
		},
		{"good with container create",
			containerInfo{
				Executable:        "/usr/bin/podman",
				ServiceName:       "jadda-jadda",
				ContainerNameOrID: "jadda-jadda",
				RestartPolicy:     "always",
				PIDFile:           "/var/run/containers/storage/overlay-containers/639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401/userdata/conmon.pid",
				StopTimeout:       10,
				PodmanVersion:     "CI",
				CreateCommand:     []string{"I'll get stripped", "container", "create", "awesome-image:latest"},
				EnvVariable:       EnvVariable,
			},
			goodContainerCreate,
			true,
			false,
			false,
		},
		{"good with journald log tag (see #9034)",
			containerInfo{
				Executable:        "/usr/bin/podman",
				ServiceName:       "jadda-jadda",
				ContainerNameOrID: "jadda-jadda",
				RestartPolicy:     "always",
				PIDFile:           "/var/run/containers/storage/overlay-containers/639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401/userdata/conmon.pid",
				StopTimeout:       10,
				PodmanVersion:     "CI",
				CreateCommand:     []string{"I'll get stripped", "create", "--name", "test", "--log-driver=journald", "--log-opt=tag={{.Name}}", "awesome-image:latest"},
				EnvVariable:       EnvVariable,
			},
			goodNewWithJournaldTag,
			true,
			false,
			false,
		},
		{"good with special chars",
			containerInfo{
				Executable:        "/usr/bin/podman",
				ServiceName:       "jadda-jadda",
				ContainerNameOrID: "jadda-jadda",
				RestartPolicy:     "always",
				PIDFile:           "/var/run/containers/storage/overlay-containers/639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401/userdata/conmon.pid",
				StopTimeout:       10,
				PodmanVersion:     "CI",
				CreateCommand:     []string{"I'll get stripped", "create", "--name", "test", "awesome-image:latest", "sh", "-c", "kill $$ && echo %\\"},
				EnvVariable:       EnvVariable,
			},
			goodNewWithSpecialChars,
			true,
			false,
			false,
		},
	}
	for _, tt := range tests {
		test := tt
		t.Run(tt.name, func(t *testing.T) {
			opts := entities.GenerateSystemdOptions{
				New:      test.new,
				NoHeader: test.noHeader,
			}
			got, err := executeContainerTemplate(&test.info, opts)
			if (err != nil) != test.wantErr {
				t.Errorf("CreateContainerSystemdUnit() error = \n%v, wantErr \n%v", err, test.wantErr)
				return
			}
			assert.Equal(t, test.want, got)
		})
	}
}
